static char rcsid[] = "@(#)$Id: outexpires.c,v 2.3 2021/09/22 17:50:58 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.3 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_addr.h"

DEBUG_VAR(Debug,__FILE__,"addr");

#include <limits.h>

#define EXP_EXPIRES_magic		0xFC08

struct expanded_expires {
    unsigned short magic;              /* EXP_EXPIRES_magic */

    struct string  * textual;
    struct tm        value;
    int              days_ahead;

    time_t           cached_time;
    
    unsigned int     print_time :1;
};

/* Caller must free */
/* May return NULL */
struct string * get_expanded_expires_textual(ptr)
     const struct expanded_expires *ptr;
{
    if (EXP_EXPIRES_magic != ptr->magic)
	panic("HEADER PANIC",__FILE__,__LINE__,"get_expanded_expires_textual",
	      "Bad magic number (expanded_expires)",0);

    if (ptr->textual)
	return dup_string(ptr->textual);

    return NULL;
}

void set_expanded_expires_textual(ptr,textual)
     struct expanded_expires * ptr;
     const struct string     * textual;
{
    if (EXP_EXPIRES_magic != ptr->magic)
	panic("HEADER PANIC",__FILE__,__LINE__,"set_expanded_expires_textual",
	      "Bad magic number (expanded_expires)",0);

    if (ptr->textual)
	free_string(& (ptr->textual));

    if (textual) {
	ptr->textual = dup_string(textual);
    }
}

/* May return NULL */
const struct tm * get_expanded_expires_value(ptr)
     const struct expanded_expires *ptr;
{
    if (EXP_EXPIRES_magic != ptr->magic)
	panic("HEADER PANIC",__FILE__,__LINE__,"set_expanded_expires_value",
	      "Bad magic number (expanded_expires)",0);

    /* If 0 == ptr->value.tm_mday, then value is not set */
    if (ptr->value.tm_mday > 0)   /* Day of the month (1-31) */
	return &(ptr->value);

    return NULL;
}

/* Return -1 if not available */
int get_expanded_expires_days_ahead(ptr)
     const struct expanded_expires *ptr;
{
    if (EXP_EXPIRES_magic != ptr->magic)
	panic("HEADER PANIC",__FILE__,__LINE__,"get_expanded_expires_days_ahead",
	      "Bad magic number (expanded_expires)",0);
    
    if (ptr->days_ahead >= 0)
	return ptr->days_ahead;

    return -1;
}

enum print_time get_expanded_expires_print_time(ptr)
     const struct expanded_expires *ptr;
{
    if (EXP_EXPIRES_magic != ptr->magic)
	panic("HEADER PANIC",__FILE__,__LINE__,"get_expanded_expires_print_time",
	      "Bad magic number (expanded_expires)",0);

    return ptr->print_time ? print_date_and_time : print_date_only;
}

time_t expanded_get_time_t(tm)
     struct tm *tm;
{
    time_t time_value = 0;

#ifdef MKTIME
    time_value = mktime(tm);
    if ((time_t)-1 == time_value) {
	DPRINT(Debug,16,(&Debug,"expanded_get_time_t: mktime failed\n"));
    }
#endif

    if (! time_value ||
	(time_t)-1 == time_value) {
	long E = make_gmttime(1900+tm->tm_year,
			      1+tm->tm_mon,
			      tm->tm_mday,
			      tm->tm_hour,
			      tm->tm_min,
			      tm->tm_sec);
		
	if (E < LONG_MAX &&  E > 0 &&
	    E < (long)time_MAX)
	    time_value = E;
    }

    DPRINT(Debug,16,(&Debug,"expanded_get_time_t=%ld\n",
		     (long)time_value));
    
    return time_value;
}

/* May return 0 if fail or year < 1970 */
time_t get_expanded_expires_cached_time(ptr)
     struct expanded_expires *ptr;
{
    if (EXP_EXPIRES_magic != ptr->magic)
	panic("HEADER PANIC",__FILE__,__LINE__,
	      "get_expanded_expires_cached_time",
	      "Bad magic number (expanded_expires)",0);
    
    if (0 == ptr->cached_time) {

	/* If 0 == ret->value.tm_mday, then value is not set */
	if (ptr->value.tm_mday > 0) {  /* Day of the month (1-31) */
	
	    time_t expires = expanded_get_time_t(& (ptr->value));
	    
	    if ((time_t)-1 != expires &&
		expires > 0) {
		ptr->cached_time = expires;
		
		DPRINT(Debug,16,(&Debug,"get_expanded_expires_cached_time: cached_time %ld\n",
				 (long)(ptr->cached_time)));
	    }
	}
    }

    DPRINT(Debug,16,(&Debug,"get_expanded_expires_cached_time=%ld\n",
		     (long)(ptr->cached_time)));

    return ptr->cached_time;
}



struct expanded_expires  * new_expanded_expires(textual,value,days_ahead,print_time)
     const struct string * textual  /* may be NULL */;
     const struct tm     * value    /* may be NULL */;
     int days_ahead      /* -1 if not given or already expired */;
     enum print_time print_time;
{
    struct expanded_expires * ret = safe_zero_alloc(sizeof (* ret));

    ret->textual = NULL;
    if (textual) {
	ret->textual = dup_string(textual);

	DPRINT(Debug,16,(&Debug,"new_expanded_expires: textual %S\n",textual));
    }

    ret->value.tm_sec  = 0;
    ret->value.tm_min  = 0;
    ret->value.tm_hour = 0;
    ret->value.tm_mday = 0;
    ret->value.tm_mon  = 0;
    ret->value.tm_year = 0;
    ret->value.tm_wday = 0;
    ret->value.tm_yday = 0;
    ret->value.tm_isdst = -1;   /* Not available */

    ret->cached_time    = 0;
    
    if (days_ahead >= 0) {
	DPRINT(Debug,16,(&Debug,
			 "new_expanded_expires: days_ahead=%d\n",
			 days_ahead));
    }
    
    if (value) {
	char *c = asctime(value);
	
	ret->value = *value;

	if (c) {
	    /* asctime() includes \n */
	    DPRINT(Debug,16,(&Debug,"new_expanded_expires: value %s",c));
	}
	
	if (-1 == days_ahead) {
	    /* Estimate days ahead (not actually correct) 	       
	     */
	    time_t is_now  = 0;

	    if (((time_t) -1) != time(&is_now)) {
   
		time_t expires = expanded_get_time_t(& (ret->value));
		
		if ((time_t)-1 != expires &&
		    expires > 0) {
		    ret->cached_time = expires;
		    
		    DPRINT(Debug,16,(&Debug,"new_expanded_expires: cached_time %ld\n",
				     (long)(ret->cached_time)));
		}
		
		if ((time_t)-1 != expires &&
		    is_now < expires) {
		    time_t diff = expires - is_now;
		    
		    diff /= 24 * 60 * 60;
		    
		    if (diff < (long)INT_MAX) {
			days_ahead = diff;
			DPRINT(Debug,16,(&Debug,
					 "new_expanded_expires: estimated days_ahead=%d\n",
					 days_ahead));
		    }
		}
	    } else {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,16,(&Debug,
				 "new_expanded_expires: Date: not given, time: %s\n",
				strerror(err)));
		
	    }
	}
    } else if (days_ahead >= 0) {
	value = days_ahead_tm(days_ahead);
	if (value) {
	    char *c = asctime(value);
	    
	    ret->value = *value;

	    if (c) {
		/* asctime() includes \n */
		DPRINT(Debug,16,(&Debug,"new_expanded_expires: calculated value %s",c));
	    }
	}
    }
    
    ret->days_ahead = days_ahead;
    ret->print_time = 1;
    
    DPRINT(Debug,16,(&Debug,
		     "new_expanded_expires: print_time %d",print_time));
    switch (print_time) {
    case print_date_only:     ret->print_time = 0; DPRINT(Debug,16,(&Debug," print_date_only"));     break;
    case print_date_and_time: ret->print_time = 1; DPRINT(Debug,16,(&Debug," print_date_and_time")); break;
    }
    DPRINT(Debug,16,(&Debug,"\n"));
    
    ret->magic = EXP_EXPIRES_magic;

    return ret;
}

void free_expanded_expires(ptr)
     struct expanded_expires **ptr;
{
    if (EXP_EXPIRES_magic != (*ptr)->magic)
	panic("HEADER PANIC",__FILE__,__LINE__,
	      "free_expanded_expires",
	      "Bad magic number (expanded_expires)",0);

    if ((*ptr)->textual)
	free_string(& ((*ptr)->textual));

    (*ptr)->magic = 0; /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}




/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
