static char rcsid[] = "@(#)$Id: references.c,v 2.9 2019/03/11 18:23:55 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.9 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>  (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_addr.h"

DEBUG_VAR(Debug,__FILE__,"addr");


static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

/*  NOTE:
           Both In-Reply-To 
            and References
	   uses same syntax.
*/

#define REFERENCES_magic     0xE803
	   
struct references {

    unsigned short magic;        /* REFERENCES_magic */

    struct ref_item {
	
	/* Message id followed by phrase 
	   either one can be NULL;
	*/
	
	struct message_id   * id;
	struct string       * phrase;
	
	struct string       * tail_comment;

    }                   * items;
    int                   item_count;  

};

static void zero_ref_item P_((struct ref_item *x));
static void zero_ref_item(x)
     struct ref_item *x;
{
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)x,sizeof (*x));

    x->id            = NULL;
    x->phrase        = NULL;
    x->tail_comment  = NULL;
}

static void clear_ref_item P_((struct ref_item *x));
static void clear_ref_item(x)
     struct ref_item *x;
{
    if (x->id)
	free_message_id(& x->id);
    if (x->phrase)
	free_string(& x->phrase);
    if (x->tail_comment)
	free_string(& x->tail_comment);
}

     
struct references * new_references(id,phrase,comment) 
     const struct message_id  * id;
     const struct string      * phrase;
     const struct string      *comment;
{
    struct references  *ret = safe_malloc(sizeof(*ret));
  
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));
    
    ret->magic =  REFERENCES_magic;

    if (!id && !phrase && !comment) {
	ret->items      = NULL;
	ret->item_count = 0;
	return ret;
    }
		
    ret->items       = safe_malloc(sizeof(ret->items[0]));
    ret->item_count  = 1;

    zero_ref_item(& ret->items[0] );

    if (id)
	ret->items[0].id              = dup_message_id(id);
    if (phrase)
	ret->items[0].phrase          = dup_string(phrase);
    if (comment)
	ret->items[0].tail_comment    = dup_string(comment);

    return ret;
}


void references_add_item(ref,id,phrase,comment)
     struct references       * ref;
     const struct message_id * id;
     const struct string     * phrase;
     const struct string     * comment;
{
    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"references_add_item",
	      "Bad magic number",0);

    if (!id && !phrase && !comment) 
	return;

    ref->items       = safe_array_realloc(ref->items,
					  (ref->item_count+1),
					  sizeof(ref->items[0]));

    
    zero_ref_item(& ref->items[ref->item_count] );

    if (id)
	ref->items[ref->item_count].id              = dup_message_id(id);
    if (phrase)
	ref->items[ref->item_count].phrase          = dup_string(phrase);
    if (comment)
	ref->items[ref->item_count].tail_comment    = dup_string(comment);

    ref->item_count++;
}

void append_references_1 P_((struct references  * ref,
			     const struct references  * ref1,
			     int nophrase));

void append_references_1(ref,ref1,nophrase)
     struct references  * ref;
     const struct references  * ref1;
     int nophrase;
{
    int i,x;
    int new_len = ref->item_count + ref1->item_count;
    
    ref->items = safe_array_realloc(ref->items,
				    new_len,
				    sizeof(ref->items[0]));

    for (x = ref->item_count, i = 0;
	 i < ref1->item_count; i++, x++) {

	if (x >= new_len)
	    panic("ID PANIC",__FILE__,__LINE__,"append_references_1",
		  "Overflow",0);

	zero_ref_item(& ref->items[x] );

	if (ref1->items[i].id)
	    ref->items[i].id           = dup_message_id(ref1->items[i].id);

	if (nophrase)
	    continue;

	if (ref1->items[i].phrase)
	    ref->items[i].phrase       = dup_string(ref1->items[i].phrase);
	if (ref1->items[i].tail_comment)
	    ref->items[i].tail_comment = dup_string(ref1->items[i].tail_comment);	
    }
    
    if (x != new_len)
	panic("ID PANIC",__FILE__,__LINE__,"append_references_1",
	      "Wrong index",0);
    ref->item_count = new_len;
}



void append_references(ref,ref1)
     struct references  * ref;
     const struct references  * ref1;
{
    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"append_references",
	      "Bad magic number (ref)",0);

    if (REFERENCES_magic != ref1->magic)
	panic("ID PANIC",__FILE__,__LINE__,"append_references",
	      "Bad magic number (ref1)",0);
    

    if (ref1->item_count < 1)
	return;


    append_references_1(ref,ref1,0);
}



struct references * dup_references(ref) 
     const struct references *ref;
{
    struct references  *ret = safe_zero_alloc(sizeof(*ret));

    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"dup_references",
	      "Bad magic number",0);

    ret->magic =  REFERENCES_magic;

    ret->items      = NULL;
    ret->item_count = 0;

    if (ref->item_count < 1)
	return ret;

    append_references(ret,ref);

    return ret;
}

struct references * dup_references_no_phrase(ref) 
     const struct references *ref;
{
    struct references  *ret =  safe_zero_alloc(sizeof(*ret));

    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"dup_references_no_phrase",
	      "Bad magic number",0);

    ret->magic =  REFERENCES_magic;

    ret->items      = NULL;
    ret->item_count = 0;

    if (ref->item_count < 1)
	return ret;

    append_references_1(ret,ref,1);
    
    return ret;
}



int same_references(ref1,ref2) 
     const struct references *ref1;
     const struct references *ref2;
{
    int i;

    if (REFERENCES_magic != ref1->magic)
	panic("ID PANIC",__FILE__,__LINE__,"same_references",
	      "Bad magic number (ref1)",0);

    if (REFERENCES_magic != ref2->magic)
	panic("ID PANIC",__FILE__,__LINE__,"same_references",
	      "Bad magic number (ref2)",0);

    if (ref1->item_count != ref2->item_count)
	return 0;

    for (i = 0; i < ref1->item_count; i++) {
	
	if (ref1->items[i].id && ref2->items[i].id) {
	    if (! same_message_id(ref1->items[i].id,
				  ref2->items[i].id,
				  0))
		return 0;
	    
	} else if (ref1->items[i].id || ref2->items[i].id)
	    return 0;

	if (ref1->items[i].phrase && ref2->items[i].phrase) {

	    if (0 != string_cmp(ref1->items[i].phrase,ref2->items[i].phrase,99))
		return 0;
	    
	} else 	if (ref1->items[i].phrase || ref2->items[i].phrase)
	    return 0;

	if (ref1->items[i].tail_comment && ref2->items[i].tail_comment) {

	    if (0 != string_cmp(ref1->items[i].tail_comment,ref2->items[i].tail_comment,99))
		return 0;
	    
	} else 	if (ref1->items[i].tail_comment || ref2->items[i].tail_comment)
	    return 0;

    }
    
    return 1;
}

void free_references(ref)
     struct references  ** ref;
{

    if (REFERENCES_magic != (*ref)->magic)
	panic("ID PANIC",__FILE__,__LINE__,"free_references",
	      "Bad magic number",0);


    if ((*ref)->items) {
	int i;

	for (i = 0; i < (*ref)->item_count; i++) {
	    clear_ref_item(& (*ref)->items[i]);
	}


	free((*ref)->items);
	(*ref)->items = NULL;
    }
    (*ref)->item_count = 0;

    (*ref)->magic = 0;  /* Invalidate */
    free(*ref);
    *ref = NULL;
}

int references_item_count(ref)
     const struct references * ref;
{
    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"references_item_count",
	      "Bad magic number",0);
    
    return ref->item_count;
}

const struct message_id * references_get_message_id(ref,idx)
     const struct references * ref;
     int idx;
{
    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"references_get_message_id",
	      "Bad magic number",0);
    
    if (idx < 0 || idx >= ref->item_count)
	panic("ID PANIC",__FILE__,__LINE__,"references_get_message_id",
	      "Bad index",0);
    

    return ref->items[idx].id;
}

const struct string * references_get_phrase(ref,idx) 
     const struct references * ref;
     int idx;
{
    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"references_get_phrase",
	      "Bad magic number",0);
    
    if (idx < 0 || idx >= ref->item_count)
	panic("ID PANIC",__FILE__,__LINE__,"references_get_phrase",
	      "Bad index",0);
    

    return ref->items[idx].phrase;
}

const struct string * references_get_comment(ref,idx)
     const struct references * ref;
     int idx;
{
    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"references_get_comment",
	      "Bad magic number",0);
    
    if (idx < 0 || idx >= ref->item_count)
	panic("ID PANIC",__FILE__,__LINE__,"references_get_comment",
	      "Bad index",0);
    

    return ref->items[idx].tail_comment;
}

int header_references_have_phrase(ref)
     const struct references * ref;
{
    int i;

    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"header_references_have_phrase",
	      "Bad magic number",0);

    for (i = 0; i < ref->item_count; i++)
	if (ref->items[i].phrase)
	    return 1;

    return 0;
}

struct string * references_to_string(ref)
     const struct references *ref;
{
    struct string * ret = NULL;
    charset_t cs = NULL;
    int several = 0;
    int i;
    
    if (REFERENCES_magic != ref->magic)
	panic("ID PANIC",__FILE__,__LINE__,"header_references_have_phrase",
	      "Bad magic number",0);
    
    if (! ref->item_count)
	return NULL;

    for (i = 0; i < ref->item_count; i++) {
	if (ref->items[i].phrase) {
	    charset_t A = get_string_type(ref->items[i].phrase);

	    if (cs) {
		if (A != cs)
		    several++;
	    } else
		cs = A;
	}
    }

    if (several)
	cs = MIME_name_to_charset("UTF-8",0);
    if (!cs)
	cs = ASCII_SET;

   
    for (i = 0; i < ref->item_count; i++) {

	if (ref->items[i].id) {
	    struct string * tmp
		= message_id_to_string(ref->items[i].id);

	    if (!ret)
		ret = new_string(cs);
	    else
		add_ascii_to_string(ret,s2us(" "));

	    append_string(&ret,tmp,1);

	    free_string(&tmp);   
	}

	if (ref->items[i].phrase) {

	    if (!ret)
		ret = new_string(cs);
	    else
		add_ascii_to_string(ret,s2us(" "));
	    
	    append_string(&ret,ref->items[i].phrase,
			  1);
	}

	if (ref->items[i].tail_comment) {
	    struct string * tmp =
		string_quote_comment(ref->items[i].tail_comment);
	    
	    if (!ret)
		ret = new_string(cs);
	    else
		add_ascii_to_string(ret,s2us(" "));

	    add_ascii_to_string(ret,s2us("("));
	    append_string(&ret,tmp,1);
	    add_ascii_to_string(ret,s2us(")"));

	    free_string(&tmp);   
	}
    }

    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
