static char rcsid[] = "@(#)$Id: oldaliases.c,v 1.8 2018/11/14 19:01:33 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.8 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 lib/mk_aliases.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *
 *****************************************************************************/


#include "def_alias.h"
#include "s_newalias.h"

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

DEBUG_VAR(Debug,__FILE__,"alias");

static int get_line P_((FILE *file, char **buffer, int *ERR));

static int get_line(file, buffer, ERR)
     FILE *file;
     char **buffer;
     int *ERR;
{
    /* Based on get_line() on lib/mkaliases.c */

    int err_flag = *ERR;

    /*
     *	Read line from file.
     *	If first_line and buff_loaded, then just return!
     *	All comment and blank lines are just ignored (never passed back).
     */
    
    int len;
        
    /*
     * We will just ignore any line that begins with comment
     */
    do {
	/* malloc_gets:
	   - does not return \n
	   - returns -1 if limit exceeded, buffer is still alloced
	   - returns -2 if error or if feof() is true before reading anything
	*/
	    
	len =  malloc_gets(buffer,32000,file);

	DPRINT(Debug,12,(&Debug,"get_line: len=%d, buffer=",
			 len));
	DEBUG_PRINT_BUFFER(Debug,12,len,s2us(*buffer));
	DPRINT(Debug,12,(&Debug,"\n"));

	if (0 == len && feof(file)) {
	    /* EOF */
	    goto FAIL;	    
	}
	
    } while (0 == len ||
	     (len > 0 && (*buffer)[0] == '#'));
	
    if (-1 == len) {
	
	lib_error(CATGETS(elm_msg_cat,
			  NewaliasSet, NewaliasSplit,
			  "Line too long, split using continuation line format (starting line\nwith whitespace):\n%.40s\n"), 
		  *buffer ? *buffer : "???");
	
	err_flag++;
	goto FAIL;
	
    } else if (-2 == len && ferror(file)) {
	lib_error(CATGETS(elm_msg_cat,
			  NewaliasSet, NewaliasReadingError,
			  "Reading error"));
	err_flag++;
	goto FAIL;		
    } else if (-2 == len) {
	/* EOF */
	goto FAIL;	
    }

    *ERR = err_flag;   
    return(0);

 FAIL:
    *ERR = err_flag;
    return -1;
}

#define BUFFER_SIZE   128000

static int get_alias P_((FILE *file, char **buffer, int *ERR, char **next_buffer));
static int get_alias(file,buffer,ERR,next_buffer)
     FILE *file;
     char **buffer;
     int *ERR;
     char **next_buffer;
{
    /* Based on get_alias() on lib/mkaliases.c */

    int err_flag = *ERR;


    /* load buffer with the next complete alias from the file.
     * (this can include reading in multiple lines and appending
     * them all together!)  Returns EOF after last entry in file.
     *
     * Lines that start with '#' are assumed to be comments and are
     * ignored.  White space as the first field of a line is taken
     * to indicate that this line is a continuation of the previous. 
     */

    int    done = 0, len;

    /** get the first line of the entry... **/

    buffer[0] = '\0';			/* zero out line */
    len = 0;

    if (*next_buffer) {
	/* have chached line */

	free(*buffer);
	*buffer = *next_buffer;
	*next_buffer = NULL;
    } else {
	/* Read first line */
	
	if (get_line(file, buffer, &err_flag) == -1) 
	    goto FAIL;
    }
       
    len = strlen(*buffer);

    /** now read in the rest (if there is any!) **/
    
    do {

	if (get_line(file, next_buffer, &err_flag) == -1) {

	    if (*next_buffer) {
		free(*next_buffer);
		*next_buffer = NULL;
	    }

	    if (err_flag)
		goto FAIL;

	    goto OUT;
	}

	done = (! whitespace((*next_buffer)[0]));

	if (! done) {
	    char *s;

	    for (s = (*next_buffer); *s && whitespace(*s); s++) ;
	    *--s = ' ';
	    len += strlen(s);

	    if (len >= BUFFER_SIZE) {
		lib_error(CATGETS(elm_msg_cat,
				  NewaliasSet, NewaliasLineToLong,
				  "Line + continuations exceeds maximum length of %ld:"),
			  BUFFER_SIZE);
		lib_error(FRM("%.40s"), buffer);
		err_flag++;
	    } else {
		
		*buffer = strmcat(*buffer,*next_buffer);
		
		free(*next_buffer);
		*next_buffer = NULL;
	    }
	}
    } while (! done);
    
 OUT:
    *ERR = err_flag;

    return(0);	/* no sweat! */

 FAIL:
    *ERR = err_flag;
    return -1;

}


static int check_alias P_((char *aliases));
static int check_alias(aliases)
     char *aliases;
{
    /* Based on check_alias() on lib/mkaliases.c */
    
    /*
     *	Check and make sure this is a legal alias.
     */
    
    char *s, *out;
    int badws_flg = 0;
    
    /*
     *	First, strip out any whitespace (and make sure it was
     *	legal whitespace).  Legal whitespace follows ','.
     */
    for (s = aliases, out = aliases; *s; s++) {
	if (whitespace(*s)) {
	    if (*(out-1) != ',') {
		badws_flg++;		/* Keep going for now */
		*out++ = *s;
	    }
	}
	else {
	    *out++ = *s;
	}
    }
    *out = '\0';

    /*
     *	We have to delay reporting the error until the (legal)
     *	spaces are striped, otherwise aliases is all screwed
     *	up and doesn't display well at all.
     */
    if (badws_flg) {
	lib_error(CATGETS(elm_msg_cat,
			  NewaliasSet, NewaliasAliasWSNotAllowed,
			  "Error - whitespace in alias '%.30s' is not allowed."),
		  aliases);
	return(-1);
    }
    
    /*
     *	Verify the alias name is valid
     */
    for (s = aliases; *s != '\0' && (ok_alias_char(*s)||*s==','); ++s ) ;
    if ( *s != '\0' ) {
	lib_error(CATGETS(elm_msg_cat,
			  NewaliasSet, NewaliasCharNotSupported,
			  "Error - character '%c' in alias '%s' is not supported."),
		  *s, aliases);
	return(-1);
    }
    return(0);
}

int check_address P_((char *addresses));
int check_address(addresses)
     char *addresses;
{

    /* Based on check_address() on lib/mkaliases.c */

    char *s, *out;
    int in_quote = FALSE;
    int badws_flg = 0;

    /*
     *	Now strip out any whitespace (and make sure it was
     *	legal whitespace).  Legal whitespace follows ','.
     *	Don't mess with white space in quotes.
     */

    for (s = addresses, out = addresses; *s; s++) {
	if (*s == '"')
	    in_quote = !in_quote;
	
	if (!in_quote && whitespace(*s)) {
	    if (*(out-1) == ',') {
		continue;
	    }
	    else {
		badws_flg++;		/* Keep going for now */
	    }
	}
	*out++ = *s;
    }
    *out = '\0';

    /*
     *	We have to delay reporting the error until the (legal)
     *	spaces are striped, otherwise addresses is all screwed
     *	up and doesn't display well at all.
     */

    if (badws_flg) {
	lib_error(CATGETS(elm_msg_cat,
			  NewaliasSet, NewaliasAddressWSNotAllowed,
			  "Error - whitespace in address '%.30s' is not allowed."),
		  addresses);
	return(-1);
    }
    return(0);
}

static void de_escape P_((char *the_string));
static void de_escape(the_string)
     char *the_string;
{
	char *s, *out;

	for (s = the_string, out = the_string; *s; s++) {
	    if (*s != '\\')
		*out++ = *s;
	    else
		*out++ = *++s;
	}
	*out = '\0';

	return;
}

static void addr_list_set_phrase P_((struct addr_list *list,
				     const struct string *phrase));
static void addr_list_set_phrase(list,phrase)
     struct addr_list *list;
     const struct string *phrase;
{

    int count             = addr_list_item_count(list);

    if (1 == count) {
	int group;
	
	const struct address *addr = 
	    addr_list_get_item(list,0,&group);
	
	const struct string * phrase1 = address_get_phrase(addr);
	const char *          asc     = address_get_ascii_addr(addr);
	const struct string * com     = address_get_comment(addr);

	struct address  *     addr2 = NULL;

	if ((phrase1 && string_len(phrase1) > 0) || !asc)
	    goto as_list;
	
	addr2 = new_address(asc,phrase,com);

	addr_list_replace_item(list,0,addr2,group);
	
	free_address(&addr2);

    } else {
	int i;
	int group;

    as_list:

	group = add_group_to_list(list,phrase);

	for (i = 0; i < count; i++) {
	    int group1;
	
	    const struct address *addr = 
		addr_list_get_item(list,i,&group1);

	    if (-1 == group1)
		group1 = group;

	    addr_list_replace_item(list,i,addr,group1);
	}
    }
}

static void add_alias P_((struct aliases_map *map,charset_t  cs,
			  char * aliases, char *lastn, char *firstn, 
			  char *comment, char *addresses, int *ERR));
static void add_alias(map,cs,aliases,lastn,firstn,comment,addresses,ERR)
     struct aliases_map *map;
     charset_t  cs;
     char * aliases; 
     char *lastn; 
     char *firstn; 
     char *comment; 
     char *addresses; 
     int *ERR;
{
    /* Partially based on add_to_table() on lib/mkaliases.c */
    int err_flag = *ERR;
    char	*s;

    char ** tokenized = rfc822_tokenize(addresses);
    char **ptr, **next = NULL;

    struct address_alias * new_alias = NULL;


    struct string *comment_s = NULL;
  
    struct alias_vector *alias_list = NULL;
    struct addr_list    *addr_list = NULL;

    
    DPRINT(Debug,9,(&Debug,
		    "add_alias: aliases=%s lastn=%s firstn=%s comment=%s addresses=%s\n",
		    aliases ? aliases : "<NULL>",
		    firstn  ? firstn  : "<NULL>",
		    lastn   ? lastn   : "<NULL>", 
		    comment ? comment : "<NULL>",
		    addresses ? addresses : "<NULL>"));   

    /* Split addresses */
    
    for (ptr = tokenized; *ptr; ptr = next) {
	int q_seen = 0;
	int q = 0;
	
	/* Skip spaces after ',' */
	while (*ptr &&
	       whitespace ((*ptr)[0]))
	    ptr++;

	if (! *ptr)
	    break;
	           
	for (next = ptr; *next; next++) { 
	    if (',' == (*next)[0] && !q) {
		break;
	    }
	    else if ('<' == (*next)[0]) {
		q++;
		q_seen++;
	    } else if ('>' == (*next)[0])
		q--;
	}
	
	/* Spaces before ',' was not allowed, so
	   there is no need for check them
	*/

	if (ptr+1 == next) {
	    char * name = *ptr;

 	    if ('"' != name[0] && '\\' != name[0] &&
		'(' != name[0]) {

		/* possible alias -- NO decoding of MIME encoding */
		struct string *temp = new_string2(cs,s2us(name));
	    
		if (!alias_list)
		    alias_list = new_alias_vector();
		alias_vector_add_alias(alias_list,temp);
		
		free_string(&temp)
;
	    } else
		goto is_address;

	} else {  /* must have address .... */
	    char * buffer1;
	    char ** scanner;
	    
	is_address:
	    buffer1 = NULL;
	    scanner = NULL;

	    for (scanner = ptr; scanner < next; scanner++) {
		if ('<' == (*scanner)[0]) 
		    q++;
		else if ('>' == (*scanner)[0])
		    q--;
		buffer1 = strmcat(buffer1,*scanner);
	    }
	    
	    if (buffer1) {

		/* Allow MIME encoding here */

		struct addr_list * address_list =
		    parse_header_address(NULL,buffer1,1,cs,NULL); 

		if (addr_list) {
		    append_addr_list(addr_list,address_list);
		    free_addr_list(&address_list);
		} else {
		    addr_list = address_list;
		    address_list = NULL;
		}
	    }
	    
	}

	if (*next) {
	    if (',' != (*next)[0])
		panic("URL PANIC",__FILE__,__LINE__,"add_alias",
		      "'next' does not pint to ','",0);

	    next++;
	}

    }
    

    if (alias_list) {
	DPRINT(Debug,9,(&Debug,
			"add_alias: %d aliases\n",
			alias_vector_item_count(alias_list)));
    }

    if (addr_list) {
	DPRINT(Debug,9,(&Debug,
			"add_alias: %d addresses\n",
			addr_list_item_count(addr_list)));
    }



    /*  NO decoding of MIME encoding */
    if (comment)
	comment_s = new_string2(cs,s2us(comment));

	
    if (alias_list && !addr_list && !firstn) {
	/* 1) Alias group */
	
	/* Allow mime encoding used on here .... */
	struct string * phrase = hdr_to_string(HDR_PHRASE,lastn,cs,1);	

	new_alias = 
	    new_group_address_alias(phrase,comment_s,alias_list);

	if (phrase)
	    free_string(&phrase);

	DPRINT(Debug,9,(&Debug,"add_alias: Alias group\n"));

    } else if (alias_list && !addr_list && firstn &&
	       1 == alias_vector_item_count(alias_list)) {
	/* 2a) Person */

	const struct string * addr_token = alias_vector_get_alias(alias_list,0);

	struct address *addr = new_address_string( addr_token,NULL,NULL);

	/* Allow mime encoding used on here .... */
	struct string *firstname = 
	    hdr_to_string(HDR_PHRASE,firstn,cs,1);
	struct string *lastname =
	    hdr_to_string(HDR_PHRASE,lastn,cs,1);

	new_alias =
	    new_person_address_alias(firstname,lastname,
				     comment_s,addr);
	
	free_string(&firstname);
	free_string(&lastname);
       
	free_address(&addr);

	DPRINT(Debug,9,(&Debug,"add_alias: Person\n"));

    } else if (addr_list && 1 == addr_list_item_count(addr_list) &&
	       !alias_list && firstn) {
	int group;
	/* 2b) Person */

	/* Allow mime encoding used on here .... */
	struct string *firstname = 
	    hdr_to_string(HDR_PHRASE,firstn,cs,1);
	struct string *lastname =
	    hdr_to_string(HDR_PHRASE,lastn,cs,1);
	const struct address *addr = 
	    addr_list_get_item(addr_list,0,&group);

	new_alias =
	    new_person_address_alias(firstname,lastname,
				     comment_s,addr);
	
	free_string(&firstname);
	free_string(&lastname);
	
	DPRINT(Debug,9,(&Debug,"add_alias: Person\n"));

    } else if (addr_list && !alias_list && !firstn &&
	       !lastn[0]) {
	/* 3) Address list */

	if (lastn[0]) {
	    struct string *phrase = hdr_to_string(HDR_PHRASE,lastn,cs,1);

	    addr_list_set_phrase(addr_list,phrase);

	    free_string(&phrase);
	}

     

	new_alias =  new_list_address_alias(comment_s,addr_list);

	DPRINT(Debug,9,(&Debug,"add_alias: Address list\n"));

    } else {
	struct string *firstname = NULL;
	struct string *lastname = NULL;
	struct string *phrase = NULL;

	/* 4) generic case */

	if (firstn) {	   
	    firstname = hdr_to_string(HDR_PHRASE,firstn,cs,1);
	    lastname  = hdr_to_string(HDR_PHRASE,lastn,cs,1);
	} else if (alias_list) 
	    phrase = hdr_to_string(HDR_PHRASE,lastn,cs,1);	
	else if (addr_list) {
	    struct string *phrase1 = hdr_to_string(HDR_PHRASE,lastn,cs,1);

	    addr_list_set_phrase(addr_list,phrase1);

	    free_string(&phrase1);
	}
	    
	new_alias =
	    new_address_alias(firstname,lastname,comment_s,
			      phrase,addr_list,alias_list);

	if (firstname)
	    free_string(&firstname);
	if (lastname)
	    free_string(&lastname);
	if (phrase)
	    free_string(&phrase);

	DPRINT(Debug,9,(&Debug,"add_alias: Mixed (alias)\n"));
    }

    if (alias_list)
	free_alias_vector(&alias_list);

    if (addr_list)
	free_addr_list(&addr_list);
    
    if (comment_s)
	free_string(&comment_s);


    /*
     *	loop over each alias in aliases (split at the ,)
     */
    while (aliases != NULL) {

	struct string * aliasname = NULL;

	if ((s = index(aliases, ',')) != NULL)
	    *s++ = '\0';

	/*
	 *  Convert alias name to lower case 
	 */
	aliases = shift_lower(aliases);

	DPRINT(Debug,9,(&Debug,"add_alias: alias key: %s\n", 
			aliases));

	aliasname = new_string2(cs,s2us(aliases));

	aliases_map_add_alias(map,aliasname,new_alias);

	free_string(&aliasname);

	aliases = s;
    }
    
    /* OUT: */
    if (tokenized)
	free_rfc822tokenized(tokenized);

    if (new_alias)
	free_address_alias(&new_alias);


    *ERR = err_flag;     
}

static void put_alias P_(( struct aliases_map *map, char * buffer,int * ERR,
			   charset_t  cs));
static void put_alias(map, buffer, ERR,cs) 
     struct aliases_map *map;
     char *buffer;
     int *ERR;
     charset_t  cs;
{

    /* Based on put_alias() on lib/mkaliases.c */
     int err_flag = *ERR;

     /*
      *
      *	parse the buffer into aliases, names, comments and addresses
      *	and then add the alias
      */

     char *s, *aliases, *lastn, *firstn, *comment, *addresses = NULL;
     int	in_quote = FALSE;
     
     /*
      *	extract the alias name, its the part up to the first =, less the
      *	white space on the end.
      */

     
     aliases = buffer;
     if ((s = index(buffer, '=')) == NULL) {
	 lib_error(CATGETS(elm_msg_cat,
			   NewaliasSet, NewaliasNoFieldSep,
			   "Error - alias \"%.40s\" missing '=' field separator."),
		   aliases);
	 err_flag++;
	 goto OUT;
     }
     
     lastn = s + 1;
     while (--s >= aliases && whitespace(*s) ) ;
     *++s = '\0';
     
     if (check_alias(aliases) == -1) {
	 err_flag++;
	 goto OUT;
     }

/*
 *	get the second field into "lastn" - putting stuff after ','
 *	into "comment".  skip over white space after = and before last =
 */
     while (*lastn != '\0' && whitespace(*lastn) )
	 lastn++;
     
     for (s = lastn; *s; s++) {
	 if (*s == '\\') {
	     s++;
	     continue;
	 }
	 
	 if (*s == '"')
	     in_quote = !in_quote;
	 
	 if (in_quote)
	     continue;
	 
	 if (*s == '=') {
	     addresses = s + 1;
	     break;
	 }
     }
     
     if (*s != '=') {
	 lib_error(CATGETS(elm_msg_cat,
			   NewaliasSet, NewaliasNoFieldSep,
			   "Error - alias \"%.40s\" missing '=' field separator."),
		   aliases);
	 err_flag++;
	 goto OUT;
     }
     
     /*
      *	Remove trailing whitespace from second field
      */

     while (--s >= lastn && whitespace(*s) ) ;
     *++s = '\0';

     /*
      *	now get anything after a comma (marks comment within name field)
      */

     for (s = lastn, comment = NULL; *s; s++) {
	 if (*s == '\\') {
	     s++;
	     continue;
	 }
	 
	 if (*s == '"')
	     in_quote = !in_quote;
	 
	 if (in_quote)
	     continue;
	 
	 if (*s == ',') {
	     comment = s + 1;
	     while (--s >= lastn && whitespace(*s) ) ;
	     *++s = '\0';		/* Trailing whitespace... */
	     break;
	 }
     }
	
     /*
      *	strip leading whitespace from comment
      */

     if (comment) {
	 while (*comment != '\0' && whitespace(*comment) )
	     comment++;
     }


     /*
      *	remainder of line is the addresses, remove leading and
      *	trailing whitespace
      */

     while (*addresses != '\0' && whitespace(*addresses) )
	 addresses++;
     
     s = addresses + strlen(addresses);
     while (--s >= addresses && whitespace(*s) ) ;
     *++s = '\0';
     
     if (check_address(addresses) == -1) {
	 err_flag++;
	 goto OUT;
     }
     

     /*
      *	split the lastn field into firstn and lastn
      */

     for (s = lastn, firstn = NULL; *s; s++) {
	 if (*s == '\\') {
	     s++;
	     continue;
	 }
	 
	 if (*s == '"')
	     in_quote = !in_quote;
	 
	 if (in_quote)
	     continue;
	 
	 if (*s == ';') {
	     firstn = s + 1;
	     while (--s >= lastn && whitespace(*s) ) ;
	     *++s = '\0';		/* Trailing whitespace... */
	     break;
	 }
     }
     
     /*
      *	strip leading whitespace from firstn
      */
     if (firstn) {
	 while (*firstn && whitespace(*firstn))
	     firstn++;
     }
     
     /*
      *  now remove 'escapes' from name/comment and aliases fields, in place
      */
     
     de_escape(lastn);
     if (firstn) {
	 de_escape(firstn);
     }
     if (comment) {
	 de_escape(comment);
     }
     de_escape(addresses);
     
     add_alias(map,cs,aliases,lastn,firstn,comment,addresses,&err_flag);
	
 OUT:
     *ERR = err_flag;     
}


struct aliases_map * load_old_aliases_map(filename)
     const char *filename;
{
    /* Based on do_newalias() on lib/mkaliases.c */

    struct aliases_map *ret = NULL;

    charset_t  cs  = system_charset;    /* Charset of file */
    FILE * in = NULL;

    char * buffer = NULL;
    char * next_buffer = NULL;
    int  err_flag = 0;			/* if errors, don't save!    */


    int err = can_open(filename,"r");
    if (err) {
	lib_error(CATGETS(elm_msg_cat,
			  NewaliasSet, NewaliasNoOpenIn,
			  "Couldn't open %s for input!"), 
		  filename);
	return NULL;
    }
    
    if ((in = fopen(filename,"r")) == NULL) {
	lib_error(CATGETS(elm_msg_cat,
			  NewaliasSet, NewaliasNoOpenIn,
			  "Couldn't open %s for input!"), 
		  filename);

	return NULL;	
    }
    
    ret= new_aliases_map();

    while (get_alias(in,&buffer,&err_flag,&next_buffer) != -1) {
	put_alias(ret,buffer,&err_flag,cs);
	if (err_flag) break;
    } 

    if (buffer)
	free(buffer);
    if (next_buffer)
	free(next_buffer);

    fclose(in);


    if (err_flag) {
	lib_error(CATGETS(elm_msg_cat, NewaliasSet, NewaliasNoSave,
			  "** Not saving tables!  Please fix and re-run!"));

	free_aliases_map(&ret);
	return NULL;
    }

    return ret;
}

extern void delete_old_alias_files(dataname)
     char *dataname;
{
    /*
     *	This routine remove all the alias hash and data files.
     *	This is called from do_newalias() when there are no user
     *	aliases to be kept.
     */

    char fname[SLEN];
    
    if (unlink(dataname)) {
	lib_error(
		  CATGETS(elm_msg_cat, NewaliasSet, 
			  NewaliasCouldntDeleteData,
			  "Could not delete alias data file %s!"), 
		  dataname);	
    }
    
    elm_sfprintf(fname,sizeof fname,FRM("%s.dir"), dataname);
    if (unlink(fname)) {
	lib_error(
		  CATGETS(elm_msg_cat, NewaliasSet, 
			  NewaliasCouldntDeleteHash,
			  "Could not delete alias hash file %s!"), 
		  fname);
    }
    
    elm_sfprintf(fname,sizeof fname,FRM("%s.pag"), dataname);
    if (unlink(fname)) {
	lib_error(
		  CATGETS(elm_msg_cat, NewaliasSet, 
			  NewaliasCouldntDeleteHash,
			  "Could not delete alias hash file %s!"), fname);
    }    
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
