static char rcsid[] = "@(#)$Id: cancel.c,v 2.11 2021/01/23 20:39:30 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.11 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>  (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/


#include "elm_defs.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"ui");

static void cancel_set_ttysig_d P_((void));
static void cancel_set_ttysig_d()
{
    /* NULL */
}

static cancel_set_ttysig_f  * SET_TTYSIG = &cancel_set_ttysig_d;

static void cancel_reset_ttysig_d P_((void));
static void cancel_reset_ttysig_d()
{
    /* NULL */
}

static cancel_reset_ttysig_f * RESET_TTYSIG = &cancel_reset_ttysig_d;

static void cancel_transient_d P_((struct string *x,int reserve));
static void cancel_transient_d(x,reserve)   
     struct string *x;
     int reserve;
{
    lib_transient(FRM("%S"),x);
}

static cancel_transient_f   * TRANSIENT = &cancel_transient_d;

static void cancel_clear_d P_((void));
static void cancel_clear_d()
{
    lib_transient(FRM(""));
}

static cancel_clear_f       * CLEAR     = &cancel_clear_d;

static void cancel_progress_d P_((struct string *x, int *reserve,
				  struct string *progress));
static void cancel_progress_d(x,reserve,progress) 
     struct string *x; 
     int *reserve;
     struct string *progress;
{
    int l = string_len(progress);

    /* DUMMY */

    if (*reserve < l)
	*reserve = l;
}


static cancel_progress_f    * PROGRESS  = &cancel_progress_d;

void setup_cancel_cb(cancel_set_ttysig,cancel_reset_ttysig,
		     cancel_transient,cancel_clear,
		     cancel_progress)
     cancel_set_ttysig_f *cancel_set_ttysig;
     cancel_reset_ttysig_f *cancel_reset_ttysig;
     cancel_transient_f  *cancel_transient;
     cancel_clear_f      *cancel_clear;
     cancel_progress_f   *cancel_progress;
{
    SET_TTYSIG    = cancel_set_ttysig;
    RESET_TTYSIG  = cancel_reset_ttysig;
    TRANSIENT     = cancel_transient;
    CLEAR         = cancel_clear;
    PROGRESS      = cancel_progress;
}

#define CANCEL_magic            0xF903

struct cancel_data {
    unsigned short              magic;    /* CANCEL_magic */

    struct string         *msg;
    volatile int          is_canceled;

    int    progress_width;
    struct cancel_data * previous;
    int                  delay_msec;

    unsigned display_delayed : 1;

} * new_cancel P_((const char * format, const char *msg, ...));

static struct cancel_data  * volatile current = NULL;

#ifdef POSIX_SIGNALS
static struct sigaction saved_state;
static              int state_set = 0;

int is_cancel_installed() {

    DPRINT(Debug,10,(&Debug,
		     "is_cancel_installed=%d\n",
		     state_set));
    return state_set;
}

static void SA_HANDLER P_((int sig));
static void SA_HANDLER(sig)
     int sig;
{
    SIGDPRINT(Debug,1,(&Debug,"Got SIGINT ...\n"));

    if (!current) {
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "SA_HANDLER",
	      "current pointer is not set",1);
	return;
    }

    if (CANCEL_magic != current->magic)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "SA_HANDLER",
	      "Bad current pointer",1);

    current->is_canceled = 1;
}
#else
int is_cancel_installed() {

    DPRINT(Debug,10,(&Debug,
		     "is_cancel_installed=0 (no POSIX_SIGNALS)\n"));
    return 0;
}
#endif

static struct cancel_data * new_cancel_internal P_((struct string  *store));
static struct cancel_data * new_cancel_internal(store)
     struct string  *store;
{
    struct cancel_data *ret = safe_zero_alloc(sizeof (*ret));

    ret->magic         = CANCEL_magic;
    ret->msg           = store;
    ret->is_canceled   = 0;
    ret->progress_width = 0;
    ret->delay_msec    = 0;
    ret->display_delayed = 0;
    ret->previous      = NULL;

    if (current) {
	if (CANCEL_magic != current->magic)
	    panic("CANCEL PANIC",__FILE__,__LINE__,
		  "new_cancel",
		  "Bad current pointer",0);

	ret->previous = current;
    } else {
#ifdef POSIX_SIGNALS
	struct sigaction act;

	bzero((void *)&act,sizeof (act));

	act.sa_handler = &SA_HANDLER;
	sigemptyset(&(act.sa_mask)); /* No signal to mask while in handler */
	act.sa_flags = 0;
#ifdef SA_INTERRUPT
	act.sa_flags |= SA_INTERRUPT;           /* SunOS */
#endif

	state_set = 0;
	bzero((void *)&saved_state,sizeof (saved_state));
	
	current = ret;

	if (0 == sigaction(SIGINT,&act,&saved_state)) {
	    DPRINT(Debug,10,(&Debug,"SIGINT handler installed\n"));
	    state_set = 1;
	} else {
	    DPRINT(Debug,10,(&Debug,"Failed to set SIGINT handler\n"));
	}

#endif
	SET_TTYSIG();
    }
    current = ret;

    return ret;
}

void set_cancel_message (
#if ANSI_C
			 struct cancel_data *cancel,
			 int delay_msec               /* -1 if not schedule cancel */,
			 const char * format, 
			 const char * msg, ...
#else
			 cancel,delay_msec,format,msg,  va_alist
#endif
			 )
#if !ANSI_C
    struct cancel_data **cancel  /* creates if not exists */;
    int delay_msec               /* -1 if not schedule cancel */;
    const char * format;
    const char * msg;
#endif
{
    va_list vl;

    struct string  *store;

    Va_start(vl, msg);           /* defined in hdrs/elm_defs.h  */

    store = elm_smessage(160,format,msg,vl);

    va_end(vl);

    DPRINT(Debug,8,(&Debug,"set_cancel_message: message=%S\n",store));

    if (CANCEL_magic != cancel ->magic)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "set_cancel_message",
	      "Bad magic number (cancel_data)",0);

    if (cancel->msg)
	free_string(&  (cancel->msg));
    cancel->msg = store;

    if (delay_msec < 0) {
	DPRINT(Debug,8,(&Debug,"set_cancel_message: Normal cancel\n"));

	TRANSIENT(store,0);
	
	cancel->delay_msec    = 0;
	cancel->display_delayed = 0;
    } else {
	DPRINT(Debug,8,(&Debug,"set_cancel_message: Schedule cancel\n"));

	cancel->delay_msec    = delay_msec;
	cancel->display_delayed = 1;
    }
}

struct cancel_data * new_cancel (
#if ANSI_C
				  const char * format, 
				  const char *msg, 
				  ...
#else
				  format,msg, va_alist
#endif
				  )
#if !ANSI_C
     const char * format; 
     const char *msg;
     va_dcl
#endif
{
    va_list vl;

    struct string  *store;
    struct cancel_data *ret;

    Va_start(vl, msg);           /*  defined in hdrs/elm_defs.h */

    store = elm_smessage(160,format,msg,vl);

    va_end(vl);
    
    DPRINT(Debug,8,(&Debug,"new_cancel: message=%S\n",store));

    TRANSIENT(store,0);

    ret = new_cancel_internal(store);

    return ret;
}

void cancel_progress(
#if ANSI_C
		     struct cancel_data *cd,
		     const char * format, 
		     const char *msg, 
		     ...			     
#else
		     cd,format,msg, va_alist
#endif
		     )
#if !ANSI_C
    struct cancel_data *cd;
    const char * format;
    const char *msg;
    va_dcl
#endif
{
    va_list vl;

    struct string  *store;

    if (CANCEL_magic != cd->magic)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "cancel_progress",
	      "Bad pointer (magic)",0);


    Va_start(vl, msg);           /*  defined in hdrs/elm_defs.h */

    store = elm_smessage(80,format,msg,vl);

    va_end(vl);

    if (cd->display_delayed) {
	TRANSIENT(cd->msg,0);
	cd->display_delayed = 0;
    }

    DPRINT(Debug,8,(&Debug,"cancel_progress: message=%S\n",store));

    PROGRESS(cd->msg, & cd->progress_width,
	     store);  

    free_string(&store);
}

/* works only if schedule is waiting this */
struct cancel_data * new_schedule_cancel(
#if ANSI_C
					 int delay_msec,
					 const char * format, 
					 const char *msg, ...
#else
					 delay_msec,format,
					 msg,va_alist
#endif
		     )
#if !ANSI_C
    int delay_msec;
    const char * format;
    const char *msg;
    va_dcl
#endif
{
    va_list vl;

    struct string  *store;
    struct cancel_data *ret;

    Va_start(vl, msg);           /*  defined in hdrs/elm_defs.h */

    store = elm_smessage(160,format,msg,vl);

    va_end(vl);
    
    DPRINT(Debug,8,(&Debug,"new_schedule_cancel: message=%S\n",store));

    ret = new_cancel_internal(store);
    ret->display_delayed = 1;
    ret->delay_msec      = delay_msec;

    return ret;
}
  
int is_schedule_cancel(cd,delay_msec)
     struct cancel_data *cd;
     int * delay_msec;
{
    if (CANCEL_magic != cd->magic)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "is_schedule_cancel",
	      "Bad pointer (magic)",0);

    if (delay_msec)
	*delay_msec = cd->delay_msec;
    
    return cd->display_delayed;
}

void schedule_cancel_timeout(cd)
     struct cancel_data *cd;
{
    if (CANCEL_magic != cd->magic)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "schedule_cancel_timeout",
	      "Bad pointer (magic)",0);

    if (cd->display_delayed) {
    	TRANSIENT(cd->msg,0);
	cd->display_delayed = 0;
    }
}

int is_canceled(cd)
     struct cancel_data *cd;
{
    int ret;
    if (CANCEL_magic != cd->magic)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "is_canceled",
	      "Bad pointer (magic)",0);

    ret = cd->is_canceled;
    if (ret) {	
	if (ret < 2) {  /* If not already show */
	    struct string * X = 
		format_string(CATGETS(elm_msg_cat, MeSet, MeCanceled,
				      "%S canceled"),
			      cd->msg);
	    
	    DPRINT(Debug,8,(&Debug,"is_canceled: showing cancel message: %S\n",
			    X));
	    
	    cd->is_canceled = 2;
			    
	    TRANSIENT(X,0);
	    free_string(&X);
	} else {
	    DPRINT(Debug,8,(&Debug,"is_canceled: cancel message already shown, message=%S\n",
			    cd->msg));
	    
	    ret = 1;
	}
	    
	cd->display_delayed = 0;
    }

    DPRINT(Debug,12,(&Debug,"is_canceled=%d (message=%S)",
		     ret,cd->msg));
    if (cd->display_delayed) {
	DPRINT(Debug,12,(&Debug,", message display delayed"));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret; 
}

static struct cancel_data * found_displayed P_((struct cancel_data *cd));
static struct cancel_data * found_displayed(cd)
     struct cancel_data *cd;
{
    while (cd) {
	if (CANCEL_magic != cd->magic)
	    panic("CANCEL PANIC",__FILE__,__LINE__,
		  "found_displayed",
		  "Bad pointer (magic)",0);

	if (! cd->display_delayed)
	    break;
	cd = cd->previous;
    }

    return cd;
}

void free_cancel(cd)
     struct cancel_data **cd;
{
    struct cancel_data *x = *cd;

    if (CANCEL_magic != x->magic)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "free_cancel",
	      "Bad pointer (magic)",0);

    if (current != x)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "free_cancel",
	      "pointer not current",0);
	
    if (x->previous &&
	CANCEL_magic != x->previous->magic)
	panic("CANCEL PANIC",__FILE__,__LINE__,
	      "free_cancel",
	      "bad previous pointer",0);

    if (x->previous) {
	struct cancel_data * v;

	DPRINT(Debug,8,(&Debug,"free_cancel: previous message=%S\n",
			x->previous->msg));

	if (NULL != (v = found_displayed(x->previous))) {
	    
	    DPRINT(Debug,8,(&Debug,"free_cancel: displayed message=%S\n",
			    v->msg));
	    TRANSIENT(v->msg,v->progress_width);
	}

    } else {
	RESET_TTYSIG();
	CLEAR();

#ifdef POSIX_SIGNALS
	if (!state_set) {
	    DPRINT(Debug,10,(&Debug,"SIGINT was not handler installed\n"));
	} else if (0 == sigaction(SIGINT,&saved_state,NULL)) {
	    DPRINT(Debug,10,(&Debug,"SIGINT handler is reset\n"));
	} else {
	    DPRINT(Debug,1,(&Debug,"failed to reset SIGINT handler\n"));
	}
	state_set = 0;
#endif	
    } 

    current = x->previous;    
  
    free_string (& (x->msg) );
    x->magic = 0;
    free(x);
    x = NULL;

    *cd =  x;
}

int refresh_cancel_message()
{
    if (current) {
	struct cancel_data * v;

	if (CANCEL_magic != current->magic)
	    panic("CANCEL PANIC",__FILE__,__LINE__,
		  "free_cancel",
		  "Bad pointer (magic)",0);

	if (NULL != (v = found_displayed(current))) {

	    DPRINT(Debug,8,(&Debug,"refresh_cancel_message: displayed message=%S\n",
			    v->msg));
	    TRANSIENT(v->msg,v->progress_width);

	    if (v->is_canceled > 1) {  /* Also refressing canceled text */
		struct string * X = 
		    format_string(CATGETS(elm_msg_cat, MeSet, MeCanceled,
					  "%S canceled"),
				  v->msg);

		DPRINT(Debug,8,(&Debug,"refresh_cancel_message: showing cancel message: %S\n",
				X));
		
		TRANSIENT(X,0);
		free_string(&X);
	    }
	}

	return 1;
    }

    return 0;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
