static char rcsid[] = "@(#)$Id: mailer.c,v 2.43 2021/07/07 16:28:00 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.43 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mailer.h"
#include "s_elm.h"
#include "mailer_imp.h"
#ifdef USE_DLOPEN
#include "shared_imp.h"
#endif

#ifdef I_SYSEXITS
#include <sysexits.h>
#endif


DEBUG_VAR(Debug,__FILE__,"mailer");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

#if ANSI_C
MO_value_hook MO_default_value;
#endif
int MO_default_value (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    if (set) {
	if (!value || !*value) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailerOptionValue,
			      "Mailer option %s requires value"),
		      L->field);
	    return 0;
	}

	if (L->malloced) {
	    free(*(L->value));
	    *(L->value) = NULL;
	}

	*(L->value) = safe_strdup(*value);
	L->malloced = 1;	
    }
    else {
	*value = *(L->value);
    }
    return 1;
}


#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

S_(mailer_set_option default_set_option)
static int default_set_option(M,X,value) 
     struct mailer_config *M;
     struct mailer_option_list *X;
     char *value;
{
    int r ;

    r = X->value_func(M,X,&value,1);
    return r;
}

S_(mailer_init_hook default_mailer_init)
static int default_mailer_init P_((struct mailer_config *M,
				   struct mail_send_state *C,
				   struct mailer_info *I));

S_(mailer_init_hook submitmail_mailer_init)
static int submitmail_mailer_init P_((struct mailer_config *M,
				      struct mail_send_state *C,
				      struct mailer_info *I));



S_(mailer_close_hook default_mailer_close)
static void default_mailer_close P_((struct mailer_config *M,
				     struct mail_send_state *C));


S_(mailer_info_init_hook default_mailer_info_init)
static int default_mailer_info_init P_((struct mailer_config *M,
					struct mailer_info *I));


S_(mailer_info_close_hook null_mailer_info_close)
static void null_mailer_info_close(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{
    /* Nothing */
}

/* Return 
   -1 if caller should result with default data (return 1)  (mi_vrfyaddr_default_ok)
   -2 if caller should just test passwd                     (mi_vrfyaddr_test_passwd)
   0  if failure                                            (mi_vrfyaddr_failure)
   1  is succees                                            (mi_vrfyaddr_success)
*/
S_(mailer_info_verify_addr default_mailer_info_verify_addr)
static enum mi_vrfyaddr_mailer_status default_mailer_info_verify_addr(M,I,text,result)
     struct mailer_config *M;
     struct mailer_info  *I;
     const char *text;
     struct addr_verify_result *result;
{
    return  mi_vrfyaddr_test_passwd;   /* caller does job */
}

S_(mailer_info_gen_def_ef default_mailer_info_gen_def_ef)
static void default_mailer_info_gen_def_ef(M,I,X)
     struct mailer_config *M;
     struct mailer_info *I;
     struct mailer_env_from *X; 
{
    /* Nothing */
}

S_(mailer_info_set_ef default_mailer_info_set_ef)
static int default_mailer_info_set_ef
P_((struct mailer_config *M,
    struct mailer_info *I,
    struct mailer_env_from *X,
    const char *value));
static int default_mailer_info_set_ef(M,I,X,value)
     struct mailer_config *M;
     struct mailer_info *I;
     struct mailer_env_from *X;
     const char *value;
{
    /* Sets env from without checking */
    X->mail_from = strmcpy(X->mail_from,value);

    DPRINT(Debug,13,(&Debug, "default_mailer_info_set_ef: Clearing MAILER_ef_verified\n"));
    X->flags &= ~MAILER_ef_verified;
    return 1;
}

/* Return
 -1  if connection lost          (mailer_disconnected)
  0  if OK                       (mailer_not_restarted)
  1  if mailer reinitialized and  query_mailer_info()
        need to be called again  (mailer_reinitialized)
*/
S_(mailer_info_restart_hook default_mailer_info_rs_hook)
static enum restart_mailer_status
   default_mailer_info_rs_hook P_((struct mailer_config *M,
				   struct mailer_info *I));
static enum restart_mailer_status default_mailer_info_rs_hook(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{
    return mailer_not_restarted;  /* Assume not restarted */
}

S_(mailer_info_set_ef sendmail_mailer_info_set_ef)
static int sendmail_mailer_info_set_ef P_((struct mailer_config *M,
					     struct mailer_info *I,
					     struct mailer_env_from *X,
					     const char *value));

S_(mailer_info_verify_addr sendmail_mailer_info_verify_addr)
static enum mi_vrfyaddr_mailer_status
   sendmail_mailer_info_verify_addr P_((struct mailer_config *M,
					struct mailer_info  *I,
					const char *text,
					struct addr_verify_result *result));

S_(mailer_info_query_hook default_mailer_info_query)
static int default_mailer_info_query P_((struct mailer_config *M,
					 struct mailer_info *I,
					 enum MI_query query));

S_(mailer_info_s_query_hook default_minfo_s_query)
static char ** default_minfo_s_query P_((size_t               * reslen_p,
					 struct mailer_config * M,
					 struct mailer_info   * I,
					 enum MI_query_string   query,
					 struct cancel_data  ** cancel_p));

S_(mailer_info_query_hook sendmail_mailer_info_query)
static int sendmail_mailer_info_query P_((struct mailer_config *M,
					  struct mailer_info *I,
					  enum MI_query query));

S_(mailer_backend3_hook unknown_mailer_backend3)
static int unknown_mailer_backend3 P_((struct mailer_config *M,
				       struct mail_send_state **C,
				       int encoding_top,
				       char * title,
				       sending_message_func *sm,
				       int *exit_status,
				       struct mailer_cancel_ticket * ticket,
				       long message_size));

S_(mailer_backend3_hook sendmail_mailer_backend3)
static int sendmail_mailer_backend3 P_((struct mailer_config *M,
					struct mail_send_state **C,
					int encoding_top,
					char * title,
					sending_message_func *sm,
					int *exit_status,
					struct mailer_cancel_ticket * ticket,
					long message_size));

S_(mailer_backend3_hook submitmail_mailer_backend3)
static int submitmail_mailer_backend3 P_((struct mailer_config *M,
					  struct mail_send_state **C,
					  int encoding_top,
					  char * title,
					  sending_message_func *sm,
					  int *exit_status,
					  struct mailer_cancel_ticket * ticket,
					  long message_size));

S_(mailer_backend3_hook execmail_mailer_backend3)
static int execmail_mailer_backend3 P_((struct mailer_config *M,
					struct mail_send_state **C,
					int encoding_top,
					char * title,
					sending_message_func *sm,
					int *exit_status,
					struct mailer_cancel_ticket * ticket,
					long message_size));

static struct mailer_config  * selected_mailer = NULL;

static char * unknown_mailer_path     = DEFAULT_MAILER_PATH;
static char * sendmail_mailer_path    = SENDMAIL_MAILER_PATH;
static char * submitmail_mailer_path  = SUBMITMAIL_MAILER_PATH;
static char * execmail_mailer_path    = EXECMAIL_MAILER_PATH;

static enum bodytypes {
    have_7bit = 0, have_8bit = 1, have_binary = 2 
} sendmail_supported_bodytype   = have_7bit;
static int sendmail_supports_dsn = 0;
static int sendmail_verify_address = 0;

#ifdef MAILER_BV_OK
static int sendmail_verify_envelope = 1;
#else
static int sendmail_verify_envelope = 0;
#endif

S_(MO_value_hook MO_sendmail_bodytype)
static int MO_sendmail_bodytype P_((struct mailer_config      * M,
				    struct mailer_option_list * L,
				    char **value,int set));
static int MO_sendmail_bodytype (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *sendmail_bodytypes[] = { "7bit", "8bit", "binary" };
    static int value_set  = 0;
    
    if (set) {
	int i;

	if (!*value) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailerOptionValue,
			      "Mailer option %s requires value"),
		      L->field);
	    return 0;
	}
	
	for (i = 0; 
	     i < sizeof sendmail_bodytypes / sizeof (sendmail_bodytypes[0]);
	     i++) {
	    if (0 == strcmp(sendmail_bodytypes[i],*value)) {
		sendmail_supported_bodytype = (enum bodytypes)i;	       
		value_set = 1;
		return 1;
	    }
	}
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSendmailBadBodytype,
			  "Bad supported-bodytype %s for sendmail"),
		  *value);
	return 0;
    }
    else {
	if (sendmail_supported_bodytype < have_7bit ||
	    sendmail_supported_bodytype > have_binary)
	    panic("MAILER PANIC",__FILE__,__LINE__,
		  "MO_sendmail_bodytype",
		  "Bad supported-bodytype",0);

	*value = sendmail_bodytypes[sendmail_supported_bodytype];
	if (sendmail_supported_bodytype > have_7bit)
	    return 1;  /* Give value always */
    }
    return value_set; /* Return value only if set at least once */
}

S_(MO_value_hook MO_sendmail_dsn)
static int MO_sendmail_dsn P_((struct mailer_config      * M,
			       struct mailer_option_list * L,
			       char **value,int set));
static int MO_sendmail_dsn (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	    sendmail_supports_dsn = 1;
	else if (0 == strcmp(*value,no_yes[0]))
	    sendmail_supports_dsn = 0;
	else if (0 == strcmp(*value,no_yes[1])) 
	    sendmail_supports_dsn = 1;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSendmailBadDsn,
			      "Bad supports-dsn=%s for sendmail"),
		      *value);
	    return 0;
	}	
	value_set = 1;
    } else {
	/* Do not return option if sendmail_supports_dsn is not supported */

	if (sendmail_supports_dsn) {
	    *value = no_yes[1];
	    return 1;
	}
	*value = no_yes[0]; /* Only show if supports-dsn=no or
			       supports-dsn=yes is given at least once 
			    */
    }    
    return value_set;
}

S_(MO_value_hook MO_sendmail_verify)
static int MO_sendmail_verify P_((struct mailer_config      * M,
			       struct mailer_option_list * L,
			       char **value,int set));
static int MO_sendmail_verify (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	     sendmail_verify_address = 1;
	else if (0 == strcmp(*value,no_yes[0]))
	    sendmail_verify_address = 0;
	else if (0 == strcmp(*value,no_yes[1])) 
	    sendmail_verify_address = 1;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSendmailBadVerify,
			      "Bad verify=%s for sendmail"),
		      *value);
	    return 0;
	}	
	value_set = 1;
    } else {
	if (sendmail_verify_address) {
	    *value = no_yes[1];
	    return 1;
	}
	*value = no_yes[0]; /* Only show if verify=no or
			       verify=yes is given at least once 
			    */
    }    
    return value_set;
}


S_(MO_value_hook MO_sendmail_verify_sender)
static int MO_sendmail_verify_sender P_((struct mailer_config      * M,
					 struct mailer_option_list * L,
					 char **value,int set));
static int MO_sendmail_verify_sender (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	    sendmail_verify_envelope = 1;
	else if (0 == strcmp(*value,no_yes[0]))
	    sendmail_verify_envelope = 0;
	else if (0 == strcmp(*value,no_yes[1])) 
	    sendmail_verify_envelope = 1;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSendmailBadVerifySender,
			      "Bad verify-set-sender=%s for sendmail"),
		      *value);
	    return 0;
	}	
	value_set = 1;
    } else {
	if (sendmail_verify_envelope) {
	    *value = no_yes[1];
	    return 1;
	}
	*value = no_yes[0]; /* Only show if verify=no or
			       verify=yes is given at least once 
			    */
    }    
    return value_set;
}


S_(MO_value_hook MO_dont_add_from)
static int MO_dont_add_from P_((struct mailer_config      * M,
			       struct mailer_option_list * L,
			       char **value,int set));
static int MO_dont_add_from (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *yes_dont[] = { "yes", "dont" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	    M -> mailer_bits  &= ~MB_DONT_ADD_FROM;
	else if (0 == strcmp(*value,yes_dont[0]))
	    M -> mailer_bits  &= ~MB_DONT_ADD_FROM;
	else if (0 == strcmp(*value,yes_dont[1])) 
	    M -> mailer_bits  |= MB_DONT_ADD_FROM;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadAddFrom,
			      "Bad add-from=%s for %s"),
		      *value, M->mailer_type);
	    return 0;
	}
	value_set = 1;
    } else {
	
	*value = yes_dont[ (M -> mailer_bits & MB_DONT_ADD_FROM) ?
			 1 : 0 ];

	if (M -> mailer_bits & MB_DONT_ADD_FROM)
	    return 1;

	/* Do not show add-from=yes if it is not set explicity */
    }

    return value_set;
}

S_(MO_value_hook MO_use_domain)
static int MO_use_domain P_((struct mailer_config      * M,
			     struct mailer_option_list * L,
			     char **value,int set));
static int MO_use_domain (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	    M -> mailer_bits  |= MB_USE_DOMAIN;
	else if (0 == strcmp(*value,no_yes[0]))
	    M -> mailer_bits  &= ~MB_USE_DOMAIN;
	else if (0 == strcmp(*value,no_yes[1])) 
	    M -> mailer_bits  |= MB_USE_DOMAIN;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadUseDomain,
			      "Bad use-domain=%s for %s"),
		      *value, M->mailer_type);
	    return 0;
	}
	value_set = 1;
    } else {
	
	*value = no_yes[ (M -> mailer_bits & MB_USE_DOMAIN) ?
			 1 : 0 ];
       
	if (!(M -> mailer_bits & MB_USE_DOMAIN))
	    return 1;

	/* Do not show use-domain=yes if it is not set explicity */
    }

    return value_set;
}

#if ANSI_C
MO_value_hook MO_allow_set_sender;
#endif
int MO_allow_set_sender (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	    M -> mailer_bits  |= MB_ALLOW_SET_SENDER;
	else if (0 == strcmp(*value,no_yes[0]))
	    M -> mailer_bits  &= ~MB_ALLOW_SET_SENDER;
	else if (0 == strcmp(*value,no_yes[1])) 
	    M -> mailer_bits  |= MB_ALLOW_SET_SENDER;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadAllowSetSender,
			      "Bad allow-set-sender=%s for %s"),
		      *value, M->mailer_type);
	    return 0;
	}
	value_set = 1;
    } else {
	
	*value = no_yes[ (M -> mailer_bits & MB_ALLOW_SET_SENDER) ?
			 1 : 0 ];
       
	if (M -> mailer_bits & MB_ALLOW_SET_SENDER)
	    return 1;

	/* Do not show use-domain=no if it is not set explicity */
    }

    return value_set;
}

/* -------- */

S_(MO_value_hook MO_sending_display)
static int MO_sending_display P_((struct mailer_config      * M,
				  struct mailer_option_list * L,
				  char **value,int set));
static int MO_sending_display (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	    M -> mailer_bits  |= MB_SENDING_DISPLAY;
	else if (0 == strcmp(*value,no_yes[0]))
	    M -> mailer_bits  &= ~MB_SENDING_DISPLAY;
	else if (0 == strcmp(*value,no_yes[1])) 
	    M -> mailer_bits  |= MB_SENDING_DISPLAY;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmBadSendingDisplay,
			      "Bad sending-display=%s for %s"),
		      *value, M->mailer_type);
	    return 0;
	}

#ifdef BACKGROUD_PROCESSES 
	if (0 != (M -> mailer_bits & MB_SENDING_DISPLAY) &&
	    background_wait_time)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmSendingDisplayNo,
			      "sending-display=%s for %s do not work with background-wait-time=%d"),
		      *value,M->mailer_type,background_wait_time);
#endif

	value_set = 1;
    } else {
	
	*value = no_yes[ (M -> mailer_bits & MB_SENDING_DISPLAY) ?
			 1 : 0 ];
       
	if (M -> mailer_bits & MB_SENDING_DISPLAY)
	    return 1;

	/* Do not show sending-display=no if it is not set explicity */
    }

    return value_set;
}


static struct mailer_option_list UNKNOWN_options[] = {
    { "path",     &unknown_mailer_path,          MO_default_value, 0 },
    { "add-from", NULL,                          MO_dont_add_from, 0 },
    { "use-domain", NULL,                        MO_use_domain, 0 },
    { "sending-display", NULL,                   MO_sending_display, 0 },
    { NULL, NULL,                                MO_default_value, 0 }    
};

static struct mailer_option_list SENDMAIL_options[] = {
    { "path",     &sendmail_mailer_path,          MO_default_value, 0 },
    { "add-from", NULL,                           MO_dont_add_from, 0 },
    { "use-domain", NULL,                         MO_use_domain, 0 },
    { "supported-bodytype", NULL,                 MO_sendmail_bodytype, 0 },
    { "supports-dsn", NULL,                       MO_sendmail_dsn,      0 },
    { "verify", NULL,                             MO_sendmail_verify,      0 },
    { "allow-set-sender",NULL,                    MO_allow_set_sender,     0 },
    { "verify-set-sender", NULL,                  MO_sendmail_verify_sender,    0 },
    { "sending-display", NULL,                    MO_sending_display, 0 },
    { NULL, NULL,                                 MO_default_value, 0 }
};

static struct mailer_option_list SUBMITMAIL_options[] = {
    { "path",     &submitmail_mailer_path,            MO_default_value, 0 },
    { "add-from", NULL,                          MO_dont_add_from, 0 },
    { "use-domain", NULL,                        MO_use_domain, 0 },
    /* sending-display can not be used */
    { NULL, NULL,                                MO_default_value, 0 }
};
static struct mailer_option_list EXECMAIL_options[] = {
    { "path",     &execmail_mailer_path,         MO_default_value, 0 },
    { "add-from", NULL,                          MO_dont_add_from, 0 },
    { "use-domain", NULL,                        MO_use_domain, 0 },
    { "sending-display", NULL,                    MO_sending_display, 0 },
    { NULL, NULL,                                MO_default_value, 0 }
};


static struct mailer_config MAILERS[] = {
    { MAILER_CONFIG_magic,
      "unknown", &(UNKNOWN_options[0]), default_set_option, 
      default_mailer_init, default_mailer_close, unknown_mailer_backend3,
      &unknown_mailer_path,  
      default_mailer_info_init, null_mailer_info_close,
      default_mailer_info_query, default_mailer_info_verify_addr,
      default_mailer_info_gen_def_ef, default_mailer_info_set_ef,
      default_mailer_info_rs_hook,default_minfo_s_query,

      MB_USE_DOMAIN
    },
    { MAILER_CONFIG_magic,
      "sendmail", &(SENDMAIL_options[0]), default_set_option, 
      default_mailer_init, default_mailer_close, sendmail_mailer_backend3,
      &sendmail_mailer_path, 
      default_mailer_info_init, null_mailer_info_close,
      sendmail_mailer_info_query, sendmail_mailer_info_verify_addr,
      default_mailer_info_gen_def_ef, sendmail_mailer_info_set_ef,  
      default_mailer_info_rs_hook, default_minfo_s_query,

      MB_DONT_ADD_FROM
    },
    { MAILER_CONFIG_magic,
      "submitmail", &(SUBMITMAIL_options[0]), default_set_option, 
      submitmail_mailer_init, default_mailer_close, submitmail_mailer_backend3,
      &submitmail_mailer_path, 
      default_mailer_info_init, null_mailer_info_close,
      default_mailer_info_query, default_mailer_info_verify_addr,
      default_mailer_info_gen_def_ef, default_mailer_info_set_ef,
      default_mailer_info_rs_hook, default_minfo_s_query,

      MB_USE_DOMAIN
    },
    { MAILER_CONFIG_magic,
      "execmail", &(EXECMAIL_options[0]), default_set_option, 
      default_mailer_init, default_mailer_close, execmail_mailer_backend3,
      &execmail_mailer_path, 
      default_mailer_info_init, null_mailer_info_close,
      default_mailer_info_query, default_mailer_info_verify_addr,
      default_mailer_info_gen_def_ef, default_mailer_info_set_ef,
      default_mailer_info_rs_hook, default_minfo_s_query,

      MB_DONT_ADD_FROM
    }
};

static int valid_mailer P_((struct mailer_config *T));
static int valid_mailer(T)
     struct mailer_config *T;
{
    int x;

    for (x = 0; x < (sizeof MAILERS) / sizeof (MAILERS[0]); x++)
	if (T == &MAILERS[x])
	    return 1;
#ifdef USE_DLOPEN
    for (x = 0; x < shared_MCF_type_count; x++)
	if (T == shared_MCF_types[x].T)
	    return 1;
#endif
    return 0;
}

static int default_mailer_info_init(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{

    if (M->mailer_path) {
	/* CHECK that mailer is executable ... */
	
	if (! *(M->mailer_path) ||
	    !(*(M->mailer_path))[0] ||
	    0 == strcmp(*(M->mailer_path),"none")
	    ) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoMailerPath,
			      "Mailer path for %s not specified"),
		  M->mailer_type);
	    return 0;
	}
	
	if (0 != access(*(M->mailer_path),EXECUTE_ACCESS)) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNotExecutable,
			      "%s not executable: %s"),
		      *(M->mailer_path),strerror(err));
	    return 0;
	}
    }

    return 1;
}

static char ** default_minfo_s_query(reslen_p,M,I,query,cancel_p)
     size_t                * reslen_p;
     struct mailer_config  * M;
     struct mailer_info    * I;
     enum MI_query_string    query;
     struct cancel_data   ** cancel_p;
{

    *reslen_p = 0;

    return NULL;
}


static int default_mailer_info_query(M,I,query)
     struct mailer_config *M;
     struct mailer_info *I;
     enum MI_query query;
{
    /* Answer to universal queries */

    switch(query) {
    case MI_DONT_ADD_FROM:
	return (M -> mailer_bits & MB_DONT_ADD_FROM) ? 1 : 0;
    case MI_USE_DOMAIN:
	return (M -> mailer_bits & MB_USE_DOMAIN) ? 1 : 0;
    case MI_HAVE_SENDING_DISPLAY:
	return (M -> mailer_bits & MB_SENDING_DISPLAY) ? 1 : 0;
    case MI_REMOTE_MAILER:
	return 0;  /* Assume local mailer */    
    default:
	return 0; /* Not available */
    }

}

static int sendmail_mailer_info_query(M,I,query)
     struct mailer_config *M;
     struct mailer_info *I;
     enum MI_query query;
{
    switch(query) {

    case MI_HAVE_8BITMIME:
	return  sendmail_supported_bodytype >= have_8bit;

    case MI_HAVE_BINARYMIME:
	return  sendmail_supported_bodytype >= have_binary;

    case MI_HAVE_DSN:
	return sendmail_supports_dsn;

    default:
	return default_mailer_info_query(M,I,query);
    }
}


static char * delay_parsing_info = NULL;

static int check_mailer P_((int read_flags /* READ_FLAG_IGNORE_MISSING */));
static int check_mailer(read_flags)
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    if (selected_mailer)
	return 1;
    if (delay_parsing_info) {
	int r = mailerfunc(&delay_parsing_info,-1,0,
			   NULL,read_flags);
	free(delay_parsing_info); delay_parsing_info = NULL;
	return r;
    }
    return 0;
}

#ifdef ANSI_C
option_func mailerfunc;
#endif
int mailerfunc(value,enter,lineno,filename,read_flags)
     char **value; 
     int enter;
     int lineno; 
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ok = 1;

    if (enter) {
	int i;
	char * WALK;

	char * temp = safe_strdup(*value);
	
	char * f = mime_parse_content_opts(temp, &WALK);
	
	if (!f) {
	    free(temp);
	    return 0;
	}

	for (i = 0; i < (sizeof MAILERS) / sizeof (MAILERS[0]); i++)
	    if (0 == strcmp(f,MAILERS[i].mailer_type))
		break;
	if (i >= (sizeof MAILERS) / sizeof (MAILERS[0])) {
	    
#ifdef USE_DLOPEN
	    if (enter > 0) {
		DPRINT(Debug,8,(&Debug,"mailerfunc: Parsing delayed\n"));

		selected_mailer = NULL;
		delay_parsing_info = strmcpy(delay_parsing_info,*value);
		free(temp);
		return 1;		    
	    }

	    selected_mailer = loc_mailer_type(f);
	    if (selected_mailer)
		goto found_shared;
#endif	    

	    if (filename)
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknowMailerTypeFile,
				  "Unknown mailer type %s on line %d on file %s"),
			  f,lineno,filename);
	    else
	    
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknowMailerType,
				  "Unknown mailer type %s"),
			  f);

	    free(temp);

	    mark_flocal_changed(mailerfunc,0);

	    return 0;
	}
	selected_mailer = &(MAILERS[i]);

#ifdef USE_DLOPEN
    found_shared:
#endif

	DPRINT(Debug,8,(&Debug,
			"mailerfunc: Selected %s\n",
			selected_mailer->mailer_type));

	while (NULL != (f = mime_parse_content_opts(NULL, &WALK))) {
	    char * q = strchr(f,'=');

	    if (q) {
		char * x = q;
 
		while (x > f && ' ' == *(x-1))
		    x--;
		*x = '\0';

		q++;
	    
		while (' ' == *q)
		    q++;
	    }

	    for (i = 0; selected_mailer->list[i].field; i++) 
		if ( 0 == strcmp(f,selected_mailer->list[i].field))
		    break;
	    if (!selected_mailer->list[i].field) {
		if (filename)
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmUnknowMailerOptionFile,
				  "Unknown mailer option %s on line %d on file %s"),
			  f,lineno,filename);

		    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUnknowMailerOption,
				  "Unknown mailer option %s"),
			  f);
		ok = 0;
		continue;
	    }

	    if (q) {
		char * val = dequote_opt(q,strlen(q));

		if (!selected_mailer->m_set_option(selected_mailer,
                                                   &(selected_mailer->list[i]),
                                                   val))
		    ok = 0;
		free(val);
	    } else {
		if (!selected_mailer->m_set_option(selected_mailer,
						   &(selected_mailer->list[i]),
						   NULL))
		    ok = 0;
	    }
	}

	if (selected_mailer->mailer_path &&
	    (!*(selected_mailer -> mailer_path) ||
	     !(*(selected_mailer -> mailer_path))[0] ||
	     0 == strcmp(*(selected_mailer -> mailer_path),"none"))) {

	    if (2 == enter) {  /* called from init_default_mailer() */
		DPRINT(Debug,8,(&Debug,
				 "mailerfunc: Mailer path for %s not specified\n",
				selected_mailer->mailer_type));

	    } else {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoMailerPath,
				  "Mailer path for %s not specified"),
			  selected_mailer->mailer_type);

	    }
	    ok = 0;
	}

	free(temp);
    } else {
	if (delay_parsing_info) {
	    *value = delay_parsing_info;
	    DPRINT(Debug,8,(&Debug,
			    "mailerfunc: Returning delayed parsing info\n"));
	} else {
	    /* static pointer to buffer accross invocations */
	    static char * return_buffer = NULL;
	    int i;
	    
	    if (!check_mailer(read_flags)) {
		selected_mailer = &(MAILERS[0]);

		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailerTypeAssumed,
				  "Mailer (type %s) assumed"),
			  selected_mailer->mailer_type);	    
	    }

	    return_buffer = strmcpy(return_buffer,
				    selected_mailer->mailer_type);  
	    
	    for (i = 0; selected_mailer->list[i].field; i++) {
		char * value = NULL;
		
		if (selected_mailer->
		    list[i].value_func(selected_mailer,
				       & (selected_mailer->list[i]),
				       &value,0)) {
		    
		    return_buffer = strmcat(return_buffer,"; ");
		    
		    if (value) {
			char *Q = elm_message(FRM("%s=%Q"),
					      selected_mailer->list[i].field,
					      value);
			return_buffer = strmcat(return_buffer,Q);
			free(Q);
		    } else
			return_buffer = strmcat(return_buffer,
						selected_mailer->list[i].field);
		}
	    }
	    *value = return_buffer;
	}
    }
    return ok;
}

void init_default_mailer(read_flags)
    int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    const char * p;
    char * type;

    /* INIT default mailer if not given on configuration file */

    if (check_mailer(read_flags))
	return;

    p = strrchr(DEFAULT_MAILER_PATH,'/');
    if (p)
	p++;
    else {
	DPRINT(Debug,1,(&Debug,
			"On mailer path %s have not / character!\n",
			DEFAULT_MAILER_PATH));
	p = DEFAULT_MAILER_PATH;
    } 

    if (0 == strcmp(p,"sendmail")) 
	type = "sendmail";
    else if (0 == strcmp(p,"submit"))
	type = "submitmail";
    else if (0 == strcmp(p,"execmail"))
	type = "execmail";
    else
	type = "unknown";

    if (!mailerfunc(&type,2,0,NULL,read_flags) ||
	!check_mailer(read_flags)) {
	DPRINT(Debug,1,(&Debug,
			"Calling of mailerfunc for setting of default mailer failed\n"));
	return;
    }

    if (selected_mailer -> mailer_path) {
	*(selected_mailer -> mailer_path) = DEFAULT_MAILER_PATH;

	if (!*(selected_mailer -> mailer_path) ||
	    !(*(selected_mailer -> mailer_path))[0] ||
	    0 == strcmp(*(selected_mailer -> mailer_path),"none")) {
	
            lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoMailerPath1,
                              "Give mailer path for mailer = %s on global elm.rc"),
                      selected_mailer->mailer_type);
	    
	    /* wait that message will be noticed ... */
	    if (POLL_method)
		wait_for_timeout(2);
	    else
		sleep(2);
        }

    } else {
	DPRINT(Debug,1,(&Debug,"Mailer type %s do not have mailer path!\n",
			selected_mailer -> mailer_type));
    }

#if defined(DEFAULT_8BITMIME)
    sendmail_supported_bodytype = have_8bit;
#endif

#if defined(DEFAULT_BINARYMIME)
    sendmail_supported_bodytype = have_binary;
#endif

#if defined(DEFAULT_DSN)
    sendmail_supports_dsn = 1;
#endif

#if defined(DEFAULT_DONT_ADD_FROM)
    selected_mailer -> mailer_bits  |= MB_DONT_ADD_FROM;
#else
    selected_mailer -> mailer_bits  &= ~MB_DONT_ADD_FROM;
#endif

#if defined(DEFAULT_USE_DOMAIN)
    selected_mailer -> mailer_bits  |= MB_USE_DOMAIN;
#else
    selected_mailer -> mailer_bits  &= ~MB_USE_DOMAIN;
#endif

}

void init_mailer_defaults(errors)
     int *errors;
{
    DPRINT(Debug,10,(&Debug,"init_mailer_defaults: starting\n"));

    /* Check if DEFAULT_MAILER_PATH is "none"
       and it is on use
    */

    if (delay_parsing_info)
	return;

    if (! selected_mailer ||
	(0 == strcmp(selected_mailer -> mailer_type,"unknown") &&
	 0 == strcmp(unknown_mailer_path,"none"))) {

	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDefineMailer,
			  "Give \"mailer\" on RC file: %s"),
		  system_rc_file);
    }

    DPRINT(Debug,10,(&Debug,"init_mailer_defaults: done\n"));
}

static void delete_mail_info P_((struct mailer_info *S));


struct mailer_info  *get_mailer_info() {
    struct mailer_info * ret;

    if (!check_mailer(0 /* read_flags -- READ_FLAG_IGNORE_MISSING */)) {
	DPRINT(Debug,7,(&Debug,"No mailer available\n"));
	return NULL;
    }
    
    if (MAILER_CONFIG_magic != selected_mailer->magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"get_mailer_info",
	      "Bad magic number (mailer_config)",0);

    DPRINT(Debug,8,(&Debug,
		    "get_mailer_info: Current mailer %s\n",
		    selected_mailer->mailer_type));

    ret = safe_zero_alloc (sizeof (*ret));

    ret->magic         = MAILER_INFO_magic;
    ret->mailer_type   = selected_mailer;

    ret->delete_pending   = 0;
    ret->p.ptr            = NULL;
    ret->first            = NULL;
    ret->first_e          = NULL;

    DPRINT(Debug,7,(&Debug,
		    "Initialising info for %s mailer, info=%p\n",
		    ret->mailer_type->mailer_type,ret));

    if (! ret->mailer_type->mi_init_hook(ret->mailer_type,ret)) {
	/* FAILURE */
	ret->delete_pending = 1;

	delete_mail_info(ret);
	return NULL;
    }

    return ret;
}

static void delete_mail_info(S)
     struct mailer_info *S;
{

    if (S->magic != MAILER_INFO_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"delete_mail_info",
	      "Bad magic number (mailer info)",0);

    if (S->first)
    	panic("MAILER PANIC",__FILE__,__LINE__,"delete_mail_info",
	      "mailer send state list not empty",
	      0);

    if (S->first_e)
    	panic("MAILER PANIC",__FILE__,__LINE__,"delete_mail_info",
	      "mailer env from list not empty",
	      0);

    if (!valid_mailer(S->mailer_type))
    	panic("MAILER PANIC",__FILE__,__LINE__,"delete_mail_info",
	      "Bad mailer type",
	      0);

    if (MAILER_CONFIG_magic != S->mailer_type->magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"delete_mail_info",
	      "Bad magic number (mailer_config)",0);

    DPRINT(Debug,7,(&Debug,
		    "Deleting info for %s mailer, info=%p\n",
		    S->mailer_type->mailer_type,S));

    S->mailer_type->mi_close_hook(S->mailer_type,S);

    if (S->p.ptr) {
	DPRINT(Debug,1,(&Debug,
			"delete_mail_info: WARNING: Private data (S->p.ptr) not free'ed\n"));		       
    }


    /* bzero is defined on hdrs/defs.h */
    bzero((void *)S, sizeof (*S));

    free(S);    
}

static int can_delete_mailer_info P_((struct mailer_info **S));
static int can_delete_mailer_info(S)
     struct mailer_info **S;
{
    if ((*S)->first)
	return 0;
    if ((*S)->first_e)
	return 0;
    delete_mail_info(*S);

    return 1;
}

void free_mailer_info(S)
     struct mailer_info **S;
{
    if (*S) {
	if ((*S)->magic != MAILER_INFO_magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,"free_mail_info",
		  "Bad magic number (mailer info)",0);
		
	(*S)->delete_pending = 1;
	if (! can_delete_mailer_info(S)) {
	    DPRINT(Debug,7,(&Debug,
			    "delaying deleting of mail info -> delete_pending, info=%p\n",
			    *S));
	}
	*S = NULL;
    }
}

/* Result is malloced */
char ** query_s_mailer_info(reslen_p,I,query,cancel_p)
     size_t                * reslen_p;
     struct mailer_info    * I;
     enum MI_query_string    query;
     struct cancel_data   ** cancel_p;
{
    char ** ret;
    size_t  reslen = 0;
    
    
    if (I->magic != MAILER_INFO_magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"query_s_mailer_info",
	      "Bad magic number (mailer info)",0);
    
    if (!valid_mailer(I->mailer_type))
    	panic("MAILER PANIC",__FILE__,__LINE__,"query_s_mailer_info",
	      "Bad mailer type",
	      0);

    if (MAILER_CONFIG_magic != I->mailer_type->magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"query_s_mailer_info",
	      "Bad magic number (mailer_config)",0);

    ret = I->mailer_type->mi_query_s_hook(&reslen,I->mailer_type,I,query,cancel_p);
    if (ret && !reslen_p) {
	size_t x = reslen+1;

	/* Add NULL element */
	
	ret = safe_array_realloc(ret,x,sizeof(ret[0]));

	ret[reslen] = NULL;
    }
    
    if (reslen)
	* reslen_p = reslen;
    
    return ret;
}

int query_mailer_info(I,query)
     struct mailer_info  *I;
     enum MI_query query;
{
    int ret;

    if (I->magic != MAILER_INFO_magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"query_mailer_info",
	      "Bad magic number (mailer info)",0);

    if (!valid_mailer(I->mailer_type))
    	panic("MAILER PANIC",__FILE__,__LINE__,"query_mailer_info",
	      "Bad mailer type",
	      0);

    if (MAILER_CONFIG_magic != I->mailer_type->magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"query_mailer_info",
	      "Bad magic number (mailer_config)",0);

    ret = I->mailer_type->mi_query_hook(I->mailer_type,I,query);

    return ret;
}

/* -1  if connection lost          (mailer_disconnected)
    0  if OK                       (mailer_not_restarted)
    1  if mailer reinitialized and  query_mailer_info()
          need to be called again  (mailer_reinitialized)
*/

enum restart_mailer_status mailer_restarted(I)
     struct mailer_info  *I;
{
    enum restart_mailer_status ret;

    if (I->magic != MAILER_INFO_magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_restarted",
	      "Bad magic number (mailer info)",0);

    if (!valid_mailer(I->mailer_type))
    	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_restarted",
	      "Bad mailer type",
	      0);

    if (MAILER_CONFIG_magic != I->mailer_type->magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_restarted",
	      "Bad magic number (mailer_config)",0);

    ret = I->mailer_type->mi_restart_hook(I->mailer_type,I);

    DPRINT(Debug,14,(&Debug,
		     "mailer_restarted=%d",
		     ret));
    switch (ret) {
    case mailer_disconnected:    DPRINT(Debug,14,(&Debug," mailer_disconnected"));  break;
    case mailer_not_restarted:   DPRINT(Debug,14,(&Debug," mailer_not_restarted")); break;
    case mailer_reinitialized:   DPRINT(Debug,14,(&Debug," mailer_reinitialized"));   break;
    }
    
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return ret;
}

static int default_mailer_init(M,C,I)
     struct mailer_config *M;
     struct mail_send_state *C;
     struct mailer_info *I;
{
    FILE * F;
    const char *tmp;
    int err = 0;
    
    if (C->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"default_mailer_init",
	      "Bad magic number (mail send state)",0);

    tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    if (!tmp)
	tmp = "/tmp/";

    C->fname = elm_message(FRM("%selm.snd-%d"), 
			tmp, getpid ());

    if (!C->fname) {
	DPRINT(Debug,1,(&Debug,   
			"couldn't make temp file nam! (mail)\n"));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldNotMakeTemp,
			  "Sorry - couldn't make temp file name."));
	return 0;
    }

    F = safeopen_rdwr(C->fname,&err);

    if (!F) {
	DPRINT(Debug,1, (&Debug,
			 "Attempt to open file %s for writing failed! (write_header_info)\n",
			 C->fname));
	DPRINT(Debug,1, (&Debug, "** %s **\n\n", strerror(err)));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorTryingToWrite,
			  "Error %s encountered trying to write to %s."), 
		  strerror(err), C->fname);

	return 0;
    }

    if (C->OUT)
	panic("MAILER PANIC",__FILE__,__LINE__,"default_mailer_init",
	      "out_state already set (mail send state)",0);

    C->OUT = new_out_state(STATE_out_file);
    set_out_state_file(F,C->OUT);

    return 1;  /* OK */
}


static void default_mailer_close(M,C)
     struct mailer_config *M;
     struct mail_send_state *C;
{
    /* close assigned FD */

    if (C->OUT) {
	FILE *F = out_state_FILE(C->OUT);

	free_out_state(& (C->OUT));

	if (F) {
	    fclose(F);
	}
    }
}



void free_mail_send_state(S)
     struct mail_send_state **S;
{
    struct mail_send_state *X = *S;
    
    if (X) {
	int i;

	if (X->magic != MAILER_magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,"free_mail_send_state",
		  "Bad magic number (mail send state)",0);

	if (!valid_mailer(X->mailer_type))
	    panic("MAILER PANIC",__FILE__,__LINE__,"free_mail_send_state",
		  "Bad mailer type",
		  0);

	if (MAILER_CONFIG_magic != X->mailer_type->magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,"free_mail_send_state",
		  "Bad magic number (mailer_config)",0);

	X->mailer_type->m_close_hook(X->mailer_type,X);

	for (i = 0; i < X->addr_count; i++) {
	    if (X->addrs[i]) {
		free(X->addrs[i]);
		X->addrs[i] = NULL;
	    }
	}
	if (X->addrs) {
	    free(X->addrs);
	    X->addrs = NULL;
	    X->addr_count = 0;
	}

	if (X->fname) {
	    if (0 == unlink(X->fname)) {
		DPRINT(Debug,8,(&Debug,"free_mail_send_state: Unlinked %s\n",
				X->fname));
		
	    }
	    free(X->fname);
	    X->fname = NULL;
	}

	if (X->mail_from) {
	    free(X->mail_from);
	    X->mail_from = NULL;
	}

	/* destroy if assigned */
	if (X->OUT) {	   
	    DPRINT(Debug,1,(&Debug,
			    "free_mail_send_state: out state is probably leaking\n"));

	    free_out_state(& (X->OUT));
	}

	if (X->msd_sd) {
	    X->msd_free_cb(&(X->msd_sd));
	    
	    if (X->msd_sd) {
		DPRINT(Debug,1,(&Debug,
				"free_mail_send_state: WARNING: sending_display (X->msd_sd) not free'ed\n")); 
	    }
	}

	if (X->head) {
	    struct mail_send_state * prev = NULL, *walk;

	    for (walk = X->head->first; walk; walk = walk -> next) {

		if (walk == X)
		    break;

		prev = walk;
	    }
	    if (walk != X)
		panic("MAILER PANIC",__FILE__,__LINE__,"free_mail_send_state",
		      "State not in list of mailer_info",0);
	    if (!prev)
		X->head->first = X->next; 
	    else
		prev->next        = X->next; 

	    if (X->head->delete_pending && can_delete_mailer_info(& (X->head))) {
		DPRINT(Debug,7,(&Debug,
				"delete_pending -> deleting mail info"));
	    }
	    X->head = NULL;
	    X->next = NULL;
	}

	/* bzero is defined on hdrs/defs.h */
	bzero((void *)X, sizeof (*X));
	free(X);
	X = NULL;
    }

    *S = X;
}

/* 0 .. not supported 
   1 .. supported 

   calls msd_inc_sending_display_cb
   if take reference from struct sending_display *sd
*/

int mailer_set_sending_display(S,sd,free_cb,inc_cb,addr_upd_cb,
			       addr_check_cb,data_upd_cb,
			       check_fail_cb,con_upd_cb,
			       check_size_cb, size_upd_cb)
     struct mail_send_state *S;
     struct sending_display *sd;
     msd_free_sending_display_cb * free_cb;
     msd_inc_sending_display_cb  * inc_cb;
     msd_update_addr_cb          * addr_upd_cb;
     msd_check_addr_cb           * addr_check_cb;
     msd_update_data_cb          * data_upd_cb;
     msd_check_fail_cb           * check_fail_cb;
     msd_update_con_cb           * con_upd_cb;
     msd_check_size_cb           * check_size_cb;
     msd_update_size_cb          * size_upd_cb;
{
    int i;

    if (S->magic != MAILER_magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_set_sending_display",
	      "Bad magic number (mail send state)",0);
    
    if (!valid_mailer(S->mailer_type))
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_set_sending_display",
	      "Bad mailer type",
	      0);

    if (MAILER_CONFIG_magic != S->mailer_type->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_set_sending_display",
	      "Bad magic number (mailer_config)",0);

    if (S->msd_sd) {
	S->msd_free_cb(&(S->msd_sd));
	
	if (S->msd_sd) {
	    DPRINT(Debug,1,(&Debug,
			    "mailer_set_sending_display: WARNING: old sending_display (S->msd_sd) not free'ed\n"));		       
	}
    }

    if (0 != (S->mailer_type->mailer_bits & MB_SENDING_DISPLAY)) {

	S->msd_sd = sd;
	
	S->msd_free_cb       = free_cb;
	S->msd_inc_cb        = inc_cb;
	S->msd_addr_upd_cb   = addr_upd_cb;
	S->msd_addr_check_cb = addr_check_cb;
	S->msd_data_upd_cb   = data_upd_cb;
	S->msd_fail_check_cb = check_fail_cb;
	S->msd_con_upd_cb    = con_upd_cb;
	S->msd_size_check_cb = check_size_cb;
	S->msd_size_upd_cb   = size_upd_cb;

	if (S->msd_sd) {
	    S->msd_inc_cb(S->msd_sd);
	    
	    if (S->mail_from) {
		struct string * A = 
		    new_string2(system_charset,
				s2us(S->mail_from));
		S->msd_addr_upd_cb(S->msd_sd,-1,A,
				   msd_addrstat_none,NULL);
		free_string(&A);
	    }
	
	    for (i = 0; i <  S->addr_count; i++) {
		if (S->addrs[i]) {
		    struct string * A = 
			new_string2(system_charset,
				    s2us(S->addrs[i]));
		    S->msd_addr_upd_cb(S->msd_sd,i,A,
				       msd_addrstat_none,NULL);
		    free_string(&A);
		}
	    }

	    return 1 ; 
	}
    }

    return 0;
}

S_(end_handler no_end_handler)
static void no_end_handler(fd,title,rs,ret,exit_stat) 
     union any_fd fd; 
     char * title; 
     struct run_state *rs; 
     int ret; 
     int exit_stat;
{
    /* Nothing */
}


/* Return NULL if no editor (and no value either) */
struct mailer_env_from * mailer_get_env_from(I)
     struct mailer_info  *I;
{
    struct mailer_env_from * ret = NULL;
    int flags                    = 0;

    if (I->magic != MAILER_INFO_magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from",
	      "Bad magic number (mailer info)",0);

    if (!valid_mailer(I->mailer_type))
    	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from",
	      "Bad mailer type", 0);

    if (MAILER_CONFIG_magic != I->mailer_type->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from",
	      "Bad magic number (mailer_config)",0);

    if (0 != (I->mailer_type->mailer_bits & MB_ALLOW_SET_SENDER))
	flags |= MAILER_ef_can_set;

    if (0 != (I->mailer_type->mailer_bits & MB_REQ_DEFAULT_SENDER))
	flags |= MAILER_ef_need_default;

    if (0 == flags) {
	DPRINT(Debug,14,(&Debug,
			 "mailer_env_from=NULL\n"));
	return NULL;
    }
    
    ret = safe_zero_alloc (sizeof (*ret));

    ret->magic      = MAILER_ef_magic;
    ret->mail_from  = NULL;
    ret->flags      = flags;
   
    ret->head       = I;
    ret->next       = I->first_e;
    I->first_e      = ret;

    if (0 != (MAILER_ef_need_default & ret->flags)) {
	DPRINT(Debug,14,(&Debug,
			 "mailer_env_from: Setting default value\n"));
	I->mailer_type->mi_def_env_from(I->mailer_type,
					I,ret);
    } else {
	DPRINT(Debug,14,(&Debug,
			 "mailer_env_from: Default value not needed\n"));

    }

    return ret;
}

void mailer_free_env_from(X)
     struct mailer_env_from **X;
{
    struct mailer_env_from * E = *X;
    
    if (E) {
	if (MAILER_ef_magic != E->magic) 
	    panic("MAILER PANIC",__FILE__,__LINE__,"mailer_free_env_from",
		  "Bad magic number (env from)",0);

	if (E->mail_from) {
	    free(E->mail_from);
	    E->mail_from = NULL;
	}

	if (E->domain) {
	    free(E->domain);
	    E->domain = NULL;
	}

	if (E->angleaddr) {
	    free(E->angleaddr);
	    E->angleaddr = NULL;
	}
	
	if (E->head) {
	    struct mailer_env_from * prev = NULL, *walk;

	    for (walk = E->head->first_e; walk; walk = walk -> next) {

		if (walk == E)
		    break;

		prev = walk;
	    }
	    if (walk != E)
		panic("MAILER PANIC",__FILE__,__LINE__,"mailer_free_env_from",
		      "Env from not in list of mailer_info",0);
	    if (!prev)
		E->head->first_e = E->next; 
	    else
		prev->next       = E->next; 

	    if (E->head->delete_pending && can_delete_mailer_info(& (E->head))) {
		DPRINT(Debug,7,(&Debug,
				"delete_pending -> deleting mail info"));
	    }
	    E->head = NULL;
	    E->next = NULL;
	}

	E->magic = 0;   /* Set bad value */

	free(E);
	E = NULL;
    }
    *X = E;
}

/* return temporary pointer  value -- do not free 
   return NULL if currently no value set (use implicit default)
  */
const char * mailer_env_from_value(X,can_edit)
     struct mailer_env_from *X;
     int *can_edit;
{
    if (MAILER_ef_magic != X->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_value",
	      "Bad magic number (env from)",0);

    if (can_edit)
	*can_edit = 0 != (X->flags & MAILER_ef_can_set);

    if (X->mail_from) {
	DPRINT(Debug,13,(&Debug,
			 "mailer_env_from_value=%s\n",
			 X->mail_from));
    }
    
    return X->mail_from;
}

/* return temporary pointer  value -- do not free 
   return NULL if currently no value set (use implicit default)
*/
const char * mailer_env_from_angleaddr(X)
     struct mailer_env_from *X;
{
    const char * ret = NULL;
    
    if (MAILER_ef_magic != X->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_angleaddr",
	      "Bad magic number (env from)",0);

    if (X->angleaddr) {
	ret = X->angleaddr;
	goto done;
    }

    if (X->mail_from) {
	char * p = X->mail_from;
	int l;
	
	DPRINT(Debug,13,(&Debug,
			 "mailer_env_from_angleaddr: address=%s\n",
			 X->mail_from));

	if ('<' == X->mail_from[0]) {
	    
	    l = strlen(X->mail_from);

	    if (l > 1 && '>' == X->mail_from[l-1]) {
		ret = X->mail_from;
		goto done;
	    }

	    DPRINT(Debug,13,(&Debug,
			     "mailer_env_from_angleaddr: Invalid address %s\n",
			     X->mail_from));
	    p = X->mail_from +1;
	}
	
	l = strlen(p);
	if (l > 0 && '>' == X->mail_from[l-1]) {
	    DPRINT(Debug,13,(&Debug,
			     "mailer_env_from_angleaddr: Invalid address %s\n",
			     X->mail_from));
	    l--;
	}

	X->angleaddr = elm_message(FRM("<%.*s>"),l,p);

	DPRINT(Debug,13,(&Debug,
			 "mailer_env_from_angleaddr: address=%s => setting angleaddr=%s\n",
			 X->mail_from,X->angleaddr));
	ret = X->angleaddr;
    }
	
 done:
    if (ret) {
	DPRINT(Debug,13,(&Debug,
			 "mailer_env_from_angleaddr=%s\n",
			 ret));
    } else {
	DPRINT(Debug,13,(&Debug,
			 "mailer_env_from_angleaddr=NULL\n"));
    }
    
    return ret;   
}

/* return temporary pointer  value -- do not free 
   return NULL if currently no value set (use implicit default)
   also return NULL is env_from is <>
  */
const char * mailer_env_from_domain(X)
     struct mailer_env_from *X;
{
    const char * ret = NULL;
    
    if (MAILER_ef_magic != X->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_domain",
	      "Bad magic number (env from)",0);

    if (X->domain)  {
	ret = X->domain;
	goto done;
    }

    if (X->mail_from) {
	char * p;
	
	DPRINT(Debug,13,(&Debug,
			 "mailer_env_from_domain: address=%s\n",
			 X->mail_from));
	
	if (0 == strcmp(X->mail_from,"<>")) {
	    ret = NULL;
	    goto done;
	}
	    
	p =  qstrpbrk(X->mail_from,"@");

	if (p) {
	    if ('<' == X->mail_from[0]) {
		/* If first character is @ then this is 
		   routed address, ... do not handle
		*/

		if (p > X->mail_from +1) {
		    int l;
		    X->domain =
			strmcpy(X->domain,p+1);
		    l = strlen(X->domain);
		    if (l > 1 && '>' == X->domain[l-1])
			X->domain[l-1] = '\0';
		    else {
			DPRINT(Debug,13,(&Debug,
					 "mailer_env_from_domain: Address %s is invalid\n",
					 X->mail_from));
		    }
		    DPRINT(Debug,13,(&Debug,
				     "mailer_env_from_domain: address=%s => setting domain=%s\n",
				     X->mail_from,X->domain));

		    ret = X->domain;
		    goto done;
		} else {
		    DPRINT(Debug,13,(&Debug,
				     "mailer_env_from_domain: Address %s is unsupported\n",
				     X->mail_from));
		}
						
	    } else {
		/* If first character is @ then this is 
		   routed address, ... do not handle
		*/
		
		if (p > X->mail_from) {
		    ret = p+1;
		    goto done;
		} else {
		    DPRINT(Debug,13,(&Debug,
				     "mailer_env_from_domain: Address %s is unsupported\n",
				     X->mail_from));
		}
	    }
	} else {
	    DPRINT(Debug,13,(&Debug,
			     "mailer_env_from_domain: Address=%s => no domain\n",
			     X->mail_from));
	}
    }

 done:
    if (ret) {
	DPRINT(Debug,13,(&Debug,
			 "mailer_env_from_domain=%s\n",
			 ret));
    } else {
	DPRINT(Debug,13,(&Debug,
			 "mailer_env_from_domain=NULL\n"));
    }
    
    return ret;
}

int mailer_env_from_changed(X)
     struct mailer_env_from *X;
{
    if (MAILER_ef_magic != X->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_changed",
	      "Bad magic number (mailer_env_from)",0);

    return 0 != (X->flags & MAILER_ef_changed);
}

/* This should be set only for local address,
   This is 250 status
   but not 252 relayed mail status
*/

int mailer_env_from_verified(X)
     struct mailer_env_from *X;
{
    if (MAILER_ef_magic != X->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_verified",
	      "Bad magic number (mailer_env_from)",0);

    /* MAILER_ef_verified is set and cleared by 
       submission mailer */
    
    return 0 != (X->flags & MAILER_ef_verified);
}

/* Return 1 if succeed */
int mailer_env_from_change(X,value)
     struct mailer_env_from *X;
     const char * value;
{
    int ret = 0;
    int r;
    const char * p = NULL;
    
    if (MAILER_ef_magic != X->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_change",
	      "Bad magic number (env from)",0);

    if (MAILER_INFO_magic != X->head->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_change",
	      "Bad magic number (mailer info)",0);

    if (!valid_mailer(X->head->mailer_type))
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_change",
	      "Bad mailer type",0);

    if (MAILER_CONFIG_magic != X->head->mailer_type->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_env_from_change",
	      "Bad magic number (mailer_config)",0);

    if (0 == (X->flags & MAILER_ef_can_set)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmChangingEnvSenderNotAllowed,
			  "Changing of envelope sender for mailer %s not allowed: %s"),
		  X->head->mailer_type->mailer_type, value);
	ret = 0;
	goto fail;
    }

    if ('<' == value[0]) {
	int l = strlen(value);

	if (l < 2) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEnvelopeSenderAddress,
			      "Envelope sender address %s is invalid."),
		      value);
	    ret = 0;
	    goto fail;
	}

	if ('>' != value[l-1]) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEnvelopeSenderAddress,
			      "Envelope sender address %s is invalid."),
		      value);
	    ret = 0;
	    goto fail;
	}
	
    } else if (! value[0]) {
	DPRINT(Debug,13,(&Debug, "mailer_env_from_change: Changed empty value to <>\n"));
	value = "<>";
    } else if (NULL != qstrpbrk_c(value,"<>")) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEnvelopeSenderAddress,
			  "Envelope sender address %s is invalid."),
		  value);
	ret = 0;
	goto fail;
    }

    p = qstrpbrk_c(value,"@");

    if (p) {
	/* Do not allow address to be end with @ -character */
	
	if (! p[1] || '>' == p[1]) {
	   lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadEnvelopeSenderAddress,
			  "Envelope sender address %s is invalid."),
		  value);
	   ret = 0;
	   goto fail;
	}
	
    } else {    
	if (0 != (MB_USE_DOMAIN & X->head->mailer_type->mailer_bits) &&
	    0 != strcmp(value,"<>")) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmEnvSenderDomainRequired,
			      "Mailer %s requires domain part on envelope sender: %s"),
		      X->head->mailer_type->mailer_type, value);
	    ret = 0;
	    goto fail;
	}
    }
    
    
    r = X->head->mailer_type->mi_set_env_from(X->head->mailer_type,
					      X->head,X,value);
    if (r) {
	X->flags |= MAILER_ef_changed;
	DPRINT(Debug,13,(&Debug, "mailer_env_from_change: Setting MAILER_ef_changed\n"));

	if (X->domain) {
	    DPRINT(Debug,13,(&Debug,
			     "mailer_env_from_change: Freeing domain=%s\n",
			     X->domain
			     ));
	    free(X->domain);
	    X->domain = NULL;
	}

	if (X->angleaddr) {
	    DPRINT(Debug,13,(&Debug,
			     "mailer_env_from_change: Freeing angleaddr=%s\n",
			     X->angleaddr
			     ));
	    free(X->angleaddr);
	    X->angleaddr = NULL;
	}
	
	ret = 1;
    }

 fail:
    DPRINT(Debug,13,(&Debug, "mailer_env_from_change=%d\n",
		     ret));
    return ret;
}


S_(msd_free_sending_display_cb NULL_msd_free_cb)
static void NULL_msd_free_cb P_((struct sending_display **sd));
static void NULL_msd_free_cb(sd)
     struct sending_display **sd;
{
    panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_free_cb",
	  "NULL_msd_free_cb called",0);

    free(sd);
    *sd = NULL;
}

S_(msd_inc_sending_display_cb NULL_msd_inc_cb);
static void NULL_msd_inc_cb P_((struct sending_display *sd));
static void NULL_msd_inc_cb(sd)
     struct sending_display *sd;
{
    panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_inc_cb",
	  "NULL_msd_inc_cb called",0);
}

S_(msd_update_addr_cb NULL_msd_addr_upd_cb)
static void NULL_msd_addr_upd_cb P_((struct sending_display *sd,
				     int                    idx,
				     struct string        * addr,
				     enum msd_addr_status   status,
				     struct string        * message /* server */
				     ));
static void NULL_msd_addr_upd_cb(sd,idx,addr,status,message)
     struct sending_display *sd;
     int                    idx;
     struct string        * addr;
     enum msd_addr_status   status;
     struct string        * message /* server */;
{
    panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_addr_upd_cb",
	  "NULL_msd_addr_upd_cb called",0);

}

S_(msd_check_addr_cb NULL_msd_addr_check_cb)
static enum msd_action NULL_msd_addr_check_cb P_((struct sending_display *sd,
						  int                    idx,
						  struct string        * addr,
						  enum msd_addr_status   status,
						  struct string  * message 
						  /* server */,
						  enum msd_can_continue  
						  can_continue));
static enum msd_action NULL_msd_addr_check_cb (sd,idx,addr,status,message,
					       can_continue)
     struct sending_display *sd;
     int                    idx;
     struct string        * addr;
     enum msd_addr_status   status;
     struct string  * message /* server */;
     enum msd_can_continue can_continue;
{
    panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_addr_check_cb",
	  "NULL_msd_addr_check_cb called",0);

    return msd_act_default;
}

S_(msd_update_data_cb NULL_msd_data_upd_cb)
static void NULL_msd_data_upd_cb P_((struct sending_display *sd,
				     enum msd_data_status   status,
				     struct string  * message /* server */));
static void NULL_msd_data_upd_cb(sd,status,message)
     struct sending_display *sd;
     enum msd_data_status   status;
     struct string  * message /* server */;
{
    panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_data_upd_cb",
	  "NULL_msd_data_upd_cb called",0);
}

S_(msd_check_fail_cb  NULL_msd_check_fail_cb)
static enum msd_action NULL_msd_check_fail_cb P_((struct sending_display *sd,
						  enum msd_data_status   status,
						  struct string  * message 
						  /* server */,
						  enum msd_can_continue 
						  can_continue));
static enum msd_action NULL_msd_check_fail_cb(sd,status,message,can_continue)
     struct sending_display *sd;
     enum msd_data_status   status;
     struct string  * message /* server */;
     enum msd_can_continue can_continue;
{
    panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_check_fail_cb",
	  "NULL_msd_check_fail_cb called",0);

    return msd_act_default;
}

S_(msd_update_con_cb NULL_msd_con_upd_cb)
static void NULL_msd_con_upd_cb P_((struct sending_display *sd,
				    enum msd_connection_info idx,
				    struct string *text));
static void NULL_msd_con_upd_cb(sd,idx,text)
     struct sending_display *sd;
     enum msd_connection_info idx;
     struct string *text;
{
    panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_con_upd_cb",
	  "NULL_msd_con_upd_cb called",0);
}

S_(msd_check_size_cb NULL_msd_size_check_cb)
static enum msd_action NULL_msd_size_check_cb P_((struct sending_display *sd,
						  long message_size,
						  long size_limit));
static enum msd_action NULL_msd_size_check_cb(sd,message_size,size_limit)
     struct sending_display *sd;
     long message_size;
     long size_limit;
{
    panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_size_check_cb",
	  "NULL_msd_size_check_cb called",0);

    return msd_act_default;
}

S_(msd_update_size_cb NULL_msd_update_size_cb)
static void NULL_msd_update_size_cb P_((struct sending_display *sd,
					enum msd_size_info idx,
					long size));
static void NULL_msd_update_size_cb(sd,idx,size)
     struct sending_display *sd;
     enum msd_size_info idx;
     long size;
{
   panic("MAILER PANIC",__FILE__,__LINE__,"NULL_msd_update_size_cb",
	  "NULL_msd_update_size_cb called",0); 

}

struct mail_send_state * mailer_init(addr_args,dsn,verbose,info,env_from)
     char **addr_args;
     int dsn;
     int verbose;
     struct mailer_info *info;
     struct mailer_env_from *env_from; 
{
    struct mail_send_state *ret;
    int i;
    int count;

    ret = safe_zero_alloc (sizeof (*ret));

    ret->magic         = MAILER_magic;

    if (info) {

	if (info->magic != MAILER_INFO_magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,"mailer_init",
		  "Bad magic number (mailer info)",0);

	ret->mailer_type   = info->mailer_type;

	ret->head          = info;
	ret->next          = info->first;
	info->first        = ret;

    } else {
	ret->mailer_type   = selected_mailer;
	ret->head          = NULL;
	ret->next          = NULL;
    }
    for (count = 0; addr_args[count]; count++);
        
    ret->addrs   = safe_calloc((count+1),sizeof (* (ret->addrs)));

    for (i = 0; i < count; i++) 
	ret->addrs[i] = safe_strdup(addr_args[i]);
    ret->addrs[count] = NULL;
    ret->addr_count   = count;
    
    ret->dsn          = dsn;
    ret->verbose      = verbose;
    ret->fname        = NULL;
    ret->mail_from    = NULL;

    ret->msd_sd            = NULL;
    ret->msd_free_cb       = NULL_msd_free_cb;
    ret->msd_inc_cb        = NULL_msd_inc_cb;
    ret->msd_addr_upd_cb   = NULL_msd_addr_upd_cb;
    ret->msd_addr_check_cb = NULL_msd_addr_check_cb;
    ret->msd_data_upd_cb   = NULL_msd_data_upd_cb;
    ret->msd_fail_check_cb = NULL_msd_check_fail_cb;
    ret->msd_con_upd_cb    = NULL_msd_con_upd_cb;
    ret->msd_size_check_cb = NULL_msd_size_check_cb;
    ret->msd_size_upd_cb   = NULL_msd_update_size_cb;

    if (env_from) {
	if (MAILER_ef_magic != env_from->magic) 
	    panic("MAILER PANIC",__FILE__,__LINE__,"mailer_init",
		  "Bad magic number (env from)",0);

	if (env_from->mail_from)
	    ret->mail_from = safe_strdup(env_from->mail_from);
    }

    ret->OUT          = NULL;   /* Nothing set */
    ret->orig_func    = no_end_handler;

    DPRINT(Debug,5,(&Debug,
		    "Initialising sending via %s mailer, %d recipients\n",
		    ret->mailer_type->mailer_type,
		    ret->addr_count));

    if (!valid_mailer(ret->mailer_type))
    	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_init",
	      "Bad mailer type",
	      0);
    
    if (MAILER_CONFIG_magic != ret->mailer_type->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_init",
	      "Bad magic number (mailer_config)",0);

    if (!ret->mailer_type->m_init_hook(ret->mailer_type,
				       ret,info)) {
	free_mail_send_state(&ret);
    }

    return ret;
}

const char *get_mailer_path(X)
     struct mail_send_state *X;
{
    if (X->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"get_mailer_path",
	      "Bad magic number (mail send state)",0);

    if (X->mailer_type->mailer_path &&
	*(X->mailer_type->mailer_path) &&
	(*(X->mailer_type->mailer_path))[0])
	return *(X->mailer_type->mailer_path);

    return X->mailer_type->mailer_type;
}


struct out_state *get_mail_outfd(X)
     struct mail_send_state  *X;
{
    if (X->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"get_mail_outfd",
	      "Bad magic number (mail send state)",0);

    if (! X->OUT) 
    	panic("MAILER PANIC",__FILE__,__LINE__,"get_mail_outfd",
	      "No out_state set",0);

    return X->OUT;
}

S_(end_handler call_end_handler)
static void call_end_handler(fd,title,rs,ret,exit_stat) 
     union any_fd fd; 
     char * title; 
     struct run_state *rs; 
     int ret; 
     int exit_stat;
{
    if (fd.mail_fd->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"call_end_handler",
	      "Bad magic number (mail send state)",0);

    DPRINT(Debug,7,(&Debug,
		    "Finished sending mail via %s mailer, ret=%d exit_stat=%d\n",
		    fd.mail_fd->mailer_type->mailer_type,
		    ret,exit_stat));
    
    fd.mail_fd->orig_func(fd,title,rs,ret,exit_stat);
    free_mail_send_state(&(fd.mail_fd));
}

/* Retrun 1 if mail is moved to ticket 

   exit_code == -1   can't run mailer
*/
static int maybe_pass_ticket P_((struct mailer_config *M,
				 struct mail_send_state *C,
				 int exit_status,
				 struct mailer_cancel_ticket * ticket,
				 struct string *message));
static int maybe_pass_ticket(M,C,exit_status,ticket,message) 
     struct mailer_config *M;
     struct mail_send_state *C;
     int exit_status;
     struct mailer_cancel_ticket * ticket;
     struct string *message;
{
    enum msd_data_status stat =  msd_mail_sent;
    int r = 0;
    
    /* Not really result from SMTP's data phase 
       but just simulate it
    */

    if (exit_status != 0)
	stat = msd_mail_fail;
    
    /* From sysexists.h */
#ifdef  EX_TEMPFAIL 
    if (exit_status == EX_TEMPFAIL)
	stat = msd_mail_tmpfail;
#endif

    if (MAILER_CANCEL_TICKET_magic != ticket->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"maybe_pass_ticket",
	      "Bad magic number (mailer_cancel_ticket)",0);

    if (C->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"maybe_pass_ticket",
	      "Bad magic number (mail_send_state)",0);

    
    if (exit_status != 0 && C->msd_sd &&

	/* submitmail_mailer_init writes addresses to
	   beginning of mail   */
	M->m_init_hook != submitmail_mailer_init) {

	ticket->msd_action = 
	    C->msd_fail_check_cb(C->msd_sd,stat,message,
				 msd_mail_failed);

	switch (ticket->msd_action) {
	case msd_act_EOF: 
	    DPRINT(Debug,4, (&Debug, 
			     "maybe_pass_ticket: action msd_act_EOF\n"));
	    break;
	    
	case msd_act_SIG:
	    DPRINT(Debug,4, (&Debug, 
			     "maybe_pass_ticket: action msd_act_SIG\n"));
	    break;

	case msd_act_need_restart:
	    DPRINT(Debug,4, (&Debug, 
			     "maybe_pass_ticket: action msd_act_need_restart\n"));
	    r = 1;
	    break;

	case msd_act_default:
	    DPRINT(Debug,4, (&Debug, 
			     "maybe_pass_ticket: action msd_act_default\n"));
	    break;

	case msd_act_cancel:
	    DPRINT(Debug,4, (&Debug, 
			     "maybe_pass_ticket: action msd_act_cancel\n"));
	    r = 1;
	    break;

	case msd_act_edit:
	    DPRINT(Debug,4, (&Debug, 
			     "maybe_pass_ticket: action msd_act_edit\n"));
	    r = 1;
	    break;

	case msd_act_send:
	    DPRINT(Debug,4, (&Debug, 
			     "maybe_pass_ticket: action msd_act_send\n"));
	    break;
	}
	 
	if (r) {

	    ticket->fname = C->fname;
	    C->fname = NULL;

	    ticket->OUT   = C->OUT;
	    C->OUT        = NULL;
	    
	    if (C->OUT) 
		out_state_fseek (C->OUT, 0);
	    
	    if (! ticket->fname) {
		DPRINT(Debug,4, (&Debug, 
				 "maybe_pass_ticket: No filename\n"));
	    }

	    if (! ticket->OUT) {
		DPRINT(Debug,4, (&Debug, 
				 "maybe_pass_ticket: No out_state\n"));
	    }

	    if (! ticket->fname && ! ticket->OUT)   
		r = 0;
	}

    } else if (C->msd_sd) 
	C->msd_data_upd_cb(C->msd_sd,stat,message);

    DPRINT(Debug,8, (&Debug, 
		     "maybe_pass_ticket=%d\n",r));

    return r;
}

static int backend_tail P_((struct mailer_config *M,
			    struct mail_send_state **C,
			    char * title,
			    sending_message_func *sm,
			    const char **argv,
			    int *exit_status,
			    struct mailer_cancel_ticket * ticket));

static int backend_tail(M,C,title,sm,argv,exit_status,ticket)
     struct mailer_config *M;
     struct mail_send_state **C;
     char * title;
     sending_message_func *sm;
     const char **argv;
     int *exit_status; 
     struct mailer_cancel_ticket * ticket;
{
    struct run_state RS;

    int options = SY_ENV_SHELL;
    int ret;
    union any_fd FD;

    if ((*C)->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"backend_tail",
	      "Bad magic number (mail_send_state)",0);

    FD.mail_fd         = *C;

    if (!(*C)->verbose)
	options |= SY_NOTTY;
    
    out_state_fseek ((*C)->OUT, 0);
#ifdef _POSIX_VERSION
    /* Synzronize underlying file descriptor */
    fflush(out_state_FILE((*C)->OUT));  
#else
    seek(fileno(out_state_FILE((*C)->OUT)),0,0);
#endif

    if ((*C)->msd_sd)
	(*C)->msd_data_upd_cb((*C)->msd_sd,
			      msd_calling_mailer,
			      NULL);

    ret=start_run(&RS, options, argv, fileno(out_state_FILE((*C)->OUT)),-1);
    
    if (ret) {
	int backgrounded = 0;
	int exit_code;
	int was_handle_sigchld = handle_sigchld;

	ret = run_already_done(&RS,&exit_code);
	if (0 == ret) {
	    sm(0);
	    	    	      	    
#ifdef BACKGROUD_PROCESSES       /* We assume POSIX in here */
	    if (background_wait_time) {
 		DPRINT(Debug,4, (&Debug, 
				 "Sleeping ( %d seconds ) for completion!\n",
				 background_wait_time));

		if (POLL_method) {
		    int err = 0;

		    enum wait_for_status tmp = wait_for_timeout_f(background_wait_time,
								  WAIT_FOR_intr |
								  (was_handle_sigchld ? 0 :  WAIT_FOR_handle_sigchld),
								  &err);
		    switch (tmp) {
		    case wait_for_done:    break;
		    case wait_for_none:
			DPRINT(Debug,4,(&Debug,  
					" -- unexpected result from wait_for_timeout\n"));
			break;
		    case  wait_for_error:
			
			DPRINT(Debug,4,(&Debug,  
					" -- sleeping failed, errno=%d, %s\n",
					err,strerror(err)));
			break;
		    }

		} else {

		    int tmp = sleep(background_wait_time);  
		    /* POSIX sleep returns time in left on
		     * interrupt -- when sendmail terminates
		     * we will get interrupt (SIGCHLD signal)
		     */
		    if (tmp > 0) {
			DPRINT(Debug,4,(&Debug,  
					" -- sleeping interrupted, %d seconds left!\n",
					tmp));
		    } else if (tmp < 0) {
			int err UNUSED_VAROK = errno;
			
			DPRINT(Debug,4,(&Debug,  
					" -- sleeping failed, errno=%d, %s\n",
					err,strerror(err)));
		    }
		}

		ret = run_already_done(&RS,&exit_code);
		if (0 == ret) 
		    ret = maybe_background(&RS,&exit_code,
					   FD,title,
					   call_end_handler,
					   NO_free_any_fd,
					   NO_inc_any_fd_refcount);
		if (0 == ret) {
		    sm(1);		    		    
		    backgrounded = 1;

		    if ((*C)->msd_sd)
			(*C)->msd_data_upd_cb((*C)->msd_sd,
					      msd_mail_backgrounded,
					      NULL);
		    
		    *C = NULL;   /* call_end_handler may free 
				    mail_send_state 
				 */
		}
	    } else
#endif
		ret = wait_end(&RS,&exit_code);
	}       
	if (!backgrounded) {

	    if (ret > 0 && exit_status)
		*exit_status = exit_code;

	    if (ret > 0 && ticket && 
		maybe_pass_ticket(M,*C,exit_code,ticket,NULL)) {
		DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
	    } else if ((*C)->msd_sd) {
		enum msd_data_status stat =  msd_mail_sent;

		/* Not really result from SMTP's data phase 
		   but just simulate it
		*/
		
		if (exit_code != 0)
		    stat = msd_mail_fail;

		/* From sysexists.h */
#ifdef  EX_TEMPFAIL 
		if (exit_code == EX_TEMPFAIL)
		    stat = msd_mail_tmpfail;
#endif

		(*C)->msd_data_upd_cb((*C)->msd_sd,stat,NULL);
	    }
		    

	    call_end_handler(FD,title,&RS,ret,exit_code);
	    *C = NULL;   /* call_end_handler free'ed mail_send_state 
			  */
	}	
    } else {
	struct string *message = NULL;

	if (RS.save_errno) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailErrno,
			      "Failed: %.30s: %.40s"),
		      argv[0],strerror(RS.save_errno));
	    
	    message = format_string(CATGETS(elm_msg_cat, ElmSet, ElmFailErrno,
					    "Failed: %.30s: %.40s"),
				    argv[0],strerror(RS.save_errno));
	}
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantStart,
			      "Can't start %.30s"),
		      argv[0]);
       
	if (ticket) {
	    if (maybe_pass_ticket(M,*C,-1,ticket,message)) {
		DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
	    }
	} else if ((*C)->msd_sd) {
	    (*C)->msd_data_upd_cb((*C)->msd_sd,msd_mail_fail,message);
	}

	if (message)
	    free_string(&message);

	free_mail_send_state(C);
    }

    return ret;
}


static int unknown_mailer_backend3(M,C,encoding_top,title,sm,exit_status,
				   ticket,message_size)
     struct mailer_config *M;
     struct mail_send_state **C;
     int encoding_top;
     char * title;
     sending_message_func *sm;
     int *exit_status;
     struct mailer_cancel_ticket * ticket;
     long message_size;
{
    char *mailerflags[20];
    const char **argv;
    int mf_idx=0;
    int r;

    if ((*C)->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"unknown_mailer_backend3",
	      "Bad magic number (mail send state)",0);


    mailerflags[mf_idx++] = unknown_mailer_path;
    mailerflags[mf_idx] = NULL;

    argv = join_argv(mailerflags,(*C)->addrs);

    r = backend_tail(M,C,title,sm,argv,exit_status,ticket);
    
    free(argv);

    return r;
}

static int sendmail_mailer_backend3(M,C,encoding_top,title,sm,exit_status,
				    ticket,message_size)
     struct mailer_config *M;
     struct mail_send_state **C;
     int encoding_top;
     char * title;
     sending_message_func *sm;
     int *exit_status;
     struct mailer_cancel_ticket * ticket;
     long message_size;
{
    char t[80];
    int r;

    char *mailerflags[22];
    const char **argv;
    int mf_idx=0;

    if ((*C)->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"sendmail_mailer_backend3",
	      "Bad magic number (mail send state)",0);

    mailerflags[mf_idx++] = sendmail_mailer_path;

    mailerflags[mf_idx++] = "-oi";
    mailerflags[mf_idx++] = "-oem";

    if ((*C)->mail_from && 0 != ( M -> mailer_bits & MB_ALLOW_SET_SENDER)) {
	mailerflags[mf_idx++] = "-f";
	mailerflags[mf_idx++] = (*C)->mail_from;
    }

    if ((*C)->verbose)
	mailerflags[mf_idx++] = "-v";
    if (metoo) 
	mailerflags[mf_idx++] = "-om";

    if (sendmail_supported_bodytype >= have_8bit) {
	if (encoding_top == ENCODING_8BIT)
	    mailerflags[mf_idx++] = "-B8BITMIME";
    }

    if (sendmail_supported_bodytype >= have_binary) {
	if (encoding_top == ENCODING_BINARY)
	    /* With -BBINARYMIME lines must terminate with \r\n
	     * Unix's \n is _NOT_ sufficient - K E H              */
	    mailerflags[mf_idx++] = "-BBINARYMIME";
    }
    
    if (sendmail_supports_dsn) {
	if ((*C)->dsn & DSN_FULL) {
	    mailerflags[mf_idx++] = "-R";
	    mailerflags[mf_idx++] = "full";
	} else if ((*C)->dsn & DSN_HDRS) {
	    mailerflags[mf_idx++] = "-R";
	    mailerflags[mf_idx++] = "hdrs";
	}
	
	if ((*C)->dsn & DSN_NEVER) {
	    mailerflags[mf_idx++] = "-N";
	    mailerflags[mf_idx++] = "never";
	} else if ((*C)->dsn & (DSN_SUCCESS|DSN_FAILURE|DSN_DELAY)) {	    
	    t[0] = '\0';
	    if ((*C)->dsn & DSN_SUCCESS)
		strfcat(t,"success", sizeof t);
	    if ((*C)->dsn & DSN_FAILURE) {
		if (t[0]) strfcat(t,",", sizeof t);
		strfcat(t,"failure", sizeof t);
	    }
	    if ((*C)->dsn & DSN_DELAY) {
		if (t[0]) strfcat(t,",", sizeof t);
		strfcat(t,"delay", sizeof t);
	    }
	    mailerflags[mf_idx++] = "-N";
	    mailerflags[mf_idx++] = t;
	}
    }

    mailerflags[mf_idx++] = "--";

    mailerflags[mf_idx] = NULL;

    argv = join_argv(mailerflags,(*C)->addrs);
    
    r = backend_tail(M,C,title,sm,argv,exit_status,ticket);
    
    free(argv);

    return r;
}

/* Return 
   -1 if caller should result with default data (return 1)  (mi_vrfyaddr_default_ok)
   -2 if caller should just test passwd                     (mi_vrfyaddr_test_passwd)
   0  if failure                                            (mi_vrfyaddr_failure)
   1  is succees                                            (mi_vrfyaddr_success)
*/
static enum mi_vrfyaddr_mailer_status sendmail_mailer_info_verify_addr(M,I,text,result)
     struct mailer_config *M;
     struct mailer_info  *I;
     const char *text;
     struct addr_verify_result *result;
{

    const char *mailerflags[5];
    int mf_idx=0;
    struct run_state RS;
    int ret,exit_code;
    enum mi_vrfyaddr_mailer_status status = mi_vrfyaddr_default_ok;
 
    if (I->magic != MAILER_INFO_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,
	      "sendmail_mailer_info_verify_addr",
	      "Bad magic number (mailer info)",0);

    if (!sendmail_verify_address) {
	DPRINT(Debug,4,(&Debug,
			"Don't call %s for verify -- falling back to default addr verify\n",
			sendmail_mailer_path));
	status =  mi_vrfyaddr_test_passwd;   /* caller does job */
	goto done;
    }

    mailerflags[mf_idx++] = sendmail_mailer_path;

    mailerflags[mf_idx++] = "-bv";

    /* Do not treat -xyc as option */
    mailerflags[mf_idx++] = "--";

    mailerflags[mf_idx++] = text;

    mailerflags[mf_idx] = NULL;

    ret = start_run(&RS,SY_NOTTY|SY_ENV_SHELL,mailerflags,-1,-1);

    if (!ret) {
	DPRINT(Debug,4,(&Debug,
			"Can't run %s for address verify -- falling back to default addr verify\n",
			sendmail_mailer_path));
	status =  mi_vrfyaddr_test_passwd;   /* caller does job */
	goto done;
    }
    ret = wait_end(&RS,&exit_code);
    if (ret < 0) {
	DPRINT(Debug,4,(&Debug,
			"%s died on signal %d -- falling back to default addr verify\n",
			sendmail_mailer_path,-ret));
	status =  mi_vrfyaddr_test_passwd;   /* caller does job */
	goto done;
    }
    if (0 == ret) {
	DPRINT(Debug,4,(&Debug,
			"%s lost? -- falling back to default addr verify\n",
			sendmail_mailer_path));
       	status =  mi_vrfyaddr_test_passwd;   /* caller does job */
	goto done;
    }

    /* -1 (mi_vrfyaddr_default_ok= is success, but caller should fill fullname */
    status = exit_code == 0 ? mi_vrfyaddr_default_ok : mi_vrfyaddr_failure;

 done:
    DPRINT(Debug,14,(&Debug,
		     "sendmail_mailer_info_verify_addr=%d",
		     status));
    switch (status) {
    case mi_vrfyaddr_test_passwd: DPRINT(Debug,14,(&Debug," mi_vrfyaddr_test_passwd")); break;
    case mi_vrfyaddr_default_ok:  DPRINT(Debug,14,(&Debug," mi_vrfyaddr_default_ok"));  break;
    case mi_vrfyaddr_failure:     DPRINT(Debug,14,(&Debug," mi_vrfyaddr_failure"));     break;
    case mi_vrfyaddr_success:     DPRINT(Debug,14,(&Debug," mi_vrfyaddr_success"));     break;
    }    
    DPRINT(Debug,14,(&Debug,"\n"));

    return status;

}

/* Return 1 if X->mail_from is changed */

static int sendmail_mailer_info_set_ef(M,I,X,value)
     struct mailer_config *M;
     struct mailer_info *I;
     struct mailer_env_from *X;
     const char *value;
{
    const char *mailerflags[5];
    int mf_idx=0;
    struct run_state RS;
    int ret,exit_code;

    if (0 == strcmp(value,"<>")) {
	/* Null envelope sender is only valid as sender (not as recipient)
	   so do not verify it
	*/
	X->mail_from = strmcpy(X->mail_from,value);

	DPRINT(Debug,13,(&Debug, "sendmail_mailer_info_set_ef: Clearing MAILER_ef_verified\n"));
	X->flags &= ~MAILER_ef_verified;
	
	return 1;
    }

    if (I->magic != MAILER_INFO_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,
	      "sendmail_mailer_info_set_ef",
	      "Bad magic number (mailer info)",0);


    if (!sendmail_verify_envelope) {
	/* Assume that every address is OK. */

	X->mail_from = strmcpy(X->mail_from,value);

	DPRINT(Debug,8,(&Debug,
			" .. envelope sender=%s\n",
			X->mail_from));

	DPRINT(Debug,13,(&Debug, "sendmail_mailer_info_set_ef: Clearing MAILER_ef_verified\n"));
	X->flags &= ~MAILER_ef_verified;
		
	return 1;
    }

    mailerflags[mf_idx++] = sendmail_mailer_path;

    mailerflags[mf_idx++] = "-bv";

    /* Do not treat -xyc as option */
    mailerflags[mf_idx++] = "--";

    mailerflags[mf_idx++] = value;

    mailerflags[mf_idx] = NULL;

    ret = start_run(&RS,SY_NOTTY|SY_ENV_SHELL,mailerflags,-1,-1);

    if (!ret) {
	DPRINT(Debug,4,(&Debug,
			"Can't run %s for address verify -- sendmail_mailer_info_set_ef failed\n",
			sendmail_mailer_path));
	goto failure;
    }

    ret = wait_end(&RS,&exit_code);
    if (ret < 0) {
	DPRINT(Debug,4,(&Debug,
			"%s died on signal %d -- sendmail_mailer_info_set_ef failed\n",
			sendmail_mailer_path,-ret));
	goto failure;
    }

    if (0 == ret) {
	DPRINT(Debug,4,(&Debug,
			"%s lost? -- sendmail_mailer_info_set_ef failed\n",
			sendmail_mailer_path));
	goto failure;
    }

    if (exit_code == 0) {       
	X->mail_from = strmcpy(X->mail_from,value);
	DPRINT(Debug,8,(&Debug,
			" .. envelope sender=%s\n",
			X->mail_from));

	/* MAILER_ef_verified  should only set for
	   250     (local mail)
	   but not
	   252     (relayed mail)
	   so do not set it
	*/
	
	DPRINT(Debug,13,(&Debug, "sendmail_mailer_info_set_ef: Clearing MAILER_ef_verified\n"));
	X->flags &= ~MAILER_ef_verified;

	return 1;
    } else {
	DPRINT(Debug,4,(&Debug,
			"... exit code %d -- envelope sender address %s bad?\n",
			exit_code,value));

    failure:
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmChangingSendmEnvSenderFailed,
			  "Changing of envelope sender for sendmail failed: %s"),
		  value);
    }

    return 0;
}

static int submitmail_mailer_init(M,C,I)
     struct mailer_config *M;
     struct mail_send_state *C;
     struct mailer_info *I;
{
    int i;

    if (C->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"submitmail_mailer_init",
	      "Bad magic number (mail send state)",0);

    if (!default_mailer_init(M,C,I))
	return 0;

    /* WARNING: Untested  --
       code copied from do_mmdf_addresses() on src/reply.c 
    */
    for (i = 0; i < C -> addr_count; i++) {
	state_puts(C -> addrs[i],C->OUT);  state_putc('\n',C->OUT);
    }
    state_putc('\n',C->OUT);

    return 1;  /* OK */
}


static int submitmail_mailer_backend3(M,C,encoding_top,title,sm,exit_status,
				      ticket,message_size)
     struct mailer_config *M;
     struct mail_send_state **C;
     int encoding_top;
     char * title;
     sending_message_func *sm;
     int *exit_status;
     struct mailer_cancel_ticket * ticket;
     long message_size;
{
    int r;
    const char *mailerflags[20];
    int mf_idx=0;

    if ((*C)->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"submitmail_mailer_backend",
	      "Bad magic number (mail send state)",0);

    /* WARNING: Untested  --
       code copied from mail_backend() on src/mailmsg2.c 
    */

    mailerflags[mf_idx++] = submitmail_mailer_path;

    mailerflags[mf_idx++] = "-mlrnv";

    mailerflags[mf_idx] = NULL;

    r = backend_tail(M,C,title,sm,mailerflags,exit_status,ticket);
    
    return r;
}

static int execmail_mailer_backend3(M,C,encoding_top,title,sm,exit_status,
				    ticket,message_size)
     struct mailer_config *M;
     struct mail_send_state **C;
     int encoding_top;
     char * title;
     sending_message_func *sm;
     int *exit_status;
     struct mailer_cancel_ticket * ticket;
     long message_size;
{
    int r;
    char *mailerflags[20];
    const char **argv;
    int mf_idx=0;

    if ((*C)->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"execmail_mailer_backend",
	      "Bad magic number (mail send state)",0);

    /* WARNING: Untested  --
       code copied from mail_backend() on src/mailmsg2.c 
    */

    mailerflags[mf_idx++] = execmail_mailer_path;

    if ((*C)->verbose)
	mailerflags[mf_idx++] = "-d";
    if (metoo)
	mailerflags[mf_idx++] = "-m";

    argv = join_argv(mailerflags,(*C)->addrs);
    
    r = backend_tail(M,C,title,sm,argv,exit_status,ticket);
    
    free(argv);

    return r;
}


/* mail_backend3 may free (*mail_fd) pointer ... */

int mail_backend3(mail_fd,func,encoding_top,title,
		  sm,exit_status,ticket,message_size)
     struct mail_send_state **mail_fd;
     end_handler *func;
     int encoding_top;
     char * title;
     sending_message_func *sm;
     int *exit_status /* may be NULL */;
     struct mailer_cancel_ticket * ticket  /* may be NULL */;
     long message_size;
{    
    int r;

    if ((*mail_fd)->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"mail_backend3",
	      "Bad magic number (mail send state)",0);

    DPRINT(Debug,7,(&Debug,
		    "Sending mail via %s mailer, %d recipients, message size %ld bytes\n",
		    (*mail_fd)->mailer_type->mailer_type,
		    (*mail_fd)->addr_count,
		    message_size));
    
    (*mail_fd)->orig_func = func;
    
    if (!valid_mailer((*mail_fd)->mailer_type))
	panic("MAILER PANIC",__FILE__,__LINE__,"mail_backend3",
	      "Bad mailer type",
	      0);

    if (MAILER_CONFIG_magic != (*mail_fd)->mailer_type->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mail_backend3",
	      "Bad magic number (mailer_config)",0);

    if ((*mail_fd)->msd_sd && message_size >= 0)
	(*mail_fd)->msd_size_upd_cb((*mail_fd)->msd_sd,
				    msd_message_size,message_size);

    if (ticket) {
	/* Clear ticket if it is reused */

	if (MAILER_CANCEL_TICKET_magic != ticket->magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,"mail_backend3",
		  "Bad magic number (mailer_cancel_ticket)",0);

	if (ticket->fname) {
	    if (! (*mail_fd)->fname ||
		((*mail_fd)->fname != ticket->fname &&
		 0 != strcmp((*mail_fd)->fname, ticket->fname))) {
		
		if (0 == unlink(ticket->fname)) {
		    DPRINT(Debug,8,(&Debug,"mail_backend3: Unlinked %s\n",
				    ticket->fname));		
		}
	    }

	    if ((*mail_fd)->fname != ticket->fname) {
		free(ticket->fname);
	    }
	    ticket->fname = NULL;	   
	}

	if (ticket->OUT) {
	    if (! (*mail_fd)->OUT ||
		(*mail_fd)->OUT != ticket->OUT) {

		/* close assigned FD */
		FILE *F = out_state_FILE(ticket->OUT);

		free_out_state(& (ticket->OUT));

		if (F) {
		    fclose(F);
		}
	    } else
		ticket->OUT = NULL;   /* ? reset shared pointer */
	}

	ticket->msd_action = msd_act_default;
    }

    DPRINT(Debug,20,(&Debug,
		     "mail_backend3: calling m_backend3_hook%s\n",
		     ticket ? ", have ticket" : ""));

    /* May free (*mail_fd) */
    r = (*mail_fd)->mailer_type->m_backend3_hook((*mail_fd)->mailer_type,
						 mail_fd,encoding_top,
						 title,sm,exit_status,
						 ticket,message_size);      

    DPRINT(Debug,20,(&Debug,"mail_backend3=%d",r));
    if (exit_status) {
	DPRINT(Debug,20,(&Debug,", *exit_status=%d",*exit_status));
    }
    if (ticket) {
	DPRINT(Debug,20,(&Debug,", ticket->msd_action=%d",
			 ticket->msd_action));
    }
    DPRINT(Debug,20,(&Debug,"\n"));
  
    return r;
}

# ifdef PWDINSYS
#  include <sys/pwd.h>
# else
#  include <pwd.h>
# endif

/* Should return 1 if verify succees and fields of result to be filled 
   -- caller should free fields of result
   Returns 0 on failure
*/
int verify_mailer_addr(I,text,result, errcode)
     struct mailer_info  *I;
     const char *text;
     struct addr_verify_result *result;
     enum mailer_errcode  * errcode;
{
    enum mi_vrfyaddr_mailer_status ret = mi_vrfyaddr_failure;
    int status = 0;
    
    if (I->magic != MAILER_INFO_magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"verify_mailer_addr",
	      "Bad magic number (mailer info)",0);

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)result, sizeof (*result));
    result -> addr      = NULL;
    result -> fullname  = NULL;
    result -> comment   = NULL;
    
    *errcode = MAILER_NOT_AVAIL;

    if (!valid_mailer(I->mailer_type))
	panic("MAILER PANIC",__FILE__,__LINE__,"verify_mailer_addr",
	      "Bad mailer type",
	      0);

    if (MAILER_CONFIG_magic != I->mailer_type->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"verify_mailer_addr",
	      "Bad magic number (mailer_config)",0);


    /* Returns 
       -1 if caller should result with default data (return 1)  (mi_vrfyaddr_default_ok)
       -2 if caller should just test passwd                     (mi_vrfyaddr_test_passwd)
       0  if failure                                            (mi_vrfyaddr_failure)
       1  is succees                                            (mi_vrfyaddr_success)
    */
    ret = I->mailer_type->mi_verify_addr(I->mailer_type,I,text,result);

    DPRINT(Debug,7,(&Debug,"Address %s verify result %d",
		    text,ret));
    switch (ret) {
    case mi_vrfyaddr_test_passwd: DPRINT(Debug,7,(&Debug," mi_vrfyaddr_test_passwd")); break;
    case mi_vrfyaddr_default_ok:  DPRINT(Debug,7,(&Debug," mi_vrfyaddr_default_ok"));  break;
    case mi_vrfyaddr_failure:     DPRINT(Debug,7,(&Debug," mi_vrfyaddr_failure"));     break;
    case mi_vrfyaddr_success:     DPRINT(Debug,7,(&Debug," mi_vrfyaddr_success"));     break;
    }
    DPRINT(Debug,7,(&Debug,"%s\n",		    
		    ret < mi_vrfyaddr_failure ? " (use default processing)" : ""));
    if (result->addr) {
	DPRINT(Debug,7,(&Debug,
			" ... resulting address %s",
			result->addr));	
    }

    switch (ret) {
    case mi_vrfyaddr_failure:
	*errcode = MAILER_NOT_EXIST;
	status = 0;
	break;
    case  mi_vrfyaddr_success:
	*errcode = MAILER_OK;
	status = 1;
	break;
    case mi_vrfyaddr_test_passwd:
    case mi_vrfyaddr_default_ok: {
	struct passwd * P = getpwnam(text);

	if (P)
	    *errcode = MAILER_OK;

	if (P && ! result -> fullname ) {
	    char * N = get_fullname1(P,text);

	    /* FIXME: Posible wrong charset */
	    if (N) {
		result -> fullname = new_string2(system_charset,s2us(N));
		DPRINT(Debug,7,(&Debug,
				"   ... adding fullname %S",
				result -> fullname));
	    }

	    status = 1;
	} else {
	    status = ret == mi_vrfyaddr_default_ok ? 1 : 0;
	}
    }
	break;
    }

    if (status && !result->addr) {

	DPRINT(Debug,7,(&Debug,
			" ... filling resulting address\n"));
	
	result->addr = safe_strdup(text);
	

	if (I->mailer_type->mi_query_hook(I->mailer_type,I,
					  MI_USE_DOMAIN)) {

	    const char * maildomain = hostfullname;

	    const char * have_mailname = useful_mailname();		   
	    
	    if (have_mailname)
		maildomain = have_mailname;
	    else {
		DPRINT(Debug,7,(&Debug,
				 "verify_mailer_addr: mailname not set or not useful, using hostfullname=%s\n"
				,maildomain));
	    }

	    result->addr = strmcat(result->addr,"@");
	    result->addr = strmcat(result->addr,maildomain);
	    DPRINT(Debug,7,(&Debug,
			    "verify_mailer_addr: address %s => %s\n",
			    text,result->addr));
	} 
    } else {
	DPRINT(Debug,7,(&Debug,"\n"));
    }

    DPRINT(Debug,7,(&Debug,"verify_mailer_addr=%d -- %s",
		    status, text));

    if (result -> addr) {
	DPRINT(Debug,7,(&Debug,"; addr=%s",
			result -> addr));
    }
    if (result -> fullname) {
	DPRINT(Debug,7,(&Debug,"; fullname=%S",
			result -> fullname));
    }
    if (result -> comment) {
	DPRINT(Debug,7,(&Debug,"; comment=%S",
			result -> comment));
    }    
    DPRINT(Debug,7,(&Debug,"\n"));
    
    return status;
}

/* Should return 1 if verify succees and fields of result to be filled 
   -- caller should free fields of result
   Returns 0 on failure
*/
int verify_mailer_domaddr(I,text,result, errcode)
     struct mailer_info  *I;
     const char *text;
     struct addr_verify_result *result;
     enum mailer_errcode  * errcode;
{
    enum mi_vrfyaddr_mailer_status ret = mi_vrfyaddr_failure;
    int status = 0;
    
    if (I->magic != MAILER_INFO_magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"verify_mailer_domaddr",
	      "Bad magic number (mailer info)",0);

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)result, sizeof (*result));
    result -> addr      = NULL;
    result -> fullname  = NULL;
    result -> comment   = NULL;
    
    *errcode = MAILER_NOT_AVAIL;

    if (!valid_mailer(I->mailer_type))
	panic("MAILER PANIC",__FILE__,__LINE__,"verify_mailer_domaddr",
	      "Bad mailer type",
	      0);
    
    if (MAILER_CONFIG_magic != I->mailer_type->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"verify_mailer_domaddr",
	      "Bad magic number (mailer_config)",0);

    /* Returns 
       -1 if caller should result with default data (return 1)  (mi_vrfyaddr_default_ok)
       -2 if caller should just test passwd                     (mi_vrfyaddr_test_passwd)
       0  if failure                                            (mi_vrfyaddr_failure)
       1  is succees                                            (mi_vrfyaddr_success)
    */
    ret = I->mailer_type->mi_verify_addr(I->mailer_type,I,text,result);

    DPRINT(Debug,7,(&Debug,"Address %s verify result %d",
		    text,ret));
    switch (ret) {
    case mi_vrfyaddr_test_passwd: DPRINT(Debug,7,(&Debug," mi_vrfyaddr_test_passwd")); break;
    case mi_vrfyaddr_default_ok:  DPRINT(Debug,7,(&Debug," mi_vrfyaddr_default_ok"));  break;
    case mi_vrfyaddr_failure:     DPRINT(Debug,7,(&Debug," mi_vrfyaddr_failure"));     break;
    case mi_vrfyaddr_success:     DPRINT(Debug,7,(&Debug," mi_vrfyaddr_success"));     break;
    }
    DPRINT(Debug,7,(&Debug,"%s\n",		    
		    ret < mi_vrfyaddr_failure ? " (use default processing)" : ""));

    if (result->addr) {
	DPRINT(Debug,7,(&Debug,
			"   ... resulting address %s",
			result->addr));	
    }

    
    switch (ret) {
    case mi_vrfyaddr_failure:
	*errcode = MAILER_NOT_EXIST;
	status = 0;
	break;
    case  mi_vrfyaddr_success:
	*errcode = MAILER_OK;
	status = 1;
	break;
    case mi_vrfyaddr_default_ok:
	status = 1;
	break;
    case mi_vrfyaddr_test_passwd:
	status = 0;
	break;
    }
    
    if (status && !result->addr) {

	DPRINT(Debug,7,(&Debug,
			"   ... filling resulting address\n",
			text));
	
	result->addr = safe_strdup(text);
    } else {
	DPRINT(Debug,7,(&Debug,"\n"));
    }

    DPRINT(Debug,7,(&Debug,"verify_mailer_domaddr=%d -- %s",
		    status, text));

    if (result -> addr) {
	DPRINT(Debug,7,(&Debug,"; addr=%s",
			result -> addr));
    }
    if (result -> fullname) {
	DPRINT(Debug,7,(&Debug,"; fullname=%S",
			result -> fullname));
    }
    if (result -> comment) {
	DPRINT(Debug,7,(&Debug,"; comment=%S",
			result -> comment));
    }    
    DPRINT(Debug,7,(&Debug,"\n"));
    

    return status;
}


char * kludge_addr(char **addr) {
    char * ptr = *addr;
    int l = strlen(ptr);
    char *res;
    
    ptr = safe_realloc(ptr,l + 1 + 2 + l + 1);

    res = ptr + l + 1;

    res[0] = '<';
    strfcpy(res+1,ptr,l+1);
    res[l+1] = '>';
    res[l+2] = '\0';

    *addr = ptr;
    return res;
}

/* ---------------------------------------------------------------------- */

struct mailer_cancel_ticket * new_mailer_cancel_ticket()
{
    struct mailer_cancel_ticket *ret = safe_zero_alloc (sizeof (*ret));

    ret->magic         = MAILER_CANCEL_TICKET_magic;

    ret->msd_action    = msd_act_default;
    ret->fname         = NULL;
    ret->OUT           = NULL;

    return ret;
}

void free_mailer_cancel_ticket(ticket) 
     struct mailer_cancel_ticket ** ticket;
{
    if (MAILER_CANCEL_TICKET_magic != (*ticket)->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"free_mailer_cancel_ticket",
	      "Bad magic number (mailer_cancel_ticket)",0);
    
    if ((*ticket)->OUT) {

	/* close assigned FD */
	FILE *F = out_state_FILE((*ticket)->OUT);
	
	free_out_state(& ((*ticket)->OUT));

	if (F) {
	    fclose(F);
	}
    }

    if ((*ticket)->fname) {

	if (0 == unlink((*ticket)->fname)) {
	    DPRINT(Debug,8,(&Debug,
			    "free_mailer_cancel_ticket: Unlinked %s\n",
			    (*ticket)->fname));		
	}
	free((*ticket)->fname);
	(*ticket)->fname = NULL;
    }
	
    (*ticket)->magic = 0;  /* Invalidate */
    free(*ticket);
    *ticket = NULL;
}

enum msd_action have_msd_action(ticket)
     struct mailer_cancel_ticket *ticket;
{
    if (MAILER_CANCEL_TICKET_magic != ticket->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"have_msd_action",
	      "Bad magic number (mailer_cancel_ticket)",0);

    return ticket->msd_action;
}
 
const char * have_cancel_filename(ticket) 
     struct mailer_cancel_ticket * ticket;
{
    if (MAILER_CANCEL_TICKET_magic != ticket->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"have_cancel_filename",
	      "Bad magic number (mailer_cancel_ticket)",0);

    return ticket->fname;
}

FILE * have_cancel_FILE(ticket)
     struct mailer_cancel_ticket * ticket;
{
    if (MAILER_CANCEL_TICKET_magic != ticket->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"have_cancel_FILE",
	      "Bad magic number (mailer_cancel_ticket)",0);

    if (ticket->OUT) {
	
	FILE *F = out_state_FILE(ticket->OUT);

	rewind(F);

	return F;
    }

    return NULL;
}
/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
