static char rcsid[] = "@(#)$Id: message-id.c,v 1.10 2025/05/29 05:42:57 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.10 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mailer.h"
#include "s_me.h"

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* For INT_MAX */
#include <limits.h>


DEBUG_VAR(Debug,__FILE__,"header");

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

#define DEVURANDOM "/dev/urandom"
#define PROCUUID   "/proc/sys/kernel/random/uuid"

static enum msg_id_localspec {
    msgidlocal_bad      = -1,
    msgidlocal_none     = 0,
    msgidlocal_md5,
    msgidlocal_devurandom,
    msgidlocal_procuuid,
    
    NUM_msgidlocal
} message_id_localspec = msgidlocal_bad;

static const char * MSG_ID_localspec[NUM_msgidlocal+1] = {
    "none",
    "md5",
    DEVURANDOM,
    PROCUUID,
    
    NULL
};

static enum msg_id_domspec {
    msgiddom_whitelisted  = -4,
    msgiddom_literal      = -3,
    msgiddom_domain       = -2,
    msgiddom_bad          = -1,
    msgiddom_default      = 0,
    msgiddom_hostfullname,
    msgiddom_maildomain,

    NUM_msgiddom
} message_id_domspec = msgiddom_default;
    
static const char * MSG_ID_domspec[NUM_msgiddom+1] = {
    "default",
    "hostfullname",
    "maildomain",

    NULL
};

static int default_timeout_sec  = 3600;

#define DOMAIN_INFO_magic	0xF20A

static struct domain_info {
    unsigned short magic;      /* DOMAIN_INFO_magic */
    
    char                    * dompart;
    enum check_dname_result   domstat;
    enum check_dname_status {
	cds_reserved    = -4,
	cds_no_routine  = -3   /* NO USE_DLOPEN                 */,
	cds_fail        = -1   /* failure or name not found     */ ,   
	cds_none         = 0   /* shared_check_dname not called */,
	cds_ok           = 1   /* name ok to use */,
	cds_bad_syntax   = 3   /* bad syntax */,
	cds_whitelisted  = 4   /* whitelisted, shared_check_dname not called */,
	cds_disabled_ok        /* name ok to use, check disabled */
    }                         domcheck;

    char                    * rewrite;
    struct schedule_timelimit valid_until;

    unsigned                busy:1;    /* check_msg_domain0   busy 
					  prevent clear_vrfy_domain_info
				       */

} * new_domain_info P_((const char *domain, enum check_dname_status st,
			const struct schedule_timelimit * default_valid_until));

static struct timed_task_handle  * cleanup_domver_handle = NULL;
static struct schedule_timelimit   next_domver_cleanup;

/* Returns 1 or 0 (number of cleared objects) 
   Updates next_domver_cleanup
*/
static int clear_vrfy_domain_info P_((struct domain_info *dom,
				      const struct schedule_timelimit *now));
static int clear_vrfy_domain_info(dom,now)
     struct domain_info *dom;
     const struct schedule_timelimit *now;
{
    int r = 0;
    
    if (DOMAIN_INFO_magic != dom->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "clear_vrfy_domain_info",
	      "Bad magic number (domain_info)",0);

    if ((cds_none != dom->domcheck || dom->rewrite)
	&&
	schedule_have_timelimit(& (dom->valid_until))) {

	if (schedule_valid_until_ok(& (dom->valid_until),now)) {
	    
	    if (schedule_shorten_next_timeout(&next_domver_cleanup,
					      & (dom->valid_until))) {
		
		char * X = schedule_timeout_string(&next_domver_cleanup);
		
		if (X) {
		    DPRINT(Debug,14,(&Debug,
				     "clear_vrfy_domain_info: [%p] %Q: setting or shortened next vrfy cleanup to %s\n",
				     dom,dom->dompart ? dom->dompart : "(not set)",
				     X));			
		     free(X);
		}	
	    }
	    
        } else if (dom->busy) {

	    DPRINT(Debug,14,(&Debug,
			     "clear_vrfy_domain_info: [%p] %Q: busy -- clearing skipped\n",
			     dom,dom->dompart ? dom->dompart : "(not set)"));

	} else {
	    char *X;
	    DPRINT(Debug,14,(&Debug,
			     "clear_vrfy_domain_info: [%p] %Q: clearing verify info",
			     dom,dom->dompart ? dom->dompart : "(not set)"));
	    
	    X = schedule_timeout_string(& (dom->valid_until));
	    
	    if (X) {
		DPRINT(Debug,14,(&Debug,", expired on %s",X));
		free(X);
	    }
	    
	    if (dom->rewrite) {
		DPRINT(Debug,14,(&Debug,", rewrite %Q",dom->rewrite));
		free(dom->rewrite);
		
	    }
	    dom->rewrite = NULL;
	    
	    DPRINT(Debug,14,(&Debug,"\n"));
	    
	    switch (dom->domcheck) {
	    case cds_reserved:   break;
	    case cds_no_routine:
	    case cds_fail:
	    case cds_ok:
		dom->domcheck = cds_none;
		break;
	    case cds_none:       break;
	    case cds_bad_syntax: break;
	    case cds_whitelisted: break;
	    case cds_disabled_ok: break;
	    }
	    
	    dom->valid_until = NO_schedule_timelimit;
	    r = 1;
	}
    }

    DPRINT(Debug,18,(&Debug,"clear_vrfy_domain_info=%d: [%p] %Q\n",
		     r,dom,dom->dompart ? dom->dompart : "(not set)"));
    return r;
}

static struct domain_info * new_domain_info(domain,st,default_valid_until)
     const char *domain;
     enum check_dname_status st;
     const struct schedule_timelimit * default_valid_until;
{
    struct domain_info *ret = safe_zero_alloc(sizeof (*ret));

    ret->magic = DOMAIN_INFO_magic;

    if (domain[0])
	ret->dompart  = safe_strdup(domain);
    else
	ret->dompart  = NULL;
    ret->domstat  = check_dname_no_routine;
    ret->domcheck = st;

    ret->rewrite  = NULL;

    if (default_valid_until)
	ret->valid_until = *default_valid_until;
    else
	ret->valid_until = NO_schedule_timelimit;
    
    ret->busy = 0;

    return ret;
}

static void free_domain_info P_((struct domain_info **ptr));
static void free_domain_info(ptr)
     struct domain_info **ptr;
{
    if (DOMAIN_INFO_magic != (*ptr)->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "free_domain_info",
	      "Bad magic number (domain_info)",0);

    if ((*ptr)->busy) {

	DPRINT(Debug,1,(&Debug,
			"free_domain_info: [%p] %Q: busy !!!!!\n",
			(*ptr),(*ptr)->dompart ? (*ptr)->dompart : "(not set)"));

	(*ptr)->busy = 0;
    }

    if ((*ptr)->dompart) {
	free((*ptr)->dompart);
	(*ptr)->dompart = NULL;
    }

    if ((*ptr)->rewrite) {
	free((*ptr)->rewrite);
	(*ptr)->rewrite = NULL;
    }

    (*ptr)->magic = 0;  /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}


enum domain_names {
    dname_domain,
    dname_hostfullname,
    dname_mailer,
    dname_maildomain,

    NUM_dname
};

static const char * domain_names_d[NUM_dname] UNUSED_VAROK = {
    "domain",
    "hostfullname",
    "mailer",
    "maildomain"
};

static struct domain_info * message_id_dinfo[NUM_dname];

static void set_msgid_dompart P_((enum domain_names d, const char *domain,
				  enum check_dname_status st,
				  const struct schedule_timelimit * default_valid_until
				  ));
static void set_msgid_dompart(d,domain,st,default_valid_until)
     enum domain_names d;
     const char *domain;
     enum check_dname_status st;
     const struct schedule_timelimit * default_valid_until;
{
    if (d < 0 || d >= NUM_dname)
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "set_msgid_dompart",
	      "Bad domain name index",0);

    if (message_id_dinfo[d]) {
	if (DOMAIN_INFO_magic != message_id_dinfo[d]->magic) 
	    panic("MAILER PANIC",__FILE__,__LINE__,
		  "set_msgid_dompart",
		  "Bad magic number (domain_info)",0);

	if (domain && domain[0]) {
	    DPRINT(Debug,14,(&Debug,
			     "set_msgid_dompart: Setting message_id #%d %s dompart=%Q\n",
			     d,domain_names_d[d], domain));
	    
	    message_id_dinfo[d]->dompart  = strmcpy(message_id_dinfo[d]->dompart,domain);
	} else if (message_id_dinfo[d]->dompart) {
	    DPRINT(Debug,14,(&Debug,
			     "set_msgid_dompart: Freeing message_id #%d %s dompart=%Q\n",
			     d,domain_names_d[d],
			     message_id_dinfo[d]->dompart));
	    
	    free(message_id_dinfo[d]->dompart);
	    message_id_dinfo[d]->dompart = NULL;    
	}
	
	if (message_id_dinfo[d]->rewrite) {
	    DPRINT(Debug,14,(&Debug,
			     "set_msgid_dompart: Freeing message_id #%d %s rewrite=%Q\n",
			     d,domain_names_d[d],
			     message_id_dinfo[d]->rewrite));
	    
	    free(message_id_dinfo[d]->rewrite);
	    message_id_dinfo[d]->rewrite = NULL;    
	}
	
	message_id_dinfo[d]->domstat  = check_dname_no_routine;
	message_id_dinfo[d]->domcheck = st;
	if (default_valid_until)
	    message_id_dinfo[d]->valid_until = *default_valid_until;
	else
	    message_id_dinfo[d]->valid_until = NO_schedule_timelimit;
	
    } else {
	DPRINT(Debug,14,(&Debug,
			 "set_msgid_dompart: Allocating message_id #%d %s dompart=%Q\n",
			 d,domain_names_d[d], domain));

	message_id_dinfo[d] = new_domain_info(domain,st,default_valid_until);
    }
}

static struct domain_info ** fallback_domain = NULL;
static size_t                fallback_domain_count = 0;

S_(timed_task_action_f cleanup_vrfy_domain_result)
static enum timed_task_r cleanup_vrfy_domain_result P_((struct timed_task *task,
							const struct schedule_timelimit *now));
static enum timed_task_r cleanup_vrfy_domain_result(task,now)
     struct timed_task *task  /* NOT USED */;
     const struct schedule_timelimit *now;
{
    enum domain_names d;
    int count UNUSED_VAROK = 0;
    char * A;

    next_domver_cleanup = NO_schedule_timelimit;

    for (d = 0; d < NUM_dname; d++) {
	
	if (message_id_dinfo[d]) {
	    if (DOMAIN_INFO_magic != message_id_dinfo[d]->magic) 
		panic("MAILER PANIC",__FILE__,__LINE__,
		      "cleanup_vrfy_domain_result",
		      "Bad magic number (domain_info)",0);

	    count +=
		clear_vrfy_domain_info(message_id_dinfo[d],now);

	}
    }

    if (fallback_domain) {
	size_t i;

	for (i = 0; i < fallback_domain_count; i++) {
	    if (fallback_domain[i]) {
		if (DOMAIN_INFO_magic != fallback_domain[i]->magic) 
		    panic("MAILER PANIC",__FILE__,__LINE__,
			  "cleanup_vrfy_domain_result",
			  "Bad magic number (domain_info)",0);
		
		count += 
		    clear_vrfy_domain_info(fallback_domain[i],now);
	    }
	}
    }

    DPRINT(Debug,9,(&Debug,
		    "cleanup_vrfy_domain_result: %d cached resuls cleaned",
		    count));

    A = schedule_timeout_string(&next_domver_cleanup);
    if (A) {
	DPRINT(Debug,9,(&Debug, ", next vrfy cleanup %s",
			A));

	free(A);
    }

    if (schedule_have_timelimit(&next_domver_cleanup)) {

	if (alter_timed_task(&cleanup_domver_handle,NULL,
			     NO_timed_task_free,
			     NO_inc_timed_task_refcount,
			     cleanup_vrfy_domain_result,
			     &next_domver_cleanup)) {
	    
	    DPRINT(Debug,9,(&Debug,
			"cleanup_vrfy_domain_result: Updated vrfy cleanup, handle %p\n",
			cleanup_domver_handle));
	}
    } else {

	DPRINT(Debug,9,(&Debug,
			"cleanup_vrfy_domain_result: freeing vrfy cleanup, handle %p\n",
			cleanup_domver_handle));

	alter_timed_task(&cleanup_domver_handle,NULL,
			 NO_timed_task_free,
			 NO_inc_timed_task_refcount,
			 NO_timed_task_action,
			 &next_domver_cleanup);

    }
	
    return timed_task_default;
}


static void update_next_domver_cleanup P_((const struct schedule_timelimit * valid_until));
static void update_next_domver_cleanup(valid_until)   
     const struct schedule_timelimit * valid_until;
{

    if (schedule_shorten_next_timeout(&next_domver_cleanup,
				      valid_until)) {

	char * X = schedule_timeout_string(&next_domver_cleanup);
			
	if (X) {
	    DPRINT(Debug,9,(&Debug,
			    "update_next_domver_cleanup: setting or shortened next vrfy cleanup to %s\n",
			    X));			
	    free(X);
	}	

	if (alter_timed_task(&cleanup_domver_handle,NULL,
			     NO_timed_task_free,
			     NO_inc_timed_task_refcount,
			     cleanup_vrfy_domain_result,
			     &next_domver_cleanup)) {
	    
	    DPRINT(Debug,9,(&Debug,
			    "update_next_domver_cleanup: Updated vrfy cleanup, timed task handle %p\n",
			    cleanup_domver_handle));
	}
    }
}

static int fallback_initialized = 0;

static void free_fallback_domains P_((void));
static void free_fallback_domains()
{
    if (fallback_domain) {
	size_t i;

	for (i = 0; i < fallback_domain_count; i++) {
	    if (fallback_domain[i]) {
		if (DOMAIN_INFO_magic != fallback_domain[i]->magic) 
		    panic("MAILER PANIC",__FILE__,__LINE__,
			  "free_fallback_domains",
			  "Bad magic number (domain_info)",0);
		
		
		DPRINT(Debug,14,(&Debug,
				 "free_fallback_domains: Freeing fallback_domain[%zu]",
				 i));
		if (fallback_domain[i]->dompart) {
		    DPRINT(Debug,14,(&Debug,
				     "=%Q",
				     fallback_domain[i]->dompart));
		}
		DPRINT(Debug,14,(&Debug,"\n"));

		free_domain_info(& (fallback_domain[i]));
	    }
	}
	free(fallback_domain);
	fallback_domain = NULL;
    }
    fallback_domain_count = 0;
    
    fallback_initialized = 1;
}

static void add_fallback_domain P_((const char * domain));
static void add_fallback_domain(domain)
     const char * domain;
{
    size_t newcount = fallback_domain_count+1;

    if (fallback_domain) {
	size_t i;

	for (i = 0; i < fallback_domain_count; i++) {
	    if (fallback_domain[i]) {
		if (DOMAIN_INFO_magic != fallback_domain[i]->magic) 
		    panic("MAILER PANIC",__FILE__,__LINE__,
			  "add_fallback_domain",
			  "Bad magic number (domain_info)",0);

		if (fallback_domain[i]->dompart &&
		    0 == strcmp(domain,fallback_domain[i]->dompart)) {
		    DPRINT(Debug,14,(&Debug,"add_fallback_domain: Found fallback_domain[%zu]=%Q\n",
				     i,fallback_domain[i]->dompart));
		    return;
		}
		
	    }	    
	}
    }

    fallback_initialized = 1;
    
    fallback_domain =
	safe_array_realloc(fallback_domain,newcount,sizeof (fallback_domain[0]));

    fallback_domain[fallback_domain_count] = new_domain_info(domain,cds_none,NULL);

    DPRINT(Debug,14,(&Debug,"add_fallback_domain: Adding fallback_domain[%zu]=%Q\n",
		     fallback_domain_count,domain));

    fallback_domain_count = newcount;
}

static int devurandom_fd = -1;
#define DEVURANDOM_LEN 16

static unsigned char devurandom_buffer[DEVURANDOM_LEN];
static int           devurandom_len = 0;
static enum buffer_usage {
    buffer_bad,
    buffer_initialized,
    buffer_used
}                    devurandom_used = buffer_bad;

static void close_devurandom P_((void));
static void close_devurandom()
{
    int r = close(devurandom_fd);
    
    if (-1 == r) {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,8,(&Debug,
			"close_devurandom: close %s (fd=%d): %s (errno=%d)\n",
			DEVURANDOM,devurandom_fd,strerror(err),err));
	
    } else if (0 == r) {
	DPRINT(Debug,8,(&Debug,
			"close_devurandom: close %s (fd=%d) closed\n",
			DEVURANDOM,devurandom_fd));
    }
    
    devurandom_fd = -1;
}

static int procuuid_fd = -1;
/* length includes \n */
#define PROCUUID_LEN 37

static unsigned char procuuid_buffer[PROCUUID_LEN];
static int           procuuid_len = 0;
static enum buffer_usage procuuid_used = buffer_bad;

static void close_procuuid P_((void));
static void close_procuuid()
{
    int r = close(procuuid_fd);
    
    if (-1 == r) {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,8,(&Debug,
			"close_procuuid: close %s (fd=%d): %s (errno=%d)\n",
			PROCUUID,procuuid_fd,strerror(err),err));
	
    } else if (0 == r) {
	DPRINT(Debug,8,(&Debug,
			"close_procuuid: close %s (fd=%d) closed\n",
			PROCUUID,procuuid_fd));
    }
    
    procuuid_fd = -1;
}

static enum msg_check_dom_mode {
    msg_checkdom_tag        = -1     /* libary tag given */,
    msg_checkdom_none       = 0,
    msg_checkdom_shared,

    NUM_msg_check_dom_mode
} msg_checkdom_mode = 
#ifdef USE_DLOPEN
    msg_checkdom_shared
#else
    msg_checkdom_none
#endif
    ;
static char * msg_verifydom_tag = NULL; 
static const char *MSG_CHECKDOM_mode[NUM_msg_check_dom_mode+1]= {
    "none",
    "shared",
    
    NULL
};

static char * saved_domspec    = NULL;
static char * saved_localspec  = NULL;
static char * saved_params = NULL;

static char * message_id_return_buffer = NULL;

void message_id_free_rc()
{
    enum domain_names d;
    
    DPRINT(Debug,8,(&Debug, "message_id_free_rc: Freeing message-id option data\n"));

    if (cleanup_domver_handle) {
    	DPRINT(Debug,9,(&Debug,
			"message_id_free_rc: Freeing domain verify cleanup handle %p\n",
			cleanup_domver_handle));

	alter_timed_task(&cleanup_domver_handle,NULL,
			 NO_timed_task_free,
			 NO_inc_timed_task_refcount,
			 NO_timed_task_action,
			 &next_domver_cleanup);
    }
	
    for (d = 0; d < NUM_dname; d++) {
	
	if (message_id_dinfo[d]) {
	    if (DOMAIN_INFO_magic != message_id_dinfo[d]->magic) 
		panic("MAILER PANIC",__FILE__,__LINE__,
		      "message_id_free_rc",
		      "Bad magic number (domain_info)",0);
	
	    DPRINT(Debug,14,(&Debug,
			     "message_id_free_rc: Freeing message_id #%d %s",
			     d,domain_names_d[d]));
	    
	    if (message_id_dinfo[d]->dompart) {
		DPRINT(Debug,14,(&Debug," dompart=%Q",
				 message_id_dinfo[d]->dompart));
	    }

	    DPRINT(Debug,14,(&Debug,"\n"));

	    free_domain_info(& (message_id_dinfo[d]));
	}
    }

    free_fallback_domains();

    if (-1 != devurandom_fd) {
	close_devurandom();	
    }

    if (devurandom_len) {
	/* bzero is defined hdrs/elm_defs.h */
	bzero(devurandom_buffer, sizeof devurandom_buffer);

	devurandom_len = 0;
    }
    
    if (-1 != procuuid_fd) {
	close_procuuid();	
    }

    if (procuuid_len) {
	/* bzero is defined hdrs/elm_defs.h */
	bzero(procuuid_buffer, sizeof procuuid_buffer);

	procuuid_len = 0;
    }

    if (saved_domspec) {
	free(saved_domspec);
	saved_domspec = NULL;
    }
    
    if (saved_localspec) {
	free(saved_localspec);
	saved_localspec = NULL;
    }
    
    if (saved_params) {
	free(saved_params);
	saved_params = NULL;
    }

    if (msg_verifydom_tag) {
	free(msg_verifydom_tag);
	msg_verifydom_tag = NULL;
    }

    if (message_id_return_buffer) {
	free(message_id_return_buffer);
	message_id_return_buffer = NULL;
    }
    
}

static const struct digest_proc_type * have_md5 = NULL;

static int local_md5_supported P_((void));
static int local_md5_supported()
{
    int ret = 0;

    if (have_md5) {
	ret = 1;
    } else {
	have_md5 =  md5_available();

	if (have_md5) {
	    ret = 1;
	} else {
	    DPRINT(Debug,14,(&Debug, "local_md5_supported: digest_proc_md5 failed\n"));
	}
    }
    
    DPRINT(Debug,14,(&Debug, "local_md5_supported=%d\n",ret));
    
    return ret;
}

static int open_devurandom P_((void));
static int open_devurandom(void)
{
    int ret = 0;

    devurandom_fd = open(DEVURANDOM,O_RDONLY);

    if (-1 == devurandom_fd) {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,8,(&Debug,
			"open_devurandom: open %s: %s (errno=%d)\n",
			DEVURANDOM,devurandom_fd,strerror(err),err));
	
    } else {
	int r;

#ifdef FD_CLOEXEC
	r = fcntl(devurandom_fd, F_SETFD, FD_CLOEXEC);
#else
	r = fcntl(devurandom_fd, F_SETFD, 1);
#endif    

	if (-1 == r) {
	    int err UNUSED_VAROK = errno;
	    
	    DPRINT(Debug,14,(&Debug, 
			     "open_devurandom: Failed to set close-on-exec flag for %s: %s\n",
			     DEVURANDOM,
			     strerror(err)));
	} else if (0 == r) {
	    DPRINT(Debug,14,(&Debug, 
			     "open_devurandom: Set close-on-exec flag for %s\n",
			     DEVURANDOM));
	}
	
	ret = 1;
    }

    DPRINT(Debug,14,(&Debug, "open_devurandom=%d\n",ret));
    
    return ret;
}

static int local_devurandom_supported P_((void));
static int local_devurandom_supported()
{
    int ret = 0;

    if (-1 == devurandom_fd) {

	if (open_devurandom()) {

	    int len;

	    devurandom_used = buffer_bad;
	    len = read(devurandom_fd,devurandom_buffer,sizeof devurandom_buffer);
	    
	    if (len == sizeof devurandom_buffer)
		ret = 1;
	    
	    if (len > 0) {
		devurandom_len  = len;
		devurandom_used = buffer_initialized;
		
	    } else if (-1 == len) {
		
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,8,(&Debug,
				"local_devurandom_supported: read %s (fd=%d): %s (errno=%d)\n",
				DEVURANDOM,devurandom_fd,strerror(err),err));		
	    }	    
	}
	
    } else
	ret = 1;
    
    DPRINT(Debug,14,(&Debug, "local_devurandom_supported=%d\n",ret));
    
    return ret;
}

static int open_procuuid P_((void));
static int open_procuuid(void)
{
    int ret = 0;

    procuuid_fd = open(PROCUUID,O_RDONLY);

    if (-1 == procuuid_fd) {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,8,(&Debug,
			"open_procuuid: open %s: %s (errno=%d)\n",
			PROCUUID,procuuid_fd,strerror(err),err));
	
    } else {
	int r;

#ifdef FD_CLOEXEC
	r = fcntl(procuuid_fd, F_SETFD, FD_CLOEXEC);
#else
	r = fcntl(procuuid_fd, F_SETFD, 1);
#endif    

	if (-1 == r) {
	    int err UNUSED_VAROK = errno;
	    
	    DPRINT(Debug,14,(&Debug, 
			     "open_procuuid: Failed to set close-on-exec flag for %s: %s\n",
			     PROCUUID,
			     strerror(err)));
	} else if (0 == r) {
	    DPRINT(Debug,14,(&Debug, 
			     "open_procuuid: Set close-on-exec flag for %s\n",
			     PROCUUID));
	}
	
	ret = 1;
    }
    
    DPRINT(Debug,14,(&Debug, "open_procuuid=%d\n",ret));
    
    return ret;
}

static int valid_procuuid P_((unsigned char buffer[],int len));
static int valid_procuuid(buffer,len)
     unsigned char buffer[];
     int len;
{
    int i;
    int ret = 0;
    
    if (len != PROCUUID_LEN) {
	DPRINT(Debug,14,(&Debug, "valid_procuuid: len %d <> %d\n",
			 len,PROCUUID_LEN));
	
	ret = 0;
	goto fail;
    }
    
    /* 2d6177aa-e245-4100-953c-21379927782a */

    if ('-'  != buffer[8] ||
	'-'  != buffer[13] ||
	'-'  != buffer[18] ||
	'-'  != buffer[23] ||
	'\n' != buffer[36]) {

	DPRINT(Debug,14,(&Debug, "valid_procuuid: '-' or newline does not match\n"));
	
	ret = 0;
	goto fail;	
    }
	
    for (i = 0; i < 8; i++) {
	if (-1 == hex(buffer[i])) {

	    DPRINT(Debug,14,(&Debug, "valid_procuuid: not a hex char on #%d\n",
			     i));
	    
	    ret = 0;
	    goto fail;	
	}
    }
    for (i = 9; i < 13; i++) {
	if (-1 == hex(buffer[i])) {

	    DPRINT(Debug,14,(&Debug, "valid_procuuid: not a hex char on #%d\n",
			     i));
	    
	    ret = 0;
	    goto fail;	
	}
    }
    for (i = 14; i < 18; i++) {
	if (-1 == hex(buffer[i])) {

	    DPRINT(Debug,14,(&Debug, "valid_procuuid: not a hex char on #%d\n",
			     i));
	    
	    ret = 0;
	    goto fail;	
	}
    }
    for (i = 19; i < 23; i++) {
	if (-1 == hex(buffer[i])) {

	    DPRINT(Debug,14,(&Debug, "valid_procuuid: not a hex char on #%d\n",
			     i));
	    
	    ret = 0;
	    goto fail;	
	}
    }
    for (i = 24; i < 36; i++) {
	if (-1 == hex(buffer[i])) {

	    DPRINT(Debug,14,(&Debug, "valid_procuuid: not a hex char on #%d\n",
			     i));
	    
	    ret = 0;
	    goto fail;	
	}
    }

    ret = 1;

 fail:
    DPRINT(Debug,14,(&Debug, "valid_procuuid=%d\n",ret));

    return ret;
}

static int local_procuuid_supported P_((void));
static int local_procuuid_supported()
{
    int ret = 0;

    if (-1 == procuuid_fd) {

	if (open_procuuid()) {

	    int len;

	    procuuid_used = buffer_bad;
	    len = read(procuuid_fd,procuuid_buffer,sizeof procuuid_buffer);

	    if (len > 0) {
		procuuid_len = len;
		
		ret = valid_procuuid(procuuid_buffer,len);
		if (ret)
		    procuuid_used = buffer_initialized;
		
	    } else if (-1 == len) {
		int err UNUSED_VAROK = errno;
	
		DPRINT(Debug,8,(&Debug,
				"local_procuuid_supported: read %s (fd=%d): %s (errno=%d)\n",
				PROCUUID,procuuid_fd,strerror(err),err));		
	    }	    
	}
	
    } else
	ret = 1;
    
    DPRINT(Debug,14,(&Debug, "local_procuuid_supported=%d\n",ret));
    
    return ret;
}

static const char INVALID[] = "(invalid)";

static char * generate_localspec P_((int *ok_p));
static char * generate_localspec(ok_p)
     int *ok_p;
{
    char * ret = NULL;

    if (message_id_localspec >= 0 && message_id_localspec < NUM_msgidlocal)
	ret = safe_strdup(MSG_ID_localspec[message_id_localspec]);
    else {
	ret = safe_strdup(INVALID);
	if (ok_p)
	    *ok_p = 0;
    }
    
    DPRINT(Debug,14,(&Debug, "generate_localspec="));
    if (ret) {
	DPRINT(Debug,14,(&Debug, "%Q",ret));
    } else {
	DPRINT(Debug,14,(&Debug, "NULL"));	
    }
    if (ok_p) {
	DPRINT(Debug,14,(&Debug, "; *ok_p=%d",*ok_p));
    }        
    DPRINT(Debug,14,(&Debug, "\n"));
    
    return ret;    
}



static char * generate_domspec P_((int *ok_p));
static char * generate_domspec(ok_p)
     int *ok_p;
{
    char * ret = NULL;

    switch (message_id_domspec) {
    default:
	if (message_id_domspec >= 0 && message_id_domspec < NUM_msgiddom)
	    ret = safe_strdup(MSG_ID_domspec[message_id_domspec]);
	else
	    goto invalid_domspec;
	
	break;
    case msgiddom_whitelisted:
    case msgiddom_literal:
    case msgiddom_domain:
	if (message_id_dinfo[dname_domain]) {
	    
	    if (DOMAIN_INFO_magic != message_id_dinfo[dname_domain]->magic) 
		panic("MAILER PANIC",__FILE__,__LINE__,
		      "generate_domspec",
		      "Bad magic number (domain_info)",0);
	    
	    if (message_id_dinfo[dname_domain]->dompart) {
		ret = safe_strdup(message_id_dinfo[dname_domain]->dompart);
	    } else
		goto invalid_domspec;
	    
	} else {
	invalid_domspec:
	    ret = safe_strdup(INVALID);
	    if (ok_p)
		*ok_p = 0;
	}
	break;
    }

    DPRINT(Debug,14,(&Debug, "generate_domspec="));
    if (ret) {
	DPRINT(Debug,14,(&Debug, "%Q",ret));
    } else {
	DPRINT(Debug,14,(&Debug, "NULL"));	
    }
    if (ok_p) {
	DPRINT(Debug,14,(&Debug, "; *ok_p=%d",*ok_p));
    }        
    DPRINT(Debug,14,(&Debug, "\n"));
    
    return ret;    
}

static unsigned short min_domain_labels = 2;
static const unsigned short MAXmin_domain_labels=9;

static const char MIN_DOMAIN_LABELS_eq[]  = "min-domain-labels=";
static const char CHECK_DOMAIN_eq[]       = "check-domain=";
static const char FALLBACK_DOMAIN_eq[]    = "fallback-domain=";
static const char DEFAULT_TIMEOUT_eq[]    = "default-cache-timeout=";

static const char PARAM_SEPARATOR[] = "; ";

static char * generate_params P_((int *ok_p));
static char * generate_params(ok_p)
     int *ok_p;
{
    char * ret = NULL;

    if (fallback_domain) {
	
	size_t i;

	for (i = 0; i < fallback_domain_count; i++) {
	    if (fallback_domain[i]) {
		
		if (DOMAIN_INFO_magic != fallback_domain[i]->magic) 
		    panic("MAILER PANIC",__FILE__,__LINE__,
			  "add_fallback_domain",
			  "Bad magic number (domain_info)",0);

		if (fallback_domain[i]->dompart) {
		    const char * dompart = fallback_domain[i]->dompart;
		
		    if (ret)
			ret = strmcat(ret,PARAM_SEPARATOR);

		    if (host_name_ok(dompart)) {
			ret = strmcat(ret,FALLBACK_DOMAIN_eq);
			ret = strmcat(ret,dompart);
		    } else {
			char * msg = elm_message(FRM("%s%Q"),
						 FALLBACK_DOMAIN_eq,
						 dompart);
			
			if (ret) {
			    ret = strmcat(ret,msg);
			    free(msg);
			} else {
			    ret = msg;
			}
			msg = NULL;
		    }
		}
	    }
	}
    }

    /* Use empty fallback domain as no fallback domain */
    if (fallback_initialized && ! fallback_domain_count && !ret &&
	message_id_localspec > msgidlocal_none) {
	ret = safe_strdup(FALLBACK_DOMAIN_eq);
    }
    
    if (default_timeout_sec >= 0) {
	char * msg = elm_message(FRM("%s%ds"),
				 DEFAULT_TIMEOUT_eq,
				 default_timeout_sec);


	
	if (ret) {
	    ret = strmcat(ret,PARAM_SEPARATOR);	    
	    ret = strmcat(ret,msg);
	    free(msg);
	} else {
	    ret = msg;
	}
	msg = NULL;	
    }

    if (msg_checkdom_mode >= 0 && msg_checkdom_mode < NUM_msg_check_dom_mode) {
	if (ret) 
	    ret = strmcat(ret,PARAM_SEPARATOR);
	
	ret = strmcat(ret,CHECK_DOMAIN_eq);
	ret = strmcat(ret,MSG_CHECKDOM_mode[msg_checkdom_mode]);
    } else if (msg_checkdom_tag == msg_checkdom_mode && msg_verifydom_tag) {
	if (ret) 
	    ret = strmcat(ret,PARAM_SEPARATOR);
	
	ret = strmcat(ret,CHECK_DOMAIN_eq);
	ret = strmcat(ret,msg_verifydom_tag);
    }

    /* min_domain_labels is unsigned */
    if ( /* min_domain_labels >= 0 && */
	min_domain_labels <= MAXmin_domain_labels) {
	char * msg = elm_message(FRM("%s%u"),
				 MIN_DOMAIN_LABELS_eq,
				 min_domain_labels);			

	if (ret) {
	    ret = strmcat(ret,PARAM_SEPARATOR);
	    ret = strmcat(ret,msg);
	    free(msg);
	} else {
	    ret = msg;
	}
	msg = NULL;
    }
    
	           
    DPRINT(Debug,14,(&Debug, "generate_params="));
    if (ret) {
	DPRINT(Debug,14,(&Debug, "%Q",ret));
    } else {
	DPRINT(Debug,14,(&Debug, "NULL"));	
    }
    if (ok_p) {
	DPRINT(Debug,14,(&Debug, "; *ok_p=%d",*ok_p));
    }        
    DPRINT(Debug,14,(&Debug, "\n"));
    
    return ret;    
}    

static int message_id_set;

static void configure_message_id P_((void));
static void configure_message_id()
{

    message_id_set = 1;
    
    DPRINT(Debug,14, (&Debug, "configure_message_id: message_id_localspec=%d",
		      message_id_localspec));
    switch (message_id_localspec) {
    case msgidlocal_bad:        DPRINT(Debug,14,(&Debug," msgidlocal_bad"));        break;
    case msgidlocal_none:       DPRINT(Debug,14,(&Debug," msgidlocal_none"));       break;
    case msgidlocal_md5:        DPRINT(Debug,14,(&Debug," msgidlocal_md5"));        break;
    case msgidlocal_devurandom: DPRINT(Debug,14,(&Debug," msgidlocal_devurandom")); break;
    case msgidlocal_procuuid:   DPRINT(Debug,14,(&Debug," msgidlocal_procuuid"));   break;
    case NUM_msgidlocal:        DPRINT(Debug,14,(&Debug," NUM_msgidlocal"));        break;
    }
    DPRINT(Debug,14,(&Debug,", message_id_domspec=%d",
		     message_id_domspec));
    switch (message_id_domspec) {
    case msgiddom_whitelisted:  DPRINT(Debug,14,(&Debug," msgiddom_whitelisted"));  break;
    case msgiddom_literal:      DPRINT(Debug,14,(&Debug," msgiddom_literal"));      break;
    case msgiddom_domain:       DPRINT(Debug,14,(&Debug," msgiddom_domain"));       break;
    case msgiddom_bad:          DPRINT(Debug,14,(&Debug," msgiddom_bad"));          break;
    case msgiddom_default:      DPRINT(Debug,14,(&Debug," msgiddom_default"));      break;
    case msgiddom_hostfullname: DPRINT(Debug,14,(&Debug," msgiddom_hostfullname")); break;
    case msgiddom_maildomain:   DPRINT(Debug,14,(&Debug," msgiddom_maildomain"));   break;
    case NUM_msgiddom:          DPRINT(Debug,14,(&Debug," NUM_msgiddom"));         break;
    }
    DPRINT(Debug,14,(&Debug,"\n"));

    if (msgidlocal_bad == message_id_localspec) {
	if (local_md5_supported()) {
	    message_id_localspec = msgidlocal_md5;
	} else if (local_devurandom_supported()) {
	    message_id_localspec = msgidlocal_devurandom;
	}

	if (message_id_localspec >= 0 && message_id_localspec <  NUM_msgidlocal) {
	    DPRINT(Debug,14, (&Debug, "configure_message_id: message_id_localspec=%d %s\n",
			      message_id_localspec, MSG_ID_localspec[message_id_localspec]));
	}
    }
	    
#ifdef FALLBACKMSGIDDOM
    if (! fallback_initialized && message_id_localspec > msgidlocal_none)
	add_fallback_domain(FALLBACKMSGIDDOM);
#endif
        
}

static int parse_fallback_domain P_((const char *value));
static int parse_fallback_domain(value)
     const char *value;
{    
    /* Use empty fallback domain as no fallback domain */
    
    if (value[0]) {
	size_t l = strlen(value);
	/* Result is malloced */
	char * val = dequote_opt(value,l);
	
	if (val[0])
	    add_fallback_domain(val);
	
	free(val); val = NULL;
    }			 		     

    return 1;  /* Parsed OK */
}

static int parse_default_timeout P_((const char *value));
static int parse_default_timeout(value)
     const char *value;
{
    int int_value = 0;
    int mul = 1;
    const char * a = value;
    
    while (isdigit(*a)) {
	int v;
	if (INT_MAX / 10 < int_value) {         
	    DPRINT(Debug,8,(&Debug,
			    "parse_default_timeout: overflows: %s\n",
			    value));
	    return 0;
	}

	int_value = (int_value*10);
	v = (*a++ - '0');
	
	if (INT_MAX -v < int_value) {
	    DPRINT(Debug,8,(&Debug,
			    "parse_default_timeout: overflows: %s\n",
			    value));
	    return 0;
	}

	int_value += v;
    }

    if (a == value) {
	DPRINT(Debug,8,(&Debug,
			"parse_default_timeout: no digits: %s\n",
			value));
	return 0;
    }
    
    if (0 == strcmp(a,"") ||
	0 == strcmp(a,"s"))
	mul = 1;
    else if (0 == strcmp(a,"m"))
	mul = 60;
    else if (0 == strcmp(a,"h"))
	mul = 3600;
    else if (0 == strcmp(a,"d"))
	mul = 86400;
    else {
	DPRINT(Debug,8,(&Debug,
			"parse_default_timeout: bad unit (%s): %s\n",
			a,value));
	return 0;
    }

    if (INT_MAX / mul < int_value) {
	DPRINT(Debug,8,(&Debug,
			"parse_default_timeout overflows: %s\n",
			value));
	return 0;
    }
    
    default_timeout_sec = int_value * mul;
    return 1;
}

static int parse_min_domain_labels P_((const char *value));
static int parse_min_domain_labels(value)
     const char *value;
{
    /* Returns -1 on error, does not accept negative numbers */
    int int_value = atonum(value);

    if (int_value >= 0 && int_value <= MAXmin_domain_labels) {
	min_domain_labels = int_value;

	return 1;
    }

    return 0;
}

    
static int parse_check_domain P_((const char *value, int read_flags));
static int parse_check_domain(value,read_flags)
     const char *value;
     int read_flags;
{
    
    if (value[0]) {
	size_t l = strlen(value);
	/* Result is malloced */
	char * val = dequote_opt(value,l);
	int found = 0;
			     
	enum msg_check_dom_mode b;
	
	for (b = 0; b < NUM_msg_check_dom_mode; b++) {
	    if (0 == strcmp(val,MSG_CHECKDOM_mode[b])) {
		found = 1;
		msg_checkdom_mode = b;
	    }
	}

#ifdef USE_DLOPEN
	if (!found) {
	    switch (mailerdom_verify_lib_status(val,read_flags)) {
	    case mailerdomlib_bad:
		found = 0;
		DPRINT(Debug,14,(&Debug,
				 "parse_check_domain: bad check-domain tag %s\n",val));
		break;
	    case mailerdomlib_not_loaded:
		DPRINT(Debug,14,(&Debug,
				 "parse_check_domain: check-domain library %s not loaded\n",val));
		break;
	    case mailerdomlib_found:
		DPRINT(Debug,14,(&Debug,
				 "parse_check_domain: check-domain library %s found\n",val));
		found = 1;
		msg_checkdom_mode   = msg_checkdom_tag;
		msg_verifydom_tag = strmcpy(msg_verifydom_tag,val);
		break;
	    case mailerdomlib_delayed:
		DPRINT(Debug,14,(&Debug,
				 "parse_check_domain: check-domain tag %s not checked\n",val));
		found = 1;
		msg_checkdom_mode   = msg_checkdom_tag;
		msg_verifydom_tag = strmcpy(msg_verifydom_tag,val);
		break;			     			     
	    }	    
	}
#endif
	
	free(val); val = NULL;
	
	if (!found)
	    return 0;

	
    }
    
    return 1;
}




static const char COMMENT_SEP[] = " # ";

int message_id_func(value,enter,lineno,filename,read_flags)
     char **value;
     int enter;
     int lineno;
     const char *filename;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    int ok = 1;

    if (filename || enter || read_flags || *value) {
	DPRINT(Debug,14, (&Debug, "message_id_func: %s",
			  enter ? "(enter) " : ""));
	if (filename) {
	    DPRINT(Debug,14, (&Debug, "filename=%Q, lineno=%d",			  
			      filename,lineno));
	}
	if (enter || read_flags) {
	    DPRINT(Debug,14, (&Debug, ", read_flags=%d",
			      read_flags));
	}
	if (*value) {
	    DPRINT(Debug,14, (&Debug, ", *value=%Q",
			      *value));
	}
	DPRINT(Debug,14, (&Debug, "\n"));
    }
	
    if (enter) {

	char * temp = safe_strdup(*value);
	const char * this_value = *value;
	char * ptr;
	char * domspec    = NULL;
	char * localspec  = temp;
	char * params = NULL;
	int   reset_saved_on_failure = 0; /* Reset saved_domspec, saved_localspec, saved_params 
					     on failure */
	 
	for (ptr = temp; *ptr; ptr++) {

	 parsenext:
	     switch (*ptr) {

	     case '\0':
		 goto doneparse;

	     case ' ':
		 if (ptr == localspec)
		     localspec = ptr+1;
		 else if (0 == strncmp(ptr,COMMENT_SEP,(sizeof COMMENT_SEP)-1)) {
		     /* Truncate on comment */

		     *ptr ='\0';
		     goto doneparse;
		     
		 } else if (!params) {
		     int len = rfc822_toklen(ptr);

		     if ('\0' == ptr[len] || ';' == ptr[len]) {
			 /* Truncate value */

			 *ptr ='\0';
			 ptr += len;
			 goto parsenext;
		     } else if (0 == strncmp(ptr+len,"# ",2)) {
			 /* Truncate on comment */
			 
			 *ptr ='\0';
			 goto doneparse;
		     }
		     		     
		     goto failchar;

		 } else if (ptr == params)
		     params = ptr+1;

		 break;

	     case '\t':
		 if (ptr == localspec)
		     localspec = ptr+1;
		 else if (!params) {
		     int len = rfc822_toklen(ptr);

		     if ('\0' == ptr[len] || ';' == ptr[len]) {
			 /* Truncate value */

			 *ptr ='\0';
			 ptr += len;
			 goto parsenext;
		     } else if (0 == strncmp(ptr+len,"# ",2)) {
			 /* Truncate on comment */
			 
			 *ptr ='\0';
			 goto doneparse;		    
		     } else {
			 unsigned char c = *ptr;
			 
			 if (filename) {
			     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadXOptFile,
					       "Bad message-id option %Q character x%02x on line %d on file %s"),
				       this_value,c,lineno,filename);
			     
			 } else {
			     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadXOp,
					       "Bad message-id option %Q character x%02x."),
				       this_value,c);
			 }
			 
			 ok = 0;
			 goto failparse;
		     }
		 } else if (ptr == params+1)
		     params = ptr+1;
		 
		 break;
		 
	     case '@':
		 
		 if (params)
		     break;

		 if (!ptr[1] || domspec) 
		     goto failchar;

		 if (ptr == localspec)
		     localspec = NULL;
		 
		 *ptr ='\0';
		 domspec = ptr+1;
		 break;
	     case '[':

		 if ((domspec && ptr == domspec+1) || params) {
		     int len = rfc822_toklen(ptr);

		     if (len < 2 || ']' != ptr[len-1]) {

			 if (filename) {
			     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadOptFile,
					       "Bad message-id option %Q token \"%c\" on line %d on file %s"),
				       this_value,*ptr,lineno,filename);
			     
			 } else {
			     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadOp,
					       "Bad message-id option %Q token \"%c\"."),
				       this_value,*ptr);
			 }
			 
			 ok = 0;
			 goto failparse;			 
		     }

		     
		     goto skip_on_parms;
		 }

		 goto failchar;

	     case '"':
		 if (params) {
		     int len = rfc822_toklen(ptr);

		     if (len < 2 || '"' != ptr[len-1]) {
			 unsigned char c = *ptr;

			 if (filename) {
			     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadXOptFile,
					       "Bad message-id option %Q character x%02x on line %d on file %s"),
				       this_value,c,lineno,filename);
			     
			 } else {
			     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadXOp,
					       "Bad message-id option %Q character x%02x."),
				       this_value,c);
			 }
			 
			 ok = 0;
			 goto failparse;			 
		     }
		     
		     goto skip_on_parms;

		 }
		     
		 goto failchar;

	     case ';':
		 if (params)
		     break;

		 if (ptr == localspec)
		     localspec = NULL;
		 if (ptr == domspec)
		     domspec = NULL;
		 
		 *ptr ='\0';
		 params = ptr+1;
		 break;

	     case '.':
		 if (domspec && ptr > domspec && '.' != ptr[-1] &&
		     '\0' != ptr[1] && ' ' != ptr[1] && '\t' != ptr[1] &&
		     ';' != ptr[1])
		     break;

		 goto failchar;

	     case '(': {
		 int len = rfc822_toklen(ptr);
		 
		 if (strncpy(ptr,INVALID,len)) {
		     
		     /* Silent failure */
		     
		     ptr += len;
		     goto parsenext;		     
		 }

		 if (len < 2 || ')' != ptr[len-1]) {
		     
		     if (filename) {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadOptFile,
					   "Bad message-id option %Q token \"%c\" on line %d on file %s"),
				   this_value,*ptr,lineno,filename);
			 
		     } else {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadOp,
					   "Bad message-id option %Q token \"%c\"."),
				   this_value,*ptr);
		     }

		     
		     ok = 0;
		     goto failparse;			 
		 }		 
	     }
		 /* FALLTHRU */
	     failchar:
	     case ')':
	     case '=':
	     case ']':
	     case '\\': 
		 
		 if (params) 
		     goto skip_on_parms;
			 
		 if (filename) {
		     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadOptFile,
				       "Bad message-id option %Q token \"%c\" on line %d on file %s"),
			       this_value,*ptr,lineno,filename);
		     
		 } else {
		     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadOp,
				       "Bad message-id option %Q token \"%c\"."),
			       this_value,*ptr);
		 }
		 
		 ok = 0;
		 goto failparse;

	     skip_on_parms:  {
		     int len = rfc822_toklen(ptr);
		 
		     while (len > 0) {
			 if (!isascii(*ptr) || !isprint(*ptr)) {
			     unsigned char c = *ptr;
			     
			     if (filename) {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadXOptFile,
						   "Bad message-id option %Q character x%02x on line %d on file %s"),
				       this_value,c,lineno,filename);
				 
			     } else {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadXOp,
						   "Bad message-id option %Q character x%02x."),
					   this_value,c);
			     }
			     
			     ok = 0;
			     goto failparse;		     
			 }
			 
			 ptr++;
			 len--;
		     }
		     
		     goto parsenext;		 
		 }
		 
	     default:

		 if (!isascii(*ptr) || !isprint(*ptr)) {
		     unsigned char c = *ptr;
		     
		     if (filename) {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadXOptFile,
					   "Bad message-id option %Q character x%02x on line %d on file %s"),
				   this_value,c,lineno,filename);
			 
		     } else {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadXOp,
					   "Bad message-id option %Q character x%02x."),
				   this_value,c);
		     }

		     ok = 0;
		     goto failparse;		     

		 }
		 break;
		 
	     }	     	    
	}

     doneparse:
	if (localspec) {
	    if (!localspec[0] || 0 == strcmp(localspec,INVALID))	       
		localspec = NULL;
	    else {
		DPRINT(Debug,15, (&Debug, "message_id_func: localspec=%s\n",localspec));		
	    }
	}
	if (domspec) {
	    if (!domspec[0] || 0 == strcmp(domspec,INVALID))		     
		domspec = NULL;
	    else {
		DPRINT(Debug,15, (&Debug, "message_id_func: domspec=%s\n",domspec));
	    }
	}
	if (params) {
	    if (!params[0])
		params = NULL;
	    else {
		DPRINT(Debug,15, (&Debug, "message_id_func: params=%s\n",params));
	    }
	}

	if (! localspec && !domspec && ! params && !message_id_set) {
	    DPRINT(Debug,15, (&Debug, "message_id_func: %Q, Null value, configuring\n",
			      this_value));
	    configure_message_id();
	}
	
	    
	if (localspec) {
	     enum msg_id_localspec localspec_val = msgidlocal_bad;
	     enum msg_id_localspec i;

	     int is_valid = 0;
	     
	     for (i = 0; i < NUM_msgidlocal; i++) {
		 if (MSG_ID_localspec[i] &&
		     0 == strcmp(localspec,MSG_ID_localspec[i])) {
		     localspec_val = i;
		     break;
		 }
	     }

	     switch (localspec_val) {
	     case msgidlocal_bad:
	     case NUM_msgidlocal:

		 if (filename) {
		     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadLocalOptFile,
				       "Invalid local part keyword %Q on message-id option %Q on line %d on file %s"),
			       localspec,this_value,lineno,filename);

		 } else {
		     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadLocalOpt,
				       "Invalid local part keyword %Q on message-id option %Q."),
			       localspec,this_value);
		 }
		 ok = 0;
		 break;
		 
	     case msgidlocal_none:
		 message_id_localspec = localspec_val;

		 if (saved_domspec || saved_params) {
		     DPRINT(Debug,14, (&Debug, "message_id_func: %Q gived. Clearing saved ",
				       localspec));

		     if (saved_domspec) {
			 DPRINT(Debug,14, (&Debug, "domain %Q", saved_domspec));
			 free(saved_domspec);
			 saved_domspec = NULL;
		     }

		     if (saved_params) {
			 DPRINT(Debug,14, (&Debug, "params %Q", saved_params));
			 free(saved_params);
			 saved_params = NULL;
		     }
		     
		     DPRINT(Debug,14, (&Debug, "\n"));
		 }
		 
		 break;
		 
		 break;
		 if (0) {
		 case msgidlocal_md5:
		     is_valid = local_md5_supported();
		 } else if (0) {		     
		 case msgidlocal_devurandom:
		     is_valid = local_devurandom_supported();
		 } else if (0) {			    
		 case msgidlocal_procuuid:
		     is_valid = local_procuuid_supported();
		 }
		 
		 
		 DPRINT(Debug,14,
			(&Debug, 
			 "message_id_func: %Q, local %s found as %d%s\n",
			 this_value,localspec,localspec_val,
			 is_valid ? ", supported" : ", fails"));

		 if (is_valid) {
		     message_id_localspec = localspec_val;
		 } else {
		     if (filename) {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDUnsLocalOptFile,
					   "Unsupported local part keyword %Q on message-id option %Q on line %d on file %s"),
				   localspec,this_value,lineno,filename);
			 
		     } else {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDUnsLocalOpt,
					   "Unsupported local part keyword %Q on message-id option %Q."),
				   localspec,this_value);
		     }
		     
		     ok = 0;
		 }
		 break;
	     }

	     saved_localspec = strmcpy(saved_localspec,localspec);
	     reset_saved_on_failure = 1;
	}
	 
	if (domspec) {

	     if ( msgidlocal_none == message_id_localspec ) {

		 if (filename) {
		     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDLocalPartNoDomFile,
				       "Local part keyword %Q does not need domain on message-id option %Q line %d on file %s"),
			       localspec,this_value,lineno,filename);

		 } else {
		     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDLocalPartNoDom,
				       "Local part keyword %Q does not need domain on message-id option %Q."),
			       localspec,this_value);
		 }
		 ok = 0;       
	     }
	     
	     if ('[' == domspec[0]) {
		 int len = rfc822_toklen(domspec);
		 
		 if (len < 2 || domspec[len] || domspec[len-1] != ']') {
		     DPRINT(Debug,14,
			    (&Debug, 
			     "message_id_func: %Q, domain %s failed to parse as literal\n",
			     this_value,domspec));

		     if (filename) {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadLitoptionFile,
					   "Invalid literal %Q on message-id option %Q on line %d on file %s"),
				   domspec,this_value,lineno,filename);

		     } else {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadLitoption,
					   "Invalid literal %Q on message-id option %Q."),
				   domspec,this_value);
		     }

		     ok =  0;
		} else {

		     DPRINT(Debug,14,
			    (&Debug, 
			     "message_id_func: %Q, domain %s parsed as literal\n",
			     this_value,domspec));
		     
		    message_id_domspec  = msgiddom_literal;

		    set_msgid_dompart(dname_domain,domspec,cds_disabled_ok,NULL);
		}
				 
	     } else {
		 int len = rfc822_toklen(domspec);

		 if ('.' == domspec[len]) {
		     const char * whitelisted =
			 is_whitelisted_msgid_domain(domspec);

		     /* Is domain name */
		     
		     if (whitelisted) {

			 DPRINT(Debug,14,
				(&Debug, 
				 "message_id_func: %Q, domain %s whitelisted on %s\n",
				 this_value,whitelisted,domspec));
			 
			 message_id_domspec = msgiddom_whitelisted;
			 set_msgid_dompart(dname_domain,domspec,cds_whitelisted,NULL);

		     } else {

			 const char * reserved_name =
			     is_special_use_domain(domspec);
			
			 if (reserved_name) {

			     DPRINT(Debug,14,
				    (&Debug, 
				     "message_id_func: %Q, domain %s reserved on %s\n",
				     this_value,reserved_name,domspec));

			     if (filename) {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDResDomOptFile,
						   "Reserved domain %s on %Q on message-id option %Q on file %s"),
					   reserved_name,domspec,this_value,
					   lineno,filename);

				 
			     } else {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDResDomOpt,
						   "Reserved domain %s on %Q on message-id option %Q."),
					   reserved_name,domspec,this_value);
			     }
			     
			     message_id_domspec = msgiddom_bad;
			     
			     set_msgid_dompart(dname_domain,NULL,cds_reserved,NULL);

			     ok = 0;
			 } else  {
			     message_id_domspec = msgiddom_domain;
			     set_msgid_dompart(dname_domain,domspec,cds_none,NULL);
			 }
		     }
			 
		 } else {
		     enum msg_id_domspec domspec_val = msgiddom_bad;
		     enum msg_id_domspec i;

		     for (i = 0; i < NUM_msgiddom; i++) {
			 if (0 == strcmp(domspec,MSG_ID_domspec[i])) {
			     domspec_val = i;
			     break;
			 }					
		     }

		     switch (domspec_val) {
		     case NUM_msgiddom:
		     case msgiddom_whitelisted:
		     case msgiddom_literal:
		     case msgiddom_domain:
		     case msgiddom_bad:

			 if (filename) {
			     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadDomOptFile,
					       "Invalid domain part keyword %Q on message-id option %Q on line %d on file %s"),
				       domspec,this_value,lineno,filename);

			 } else {
			     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadDomOpt,
					       "Invalid domain part keyword %Q on message-id option %Q."),
				       localspec,this_value);
			 }
			 ok = 0;
			 
			 break;

		     case msgiddom_default:
			 set_msgid_dompart(dname_maildomain,useful_mailname(),cds_none,NULL);
			 /* FALLTHROUG */
			 
		     case msgiddom_hostfullname:
			 DPRINT(Debug,14,
				(&Debug, 
				 "message_id_func: %Q, domain %s found as %d\n",
				 this_value,domspec,domspec_val));
			 
			 message_id_domspec = domspec_val;
			 set_msgid_dompart(dname_hostfullname,hostfullname,cds_none,NULL);
			 break;
			 
		     case msgiddom_maildomain:

			 DPRINT(Debug,14,
				(&Debug, 
				 "message_id_func: %Q, domain %s found as %d\n",
				 this_value,domspec,domspec_val));
			 
			 message_id_domspec = domspec_val;
			 set_msgid_dompart(dname_maildomain,useful_mailname(),cds_none,NULL);
			 break;
		     }
		 }
	     }

	     saved_domspec = strmcpy(saved_domspec,domspec);
	     reset_saved_on_failure = 1;
	} 
	 	 
	if (params) {
	     char * WALK;	 
	     char * f;
	     int clear_fallback = 1;
	     int default_timout_seen = 0;
	     int check_domain_seen = 0;
	     int min_domain_labels_seen = 0;

	     if ( msgidlocal_none == message_id_localspec ) {

		 if (filename) {
		     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDLocalPartNoParmFile,
				       "Local part keyword %Q does not need parameters on message-id option %Q line %d on file %s"),
			       localspec,this_value,lineno,filename);

		 } else {
		     lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDLocalPartNoParam,
				       "Local part keyword %Q does not need parameters on message-id option %Q."),
			       localspec,this_value);
		 }
		 ok = 0;       
	     }
	     
	     saved_params = strmcpy(saved_params,params);
	     reset_saved_on_failure = 1;
	     
	     for (f = mime_parse_content_opts(params, &WALK); f;
		  f = mime_parse_content_opts(NULL, &WALK)) {

		 char * x = strchr(f,'=');

		 if (x) {
		     size_t len = (x - f) +1;
		     
		     if (0 == strncmp(f,FALLBACK_DOMAIN_eq,len)) {
			 char * a = f+len;

			 if (clear_fallback) {
			     free_fallback_domains();
			     clear_fallback = 0;
			 }
			 
			 if (parse_fallback_domain(a)) {
			     DPRINT(Debug,14,(&Debug,"message_id_func: Parsed %s\n",
					      f));
			 } else {
			     goto bad_param;
			 }

		     } 	else if (0 == strncmp(f,DEFAULT_TIMEOUT_eq,len)) {
			 char * a = f+len;
			 
			 if (default_timout_seen) {
			     if (filename) {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDDubParamFile,
						   "Duplicate paramater %Q on message-id option %Q on line %d on file %s"),
					   f,this_value,lineno,filename);
				 
			     } else {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDDubParam,
						   "Duplicate paramater %Q on message-id option %Q."),
					   f,this_value);
			     }
			     
			     ok = 0;
			 }

			 if (parse_default_timeout(a)) {
			     DPRINT(Debug,14,(&Debug,"message_id_func: Parsed %s\n",
					      f));
			     default_timout_seen = 1;
			 } else {
			     goto bad_param;
			 }

		     }  else if (0 == strncmp(f,CHECK_DOMAIN_eq,len)) {
			 char * a = f+len;

			 if (check_domain_seen) {
			     if (filename) {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDDubParamFile,
						   "Duplicate paramater %Q on message-id option %Q on line %d on file %s"),
					   f,this_value,lineno,filename);
				 
			     } else {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDDubParam,
						   "Duplicate paramater %Q on message-id option %Q."),
					   f,this_value);
			     }
			     
			     ok = 0;
			 }

			 if (parse_check_domain(a,read_flags)) {
			     DPRINT(Debug,14,(&Debug,"message_id_func: Parsed %s\n",
					      f));
			     check_domain_seen = 1;
			 } else {
			     goto bad_param;
			 }			 			
			 
		     } 	else if (0 == strncmp(f,MIN_DOMAIN_LABELS_eq,len)) {
			 char * a = f+len;
			 
			 if (min_domain_labels_seen) {
			     if (filename) {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDDubParamFile,
						   "Duplicate paramater %Q on message-id option %Q on line %d on file %s"),
					   f,this_value,lineno,filename);
				 
			     } else {
				 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDDubParam,
						   "Duplicate paramater %Q on message-id option %Q."),
					   f,this_value);
			     }
			     
			     ok = 0;
			 }

			 
			 if (parse_min_domain_labels(a)) {
			     DPRINT(Debug,14,(&Debug,"message_id_func: Parsed %s\n",
					      f));
			     min_domain_labels_seen = 1;
			 } else {
			     goto bad_param;
			 }			 	
			 			 
		     } else
			 goto bad_param;
		     			 
		 } else {
		 bad_param:

		     if (filename) {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadParamFile,
					   "Invalid parameter %Q on message-id option %Q on line %d on file %s"),
				   f,this_value,lineno,filename);

		     } else {
			 lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDBadParam,
					   "Invalid parameter %Q on message-id option %Q."),
				   f,this_value);
		     }
		     ok = 0;		     
		 }
	     }
	}

#ifdef FALLBACKMSGIDDOM
	if (! fallback_initialized && message_id_localspec > msgidlocal_none)
	    add_fallback_domain(FALLBACKMSGIDDOM);
#endif

    failparse:
	free(temp); temp = NULL;
	
	if (reset_saved_on_failure && !ok) {
	    DPRINT(Debug,14, (&Debug, "message_id_func: resetting unparsed saved value: "));

	    if (saved_localspec) {
		DPRINT(Debug,14, (&Debug, "%s", saved_localspec));
		free(saved_localspec);
		saved_localspec = NULL;
	    }
	    
	    if (saved_domspec) {
		DPRINT(Debug,14, (&Debug, "@%s", saved_domspec));
		free(saved_domspec);
		saved_domspec = NULL;
	    }

	    if (saved_params) {
		DPRINT(Debug,14, (&Debug, "; %s", saved_params));
		free(saved_params);
		saved_params = NULL;
	    }
	    
	    DPRINT(Debug,14, (&Debug, "\n"));
	}

	if (ok) {
	    message_id_set = 1;
	    
	    DPRINT(Debug,14, (&Debug, "message_id_func: unparsed saved value: "));

	    if (saved_localspec) {
		DPRINT(Debug,14, (&Debug, "%s", saved_localspec));
	    }
	    
	    if (saved_domspec) {
		DPRINT(Debug,14, (&Debug, "@%s", saved_domspec));
	    }

	    if (saved_params) {
		DPRINT(Debug,14, (&Debug, "; %s", saved_params));
	    }
	    
	    DPRINT(Debug,14, (&Debug, "\n"));
	}
		
     } else {  /* not enter -- return message-id setting on *value */
	 char * result = NULL;

	 char * generated_localspec = NULL;
	 char * generated_domspec = NULL;
	 char * generated_params  = NULL;
	 char * comment = NULL;

	 if (! message_id_set) {
	     configure_message_id();
	 }

	 if (saved_localspec)
	     result = strmcat(result,saved_localspec);

	 else if (generated_localspec ||
		  (generated_localspec = generate_localspec(&ok))) {
	     result = strmcat(result,generated_localspec);
	 } else 
	     result = strmcat(result,INVALID);

	 if (saved_domspec) {
	     result = strmcat(result,"@");
	     result = strmcat(result,saved_domspec);

	 } else if (generated_domspec ||
		    (message_id_localspec != msgidlocal_none &&
		     (generated_domspec = generate_domspec(&ok)))) {
	     result = strmcat(result,"@");
	     result = strmcat(result,generated_domspec);
	 } else if (message_id_localspec != msgidlocal_none) {
	     result = strmcat(result,"@");
	     result = strmcat(result,INVALID);
	 }

	 if (saved_params) {
	     result = strmcat(result,PARAM_SEPARATOR);
	     result = strmcat(result,saved_params);
	 } else if (generated_params ||
		    (message_id_localspec != msgidlocal_none &&
		     (generated_params = generate_params(&ok)))) {
	     result = strmcat(result,PARAM_SEPARATOR);

	     result = strmcat(result,generated_params);
	 }
	 
	 /* Generate expanded value as comment */
	 
	 if (generated_localspec ||
	    (generated_localspec = generate_localspec(&ok))) {
	      comment = strmcat(comment,generated_localspec);
          } else 
	     comment = strmcat(comment,INVALID);

	 if (generated_domspec ||
	     (message_id_localspec != msgidlocal_none &&
	      (generated_domspec = generate_domspec(&ok)))) {
	      comment = strmcat(comment,"@");
	      comment = strmcat(comment,generated_domspec);
	 } else if (message_id_localspec != msgidlocal_none) {
	     comment = strmcat(comment,"@");
	     comment = strmcat(comment,INVALID);
         }

	 if (generated_params ||
	     (message_id_localspec != msgidlocal_none &&
	      (generated_params = generate_params(&ok)))) {
	     comment = strmcat(comment,PARAM_SEPARATOR);
	     
	     comment = strmcat(comment,generated_params);
	 }
	 
	 if (result && comment) {
	       if (0 != strcmp(result,comment)) {
		 result = strmcat(result,COMMENT_SEP);
		 result = strmcat(result,comment);
	     }
         }

	 /* static return buffer */
	 
	 if (message_id_return_buffer) {
	     free(message_id_return_buffer);
	 }
	 message_id_return_buffer = result;
	 *value = message_id_return_buffer;
	 result = NULL;
	     
	 if (generated_localspec) {
	     free(generated_localspec);
	     generated_localspec = NULL;
	 }
	 if (generated_domspec) {
	     free(generated_domspec);
	     generated_domspec = NULL;
	 }
	 if (generated_params) {
	     free(generated_params);
	     generated_params = NULL;
	 }
	 if (comment) {
	     free(comment);
	     comment = NULL;
	 }
	 
     }

     DPRINT(Debug,14, (&Debug, "message_id_func=%d%s%s",
		       ok,
		       ok    ? " (succeed)" : " FAILED",
		       enter ? " (enter) " : " (return value)"));
     if (*value) {
	 DPRINT(Debug,14, (&Debug, "; *value=%Q",*value));
     }
     
     DPRINT(Debug,14, (&Debug, "\n"));
     
    return ok;
}

/* Does NOT include all letters -- not \0 terminated */
static const char SYMBOLS[] = {    
    'A','B','C','D','E','F','G','H',    'J','K','L','M','N','P','Q','R','S','T','U','V','W','X','Y','Z',
    'a','b','c','d','e','f','g','h','i','j','k',    'm','n','p',    'r','s','t','u','v','w','x','y','z',
    '2','3','4','5','6','7','8','9'
};

/* Generate left side of message-id for 
   msgidlocal_md5, and
   msgidlocal_devurandom

   result is malloced
*/
static char * octectvalue_to_left P_((const unsigned char data[],
				      const size_t        data_len));
static char * octectvalue_to_left(data,data_len)
     const unsigned char data[];
     const size_t        data_len;
{
    /* This produces different result on 64-bit ad 32-bit enviroment 

       And different result if HAVE_INT128 is defined and __int128
       is available.
     */


#ifdef HAVE_INT128
    typedef unsigned __int128  data_chunk;
#else
    typedef unsigned long  data_chunk;
#endif
    
    static const size_t base = sizeof SYMBOLS;
    static unsigned int bits = 0;
    static int check = 0;   /* Calculate bits from base once */
    
    char * ret = NULL;
    size_t ret_len = 0;
    size_t malloc_len = 1 + data_len / sizeof (data_chunk); /* \0 and separating . characters */
    size_t expand = 0;
    size_t i;
    char * temp = NULL;
    size_t tempsize;
    
    if (!check) {
	size_t x;
	
	DPRINT(Debug,14,(&Debug, "octectvalue_to_left: base=%zu\n",base)); 
	
	for (i = 0; i < base; i++) {
	    if (!SYMBOLS[i] || !isascii(SYMBOLS[i]) || !isprint(SYMBOLS[i])) {
		DPRINT(Debug,1,(&Debug, "octectvalue_to_left: #%zu bad symbol %d\n",
				i,SYMBOLS[i]));

		panic("MAILER PANIC",__FILE__,__LINE__,
		      "octectvalue_to_left",
		      "Bad symbol",0);
	    }

	}

	i = base;
	bits = 0;
	
	while (i > 0) {
	    bits++;
	    i >>= 1;
	}

	x = 1 << bits;

	DPRINT(Debug,14,(&Debug, "octectvalue_to_left: bits=%u => %zu",
			 bits,x));

	if (x > base)  {   /* Round down bits */
	    DPRINT(Debug,14,(&Debug, " > base %zu",base));

	    bits--;

	    x = 1 << bits;
	    DPRINT(Debug,14,(&Debug, ", reducing bits=%u (=> %zu)",
			     bits,x));
	} else {
	    
	    DPRINT(Debug,14,(&Debug, " OK -- base %zu",base));
	}
	DPRINT(Debug,14,(&Debug, "\n"));
	    	
	check = 1;
    }

    /* Is this enough ? */
    
    tempsize = (sizeof (data_chunk) * 8) / bits +1;

    temp = safe_malloc(tempsize);
    
    /* Does not need to be correct */
    
    expand = (data_len * 8) / bits;
    
    DPRINT(Debug,14,(&Debug, "octectvalue_to_left: data_len=%zu => expand=%zu",
		     data_len,expand));
    malloc_len += expand;
    DPRINT(Debug,14,(&Debug, ", malloc_len=%zu, bits=%u => tempsize=%zu\n",
		     malloc_len,bits,tempsize));
    
    ret = safe_malloc(malloc_len);

#define ADD(x) { \
        if (malloc_len < ret_len+1) { \
	    size_t old = malloc_len; \
	    malloc_len = ret_len+8; \
	    DPRINT(Debug,14,(&Debug, \
			     "octectvalue_to_left: malloc_len=%zu => %zu\n", \
			     old,malloc_len));	\
	    ret = safe_realloc(ret,malloc_len); \
	} \
	ret[ret_len++] = (x); \
    } while(0)
    
    for (i = 0; i < data_len; ) {
	size_t j;
	size_t y;
	data_chunk r;
	
	data_chunk unit = 0;

	/* Collect unsigned long or data_chunk worth of octets (if available) */
	
	for (j = 0; j < sizeof unit && i < data_len; j++) {

	    unit <<= 8;
	    unit |= data[i++];
	}

	/* Add . between values */
	if (ret_len > 0)
	    ADD('.');
	
	y = tempsize;

	do {
	    data_chunk t;
	    
	    t = (unit % base );
	    r = (unit / base );
	    unit = r;

	    y--;
	    temp[y] = SYMBOLS[t];

	} while (r && y > 0);

	if (r) {
	    DPRINT(Debug,14,(&Debug, "octectvalue_to_left: temp buffer too small\n"));
	    ADD('~');
	}

	while (y < tempsize) {
	    ADD(temp[y]);
	    y++;
	}	
    }

    free(temp);

    if (malloc_len != ret_len+1) {  /* Adjuts for final \0 */
	size_t old = malloc_len;    
	malloc_len = ret_len+1;	 
	DPRINT(Debug,14,(&Debug, "octectvalue_to_left: malloc_len=%zu => %zu\n",old,malloc_len)); 
	ret = safe_realloc(ret,malloc_len); 
    }
    ret[ret_len] = '\0';

#undef ADD
    
    DPRINT(Debug,14,(&Debug, "octectvalue_to_left=%Q, ret_len=%zu, malloc_len=%zu\n",ret,ret_len,malloc_len));
        
    return ret;
}

static int match_to_domain P_((enum domain_names d, const char *old_domain_part));
static int match_to_domain(d,old_domain_part)
     enum domain_names d;
     const char *old_domain_part;
{
    int ourdom = 0;
    
    if (d < 0 || d >= NUM_dname)
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "match_to_domain",
	      "Bad domain name index",0);

    if (message_id_dinfo[d]) {
	if (DOMAIN_INFO_magic != message_id_dinfo[d]->magic) 
	    panic("MAILER PANIC",__FILE__,__LINE__,
		  "match_to_domain",
		  "Bad magic number (domain_info)",0);
	
	if (message_id_dinfo[d]->dompart &&
	    message_id_dinfo[d]->dompart[0] &&
	    0 == strcmp(message_id_dinfo[d]->dompart,
			old_domain_part)) {

	    DPRINT(Debug,14,(&Debug,
			     "match_to_domain: old message-id domain %Q matches to generated domain part (message_id #%d %s)\n",
			     old_domain_part,
			     d,domain_names_d[d]));
	    ourdom = 1;
	}
	
	if (message_id_dinfo[d]->rewrite &&
	    message_id_dinfo[d]->rewrite[0] &&
	    0 == strcmp(message_id_dinfo[d]->rewrite,
			old_domain_part)) {
	    DPRINT(Debug,14,(&Debug,
			     "match_to_domain: old message-id domain %Q matches to rewritten domain name on name check (message_id #%d %s)\n",
			     old_domain_part,
			     d,domain_names_d[d]));
	    ourdom = 1;
	}
    }
    
    return ourdom;
}

/* Result is malloced */

static char *get_msg_domain P_((enum domain_names d));
static char *get_msg_domain(d)
     enum domain_names d;
{
   if (d < 0 || d >= NUM_dname)
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "get_msg_domain",
	      "Bad domain name index",0);

    if (message_id_dinfo[d]) {
	if (DOMAIN_INFO_magic != message_id_dinfo[d]->magic) 
	    panic("MAILER PANIC",__FILE__,__LINE__,
		  "get_msg_domain",
		  "Bad magic number (domain_info)",0);


	if (message_id_dinfo[d]->dompart &&
	    message_id_dinfo[d]->dompart[0]) {

	    DPRINT(Debug,14,(&Debug,
			     "get_msg_domain: message_id #%d %s is %Q\n",
			     d,domain_names_d[d],
			     message_id_dinfo[d]->dompart));

	    return safe_strdup(message_id_dinfo[d]->dompart);
	}
    }

    return NULL;
}


/* Count numbers of domain labels, domain literals are 
   assumed to be handled separately and not accepted here

   Does not count empty domain labels (these also flag error)
*/

static int count_domain_labels(const char *dom,int * ok_p);
static int count_domain_labels(dom,ok_p)
     const char *dom;
     int * ok_p;
{
    int count = 0;
    const char * a;
    int len = 0;
    
    int start_of_label = 1;

    /* Note: \.  is not considered as domain label separator */

    if (ok_p)
	*ok_p = (dom[0] != '\0');
    
    for (a = dom; *a; a += len) {
	len = rfc822_toklen(a);   /* Returns 2 for \. */

	if (1 == len && '.' == *a)  {
	    if (start_of_label) {
		DPRINT(Debug,14,(&Debug,
				 "count_domain_labels: %Q -- several dots (..) or dot (.) on beginning of domain\n",
				 dom));

		if (ok_p)
		    *ok_p = 0;
	    } else {
		start_of_label = 1;
	    }

	} else if (start_of_label) {
	    start_of_label = 0;
	    count++;
	}

	if ('.' != *a && '\\' != *a && rfc822_special(*a)) {

	    /* !!!! %.*Q can not used here as format argument */
	    DPRINT(Debug,14,(&Debug,
			     "count_domain_labels: %Q -- rfc822 special (%.*s) seen\n",
			     dom,len,a));

	    if (ok_p)
		*ok_p = 0;
	}

	if (whitespace(*a)) {
	    /* !!!! %.*Q can not used here as format argument */
	    DPRINT(Debug,14,(&Debug,
			     "count_domain_labels: %Q -- whitespace (%.*s) seen\n",
			     dom,len,a));
	    
	    if (ok_p)
		*ok_p = 0;
	    
	}
    }

    if (start_of_label) {
	DPRINT(Debug,14,(&Debug,
			 "count_domain_labels: %Q -- dot (.) on end of domain or empty string given\n",
			 dom));
	
	if (ok_p)
	    *ok_p = 0;
    }

    DPRINT(Debug,14,(&Debug,
		     "count_domain_labels=%d labels",
		     count));
    if (ok_p) {
	DPRINT(Debug,14,(&Debug,
			 ", *ok_p=%d%s",
			 *ok_p,
			 *ok_p ? " OK" : " FAILED"));
    }
    
    DPRINT(Debug,14,(&Debug,", domain %Q\n",dom));
        
    return count;
}

/* Return rewritten string if domain check succeed, also update  message_id_dinfo 
   resuld is malloced
*/

static char * check_msg_domain0 P_((struct domain_info              * I,
				    const char                      * dom,
				    struct cancel_data             ** cancel_p,
				    const struct schedule_timelimit * now,
				    const struct schedule_timelimit * default_valid_until  /* for other data */
				    ));

static char * check_msg_domain0(I,dom0,cancel_p,now,default_valid_until)
     struct domain_info              * I;
     const char                      * dom0;
     struct cancel_data             ** cancel_p;
     const struct schedule_timelimit * now;
     const struct schedule_timelimit * default_valid_until  /* for other data */;
{
    char * ret = NULL;
    
    int busy;

    /* check_msg_domain0() is called
       with I->dompart == dom0
    */
    char * backup_dom = safe_strdup(dom0);

    if (DOMAIN_INFO_magic != I->magic) 
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "check_msg_domain0",
	      "Bad magic number (domain_info)",0);

    busy = I->busy;
    I->busy = 1;
    

    DPRINT(Debug,5,(&Debug, 
		    "check_msg_domain0: [%p] %Q was already busy !!\n",
		    I,I->dompart ? I->dompart : "(not set)"));


    if (dom0 == I->dompart)  {
	DPRINT(Debug,14,(&Debug, 
			 "check_msg_domain0: [%p] domain is same as %Q argument\n",
			 I,dom0));
			        
    }


    if ('[' == backup_dom[0]) {
	int len = rfc822_toklen(backup_dom);
	
	if (len < 2 || backup_dom[len] || backup_dom[len-1] != ']') {
	    DPRINT(Debug,14,(&Debug, 
			     "check_msg_domain0: domain %Q failed to parse as literal\n",
			     backup_dom));
	    I->domstat  = check_dname_no_routine;
	    I->domcheck = cds_bad_syntax;
	    
	    if (I->dompart)
		free(I->dompart);
	    I->dompart = NULL;
	    if (I->rewrite)
		free(I->rewrite);
	    I->rewrite = NULL;

	    I->valid_until = NO_schedule_timelimit;
	    
	    ret = NULL;
	    goto done;
	} else {
	    DPRINT(Debug,14,(&Debug, 
			     "check_msg_domain0: domain %Q parsed as literal\n",
			     backup_dom));
	    I->domstat  = check_dname_no_routine;
	    I->domcheck = cds_disabled_ok;
	    I->dompart = strmcpy(I->dompart,backup_dom);
	    if (I->rewrite)
		free(I->rewrite);
	    I->rewrite = NULL;
	    
	    I->valid_until = NO_schedule_timelimit;
	    
	    ret = safe_strdup(backup_dom);
	    goto done;
	}
	
    } else {
	const char * whitelisted =
	    is_whitelisted_msgid_domain(backup_dom);

	const char * reserved_name = NULL;
	const char * dummy_name = NULL;
	
	if (whitelisted) {
	    
	    DPRINT(Debug,14,(&Debug, 
			     "check_msg_domain0: domain %Q whitelisted on %Q\n",
			     whitelisted,backup_dom));
	    
	    I->domstat  = check_dname_no_routine;
	    I->domcheck = cds_whitelisted;
	    I->dompart = strmcpy(I->dompart,backup_dom);
	    if (I->rewrite)
		free(I->rewrite);
	    I->rewrite = NULL;

	    I->valid_until = NO_schedule_timelimit;
	    
	    ret = safe_strdup(backup_dom);
	    goto done;
	    
	} else if ((reserved_name = is_special_use_domain(backup_dom))) {

	    DPRINT(Debug,14,(&Debug, 
			     "check_msg_domain0: reserved domain %Q usupported on %Q\n",
			     reserved_name,backup_dom));
	    
	    I->domstat  = check_dname_no_routine;
	    I->domcheck = cds_reserved;
	    
	    I->dompart = strmcpy(I->dompart,backup_dom);
	    if (I->rewrite)
		free(I->rewrite);
	    I->rewrite = NULL;
	    
	    I->valid_until = NO_schedule_timelimit;
	    
	    goto done;

	} else if ((dummy_name = matches_dummy_domain(backup_dom))) {

	    DPRINT(Debug,14,(&Debug, 
			     "check_msg_domain0: dummy domain %Q rejected on %Q\n",
			     dummy_name,backup_dom));
	    
	    I->domstat  = check_dname_no_routine;
	    I->domcheck = cds_fail;
	    
	    if (I->rewrite)
		free(I->rewrite);
	    I->rewrite = NULL;
	    
	    I->valid_until = NO_schedule_timelimit;
	    
	    goto done;
	    
	} else if (I->dompart &&
		   0 == strcmp(I->dompart,backup_dom)) {

	    if (schedule_have_timelimit(& (I->valid_until))) {
	    
		if (schedule_valid_until_ok(& (I->valid_until),now)) {
	    
		    char * A = schedule_timeout_string(& (I->valid_until));
		    
		    DPRINT(Debug,14,(&Debug, 
				     "check_msg_domain0: Entry %Q valid (not expired)",
				     I->dompart));
		    if (A) {
			DPRINT(Debug,14,(&Debug, ", valid until %s",A));
			free(A); A = NULL;
		    }
		} else
		    goto done;
	    } else {
		DPRINT(Debug,14,(&Debug, 
				 "check_msg_domain0: Entry %Q valid (no timelimit)",
				 I->dompart));
	    }
		    
	    DPRINT(Debug,14,(&Debug, ", domcheck = %d",I->domcheck));
	    switch (I->domcheck) {
	    case cds_reserved:
		DPRINT(Debug,14,(&Debug, " cds_reserved\n"));
		goto done;
	    case cds_no_routine:
		DPRINT(Debug,14,(&Debug, " cds_no_routine"));
		break;		
	    case cds_fail:
		DPRINT(Debug,14,(&Debug, " cds_fail"));
		break;
	    case cds_none:
		DPRINT(Debug,14,(&Debug, " cds_none"));
		break;
	    case cds_ok:
		DPRINT(Debug,14,(&Debug, " cds_ok"));
		if (I->rewrite) {
		    DPRINT(Debug,14,(&Debug, ", rewrite %Q",I->rewrite));
		    ret = safe_strdup(I->rewrite);
		} else
		    ret = safe_strdup(I->dompart);
		DPRINT(Debug,14,(&Debug, "\n"));
		goto done;
	    case cds_bad_syntax:
		DPRINT(Debug,14,(&Debug, " cds_bad_syntax\n"));
		goto done;
	    case cds_whitelisted:
		DPRINT(Debug,14,(&Debug, " cds_whitelisted\n"));
		ret = safe_strdup(I->dompart);
		goto done;
	    case cds_disabled_ok:
		DPRINT(Debug,14,(&Debug, " cds_disabled_ok\n"));
		ret = safe_strdup(I->dompart);
		goto done;		
	    }

	    DPRINT(Debug,14,(&Debug, ", domstat = %d",I->domstat));
	    switch (I->domstat) {
	    case check_dname_no_routine:
		DPRINT(Debug,14,(&Debug, " check_dname_no_routine"));
		break;
	    case check_dname_query_blacklisted:
		DPRINT(Debug,14,(&Debug, " check_dname_query_blacklisted\n"));
		goto done;
	    case check_dname_failure:
		DPRINT(Debug,14,(&Debug, " check_dname_failure"));
		/* Assume temporal failure */
		break;
	    case check_dname_not_found:
		DPRINT(Debug,14,(&Debug, " check_dname_not_found\n"));
		goto done;
	    case check_dname_have_name:
		DPRINT(Debug,14,(&Debug, " check_dname_have_name"));
		/* Should not happen */
		if (I->rewrite) {
		    DPRINT(Debug,14,(&Debug, ", rewrite %Q",I->rewrite));
		    ret = safe_strdup(I->rewrite);
		} else
		    ret = safe_strdup(I->dompart);
		DPRINT(Debug,14,(&Debug, "\n"));
		goto done;
	    case check_dname_bad_syntax:
		DPRINT(Debug,14,(&Debug, " check_dname_bad_syntax\n"));
		goto done;
	    }
	    DPRINT(Debug,14,(&Debug, "\n"));
	    
	}

	if (!ret) {	    
	    I->domstat  = check_dname_no_routine;
	    I->domcheck = cds_none;

	    I->dompart = strmcpy(I->dompart,backup_dom);
	    if (I->rewrite)
		free(I->rewrite);
	    I->rewrite = NULL;
	    
	    I->valid_until = NO_schedule_timelimit;

	    if (min_domain_labels > 0) {
		int ok = 1;
		
		int num_labels = count_domain_labels(backup_dom,&ok);

		DPRINT(Debug,14,(&Debug,
				 "check_msg_domain0: domain %Q have %d domain label(s), %s",
				 backup_dom,num_labels,ok ? "syntax OK" : "parse failed"));
		
		if (!ok) {
		    DPRINT(Debug,14,(&Debug, "\n"));
		    I->domcheck = cds_bad_syntax;
		    goto done;
		} else if (num_labels < min_domain_labels) {
		    DPRINT(Debug,14,(&Debug,
				     ", skipping DNS check, %d domain labels required\n",
				     min_domain_labels));
		    I->domcheck = cds_fail;
		    goto done;
		}

		DPRINT(Debug,14,(&Debug, "\n"));
	    }


	    
	    switch (msg_checkdom_mode) {
		
	    case msg_checkdom_tag:
#ifdef USE_DLOPEN
		I->domstat  = shared_check_dname(I->dompart,&(I->rewrite),
						 msg_verifydom_tag,
						 now,
						 default_valid_until,
						 &(I->valid_until),
						 NULL /* No error message? */,
						 cancel_p);
#else
		DPRINT(Debug,14,(&Debug, "check_msg_domain0: no shared routines available\n"));
		I->domstat  = check_dname_no_routine;
#endif
		break;
	    case msg_checkdom_none:
		DPRINT(Debug,14,(&Debug, "check_msg_domain0: Domain verify disabled.\n"));
		break;
	    case msg_checkdom_shared:
#ifdef USE_DLOPEN
		I->domstat  = shared_check_dname(I->dompart,&(I->rewrite),
						 NULL /* No tag */,
						 now,
						 default_valid_until,
						 &(I->valid_until),
						 NULL /* No error message? */,
						 cancel_p);
#else
		DPRINT(Debug,14,(&Debug, "check_msg_domain0: no shared routines available\n"));
		I->domstat  = check_dname_no_routine;
#endif
		break;
	    case NUM_msg_check_dom_mode:
		break;
	    }
		
	    DPRINT(Debug,14,(&Debug, "check_msg_domain0: %Q: domstat = %d",
			     I->dompart,I->domstat));
	    switch (I->domstat) {
	    case check_dname_no_routine:
		DPRINT(Debug,14,(&Debug, " check_dname_no_routine"));
		I->domcheck = cds_no_routine;

		if (msg_checkdom_mode != msg_checkdom_none) {
		    DPRINT(Debug,14,(&Debug, " (disabling domain verify)"));
		    msg_checkdom_mode = msg_checkdom_none;
		}		
		break;
	    case check_dname_query_blacklisted:
		DPRINT(Debug,14,(&Debug, " check_dname_query_blacklisted"));
		I->domcheck = cds_reserved;
		break;
	    case check_dname_failure:
		DPRINT(Debug,14,(&Debug, " check_dname_failure"));
		I->domcheck = cds_fail;
		break;
	    case check_dname_not_found:
		DPRINT(Debug,14,(&Debug, " check_dname_not_found"));
		I->domcheck = cds_fail;
		break;
	    case check_dname_have_name:
		DPRINT(Debug,14,(&Debug, " check_dname_have_name"));
		I->domcheck = cds_ok;
		if (I->rewrite) {
		    DPRINT(Debug,14,(&Debug, ", rewrite %Q",I->rewrite));
		    ret = safe_strdup(I->rewrite);
		} else
		    ret = safe_strdup(I->dompart);
		break;
	    case check_dname_bad_syntax:
		DPRINT(Debug,14,(&Debug, " check_dname_bad_syntax"));
		I->domcheck = cds_bad_syntax;
		break;
	    }

	    
	    if (schedule_have_timelimit(& (I->valid_until))) {
		char * A = schedule_timeout_string(& (I->valid_until));

		if (A) {
		    DPRINT(Debug,14,(&Debug, ", valid until %s",A));
		    free(A); A = NULL;
		}
		DPRINT(Debug,14,(&Debug, "\n"));
		
		update_next_domver_cleanup(& (I->valid_until));
	    } else {
		DPRINT(Debug,14,(&Debug, ", no timelimit\n"));
	    }

	    if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		DPRINT(Debug,14,(&Debug, "check_msg_domain0: DNS lookup canceled\n"));
	    }	    	    
	}
    }

 done:

    if (! I->busy) {
	DPRINT(Debug,5,(&Debug, 
			"check_msg_domain0: [%p] %Q busy was cleared!!\n",
			I,I->dompart ? I->dompart : "(not set)"));
    }

    I->busy = busy;

    if (I->busy) {
	DPRINT(Debug,5,(&Debug, 
			"check_msg_domain0: [%p] %Q still busy on end!!\n",
			I,I->dompart ? I->dompart : "(not set)"));
    }


    DPRINT(Debug,14,(&Debug, 
		     "check_msg_domain0="));
    if (ret) {
	DPRINT(Debug,14,(&Debug,"%Q",ret));
    } else {
	DPRINT(Debug,14,(&Debug,"NULL (failure)"));
    }
    DPRINT(Debug,14,(&Debug,"; dom=%Q\n",backup_dom));

    free(backup_dom);
    backup_dom = NULL;

	    
    return ret;
}
				    
static char * check_msg_domain P_((enum domain_names d,const char *dom,
				   struct cancel_data            ** cancel_p,
				   const struct schedule_timelimit * now,				   
				   const struct schedule_timelimit * default_valid_until  /* for other data */));
static char * check_msg_domain(d,dom,cancel_p,now,default_valid_until)
     enum domain_names d;
     const char *dom;
     struct cancel_data            ** cancel_p;
     const struct schedule_timelimit * now;
     const struct schedule_timelimit * default_valid_until  /* for other data */;
{
    if (d < 0 || d >= NUM_dname)
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "check_msg_domain",
	      "Bad domain name index",0);

    if (! message_id_dinfo[d])
	message_id_dinfo[d] = new_domain_info(dom,cds_none, default_valid_until);
    
    return check_msg_domain0(message_id_dinfo[d],dom,cancel_p,
			     now,
			     default_valid_until);
}


static void free_domlist P_((char ***domlist,
			    size_t  *domlist_len));
static void free_domlist(domlist,domlist_len)
     char ***domlist;
     size_t  *domlist_len;
{
    if (*domlist) {
	size_t x;

	for (x =  0; x < *domlist_len; x++) {
	    if ((*domlist)[x]) {
		free((*domlist)[x]);
		(*domlist)[x] = NULL;
	    }
	}	
	free(*domlist);
	*domlist = NULL;
    }
    *domlist_len = 0;
}

/* Result is malloced */
static char * left_part_from_devurandom P_((enum message_id_status *ret_p));
static char * left_part_from_devurandom(ret_p)
     enum message_id_status *ret_p;
{
    
    char * left_part = NULL;
    int retry = 0;
    int last_retry;
    
    
    do {
	last_retry = retry;
	
	DPRINT(Debug,14, (&Debug,"left_part_from_devurandom: devurandom_used=%d ",devurandom_used));
	switch (devurandom_used) {
	case buffer_bad:         DPRINT(Debug,14, (&Debug," buffer_bad")); break;
	case buffer_initialized: DPRINT(Debug,14, (&Debug," buffer_initialized devurandom len %d",
						   devurandom_len));
	    if (devurandom_len > 0) {
		size_t x;
		DPRINT(Debug,14,(&Debug," ["));
		
		for (x = 0; x < devurandom_len && x < sizeof procuuid_buffer; x++) {
		    DPRINT(Debug,14,(&Debug,"%02x",devurandom_buffer[x]));
		}
		DPRINT(Debug,14,(&Debug,"]"));
	    }
	    
	    break;
	case buffer_used:        DPRINT(Debug,14, (&Debug," buffer_used")); break;
	}
	if (retry) {
	    DPRINT(Debug,14, (&Debug," retry=%d",retry));
	}
	DPRINT(Debug,14, (&Debug,"\n"));
	
	switch (devurandom_used) {
	    
	case buffer_initialized:
	    
	    /* Have data */
	    if (devurandom_len == sizeof devurandom_buffer) {
		
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: have full %d bytes on devurandom buffer\n",
				  devurandom_len));
		left_part = octectvalue_to_left(devurandom_buffer,devurandom_len);
		
		devurandom_used = buffer_used;
		
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: consumed devurandom buffer\n"));
		goto done;
	    } else if (devurandom_len < sizeof devurandom_buffer) {
		int need = (sizeof devurandom_buffer) - devurandom_len;
		int len;
		
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: have data %d bytes on devurandom buffer, need %d bytes more\n",
				  devurandom_len,need));
		
		if (-1 ==  devurandom_fd &&
		    ! open_devurandom()) {
		    DPRINT(Debug,14, (&Debug,
				      "left_part_from_devurandom: devurandom is not open.\n"));
		    
		    goto done;
		}
		
		len = read(devurandom_fd,
			   devurandom_buffer + devurandom_len,
			   need);
		
		if (len > 0) {
		    DPRINT(Debug,14, (&Debug,
				      "left_part_from_devurandom: Got %d bytes.\n",len));
		    
		    devurandom_len += len;
		    retry++;
		    
		} else if (-1 == len) {
		    int err UNUSED_VAROK = errno;
		    
		    DPRINT(Debug,14,(&Debug,
				     "left_part_from_devurandom: read %s (fd=%d): %s (errno=%d)\n",
				     DEVURANDOM,devurandom_fd,
				     strerror(err),err));
		    goto done;
		    
		} else if (0 == len) {
		    DPRINT(Debug,14, (&Debug,
				      "left_part_from_devurandom: Got EOF.\n"));
		    retry++;
		    close_devurandom();
		} else {
		    goto done;
		}
		
	    } else {
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: have data %d bytes on devurandom buffer, but buffer is %zu bytes\n",
				  devurandom_len,sizeof devurandom_buffer));
		
		devurandom_used = buffer_bad;
		goto done;		
	    }
	    break;
	    
	case buffer_used: {
	    static unsigned char newrandom[DEVURANDOM_LEN];
	    int len;
	    
	    /* Buffer data is used, read new data and check that it differ
	       from used data */
	    
	    if (-1 ==  devurandom_fd &&
		! open_devurandom()) {
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: devurandom is not open.\n"));
		
		goto done;
	    }
	    
	    len = read(devurandom_fd,newrandom,sizeof newrandom);
	    
	    if (len > 0) {
		int i;
		int differ = 0;
		
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: Got %d bytes.\n",len));
		
		for (i = 0;
		     i < len && i < sizeof newrandom &&
			 i < devurandom_len && i < sizeof devurandom_buffer;
		     i++) {
		    if (newrandom[i] != devurandom_buffer[i])
			differ++;
		}
		
		if (differ) {
		    DPRINT(Debug,14, (&Debug,
				      "left_part_from_devurandom: %d bytes differ from previous devurandom context\n",
				      differ));
		    
		    if (len < sizeof devurandom_buffer) {
			memcpy(devurandom_buffer,newrandom,len);
			devurandom_len = len;
		    } else {
			memcpy(devurandom_buffer,newrandom,sizeof devurandom_buffer);
			devurandom_len = sizeof devurandom_buffer;
		    }
		    
		    devurandom_used = buffer_initialized;
		    retry++;
		} else {
		    DPRINT(Debug,14, (&Debug,
				      "left_part_from_devurandom: Context does not differ from preivious devurandom_buffer\n"));
		    goto done;
		}
		
	    } else if(-1 == len) {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,14,(&Debug,
				 "left_part_from_devurandom: read %s (fd=%d): %s (errno=%d)\n",
				 DEVURANDOM,devurandom_fd,
				 strerror(err),err));
		goto done;
		
	    } else if (0 == len) {
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: Got EOF.\n"));
		retry++;
		close_devurandom();
	    } else {
		goto done;
	    }
	    
	}
	    break;
	    
	case buffer_bad: {
	    int len;
	    
	    if (-1 ==  devurandom_fd &&
		! open_devurandom()) {
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: devurandom is not open.\n"));
		
		goto done;
	    }
	    
	    len = read(devurandom_fd, devurandom_buffer, sizeof devurandom_buffer);
	    
	    if (len > 0) {
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: Got %d bytes.\n",len));
		
		devurandom_len = len;
		devurandom_used = buffer_initialized;
		retry++;
	    } else if(-1 == len) {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,14,(&Debug,
				 "left_part_from_devurandom: read %s (fd=%d): %s (errno=%d)\n",
				 DEVURANDOM,devurandom_fd,
				 strerror(err),err));
		goto done;
			    
	    } else if (0 == len) {
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_devurandom: Got EOF.\n"));
		retry++;
		close_devurandom();
	    } else {
		goto done;
	    }				      
	}
	    break;
	}
		
	if (retry > 10) {
	    DPRINT(Debug,14,(&Debug,
			     "left_part_from_devurandom: retry=%d -- failure\n",
			     retry));
	    break;
	}
	
    } while (retry > last_retry && !left_part && msid_stat_none == *ret_p);
    
done:
    DPRINT(Debug,14, (&Debug,"left_part_from_devurandom="));
    if (left_part) {
	DPRINT(Debug,14, (&Debug,"%Q",left_part));
    } else {
	DPRINT(Debug,14, (&Debug,"NULL"));
    }
    DPRINT(Debug,14, (&Debug,"; *ret_p=%d",*ret_p));
    switch (*ret_p) {
    case msid_stat_need_hash:    DPRINT(Debug,14, (&Debug," msid_stat_need_hash"));    break;
    case msid_stat_need_restart: DPRINT(Debug,14, (&Debug," msid_stat_need_restart")); break;
    case msid_stat_none:         DPRINT(Debug,14, (&Debug," msid_stat_none"));         break;
    case msid_stat_updated:      DPRINT(Debug,14, (&Debug," msid_stat_updated"));      break;
    case msid_stat_no_changes:   DPRINT(Debug,14, (&Debug," msid_stat_no_changes"));   break;
    }    
    DPRINT(Debug,14, (&Debug,"\n"));
    
    return left_part;
}

/* Result is malloced */
static char * left_part_from_procuuid P_((enum message_id_status *ret_p));
static char * left_part_from_procuuid(ret_p)
     enum message_id_status *ret_p;
{
    char * left_part = NULL;
    int retry = 0;
    int last_retry;
    
    do {
	last_retry = retry;

	DPRINT(Debug,14, (&Debug,"left_part_from_procuuid: procuuid_used=%d ",procuuid_used));
	switch (procuuid_used) {
	case buffer_bad:         DPRINT(Debug,14, (&Debug," buffer_bad")); break;
	case buffer_initialized: DPRINT(Debug,14, (&Debug," buffer_initialized procuuid len %d",
						   procuuid_len));
	    if (procuuid_len > 0) {
		int len = procuuid_len;
		if (len > sizeof procuuid_buffer)
		    len = sizeof procuuid_buffer;
		/* Do not debug newline on end of buffer */
		if (len > 0 && '\n' == procuuid_buffer[len-1])
		    len--;
		
		DPRINT(Debug,14, (&Debug," ["));
		DEBUG_PRINT_BUFFER(Debug,14, len,procuuid_buffer);
		DPRINT(Debug,14,(&Debug,"]"));
	    }
	    break;	
	case buffer_used:        DPRINT(Debug,14, (&Debug," buffer_used")); break;
	}
	if (retry) {
	    DPRINT(Debug,14, (&Debug," retry=%d",retry));
	}
	DPRINT(Debug,14, (&Debug,"\n"));

	switch (procuuid_used) {
	case buffer_initialized:

	    /* Have data */
	    if (procuuid_len == sizeof procuuid_buffer) {
		DPRINT(Debug,14,(&Debug,
				 "left_part_from_procuuid: have full %d characters on procuuid buffer\n",
				 procuuid_len));

		if (valid_procuuid(procuuid_buffer,procuuid_len)) {
		    int len = procuuid_len;

		    /* Do not copy newline on end of buffer */
		    if (len > 0 && '\n' == procuuid_buffer[len-1])
			len--;
		    
		    procuuid_used = buffer_used;
		    left_part = safe_malloc(len+1);
		    if (len)
			memcpy(left_part,procuuid_buffer,len);
		    left_part[len] = '\0';
		    goto done;
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "left_part_from_procuuid: buffer not valid"));

		    if (-1 != procuuid_fd ) {
			DPRINT(Debug,14,(&Debug,", closing procuuid\n"));
			close_procuuid();
		    } else {
			DPRINT(Debug,14,(&Debug,"\n"));
		    }
		    procuuid_used = buffer_bad;
		    retry++;
		}

	    } else if (procuuid_len < sizeof procuuid_buffer) {
		int need = (sizeof procuuid_buffer) - procuuid_len;
		int len;
		
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_procuuid: have %d characters on procuuid buffer, need %d characters more\n",
				  procuuid_len,need));

		if (-1 == procuuid_fd &&
		    ! open_procuuid()) {
		    
		    DPRINT(Debug,14,(&Debug,
				     "left_part_from_procuuid: procuuid is not open.\n"));
		    goto done;
		}

		len = read(procuuid_fd,
			   procuuid_buffer+procuuid_len,
			   need);

		if (len > 0) {
		    DPRINT(Debug,14, (&Debug,
				      "left_part_from_procuuid: Got %d characters.\n",len));

		    procuuid_len += len;
		    retry++;

		} else if (-1 == len) {
		    int err UNUSED_VAROK = errno;
		    
		    DPRINT(Debug,14,(&Debug,
				     "left_part_from_procuuid: read %s (fd=%d): %s (errno=%d)\n",
				     PROCUUID,procuuid_fd,strerror(err),err));
		    
		    goto done;
		} else if (0 == len) {
		    DPRINT(Debug,14,(&Debug,
				     "left_part_from_procuuid: Got EOF.\n"));
		    retry++;
		    close_procuuid();

		} else {
		    goto done;
		}
		    
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "left_part_from_procuuid: have %d characters on procuuid buffer, but buffer is %zu bytes.",
				 procuuid_len,sizeof procuuid_buffer));
		procuuid_used = buffer_bad;

		goto done;
	    }
	    break;

	case buffer_used: {
	    static unsigned char newuuid[PROCUUID_LEN];
	    int len;

	    /* Buffer data is used, read new data and check that it differ
		from used data */

	    if (-1 == procuuid_fd &&
		! open_procuuid()) {

		DPRINT(Debug,14,(&Debug,
				 "left_part_from_procuuid: procuuid is not open.\n"));
		goto done;
	    }

	    len = read(procuuid_fd,newuuid, sizeof newuuid);

	    if (len > 0) {
		int i;
		int differ = 0;
		
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_procuuid: Got %d characters.\n",len));
		
		for (i = 0;
		     i < len && i < sizeof newuuid &&
			 i < procuuid_len && i < sizeof procuuid_buffer;
		     i++) {
		    if (newuuid[i] != procuuid_buffer[i])
			differ++;		    
		}

		if (differ) {
		    DPRINT(Debug,14, (&Debug,
				      "left_part_from_procuuid: %d characters differ from previous procuuid context\n",
				      differ));

		    if (len < sizeof procuuid_buffer) {
			memcpy(procuuid_buffer,newuuid,len);
			procuuid_len = len;
		    } else {
			memcpy(procuuid_buffer,newuuid,sizeof procuuid_buffer);
			procuuid_len = sizeof procuuid_buffer;
		    }
		    procuuid_used = buffer_initialized;
		    retry++;

		} else {
		    DPRINT(Debug,14, (&Debug,
				      "left_part_from_procuuid: Context does not differ from preivious procuuid_buffer\n"));
		    goto done;
		}
	    } else if (-1 == len) {
		int err UNUSED_VAROK = errno;
	
		DPRINT(Debug,14,(&Debug,
				 "left_part_from_procuuid: read %s (fd=%d): %s (errno=%d)\n",
				 PROCUUID,procuuid_fd,strerror(err),err));

		goto done;
	    } else if (0 == len) {
		DPRINT(Debug,14,(&Debug,
				 "left_part_from_procuuid: Got EOF (expected after read).\n"));
		retry++;
		close_procuuid();
	    } else {
		goto done;
	    }

	}
	    break;
	    
	case buffer_bad: {
	    int len;

	    if (-1 == procuuid_fd &&
		! open_procuuid()) {

		DPRINT(Debug,14,(&Debug,
				 "left_part_from_procuuid: procuuid is not open.\n"));
		goto done;
	    }

	    len = read(procuuid_fd,procuuid_buffer, sizeof procuuid_buffer);

	    if (len > 0) {
		DPRINT(Debug,14, (&Debug,
				  "left_part_from_procuuid: Got %d characters.\n",len));
		
		procuuid_len = len;
		procuuid_used = buffer_initialized;
		retry++;
	    } else if (-1 == len) {
		int err UNUSED_VAROK = errno;
	
		DPRINT(Debug,14,(&Debug,
				 "left_part_from_procuuid: read %s (fd=%d): %s (errno=%d)\n",
				 PROCUUID,procuuid_fd,strerror(err),err));

		goto done;
	    } else if (0 == len) {
		DPRINT(Debug,14,(&Debug,
				 "left_part_from_procuuid: Got EOF (expected after read).\n"));
		retry++;
		close_procuuid();
	    } else {
		goto done;
	    }
	}
	    break;

	    
	}

	if (retry > 10) {
	    DPRINT(Debug,14,(&Debug,
			     "left_part_from_procuuid: retry=%d -- failure\n",
			     retry));
	    break;
	}
	
    } while (retry > last_retry && !left_part && msid_stat_none == *ret_p);

 done:
    DPRINT(Debug,14, (&Debug,"left_part_from_procuuid="));
    if (left_part) {
	DPRINT(Debug,14, (&Debug,"%Q",left_part));
    } else {
	DPRINT(Debug,14, (&Debug,"NULL"));
    }
    DPRINT(Debug,14, (&Debug,"; *ret_p=%d",*ret_p));
    switch (*ret_p) {
    case msid_stat_need_hash:    DPRINT(Debug,14, (&Debug," msid_stat_need_hash"));    break;
    case msid_stat_need_restart: DPRINT(Debug,14, (&Debug," msid_stat_need_restart")); break;
    case msid_stat_none:         DPRINT(Debug,14, (&Debug," msid_stat_none"));         break;
    case msid_stat_updated:      DPRINT(Debug,14, (&Debug," msid_stat_updated"));      break;
    case msid_stat_no_changes:   DPRINT(Debug,14, (&Debug," msid_stat_no_changes"));   break;
    }    
    DPRINT(Debug,14, (&Debug,"\n"));
    
    return left_part;
}
    
enum message_id_status update_message_id(msg_id_res,old_hash,current_hash,info,need_hash,
					 cancel_p)
     struct message_id      ** msg_id_res;
     struct digest_proc     * old_hash;
     struct digest_proc     * current_hash;
     struct mailer_info     * info;
     const struct digest_proc_type ** need_hash;
     struct cancel_data            ** cancel_p; /* *cancel_p is set if DNS lookup is cancelable */
{
    enum message_id_status ret = msid_stat_none;

    struct schedule_timelimit now                  = NO_schedule_timelimit;
    struct schedule_timelimit default_valid_until  = NO_schedule_timelimit;
    char * A;
    
    
    schedule_set_current_time(&now);
    A = schedule_timeout_string(&now);
    if (A) {
	DPRINT(Debug,14,(&Debug,
			 "update_message_id: now %s\n",
			 A));
	free(A); A = NULL;
    }
    
    if (default_timeout_sec >= 0) {
	DPRINT(Debug,14,(&Debug,
			 "update_message_id: default_timeout_sec=%d\n",
			 default_timeout_sec));

	if (! schedule_have_timelimit(&now)) {

	    DPRINT(Debug,14,(&Debug,
			    "update_message_id:  No time\n"));
	    	    
	} else if (schedule_set_next_timeout0(&default_valid_until,&now,
					      default_timeout_sec,
					      /* Use time_MAX -1 on failure
						 May fail at Tue Jan 19 03:14:06 2038 */
					      se_use_before_time_MAX)) {
	    A = schedule_timeout_string(&default_valid_until);
	    if (A) {
		DPRINT(Debug,14,(&Debug,
				 "update_message_id: default valid until %s\n",
				 A));
		free(A); A = NULL;
	    }
	}
    }

    if (! message_id_set) {
	configure_message_id();
    }
    
    DPRINT(Debug,14,(&Debug,
		     "update_message_id: message_id_localspec=%d",
		     message_id_localspec));
    switch (message_id_localspec) {
    case msgidlocal_bad:        DPRINT(Debug,14,(&Debug," msgidlocal_bad"));        break;
    case msgidlocal_none:       DPRINT(Debug,14,(&Debug," msgidlocal_none"));       break;
    case msgidlocal_md5:        DPRINT(Debug,14,(&Debug," msgidlocal_md5"));        break;
    case msgidlocal_devurandom: DPRINT(Debug,14,(&Debug," msgidlocal_devurandom")); break;
    case msgidlocal_procuuid:   DPRINT(Debug,14,(&Debug," msgidlocal_procuuid"));   break;
    case NUM_msgidlocal:        DPRINT(Debug,14,(&Debug," NUM_msgidlocal"));        break;
    }
    DPRINT(Debug,14,(&Debug,", message_id_domspec=%d",
		     message_id_domspec));
    switch (message_id_domspec) {
    case msgiddom_whitelisted:  DPRINT(Debug,14,(&Debug," msgiddom_whitelisted"));  break;
    case msgiddom_literal:      DPRINT(Debug,14,(&Debug," msgiddom_literal"));      break;
    case msgiddom_domain:       DPRINT(Debug,14,(&Debug," msgiddom_domain"));       break;
    case msgiddom_bad:          DPRINT(Debug,14,(&Debug," msgiddom_bad"));          break;
    case msgiddom_default:      DPRINT(Debug,14,(&Debug," msgiddom_default"));      break;
    case msgiddom_hostfullname: DPRINT(Debug,14,(&Debug," msgiddom_hostfullname")); break;
    case msgiddom_maildomain:   DPRINT(Debug,14,(&Debug," msgiddom_maildomain"));   break;
    case NUM_msgiddom:          DPRINT(Debug,14,(&Debug," NUM_msgiddom"));         break;
    }
    DPRINT(Debug,14,(&Debug,"\n"));
    
    /* This is called if message-id is not check, or when hash changes
       or mailer is reconnected (which may imply domain name change if
       msgiddom_default is used 
    */

    if (* msg_id_res) {
	const char * old_left_part   = message_id_left(* msg_id_res);
	const char * old_domain_part = message_id_domain(* msg_id_res);

	if (old_left_part   && old_left_part[0]
	    &&
	    old_domain_part && old_domain_part[0]) {
	    int ourdom = 0;
	    int ourleft = 0;

	    /* Check domain name ------------------------------ */

	    switch (message_id_domspec) {
	    case msgiddom_whitelisted:
	    case msgiddom_literal:
	    case msgiddom_domain:
		if (match_to_domain(dname_domain,old_domain_part))
		    ourdom = 1;
		break;
	    case msgiddom_default:
		if (match_to_domain(dname_hostfullname,old_domain_part))
		    ourdom = 1;
		if (match_to_domain(dname_mailer,old_domain_part))
		    ourdom = 1;
		if (match_to_domain(dname_maildomain,old_domain_part))
		    ourdom = 1;
		break;
	    case msgiddom_hostfullname:
		if (match_to_domain(dname_hostfullname,old_domain_part))
		    ourdom = 1;
		break;
	    case msgiddom_maildomain:
		if (match_to_domain(dname_hostfullname,old_domain_part))
		    ourdom = 1;
		break;		
	    case msgiddom_bad:
	    case NUM_msgiddom:
		DPRINT(Debug,14,(&Debug,
				 "update_message_id: Message-id not generated here\n"));
		
		ret = msid_stat_no_changes;
		goto nochange;		
	    }
	    
	    if (fallback_domain) {
		size_t i;
		
		for (i = 0; i < fallback_domain_count; i++) {
		    if (fallback_domain[i]) {
			
			if (DOMAIN_INFO_magic != fallback_domain[i]->magic) 
			    panic("MAILER PANIC",__FILE__,__LINE__,
				  "update_message_id",
				  "Bad magic number (domain_info)",0);

			if (fallback_domain[i]->dompart &&
			    0 == strcmp(fallback_domain[i]->dompart,
					old_domain_part)) {
			    
			    DPRINT(Debug,14,(&Debug,
					     "update_message_id: old message-id domain %s matches to fallback domain\n",
					     old_domain_part));
			    ourdom = 1;
			}
		    }
		}
	    }
	    
	    /* Check local part --------------------------------- */

	    switch (message_id_localspec) {

	    case msgidlocal_md5:
		if (old_hash) {
		    size_t old_hash_size = len_digest_proc(old_hash);
		    unsigned char * data = safe_malloc(old_hash_size);
		    size_t data_len = result_digest_proc(old_hash,data,old_hash_size);

		    if (data_len) {
			size_t x;
			char * old_hash_left;
			
			DPRINT(Debug,14,(&Debug,
					 "update_message_id: old_hash len %zu [",
					 data_len));
			for (x = 0; x < data_len; x++) {
			    DPRINT(Debug,14,(&Debug,"%02x",data[x]));
			}
			DPRINT(Debug,14,(&Debug,"]\n"));

			old_hash_left = octectvalue_to_left(data,data_len);
			if (old_hash_left) {
			    DPRINT(Debug,14,(&Debug,
					     "update_message_id: left side from old hash is %s",
					     old_hash_left));
			    
			    if (0 == strcmp(old_left_part,old_hash_left)) {
				DPRINT(Debug,14,(&Debug,
						 ", old message-id left side %s matches to old hash",
						 old_left_part));
				ourleft = 1;
			    }
			    DPRINT(Debug,14,(&Debug,"\n"));
			    
			    free(old_hash_left);
			}
		    }
		    		    
		    free(data);
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "update_message_id: old_hash not given\n"));
		    
		}
		break;

	    case msgidlocal_devurandom:
		if (devurandom_len > 0 && buffer_used == devurandom_used) {
		    int x;
		    char * old_urandom_left;

		    DPRINT(Debug,14,(&Debug,
				     "update_message_id: devuramdom len %d [",
				     devurandom_len));
		    
		    for (x = 0; x < devurandom_len; x++) {
			    DPRINT(Debug,14,(&Debug,"%02x",devurandom_buffer[x]));
		    }
		    DPRINT(Debug,14,(&Debug,"]\n"));
		    
		    old_urandom_left = octectvalue_to_left(devurandom_buffer,devurandom_len);
		    if (old_urandom_left) {
			DPRINT(Debug,14,(&Debug,
					 "update_message_id: left side from old devurandom is %s",
					 old_urandom_left));
			
			if (0 == strcmp(old_left_part,old_urandom_left)) {
			    DPRINT(Debug,14,(&Debug,
					     ", old message-id left side %s matches to old devurandom data",
					     old_left_part));
			    ourleft = 1;
			}
			DPRINT(Debug,14,(&Debug,"\n"));
			
			free(old_urandom_left);
		    }

		} else {
		    DPRINT(Debug,14,(&Debug,
				     "update_message_id: no used data on devurandom_buffer\n"));
		}
		break;

	    case msgidlocal_procuuid:
		if (procuuid_len > 0 && buffer_used == procuuid_used) {
		    int len = procuuid_len;

		    /* Remove \n if still on buffer */
		    if ('\n' == procuuid_buffer[procuuid_len-1])
			len = procuuid_len-1;
		    
		    DPRINT(Debug,14,(&Debug,
				     "update_message_id: procuuid len %d (without newline) [%.*s]",
				     len,len,procuuid_buffer));

		    if (0 == strncmp(old_left_part,us2s(procuuid_buffer),len) &&
			'\0' == old_left_part[len]) {
			DPRINT(Debug,14,(&Debug,
					 ", old message-id left side %s matches to old procuid",
					 old_left_part));
			ourleft = 1;
		    }
		    DPRINT(Debug,14,(&Debug,"\n"));
		
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "update_message_id: no used data on procuuid_buffer\n"));
		}
		break;

	    case msgidlocal_bad:
	    case msgidlocal_none:
	    case NUM_msgidlocal:
		DPRINT(Debug,14,(&Debug,
				 "update_message_id: Message-id not generated here\n"));
		
		ret = msid_stat_no_changes;
		goto nochange;		
	    }


	    /* OK ----------------------------------------------- */
	    
	    if (!ourdom || !ourleft) {
		DPRINT(Debug,14,(&Debug,
				 "update_message_id: not our"));

		if (!ourleft) {
		    DPRINT(Debug,14,(&Debug,", old message-id left side %s is not generated here",
				     old_left_part));		    
		}
		
		if (!ourdom) {
		    DPRINT(Debug,14,(&Debug,", old message-id domain %s is not generated here",
				 old_domain_part));
		}

		DPRINT(Debug,14,(&Debug,"\n"));
		
		ret = msid_stat_no_changes;
		goto nochange;
	    }
	    
	}
    }
    

    switch (message_id_localspec) {

	char * left_part;
	char * domain_part;

    case msgidlocal_md5:
	if (!current_hash) {
	    DPRINT(Debug,14, (&Debug, "update_message_id: hash needed\n"));
	    ret = msid_stat_need_hash;
	    if (need_hash) {
		*need_hash = md5_available();
		if (!*need_hash) {
		    DPRINT(Debug,14, (&Debug, "update_message_id: MD5 not available\n"));
		}
	    }
	    goto nochange;
	}
	
	if (0) {
	case msgidlocal_devurandom:

	    if (! local_devurandom_supported()) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDUnSupKw,
				  "Message-ID keyword %s unsupported."),
			  MSG_ID_localspec[message_id_localspec]);
		message_id_localspec = msgidlocal_bad;
		goto failure;		
	    }

	}

	if (0) {
	case msgidlocal_procuuid:
	    if (! local_procuuid_supported()) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeMsgIDUnSupKw,
				  "Message-ID keyword %s unsupported."),
			  MSG_ID_localspec[message_id_localspec]);
		message_id_localspec = msgidlocal_bad;
		goto failure;		
	    }	    
	}
	
	left_part   = NULL;
	domain_part = NULL;
	
	switch (message_id_domspec) {
	    const char * dom;
	    char * tempdom;
	    int q;
	    
	case msgiddom_domain:
	    
	    tempdom = get_msg_domain(dname_domain);
	    if (tempdom) {
		DPRINT(Debug,14, (&Debug,
				  "update_message_id: domain part candinate %s\n",
				  tempdom));
		
		domain_part = check_msg_domain(dname_domain,tempdom,cancel_p,
					       &now,&default_valid_until);
		if (domain_part) {
		    DPRINT(Debug,14, (&Debug,
				      "update_message_id: domain part %s succeed as %s\n",
				      tempdom,domain_part));

		    free(tempdom); tempdom = NULL;
		    goto found_domain;
		} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
		    DPRINT(Debug,7,(&Debug,
				    "update_message_id: %s: DNS lookup canceled\n",
				    tempdom));
		    free(tempdom); tempdom = NULL;
		    goto cleanup;
		}

		free(tempdom); tempdom = NULL;

	    }
	    break;
	    
	case  msgiddom_hostfullname:

	    if (hostfullname[0]) {
		DPRINT(Debug,14, (&Debug, "update_message_id: domain part candinate %s\n",
				  hostfullname));

		domain_part = check_msg_domain(dname_hostfullname,hostfullname,cancel_p,
					       &now,&default_valid_until);
		if (domain_part) {
		    DPRINT(Debug,14, (&Debug,
				      "update_message_id: domain part %s succeed as %s\n",
				      hostfullname,domain_part));
		    goto found_domain;
				    
		} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
		    DPRINT(Debug,7,(&Debug,
				    "update_message_id: %s: DNS lookup canceled\n",
				    hostfullname));
		    goto cleanup;
		}
	    }
	    break;
	    
	case msgiddom_maildomain:
	    dom = useful_mailname();
	    if (dom) {
		DPRINT(Debug,14, (&Debug,
				  "update_message_id: domain part candinate %s\n",
				  dom));
		
		domain_part = check_msg_domain(dname_maildomain,dom,cancel_p,
					       &now,&default_valid_until);
		if (domain_part) {
		    DPRINT(Debug,14, (&Debug, "update_message_id: domain part %s succeed as %s\n",
				      dom,domain_part));
		    goto found_domain;
		} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
		    DPRINT(Debug,7,(&Debug,
				    "update_message_id: %s: DNS lookup canceled\n",
				    dom));
		    goto cleanup;
		}
	    }
	    break;
	    
	case msgiddom_default:
	    
	    /* 1) hostname */
	    
	    if (hostfullname[0]) {
		DPRINT(Debug,14, (&Debug,
				  "update_message_id: domain part candinate %s\n",
				  hostfullname));
		
		domain_part = check_msg_domain(dname_hostfullname,hostfullname,cancel_p,
					       &now,&default_valid_until);
		if (domain_part) {
		    DPRINT(Debug,14, (&Debug,
				      "update_message_id: domain part %s succeed as %s\n",
				      hostfullname,domain_part));
		    goto found_domain;
		    
		} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
		    DPRINT(Debug,7,(&Debug, "update_message_id: %s: DNS lookup canceled\n",
				    hostfullname));
		    goto cleanup;
		}
	    }
	    
	    /* 2) interface address from mailer */
	    
	    q = query_mailer_info(info,MI_REMOTE_MAILER);
	    DPRINT(Debug,14, (&Debug,
			      "update_message_id: MI_REMOTE_MAILER gives %d\n",q));
	    if (q) {
		char * *domlist = NULL;
		size_t  domlist_len = 0;

		
		/* Get name from local addess of connection */
		domlist = query_s_mailer_info(&domlist_len,
					      info,MI_con_name_local,cancel_p);

		if (domlist) {
		    int ok = 0;
		    size_t x;
		    
		    DPRINT(Debug,14, (&Debug,
				      "update_message_id: Got %zu mailer addresses for MI_con_name_local\n",
				      domlist_len));
		    
		    tempdom = get_msg_domain(dname_mailer);
		    
		    if (tempdom) {
			size_t i;

			DPRINT(Debug,14, (&Debug,
					  "update_message_id: Checking if cached %s is one of mailer local addresses",
					  tempdom));

			for (i = 0; i < domlist_len; i++) {
			    if (domlist[i]) {
				if (0 == strcmp(domlist[i],tempdom)) {
				    DPRINT(Debug,14, (&Debug,"; found from index #%zu",
						      i));
				    ok = 1;
				    break;
				}
			    }
			}

			if (!ok) {
			    DPRINT(Debug,14, (&Debug,"; not found"));
			}
			DPRINT(Debug,14, (&Debug,"\n"));
			
			if (ok) {
			    DPRINT(Debug,14, (&Debug,
					      "update_message_id: domain part candinate %s from local address of mailer connection\n",
					      tempdom));
			    
			    domain_part = check_msg_domain(dname_mailer,tempdom,cancel_p,
							   &now,&default_valid_until);
			    if (domain_part) {
				DPRINT(Debug,14, (&Debug,
						  "update_message_id: domain part %s succeed as %s\n",
						  tempdom,domain_part));

				free_domlist(&domlist,&domlist_len);
				free(tempdom);
				
				goto found_domain;

			    } else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
				DPRINT(Debug,7,(&Debug,
						"update_message_id: %s: DNS lookup canceled\n",
						tempdom));

				free_domlist(&domlist,&domlist_len);
				free(tempdom); tempdom = NULL;

				goto cleanup;
			    }			    
			}
						
			free(tempdom); tempdom = NULL;
		    }

		    DPRINT(Debug,14, (&Debug,
				      "update_message_id: Checking mailer local addresses\n"));
		    
		    for (x = 0; x < domlist_len; x++) {
			if (domlist[x]) {
			    DPRINT(Debug,14, (&Debug,
					      "update_message_id: domain part candinate %s from local address #%zu of mailer connection\n",
					      domlist[x],x));
			    
			    
			    domain_part = check_msg_domain(dname_mailer,domlist[x],cancel_p,
							   &now,&default_valid_until);
				if (domain_part) {
				    DPRINT(Debug,14, (&Debug,
						      "update_message_id: domain part %s succeed as %s\n",
						      domlist[x],domain_part));
				    
				    free_domlist(&domlist,&domlist_len);				    
				    goto found_domain;

				} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
				    DPRINT(Debug,7,(&Debug,
						    "update_message_id: %s: DNS lookup canceled\n",
						    domlist[x]));
				    
				    free_domlist(&domlist,&domlist_len);
				    goto cleanup;
				}			    								
			}
		    }
				    
		    free_domlist(&domlist,&domlist_len);
		    
		} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
		    DPRINT(Debug,7,(&Debug, "update_message_id: lookup canceled\n"));
		    goto cleanup;
		}	       		
	    }
	    
	    /* 3) maildomain */
	    
	    dom = useful_mailname();
	    if (dom) {
		DPRINT(Debug,14, (&Debug,
				  "update_message_id: domain part candinate %s\n",
				  dom));
		
		domain_part = check_msg_domain(dname_maildomain,dom,cancel_p,
					       &now,&default_valid_until);
		if (domain_part) {
		    DPRINT(Debug,14, (&Debug,
				      "update_message_id: domain part %s succeed as %s\n",
				      dom,domain_part));
		    goto found_domain;
		    
		} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
		    DPRINT(Debug,7,(&Debug,
				    "update_message_id: %s: DNS lookup canceled\n",
				    dom));
		    goto cleanup;
		}
		
	    }
	    break;
	    
	    if (0) {
	    case msgiddom_whitelisted:
	    case msgiddom_literal:
		domain_part = get_msg_domain(dname_domain);
		if (domain_part) {
		    DPRINT(Debug,14, (&Debug, "update_message_id: domain part %s\n",
				      domain_part));
		    
		    /* No checks needed */
		    goto found_domain;
		}
	    }
	    
	case msgiddom_bad:
	case NUM_msgiddom:
	    DPRINT(Debug,14, (&Debug, "update_message_id: Message-id domain not available\n"));
	    
	    goto cleanup;
	}
	
	if (!domain_part && fallback_domain) {
	    size_t i;
	    
	    for (i = 0; i < fallback_domain_count && !domain_part; i++) {
		if (fallback_domain[i]) {
		    
		    if (DOMAIN_INFO_magic != fallback_domain[i]->magic) 
			panic("MAILER PANIC",__FILE__,__LINE__,
			      "update_message_id",
			      "Bad magic number (domain_info)",0);
		    
		    if (fallback_domain[i]->dompart) {			
			DPRINT(Debug,14, (&Debug,
					  "update_message_id: domain part candinate %s (fallback)\n",
					  fallback_domain[i]->dompart));
			
			
			domain_part = check_msg_domain0(fallback_domain[i],
							fallback_domain[i]->dompart,
							cancel_p,
							&now,
							&default_valid_until);
			if (domain_part) {
			    DPRINT(Debug,14, (&Debug,
					      "update_message_id: domain part %s succeed as %s\n",
					      fallback_domain[i]->dompart,domain_part));
			    goto found_domain;
			} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {       
			    DPRINT(Debug,7,(&Debug,
					    "update_message_id: %s: DNS lookup canceled\n",
					    fallback_domain[i]->dompart));
			    goto cleanup;
			}			
		    }
		}
	    }
	}

    found_domain:
	if (domain_part) {
	    
	    switch (message_id_localspec) {
	    case msgidlocal_md5:
		if (current_hash) {
		    size_t          buffer_len = len_digest_proc(current_hash);
		    unsigned char * buffer     = safe_zero_alloc(buffer_len);
		    size_t          hash_len   = result_digest_proc(current_hash,
								    buffer,buffer_len);

		    if (hash_len) {
			size_t x;
			
			DPRINT(Debug,14,(&Debug,
					 "update_message_id: hash len %zu [",
					 hash_len));

			for (x = 0; x < hash_len; x++) {
			    DPRINT(Debug,14,(&Debug,"%02x",buffer[x]));
			}
			DPRINT(Debug,14,(&Debug,"]\n"));

			left_part = octectvalue_to_left(buffer,hash_len);
		    }

		    free(buffer); buffer = NULL;
		} else {
		    DPRINT(Debug,14, (&Debug,
				      "update_message_id: NO hash\n"));
		}
		break;

	    case msgidlocal_devurandom:

		left_part = left_part_from_devurandom(&ret);				
		break;

	    case msgidlocal_procuuid:

		left_part = left_part_from_procuuid(&ret);
		break;

	    case msgidlocal_bad:
	    case msgidlocal_none:
	    case NUM_msgidlocal:
		break;
	    }

	    if (left_part) {
		struct message_id  * id = new_message_id(left_part,domain_part,NULL);
		
		DPRINT(Debug,14,(&Debug,
				 "update_message_id: left part %s\n",left_part));

		if (id) {

		    if (* msg_id_res) 
			free_message_id(msg_id_res);
			
		    * msg_id_res = id;
		    ret = msid_stat_updated;

		}
	    } else {
		DPRINT(Debug,14, (&Debug,
				  "update_message_id: message-id left part is not available\n"));

	    }

	} else {
	    DPRINT(Debug,14, (&Debug,
			      "update_message_id: message-id domain is not available\n"));
	}

    cleanup:
	if (left_part)
	    free(left_part);
	if (domain_part)
	    free(domain_part);

	
	if (msid_stat_none == ret) {	    
	failure:
	case NUM_msgidlocal:
	case msgidlocal_bad:
	case msgidlocal_none:
	    DPRINT(Debug,14, (&Debug, "update_message_id: Message-id not available\n"));
	
	    if (* msg_id_res) 
		ret = msid_stat_no_changes;
	}
	break;
    }

 nochange:
    
    DPRINT(Debug,14, (&Debug, "update_message_id=%d",ret));
    switch (ret) {
    case msid_stat_need_hash:    DPRINT(Debug,14, (&Debug, " msid_stat_need_hash"));    break;
    case msid_stat_need_restart: DPRINT(Debug,14, (&Debug, " msid_stat_need_restart")); break;
    case msid_stat_none:         DPRINT(Debug,14, (&Debug, " msid_stat_none"));         break;
    case msid_stat_updated:      DPRINT(Debug,14, (&Debug, " msid_stat_updated"));      break;
    case msid_stat_no_changes:   DPRINT(Debug,14, (&Debug, " msid_stat_no_changes"));   break;
    }
    if (need_hash && *need_hash) {
	DPRINT(Debug,14, (&Debug, "; *need_hash=%p",*need_hash));
    }
    if (* msg_id_res) {
	char * A = get_message_id_ascii_brackets(* msg_id_res);
	
	DPRINT(Debug,14, (&Debug, "; * msg_id_res=%p",* msg_id_res));
	if (A) {
	    DPRINT(Debug,14, (&Debug, " = %s",A));
	    free(A);
	}
    }
    
    DPRINT(Debug,14, (&Debug,"\n"));
    
    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
