static char rcsid[] = "@(#)$Id: senderhelper.c,v 2.5 2020/11/15 16:02:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.5 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mailer.h"
#include "s_elm.h"
#include "s_me.h"

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

DEBUG_VAR(Debug,__FILE__,"header");

void sndhelper_get_sender(add_sender_h,mailer_info,sender_domain_p,
			  sender_domain_verify_p,sender_addr_p,
			  from_addr_len)
     const enum add_sender_v add_sender_h;
     struct mailer_info    * mailer_info;
     char                 ** sender_domain_p;
     int                   * sender_domain_verify_p;
     char                 ** sender_addr_p;
     const int               from_addr_len;
{
    char * sender_addr           = *sender_addr_p;
    char * sender_domain         = *sender_domain_p;
    int    sender_domain_verify  = *sender_domain_verify_p;
	    
    switch (add_sender_h) {
	
    case  add_sender_no:
	if (from_addr_len > 1) {
	    
	    /* Result is malloced */		    
	    sender_addr = from_addr_literal(mailer_info,
					    &sender_domain);
	    
	    DPRINT(Debug,9,(&Debug,
			    "sndhelper_get_sender: Need add Sender %s because of %d addresses on From\n",
			    sender_addr,from_addr_len));
	    
	    break;
	    
	case add_sender_auto:
	    sender_domain_verify = query_mailer_info(mailer_info,
						     MI_REMOTE_MAILER);
	    if (sender_domain_verify) {
		DPRINT(Debug,9,(&Debug,
				"sndhelper_get_sender: remote mailer, Sender address domain need to be verified before adding\n"));
	    }
		    
	    /* FALLTHRU */
	    
	case add_sender_yes:
	    /* Result is malloced */		    
	    sender_addr = from_addr_literal(mailer_info,
					    &sender_domain);
	    break;
	}
	break;
    case NUM_add_sender: /* not used */ break;
    }

    *sender_addr_p          = sender_addr;
    *sender_domain_p        = sender_domain;
    *sender_domain_verify_p = sender_domain_verify;    
}

enum sndhelper_status sndhelper_handle_sender(add_sender_h,mailer_info,sender_domain_p,
			     sender_domain_verify_p,sender_addr_p,
			     from_addr_len, sender_ok_p,headers)
     const enum add_sender_v   add_sender_h;
     struct mailer_info      * mailer_info;
     char                   ** sender_domain_p;
     int                     * sender_domain_verify_p;
     char                   ** sender_addr_p;
     const int                 from_addr_len;
     int                     * sender_ok_p;
     struct mailing_headers  * headers;
{
    char * sender_addr           = *sender_addr_p;
    char * sender_domain         = *sender_domain_p;
    int    sender_domain_verify  = *sender_domain_verify_p;
    int    sender_ok             = *sender_ok_p;

    enum sndhelper_status ret = sndhelper_none;
    
    if (headers->env_from &&
	mailer_env_from_changed(headers->env_from) &&
	! sender_addr) {
	
	switch (add_sender_h) {
	case add_sender_auto:
	    sender_domain_verify = query_mailer_info(mailer_info,
						     MI_REMOTE_MAILER);
	    
	    if (sender_domain_verify) {
		DPRINT(Debug,9,(&Debug,
				"sndhelper_handle_sender: remote mailer, Sender address domain need to be verified before adding\n"));
	    }
	    
	    /* FALLTHRU */
	    
	case add_sender_yes:
	    /* Result is malloced */		
	    sender_addr = from_addr_literal(mailer_info,
					    &sender_domain);
	    break;
	case  add_sender_no:
	case NUM_add_sender: /* not used */ break;
	}
    }

    if (sender_addr) {
	int          is_literal = 0;
	const char * whitelisted_name = NULL;
	
	DPRINT(Debug,6,(&Debug,"sndhelper_handle_sender: Possible Sender address %s",
			sender_addr));
	if (sender_domain) {
	    const char * reserved_name    = NULL;
	    
	    DPRINT(Debug,6,(&Debug,", domain %s\n",
			    sender_domain));
	    
	    sender_ok =
		build_address_classify_domain(sender_domain,
					      &is_literal,
					      &whitelisted_name,
					      &reserved_name);
	    
	    if (!sender_ok) {
		build_address_classify_message(sender_addr,
					       sender_domain,
					       is_literal,
					       whitelisted_name,
					       reserved_name);
		
		ret = sndhelper_domain_failure;
		
	    }
				
	} else {
	    DPRINT(Debug,6,(&Debug,", no domain\n"));
	}
	
	if (sender_ok) {
	    int val = give_dt_enumerate_as_int(&verify_domain);
	    
	    DPRINT(Debug,9,(&Debug, 
			    "sndhelper_handle_sender: %s: verify-domain =  %d%s, sender_domain_verify=%d, is_literal=%d, %d from address(es)",
			    sender_addr,						
			    val,
			    val  < 0 ? ", uses shared library" : "",
			    sender_domain_verify,
			    is_literal,
			    from_addr_len));
	    
	    if (whitelisted_name) {
		DPRINT(Debug,9,(&Debug, ", whitelisted %s",
				whitelisted_name));
	    }
	    
	    if (sender_domain_verify && sender_domain && val &&
		!whitelisted_name && !is_literal) {
		
		struct cancel_data  *main_cancel;
		enum verify_domain_result r;
		char *rewrite  = NULL;
		
		DPRINT(Debug,9,(&Debug, ", verifying domain %s\n",
				sender_domain));
		
		main_cancel =
		    build_address_cancel_mode(sender_domain);
		
		if (main_cancel) {
		    DPRINT(Debug,9,(&Debug, "  ... have cancel mode\n"));
		}
		r = verify_mail_domain_util(sender_domain,&rewrite,&main_cancel);
		
		if (from_addr_len < 2) {
		    
		    switch (r) {
		    case verify_domain_not_found:
		    case verify_domain_no_mail:  
		    case verify_domain_bad_syntax:
			
			DPRINT(Debug,9,(&Debug, "sndhelper_handle_sender: Sender address %s will not set\n",
					sender_addr));
			
			free(sender_addr);
			sender_addr = NULL;
			
		    case verify_domain_ok:
		    case verify_domain_not_available:
		    case verify_domain_fallback:
		    case verify_domain_failure:
			break;
		    }
		} else {
		    DPRINT(Debug,9,(&Debug, "sndhelper_handle_sender: %s: Verify result not used, %d from addresses\n",
				    sender_addr,from_addr_len));
		}
		
		if (rewrite) {
		    DPRINT(Debug,9,(&Debug, "sndhelper_handle_sender: %s: domain %s rewrite is %s\n",
				    sender_addr,sender_domain,rewrite));
		}
		
		if (sender_addr && verify_domain_ok == r) {
		    char *sep = qstrpbrk(sender_addr,"!:@");
		    
		    if (sep) {
			
			DPRINT(Debug,9,
			       (&Debug, 
				"sndhelper_handle_sender: addr %s separator %c\n",
				sender_addr, *sep));
			
			if ('@' == *sep && sep > sender_addr && rewrite) {
			    int len = sep-sender_addr;
			    
			    char * addr1 = 
				elm_message(FRM("%.*s@%s"),
					    len,sender_addr,
					    rewrite);
			    
			    DPRINT(Debug,9,(&Debug,
					    "sndhelper_handle_sender: rewiting address %s to %s\n",
					    sender_addr,addr1));
			    free(sender_addr);
			    sender_addr = addr1;
			}
		    }			
		}
		
		if (main_cancel)
		    free_cancel(&main_cancel);
		
		if (rewrite) {
		    free (rewrite);
		    rewrite = NULL;
		}
	    } else {
		DPRINT(Debug,9,(&Debug, ", verify skipped\n"));
	    }
	    
	    if (sender_addr) {
		headers->sender = sender_addr;
		sender_addr = NULL;
		
		DPRINT(Debug,6,(&Debug,
				"sndhelper_handle_sender: setting Sender address: %s\n",
				headers->sender));
	    }
	}	    	    
    }

    DPRINT(Debug,9,(&Debug,
		    "sndhelper_handle_sender=%d",ret));
    switch (ret) {
    case sndhelper_domain_failure:   DPRINT(Debug,9,(&Debug," sndhelper_domain_failure")); break;
    case sndhelper_none:             DPRINT(Debug,9,(&Debug," sndhelper_none"));           break;
    }
    *sender_addr_p          = sender_addr;
    *sender_domain_p        = sender_domain;
    *sender_domain_verify_p = sender_domain_verify;    
    *sender_ok_p            = sender_ok;

    DPRINT(Debug,9,(&Debug,"\n"));
    
    return ret;
}

/* Returns ".localdomain" if domain ends .localdomain and
   Configure or setup_names() selected that for placeholder
   domain.
*/

const char * matches_dummy_domain(domain)
     const char  * domain;
{

    if (have_dummy_domain && '.' == have_dummy_domain[0]) {
	
	const char * x;

	for (x = domain; *x; x++) {

	    if ('.' == *x &&
		0 == strcmp(x,have_dummy_domain)) {

		DPRINT(Debug,12,(&Debug,
				 "matches_dummy_domain=%s: Domain %s matches to dummy domain %s\n",
				 have_dummy_domain,domain,have_dummy_domain));

		return have_dummy_domain;
	    }
	}	       
    }

    return NULL;
}



/* Return 1 if OK, 0 if invalid */

int build_address_classify_domain(domain, is_literal_p, whitelisted_name_p, reserved_name_p)
     const char  * domain;
     int         * is_literal_p;
     const char ** whitelisted_name_p;
     const char ** reserved_name_p;
{
    int          ret              = 1;
    int          is_literal       = 0;

    const char * whitelisted_name = NULL;
    const char * reserved_name    = NULL;
    
    if (!domain[0]) {
	 DPRINT(Debug,12,(&Debug,
			  "build_address_classify_domain: empty domain.\n"));

	 ret = 0;
		
    } else if ('[' == domain[0]) {   

	/* Address literal must be one token */
	
	int len = rfc822_toklen(domain);
	 
	 if (len < 2 || domain[len] || domain[len-1] != ']') {
	     
	     DPRINT(Debug,12,(&Debug,
			      "build_address_classify_domain: %s failed to parse as literal\n",domain));
	     
	     ret = 0;
	     is_literal = -1;
	     goto done;
	 }
	 
	 is_literal = 1;
	 
	 DPRINT(Debug,12,(&Debug,
			  "build_address_classify_domain: %s is literal\n",domain));
		 
    } else {	
	whitelisted_name =
	    is_whitelisted_valid_domain(domain);
	
	if (whitelisted_name) {
	    DPRINT(Debug,12,(&Debug,
			     "build_address_classify_domain: Domain %s is whitelisted on %s\n",
			     whitelisted_name,
			     domain));

	} else {
	    
	    reserved_name = is_special_use_domain(domain);
	    
	    if (reserved_name) {
		
		DPRINT(Debug,12,
		       (&Debug, 
			"build_address_classify_domain: Special use domain %s on %s\n",
			reserved_name,domain));
		
		ret = 0;
	    } else {
		const char * dummy_name =
		    matches_dummy_domain(domain);
		
		if (dummy_name) {
		    
		    DPRINT(Debug,12,(&Debug,
				     "build_address_classify_domain: Domain %s matches to dummy domain %s -- IGNORED\n",
				     domain,dummy_name));
		}
	    }
	}
    }

 done:

    DPRINT(Debug,12, 
	   (&Debug, "build_address_classify_domain=%d; domain=%s",
	    ret,domain));
    if (is_literal_p) {
	*is_literal_p       = is_literal;
	DPRINT(Debug,12, (&Debug, ", is_literal=%d",is_literal));
    }
    if (whitelisted_name_p) {
	*whitelisted_name_p = whitelisted_name;
	DPRINT(Debug,12, (&Debug, ", whitelisted_name=%s",
			  whitelisted_name ? whitelisted_name : "(none)"));
    }
    if (reserved_name_p) {
	*reserved_name_p    = reserved_name;
	DPRINT(Debug,12, (&Debug, ", reserved_name=%s",
			  reserved_name ? reserved_name : "(none)"));
    }
    DPRINT(Debug,12, (&Debug, "\n"));
    
    return ret;
}


void build_address_classify_message(addr,domain,is_literal,
				    whitelisted_name,reserved_name)
     const char * addr;
     const char * domain;
     int          is_literal;
     const char * whitelisted_name;
     const char * reserved_name;
{
    if (!addr)
	addr = "(?)";
    if (!domain)
	domain = "(?)";
    else if (! domain[0]) {
	DPRINT(Debug,9,
	       (&Debug, "build_address_classify_message: %s: empty domain\n",
		addr));
	lib_error(CATGETS(elm_msg_cat, 
			  ElmSet, 
			  ElmVerifyEmptyDomainAddr,
			  "Empty domain on address %s."),
		  addr);	
    }
    
    if (is_literal < 0) {

	DPRINT(Debug,9,
	       (&Debug, 
		"build_address_classify_message: %s: invalid literal %s\n",
		addr,domain));

	lib_error(CATGETS(elm_msg_cat, 
			  ElmSet, 
			  ElmVerifyBadAddressLiteral,
			  "Invalid literal %s on address %s."),
		  domain,addr);
	
    } else if (is_literal) {
	DPRINT(Debug,12,(&Debug,
			 "build_address_classify_message: %s: Domain %s is literal%s\n",
			 addr,
			 domain));		       	
    }
    
    if (whitelisted_name) {
	DPRINT(Debug,12,(&Debug,
			 "build_address_classify_message: %s: Domain %s is whitelisted\n",
			 addr,
			 whitelisted_name));		       	
    }
			    
    if (reserved_name) {
	DPRINT(Debug,9,
	       (&Debug, 
		"build_address_classify_message: %s: special use domain %s\n",
		addr,reserved_name));
	
	lib_error(CATGETS(elm_msg_cat, 
			  ElmSet, 
			  ElmVerifyReservedDomain,
			  "Reserved domain %s is unsupported on address %s."),
		  reserved_name,addr);
    }

}

struct cancel_data  *build_address_cancel_mode(domain)
     const char *domain;
{
	struct cancel_data  * main_cancel = NULL;
	enum name_lookup_cancel_v cancel_mode =
#ifdef REMOTE_MBX
	    give_dt_enumerate_as_int(&name_lookup_cancel)
#else
	    name_lookup_cancel_disabled
#endif
	    ;

	DPRINT(Debug,9,(&Debug,
			"build_address_cancel_mode: name-lookup-cancel = %d, domain = %s\n",
			cancel_mode,domain));
	
	switch (cancel_mode) {
	case name_lookup_cancel_enabled:
	    main_cancel =
		new_schedule_cancel(100  /* 100 ms,
					    do not print this now, this text is
					    reset before (some) hostname lookups */,
				    CATGETS(elm_msg_cat, MeSet,MeLookingUp,
					    "Looking up %s ..."),
				    domain);	
	    
	    break;
	case name_lookup_cancel_disabled:
	case name_lookup_cancel_auto:
	case NUM_name_lookup_cancel:
	    break;
	}
  
	return main_cancel;
}	    

#define DEFAULT_VERIFY_TIME 1000

static char * last_verify_domain    = NULL;
static char * last_rewrite          = NULL;
static struct timed_task_handle  * free_verify_handle = NULL;
static struct schedule_timelimit  last_verify_valid_until;


/* Used as free_rc hook */
void free_senderhelper()
{
    DPRINT(Debug,20,(&Debug, "free_senderhelper: called\n"));
    
    if (last_verify_domain) {

	DPRINT(Debug,9,(&Debug, "free_senderhelper: freeing cached domain: %s\n",
			last_verify_domain));
	
	free(last_verify_domain);
	last_verify_domain = NULL;
	
    }
    
    if (last_rewrite) {
	
	DPRINT(Debug,9,(&Debug, "free_senderhelper: freeing cached rewrite: %s\n",
			last_rewrite));

	free(last_rewrite);
	last_rewrite = NULL;
    }
    
    if (free_verify_handle) {
	DPRINT(Debug,9,(&Debug, "free_senderhelper: freeing timed task handle %p\n",
			free_verify_handle));

	alter_timed_task(&free_verify_handle,
			 NULL,
			 NO_timed_task_free,
			 NO_inc_timed_task_refcount,
			 NO_timed_task_action,
			 &last_verify_valid_until);
    }
}

S_(timed_task_action_f free_last_verify_domain)
static enum timed_task_r  free_last_verify_domain P_((struct timed_task *task,
					 const struct schedule_timelimit *now));
static enum timed_task_r  free_last_verify_domain(task,now)
     struct timed_task *task  /* NOT USED */;
     const struct schedule_timelimit *now;
{
    free_senderhelper();

    return timed_task_disable;
}

enum verify_domain_result verify_mail_domain_util(domain,rewrite,cancel_p)
     const char *domain;
     char **rewrite;
     struct cancel_data             ** cancel_p
     /* May be NULL, Used if dns lookup was cancelable */;
{
    enum verify_domain_result result  = verify_domain_failure;
    
    const char * reserved_name = is_special_use_domain(domain);
    const char * dummy_name = NULL;
    
    if (reserved_name) {
	
	lib_error(CATGETS(elm_msg_cat, MeSet,
			  MeReservedDomainUnsupported,
			  "Reserved domain %s is unsupported: %s"),
		  reserved_name,domain);
	
	DPRINT(Debug,9,(&Debug, 
			"verify_mail_domain_util: %s: special use domain %s\n",
			domain,reserved_name));
	
	result = verify_domain_not_found;

    } else if ((dummy_name = matches_dummy_domain(domain))) {
	    
	DPRINT(Debug,9,(&Debug, 
			"verify_mail_domain_util: Domain %s matches to dummy domain %s\n",
			domain,dummy_name));
	
	result = verify_domain_not_found;
	
    } else {

	static enum verify_domain_result  
	    last_verify_result  = verify_domain_failure;
	static int last_rewrite_disable = 0;

	struct schedule_timelimit now = NO_schedule_timelimit;
	
	schedule_set_current_time(&now);
	
	if (last_verify_domain &&
	    schedule_valid_until_ok(&last_verify_valid_until,
				    &now) &&

	    ( rewrite ?    /* case sensitive, so that cached 
			      rewrite does not change case */
	      0 == strcmp(last_verify_domain,domain) :
	      0 == istrcmp(last_verify_domain,domain)
	      )) {

	    char * A = schedule_timeout_string(&last_verify_valid_until);
	    if (A) {
		DPRINT(Debug,9,(&Debug, 
				"verify_mail_domain_util: %s: domain %s result is still valid until %s\n",
				domain,last_verify_domain,A));
		free(A);
	    }
	   
	    result = last_verify_result;
	    
	    switch (result) {
	    case verify_domain_fallback:
		DPRINT(Debug,9,(&Debug, 
				"verify_mail_domain_util: %s: cached result -- verify domain fallback\n",
				domain));
#ifdef I_NETDB
		goto fallback;
#endif
	    case verify_domain_failure:
		DPRINT(Debug,9,(&Debug, 
				"verify_mail_domain_util: %s: ignoring cached result\n",
				domain));

		goto retry;
	    default:
		DPRINT(Debug,9,(&Debug, 
				"verify_mail_domain_util: %s: using cached result\n",
				domain));
		break;
	    }
	    
	    if (rewrite) {
		if (!last_rewrite) {

		    if (last_rewrite_disable && !*rewrite) {

			DPRINT(Debug,9,(&Debug, 
					"verify_mail_domain_util: %s: no rewrite data, ok\n",
					domain));

		    } else {

			DPRINT(Debug,9,(&Debug, 
					"verify_mail_domain_util: %s: no rewrite data, retrying\n",
					domain));
			goto retry;
		    }
		} else  {
		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: %s: using cached rewrite: %s\n",
				    domain,last_rewrite));
		    *rewrite = strmcpy(*rewrite,last_rewrite);
		} 
	    }

	   
	} else {  /* Not cached */

	    const char *tag;
	    int         translated_value ;

	    const struct dt_enum_shared * is_shared_value;
	    struct schedule_timelimit verify_timelimit;
	    
	retry:
	    verify_timelimit = NO_schedule_timelimit;

	    if (schedule_set_next_timeout0(&verify_timelimit,&now,
					   DEFAULT_VERIFY_TIME,
					   se_use_before_time_MAX)) {
		char * A = schedule_timeout_string(&verify_timelimit);
		if (A) {
		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: %s: default verify timelimit is %s\n",
				    domain,A));
		    free(A);
		}
	    }
	    
	    tag = 0;
	    translated_value = 0;
	    is_shared_value = give_dt_shared_value(&verify_domain,
						   &tag,&translated_value);
	    	    
	    if (is_shared_value) {
		
		DPRINT(Debug,9,(&Debug, 
				"verify_mail_domain_util: %s: using shared routine for tag %s\n",
				domain,tag));

#ifdef USE_DLOPEN
		result = shared_verify_domain(domain,rewrite,tag,
					      translated_value,is_shared_value,
					      &now,
					      &verify_timelimit  /* valid until */,
					      cancel_p);
#else
		DPRINT(Debug,9,(&Debug, 
				"verify_mail_domain_util: no shared routines available\n"));
		result = verify_domain_not_available;
#endif

#ifdef I_NETDB
		if (verify_domain_fallback == result)
		    goto fallback;
#endif	


	    } else if (!tag) { 

		switch(translated_value) {
		case verify_domain_shared:
		    
		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: %s: using shared routine\n",
				    domain));
	    
#ifdef USE_DLOPEN
		    /* search any verifier module */
		    result = shared_verify_domain(domain,rewrite,NULL,translated_value,NULL,
						  &now,
						  &verify_timelimit  /* valid until */,
						  cancel_p);
#else
		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: no shared routines available\n"));
		    result = verify_domain_not_available;
#endif

#ifdef I_NETDB
		    if (verify_domain_fallback == result)
			goto fallback;
#endif
		    break;
	    
#ifdef I_NETDB	    
		case verify_domain_yes:

		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: %s: trying shared routine first\n",
				    domain));

#ifdef USE_DLOPEN
		    /* search any verifier module */
		    result = shared_verify_domain(domain,rewrite,NULL,
						  translated_value,NULL,
						  &now,
						  &verify_timelimit  /* valid until */,
						  cancel_p);
#else
		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: no shared routines available\n"));
		    result = verify_domain_not_available;
#endif

		    if (verify_domain_not_available != result &&
			verify_domain_fallback != result)
			break;

		    /* FALLTHROUGH */
	    	    
		fallback:

		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: %s: using fallback or default routine\n",
				    domain));

		case verify_domain_hostname: {
		    
		    struct hostent *he = NULL;

		    if ( cancel_p && *cancel_p) { /* Interrupting of getaddrinfo() or
						     gethostbyname() is not really
						     supported, result is uncertain */
			
			if (is_schedule_cancel(*cancel_p,NULL))
			    set_cancel_message(*cancel_p,-1 /* also show it now */,
					       CATGETS(elm_msg_cat, MeSet,MeLookingUp,
						       "Looking up %s ..."),
					       domain);
			else {
			    DPRINT(Debug,14,(&Debug,
					     "verify_mail_domain_util: 'Looking up' already printed.\n"));
			}
	    
		    } else
			lib_transient(CATGETS(elm_msg_cat, MeSet,
					      MeLookingUp,
					      "Looking up %s ..."),
				      domain);

		    errno = 0;
		    
		    he = gethostbyname(domain);
		    
		    if (!he) {
			int err = errno; 
			
			DPRINT(Debug,9,(&Debug, 
					"verify_mail_domain_util: %s: h_errno=%d",
					domain,h_errno));
			if (err) {
			    DPRINT(Debug,9,(&Debug, " errno=%d (%s)",
					    err,strerror(err)));
			}
			DPRINT(Debug,9,(&Debug, "\n"));

			if (HOST_NOT_FOUND != h_errno &&
			    NO_DATA        != h_errno &&
			    NO_RECOVERY    != h_errno &&
			    EINTR == err &&  cancel_p && *cancel_p &&
			    is_canceled(*cancel_p)) {
			    DPRINT(Debug,9,(&Debug, 
					    "verify_mail_domain_util: %s: query canceled\n",
					    domain));

			    result = verify_domain_failure;
			    goto fail;

			} else {
			    switch(h_errno) {
			    case HOST_NOT_FOUND:
				result = verify_domain_not_found;
				break;
				
			    case NO_DATA:
				result = verify_domain_ok;
				
			    }
			}
		    } else
			result = verify_domain_ok;
		    
		    if (result >= verify_domain_not_found) {
					
			switch(result) {
			case verify_domain_ok:
			    lib_transient(CATGETS(elm_msg_cat, MeSet,MeLookingUpOK,
						  "Looking up %s ... OK"),
					  domain);
			    break;
			case verify_domain_not_found:
			    lib_transient(CATGETS(elm_msg_cat, MeSet,MeLookingUpNotExists,
						  "Looking up %s ... Not exists"),
					  domain);
			break;
			case verify_domain_not_available:
			case verify_domain_fallback:
			case verify_domain_failure:
			case verify_domain_no_mail:
			case verify_domain_bad_syntax:
			    break;
			}
		    } else
			lib_transient(CATGETS(elm_msg_cat, MeSet,
					      MeLookingUpFailed,
					      "Looking up %s ... Failed"),
				      domain);	
		}
		    break;
#endif
		}
	    } /* !tag */

	    if (result >= verify_domain_not_found) {
		char * A;
		last_verify_domain = strmcpy(last_verify_domain,domain);
		last_verify_result = result;
		last_verify_valid_until = verify_timelimit;

		last_rewrite_disable = rewrite && !*rewrite;
		if (last_rewrite_disable) {
		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: %s: caching no rewrite\n",
				    last_verify_domain));
		}
		
		if (rewrite && *rewrite) {
		    last_rewrite = strmcpy(last_rewrite,*rewrite);

		    DPRINT(Debug,9,(&Debug, 
				    "verify_mail_domain_util: %s: caching rewrite: %s\n",
				    last_verify_domain,last_rewrite));

		} else if (last_rewrite) {
		    free(last_rewrite);
		    last_rewrite = NULL;
		}

		A = schedule_timeout_string(&last_verify_valid_until);
		if (A) {
		     DPRINT(Debug,9,(&Debug, 
				     "verify_mail_domain_util: %s: caching result to %s\n",
				     last_verify_domain,A));
		     free(A);
		}

		if (alter_timed_task(&free_verify_handle,
				     NULL,
				     NO_timed_task_free,
				     NO_inc_timed_task_refcount,
				     free_last_verify_domain,
				     &last_verify_valid_until)) {
		     DPRINT(Debug,9,(&Debug, 
				     "verify_mail_domain_util: Scheduled removal of verify result, timed task handle %p\n",
				     free_verify_handle));
		}
	    }
	    
	} /* not cached */
    } /* not reverved name */

 fail:
    DPRINT(Debug,9,(&Debug, 
		    "verify_mail_domain_util=%d: %s",
		    result,domain));

    switch (result) {
    case verify_domain_not_available: 
	DPRINT(Debug,9,(&Debug,
			" verify_domain_not_available"));
	break;
    case verify_domain_fallback:
	DPRINT(Debug,9,(&Debug,
			" verify_domain_fallback"));
	break;
    case verify_domain_failure:
	DPRINT(Debug,9,(&Debug,
			" verify_domain_failure"));
	break;
    case verify_domain_not_found:
	DPRINT(Debug,9,(&Debug,
			" verify_domain_not_found"));
	break;
    case verify_domain_ok:
	DPRINT(Debug,9,(&Debug,
			" verify_domain_ok"));
	break;
    case verify_domain_no_mail:
	DPRINT(Debug,9,(&Debug,
			" verify_domain_no_mail"));
	break;
    case verify_domain_bad_syntax:
	DPRINT(Debug,9,(&Debug,
			" verify_domain_bad_syntax"));
	break;
    }

    if (rewrite && *rewrite) {
	DPRINT(Debug,9,(&Debug,", rewrite %s",
			*rewrite));
    }

    DPRINT(Debug,9,(&Debug, "\n"));

    return result;
}

void sndhelper_update_user_agent(headers,hdr_charset,enmime,default_xmailer)
     struct mailing_headers * headers;
     charset_t                hdr_charset;
     int                      enmime;
     default_xmailer_cb     * default_xmailer;
{
    char * old_value = headers->encoded_user_agent;
    
    headers->encoded_user_agent = NULL;
    if (dt_flag_is_set(&program_snd_ident,snd_ident_user_agent)) {
	struct string * C = NULL;
	
	if (dt_flag_is_set(&program_snd_ident,snd_ident_version)) {
	    const char * v = VERSIONTAG;

	    if ('.' == *v)
		v++;
	    if ('\0' == *v)
		goto no_version;

	    headers->encoded_user_agent = elm_message(FRM("elmme/%s"),
						      v);
	    
	} else {
	no_version:
	    headers->encoded_user_agent = safe_strdup("elmme");
	}

	if (headers->xmailer && string_len(headers->xmailer))
	    C = ascify_string(headers->xmailer);
	else
	    C = default_xmailer();

	if (C) {
	    char * s_comment = string_to_hdr(HDR_COMMENT,C,
					     hdr_charset,
					     enmime,NULL,0);
	    
	    if (s_comment) {
		headers->encoded_user_agent =
		    strmcat(headers->encoded_user_agent," (");
		headers->encoded_user_agent =
		    strmcat(headers->encoded_user_agent,s_comment);
		headers->encoded_user_agent =
		    strmcat(headers->encoded_user_agent,")");
		
		free(s_comment);
		s_comment = NULL;
	    }
	}
	
	free_string(&C);
    }

    if (old_value){
	DPRINT(Debug,14,(&Debug,
			"sndhelper_update_user_agent: User-Agent: Changing %s => %s\n",
			old_value,
			headers->encoded_user_agent ? headers->encoded_user_agent : "(unset)"));
	free(old_value);
    } else if (headers->encoded_user_agent) {
	DPRINT(Debug,14,(&Debug,
			"sndhelper_update_user_agent: User-Agent: Setting => %s\n",
			headers->encoded_user_agent));
    }
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
