static char rcsid[] = "@(#)$Id: connection.c,v 2.12 2020/05/28 05:02:12 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.12 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "ss_imp.h"

DEBUG_VAR(Debug,__FILE__,"net");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#ifdef REMOTE_MBX

static struct connection_cache * CACHE_LIST = NULL;


int connection_match_username_host(c,username,host)
     const struct connection_cache *c;
     const char *username;
     const char *host;
{   
    return  remote_account_match(&(c->C),username,host);
}

/* May not be 
       uin16 port
   because this is promoted to int
*/

struct connection_cache * locate_from_cache_1(ra,con_type,port) 
     const struct remote_account *ra;
     const struct connection_type  *con_type;
     int port;
{
    struct connection_cache * ptr;
    char mybuffer[256];
    const char * s UNUSED_VAROK = NULL;


    if (REMOTE_ACCOUNT_magic != ra->magic)
	 panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "locate_from_cache_1",
		  "Bad magic (remote_account)",0);

    if (ra->hostaddr.generic)
	s = give_SOCKADDR_ptr_as_string(ra->hostaddr,
					ra->addrsize,
					mybuffer,sizeof mybuffer);

    for (ptr = CACHE_LIST; ptr; ptr = ptr -> next) {

	if (REMOTE_ACCOUNT_magic != ptr->C.magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "locate_from_cache_1",
		  "Bad magic (remote_account on cache list)",0);
	
	if (same_remote_account_host_user(ra,&(ptr->C)) &&
	    port == ptr->port &&
	    (( ! ra->hostname && ! ptr->C.hostname) 
	     ||
	     ( ra->hostname && ptr->C.hostname &&
	       0 == strcmp(ra->hostname,ptr->C.hostname))
	     ) 
	    &&
	    (( ! ra->hostaddr.dummy && ! ptr->C.hostaddr.dummy)
	     ||
	     (ra->hostaddr.dummy && ptr->C.hostaddr.dummy &&
	      ra->addrsize == ptr->C.addrsize &&
	      ra->addrsize > 0 &&
	      0 == memcmp(ra->hostaddr.dummy,ptr->C.hostaddr.dummy,
			  ra->addrsize)) )
	    ) {
	    
	    if ((!con_type || con_type == ptr->type) &&
		(ra->service == ptr->C.service ||
		 (ra->service && ptr->C.service &&
		  is_similar_service_entry(ra->service,
					   ptr->C.service))
		  )) {
		    
		DPRINT(Debug,11,(&Debug,
				 "locate_from_cache_1=%p;  username=%s, host=%s, port=%d; hostname=%s, addr=%s\n",
				 ptr,
				 ra->username ? ra->username : "<NULL>",
				 ra->host ? ra->host : "<NULL>",
				 port,
				 ra->hostname ? ra->hostname : "<NULL>",
				 s ? s : "<unknown>"));

		return ptr;
	    }

	    DPRINT(Debug,11,(&Debug,
			     "locate_from_cache_1: %p:  username=%s, host=%s, port=%d; hostname=%s, addr=%s\n",
			     ra->username ? ra->username : "<NULL>",
			     ra->host ? ra->host : "<NULL>",
			     port,
			     ra->hostname ? ra->hostname : "<NULL>",
			     s ? s : "<unknown>"));
	}
    }

    DPRINT(Debug,11,(&Debug,
		     "locate_from_cache_1=NULL;  username=%s, host=%s, port=%d; hostname=%s, addr=%s\n",
		     ra->username ? ra->username : "<NULL>",
		     ra->host ? ra->host : "<NULL>",port,		     
		     ra->hostname ? ra->hostname : "<NULL>",
		     s ? s : "<unknown>"));

    return NULL;
}

/* May not be 
       uin16 port
   because this is promoted to int
*/
struct connection_cache * locate_from_cache(username,host,con_type,port,se)
     const char * username;
     const char * host;
     const struct connection_type  *con_type;
     int port;     /* 0 == default port */
     struct service_entry     * se;    /* extra constrait */
{
    struct connection_cache * ptr;
    
    for (ptr = CACHE_LIST; ptr; ptr = ptr -> next) {

	if (REMOTE_ACCOUNT_magic != ptr->C.magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "locate_from_cache",
		  "Bad magic (remote_account on cache list)",0);

	if (!ptr->C.username || !ptr->C.host) {
	    DPRINT(Debug,11,(&Debug,
			     "locate_from_cache: %p: No host or username on cache!\n",
			     ptr));
	    continue;
	}
	

	if (0 == strcmp(username,ptr->C.username) &&
	    0 == istrcmp(host,ptr->C.host) &&
	    port == ptr->port) {
	    
	    if ((!con_type || con_type == ptr->type) &&
		(!se || se == ptr->C.service ||
		 (ptr->C.service &&
		  is_similar_service_entry(se,ptr->C.service))
		 )) {
		
		DPRINT(Debug,11,(&Debug,
				 "locate_from_cache=%p;  username=%s, host=%s, port=%d\n",
				 ptr,username, host,port));
		return ptr;	    
	    }

	    DPRINT(Debug,11,(&Debug,			    
			    "locate_from_cache: %p: same  username=%s, host=%s, port=%d\n",
			    ptr,username, host,port));
	}		
    }

    DPRINT(Debug,11,(&Debug,
		     "locate_from_cache=NULL;  username=%s, host=%s, port=%d\n",
		     username, host,port));

   
    return NULL;
}

static void remove_from_cache P_((struct connection_cache *con,
				  int maybe));
static void remove_from_cache(con,maybe)
     struct connection_cache *con;
     int maybe;
{
    struct connection_cache * ptr, *prev = NULL;

    for (ptr = CACHE_LIST; ptr; prev=ptr, ptr = ptr -> next) {
	if (ptr == con) {
	    if (NULL == prev)
		CACHE_LIST = ptr->next;
	    else
		prev->next = ptr->next;
	    ptr->next = NULL;

	    DPRINT(Debug,11,(&Debug,
			     "remove_from_cache: con=%p (%s@%s), type=%p (%s): removed from cache\n",
			     con,
			     con->C.username ? con->C.username : "<NULL>",
			     con->C.host ? con->C.host : "<NULL>",
			     con->type,
			     con->type->type_name));

	    return;
	}
    }
    con->next = NULL;
    
    if (maybe) {
	DPRINT(Debug,11,(&Debug,
			 "remove_from_cache: con=%p (%s@%s), type=%p (%s): Not in cache\n",
			 con,
			 con->C.username ? con->C.username : "<NULL>",
			 con->C.host ? con->C.host : "<NULL>",
			 con->type,
			 con->type->type_name));
	
    } else {
	DPRINT(Debug,1,(&Debug,
			"remove_from_cache: con=%p (%s@%s), type=%p: Not in cache\n",
			con,
			con->C.username ? con->C.username : "<NULL>",
			con->C.host ? con->C.host : "<NULL>",
			con->type));
	
	panic("CONNECTION PANIC",__FILE__,__LINE__,"remove_from_cache",
	      "Connection not in connection cache",0);
    }
}



struct connection_cache *  create_connection(T)
     struct connection_type *T;
{
    struct connection_cache *ret;

    if (CONNECTION_TYPE_magic != T->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "create_connection",
              "Bad connection type",0);

    DPRINT(Debug,10,(&Debug,
		     "create_connection: type=%p (%s)\n",
		     T,T->type_name));

    ret = safe_zero_alloc(sizeof (*ret));

    ret->magic = CONNECTION_CACHE_magic;
    ret->type  = T;
    ret->next  = NULL;
    ret->f     = NULL;
    ret->d     = NULL;
    ret->state = CON_error;
    ret->seen_badpid = 0;
    ret->a.any = NULL;     /* Initilize to NULL */
    ret->port  = 0;        /* Default port */

    ret->quota = NULL;

    zero_remote_account(&(ret->C));

    ret->type->cache_zero_it(ret);

    DPRINT(Debug,10,(&Debug,
		     "create_connection=%p\n",ret));

    return ret;
}


int free_connection(c,cd)
     struct connection_cache **c;
     struct cancel_data  * cd;
{
    int ret = 1;
    
    if (CONNECTION_CACHE_magic != (*c)->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "free_connection",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != (*c)->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "free_connection",
              "Bad connection type",0);

    DPRINT(Debug,10,(&Debug,
		     "free_connection: con=%p (%s@%s), type=%p (%s)",
		     (*c),
		     (*c)->C.username ? (*c)->C.username : "<NULL>",
		     (*c)->C.host ? (*c)->C.host : "<NULL>",
		     (*c)->type,
		     (*c)->type->type_name));
    
    DPRINT(Debug,10,(&Debug,", state=%d",
		     (*c)->state));
    switch ((*c)->state) {
    case CON_verify_failure: DPRINT(Debug,10,(&Debug," CON_verify_failure"));
	break;
    case CON_greeting:  DPRINT(Debug,10,(&Debug," CON_greeting")); break;
    case CON_error:     DPRINT(Debug,10,(&Debug," CON_error")); break;
    case CON_open:      DPRINT(Debug,10,(&Debug," CON_open")); break;
    case CON_logged:    DPRINT(Debug,10,(&Debug," CON_logged")); break;
    }
    if ((*c)->seen_badpid) {
	DPRINT(Debug,10,(&Debug,", not our stream, owner pid %d",
			 (*c)->seen_badpid));
	
    }    
    DPRINT(Debug,10,(&Debug,"\n"));
    
    if (CACHE_LIST)
	remove_from_cache((*c),1);

    if (NULL != (*c)->C.stream &&
	(*c)->state != CON_error) {
	
	ret = (*c)->type->cache_close_it((*c),cd);
    }

    (*c)->type->cache_free_it((*c));

    clear_remote_account0(&((*c)->C),
			  (*c)->seen_badpid);

    (*c)->state = CON_error;
    (*c)->f    = NULL;
    (*c)->d    = NULL;
    (*c)->next = NULL;
    (*c)->type = NULL;

    if ((*c)->quota) {
	mail_quota_unlink_con((*c)->quota,*c);
	(*c)->quota = NULL;
    }
    
    if ((*c)->this_server) {
        remove_remote_server_from_connection(*c,
					     (*c)->this_server);
	(*c)->this_server = NULL;
    }

    free(*c);
    *c = NULL;

     DPRINT(Debug,10,(&Debug,
		      "free_connection=%d\n",
		      ret));
    
    return ret;
}

void connection_set_reset_quota(con,quota)
     struct connection_cache * con;
     struct mail_quota       * quota;
{
    if (CONNECTION_CACHE_magic != con->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "connection_set_reset_quota",
	      "Bad magic number (connection_cache)",0);

    if (con->quota != quota) {
	if (con->quota) {
	    mail_quota_unlink_con(con->quota,con);
	}
	
	con->quota = quota;
    }
}


S_(ra_verify_peer_cb_f connection_verify_peer_cb)
static int connection_verify_peer_cb P_((struct remote_account *ra,
					 union ra_verify_peer_cb_data *data,
					 const struct string * Server,
					 const struct string * server));
static int connection_verify_peer_cb(ra,data,Server,server)
   struct remote_account *ra;
   union ra_verify_peer_cb_data *data;
   const struct string * Server;
   const struct string * server;
{
    struct connection_cache *c = data->connection;
    int ret;

    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
              "connection_verify_peer_cb",
	      "Bad connection type",0);

    ret = c->type->cache_verify_cb_connection(c,ra,Server,server);

    DPRINT(Debug,12,(&Debug,
		     "connection_verify_peer_cb=%d  (state=%d)\n",
		     ret,c->state));

    return ret;
}

/* return 1 if cb success, 0 failure return -1 no connection
 */
int prepare_connection_verify(c)
     struct connection_cache *c;
{
    int ret;

    struct string * Server = NULL;
    struct string * server = NULL;

    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "prepare_connection_verify",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "connection_verify_peer",
              "Bad connection type",0);  
  
    server = c->type->cache_verify_server_const();
    Server = c->type->cache_verify_Server_const();
   
    ret = c->type->cache_verify_cb_connection(c,& (c->C),Server,server);

    DPRINT(Debug,12,(&Debug,
		     "prepare_connection_verify=%d  (state=%d)\n",
		     ret,c->state));

    if (server)
       free_string(&server);
   
   if (Server)
       free_string(&Server);

   return ret;
}

/* 0 = failed
   1 = OK
*/
static int connection_verify_peer P_((struct connection_cache *c));
static int connection_verify_peer(c)
     struct connection_cache *c;
{
    int status = 1;
    struct string * Server = NULL;
    struct string * server = NULL;
    union ra_verify_peer_cb_data cbdata;
    int r;

    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "connection_verify_peer",
	      "Bad magic number (connection_cache)",0);

    cbdata.connection = c;  
    
    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "connection_verify_peer",
              "Bad connection type",0);  

   server = c->type->cache_verify_server_const();
   Server = c->type->cache_verify_Server_const();

   if ((r = remote_account_verify_peer(& (c->C), &cbdata,
				       connection_verify_peer_cb,
				       Server,server)) < 1) {

       if (r < 0)
	   c->state = CON_error;
       else
	   c->state = CON_verify_failure;

       status = 0;
   }
   
   if (server)
       free_string(&server);
   
   if (Server)
       free_string(&Server);

   DPRINT(Debug,11,(&Debug,
		    "connection_verify_peer=%d  (state=%d)\n",
		    status,c->state));
   return status;
}

enum connection_state get_connection_state(c)
      const struct connection_cache *c;
{

    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "get_connection_state",
	      "Bad magic number (connection_cache)",0);


    return c->state;
}

void set_connection_state(c,s)
      struct connection_cache *c;
      enum connection_state s;
{
    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "set_connection_state",
	      "Bad magic number (connection_cache)",0);

    c->state = s;
}

static void join_connection_common P_((struct connection_cache *c,
				       struct remote_account *X,
				       enum connection_state st));
static void join_connection_common(c,X,st)
     struct connection_cache *c;
     struct remote_account *X;
     enum connection_state st;
{
    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "join_connection_common",
	      "Bad magic number (connection_cache)",0);

    clear_remote_account(&(c->C));
    
    c->C     = *X;
    c->state = st;
    
    zero_remote_account(X);     /* Avoid double free */
}

int  join_connection(c,X,st)
     struct connection_cache *c;
     struct remote_account *X;
     enum connection_state st;
{
    int ret = 0;

    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "join_connection",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "join_connection",
              "Bad connection type",0);

    DPRINT(Debug,10,(&Debug,
		     "join_connection: con=%p, type=%p (%s), X=%p (%s@%s), st=%d\n",
		     c,
		     c->type,
		     c->type->type_name,
		     X,
		     X->username ? X->username : "<NULL>",
		     X->host ? X->host : "<NULL>",
		     st));

    join_connection_common(c,X,st);

    ret = c->type->cache_open_it(c);

    if (ret)
	ret = connection_verify_peer(c);

    DPRINT(Debug,10,(&Debug,
		     "join_connection=%d  (state=%d)\n",
		     ret,c->state));
    return ret;
}

int  join_connection_hm(c,X,st,passhm)
     struct connection_cache *c;
     struct remote_account *X;
     enum connection_state st;
     struct browser_passhm *passhm;
{
    int ret = 0;
    int use_passhm = 0;

    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "join_connection_hm",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "join_connection",
              "Bad connection type",0);

    DPRINT(Debug,10,(&Debug,
		     "join_connection_hm: con=%p, type=%p (%s), X=%p (%s@%s), st=%d\n",
		     c,
		     c->type,
		     c->type->type_name,
		     X,
		     X->username ? X->username : "<NULL>",
		     X->host ? X->host : "<NULL>",
		     st));

    join_connection_common(c,X,st);

    if (passhm) {
	if (BROWSER_PASSHM_magic != passhm->magic)
	    panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
		  "join_connection_hm",
		  "Bad magic number (browser_passhm)",0);

	ret = c->type->cache_open_hm_it(c,passhm,&use_passhm);

    } else
	ret = c->type->cache_open_it(c);

    if (ret)
	ret = connection_verify_peer(c);

    if (ret) {
	if (passhm) {
	    if (use_passhm) {
		ret = browser_passhm_verify_ra_con(passhm,
						   &(c->C));
		if (!ret) {
		    DPRINT(Debug,10,(&Debug,
				     "join_connection_hm: Connection have (hashmark) verify failure\n"));
		}  
	    } else {
		 DPRINT(Debug,10,(&Debug,
				  "join_connection_hm: Skipping browser_passhm connection verify\n"));
	    }
	}
    }

    DPRINT(Debug,10,(&Debug,
		     "join_connection_hm=%d  (state=%d)\n",
		     ret,c->state));


    return ret;
}

static int login_connection_common P_((struct connection_cache *c));
static int login_connection_common(c)
     struct connection_cache *c;
{
    int ret = 1;

    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "login_connection_common",
	      "Bad magic number (connection_cache)",0);

    if (c->state == CON_verify_failure) {
	DPRINT(Debug,10,(&Debug,
			 "login_connection_common: Connection have verify failure\n"));

	ret = 0;

    } else if (c->state != CON_open) {

	DPRINT(Debug,10,(&Debug,
			 "login_connection_common: Opening it first...\n")); 
	ret = c->type->cache_open_it(c);

	if (ret)
	    ret = connection_verify_peer(c);

    }

    DPRINT(Debug,15,(&Debug,
		     "login_connection_common=%d: %s@%s\n",
		     ret,c->C.username ? c->C.username : "<NULL>",
		     c->C.host ? c->C.host : "<NULL>"));

    return ret;
}


int login_connection_hm(c,passhm)
     struct connection_cache *c;
     struct browser_passhm *passhm;
{
    int ret = 1;
    int use_passhm = 0;

    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "login_connection_hm",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "login_connection_hm",
              "Bad connection type",0);
    
    DPRINT(Debug,10,(&Debug,
		     "login_connection_hm: con=%p (%s@%s), type=%p (%s)\n",
		     c,
		     c->C.username ? c->C.username : "<NULL>",
		     c->C.host ? c->C.host : "<NULL>",
		     c->type,c->type->type_name));

    if (passhm) {
	if (BROWSER_PASSHM_magic != passhm->magic)
	    panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
		  "login_connection_hm",
		  "Bad magic number (browser_passhm)",0);
	
	if (c->state == CON_verify_failure) {
	    DPRINT(Debug,10,(&Debug,
			     "login_connection_hm: Connection have verify failure\n"));

	    ret = 0;
	} else if (c->state != CON_open) {

	    DPRINT(Debug,10,(&Debug,
			     "login_connection_hm: Opening it first...\n")); 

	    ret = c->type->cache_open_hm_it(c,passhm,&use_passhm);

	    if (ret)
		ret = connection_verify_peer(c);
	}

    } else
	ret = login_connection_common(c);

    if (ret) {

	if (passhm) {

	    if (use_passhm) {
		ret = browser_passhm_verify_ra_con(passhm,
						   &(c->C));
		if (!ret) {
		    DPRINT(Debug,10,(&Debug,
				     "login_connection_hm: Connection have (hashmark) verify failure\n"));
		    goto fail;
		}
	    } else {
		DPRINT(Debug,10,(&Debug,
				 "login_connection_hm: Skipping browser_passhm connection verify\n"));
	    }
	}

	ret = c->type->cache_login_it(c,NULL);
    }

    fail:
    DPRINT(Debug,10,(&Debug,
		     "login_connection_hm=%d\n",ret));
    return ret;
}


int  login_connection(c,password)
     struct connection_cache *c;
     const struct string *password; /* May be NULL */
{
    int ret = 0;

    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "login_connection",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "login_connection",
              "Bad connection type",0);

    DPRINT(Debug,10,(&Debug,
		     "login_connection: con=%p (%s@%s), type=%p (%s)\n",
		     c,
		     c->C.username ? c->C.username : "<NULL>",
		     c->C.host ? c->C.host : "<NULL>",
		     c->type,c->type->type_name));

    ret = login_connection_common(c);
    if (ret) {
	ret = c->type->cache_login_it(c,password);
    }

    DPRINT(Debug,10,(&Debug,
		     "login_connection=%d\n",ret));
    return ret;
}

void folder_from_connection(c,f)
     struct connection_cache *c;
     struct folder_info *f;
{
    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "folder_from_connection",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "folder_from_connection",
              "Bad connection type",0);

    DPRINT(Debug,10,(&Debug,
		     "folder_from_connection: con=%p (%s@%s), type=%p (%s)\n",
		     c,
		     c->C.username ? c->C.username : "<NULL>",
		     c->C.host ? c->C.host : "<NULL>",
		     c->type,c->type->type_name));
    DPRINT(Debug,10,(&Debug,
		     "                      : folder=%p\n",f));

    /* Must have be on connection cache */
    remove_from_cache(c,0);

    c->f = f;
    c->type->cache_folder_from_it(c,f);
}

/* Sets *c = NULL */
void browser_from_connectionX(c,d)
     struct connection_cache **c;
     struct folder_browser *d;
{
    if (CONNECTION_CACHE_magic != (*c)->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "browser_from_connectionX",
	      "Bad magic number (connection_cache)",0);
        
    if (CONNECTION_TYPE_magic != (*c)->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "browser_from_connectionX",
              "Bad connection type",0);

    DPRINT(Debug,10,(&Debug,
		     "browser_from_connectionX: con=%p (%s@%s), type=%p (%s)\n",
		     *c,
		     (*c)->C.username ? (*c)->C.username : "<NULL>",
		     (*c)->C.host ? (*c)->C.host : "<NULL>",
		     (*c)->type,(*c)->type->type_name));
    DPRINT(Debug,10,(&Debug,
		     "                      : browser=%p\n",d));

    /* Must have be on connection cache */
    remove_from_cache(*c,0);

    (*c)->d = d;
    (*c)->type->cache_browser_from_it(*c,d);
    *c = NULL;  /* Assumed that cache_browser_from_it stores pointer */
}

int close_connection(con, free_stream_only,cd)
     struct connection_cache *con;
     int free_stream_only;
     struct cancel_data  * cd;
{
    int ret = 1;
    
    if (CONNECTION_CACHE_magic != con->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "close_connection",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != con->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "close_connection",
              "Bad connection type",0);
 
    DPRINT(Debug,10,(&Debug,
		     "close_connection: con=%p (%s@%s), type=%p (%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>",
		     con->type,con->type->type_name));

    if (free_stream_only) {
	DPRINT(Debug,10,(&Debug,
			 "close_connection: Obly freeing stream\n"));
    } else {
	ret = con->type->cache_close_it(con,cd);
    }
    
    if (NULL != con->C.stream ) {
	DPRINT(Debug,10,(&Debug,
			 "close_connection: Closing socket"));
	if (con->seen_badpid) {
	    DPRINT(Debug,10,(&Debug,", not our stream, owner pid %d",
			     con->seen_badpid));	    
	}    
	DPRINT(Debug,10,(&Debug,"\n"));
	
	FreeStreamStack0(&(con->C.stream),
			 con->seen_badpid,
			 1 /* Force clearing of stack even when this is not last reference */);
    }	
    
    con->state = CON_error;
    
    DPRINT(Debug,10,(&Debug,
		     "close_connection=%d\n",
		     ret));
    return ret;
}

int close_cached_connections(cd)
    struct cancel_data  * cd;
{
    struct connection_cache * ptr, *next;
    int X = 0;

    DPRINT(Debug,4,(&Debug,
		"Closing cached connections...\n"));
    for (ptr = CACHE_LIST; ptr; ptr = next) {

	if (CONNECTION_CACHE_magic != ptr->magic)
	    panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
		  "close_cached_connections",
		  "Bad magic number (connection_cache)",0);

	next = ptr -> next;
	if (!close_connection(ptr,0,cd)) {
	    DPRINT(Debug,10,(&Debug,"close_cached_connections: close_connection failed\n"));
	}
	X++;
	if (!free_connection(&ptr,cd)) {
	    DPRINT(Debug,10,(&Debug,"close_cached_connections: free_connection failed\n"));
	}
    }
    DPRINT(Debug,4,(&Debug,
		    "Cached connections closed\n"));


    DPRINT(Debug,10,(&Debug,"close_cached_connections=%d\n",
		     X));
    return X;
}

void  cache_connection(c)
     struct connection_cache *c;
{    
    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "cache_connection",
	      "Bad magic number (connection_cache)",0);
    
    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "cache_connection",
              "Bad connection type",0);

    DPRINT(Debug,10,(&Debug,
		     "cache_connection: con=%p (%s@%s), type=%p (%s)\n",
		     c,
		     c->C.username ? c->C.username : "<NULL>",
		     c->C.host ? c->C.host : "<NULL>",
		     c->type,c->type->type_name));

    c->f    = NULL;            /* Detach from folder */
    c->d    = NULL;            /* Detach from browser */
    c->next = CACHE_LIST;
    CACHE_LIST = c;    
}

const struct remote_account * give_con_remote_account(c)
     struct connection_cache *c;
{
    if (CONNECTION_CACHE_magic != c->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "give_con_remote_account",
	      "Bad magic number (connection_cache)",0);

    if (CONNECTION_TYPE_magic != c->type->magic)
	panic("REMOTE CONNECTION PANIC",__FILE__,__LINE__,
	      "give_con_remote_accoun",
              "Bad connection type",0);

    return &( c->C );
}


#endif


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

