static char rcsid[] = "@(#)$Id: last_read.c,v 2.1 2024/06/10 18:05:34 hurtta Exp $";

/************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.1 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ************************************************************************/

#include "def_mbox.h"
#include "mbxlocal_imp.h"
#include "s_me.h"

#include "commentfile.h"
#include "filelock.h"

DEBUG_VAR(Debug,__FILE__,"file");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static const unsigned char * cs2us P_((const char *str));
static const unsigned char * cs2us(str)
     const char *str;
{
    return (unsigned char *)str;
}

#define LAST_READ_CACHE_magic	0xFA0A

/*  File .elm-last-read-{uid} */
static struct last_read_cache {
    unsigned short               magic;   /* LAST_READ_CACHE_magic */

    int                       refcount;
    
    char * sys_dir;             /* NULL, if fallback file
				   ~/.elm/last.read
				*/

    struct string_sort * cached_name; /* helper for sortlist functions */

    
    char * cache_file_name;  
    
    /* Store only inode number if st_dev is same than on cache 
       cache_file.dev is implied
     */
    struct file_changes      cache_file;
    
    struct sortlist * entries;


    unsigned modified:    1; /* Need write when
				flush_last_read()
				is called 
			     */
    unsigned read:    1;    /* Read from file */

} * malloc_last_read_cache P_((const char                 * sys_dir,
			       const struct stat          * st,
			       size_t                       prealloc,
			       const struct file_changes  * fchanges,
			       const char                 * filename
			       ));

static void inc_last_read_cache_refcount P_((struct last_read_cache * cache));

#define LAST_READ_CACHE_DEF_magic	0xFA0C

static const struct last_read_cache_def {
    unsigned short   magic; /* LAST_READ_CACHE_DEF_magic */
    
    const struct stat         * st;
    size_t                      prealloc;
    const struct file_changes * fchanges;
    const char                * cache_file_name;
} NULL_last_read_cache_def = {
    LAST_READ_CACHE_DEF_magic,
    NULL,
    0,
    NULL,
    NULL
};


#define LAST_READ_ENTRY_magic	0xFA0B

enum LAST_READ_bits {
    LAST_READ_bit_have_st_ino,  /* Have inode number */
    LAST_READ_bit_modified,     /* entry modified - do no read
				 from .elm-last-read-{uid}
			        */
    LAST_READ_bit_flag,         /* entry flagged - found
				   from directory listing
				*/
    LAST_READ_bit_delete,       /* entry marked for delete -
				   not written -
				   deleted after write
				*/

    LAST_READ_bit_old,

    LAST_READ_bit_valid,        /* if not set, entry not set yet */

    
    LAST_READ_bit_purge,        /* entry marked for delete
				   after read -- unmarked
				   if seen on read
				*/

    LAST_READ_bit_memonly,      /* Not saved to disk */
    
    LAST_READ_bit_count
};

#define LAST_READ_have_st_ino  (1 << LAST_READ_bit_have_st_ino)  
#define LAST_READ_modified     (1 << LAST_READ_bit_modified)
#define LAST_READ_flag         (1 << LAST_READ_bit_flag)
#define LAST_READ_delete       (1 << LAST_READ_bit_delete)
#define LAST_READ_old          (1 << LAST_READ_bit_old)
#define LAST_READ_valid        (1 << LAST_READ_bit_valid)
#define LAST_READ_purge        (1 << LAST_READ_bit_purge)
#define LAST_READ_memonly      (1 << LAST_READ_bit_memonly)


#define LAST_READ_ENTRY_DEF_magic	0xFA0D

struct last_read_entry_def {
    unsigned short               magic;   /* LAST_READ_ENTRY_DEF_magic */

    unsigned entry_flags : LAST_READ_bit_count;
};

static struct last_read_entry {

    unsigned short               magic; /* LAST_READ_ENTRY_magic */

    int                       refcount;
    
    time_t   last_read;       /* Last read timestamp */
    
    /* Does not include path
       if on same directory than
       .elm-last-read-{uid} file
    */
    char               * folder_sys;
    
    struct string_sort * cached_name; /* helper for sortlist functions */
    
    ino_t     st_ino;         /* Inode number of folder */

    unsigned entry_flags : LAST_READ_bit_count;
} * malloc_last_read_entry P_((const char                   * folder_sys,
			       const struct last_read_entry_def * def
			       ));

static struct last_read_entry * malloc_last_read_entry(folder_sys,def)
     const char                       * folder_sys;
     const struct last_read_entry_def * def;
{
    struct last_read_entry * ret =
	safe_zero_alloc(sizeof (*ret));

    ret -> magic       =  LAST_READ_ENTRY_magic;

    ret -> refcount    = 1;

    ret -> last_read   = 0;

    ret -> folder_sys  = safe_strdup(folder_sys);
    ret -> cached_name = NULL;  /* Filled later */
    
    ret -> st_ino      = 0;

    ret -> entry_flags = 0;

    if (def) {
	if (LAST_READ_ENTRY_DEF_magic != def->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "malloc_last_read_entry",
		  "Bad magic number (last_read_entry_def)",0);

	ret -> entry_flags = def->entry_flags;
    }
    
    return ret;
}


static void free_last_read_entry P_((struct last_read_entry ** entry));
static void free_last_read_entry(entry)
     struct last_read_entry ** entry;
{
    if (LAST_READ_ENTRY_magic != (*entry)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_last_read_entry",
              "Bad magic type (last_read_entry)",0);

    if ((*entry)->refcount < 1)
	panic("MBX PANIC",__FILE__,__LINE__,"free_last_read_entry",
              "Bad refcount",0);

    (*entry)->refcount--;

    if ((*entry)->refcount > 0) {
	(*entry) = NULL;

	return;
    }

    if ((*entry)->folder_sys) {
	free((*entry)->folder_sys);

	(*entry)->folder_sys = NULL;	
    }

    if ((*entry)->cached_name)
	free_string_sort(& ((*entry)->cached_name));

    (*entry)->magic = 0; /* invalidate */
    free(*entry);

    *entry = NULL;
}

static void inc_last_read_entry_refcount P_((struct last_read_entry * entry));
static void inc_last_read_entry_refcount(entry)
     struct last_read_entry * entry;
{
    if (LAST_READ_ENTRY_magic != entry->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"inc_last_read_entry_refcount",
              "Bad magic type (last_read_entry)",0);
    
    entry->refcount++;
}

    

/* ------------------------------------------------------------------ */

S_(alloc_sort_item_f alloc_read_entry)
static void alloc_read_entry P_((union sort_item             * res,
				 const union sort_key          key,
				 const union sort_item_default def));
static void alloc_read_entry(res,key,def)
     union sort_item             * res;
     const union sort_key          key;
     const union sort_item_default def;
{
    res->last_read_entry =  malloc_last_read_entry(key.str,
						   def.last_read_entry);     
}

S_(free_sort_item_f free_read_entry)
static void free_read_entry  P_((union sort_item      * ptr)); /* Decrements refcount */
static void free_read_entry(ptr)
     union sort_item      * ptr;
{
    free_last_read_entry(& (ptr->last_read_entry));     
}

S_(inc_sort_item_refcount_f inc_read_entry_refcount)
static void inc_read_entry_refcount P_((union sort_item item));
static void inc_read_entry_refcount(item)
     union sort_item item;
{
    inc_last_read_entry_refcount(item.last_read_entry);
}

S_(sort_item_debug_name_f read_entry_name)
static struct string * read_entry_name P_((const union sort_item item));
static struct string * read_entry_name(item)
     const union sort_item item;
{
    struct string          * ret   = NULL;
    struct last_read_entry * entry = item.last_read_entry;
    
    if (LAST_READ_ENTRY_magic != entry->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"read_entry_name",
              "Bad magic type (last_read_entry)",0);

    if (entry->cached_name) {
	/* Increments refcount */
	ret = give_string_from_string_sort(entry->cached_name);
    } else {
	ret = new_string2(local_fs_charset,cs2us(entry->folder_sys));

	/* Cache reference of string (increments refcount) */
	entry->cached_name = new_string_sort(ret);
    }

    return ret;
}

S_(compare_sort_key_to_item_f cmp_key_read_entry)
static int cmp_key_read_entry P_((const union sort_key key,
				  const union sort_item item));
static int cmp_key_read_entry(key,item)
     const union sort_key key;
     const union sort_item item;
{
    int ret;

    struct last_read_entry * entry = item.last_read_entry;

    if (LAST_READ_ENTRY_magic != entry->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"cmp_key_read_entry",
              "Bad magic type (last_read_entry)",0);

    ret = strcmp(key.str,entry->folder_sys);

    return ret;
}

S_(sort_key_debug_name_f read_entry_key_name)
static struct string * read_entry_key_name  P_((const union sort_key key));
static struct string * read_entry_key_name(key)
     const union sort_key key;
{
    struct string          * ret   = new_string2(local_fs_charset,
						 cs2us(key.str));

    return ret;
}

static  struct sort_operation last_read_entry_op = {
    SORT_OPERATION_magic,

    alloc_read_entry,
    free_read_entry,
    inc_read_entry_refcount,
    read_entry_name,

    cmp_key_read_entry,

    read_entry_key_name
};



/* ------------------------------------------------------------------ */

S_(alloc_sort_item_f alloc_read_cache)
static void alloc_read_cache P_((union sort_item             * res,
				 const union sort_key          key,
				 const union sort_item_default def));
static void alloc_read_cache(res,key,def)
     union sort_item             * res;
     const union sort_key          key;
     const union sort_item_default def;
{
    const struct last_read_cache_def * X = def.last_read_cache;

    if (LAST_READ_CACHE_DEF_magic != X->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"alloc_read_cache",
              "Bad magic type (last_read_cache_def)",0);
    
    res->last_read_cache = malloc_last_read_cache(key.str,
						  X->st,
						  X->prealloc,
						  X->fchanges,
						  X->cache_file_name);
}

S_(free_sort_item_f free_read_cache)
static void free_read_cache  P_((union sort_item      * ptr)); /* Decrements refcount */
static void free_read_cache(ptr)
     union sort_item      * ptr;
{
    free_last_read_cache(& (ptr->last_read_cache));
}

S_(inc_sort_item_refcount_f inc_read_cache_refcount)
static void inc_read_cache_refcount P_((union sort_item item));
static void inc_read_cache_refcount(item)
     union sort_item item;
{   
    inc_last_read_cache_refcount(item.last_read_cache);
}

S_(sort_item_debug_name_f read_cache_name)
static struct string * read_cache_name P_((const union sort_item item));
static struct string * read_cache_name(item)
     const union sort_item item;
{
    struct string          * ret   = NULL;
    struct last_read_cache * cache = item.last_read_cache;

    if (LAST_READ_CACHE_magic != cache->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"read_cache_name",
              "Bad magic type (last_read_cache)",0);

    if (cache->cached_name) {
	/* Increments refcount */
	ret = give_string_from_string_sort(cache->cached_name);
    } else {
	ret = new_string2(local_fs_charset,cs2us(cache->sys_dir));

	/* Cache reference of string (increments refcount) */
	cache->cached_name = new_string_sort(ret);
    }

    return ret;
}

S_(compare_sort_key_to_item_f cmp_key_read_cache)
static int cmp_key_read_cache P_((const union sort_key key,
				  const union sort_item item));
static int cmp_key_read_cache(key,item)
     const union sort_key key;
     const union sort_item item;
{
    int ret;
    
    struct last_read_cache * cache = item.last_read_cache;

    if (LAST_READ_CACHE_magic != cache->magic)
	panic("MBX PANIC",__FILE__,__LINE__," cmp_key_read_cache",
              "Bad magic type (last_read_cache)",0);

    ret = strcmp(key.str,cache->sys_dir);

    return ret;
}

S_(sort_key_debug_name_f read_cache_key_name)
static struct string * read_cache_key_name  P_((const union sort_key key));
static struct string * read_cache_key_name(key)
     const union sort_key key;
{
    struct string          * ret   = new_string2(local_fs_charset,
						 cs2us(key.str));

    return ret;
}

static  struct sort_operation last_read_cache_op = { 
    SORT_OPERATION_magic,

    alloc_read_cache,
    free_read_cache,
    inc_read_cache_refcount,
    read_cache_name,

    cmp_key_read_cache,

    read_cache_key_name
};


/* ------------------------------------------------------------------ */


static const char LAST_READ_HEADER[] = "ELMME+ READ\n";
static const size_t LAST_READ_HEADER_len = (sizeof LAST_READ_HEADER) - 1;

#define NAME_LIMIT 2048

static int parse_last_read_cache P_((const char          * filename,
				     FILE                * f,
				     struct file_changes * cache_file,
				     struct sortlist    ** cache_entries,
				     int                 * errors,
			 const struct last_read_entry_def * entry_default,
				     int                 * suggest_rewrite
				     ));

#if DEBUG
/* Returns pointer to static storage */
static char * give_entry_flags P_((unsigned entry_flags));
static char * give_entry_flags(entry_flags)
     unsigned entry_flags;
{
    static char buffer[25*LAST_READ_bit_count];

    enum LAST_READ_bits b;

    if (!entry_flags)
	return "none";

    buffer[0] = '\0';
    buffer[1] = '\0'; /* In case there is unknown bits set */

    for (b = 0; b < LAST_READ_bit_count; b++) {
	if (ison(entry_flags,(1 << b))) {
	    switch (b) {
	    case LAST_READ_bit_have_st_ino:
		strfcat(buffer," LAST_READ_have_st_ino", sizeof buffer);
		break;
	    case LAST_READ_bit_modified:
		strfcat(buffer," LAST_READ_modified", sizeof buffer);
		break;
	    case LAST_READ_bit_flag:
		strfcat(buffer," LAST_READ_flag", sizeof buffer);
		break;
	    case LAST_READ_bit_delete:
		strfcat(buffer," LAST_READ_delete", sizeof buffer);
		break;
	    case LAST_READ_bit_old:
		strfcat(buffer," LAST_READ_old", sizeof buffer);
		break;
	    case LAST_READ_bit_valid:
		strfcat(buffer," LAST_READ_valid", sizeof buffer);
		break;
	    case LAST_READ_bit_purge:
		strfcat(buffer," LAST_READ_purge", sizeof buffer);
		break;
	    case LAST_READ_bit_memonly:
		strfcat(buffer," LAST_READ_memonly", sizeof buffer);
		break;
	    case  LAST_READ_bit_count:
		break;
	    }
	}
    }
    
    return buffer+1;
}
#endif

static int stat_last_read_cache P_((const char          * filename,
				    FILE                * f,
				    struct file_changes * cache_file));

static int stat_last_read_cache(filename,f,cache_file)
     const char          * filename;
     FILE                * f;
     struct file_changes * cache_file;
{
    int ret = 1;
    
    enum syscall_status  r_stat;
    struct stat          stat_buffer;
    int            fd = fileno(f);
    
    r_stat = fstat(fileno(f),&stat_buffer);
    switch (r_stat) {
	int err UNUSED_VAROK;
	char *X;
    case syscall_success:
	DPRINT(Debug,14,(&Debug,
			 "stat_last_read_cache: fstat %d: %s succeed: dev %lu ino %lu size %ld modified %ld",
			 fd,filename,
			 (unsigned long)stat_buffer.st_dev,
			 (unsigned long)stat_buffer.st_ino,
			 (long)stat_buffer.st_size,
			 (long)stat_buffer.st_mtime));
	X = ctime(& (stat_buffer.st_mtime));
	if (X) { /* ctime() includes newline */
	    DPRINT(Debug,14,(&Debug," -- %s",X));
	} else {
	    DPRINT(Debug,14,(&Debug,"\n"));
	}
	
	stat_to_file_changes(&stat_buffer,cache_file);
	break;
    case syscall_error:
	err = errno;
	
	DPRINT(Debug,14,(&Debug,
			 "stat_last_read_cache: fstat %d: %s failed: %s\n",
			 fd,filename,strerror(err)));
	ret = 0;
	break;
    }		

    DPRINT(Debug,14,(&Debug,
		     "stat_last_read_cache=%d; filename %s, fileno %d\n",
		     ret,filename,fd));

    return ret;
}

static int parse_last_read_cache(filename,f,cache_file,
				cache_entries,errors,entry_default,
				suggest_rewrite)
     const char          * filename;
     FILE                * f;
     struct file_changes * cache_file;
     struct sortlist    ** cache_entries;
     int                 * errors;
     const struct last_read_entry_def * entry_default;
     int                 * suggest_rewrite;
{
    
    char buffer[100];
    int lineno = 1;
    int c;
    int corrupt_note = 0;
    size_t idx = 0;  /* append mode position */
    size_t cache_len = 0;

    char * entry_name = NULL;
    size_t entry_name_alloced = 0;
    int ret = 0;
    long pos = 0;

    /* Assumed to be unsigned */
    ino_t MAX_ino     = ~(ino_t)0;
    time_t now = time(NULL);
    
    
    if (((time_t) -1) == now) {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,14,(&Debug,
			 "parse_last_read_cache: time() failed: %s\n",
			 strerror(err)));

    }

    DPRINT(Debug,14,(&Debug,
		     "parse_last_read_cache:  MAX_ino = %lu (unsigned assumed)\n",
		     (unsigned long)MAX_ino));

    
    
    if ((ret = mail_gets(buffer, sizeof buffer,f)) > 0) {
	if (ret != LAST_READ_HEADER_len || 0 != memcmp(buffer,LAST_READ_HEADER,
						       LAST_READ_HEADER_len)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileCorrupted,
			      "File %s is corrupted."),
		      filename);
	    
	    if (errors)
		(* errors)++;
	    
	    return 0;
	}
	lineno++;  /* mail_gets() reads whole line */
    }



    /* skip comment block */
    pos = ftell(f);
    c = fgetc(f);
    if (EOF != c) {

	if ('#' == c || '\n' == c) {

	    int lines = 0;
	    
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: Skipping comment block from offset %ld at %s\n",
			     pos,filename));
	
	    while ('#' == c || '\n' == c) {
		lines++;
		
		while (EOF != c && '\n' != c) {
		    c = fgetc(f);
		}
		
		if ('\n' == c) {
		    lineno++;

		    pos = ftell(f);
		    c = fgetc(f);
		} else
		    break;
	    }

	    if (lines) {
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: Skipped %d lines (last line offset %ld)\n",
				 filename,lines,pos));
	    }
	    
	}

	if (EOF != c)
	    ungetc(c,f);	
    }

    if (EOF == c) {
	pos = ftell(f);
	
	DPRINT(Debug,14,(&Debug,
			 "parse_last_read_cache: File %s do not have entries, offset %ld\n",
			 filename,pos));
    }

    if (* cache_entries) {
	size_t i;
	
	cache_len = sort_list_len(* cache_entries);

	DPRINT(Debug,14,(&Debug,
			 "parse_last_read_cache: %zu items on cache_entries already\n",
			 cache_len));

	for (i = 0; i < cache_len; i++) {

	    union sort_item res;

	    res.last_read_entry = NULL;

	    /* Increments refcount */
	    get_sort_list_item(* cache_entries,sort_list_get_normal,i,
			       &res);

	    if (res.last_read_entry) {
		struct last_read_entry * entry;
		
		if (LAST_READ_ENTRY_magic != res.last_read_entry->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "parse_last_read_cache",
			  "Bad magic type (last_read_entry)",0);

		entry = res.last_read_entry;
		
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: entry %zu = %Q",
				 i,entry->folder_sys));

		if (ison(entry->entry_flags,LAST_READ_have_st_ino)) {

		    DPRINT(Debug,14,(&Debug,", st_ino=%lu",
				     (unsigned long)(entry->st_ino)));
		}

		if (entry->entry_flags) {
		    DPRINT(Debug,14,(&Debug,", entry_flags=%u %s",
				     entry->entry_flags,
				     give_entry_flags(entry->entry_flags)
				     ));

		}

		/* Delete if not seen on read */
		
		if (isoff(entry->entry_flags,LAST_READ_delete) &&
		    isoff(entry->entry_flags,LAST_READ_modified) &&
		    isoff(entry->entry_flags,LAST_READ_memonly))
		    setit(entry->entry_flags,LAST_READ_purge);
		
		if (entry->last_read > 0) {
		    char * X;
		    
		    DPRINT(Debug,14,(&Debug,", last_read=%ld",
				     (long)(entry->last_read)));

		    X = ctime(&(entry->last_read));

		    /* ctime() includes newline \n to result */
		    if (X) {
			DPRINT(Debug,14,(&Debug," %s",
					 X));
		    } else {
			DPRINT(Debug,14,(&Debug,"\n"));
		    }		    
		} else {
			DPRINT(Debug,14,(&Debug,"\n"));
		}
		
		free_last_read_entry(& res.last_read_entry);
	    }	   
	}
    } else {
	size_t count = 0;
	int last_c = '\n';
	enum syscall_status r;

	DPRINT(Debug,14,(&Debug,
			 "parse_last_read_cache: Creating cache_entries, counting entries from offset %ld at %s\n",
			 pos,filename));
	
	while (EOF != (c = fgetc(f))) {
	    if (c == '\n')
		count++;
	    last_c = c;
	}

	if ('\n' != last_c) {
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: Last character was not newline\n"));
	    count ++;
	}

	DPRINT(Debug,14,(&Debug,
			 "parse_last_read_cache: Creating cache_entries, count=%zu of %s\n",
			 count,filename));

	* cache_entries = alloc_sort_list(&last_read_entry_op,count);

	r = fseek(f,pos,SEEK_SET);
	switch (r) {
	    int err UNUSED_VAROK;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: Reading again from offset %ld at %s\n",
			     pos,filename));
	    break;
	case syscall_error:
	    err = errno;
	    
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: Failed to seek to offset %ld at %s: %s\n",
			     pos,filename,strerror(err)));

	    if (errors)
		(* errors)++;

	    return 0;
	}
    }


    entry_name_alloced = SLEN;
    entry_name = safe_malloc(entry_name_alloced);
    
    while (EOF != (c = fgetc(f))) {

	size_t entry_name_len = 0;
	int q = 0;

	union sort_key            key;
	union sort_item           res;
	union sort_item_default   def;

	key.str             = NULL;  
	def.last_read_entry = entry_default;
	res.dummy           = NULL;
	
	if ('\n' == c) {
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: %s: %d: empty line\n",
			     filename,lineno));

	    lineno++;

	    if (suggest_rewrite)
		(*suggest_rewrite)++;
	    
	    break;
	}

	if (!c) {
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: %s: %d: NUL not allowed\n",
			     filename,lineno));

	    if (!corrupt_note) {
	    
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileCorrupted,
				  "File %s is corrupted."),
			  filename);

		if (errors)
		    (* errors)++;
		
		corrupt_note = 1;
	    }
	    
	    goto fail_line;
	}

	/* space or TAB */
	if (whitespace(c)) {
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: %s: %d: no filename\n",
			     filename,lineno));

	    goto fail_line;
	}

	if ('#' == c) {  /* Ignore lines starting with # */

	    
	    goto end_line;
	}
	
	if ('"' == c) {
	    /*  " and \   are quoted -- filename with NL \n is not allowed 
		end with "
	    */

	    q = 1;
	} else {
	    /* Filename but \  " or  NL \n is not allowed 
	       end with whitespace
	     */

	    q = 0;

	    entry_name[0]  = c;
	    entry_name_len = 1;
	}

	while (EOF != c) {
	    c = fgetc(f);
	    
	    if ('"' == c) {
		if (q)
		    break;

		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: %c is not allowed\n",
				 filename,lineno,c));
		goto fail_line;
	    }
	    
	    if (whitespace(c)) {
		if (!q) {
		    break;
		}		
	    }
	    
	    if ('\\' == c) {

		if (q) {
		    c = fgetc(f);

		    if (EOF == c)
			goto fail_line;
		    
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "parse_last_read_cache: %s: %d: %c is not allowed\n",
				     filename,lineno,c));
		    goto fail_line;
		}
	    }

	    if (!c) {
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: NUL not allowed\n",
				 filename,lineno));
		
		if (!corrupt_note) {
		    
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileCorrupted,
				      "File %s is corrupted."),
			      filename);
		    
		    if (errors)
			(* errors)++;
		    
		    corrupt_note = 1;
		}
		
		goto fail_line;
	    } 

	    if (entry_name_len > 0 &&
		'/' == c && '/' != entry_name[0]) {
		
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: / is only allowed on absolute paths\n",
				 filename,lineno));

		goto fail_line;
	    }
			    
	    if ('\n' == c) {
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: Unexpected NL\n",
				 filename,lineno));
		goto fail_line;
	    }

	    if (entry_name_len+1 >= entry_name_alloced &&
		entry_name_alloced < (size_t)(NAME_LIMIT+1)) {

		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: Expanding name buffer %zu => %d\n",
				 filename,lineno,entry_name_alloced,NAME_LIMIT+1));
		
		entry_name_alloced = NAME_LIMIT+1;
		entry_name = safe_realloc(entry_name,entry_name_alloced);
	    }
	    
	    if (entry_name_len+1 >= entry_name_alloced) {
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: Too long name\n",
				 filename,lineno));
		goto fail_line;
	    }
	    entry_name[entry_name_len++] = c;
	}

	if (0 == entry_name_len) {
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: %s: %d: No name\n",
			     filename,lineno));
	    goto fail_line;
	} else if (entry_name_len < entry_name_alloced) {
	    entry_name[entry_name_len] = '\0';
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: %s: %d: Too long name\n",
			     filename,lineno));
	    goto fail_line;
	}

	if (q) {
	    if ('"' != c) {
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: \" expected\n",
				 filename,lineno));
		goto fail_line;		 
	    }
	    c = fgetc(f);	    
	}
	
	if (EOF == c)
	    goto fail_line;

	/* Not space or TAB */
	if (! whitespace(c)) {
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: %s: %d: %Q: whitespace expected\n",
			     filename,lineno,entry_name));
	    goto fail_line;	
	}
	
	key.str             = entry_name;
	def.last_read_entry = entry_default;
	res.dummy       = NULL;

	/* Increments refcount, returns 1 if found or created */
	if (search_sort_list_item(* cache_entries,
				  sort_list_insert_hint /* append mode */,
				  key,def,&res,&idx,
				  suggest_rewrite)) {


	    if (res.last_read_entry) {
		struct last_read_entry * entry = res.last_read_entry;
		
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: %Q: found or created item %zu\n",
				 filename,lineno,entry_name, idx));
		

		       				 
		if (LAST_READ_ENTRY_magic != entry->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"parse_last_read_cache",
			  "Bad magic type (last_read_entry)",0);

		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: #%zu folder_sys=%s",
				 idx,entry->folder_sys));

		if (entry->entry_flags) {		    
		    DPRINT(Debug,14,(&Debug,", entry_flags=%u %s",
				     entry->entry_flags,
				     give_entry_flags(entry->entry_flags)
				     ));
		}

		DPRINT(Debug,14,(&Debug,"\n"));

		/* Seen - do not delete after read */
		clearit(entry->entry_flags,LAST_READ_purge);
		
		/* Format:

		   filename ino time
		*/

		if (isoff(entry->entry_flags,LAST_READ_delete)) {

		    unsigned entry_flags = 0;
		    
		    ino_t     st_ino     = 0;
		    time_t    last_read  = 0;

		    char comment[SLEN];
		    int comment_len = 0;
		    
		    /* space or TAB */
		    while (whitespace(c))
			c = fgetc(f);

		    /* - means no inode */
		    
		    if ('-' == c) {
			
			c = fgetc(f);
			if ('0' <= c && c <= '9') {

			    DPRINT(Debug,14,(&Debug,
					     "parse_last_read_cache: %s: %d: %Q: %c is not allowed (negative inode)\n",
					     filename,lineno,entry_name,c));
			    
			    if (suggest_rewrite)
				(*suggest_rewrite)++;
			    
			    goto skip_line;

			}
  
			
		    } else if ('0' <= c && c <= '9') {
			
			while ('0' <= c && c <= '9') {

			    /* st_ino assumed to be unsigned */
			    
			    if (st_ino >= MAX_ino/10 &&
				MAX_ino > 0) {
				DPRINT(Debug,14,(&Debug,
					     "parse_last_read_cache: %s: %d: %Q: possible ino overflow\n",
					     filename,lineno,entry_name));
			    
				if (suggest_rewrite)
				    (*suggest_rewrite)++;
				
				goto skip_line;	
			    }
			    
			    st_ino = st_ino * 10 + (c - '0');
			    c = fgetc(f);	    
			}

			setit(entry_flags,LAST_READ_have_st_ino);
			
		    } else if (!c) {
			DPRINT(Debug,14,(&Debug,
					 "parse_last_read_cache: %s: %d: %Q: NUL not allowed\n",
					 filename,lineno,entry_name));
			
			if (!corrupt_note) {
			    
			    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileCorrupted,
					      "File %s is corrupted."),
			      filename);
			    
			    if (errors)
				(* errors)++;
			    
			    corrupt_note = 1;
			}

			goto skip_line;
		    } else {
			DPRINT(Debug,14,(&Debug,
					 "parse_last_read_cache: %s: %d: %Q: %c is not allowed\n",
					 filename,lineno,entry_name,c));

			if (suggest_rewrite)
			    (*suggest_rewrite)++;
			
			goto skip_line;
		    }

		    /* then time - must be positive */

		    if (! whitespace(c)) {
			DPRINT(Debug,14,(&Debug,
					 "parse_last_read_cache: %s: %d: %Q: whitespace expected\n",
					 filename,lineno,entry_name));

			if (suggest_rewrite)
			    (*suggest_rewrite)++;
			
			goto skip_line;	
		    }

		    while (EOF != c && whitespace(c))
			c = fgetc(f);

		    while ('0' <= c && c <= '9') {
			if (last_read >= time_MAX/10) {
			    DPRINT(Debug,14,(&Debug,
					     "parse_last_read_cache: %s: %d: %Q: possible time overflow\n",
					     filename,lineno,entry_name));
			    
			    if (suggest_rewrite)
				(*suggest_rewrite)++;
			    
			    goto skip_line;	
			}

			last_read = last_read * 10 + (c - '0');
			c = fgetc(f);			
		    }
		    
		    /* space or TAB */
		    
		    while (whitespace(c))
			c = fgetc(f);
		    
		    if ('#' == c) {
		        /* Ignore comment (#) on end of line */
		    
		        while (EOF != c && '\n' != c) {
			    c = fgetc(f);

			    if (comment_len+1 < sizeof comment) {
				comment[comment_len++] = c;
			    }
			    
			}
		    }
		    
		    if ('\n' == c) {

			int entry_same_file = 0;
			
			setit(entry_flags,LAST_READ_valid);
			setit(entry_flags,LAST_READ_old);
		       	
			if (ison(entry->entry_flags,LAST_READ_valid)) {
			    entry_same_file =
				( isoff(entry->entry_flags,LAST_READ_have_st_ino) &&
				  isoff(entry_flags,LAST_READ_have_st_ino) )
				||
				( ison(entry->entry_flags,LAST_READ_have_st_ino) &&
				  ison(entry_flags,LAST_READ_have_st_ino) &&
				  st_ino == entry->st_ino );			    
			}

			if ((time_t) -1 != now && last_read > now) {
			    DPRINT(Debug,14,(&Debug,
					    "parse_last_read_cache: %s: %d: %Q: %zu -- timestamp in future, last_read = %d > now = %d\n",
					    filename,lineno,entry_name, idx,
					    last_read,now));
			    
		        } else if (ison(entry->entry_flags,LAST_READ_old) ||
				   isoff(entry->entry_flags,LAST_READ_valid) ||
				   
				   ( entry_same_file &&
				     entry->last_read < last_read) ||
				   
				   ( isoff(entry->entry_flags,LAST_READ_modified) &&
				     entry->last_read < last_read)
				   ) {
			    
			    /* Update from disk */
			    
			    entry->entry_flags = entry_flags |
				(  entry->entry_flags &
				   LAST_READ_flag );

			    DPRINT(Debug,14,(&Debug,
					     "parse_last_read_cache: %s: %d: %Q: %zu -- %s",
					     filename,lineno,entry_name, idx,
					     ison(entry->entry_flags,LAST_READ_valid) ?
					     "updating" : "setting"));

			    if (ison(entry->entry_flags,LAST_READ_valid)) {
				if (entry->last_read != last_read) {
				    DPRINT(Debug,14,(&Debug, " last read = %d -> %d",
						     entry->last_read,last_read));
				}
				
				if (ison(entry_flags,LAST_READ_have_st_ino) &&
				    entry->st_ino != st_ino) {
				    DPRINT(Debug,14,(&Debug, " st_ino = %lu -> %lu",
						     (unsigned long)entry->st_ino,
						     (unsigned long)st_ino));
				}
			    } else {
				DPRINT(Debug,14,(&Debug, " last read = %d",
						 last_read));

				if (ison(entry_flags,LAST_READ_have_st_ino)) {
				    DPRINT(Debug,14,(&Debug, " st_ino = %lu",
						     (unsigned long)st_ino));
				}
			    }

			    if (comment_len > 0) {
				comment[comment_len] = '\0';

				DPRINT(Debug,14,(&Debug, " comment = %.*s",
						 comment_len,comment));
			    }
			    
			    DPRINT(Debug,14,(&Debug,"\n"));
			    
			    entry->last_read = last_read;
			    entry->st_ino    = st_ino;
			    
			} else {
			    
			    DPRINT(Debug,14,(&Debug,
					     "parse_last_read_cache: %s: %d: %Q: %zu -- skipping update from disk, %s\n",
					     filename,lineno,entry_name, idx,
					     entry_same_file ? "is same file" : "not same file"));
			}
				
			/* lineno is incremented later */
		    } else {
			DPRINT(Debug,14,(&Debug,
					 "parse_last_read_cache: %s: %d: %Q: newline expected\n",
					 filename,lineno,entry_name));
			
			if (suggest_rewrite)
			    (*suggest_rewrite)++;
			
			goto skip_line;	
		    }
		    		
		} else {
		    DPRINT(Debug,14,(&Debug,
				     "parse_last_read_cache:  %s: %d: %Q: Skipping line\n",
				     filename,lineno,entry_name));

		skip_line:
		    
		    while (EOF != c && '\n' != c)
			c = fgetc(f);

		}
		
		/* Decrements refcount */
		free_last_read_entry(& res.last_read_entry);
		

	    } else {
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: %Q: search_sort_list_item did not set last_read_entry\n",
				 filename,lineno,entry_name));
		goto fail_line;
	    }
			    
	} else {
	    
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: %s: %d: %Q: search_sort_list_item failed to create item\n",
			     filename,lineno,entry_name));
			     
	fail_line:
	    if (suggest_rewrite)
		(*suggest_rewrite)++;

	end_line:
	    while (EOF != c && '\n' != c)
		c = fgetc(f);
	}
	    
	switch (c) {
	case EOF:
	    
	    if (feof(f)) {		
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: Unexpect end of file\n",
				 filename,lineno));
	    }
	    
	    if (ferror(f)) {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,14,(&Debug,
				 "parse_last_read_cache: %s: %d: %s\n",
				 filename,lineno,strerror(err)));		    
	    }
	    break;
	    
	case '\n':
	    lineno++;
	    break;
	}	    
    }
    
    if (entry_name) {
	free(entry_name);
	entry_name = NULL;
    }
    entry_name_alloced = 0;

    if (cache_file) {

	if (! stat_last_read_cache(filename,f,cache_file)) {
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: stat %s failed\n",
			     filename));
	    ret = 0;
	    goto fail;
	}	
    }

    cache_len = sort_list_len(* cache_entries);
    
    if (feof(f)) {
	
	DPRINT(Debug,14,(&Debug,
			 "parse_last_read_cache: %s: %zu items\n",
			 filename,cache_len));

	if (cache_len > 0) {
	    size_t i = cache_len -1;
	    size_t deleted  = 0;
	    size_t valid    = 0;
	    size_t purge    = 0;
	    size_t modified = 0;
	    
	    while (1) {
		union sort_item res;

		res.last_read_entry = NULL;
		
		/* Increments refcount */
		get_sort_list_item(* cache_entries,sort_list_get_normal,i,
				   &res);
		
		if (res.last_read_entry) {
		    struct last_read_entry * entry;
		    
		    if (LAST_READ_ENTRY_magic != res.last_read_entry->magic)
			panic("MBX PANIC",__FILE__,__LINE__,
			      "parse_last_read_cache",
			      "Bad magic type (last_read_entry)",0);
		    
		    entry = res.last_read_entry;

		    if (ison(entry->entry_flags,LAST_READ_purge))
			purge++;
		    if (ison(entry->entry_flags,LAST_READ_valid))
			valid++;
		    if (ison(entry->entry_flags,LAST_READ_modified))
			modified++;

		    if (isoff(entry->entry_flags,LAST_READ_valid) ||
			(ison(entry->entry_flags,LAST_READ_purge) &&
			 isoff(entry->entry_flags,LAST_READ_modified))) {

			free_last_read_entry(& res.last_read_entry);

			/* Increments refcount */
			get_sort_list_item(* cache_entries,sort_list_get_remove,i,
					   &res);

			if (res.last_read_entry) {
			    struct last_read_entry * entry;
			    
			    if (LAST_READ_ENTRY_magic != res.last_read_entry->magic)
				panic("MBX PANIC",__FILE__,__LINE__,
				      "parse_last_read_cache",
				      "Bad magic type (last_read_entry)",0);

			    entry = res.last_read_entry;
			    
			    DPRINT(Debug,14,(&Debug,
					     "parse_last_read_cache: #%zu folder_sys=%s - deleted\n",
					     i,entry->folder_sys));
			    
			    
			    deleted++;
			}


			
		    } else
			clearit(entry->entry_flags,LAST_READ_purge);
		    
		    if (res.last_read_entry) 
			free_last_read_entry(& res.last_read_entry);
		}	   

		if (i > 0)
		    i--;
		else
		    break;
	    }
	    
	    DPRINT(Debug,14,(&Debug,
			     "parse_last_read_cache: %s: %zu valid, %zu flagged purge, %zu modified items - %zu deleted.\n",
			     filename,valid,purge,modified,deleted));	    
	}
	
	ret = 1;
    } else {
	DPRINT(Debug,14,(&Debug,
			 "parse_last_read_cache: %s: All items not readed\n",
			 filename));
    }

 fail:
    DPRINT(Debug,14,(&Debug,
		     "parse_last_read_cache=%d\n",
		     ret));
    
    return ret;
}



static enum quote_name_res {
    quote_name_bad = -1   /* Bad name */,
    quote_name_no  =  0   /* do not quote */,
    quote_name_needed

} quote_name P_((const char *folder_sys));
static enum quote_name_res  quote_name(folder_sys)
     const char *folder_sys;
{
    const char * c;

    enum quote_name_res ret = quote_name_no;
    
    
    if (! folder_sys[0])
	return quote_name_bad;

    if ('#' == folder_sys[0])
	ret = quote_name_needed;
    
    for (c = folder_sys; *c; c++) {

	switch (*c) {

	case '\n':
	    return quote_name_bad;

	case '"':
	case '\\':
	    ret = quote_name_needed;
	    break;
	   	    
	case '\r':
	    return quote_name_bad;

	case '/':

	    /* / is only allowed on absolute paths */
	    
	    if ('/' != folder_sys[0])
		return quote_name_bad;

	    break;
	    
	default:
	    if (whitespace(*c))
		ret = quote_name_needed;
		
	    break;	    
	}

	if (c - folder_sys >= NAME_LIMIT)
	    return quote_name_bad;
    }
        
    return ret;
}


/* This is written to temporary file */
static int dump_last_read_cache P_((FILE *f,
				    const char          * filename /* temp file name */,
				    struct file_changes * cache_file,
				    struct sortlist     * cache_entries,
				    int                 * errno_res,

				    /* May be NULL */
				    FILE *commentfile, 
				    const char *actor,
				    char *version_buff
				    )); 
static int dump_last_read_cache(f,filename,cache_file,cache_entries,errno_res,
				commentfile,actor,version_buff)
     FILE *f;
     const char          * filename /* temp file name */;
     struct file_changes * cache_file;
     struct sortlist     * cache_entries;
     int                 * errno_res;
     
     /* May be NULL */
     FILE                * commentfile; 
     const char          * actor;
     char                * version_buff;     
{
    int ret = 0;
    size_t cache_len = 0;
    int r;
    size_t z1;

    static char actor1[SLEN]   = "";
    static char version1[SLEN] = "";

    if (actor && version_buff) {
	strfcpy(actor1,actor,sizeof actor1);
	strfcpy(version1,version_buff,sizeof version1);
    } else if (actor1[0]) {   /* Sticky */
	actor        = &(actor1[0]);
	version_buff = &(version1[0]);
    } 
        
    if (! cache_entries) {
	ret = 0;

	goto fail;
    }
    
    cache_len = sort_list_len(cache_entries);
    
    if ((z1=fwrite(LAST_READ_HEADER,1, LAST_READ_HEADER_len,f)) <
	LAST_READ_HEADER_len) {

	if (ferror(f)) {
	    int err = errno;
	    
	    DPRINT(Debug,14,(&Debug,
			     "dump_last_read_cache: %s: fwrite failed: %s\n",
			     filename,strerror(err)));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeLastReadWriteTempFail,
			      "Write failed for last-read temp file %s: %s"),
		      filename,strerror(err));

	    if (errno_res)
		*errno_res = err;
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "dump_last_read_cache: %s: short fwrite = %zu < %zu\n",
			     filename,z1,LAST_READ_HEADER_len));
	}
	
	ret = 0;

	goto fail;
    }

    /* commentfile,actor,version_buff may be NULL */
    insert_commentfile(f,ELMLASTREAD_INFO,commentfile,
		       actor,version_buff);
    
    if (cache_len > 0) {
	size_t i;

	size_t need_delete = 0;
	
	DPRINT(Debug,14,(&Debug,
			 "dump_last_read_cache: %s: cache_len=%zu\n",
			 filename,cache_len));
	
	for (i = 0; i < cache_len; i++) {
	    
	    union sort_item res;
	    
	    res.last_read_entry = NULL;
	    
	    /* Increments refcount */
	    get_sort_list_item(cache_entries,sort_list_get_normal,i,
			       &res);
	    
	    if (res.last_read_entry) {
		struct last_read_entry * entry;
		enum quote_name_res      q;
		char * X;
		
		if (LAST_READ_ENTRY_magic != res.last_read_entry->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "dump_last_read_cache",
			  "Bad magic type (last_read_entry)",0);
		
		entry = res.last_read_entry;
		
		DPRINT(Debug,14,(&Debug,
				 "dump_last_read_cache: entry %zu = %Q",
				 i,entry->folder_sys));
		
		DPRINT(Debug,14,(&Debug,", entry_flags=%u %s",
				 entry->entry_flags,
				 give_entry_flags(entry->entry_flags)
				 ));

		if (ison(entry->entry_flags,
			 LAST_READ_delete)) {
		    need_delete++;
		    
		    DPRINT(Debug,14,(&Debug," -- delete, skipping\n"));
		    goto skip_this_entry;
		}
				
		if (isoff(entry->entry_flags,
			  LAST_READ_valid)) {
		    DPRINT(Debug,14,(&Debug," -- not valid, skipping\n"));
		    goto skip_this_entry;
		}
		
		if (ison(entry->entry_flags,
			 LAST_READ_memonly)) {
		    DPRINT(Debug,14,(&Debug," -- mem only, skipping\n"));
		    goto skip_this_entry;
		}
		
		q = quote_name(entry->folder_sys);
		DPRINT(Debug,14,(&Debug," quote name = %d",
				 q));
		switch (q) {
		case quote_name_bad:
		    DPRINT(Debug,14,(&Debug," quote_name_bad"));    break;
		case quote_name_no:
		    DPRINT(Debug,14,(&Debug," quote_name_no"));     break;
		case quote_name_needed:
		    DPRINT(Debug,14,(&Debug," quote_name_needed")); break;
		}
		
		if (q <= quote_name_bad) {
		    DPRINT(Debug,14,(&Debug,", skipping\n"));
		    setit(entry->entry_flags,LAST_READ_memonly);
		    goto skip_this_entry;
		} else if (q > quote_name_no) {
		    DPRINT(Debug,14,(&Debug,", quoting\n"));
		    
		    elm_fprintf(f,FRM("%Q"),entry->folder_sys);
		} else {
		    DPRINT(Debug,14,(&Debug,"\n"));
		    
		    fputs(entry->folder_sys,f);
		}
		
		/* Both space and TAB allowed */
		putc('\t',f);
		
		if (ison(entry->entry_flags,
			 LAST_READ_have_st_ino)) {
		    
		    fprintf(f,"%lu",(unsigned long)(entry->st_ino));
		    
		} else
		    putc('-',f);  /* Placeholder */
		
		/* Both space and TAB allowed */
		putc('\t',f);
		
		fprintf(f,"%ld",(long)(entry->last_read));
		
		X = ctime(&(entry->last_read));
		
		/* ctime() includes newline \n to result */
		if (X) {
		    fprintf(f,"\t# %s",
			    X);
		    
		} else
		    putc('\n',f);
		
	    skip_this_entry:
		free_last_read_entry(& res.last_read_entry);
		
	    }
	}


	if (ferror(f)) {
	    int err = errno;
	    DPRINT(Debug,14,(&Debug,
			     "dump_last_read_cache: %s: write failed: %s\n",
			     filename,strerror(err)));

	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeLastReadWriteTempFail,
			      "Write failed for last-read temp file %s: %s"),
		      filename,strerror(err));

	    if (errno_res)
		*errno_res = err;
	    
	    ret = 0;
	    goto fail;
	}
	
	r = fflush(f);
	switch(r) {
	    int err;
	case 0:
	    DPRINT(Debug,14,(&Debug,
			     "dump_last_read_cache: %s: written\n",
			     filename));
	    ret = 1;
	    break;
	case EOF:
	    err = errno;
	    
	    DPRINT(Debug,14,(&Debug,
			     "dump_last_read_cache: %s: flush failed: %s\n",
			     filename,strerror(err)));

	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeLastReadWriteTempFail,
			      "Write failed for last-read temp file %s: %s"),
		      filename,strerror(err));

	    if (errno_res)
		*errno_res = err;
	    
	    ret = 0;
	    goto fail;
	}

	if (cache_file) {
	    if (! stat_last_read_cache(filename,f,cache_file)) {
		DPRINT(Debug,14,(&Debug,
				 "dump_last_read_cache: stat %s failed\n",
				 filename));
		ret = 0;
		goto fail;
	    }				 
	}

	if (need_delete > 0) {

	    size_t i = cache_len -1;
	    size_t deleted  = 0;
	    size_t valid    = 0;
	    
	    DPRINT(Debug,14,(&Debug,
			     "dump_last_read_cache: %s: %zu entries marked for delete\n",
			     filename,need_delete));

	    while (1) {
		union sort_item res;

		res.last_read_entry = NULL;
		
		/* Increments refcount */
		get_sort_list_item(cache_entries,sort_list_get_normal,i,
				   &res);
		
		if (res.last_read_entry) {
		    struct last_read_entry * entry;
		    
		    if (LAST_READ_ENTRY_magic != res.last_read_entry->magic)
			panic("MBX PANIC",__FILE__,__LINE__,
			      "dump_last_read_cache",
			      "Bad magic type (last_read_entry)",0);
		    
		    entry = res.last_read_entry;

		    if (ison(entry->entry_flags,LAST_READ_valid))
			valid++;

		    if (ison(entry->entry_flags,LAST_READ_delete)) {

			free_last_read_entry(& res.last_read_entry);
			
			/* Increments refcount */
			get_sort_list_item(cache_entries,sort_list_get_remove,i,
					   &res);

			if (res.last_read_entry) {
			    struct last_read_entry * entry;
			    
			    if (LAST_READ_ENTRY_magic != res.last_read_entry->magic)
				panic("MBX PANIC",__FILE__,__LINE__,
				      "dump_last_read_cache",
				      "Bad magic type (last_read_entry)",0);

			    entry = res.last_read_entry;
			    
			    DPRINT(Debug,14,(&Debug,
					     "dump_last_read_cache: #%zu folder_sys=%s - deleted\n",
					     i,entry->folder_sys));
			    
			    
			    deleted++;
			}
		    }

		    if (res.last_read_entry) 
			free_last_read_entry(& res.last_read_entry);
		}	   
		
		if (i > 0)
		    i--;
		else
		    break;
	    }

	    DPRINT(Debug,14,(&Debug,
			     "dump_last_read_cache: %s: %zu valid, %zu deleted.\n",
			     filename,valid,deleted));
	}

    } else {
	DPRINT(Debug,14,(&Debug,
			 "dump_last_read_cache: %s: No entries\n",
			 filename
			 ));
    }

    
    
 fail:

    DPRINT(Debug,14,(&Debug,
		     "dump_last_read_cache=%d%s; filename=%s\n",
		     ret,
		     ret ? " (succeed)" : "",
		     filename));
    return ret;
}



static struct last_read_cache *malloc_last_read_cache(sys_dir,st,
						      prealloc,fchanges,
						      filename)
     const char                 * sys_dir;
     const struct stat          * st;
     size_t                       prealloc;
     const struct file_changes  * fchanges;
     const char                 * filename;
{
    struct last_read_cache * ret = 
	safe_zero_alloc(sizeof (*ret));

    ret -> magic = LAST_READ_CACHE_magic;

    ret -> refcount = 1;
    
    ret -> sys_dir = sys_dir ? safe_strdup(sys_dir) : NULL;
    ret -> cached_name = NULL;
    

    if (filename) {
	ret -> cache_file_name = safe_strdup(filename);
    } else
	ret -> cache_file_name = NULL;   /* Filled later */
	
    ret->cache_file = NULL_file_changes;

    if (fchanges)
	ret->cache_file = *fchanges;
    
    if (st)
	stat_to_file_changes(st,& (ret->cache_file));

    ret->entries = alloc_sort_list(&last_read_entry_op,
				   prealloc);
    
    ret->modified  = 0;
    ret->read      = 0;
    
    return ret;
}

enum last_read_mode {
    last_read_read_only = 0,
    last_read_check,
    last_read_read_update  /* opended for rewrite */,
    last_read_create  /* Create for locking        */
};


enum LREAD_OPEN_bits {
    LREAD_OPEN_bit_SUGGEST_REWRITE,
    LREAD_OPEN_bit_FILE_FD_LOCKED,
    LREAD_OPEN_bit_WRITTEN,
    
    LREAD_OPEN_bit_count
};

#define LREAD_OPEN_SUGGEST_REWRITE   (1 << LREAD_OPEN_bit_SUGGEST_REWRITE)
#define LREAD_OPEN_FILE_FD_LOCKED    (1 << LREAD_OPEN_bit_FILE_FD_LOCKED)
#define LREAD_OPEN_WRITTEN           (1 << LREAD_OPEN_bit_WRITTEN)


#define LAST_READ_OPEN_magic 	0xFA0E

static struct last_read_open {
    unsigned short               magic; 	/* LAST_READ_OPEN_magic */

    int    file_fd;
    FILE * fh;

    const char           * filename;
    struct dt_flags_info * locking_flags;

    enum last_read_mode    is_update;

    unsigned int  lread_open_flags :  LREAD_OPEN_bit_count;
    
} * open_last_read_cache P_((const char           * filename,
			     struct dt_flags_info * locking_flags,
			     enum last_read_mode    is_update,
			     struct file_changes  * file_information,
			     int                  * errno_res
			     )); 

static struct last_read_open * open_last_read_cache(filename,
						    locking_flags,
						    is_update,
						    file_information,
						    errno_res)
     const char           * filename;
     struct dt_flags_info * locking_flags;
     enum last_read_mode    is_update;
     struct file_changes  * file_information;
     int                  * errno_res;
{
    struct last_read_open * ret = NULL;
    
    enum FLOCKING_mode fm = FLOCKING_exclusive;
    char *          openm = "r+";
    int              accm = O_RDWR;
    mode_t         creatm = 0600;   /* Supposed to be read only
				       by user */
    
    int file_fd = -1;
    int file_fd_locked = 0;
    int loop_count = 0;
    int loop_message_printed = 0;

    FILE * f = NULL;
    
    struct stat file_fd_stat;
    struct stat filename_stat;
    
    int have_file_fd_stat = 0;
    int have_filename_stat = 0;
    int have_creat_retry   = 0;
    
    
    DPRINT(Debug,10,(&Debug,
		     "open_last_read_cache: is_update=%d",
		     is_update));

    switch (is_update) {
    case last_read_read_only:
	DPRINT(Debug,10,(&Debug," last_read_read_only"));
	fm     = FLOCKING_shared;
	openm  = "r";
	accm   = O_RDONLY;

	break;

    case last_read_check:
	DPRINT(Debug,10,(&Debug," last_read_check"));

	if (0) {
	case last_read_read_update:
	    DPRINT(Debug,10,(&Debug," last_read_read_update"));
	}
	fm = FLOCKING_exclusive;
	openm = "r+";
	accm = O_RDWR;
	break;
	    	
    case last_read_create:
	DPRINT(Debug,10,(&Debug," last_read_create"));
	fm = FLOCKING_exclusive;
	openm = "r+";
	accm = O_RDWR;
	break;
    }
    DPRINT(Debug,10,(&Debug," filename=%Q\n",
		     filename));
    
    while (-1 == file_fd) {
	enum syscall_status  r;
	enum FLOCKING_status r1;
	
	int err = can_open(filename,openm);
	int creatf = 0;

	file_fd_locked = 0;

	if (err) {
	    DPRINT(Debug,2,(&Debug,
			    "open_last_read_cache: %s: %s (can_open)\n",
			    filename,strerror(err)));
	    
	    if (EINTR == err) {
		continue;
	    }

	    if (ENOENT == err &&
		last_read_create == is_update &&
		! have_creat_retry) {

		DPRINT(Debug,10,(&Debug,
				 "open_last_read_cache: %s: Trying create file\n",
				 filename));
		
		creatf = O_CREAT|O_EXCL;		
	    } else {
		if (loop_message_printed) {
		    lib_transient(CATGETS(elm_msg_cat, MeSet,
					  MeLastReadWaitingFail,
					  "Waiting last-read file %s to be updated... Fail: %s"),
				  filename,strerror(err));
		}

		if (errno_res)
		    *errno_res = err;
		
		return NULL;
	    }
	    	    
	} else {
	    have_creat_retry = 0;
	}

	file_fd = open(filename,accm|creatf,creatm);
	
	if (-1 == file_fd) {
	    err = errno;

	    DPRINT(Debug,10,(&Debug,"open_last_read_cache: %s: %s\n",
			     filename,strerror(err)));

	     if (EINTR == err) {
		 continue;
	     }

	     if (EEXIST == err) {
		 /* Must have create failure 
		    -- retry with canopen()
		 */

		 DPRINT(Debug,10,(&Debug,
				  "open_last_read_cache: %s: Trying open file\n",
				  filename));
		 
		 have_creat_retry = 1;
		 continue;
	     }

	     if (loop_message_printed) {
		 lib_transient(CATGETS(elm_msg_cat, MeSet,
				       MeLastReadWaitingFail,
				       "Waiting last-read file %s to be updated... Fail: %s"),
			       filename,strerror(err));
	     }

	     if (errno_res)
		 *errno_res = err;
	     
	     return NULL;
	}

	r = fstat(file_fd,&file_fd_stat);
	switch (r) {
	    char *X;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: fstat %d: %s succeed: dev %lu ino %lu size %ld modified %ld",
			     file_fd,filename,
			     (unsigned long)file_fd_stat.st_dev,
			     (unsigned long)file_fd_stat.st_ino,
			     (long)file_fd_stat.st_size,
			     (long)file_fd_stat.st_mtime));

	    X = ctime(& (file_fd_stat.st_mtime));
	    if (X) { /* ctime() includes newline */
		DPRINT(Debug,14,(&Debug," -- %s",X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }

	    have_file_fd_stat = 1;
	    
	    break;
	case syscall_error:
	    err = errno;

	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: fstat %d: %s failed: %s\n",
			     file_fd,filename,strerror(err)));

	    if (EINTR == err) {
		goto oops;
	    }
	    
	    /* Just read old file ? */
	    goto quit_locking_loop;
	}

	/* <last_read> is locked with exclusive lock
	   when new file <last_read>.N is written
	   and then renamed to <last_read>

	*/

	r1 = filelock_fd(file_fd,fm,
			 locking_flags,
			 filename,
			 FLOCKING_non_block,
			 &err);

	switch (r1) {
	case FLOCKING_FAIL:
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: filelock_fd %d: %s locking failed: %s\n",
			     file_fd,filename,strerror(err)));

	    if (errno_res)
		*errno_res = err;
	    
		/* Just read old file ? */
	    goto quit_locking_loop;
	case FLOCKING_OK:
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: filelock_fd %d: %s locked\n",
			     file_fd,filename));
	    file_fd_locked = 1;
	    break;
	case FLOCKING_RETRY:
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: filelock_fd %d: %s locking failed (retry)",
			     file_fd,filename));
	    if (err) {
		DPRINT(Debug,14,(&Debug,": %s",
				 strerror(err)));
	    }
	    DPRINT(Debug,14,(&Debug,"\n"));
	    
	    if (EINTR == err) {
		goto oops;
	    }
	    	    
	    goto wait_to_change;
	case FLOCKING_UNSUPPORTED:
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: filelock_fd %d: %s locking not supported\n",
			     file_fd,filename));
	    /* Just read old file ? */
	    goto quit_locking_loop;
	}

	r = stat(filename,&filename_stat);
	switch (r) {
	    char *X;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: stat %s succeed: dev %lu ino %lu size %ld modified %ld",
			     filename,
			     (unsigned long)filename_stat.st_dev,
			     (unsigned long)filename_stat.st_ino,
			     (long)filename_stat.st_size,
			     (long)filename_stat.st_mtime));
	    X = ctime(& (filename_stat.st_mtime));
	    if (X) { /* ctime() includes newline */
		DPRINT(Debug,14,(&Debug," -- %s",X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }
	    
	    have_filename_stat = 1;
	    
	    break;
	case syscall_error:
	    err = errno;
	    
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: stat %s: %s\n",
			     filename,strerror(err)));
	    if (EINTR == err) {
		goto oops;
	    }
	    
	    /* Just read old file ? */
	    goto quit_locking_loop;
	}
	
	if (!have_filename_stat || !have_file_fd_stat) {

	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: file %s - no stat?\n",
			     filename));

	    goto quit_locking_loop;
	    
	} else if (filename_stat.st_dev   == file_fd_stat.st_dev  &&
		   filename_stat.st_ino   == file_fd_stat.st_ino  &&
		   filename_stat.st_size  == file_fd_stat.st_size &&
		   filename_stat.st_mtime == file_fd_stat.st_mtime) {
	    
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: file %s not changed (since open)\n",
			     filename));

	    if (loop_message_printed) {
		lib_transient(CATGETS(elm_msg_cat, MeSet,
				      MeLastReadWaitingOK,
				      "Waiting last-read file %s to be updated... OK"),
			      filename);
		loop_message_printed = 0;
	    }
	    
	    goto quit_locking_loop;

	} else {
	    int wait_it;
	    
	    DPRINT(Debug,14,(&Debug,
			     "open_last_read_cache: file %s changed after open\n",
			     filename));
	    
	oops:
	    wait_it = 0;
	    
	    if (0) {
	    wait_to_change:
		wait_it = 1;
	    }
	    
	    if (loop_count++ > 10) {
		DPRINT(Debug,14,(&Debug,
				 "open_last_read_cache: file %s - try #%d -- quiting\n",
				 filename,loop_count));
		goto quit_locking_loop;
	    }

	    if (wait_it) {
		DPRINT(Debug,14,(&Debug,
				 "open_last_read_cache: file %s - try #%d -- waiting\n",
				 filename,loop_count));

		if (!loop_message_printed) {
		    lib_transient(CATGETS(elm_msg_cat, MeSet,
					  MeLastReadWaiting,
					  "Waiting last-read file %s to be updated..."),
				  filename);
		    loop_message_printed = 1;		    
		}

		if (POLL_method)
		    wait_for_timeout(5);
		else
		    sleep(5);
		
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "open_last_read_cache: file %s - try #%d -- looping\n",
				 filename,loop_count));
	    }

	    if (file_fd_locked) { /* Just ignore error --
				     close() should release this anyway
				  */
		filelock_fd(file_fd,FLOCKING_release,locking_flags,
			    filename,FLOCKING_non_block,NULL);
	    }
	    
	    close(file_fd);  /* Result ignored */
	    file_fd = -1;	    
	}	
    }

 quit_locking_loop:

    if (loop_message_printed) {
	lib_transient(CATGETS(elm_msg_cat, MeSet,
			      MeLastReadWaitingError,
			      "Waiting last-read file %s to be updated... Error"),
		      filename);
    }			      

    if (file_information) {
	if (FILE_CHANGES_magic != file_information->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"open_last_read_cache",
		  "Bad magic number (file_changes)",0);

	if (sizeof (*file_information) != file_information->self_size) {
	    
	    DPRINT(Debug,1,(&Debug,
			    "open_last_read_cache: size mismatch file_changes size %zu != self size %zu\n",
			    sizeof (*file_information),file_information->self_size));
	    	    
	} else {
	    stat_to_file_changes(have_file_fd_stat ? &file_fd_stat : NULL,
				 file_information);
	}
    }

    f = fdopen(file_fd,openm);
    if (!f) {
 	int err = errno;
	DPRINT(Debug,10,(&Debug,
			 "open_last_read_cache: fdopen %d: %s: %s\n",
			 file_fd,filename,strerror(err)));
	
	if (file_fd_locked) { /* Just ignore error --
				 close() should release this anyway
			      */
	    filelock_fd(file_fd,FLOCKING_release,locking_flags,
			filename,FLOCKING_non_block,NULL);
	}
	
	close(file_fd);  /* Result ignored */

	if (errno_res)
	    *errno_res = err;
	
	return NULL;
    }


    ret = safe_zero_alloc(sizeof (*ret));

    ret->magic    = LAST_READ_OPEN_magic;
    ret->file_fd  = file_fd;
    ret->fh       = f;
    ret->filename = filename;   /* shared */
    ret->locking_flags = locking_flags;
    ret->is_update     = is_update;

#define IFSET(var,flag) (var ? flag : 0)
    
    ret->lread_open_flags =
	IFSET(file_fd_locked,LREAD_OPEN_FILE_FD_LOCKED);
   
    DPRINT(Debug,10,(&Debug,
		     "open_last_read_cache: %s open%s\n",
		     filename,
		     file_fd_locked ? ", file_fd locked" : ""));

    return ret;
}

static void close_last_read_open P_((struct last_read_open *ptr));

/* Closes ptr->fh if not update */
static int read_last_read_cache P_((struct last_read_open *ptr,
				    struct file_changes * cache_file,
				    struct sortlist    ** cache_entries,
				    int                 * errors,
				    const struct last_read_entry_def * entry_default));
static int read_last_read_cache(ptr,cache_file,cache_entries,errors,entry_default)
     struct last_read_open *ptr;
     struct file_changes * cache_file;
     struct sortlist    ** cache_entries;
     int                 * errors;
     const struct last_read_entry_def * entry_default;
{
    int ret = 0;
    int suggest_rewrite = 0;

    
    if (LAST_READ_OPEN_magic != ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "read_last_read_cache",
	      "Bad magic number (last_read_open)",0);

    if (! ptr->fh) {
	ret = 0;
	goto fail;
    }
	
    rewind(ptr->fh);

    ret = parse_last_read_cache(ptr->filename,
				ptr->fh,
				cache_file,
				cache_entries,
				errors,
				entry_default,
				&suggest_rewrite);

    if (ptr -> is_update < last_read_check ||
	(last_read_check == ptr -> is_update &&
	 ! suggest_rewrite)) {
	close_last_read_open(ptr);	
    }

    if (suggest_rewrite)
	setit(ptr->lread_open_flags,
	      LREAD_OPEN_SUGGEST_REWRITE);
    
 fail:

    DPRINT(Debug,14,(&Debug,
		     "read_last_read_cache=%d; filename=%s\n",
		     ret,ptr->filename));

    
    return ret;
    
}
					  	
static void close_last_read_open(ptr)
    struct last_read_open *ptr;
{
    if ( LAST_READ_OPEN_magic != ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "close_last_read_entry",
	      "Bad magic number (last_read_open)",0);
    
    if (-1 != ptr->file_fd) {
	if (ison(ptr->lread_open_flags,
		 LREAD_OPEN_FILE_FD_LOCKED) &&
	    ptr->locking_flags) {
	    /* Just ignore error --
	       close() should release this anyway
	    */
	    
	    filelock_fd(ptr->file_fd,FLOCKING_release,
			ptr->locking_flags,
			ptr->filename,
			FLOCKING_non_block,NULL);

	    clearit(ptr->lread_open_flags,
		    LREAD_OPEN_FILE_FD_LOCKED);
	}
	
	if (! ptr->fh ||
	    fileno(ptr->fh) != ptr->file_fd) {

	    close(ptr->file_fd);
	}
	ptr->file_fd = -1;
    }

    if (ptr->fh) {
	int r = fclose(ptr->fh);
	switch (r) {
	    int err UNUSED_VAROK;
	case 0:
	    DPRINT(Debug,14,(&Debug,
			     "close_last_read_open: %s: closed\n",
			     ptr->filename));
	    break;
			
	case EOF:
	    err = errno;
	    DPRINT(Debug,14,(&Debug,
			     "close_last_read_open: %s: close failed: %s\n",
			     ptr->filename,strerror(err)));
	    break;
	}
	ptr->fh = NULL;
    }

}
	
static void free_last_read_open P_((struct last_read_open **ptr)); 
static void free_last_read_open(ptr)
     struct last_read_open **ptr;
{
    if (LAST_READ_OPEN_magic != (*ptr)->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "free_last_read_open",
		  "Bad magic number (last_read_open)",0);

    if ((*ptr)->filename) {
	DPRINT(Debug,14,(&Debug,
			 "free_last_read_open: Freeing %s\n",
			 (*ptr)->filename));
    }

    
    close_last_read_open(*ptr);

    (*ptr)->magic = 0; /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

    
/* Ignores / on end */
int same_dir_str P_((const char * dir1, const char * dir2));
int same_dir_str(dir1,dir2)
     const char * dir1;
     const char * dir2;
{
    int r = 0;
    
    size_t l1 = strlen(dir1);
    size_t l2 = strlen(dir2);

    if (l1 > 1 && '/' == dir1[l1-1])
	l1--;
    if (l2 > 1 && '/' == dir2[l2-1])
	l2--;
	
    if (l1 == l2) {
	r = 0 == strncmp(dir1,dir2,l1);
	
    } else {
	r = 0;
    }
	
    return r;
}

static int same_dir_stat P_((const char  * dir1,
		      struct stat * buf_dir1,
		      const char  * dir2));
static int same_dir_stat(dir1,buf_dir1,dir2)
     const char * dir1;
     struct stat * buf_dir1;
     const char * dir2;
{
    struct stat buf2;
    enum syscall_status  r = stat(dir2, &buf2);
    
    switch (r) {
    case syscall_error /* -1 */: {
	int err UNUSED_VAROK = errno;

	DPRINT(Debug,10,(&Debug,
			 "same_dir_stat: stat %s failed %s (errno=%d)\n",
			 dir2,strerror(err), err));
		
    }
	break;
    case syscall_success /* 0 */:
	
	DPRINT(Debug,10,(&Debug,
			 "same_dir_stat: stat %s succeed\n",
			 dir2));
	
	if (buf_dir1->st_ino == buf2.st_ino && buf_dir1->st_dev == buf2.st_dev) {

	    DPRINT(Debug,10,(&Debug,
			     "same_dir_stat: Same dir %s and %s\n",
			     dir1,dir2));
	    return 1;
	}
	break;
    }
    
    return 0;
}

/* Is sys_dir is relative, may return read_loc_none */

static enum last_read_location {
    read_loc_none = 0,
    read_loc_dir,
    read_loc_def } last_read_dir_to_loc P_((const char * sys_dir,
					    const char **res_dir));

static enum last_read_location last_read_dir_to_loc(sys_dir,
						    res_dir)
     const char * sys_dir;
     const char **res_dir;
{
    enum last_read_location ret = read_loc_none;

    enum syscall_status  r_sys_dir = syscall_error;
    struct stat        buf_sys_dir;
    
    /* give_dt_estr_as_str adds / to end */
    const char * mailhome_val =
	give_dt_estr_as_str(&mailhome_dir_e,
			    "mailhome-dir",
			    NULL,NULL);   /* XXXX */

    const char * mbx_dir =
	give_dt_estr_as_str(&extra_mailbox_dir_e,
			    "extra-mailbox-dir",
			    NULL,NULL);
    
    const char * folder_dir =
	give_dt_estr_as_str(&folders_e,
			    "maildir",
			    NULL,NULL);
    
    if (res_dir)
	*res_dir = NULL;
    
    if (mailhome_val) {
	if (same_dir_str(mailhome_val,sys_dir)) {
	    ret =  read_loc_def;
	    if (res_dir)
		*res_dir = mailhome_val;
	}
    }

    if (read_loc_none == ret && mbx_dir) {	
	if (same_dir_str(mbx_dir,sys_dir)) {
	    ret =  read_loc_dir;
	    if (res_dir)
		*res_dir = mbx_dir;
	}	
    }

    if (read_loc_none == ret && folder_dir) {	
	if (same_dir_str(folder_dir,sys_dir)) {
	    ret =  read_loc_dir;
	    if (res_dir)
		*res_dir = folder_dir;
	}	
    }

    if (read_loc_none == ret) {	
	if (same_dir_str(home,sys_dir)) {
	    ret =  read_loc_dir;
	    if (res_dir)
		*res_dir = home;
	}	
    }
    
    if (read_loc_none == ret) {
	char * pwd_val = getenv("PWD");
	
	r_sys_dir = stat(sys_dir,&buf_sys_dir);

	switch (r_sys_dir) {
	case syscall_error /* -1 */: {
	    int err UNUSED_VAROK = errno;
	    
	    DPRINT(Debug,10,(&Debug,
			     "last_read_dir_to_loc: stat %s failed %s (errno=%d)\n",
			     sys_dir,strerror(err), err));
	}
	    break;
	case syscall_success /* 0 */:
	    DPRINT(Debug,10,(&Debug,
			     "last_read_dir_to_loc: stat %s succeed\n",
			     sys_dir));

	    if (mailhome_val) {
		if (same_dir_stat(sys_dir,&buf_sys_dir,
				  mailhome_val)) {
		    
		    ret =  read_loc_def;
		    if (res_dir)
			*res_dir = mailhome_val;
		    
		}	    
	    }

	    if (read_loc_none == ret && mbx_dir) {	
		if (same_dir_stat(sys_dir,&buf_sys_dir,
				  mbx_dir)) {
		    ret =  read_loc_dir;
		    if (res_dir)
			*res_dir = mbx_dir;
		}	
	    }
	    
	    if (read_loc_none == ret && folder_dir) {
		if (same_dir_stat(sys_dir,&buf_sys_dir,
				  folder_dir)) {
		    ret =  read_loc_dir;
		    if (res_dir)
			*res_dir = folder_dir;
		}	
	    }
	    
	    if (read_loc_none == ret && folder_dir) {
		if (same_dir_stat(sys_dir,&buf_sys_dir,
				  folder_dir)) {
		    ret =  read_loc_dir;
		    if (res_dir)
			*res_dir = folder_dir;
		}	
	    }
	    
	    if (read_loc_none == ret) {
		if (same_dir_stat(sys_dir,&buf_sys_dir,
				  home)) {
		    ret =  read_loc_dir;
		    if (res_dir)
			*res_dir = home;
		}	
	    }
	    
	    if (read_loc_none == ret && pwd_val && '/' == pwd_val[0]) {
		
		if (same_dir_stat(sys_dir,&buf_sys_dir,
				  pwd_val)) {
		    ret =  read_loc_dir;
		    if (res_dir)
			*res_dir = pwd_val;
		}	
	    }

	    break;
	}

	if (read_loc_none == ret && '/' == sys_dir[0]) {
	    ret =  read_loc_dir;
	    if (res_dir)
		*res_dir = sys_dir;
	}
    }

    
    DPRINT(Debug,10,(&Debug,
		     "last_read_dir_to_loc=%d",
		     ret));

    switch (ret) {
    case read_loc_none: DPRINT(Debug,10,(&Debug, " read_loc_none")); break;
    case read_loc_dir:  DPRINT(Debug,10,(&Debug, " read_loc_dir"));  break;
    case read_loc_def:  DPRINT(Debug,10,(&Debug, " read_loc_def")); break;
    }
    DPRINT(Debug,10,(&Debug," sys_dir=%Q",sys_dir));
    if (res_dir && *res_dir) {
	DPRINT(Debug,10,(&Debug,"; *res_dir=%Q",*res_dir));
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return ret;
}

/* Replaces file */
static int write_last_read_cache P_((struct last_read_open  * H,
				     struct last_read_cache * cache,
				     struct dt_flags_info   * locking_flags,
				     int                    * errno_res,

				     /* May be NULL */
				     FILE *commentfile, 
				     const char *actor,
				     char *version_buff
				     ));
static int write_last_read_cache(H,cache,locking_flags,errno_res,
				 commentfile,actor,version_buff
				 )
     struct last_read_open  * H;
     struct last_read_cache * cache;
     struct dt_flags_info   * locking_flags;
     int                    * errno_res;

     /* May be NULL */
     FILE                * commentfile; 
     const char          * actor;
     char                * version_buff;     
     
{
    int ret = 0;
    
    if (LAST_READ_OPEN_magic != H->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "write_last_read_cache",
	      "Bad magic number (last_read_open)",0);

    if (LAST_READ_CACHE_magic != cache->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"write_last_read_cache",
              "Bad magic type (last_read_cache)",0);

    
    if (H->filename) {  /* shared pointer */
        char          * temp_filename = NULL;
	int             unlink_temp   = 0;
	
	int             temp_file_fd  = -1;
	int             file_fd_locked = 0;

	FILE          * temp_fh       = NULL;
	
	int err;
	enum FLOCKING_status r2;
		

	temp_filename =
	    elm_message(FRM("%s.N"),
			H->filename);

	err = can_open(temp_filename,"r+");

	if (err) {
	    DPRINT(Debug,14,(&Debug,
			     "write_last_read_cache: %s canopen: %s\n",
			     temp_filename,strerror(err)));

	    if (ENOENT != err) {
	    
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeLastReadCheckTempFail,
				  "Failed to check last-read temp file %s: %s"),
			  temp_filename,strerror(err));

		if (errno_res)
		    *errno_res = err;
		
		goto fail;
	    }
	}

	temp_file_fd = open(temp_filename,O_CREAT|O_EXCL|O_RDWR,
			    0600   /* Supposed to be read only
				      by user */);

	if (-1 == temp_file_fd) {
	    err = errno;

	    DPRINT(Debug,14,(&Debug,
			     "write_last_read_cache: %s open: %s\n",
			     temp_filename,strerror(err)));

 	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeLastReadCreatTempFail,
			      "Failed to create last-read temp file %s: %s"),
		      temp_filename,strerror(err));

	    if (errno_res)
		*errno_res = err;
	    
	    goto fail;
	}
	
	DPRINT(Debug,14,(&Debug,
			 "write_last_read_cache: %s created\n",
			 temp_filename));


	/* Lock this, because this is moved to  struct last_read_open */
	
	r2 = filelock_fd(temp_file_fd,FLOCKING_exclusive,locking_flags,
			 temp_filename,FLOCKING_non_block,
			 &err);

	switch (r2) {
	case FLOCKING_FAIL:
	    DPRINT(Debug,14,(&Debug,
			     "write_last_read_cache: filelock_fd %d: %s locking failed: %s\n",
			     temp_file_fd, temp_filename,strerror(err)));

	    if (errno_res)
		*errno_res = err;
	    
	    unlink_temp = 1;
	    goto fail;
	case FLOCKING_OK:
	    DPRINT(Debug,14,(&Debug,
			     "write_last_read_cache: filelock_fd %d: %s locked\n",
			     temp_file_fd, temp_filename));
	    file_fd_locked = 1;
	    break;
	case FLOCKING_RETRY:
	    DPRINT(Debug,14,(&Debug,
			     "write_last_read_cache: filelock_fd %d: %s locking failed (retry)",
			     temp_file_fd, temp_filename));
	    if (err) {
		DPRINT(Debug,14,(&Debug,": %s",
				 strerror(err)));
	    }
	    DPRINT(Debug,14,(&Debug,"\n"));

	    if (errno_res)
		*errno_res = err;
	    
	    unlink_temp = 1;
	    goto fail;
	case FLOCKING_UNSUPPORTED:
	    DPRINT(Debug,14,(&Debug,
			     "write_last_read_cache: filelock_fd %d: %s locking not supported\n",
			     temp_file_fd, temp_filename));
	    break;
	}

	temp_fh = fdopen(temp_file_fd,"w+");
	if (temp_fh) {
	    struct file_changes      cache_file = cache->cache_file;

	    if (dump_last_read_cache(temp_fh,
				     temp_filename,
				     &cache_file,
				     cache->entries,
				     errno_res,

				     /* commentfile,actor,version_buff
					may be NULL
				     */
				     commentfile,actor,version_buff
				     )) {
		enum syscall_status  r_rename =
		    rename(temp_filename,
			   H->filename);

		switch (r_rename) {
		case syscall_success:
		    DPRINT(Debug,14,(&Debug,
				     "write_last_read_cache: %s renamed to %s\n",
				     temp_filename,
				     H->filename));
			   
		    close_last_read_open(H);
		    
		    cache->cache_file_name = strmcpy(cache->cache_file_name,
						     H->filename);
		    cache->cache_file = cache_file;
		    cache->modified   = 0;
		    
		    H->filename = cache->cache_file_name;
		    H->file_fd  = temp_file_fd; temp_file_fd = -1;
		    H->fh       = 	temp_fh; temp_fh = NULL;
		    H->lread_open_flags =
			LREAD_OPEN_WRITTEN |
			(file_fd_locked ? LREAD_OPEN_FILE_FD_LOCKED : 0);

		    ret = 1;


		    
		    break;

		case syscall_error /* -1 */:
		    err = errno;

		    DPRINT(Debug,14,(&Debug,
				     "write_last_read_cache: %s rename to %s failed: %s\n",
				     temp_filename,
				     H->filename,
				     strerror(err)));

		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeLastReadRenameFail,
				      "Failed to rename last-read file to %s: %s"),
			      H->filename,strerror(err));

		    if (errno_res)
			*errno_res = err;
		    
		    unlink_temp = 1;
		    break;
		}
	    } else {
		
		DPRINT(Debug,14,(&Debug,
				 "write_last_read_cache: Failed to write last-read temp file %s\n",
				 temp_filename));

		unlink_temp = 1;
	    }
		
	} else {
	    err = errno;
	    
	    DPRINT(Debug,14,(&Debug,
			     "write_last_read_cache: fdopen %d: %s open failed: %s\n",
			     temp_file_fd, temp_filename,strerror(err)));

	    if (errno_res)
		*errno_res = err;
	    
	    unlink_temp = 1;
	}
		
    fail:
	if (temp_file_fd != -1) {
	    
	    if (temp_file_fd != H->file_fd) {
		if (file_fd_locked) {
		    /* Just ignore error --
		       close() should release this anyway
		    */
		    
		    filelock_fd(temp_file_fd,FLOCKING_release,locking_flags,
				temp_filename,FLOCKING_non_block,
				NULL);
		}
		
		if (! temp_fh || temp_file_fd != fileno(temp_fh)) {

		    unlink_temp = 1;
		    close(temp_file_fd);
		}
	    }
	    temp_file_fd = -1;
	}
	
	if (temp_fh) {
	    if (temp_fh != H->fh) {
		
		unlink_temp = 1;
		fclose(temp_fh);
	    }
	    temp_fh = NULL;
	}
	
	if (temp_filename) {
	    
	    if (unlink_temp) {
		enum syscall_status r1 = unlink(temp_filename);
		switch (r1) {
		case syscall_success:
		    DPRINT(Debug,14,(&Debug,
				     "write_last_read_cache: %s unlinked\n",
				     temp_filename));
		    break;
		case syscall_error:
		    err = errno;
		    DPRINT(Debug,14,(&Debug,
				     "write_last_read_cache: %s unlink: %s\n",
				     temp_filename,strerror(err)));

		    if (errno_res)
			*errno_res = err;
		    
		    break;
		}
	    }
	    
	    free(temp_filename);
	    temp_filename = NULL;
	}
    } else {
	DPRINT(Debug,14,(&Debug,
			 "write_last_read_cache: No filename\n"));
    }


    DPRINT(Debug,14,(&Debug,
		     "write_last_read_cache=%d%s\n",
		     ret,
		     ret ? " (succeed)" : ""));
    
    return ret;
}

static struct last_read_cache * load_last_read_cache
                P_((const char * filename,
		    const char * sys_dir /* may be NULL */,
		    struct last_read_open  **hdl,
		    int *errors,
		    struct dt_flags_info * locking_flags,
		    enum last_read_mode    is_update
		    ));

static void reload_last_read_cache P_((struct last_read_cache *cache,
				       struct last_read_open  **hdl,
				       int *errors,
				       struct dt_flags_info * locking_flags,
				       enum last_read_mode    is_update,
				       int                  * errno_res));

static void reload_last_read_cache(cache,
				   hdl,errors,locking_flags,
				   is_update,errno_res)
     struct last_read_cache *cache;
     struct last_read_open  **hdl;
     int *errors;
     struct dt_flags_info * locking_flags;
     enum last_read_mode    is_update;
     int                  * errno_res;
{

    const char * filename                 = NULL;

    struct last_read_open  * H = NULL;

    struct file_changes    fchanges = NULL_file_changes;

    if (LAST_READ_CACHE_magic != cache->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"reload_last_read_cache",
              "Bad magic type (last_read_cache)",0);

    filename = cache->cache_file_name;
        
    if (hdl && *hdl) 
	free_last_read_open(hdl);
       
    H = open_last_read_cache(filename,locking_flags,
			     is_update,
			     &fchanges,
			     errno_res);

    if (H) {

	static struct last_read_entry_def def = {
	    LAST_READ_ENTRY_DEF_magic,

	    
	    LAST_READ_old /* Is this needed ? */
	};

	if (LAST_READ_OPEN_magic != H->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "reload_last_read_cache",
		  "Bad magic number (last_read_open)",0);

	if (H->fh) {
	    
	    if (read_last_read_cache(H,
				     & (cache->cache_file),
				     & (cache->entries),
				     errors,
				     &def)) {

		/* Update file information */
		cache->cache_file = fchanges; 
		cache->read       = 1;  /* Mark that read once */
		
		DPRINT(Debug,14,(&Debug,
				 "reload_last_read_cache: %s: load succeed\n",
				 filename));
		
		if (H->fh &&
		    ison(H->lread_open_flags,
			 LREAD_OPEN_SUGGEST_REWRITE) &&
		    (is_update == last_read_read_update ||
		     !hdl)) {
		    
		    /* Write_update */
		    
		    if (write_last_read_cache(H,cache,locking_flags,
					      NULL /* errno_res */,
					      
					      /* commentfile,actor,version_buff */
					      NULL,NULL,NULL
					      )) {
			DPRINT(Debug,14,(&Debug,
					 "reload_last_read_cache: %s: updated\n",
					 filename));
		    }
		}
	    }

	    
	    if (H->fh && is_update < last_read_read_update)
		close_last_read_open(H);
	    
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "reload_last_read_cache: %s: No file handle\n",
			     filename));
	}
    
    	if (hdl) {
	    *hdl = H;
	    H = NULL;
	} else
	    free_last_read_open(&H);
    }
}
				       
/* For loading new cache */
struct last_read_cache * load_last_read_cache(filename,
					      sys_dir,
					      hdl,
					      errors,locking_flags,is_update)
     const char * filename;
     const char * sys_dir /* may be NULL */;
     struct last_read_open  **hdl;
     int *errors;
     struct dt_flags_info * locking_flags;
     enum last_read_mode    is_update;
{
    struct last_read_open  * H = NULL;
    struct last_read_cache * ret = NULL;

    struct file_changes    fchanges = NULL_file_changes;
    
    if (hdl && *hdl) 
	free_last_read_open(hdl);
    
    H = open_last_read_cache(filename,locking_flags,is_update,
			     &fchanges,
			     NULL /* errno_res */);

    if (H) {

	static struct last_read_entry_def def = {
	    LAST_READ_ENTRY_DEF_magic,

	    
	    LAST_READ_old /* Is this needed ? */
	};

	if (LAST_READ_OPEN_magic != H->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "load_last_read_cache",
		  "Bad magic number (last_read_open)",0);

	if (H->fh) {
	    int c = '\0';
	    
	    size_t prealloc = 0;
	    int last_c = '\0';  /* ignore first line
				   (header) on prealloc
				   counting
				*/
	    
	    while (EOF != (c = getc(H->fh))) {
		if ('\n' == last_c &&
		    '#'  != c && '\n' != c)
		    prealloc++;
		last_c = c;
	    }
	
	    if (ferror(H->fh)) {
		int err UNUSED_VAROK = errno;
		DPRINT(Debug,14,(&Debug,
				 "load_last_read_cache: %s: read failed: %s\n",
				 filename,strerror(err)));
	    }

	    if (feof(H->fh)) {
		
		DPRINT(Debug,14,(&Debug,
				 "load_last_read_cache: %s: prealloc %zu entries\n",
				 filename,prealloc));	       
	    }

	    rewind(H->fh);

	    ret = malloc_last_read_cache(sys_dir,NULL,prealloc,
					 &fchanges,filename);
	    
	    if (read_last_read_cache(H,
				     & (ret->cache_file),
				     & (ret->entries),
				     errors,
				     &def)) {

		/* Update file information */
		ret->cache_file = fchanges; 
		ret->read       = 1;  /* Mark that read once */
		
		DPRINT(Debug,14,(&Debug,
				 "load_last_read_cache: %s: load succeed\n",
				 filename));

		if (H->fh &&
		    ison(H->lread_open_flags,
			 LREAD_OPEN_SUGGEST_REWRITE) &&
		    (is_update == last_read_read_update ||
		     !hdl)) {
		    
		    /* Write_update */

		    if (write_last_read_cache(H,ret,locking_flags,
					      NULL /* errno_res */,

					      /* commentfile,actor,version_buff */
					      NULL,NULL,NULL
					      )) {
			DPRINT(Debug,14,(&Debug,
					 "load_last_read_cache: %s: updated\n",
					 filename));
		    }
		    
		}
	    }

	    if (H->fh && is_update < last_read_read_update)
		close_last_read_open(H);
	    
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "load_last_read_cache: %s: No file handle\n",
			     filename));
	}
    
	if (hdl) {
	    *hdl = H;
	    H = NULL;
	} else
	    free_last_read_open(&H);

    }
    
    DPRINT(Debug,14,(&Debug,
		     "load_last_read_cache"));

    if (ret) {
	DPRINT(Debug,14,(&Debug,"=%p (succeed)\n",
			 ret));
    } else {
	DPRINT(Debug,14,(&Debug,"=NULL\n"));
    }

    return ret;
}

static struct last_read_cache * last_read_def_cache    = NULL;
static int                      last_read_def_rewrite  = 0; /* not used */
static struct file_changes      last_read_def_fchanges = FILE_CHANGES_INIT;

/* write_conf() lib/misc/conf_writer.c does locking */
S_(merge_conf_map_f merge_user_last_read)
static int merge_user_last_read  P_((const char  * fname,
				     FILE        * f));
static int merge_user_last_read(fname,f)
     const char  * fname;
     FILE        * f;
{
    int r = 0;

	static struct last_read_entry_def def = {
	    LAST_READ_ENTRY_DEF_magic,

	    
	    LAST_READ_old /* Is this needed ? */
	};

    
    if (! last_read_def_cache) {
	int c = '\0';	    
	size_t prealloc = 0;
	
	while (EOF != (c = getc(f))) {
	    if ('\n' == c)
		prealloc++;
	}
	
	if (ferror(f)) {
	    int err UNUSED_VAROK = errno;
	    DPRINT(Debug,14,(&Debug,
			     "merge_user_last_read: %s: read failed: %s\n",
			     fname,strerror(err)));
	}
	
	if (feof(f)) {
	    /* This may give one line too much (header) */
	    
	    DPRINT(Debug,14,(&Debug,
			     "merge_user_last_read: %s: prealloc %zu entries\n",
			     fname,prealloc));	       
	}
	
	rewind(f);

	last_read_def_cache = malloc_last_read_cache(NULL /* sys_dir */,
						     NULL /* struct stat */,
						     prealloc,
						     &last_read_def_fchanges,
						     fname);
    }

    if (LAST_READ_CACHE_magic != last_read_def_cache->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"merge_user_last_read",
              "Bad magic type (last_read_cache)",0);

    r =  parse_last_read_cache(fname,f,&last_read_def_fchanges,
			       & (last_read_def_cache->entries),
			       NULL /* Errors */,
			       &def,
			       &last_read_def_rewrite);
				   
    DPRINT(Debug,14,(&Debug,
		     "merge_user_last_read=%d%s\n",
		     r,
		     r ? " (succeed)" : ""));
    
    return r;				   
}


/* write_conf() lib/misc/conf_writer.c does locking */
S_(dump_conf_map_f dump_user_last_read)
static int dump_user_last_read P_((FILE *f,const char *actor,
				   char *version_buff,
				   int * errno_res));
static int dump_user_last_read(f,actor,version_buff,errno_res) 
     FILE *f;
     const char *actor;
     char *version_buff;
     int * errno_res;
{
    int r = 0;
    
    if (last_read_def_cache) {

	if (LAST_READ_CACHE_magic != last_read_def_cache->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"dump_user_last_read",
		  "Bad magic type (last_read_cache)",0);

	if (last_read_def_cache->entries) {
	
	    r = dump_last_read_cache(f,user_last_read_def,
				     &last_read_def_fchanges,
				     last_read_def_cache->entries,
				     errno_res,

				     NULL /* commentfile */,
				     actor,version_buff);
	    
	    if (r) {
		DPRINT(Debug,14,(&Debug,
				 "dump_user_last_read: %s written\n",
				 user_last_read_def));
		
		last_read_def_rewrite  = 0;
	    }
	}
    }

    DPRINT(Debug,14,(&Debug,
		     "dump_user_last_read=%d%s",
		     r,
		     r ? " (succeed)" : ""));
    if (errno_res) {
	DPRINT(Debug,14,(&Debug,"; *errno_res=%d",
			 *errno_res));

	if (*errno_res) {
	    DPRINT(Debug,14,(&Debug,": %s",
			     strerror(*errno_res)));
	}
    }    
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return r;
}

S_(dump_message_f user_last_read_written)
static void user_last_read_written P_((char *fname));
static void user_last_read_written(fname)
     char *fname;
{
    lib_error(CATGETS(elm_msg_cat, MeSet, 
		      MeDefaultLastReadW,
		      "Default last read file %s written."),
	      fname);
}


/* system init hook */
void init_last_read_defaults(errors)
     int *errors;
{

    DPRINT(Debug,10,(&Debug,"init_last_read_defaults: starting\n"));

    register_conf_write(USER_LAST_READ,
			user_last_read_def,
			dump_user_last_read,
			user_last_read_written,
			&last_read_def_rewrite, /* not used */
			merge_user_last_read);

    DPRINT(Debug,10,(&Debug,"init_last_read_defaults: done\n"));
}

static struct sortlist        * last_read_dir_cache = NULL;       


/* User init hook */
void last_read_rc_file(errors)
     int *errors;
{
    enum use_lastread_v use_lastread =
	give_dt_enumerate_as_int(&use_last_read_file);
    
    if (use_lastread > use_lastread_no) {
	struct last_read_open  * last_read_hdl_cache = NULL;

	/* Also give_last_read() can set last_read_dir_cache */
	
	if (last_read_dir_cache) {
	    DPRINT(Debug,13,(&Debug,
			     "last_read_rc_file: last_read_dir_cache already set\n"));
	} else
	    last_read_dir_cache = alloc_sort_list(& last_read_cache_op,2);

	/* Also give_last_read() can set last_read_def_cache */

	if (last_read_def_cache) {

	    DPRINT(Debug,13,(&Debug,
			     "last_read_rc_file: default cache %s already set\n",
			     user_last_read_def));
	    
	} else
	    last_read_def_cache =
		load_last_read_cache(user_last_read_def,
				     NULL /* user default file */,
				     &last_read_hdl_cache,
				     errors,
				     &conf_merge_locking,
				     last_read_check
				     );

	if (last_read_hdl_cache) {

	    if (LAST_READ_OPEN_magic != last_read_hdl_cache->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "last_read_rc_file",
		      "Bad magic number (last_read_open)",0);

	    last_read_def_rewrite = ison(last_read_hdl_cache->lread_open_flags,
					 LREAD_OPEN_SUGGEST_REWRITE);
	    
	    free_last_read_open(& last_read_hdl_cache);
	}
    }
}

static char * give_cache_file_name P_((const char       * sys_dir,
				       enum last_read_location last_read_loc));
static char * give_cache_file_name(sys_dir,last_read_loc)
     const char       * sys_dir;
     enum last_read_location last_read_loc;
{
    static char * ret = NULL;

    switch (last_read_loc) {
    case read_loc_none:
	ret = NULL;
	break;
    case read_loc_dir:
	if (sys_dir) {
	    int L;

	    char temp[STRING];

	    elm_sfprintf(temp,sizeof temp,
			 FRM(".elm-last-read-%04x"),
			 userid);
	    
	    ret = safe_strdup(sys_dir);
	    L = strlen(ret);
	    if (L > 0 && ret[L-1] != '/') {
		ret = strmcat(ret,"/");
	    }
	    ret = strmcat(ret,temp);
	}
	break;
    case read_loc_def:
	ret = safe_strdup(user_last_read_def);
	break;
    }

    if (ret) {
	DPRINT(Debug,13,(&Debug,"give_cache_file_name=%Q",
			 ret));
    } else {
	DPRINT(Debug,13,(&Debug,"give_cache_file_name=NULL"));
    }
    DPRINT(Debug,13,(&Debug,"; last_read_loc=%d",last_read_loc));
    switch (last_read_loc) {
    case read_loc_none: DPRINT(Debug,13,(&Debug," read_loc_none"));  break;
    case read_loc_dir:  DPRINT(Debug,13,(&Debug," read_loc_dir"));   break;
    case read_loc_def:  DPRINT(Debug,13,(&Debug," read_loc_def"));   break;
    }
    if (sys_dir) {
	DPRINT(Debug,13,(&Debug," sys_dir=%Q",sys_dir));
    }
    DPRINT(Debug,13,(&Debug,"\n"));
    
    return ret;
}


void free_last_read()
{
    if (last_read_def_cache)
	free_last_read_cache(& last_read_def_cache);

    if (last_read_dir_cache)
	free_sort_list(& last_read_dir_cache);
}

static struct dt_flags_info * cache_guess_locking
   P_((struct last_read_cache * dir_last_read));
static struct dt_flags_info * cache_guess_locking(dir_last_read)
     struct last_read_cache * dir_last_read;
{
    struct dt_flags_info *ret = &folder_locking;
    
    const char * mbx_dir =
	give_dt_estr_as_str(&extra_mailbox_dir_e,
			    "extra-mailbox-dir",
			    NULL,NULL);

    
    if (LAST_READ_CACHE_magic != dir_last_read->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"cache_guess_locking",
	      "Bad magic type (last_read_cache)",0);

    if (! dir_last_read->sys_dir) {
	/*  NULL, if fallback file
	    ~/.elm/last.read
	*/

	DPRINT(Debug,14,(&Debug,
			 "cache_guess_locking=&conf_merge_locking; file %s\n",
			 dir_last_read->cache_file_name));
	
	return &conf_merge_locking;
    }

    if (dt_flag_is_set(&last_read_fbck_locking, lock_prefer_flag)) {

	ret =  &last_read_fbck_locking;

	DPRINT(Debug,14,(&Debug,
			 "cache_guess_locking: last-read-fallback-locking preferred\n"));
	
    } else {
	int ok = 0;
	
	if (mbx_dir && same_dir_str(dir_last_read->sys_dir,
				    mbx_dir)) {    
	    ret =  &mailbox_locking;

	    DPRINT(Debug,14,(&Debug,
			     "cache_guess_locking: %s on extra-mailbox-dir, suggest mailbox-locking\n",
			     dir_last_read->sys_dir));
	    
	}

	if (dt_flag_is_set(ret,lock_flock_flag)) {
	    DPRINT(Debug,14,(&Debug,
			     "cache_guess_locking: flock locking set\n"));
	    ok = 1;
	}
	
	if (dt_flag_is_set(ret,lock_fcntl_flag)) {
	    DPRINT(Debug,14,(&Debug,
			     "cache_guess_locking: fcntl locking set\n"));
	    ok = 1;
	}

	if (!ok) {
		
	    ret =  &last_read_fbck_locking;

	    DPRINT(Debug,14,(&Debug,
			     "cache_guess_locking: falling to last-read-fallback-locking\n"));
	    
	}
    }

    DPRINT(Debug,14,(&Debug,
		     "cache_guess_locking=%p",ret));

    if ( &folder_locking == ret) {
	DPRINT(Debug,14,(&Debug," &folder_locking"));
    }
    if (&mailbox_locking == ret) {
	DPRINT(Debug,14,(&Debug," &mailbox_locking"));
    }
    if (&last_read_fbck_locking == ret) {
	DPRINT(Debug,14,(&Debug," (&last_read_fbck_locking"));
    }    
    DPRINT(Debug,14,(&Debug,"; dir %s, file %s\n",
		     dir_last_read->sys_dir,
		     dir_last_read->cache_file_name));
	
    return ret;
}


/* Write struct last_read_cache if modified */
void flush_last_read(dir_last_read)
     struct last_read_cache * dir_last_read;
{

    if (dir_last_read) {
	
	if (LAST_READ_CACHE_magic != dir_last_read->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"flush_last_read",
		  "Bad magic type (last_read_cache)",0);

	if (dir_last_read->modified) {

	    struct dt_flags_info   * locking =
		cache_guess_locking(dir_last_read);
	    struct last_read_open  * H = NULL;

	    DPRINT(Debug,14,(&Debug,
			     "write_last_read_cache: %s: %s need update\n",
			     dir_last_read->sys_dir ?
			     dir_last_read->sys_dir : "fallback file",
			     dir_last_read->cache_file_name));
	    
	    /* Need read and lock */

	    reload_last_read_cache(dir_last_read,&H,
				   NULL /* errors */,
				   locking,
				   last_read_read_update,
				   NULL /* errno_res */);

	    if (H) {

		if (LAST_READ_OPEN_magic != H->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "write_last_read_cache",
			  "Bad magic number (last_read_open)",0);
		
		if (last_read_def_cache == dir_last_read) {
		    last_read_def_rewrite = ison(H->lread_open_flags,
						 LREAD_OPEN_SUGGEST_REWRITE);
		}


		if (write_last_read_cache(H,
					  dir_last_read,
					  locking,
					  NULL /* errno_res */,

					  /* commentfile,actor,version_buff */
					  NULL,NULL,NULL
					  )) {
		    
		     DPRINT(Debug,14,(&Debug,
				      "write_last_read_cache: %s: %s updated\n",
				      dir_last_read->sys_dir ?
				      dir_last_read->sys_dir : "fallback file",
				      dir_last_read->cache_file_name));

		     if (last_read_def_cache == dir_last_read) {
			 last_read_def_rewrite = 0;
		     }
		     
		}

		free_last_read_open(&H);
	    }
	}
    }
	
}

enum last_read_default { last_read_def_existing = 0,
			 last_read_def_create };

static  struct last_read_cache * give_last_read P_((const char       * sys_dir,
						    enum last_read_location last_read_loc,
						    const char * res_dir,
						    enum last_read_default default_create
						    ));
						    
static  struct last_read_cache * give_last_read(sys_dir,last_read_loc,res_dir,
						default_create)
     const char            * sys_dir;
     enum last_read_location last_read_loc;
     const char            * res_dir;
     enum last_read_default  default_create;
{
    struct last_read_cache * ret   = NULL;
    char                   * fname = NULL;

    DPRINT(Debug,13,(&Debug,"give_last_read: sys_dir=%s last_read_loc=%d",sys_dir,last_read_loc));
    switch (last_read_loc) {
    case read_loc_none: DPRINT(Debug,13,(&Debug," read_loc_none"));     break;
    case read_loc_dir:  DPRINT(Debug,13,(&Debug," read_loc_dir"));      break;
    case read_loc_def:  DPRINT(Debug,13,(&Debug," read_loc_def"));      break;	
    }
    if (res_dir) {
	DPRINT(Debug,13,(&Debug," res_dir=%s",res_dir));
    }
    DPRINT(Debug,13,(&Debug," default_create=%d",default_create));
    switch (default_create) {
    case last_read_def_existing:  DPRINT(Debug,13,(&Debug," last_read_def_existing")); break;
    case last_read_def_create:    DPRINT(Debug,13,(&Debug," last_read_def_create"));   break;
    }
    DPRINT(Debug,13,(&Debug,"\n"));
    
    fname = give_cache_file_name(sys_dir,last_read_loc);

    if (fname) {	    
	enum sort_list_search_op op = sort_list_search_normal;
	enum use_lastread_v use_lastread =
	    give_dt_enumerate_as_int(&use_last_read_file);
	
	
	switch (use_lastread) {
	case use_lastread_no:
	    DPRINT(Debug,13,(&Debug,
			     "give_last_read: %s - use-last-read disabled\n",
			     sys_dir));
	    goto fail;
	    
	case use_lastread_yes:
	    DPRINT(Debug,13,(&Debug,
			     "give_last_read: %s - use-last-read enabled\n",
			     sys_dir));
	    op = sort_list_search_create;
	    break;
	case use_lastread_auto:
	    DPRINT(Debug,13,(&Debug,
			     "give_last_read: %s - use-last-read auto\n",
			     sys_dir));

	    if (0) {
	    case use_lastread_existing:
		DPRINT(Debug,13,(&Debug,
				 "give_last_read: %s - use-last-read existing\n",
				 sys_dir));	    	    
	    }

	    switch (default_create) {
	    case last_read_def_create:
		op = sort_list_search_create;
		break;
	    case last_read_def_existing: {
		enum syscall_status  r_access =
		    access(fname, ACCESS_EXISTS);
		
		switch ( r_access) {
		    int err UNUSED_VAROK;
		case syscall_success:
		    DPRINT(Debug,13,(&Debug,
				     "give_last_read: %s - %s exists\n",
				     sys_dir,fname));
		    op = sort_list_search_create;
		    break;
		case syscall_error:
		    err = errno;
		    DPRINT(Debug,13,(&Debug,
				     "give_last_read: %s - %s gives: %s\n",
				     sys_dir,fname,
				     strerror(err)));
		    break;
		}		    						
	    }
		break;
	    }
		    
	    break;
	case NUM_use_lastread:
	    break;
	}
		 	    
	switch (last_read_loc) {
	    
	case read_loc_none:
	    break;
	case read_loc_dir:
	    
	    if (res_dir) {
		struct last_read_cache_def def = NULL_last_read_cache_def;
		union sort_item_default    def0;
		union sort_item            res;
		size_t                     idx;
		union sort_key             key;
		
		if (! last_read_dir_cache) {
		    DPRINT(Debug,13,(&Debug,
				     "give_last_read: %s -- setting last_read_dir_cache\n",
				     sys_dir));
		    
		    last_read_dir_cache = alloc_sort_list(& last_read_cache_op,2);
		}
		
		def.st              = NULL;
		def.prealloc        = 5;
		def.fchanges        = NULL;
		
		def0.last_read_cache = &def;
		def.cache_file_name = fname;
		
		res.last_read_cache  = NULL;
		key.str = res_dir;
		idx = 0;
		
		/* Increments refcount */;
		if (search_sort_list_item(last_read_dir_cache,
					  op,
					  key,def0,&res,&idx,NULL)) {
		    
		    if (res.last_read_cache) {
			if (LAST_READ_CACHE_magic != res.last_read_cache->magic)
			    panic("MBX PANIC",__FILE__,__LINE__,"give_last_read",
				  "Bad magic type (last_read_cache)",0);
				    			
			DPRINT(Debug,13,(&Debug,
					 "give_last_read: %s -- last read %Q: idx #%zu found\n",
					 sys_dir,res_dir,idx));
			
			ret = res.last_read_cache;

			if (! ret->read) {
			    DPRINT(Debug,13,(&Debug,
					     "give_last_read: %s -- need initial read\n",
					     sys_dir));
			    
			    reload_last_read_cache(ret,NULL,NULL,
						   cache_guess_locking(ret),
						   last_read_check,
						   NULL /* errno_res */);
			} else {
			    int err = 0;
			    enum file_changes_result change_status
				= test_file_changes(fname,
						    & ret->cache_file,
						    & err);

			    switch (change_status) {
			    case no_file_changes:
				DPRINT(Debug,13,(&Debug,
						 "give_last_read: %s last_read %s not changed\n",
						 sys_dir,fname));
				break;
			    case file_changes_error:
				DPRINT(Debug,13,(&Debug,
						 "give_last_read: %s last_read %s not available %s\n",
						 sys_dir,fname,strerror(err)));
				break;
			    case have_file_changes:
				DPRINT(Debug,13,(&Debug,
						 "give_last_read: %s last_read %s need reload\n",
						 sys_dir,fname));
				
				reload_last_read_cache(ret,NULL,NULL,
						       cache_guess_locking(ret),
						       last_read_check,
						       NULL /* errno res */);
				break;
			    }
			}
			    						
		    } else {
			DPRINT(Debug,13,(&Debug,
					 "give_last_read: %s -- last read for %Q: search_sort_list_item did not set last_read_cache, idx #%zu\n",
					 sys_dir,res_dir,idx));
		    }
		    
		} else {
		    DPRINT(Debug,13,(&Debug,
				     "give_last_read: %s -- last read for %Q not found or created\n",
				     sys_dir,res_dir));
		}
	    } else {
		DPRINT(Debug,13,(&Debug,
				 "give_last_read: %s -- target dir not set\n",
				 sys_dir));
	    }
	    break;
	case read_loc_def: {
	    struct last_read_open  * H = NULL;
	    
	    if (last_read_def_cache) {
		
		ret = last_read_def_cache;
		inc_last_read_cache_refcount(ret);

		if (! ret->read) {
		    DPRINT(Debug,13,(&Debug,
				     "give_last_read: %s -- need initial read for default cache %s\n",
				     sys_dir,fname));
		    
		    reload_last_read_cache(ret,&H,NULL,
					   &conf_merge_locking,
					   last_read_check,
					   NULL /* errno_res */);
		} else {

		    int err = 0;
		    enum file_changes_result change_status
			= test_file_changes(fname,
					    & ret->cache_file,
					    & err);
		    
		    switch (change_status) {
		    case no_file_changes:
			DPRINT(Debug,13,(&Debug,
					 "give_last_read: %s last_read %s not changed\n",
					 sys_dir,fname));
			break;
		    case file_changes_error:
			DPRINT(Debug,13,(&Debug,
					 "give_last_read: %s last_read %s not available %s\n",
					 sys_dir,fname,strerror(err)));
			break;
		    case have_file_changes:
			DPRINT(Debug,13,(&Debug,
					 "give_last_read: %s last_read %s need reload\n",
					 sys_dir,fname));
			
			reload_last_read_cache(ret,&H,NULL,
					       &conf_merge_locking,
					       last_read_check,
					       NULL /* errno_res */);
			break;
		    }
		}
		    
	    } else {
		DPRINT(Debug,13,(&Debug,
				 "give_last_read: %s -- default cache %s not read\n",
				 sys_dir,fname));

		last_read_def_cache =
		    load_last_read_cache(user_last_read_def,NULL,
					 &H,NULL, &conf_merge_locking,
					 last_read_check);

		if (last_read_def_cache) {
		    ret = last_read_def_cache;
		    inc_last_read_cache_refcount(ret);

		    DPRINT(Debug,13,(&Debug,
				     "give_last_read: %s -- succeed to read default cache %s\n",
				     sys_dir,fname)); 
		} else {
		    switch (default_create) {
		    case last_read_def_create:
			last_read_def_cache = malloc_last_read_cache(NULL /* sys_dir */,
								     NULL /* struct stat */,
								     1,
								     &last_read_def_fchanges,
								     fname);
			DPRINT(Debug,13,(&Debug,
					 "give_last_read: %s - created dummy default cache %s\n",
					 sys_dir,fname));

			ret = last_read_def_cache;
			inc_last_read_cache_refcount(ret);
			
			break;
		    case last_read_def_existing:
			break;
		    }
		}
	    }

	    if (H) {
		
		if (LAST_READ_OPEN_magic != H->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "write_last_read_cache",
			  "Bad magic number (last_read_open)",0);
		
		if (last_read_def_cache) {
		    last_read_def_rewrite = ison(H->lread_open_flags,
						 LREAD_OPEN_SUGGEST_REWRITE);
		}

		free_last_read_open(&H);
	    }
	    
	}
	    break;
	}
	    
    fail:
	free(fname);
	fname = NULL;       	    
    } else {
	DPRINT(Debug,13,(&Debug,
			 "give_last_read: %s -- cache file name not set\n",
			 sys_dir));
    }
    
    if (ret) {
	if (LAST_READ_CACHE_magic != ret->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"give_last_read",
		  "Bad magic type (last_read_cache)",0);

	DPRINT(Debug,13,(&Debug,
			 "give_last_read=%p",ret));
	if (ret->sys_dir) {
	    DPRINT(Debug,13,(&Debug," sys_dir=%s",ret->sys_dir));
	}
	if (ret->cache_file_name) {
	    DPRINT(Debug,13,(&Debug," cache_file_name=%s",ret->cache_file_name));
	}	
    } else {
	DPRINT(Debug,13,(&Debug,
			 "give_last_read=NULL"));
    }

    DPRINT(Debug,13,(&Debug,"; requested sys_dir=%s",sys_dir));
    if (res_dir) {
	DPRINT(Debug,13,(&Debug," res_dir=%s",res_dir));
    }
    DPRINT(Debug,13,(&Debug,"\n"));
    
    return ret;
}
					     
/* Result is malloced if *free_sys_dir is set */
static char * stat_file_to_sys_dir P_((const char   * stat_file,
				       int          * free_sys_dir_p
				       ));
static char * stat_file_to_sys_dir (stat_file,free_sys_dir_p)
     const char   * stat_file;
     int          * free_sys_dir_p;
{
     char       * sys_dir  = NULL;
     int          free_sys_dir = * free_sys_dir_p;

     const char * p        = strrchr(stat_file,'/');
    
     if (p) {
	 size_t len = (p - stat_file);
	 	 
	 sys_dir = safe_malloc(len+1);
	 free_sys_dir = 1;
	 strnfcpy(sys_dir,stat_file,len,len+1,NULL);	
	
     } else {
	 sys_dir      = ".";
	 free_sys_dir = 0;
     }
     
     
     DPRINT(Debug,14,(&Debug,
		      "stat_file_to_sys_dir=%s; stat_file=%s\n",
		     sys_dir,
		     stat_file));
    
     * free_sys_dir_p = free_sys_dir;
     return sys_dir;
}

/* Increments refcount - may change or open new cache 
   does not reload cache if directory is corrent 
*/
struct last_read_cache * upgrade_last_read(dir_last_read,
					   stat_file,
 					   stat_buf)
     struct last_read_cache * dir_last_read;
     const char             * stat_file;
     struct stat            * stat_buf;
{
    struct last_read_cache *ret = NULL;
    int need_upgrade    = 0;

    int          free_sys_dir = 0;
    char       * sys_dir      =  stat_file_to_sys_dir(stat_file,
						      &free_sys_dir);
    const char * res_dir = NULL;
    
    enum last_read_location last_read_loc =  read_loc_none;

    
    DPRINT(Debug,13,(&Debug,
		     "upgrade_last_read: %s => sys_dir=%Q\n",
		     stat_file,sys_dir));

    last_read_loc = last_read_dir_to_loc(sys_dir,&res_dir);

    DPRINT(Debug,13,(&Debug,
		     "upgrade_last_read: %s => location %d",
		     stat_file,
		     last_read_loc));

    switch (last_read_loc) {
    case read_loc_none: DPRINT(Debug,13,(&Debug," read_loc_none"));
	break;
    case read_loc_dir:  DPRINT(Debug,13,(&Debug," read_loc_dir"));
	break;
    case read_loc_def:  DPRINT(Debug,13,(&Debug," read_loc_def"));
	break;
    }
    
    if (res_dir) {
	DPRINT(Debug,13,(&Debug," res_dir=%Q",res_dir));
    }
    DPRINT(Debug,13,(&Debug,"\n"));
    
    if (dir_last_read) {

	if (LAST_READ_CACHE_magic != dir_last_read->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"upgrade_last_read",
		  "Bad magic type (last_read_cache)",0);

	DPRINT(Debug,13,(&Debug,
			 "upgrade_last_read: %s dir_last_read=%p",
			 stat_file,dir_last_read));
	if (dir_last_read->sys_dir) {
	    DPRINT(Debug,13,(&Debug," sys_dir=%s",dir_last_read->sys_dir));
	}
	if (dir_last_read->cache_file_name) {
	    DPRINT(Debug,13,(&Debug," cache_file_name=%s",
			     dir_last_read->cache_file_name));
	}	
	DPRINT(Debug,13,(&Debug,"\n"));
	
	switch (last_read_loc) {
	case read_loc_none: need_upgrade = 1;  break;
	case read_loc_dir:
	    if (res_dir && dir_last_read->sys_dir &&
		same_dir_str(res_dir,dir_last_read->sys_dir)) {

		if (FILE_CHANGES_magic != dir_last_read->cache_file.magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"upgrade_last_read",
			  "Bad magic type (file_changes)",0);

		if (stat_buf &&
		    dir_last_read->cache_file.dev != stat_buf->st_dev) {

		    DPRINT(Debug,13,(&Debug,
				     "upgrade_last_read: %s -- last_read dev %ld does not match %Q dev %ld\n",
				     stat_file,
				     (long)(dir_last_read->cache_file.dev),
				     res_dir,
				     (long)(stat_buf->st_dev)));
		    need_upgrade = 1;
		}
		
	    } else {
		need_upgrade = 1;
		if (res_dir && dir_last_read->sys_dir) {
		    DPRINT(Debug,13,(&Debug,
				     "upgrade_last_read: %s -- last_read %Q does not match to %Q\n",
				     stat_file,dir_last_read->sys_dir,
				     res_dir));
				     
		}
	    }
	    break;
	case read_loc_def:
	    if (last_read_def_cache != dir_last_read)
		need_upgrade = 1;
	    break;
	}

	if (need_upgrade) {
	    DPRINT(Debug,13,(&Debug,
			     "upgrade_last_read: %s -- last read %s need upgrade\n",
			     stat_file,
			     dir_last_read->cache_file_name));
	}
    }

        
    if (need_upgrade || !dir_last_read) {
	
	ret = give_last_read(sys_dir,last_read_loc,res_dir,
			     dir_last_read ? last_read_def_create :
			     last_read_def_existing
			     );
	
	DPRINT(Debug,13,(&Debug,
			 "upgrade_last_read: %s ",
			 stat_file));

	if (ret) {
	    DPRINT(Debug,13,(&Debug," last_read for %s changed or set",
			     sys_dir));
	} else {
	    DPRINT(Debug,13,(&Debug," last_read for %s failed",
			     sys_dir));
	}
	DPRINT(Debug,13,(&Debug,"\n"));
	
    } else {
	if (FILE_CHANGES_magic != dir_last_read->cache_file.magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"upgrade_last_read",
			  "Bad magic type (file_changes)",0);
	
	ret = dir_last_read;
	inc_last_read_cache_refcount(ret);

	DPRINT(Debug,13,(&Debug,
			 "upgrade_last_read: %s last_read %s not changed\n",
			 stat_file,dir_last_read->cache_file_name));
	
    }

    if (sys_dir) {
	if (free_sys_dir)
	    free(sys_dir);
	sys_dir = NULL;
    }

    if (ret) {
	if (LAST_READ_CACHE_magic != ret->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"upgrade_last_read",
		  "Bad magic type (last_read_cache)",0);

	DPRINT(Debug,13,(&Debug,
			 "upgrade_last_read=%p",ret));
	if (ret->sys_dir) {
	    DPRINT(Debug,13,(&Debug," sys_dir=%s",ret->sys_dir));
	}
	if (ret->cache_file_name) {
	    DPRINT(Debug,13,(&Debug," cache_file_name=%s",ret->cache_file_name));
	}	
    } else {
	DPRINT(Debug,13,(&Debug,
			 "upgrade_last_read=NULL"));
    }
    DPRINT(Debug,13,(&Debug,"; stat_file=%s\n",stat_file));
    
    return ret;
}

void inc_last_read_cache_refcount(cache)
     struct last_read_cache * cache;
{
    if (LAST_READ_CACHE_magic != cache->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"inc_last_read_cache_refcount",
              "Bad magic type (last_read_cache)",0);

    cache->refcount++;
}
    
/* Decremetns refcount */
void free_last_read_cache(cache)
     struct last_read_cache ** cache;
{

    if (LAST_READ_CACHE_magic != (*cache)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_last_read_cache",
              "Bad magic type (last_read_cache)",0);

    if ((*cache)->refcount < 1)
	panic("MBX PANIC",__FILE__,__LINE__,"free_last_read_cache",
              "Bad refcount",0);

    (*cache)->refcount--;

    if ((*cache)->refcount > 0) {
	(*cache) = NULL;

	return;
    }
    

    if ((*cache)->sys_dir) {
	free((*cache)->sys_dir);
	    
	(*cache)->sys_dir = NULL;
    }

    if ((*cache)->cached_name)
	free_string_sort(& ((*cache)->cached_name));
    
    if ((*cache)->cache_file_name) {
	free((*cache)->cache_file_name);

	(*cache)->cache_file_name = NULL;
    }

    
    if ((*cache)->entries)
	free_sort_list(& ((*cache)->entries));
		       
    (*cache)->magic = 0; /* Invalidate */
    
    free(*cache);
    *cache = NULL;
}


 /* NULL if / without absolute path */
static const char * pathname_to_entryname P_((struct last_read_cache * this_last_read,
					      const char             * pathname,
					      struct stat            * file_stat));
static const char * pathname_to_entryname(this_last_read,pathname,file_stat)
     struct last_read_cache * this_last_read;
     const char             * pathname;
     struct stat            * file_stat;
{
    const char             * name1 = pathname;

    if (LAST_READ_CACHE_magic != this_last_read->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"pathname_to_entryname",
	      "Bad magic type (last_read_cache)",0);

    if (this_last_read->sys_dir &&
	in_directory(file_stat,pathname,this_last_read->sys_dir)) {

	char * x = strrchr(pathname,'/');
	
	DPRINT(Debug,13,(&Debug,
			 "pathname_to_entryname: %s on directory %s",
			 pathname,this_last_read->sys_dir));

	if (x && x[1]) {
	    name1 = x+1;

	    DPRINT(Debug,13,(&Debug,", name => %s",
			     name1));
	}
	DPRINT(Debug,13,(&Debug,"\n"));		
    }

    if (! name1[0]) {
	DPRINT(Debug,13,(&Debug,
			 "pathname_to_entryname: %s - empty name\n",
			 pathname));
	name1 = NULL;
    } else if ('/' != name1[0] ) {

	const char * x1;

	for (x1 = name1; *x1; x1++) {
	    if ('/' == *x1) {
		DPRINT(Debug,13,(&Debug,
				 "pathname_to_entryname: %s - invalid name %s\n",
				 pathname,name1));
		name1 = NULL;
		goto fail;
	    }
	}
    }

 fail:
    return name1;
}

/* Cache assumed to point correct directory */
int have_last_read(this_last_read,pathname,file_stat,last_read_time)
     struct last_read_cache * this_last_read;
     const char             * pathname;
     struct stat            * file_stat;
     time_t                 * last_read_time;
{
    /*  this_last_read is from upgrade_last_read() */

    const char             * name1 = pathname;
    int ret = 0;
    
    if (LAST_READ_CACHE_magic != this_last_read->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"have_last_read",
	      "Bad magic type (last_read_cache)",0);

    name1 = pathname_to_entryname(this_last_read,pathname,file_stat);

    if (!name1) {
	DPRINT(Debug,13,(&Debug,
			 "have_last_read: %s: Invalid name\n",
			 pathname));
	goto fail;
    }
    
    if (this_last_read->entries) {
	union sort_key            key;
	union sort_item           res;
	union sort_item_default   def;
	size_t idx = 0;
	
	key.str             = name1;
	def.last_read_entry = NULL;
	res.dummy           = NULL;

	if (search_sort_list_item(this_last_read->entries,
				  sort_list_search_normal,
				  key,def,&res,&idx,
				  NULL)) {

	    if (res.last_read_entry) {
		struct last_read_entry * entry = res.last_read_entry;

		DPRINT(Debug,13,(&Debug,
				 "have_last_read: %s: %s found from %s idx %zu\n",
				 pathname,name1,
				 this_last_read->cache_file_name ?
				 this_last_read->cache_file_name : "(no name)",
				 idx));
		
		if (LAST_READ_ENTRY_magic != entry->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"have_last_read",
			  "Bad magic type (last_read_entry)",0);

		if (ison(entry->entry_flags,LAST_READ_valid)) {

		    if (FILE_CHANGES_magic != this_last_read->cache_file.magic)
			panic("MBX PANIC",__FILE__,__LINE__,"have_last_read",
			      "Bad magic number (file_changes)",0);
		    
		    if (file_stat && this_last_read->cache_file.valid) {
			
			if (this_last_read->cache_file.dev != file_stat->st_dev) {
	
			    DPRINT(Debug,13,(&Debug,
					     "have_last_read: %s:  directory %s st_dev %ld != stat dev %ld\n",
					     pathname,
					     this_last_read->sys_dir ?
					     this_last_read->sys_dir :
					     "(default)",
					     (long) this_last_read->cache_file.dev,
					     (long) file_stat->st_dev));

			    if (ison(entry->entry_flags, LAST_READ_have_st_ino)) {
				DPRINT(Debug,13,(&Debug,
						 "have_last_read: %s: idx %zu %s have st_ino -- no match\n",
						 pathname,idx,
						 entry->folder_sys));
				goto fail_match;
			    }
			    
			} else {		       
			    if (isoff(entry->entry_flags, LAST_READ_have_st_ino)) {
				DPRINT(Debug,13,(&Debug,
						 "have_last_read: %s: idx %zu %s have no st_ino -- no match\n",
						 pathname,idx,
						 entry->folder_sys));
				goto fail_match;
			    }

			    if (entry->st_ino != file_stat->st_ino) {
				DPRINT(Debug,13,(&Debug,
						 "have_last_read: %s: idx %zu %s st_ino %lu != stat ino %lu -- no match\n",
						 pathname,idx,
						 (unsigned long) entry->st_ino,
						 (unsigned long) file_stat->st_ino,
						 entry->folder_sys));
				
				goto fail_match;
			    }
			}
		    }

		    if (entry->last_read > 0 &&
			(time_t)-1 != entry->last_read) {
			char *X = ctime(& (entry->last_read));

			DPRINT(Debug,13,(&Debug,
					 "have_last_read: %s: entry %zu %s last_read %ld",
					 pathname,idx,
					 entry->folder_sys,
					 (long)entry->last_read));

			if (X) { /* ctime() includes newline */
			    DPRINT(Debug,13,(&Debug," -- %s",X));
			} else {
			    DPRINT(Debug,13,(&Debug,"\n"));
			}
					 			
			if (last_read_time)
			    *last_read_time = entry->last_read;

			ret = 1;
		    }
		    
			
		} else {
		    DPRINT(Debug,13,(&Debug,
				     "have_last_read: %s: entry %zu %s not valid\n",
				     pathname,idx,
				     entry->folder_sys));
		}

	    fail_match:
		    
		/* Decrements refcount */
		free_last_read_entry(& res.last_read_entry);
		
	    } else {
		 DPRINT(Debug,13,(&Debug,
				  "have_last_read: %s: %s found from %s idx %zu - search_sort_list_item did not set last_read_entry\n",
				  pathname,name1,
				  this_last_read->cache_file_name ?
				  this_last_read->cache_file_name : "(no name)",
				  idx));
	    }

	} else {
	    DPRINT(Debug,13,(&Debug,
			     "have_last_read: %s: %s not found from %s\n",
			     pathname,name1,
			     this_last_read->cache_file_name ?
			     this_last_read->cache_file_name : "(no name)"));
	}
	    
    } else {
	DPRINT(Debug,13,(&Debug,
			 "have_last_read: %s no entries on %s\n",
			 pathname,
			 this_last_read->cache_file_name ?
			 this_last_read->cache_file_name : "(no name)"));
    }


 fail:
    DPRINT(Debug,13,(&Debug,
		     "have_last_read=%d; pathname=%s",
		     ret,pathname));

    if (last_read_time && *last_read_time > 0 && (time_t)-1 != *last_read_time) {
	char *X = ctime(last_read_time);
	DPRINT(Debug,13,(&Debug,
			 " *last_read_time=%ld",
			 *last_read_time));
	
	if (X) { /* ctime() includes newline */
	    DPRINT(Debug,13,(&Debug," -- %s",X));
	} else {
	    DPRINT(Debug,13,(&Debug,"\n"));
	}	
    }

    DPRINT(Debug,13,(&Debug,"\n"));
    
    return ret;
}

/* Possible allocates new, decrement old refcount if name is changed */
int update_last_read(dir_last_read,sys_dir)
     struct last_read_cache ** dir_last_read;
     const char * sys_dir;
{
    int cache_changed = 0;
    int ret = 0;

    const char * resolved_dir = NULL;
    
    enum last_read_location new_loc = last_read_dir_to_loc(sys_dir,
							  &resolved_dir);
    
    if (*dir_last_read) {
	if (LAST_READ_CACHE_magic != (*dir_last_read)->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"update_last_read",
		  "Bad magic type (last_read_cache)",0);

	DPRINT(Debug,13,(&Debug,
			 "update_last_read: old cache %p",
			 *dir_last_read));
	
	if ((*dir_last_read)->sys_dir) {
	    DPRINT(Debug,13,(&Debug,"; sys_dir=%s",
			     (*dir_last_read)->sys_dir));
	}
	if (last_read_def_cache == *dir_last_read) {
	    DPRINT(Debug,13,(&Debug,"; default cache"));
	}
	if ((*dir_last_read)->cache_file_name) {
	    DPRINT(Debug,13,(&Debug,"; cache_file_name=%s",
			     (*dir_last_read)->cache_file_name));
	}       
	DPRINT(Debug,13,(&Debug,"\n"));
	
	switch (new_loc) {
	case read_loc_none: cache_changed = 1; break;
	case read_loc_dir:
	    if (! (*dir_last_read)->sys_dir ||
		! resolved_dir              ||
		! same_dir_str((*dir_last_read)->sys_dir,
			       resolved_dir))
		cache_changed = 1;
	    break;
	case read_loc_def:
	    if ((*dir_last_read)->sys_dir)
		cache_changed = 1;
	    
	}

	if (cache_changed)
	    free_last_read_cache(dir_last_read);
	
    } else
	cache_changed = 1;

    if (cache_changed) {

	if (*dir_last_read)
	    panic("MBX PANIC",__FILE__,__LINE__,"update_last_read",
		  "Cache still set",0);
	
	if (read_loc_dir == new_loc) {
	    enum syscall_status  r;
	    	    
	    r = access(sys_dir,WRITE_ACCESS|EXECUTE_ACCESS);
	    
	    switch (r) {
		int err UNUSED_VAROK;
	    case syscall_error /* -1 */: 
		err = errno;
		
		DPRINT(Debug,10,(&Debug,
				 "update_last_read: No write access (and search) for %s:  %s (errno=%d)\n",
				 sys_dir,strerror(err), err));
		
		/* No last_read_cache, if directory
		   is not writable and executable (searchable)
		*/
		ret = 0;
		goto out;
	    case syscall_success /* 0 */:
		DPRINT(Debug,10,(&Debug,
				 "update_last_read: %s is accessible\n",
				 sys_dir));
		break;
	    }
	}

	/* locate from last_read_dir_cache */
	/* or last_read_def_cache      */

	(*dir_last_read) = give_last_read(sys_dir,
					  new_loc,resolved_dir,
					  last_read_def_existing);
    }

    if (*dir_last_read) {
	if (LAST_READ_CACHE_magic != (*dir_last_read)->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"update_last_read",
		  "Bad magic type (last_read_cache)",0);

	DPRINT(Debug,10,(&Debug,
			 "update_last_read: %s: cache %p",
			 sys_dir,*dir_last_read));
	
	if ((*dir_last_read)->sys_dir) {
	    DPRINT(Debug,10,(&Debug,"; sys_dir=%s",
			     (*dir_last_read)->sys_dir));
	}
	if (last_read_def_cache == *dir_last_read) {
	    DPRINT(Debug,10,(&Debug,"; default cache"));
	}
	if ((*dir_last_read)->cache_file_name) {
	    DPRINT(Debug,10,(&Debug,"; cache_file_name=%s",
			     (*dir_last_read)->cache_file_name));
	}       

	ret = 1;
    } else {
	DPRINT(Debug,10,(&Debug,
			 "update_last_read: %s: No cache",
			 sys_dir));

	ret = 0;
    }

    if (cache_changed) {
	DPRINT(Debug,10,(&Debug,"; cache changed\n"));
    }
    DPRINT(Debug,10,(&Debug,"\n"));

 out:
    DPRINT(Debug,12,(&Debug,
		     "update_last_read=%d%s\n",
		     ret,
		     ret ? " last_read active" : ""));

    return ret;
}


/* Increments refcount */


struct last_read_cache * last_read_enabled(use_lastread,cur_folder_sys,
					   oldstat,curstat)
     enum use_lastread_v use_lastread;
     char             * cur_folder_sys;
     struct stat      * oldstat;
     struct stat      * curstat;
{
    struct last_read_cache *ret = NULL;

    int          free_sys_dir = 0;
    char       * sys_dir      =  stat_file_to_sys_dir(cur_folder_sys,
						      &free_sys_dir);
    enum last_read_location last_read_loc =  read_loc_none;
    const char * res_dir = NULL;

    enum last_read_default may_create = last_read_def_existing;
    struct last_read_cache * last_read = NULL;
    
    DPRINT(Debug,13,(&Debug,
		     "last_read_enabled: use_lastread=%d",
		     use_lastread));

    switch (use_lastread) {
    case use_lastread_no:       DPRINT(Debug,13,(&Debug," use_lastread_no\n"));     goto fail;
    case use_lastread_yes:      DPRINT(Debug,13,(&Debug," use_lastread_yes"));
	may_create = last_read_def_create;
	break;
    case use_lastread_auto:     DPRINT(Debug,13,(&Debug," use_lastread_auto"));     break;
    case use_lastread_existing: DPRINT(Debug,13,(&Debug," use_lastread_existing")); break;
    case NUM_use_lastread:                                                          break;
    }

    DPRINT(Debug,13,(&Debug,", cur_folder_sys=%s", cur_folder_sys));

    if (oldstat) {
	DPRINT(Debug,13,(&Debug,
			 ", oldstat dev/ino=%lu/%lu atime=%ld",
			 (unsigned long)(oldstat->st_dev),
                         (unsigned long)(oldstat->st_ino),
			 (long)oldstat->st_atime));
    }
    if (curstat) {
	DPRINT(Debug,13,(&Debug,
			 ", curstat dev/ino=%lu/%lu atime=%ld",
			 (unsigned long)(curstat->st_dev),
                         (unsigned long)(curstat->st_ino),
			 (long)curstat->st_atime));
    }
    
    if (oldstat && curstat) {
	if (oldstat->st_dev == curstat->st_dev &&
	    oldstat->st_ino == curstat->st_ino) {
	    if (oldstat->st_atime == curstat->st_atime){
		DPRINT(Debug,13,(&Debug,", time stampt not updated"));
		if (use_lastread_auto == use_lastread) {
		    may_create = last_read_def_create;
		    DPRINT(Debug,13,(&Debug,", use last read"));
		}
	    } else if (oldstat->st_atime < curstat->st_atime) {
		DPRINT(Debug,13,(&Debug,", time stamp advanced"));
	    }
	} else {
	    DPRINT(Debug,13,(&Debug,", file changed"));
	}
    }
    
    DPRINT(Debug,13,(&Debug,"; sys_dir=%s\n",sys_dir));

    last_read_loc = last_read_dir_to_loc(sys_dir,&res_dir);

    DPRINT(Debug,13,(&Debug,
		     "last_read_enabled: sys_dir=%s => location %d",
		     sys_dir,
		     last_read_loc));

    switch (last_read_loc) {
    case read_loc_none: DPRINT(Debug,13,(&Debug," read_loc_none\n"));   goto fail;
    case read_loc_dir:  DPRINT(Debug,13,(&Debug," read_loc_dir"));      break;
    case read_loc_def:  DPRINT(Debug,13,(&Debug," read_loc_def"));      break;
    }

    if (res_dir) {
	DPRINT(Debug,13,(&Debug,", res_dir=%s",res_dir));
    }
    DPRINT(Debug,13,(&Debug,"\n"));

    if (read_loc_dir == last_read_loc) {

	enum syscall_status  r = access(sys_dir,WRITE_ACCESS|EXECUTE_ACCESS);

	switch (r) {
	    int err UNUSED_VAROK;
	case syscall_error /* -1 */: 
	    err = errno;
	    
	    DPRINT(Debug,10,(&Debug,
			     "last_read_enabled: No write access (and search) for %s:  %s (errno=%d)\n",
			     sys_dir,strerror(err), err));
	    
	    /* No last_read_cache, if directory
	       is not writable and executable (searchable)
	    */
	    goto fail;
	case syscall_success /* 0 */:
	    DPRINT(Debug,10,(&Debug,
			     "last_read_enabled: %s is accessible\n",
			     sys_dir));
	    break;
	}
    }

    last_read =  give_last_read(sys_dir,last_read_loc,res_dir,
				may_create);

    if (last_read) {
	if (LAST_READ_CACHE_magic != last_read->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"last_read_enabled",
		   "Bad magic type (last_read_cache)",0);
	
	switch (may_create) {
	case  last_read_def_existing:

	    if (last_read->entries) {
		const char             * name1 
		    = pathname_to_entryname(last_read,
					    cur_folder_sys,
					    curstat);

		if (name1) {  /* NULL if / without absolute path */

		    union sort_key            key;
		    union sort_item           res;
		    union sort_item_default   def;
		    size_t idx = 0;
		    
		    key.str             = name1;
		    def.last_read_entry = NULL;
		    res.dummy           = NULL;

		    if (search_sort_list_item(last_read->entries,
					      sort_list_search_normal,
					      key,def,&res,&idx,
					      NULL)) {

			if (res.last_read_entry) {
			    struct last_read_entry * entry = res.last_read_entry;
			    DPRINT(Debug,13,(&Debug,
					     "last_read_enabled: %s: %s found from %s idx %zu\n",
					     cur_folder_sys,name1,
					     last_read->cache_file_name ?
					     last_read->cache_file_name : "(no name)",
					     idx));

			    if (LAST_READ_ENTRY_magic != entry->magic)
				panic("MBX PANIC",__FILE__,__LINE__,"last_read_enabled",
				      "Bad magic type (last_read_entry)",0);
			    
			    ret = last_read;
			    last_read = NULL;

			    /* Decrements refcount */
			    free_last_read_entry(& res.last_read_entry);
			    
			} else {
			    DPRINT(Debug,13,(&Debug,
					     "last_read_enabled: %s: %s found from %s idx %zu - search_sort_list_item did not set last_read_entry\n",
					     cur_folder_sys,name1,
					     last_read->cache_file_name ?
					     last_read->cache_file_name : "(no name)",
					     idx));
			}
		    } else {
			DPRINT(Debug,13,(&Debug,
					 "last_read_enabled: %s: %s not found from %s\n",
					 cur_folder_sys,name1,
					 last_read->cache_file_name ?
					 last_read->cache_file_name : "(no name)"));
		    }
		} else {
		    DPRINT(Debug,13,(&Debug,
				     "last_read_enabled: %s: not valid\n",
				     cur_folder_sys));
		}

	    } else {
		DPRINT(Debug,13,(&Debug,
				 "last_read_enabled: %s: no entries on %s\n",
				 cur_folder_sys,
				 last_read->cache_file_name ?
				 last_read->cache_file_name : "(no name)"));
				 
	    }

	    break;

	case last_read_def_create:
	    DPRINT(Debug,13,(&Debug,
			     "last_read_enabled: using last read cache %s",
			     last_read->cache_file_name ?
			     last_read->cache_file_name : "(no name)"));
	    if (last_read->sys_dir) {
		DPRINT(Debug,13,(&Debug," for %s",
				 last_read->sys_dir));
	    }
		       
	    DPRINT(Debug,13,(&Debug,"\n"));
	    ret = last_read;
	    last_read = NULL;
	    break;
	}

    } else {
	DPRINT(Debug,13,(&Debug,
			 "last_read_enabled: last read cache not available, sys_dir=%s\n",
			 sys_dir));
    }
    
 fail:
    if (last_read)
	free_last_read_cache(& last_read);
    
    if (sys_dir) {
	if (free_sys_dir)
	    free(sys_dir);
	sys_dir = NULL;
    }


     if (ret) {
	 if (LAST_READ_CACHE_magic != ret->magic)
	     panic("MBX PANIC",__FILE__,__LINE__,"last_read_enabled",
		   "Bad magic type (last_read_cache)",0);
	 
	 DPRINT(Debug,13,(&Debug,
			  "last_read_enabled=%p",ret));
	 if (ret->sys_dir) {
	     DPRINT(Debug,13,(&Debug," sys_dir=%s",ret->sys_dir));
	 }
	 if (ret->cache_file_name) {
	     DPRINT(Debug,13,(&Debug," cache_file_name=%s",ret->cache_file_name));
	 }	
     } else {
	 DPRINT(Debug,13,(&Debug,
			  "last_read_enabled=NULL"));
     }
     DPRINT(Debug,13,(&Debug,"; cur_folder_sys=%s\n",cur_folder_sys));
    
    return ret;
}

/* Writes cache */

void mark_last_read(cache,
		    cur_folder_sys,
		    curstat)
     struct last_read_cache * cache;
     char                   * cur_folder_sys;
     struct stat            * curstat;
{
    struct dt_flags_info   * locking =   
	cache_guess_locking(cache);
    struct last_read_open  * H       = NULL; 
    const char             * name1   = NULL;

    enum last_read_mode      read_mode = last_read_read_update;
     
    if (LAST_READ_CACHE_magic != cache->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mark_last_read",
	      "Bad magic type (last_read_cache)",0);

    if (cache->cache_file_name && ! cache->read) {
	enum syscall_status  r_access =
	    access(cache->cache_file_name, ACCESS_EXISTS);

	switch ( r_access) {
	    int err UNUSED_VAROK;
	case syscall_success:
	    DPRINT(Debug,13,(&Debug,
			     "mark_last_read: %s: cache file %s exists\n",
			     cur_folder_sys,cache->cache_file_name));
	    break;
	case syscall_error:
	    err = errno;

	    DPRINT(Debug,13,(&Debug,
			     "mark_last_read: %s: cache file %s gives: %s\n",
			     cur_folder_sys,cache->cache_file_name,
			     strerror(err)));

	    if (ENOENT == err) {
		DPRINT(Debug,13,(&Debug,
				 "mark_last_read: Need create %s\n",
				 cache->cache_file_name));
		read_mode = last_read_create;
	    }
	}	
    }
    
    reload_last_read_cache(cache,&H,NULL,locking,read_mode,
			   NULL /* errno_res */);

    name1 = pathname_to_entryname(cache,
				  cur_folder_sys,
				  curstat);
    
    if (name1) {
	
	 union sort_key            key;
	 union sort_item           res;
	 union sort_item_default   def;
	 size_t idx = 0;

	 static struct last_read_entry_def def1 = {
	     LAST_READ_ENTRY_DEF_magic,
	     0 /* entry_flags */
	 };
	 
	 key.str             = name1;
	 def.last_read_entry = &def1;
	 res.dummy           = NULL;

	 if (search_sort_list_item(cache->entries,
				   sort_list_search_create,
				   key,def,&res,&idx,
				   NULL)) {

	     if (res.last_read_entry) {
		 struct last_read_entry * entry = res.last_read_entry;

		 DPRINT(Debug,13,(&Debug,
				  "mark_last_read: %s: %s found (or created) from %s idx %zu\n",
				  cur_folder_sys,name1,
				  cache->cache_file_name ?
				  cache->cache_file_name : "(no name)",
				  idx));
		 
		 if (LAST_READ_ENTRY_magic != entry->magic)
		     panic("MBX PANIC",__FILE__,__LINE__,
			   "mark_last_read",
			   "Bad magic type (last_read_entry)",0);
		 
		 cache->modified    = 1;  /* mark need write */
		 entry->entry_flags = LAST_READ_modified;
		 
		 entry->last_read   = 0;
		 entry->st_ino      = 0;

		 if (FILE_CHANGES_magic != cache->cache_file.magic)
		     panic("MBX PANIC",__FILE__,__LINE__,
			   "mark_last_read",
			   "Bad magic type (file_changes)",0);

		 if (curstat && cache->cache_file.valid &&
		     cache->cache_file.dev == curstat->st_dev) {
		     
		     entry->st_ino = curstat->st_ino;
		     setit(entry->entry_flags, LAST_READ_have_st_ino);
		 }

		 /* XXX  is this correct? */
		 if ((time_t)-1 != time(& (entry->last_read))) {
		     setit(entry->entry_flags, LAST_READ_valid);
		 }
		 
		 /* Decrements refcount */
		 free_last_read_entry(& res.last_read_entry);
			    
	     } else {
		 DPRINT(Debug,13,(&Debug,
				  "mark_last_read: %s: %s found (or created) from %s idx %zu - search_sort_list_item did not set last_read_entry\n",
				  cur_folder_sys,name1,
				  cache->cache_file_name ?
				  cache->cache_file_name : "(no name)",
				  idx));
	     }
		 
	 } else { 	     
	     DPRINT(Debug,13,(&Debug,
			      "mark_last_read: %s: %s not found (or created) from %s\n",
			      cur_folder_sys,name1,
			      cache->cache_file_name ?
			      cache->cache_file_name : "(no name)"));
	 }
	 
    } else {
	DPRINT(Debug,13,(&Debug,
			 "mark_last_read: %s: not valid\n",
			 cur_folder_sys));
    }
        
    if (H) {

	if (LAST_READ_OPEN_magic != H->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "mark_last_read",
		  "Bad magic number (last_read_open)",0);
	
	if (last_read_def_cache == cache) {
	    last_read_def_rewrite =  ison(H->lread_open_flags,
					 LREAD_OPEN_SUGGEST_REWRITE);
	}
		
	if (cache->modified || ison(H->lread_open_flags,
				    LREAD_OPEN_SUGGEST_REWRITE)) {

	    if (write_last_read_cache(H,cache,locking,
				      NULL /* errno_res */,

				      /* commentfile,actor,version_buff */
				      NULL,NULL,NULL
				      )) {

		DPRINT(Debug,13,(&Debug,
				 "mark_last_read: %s: cache %s updated\n",
				 cur_folder_sys,
				 cache->cache_file_name ?
				 cache->cache_file_name : "(no name)"));

		if (last_read_def_cache == cache) {
		    last_read_def_rewrite = 0;
		}
		
	    }
	} else {
	    DPRINT(Debug,13,(&Debug,
			     "mark_last_read: %s: cache %s not modified\n",
			     cur_folder_sys,
			     cache->cache_file_name ?
			     cache->cache_file_name : "(no name)"));
	}

	free_last_read_open(&H);
    } else {
	DPRINT(Debug,13,(&Debug,
			 "mark_last_read: %s: cache %s not open\n",
			 cur_folder_sys,
			 cache->cache_file_name ?
			 cache->cache_file_name : "(no name)"));
    }
}


static int flag_cache_entry P_((struct sortlist     * cache_entries,
				const char          * sys_dir,
				const char          * entryname,
				const char          * cache_file_name
				));
static int flag_cache_entry(cache_entries,sys_dir,entryname,cache_file_name)
     struct sortlist     * cache_entries;
     const char          * sys_dir;
     const char          * entryname;
     const char          * cache_file_name;
{
    int ret = 0;
    union sort_key            key;
    union sort_item           res;
    union sort_item_default   def;
    size_t idx = 0;
    
    key.str             = entryname;
    def.last_read_entry = NULL;
    res.dummy           = NULL;

    
    if (search_sort_list_item(cache_entries,
			      sort_list_search_normal,
			      key,def,&res,&idx,
			      NULL)) {

	if (res.last_read_entry) {
	    struct last_read_entry * entry = res.last_read_entry;
	    
	    DPRINT(Debug,13,(&Debug,
			     "flag_cache_entry: %s: %s found from %s idx %zu\n",
			     sys_dir,entryname,
			     cache_file_name ?
			     cache_file_name : "(no name)",
			     idx));

	    if (LAST_READ_ENTRY_magic != entry->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "flag_cache_entry",
		      "Bad magic type (last_read_entry)",0);

	    
	    setit(entry->entry_flags,LAST_READ_flag);

	    ret = 1;
	    
	    /* Decrements refcount */
	    free_last_read_entry(& res.last_read_entry);
		
	} else {
	    DPRINT(Debug,13,(&Debug,
			     "flag_cache_entry: %s: %s found from %s idx %zu - search_sort_list_item did not set last_read_entry\n",
			     sys_dir,entryname,
			     cache_file_name ?
			     cache_file_name : "(no name)",			     
			     idx));
	}

    } else {
	DPRINT(Debug,15,(&Debug,
			 "flag_cache_entry: %s: %s not found from %s\n",
			 sys_dir,entryname,
			 cache_file_name ?
			 cache_file_name : "(no name)"));
    }

    return ret;
}

/* Flag local entries  and matching with sys_dir prefix */
void flag_last_read(dir_last_read,sys_dir,entryname)
     struct last_read_cache * dir_last_read;
     const char * sys_dir;
     const char * entryname;
     
{    
    if (LAST_READ_CACHE_magic != dir_last_read->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"flag_last_read",
	      "Bad magic type (last_read_cache)",0);

    if (sys_dir && dir_last_read->sys_dir &&
	! same_dir_str(sys_dir,dir_last_read->sys_dir)) {

	DPRINT(Debug,15,(&Debug,
			 "flag_last_read: dir_last_read=%p sys_dir=%s entryname=%s does dor match to cache dir %s\n",
			 dir_last_read,sys_dir,entryname,dir_last_read->sys_dir));	
    }

    if (dir_last_read->entries) {

	int r = 0;

	r += flag_cache_entry(dir_last_read->entries,
			      sys_dir,entryname,
			      dir_last_read->cache_file_name);

	if ('/' != entryname[0]) {
	    char * str = cat_dir_entry(sys_dir,entryname);

	    r += flag_cache_entry(dir_last_read->entries,
				  sys_dir,str,
				  dir_last_read->cache_file_name);

	    free(str);
	}

	if (r) {
	    DPRINT(Debug,15,(&Debug,
			     "flag_last_read: dir_last_read=%p sys_dir=%s entryname=%s flagged\n",
			     dir_last_read,sys_dir,entryname));
	}
    }
}

/* deletes not flagged local entries (and absolute path entries
   matching with sys_dir prefix) 
*/
void purge_last_read(dir_last_read,sys_dir)
     struct last_read_cache * dir_last_read;
     const char * sys_dir;
{
    if (LAST_READ_CACHE_magic != dir_last_read->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"purge_last_read",
	      "Bad magic type (last_read_cache)",0);

    if (dir_last_read->entries) {
	size_t  cache_len = sort_list_len(dir_last_read->entries);

	DPRINT(Debug,15,(&Debug,
			 "purge_last_read: dir_last_read=%p sys_dir=%s  %zu items\n",
			 dir_last_read,sys_dir,cache_len));

	if (cache_len > 0) {
	    size_t i = cache_len -1;
	    size_t deleted  = 0;
	    size_t valid    = 0;
	    size_t flagged    = 0;
	    size_t bad_prefix = 0;
	    size_t modified = 0;

	    size_t sys_dir_len = strlen(sys_dir);

	    if (sys_dir_len > 0 && '/' == sys_dir[sys_dir_len-1])
		sys_dir_len--;
	    		
	    while (1) {
		union sort_item res;

		res.last_read_entry = NULL;
		
		/* Increments refcount */
		get_sort_list_item(dir_last_read->entries,sort_list_get_normal,i,
				   &res);

		if (res.last_read_entry) {
		    struct last_read_entry * entry;

		    int need_delete = 0;
		    
		    if (LAST_READ_ENTRY_magic != res.last_read_entry->magic)
			panic("MBX PANIC",__FILE__,__LINE__,
			      "purge_last_read",
			      "Bad magic type (last_read_entry)",0);
		    
		    entry = res.last_read_entry;
		    
		    if (ison(entry->entry_flags,LAST_READ_valid))
			valid++;
		    if (ison(entry->entry_flags,LAST_READ_modified))
			modified++;

		    if (ison(entry->entry_flags,LAST_READ_flag))
			flagged++;
		    else {

			if ('/' != entry->folder_sys[0]) {
			    need_delete = 1;
			    
			} else if (sys_dir_len > 0 &&
				   0 == strncmp(entry->folder_sys,
						sys_dir,sys_dir_len) &&
				   '/' == entry->folder_sys[sys_dir_len]) {
			    
			    /* Matches prefix */

			    need_delete = 1;

			} else
			    bad_prefix++;
			    
		    }

		    if (isoff(entry->entry_flags,LAST_READ_valid) ||
			(need_delete &&
			 isoff(entry->entry_flags,LAST_READ_modified))) {

			free_last_read_entry(& res.last_read_entry);

			/* Increments refcount */
			get_sort_list_item(dir_last_read->entries,sort_list_get_remove,i,
					   &res);
			
			if (res.last_read_entry) {
			    struct last_read_entry * entry;
			    
			    if (LAST_READ_ENTRY_magic != res.last_read_entry->magic)
				panic("MBX PANIC",__FILE__,__LINE__,
				      "purge_last_read",
				      "Bad magic type (last_read_entry)",0);
			    
			    entry = res.last_read_entry;
			    
			    DPRINT(Debug,14,(&Debug,
					     "purge_last_read: dir_last_read=%p #%zu folder_sys=%s - deleted\n",
					     dir_last_read,i,entry->folder_sys));
			    
			    
			    deleted++;
			}
		    } else
			clearit(entry->entry_flags,LAST_READ_flag);
		    
		    if (res.last_read_entry) 
			free_last_read_entry(& res.last_read_entry);
		}	   

		if (i > 0)
		    i--;
		else
		    break;
	    }
	    
	    DPRINT(Debug,14,(&Debug,
			     "purge_last_read: dir_last_read=%p %zu valid, %zu flagged, %zu bad prefix, %zu modified - %zu deleted.\n",
			     dir_last_read,valid,flagged,bad_prefix,modified,deleted));	    
	}

    }
    
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
