static char rcsid[] = "@(#)$Id: remote_server.c,v 2.26 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.26 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                  (was hurtta+elm@posti.FMI.FI)  
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "s_me.h"
#include "mbxlocal_imp.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char * cs2us P_((const char *str));
static const unsigned char * cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}

/* ------------------------------------------------------------------ */

/* REMOTE browser */

#define REMOTE_BROWSER_magic	0xF516

struct REMOTE_BROWSER {
    unsigned short            magic;     /* REMOTE_BROWSER_magic */

    struct remote_server    * current_server;
    
#if REMOTE_DIR_LISTING
    /* Assumed that current separator is ASCII here ... */
    char cur_sep;
    
    struct string           * dir_name_cache;
#endif

}; 

static struct REMOTE_BROWSER * malloc_REMOTE_BROWSER P_((void));
static struct REMOTE_BROWSER * malloc_REMOTE_BROWSER()
{
    struct REMOTE_BROWSER * ret = safe_zero_alloc(sizeof (*ret));

    ret->magic            = REMOTE_BROWSER_magic;

    ret->current_server   = NULL;

#if REMOTE_DIR_LISTING
    ret->cur_sep          = '\0';
    ret->dir_name_cache   = NULL;
#endif

    return ret;
}

static void remote_server_dec P_((struct remote_server **server));

static void free_REMOTE_BROWSER P_((struct REMOTE_BROWSER **ptr));
static void free_REMOTE_BROWSER(ptr)
     struct REMOTE_BROWSER **ptr;
{
    if (REMOTE_BROWSER_magic != (*ptr)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_REMOTE_BROWSER",
	      "Bad magic type",0);

    if ((*ptr)->current_server) 
	remote_server_dec(& (*ptr)->current_server);
   
#if REMOTE_DIR_LISTING
    if ((*ptr)->dir_name_cache) 
	free_string(& (*ptr)->dir_name_cache);
#endif


    (*ptr)->magic   = 0;        /* Invalidate */

    free(*ptr);
    (*ptr)          = NULL;
}

/* ---------------------------------------------------------------------- */

#ifdef REMOTE_MBX

static char * remote_browser_full_path P_((struct folder_browser *dir));
static char * remote_browser_full_path(dir)
     struct folder_browser *dir;
{
    char * ret = NULL;

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"remote_browser_full_path",
	      "Bad magic type (dir remote_browser)",0);

    if (!dir->sys_dir) {
	
	    DPRINT(Debug,17,(&Debug, 
			     "remote_browser_full_path: On default directory\n"));

	    if (dir->selection->sys_name) {
		ret = safe_strdup(dir->selection->sys_name);
	    } else {
		ret = safe_strdup("INBOX");

		DPRINT(Debug,17,(&Debug, 
				 "remote_browser_full_path: Assuming INBOX folder\n"));

	    }
	} else {
#if REMOTE_DIR_LISTING
	    int l1 = strlen(dir->sys_dir);
#endif
	    DPRINT(Debug,17,(&Debug, 
			     "remote_browser_full_path: On directory %s\n",
			     dir->sys_dir
			     ));

#if REMOTE_DIR_LISTING
	    if (!dir->a.remote_browser->cur_sep) {
		DPRINT(Debug,17,(&Debug, 
				 "remote_browser_full_path: No separator\n"));
		goto fail;
	    }

	    if (!dir->selection->sys_name) {
		DPRINT(Debug,17,(&Debug, 
				 "remote_browser_full_path: No folder name\n"));
		goto fail;
	    }

	    ret = safe_strdup(dir->sys_dir);

	    if (dir->a.remote_browser->cur_sep !=
		 dir->sys_dir[l1-1]) {

		char buf[2];
		buf[0] = dir->a.remote_browser->cur_sep;
		buf[1] = '\0';
		
		ret = strmcat(ret,buf);
	    }
	    
	    ret = strmcat(ret,dir->selection->sys_name);
#else
	    goto fail;
#endif
    }

 fail:

    if (ret) {
	    DPRINT(Debug,17,(&Debug, 
			     "remote_browser_full_path=%s\n",ret));
    } else {
	    DPRINT(Debug,17,(&Debug, 
			     "remote_browser_full_path=NULL\n"));

    }

    return ret;

}
#endif

/* ---------------------------------------------------------------------- */

#define RSERVER_WS_magic	0xF90A

struct remote_server_ws { /* remote server write state */
    unsigned short magic;      /* RSERVER_WS_magic */

    struct browser_type  * slave_type;
    WRITE_STATE            slave_write_state;
};

static struct remote_server_ws * malloc_remote_server_ws P_((struct browser_type  * slave_type));
static struct remote_server_ws * malloc_remote_server_ws(slave_type) 
     struct browser_type  * slave_type;
{
    struct remote_server_ws *ret = safe_zero_alloc(sizeof (*ret));
 
    ret->magic = RSERVER_WS_magic;

    ret->slave_type = slave_type;
    malloc_keep_browser_write_state(& (ret->slave_write_state));
    
    slave_type->zero_ws_fields_it(ret->slave_write_state);

    return ret;
}

static void free_remote_server_ws P_((struct remote_server_ws **ptr));
static void free_remote_server_ws(ptr)
     struct remote_server_ws **ptr;
{
    if ((*ptr)->magic != RSERVER_WS_magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_remote_server_ws",
	      "Bad magic type",0);
    	
    (*ptr)->slave_type->free_ws_fields_it((*ptr)->slave_write_state);
    free_keep_browser_write_state(& ((*ptr)->slave_write_state));
    
    (*ptr)->magic = 0;    /* Invalidate */
    free(*ptr);
    *ptr = NULL;
    
}

/* ---------------------------------------------------------------------- */

#define RSERVER_TYPE_magic	0xF909

typedef void rs_free_record P_((struct remote_server *rec));
typedef void rs_init_record P_((struct remote_server *rec));
typedef struct browser_type * rs_dir_type P_((struct remote_server *rec));
typedef int rs_connect_browser P_((struct remote_server *server,
				   struct folder_browser *dir));
typedef struct folder_info   * rs_folder_from_remote P_((struct remote_server *server,
							 struct folder_browser *dir,
							 int treat_as_spooled));


typedef int  rs_selection_is_folder P_((struct remote_server *server,
					struct folder_browser *dir,
					struct folder_info *folder));
typedef int  rs_create_selection_folder P_((struct remote_server *server,
					    struct folder_browser *dir));
typedef int rs_prepare_write_folder P_((struct remote_server *server,
					struct folder_browser *dir,
					WRITE_STATE ptr));
typedef int rs_end_write_folder P_((struct remote_server *server,
				    struct folder_browser *dir,
				    WRITE_STATE ptr));

typedef int rs_start_we_dir P_((struct remote_server *server,
				struct folder_browser *dir,
				WRITE_STATE write_state_ptr,
				int write_envelope,
				struct header_rec *current_header,
				int *env_flags));
typedef int rs_end_we_dir P_((struct remote_server *server,
			      struct folder_browser *dir,
			      WRITE_STATE write_state_ptr,
			      int write_envelope,
			      struct header_rec *current_header));

typedef int rs_write_dir_ws P_((struct remote_server *server,
				struct folder_browser *dir,
				WRITE_STATE ptr,
				int l, const char *buffer));


/* Returns -1L  on failure */
typedef long rs_tell_dir_ws  P_((struct remote_server *server,
				 struct folder_browser *dir,
				 WRITE_STATE write_state_ptr));

/* Returns 0 on failure */
typedef int rs_seek_dir_ws P_((struct remote_server *server,
			       struct folder_browser *dir,
			       WRITE_STATE write_state_ptr,
			       long pos));


/* Returns 0 on failure */
typedef int rs_select_dir P_((struct remote_server *server,
			      struct folder_browser *dir,
			      const struct string *rel_itemname, /*NONNULL */
			      struct string **dispname,
			      int *newpos));

/* Returns 0 on failure */
typedef int rs_change_dir P_((struct remote_server *server,
			      struct folder_browser *dir,
			      const struct string *rel_dirname,
			      struct string **dispname));
	
/* Gives server type and username@server handle as string -- 
   caller must free_string() */
typedef struct string * rs_give_server_title P_((const struct remote_server *server));
		       
struct remote_server_type {
    unsigned short                magic;  /* RSERVER_TYPE_magic */

    rs_free_record               * rs_free_it;
    rs_init_record               * rs_init_it;
    rs_dir_type                  * rs_dir_type_it; 
    rs_connect_browser           * rs_connect_browser_it; 
    rs_folder_from_remote        * rs_folder_from_it;  
    rs_selection_is_folder       * rs_selection_is_it;
    rs_create_selection_folder   * rs_create_selection_it;
    rs_prepare_write_folder      * rs_prepare_write_it;
    rs_end_write_folder          * rs_end_write_it;

    rs_start_we_dir              * rs_start_we_it;
    rs_end_we_dir                * rs_end_we_it;
    rs_write_dir_ws              * rs_write_it_ws;
    rs_tell_dir_ws               * rs_tell_it_ws;
    rs_seek_dir_ws               * rs_seek_it_ws;

    rs_select_dir                * rs_select_it;
    rs_change_dir                * rs_change_it;
    rs_give_server_title         * rs_give_it_title;
};


#define REMOTE_SERVER_magic     0xF908

struct remote_server {
    unsigned short               magic;   /* REMOTE_SERVER_magic */

    struct remote_server_type  * server_type;

    int                          browser_count;
   
    struct folder_info         * current_folder;             /* POP  */
    struct connection_cache    * current_connection;         /* IMAP */

    struct browser_passhm      * passhm;

    struct remote_server       * next_server;
};

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


/* ==== LOCAL MAILBOX ========================================== */

S_(rs_free_record rs_free_static)
static void rs_free_static P_((struct remote_server *rec));
static void rs_free_static(rec)
     struct remote_server *rec;
{
    /* static record do not free */

    panic("MBX PANIC",__FILE__,__LINE__,"rs_free_static",
	  "rs_free_static() called",0);

}

S_(rs_init_record rs_init_static)
static void rs_init_static P_((struct remote_server *rec));
static void rs_init_static(rec)
     struct remote_server *rec;
{
    /* static record do not alloc */

    panic("MBX PANIC",__FILE__,__LINE__,"rs_init_static",
	  "rs_init_static() called",0);
}



S_(rs_dir_type  rs_dir_static_type)
struct browser_type * rs_dir_static_type P_((struct remote_server *rec));
struct browser_type * rs_dir_static_type(rec)
     struct remote_server *rec;
{
    return &dummy_browser;
}


S_(rs_connect_browser rs_connect_local)
static int rs_connect_local P_((struct remote_server *server,
				struct folder_browser *dir));
static int rs_connect_local(server,dir) 
     struct remote_server *server;
     struct folder_browser *dir;
{
    if (&dummy_browser != dir->type)
	panic("BROWSER PANIC",__FILE__,__LINE__,"rs_connect_local",	
	      "Browser type is not dummy_browser",0);

    if (REMOTE_SERVER_magic != server->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"rs_connect_local",
	      "Bad magic type",0);

    DPRINT(Debug,16,(&Debug,
		     "rs_connect_local: server=%p, dir=%p\n",
		     server,dir)); 

    dummy_browser_set_remote(dir,NULL,
			     server->passhm  /* This is not needed ? */);

    DPRINT(Debug,16,(&Debug, "rs_connect_local=1\n"));
    return 1;
}

S_(rs_folder_from_remote rs_folder_from_local)
static struct folder_info   * rs_folder_from_local P_((struct remote_server *server,
						       struct folder_browser *dir,
						       int treat_as_spooled));

static struct folder_info   * rs_folder_from_local(server,dir,treat_as_spooled)
     struct remote_server *server; /* NOT USED */
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info *res = NULL;

    DPRINT(Debug,16,(&Debug,
		     "rs_folder_from_local: server=%p, dir=%p, treat_as_spooled=%d\n",
		     server,dir,treat_as_spooled)); 

    /* real_folder_from_local can be called with any folder_browser type */

    res = real_folder_from_local(dir,treat_as_spooled,1); 

    DPRINT(Debug,16,(&Debug,
		     "rs_folder_from_local=%d\n",res));

    return res;
}


S_(rs_selection_is_folder rs_selection_is_local)
static int  rs_selection_is_local P_((struct remote_server *server,
				      struct folder_browser *dir,
				      struct folder_info *folder));
static int  rs_selection_is_local(server,dir,folder) 
     struct remote_server *server;  /* NOT USED */
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int res = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_selection_is_local: server=%p, dir=%p, folder=%d\n",
		     server,dir,folder)); 


    res = real_selection_is_local(dir,folder,1);

    DPRINT(Debug,16,(&Debug,
		     "rs_selection_is_local=%d\n",res));

    return res;
}

S_(rs_create_selection_folder rs_create_selection_local)
static int  rs_create_selection_local P_((struct remote_server *server,
					  struct folder_browser *dir));
static int  rs_create_selection_local(server,dir)
     struct remote_server *server; /* NOT USED */
     struct folder_browser *dir;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_create_selection_local: server=%p, dir=%p\n",
		     server,dir)); 

    ret = real_create_selection_local(dir,1);

    DPRINT(Debug,16,(&Debug,
		     "rs_create_selection_local=%d\n",ret));

    return ret;
}

S_(rs_prepare_write_folder rs_prepare_write_local)
static int rs_prepare_write_local P_((struct remote_server *server,
				      struct folder_browser *dir,
				      WRITE_STATE ptr));
static int rs_prepare_write_local(server,dir,ptr)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;
    char * name = NULL;

    DPRINT(Debug,16,(&Debug,
		     "rs_prepare_write_local: server=%p, dir=%p\n",
		     server,dir)); 

    name =  resolve_selection_local(dir,1);
    if (name) {

	if (! ptr->a.remote) 
	    ptr->a.remote =  malloc_remote_server_ws(&dummy_browser);
	else if (RSERVER_WS_magic != ptr->a.remote->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"rs_prepare_write_local",
		  "Bad magic type",0);
	
	if (&dummy_browser != ptr->a.remote->slave_type)
	    panic("MBX PANIC",__FILE__,__LINE__,"rs_prepare_write_local",
		  "Bad slave type",0);
		
	ret = real_prepare_write_local(dir, ptr->a.remote->slave_write_state,
				       name);
	
	free(name);
    } else {

	DPRINT(Debug,16,(&Debug,
			 "rs_prepare_write_local: no name\n"));
	
    }

    DPRINT(Debug,16,(&Debug,
		     "rs_prepare_write_local=%d\n",ret));


    return ret;
}

S_(rs_end_write_folder rs_end_write_local)
static int rs_end_write_local P_((struct remote_server *server,
				      struct folder_browser *dir,
				      WRITE_STATE ptr));
static int rs_end_write_local(server,dir,ptr)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;
    char * name = NULL;

    DPRINT(Debug,16,(&Debug,
		     "rs_end_write_local: server=%p, dir=%p\n",
		     server,dir)); 

    name =  resolve_selection_local(dir,1);
    if (name) {

	if (! ptr->a.remote) {
	    DPRINT(Debug,16,(&Debug,
			     "rs_end_write_local: no remote write state\n"));

	    goto fail;
	}

	if (RSERVER_WS_magic != ptr->a.remote->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"rs_end_write_local",
		  "Bad magic type",0);
	
	if (&dummy_browser != ptr->a.remote->slave_type)
	    panic("MBX PANIC",__FILE__,__LINE__,"rs_end_write_local",
		  "Bad slave type",0);
		
	ret = real_end_write_local(dir, ptr->a.remote->slave_write_state,
				   name);
	
    fail:
	free(name);
    } else {

	DPRINT(Debug,16,(&Debug,
			 "rs_end_write_local: no name\n"));
	
    }

    DPRINT(Debug,16,(&Debug,
		     "rs_end_write_local=%d\n",ret));


    return ret;
}

S_(rs_start_we_dir rs_start_we_local)
static int rs_start_we_local P_((struct remote_server *server,
				 struct folder_browser *dir,
				 WRITE_STATE write_state_ptr,
				 int write_envelope,
				 struct header_rec *current_header,
				 int *env_flags));
static int rs_start_we_local(server,dir,write_state_ptr,write_envelope,
			     current_header,env_flags)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
     int *env_flags;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_start_we_local: server=%p, dir=%p\n",
		     server,dir)); 

    *env_flags = 0;

    if (write_envelope) {
	if (! write_state_ptr->a.remote) {
	    DPRINT(Debug,16,(&Debug,
			     "rs_start_we_local: no remote write state\n"));
	    
	    goto fail;
	}
	
	if (RSERVER_WS_magic != write_state_ptr->a.remote->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"rs_start_we_local",
		  "Bad magic type",0);
	
	if (&dummy_browser != write_state_ptr->a.remote->slave_type)
	    panic("MBX PANIC",__FILE__,__LINE__,"rs_start_we_local",
		  "Bad slave type",0);
	
	ret = real_start_we_local(dir,
				  write_state_ptr->a.remote->slave_write_state,
				  current_header,
				  env_flags);
    } else
	ret = 1;

 fail:
    DPRINT(Debug,16,(&Debug,
		     "rs_start_we_local=%d\n",ret));
    return ret;
}

S_(rs_end_we_dir rs_end_we_local)
static int rs_end_we_local P_((struct remote_server *server,
			       struct folder_browser *dir,
			       WRITE_STATE write_state_ptr,
			       int write_envelope,
			       struct header_rec *current_header));
static int rs_end_we_local(server,dir,write_state_ptr,write_envelope,
			   current_header)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_end_we_local: server=%p, dir=%p\n",
		     server,dir)); 

    if (write_envelope) {
	if (! write_state_ptr->a.remote) {
	    DPRINT(Debug,16,(&Debug,
			     "rs_end_we_local: no remote write state\n"));
	    
	    goto fail;
	}
	
	if (RSERVER_WS_magic != write_state_ptr->a.remote->magic) 
	    panic("MBX PANIC",__FILE__,__LINE__,"rs_end_we_local",
		  "Bad magic type",0);
	
	if (&dummy_browser != write_state_ptr->a.remote->slave_type)
	    panic("MBX PANIC",__FILE__,__LINE__,"rs_end_we_local",
		  "Bad slave type",0);
	
	ret = real_end_we_local(dir,
				  write_state_ptr->a.remote->slave_write_state,
				  current_header);

    } else
	ret = 1;

 fail:
    DPRINT(Debug,16,(&Debug,
		     "rs_end_we_local=%d\n",ret));

    return ret;
}

S_(rs_write_dir_ws rs_write_local_ws)
static int rs_write_local_ws P_((struct remote_server *server,
				  struct folder_browser *dir,
				  WRITE_STATE ptr,
				  int l, const char *buffer));
static int rs_write_local_ws(server,dir,ptr,l,buffer)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE ptr;
     int l; 
     const char *buffer;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_write_local_ws: server=%p, dir=%p\n",
		     server,dir)); 
    
    if (! ptr->a.remote) {
	DPRINT(Debug,16,(&Debug,
			 "rs_write_local_ws: no remote write state\n"));
	
	goto fail;
    }


    if (RSERVER_WS_magic != ptr->a.remote->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"rs_write_local_ws",
	      "Bad magic type",0);
    
    if (&dummy_browser != ptr->a.remote->slave_type)
	panic("MBX PANIC",__FILE__,__LINE__,"rs_write_local_ws",
	      "Bad slave type",0);

    ret = real_browser_write_ws(dir,ptr->a.remote->slave_write_state,l,buffer);


 fail:
    DPRINT(Debug,16,(&Debug,
		     "rs_write_local_ws=%d\n",ret));

    return ret;
}


/* Returns -1L on failure */
S_(rs_tell_dir_ws rs_tell_local_ws)
static long rs_tell_local_ws  P_((struct remote_server *server,
				  struct folder_browser *dir,
				  WRITE_STATE write_state_ptr));

static long rs_tell_local_ws(server,dir,write_state_ptr)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    int ret = -1;

    DPRINT(Debug,16,(&Debug,
		     "rs_tell_local_ws: server=%p, dir=%p\n",
		     server,dir)); 
    
    if (! write_state_ptr->a.remote) {
	DPRINT(Debug,16,(&Debug,
			 "rs_tell_local_ws: no remote write state\n"));
	
	goto fail;
    }


    if (RSERVER_WS_magic != write_state_ptr->a.remote->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"rs_tell_local_ws",
	      "Bad magic type",0);
    
    if (&dummy_browser != write_state_ptr->a.remote->slave_type)
	panic("MBX PANIC",__FILE__,__LINE__,"rs_tell_local_ws",
	      "Bad slave type",0);

    ret = real_browser_tell_ws(dir,write_state_ptr->a.remote->slave_write_state);

 fail:
    DPRINT(Debug,16,(&Debug,
		     "rs_tell_local_ws=%ld\n",ret));

    return ret;

}

/* Returns 0 on failure */
S_(rs_seek_dir_ws rs_seek_local_ws)
static int rs_seek_local_ws P_((struct remote_server *server,
				struct folder_browser *dir,
				WRITE_STATE write_state_ptr,
				long pos));
static int rs_seek_local_ws(server,dir,write_state_ptr,pos)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     long pos;
{
    int ret= 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_seek_local_ws: server=%p, dir=%p\n",
		     server,dir)); 
    
    if (! write_state_ptr->a.remote) {
	DPRINT(Debug,16,(&Debug,
			 "rs_seek_local_ws: no remote write state\n"));
	
	goto fail;
    }


    if (RSERVER_WS_magic != write_state_ptr->a.remote->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"rs_seek_local_ws",
	      "Bad magic type",0);
    
    if (&dummy_browser != write_state_ptr->a.remote->slave_type)
	panic("MBX PANIC",__FILE__,__LINE__,"rs_seek_local_ws",
	      "Bad slave type",0);

    ret = real_browser_seek_ws(dir,write_state_ptr->a.remote->slave_write_state,
			       pos);

 fail:
    DPRINT(Debug,16,(&Debug,
		     "rs_seek_local_ws=%d\n",ret));
    return ret;
}

/* Return 1 if changed */
static int browser_change_to_real P_((struct folder_browser *dir));

/* Returns 0 on failure */
S_(rs_select_dir rs_select_local)
static int rs_select_local P_((struct remote_server *server,
			       struct folder_browser *dir,
			       const struct string *rel_itemname, /*NONNULL */
			       struct string **dispname,
			       int *newpos));
static int rs_select_local(server,dir,rel_itemname,dispname,newpos) 
     struct remote_server *server;
     struct folder_browser *dir;
     const struct string *rel_itemname; /*NONNULL */
     struct string **dispname;
     int *newpos;
{
    int ret = 0;

    struct string * INBOX = 
	new_string2(system_charset,s2us("INBOX"));


    DPRINT(Debug,16,(&Debug,
		     "rs_select_local: server=%p, dir=%p\n",
		     server,dir));


    if (0 == string_cmp(rel_itemname,INBOX,-99 /* Unknown indicator */))  {

	unsigned int defaultfile_keyword = 0;
	unsigned int defaultfile_cmask   = 0;

	const char * default_val = give_dt_estr_as_str(&defaultfile_e,
						       "incoming-mailbox",
						       &defaultfile_keyword,
						       &defaultfile_cmask);
	
	DPRINT(Debug,16,(&Debug,
			 "rs_select_local: Is incoming mailbox: No directory, %S\n",
			 rel_itemname));
	
	if (dir->dirname || dir->sys_dir) {
	    /* Make empty selection folder */
	    clear_dir_vector(dir);
	}
	
	if (dir->dirname)
	    free_string(&dir->dirname);

	if (dir->sys_dir) {
	    free(dir->sys_dir);
	    dir->sys_dir = NULL;
	}
	
	if (default_val) {

	    /* Explicite disallow $MAIL to be interpreted as remote mailbox */
	    if (0 != (defaultfile_cmask & FOLDER_is_local_file)) {

		int F = 0;
		
		DPRINT(Debug,16,(&Debug,
				 "rs_select_local: FOLDER_is_local_file is set for value\n"));

		F = local_selection_stat(default_val,
					 /* struct last_read_cache * */
					 NULL);
	    
		/* WARNING: set_dir_selection does not allocate strings -- 
		 *          it just assign pointers!
		 */
		set_dir_selection(dir,safe_strdup("INBOX"),
				  dup_string(rel_itemname),0|F);
		
		ret = 1;
		
	    } else  if ('/' == default_val[0]) {
		int F = 0;
	    
		DPRINT(Debug,16,(&Debug,
				 "rs_select_local: Using local incoming mailbox\n"));
		
		F = local_selection_stat(default_val,
					 /* struct last_read_cache * */
					 NULL);
		
		/* WARNING: set_dir_selection does not allocate strings -- 
		 *          it just assign pointers!
		 */
		set_dir_selection(dir,safe_strdup("INBOX"),
				  dup_string(rel_itemname),0|F);
		
		ret = 1;		
	    } 

	}

    } else {
	/* & must refer to maildir  */

	/* give_dt_estr_as_str adds / to end */
	const char * folders_val = give_dt_estr_as_str(&folders_e,"maildir",
						       NULL,NULL);

	if (folders_val) {


	    if (change_folder_browser_type(dir,
#ifdef DIROPS
					   &local_browser
#else
					   &dummy_browser
#endif
					   )) {

		struct string * new_buffer = new_string2(local_fs_charset,
							 cs2us(folders_val));

		append_string(&new_buffer,rel_itemname,0);


		ret = dir->type->browser_select_it(dir,new_buffer,dispname,
						   newpos);
		
		if (!ret) {
		    DPRINT(Debug,15,(&Debug,"rs_select_local: real browser_select_it [relative=%S] FAILED\n",
				     new_buffer));	
		}

		free_string(&new_buffer);

	    } else {
		DPRINT(Debug,16,(&Debug,"rs_select_local: .. remote browser does not change type ?? ..\n"));
	    }
	
	} else {
	    DPRINT(Debug,16,(&Debug,
			     "rs_select_local: maildir not set\n"));
	}
    }


    free_string(&INBOX);
    

    DPRINT(Debug,16,(&Debug,
		     "rs_select_local=%d\n",
		     ret)); 
    
    return ret;
}

S_(rs_change_dir rs_change_local)
int rs_change_local P_((struct remote_server *server,
			struct folder_browser *dir,
			const struct string *rel_dirname,
			struct string **dispname));
int rs_change_local(server,dir,rel_dirname,dispname)
     struct remote_server *server;
     struct folder_browser *dir;
     const struct string *rel_dirname;
     struct string **dispname;
{
    int ret = 0;

    /* give_dt_estr_as_str adds / to end */
    const char * folders_val = give_dt_estr_as_str(&folders_e,"maildir",
						   NULL,NULL);
    
    DPRINT(Debug,16,(&Debug,
		     "rs_change_local: server=%p, dir=%p\n",
		     server,dir)); 

    /* & must refer to maildir  */

    if (folders_val) {
	
	if (change_folder_browser_type(dir,
#ifdef DIROPS
				       &local_browser
#else
				       &dummy_browser
#endif
				       )) {
	    
	    
	    struct string * new_buffer = new_string2(local_fs_charset,
						     cs2us(folders_val));
	    
	    if (rel_dirname)
		append_string(&new_buffer,rel_dirname,0);
	    
	    
	    ret = dir->type->browser_change_it(dir,new_buffer,dispname);
	    
	    if (!ret) {
		DPRINT(Debug,15,(&Debug,"rs_change_local: real browser_change_it [relative=%S] FAILED\n",
				 new_buffer));	
	    }
	    
	    free_string(&new_buffer);

	} else {

	    DPRINT(Debug,16,(&Debug,"rs_change_local: .. remote browser does not change type ?? ..\n"));

	}
	
    } else {
	DPRINT(Debug,16,(&Debug,
			 "rs_change_local: maildir not set\n"));
    }

    DPRINT(Debug,16,(&Debug,
		     "rs_change_local=%d\n",
		     ret)); 

    return ret;
}

/* Gives server type and username@server handle as string -- 
   caller must free_string() */
S_(rs_give_server_title rs_give_local_title)
static struct string * rs_give_local_title P_((const struct remote_server 
					       *server));
static struct string * rs_give_local_title(server)
     const struct remote_server *server;
{
    const char * folders_val = give_dt_estr_as_str(&folders_e,"maildir",
						   NULL,NULL);

    if (folders_val)
	return format_string(CATGETS(elm_msg_cat, MeSet, 
				     MeRemBrowserLocalTitleDir,
				     "local incoming mailbox or folder on directory %s"),
			     folders_val);
	

    return format_string(CATGETS(elm_msg_cat, MeSet, MeRemBrowserLocalTitle,
			 "local mailbox"));
}

static struct remote_server_type local_server_type = {
    RSERVER_TYPE_magic,

    rs_free_static,
    rs_init_static,
    rs_dir_static_type,
    rs_connect_local,
    rs_folder_from_local,
    rs_selection_is_local,
    rs_create_selection_local,
    rs_prepare_write_local,
    rs_end_write_local,
    rs_start_we_local,
    rs_end_we_local,
    rs_write_local_ws,
    rs_tell_local_ws,
    rs_seek_local_ws,
    rs_select_local,
    rs_change_local,
    rs_give_local_title
};

static struct remote_server local_server = {
    REMOTE_SERVER_magic,
    & local_server_type,
    0,

    NULL
};

struct remote_server  * const LOCAL_SERVER = &local_server;

/* =========== POP SERVER TYPE ====================================== */

S_(rs_free_record rs_free_real)
static void rs_free_real P_((struct remote_server *rec));
static void rs_free_real(rec)
     struct remote_server *rec;
{
    /* No private data */
}

S_(rs_init_record rs_init_real)
static void rs_init_real P_((struct remote_server *rec));
static void rs_init_real(rec)
     struct remote_server *rec;
{
    /* No private data */
}


S_(rs_connect_browser rs_connect_pop)
static int rs_connect_pop P_((struct remote_server *server,
				struct folder_browser *dir));
static int rs_connect_pop(server,dir) 
     struct remote_server *server;
     struct folder_browser *dir;
{
    
    if (&dummy_browser != dir->type)
	panic("BROWSER PANIC",__FILE__,__LINE__,"rs_connect_pop",	
	      "Browser type is not dummy_browser",0);

    if (REMOTE_SERVER_magic != server->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"rs_connect_pop",
	      "Bad magic type",0);

    DPRINT(Debug,16,(&Debug,
		     "rs_connect_pop: server=%p, dir=%p\n",
		     server,dir)); 

    dummy_browser_set_remote(dir,NULL,
			     server->passhm);

#ifdef REMOTE_MBX

    if (server->current_folder) {
	const struct remote_account *ra = 
	    give_pop_remote_account(server->current_folder);
	char * remote = NULL;

	if (!ra) {
	    DPRINT(Debug,11,(&Debug,"rs_connect_pop: no remote account\n"));

	    goto fail;
	}

	remote = elm_message(FRM("%s@%s"),ra->username,ra->host);
	
	dummy_browser_set_remote(dir,remote,
				 server->passhm);

	DPRINT(Debug,12,(&Debug,
			 "rs_connect_pop: remote=%s\n",
			 remote));


	DPRINT(Debug,16,(&Debug,
			 "rs_connect_pop=1\n"));

	return 1;
    }

 fail:
#endif

    DPRINT(Debug,16,(&Debug,
		     "rs_connect_pop=0\n"));

    return 0;
}


S_(rs_folder_from_remote rs_folder_from_pop)
static struct folder_info   * rs_folder_from_pop P_((struct remote_server *server,
						       struct folder_browser *dir,
						       int treat_as_spooled));

static struct folder_info   * rs_folder_from_pop(server,dir,treat_as_spooled)
     struct remote_server *server;
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info *res = NULL;

    DPRINT(Debug,16,(&Debug,
		     "rs_folder_from_pop: server=%p, dir=%p, treat_as_spooled=%d\n",
		     server,dir,treat_as_spooled)); 


    if (dir->sys_dir && dir->sys_dir[0]) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeRemBrowserNotPopDir,
			  "Current remote server is POP type. Directory %S does not exist."),
		  dir->dirname);

	
	DPRINT(Debug,16,(&Debug,
			 "rs_folder_from_pop=NULL\n"));

	return NULL;
    }

    if (dir->selection->sys_name &&
	0 != istrcmp(dir->selection->sys_name,"INBOX")) {

	lib_error(CATGETS(elm_msg_cat, MeSet, MeRemBrowserNotPopFolder,
			  "Current remote server is POP type. Folder %S does not exist."),
		  dir->selection->disp_name);

	DPRINT(Debug,16,(&Debug,
			 "rs_folder_from_pop=NULL\n"));

	return NULL;		  
    }

#ifdef REMOTE_MBX

    /* NOTE:   This actually can not succeed, because session to same
              user can not open to same pop server on same time

	      Also rs_selection_is_pop() reports that folder is same,
	      so that this is not called, when folder is tried to change
    */

    if (server->current_folder) {
	
	struct remote_account  current_ra;
	
	const struct remote_account *ra = 
	    give_pop_remote_account(server->current_folder);
	
	zero_remote_account(& current_ra);

	if (connect_remote_account_1(& current_ra,ra)) {

	    struct remote_account *status = NULL;
	    enum itls_status have_initial_tls = itls_none;

	    res = mbx_new_folder();
	    
	    res -> cur_folder_sys = 
		elm_message(FRM("%s@%s"),ra->username,ra->host);
	    res -> cur_folder_disp =
		format_string(FRM("%s@%s"),ra->username,ra->host);

	    switch((have_initial_tls = 
		    remote_account_init_tls(& current_ra))) {

	    case itls_unsupported:
	    case itls_failed:
		
		goto set_dummy;

	    case itls_none:
		break;
		
	    case itls_done:
		DPRINT(Debug,8,(&Debug,
				"rs_folder_from_pop: Initial TLS done (no STLS)\n"));
	    }
	    	   
	    res -> folder_type = POP_MBX;
	    res -> folder_type->init_it(res);
	    
	    status = set_pop_connection(res,&current_ra,
					server->passhm);
	    
	    if (status) {
		/* In here we can use temp_dir instead of default_temp because
		 * that file does not act as session lock ...
		 */
		
		set_remote_tempmbox(res,status);
		
	    } else {
	    set_dummy:
		DPRINT(Debug,12,(&Debug,
				 "rs_folder_from_pop: Returning dummy folder\n"));
		
		res -> folder_type = NO_NAME;
		res -> folder_type->init_it(res);
	    }
	} else {
	    DPRINT(Debug,16,(&Debug,
			     "rs_folder_from_pop: connect to remote account failed\n"));
	}
	
	clear_remote_account(&current_ra);    
    } else {
	DPRINT(Debug,16,(&Debug,
			 "rs_folder_from_pop: No current folder\n"));
    }
#endif
    
    DPRINT(Debug,16,(&Debug,
		     "rs_folder_from_pop=%p\n",res));

    return res;
}

S_(rs_selection_is_folder rs_selection_is_pop)
static int  rs_selection_is_pop P_((struct remote_server *server,
				      struct folder_browser *dir,
				      struct folder_info *folder));
static int  rs_selection_is_pop(server,dir,folder) 
     struct remote_server *server;  
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_selection_is_pop: server=%p, dir=%p, folder=%d\n",
		     server,dir,folder)); 

#ifdef REMOTE_MBX

    if (folder->folder_type != &pop_mbx) {

	DPRINT(Debug,11,(&Debug,
			 "real_selection_is_pop: Not a POP folder\n"));
	ret = 0;

    } else if (server->current_folder) {

	const struct remote_account *ra = 
	    give_pop_remote_account(server->current_folder);
	const struct remote_account *ra1 = 
	    give_pop_remote_account(folder);


	if (!ra || !ra1) {
	    DPRINT(Debug,16,(&Debug,"rs_selection_is_pop: no remote account\n"));

	    goto fail;
	}

	if (0 != strcmp(ra->username,ra1->username) ||
	    0 != strcmp(ra->host,ra1->host)) {
	    DPRINT(Debug,16,(&Debug, "rs_selection_is_pop: Host or username does not match\n"));
	    ret = 0;
	} else if (dir->sys_dir && dir->sys_dir[0]) {

	    DPRINT(Debug,16,(&Debug, "rs_selection_is_pop: directory %s -- not POP\n",
			     dir->sys_dir)); 
	    ret = 0;

	} else if (dir->selection->sys_name &&
		   0 != istrcmp(dir->selection->sys_name,"INBOX")) {
	    
	    DPRINT(Debug,16,(&Debug, "rs_selection_is_pop: folder %s -- not POP\n",
			     dir->selection->sys_name)); 
	    ret = 0;
	    
	} else {
	    ret = 1;
	}
    } else {
	DPRINT(Debug,16,(&Debug, "rs_selection_is_pop: remote server - no current folder!\n"));
    }

 fail:

#endif

    DPRINT(Debug,16,(&Debug,
		     "rs_selection_is_pop=%d\n",ret));

    return ret;
}

S_(rs_create_selection_folder rs_create_selection_pop)
static int  rs_create_selection_pop P_((struct remote_server *server,
					  struct folder_browser *dir));
static int  rs_create_selection_pop(server,dir)
     struct remote_server *server;
     struct folder_browser *dir;
{

#ifdef REMOTE_MBX

    if (server->current_folder) {
	
	const struct remote_account *ra = 
	    give_pop_remote_account(server->current_folder);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeAtCreatRemoteFolderNotSupp,
			  "Creating of remote folder for %s@%s not supported"),
		  ra->username,ra->host);
    }

#endif

    DPRINT(Debug,16,(&Debug,
		     "rs_create_selection_pop=0\n"));

    return 0;
}

S_(rs_prepare_write_folder rs_prepare_write_proxy)
static int rs_prepare_write_proxy P_((struct remote_server *server,
				      struct folder_browser *dir,
				      WRITE_STATE ptr));
static int rs_prepare_write_proxy(server,dir,ptr)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_prepare_write_proxy: server=%p, dir=%p\n",
		     server,dir)); 



    /* XXXX we change folder type so that writing is possible
     *
     *      compare to browser_prepare_write_dummy()
     *
     */

    dir->type->free_ws_fields_it(ptr);       /* Need reinit of state */

    if (browser_change_to_real(dir)) {
	
	dir->type->zero_ws_fields_it(ptr);   /* Hopefully ok ....... */
	ret = dir->type->browser_prepare_write_it(dir,ptr);
	
	if (!ret) {
	    DPRINT(Debug,16,(&Debug,
			     "rs_prepare_write_proxy: real browser_prepare_write_it FAILED\n"));
	}


    } else {
	DPRINT(Debug,16,(&Debug,
			 "rs_prepare_write_proxy: failed to change browser type to real type\n"));
    }

    DPRINT(Debug,16,(&Debug,
		     "rs_prepare_write_proxy=%d\n",ret));

    return ret;
}


S_(rs_end_write_folder rs_end_write_proxy)
static int rs_end_write_proxy P_((struct remote_server *server,
				  struct folder_browser *dir,
				  WRITE_STATE ptr));
static int rs_end_write_proxy(server,dir,ptr)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE ptr;
{

    DPRINT(Debug,16,(&Debug,
		     "rs_end_write_proxy=0: server=%p, dir=%p\n",
		     server,dir)); 
    
    return 0;
}


S_(rs_start_we_dir rs_start_we_dummy)
static int rs_start_we_dummy P_((struct remote_server *server,
				 struct folder_browser *dir,
				 WRITE_STATE write_state_ptr,
				 int write_envelope,
				 struct header_rec *current_header,
				 int *env_flags));
static int rs_start_we_dummy(server,dir,write_state_ptr,write_envelope,
			     current_header,env_flags)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
     int *env_flags;
{

    DPRINT(Debug,16,(&Debug,
		     "rs_start_we_dummy=0: server=%p, dir=%p\n",
		     server,dir)); 
    
    return 0;
}

S_(rs_end_we_dir rs_end_we_dummy)
static int rs_end_we_dummy P_((struct remote_server *server,
			       struct folder_browser *dir,
			       WRITE_STATE write_state_ptr,
			       int write_envelope,
			       struct header_rec *current_header));
static int rs_end_we_dummy(server,dir,write_state_ptr,write_envelope,
			   current_header)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
{

    DPRINT(Debug,16,(&Debug,
		     "rs_end_we_dummy=0: server=%p, dir=%p\n",
		     server,dir)); 
    
    return 0;
}


S_(rs_write_dir_ws rs_write_dummy_ws)
static int rs_write_dummy_ws P_((struct remote_server *server,
				 struct folder_browser *dir,
				 WRITE_STATE ptr,
				 int l, const char *buffer));
static int rs_write_dummy_ws(server,dir,ptr,l,buffer)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE ptr;
     int l; 
     const char *buffer;
{
    DPRINT(Debug,16,(&Debug,
		     "rs_write_dummy_ws=0: server=%p, dir=%p\n",
		     server,dir)); 

    return 0;
}


/* Returns -1L on failure */
S_(rs_tell_dir_ws rs_tell_dummy_ws)
static long rs_tell_dummy_ws  P_((struct remote_server *server,
				  struct folder_browser *dir,
				  WRITE_STATE write_state_ptr));

static long rs_tell_dummy_ws(server,dir,write_state_ptr)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    DPRINT(Debug,16,(&Debug,
		     "rs_tell_dummy_ws=-1: server=%p, dir=%p\n",
		     server,dir)); 

    return -1L;
}


/* Returns 0 on failure */
S_(rs_seek_dir_ws rs_seek_dummy_ws)
static int rs_seek_dummy_ws P_((struct remote_server *server,
				struct folder_browser *dir,
				WRITE_STATE write_state_ptr,
				long pos));
static int rs_seek_dummy_ws(server,dir,write_state_ptr,pos)
     struct remote_server *server;
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     long pos;
{
    DPRINT(Debug,16,(&Debug,
		     "rs_seek_dummy_ws=0: server=%p, dir=%p\n",
		     server,dir)); 

    return 0;
}



/* Returns 0 on failure */
S_(rs_select_dir rs_select_proxy)
static int rs_select_proxy P_((struct remote_server *server,
			     struct folder_browser *dir,
			     const struct string *rel_itemname, /*NONNULL */
			     struct string **dispname,
			     int *newpos));
static int rs_select_proxy(server,dir,rel_itemname,dispname,newpos) 
     struct remote_server *server;
     struct folder_browser *dir;
     const struct string *rel_itemname; /*NONNULL */
     struct string **dispname;
     int *newpos;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_select_proxy: server=%p, dir=%p\n",
		     server,dir)); 


    if (browser_change_to_real(dir)) {

	/* POP and INBOX lead finally to IMAP mailbox on select */

	ret = dir->type->browser_select_it(dir,rel_itemname,dispname,newpos);
	if (!ret) {
	    DPRINT(Debug,15,(&Debug,"rs_select_proxy: real browser_select_it FAILED\n"));	
	}
	    
    } else {
	DPRINT(Debug,16,(&Debug,
			 "rs_select_proxy: failed to change browser type to real type\n"));
    }

    DPRINT(Debug,16,(&Debug,
		     "rs_select_proxy=%d\n",
		     ret)); 


    return ret;
}

S_(rs_change_dir rs_change_proxy)
int rs_change_proxy P_((struct remote_server *server,
			struct folder_browser *dir,
			const struct string *rel_dirname,
			struct string **dispname));
int rs_change_proxy(server,dir,rel_dirname,dispname)
     struct remote_server *server;
     struct folder_browser *dir;
     const struct string *rel_dirname;
     struct string **dispname;
{
    int ret = 0;
    
    DPRINT(Debug,16,(&Debug,
		     "rs_change_proxy: server=%p, dir=%p\n",
		     server,dir)); 


    if (browser_change_to_real(dir)) {
	
	/* POP and INBOX lead finally to IMAP mailbox on change */
	
	ret = dir->type->browser_change_it(dir,rel_dirname,dispname);
	if (!ret) {
	    DPRINT(Debug,15,(&Debug,"rs_change_proxy: real browser_change_it FAILED\n"));	
	}
	    
    } else {
	DPRINT(Debug,16,(&Debug,
			 "rs_change_proxy: failed to change browser type to real type\n"));
    }

    DPRINT(Debug,16,(&Debug,
		     "rs_change_proxy=%d\n",
		     ret)); 


    return ret;
}

/* Gives server type and username@server handle as string -- 
   caller must free_string(), may return NULL */
S_(rs_give_server_title rs_give_pop_title)
static struct string * rs_give_pop_title P_((const struct remote_server 
						*server));
static struct string * rs_give_pop_title(server)
     const struct remote_server *server;
{
#ifdef REMOTE_MBX
    if (server->current_folder) {
	const struct remote_account *ra = 
	    give_pop_remote_account(server->current_folder);
	struct string * S = NULL;

	if (!ra) {
	    DPRINT(Debug,16,(&Debug,
			     "rs_give_pop_title=NULL\n"));


	    return NULL;
	}

	S = format_string(CATGETS(elm_msg_cat, MeSet, MeRemBrowserPopTitle,
				  "POP or IMAP folder on %s@%s"),
			  ra->username,ra->host);

	DPRINT(Debug,16,(&Debug,
			 "rs_give_pop_title=%S\n",S));
		
	return S;
    } 
#endif

    DPRINT(Debug,16,(&Debug,
		     "rs_give_pop_title=NULL\n"));

    return NULL;  /* may return NULL */    
}


static struct remote_server_type pop_server_type = {
    RSERVER_TYPE_magic,

    rs_free_real,
    rs_init_real,
    rs_dir_static_type,
    rs_connect_pop,
    rs_folder_from_pop,
    rs_selection_is_pop,
    rs_create_selection_pop,
    rs_prepare_write_proxy,
    rs_end_write_proxy,
    rs_start_we_dummy,
    rs_end_we_dummy,
    rs_write_dummy_ws,
    rs_tell_dummy_ws,
    rs_seek_dummy_ws,
    rs_select_proxy,
    rs_change_proxy,
    rs_give_pop_title
};

struct remote_server_type * const POP_SERVER_TYPE = &pop_server_type;

/* =========== IMAP SERVER TYPE ====================================== */

S_(rs_dir_type  rs_dir_imap_type)
struct browser_type * rs_dir_imap_type P_((struct remote_server *rec));
struct browser_type * rs_dir_imap_type(rec)
     struct remote_server *rec;
{
#ifdef REMOTE_MBX
    return &imap_browser;
#else
    return &dummy_browser;
#endif
}

S_(rs_connect_browser rs_connect_imap)
static int rs_connect_imap P_((struct remote_server *server,
				struct folder_browser *dir));
static int rs_connect_imap(server,dir) 
     struct remote_server *server;
     struct folder_browser *dir;
{

    int ret = 0;

    if (REMOTE_SERVER_magic != server->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"rs_connect_imap",
	      "Bad magic type",0);

    DPRINT(Debug,16,(&Debug,
		     "rs_connect_imap: server=%p, dir=%p\n",
		     server,dir)); 

#ifdef REMOTE_MBX

    if (&imap_browser != dir->type)
	panic("BROWSER PANIC",__FILE__,__LINE__,"rs_connect_imap",	
	      "Browser type is not imap_browser",0);

    if (server->current_connection) {

	const struct remote_account *ra = 
	    give_con_remote_account(server->current_connection);

	struct connection_cache  * CX
	    = locate_from_cache_1(ra,
				  &IMAP_connection,
				  0  /* Default port */);
	
	if (CX) {
	    DPRINT(Debug,16,(&Debug,
			     "rs_connect_imap: Found cached connection\n"));

	    /*  browser_from_connectionX sets CX = NULL */
	    browser_from_connectionX(&CX,dir);

	    /* passhm is null if wrong type */
	    imap_browser_set_passhm(dir,server->passhm);
	    	    
	    ret = 1;

	} else {
			       
	    struct remote_account X;

	    zero_remote_account(&X);

	    if (connect_remote_account_1(&X,ra)) {
	    
		enum itls_status have_initial_tls = itls_none;

		switch((have_initial_tls = 
			remote_account_init_tls(& X))) {
		
		case itls_unsupported:
		case itls_failed:
		
		    goto fail;

		case itls_none:
		    break;
		
		case itls_done:
		    DPRINT(Debug,8,(&Debug,
				    "rs_connect_imap: Initial TLS done (no STLS)\n"));
		}

		if (server->passhm &&
		    browser_passhm_check_type(server->passhm,hmcon_imap)) {
		    
		    ret = join_connection_hm(give_imap_connection(dir),&X,
					     CON_greeting,server->passhm);

		    imap_browser_set_passhm(dir,server->passhm);
		} else {
		    
		    if (server->passhm) {
			DPRINT(Debug,12,(&Debug,
					 "rs_connect_imap: Discarding browser_passhm\n"));
		    }
		    
		    ret = join_connection(give_imap_connection(dir),&X,
					  CON_greeting);
		}
	    	    
		DPRINT(Debug,12,(&Debug,
				 "rs_connect_imap: %s (%d)\n",
				 ret ? "joined" : "join failed",
				 ret));
	    
	    }

	fail:
	    clear_remote_account(&X);    
	}
    }

#endif

    DPRINT(Debug,16,(&Debug,
		     "rs_connect_imap=%d\n",ret));

    return ret;
}

#ifdef REMOTE_MBX
static struct folder_info  * rs_folder_from_imap_1 P_((struct folder_browser *dir,
						       const struct remote_account *ra));
static struct folder_info  * rs_folder_from_imap_1(dir,ra)
     struct folder_browser *dir;
     const struct remote_account *ra;
{
    struct folder_info *res = NULL;

    struct string * path_w  = NULL;
    char          * path    = NULL;
    
    char *  path_head = "";     
    
    if (dir->sys_dir && dir->sys_dir[0]) {
#if REMOTE_DIR_LISTING
	int l = strlen(dir->sys_dir);
	
	if (!dir->a.remote_browser->dir_name_cache) {
	    DPRINT(Debug,1,(&Debug,
			    "rs_folder_from_imap_1: No dir_name_cache, sys_dir=%s\n",
			    dir->sys_dir));
	    
	    goto fail2;
	}
	
	if (! dir->a.remote_browser->cur_sep) {
	    DPRINT(Debug,1,(&Debug,
			    "rs_folder_from_imap: No cur_sept, dir_name_cache=%S, sys_dir=%s\n",
			    dir->a.remote_browser->dir_name_cache,
			    dir->sys_dir));
	    
	    goto fail2;
	}
			    
	if (dir->sys_dir[0] != '/') {
	    path_head = ":";
	}
	
	path  = safe_strdup(dir->sys_dir);
	path_w = dup_string(dir->a.remote_browser->dir_name_cache);
			    
	if (dir->sys_dir[l-1] != dir->a.remote_browser->cur_sep) {
	    
	    char buf[2];
	    buf[0] = dir->a.remote_browser->cur_sep;
	    buf[1] = '\0';
	    
	    path = strmcat(path,buf);
	    fill_ascii_to_string(path_w,1,dir->a.remote_browser->cur_sep);
	}
#else
	goto fail2;
#endif
	
    } else {
	path_head = ":";
    } 
	    
    if (! dir->selection->sys_name) {
	
	struct string * X = new_string2(display_charset,s2us("INBOX"));
	
	path = strmcat(path,"INBOX");
	
	append_string(&path_w,X,1);
	
	free_string(&X);
    } else {
	
	path = strmcat(path,dir->selection->sys_name);
	append_string(&path_w,dir->selection->disp_name,1);
	
    }
    
    res = mbx_new_folder();
    
    res -> cur_folder_sys = 
	elm_message(FRM("%s@%s%s%s"),ra->username,ra->host,
		    path_head,path);
    
    res -> cur_folder_disp =		
	format_string(FRM("%s@%s%s%S"),ra->username,ra->host,
		      path_head,path_w);
    
    
    res -> folder_type = IMAP_MBX;
    res -> folder_type->init_it(res);
    
    res->p->a.imap_mbx.folder            = safe_strdup(path);
    res->p->a.imap_mbx.folder_name_cache = dup_string(path_w);
    
 fail2:
    
    if (path)
	free(path);

    if (path_w)
	free_string(&path_w);

    DPRINT(Debug,15,(&Debug,
		    "rs_folder_from_imap_1=%p (%s)\n",
		     res,
		     res ? "succeed" : "failed"));
    return res;
}
#endif


S_(rs_folder_from_remote rs_folder_from_imap)
static struct folder_info   * rs_folder_from_imap P_((struct remote_server *server,
						      struct folder_browser *dir,
						      int treat_as_spooled));

static struct folder_info   * rs_folder_from_imap(server,dir,treat_as_spooled)
     struct remote_server *server;
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info *res = NULL;

    DPRINT(Debug,16,(&Debug,
		     "rs_folder_from_imap: server=%p, dir=%p, treat_as_spooled=%d\n",
		     server,dir,treat_as_spooled)); 


    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"rs_folder_from_imap",
	      "Bad magic type (dir remote_browser)",0);

    if (REMOTE_SERVER_magic != server->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"rs_folder_from_imap",
	      "Bad magic type (server)",0);
    
#ifdef REMOTE_MBX

    if (server->current_connection) {
	
	
	const struct remote_account *ra = 
	    give_con_remote_account(server->current_connection);
	

	struct connection_cache  * CX
	    = locate_from_cache_1(ra,
				  &IMAP_connection,
				  0  /* Default port */);
	
	if (CX) {
	    DPRINT(Debug,16,(&Debug,
			     "rs_folder_from_imap: Found cached connection\n"));

	    res = rs_folder_from_imap_1(dir,ra);
	    if (!res) {
		     DPRINT(Debug,16,(&Debug,
				      "rs_folder_from_imap: Setup rs_folder_from_imap_1() failed\n"));
		     goto fail_2;
	    }

	    folder_from_connection(CX,res);

	    imap_mbx_set_passhm(res,server->passhm);

	    /* IMAP temporary mailboxs are always put to
	       temp_dir */
	    set_remote_tempmbox(res,
				&(res->p->a.imap_mbx.Ch->C));
	   

	} else {

	    struct remote_account current_ra;

	    zero_remote_account(&current_ra);  

	    if (connect_remote_account_1(&current_ra,ra)) {
	   
		struct connection_cache *XX;

		enum itls_status have_initial_tls = itls_none;

		switch((have_initial_tls = 
			remote_account_init_tls(& current_ra))) {

		case itls_unsupported:
		case itls_failed:
		
		    goto fail_1;
		    
		case itls_none:
		    break;
		    
		case itls_done:
		    DPRINT(Debug,8,(&Debug,
				    "rs_folder_from_imap: Initial TLS done (no STLS)\n"));
		}
     
		res = rs_folder_from_imap_1(dir,ra);

		if (!res) {
		     DPRINT(Debug,16,(&Debug,
				      "rs_folder_from_imap: Setup rs_folder_from_imap_1() failed\n"));
		    goto fail_1;
		}
		
		/* IMAP temporary mailboxs are always put to
		   temp_dir */
		set_remote_tempmbox(res,&current_ra);
		
		/* pick connection */
		XX = res->p->a.imap_mbx.Ch;
		
		if (!join_connection_hm(XX,&current_ra,CON_greeting,
					server->passhm)) {
		    set_imap_connection_state(XX,IMAP_error);
		} else {
		    set_imap_connection_state(XX,IMAP_idle);
		}
		
		imap_mbx_set_passhm(res,server->passhm);
		
		/* Signal to browser_update_imaplisting() for reopening ...*/
		if (NULL == XX->C.stream)
		    XX->state = CON_error;
		
	    } else {
		DPRINT(Debug,16,(&Debug,
				 "rs_folder_from_imap: connect to remote account failed\n"));

	    }

	fail_1:

	    clear_remote_account(&current_ra);  
	}
    } else {
	DPRINT(Debug,16,(&Debug,
			 "rs_folder_from_imap: No current connection\n"));

    }

 fail_2:
#endif

    DPRINT(Debug,16,(&Debug,
		     "rs_folder_from_imap=%p\n",res));

    return res;
}

S_(rs_selection_is_folder rs_selection_is_imap)
static int  rs_selection_is_imap P_((struct remote_server *server,
				     struct folder_browser *dir,
				     struct folder_info *folder));
static int  rs_selection_is_imap(server,dir,folder) 
     struct remote_server *server;  
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_selection_is_imap: server=%p, dir=%p, folder=%d\n",
		     server,dir,folder)); 

#ifdef REMOTE_MBX

    if (folder->folder_type != &imap_mbx) {
	DPRINT(Debug,11,(&Debug, 
			 "rs_selection_is_imap: Not a imap folder\n"));
	ret = 0;
    } else  if (server->current_connection) {

	const struct remote_account *ra = 
	    give_con_remote_account(server->current_connection);

	struct connection_cache *con1 = folder->p->a.imap_mbx.Ch;

	const struct remote_account *ra1 = 
	    give_con_remote_account(con1);

	if (!ra || !ra1) {
	    DPRINT(Debug,16,(&Debug,"rs_selection_is_imap: no remote account\n"));

	    goto fail;
	}

	if (0 != strcmp(ra->username,ra1->username) ||
	    0 != strcmp(ra->host,ra1->host)) {
	    DPRINT(Debug,16,(&Debug, "rs_selection_is_imap: Host or username does not match\n"));
	    ret = 0;
	} else {

	    /* Malloced */
	    char * full_path = remote_browser_full_path(dir);

	    if (!full_path) {
		DPRINT(Debug,16,(&Debug, "rs_selection_is_imap: No full path\n"));
		goto fail;
	    }

	    ret = 0 == strcmp(full_path,folder->p->a.imap_mbx.folder);

	    if (!ret) {
		DPRINT(Debug,11,(&Debug, 
				 "rs_selection_is_imap: %s <> %s\n",
				 full_path,folder->p->a.imap_mbx.folder));
	    }

	    free(full_path);

	}

    } else {
	DPRINT(Debug,16,(&Debug, "rs_selection_is_imap: remote server - no current connection!\n"));
    }

 fail:

#endif

    DPRINT(Debug,16,(&Debug,
		     "rs_selection_is_imap=%d\n",ret));

    return ret;
}

S_(rs_create_selection_folder rs_create_selection_imap)
static int  rs_create_selection_imap P_((struct remote_server *server,
					 struct folder_browser *dir));
static int  rs_create_selection_imap(server,dir)
     struct remote_server *server;
     struct folder_browser *dir;
{
    int ret = 0;

    DPRINT(Debug,16,(&Debug,
		     "rs_create_selection_imap: server=%p, dir=%p\n",
		     server,dir)); 

#ifdef REMOTE_MBX

    /* XXXX we change folder type so that remote browser do not
     *      need to able to do folder listing 
     *
     */

    if (!change_folder_browser_type(dir, &imap_browser)) {
	DPRINT(Debug,16,(&Debug,
			 "rs_create_selection_imap: .. remote browser does not change type ?? ..\n"));

    }

    if (rs_connect_imap(server,dir)) {
	DPRINT(Debug,16,(&Debug,
			 "rs_create_selection_imap: connection to imap server failed\n"));

	/* CANCEL type change  */
	
	change_folder_browser_type(dir,&remote_browser);
	
	goto fail;
    }

    ret =  dir->type->browser_create_selection_it(dir);

    if (!ret) {
	DPRINT(Debug,11,(&Debug,
			 "rs_create_selection_imap: real browser_create_selection_it FAILED\n"));
    }

 fail:
#endif

    DPRINT(Debug,16,(&Debug,
		     "rs_create_selection_imap=%d\n",ret));

    return ret;
}

/* Gives server type and username@server handle as string -- 
   caller must free_string(), may return NULL */
S_(rs_give_server_title rs_give_imap_title)
static struct string * rs_give_imap_title P_((const struct remote_server 
						*server));
static struct string * rs_give_imap_title(server)
     const struct remote_server *server;
{
#ifdef REMOTE_MBX
    if (server->current_connection) {
	const struct remote_account *ra = 
	    give_con_remote_account(server->current_connection);
	struct string * S = NULL;

	if (!ra) {
	    DPRINT(Debug,16,(&Debug,
			     "rs_give_imap_title=NULL\n"));

	    return NULL;
	}

	S = format_string(CATGETS(elm_msg_cat, MeSet, MeRemBrowserImapTitle,
				  "IMAP folder on %s@%s"),
			  ra->username,ra->host);

	DPRINT(Debug,16,(&Debug,
			 "rs_give_imap_title=%S\n",S));


	return S;
    }

#endif
	    DPRINT(Debug,16,(&Debug,
			     "rs_give_imap_title=NULL\n"));

    return NULL;
}

static struct remote_server_type imap_server_type = {
    RSERVER_TYPE_magic,

    rs_free_real,
    rs_init_real,
    rs_dir_imap_type,
    rs_connect_imap,
    rs_folder_from_imap,
    rs_selection_is_imap,
    rs_create_selection_imap,
    rs_prepare_write_proxy,
    rs_end_write_proxy,
    rs_start_we_dummy,
    rs_end_we_dummy,
    rs_write_dummy_ws,
    rs_tell_dummy_ws,
    rs_seek_dummy_ws,
    rs_select_proxy,
    rs_change_proxy,
    rs_give_imap_title
};

struct remote_server_type * const IMAP_SERVER_TYPE = &imap_server_type;


/* =================================================================== */

static struct remote_server * FIRST_SERVER = &local_server;

static void remote_server_free P_((struct remote_server *server));
static void remote_server_free(server)
     struct remote_server *server;
{
    struct remote_server *ptr;
    struct remote_server *previous = NULL;

    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_free",	
	      "Bad server magic",0);
   
    for (ptr =  FIRST_SERVER; ptr; ptr = ptr->next_server) {
	if (ptr == server)
	    break;

	previous = ptr;
    }

    if (ptr != server)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_free",
	      "Server not in list",0);

    if (previous)
	previous->next_server = server->next_server;
    else
	FIRST_SERVER          = server->next_server;
    server->next_server = NULL;


    if (RSERVER_TYPE_magic != server->server_type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_free",	
	      "Bad server type magic",0);

    server->server_type->rs_free_it(server);

    if (server->current_folder)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_free",	
	      "current_folder is set",0);
    if (server->browser_count)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_free",	
	      "browser_count is not zero",0);
    if (server->current_connection)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_free",	
	      "current_connection is set",0);

    if (server->passhm)
	free_browser_passhm(& (server->passhm));

    server->magic  = 0;   /* Invalidate */
    free(server);        
}

struct browser_type * remote_server_real_type(server)
     struct remote_server *server;
{    
    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_real_type", 
	      "Bad server magic",0);
    
    if (RSERVER_TYPE_magic != server->server_type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_real_type", 
	      "Bad server type magic",0);    

    return server->server_type->rs_dir_type_it(server);
}

int connect_remote_server_to_browser(server,dir)
     struct remote_server *server;
     struct folder_browser *dir;
{
    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "connect_remote_server_to_browser", 
	      "Bad server magic",0);
    
    if (RSERVER_TYPE_magic != server->server_type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "connect_remote_server_to_browser", 
	      "Bad server type magic",0);    

    return server->server_type->rs_connect_browser_it(server,dir);

}

/* Gives server type and username@server handle as string -- 
   caller must free_string() */
struct string * give_remote_server_title(server)
     const struct remote_server *server;
{
    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "give_remote_server_title", 
	      "Bad server magic",0);
    
    if (RSERVER_TYPE_magic != server->server_type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "give_remote_server_title", 
	      "Bad server type magic",0);    

    return server->server_type->rs_give_it_title(server);
}

/* Caller must free this  */
struct address * give_remote_server_useraddr(server, useraddr_flags)
     const struct remote_server * server;
     int                        * useraddr_flags;
{
     struct address * ret = NULL;

     if (REMOTE_SERVER_magic != server->magic)
	 panic("BROWSER PANIC",__FILE__,__LINE__,
		   "give_remote_server_useraddr", 
		   "Bad server magic",0);

     if (useraddr_flags) 
	 * useraddr_flags = 0;

#ifdef REMOTE_MBX
     if (server->passhm) {
	 int r  UNUSED_VAROK;

	 DPRINT(Debug,14,(&Debug,
			  "give_remote_server_useraddr: have browser_passhm\n"));

         r = browser_passhm_remote_lookup(server->passhm,
					  NULL,0,NULL,NULL,
					  &ret,
					  useraddr_flags,
					  NULL);
					      

	 DPRINT(Debug,14,(&Debug,
			  "give_remote_server_useraddr: %s user address, browser_passhm_remote_lookup=%d",
			  ret ? "go" : "no", r));
	 
	 switch (r) {
	 case -1 : DPRINT(Debug,14,(&Debug," use fallabck"));   break;
	 case  0 : DPRINT(Debug,14,(&Debug," lookup failed"));  break;
	 case  1 : DPRINT(Debug,14,(&Debug," ok"));             break;
	 }

        if (useraddr_flags) {
               DPRINT(Debug,14,(&Debug,", *useraddr_flags=%d%s%s",
				    *useraddr_flags,
				    (*useraddr_flags) &  USERADDR_SELF_CC ?
				    "  USERADDR_SELF_CC" : "",
				    (*useraddr_flags) &  USERADDR_SELF_BCC ?
				    "  USERADDR_SELF_BCC" : ""));
           
         }

	 DPRINT(Debug,14,(&Debug,"\n"));

     } else {
	 DPRINT(Debug,14,(&Debug,
			  "give_remote_server_useraddr: no browser_passhm\n"));
     }
#endif
     
     return ret;
}

/* ..... */

static struct remote_server * malloc_remote_server P_((struct remote_server_type *t));
static struct remote_server * malloc_remote_server(t)
     struct remote_server_type *t; 
{
    struct remote_server *ret = safe_zero_alloc(sizeof (*ret));
 
    if (t->magic != RSERVER_TYPE_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"malloc_remote_server",
	      "Bad (type) magic",0);

    ret->magic         = REMOTE_SERVER_magic;
    ret->server_type   = t;
    ret->browser_count = 0;
    ret->current_folder      = NULL;
    ret->current_connection  = NULL;
    ret->passhm              = NULL;

    ret->server_type->rs_init_it(ret);

    ret->next_server = FIRST_SERVER;
    
    FIRST_SERVER = ret;

    return ret;
}

static void server_check_free P_((struct remote_server *server));
static void server_check_free(server)
     struct remote_server *server;
{
    if (! server->current_folder &&
	! server->current_connection &&
	! server->browser_count ) {

	if (server == &local_server) {
	    DPRINT(Debug,11,(&Debug,
			     "server_check_free: %p, local_server (static), will not free\n",
			     server));

	} else
	    remote_server_free(server);
    }
}

static void remote_server_dec(server)
     struct remote_server **server;
{
    if (REMOTE_SERVER_magic != (*server)->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_browser_dec",	
	      "Bad (server) magic",0);

    if ((*server)->browser_count < 1)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_browser_dec",
	      "Bad browser count",0);

    (*server)->browser_count--;

    DPRINT(Debug,16,(&Debug,
		     "remote_browser_dec: remote server %p browser count decremented to %d\n",
		     (*server),(*server)->browser_count));

    if (0 == (*server)->browser_count)
	server_check_free(*server);

    (*server) = NULL;  /* No reference via this pointrer */
}

void remote_server_dec_dir(browser)
     struct folder_browser *browser;
{
    if (! browser->default_server)
  	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_dec_dir",
	      "No default server",0);

    remote_server_dec(& browser->default_server);
}

static void remote_browser_dec_current P_((struct folder_browser *browser));
static void remote_browser_dec_current(browser)
     struct folder_browser *browser;
{
    if (REMOTE_BROWSER_magic != browser->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"remote_browser_dec_current",
	      "Bad magic type",0);
    
    if (! browser->a.remote_browser->current_server)
  	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_browser_dec_current",
	      "No current server",0);

    remote_server_dec(& (browser->a.remote_browser->current_server));
}

static struct remote_server * remote_server_inc P_((const struct remote_server *server));
static struct remote_server * remote_server_inc(server)
     const struct remote_server *server;
{
    struct remote_server *ptr;

    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_inc",	
	      "Bad (server) magic",0);
    
    for (ptr =  FIRST_SERVER; ptr; ptr = ptr->next_server)
	if (ptr == server)
	    break;

    if (ptr != server)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_inc",
	      "Server not in list",0);
    
    ptr->browser_count++;

    DPRINT(Debug,16,(&Debug,
		     "remote_browser_inc: remote server %p browser count incremented to %d\n",
		     ptr,ptr->browser_count));

    return ptr;
}


void remote_server_inc_dir(browser,server)
     struct folder_browser *browser;
     const struct remote_server *server;
{
    if (browser->default_server)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_server_inc_dir",
	      "Double assigment",0);

    browser->default_server = remote_server_inc(server);
}

static void remote_browser_inc P_((struct folder_browser *browser,
					   struct remote_server *server));
static void remote_browser_inc(browser,server)
     struct folder_browser *browser;
     struct remote_server  *server;
{
    if (REMOTE_BROWSER_magic != browser->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"remote_browser_inc",
	      "Bad magic type",0);

    if (browser->a.remote_browser->current_server)
  	panic("BROWSER PANIC",__FILE__,__LINE__,"remote_browser_inc",
	      "Double assigment",0);

    browser->a.remote_browser->current_server = remote_server_inc(server);      
}

struct remote_server * give_remote_server_from_folder(folder,t,passhm)
     struct folder_info *folder;
     struct remote_server_type *t;
     struct browser_passhm * passhm;
{
    struct remote_server *ptr;

      for (ptr =  FIRST_SERVER; ptr; ptr = ptr->next_server)
	  if (ptr->current_folder == folder &&
	      ptr->server_type    == t &&
	      ptr->passhm         == passhm) {

	      DPRINT(Debug,10,(&Debug,
			       "give_remote_server_from_folder=%p; folder=%p, server type=%p\n",
			       ptr,folder,t));

	      return ptr;
	  }

      /* ptr->current_folder protects freeing
	 ptr->browser_count is not incremented
      */

      ptr = malloc_remote_server(t);

      ptr->current_folder = folder;
      ptr->passhm         = passhm;
      if (ptr->passhm)
	  inc_browser_passhm_refcount(ptr->passhm);

      DPRINT(Debug,10,(&Debug,
		       "give_remote_server_from_folder=%p; folder=%p (new server), server type=%p\n",
		       ptr,folder,t));

	  
      return ptr;
}

void remove_remote_server_from_folder(folder,server)
     struct folder_info *folder;
     struct remote_server *server;
{
    
    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remove_remote_server_from_folder",	
	      "Bad (server) magic",0);

    if (server->current_folder != folder)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remove_remote_server_from_folder",	
	      "Folder not current",0);
    
    /* server->current_folder protects freeing
       server->browser_count is not incremented/decremented
    */

    server->current_folder = NULL;

    server_check_free(server);
}


struct remote_server * give_remote_server_from_connection(con,t,passhm)
     struct connection_cache   *con;
     struct remote_server_type *t;
     struct browser_passhm * passhm;
{
    struct remote_server *ptr;

    for (ptr =  FIRST_SERVER; ptr; ptr = ptr->next_server)
	if (ptr->current_connection == con &&
	    ptr->server_type    == t)
	    
	    return ptr;


    /* ptr->current_connection protects freeing
       ptr->browser_count is not incremented
    */


    ptr = malloc_remote_server(t);

    ptr->current_connection = con;
    ptr->passhm         = passhm;
    if (ptr->passhm)
	inc_browser_passhm_refcount(ptr->passhm);

    return ptr;
}


void remove_remote_server_from_connection(con,server)
     struct connection_cache *con;
     struct remote_server    *server;
{
    
    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remove_remote_server_from_connection",	
	      "Bad (server) magic",0);

    if (server->current_connection != con)
	panic("BROWSER PANIC",__FILE__,__LINE__,"remove_remote_server_from_connection",	
	      "Folder not current",0);


    /* server->current_connection protects freeing
       server->browser_count is not incremented/decremented
    */

    server->current_connection = NULL;

    server_check_free(server);
}


/* ======================================================================= */

/* Remote server browser */

S_(browser_zero_dir browser_zero_remote)
static  void browser_zero_remote P_((struct folder_browser *dir));
static  void browser_zero_remote(dir) 
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug,"browser_zero_remote: dir=%p\n", dir));
    
    dir->a.remote_browser = malloc_REMOTE_BROWSER();

}

S_(browser_free_dir browser_free_remote)
static void browser_free_remote P_((struct folder_browser *dir));
static void browser_free_remote(dir)
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug,"browser_free_remote: dir=%p\n", dir));

    free_REMOTE_BROWSER(& dir->a.remote_browser);
}


/* Return 1 if changed */
static int browser_change_to_real(dir)
     struct folder_browser *dir;
{
    int ret = 0;
    struct remote_server * server  = NULL;
    struct browser_type  * new_type = NULL;

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_to_real",
	      "Bad magic type",0);

    if (! dir->a.remote_browser->current_server) {

	DPRINT(Debug,12,(&Debug,"browser_change_to_real: No current remote server\n"));	
	goto fail;
    }

    server = dir->a.remote_browser->current_server;
    
    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_change_to_real",	
	      "Bad server magic",0);

    if (RSERVER_TYPE_magic != server->server_type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_change_to_real",	
	      "Bad server type magic",0);

    new_type = server->server_type->rs_dir_type_it(server);
    if (! new_type) {
	DPRINT(Debug,12,(&Debug,"browser_change_to_real: failed to get new type\n"));
	goto fail;
    }

    /* change_folder_browser_type() does clear_dir_vector() */
    if (!change_folder_browser_type(dir,new_type)) {
	DPRINT(Debug,12,(&Debug,"browser_change_to_real: .. remote browser does not change type ?? ..\n"));
	goto fail;
    }

    ret = server->server_type->rs_connect_browser_it(server,dir);
    if (!ret) {
	DPRINT(Debug,12,(&Debug,"browser_change_to_real: failed to connect remote mailbox"));

	/* CANCEL type change  */
	
	change_folder_browser_type(dir,&remote_browser);

	goto fail;
    }

 fail:

    DPRINT(Debug,12,(&Debug,"browser_change_to_real=%d\n",ret));
    return ret;
}

S_(browser_change_dir browser_change_remote)
static int browser_change_remote P_((struct folder_browser *dir,
				    const struct string *rel_dirname,
				    struct string **dispname));
static int browser_change_remote(dir,rel_dirname,dispname)
     struct folder_browser *dir;
     const struct string *rel_dirname;
     struct string **dispname;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_change_remote: dir=%p\n", dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_remote",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
	
    if (! server) {
	    DPRINT(Debug,11,(&Debug,
			     "browser_change_remote: No current remote server\n"));
	    goto fail;
    }
	
    if (REMOTE_SERVER_magic != server->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"browser_change_remote",	
		  "Bad server magic",0);
    
    ret = server->server_type->rs_change_it(server,dir,rel_dirname,
						dispname);
	
 fail:
    DPRINT(Debug,11,(&Debug,"browser_change_remote=%d\n", ret));

    return ret;
}

S_(browser_select_dir browser_select_remote)
static int browser_select_remote P_((struct folder_browser *dir,
				     const struct string *rel_itemname,
				     struct string **dispname,
				     int *newpos));
static int browser_select_remote(dir,rel_itemname,dispname,newpos)
     struct folder_browser *dir;
     const struct string *rel_itemname;
     struct string **dispname;
     int *newpos;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_select_remote: dir=%p\n", dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_select_remote",
	      "Bad magic type",0);
    
    if (rel_itemname) {
	struct remote_server * server  = NULL;

	DPRINT(Debug,11,(&Debug,"browser_select_remote: rel_itemname=%S\n", 
			 rel_itemname));
	
	server = dir->a.remote_browser->current_server;
	
	if (! server) {
	    DPRINT(Debug,11,(&Debug,
			     "browser_select_remote: No current remote server\n"));
	    goto fail;
	}
	
	if (REMOTE_SERVER_magic != server->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"browser_select_remote",	
		  "Bad server magic",0);

	ret = server->server_type->rs_select_it(server,dir,rel_itemname,
						dispname,newpos);
	
    	
    } else {

	/* Similar than browser_select_dummy() */

	DPRINT(Debug,11,(&Debug,"browser_select_remote: rel_itemname=NULL\n"));

	/* Null selection when remote mode ... */
	set_dir_selection(dir,NULL,NULL,
			  BROWSER_EXIST       /* Assume existance */
			  |BROWSER_MAILFILE   /* Assume folder    */
			  );

	/* browser_folder_from_remote() will be called */

	ret = 1;
    }


 fail:
    DPRINT(Debug,11,(&Debug,"browser_select_remote=%d\n", ret));
    
    return ret;
}

S_(browser_change_v_dir browser_change_v_remote)
static int browser_change_v_remote P_((struct folder_browser *dir,
				      struct name_vector *X,
				      struct string **dispname));
static int browser_change_v_remote(dir,X,dispname)
     struct folder_browser *dir;
     struct name_vector *X;
     struct string **dispname;
{
    DPRINT(Debug,11,(&Debug,"browser_change_v_remote=0: dir=%p\n", dir));

    /* Not needed? */

    return 0;     /* UNSUPPORTED */
}

S_(browser_change_up_dir browser_change_up_remote)
static int browser_change_up_remote P_((struct folder_browser *dir,
				       struct string **dispname,
				       struct string ** disp_tail));
static int browser_change_up_remote(dir,dispname,disp_tail)
     struct folder_browser *dir;
     struct string **dispname;
     struct string ** disp_tail;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_change_up_remote: dir=%p\n", dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_up_remote",
	      "Bad magic type",0);

    /* XXXX Currently that is not mostly needed because remote browser does not do
            directory listing
    */

#if REMOTE_DIR_LISTING

    if (dir->a.remote_browser->dir_name_cache && 
	dir->a.remote_browser->cur_sep != '\0' &&
	dir->a.remote_browser->current_server) {

	/* XXXX   Not used */

	const struct string * relative1 = dir->a.remote_browser->dir_name_cache;

	int L         = string_len(relative1);

	if (L > 0) {

	    /* ASCII assumed on separator */
	    int L1 = browser_change_up_helper(relative1,dir->a.remote_browser->cur_sep);
					      
	    int X = 0;
	    struct string * A1 = clip_from_string(relative1,&X,L1);
	    
	    if (disp_tail) 
		*disp_tail = browser_disp_tail_helper(relative1,X,dir->a.remote_browser->cur_sep);

	    /* Need not generate actual directory listing */
	    
	    if (*dispname)
		free_string(dispname);	    
	    *dispname = format_string(FRM("&%S"),A1);	    

	    ret = 1;
	} else
	    /* Just return default menu */
	    ret = -1;
       
    } else 
#endif
	/* Just return default menu */
	ret = -1;


    DPRINT(Debug,11,(&Debug,"browser_change_up_remote=%d\n", ret));
    return ret;   
}


S_(browser_give_title_dir browser_give_title_remote)
static struct string * browser_give_title_remote P_((struct folder_browser *dir));
static struct string * browser_give_title_remote(dir)
     struct folder_browser *dir;
{
    static struct string *ret = NULL;

    DPRINT(Debug,11,(&Debug,"browser_give_title_remote: dir=%p\n", dir));


    /* XXXX Currently should not be needed  because remote browser does not do
            directory listing
    */

    if (dir->dirname)
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeNoDirListing,
				    "Directory listing for %S not available"),
				    dir->dirname);
    else 
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeDefaultDir,
				    "Default directory"));

    DPRINT(Debug,11,(&Debug,"browser_give_title_remote=%S\n", ret));
    
    return ret;
}

S_(browser_separator_dir browser_separator_remote)
static char browser_separator_remote P_((struct folder_browser *dir));
static char browser_separator_remote(dir)
     struct folder_browser *dir;
{
    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_separator_remote",
	      "Bad magic type",0);
 
#if REMOTE_DIR_LISTING
    if (dir->a.remote_browser->current_server) {       
	return dir->a.remote_browser->cur_sep;

    } else  /* Local file system so separator is '/'  (unix) */
	return '/';

#else
    return '\0';
#endif

}

/* ---------------- */

S_(browser_name_dir browser_name_remote)
static struct string * browser_name_remote P_((struct folder_browser *dir));
static struct string * browser_name_remote(dir)
     struct folder_browser *dir;
{
    static struct string *ret = NULL;

    DPRINT(Debug,11,(&Debug,"browser_name_remote: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_name_remote",
	      "Bad magic type",0);

    /* XXXXX OOPS? */   
    if (!dir->dirname) 	{
	if (dir->sys_dir)
	    dir->dirname = format_string(FRM("&%s"),
					 dir->sys_dir);
	else
	    dir->dirname = format_string(FRM("&"));
    }

    /* XXXX   not really used */
    
    if (dir->filter)
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeRemoteDirFilter,
				    "Remote directory %S with filter %S"),
			    dir->dirname,dir->filter);
    else
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeRemoteDir,
				    "Remote directory %S"),
			    dir->dirname);
    
    DPRINT(Debug,11,(&Debug,"browser_name_remote=%S\n", ret));

    return ret;
}

S_(browser_cat_dir browser_cat_remote)
static struct string * browser_cat_remote P_((struct folder_browser *dir,
					      const struct string * item));
static struct string * browser_cat_remote(dir,item)
     struct folder_browser *dir;
     const struct string * item;
{
    struct string * ret = NULL;

    DPRINT(Debug,11,(&Debug,"browser_cat_remote: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_cat_remote",
	      "Bad magic type",0);


    /* XXXX not really used? */

    if (dir->dirname) {

	struct string * relative1 = NULL;
	    
#if REMOTE_DIR_LISTING

	int L;

	if (!dir->a.remote_browser->dir_name_cache) {
	    DPRINT(Debug,11,(&Debug, "browser_cat_remote: No dir_name_cache!\n"));
	    goto fail;
	}
	relative1 = dup_string(dir->a.remote_browser->dir_name_cache);

	L = string_len(relative1);

	if (L > 0) {
	    if (dir->a.remote_browser->cur_sep == '\0') {
		DPRINT(Debug,11,(&Debug, "browser_cat_remote: No separator!\n"));
		goto fail;
		
	    } else if ( dir->a.remote_browser->cur_sep !=   /* ASCII assumed! */
			give_unicode_from_string(dir->dirname,L-1))
		fill_ascii_to_string(relative1,1,dir->a.remote_browser->cur_sep);
	}
	
	ret = format_string(FRM("&%S"),relative1);

	/* Some remote directory! */
	if (item) {
	   append_string(&ret, item, 0);
	}

    fail:

#endif

	if (relative1)
	    free_string(&relative1);

    } else {   /* !!!! XXXXX Should not happen */
	ret = format_string(FRM("&"));

	if (item) {
	   append_string(&ret, item, 0);
	}	
    }

    
    if (ret) {
	DPRINT(Debug,11,(&Debug,"browser_cat_remote=%S\n",ret));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_cat_remote=NULL\n"));
    }
    return ret;
}


S_(browser_folder_from_dir browser_folder_from_remote)
static struct folder_info * 
browser_folder_from_remote P_((struct folder_browser *dir,
			      int treat_as_spooled));

static struct folder_info * browser_folder_from_remote(dir, treat_as_spooled)
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info   * res     = NULL;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_folder_from_remote: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_folder_from_remote",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;

    if (! server) {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_remote: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_folder_from_remote",	
	      "Bad server magic",0);

    res = server->server_type->rs_folder_from_it(server,dir,treat_as_spooled);

 fail:
    if (res) {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_remote=%p\n",res));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_remote=NULL\n"));
    }
    return res;
}


S_(browser_create_selection_dir browser_create_selection_remote)
static int browser_create_selection_remote P_((struct folder_browser *dir));

static int browser_create_selection_remote(dir)
     struct folder_browser *dir;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,
		     "browser_create_selection_remote: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_create_selection_remote",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
    
    if (! server) {
	DPRINT(Debug,11,(&Debug,"browser_create_selection_remote: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_create_selection_remote",	
	      "Bad server magic",0);

    ret = server->server_type->rs_create_selection_it(server,dir);

 fail:
    DPRINT(Debug,11,(&Debug,"browser_create_selection_remote=%d\n", ret));
    
    return ret;
}

S_(zero_ws_fields_browser zero_ws_fields_remote)
static void zero_ws_fields_remote P_((WRITE_STATE ptr));
static void zero_ws_fields_remote(ptr)
     WRITE_STATE ptr;
{
    ptr->a.remote = NULL;
}

S_(free_ws_fields_browser free_ws_fields_remote)
static void free_ws_fields_remote P_((WRITE_STATE ptr));
static void free_ws_fields_remote(ptr)
     WRITE_STATE ptr;
{
    if (ptr->a.remote) {
	free_remote_server_ws(& (ptr->a.remote));
    }
}

S_(browser_prepare_write_dir browser_prepare_write_remote)
static int browser_prepare_write_remote P_((struct folder_browser *dir,
					   WRITE_STATE ptr));
static int browser_prepare_write_remote(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_prepare_write_remote: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_prepare_write_remote",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
    
    if (! server) {
	DPRINT(Debug,11,(&Debug,"browser_prepare_write_remote: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_prepare_write_remote",	
	      "Bad server magic",0);

    ret = server->server_type->rs_prepare_write_it(server,dir,ptr);

 fail:
    DPRINT(Debug,11,(&Debug,"browser_prepare_write_remote=%d\n", ret));

    return ret;
}

S_(browser_end_write_dir browser_end_write_remote)
static int browser_end_write_remote P_((struct folder_browser *dir,
			      WRITE_STATE ptr));
static int browser_end_write_remote(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_end_write_remote: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_end_write_remote",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
    
    if (! server) {
	DPRINT(Debug,11,(&Debug,"browser_end_write_remote: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_end_write_remote",	
	      "Bad server magic",0);

    ret = server->server_type->rs_end_write_it(server,dir,ptr);

 fail:
 
    DPRINT(Debug,11,(&Debug,"browser_end_write_remote=%d\n", ret));
    
    return ret;
}

S_(browser_sync_write_dir browser_sync_write_remote)
static int browser_sync_write_remote P_((struct folder_browser *dir,
					WRITE_STATE ptr));
static int browser_sync_write_remote(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_sync_write_remote: dir=%p\n", 
		     dir));

    /* sync_write_folder() is only used on src/leavembox.c
     *
     * That does not refer & (remote server), so we
     * do not support that now.
     */
 
    DPRINT(Debug,11,(&Debug,"browser_sync_write_remote=%d (unsupported)\n", 
		     ret));
    
    return ret;
}


/* Returns -1L on failure */
S_(browser_tell_dir_ws browser_tell_remote_ws)
static long browser_tell_remote_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr));
static long browser_tell_remote_ws(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    long ret = -1L;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_tell_remote_ws: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_tell_remote_ws",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
    
    if (! server) {
	DPRINT(Debug,11,(&Debug,"browser_tell_remote_ws: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_tell_remote_ws",	
	      "Bad server magic",0);

    ret = server->server_type->rs_tell_it_ws(server,dir,write_state_ptr);
    
 fail:
    DPRINT(Debug,11,(&Debug,"browser_tell_remote_ws=%ld\n",ret));
    return ret;
}

S_(browser_seek_dir_ws browser_seek_remote_ws)
/* Returns 0 on failure */
static int browser_seek_remote_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      long pos));
static int browser_seek_remote_ws(dir,write_state_ptr,pos)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     long pos;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_seek_remote_ws: dir=%p\n", 
		     dir));


    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_seek_remote_ws",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
    
    if (! server) {
	DPRINT(Debug,11,(&Debug,
			 "browser_seek_remote_ws: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_seek_remote_ws",	
	      "Bad server magic",0);

    ret = server->server_type->rs_seek_it_ws(server,dir,write_state_ptr,pos);

 fail:
    DPRINT(Debug,11,(&Debug,"browser_seek_remote_ws=%d\n",ret));
    return ret;
}

/* Return 0 on failure */
S_(browser_write_dir_ws browser_write_remote_ws)
static int browser_write_remote_ws P_((struct folder_browser *dir,
				      WRITE_STATE ptr,
				      int l, const char *buffer));
static int browser_write_remote_ws(dir,ptr,l,buffer)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     int l; 
     const char *buffer;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_write_remote_ws: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_start_we_remote",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
    
    if (! server) {
	DPRINT(Debug,11,(&Debug,
			 "browser_write_remote_ws: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_write_remote_ws",	
	      "Bad server magic",0);

    ret = server->server_type->rs_write_it_ws(server,dir,ptr,l,buffer);

    
 fail:
    DPRINT(Debug,11,(&Debug,"browser_write_remote_ws=%d\n",ret));
    return ret;
}


S_(browser_start_we_dir browser_start_we_remote)
static int browser_start_we_remote P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      int write_envelope,
				      struct header_rec *current_header,
				      int *env_flags));
static int browser_start_we_remote(dir,write_state_ptr,
				   write_envelope,current_header,
				   env_flags)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
     int *env_flags;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_start_we_remote: dir=%p\n", 
		     dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_start_we_remote",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
    
    if (! server) {
	DPRINT(Debug,11,(&Debug,"browser_start_we_remote: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_start_we_remote",	
	      "Bad server magic",0);

    ret = server->server_type->rs_start_we_it(server,
					      dir,write_state_ptr,
					      write_envelope,current_header,
					      env_flags);

 fail:
    	    
    DPRINT(Debug,11,(&Debug,"browser_start_we_remote=%d\n",ret));
    return ret;
}

S_(browser_end_we_dir browser_end_we_remote)
static int browser_end_we_remote P_((struct folder_browser *dir,
				  WRITE_STATE write_state_ptr,
				  int write_envelope,
				  struct header_rec *current_header));
static int browser_end_we_remote(dir,write_state_ptr,
			      write_envelope,current_header)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_end_we_remote: dir=%p\n", 
		dir));

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_end_we_remote",
	      "Bad magic type",0);

    server = dir->a.remote_browser->current_server;
    
    if (! server) {
	DPRINT(Debug,11,(&Debug,"browser_end_we_remote: No current remote server\n"));
	goto fail;
    }


    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_end_we_remote",	
	      "Bad server magic",0);

    ret = server->server_type->rs_end_we_it(server,
					      dir,write_state_ptr,
					      write_envelope,current_header);

 fail:
    DPRINT(Debug,11,(&Debug,"browser_end_we_remote=%d\n",ret));
    return ret;
}

S_(browser_selection_is_folder browser_selection_is_remote)
static int browser_selection_is_remote P_((struct folder_browser *dir,
					  struct folder_info *folder));

static int browser_selection_is_remote(dir,folder)
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int ret = 0;
    struct remote_server * server  = NULL;

    DPRINT(Debug,11,(&Debug,"browser_selection_is_remote: dir=%p\n", 
		     dir));

    if (dir->sel_type != selection_folder)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_selection_is_remote",
	      "Bad selection type",0);
   
    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_selection_is_remote",
	      "Bad magic type",0);

    
    server = dir->a.remote_browser->current_server;

    if (! server) {
	DPRINT(Debug,11,(&Debug,"browser_selection_is_remote: No current remote server\n"));
	goto fail;
    }

    if (REMOTE_SERVER_magic != server->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_selection_is_remote",	
	      "Bad server magic",0);

    ret = server->server_type->rs_selection_is_it(server,dir,folder);
 
 fail:    
    DPRINT(Debug,11,(&Debug,
		     "browser_selection_is_remote=%d\n",ret));
    return ret;
}

S_(browser_make_ref_folder browser_make_ref_remote)
static int browser_make_ref_remote P_((struct folder_browser *dir,
				      char **refname, int *iscopy,
				      int is_text));
static int browser_make_ref_remote(dir,refname,iscopy,is_text)
     struct folder_browser *dir;
     char **refname; 
     int *iscopy;
     int is_text;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_make_ref_remote: dir=%p\n", 
		     dir));

    if (dir->sel_type != selection_file)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_make_ref_remote",
	      "Bad selection type",0);

    /* Unsupported */

    DPRINT(Debug,11,(&Debug,"browser_make_ref_remote=%d\n",ret));
    
    return ret;
}

S_(browser_update_dir browser_update_remote)
static void browser_update_remote P_((struct folder_browser *dir));
static void browser_update_remote(dir)
     struct folder_browser *dir;
{
    clear_dir_vector(dir);

    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_update_remote",
	      "Bad magic type",0);

    panic("BROWSER PANIC",__FILE__,__LINE__,"browser_update_remote",
	  "browser_update_remote() called",0);	
}


/* Call only if BROWSER_NEEDSTAT is set */
S_(browser_do_stat browser_remote_do_stat)
static void browser_remote_do_stat P_((struct folder_browser *dir,
				      int idx));
static void browser_remote_do_stat(dir,idx)
     struct folder_browser *dir;
     int idx;
{
    char * entryname UNUSED_VAROK;
    int flags        UNUSED_VAROK;

    DPRINT(Debug,13,(&Debug,
		     "browser_remote_do_stat: dir=%p, idx=%d\n", 
		     dir,idx));

    if (idx < 0 || idx >= dir->vector_len)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_remote_do_stat",
	      "Bad index",0);

    flags     = dir->vector[idx].flags;
    entryname = dir->vector[idx].sys_name;

    flags &= ~BROWSER_NEEDSTAT;

    
    /* Not used */
    

    DPRINT(Debug,13,(&Debug,
		     "browser_remote_do_stat: changing flags %X -> %X\n",
		     dir->vector[idx].flags,flags));

    dir->vector[idx].flags = flags;


    panic("BROWSER PANIC",__FILE__,__LINE__,"browser_remote_do_stat",
	  "browser_remote_do_stat() called",0);	
    
}

S_(browser_folder_sort_dir browser_folder_sort_remote)
static void browser_folder_sort_remote P_((struct folder_browser *dir,
					  print_sort_message * print));
static void browser_folder_sort_remote(dir,print)
     struct folder_browser *dir;
     print_sort_message * print;
{
    if (dir->vector_len < 2)
	return;

    DPRINT(Debug,13,(&Debug,
		     "browser_folder_sort_remote() unsupported\n"));
}


S_(browser_free_name_vector_dir browser_free_name_vector_remote)
static void browser_free_name_vector_remote P_((struct folder_browser *dir,
					       struct name_vector *entry));
static void browser_free_name_vector_remote(dir,entry)
     struct folder_browser *dir;
     struct name_vector *entry;
{

    entry->a.dummy = NULL;

}

S_(browser_line_idx_dir browser_line_idx_remote)
static const struct string * browser_line_idx_remote 
                      P_((struct folder_browser *dir,
			  struct name_vector *entry,
			  const struct string  **comment_p));
static const struct string * browser_line_idx_remote(dir,entry,comment_p)
     struct folder_browser *dir;
     struct name_vector *entry;
     const struct string  **comment_p;
{

    DPRINT(Debug,11,(&Debug,"browser_line_idx_remote=NULL\n"));

    return NULL;
}

S_(browser_reload_dir browser_reload_remote)
static int browser_reload_remote P_((struct folder_browser *dir));
static int browser_reload_remote(dir)
     struct folder_browser *dir;
{
    int ret = 0;

    /* Not used? 
     *
     * remote browser do not able to do folder listing 
     */

    DPRINT(Debug,11,(&Debug, "browser_reload_remote=%d (unsupported)\n", ret));

    return ret;    
}

struct browser_type remote_browser = {
    BROWSER_TYPE_magic,
    browser_zero_remote,
    browser_free_remote,
    browser_change_remote,
    browser_give_title_remote,
    browser_separator_remote,
    browser_name_remote,
    browser_cat_remote,
    browser_select_remote,    
    browser_folder_from_remote,
    browser_change_v_remote,
    browser_change_up_remote,
    browser_create_selection_remote,
    zero_ws_fields_remote,
    free_ws_fields_remote,
    browser_prepare_write_remote,
    browser_end_write_remote,
    browser_tell_remote_ws,
    browser_seek_remote_ws,
    browser_write_remote_ws,
    browser_start_we_remote,
    browser_end_we_remote,
    browser_selection_is_remote,
    browser_make_ref_remote,
    browser_update_remote,      
    browser_remote_do_stat,     
    browser_sync_write_remote,
    browser_folder_sort_remote,
    browser_free_name_vector_remote,
    browser_reset_filter_generic,
/* Not called, select_dir_item_by_idx() is used
   only on fbrowser (Elm 2.5 stype file browser) */
    browser_select_idx_generic,
    browser_line_idx_remote,
    browser_reload_remote,
    browser_gethm_name_vector_null,
    browser_sethm_name_vector_null
};

/* ------------------------------------------------------------------- */

/* Remote server is not available                                 */
void remote_browser_failure(dir)
     struct folder_browser *dir;
{
    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"remote_browser_failure",
	      "Bad magic type",0);
        
    if (dir->a.remote_browser->current_server)
	remote_browser_dec_current(dir);

    lib_error(CATGETS(elm_msg_cat, MeSet, MeRemBrowserNotSet,
		      "Current remote server not set (unable to expand &)"));
}

/* Changes current server to default server ---------------------- */
int change_remote_browser_server(dir)
     struct folder_browser *dir;
{
    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"change_remote_browser_server",
	      "Bad magic type",0);


    if (dir->a.remote_browser->current_server != dir->default_server) {
	
	if (dir->a.remote_browser->current_server) {
	    DPRINT(Debug,16,(&Debug,
			     "change_remote_browser_server: changing current server from %p to %p\n",
			     dir->a.remote_browser->current_server,dir->default_server));

	    remote_browser_dec_current(dir);
	} else {
	    DPRINT(Debug,16,(&Debug,
			     "change_remote_browser_server: setting current server to %p\n",
			     dir->default_server));
	}
		
	remote_browser_inc(dir, dir->default_server);
   
	
	/* Clear directory listing ..... */
	clear_dir_vector(dir);

	DPRINT(Debug,16,(&Debug,
			 "change_remote_browser_server=1\n"));
	return 1;
    }


    DPRINT(Debug,16,(&Debug,
		     "change_remote_browser_server=0\n"));

    return 0;
}

struct remote_server * remote_browser_server(dir)
     struct folder_browser *dir;
{
    if (REMOTE_BROWSER_magic != dir->a.remote_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"remote_browser_server",
	      "Bad magic type",0);

    return dir->a.remote_browser->current_server;    
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
