static char rcsid[] = "@(#)$Id: pager_lineext.c,v 2.6 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.6 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "pg_lineext_imp.h"
#include "sb_imp.h"

DEBUG_VAR(DebugIO,__FILE__,"stateio");

struct pager_lineext  * malloc_pager_lineext(lineext_type, stringbuffer)
     struct lineext_type * lineext_type;
     struct stringbuffer * stringbuffer;
{
    struct pager_lineext * ret = safe_zero_alloc(sizeof (*ret));

    if (LINEEXT_TYPE_magic != lineext_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"malloc_pager_lineext",
	      "Bad magic number (lineext_type)",0);
    
    ret->lineext_type = lineext_type;
    ret->refcount     = 1;
    ret->a.dummy      = NULL;
    ret->magic        = PAGER_LINEEXT_magic;

    ret->stringbuffer   = stringbuffer;

    if (ret->stringbuffer)
	inc_stringbuffer_refcount(ret->stringbuffer);

    ret->lineext_type->plext_init_it(ret,ret->stringbuffer);

    return ret;
}

/* Default routine */
extern struct string * plext_walk_stringbuffer(ext,walk_extension,pg_flags,
					       range)
     struct pager_lineext  * ext;
     struct pager_lineext ** walk_extension;
     int                   * pg_flags;
     struct pager_range   ** range;
{
    struct string * ret = NULL;

    if (PAGER_LINEEXT_magic != ext->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "plext_walk_stringbuffer",
	      "Bad magic number (pager_lineext)",0);

    ret = walk_lineext_on_stringbuffer(ext->stringbuffer,
				       walk_extension,pg_flags,
				       range);

    return ret;
}

static void report_range_change P_((enum range_changed *range_changed,
				    struct pager_range **range,
				    struct pager_range * new_range));
static void report_range_change(range_changed,range,new_range)
     enum range_changed *range_changed;
     struct pager_range **range;
     struct pager_range * new_range;
{
    if (range_changed) {
	DPRINT(DebugIO,10,(&DebugIO, 
			   "report_range_change: range changed"));
	
	if (*range) {
	    int flags UNUSED_VAROK = get_pager_range_flags(*range);
	    int ql    UNUSED_VAROK = get_pager_range_quote_level(*range);
	    DPRINT(DebugIO,10,(&DebugIO, 
			       ", old range %p flags=%d quote level=%d",
			       *range,flags,ql));
	}
	if (new_range) {
	    int flags UNUSED_VAROK = get_pager_range_flags(new_range);
	    int ql    UNUSED_VAROK = get_pager_range_quote_level(new_range);
	    DPRINT(DebugIO,10,(&DebugIO, 
			       ", new range %p flags=%d quote level=%d",
			       new_range,flags,ql));
	}
	
	*range_changed = get_pager_range_change(*range,new_range);
	switch (*range_changed) {
	case rc_same_range: DPRINT(DebugIO,10,(&DebugIO,", same range?"));
	    break;
	case rc_newline: DPRINT(DebugIO,10,(&DebugIO,", newline"));
	    break;
	case rc_same_line: DPRINT(DebugIO,10,(&DebugIO,", same line"));
	    break;
	default: DPRINT(DebugIO,10,(&DebugIO,", bad value (%d)",
				    *range_changed));
	    break;
	}
	
	DPRINT(DebugIO,10,(&DebugIO, "\n"));
	
    }
}

/* Changes *walk_extension to point next line extension or to NULL */
struct string *get_lineext_and_walk(walk_extension,pg_flags,
				    range_changed,range)
     struct pager_lineext **walk_extension;
     int *pg_flags;
     enum range_changed *range_changed;
     struct pager_range **range;
{
    struct string * ret = NULL;
    struct pager_lineext *Walk_extension = NULL;
    int PG_FLAGS = 0;
    struct pager_range * new_range = NULL;

    if (PAGER_LINEEXT_magic != (*walk_extension)->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"get_lineext_and_walk",
	      "Bad magic number (pager_lineext)",0);
    
    if (range_changed)
	*range_changed = rc_same_range;
    
    if (LINEEXT_TYPE_magic != (*walk_extension)->lineext_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"get_lineext_and_walk",
	      "Bad magic number (lineext_type)",0);

    Walk_extension = *walk_extension;
    inc_pager_lineext_refcount(Walk_extension);

    ret = (*walk_extension)->lineext_type->plext_walk_it(*walk_extension,
							 & Walk_extension,
							 &PG_FLAGS,
							 range ? &new_range : 
							 NULL);

    
    if (range && *range != new_range) {

	report_range_change(range_changed,range,new_range);

	if (*range)
	    free_pager_range(range);
	
	*range = new_range;
	new_range = NULL;
    }

    if (new_range)
	free_pager_range(&new_range);
    
    if (pg_flags)
	*pg_flags = PG_FLAGS;

    free_pager_lineext(walk_extension);
    *walk_extension = Walk_extension;

    return ret;
}

struct string_alt * get_lineext_alt_and_walk(walk_extension,
					     pg_flags,
					     range_changed,
					     range)
     struct pager_lineext **walk_extension;
     int *pg_flags;
     enum range_changed * range_changed;
     struct pager_range **range;
{
    struct string_alt * ret = NULL;
    struct pager_lineext *Walk_extension = NULL;
    int PG_FLAGS = 0;
    struct pager_range * new_range = NULL;

    if (PAGER_LINEEXT_magic != (*walk_extension)->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"get_lineext_alt_and_walk",
	      "Bad magic number (pager_lineext)",0);
    
    if (range_changed)
	*range_changed = rc_same_range;

    if (LINEEXT_TYPE_magic != (*walk_extension)->lineext_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"get_lineext_alt_and_walk",
	      "Bad magic number (lineext_type)",0);

    Walk_extension = *walk_extension;
    inc_pager_lineext_refcount(Walk_extension);

    ret =  (*walk_extension)->lineext_type->plext_walk_alt_it(*walk_extension,
							      & Walk_extension,
							      &PG_FLAGS,
							      range ? &new_range : 
							      NULL);
    
    if (range && *range != new_range) {

	report_range_change(range_changed,range,new_range);

	if (*range)
	    free_pager_range(range);
	
	*range = new_range;
	new_range = NULL;
    }

    if (new_range)
	free_pager_range(&new_range);
    
    if (pg_flags)
	*pg_flags = PG_FLAGS;

    free_pager_lineext(walk_extension);
    *walk_extension = Walk_extension;

    return ret;
}


void free_pager_lineext(extension)
     struct pager_lineext **extension;
{
    if (PAGER_LINEEXT_magic != (*extension)->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"free_pager_lineext",
	      "Bad magic number (pager_lineext)",0);

    (*extension)->refcount--;

    if ((*extension)->refcount > 0) {  /* Do not free */
	(*extension) = NULL;           /* Refefence count for this pointer is     */
	return;			       /* decrement, so this must have be reset   */	 
    }

    if (LINEEXT_TYPE_magic != (*extension)->lineext_type->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"free_pager_lineext",
	      "Bad magic number (lineext_type)",0);

    (*extension)->lineext_type->plext_dest_it(*extension,
					      (*extension)->stringbuffer);

    if ((*extension)->stringbuffer)
	free_stringbuffer(& ((*extension)->stringbuffer));

    (*extension)->magic        = 0;  /* Invalidate */
    free(*extension);
    *extension = NULL;
}

void inc_pager_lineext_refcount(extension)
     struct pager_lineext *extension;
{
    if (PAGER_LINEEXT_magic != extension->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"inc_pager_lineext_refcount",
	      "Bad magic number (pager_lineext)",0);

    extension->refcount++;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
