static char rcsid[] = "@(#)$Id: pgrng_bfr_binding.c,v 2.5 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.5 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "pg_range_imp.h"

/* Needs STRING_magic */
#include "elmme-magic.h"
#include "pg_params_imp.h"

DEBUG_VAR(DebugIO,__FILE__,"stateio");

#define PGRNG_BFR_magic	     0xEB0F

struct pgrng_bfr_binding {
    unsigned short              magic; /* PGRNG_BFR_magic */

    int refcount;

    /* target's refcount is NOT incremented: */

    struct pager_range  *range;
    struct stringbuffer *buffer;
    int                  buffer_index;

    /* range and buffer also count as refcount */

    struct {
	struct pager_param_value  * value; /* does not need refcount */

	int                         buffer_index;
    }                             * values;      
    int                          value_count;

};

/* 1) binding -> pager_range -------------------------------------------- */

static void clear_pgrng_bfr_binding P_((struct pgrng_bfr_binding **binding));

void pgrng_bfr_bnd_clear_range(binding,range)
     struct pgrng_bfr_binding **binding;
     const struct pager_range *range;
{
    if (PGRNG_BFR_magic != (*binding)->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_clear_range",
	      "Bad magic number",0);

    if (range != (*binding)->range)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_clear_range",
	      "Link (range) failure",0);
    (*binding)->range = NULL;
    
    clear_pgrng_bfr_binding(binding);
}

void pgrng_bfr_bnd_set_range(binding,range)
     struct pgrng_bfr_binding *binding;
     struct pager_range *range;
{
    if (PGRNG_BFR_magic != binding->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_set_range",
	      "Bad magic number",0);

    if (binding->range)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_set_range",
	      "Link (range) failure",0);

    binding->range = range;
}

/* Caller must call free_pager_range() */
struct pager_range * pgrng_bfr_bnd_get_range(binding)
     const struct pgrng_bfr_binding *binding;
{
    if (PGRNG_BFR_magic != binding->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_get_range",
	      "Bad magic number",0);

    if (binding->range)
	inc_pager_range_refcount(binding->range);

    return binding->range;
}

/* 2) binding -> stringbuffer ------------------------------------------- */

void pgrng_bfr_bnd_clear_stringbuffer(binding,buffer,index)
     struct pgrng_bfr_binding **binding;
     const struct stringbuffer *buffer;
     int index;
{
    if (PGRNG_BFR_magic != (*binding)->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_clear_stringbuffer",
	      "Bad magic number",0);

    if (buffer != (*binding)->buffer ||
	index  != (*binding)->buffer_index)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_clear_stringbuffer",
	      "Link (stringbuffer) failure",0);	
    (*binding)->buffer = NULL;
    (*binding)->buffer_index = -1;

    clear_pgrng_bfr_binding(binding);
}

void pgrng_bfr_bnd_set_stringbuffer(binding,buffer,index)
     struct pgrng_bfr_binding *binding;
     struct stringbuffer *buffer;
     int index;
{
    if (PGRNG_BFR_magic != binding->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_clear_stringbuffer",
	      "Bad magic number",0);

    if (binding->buffer ||
	-1 != binding->buffer_index)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_set_stringbuffer",
	      "Link (stringbuffer) failure",0);	

    binding->buffer       = buffer;
    binding->buffer_index = index;
}

/* Caller must call free_stringbuffer() */
struct stringbuffer * pgrng_bfr_bnd_get_stringbuffer(binding,index)
     struct pgrng_bfr_binding *binding; 
     int *index;
{
    if (PGRNG_BFR_magic != binding->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrng_bfr_bnd_get_stringbuffer",
	      "Bad magic number",0);

    if (index)
	*index = binding->buffer_index;

    if (binding->buffer)
	inc_stringbuffer_refcount(binding->buffer);

    return binding->buffer;
}


/* 3) general ---------------------------------------------------------- */

struct pgrng_bfr_binding * new_pgrng_bfr_binding() 
{
    struct pgrng_bfr_binding *ret = safe_zero_alloc(sizeof (*ret));

    ret->magic = PGRNG_BFR_magic;
    ret->refcount = 1;
    
    ret->range = NULL;
    ret->buffer = NULL;
    ret->buffer_index = -1;

    ret->values      = NULL;
    ret->value_count = 0;

    return ret;
}


void free_pgrng_bfr_binding(binding)
     struct pgrng_bfr_binding **binding;
{

    if (PGRNG_BFR_magic != (*binding)->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "free_pgrng_bfr_binding",
	      "Bad magic number",0);

    (*binding)->refcount--;

    clear_pgrng_bfr_binding(binding);
}

void inc_pgrng_bfr_binding_refcount(binding)
     struct pgrng_bfr_binding *binding;
{
    if (PGRNG_BFR_magic != binding->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "inc_pgrng_bfr_binding_refcount",
	      "Bad magic number",0);

    binding->refcount++;
}

static void clear_pgrng_bfr_binding(binding)
     struct pgrng_bfr_binding **binding;
{
    if (PGRNG_BFR_magic != (*binding)->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "clear_pgrng_bfr_binding",
	      "Bad magic number",0);

    if ((*binding)->refcount < 0)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "clear_pgrng_bfr_binding",
	      "Bad recount",0);

    if (((*binding)->buffer   && -1 == (*binding)->buffer_index) ||
	(! (*binding)->buffer && -1 != (*binding)->buffer_index))
		panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "clear_pgrng_bfr_binding",
	      "Bad buffer_index",0);
	
    if ((*binding)->refcount ||
	(*binding)->range ||
	(*binding)->buffer) 
	return;

    if ((*binding)->values) {
	int i;

	for (i = 0; i < (*binding)->value_count; i++) {
	    if ((*binding)->values[i].value) {

		reset_pager_param_value_bcklink(&((*binding)->values[i].
						  value),
						*binding);
	       
	    }
	    (*binding)->values[i].buffer_index = -1;
	}

	free((*binding)->values);
	(*binding)->values = NULL;
    }
    (*binding)->value_count = 0;
    
    (*binding)->magic = 0;  /* Invalidate */
    free(*binding);
    *binding = NULL;
}

/* -------------------------------------------------------------------- */

void pgrng_binding_add_value(backlink,value,buffer_index)
     struct pgrng_bfr_binding * backlink;
     struct pager_param_value * value;
     int  buffer_index;
{

    if (PGRNG_BFR_magic != backlink->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrn_binding_add_value",
	      "Bad magic number (pgrng_bfr_binding)",0);

    if (PAGER_PARAM_VALUE_magic != value->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrn_binding_add_value",
	      "Bad magic number (pager_param_value)",0);
    
    if (value->backlink) {

	if (value->backlink != backlink)
	    panic("PAGER RANGE PANIC",__FILE__,__LINE__,
		  "pgrn_binding_add_value",
		  "Bad backlink",0);

    } else
	value->backlink = backlink;
    
    if (backlink->values && buffer_index >= 0) {
	int i;
	
	for (i = 0; i < backlink->value_count; i++) {

	    if (buffer_index == backlink->values[i].buffer_index) {

		if (value ==  backlink->values[i].value)
		    return;   /* Skip duplicate insert */

		panic("PAGER RANGE PANIC",__FILE__,__LINE__,
		      "pgrn_binding_add_value",
		      "Duplicate buffer index",0);		
	    }	    
	}
    }
    
    
    backlink->values = safe_array_realloc(backlink->values,
					  backlink->value_count+1,
					  sizeof (backlink->values[0]));

    /* Does not need refcount */
    
    backlink->values[backlink->value_count].value = value;
    backlink->values[backlink->value_count].buffer_index = buffer_index;
    backlink->value_count++;
}

/* Increments refcount */
struct pager_param_value * pgrng_binding_get_value(backlink,buffer_index)
     struct pgrng_bfr_binding * backlink;
     int  buffer_index;
{
    struct pager_param_value * ret = NULL;

    if (PGRNG_BFR_magic != backlink->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrn_binding_get_value",
	      "Bad magic number (pgrng_bfr_binding)",0);

    if (backlink->values) {
	int i;
	
	for (i = 0; i < backlink->value_count; i++) {
	    
	    if (buffer_index == backlink->values[i].buffer_index) {

		ret = backlink->values[i].value;

		break;
	    }
	}
    }

    if (ret)
	inc_pager_param_value_refcount(ret);
		
    return ret;
}

/* Increments refcount */
struct pager_param_value *  pgrng_binding_search_helper(backlink,
						       next_pos,
						       found_pos)
     struct pgrng_bfr_binding * backlink;
     int                      * next_pos /* iterator */;
     int                      * found_pos /* -1 if not found */;
{
    struct pager_param_value * ret = NULL;
    int                        min_pos = 0;
    int                        found = -1;
    
    if (PGRNG_BFR_magic != backlink->magic)
	panic("PAGER RANGE PANIC",__FILE__,__LINE__,
	      "pgrn_binding_get_value",
	      "Bad magic number (pgrng_bfr_binding)",0);

    if (found_pos)
	*found_pos = -1;

    if (next_pos) {
	min_pos = *next_pos;

	if (min_pos < 0)
	    panic("PAGER RANGE PANIC",__FILE__,__LINE__,
		  "pgrn_binding_get_value",
		  "Bad start index",0);
	
	*next_pos = -1;
    }
	
    if (backlink->values) {
	int i;
	
	for (i = 0; i < backlink->value_count; i++) {

	    if (min_pos <= backlink->values[i].buffer_index &&
		(-1 == found ||
		 backlink->values[i].buffer_index < found)
		) {

		ret   = backlink->values[i].value;
		found = backlink->values[i].buffer_index;

		if (next_pos)
		    *next_pos = found +1;
		if (found_pos)
		    *found_pos = found;

		if (min_pos == backlink->values[i].buffer_index)
		    break;
	    }		
	}	    
    }
    
    if (ret)
	inc_pager_param_value_refcount(ret);
    
    return ret;
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
