static char rcsid[] = "@(#)$Id: rc_parse.c,v 2.16 2025/05/29 05:42:56 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.16 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *
 *  Moved do_rc() from lib/read.rc and splitted to do_rc_parse()
 *  and do_rc_process()
 * 
 *  Based on Elm 2.4 src/read_rc.c. It have following copyright:
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

#include "elm_defs.h"
#include "rc_imp.h"
#include "s_elm.h"
#include "save_opts.h"

/* For INT_MAX */
#include <limits.h>


DEBUG_VAR(Debug,__FILE__,"config");

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

#if 0
static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}
#endif

#define PARSED_RC_magic		0xF405

static struct parsed_rc {
    unsigned short magic;   /*    PARSED_RC_magic */

    struct buffer_data {
	char * buffer;
	int   buffer_len;
	int lineno;
    } * head_comments;
    size_t head_comments_len;

    struct parsed_line {
	struct rc_save_info_rec * record;
	
	char     * rest;
	size_t     restlen;

	int lineno;

	struct hist_record *hist_record;

	enum rec_line_mode  negate_continue;
    }    * setting_lines;
    size_t setting_lines_len;

    struct cached_blocks {
	int lineno;
	char * tag;

	struct cached_line * cached_lines;
	size_t               cached_lines_len;
	

    }    * cached_blocks;
    size_t cached_blocks_len;
    
} * malloc_parsed_rc P_((void));
static struct parsed_rc  * malloc_parsed_rc()
{
    struct parsed_rc * rc = safe_zero_alloc(sizeof (*rc));

    rc->magic = PARSED_RC_magic;

    rc->head_comments     = NULL;
    rc->head_comments_len = 0;

    rc->setting_lines     = NULL;
    rc->setting_lines_len = 0;

    rc->cached_blocks     = NULL;
    rc->cached_blocks_len = 0;

    return rc;
}

static void rc_parsed_add_setting_line P_((struct parsed_rc        * rc,
					   struct rc_save_info_rec * r,
					   const char              * rest,
					   size_t                    restlen,
					   int                       lineno,
					   struct hist_record      * hist,
					   enum rec_line_mode        negate_continue));

static void rc_parsed_add_setting_line(rc,r,rest,restlen,lineno,hist,
				       negate_continue)
     struct parsed_rc        * rc;
     struct rc_save_info_rec * r;
     const char              * rest;
     size_t                    restlen;
     int                       lineno;
     struct hist_record      * hist;
     enum rec_line_mode        negate_continue;
{
    size_t y;
    
    if (PARSED_RC_magic != rc->magic)
	panic("RC PANIC",__FILE__,__LINE__,
	      "rc_parsed_add_setting_line",
	      "Bad magic number",0);


    rc->setting_lines = 
	safe_array_realloc(rc->setting_lines,
			   rc->setting_lines_len +1,
			   sizeof(rc->setting_lines[0]));

    y = rc->setting_lines_len++;
    
    rc->setting_lines[y].rest =
	safe_malloc(restlen+1);
    memcpy(rc->setting_lines[y].rest,rest,restlen+1);
    rc->setting_lines[y].restlen = restlen;
    
    rc->setting_lines[y].lineno          = lineno;
    rc->setting_lines[y].negate_continue = negate_continue;
		    
    rc->setting_lines[y].record      = r;
    rc->setting_lines[y].hist_record = hist;
    
}

static void parsed_rc_add_cache_line P_((struct parsed_rc        * rc,
					 const char              * tag,
					 /* keyword == NULL if continue line */
					 const char              * keyword,
					 const char              * rest,
					 size_t                    restlen,
					 int                       lineno));
static void parsed_rc_add_cache_line(rc,tag,keyword,rest,restlen,lineno)
     struct parsed_rc        * rc;
     const char              * tag;
     /* keyword == NULL if continue line */
     const char              * keyword;
     const char              * rest;
     size_t                    restlen;
     int                       lineno;
{
    size_t x,y;
    
    if (PARSED_RC_magic != rc->magic)
	panic("RC PANIC",__FILE__,__LINE__,
	      "parsed_rc_add_cache_line",
	      "Bad magic number",0);

    if (rc->cached_blocks_len < 1 ||
	0 != strcmp(rc->cached_blocks[rc->cached_blocks_len-1].tag,
		    tag)) {

	DPRINT(Debug,10,(&Debug,"parsed_rc_add_cache_line: Added new block %lu for tag %s, lineno %d\n",
			 (unsigned long)(rc->cached_blocks_len),
			 tag,lineno));

	rc->cached_blocks =
	    safe_array_realloc(rc->cached_blocks,
			       rc->cached_blocks_len+1,
			       sizeof (rc->cached_blocks[0]));

	x = rc->cached_blocks_len++;

	rc->cached_blocks[x].lineno = lineno;
	rc->cached_blocks[x].tag =
	    safe_strdup(tag);
	rc->cached_blocks[x].cached_lines     = NULL;
	rc->cached_blocks[x].cached_lines_len = 0;
	
    } else
	x = rc->cached_blocks_len-1;

    rc->cached_blocks[x].cached_lines =
	safe_array_realloc(rc->cached_blocks[x].cached_lines,
			   rc->cached_blocks[x].cached_lines_len +1,
			   sizeof (rc->cached_blocks[x].cached_lines[0]));

    y = rc->cached_blocks[x].cached_lines_len++;
    
    rc->cached_blocks[x].cached_lines[y].keyword =
	keyword ? safe_strdup(keyword) : NULL;
    rc->cached_blocks[x].cached_lines[y].rest =
	safe_malloc(restlen+1);
    memcpy(rc->cached_blocks[x].cached_lines[y].rest,rest,restlen+1);
    rc->cached_blocks[x].cached_lines[y].restlen = restlen;
    rc->cached_blocks[x].cached_lines[y].lineno  = lineno;
}

static void free_cached_lines P_((struct cached_line ** cached_lines_p,
				  size_t * cached_lines_len_p));

static void free_cached_lines(cached_lines_p,cached_lines_len_p)
     struct cached_line ** cached_lines_p;
     size_t              * cached_lines_len_p;
{
    struct cached_line * cached_lines     = *cached_lines_p;
    size_t               cached_lines_len = *cached_lines_len_p;

    if (cached_lines) {
	size_t j;

	for (j = 0; j < cached_lines_len; j++) {

	    if (cached_lines[j].keyword) {
		free(cached_lines[j].keyword);
		cached_lines[j].keyword = NULL;
	    }

	    if (cached_lines[j].rest) {
		free(cached_lines[j].rest);
		cached_lines[j].rest = NULL;
	    }
	    cached_lines[j].restlen = 0;
		    
	}
	
	free(cached_lines);
	cached_lines = NULL;
    }
    cached_lines_len = 0;

    *cached_lines_p     = cached_lines;
    *cached_lines_len_p = cached_lines_len;
    
}


void free_parsed_rc(rc)
     struct parsed_rc **rc;
{
    if (PARSED_RC_magic != (*rc)->magic)
	panic("RC PANIC",__FILE__,__LINE__,"free_parsed_rc",
	      "Bad magic number",0);

    if ((*rc)->head_comments) {
	size_t i;

	for (i = 0; i < (*rc)->head_comments_len; i++) {
	    if ((*rc)->head_comments[i].buffer) {
		free((*rc)->head_comments[i].buffer);
		(*rc)->head_comments[i].buffer = NULL;
	    }
	    (*rc)->head_comments[i].buffer_len = 0;		
	}

	free((*rc)->head_comments);
	(*rc)->head_comments = NULL;
    }
    (*rc)->head_comments_len = 0;

    if ((*rc)->setting_lines) {
	size_t i;

	for (i = 0; i < (*rc)->setting_lines_len; i++) {

	    if ((*rc)->setting_lines[i].rest) {
		free((*rc)->setting_lines[i].rest);
		(*rc)->setting_lines[i].rest = NULL;
	    }
	    (*rc)->setting_lines[i].restlen = 0;

	}
	
	free((*rc)->setting_lines);
	(*rc)->setting_lines = NULL;
    }
    (*rc)->setting_lines_len = 0;

    if ((*rc)->cached_blocks) {

	size_t i;

	for (i = 0; i < (*rc)->cached_blocks_len; i++) {

	    free_cached_lines(& ((*rc)->cached_blocks[i].cached_lines),
			      & ((*rc)->cached_blocks[i].cached_lines_len));
	    

	}
	

	free((*rc)->cached_blocks);
	(*rc)->cached_blocks = NULL;
    }
    (*rc)->cached_blocks_len = 0;
    
    (*rc)->magic = 0;   /* Invalidate */
    free(*rc);
    *rc = NULL;
}


struct string * cs_option_value(rest,restlen,lineno,filename,cs)
     const char *rest;
     size_t restlen;
     int lineno;
     const char *filename;
     charset_t cs;
{
    struct string * rest_S = new_string(cs);
    const char * csn  = get_charset_MIME_name(cs);
    int ERRORS;
    int rs = 0;

    if (restlen <= (size_t)INT_MAX) 
	rs =  add_streambytes_to_string(rest_S,(int)restlen,cs2us(rest),&ERRORS);

    if (rs < restlen) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedAsCharset,
			  "Failed to parse line %d as charset %s in \"%s\" file"),
		  lineno,
		  csn ? csn : "<no MIME name>",filename);
	
	free_string(&rest_S);
	
	return NULL;
    }
    
    if (ERRORS) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFileAsCharsetErrors,
			  "There is %d errors when parsing line %d as charset %s in \"%s\" file"),
		  ERRORS,lineno, csn ? csn : "<no MIME name>",filename);
	
	free_string(&rest_S);
	
	return NULL;	
    }
    
    return rest_S;
}

static struct string * ascii_option_value P_((const char *rest,
					      size_t restlen,
					      int lineno,
					      const char *filename));
static struct string * ascii_option_value(rest,restlen,lineno,filename)
     const char *rest; 
     size_t restlen;
     int lineno; 
     const char *filename;
{
    return cs_option_value(rest,restlen,lineno,filename,ASCII_SET);
}

void rc_line_ignored_message(rest,restlen,lineno,filename)
     const char              * rest;
     size_t                    restlen;
     int                       lineno;
     const char              * filename;
{
    size_t a;

    for (a = 0; a < restlen; a++) {
	if (! rest[a] ||
	    ! whitespace( rest[a]))
	    break;
    }
    
    if (a < restlen && rest[a] && '#' != rest[a]) {
	char * x = "...";
	int l = 60;
	if (restlen - a < l+3) {
	    l = restlen - a;
	    x = "";
	}
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIgnoredLineElmrc,
			  "Line %d in \"%s\" file ignored: %.*s%s"),
		  lineno,filename,l,rest+a,x);
    }
    
}

struct parsed_rc * do_rc_parse(file,lcl,filename,errors_p,
			       read_flags)
     FILE *file;
     enum record_mode lcl;
     const char *filename;
     int  * errors_p;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    struct parsed_rc *rc = NULL;
    struct rc_save_info_rec * prev_type = NULL;
    struct hist_record      * rec       = NULL;

    char current_tag[SLEN];
    int lineno = 0;

    enum ignore_mode {
	ig_process   = 0,
	ig_ignored  = 1,
	ig_cached   = 2
    } ignored =  ig_process;
       
    char * buffer = NULL;
    int r;  /* length of line or error/eof status */

    int errors = *errors_p;
    int last_was_empty = 1;
    int option_seen    = 0;
    
    if (!file) 
	return NULL;
    
    rc = malloc_parsed_rc();
    
    current_tag[0] = '\0';

    while (0 <= ( r = malloc_gets(&buffer,MAX_LINE,file))) {
	
	char   * tag         = NULL;
	char   keyword[SLEN];
	char   * rest        = NULL;
	int      restlen     = 0;

	int    assig  = 0;

	lineno++;

	if (!buffer)           /* EOF           */
	    break;
	if (r < 1) {            /* empty line    */
	    last_was_empty = 1;
	    continue;
	}
	    
	if (buffer[0] == '#') {   /* comment       */

	    if (lineno < 3) {
		
		rc->head_comments = 
		    safe_realloc(rc->head_comments,
				 (rc->head_comments_len +1) *
				 sizeof (rc->head_comments[0]));

		rc->head_comments[rc->head_comments_len].buffer
		    = safe_malloc(r+1);
		memcpy(rc->head_comments[rc->head_comments_len].buffer,
		       buffer,r+1);
		rc->head_comments[rc->head_comments_len].buffer_len = r;
		rc->head_comments[rc->head_comments_len].lineno = lineno;
		
		rc->head_comments_len++;		    
	    }

	    if (last_was_empty ||
		(r > 4 && 0 == strncmp(buffer,"### ",4))
		) {
	    
		if (prev_type) {
		    if (RCTYPE_magic != prev_type->dt_type->magic)
			panic("RC PANIC",__FILE__,__LINE__,"do_rc_parse",
			      "Bad prev_type",0);
		    
		    DPRINT(Debug,10,(&Debug,"%s:%d: forgotting 'last option': %s\n",
				     filename,lineno,prev_type->name));
		    prev_type = NULL;
		    
		}

		if (option_seen) {
		    DPRINT(Debug,10,(&Debug,"%s:%d: Resetting 'option seen'\n",
				     filename,lineno,prev_type->name));

		    option_seen = 0;
		}
	    }
		
	    continue;
	}
	last_was_empty = 0;

	
	rest = rc_split_keyword(buffer,r,keyword, sizeof keyword,&tag,
				&assig, &restlen);

	if (tag && !keyword[0]) {
	    strfcpy(current_tag, tag, sizeof current_tag);
	    
	    if (rest[0] && '#' != rest[0]) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIgnoredLine,
				  "Rest of line ignored after \"%s:\" on line %d on file %s"),
			  tag, lineno,filename);
		errors++;
	    }

	    if (prev_type) {
		if (RCTYPE_magic != prev_type->dt_type->magic)
		    panic("RC PANIC",__FILE__,__LINE__,"do_rc_parse",
			  "Bad prev_type",0);
		
		DPRINT(Debug,10,(&Debug,"%s:%d: forgotting 'last option': %s\n",
				 filename,lineno,prev_type->name));
		prev_type = NULL;
		
	    }
	    
	    if (option_seen) {
		DPRINT(Debug,10,(&Debug,"%s:%d: Resetting 'option seen'\n",
				 filename,lineno,prev_type->name));
		
		option_seen = 0;
	    }

	} else if (!keyword[0]) {
	    struct string * rest_S UNUSED_VAROK = NULL;
	   	    
	    if (!option_seen) {

		size_t a;

		for (a = 0; a < restlen; a++) {
		    if (! rest[a] ||
			! whitespace( rest[a]))
			break;
		}
    
		if (a < restlen) {
		    char * x = "...";

		    int l = 60;
		    if (restlen - a < l+3) {
			l = restlen - a;
			x = "";
		    }
		
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmContLineIgnored,
				      "Continuation line %d on file \"%s\" ignored: %.*s%s"),
				      lineno,filename,l,rest+a,x);

		    errors++;
		}
		
		DPRINT(Debug,10,(&Debug,"%s:%d: IGNORED [cont] %s\n",
				 filename,lineno,rest));
				
		continue;
	    }


	    switch (ignored) {
	    case ig_process:
		if (prev_type != NULL) {
		    
		    if (RCTYPE_magic != prev_type->dt_type->magic)
			panic("RC PANIC",__FILE__,__LINE__,"do_rc_parse",
			      "Bad prev_type",0);
		    
		    if (rec) 
			hist_record_add_line(rec,rest);
		    
		    if (0 != (prev_type->flags & FL_SYS) && 
			lcl == LOCAL_RC) {

			DPRINT(Debug,10,(&Debug,"%s:%d: NOT ALLOWED [cont] %s=%s\n",
					 filename,lineno,prev_type->name,rest));
			
			continue;
		    }

		    		    
		    if (0 != (prev_type->flags & FL_ASCIIFIRST)) {
			struct string * rest_S = ascii_option_value(rest,restlen,
								    lineno,filename);
			
			if (! rest_S) 
			    goto have_errors_cline;


			if (!rc_eval_tail_cline(prev_type,
						lcl,
						rest_S,
						lineno,
						filename,
						read_flags)) {
			    
			have_errors_cline:
			    
			    errors++;
			    
			    /* If there was errors, then assume that rewriting
			       of elmrc entry causes it to be rewritten
			    */
			    if (rec)
				hist_record_changed(rec);
			}
			
			if (rest_S)
			    free_string(&rest_S);

		    } else {

			rc_parsed_add_setting_line(rc,prev_type,
						   rest,restlen,lineno,rec,
						   line_continue);
			
		    }
		} else 
		    rc_line_ignored_message(rest,restlen,lineno,filename);
		    
		break;
	    case ig_ignored:
		break;
	    case ig_cached:
		if (tag || current_tag[0]) {
		    parsed_rc_add_cache_line(rc,
					     tag ?  tag : current_tag,
					     NULL,
					     rest,restlen,
					     lineno);
		} else
		    rc_line_ignored_message(rest,restlen,lineno,filename);
		break;		
	    }

	} else if (assig) {  
	    int negate = 0;
	    
	    option_seen = 1;
		    
	    
	    if (tag || current_tag[0]) {
#ifdef USE_DLOPEN
		struct rc_save_info_rec * rc_options      = NULL;
	        size_t                    rc_option_count = 0;
		enum shared_have_code     have_code = have_code_invalid;
#endif		
		if (!tag) 
		    tag = current_tag;
		else
		    ignored = ig_process;
		
#ifdef USE_DLOPEN
		if (give_options(tag,&rc_options,&rc_option_count,&have_code)) {
		    
		    prev_type = rc_locate_option(rc_options,rc_option_count,
						 keyword,&negate);
		    ignored = ig_process;
		    
		    if (!prev_type) {
			errors++;
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeyTagInElmrc,
					  "I can't understand keyword \"%s\" (section %s) in line %d in \"%s\" file"),
				  keyword, tag,lineno,filename);
			continue;
		    }
		    
		    
		} else {

		    DPRINT(Debug,10,(&Debug,
				     "do_rc_parse: file %s:%d: tag %s not found, have_code=%d",
				     filename,lineno,tag,have_code));
		    switch(have_code) {
		    case  have_code_invalid:
			DPRINT(Debug,10,(&Debug," (have_code_invalid)"));
			break;
		    case have_code_not:
			DPRINT(Debug,10,(&Debug," (have_code_not)"));
			break;
		    case have_code_loaded:
			DPRINT(Debug,10,(&Debug," (have_code_loaded)"));
			break;
		    case have_code_delayed:
			DPRINT(Debug,10,(&Debug," (have_code_delayed)"));
			break;
		    }
		    DPRINT(Debug,10,(&Debug,"\n"));
		    
		    switch(have_code) {
						
		    case  have_code_invalid:
			
			switch (ignored) {
			case ig_process:
			    ignored = ig_ignored;
			    
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIgnoredSectionInv,
					      "Keywords on section \"%s\" ignored starting from line %d on file %s, tag %s: is invalid."),
				      tag, lineno,filename,
				      tag);
			    break;
			case  ig_ignored:
			    break;
			case ig_cached:
			    /* Should not happed ! */
			    parsed_rc_add_cache_line(rc,tag,
						     keyword,
						     rest,restlen,
						     lineno);
			    break;
			    
			}
			break;

		    case have_code_delayed:
		    case have_code_not:
			
			ignored = ig_cached;
			parsed_rc_add_cache_line(rc,tag,
						 keyword,
						 rest,restlen,
						 lineno);

			break;

		    case have_code_loaded:
			DPRINT(Debug,10,(&Debug," (have_code_loaded)\n"));
			/* Should not happed ! */

			switch (ignored) {
			case ig_process:
			    ignored = ig_ignored;
			    			    			    
			    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIgnoredSection,
					      "Keywords on section \"%s\" ignored starting from line %d on file %s"),
				      tag, lineno,filename);
			case  ig_ignored:
			    break;
			case ig_cached:
			    /* Should not happed ! */
			    parsed_rc_add_cache_line(rc,tag,
						     keyword,
						     rest,restlen,
						     lineno);
			    break;
			    
			}
			break;
		    }
		}
#else
		if (ig_ignored != ignored) {
		    ignored = ig_ignored;
		    
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSharedIgnoredSection,
				      "Shared libraries not supported. Section \"%s\" ignored starting from line %d on file %s"),
			      tag, lineno,filename);
		    errors++;		
		}
#endif
		
	    } else {
		
		prev_type = rc_locate_option(save_info,NUMBER_OF_SAVEABLE_OPTIONS,
					     keyword,&negate);
		
		if (!prev_type) {
		    errors++;
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeyInElmrc,
				      "I can't understand keyword \"%s\" in line %d in \"%s\" file"),
			      keyword, lineno,filename);
		    
		    continue;
		}
		
	    }
	   	    
	    if (!prev_type) 
		continue;
	    
	    rec = rc_need_hist_record(lcl,prev_type,tag);
	    if (rec) 
		hist_record_first_line(rec,rest);
	    	    
	    if (ig_process != ignored)
		continue;
	    
	    
	    if (0 != (prev_type->flags & FL_SYS) && 
		lcl == LOCAL_RC) {

		DPRINT(Debug,10,(&Debug,"%s:%d: NOT ALLOWED %s=%s\n",
				 filename,lineno,prev_type->name,rest));
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmBadLocalKeyInElmrc,
				  "Keyword \"%s\" is not valid on user file %s"),
			  prev_type->name,filename);
		continue;
	    }
	    
	    if (0 != (prev_type->flags & FL_ASCIIFIRST)) {
		struct string * rest_S = ascii_option_value(rest,restlen,
							    lineno,filename);
		
		if (! rest_S) 
		    goto have_errors_line;
		
		if (!rc_eval_tail(prev_type,lcl,rest_S,lineno,filename,negate,
				  read_flags)) {
		    
		have_errors_line:
		    errors++;
		    
		    /* If there was errors, then assume that rewriting
		       of elmrc entry causes it to be rewritten
		    */
		    if (rec)
			hist_record_changed(rec);
		    
		}
		
		if (rest_S)
		    free_string(&rest_S);
		
	    } else {

		rc_parsed_add_setting_line(rc,prev_type,
					    rest,restlen,lineno,rec,
					    negate ? line_negate  :  line_normal);
		
	    }
	    
	} else {
	    prev_type = NULL;
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadLineInElmrc,
			      "Keyword \"%s\" without = -character in line %d in \"%s\" file"),
		      keyword, lineno,filename);
	    errors++;
	}
    }
    
    if (-1 == r) {	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmTooLongLine,
			  "%.30s: %d: Too long line: %.30s..."),
		  filename,lineno+1,
		  buffer ? buffer : "<not available>");
	errors++;
    }
    
    if (ferror(file)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorReading,
			  "Error reading from %.50s"),
		  filename);
	errors++;
    }
    
    if (buffer)
	free(buffer);
    
    *errors_p = errors;
    
    return rc;
}

#define DELAYED_BLOCK_magic   	0xF409

static struct delayed_block {
    unsigned short         magic;    /* DELAYED_BLOCK_magic */
    
    int                    lineno;
    char                 * tag;
    enum record_mode       lcl;
    char                 * filename; 
    charset_t              rcset;
    
    struct cached_line * cached_lines;
    size_t               cached_lines_len;

    struct delayed_block   * next;

}   * first_delayed_block, * last_delayed_block = NULL;


static void add_delayed_block P_((struct cached_blocks  * src /* resets  cached_lines */,
				  enum record_mode        lcl,
				  const char            * filename,
				  charset_t               rcset)) UNUSED_FUNCOK;
static void add_delayed_block(src,lcl,filename,rcset)
     struct cached_blocks  * src /* resets  cached_lines */;
     enum record_mode        lcl;
     const char            * filename;
     charset_t               rcset;
{

    struct delayed_block  * block = safe_zero_alloc(sizeof (*block));

    block->magic = DELAYED_BLOCK_magic;

    block->lineno  = src->lineno;
    block->tag     = safe_strdup(src->tag);

    block->lcl           = lcl;
    block->filename      = safe_strdup(filename);
    block->rcset         = rcset;

    block->cached_lines     = src->cached_lines;
    block->cached_lines_len = src->cached_lines_len;

    src->cached_lines     = NULL;
    src->cached_lines_len = 0;
    
    block->next  = NULL;

    DPRINT(Debug,10,(&Debug,
		     "add_delayed_block: Added delayed block for tag %s, lineno %d, file %s; %lu cached lines\n",
		     block->tag,
		     block->lineno,filename,
		     (unsigned long)(block->cached_lines_len)));
    
    if (first_delayed_block) {
	if (DELAYED_BLOCK_magic != first_delayed_block->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"add_delayed_block",
		  "Bad magic number (delayed_block, first_delayed_block)",0);

	if (!last_delayed_block)
	    panic("RC PANIC",__FILE__,__LINE__,"add_delayed_block",
		  "first_delayed_block set, last_delayed_block == NULL",0);
    } else {
	if (last_delayed_block)
	    panic("RC PANIC",__FILE__,__LINE__,"add_delayed_block",
		  "first_delayed_block == NULL, last_delayed_block set",0);
	
	first_delayed_block = block;
    }
    
    if (last_delayed_block) {
	if (DELAYED_BLOCK_magic != last_delayed_block->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"add_delayed_block",
		  "Bad magic number (delayed_block, last_delayed_block)",0);

	last_delayed_block->next = block;
    }
	
    last_delayed_block = block;
}

void clear_delayed_block P_((struct delayed_block *b));
void clear_delayed_block(b)
     struct delayed_block *b;
{
    if (DELAYED_BLOCK_magic != b->magic)
	panic("RC PANIC",__FILE__,__LINE__,"clear_delayed_block",
	      "Bad magic number (delayed_block)",0);

    DPRINT(Debug,10,(&Debug,"clear_delayed_block: Clearing delayed block for"));
    if (b->tag) {
	DPRINT(Debug,10,(&Debug," tag %s, ",
			 b->tag));
	free(b->tag);
	b->tag = NULL;
    }
    
    DPRINT(Debug,10,(&Debug,"lineno %d",
		     b->lineno));
    b->lineno = -1;
    b->lcl    = RC_MODE_COUNT;   /* Invalid value */
    b->rcset  = system_charset;  /* Default value */
    
    if (b->filename) {
	DPRINT(Debug,10,(&Debug,", file %s",
			 b->filename));
	
	free(b->filename);
	b->filename = NULL;
    }
    DPRINT(Debug,10,(&Debug,"; %lu cached lines\n",
		     (unsigned long)(b->cached_lines_len)));
    
    free_cached_lines(& (b->cached_lines),
		      & (b->cached_lines_len));    
}

void free_delayed_blocks()
{
    struct delayed_block *b;

    for (b = first_delayed_block; b; b = first_delayed_block) {

	if (DELAYED_BLOCK_magic != b->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"free_delayed_blocks",
		  "Bad magic number (delayed_block)",0);
	
	first_delayed_block = b->next;
	b->next = NULL;

	clear_delayed_block(b);

	b->magic = 0; /* Invalidate */
	free(b);
	b = NULL;
    }
    last_delayed_block = NULL;
   
}


static void process_cached_lines P_((const char              * tag,
				     enum record_mode          lcl,
				     const char              * filename,
				     charset_t                 rcset,
				     struct cached_line      * cached_lines,
				     size_t                    cached_lines_len,
				     struct rc_save_info_rec * rc_options,
				     size_t                    rc_option_count,
				     int                     * errors_p,
				     int read_flags /* READ_FLAG_IGNORE_MISSING */));

static void process_cached_lines(tag,lcl,filename,rcset,
				 cached_lines,cached_lines_len,
				 rc_options,rc_option_count,
				 errors_p,read_flags)
     const char              * tag;
     enum record_mode          lcl;
     const char              * filename;
     charset_t                 rcset;
     struct cached_line      * cached_lines;
     size_t                    cached_lines_len;
     struct rc_save_info_rec * rc_options;
     size_t                    rc_option_count;
     int                     * errors_p;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{

    if (cached_lines) {

	struct rc_save_info_rec * prev_type = NULL;
	struct hist_record      * rec       = NULL;
	size_t x;

	DPRINT(Debug,10,(&Debug,
			 "process_cached_lines: file %s, tag %s, %lu cached lines, %lu rc_options\n",
			 filename,tag,
			 (unsigned long)cached_lines_len,
			 (unsigned long)rc_option_count));
	
	for (x = 0;  x < cached_lines_len; x++) {
	    struct string           * rest_S    = NULL;
	    int code                            = 0;
			  
	    if (cached_lines[x].keyword) {
		int negate = 0;

		prev_type = rc_locate_option(rc_options,rc_option_count,
					     cached_lines[x].keyword,
					     &negate);

		if (!prev_type) {
		    
		    if (errors_p)
			(*errors_p)++;
		    
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBadKeyTagInElmrc,
				      "I can't understand keyword \"%s\" (section %s) in line %d in \"%s\" file"),
			      cached_lines[x].keyword,
			      tag,
			      cached_lines[x].lineno,filename);
		    continue;
		}
		
		if (RCTYPE_magic != prev_type->dt_type->magic)
		    panic("RC PANIC",__FILE__,__LINE__,"process_cached_lines",
			  "Bad prev_type",0);

		if (0 != (prev_type->flags & FL_SYS) && 
		    lcl == LOCAL_RC) {
				  
		    DPRINT(Debug,10,(&Debug,"%s:%d: NOT ALLOWED %s=%s\n",
				     filename,
				     cached_lines[x].lineno,
				     prev_type->name,
				     cached_lines[x].rest));
				  
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmBadLocalKeyInElmrc,
				      "Keyword \"%s\" is not valid on user file %s"),
			      
			      prev_type->name,filename);
		    continue;
		}
					    
		rec = rc_need_hist_record(lcl,prev_type,
					  tag);
			      
		if (rec) 
		    hist_record_first_line(rec,
					   cached_lines[x].rest);
			      
		if (0 != (prev_type->flags & FL_ASCIIFIRST)) 
		    rest_S = ascii_option_value(cached_lines[x].rest,
						cached_lines[x].restlen,
						cached_lines[x].lineno,
						filename);
		else 
		    rest_S = cs_option_value(cached_lines[x].rest,
					     cached_lines[x].restlen,
					     cached_lines[x].lineno,
					     filename,rcset);
			      
		if (!rest_S)
		    goto have_errors_1;

		code = rc_eval_tail(prev_type,lcl,rest_S,
				    cached_lines[x].lineno,
				    filename,negate,read_flags);
			      
	    } else if (prev_type) {
		
		if (RCTYPE_magic != prev_type->dt_type->magic)
		    panic("RC PANIC",__FILE__,__LINE__,"process_cached_lines",
			  "Bad prev_type",0);
		
		if (rec) 
		    hist_record_add_line(rec,
					 cached_lines[x].rest);
		
		if (0 != (prev_type->flags & FL_SYS) && 
		    lcl == LOCAL_RC) {
		    
		    DPRINT(Debug,10,(&Debug,"%s:%d: NOT ALLOWED [cont] %s=%s\n",
				     filename,
				     cached_lines[x].lineno,
				     prev_type->name,
				     cached_lines[x].rest));
				  
		    continue;
		}

		if (0 != (prev_type->flags & FL_ASCIIFIRST)) 
		    rest_S = ascii_option_value(cached_lines[x].rest,
						cached_lines[x].restlen,
						cached_lines[x].lineno,
						filename);
		else 
		    rest_S = cs_option_value(cached_lines[x].rest,
					     cached_lines[x].restlen,
					     cached_lines[x].lineno,
					     filename,rcset);
			      
		if (!rest_S)
		    goto have_errors_1;
			      
		code = rc_eval_tail_cline(prev_type,
					  lcl,
					  rest_S,
					  cached_lines[x].
					  lineno,filename,
					  read_flags);
			      
	    } else {
		rc_line_ignored_message(cached_lines[x].rest,
					cached_lines[x].restlen,
					cached_lines[x].lineno,
					filename);
		code = -1;
	    }

	    if (0 == code) {
	    have_errors_1:

		if (errors_p)
		    (*errors_p)++;
		
		/* If there was errors, then assume that rewriting
		   of elmrc entry causes it to be rewritten
		*/
		if (rec)
		    hist_record_changed(rec);
	    }
			  
	    if (rest_S)
		free_string(&rest_S);
	}
    }
}

int give_cached_lines(tag,walk,lcl,rcset,filename,cached_lines,cached_lines_len)
     const char            * tag;
     /* In out */
     struct delayed_block ** walk;
				 
     /* Output: */
     enum record_mode      * lcl;
     charset_t             * rcset;
     char                 ** filename;
     struct cached_line   ** cached_lines;
     size_t                * cached_lines_len;
{
    struct delayed_block *b = *walk;

    if (b) {
	if (DELAYED_BLOCK_magic != b->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"give_cached_lines",
		  "Bad magic number (delayed_block)",0);

	b = b->next;
    } else
	b = first_delayed_block;

    while (b) {
	if (DELAYED_BLOCK_magic != b->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"give_cached_lines",
		  "Bad magic number (delayed_block)",0);

	if (b->tag &&
	    0 == strcmp(b->tag,tag)) {

	    DPRINT(Debug,10,(&Debug,
			     "give_cached_lines: Returning delayed block for %s, filename %s, lineno %d, lcl %d",
			     b->tag,b->filename,
			     b->lineno,b->lcl));  
	    switch (b->lcl) {
	    case SYSTEM_RC:     DPRINT(Debug,10,(&Debug," (SYSTEM_RC)"));     break;
	    case LOCAL_RC:      DPRINT(Debug,10,(&Debug," (LOCAL_RC)"));      break;
	    case RC_MODE_COUNT: DPRINT(Debug,10,(&Debug," (RC_MODE_COUNT)")); break;
	    }
	    DPRINT(Debug,10,(&Debug,"\n"));
	    
	    if (lcl)
		*lcl              = b->lcl;
	    if (rcset)
		*rcset            = b->rcset;
	    if (filename)
		*filename         = b->filename;
	    if (cached_lines)
		*cached_lines     = b->cached_lines;
	    if (cached_lines_len)
		*cached_lines_len = b->cached_lines_len;

	    *walk = b;
	    return 1;
	}
	b = b->next;
    }
    
    return 0;
}


/* Removes processed block */
void process_delayed_blocks(tag,rc_options,rc_option_count,
			    errors_p,read_flags)
     const char              * tag;     
     struct rc_save_info_rec * rc_options;
     size_t                    rc_option_count;
     int                     * errors_p;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
   struct delayed_block *b;

   struct delayed_block *next_block      = NULL;
   struct delayed_block *previous_block  = NULL;
   
   for (b = first_delayed_block; b; b = next_block) {

	if (DELAYED_BLOCK_magic != b->magic)
	    panic("RC PANIC",__FILE__,__LINE__,"free_delayed_blocks",
		  "Bad magic number (delayed_block)",0);
	
	if (b->tag &&
	    0 == strcmp(b->tag,tag)) {

	    DPRINT(Debug,10,(&Debug,
			     "process_delayed_blocks: Processing delayed block for %s, filename %s, lineno %d, lcl %d",
			     b->tag,b->filename,
			     b->lineno,b->lcl));  
	    switch (b->lcl) {
	    case SYSTEM_RC:     DPRINT(Debug,10,(&Debug," (SYSTEM_RC)"));     break;
	    case LOCAL_RC:      DPRINT(Debug,10,(&Debug," (LOCAL_RC)"));      break;
	    case RC_MODE_COUNT: DPRINT(Debug,10,(&Debug," (RC_MODE_COUNT)")); break;
	    }
	    DPRINT(Debug,10,(&Debug,"\n"));
	    
	    process_cached_lines(b->tag,
				 b->lcl,
				 b->filename,
				 b->rcset,
				 b->cached_lines,
				 b->cached_lines_len,
				 rc_options,
				 rc_option_count,
				 errors_p,
				 read_flags);

	    clear_delayed_block(b);

	    if (previous_block) {
		if (DELAYED_BLOCK_magic != previous_block->magic)
		    panic("RC PANIC",__FILE__,__LINE__,"free_delayed_blocks",
			  "Bad magic number (delayed_block, previous_block)",0);
		previous_block->next = b->next;
	    } else
		first_delayed_block = b->next;

	    if (last_delayed_block == b)
		last_delayed_block = previous_block;

	    next_block      = b->next;

	    b->magic = 0; /* Invalidate */
	    free(b);
	    b = NULL;
	    	    
	} else {
	    next_block      = b->next;
	    previous_block  = b;
	}
    }
}

struct editor_propline * current_editor_propline;

void do_rc_process(rc,lcl,filename,elmrc_charset,
		   errors_p,read_flags)
     struct parsed_rc *rc; 
     enum record_mode lcl;
     const char *filename; 
     charset_t elmrc_charset; /* NULL, if no @charset tag */
     int *errors_p;
     int read_flags /* READ_FLAG_IGNORE_MISSING */;
{
    charset_t rcset = system_charset;
    const char * csn UNUSED_VAROK;

    int errors = *errors_p;

     if (PARSED_RC_magic != rc->magic)
	panic("RC PANIC",__FILE__,__LINE__,"do_rc_process",
	      "Bad magic number (parsed_rc)",0);

     /* Nothing detected yet */
     if (current_editor_propline)
	 free_editor_propline(& current_editor_propline);
   
     if (editor_propline_ignore != editor_elmrc_propline.val) {
	 size_t i;
	 
	 for (i = 0; 
	      i < rc->head_comments_len &&
		  !current_editor_propline; 
	      i++) {

	     current_editor_propline =
		 detect_editor_propline(editor_elmrc_propline.val,
					rc->head_comments[i].buffer,
					rc->head_comments[i].buffer_len,
					filename,
					rc->head_comments[i].lineno,&rcset);

	 }
     }

     if (elmrc_charset) {
	 csn  = get_charset_MIME_name(elmrc_charset);

	 rcset = elmrc_charset;
	 
	 DPRINT(Debug,20,(&Debug,
			  "do_rc_process: %s: Given %s charset\n",
			  filename,csn ? csn : "<no MIME name>"));
     } else {
	 csn  = get_charset_MIME_name(rcset);

	 DPRINT(Debug,20,(&Debug,
			  "do_rc_process: %s: Using %s as charset\n",
			  filename,csn ? csn : "<no MIME name>"));
     }

     if (rc->setting_lines) {
         size_t i;
	 
	 for (i = 0; i < rc->setting_lines_len; i++) {
	     struct string * rest_S =
		 cs_option_value(rc->setting_lines[i].rest,
				 rc->setting_lines[i].restlen,
				 rc->setting_lines[i].lineno,
				 filename,rcset);
	     int code = 0;
	     
	     if (!  rest_S) 
		 goto have_errors_sline;
	     
	     	     
	     if (RCTYPE_magic != rc->setting_lines[i].record -> dt_type->magic)
		 panic("RC PANIC",__FILE__,__LINE__,"do_rc_process",
		       "Bad magic number for record",0);

	     code =  rc_eval_tail_line(rc->setting_lines[i].record,
				       lcl,rest_S,
				       rc->setting_lines[i].lineno,
				       filename,
				       rc->setting_lines[i].negate_continue,
				       read_flags);
	     				 
	     if (!code) {
	     have_errors_sline:
		 errors++;
		 
		 /* If there was errors, then assume that rewriting
		    of elmrc entry causes it to be rewritten
		 */
		 if (rc->setting_lines[i].hist_record)
		     hist_record_changed(rc->setting_lines[i].hist_record);
	     }

	     if (rest_S)
		 free_string(&rest_S);
	 }
     }

     if (rc->cached_blocks) {
	 size_t b;
	 char * ignored = NULL;
	 
	 for (b = 0; b <  rc->cached_blocks_len; b++) {

#ifdef USE_DLOPEN
	     struct rc_save_info_rec * rc_options      = NULL;
	     size_t                    rc_option_count = 0;
	     enum shared_have_code     have_code = have_code_invalid;

	     
	     if (give_options(rc->cached_blocks[b].tag,
			      &rc_options,&rc_option_count,
			      &have_code)) {
		 
		  ignored = NULL;

		  process_cached_lines(rc->cached_blocks[b].tag,
				       lcl,filename,rcset,
				       rc->cached_blocks[b].cached_lines,
				       rc->cached_blocks[b].cached_lines_len,
				       rc_options,
				       rc_option_count,
				       &errors,read_flags);
		  		 
	     } else {

		 DPRINT(Debug,10,(&Debug,
				  "do_rc_process: file %s:%d: tag %s not found, have_code=%d",
				  filename,rc->cached_blocks[b].lineno,
				  rc->cached_blocks[b].tag,have_code));
		 switch(have_code) {
		 case  have_code_invalid:
		     DPRINT(Debug,10,(&Debug," (have_code_invalid)"));
		     break;
		 case have_code_not:
		     DPRINT(Debug,10,(&Debug," (have_code_not)"));
		     break;
		 case have_code_loaded:
		     DPRINT(Debug,10,(&Debug," (have_code_loaded)"));
		     break;
		 case have_code_delayed:
		     DPRINT(Debug,10,(&Debug," (have_code_delayed)"));
		     break;
		 }
		 DPRINT(Debug,10,(&Debug,"\n"));
		 
		 if (have_code_delayed == have_code) {
		     ignored = NULL;

		     add_delayed_block(& (rc->cached_blocks[b]),
				       lcl,filename,rcset);
		     
		 } else if (!ignored ||
		     0 != strcmp(ignored,rc->cached_blocks[b].tag)) {
		     
		     ignored = rc->cached_blocks[b].tag;

		     lib_error(CATGETS(elm_msg_cat, ElmSet,
				       ElmIgnoredSection,
					      "Keywords on section \"%s\" ignored starting from line %d on file %s"),
			       rc->cached_blocks[b].tag,
			       rc->cached_blocks[b].lineno,
			       filename);
		     
		 }
	     }
#else
	      if (!ignored ||
		  0 != strcmp(ignored,rc->cached_blocks[b].tag)) {
		  
		  ignored = rc->cached_blocks[b].tag;
		  
		  lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSharedIgnoredSection,
				    "Shared libraries not supported. Section \"%s\" ignored starting from line %d on file %s"),
			    rc->cached_blocks[b].tag,
			    rc->cached_blocks[b].lineno,
			    filename);
		  errors++;	
	      }
#endif
	 }
     }
     
     *errors_p = errors;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
