static char rcsid[] = "@(#)$Id: sb_mem.c,v 2.11 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.11 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "s_me.h"
#include "sb_imp.h"
#include "pg_range_imp.h"
#include "pg_lineext_imp.h"

/* Needs STRING_magic */
#include "elmme-magic.h"
#include "pg_params_imp.h"


DEBUG_VAR(Debug,__FILE__,"charset");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#define SB_in_mem_magic	     0xEB08

struct sb_in_mem {
    unsigned short magic;     /* SB_in_mem_magic */

    int              string_count;
    struct sb_line_data {
	struct string *line;
	int            pg_flags;
	struct pager_range *range;
	struct plext_in_mem *next_extension;
    } * strings;

    struct plext_in_mem *last_extension;

    struct sb_pager_range {
	/* Not really used */
	struct pgrng_bfr_binding *pager_range;

	struct pager_param_value ** param_value;
	int param_value_count;
	
    } * pager_range;
    int pager_range_count;
    
};


S_(sb_init_buffer sb_init_mem)
static int sb_init_mem P_((struct stringbuffer *buffer));
static int sb_init_mem(buffer)
     struct stringbuffer *buffer;
{
    buffer->a.mem = safe_zero_alloc(sizeof (* (buffer->a.mem)));

    buffer->a.mem->magic = SB_in_mem_magic;

    buffer->a.mem->string_count = 0;
    buffer->a.mem->strings      = NULL;

    buffer->a.mem->last_extension = NULL;

    buffer->a.mem->pager_range = NULL;
    buffer->a.mem->pager_range_count = 0;

    return 1;
}


static void free_plext_in_mem P_((struct plext_in_mem **mem));

S_(sb_free_buffer sb_free_mem)
static void sb_free_mem P_((struct stringbuffer *buffer));
static void sb_free_mem(buffer)
     struct stringbuffer *buffer;
{
    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"sb_free_mem",
	      "Bad magic number",0);
        
    if (buffer->a.mem->strings) {
	int i;

	for (i = 0; i < buffer->a.mem->string_count; i++) {
	    if (buffer->a.mem->strings[i].line)
		free_string(&(buffer->a.mem->strings[i].line));
	    if (buffer->a.mem->strings[i].range)
		free_pager_range(&(buffer->a.mem->strings[i].range));
	    if (buffer->a.mem->strings[i].next_extension)
		free_plext_in_mem(& (buffer->a.mem->strings[i].next_extension));	
	}

	free(buffer->a.mem->strings);
	buffer->a.mem->strings      = NULL;
    }
    buffer->a.mem->string_count = 0;

    if (buffer->a.mem->last_extension)
	free_plext_in_mem(& (buffer->a.mem->last_extension));

    if (buffer->a.mem->pager_range) {
	int i;

	for (i = 0; i < buffer->a.mem->pager_range_count; i++) {
	    if (buffer->a.mem->pager_range[i].pager_range) {
		pgrng_bfr_bnd_clear_stringbuffer(& (buffer->a.mem->
						    pager_range[i].pager_range),
						 buffer,i);
	    }

	    if (buffer->a.mem->pager_range[i].param_value) {
		int j;

		for (j = 0;
		     j < buffer->a.mem->pager_range[i].param_value_count;
		     j++) {
		    
		    if (buffer->a.mem->pager_range[i].param_value[j])
			
			free_pager_param_value(& (buffer->a.mem->pager_range[i].
						  param_value[j]));
		    
		}
		
		free(buffer->a.mem->pager_range[i].param_value);
		buffer->a.mem->pager_range[i].param_value = NULL;

	    }
	    buffer->a.mem->pager_range[i].param_value_count = 0;
	}
	
	free(buffer->a.mem->pager_range);
	buffer->a.mem->pager_range = NULL;
    }
    buffer->a.mem->pager_range_count = 0;

    buffer->a.mem->magic = 0;   /* Invalidate */
    free(buffer->a.mem);
    buffer->a.mem = NULL;
}

S_(sb_add_line_to_stringbuffer sb_add_line_to_mem)
static void sb_add_line_to_mem P_((struct stringbuffer *buffer,
				   const struct string *string,
				   int pg_flags,struct pager_range *range,
				   struct pgrng_bfr_binding  * binding));
static void sb_add_line_to_mem(buffer,string,pg_flags,range,binding)
     struct stringbuffer *buffer;
     const struct string *string;
     int pg_flags;
     struct pager_range *range;    
     struct pgrng_bfr_binding  * binding;  /* NOT USED */
{
    int ptr;

    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"sb_add_line_to_mem",
	      "Bad magic number",0);

    ptr = buffer->a.mem->string_count;
    
    buffer->a.mem->strings = safe_array_realloc(buffer->a.mem->strings,
						(buffer->a.mem->string_count+1),
						sizeof (buffer->a.mem->strings[0]));

    buffer->a.mem->strings[ptr].line     = dup_string(string);
    buffer->a.mem->strings[ptr].pg_flags = pg_flags;
    buffer->a.mem->strings[ptr].range     = range;
    buffer->a.mem->strings[ptr].next_extension = NULL;
    if (buffer->a.mem->strings[ptr].range)
	inc_pager_range_refcount(buffer->a.mem->strings[ptr].range);

    buffer->a.mem->string_count++;

    /* Mark beginning of line */
    if (buffer->a.mem->last_extension)
	free_plext_in_mem(& (buffer->a.mem->last_extension));
}

static struct plext_in_mem * malloc_plext_in_mem P_((void));
static void inc_plext_in_mem_refcount P_((struct plext_in_mem *mem));

S_(sb_add_lineext_to_stringbuffer sb_add_lineext_to_mem)
static void sb_add_lineext_to_mem P_((struct stringbuffer *buffer,
				      const struct string *string,
				      int pg_flags,struct pager_range *range,
				      struct pgrng_bfr_binding  * binding));


S_(sb_linecount_stringbuffer sb_linecount_mem)
static int sb_linecount_mem P_((const struct stringbuffer *ptr));
static int sb_linecount_mem(ptr)
     const struct stringbuffer *ptr;
{
    if (SB_in_mem_magic != ptr->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"sb_linecount_mem",
	      "Bad magic number",0);

    return ptr->a.mem->string_count;
}

static struct pager_lineext * sb_mem_copy_extension P_((struct stringbuffer *buffer,
							int ptr));


S_(sb_get_line_from_stringbuffer sb_get_line_from_mem)
static struct string *sb_get_line_from_mem P_((struct stringbuffer *buffer,
					       int ptr, int *pg_flags,
					       struct pager_range **range,
					       struct pager_lineext **first_extension));
static struct string *sb_get_line_from_mem(buffer,ptr,pg_flags,range,first_extension)
     struct stringbuffer *buffer;
     int ptr;
     int *pg_flags;
     struct pager_range **range;  
     struct pager_lineext **first_extension;
{
    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"sb_get_line_from_mem",
	      "Bad magic number",0);

    if (ptr < 0 || ptr >= buffer->a.mem->string_count)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"sb_get_line_from_mem",
	      "Bad index",0);

    if (range &&
	*range != buffer->a.mem->strings[ptr].range) {
	
	if (*range)
	    free_pager_range(range);

	*range = buffer->a.mem->strings[ptr].range;
	if (*range)
	    inc_pager_range_refcount(*range);	    
    }

    if (first_extension) {
	if (*first_extension)
	    free_pager_lineext(first_extension);

	if (buffer->a.mem->strings[ptr].next_extension)
	    *first_extension = sb_mem_copy_extension(buffer,ptr);
    }

    *pg_flags = buffer->a.mem->strings[ptr].pg_flags;
    return dup_string(buffer->a.mem->strings[ptr].line);
}

S_(sb_walk_lineext_on_stringbuffer sb_walk_lineext_on_mem)
static struct string *sb_walk_lineext_on_mem
                  P_((struct stringbuffer *buffer,
		      struct pager_lineext **walk_extension,
		      int *pg_flags,
		      struct pager_range **range));

S_(sb_add_pager_range_to_stringbuffer sb_add_pager_range_to_mem)
static void sb_add_pager_range_to_mem P_((struct stringbuffer * buffer,
					  struct pgrng_bfr_binding *
					  binding,
					  struct pager_range * range));
static void sb_add_pager_range_to_mem(buffer,binding,range)
     struct stringbuffer * buffer;
     struct pgrng_bfr_binding * binding;
     struct pager_range * range;
{
    int ptr;

    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_pager_range_to_mem",
	      "Bad magic number",0);

    /* Because index numbers are not required,
       also this cache is not needed really */

    ptr = buffer->a.mem->pager_range_count;

    buffer->a.mem->pager_range = 
	safe_array_realloc(buffer->a.mem->pager_range,
			   (buffer->a.mem->pager_range_count +1),
			   sizeof (buffer->a.mem->pager_range[0]));
    buffer->a.mem->pager_range[ptr].pager_range = binding;

    buffer->a.mem->pager_range[ptr].param_value = NULL;
    buffer->a.mem->pager_range[ptr].param_value_count = 0;

    
    pgrng_bfr_bnd_set_stringbuffer(buffer->a.mem->pager_range[ptr].
				   pager_range,
				   buffer,ptr);
    buffer->a.mem->pager_range_count++;

}

S_(sb_get_pager_range_from_stringbuffer sb_get_pager_range_from_mem)
static struct pager_range * sb_get_pager_range_from_mem
  P_((struct stringbuffer * buffer,int index));
static struct pager_range * sb_get_pager_range_from_mem(buffer,index)
     struct stringbuffer * buffer;
     int index;
{
    struct pager_range *ret = NULL;
    /* Not used */

    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_range_from_mem",
	      "Bad magic number",0);

    if (index < 0 || index >= buffer->a.mem->pager_range_count)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_range_from_mem",
	      "Bad index",0);
	
    if (! buffer->a.mem->pager_range[index].pager_range)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_range_from_mem",
	      "No binding",0);

    ret = pgrng_bfr_bnd_get_range(buffer->a.mem->pager_range[index].pager_range);

    if (!ret) {
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_range_from_mem",
	      "pager range deallocated?",0);
    }
    
    return ret;
}

static void sb_add_string_alt_to_mem
   P_((struct stringbuffer * buffer,
       struct string_alt   * string_alt,
       int                   pg_flags,
       struct pager_range  * range));

/* entity is processed as lineext */
S_(sb_add_entity_to_stringbuffer sb_add_entity_to_mem)
static  enum stringbuffer_entity_result sb_add_entity_to_mem
   P_((struct stringbuffer *buffer,
       const struct out_entity *oe,
       int pg_flags,struct pager_range *range,
       struct pgrng_bfr_binding  * binding));
static  enum stringbuffer_entity_result sb_add_entity_to_mem(buffer,oe,
							     pg_flags,range,
							     binding)
     struct stringbuffer *buffer;
     const struct out_entity *oe;
     int pg_flags;
     struct pager_range *range;
     struct pgrng_bfr_binding  * binding;
{
    struct out_entity * oe2 = dup_out_entity(oe);  /* drop const */
    
    struct string_alt * strX =
	new_string_alt_entity(oe2);     
       
    sb_add_string_alt_to_mem(buffer,strX,pg_flags,range);

    free_string_alt(& strX);
    free_out_entity(& oe2); 

    return sb_entity_succeed;
}

S_(sb_add_pager_param_to_stringbuffer sb_add_pager_param_to_mem)
static void sb_add_pager_param_to_mem 
    P_((struct stringbuffer       * buffer,
	struct pgrng_bfr_binding  * binding,
	struct pager_range        * range,
	enum pager_param_opcode     opcode,
	struct pager_param_value  * param_value));
static void sb_add_pager_param_to_mem (buffer,binding,range,opcode,
				       param_value)
     struct stringbuffer       * buffer;
     struct pgrng_bfr_binding  * binding;
     struct pager_range        * range;
     enum pager_param_opcode     opcode;
     struct pager_param_value  * param_value;
{

    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_pager_param_to_mem",
	      "Bad magic number",0);

    if (binding) {
	int range_index;
	struct stringbuffer *bf = 
	    pgrng_bfr_bnd_get_stringbuffer(binding,&range_index);
	struct sb_pager_range * sb_range;

	int ptr;
	
    	if (bf != buffer)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_add_pager_param_to_mem",
		  "pager_range (binding) added to wrong stringbuffer",0);

	if (range_index < 0 || 
	    range_index >=  buffer->a.mem->pager_range_count ||
	    buffer->a.mem->pager_range[range_index].pager_range != binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_add_pager_param_to_mem",
		  "bad range_index",0);
	sb_range = &(buffer->a.mem->pager_range[range_index]);

	ptr = sb_range->param_value_count;

	sb_range->param_value =
	    safe_array_realloc(sb_range->param_value,
			       sb_range->param_value_count+1,
			       sizeof (sb_range->param_value[0]));

	sb_range->param_value[ptr] = param_value;
	inc_pager_param_value_refcount(sb_range->param_value[ptr]);

	sb_range->param_value_count++;

	/* Not really needed */
	pgrng_binding_add_value(binding,param_value,ptr);

	free_stringbuffer(&bf);

    } else
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_pager_param_to_mem",
	      "pager_range (binding) missing",0);

}

S_(sb_get_pager_param_from_stringbuffer sb_get_pager_param_from_mem)
static struct pager_param_value * sb_get_pager_param_from_mem
     P_((struct stringbuffer      * buffer,
	 struct pager_range       * range,
	 enum pager_param_opcode    opcode,
	 int                      * next_pos,
	 struct pgrng_bfr_binding * binding,
	 int                      * found_pos));
static struct pager_param_value * sb_get_pager_param_from_mem(buffer,
							      range,
							      opcode,
							      next_pos,
							      binding,
							      found_pos)
     struct stringbuffer      * buffer;
     struct pager_range       * range;
     enum pager_param_opcode    opcode;
     int                      * next_pos;
     struct pgrng_bfr_binding * binding;
     int                      * found_pos;
{
    struct pager_param_value * ret = NULL;
    int                        min_pos = 0;
    
    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_param_from_mem",
	      "Bad magic number",0);

    if (found_pos)
	 *found_pos = -1;

    if (next_pos) {
	min_pos = *next_pos;

	if (min_pos < 0)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_pager_param_from_mem",
		  "Bad start index",0);
	
	*next_pos = -1;
    }
    
    if (binding) {
	int range_index;
	struct stringbuffer *bf = 
	    pgrng_bfr_bnd_get_stringbuffer(binding,&range_index);
	struct sb_pager_range * sb_range;
	
    	if (bf != buffer)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_pager_param_from_mem",
		  "pager_range (binding) added to wrong stringbuffer",0);

	if (range_index < 0 || 
	    range_index >=  buffer->a.mem->pager_range_count ||
	    buffer->a.mem->pager_range[range_index].pager_range != binding)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_get_pager_param_from_mem",
		  "bad range_index",0);
	sb_range = &(buffer->a.mem->pager_range[range_index]);

	if (sb_range->param_value) {
	    int i;
	    
	    for (i = min_pos; i < sb_range->param_value_count; i++) {

		if (sb_range->param_value[i]) {

		    if (match_pager_param_value(sb_range->param_value[i],
						opcode)) {
			ret =  sb_range->param_value[i];

			inc_pager_param_value_refcount(ret);
			if (next_pos)
			    *next_pos = i+1;
			if (found_pos)
			    *found_pos = i; 
			break;
		    }
		}
	    }
	}

	free_stringbuffer(&bf);
	
    } else
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_get_pager_param_from_mem",
	      "pager_range (binding) missing",0);
   
    /* Already done:
       inc_pager_param_value_refcount(ret);
    */
    
    return ret;
}


struct sb_type   sb_in_mem = {
    SB_TYPE_magic,

    sb_init_mem,
    sb_free_mem,
    sb_add_line_to_mem,
    sb_linecount_mem,
    sb_get_line_from_mem,
    sb_add_pager_range_to_mem,
    sb_get_pager_range_from_mem,
    sb_add_lineext_to_mem,
    sb_walk_lineext_on_mem,
    sb_add_entity_to_mem,
    sb_add_pager_param_to_mem,
    sb_get_pager_param_from_mem
};

/* ------------------------------------------------------------------ */

#define PLEXT_IN_MEM_magic		0xEB17

struct plext_in_mem {
    unsigned short            magic;       /* PLEXT_IN_MEM_magic */

    int            refcount;

    struct string_alt   * line_or_other;
    int                   pg_flags;
    struct pager_range  * range;
    
    struct plext_in_mem * next_extension;
};

static void sb_add_string_alt_to_mem(buffer,string_alt,
				     pg_flags,range)
     struct stringbuffer * buffer;
     struct string_alt   * string_alt;
     int                   pg_flags;
     struct pager_range  * range;
{

     struct plext_in_mem * ext;
     union string_alt_value u UNUSED_VAROK;  /* Debug */
     enum string_alt_type   t = string_alt_none;
     int ptr;

    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_string_alt_to_mem",
	      "Bad magic number",0);

    if (buffer->a.mem->string_count < 0) {
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_add_string_alt_to_mem",
	      "No line",0);
    }

    u = get_string_alt_value(string_alt,&t);

    DPRINT(Debug,25,(&Debug,"sb_add_string_alt_to_mem: "));
    switch (t) {
    case string_alt_none:    DPRINT(Debug,25,(&Debug,"string_alt_none")); break;
    case string_alt_text:   DPRINT(Debug,25,(&Debug,"string_alt_text")); break;
    case string_alt_entity: DPRINT(Debug,25,(&Debug,"string_alt_entity")); break; 
    }
    DPRINT(Debug,25,(&Debug,"\n"));
    
    
    ptr     = buffer->a.mem->string_count-1;

    ext =  malloc_plext_in_mem();

    ext->line_or_other     = string_alt;
    inc_string_alt_refcount(ext->line_or_other);
    
    ext->pg_flags          = pg_flags;
    ext->range             = range;
    if (ext->range)
	inc_pager_range_refcount(ext->range);

    if (buffer->a.mem->last_extension) {
	if (buffer->a.mem->last_extension->next_extension)
	    panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		  "sb_add_string_alt_to_mem",
		  "next_extension already set",0);

	buffer->a.mem->last_extension->next_extension = ext;
	inc_plext_in_mem_refcount(buffer->a.mem->
				  last_extension->next_extension);

	free_plext_in_mem(& (buffer->a.mem->last_extension));
    } else {
	if (buffer->a.mem->strings[ptr].next_extension)
	    	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
		      "sb_add_string_alt_to_mem",
		      "next_extension already set",0);

	buffer->a.mem->strings[ptr].next_extension = ext;
	inc_plext_in_mem_refcount(buffer->a.mem->strings[ptr].next_extension);
    }
    buffer->a.mem->last_extension = ext;
   
}

static struct plext_in_mem * malloc_plext_in_mem()
{
    struct plext_in_mem *ret = safe_zero_alloc(sizeof (* ret));

    ret->magic = PLEXT_IN_MEM_magic;
    ret->refcount = 1;

    ret->line_or_other = NULL;
    ret->pg_flags      = 0;
    ret->range         = NULL;

    return ret;
}

static void free_plext_in_mem(mem) 
    struct plext_in_mem **mem;
{
    if (PLEXT_IN_MEM_magic != (*mem)->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"free_plext_in_mem",
	      "Bad magic number",0);

   (*mem)->refcount--;

    if ((*mem)->refcount > 0) {  /* Do not free */
        
        (*mem) = NULL;           /* Refefence count for this pointer is     */
        return;                  /* decrement, so this must have be reset   */
                                         
    }

    if ((*mem)->line_or_other)
	free_string_alt(&((*mem)->line_or_other));
    if ((*mem)->range)
	free_pager_range(&((*mem)->range));
    if ((*mem)->next_extension)
	free_plext_in_mem(& ((*mem)->next_extension));

    (*mem)->magic = 0;          /* Invalidate */
    free(*mem);
    *mem = NULL;
}

static void inc_plext_in_mem_refcount(mem)
     struct plext_in_mem *mem;
{
    if (PLEXT_IN_MEM_magic != mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "inc_plext_in_mem_refcount",
	      "Bad magic number",0);

    mem->refcount++;
}

S_(plext_init_extension plext_init_mem)
static void plext_init_mem P_((struct pager_lineext *ext,
			       struct stringbuffer * stringbuffer));
static void plext_init_mem(ext,stringbuffer)
     struct pager_lineext * ext;
     struct stringbuffer * stringbuffer; /* This is stored by caller */
{
    ext->a.mem = malloc_plext_in_mem();
}

S_(plext_dest_extension plext_dest_mem)
static void plext_dest_mem P_((struct pager_lineext *ext,
			       struct stringbuffer * stringbuffer));
static void plext_dest_mem(ext,stringbuffer)
     struct pager_lineext * ext;
     struct stringbuffer * stringbuffer; /* This is stored by caller */
{
    if (PLEXT_IN_MEM_magic != ext->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"plext_dest_mem",
	      "Bad magic number",0);

    free_plext_in_mem(& (ext->a.mem));
}

S_(plext_walk_alt_extension plext_walk_alt_mem)
static struct string_alt * plext_walk_alt_mem
                 P_((struct pager_lineext *ext,
		     struct pager_lineext **
		     walk_extension,
		     int *pg_flags,
		     struct pager_range **range));

struct lineext_type plext_in_mem = {
    LINEEXT_TYPE_magic,
    
    plext_init_mem,
    plext_dest_mem,
    plext_walk_stringbuffer,
    plext_walk_alt_mem
};

static struct string_alt * plext_walk_alt_mem(ext,walk_extension,pg_flags,
					       range)
     struct pager_lineext *ext;
     struct pager_lineext **walk_extension;
     int *pg_flags;
     struct pager_range **range;
{
    struct stringbuffer  * buffer;
    struct pager_lineext * this_lineext = NULL;	
    struct string_alt    * res = NULL;
        
    if (PAGER_LINEEXT_magic != ext->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "plext_walk_alt_mem",
	      "Bad magic number (pager_lineext)",0);
    
    buffer = ext->stringbuffer;
    
    if (STRINGBUFFER_magic != buffer->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "plext_walk_alt_mem",
	      "Bad magic number (stringbuffer)",0);
    
    if (&sb_in_mem != buffer->buffer_type)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "plext_walk_alt_mem",
	      "Bad buffer type",0);
    
    if (PLEXT_IN_MEM_magic != (*walk_extension)->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_mem",
	      "Bad magic number (plext_in_mem)",0);
    
    this_lineext = *walk_extension;
    *walk_extension = NULL;
    
    if (range &&
	*range != this_lineext->a.mem->range) {
	
	if (*range)
	    free_pager_range(range);
	
	*range = this_lineext->a.mem->range;
	if (*range)
	    inc_pager_range_refcount(*range);
    }
    
    *pg_flags = this_lineext->a.mem->pg_flags;
    
    res = this_lineext->a.mem->line_or_other;
    inc_string_alt_refcount(res);
    
    if (this_lineext->a.mem->next_extension) {
	*walk_extension = malloc_pager_lineext(&plext_in_mem,buffer);

	free_plext_in_mem(& ((*walk_extension)->a.mem));
	
	(*walk_extension)->a.mem = this_lineext->a.mem->next_extension;
	
	inc_plext_in_mem_refcount((*walk_extension)->a.mem);
    }

    free_pager_lineext(&this_lineext);
    
    return res;
}

static struct pager_lineext * sb_mem_copy_extension(buffer,ptr)
     struct stringbuffer *buffer; 
     int ptr;
{
    struct pager_lineext *first_extension = 
	malloc_pager_lineext(&plext_in_mem,buffer);
        
    free_plext_in_mem(& (first_extension->a.mem));

    first_extension->a.mem = buffer->a.mem->strings[ptr].next_extension;
    inc_plext_in_mem_refcount(first_extension->a.mem);

    return first_extension;
}


static void sb_add_lineext_to_mem(buffer,string,pg_flags,range,binding)
     struct stringbuffer      * buffer;
     const struct string      * string;
     int                        pg_flags;
     struct pager_range       * range;
     struct pgrng_bfr_binding * binding;  /* NOT USED */
{
    struct string_alt * strX = new_string_alt_text(string);


    sb_add_string_alt_to_mem(buffer,strX,pg_flags,range);

    free_string_alt(& strX);

    
}

static struct string *sb_walk_lineext_on_mem(buffer,walk_extension,pg_flags,range)
     struct stringbuffer *buffer;
     struct pager_lineext **walk_extension;
     int *pg_flags;
     struct pager_range **range;
{
    struct pager_lineext * this_lineext = NULL;
    struct string * res = NULL;
    
    if (SB_in_mem_magic != buffer->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,"sb_walk_lineext_on_mem",
	      "Bad magic number (sb_in_mem)",0);
    
    if (PLEXT_IN_MEM_magic != (*walk_extension)->a.mem->magic)
	panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_mem",
	      "Bad magic number (plext_in_mem)",0);
    
   this_lineext = *walk_extension;
   *walk_extension = NULL;

   if (range &&
	*range != this_lineext->a.mem->range) {

       if (*range)
	    free_pager_range(range);

       *range = this_lineext->a.mem->range;
       if (*range)
	    inc_pager_range_refcount(*range);
   }

   *pg_flags = this_lineext->a.mem->pg_flags;
   
   { 
       enum string_alt_type  value_type;
       const union string_alt_value alt_value =
	   get_string_alt_value(this_lineext->a.mem->line_or_other,
				&value_type);
       
       switch(value_type) {
	   const struct string  * text;
       case string_alt_none:
	   break;
       case string_alt_text:
	   res = dup_string(alt_value.text);
	   break;
       case string_alt_entity:
	   
	   text = out_entity_text_value(alt_value.entity);
	   
	   if (text) {
	       
	       res = dup_string(text);
	       
	   } else {
	       const uint16 code = out_entity_unicode_value(alt_value.entity);
	       
	       if (UNICODE_BAD_CHAR != code &&
		   charset_have_unicode ==
		   string_type_have_unicode(display_charset,
					    code)) {
		   
		   res = new_string(display_charset);
		   add_unicode_to_string(res,1,&code);
		   
	       } else {
		   
		   int flags2 = 0;
		   
		   const struct string  * refkey =  
		       out_entity_reference_key(alt_value.entity,&flags2);
		   
		   res = dup_string(refkey);
		   *pg_flags |= flags2;
	       }	               
	   }
       }
   }

   if (!res)
       panic("STRINGBUFFER PANIC",__FILE__,__LINE__,
	      "sb_walk_lineext_on_mem",
	     "Unsupported lineext string_alt_type type",0);
   
   
   if (this_lineext->a.mem->next_extension) {
       *walk_extension = 	malloc_pager_lineext(&plext_in_mem,buffer);
       
       free_plext_in_mem(& ((*walk_extension)->a.mem));
       
       (*walk_extension)->a.mem = this_lineext->a.mem->next_extension;
       
       inc_plext_in_mem_refcount((*walk_extension)->a.mem);
   }
   
   free_pager_lineext(&this_lineext);
   
   return res;
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

