static char rcsid[] = "@(#)$Id: streamsched.c,v 2.22 2022/02/20 07:34:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.22 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elm_defs.h"
#include "ss_imp.h"
#include "schedule_time.h"
#include "connection_imp.h"
#ifdef USE_DLOPEN
#include "shared_imp.h"
#endif

DEBUG_VAR(Debug,__FILE__,"net");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* ------------------------------------------------------------------------ */

/* Simple stream implementation */

struct simple_type {
    struct stream_type    *type;
    int                   socket;
};

S_(ss_ReadFromStream ss_ReadFromSocket)
static int ss_ReadFromSocket P_((struct streamsched *ss,
				 int stack_idx,
				 struct Read_Buffer *buffer,
				 int wanted));
static int ss_ReadFromSocket(ss,stack_idx,buffer,wanted)	     
     struct streamsched *ss;
     int stack_idx;
     struct Read_Buffer *buffer;
     int wanted;
{
    int n = 0;
    int err = errno;

    DPRINT(Debug,15,(&Debug,
		     "ss_ReadFromSocket: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].simple ||
	ss->this_stack[stack_idx].simple->type != &SocketStream)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ss_ReadFromSocket",
	      "Bad stack index or stack",0);

    n = ReadFromSocket(ss->this_stack[stack_idx].simple->socket,
		       buffer,wanted);
    
    if (n < 0) {
	err = errno;
	DPRINT(Debug,8,(&Debug,
			"ss_ReadFromSocket: read error %s (errno %d)\n",
			strerror(err),err));
    }

    if (n < 0 && err != EINTR && err != EAGAIN && err != EWOULDBLOCK) {
	ss->error_state = strmcpy(ss->error_state,
 				  strerror(err));
    }

    if (n > 0 && ss->stack_len > 1 && transaction_file) {
	int i;	

	time_t now = 0;
	
	if (((time_t) -1) != time(&now)) {
	    struct tm * zz = localtime(&now);
	    
	    if (!zz)
		goto no_time5;
	
	    fprintf(transaction_file,
		    "%d [%d] %02d:%02d:%02d R<< [len %4d] ",
		    getpid(),
		    ss->this_stack[stack_idx].simple->socket,
		    zz->tm_hour,
		    zz->tm_min,
		    zz->tm_sec,
		    n);
	} else {
	no_time5:
	    
	    fprintf(transaction_file,
		    "%d [%d] R<< [len %4d] ",
		    getpid(),
		    ss->this_stack[stack_idx].simple->socket,
		    n);
	    
	}
	    
	for (i = 0; i < n && i < 13; i++) {
	    size_t x = buffer -> read_len + i;
	    
	    unsigned char c = 
		(unsigned char) buffer -> read_buffer[x];
	    fprintf(transaction_file," %02X",c);
	}
	if (i < n) 
	    fprintf(transaction_file," ...");
	fprintf(transaction_file,"\n");
    }

    return n;
}

S_(ss_WriteToStream ss_WriteToSocket)
static int ss_WriteToSocket P_((struct streamsched *ss,
				int stack_idx,
				struct Write_Buffer *buffer));
static int ss_WriteToSocket(ss,stack_idx,buffer)
     struct streamsched *ss;
     int stack_idx;
     struct Write_Buffer *buffer;
{
    int n;
    int err = 0;

    DPRINT(Debug,15,(&Debug,
		     "ss_WriteToSocket: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].simple ||
	ss->this_stack[stack_idx].simple->type != &SocketStream)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ss_WriteToSocket",
	      "Bad stack index or stack",0);

    
    n = WriteToSocket(ss->this_stack[stack_idx].simple->socket,buffer);

    if (n < 0) {
	err = errno;
	DPRINT(Debug,8,(&Debug,
			"ss_WriteToSocket: read error %s (errno %d)\n",
			strerror(err),err));
    }

    if (n < 0 && err != EINTR && err != EAGAIN && err != EWOULDBLOCK) {
	ss->error_state = strmcpy(ss->error_state,
				  strerror(err));
    }

    if (n > 0 && ss->stack_len > 1 && transaction_file) {
	int i;	

	time_t now = 0;
	
	if (((time_t) -1) != time(&now)) {
	    struct tm * zz = localtime(&now);
	    
	    if (!zz)
		goto no_time6;
	
	    fprintf(transaction_file,
		    "%d [%d] %02d:%02d:%02d R>> [len %4d] ",
		    getpid(),
		    ss->this_stack[stack_idx].simple->socket,
		    zz->tm_hour,
		    zz->tm_min,
		    zz->tm_sec,
		    n);
	} else {
	no_time6:
	    fprintf(transaction_file,
		    "%d [%d] R>> [len %4d] ",
		    getpid(),
		    ss->this_stack[stack_idx].simple->socket,
		    n);
	}
	    
	for (i = 0; i < n && i < 13; i++) {
	    unsigned char c = 
		(unsigned char) buffer -> write_buffer[i];
	    fprintf(transaction_file," %02X",c);
	}
	if (i < n) 
	    fprintf(transaction_file," ...");
	fprintf(transaction_file,"\n");
    }

    return n;
}

S_(ss_FreeThis ss_FreeSocket)
static void ss_FreeSocket P_((struct streamsched *ss,int stack_idx,
			      int badpid
			      ));
static void ss_FreeSocket(ss, stack_idx,badpid)
     struct streamsched *ss;
     int stack_idx;
     int badpid;
{

    DPRINT(Debug,15,(&Debug,
		     "ss_FreeSocket: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].simple ||
	ss->this_stack[stack_idx].simple->type != &SocketStream)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ss_FreeSocket",
	      "Bad stack index or stack",0);
    
    if (ss->this_stack[stack_idx].simple->socket != -1) {
	int r;
	
	DPRINT(Debug,15,(&Debug,
			 "ss_FreeSocket: socket=%d -- clear action\n",
			 ss->this_stack[stack_idx].simple->socket));
	
	/* No longer schedule for this! */
	clear_action0(ss->this_stack[stack_idx].simple->socket,
		      badpid);

	
	r = close(ss->this_stack[stack_idx].simple->socket);

	DPRINT(Debug,15,(&Debug,
			 "ss_FreeSocket: closed fd %d",
			 ss->this_stack[stack_idx].simple->socket));
	if (-1 == r) {
	    int err UNUSED_VAROK = errno;

	    DPRINT(Debug,15,(&Debug,", failed; errno %d (%s)",
			     err,strerror(err)));
	}
	DPRINT(Debug,15,(&Debug,"\n"));
	
	ss->this_stack[stack_idx].simple->socket = -1;
    }

    ss->this_stack[stack_idx].simple->type = NULL;

    free(ss->this_stack[stack_idx].simple);
    ss->this_stack[stack_idx].simple = NULL;
}

S_(ss_StreamAction ss_StreamNoAction)
static enum action_status ss_StreamNoAction P_((struct streamsched *ss, int stack_idx,
						const struct schedule_timelimit * now));
static enum action_status ss_StreamNoAction(ss,stack_idx,now)
     struct streamsched *ss;
     int stack_idx;
     const struct schedule_timelimit * now;
{

    DPRINT(Debug,15,(&Debug,
		     "ss_StreamNoAction: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ss_StreamNoAction",
	  "ss_StreamNoAction called",0);    	

    return action_disable;
}

S_(ss_StreamSchedule ss_SocketSchedule)
static void ss_SocketSchedule P_((struct streamsched *ss, int stack_idx,
			   int previous_needs, 
			   int *needs, int *provides));
static void ss_SocketSchedule(ss,stack_idx,previous_needs,needs,provides)
     struct streamsched *ss; 
     int stack_idx;
     int previous_needs; 
     int *needs; 
     int *provides;
{
    DPRINT(Debug,15,(&Debug,
		     "ss_SocketSchedule: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].simple ||
	ss->this_stack[stack_idx].simple->type != &SocketStream)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ss_SocketSchedule",
	      "Bad stack index or stack",0);


    *needs = previous_needs &
	(SS_read_act|SS_write_act|SS_timeout_act);

    /* We indicate nothing because read_engine and write_engine
       will call action routines ...
    */
    *provides = 0;
}

S_(ss_StreamInfo ss_SocketInfo)
static void ss_SocketInfo P_((struct streamsched *ss,
			      int stack_idx,
			      enum SS_info function,
			      int *int_val,
			      char **str_val,
			      struct string **string_val));
static void ss_SocketInfo(ss,stack_idx,function,int_val,str_val, string_val)
     struct streamsched *ss; 
     int stack_idx;
     enum SS_info function;
     int *int_val; 
     char **str_val;
     struct string **string_val;
{
    DPRINT(Debug,15,(&Debug,
		     "ss_SocketInfo: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].simple ||
	ss->this_stack[stack_idx].simple->type != &SocketStream)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ss_SocketInfo",
	      "Bad stack index or stack",0);

    
    /* We are not intresting any of functions currectly
       so do nothing ...
    */
}

S_(ss_StreamVerifyName ss_SocketVerifyName)
static int ss_SocketVerifyName P_((struct streamsched *ss, int stack_idx,
				   enum SS_name_type name_type,
				   const struct string *required_name,
				   struct string      **returned_name));
static int ss_SocketVerifyName(ss,stack_idx,name_type,required_name,
			       returned_name)
     struct streamsched *ss; 
     int stack_idx;
     enum SS_name_type name_type;
     const struct string *required_name;
     struct string      **returned_name;
{
    DPRINT(Debug,15,(&Debug,
		     "ss_SocketVerifyName: ss=%p, stack_idx=%d\n",
		     ss,stack_idx));

    if (stack_idx < 0 || stack_idx >= ss->stack_len ||
	!ss->this_stack || !ss->this_stack[stack_idx].simple ||
	ss->this_stack[stack_idx].simple->type != &SocketStream)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ss_SocketInfo",
	      "Bad stack index or stack",0);

    
    /* We are not intresting any of functions currectly
       so do nothing ...
    */

    return 0;
}

struct stream_type SocketStream = 
{
    SS_type_magic,	
    "Socket",
    ss_ReadFromSocket,
    ss_WriteToSocket,
    ss_FreeSocket,
    ss_StreamNoAction,
    ss_StreamNoAction,
    ss_SocketSchedule,
    ss_StreamNoAction,
    ss_StreamNoAction,
    ss_SocketInfo,
    ss_SocketVerifyName,
};


/* ----------------------------------------------------------------------- */

static char * return_stream_ident P_((struct streamsched *ss));
static char * return_stream_ident(ss)
     struct streamsched *ss;
{
    static char BUFFER[10];

    if (ss->stack_len < 1 ||
	ss->this_stack[ss->stack_len-1].simple->type != 
	&SocketStream)
	return "*";

    if (ss->this_stack[ss->stack_len-1].simple->socket == -1)
	return "*";

    elm_sfprintf(BUFFER,sizeof BUFFER,
		 FRM("%d"),
		 ss->this_stack[ss->stack_len-1].simple->socket);

    return BUFFER;
}


static void free_error_state P_((struct streamsched *ss));
static void free_error_state(ss)
     struct streamsched *ss;
{
    if (ss->error_state) {
	free(ss->error_state);
	ss->error_state = NULL;
    }    
}

static enum action_status call_user_read_callback P_((struct streamsched *ss,
						      const struct schedule_timelimit * now));
static enum action_status call_user_read_callback(ss,now)
     struct streamsched *ss;
     const struct schedule_timelimit * now;     
{
    
    enum action_status ret = ss->read_act(ss,ss->data,now);

    DPRINT(Debug,20,(&Debug,"call_user_read_callback: read_act gives %d",
		     ret));
    
    switch (ret) {
    case action_disable:
	ss->read_act = ss_noaction_routine;
	DPRINT(Debug,20,(&Debug," (action_disable)"));
	break;
    case action_continue:
	DPRINT(Debug,20,(&Debug," (action_continue)"));
	break;			    			    
    }
    DPRINT(Debug,20,(&Debug,"\n"));
	
    ss->active_flags &= ~SS_read_act;
    return ret;
}

static enum action_status call_user_write_callback P_((struct streamsched *ss,
						       const struct schedule_timelimit * now));
static enum action_status call_user_write_callback(ss,now)
     struct streamsched *ss;
     const struct schedule_timelimit * now;
{
    enum action_status ret = ss->write_act(ss,ss->data,now);

    DPRINT(Debug,20,(&Debug,"call_user_write_callback: write_act gives %d",
		     ret));
    
    switch (ret) {
    case action_disable:
	ss->write_act = ss_noaction_routine;
	DPRINT(Debug,20,(&Debug," (action_disable)"));
	break;
    case action_continue:
	DPRINT(Debug,20,(&Debug," (action_continue)"));
	break;			    			    
    }
    DPRINT(Debug,20,(&Debug,"\n"));
	
    ss->active_flags &= ~SS_write_act;
    return ret;
}

static enum action_status call_user_timeout_callback P_((struct streamsched *ss,
							 const struct schedule_timelimit * now));
static enum action_status call_user_timeout_callback(ss,now)
     struct streamsched *ss;
     const struct schedule_timelimit * now;
{
    enum action_status ret = ss->timeout_act(ss,ss->data,now);

    DPRINT(Debug,20,(&Debug,"call_user_timeout_callback: timeout_act gives %d",
		     ret));

    switch (ret) {
    case action_disable:
	ss->timeout_act = ss_noaction_routine;
	DPRINT(Debug,20,(&Debug," (action_disable)"));
	break;
    case action_continue:
	DPRINT(Debug,20,(&Debug," (action_continue)"));
	break;			    			    
    }
    DPRINT(Debug,20,(&Debug,"\n"));
	
    ss->active_flags &= ~SS_timeout_act;
    return ret;
}

static void xx_will P_((struct streamsched *ss, char *Tag, 
			 int level, char *action));
static void xx_will(ss,Tag, level, action)
     struct streamsched *ss; 
     char *Tag;
     int level;
     char * action;
{
    char *s UNUSED_VAROK =  return_stream_ident(ss);

    DPRINT(Debug,20,(&Debug,
		     " ... [%s] %*s %s will %s\n",
		     s, level*2+3, "",
		     Tag,action));
}

static enum action_status call_read_callback P_((struct streamsched *ss, int level,
						 const struct schedule_timelimit * now));
static enum action_status call_read_callback(ss,level,now)
     struct streamsched *ss; 
     int level;
     const struct schedule_timelimit * now;
{
    enum action_status ret = action_continue;
    
    if (level >= 0) {
	xx_will(ss,(*(ss -> this_stack[level].TYPE))->tag,level,
		"read");
	ret = (*(ss -> this_stack[level].TYPE))-> read_action(ss,level,now);
    } else {
	if (ss->read_act != ss_noaction_routine) {
	    xx_will(ss,"Client",level,"read");
	    ret = call_user_read_callback(ss,now);
	} else
	    ret = action_disable;
    }
    return ret;
}

static enum action_status call_write_callback
             P_((struct streamsched *ss, int level,
		 const struct schedule_timelimit * now));
static enum action_status call_write_callback(ss,level,now)
     struct streamsched *ss; 
     int level;
     const struct schedule_timelimit * now;
{
    enum action_status ret = action_continue;
    
    if (level >= 0) {
	xx_will(ss,(*(ss -> this_stack[level].TYPE))->tag,level,
		"write");
	ret = (*(ss -> this_stack[level].TYPE))-> write_action(ss,level,
							       now);
    } else {
	 if (ss->write_act != ss_noaction_routine) {
	     xx_will(ss,"Client",level,"write");
	     ret = call_user_write_callback(ss,now);
	 } else
	     ret = action_disable;
    }

    return ret;
}

static void after_action P_((struct streamsched *ss,
			     const struct schedule_timelimit * now));

S_(action_routine read_engine)
static enum action_status read_engine P_((int fd,
					  union action_routine_data data,
					  const struct schedule_timelimit * now));
static enum action_status read_engine(fd,data,now)
     int                        fd; 
     union action_routine_data  data;
     const struct schedule_timelimit * now;
{
    struct streamsched *ss = data.stream;
    enum action_status ret = action_disable;

    DPRINT(Debug,19,(&Debug,
		     "read_engine: ss=%p\n",ss));

    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"read_engine",
	      "Bad stream (magic)",0);
    
    
    if (ss->stack_len > 0) {
    
	/* Bottom of stack is never called ... 
	   instead call previous routine ...
	   
	   ss->stack_len-2 aka level can be < 0 
	*/
	ret = call_read_callback(ss,ss->stack_len-2,now);

	after_action(ss,now);

    } else {
	DPRINT(Debug,19,(&Debug,
			 "read_engine: stack disabled, stack len = %d\n",
			 ss->stack_len));
    }
	
    DPRINT(Debug,19,(&Debug,
		     "read_engine=%d %s\n",
		     ret, ret == action_disable ? "(disabling)" : ""));
    
    return ret;
}

S_(action_routine write_engine);
static enum action_status write_engine P_((int                               fd,
					   union action_routine_data         data,
					   const struct schedule_timelimit * now));
static enum action_status write_engine(fd,data,now)
     int                               fd; 
     union action_routine_data         data;
     const struct schedule_timelimit * now;
{
    enum action_status ret = action_disable;
    struct streamsched *ss = data.stream;

    DPRINT(Debug,19,(&Debug,
		     "write_engine: ss=%p\n",ss));

    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"write_engine",
	      "Bad stream (magic)",0);


    if (ss->stack_len > 0) {
	/* Bottom of stack is newer called ... 
	   instead call previous routine ...

	   ss->stack_len-2 aka level can be < 0 
	*/
	ret = call_write_callback(ss,ss->stack_len-2,now);
    
	after_action(ss,now);
    } else {
	DPRINT(Debug,19,(&Debug,
			 "write_engine: stack disabled, stack len = %d\n",
			 ss->stack_len));
    }

	
    DPRINT(Debug,19,(&Debug,
		     "write_engine=%d %s\n",
		     ret, ret == action_disable ? "(disabling)" : ""));

    return ret;
}

S_(action_routine timeout_engine);
static enum action_status timeout_engine P_((int                               fd,
					     union action_routine_data         data,
					     const struct schedule_timelimit * now));
static enum action_status timeout_engine(fd,data,now)
     int                               fd; 
     union action_routine_data         data;
     const struct schedule_timelimit * now;
{
    enum action_status ret = action_disable;
    struct streamsched *ss = data.stream;

    DPRINT(Debug,19,(&Debug,
		     "timeout_engine: ss=%p\n",ss));

    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"timeout_engine",
	      "Bad stream (magic)",0);

    if (ss->stack_len > 0) {
	ret = call_user_timeout_callback(ss,now);

	after_action(ss,now);
    } else {
	DPRINT(Debug,19,(&Debug,
			 "timeout_engine: stack disabled, stack len = %d\n",
			 ss->stack_len));
    }

    DPRINT(Debug,19,(&Debug,
		     "timeout_engine=%d %s\n",
		     ret, ret ==action_disable  ? "(disabling)" : ""));

    return ret;
}

S_(free_action_data_f free_ss_data)
static  void free_ss_data P_((union action_routine_data * data,
			      int fd,
			      int badpid));
static  void free_ss_data(data,fd,badpid)
     union action_routine_data * data;
     int fd;
     int badpid;
{
    struct streamsched *ss = data->stream;

    DPRINT(Debug,19,(&Debug,
		     "free_ss_data: ss=%p\n",ss));

    FreeStreamStack0(&ss,badpid,
		     0 /* No reset, just decrement refcount */);

    data->stream = ss;
}

S_(inc_action_data_refcount_f inc_ss_data_refcount)
static void inc_ss_data_refcount P_((union action_routine_data data));
static void inc_ss_data_refcount(data)
     union action_routine_data data;
{
    struct streamsched *ss = data.stream;

    DPRINT(Debug,19,(&Debug,
		     "inc_ss_data_refcount: ss=%p\n",ss));
    
    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"inc_ss_data_refcount",
	      "Bad stream (magic)",0);

    IncStreamStackRefcount(ss);
}

static void free_stack P_((struct streamsched *ss, int badpid));
static void free_stack(ss,badpid)
     struct streamsched *ss;
     int badpid;
{
    DPRINT(Debug,19,(&Debug,
		     "free_stack: ss=%p; stack len %d\n",
		     ss,
		     ss->stack_len));

    if (ss->free_stack_lock) {  /* Avoid recursive call */
	DPRINT(Debug,19,(&Debug,
			 "free_stack: ss=%p; already in progress\n"));
	
    } else {
	ss->free_stack_lock = 1;
	
	if (ss -> this_stack) {
	    int i;
	    
	    /* NOTE: free_it may start shutdown_act 
	       
	       if badpid != 0, do NOT start shutdown_act 
	    */
	    
	    for (i = 0; i < ss->stack_len; i++) {
		DPRINT(Debug,19,(&Debug,
				 "free_stack: ss=%p; freeing [%d] type %s\n",
				 ss,i,(*(ss -> this_stack[i].TYPE))->tag));
		
		(*(ss -> this_stack[i].TYPE))->free_it(ss,i,badpid);
	    }
	    
	    free(ss -> this_stack);
	    ss -> this_stack = NULL;
	    
	    DPRINT(Debug,19,(&Debug,
			     "free_stack: ss=%p; done\n",
			     ss));
	}
	ss->stack_len       = 0;
	ss->free_stack_lock = 0;
    }
    
}

S_(badpid_action_f badpid_engine)
static enum badpid_status badpid_engine P_((int fd,
					    union action_routine_data data,
					    int badpid));
static enum badpid_status badpid_engine(fd,data,badpid)
     int fd;
     union action_routine_data data;
     int badpid;
{
    struct streamsched *ss = data.stream;

    enum badpid_status ret = badpid_remove;
    
    DPRINT(Debug,19,(&Debug,
		     "badpid_engine: ss=%p\n",ss));
    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"badpid_engine",
	      "Bad stream (magic)",0);
    
    
    if (ss->stack_len > 0) {
		
	ret = ss->badpid_act(ss,ss->data,badpid);
	
	switch (ret) {
	case badpid_disable:
	    DPRINT(Debug,19,(&Debug,
			     "badpid_engine: ... badpid_disable\n"));
	    break;
	case badpid_ignore:
	    DPRINT(Debug,19,(&Debug,
			     "badpid_engine: ... badpid_ignore\n"));
	    
	case badpid_change:
	    DPRINT(Debug,19,(&Debug,
			     "badpid_engine: ... badpid_change\n"));
	    break;
	case badpid_remove:
	    DPRINT(Debug,19,(&Debug,
			     "badpid_engine: ... badpid_remove\n"));


	    DPRINT(Debug,19,(&Debug,
			     "badpid_engine: Freeing stack, stack len = %d\n",
			     ss->stack_len));
    
	    free_stack(ss,badpid);
	    break;
	}
	
    } else {
	DPRINT(Debug,19,(&Debug,
			 "badpid_engine: stack disabled, stack len = %d\n",
			 ss->stack_len));
    }
    
    DPRINT(Debug,19,(&Debug,
		     "badpid_engine=%d\n",ret));
    
    return ret;
}


static int prewait P_((struct streamsched *ss, int *restart,
		       const struct schedule_timelimit * now));

S_(schedule_routine  schedule_engine);
static enum schedule_return schedule_engine P_((int fd,
						union schedule_routine_data
						data,
						const struct schedule_timelimit * now));
static enum schedule_return schedule_engine(fd,data,now)
     int                         fd; 
     union schedule_routine_data data;
     const struct schedule_timelimit * now;
{
    enum schedule_return ret = schedule_done;
    struct streamsched *ss = data.stream;
    
    DPRINT(Debug,19,(&Debug,
		     "schedule_engine: ss=%p\n",ss));

    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"schedule_engine",
	      "Bad stream (magic)",0);

    if (ss->stack_len > 0) {
	int retry = 0;
	int mask  = 0; 
	char * Ident UNUSED_VAROK;
	union action_routine_data action_data;
	
	int old_values;

	action_data.stream = ss;
	
	Ident =  return_stream_ident(ss);
	old_values = ss->active_flags;

	mask = prewait(ss,&retry,now);

	if (ss->timeout_act != ss_noaction_routine) {
	    DPRINT(Debug,20,(&Debug,
			 " ... [%s] stream timeout=%d\n",
			     Ident,ss->timeout_sec));
	    mask |= SS_timeout_act;	
	}
	
	if (ss->this_stack[ss->stack_len-1].simple->type != 
	    &SocketStream) {
	    DPRINT(Debug,1,(&Debug,
			    "bad stream -- length of stream = %d\n",
			    ss->stack_len));

	    DPRINT(Debug,1,(&Debug, 
			    "           -- end of stream %s\n",
			    SS_type_magic == ss->this_stack[ss->stack_len-1].
			    simple->type->magic ? 
			    "valid type" : "not valid"));

	    panic("STREAMSCHED PANIC",__FILE__,__LINE__,"schedule_engine",
		  "No socket on end of stream",0);
	} else if (ss->this_stack[ss->stack_len-1].simple->socket ==  -1)
	    panic("STREAMSCHED PANIC",__FILE__,__LINE__,"schedule_engine",
		  "Socket closed on end of stream",0);
	else
	    change_action2(ss->this_stack[ss->stack_len-1].simple->socket,
			   ss->timeout_sec,
			   (mask & SS_read_act) ? 
			   read_engine : no_action_routine,
			   (mask & SS_write_act) ? 
			   write_engine : no_action_routine,
			   (mask & SS_timeout_act) ? 
			   timeout_engine : no_action_routine,
			   free_ss_data,
			   inc_ss_data_refcount,
			   badpid_engine,
			   action_data);
	
	if (old_values != ss->active_flags) {
	    DPRINT(Debug,20,(&Debug,
			     " ... [%s] active flags changed ... quit waiting\n",
			     Ident));
	    ret = schedule_have_action;
	} else if (retry > 0) {
	    DPRINT(Debug,20,(&Debug,
			     " ... [%s] retrying without waiting\n",
			     Ident));
	    ret = schedule_reconsider;
	} 
	else if (retry < 0) {
	    DPRINT(Debug,20,(&Debug,
			     " ... [%s] Signaling wait ready\n",
			     Ident));
	    ret = schedule_have_action;
	}

    } else {
	DPRINT(Debug,19,(&Debug,
			 "schedule_engine: stack disabled, stack len = %d\n",
			 ss->stack_len));

	ret = schedule_remove;
    }
    

    DPRINT(Debug,19,(&Debug,
		     "schedule_engine=%d\n",ret));
    return ret;
}

static void xx_wants P_((struct streamsched *ss, char *Tag, 
			 int level, int mask));
static void xx_wants(ss,Tag, level, mask)
     struct streamsched *ss; 
     char *Tag;
     int level;
     int mask;
{
    char *s UNUSED_VAROK =  return_stream_ident(ss);

    DPRINT(Debug,20,(&Debug,
		     " ... [%s] %*s %s wants",
		     s, level*2+3, "",
	       Tag));
	   
    if (mask & SS_read_act)
	DPRINT(Debug,20,(&Debug," read"));
    if (mask & SS_write_act)
	DPRINT(Debug,20,(&Debug," write"));
    if (mask & SS_timeout_act)
	DPRINT(Debug,20,(&Debug," timeout"));
    if (mask & SS_setup_act)
	DPRINT(Debug,20,(&Debug," setup"));
    if (mask & SS_shutdown_act)
	DPRINT(Debug,20,(&Debug," shutdown"));
    if (0 == mask)
	DPRINT(Debug,20,(&Debug," NONE"));
    DPRINT(Debug,20,(&Debug,"\n"));
}

static void xx_provides P_((struct streamsched *ss, char *Tag, 
			 int level, int mask));
static void xx_provides(ss,Tag, level, mask)
     struct streamsched *ss; 
     char *Tag;
     int level;
     int mask;
{
    char *s UNUSED_VAROK =  return_stream_ident(ss);

    DPRINT(Debug,20,(&Debug,
	       " ... [%s] %*s %s provides",
	       s, level*2+3, "",
	       Tag));
	   
    if (mask & SS_read_act)
	DPRINT(Debug,20,(&Debug," read"));
    if (mask & SS_write_act)
	DPRINT(Debug,20,(&Debug," write"));
    if (mask & SS_timeout_act)
	DPRINT(Debug,20,(&Debug," timeout"));
    if (mask & SS_setup_act)
	DPRINT(Debug,20,(&Debug," setup"));
    if (mask & SS_shutdown_act)
	DPRINT(Debug,20,(&Debug," shutdown"));
    if (0 == mask)
	DPRINT(Debug,20,(&Debug," NONE"));
    DPRINT(Debug,20,(&Debug,"\n"));
}

static int prewait(ss,restart,now)
     struct streamsched *ss;
     int *restart;
     const struct schedule_timelimit * now;
{
    int return_mask = 0;
    int want_mask = 0;

    int level;
    char * Ident UNUSED_VAROK;

    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"prewait",
	      "Bad stream (magic)",0);
    Ident =  return_stream_ident(ss);

    if (ss_noaction_routine != ss->read_act)
	want_mask |= SS_read_act;
    if (ss_noaction_routine != ss->write_act)
	want_mask |= SS_write_act;
    if (ss_noaction_routine != ss->timeout_act)
	want_mask |= SS_timeout_act;
    
    DPRINT(Debug,20,(&Debug,
		     " ... [%s] --- before action:\n",Ident));
    
    xx_wants(ss,"Client",-1,want_mask);

    for (level =  0; level < ss->stack_len; level++) {
	int old_mask = want_mask;
	int provide_mask;
	int reconsider = 0;
	int first = 1;
	int count = 0;

	do {

	    if (reconsider) {
		count ++;
		
		if (count > 2) {
		    DPRINT(Debug,1,(&Debug,
				    "prewait [%s] recondsidering [%d] delayed!\n",
				    Ident,count));
		    *restart = 1;
		    break;
		}

		DPRINT(Debug,20,(&Debug,
				 " ... [%s] %*s ... reconsidering  [%d]\n",
				 Ident,level*2+3,"",count));
		reconsider = 0;
	    }

	    (*(ss -> this_stack[level].TYPE))->
		schedule_data(ss,
			      level,old_mask,
			      &want_mask,&provide_mask);
	    
	    if (level == ss->stack_len-1)
		return_mask = want_mask;
       
	    xx_wants(ss,(*(ss -> this_stack[level].TYPE))->tag,level,want_mask);
	    xx_provides(ss,(*(ss -> this_stack[level].TYPE))->tag,level,
			provide_mask);

	    if (provide_mask & SS_write_act) {		
		if(call_write_callback(ss,level-1,now)) {
		    *restart = 1;
		    DPRINT(Debug,20,(&Debug,
				     " ... [%s] %*s ... from beginning again\n",
				     Ident,level*2+3,""));
		    break;
		} else
		    reconsider = 1;
	    }

	    if (provide_mask & SS_read_act) {
		if(call_read_callback(ss,level-1,now)) {
		    *restart = 1;
		    DPRINT(Debug,20,(&Debug,
				     " ... [%s] %*s ... from beginning again\n",
				     Ident,level*2+3,""));
		    break;
		} else
		    reconsider = 1;
	    }
	    
	    if (first) {
		switch(want_mask & (SS_setup_act|SS_shutdown_act)) {

		case SS_setup_act:
		    /* Setup is just internal action */
		    
		    xx_will(ss,(*(ss -> this_stack[level].TYPE))->tag,level,
			    "setup");
		    
		    if (!(*(ss -> this_stack[level].TYPE))->
			setup_action(ss,level,now)) {
			*restart = -1;
			DPRINT(Debug,20,(&Debug,
					 " ... [%s] %*s ... from beginning again (quiting)\n",
					 Ident,level*2+3,""));
		    } else 
			reconsider = 1;
		    
		    ss->active_flags &= ~SS_setup_act;		
		    break;

		case SS_shutdown_act:
		    /* Shutdown is just internal action */
		    
		    xx_will(ss,(*(ss -> this_stack[level].TYPE))->tag,level,
			    "shutdown");
		    
		    if (!(*(ss -> this_stack[level].TYPE))->
			shutdown_action(ss,level,now)) {
			*restart = -1;
			DPRINT(Debug,20,(&Debug,
					 " ... [%s] %*s ... from beginning again (quiting)\n",
					 Ident,level*2+3,""));
		    } else 
			reconsider = 1;
		    
		    ss->active_flags &= ~SS_shutdown_act;		
		    break;

		case SS_shutdown_act|SS_setup_act:
		    DPRINT(Debug,1,(&Debug,
				    " ... [%s] %*s ... BOTH setup and shutdown wanted?",
				    Ident,level*2+3,""));
		    panic("STREAMSCHED PANIC",__FILE__,__LINE__,
			  "prewait",
			  "Both setup and shutdown wanted to same stream",0);
		}
		first = 0;
	    }

	} while (reconsider && !ss->error_state);	

	if (ss->error_state) {
	    DPRINT(Debug,20,(&Debug,
			     " ... [%s] Error state: %s\n",Ident,
			     ss->error_state));
	    *restart = 0;
	}
    }
    
    return return_mask;
}

static void after_action(ss,now)
     struct streamsched *ss;
     const struct schedule_timelimit * now;
{
    int i;
    int want_mask;
    char *s UNUSED_VAROK;

    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"after_action",
	      "Bad stream (magic)",0);

    want_mask = ss->active_flags;
    
    s =  return_stream_ident(ss);
    DPRINT(Debug,20,(&Debug,
		     " ... [%s] --- after action:\n",s));

    for (i = ss->stack_len-2; i >= 0;  i--) {	
	int mask;

	(*(ss -> this_stack[i].TYPE))->
	    schedule_data(ss,i,want_mask,&want_mask,&mask);
		
	xx_provides(ss,(*(ss -> this_stack[i].TYPE))->tag,i,mask);
	xx_wants(ss,(*(ss -> this_stack[i].TYPE))->tag,i,want_mask);   

	if (mask & SS_read_act)
	    call_read_callback(ss,i-1,now);
	if (mask & SS_write_act)
	    call_write_callback(ss,i-1,now);	    
    }    
}

static void WaitStreamFor_setup P_((struct streamsched *ss));
static void WaitStreamFor_setup(ss)
     struct streamsched *ss;
{
    union schedule_routine_data data;

    data.stream = ss;
    
    if (ss->stack_len < 1 ||
	ss->this_stack[ss->stack_len-1].simple->type != 
	&SocketStream) {
	DPRINT(Debug,1,(&Debug,
			"bad stream -- length of stream = %d\n",
			ss->stack_len));
	if (ss->stack_len > 0) {
	    DPRINT(Debug,1,(&Debug, 
			    "           -- end of stream %s\n",
			    SS_type_magic == ss->this_stack[ss->stack_len-1].
			    simple->type->magic ? 
			    "valid type" : "not valid"));
	}
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"WaitStreamFor_setup",
	      "No socket on end of stream",0);
    } else if (ss->this_stack[ss->stack_len-1].simple->socket ==  -1)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"WaitStreamFor_setup",
	      "Socket closed on end of stream",0);
    else	    
	set_schedule_action(ss->this_stack[ss->stack_len-1].simple->socket,
			    schedule_engine,
			    no_free_schedule_data,
			    no_inc_schedule_data_refcount,
			    data);
}


static void WaitStreamFor_errcheck P_((struct streamsched *ss,char * Ident,
				       const struct schedule_timelimit * now));
static void WaitStreamFor_errcheck(ss,Ident,now)
     struct streamsched *ss;
     char * Ident;
     const struct schedule_timelimit * now;
{
    if (ss->error_state) {
	
	/* Let's hope that read_callback is available ... */
	if ( ss_noaction_routine == ss->read_act) {
	    DPRINT(Debug,4,(&Debug,
			    " ... [%s] Can't signal on error state: %s\n",
			    Ident,ss->error_state));
	    
	} else {
	    DPRINT(Debug,4,(&Debug,
			    " ... [%s] Signaling on error state: %s\n",
			    Ident,ss->error_state));
	    call_user_read_callback(ss,now);	    
	}
    }
}


void WaitStreamFor(ss,flags)
     struct streamsched *ss; 
     int flags;
{
    DPRINT(Debug,19,(&Debug,
		     "WaitStreamFor: ss=%p\n",
		     ss));
    
    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"WaitStreamFor",
	      "Bad stream (magic)",0);

    DPRINT(Debug,20,(&Debug,
		     "WaitStreamFor: ss=%p, flags=%d  <--- START\n",
		     ss,flags));


    if (ss->stack_len > 0) {
	struct schedule_timelimit current =  NO_schedule_timelimit;
    
	char * Ident = return_stream_ident(ss);
	
	ss->active_flags     = flags;
	
	while (ss->active_flags == flags) {
	    enum wait_for_status r;
	    
	    WaitStreamFor_setup(ss);
	    
	    r = wait_for_any_action_settime(&current);
	    
	    DPRINT(Debug,20,(&Debug,"WaitStreamFor: wait_for_any_action_settime=%d\n",r));
	
	    switch (r) {
		int err UNUSED_VAROK;
	    case wait_for_done:             break;
	    case wait_for_none:
		DPRINT(Debug,20,(&Debug,"WaitStreamFor: wait interrupted\n"));
		break;
	    case wait_for_error:
		err = errno;
		
		DPRINT(Debug,20,(&Debug,"WaitStreamFor:  errno=%d, %s\n",
				 err,strerror(err)));
		break;
	    }
	    
	    WaitStreamFor_errcheck(ss,Ident,&current);
	}

    } else {
	DPRINT(Debug,19,(&Debug,
			 "WaitStreamFor: stack disabled, stack len = %d\n",
			 ss->stack_len));
    }
    
    DPRINT(Debug,20,(&Debug,
		     "WaitStreamFor  <--- END\n"));

    return;
}

int WaitStreamFor_c(ss,flags,cd)
     struct streamsched *ss; 
     int flags;
     struct cancel_data *cd;
{    
    int ret = 0;

    DPRINT(Debug,19,(&Debug,
		     "WaitStreamFor_c: ss=%p\n",
		     ss));
   
    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"WaitStreamFor_c",
	      "Bad stream (magic)",0);

    DPRINT(Debug,20,(&Debug,
		     "WaitStreamFor_c: ss=%p, flags=%d, %s cancel data  <--- START\n",
		     ss,flags,
		     cd ? "have" : "no"));

    if (ss->stack_len > 0) {
	char * Ident = return_stream_ident(ss);
	int canceled = 0;
	struct schedule_timelimit current = NO_schedule_timelimit;
       
	ss->active_flags     = flags;
	
	while (ss->active_flags == flags && !canceled) {
	    enum wait_for_status r;
	    
	    WaitStreamFor_setup(ss);
	    
	    r = wait_for_any_action_settime_c(&current,cd);
	    
	    DPRINT(Debug,20,(&Debug,"WaitStreamFor_c: wait_for_any_action_settime_c=%d\n",r));
	    
	    switch (r) {
		int err;
	    case wait_for_done:                   break;
	    case wait_for_none:
		DPRINT(Debug,20,(&Debug,"WaitStreamFor_c: wait interrupted\n"));
		break;
	    case wait_for_error:
		err = errno;
		
		DPRINT(Debug,20,(&Debug,"WaitStreamFor_c:  errno=%d, %s\n",
				 err,strerror(err)));
		
		if (EINTR == err && cd && is_canceled(cd)) {
		    
		    DPRINT(Debug,20,(&Debug,"WaitStreamFor_c: Wait canceled\n"));
		    
		    canceled = 1;
		}
		break;
	    }
	    
	    WaitStreamFor_errcheck(ss,Ident,&current);
	}
		
	ret = ss->active_flags != flags;
    } else {
	DPRINT(Debug,19,(&Debug,
			 "WaitStreamFor_c: stack disabled, stack len = %d\n",
			 ss->stack_len));
    }
    
    DPRINT(Debug,20,(&Debug,
		     "WaitStreamFor_c=%d  <--- END\n",
		     ret));

    return ret;
}

int ReadFromStream(ss,buffer,wanted)
     struct streamsched *ss;
     struct Read_Buffer *buffer;
     int wanted;
{
    int n = 0;

    DPRINT(Debug,19,(&Debug,
		     "ReadFromStream: ss=%p\n",
		     ss));
    
    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ReadFromStream",
	      "Bad stream (magic)",0);

    if (ss->stack_len > 0) {
    
	n = (*(ss -> this_stack[0].TYPE))->read_from_it(ss,0,buffer,wanted);

	if (n > 0) {
	    
	    if (transaction_file) {
		int i;
		int q = 0;
		char * id = return_stream_ident(ss);

		time_t now = 0;
		struct tm * zz = NULL;
		
		if (((time_t) -1) != time(&now)) {
		    zz = localtime(&now);
	    
		    if (!zz)
			goto no_time7;
		
		    fprintf(transaction_file,
			    "%d [%s] %02d:%02d:%02d %c<< [len %4d] ",
			    getpid(),id,
			    zz->tm_hour,
			    zz->tm_min,
			    zz->tm_sec,
			    ss->stack_len > 1 ? 'T' : '<',
			    n);
		} else {
		no_time7:
		    fprintf(transaction_file,
			    "%d [%s] %c<< [len %4d] ",
			    getpid(),id,
			    ss->stack_len > 1 ? 'T' : '<',
			    n);

		}
		
		for (i = 0; i < n; i++) {
		    unsigned char c = 
			(unsigned char) buffer -> read_buffer[buffer -> read_len 
							      + i];
		    
		    if (isascii(c) && isprint(c) && c != '"') {
			if (!q) {
			    putc(' ',transaction_file);
			    putc('"',transaction_file);
			    q++;
			}
			putc(c,transaction_file);
		    } else {
			if (q) {
			    putc('"',transaction_file);
			    q = 0;
			}
			fprintf(transaction_file," %02X",c);
			
			if (c == 0x0A && i < n-1) {
			    
			    if (zz) {
				fprintf(transaction_file,
					"\n%d [%s] %02d:%02d:%02d %c<< [continue] ",
					getpid(),id,
					zz->tm_hour,
					zz->tm_min,
					zz->tm_sec,
					ss->stack_len > 1 ? 'T' : '<');
			    } else {
				fprintf(transaction_file,
					"\n%d [%s] %c<< [continue] ",
					getpid(),id,
					ss->stack_len > 1 ? 'T' : '<');

			    }
			}
		    }		
		}
		if (q) {
		    putc('"',transaction_file);
		}
		putc('\n',transaction_file);
	    }
	    
	    buffer -> read_len += n;
	    DPRINT(Debug,8,(&Debug,
			    "ReadFromStream: Read %d bytes (%d total)\n",
			    n,buffer->read_len));
	}


    } else {
	DPRINT(Debug,19,(&Debug,
			 "ReadFromStream: stack disabled, stack len = %d\n",
			 ss->stack_len));
    }
    return n;
}

char * RemoveStreamError(ss)
     struct streamsched *ss;
{
    char * ret;

    DPRINT(Debug,19,(&Debug,
		     "RemoveStreamError: ss=%p\n",
		     ss));
    
    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"RemoveStreamError",
	      "Bad stream (magic)",0);

    ret = ss->error_state;
    ss->error_state = NULL; 

    if (ret) {
	DPRINT(Debug,19,(&Debug,
			 "RemoveStreamError=%s\n",ret));
    }
    
    return ret;
}
					    
int WriteToStream(ss,buffer)
     struct streamsched *ss;
     struct Write_Buffer *buffer;
{
    int n = 0;

     DPRINT(Debug,19,(&Debug,
		      "WriteToStream: ss=%p\n",
		      ss));
    
    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"WriteToStream",
	      "Bad stream (magic)",0);

    if (ss->stack_len > 0) {
	n = (*(ss -> this_stack[0].TYPE))->write_to_it(ss,0,buffer);
	
	if (n > 0) {
	    if (transaction_file) {
		int i;
		int q = 0;
		char * id = return_stream_ident(ss);

		time_t now = 0;
		struct tm * zz = NULL;
		
		if (((time_t) -1) != time(&now)) {
		    zz = localtime(&now);
		    
		    if (!zz)
			goto no_time8;
		
		    fprintf(transaction_file,
			    "%d [%s] %02d:%02d:%02d %c>> [len %4d] ",
			    getpid(),id,
			    zz->tm_hour,
			    zz->tm_min,
			    zz->tm_sec,
			    ss->stack_len > 1 ? 'T' : '>',
			    n);
		} else {
		no_time8:
		    fprintf(transaction_file,
			    "%d [%s] %c>> [len %4d] ",
			    getpid(),id,
			    ss->stack_len > 1 ? 'T' : '>',
			    n);		    
		}
		    
		for (i = 0; i < n; i++) {
		    unsigned char c = 
			(unsigned char) buffer->write_buffer[i];
		    
		    if (isascii(c) && isprint(c) && c != '"') {
			if (!q) {
			    putc(' ',transaction_file);
			    putc('"',transaction_file);
			    q++;
			}
			putc(c,transaction_file);
		    } else {
			if (q) {
			    putc('"',transaction_file);
			    q = 0;
			}
			fprintf(transaction_file," %02X",c);
			if (c == 0x0A && i < n-1) {
			    
			    if (zz) {
				fprintf(transaction_file,
					"\n%d [%s] %02d:%02d:%02d %c>> [continue] ",
					getpid(),id,
					zz->tm_hour,
					zz->tm_min,
					zz->tm_sec,
					ss->stack_len > 1 ? 'T' : '>');
			    } else {
				fprintf(transaction_file,
					"\n%d [%s] %c>> [continue] ",
					getpid(),id,
					ss->stack_len > 1 ? 'T' : '>');
			    }
			}
			
		    }
		}
		if (q) {
		    putc('"',transaction_file);
		}
		putc('\n',transaction_file);
	    }
	}
    } else {
	DPRINT(Debug,19,(&Debug,
			 "WriteToStream: stack disabled, stack len = %d\n",
			 ss->stack_len));
    }

    return n;
}
       
static union stream_types alloc_simple P_((int socket));
static union stream_types alloc_simple(socket)
     int socket;
{
    union stream_types ret;

    struct simple_type * S = safe_zero_alloc(sizeof (struct simple_type));

    S->type    = &SocketStream;
    S->socket  = socket;

    ret.simple = S;
    
    return ret;
}


void add_stream_to_head(ss,stream)
     struct streamsched *ss;
     union stream_types stream;
{
    int i;

     DPRINT(Debug,19,(&Debug,
		      "add_stream_to_head: ss=%p\n",
		      ss));

#ifdef USE_DLOPEN
    load_shared_libs1(&use_shared_connect);
#endif


    if (SS_type_magic != (*(stream.TYPE))->magic) {
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"add_stream_to_head",
	      "Bad streamtupe (magic)",0);
    }
	

    DPRINT(Debug,19,(&Debug,
		     "add_stream_to_head: stream type %s\n",
		     (*(stream.TYPE))->tag));
    
    ss->this_stack = safe_array_realloc(ss->this_stack,
					(ss->stack_len+1),
					sizeof (ss->this_stack[0]));

    for (i = ss->stack_len; i > 0; i--)
	ss->this_stack[i] = ss->this_stack[i-1];
    ss->this_stack[0] = stream;
    ss->stack_len++;

    if (transaction_file) {

	char * id = return_stream_ident(ss);
	int i;
	time_t now = 0;
			
	if (((time_t) -1) != time(&now)) {
	    struct tm * zz = localtime(&now);
		    
	    if (!zz)
		goto no_time9;
	
	    fprintf(transaction_file,
		    "%d [%s] %02d:%02d:%02d ---",
		    getpid(),id,
		    zz->tm_hour,
		    zz->tm_min,
		    zz->tm_sec);
	} else {
	no_time9:
	    fprintf(transaction_file,
		    "%d [%s] ---",
		    getpid(),id);

	}

	for (i = 0; i < ss->stack_len; i++)
	    fprintf(transaction_file," %s",
		    (*(ss->this_stack[i].TYPE))->tag);
	fprintf(transaction_file,"\n");
    }
}

int ss_noaction_routine(ss,data,now)
     struct streamsched              * ss; 
     union ss_action_routine_data      data; 
     const struct schedule_timelimit * now;
{
    panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ss_noaction_routine",
	  "ss_noaction_routine called",0);
    return 0;
}

void ss_nofree_action_rt_data(ss,data,badpid)
     struct streamsched *ss;
     union  ss_action_routine_data * data;
     int badpid;
{
    /* Do not free */

    DPRINT(Debug,19,(&Debug,
		     "ss_nofree_action_rt_data: ss=%p, no refcount, real data %p, badpid=%d\n",
		     ss,data->data,badpid));
    
    data->data = NULL;
}

void ss_noinc_action_rt_data_refcount(data)
     union  ss_action_routine_data data;
{
    /* No action */

    DPRINT(Debug,19,(&Debug,
		     "ss_noinc_action_rt_data_refcount: no refcount, real data %p\n",
		     data.data));

}

enum badpid_status ss_nobadpid_routine(ss,data,badpid)
     struct streamsched *ss;
     union  ss_action_routine_data data;
     int badpid;
 {
     /* No action */

    DPRINT(Debug,19,(&Debug,
		     "ss_nobadpid_routine=badpid_disable; ss=%p, badpid=%d\n",
		     ss,badpid));
     
     return badpid_disable /* action_disable */;
}    

enum badpid_status ss_badpid_remove(ss,data,badpid)
     struct streamsched *ss;
     union  ss_action_routine_data data;
     int badpid;
{
    DPRINT(Debug,19,(&Debug,
		     "ss_badpid_remove=badpid_remove; ss=%p, badpid=%d\n",
		     ss,badpid));
    
    return badpid_remove;  /* remove action */
}

struct streamsched * returnSimpleStream(socket)
     int socket;
{
    struct streamsched * ret = safe_zero_alloc(sizeof (struct streamsched));
    union ss_action_routine_data data;

    data.data = NULL;
    
    ret->this_stack  = NULL;
    ret->stack_len   = 0;
    ret->refcount    = 1;
    
    ret->read_act     = ss_noaction_routine;
    ret->write_act    = ss_noaction_routine;
    ret->timeout_act  = ss_noaction_routine;
    ret->free_act     = ss_nofree_action_rt_data;
    ret->inc_refcount = ss_noinc_action_rt_data_refcount;
    ret->badpid_act   =  ss_badpid_remove;
    ret->timeout_sec  = 0;
    ret->error_state  = NULL;
    ret->active_flags = 0;
    ret->data         = data;
    ret->magic        = SS_magic;
    
    ret->free_stack_lock = 0;
    ret->nested_FreeStreamStack0 = 0;

    add_stream_to_head(ret,alloc_simple(socket));

    return ret;
}

void ConfigStream0(ss,read_act,write_act,timeout_act,free_act,
		   inc_refcount,badpid_act,timeout_sec,data,
		   badpid)
     struct streamsched * ss;
     ss_action_routine  * read_act;
     ss_action_routine  * write_act;
     ss_action_routine  * timeout_act;
     ss_free_action_rt_data * free_act;
     ss_inc_action_rt_data_refcount * inc_refcount;
     ss_badpid_routine  * badpid_act;
     int                  timeout_sec;
     union ss_action_routine_data data;
     int badpid;  /* should only be used on reset */
{
    DPRINT(Debug,10,(&Debug,
		     "ConfigStream0: ss=%p, badpid=%d\n",
		     ss,badpid));

    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ConfigStream0",
	      "Bad stream (magic)",0);

    ss->read_act     = read_act;
    ss->write_act    = write_act;
    ss->timeout_act  = timeout_act;
    ss->timeout_sec  = timeout_sec;

    if (ss->data.data != data.data) {
	if (ss->data.data) {
	    DPRINT(Debug,10,(&Debug,
			     "ConfigStream0: freeing old data %p\n",
			     ss->data.data));
	    
	    ss->free_act(ss,&(ss->data),
			 badpid);
	}
		     
	ss->data         = data;
	if (ss->data.data) {
	    DPRINT(Debug,10,(&Debug,
			     "ConfigStream0: Adding new data %p\n",
			     ss->data.data));

	    inc_refcount(ss->data);
	}
    }
    ss->free_act     = free_act;
    ss->inc_refcount = inc_refcount;
    ss->badpid_act   = badpid_act;

}

void ConfigStream2(ss,read_act,write_act,timeout_act,free_act,
		   inc_refcount,badpid_act,timeout_sec,data)
     struct streamsched * ss;
     ss_action_routine  * read_act;
     ss_action_routine  * write_act;
     ss_action_routine  * timeout_act;
     ss_free_action_rt_data * free_act;
     ss_inc_action_rt_data_refcount * inc_refcount;
     ss_badpid_routine  * badpid_act;
     int                  timeout_sec;
     union ss_action_routine_data data;
{
    ConfigStream0(ss,read_act,write_act,timeout_act,free_act,
		  inc_refcount,badpid_act,timeout_sec,data,
		  0 /* Assume no badpid */ );
}


void ConfigStream(ss,read_act,write_act,timeout_act,timeout_sec,data)
     struct streamsched         * ss;
     ss_action_routine          * read_act;
     ss_action_routine          * write_act;
     ss_action_routine          * timeout_act;
     int                          timeout_sec;
     union ss_action_routine_data data;
{
    DPRINT(Debug,10,(&Debug,
		     "ConfigStream: ss=%p\n",
		     ss));


    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"ConfigStream",
	      "Bad stream (magic)",0);

    if (ss_noaction_routine == read_act &&
	ss_noaction_routine == write_act &&
	ss_noaction_routine == timeout_act)
	ConfigStream2(ss,read_act,write_act,timeout_act,
		      ss_nofree_action_rt_data,
		      ss_noinc_action_rt_data_refcount,
		      ss_nobadpid_routine,
		      timeout_sec,data);
    else
	ConfigStream2(ss,read_act,write_act,timeout_act,
		      ss_nofree_action_rt_data,
		      ss_noinc_action_rt_data_refcount,
		      ss_badpid_remove,
		      timeout_sec,data);
}

void FreeStreamStack0(ss,badpid,reset)
     struct streamsched **ss;
     int badpid;
     int reset;
{

    DPRINT(Debug,10,(&Debug,
		     "FreeStreamStack0: *ss=%p, badpid=%d\n",
		     *ss,badpid));

    
    if ((*ss)->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"FreeStreamStack0",
	      "Bad stream (magic)",0);

    if ((*ss)->refcount < 1) 
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"FreeStreamStack0",
	      "Bad refcount",0);

    DPRINT(Debug,10,(&Debug,
		     "FreeStreamStack0: refcount=%d nested_FreeStreamStack0=%d",
		     (*ss)->refcount,(*ss)->nested_FreeStreamStack0));
    
    
    (*ss)->refcount--;

    if ((*ss)->refcount > 0) {   /* Do not free */
	int old_refcount = (*ss)->refcount;

	if (reset) {

	    (*ss)->nested_FreeStreamStack0 ++;
	    
	    DPRINT(Debug,10,(&Debug,", refcount => %d, reset only\n",
			     (*ss)->refcount));
	    free_stack(*ss,badpid);  /* Reset will trigger another call of FreeStreamStack0 */

	    (*ss)->nested_FreeStreamStack0 --;
	    
	} else {
	    DPRINT(Debug,10,(&Debug,", refcount => %d, not freeing\n",
			     (*ss)->refcount));
	}
	
	if ((*ss)->refcount > 0) {
	    int was_refcount = (*ss)->refcount;
	    
	    (*ss) = NULL;           /* Refefence count for this pointer is decremented, */
	                            /* so this must have be reset */

	    DPRINT(Debug,10,(&Debug,
			     "FreeStreamStack0:  refcount=%d => %d, returning\n",
			     old_refcount,was_refcount));
	    
	    return;
	}

	DPRINT(Debug,10,(&Debug,
			 "FreeStreamStack0:  refcount=%d => %d, continuing",
			 old_refcount,(*ss)->refcount));
    } else if ((*ss)->nested_FreeStreamStack0) {
	DPRINT(Debug,10,(&Debug,", not feeing (returning); FreeStreamStack0 already on progress\n",
			 (*ss)->nested_FreeStreamStack0));
 
	(*ss) = NULL;           /* Refefence count for this pointer is decremented, */
 	                        /* so this must have be reset */
	
	return;
    }

    DPRINT(Debug,10,(&Debug,", freeing\n"));

    free_stack(*ss,badpid);

    free_error_state(*ss);
    
    free(*ss);
    *ss = NULL;

    DPRINT(Debug,10,(&Debug,
		     "FreeStreamStack0: done, returning\n")); 
}

void FreeStreamStack(ss)
     struct streamsched **ss;
{
    FreeStreamStack0(ss,
		     0 /* Assume not badpid */,
		     1 /* Assume reset */);
}

void FreeStreamStack2(ss,reset)
     struct streamsched **ss;
     int reset;
{
    FreeStreamStack0(ss,
		     0 /* Assume not badpid */,
		     reset);
}

void IncStreamStackRefcount(ss)
     struct streamsched *ss;
{
    DPRINT(Debug,10,(&Debug,
		     "IncStreamStackRefcount: ss=%p\n",
		     ss));

    if (ss->magic != SS_magic)
	panic("STREAMSCHED PANIC",__FILE__,__LINE__,"IncStreamStackRefcount",
	      "Bad stream (magic)",0);

    DPRINT(Debug,10,(&Debug,
		     "IncStreamStackRefcount: refcount=%d",ss->refcount));

    ss->refcount++;

    DPRINT(Debug,10,(&Debug," => %d\n",
		     ss->refcount));
}


/* Unknown functions does not modify *int_val, *str_val 
   or *string_val
*/

void StreamInfo(ss,function,int_val,str_val,string_val)
     struct streamsched *ss;
     enum SS_info function;
     int *int_val;  
     char **str_val;
     struct string **string_val;
{
    int max  = 0;
    struct string * name = NULL;
    int i;

    DPRINT(Debug,10,(&Debug,
		     "StreamInfo: ss=%p, function=%d\n",
		     ss,function));

    
    switch (function) {
    case SS_ssf:

	DPRINT(Debug,10,(&Debug,
			 "StreamInfo: ... SS_ssf\n"));
	
	/* "Security Strength Factor" */
	/* return maximum of values */
	
	for (i = 0; i < ss->stack_len; i++) {
	    int val = 0;        /* Default is 0 */

	    (*(ss -> this_stack[i].TYPE))->query_it(ss,i,
						    function,
						    &val,NULL,
						    NULL);

	    if (val > max)
		max = val;
	}

	if (int_val)
	    *int_val = max;
	break;


    case SS_verify:

	DPRINT(Debug,10,(&Debug,
			 "StreamInfo: ... SS_verify\n"));

	/* verified name from certificate */

	for (i = 0; i < ss->stack_len; i++) {
	    int val                      = 0;        /* Default is 0 
							0 == not verified
                                                       -1 == verify failure
						        1 == verified (possible name given)

						      */
	    struct string * peer_name   = NULL;
	    
	    (*(ss -> this_stack[i].TYPE))->query_it(ss,i,
						    function,
						    &val,NULL,
						    &peer_name);

	    if (max == 0 && val < 0) {
		max = val;

		if (peer_name)
		    free_string(&peer_name);

	    } else if (val > max && val > 0) {

		max = val;
		if (name)
		    free_string(& name);

		name = peer_name;
	    } else {

		if (peer_name)
		    free_string(&peer_name);

	    }	    
	} 

	if (int_val)
	    *int_val    = max;

	if (string_val)
	    *string_val = name;
	else if (name)
	    free_string(& name);
	    
	break;

    }
}

int StreamVerifyName(ss,name_type,required_name,returned_name)
     struct streamsched *ss;
     enum SS_name_type name_type;
     const struct string *required_name;
     struct string      **returned_name;
{
    int i;
    int ret = 0;

    DPRINT(Debug,10,(&Debug,
		     "StreamVerifyName: ss=%p, name_type=%d",
		     ss,name_type));
    switch(name_type) {
    case SS_peer_cn:      DPRINT(Debug,10,(&Debug," SS_peer_cn")); break;
    case SS_check_host:   DPRINT(Debug,10,(&Debug," SS_check_host")); break;
    }    
    DPRINT(Debug,10,(&Debug,", required_name=%S\n",
		     required_name));

    for (i = 0; i < ss->stack_len; i++) {
	 struct string * peer_name   = NULL;

	 /* Unknown functions does not modify **returned_name 
	    0 = not implemented or verify failed, may set *returned_name
	    1 = name verified
	 */

	 
	 int val =  (*(ss -> this_stack[i].TYPE))->
	     verify_name_it(ss,i,name_type,required_name,
			    &peer_name);

	 DPRINT(Debug,10,(&Debug,
			  "StreamVerifyName: ss=%p, stack_idx=%d verify_name_it=%d",
			  ss,i,val));
	 if (peer_name) {
	     DPRINT(Debug,10,(&Debug,", peer_name=%S",peer_name));
	 }
	 DPRINT(Debug,10,(&Debug,"\n"));

	 if (val > 0 && returned_name && *returned_name)
	     free_string(returned_name);

	 if (peer_name) {
	     if (returned_name) {
		 if (*returned_name)
		     free_string(returned_name);
		 *returned_name = peer_name;
		 peer_name = NULL;
	     } else
		 free_string(&peer_name);
	 }		 

	 if (val > 0) {
	     ret = val;
	     break;
	 }
    }

    DPRINT(Debug,10,(&Debug,
		     "StreamVerifyName=%d",
		     ret));

    if (returned_name && *returned_name) {
	DPRINT(Debug,10,(&Debug,", *returned_name=%S",*returned_name));
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    return ret;
}

int StreamOK(st_size)
     size_t st_size;
{
    if (st_size == sizeof (struct stream_type)) 
	return 1;

    DPRINT(Debug,1,(&Debug,
		    "StreamOK: Size mismatch st_size=%d != %d\n",
		    st_size,sizeof (struct stream_type)));
    
    return 0;
} 

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
