static char rcsid[] = "@(#)$Id: string.c,v 2.25 2023/12/13 16:55:32 hurtta Exp $";

/*************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.25 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ************************************************************************/

/* For LONG_MAX */
#include <limits.h>

#include "elm_defs.h"
#include "s_me.h"
#include "cs_imp.h"
#ifdef USE_DLOPEN
#include "shared_imp.h"
#endif

DEBUG_VAR(Debug,__FILE__,"charset");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}
static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}
static const char *cus2s P_((const unsigned char *str));
static const char *cus2s(str) 
     const unsigned char *str;
{
    return (const char *)str;
}

#define INVALID_MAP(cs) ((cs)->map_info && \
   (cs)->map_info->map_type != (cs)->charset_type)



int verify_string(str)
     struct string *str;
{
     
     if (!str) {
	 DPRINT(Debug,1,(&Debug,"verify_string=0:  FAILURE  str = NULL\n"));
	 return 0;
     }

     if (!str->string_type) {
	 DPRINT(Debug,1,(&Debug,
			 "verify_string=0:  FAILURE  str->string_type = NULL  (str=%p)\n",
			 str));
	  return 0;
     }

     if (CS_charset_magic != str->string_type->magic) {
	 DPRINT(Debug,1,(&Debug,
			 "verify_string=0:  FAILURE  Bad magic  (str=%p)\n",
			 str));
       return 0;
     }

     if (INVALID_MAP(str->string_type)) {
	 DPRINT(Debug,1,(&Debug,
			 "verify_string=0:  FAILURE  Bad map  (str=%p)\n",
			 str));
       return 0;
     }


     return 1;
}

static struct string * malloc_string P_((charset_t set));
static struct string * malloc_string(set) 
     charset_t set;
{
    struct string * res = safe_zero_alloc(sizeof (*res));

    res->magic = STRING_magic;

     if (CS_charset_magic != set->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"malloc_string",
	      "Bad magic number (set)",0);

    if (INVALID_MAP(set)) 
	panic("STRING PANIC",__FILE__,__LINE__,"malloc_string",
	      "Bad map (set)",0);


    /* Initialize associated map if not done already ... */       
    if (set->map_info && !set->map_info->map_initialized)
	set->map_info->map_init_it(set->map_info);

    res->string_type = set;
    res->refcount    = 1;
    
    res->p = safe_zero_alloc(sizeof (struct str_private_data));
    res->p->magic             = CS_str_magic;
    res->p->len               = 0;
    res->p->state             = NULL;
    res->p->private_flag      = 0;     /* Mostly not used */
    res->p->lang              = NULL;

    res->string_type->charset_type->cs_init_it(res);
    return res;
}

/* ------------------------------------------------------------------------- */

struct string * new_string(set)
     charset_t set;
{
    struct string *res;

    DPRINT(Debug,60,(&Debug,"new_string(set=%p '%s')\n",
		     set,
		     set->MIME_name ? set->MIME_name : "<none>"));

    res = malloc_string(set);
    
    DPRINT(Debug,60,(&Debug,"new_string=%p  (type=%p '%s')\n",
		     res,
		     res->string_type->charset_type,
		     res->string_type->charset_type->type_name));

    return res;
}

struct string * new_langstring(set,lang)
     charset_t set; 
     char *lang;
{
    struct string *res;

    DPRINT(Debug,60,(&Debug,"new_langstring(set=%p '%s', lang=%s)\n",
		     set,
		     set->MIME_name ? set->MIME_name : "<none>",
		     lang));

    res = malloc_string(set);
    res->p->lang   = safe_strdup(lang);

    DPRINT(Debug,60,(&Debug,"new_langstring=%p  (type=%p '%s')\n",
		     res,
		     res->string_type->charset_type,
		     res->string_type->charset_type->type_name));

    return res;
}

struct string * new_string2 P_((charset_t set, 
				const unsigned char *data));
struct string * new_string2(set,data)
     charset_t set; 
     const unsigned char *data;
{
    struct string * res;
    int ERRORS = 0;

    DPRINT(Debug,60,(&Debug,
	       "new_string2: set=%p '%s'\n",
	       set,
	       set->MIME_name ? set->MIME_name : "<none>"));
    if (set == display_charset || set == system_charset) {
	DPRINT(Debug,61,(&Debug,
			 "new_string2 -- data=%s\n",data));
    }

    res = malloc_string(set);
    res->string_type->charset_type->
	cs_add_streambytes_to_it(res,
				 strlen(cus2s(data)),
				 data,&ERRORS);

    DPRINT(Debug,60,(&Debug,
		     "new_string2=%p  (type=%p '%s'): len=%d  ERRORS=%d\n",
		     res,
		     res->string_type->charset_type,
		     res->string_type->charset_type->type_name,
		     res->p->len,
		     ERRORS));
    return res;
}

void free_string(str)
     struct string **str;
{
    if (STRING_magic != (*str)->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"free_string",
	      "Bad magic number",0);

    if (CS_charset_magic != (*str)->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"free_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP((*str)->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"free_string",
	      "Bad map (string type)",0);


    DPRINT(Debug,60,(&Debug,"free_string(*str=%p) ('%s'; type=%p '%s')\n",
		     (*str),
		     (*str)->string_type->MIME_name ? 
		     (*str)->string_type->MIME_name : "<none>",
		     (*str)->string_type->charset_type,
		     (*str)->string_type->charset_type->type_name));
    
    if ((*str)->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"free_string",
	      "Bad magic number (str_private_data)",0);

    if ((*str)->p->state && 
	(*str)->p->state->charset != (*str)->string_type)
	panic("STRING PANIC",__FILE__,__LINE__,"free_string",
	      "Bad magic number (state charset)",0);

    if ((*str)->refcount < 1)
	panic("STRING PANIC",__FILE__,__LINE__,"free_string",
	      "Bad refcount",0);

    DPRINT(Debug,60,(&Debug,
		     "free_string(*str=%p): refcount=%d",
		     (*str),(*str)->refcount));
		     
    
    (*str)->refcount--;

    DPRINT(Debug,60,(&Debug," => %d\n",
		     (*str)->refcount));
    
    if ((*str)->refcount > 0) {
	(*str) = NULL;
	
	return;	
    }
    

    
    (*str)->string_type->charset_type->cs_free_it(*str);

    if ((*str)->p->state) {
	free_state_internal(&((*str)->p->state));		
    }

    if ((*str)->p->lang) {
	free((*str)->p->lang);
	(*str)->p->lang = NULL;
    }

    free((*str)->p);
    (*str)->p = NULL;
    (*str)->string_type = NULL;

    (*str)->magic    = 0;    /* Invalidate */
    free(*str);
    *str = NULL;
}

void inc_string_refcount(s)
     struct string *s;
{
    if (STRING_magic != s->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"inc_string_refcount",
	      "Bad magic number",0);

    if (CS_charset_magic != s->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"inc_string_refcount",
	      "Bad magic number (string type)",0);

    DPRINT(Debug,60,(&Debug,"inc_string_refcount(%p) ('%s'; type=%p '%s') refcount=%d",
		     s,
		     s->string_type->MIME_name ? 
		     s->string_type->MIME_name : "<none>",
		     s->string_type->charset_type,
		     s->string_type->charset_type->type_name,
		     s->refcount
		     ));
    
    s->refcount++;
    
    DPRINT(Debug,60,(&Debug," => %d\n",
		     s->refcount));
    
}


const char * get_string_lang(str)
     const struct string *str;
{
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"get_string_lang",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"get_string_lang",
	    "Bad magic number (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"get_string_lang",
	      "Bad magic number (str_private_data)",0);

    return str->p->lang;
}

const char * get_string_MIME_name(str)
     const struct string *str;
{
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"get_string_MIME_name",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"get_string_MIME_name",
	      "Bad magic number (string type)",0);

    return str->string_type->MIME_name;
}

const struct  charset_type * get_string_charset_type(str)
     const struct string *str;
{
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"get_string_charset_type",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"get_string_charset_type",
	      "Bad magic number (string type)",0);

    return str->string_type->charset_type;
}

charset_t get_string_type(str) 
     const struct string *str;
{
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"get_string_type",
	      "Bad magic number",0);

    return str->string_type;
}

int add_streambyte_to_string(str,ch)
     struct string *str; 
     int ch;
{
    int r;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambyte_to_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambyte_to_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambyte_to_string",
	      "Bad map (string type)",0);


    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambyte_to_string",
	      "Bad magic number (str_private_data)",0);

    if (str->p->state && 
	str->p->state->charset != str->string_type)
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambyte_to_string",
	      "Bad magic number (state charset)",0);

    DPRINT(Debug,60,(&Debug,
		     "add_streambyte_to_string: str=%p ('%s'; type=%p '%s')\n",
		     str,	       
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    if (str->string_type == display_charset || 
	str->string_type == system_charset) {
	DPRINT(Debug,61,(&Debug,
			 "add_streambyte_to_string -- ch=%d '%c'\n",ch,ch));    
    } else {
	DPRINT(Debug,61,(&Debug,
			 "add_streambyte_to_string -- ch=%d\n",ch));    
    }


    r = str->string_type->charset_type->cs_add_streambyte_to_it(str,ch);
    
    DPRINT(Debug,60,(&Debug,
		     "add_streambyte_to_string=%d\n",
		     r));

    return r;
}

int add_streambytes_to_string(str,count,data, errors)
     struct string *str; 
     int count; 
     const unsigned char *data;
     int *errors;
{
    int r;

    int ERRORS = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambytes_to_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambytes_to_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambytes_to_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambytes_to_string",
	      "Bad magic number (str_private_data)",0);

    if (str->p->state && 
	str->p->state->charset != str->string_type)
	panic("STRING PANIC",__FILE__,__LINE__,"add_streambytes_to_string",
	      "Bad magic number (state charset)",0);

    DPRINT(Debug,60,(&Debug,
		     "add_streambytes_to_string: str=%p, count=%d, data=%p ('%s'; type=%p '%s')\n",
		     str,count,data,	       
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    if (str->string_type == display_charset || 
	str->string_type == system_charset) {
	DPRINT(Debug,61,(&Debug,
			 "add_streambytes_to_string -- data='%.*s'\n",
			 count,data));    
    } 

    r = str->string_type->charset_type->cs_add_streambytes_to_it(str, count,data, &ERRORS);
    
    if (errors)
	*errors = ERRORS;

    DPRINT(Debug,60,(&Debug,"add_streambytes_to_string=%d ERRORS=%d\n",
		     r,ERRORS));

    return r;
}

static const unsigned char * csUs P_((const char *str));
static const unsigned char * csUs(str)
     const char *str;
{
    return (const unsigned char *)str;
}

static const char * cUss P_((const unsigned char *str));
static const char * cUss(str)
     const unsigned char *str;
{
    return (const char *)str;
}

static void add_name_to_string P_((struct string *str,
				   const unsigned char *ascii));

static void add_name_to_string(str,ascii)
     struct string *str; 
     const unsigned char *ascii;
{
    int l = strlen(cUss(ascii));
    
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_name_to_string",
	      "Bad magic number",0);

    if ( l> 0) { /* Avoid allocing zero length vector */
	int FAILCOUNT = 0;

	int i;

	uint16 * vector = safe_calloc ( (l), sizeof (uint16));
	
	/* Ascii and UNICODE have save codes ... */
	
	for (i = 0; i < l ; i++)
	    /* ascii string is unsigned ... */
	    if (ascii[i] < 128) 
		vector[i] = ascii[i];
	    else
		vector[i] = '?';
    
	str->string_type->charset_type->cs_add_unicodedata_to_it(str,l,vector,
							     &FAILCOUNT);
	free(vector);
    }
}

enum charset_unicode_stat string_type_have_unicode(set,unicode)
     charset_t set;
     int unicode;
{
    enum charset_unicode_stat ret = charset_unicode_bad;
    
    if (CS_charset_magic != set->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_type_have_unicode",
	      "Bad magic number (set,charcode_info)",0);

    if (INVALID_MAP(set))
	panic("STRING PANIC",__FILE__,__LINE__,"string_type_have_unicode",
	      "Bad map (set)",0);

    DPRINT(Debug,60,(&Debug,
		     "string_type_have_unicode(set=%p %s,unicode=%04x)\n",
		     set,
		     set->MIME_name ? set->MIME_name : "",
		     unicode));
    
    if (unicode >= 0 && unicode <= 0xFFFF)
	ret = set->charset_type->cs_it_have_unicode(set,unicode);
    
    DPRINT(Debug,60,(&Debug,
		     "string_type_have_unicode=%d",ret));

    switch(ret) {
    case charset_unicode_bad:     DPRINT(Debug,60,(&Debug," charset_unicode_bad"));     break;
    case charset_unicode_unknown: DPRINT(Debug,60,(&Debug," charset_unicode_unknown")); break;
    case charset_missing_unicode: DPRINT(Debug,60,(&Debug," charset_missing_unicode")); break;
    case charset_have_unicode:    DPRINT(Debug,60,(&Debug," charset_have_unicode"));    break;
    }
    
    DPRINT(Debug,60,(&Debug,"\n"));
    
    return ret;
}

void add_unicode_to_string(str,len,vector)
     struct string *str;
     int len;
     const uint16 *vector;
{
    int FAILCOUNT = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_unicode_to_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"add_unicode_to_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"add_unicode_to_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_unicode_to_string",
	      "Bad magic number (str_private_data)",0);

    if (len < 0)
	panic("STRING PANIC",__FILE__,__LINE__,"add_unicode_to_string",
	      "Bad vector len",0);

    DPRINT(Debug,60,(&Debug,
		     "add_unicode_to_string(%p,len=%d)  ('%s'; type=%p '%s')\n",
		     str,len,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    str->string_type->charset_type->cs_add_unicodedata_to_it(str,len,vector,
							     &FAILCOUNT);
    
    DPRINT(Debug,60,(&Debug,
		     "add_unicode_to_string: len = %d\n",str->p->len));	     
}

static struct string * cat_strings_internal P_((const struct string *str1,
						const struct string *str2,
						int printind,
						int *errors));
static struct string * cat_strings_internal(str1,str2,printind,errors)
     const struct string *str1;
     const struct string *str2;
     int printind;
     int *errors;
{
    struct string *ret = NULL;

    int str1_prop;
    int str2_prop;

    if (STRING_magic != str1->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings_internal",
	      "Bad magic number (str1)",0);
    if (STRING_magic != str2->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings_internal",
	      "Bad magic number (str2)",0);

    str1_prop = charset_properties(str1 -> string_type);
    str2_prop = charset_properties(str2 -> string_type);

    *errors = 0;

    /* Check lengths and compress strings */
    str1->string_type->charset_type->cs_check_length_it(str1);
    str2->string_type->charset_type->cs_check_length_it(str2);

    
    /* Either one is empty, produce other,
       but not if other is not printable and printind is set.
     */
    if (str1->p->len == 0 &&
	(!printind || 0 != (str2_prop & CS_mapping))) {
	ret = malloc_string(str2 -> string_type);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);

	if (str2->p->lang) {
	    ret->p->lang = safe_strdup(str2->p->lang);
	}

    } else if (str2->p->len == 0 &&
	       (!printind || 0 != (str1_prop & CS_mapping))) {
	ret = malloc_string(str1 -> string_type);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);

	if (str1->p->lang) {
	    ret->p->lang = safe_strdup(str1->p->lang);
	}

    } else if (str1 -> string_type == str2 -> string_type) {
	/* 1) Same CHARSET */

	ret = malloc_string(str1 -> string_type);

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);

	if (str1->p->lang && str2->p->lang &&
	    0 == strcmp(str1->p->lang, str2->p->lang)) {
	    ret->p->lang = safe_strdup(str1->p->lang);
	}

    } else if (str1 -> string_type -> charset_type == 
	       str2 -> string_type -> charset_type &&
	       charset_superset_of(str1 -> string_type,str2 -> string_type)) {
	/* 2)  CHARSET of str1 is superset of CHARSET of str2 */

	ret = malloc_string(str1 -> string_type);

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);

	if (str1->p->lang && str2->p->lang &&
	    0 == strcmp(str1->p->lang, str2->p->lang)) {
	    ret->p->lang = safe_strdup(str1->p->lang);
	}

    } else if (str1 -> string_type -> charset_type == 
	       str2 -> string_type -> charset_type &&
	       charset_superset_of(str2 -> string_type,str1 -> string_type)) {
	/* 3)  CHARSET of str2 is superset of CHARSET of str1 */

	ret = malloc_string(str2 -> string_type);

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);
	
	if (str1->p->lang && str2->p->lang &&
	    0 == strcmp(str1->p->lang, str2->p->lang)) {
	    ret->p->lang = safe_strdup(str1->p->lang);
	}

    } else if (printind &&
	       0 != (str1_prop & CS_mapping) &&
	       0 == (str2_prop & CS_mapping) &&
	       str2 -> string_type -> MIME_name) {
	/* Add just str1 and indicate that 2 is not mappable */
	
	ret = malloc_string(str1 -> string_type);

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str1);
	add_name_to_string(ret,csUs("[?"));
	add_name_to_string(ret,csUs(str2 -> string_type -> MIME_name));
	add_name_to_string(ret,csUs("?]"));

	*errors = str2->p->len;

    } else if (printind &&
	       0 != (str2_prop & CS_mapping) &&
	       0 == (str1_prop & CS_mapping) &&
	       str1 -> string_type -> MIME_name) {
	/* Add just str2 and indicate that 1 is not mappable */
	
	ret = malloc_string(str2 -> string_type);

	add_name_to_string(ret,csUs("[?"));
	add_name_to_string(ret,csUs(str1 -> string_type -> MIME_name));
	add_name_to_string(ret,csUs("?]"));

	*errors = str1->p->len;

	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str2);

    } else if (0 == str1->p->len && 0 == str2->p->len) {
	/* Avoid mallocing null vector */
	ret = malloc_string(display_charset);

    } else {
	/* 4) Unrelated CHARSETs -- so we try use display charset or UTF-8
	   instead 
	*/
	int i;
	int l = 0;
	int FAILCOUNT = 0;
	int FOUNDERR  = 0;

 	uint16 * vector;
	charset_t target_set = MIME_name_to_charset("UTF-8",0);

	
	vector = safe_calloc ( (str1->p->len + str2->p->len),
			       sizeof (uint16));
	
	if (!target_set)
	    panic("CHARSET PANIC",__FILE__,__LINE__,"cat_strings_internal",
		  "UTF-8 not found",0);

	if (0 != (str1_prop & CS_mapping) &&
	    0 != (str2_prop & CS_mapping) &&
	    ISO2022_superset_of(str2 -> string_type,str1 -> string_type)) {

		/* CHARSET of str2 is superset of CHARSET of str1 */
	    target_set = str2->string_type;
		
	} else 	if (0 != (str1_prop & CS_mapping) &&
		    0 != (str2_prop & CS_mapping) &&
		    ISO2022_superset_of(str1 -> string_type,
					str2 -> string_type)) {
	    
	    /* CHARSET of str1 is superset of CHARSET of str2 */
	    target_set = str1->string_type;
	    
	} else if (((0 != (charset_properties(display_charset) & CS_mapping) 
		     &&
		     charset_superset_of(display_charset, 
					 str1 -> string_type) 
		     &&
		     charset_superset_of(display_charset, str2 -> string_type))
		    ||
		    (0 != (charset_properties(display_charset) & 
			   CS_universal_set)
		     ))) {
	    
	    /* If display charset is superset of charset of strings
	       use it 
	    */
	    
	    
	    target_set = display_charset;
	}

	if (0 == (charset_properties(target_set) & CS_mapping)) {
	    DPRINT(Debug,5,(&Debug,
		       "cat_strings_internal: selected charset do not have mapping info, using ASCII as target set\n"));
	    target_set = ASCII_SET;
	}

	ret = malloc_string(target_set);

	for (i = 0; i < str1->p->len; i++) {
	    int found = 0;
	    uint16 c = 
		str1->string_type->charset_type->
		cs_give_unicode_from_it(str1,i,&found);
	    vector[l++] = c;
	    if (!found)
		FOUNDERR++;
	}

	for (i = 0; i < str2->p->len; i++) {
	    int found;
	    uint16 c = 
		str2->string_type->charset_type->
		cs_give_unicode_from_it(str2,i,&found);
	    vector[l++] = c;
	    if (!found)
		FOUNDERR++;
	}

	ret->string_type->charset_type->cs_add_unicodedata_to_it(ret,l,vector,
								 &FAILCOUNT);
	free(vector);

	/* Not necessary correct .... */
	*errors = FAILCOUNT;
	if (FOUNDERR > FAILCOUNT)
	    *errors = FOUNDERR; 

	DPRINT(Debug,60,(&Debug,
			 "cat_strings_internal: FOUNDERR=%d FAILCOUNT=%d\n",
			 FOUNDERR,FAILCOUNT));

	if (str1->p->lang && str2->p->lang &&
	    0 == strcmp(str1->p->lang, str2->p->lang)) {
	    ret->p->lang = safe_strdup(str1->p->lang);
	}
    }

    return ret;
}



struct string * cat_strings(str1,str2,printind)
     const struct string *str1; 
     const struct string *str2;
     int printind;
{
    struct string *ret = NULL;
    int ERROR = 0;

    if (STRING_magic != str1->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad magic number (str1)",0);

    if (STRING_magic != str2->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad magic number (str2)",0);


    if (CS_charset_magic != str1->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad magic number (string type)",0);

    if (CS_charset_magic != str2->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str1->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad map (string type)",0);

    if (INVALID_MAP(str2->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad map (string type)",0);


    if (str1->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad magic number (str_private_data)",0);

    if (str2->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cat_strings",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "cat_strings(%p,%p) ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
	       str1,str2,
	       str1->string_type->MIME_name ? 
	       str1->string_type->MIME_name : "<none>",
	       str1->string_type->charset_type,
	       str1->string_type->charset_type->type_name,
	       str1->p->len,
	       str2->string_type->MIME_name ? 
	       str2->string_type->MIME_name : "<none>",
	       str2->string_type->charset_type,
	       str2->string_type->charset_type->type_name,
	       str2->p->len));

    /* Check lengths */
    str1->string_type->charset_type->cs_check_length_it(str1);
    str2->string_type->charset_type->cs_check_length_it(str2);

    ret = cat_strings_internal(str1,str2,printind,&ERROR);

    DPRINT(Debug,60,(&Debug,
		     "cat_strings=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));

    return ret;
}

int string_len(str)
     const struct string *str;
{
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_len",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_len",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"string_len",
	      "Bad map (string type)",0);


    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_len",
	      "Bad magic number (str_private_data)",0);

    /* Check length */
    str->string_type->charset_type->cs_check_length_it(str);

    DPRINT(Debug,60,(&Debug,"string_len(%p)=%d ('%s'; type=%p '%s')\n",
		     str,str->p->len,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    return str->p->len;
}

int string_cmp(str1,str2,unknown_val)
     const struct string *str1; 
     const struct string *str2;
     int unknown_val;
{
    int ret = 0;
    int not_found = 0;
    int i;
    int str1_len;
    int str2_len;
    cs_give_unicode_from_string *str1_get_unicode;
    cs_give_unicode_from_string *str2_get_unicode;

    if (STRING_magic != str1->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad magic number (str1)",0);
    if (STRING_magic != str2->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad magic number (str2)",0);

    if (CS_charset_magic != str1->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad magic number (string type)",0);

    if (CS_charset_magic != str2->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str1->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad map (string type)",0);

    if (INVALID_MAP(str2->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad map (string type)",0);

    if (str1->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad magic number (str_private_data)",0);

    if (str2->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_cmp",
	      "Bad magic number (str_private_data)",0);


    DPRINT(Debug,60,(&Debug,
		     "string_cmp(%p,%p,unknown_val=%d) ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
		     str1,str2,unknown_val,
		     str1->string_type->MIME_name ? 
		     str1->string_type->MIME_name : "<none>",
		     str1->string_type->charset_type,
		     str1->string_type->charset_type->type_name,
		     str1->p->len,
		     str2->string_type->MIME_name ? 
		     str2->string_type->MIME_name : "<none>",
		     str2->string_type->charset_type,
		     str2->string_type->charset_type->type_name,
		     str2->p->len));

    /* Check lengths -- do compressions and so on */
    str1->string_type->charset_type->cs_check_length_it(str1);
    str2->string_type->charset_type->cs_check_length_it(str2);


    str1_len = str1->p->len;
    str2_len = str2->p->len;

    str1_get_unicode = str1->string_type->charset_type->cs_give_unicode_from_it;
    str2_get_unicode = str2->string_type->charset_type->cs_give_unicode_from_it;
    /* Make unicode compare first to make comparision to stable */

    for (i = 0; i < str1_len && i < str2_len; i++) {
	int found1,found2;
	uint16 c1 = str1_get_unicode(str1,i,&found1);
	uint16 c2 = str2_get_unicode(str2,i,&found2);

	if (!found1 || !found2) {
	    ret = 0;
	    not_found = 1;
	    break;
	} else if (c1 < c2) 
	    ret = -1;
	else if (c1 > c2)
	    ret = 1;

	if (ret != 0)	    
	    break;
    }

    DPRINT(Debug,61,(&Debug,
		     "string_cmp: i=%d, str1 len=%d, str2 len=%d, not_found=%d\n",
		     i,str1->p->len,str2->p->len,not_found));

    if (!not_found) {
	if (0 == ret && i < str1->p->len)
	    ret = 1;
	if (0 == ret && i < str2->p->len)
	    ret = -1;
	
	DPRINT(Debug,61,(&Debug,
		   "string_cmp: unicode cmp ret=%d\n",ret)); 
    }

    if (0 == ret && str1 -> string_type == str2 -> string_type) {
	ret = str1 -> string_type->charset_type->cs_cmp_it(str1,str2);
	DPRINT(Debug,61,(&Debug,
			 "string_cmp: cs_cmp_it ret=%d\n",ret)); 
	
	if (ret < 0)
	    ret = -1;
	else if (ret > 0)
	    ret = 1;

    } else if (0 == ret && not_found) {
	DPRINT(Debug,61,(&Debug,"string_cmp: Unable to compare values!\n"));
	ret = unknown_val;
    }

    DPRINT(Debug,60,(&Debug,"string_cmp=%d\n",ret));
    return ret;
}

/* Returns prefix len, 0 on no match and -1 if unknown character 

   Note: because this returns prefix len, 
         empty prefix gives 0 which is same than no match
*/
int string_have_prefix(str,prefix)
     const struct string *str;
     const struct string *prefix;
{
    int ret = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_prefix",
	      "Bad magic number (str)",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_prefix",
	      "Bad magic number (string type) (str)",0);
    
    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_prefix",
	      "Bad map (string type) (str)",0);

    if (STRING_magic != prefix->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_prefix",
	      "Bad magic number (prefix)",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_prefix",
	      "Bad magic number (string type) (prefix)",0);
    
    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_prefix",
	      "Bad map (string type) (prefix)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_prefix",
	      "Bad magic number (str_private_data) (str)",0);

    if (prefix->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_prefix",
	      "Bad magic number (str_private_data)",0);

     DPRINT(Debug,60,(&Debug,
		      "string_have_prefix(%p,%p)  ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
		      str,prefix,
		      str->string_type->MIME_name ? 
		      str->string_type->MIME_name : "<none>",
		      str->string_type->charset_type,
		      str->string_type->charset_type->type_name,
		      str->p->len,
		      prefix->string_type->MIME_name ? 
		      prefix->string_type->MIME_name : "<none>",
		      prefix->string_type->charset_type,
		      prefix->string_type->charset_type->type_name,
		      prefix->p->len));


    /* Check lengths -- do compressions and so on */
    str->string_type->charset_type->cs_check_length_it(str);
    prefix->string_type->charset_type->cs_check_length_it(prefix);

    if (prefix->p->len > str->p->len) {
	DPRINT(Debug,60,(&Debug,
			 "string_have_prefix: prefix len = %d > string len = %d\n",
			 prefix->p->len,
			 str->p->len));      

    } else {
	int i;
	int str_len    = str->p->len;
	int prefix_len = prefix->p->len;
	cs_give_unicode_from_string *str_get_unicode =
	    str->string_type->charset_type->cs_give_unicode_from_it;
	cs_give_unicode_from_string *prefix_get_unicode =
	    prefix->string_type->charset_type->cs_give_unicode_from_it;

	
	for (i = 0; i < prefix_len && i < str_len; i++) {
	    int found1,found2;
	    uint16 c1 = str_get_unicode(str,i,&found1);
	    uint16 c2 = prefix_get_unicode(prefix,i,&found2);

	    if (!found1 | !found2) {
		ret = -1;
		goto fail;
	    }

	    if (c1 != c2) {
		ret = 0;
		goto fail;
	    }
	}

	DPRINT(Debug,60,(&Debug,
			 "string_have_prefix: preix matches, len %d\n",
			 i));
	
	ret = i;
    }

 fail:
    DPRINT(Debug,60,(&Debug,
		     "string_have_prefix=%d\n",ret));
    return ret;
}


struct string * dup_string(str)
     const struct string *str;
{
    struct string *ret;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"dup_string",
	      "Bad magic number",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"dup_string",
	      "Bad magic number (str_private_data)",0);
    
    DPRINT(Debug,60,(&Debug,"dup_string: str=%p ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    ret = malloc_string(str -> string_type);
    ret->string_type->charset_type->cs_add_intdata_to_it(ret,str);

    if (str->p->lang) {
	ret->p->lang = safe_strdup(str->p->lang);
    }


    DPRINT(Debug,60,(&Debug,"dup_string=%p ('%s'; type=%p '%s')\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name));

    return ret;
}

static struct string * convert_string_internal P_((charset_t set,
						   const struct string *str,
						   int printind,
						   int *failcount));

static struct string * convert_string_internal(set,str,printind,failcount)
     charset_t set;
     const struct string *str;
     int printind;
     int *failcount;
{
    struct string *ret = malloc_string(set);

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string_internal",
	      "Bad magic number",0);

    /* Check lengths -- do compressions and so on */
    str->string_type->charset_type->cs_check_length_it(str);


    if (0 == str->p->len) {
	/* 0) Empty string is always convertable
	      because nothing need to be added to result
	 */

	*failcount = 0;

    } else if (set == str->string_type) {
	/* 1) Same charset */
	ret->string_type->charset_type->cs_add_intdata_to_it(ret,str);

	if (str->p->lang) {
	    ret->p->lang = safe_strdup(str->p->lang);
	}

	*failcount = 0;

    } else if (printind &&
	       0 != (charset_properties(set) & CS_mapping) &&
	       0 == (charset_properties(str -> string_type) & CS_mapping) &&
	       str -> string_type -> MIME_name) {
	/* Just indicate that str is not mappable */
	
	add_name_to_string(ret,csUs("[?"));
	add_name_to_string(ret,csUs(str -> string_type -> MIME_name));
	add_name_to_string(ret,csUs("?]"));

	*failcount = str->p->len; 

    } else {
	/* 2) Different charset */
	int i;
	int l = 0;
       
	uint16 * vector;
	int FAILCOUNT = 0;
	
	vector = safe_calloc ( (str->p->len), sizeof (uint16));
	*failcount = 0;

	for (i = 0; i < str->p->len; i++) {
	    int found;
	    uint16 c = 
		str->string_type->charset_type->
		cs_give_unicode_from_it(str,i,&found);
	    vector[l++] = c;

	    if (!found)
		(*failcount)++;
	}



	ret->string_type->charset_type->cs_add_unicodedata_to_it(ret,l,vector,
								 &FAILCOUNT);

	DPRINT(Debug,61,(&Debug,
			 "convert_string_internal: failcount from cs_add_unicodedata_to_it: %d\n",
			 FAILCOUNT));

	/* TODO: What is correct ??? */
	if (FAILCOUNT > (*failcount))
	    (*failcount) = FAILCOUNT;

	free(vector);

	if (str->p->lang) {
	    ret->p->lang = safe_strdup(str->p->lang);
	}

    }

    return ret;
}

struct string * convert_string(set,str,printind)
     charset_t set;
     const struct string *str;
     int printind;
{
    struct string *ret = NULL;
    int FAILCOUNT = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string",
	      "Bad map (string type)",0);
	
    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "convert_string(set=%p '%s',str=%p)  ('%s', type=%p '%s', len=%d)\n",
		     set,
		     set->MIME_name ? set->MIME_name : "<none>",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
    
    ret = convert_string_internal(set,str,printind,&FAILCOUNT);
    DPRINT(Debug,60,(&Debug,
		     "convert_string: failcount = %d\n",
		     FAILCOUNT));
    
    DPRINT(Debug,60,(&Debug,
		     "convert_string=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));

    return ret;
}

struct string * convert_string2(set,str,failcount)
     charset_t set;
     const struct string *str;
     int *failcount; 
{
    struct string *ret = NULL;
    int FAILCOUNT = 0;
    
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string2",
	      "Bad magic number",0);


    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string2",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string2",
	      "Bad map (string type)",0);
	
    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"convert_string2",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "convert_string2(set=%p '%s',str=%p)  ('%s', type=%p '%s', len=%d)\n",
		     set,
		     set->MIME_name ? set->MIME_name : "<none>",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
    
    if (!failcount)
	failcount = &FAILCOUNT;

    ret = convert_string_internal(set,str,0,failcount);

    DPRINT(Debug,60,(&Debug,
		     "convert_string2: failcount = %d\n",
		     *failcount));
    
    DPRINT(Debug,60,(&Debug,
		     "convert_string2=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));

    return ret;
}

unsigned char *stream_from_string(str, printable,terminal)
     const struct string *str;
     int printable;
     screen_info_p terminal; 
{
    int reslen;
    unsigned char * ret;
    char *ret0 = NULL;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"stream_from_string",
	      "Bad magic number",0);
    
    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"stream_from_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"stream_from_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"stream_from_string",
	      "Bad magic number (str_private_data)",0);
    
    DPRINT(Debug,60,(&Debug,"stream_from_string: ret=%p ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    if (terminal && str->string_type->iso2022_info) 
	ret0 = terminal_charset_post_init(terminal,
					  str->string_type->iso2022_info);


    /* Catenate order is ret == ret0 + ret */
    	    
    ret = str->string_type->charset_type->cs_stream_from_it(str,
							    printable,
							    terminal,
							    &reslen);
    DPRINT(Debug,60,(&Debug,
		     "stream_from_string -- reslen=%d\n",reslen));

    if (str->string_type == display_charset) {
	DPRINT(Debug,61,(&Debug,
			 "stream_from_string -- ret='%s'%s\n",
			 ret,
			 ret0 ? " (without post init)" : ""));
    }


    if (ret0) {
	int i;
	DPRINT(Debug,8,(&Debug,"Post init terminal charset %s (",
			str->string_type->MIME_name ? 
			str->string_type->MIME_name : 
			"<no MIME name>"));
	for (i = 0; ret0[i]; i++) {
	    DPRINT(Debug,8,(&Debug,"%s%02X",
			    i ? " " : "", ret0[i]));
	}
	DPRINT(Debug,8,(&Debug,")\n"));

	/* Catenate order is ret == ret0 + ret */

	ret0 = strmcat(ret0,us2s(ret));
	free(ret);
	ret = s2us(ret0);
    }

    DPRINT(Debug,60,(&Debug,"stream_from_string=%p\n",
	       ret));

    return ret;
}

/* result is malloced */
void bytestream_from_string(str,res,reslen)
     const struct string *str;
     char **res;
     int *reslen;
{
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"bytestream_from_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"bytestream_from_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"bytestream_from_string",
	      "Bad map (string type)",0);
	
    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"bytestream_from_string",
	      "Bad magic number (str_private_data)",0);
    
    DPRINT(Debug,60,(&Debug,
		     "bytestream_from_string: str=%p ('%s'; type=%p '%s', len=%d) reslen=%p\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len,
		     reslen));


    *res = us2s(str->string_type->charset_type->cs_stream_from_it(str,0,NULL,
								  reslen));

    DPRINT(Debug,60,(&Debug,
		     "bytestream_from_string: *res=%p, *reslen=%d\n",
		     *res,*reslen));
    
}


unsigned char *streamclip_from_string(str,pos,len,terminal,printable_len)
     const struct string *str;
     int *pos; 
     int len; 
     screen_info_p terminal;   
     struct cs_printable_len *printable_len;
{
    int p UNUSED_VAROK = *pos;
    unsigned char * ret;
    char *ret0 = NULL;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"streamclip_from_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"streamclip_from_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"streamclip_from_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"streamclip_from_string",
	      "Bad magic number (str_private_data)",0);
	
    if (printable_len && !terminal)
	panic("STRING PANIC",__FILE__,__LINE__,"streamclip_from_string",
	      "printable_len argument requires terminal argument",0);
	
    if (printable_len && 
	0 == ( CS_printable_len & str->string_type->charset_type->cs_it_properties(str->string_type)))
	panic("STRING PANIC",__FILE__,__LINE__,"streamclip_from_string",
	      "Charset does not support printable_len argument",0);


    if (terminal && str->string_type->iso2022_info) 
	ret0 = terminal_charset_post_init(terminal,
					  str->string_type->iso2022_info);

    /* Catenate order is ret == ret0 + ret */
     
    DPRINT(Debug,60,(&Debug,
		     "streamclip_from_string(%p,%d,%d)    ('%s'; type=%p '%s')\n",
		     str,p,len,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    
    ret = 
	str->string_type->charset_type->cs_streamclip_from_it(str,pos,len,
							      terminal,
							      printable_len);

    if (str->string_type == display_charset) {
	DPRINT(Debug,61,(&Debug,
			 "streamclip_from_string -- ret='%s'%s\n",
			 ret,
			 ret0 ? " (without post init)" : ""));
    }

    if (ret0) {
	int i;
	DPRINT(Debug,8,(&Debug,
			"Post init terminal charset %s (",
			str->string_type->MIME_name ? 
			str->string_type->MIME_name : 
			"<no MIME name>"));
	for (i = 0; ret0[i]; i++) {
	    DPRINT(Debug,8,(&Debug,"%s%02X",
			    i ? " " : "", ret0[i]));
	}
	DPRINT(Debug,8,(&Debug,")\n"));

	/* Catenate order is ret == ret0 + ret */

	ret0 = strmcat(ret0,us2s(ret));
	free(ret);
	ret = s2us(ret0);
    }
    
    DPRINT(Debug,60,(&Debug,
		     "streamclip_from_string [%d(..%d)] =%p ('%s'; type=%p '%s')",
		     p,*pos,ret,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    
    if (printable_len) {
	DPRINT(Debug,60,(&Debug,
			 " printable_len: max_len=%d ret_len=%d",
			 printable_len->max_len,
			 printable_len->ret_len));
    }
    DPRINT(Debug,60,(&Debug,"\n"));

    return ret;
}

int estimate_clip_string(str,pos,len,terminal,printable_len)
     const struct string *str;
     int pos; 
     int len;      /* UPPER LIMIT */
     screen_info_p terminal;
     struct cs_printable_len *printable_len;
{
    int ret;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"estimate_clip_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"estimate_clip_string",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"estimate_clip_string",
	      "Bad map (string type)",0);


    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"estimate_clip_string",
	      "Bad magic number (str_private_data)",0);


    if (0 == ( CS_printable_len & 
	       str->string_type->charset_type->cs_it_properties(str->string_type)))
	panic("STRING PANIC",__FILE__,__LINE__,"estimate_clip_string",
	      "Charset does not support function",0);

    if (!printable_len || !terminal)
	panic("STRING PANIC",__FILE__,__LINE__,"estimate_clip_string",
	      "printable_len and terminal required",0);

    
    ret = 
	str->string_type->charset_type->cs_estimate_clip_it(str,pos,len,
							    terminal,
							    printable_len);

    
    DPRINT(Debug,60,(&Debug,
		     "estimate_clip_it(%p,%d,%d,...) =%d ('%s'; type=%p '%s')",
		     str,pos,len,ret,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    DPRINT(Debug,60,(&Debug,
		     " printable_len: max_len=%d ret_len=%d\n",
		     printable_len->max_len,
		     printable_len->ret_len));
    
    return  ret;
}

struct string *clip_from_string(str,pos,len)
     const struct string *str;
     int *pos; 
     int len;
{
    struct string *ret = malloc_string(str->string_type);

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"clip_from_string",
	      "Bad magic number",0);
    
    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"clip_from_string",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "clip_from_string(%p,%d,%d)  ('%s'; type=%p '%s')\n",
		     str,*pos,len,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));

    /* It is assumed that compressions are done
       .. ie string_length() function
       is called 
    */

    str->string_type->charset_type->cs_clip_from_it(ret,str,pos,len);

    DPRINT(Debug,60,(&Debug,
		     "clip_from_string=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));
    
    return ret;
}

struct string *ascify_string(str)
     const struct string *str;
{
    charset_t ascii_ptr = MIME_name_to_charset("US-ASCII",0);
    struct string *ret;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"ascify_string",
	      "Bad magic number",0);

    if (!ascii_ptr)
	panic("STRING PANIC",__FILE__,__LINE__,"ascify_string",
	      "US-ASCII not found",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"ascify_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))
	panic("STRING PANIC",__FILE__,__LINE__,"ascify_string",
	      "Bad map (string type)",0);


    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"ascify_string",
	      "Bad magic number (str_private_data)",0);

        DPRINT(Debug,60,(&Debug,
			 "ascify_string(%p) ('%s'; type=%p '%s', len=%d)\n",
			 str,
			 str->string_type->MIME_name ? 
			 str->string_type->MIME_name : "<none>",
			 str->string_type->charset_type,
			 str->string_type->charset_type->type_name,
			 str->p->len));

    /* Check lengths -- do compressions and so on */
    str->string_type->charset_type->cs_check_length_it(str);

    if (0 == str->p->len) {
	/*  Empty string is always convertable
	    because nothing need to be added to result
	    
	    This avoids allocating empty vector
	*/

	ret = malloc_string(ascii_ptr);

    } else {
	int can_ascii = 
	    str->string_type->charset_type->cs_can_ascii_it(str);

	if (can_ascii) {
	    int i;
	    int l = 0;
	    uint16 * vector = safe_calloc ( (str->p->len), sizeof (uint16));
	    int FAILCOUNT = 0;
	    
	    ret = malloc_string(ascii_ptr);
	    
	    
	    for (i = 0; i < str->p->len; i++) {
		int found;
		uint16 c = 
		    str->string_type->charset_type->
		    cs_give_unicode_from_it(str,i,&found);
		vector[l++] = c;
	    }
	    
	    ret->string_type->charset_type->cs_add_unicodedata_to_it(ret,l,vector,
								     &FAILCOUNT);
	    free(vector);
	} else {
	    ret = malloc_string(str -> string_type);
	    ret->string_type->charset_type->cs_add_intdata_to_it(ret,str);   
	}
    }

    if (str->p->lang) {
	ret->p->lang = safe_strdup(str->p->lang);
    }

    DPRINT(Debug,60,(&Debug,
		     "ascify_string=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));

    return ret;
}

int can_ascii_string(str)
     const struct string *str;
{
    int ret;
    
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"can_ascii_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"can_ascii_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"can_ascii_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"can_ascii_string",
	      "Bad magic number (str_private_data)",0);
    
    ret= str->string_type->charset_type->cs_can_ascii_it(str);
    
    DPRINT(Debug,60,(&Debug,
		     "can_ascii_string(%p)=%d ('%s'; type=%p '%s', len=%d)\n",
		     str,ret,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
    
    return ret;
}

void add_ascii_to_string(str,ascii)
     struct string *str; 
     const unsigned char *ascii;
{

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_ascii_to_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"add_ascii_to_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"add_ascii_to_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_ascii_to_string",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "add_ascii_to_string(%p)  ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    
    DPRINT(Debug,61,(&Debug,
		     "add_ascii_to_string -- ascii=%s\n",ascii));

    add_name_to_string(str,ascii);

    DPRINT(Debug,60,(&Debug,
		     "add_ascii_to_string: len = %d\n",str->p->len));	   
}

void fill_ascii_to_string(str,count,ascii)
     struct string *str; 
     int count; 
     unsigned int ascii;
{
    int i;
    uint16 * vector;
    int FAILCOUNT = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
	      "Bad map (string type)",0);

    if (count < 1)
	panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
	      "bad count",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
	      "Bad magic number (str_private_data)",0);

    if (str->p->state) {
	if (str->p->state->charset != str->string_type)
	    panic("STRING PANIC",__FILE__,__LINE__,"fill_ascii_to_string",
		  "Bad magic number (state charset)",0);
	free_state_internal(&(str->p->state));
    }

    vector = safe_calloc ( count, sizeof (uint16));

    /* Ascii and UNICODE have same codes ... */
    
    for (i = 0; i < count ; i++)
	/* ascii is unsigned ... */
	if (ascii < 128) 
	    vector[i] = ascii;
	else
	    vector[i] = '?';  /* XXX */


    str->string_type->charset_type->cs_add_unicodedata_to_it(str,count,vector,
							     &FAILCOUNT);
    free(vector);

    DPRINT(Debug,60,(&Debug,
		     "fill_ascii_to_string(%p,%d,%d)  ('%s'; type=%p '%s'): len=%d\n",
		     str,count,ascii,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
}

struct string *collapse_spaces_from_string(str)
     const struct string *str;
{
    struct  charset_type  * cstype;

    struct string *ret;
    int i, j = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"collapse_spaces_from_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"collapse_spaces_from_string",
	      "Bad magic number (string type)",0);

    cstype = str->string_type->charset_type;

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"collapse_spaces_from_string",
	      "Bad magic number (str_private_data)",0);


    DPRINT(Debug,60,(&Debug,
		     "collapse_spaces_from_string(%p) ('%s'; type=%p '%s', len=%d)\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     cstype,
		     cstype->type_name,
		     str->p->len));


    /* Check lengths -- do compressions and so on */
    str->string_type->charset_type->cs_check_length_it(str);

    ret = malloc_string(str->string_type);

    for (i = 0; i < str->p->len; i = j) {

	int is_space = 0;
	int POS;
	int len;

	/* Scan to next space */
	for (j = i; j < str->p->len; j++) {

	    int found;
	    
	    uint16 c = cstype->cs_give_unicode_from_it(str,j,&found);
	    	    
	    if (!found) {
		DPRINT(Debug,61,(&Debug,
				 "collapse_spaces_from_string -- failing at %d -- no unicode\n",
				 i));
	    } else
		is_space = unicode_ch(c,UOP_space);

	    if (is_space)
		break;

	}

	POS = i;
	len = j-i;	
	
	if (len > 0) {
	    struct string *tmp = malloc_string(str->string_type);

	    DPRINT(Debug,61,(&Debug,
			     "collapse_spaces_from_string: clipping no space from %d -- len %d\n",
			     POS,len));
	    
	    cstype->cs_clip_from_it(tmp,str,&POS,len);
	    cstype->cs_add_intdata_to_it(ret,tmp);
	    free_string(&tmp);
	    
	}

	/* Skip over space */
	if (is_space) {
	    uint16 SPACE = 0x0020;
	    int FAIL = 0;

	    cstype->cs_add_unicodedata_to_it(ret,1,&SPACE,&FAIL);

	    for (; j < str->p->len; j++) {
		
		int found;
		
		uint16 c = cstype->cs_give_unicode_from_it(str,j,&found);
		
		if (!found) {
		    DPRINT(Debug,61,(&Debug,
				 "collapse_spaces_from_string -- failing at %d -- no unicode\n",
				     j));
		} else
		    is_space = unicode_ch(c,UOP_space);
		
		if (!is_space)
		    break;
	    }			       		
	}
    }


    DPRINT(Debug,60,(&Debug,
		     "collapse_spaces_from_string=%p  ('%s'; type=%p '%s'): len=%d\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len));
    
    return ret;
}

struct string *skip_ascii_head_from_string(str,ascii,ignore_case)
     const struct string *str;
     const unsigned char *ascii;
     int ignore_case;
{
    struct string *ret = malloc_string(str->string_type);
    int X = 0;
    int x UNUSED_VAROK;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"skip_ascii_head_from_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"skip_ascii_head_from_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"skip_ascii_head_from_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"skip_ascii_head_from_string",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "skip_ascii_head_from_string(%p) ('%s'; type=%p '%s', len=%d)\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));
    DPRINT(Debug,60,(&Debug,
		     "skip_ascii_head_from_string -- ascii=%s\n",ascii));
    
    /* Check lengths -- do compressions and so on */
    str->string_type->charset_type->cs_check_length_it(str);

    while(1) {
	int i;
	const unsigned char * p = ascii;

	DPRINT(Debug,61,(&Debug,
			 "skip_ascii_head_from_string -- starting pos %d\n",
			 X));


	for (i = X;  
	     /* ascii string is unsigned ... */
	     *p && *p < 128 && i < str->p->len; 
	     i++, p++) {

	    int found;
	    uint16 c =
		str->string_type->charset_type->
		cs_give_unicode_from_it(str,i,&found);
	    unsigned char c1 = *p;

	    if (!found) {
		DPRINT(Debug,61,(&Debug,
				 "skip_ascii_head_from_string -- failing at %d -- no unicode\n",
				 i));
		break;
	    }
	    if (ignore_case) {
		c = unicode_ch(c,UOP_lowercase);
		if (!c) {	
		    DPRINT(Debug,61,(&Debug,
				     "skip_ascii_head_from_string -- failing at %d -- no lowercasing\n",
				     i));
		    break;
		}
	    }

	    if (ignore_case && c1 >= 0x41 && c1 <= 0x5A) {
	        c1 = ( c1 - 0x41) + 0x61;
	    }

	    if (c1 != c) {
		DPRINT(Debug,61,(&Debug,
				 "skip_ascii_head_from_string -- failing at %d -- mismatch %02X (%c) <> %04X\n",
				 i,c1,c1,c));
		break;
	    }
	}
	if (*p) {
	    DPRINT(Debug,61,(&Debug,
			     "skip_ascii_head_from_string -- hit end at %d\n",
			     i));
	    break;
	}
       
	/* Try next sequence */
	X = i;
	DPRINT(Debug,61,(&Debug,
			 "skip_ascii_head_from_string --  got %d\n",
			 X));
    }	    

    x = X;
    str->string_type->charset_type->cs_clip_from_it(ret,str,&X,str->p->len);

    DPRINT(Debug,60,(&Debug,
		     "skip_ascii_head_from_string=%p  ('%s'; type=%p '%s'): len=%d (clip=%d..%d)\n",
		     ret,
		     ret->string_type->MIME_name ? 
		     ret->string_type->MIME_name : "<none>",
		     ret->string_type->charset_type,
		     ret->string_type->charset_type->type_name,
		     ret->p->len,x,X));

    return ret;
}

/* WARNING:  get_word_from_string does everything via Unicode
 *           may lose information on some situations
 */
             
int get_word_from_string(str,word,pos,flags,ascii_delim, found_delim)
     const struct string *str;
     struct string **word;
     int *pos;
     int flags;
     const unsigned char *ascii_delim;
     uint16 *found_delim;
{
    int lowercase = flags & GWF_lowercase;
    int trim      = flags & GWF_trim_space;
    
    int X = *pos;
    int last_X;
    int X1 UNUSED_VAROK;

    uint16 *data = NULL;
    int dx = 0;
    int ret = 0;
    int FAILCOUNT = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"get_word_from_string",
	      "Bad magic number",0);


    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"get_word_from_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"get_word_from_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"get_word_from_string",
	      "Bad magic number (str_private_data)",0);

    *word = malloc_string(str->string_type);

    DPRINT(Debug,60,(&Debug,
		     "get_word_from_string(%p) ('%s'; type=%p '%s', len=%d), *pos=%d\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len,
		     *pos));

    if (found_delim)
	*found_delim = 0;

    if (0 == *pos) {
	/* Check lengths -- do compressions and so on */
	str->string_type->charset_type->cs_check_length_it(str);
    }

    if (str->p->len < 1)
	goto fail;

    /* SKIP delim */
    while (X < str->p->len) {
	int found;
	int sp;
       
	uint16 c = str->string_type->charset_type->
	    cs_give_unicode_from_it(str,X,&found);

	/* ascii string is unsigned ... */
	const unsigned char *y = NULL;

	if (!found) {
	    DPRINT(Debug,57,(&Debug,
			     "get_word_from_string -- failing at %d -- no unicode\n",
			     X));

	    break;
	}

	/* Is space?    (returns 0 or original character) */
	sp = unicode_ch(c,UOP_space);
	if (sp) {
	    if (trim) {
		X++;
		continue;
	    }
		
	    c = 0x0020;    /* Treat all unicode space characters as ascii space */
	}

	for (y = ascii_delim; *y; y++)
	    if (*y == c || *y == sp)
		break;
	    
	/* Delim not found ? */
	if (! *y)
	    break;

	X++;
    }

    X1 = X;    
    data = safe_calloc(str->p->len, sizeof (data[0]));
    dx = 0;

    last_X = 0;
    /* Read first word */
    while (X < str->p->len) {
	int found;
	uint16 c = str->string_type->charset_type->
	    cs_give_unicode_from_it(str,X,&found);

	int sp;
	/* ascii string is unsigned ... */
	const unsigned char *y = NULL;

	if (!found) {
	    DPRINT(Debug,57,(&Debug,
			     "get_word_from_string -- failing at %d -- no unicode\n",
			     X));

	    ret = 0;
	    goto fail;
	}

	/* Is space?    (returns 0 or original character) */
	sp = unicode_ch(c,UOP_space);
	if (sp)
	    c = 0x0020;    /* Treat all unicode space characters as ascii space */

	for (y = ascii_delim; *y; y++)
	    if (*y == c || *y == sp)
		break;

	/* Delim found */
	if (*y) {
	    if (found_delim)
		*found_delim = c;
	    break;
	}
	data[dx++] = lowercase ? unicode_ch(c,UOP_lowercase) : c;

	if (!sp)
	    last_X = dx;

	X++;
	
    }

    if (trim && last_X < dx)
	dx = last_X;

    (*word)->string_type->charset_type->cs_add_unicodedata_to_it(*word,dx,data,
								 &FAILCOUNT);

    ret = X -*pos;
    *pos = X;

 fail:
    if (data)
	free(data);

    DPRINT(Debug,58,(&Debug,
		     "get_word_from_string=%d, *pos=%d",ret,*pos));

    if (found_delim) {
	DPRINT(Debug,60,(&Debug,", *found_delim=%04x",*found_delim));
    }

    DPRINT(Debug,58,(&Debug,"\n"));

    return ret;
}


int locate_unicode_from_string(str,unicode)
     const struct string *str;
     int unicode;
{
    /* NOTE: We can not trim string !!! */
    int i;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"locate_unicode_from_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"locate_unicode_from_string",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"locate_unicode_from_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"locate_unicode_from_string",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "locate_unicode_from_string(%p) ('%s'; type=%p '%s', len=%d)\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len));



    for (i = 0; i < str->p->len; i++) {
	int found;
	uint16 c = str->string_type->charset_type->
	    cs_give_unicode_from_it(str,i,&found);

	if (found && c == unicode) {
	    DPRINT(Debug,60,(&Debug,
			     "locate_unicode_from_string=%d\n",i));
	    return i;
	}
    }
    
    DPRINT(Debug,60,(&Debug,
		     "locate_unicode_from_string=-1 (not found)\n"));
    return -1;
}

int find_pattern_from_string(str,pattern,ignore_case)
     const struct string *str;
     const struct string *pattern;
     int ignore_case;
{
    int ret = -1;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad magic number (str)",0);

    if (STRING_magic != pattern->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad magic number (pattern)",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad magic number (string type)",0);

    if (CS_charset_magic != pattern->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad map (string type)",0);

    if (INVALID_MAP(pattern->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad magic number (str_private_data)",0);

    if (pattern->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"find_pattern_from_string",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "find_pattern_from_string(%p,%p,%d) ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
		     str,pattern,ignore_case,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     str->p->len,
		     pattern->string_type->MIME_name ? 
		     pattern->string_type->MIME_name : "<none>",
		     pattern->string_type->charset_type,
		     pattern->string_type->charset_type->type_name,
		     pattern->p->len));

    /* Check lengths -- do compressions and so on */
    str->string_type->charset_type->cs_check_length_it(str);
    pattern->string_type->charset_type->cs_check_length_it(pattern);

    
    if (0 == str->p->len) {
	/* Nothing can be found from empty string */
	
	DPRINT(Debug,63,(&Debug,
			 "cs_find_pattern_from_string=0  -- empty string\n"));
	ret = 0;
    } else if (0 == pattern->p->len) {
	/* Empty pattern can found from anywhere */
	
	DPRINT(Debug,63,(&Debug,
			 "cs_find_pattern_from_string=1  -- empty pattern\n"));
	ret = 1;
    } else if (str -> string_type == pattern -> string_type)
	ret = str->string_type->charset_type->
	    cs_find_pattern_from_it(str,pattern,ignore_case);

    /* -1 indicates that unicode values should be used on comparision ... */

    if (ret < 0) {
	int l1 = 0;
	uint16 * vector1 = safe_calloc ( (str->p->len), sizeof (uint16));
	int l2 = 0;
	uint16 * vector2 = safe_calloc ( (pattern->p->len), sizeof (uint16));
	int i,j;
	
	ret = 0;
	
	for (i = 0;  i < str->p->len; i++) {
	    int found;
            uint16 c =
                str->string_type->charset_type->
		cs_give_unicode_from_it(str,i,&found);

	    /* TODO: Should lowercase ANY unicode value ... */

	    if (!found)
		c = MAPPING_NONE;
	    else if (ignore_case) {
		c = unicode_ch(c,UOP_lowercase);
		if (!c)
		    c = MAPPING_NONE;
	    }
	    vector1[l1++] = c;
	}

	for (i = 0;  i < pattern->p->len; i++) {
	    int found;
            uint16 c =
                pattern->string_type->charset_type->
		cs_give_unicode_from_it(pattern,i,&found);

	    if (!found)
		c = MAPPING_NONE;
	    else if (ignore_case) { 
		c = unicode_ch(c,UOP_lowercase);
		if (!c)
		    c = MAPPING_NONE;
	    }
	    vector2[l2++] = c;
	}

	for (i = 0; i < l1; ) {
	    const int s = i + 1;
	    for (j = 0; j < l2 && i < l1; j++,i++) {
		if (vector1[i] != vector2[j])
		    break;
		/* MAPPING_NONE matches newer ... */
		if (MAPPING_NONE == vector1[i] ||
		    MAPPING_NONE == vector2[j])
		    break;
	    }
	    if (j >= l2) {
		DPRINT(Debug,63,(&Debug,
				 "cs_find_pattern_from_string=1 MATCH\n"));

		ret = 1;
		break;
	    }
	    i = s;
	}	     

	if (!ret) {
	    DPRINT(Debug,63,(&Debug,
			     "cs_find_pattern_from_string=0 NO MATCH\n"));
	}
	
	free(vector1);
	free(vector2);
    }
								      
    DPRINT(Debug,60,(&Debug,
		     "find_pattern_from_string=%d\n",ret));

    return ret;
}


static int string_match_part P_((const struct string * name, int name_X,
				 const struct string * pat, int pat_X,
				 int ignore_case));

static int string_match_part(name,name_X,pat,pat_X,ignore_case)
     const struct string * name;
     int name_X;
     const struct string * pat;
     int pat_X;
     int ignore_case;
{
    int r = 1;

    if (STRING_magic != name->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_match_part",
	      "Bad magic number (name)",0);

    if (STRING_magic != pat->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_match_part",
	      "Bad magic number (pat)",0);

    /* It is assumed that compressions are already done ... */

    while (pat_X < pat->p->len) {
	int pat_found = 0;
	uint16 code_pat  = 
	    pat->string_type->charset_type->
	    cs_give_unicode_from_it(pat,
				    pat_X,
				    &pat_found);
	
	if (pat_found && 
	    0x002A /* '*' */ == code_pat) {
	    
	    if (name_X < name->p->len) {
		if (string_match_part(name,name_X,pat,pat_X+1,
				      ignore_case)) {
		    /* Tail matches */
		
		    goto succeed;
		}
		/* Try match to next starting position */
		
		name_X++;
		
	    } else {
		/* '*' matches to empty string */
		pat_X++;
	    }	    
	} else if (name_X < name->p->len) {
	    int name_found = 0;
	    uint16 code_name = 
		name->string_type->charset_type->
		cs_give_unicode_from_it(name,
					name_X,
					&name_found);
	    
	    
	    if (pat_found &&
		0x003F /* '?' */ == code_pat) {

		/* Accept any character */
		name_X++;
		pat_X++;
		
	    } else if (pat_found && name_found &&
		       code_pat == code_name) {

		    DPRINT(Debug,56,(&Debug,
				     "-- Matches on pat_X=%d name_X=%d    \t(%04X)\n",
				     pat_X,name_X,code_name));

		
		/* Accept character */
		name_X++;
		pat_X++;

	    } else if (pat_found && name_found && ignore_case) {

		uint16 c1 = unicode_ch(code_pat,UOP_lowercase);
		uint16 c2 = unicode_ch(code_name,UOP_lowercase);

		if (!c1 || !c2) {
		    DPRINT(Debug,56,(&Debug,
				     "-- no lowercase character pat_X=%d (%04X) name_X=%d (%04X)\n",
				     pat_X,code_pat,name_X,code_name));

		    goto no_match;
		}

		if (c1 == c2) {

		    DPRINT(Debug,56,(&Debug,
				     "-- Lowercase matches on pat_X=%d (%04X) name_X=%d (%04X)   \t(%04X)\n",
				     pat_X,code_pat,name_X,code_name,c1));


		    /* Accept character */
		    name_X++;
		    pat_X++;

		} else
		    goto no_match;

	    } else {
	    no_match:
		r = 0;

		DPRINT(Debug,55,(&Debug,
			   "-- Match fail on pat_X=%d (%04X) name_X=%d (%04X)\n",
			   pat_X,code_pat,name_X,code_name));
		break;
	    }		
	} else {
	    r = 0;
	    DPRINT(Debug,55,(&Debug,
			     "-- Match fail on pat_X=%d name_X=%d (end of name)\n",
			     pat_X,name_X));
	    break;	    
	}  
    }
    
    if (name_X != name->p->len ||
	pat_X != pat->p->len) {

	DPRINT(Debug,55,(&Debug,
			 "-- Match fail on pat_X=%d name_X=%d (failure?)\n",
			 pat_X,name_X));

	r = 0;
    }
 succeed:
    return r;
}

/* return 1 on succeed and 0 on failure */
int string_match(name,pat,ignore_case)
     const struct string * name;
     const struct string * pat;
     int ignore_case;
{
    int r = 0;

    if (STRING_magic != name->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad magic number (name)",0);

    if (STRING_magic != pat->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad magic number (pat)",0);

    if (CS_charset_magic != name->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad magic number (string type)",0);

    if (CS_charset_magic != pat->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(name->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad map (string type)",0);

    if (INVALID_MAP(pat->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad map (string type)",0);

    if (name->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad magic number (str_private_data)",0);
    if (pat->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_match",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,60,(&Debug,
		     "string_match(%p,%p,%d) ('%s'; type=%p '%s', len=%d), ('%s'; type=%p '%s', len=%d)\n",
		     name,pat,ignore_case,
		     name->string_type->MIME_name ? 
		     name->string_type->MIME_name : "<none>",
		     name->string_type->charset_type,
		     name->string_type->charset_type->type_name,
		     name->p->len,
		     pat->string_type->MIME_name ? 
		     pat->string_type->MIME_name : "<none>",
		     pat->string_type->charset_type,
		     pat->string_type->charset_type->type_name,
		     pat->p->len));

    /* Check lengths -- do compressions and so on */
    name->string_type->charset_type->cs_check_length_it(name);
    pat->string_type->charset_type->cs_check_length_it(pat);

    r = string_match_part(name,0,pat,0,ignore_case);
    
    DPRINT(Debug,56,(&Debug,
		     "%S %s %S %s\n",
		     name,r ? "matches" : "do not match",pat,
		     ignore_case ? "with ignoring case" : ""));

    DPRINT(Debug,60,(&Debug,
		     "string_match=%d\n",r));

    return r;
}


uint16 give_unicode_from_string(str,pos)
     const struct string *str;
     int pos;
{
    int found;
    uint16 ret;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"give_unicode_from_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"give_unicode_from_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"give_unicode_from_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"give_unicode_from_string",
	      "Bad magic number (str_private_data)",0);

    /* It is assumed that compressions are done
       .. ie string_length() function
       is called 
    */

    ret = str->string_type->charset_type->
	cs_give_unicode_from_it(str,pos, &found);

    DPRINT(Debug,60,(&Debug,
		     "give_unicode_from_string(%p,%d)=%d  ('%s'; type=%p '%s')%s\n",
		     str,pos,ret,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     found ? "" : " [NOT FOUND]"));
    
    return ret;
}

unsigned char cs_give_bytecode_from_null(str,pos,found)
     const struct string *str;
     int pos; 
     int *found;
{
    DPRINT(Debug,61,(&Debug,"cs_give_bytecode_from_null called\n"));

    *found = 0;

    return 0;
}

/* Returns flags what values are valid */
int  give_character_from_string(str,pos,unicode,bytecode)
     const struct string *str;
     int pos;
     uint16              *unicode;
     unsigned char       *bytecode;
{
    int flags = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"give_character_from_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"give_character_from_string",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"give_character_from_string",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"give_character_from_string",
	      "Bad magic number (str_private_data)",0);

    if (unicode) {
	int found;

	*unicode = str->string_type->charset_type->
	    cs_give_unicode_from_it(str,pos, &found);

	if (found)
	    flags |= GCHAR_unicode;
    }

    if (bytecode) {
	int found;
	
	*bytecode = str->string_type->charset_type->
	    cs_give_bytecode_from_it(str,pos, &found);
	
	if (found)
	    flags |= GCHAR_bytecode;
    }
    
    return flags;
}

void remove_control(str)
     const struct string *str;
{
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"remove_control",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"remove_control",
	      "Bad magic number (string type)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"remove_control",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"remove_control",
	      "Bad magic number (str_private_data)",0);

    if (str->p->state) {
	if (str->p->state->charset != str->string_type)
	    panic("STRING PANIC",__FILE__,__LINE__,"remove_control",
		  "Bad magic number (state charset)",0);
	free_state_internal(&(str->p->state));
    }

    DPRINT(Debug,60,(&Debug,
		     "remove_control(%p)   ('%s'; type=%p '%s')\n",
		     str,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name));
    
    str->string_type->charset_type->cs_remove_control_it(str);	
}

void add_state_to_string(str,ch)
     struct string *str; 
     struct charset_state *ch;
{
    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
	      "Bad magic number (string type)",0);
    
    if (CS_charset_magic != ch->charset->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
	      "Bad magic number (charset)",0);

    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
	      "Bad map (string type)",0);

    if (INVALID_MAP(ch->charset))    
	panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
	      "Bad map (charset)",0);


    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
	      "Bad magic number (str_private_data)",0);

    if (str->p->state) {
	if (str->p->state->charset != str->string_type)
	    panic("STRING PANIC",__FILE__,__LINE__,"add_state_to_string",
		  "Bad magic number (state charset)",0);
	free_state_internal(&(str->p->state));
    }

    DPRINT(Debug,60,(&Debug,
		     "add_state_to_it(%p,%p)   ('%s'; type=%p '%s', '%s'; type=%p '%s')\n",
		     str,ch,
		     str->string_type->MIME_name ? 
		     str->string_type->MIME_name : "<none>",
		     str->string_type->charset_type,
		     str->string_type->charset_type->type_name,
		     ch->charset->MIME_name ? 
		     ch->charset->MIME_name : "<none>",
		     ch->charset->charset_type,
		     ch->charset->charset_type->type_name));

    if (ch->charset == str->string_type)
	str->string_type->charset_type->cs_add_state_to_it(str,ch);
    else {
	int found;
	int FAILCOUNT = 0;
	uint16 val = 
	    ch->charset->charset_type->cs_give_unicode_from_s_it(ch,&found);
	
	str->string_type->charset_type->
	    cs_add_unicodedata_to_it(str,1,&val,&FAILCOUNT);
    }
}

int charset_properties(ptr)
     charset_t ptr;
{
    int prop = 0;

    if (CS_charset_magic != ptr->magic) 
 	panic("STRING PANIC",__FILE__,__LINE__,"charset_properties",
	      "Bad magic number (charset)",0);

    if (INVALID_MAP(ptr))    
 	panic("STRING PANIC",__FILE__,__LINE__,"charset_properties",
	      "Bad map (charset)",0);


    prop = ptr->charset_type->cs_it_properties(ptr);

    DPRINT(Debug,18,(&Debug, 
		    "charset_properties(%p '%s'; type=%p)=%d\n",
		    ptr,
		    ptr->MIME_name ? ptr->MIME_name : "<none>",
		    ptr->charset_type,
		    prop));
    return prop;
}

/* bad_pos is -1 if string is OK */
long string_to_long(str,bad_pos)
     const struct string *str; 
     int *bad_pos;
{
    long val = 0L;
    int found;
    int pos = 0;
    uint16 c;
    int plus_minus = 1;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_to_long",
	      "Bad magic number",0);
    
    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_to_long",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_to_long",
	      "Bad magic number (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_to_long",
	      "Bad magic number (str_private_data)",0);

    *bad_pos = -1;
    
    pos = 0;

    /* Check lengths */
    str->string_type->charset_type->cs_check_length_it(str);

    
    if (str->p->len < 1)
	goto fail;

    c = str->string_type->charset_type->
	cs_give_unicode_from_it(str,pos,&found);

    if (!found) 
	goto fail;

    if (0x002B /* + */ == c) {
	plus_minus = 1;
	pos++;
	
    } else if (0x002D /* - */ == c) {
	plus_minus = -1;
	pos++;
    }


    while (pos < str->p->len) {
	int num;
	long l1;

	c = str->string_type->charset_type->
	    cs_give_unicode_from_it(str,pos,&found);
	if (!found) 
	    goto fail;

	if (c < 0x0030  /* 0 */ ||
	    c > 0x0039  /* 9 */)
	    goto fail;

	num = c - 0x0030;

	if (val > LONG_MAX / 10)
	    goto fail;     /* Overflow */
	l1 = val * 10;
	
	if (LONG_MAX - l1 < num)
	    goto fail;     /* Overflow */
	
	l1 += num;

	val = l1;	
	pos++;
    }

    val = val * plus_minus;

    DPRINT(Debug,30,(&Debug,
		     "string_to_long: %S gives value %ld\n",
		     str,val));

    return val;

 fail:
    *bad_pos = pos;

    val = val * plus_minus;

    DPRINT(Debug,30,(&Debug,
		     "string_to_long: %S gives value %ld, error on position %d\n",
		     str,val,pos));

    return val;
}


static int string_matches_ascii_1 P_((const struct string *str,
				      const unsigned char *ascii,
				      int ignore_case,
				      int match_prefix,
				      int return_match_len,
				      int return_order));

static int string_matches_ascii_1(str,ascii,ignore_case,match_prefix,
				  return_match_len,return_order)
     const struct string *str;
     const unsigned char *ascii;
     int ignore_case;
     int match_prefix;
     int return_match_len;
     int return_order;
{
    int i;
    int r;
    
    for (i = 0; i < str->p->len; i++) {
	uint16 c;
	int found;
	unsigned char c1 = ascii[i];

	c = str->string_type->charset_type->
	    cs_give_unicode_from_it(str,i,&found);

	if (!found) {
	    r = 0;
	    if (return_order) r = 1;
	    
	    DPRINT(Debug,30,(&Debug,
			     "string_matches_ascii_1=%d on pos %d (unicode not found)\n",
			     r,i));
	    return r;
	}
	if (ignore_case) {
		int c_a = unicode_ch(c,UOP_lowercase);
		if (!c_a && !return_order) {
		    DPRINT(Debug,30,(&Debug,
				     "string_matches_ascii_1=0 on pos %d (no lowercasing)\n",
				     i));
		    return 0;
		}
		if (c_a) c = c_a;
	}
	if (!c1) {
	    if (match_prefix) {
		
		if (return_match_len) {
		    DPRINT(Debug,30,(&Debug,
				     "string_matches_ascii_1=%d (ascii string ended, prefix match)\n",
				     i));
		    
		    return i;
		}

		r = 1;
		if (return_order) r = 0;
		
		DPRINT(Debug,30,(&Debug,
				 "string_matches_ascii_1=%d on pos %d (ascii string ended, prefix match)\n",
				 r,i));
		return r;
	    } else {

		r = 0;
		if (return_order) r = 1;

		
		DPRINT(Debug,30,(&Debug,
				 "string_matches_ascii_1=0 on pos %d (ascii string ended)\n",
				 r,i));
		return r;
	    }
	}

	if (ignore_case && c1 >= 0x41 && c1 <= 0x5A) {
	    c1 = ( c1 - 0x41) + 0x61;
	}

	if (c != c1) {
	    r = 0;

	    if (return_order && c < c1) r = -1;
	    if (return_order && c > c1) r =  1;
	    
	    DPRINT(Debug,30,(&Debug,
		     "string_matches_ascii_1=%d on pos %d (mismatch)\n",
			     r,i));
	    return r;	    
	}
    }

    if (ascii[i]) {
	r = 0;
	if (return_order) r = -1;
	
	DPRINT(Debug,30,(&Debug,
			 "string_matches_ascii_1=%d on pos %d (ascii string not ended)\n",
			 r,i));
	return r;
    }


    if (return_match_len) {
	DPRINT(Debug,30,(&Debug,
			 "string_matches_ascii_1=%d (matches)\n",
			 i));
	
	return i;
    }

    r = 1;

    if  (return_order) r = 0;
    
    DPRINT(Debug,30,(&Debug,
		     "string_matches_ascii_1=%d on pos %d (matches)\n",
		     r,i));
    return r;

}

/* Return -1 on failure */

static int string_find_ascii_1 P_((const struct string *str,
				   const unsigned char *ascii,
				   int ignore_case,
				   int return_match_len));

static int string_find_ascii_1(str,ascii,ignore_case,return_match_len)
     const struct string *str;
     const unsigned char *ascii;
     int ignore_case;
     int return_match_len;
{
    int startpos;
    
    for (startpos = 0; startpos < str->p->len; startpos++) {
	
	int i;

	for (i = 0; i <  str->p->len - startpos; i++) {

	    int spos = startpos + i;

	    uint16 c;
	    int found;
	    unsigned char c1 = ascii[i];

	    c = str->string_type->charset_type->
		cs_give_unicode_from_it(str,spos,&found);

	    if (!found) {
		DPRINT(Debug,30,(&Debug,
				 "string_find_ascii_1: unicode not found on pos %d\n",
				 spos));

		startpos = spos;
		goto restart;
	    }

	    if (ignore_case) {
		c = unicode_ch(c,UOP_lowercase);
		if (!c) {
		    DPRINT(Debug,30,(&Debug,
				     "string_find_ascii_1: no lowercasing on pos %d\n",
				     spos));
		    
		    startpos = spos;
		    goto restart;	
		}
	    }

	    if (!c1) {
		if (return_match_len) {
		    DPRINT(Debug,30,(&Debug,
				     "string_find_ascii_1=%d (ascii string ended, match from pos %d)\n",
				     i,startpos));
		    
		    return i;
		}
	
		DPRINT(Debug,30,(&Debug,
				 "string_find_ascii_1=%d (ascii string ended on len %d, match from pos %d))\n",
				 startpos,i,startpos));

		return spos;
	    }
	    
	    if (ignore_case && c1 >= 0x41 && c1 <= 0x5A) {
		c1 = ( c1 - 0x41) + 0x61;
	    }

	    if (c != c1) {
		DPRINT(Debug,30,(&Debug,
				 "string_find_ascii_1: mismatch on pos %d, ascii pos %d\n",
				 spos,i));
		goto fail;
	    }
	}

	if (ascii[i]) {
	    DPRINT(Debug,30,(&Debug,
			     "string_find_ascii_1=-1 - ascii string not end on len %d, match from pos %d))\n",
			     i,startpos));
	    return -1;
	}

	if (return_match_len) {
	    DPRINT(Debug,30,(&Debug,
			     "string_find_ascii_1=%d (ascii string ended, match from pos %d - full match)\n",
			     i,startpos));
	    
	    return i;
	}

	DPRINT(Debug,30,(&Debug,
			 "string_find_ascii_1=%d (ascii string ended on len %d, match from pos %d - full match))\n",
			 startpos,i,startpos));
	
	return startpos;
	
    fail:
	    
	if (0) {
	restart:
	    DPRINT(Debug,30,(&Debug,
			     "string_find_ascii_1: restarting after pos %d\n",startpos));

	}	   
    }

    DPRINT(Debug,30,(&Debug,
		     "string_find_ascii_1=-1 - search failed\n"));

    return -1;
}


/* Return 1 is matches, otherwise 0 
   Return match len if flag&SMA_return_len

   if op == SMA_op_return_order:

   return 1 if unicode not found
   return 1 if ascii ends first
   
   return -1 if str < ascii
   return 1 if str > ascii
   
   return -1 if str ends first
   return 0 if str matches ascii
*/
int string_matches_ascii(str,ascii,flag,op)
     const struct string *str;
     const unsigned char *ascii;
     int flag;
     enum SMA_operation op;
{
    int ignore_case       = (flag & SMA_ignore_case);
    int return_match_len  = (flag & SMA_return_len);

    int match_prefix      = 0;
    int find_ascii_substr = 0;  /* Find ascii subtring, 
				  return start pos or -1 
				  if SMA_return_len is not set
				  
				  Both SMA_match_prefix and
				  SMA_find_ascii must not
				  be set;
							 
			       */

    int return_order = 0;    /* retun 0 on match */
    
    int ret = 0;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_matches_ascii",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_matches_ascii",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_matches_ascii",
	      "Bad map (string type)",0);

    /* Check lengths -- do compressions and so on */
    str->string_type->charset_type->cs_check_length_it(str);
    
    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_matches_ascii",
	      "Bad magic number (str_private_data)",0);

    DPRINT(Debug,25,(&Debug,
		     "string_matches_ascii: str=%S\n",str));
    DPRINT(Debug,25,(&Debug,
		     "                    ascii=%Q\n",ascii));
    DPRINT(Debug,25,(&Debug,
		     "                    flag=%d",flag));
    if (ignore_case)       { DPRINT(Debug,25,(&Debug,", ignore case"));       }
    if (return_match_len)  { DPRINT(Debug,25,(&Debug,", return match len"));  }

    DPRINT(Debug,25,(&Debug,"\n"));
    
    DPRINT(Debug,25,(&Debug,
		     "                    op=%d",op));
    switch (op) {
    case SMA_op_normal:  DPRINT(Debug,25,(&Debug, " SMA_op_normal"));  break;
    case SMA_op_match_prefix:
	DPRINT(Debug,25,(&Debug, " SMA_op_match_prefix")); 
	match_prefix = 1;
	break;
    case SMA_op_find_ascii:
	DPRINT(Debug,25,(&Debug, " SMA_op_find_ascii"));
	find_ascii_substr = 1;
	break;
    case SMA_op_return_order:
	DPRINT(Debug,25,(&Debug, " SMA_op_return_order"));
	return_order = 1;
	break;
    }
    
    
    if (return_order && return_match_len) {
	panic("STRING PANIC",__FILE__,__LINE__,
	      "string_matches_ascii",
	      "Both return_order and return_match_len are set",0);

    }

    if (find_ascii_substr) {
	
	if (match_prefix) {
	    panic("STRING PANIC",__FILE__,__LINE__,
		  "string_matches_ascii",
		  "Both match_prefix and find_ascii_substr are set",0);
	} 
	
	if (return_order) {
	    panic("STRING PANIC",__FILE__,__LINE__,
		  "string_matches_ascii",
		  "Both return_order and find_ascii_substr are set",0);
	}
	
	ret = string_find_ascii_1(str,ascii,ignore_case,return_match_len);
    } else {
	ret = string_matches_ascii_1(str,ascii,ignore_case,match_prefix,
				     return_match_len,return_order);
    }

    DPRINT(Debug,25,(&Debug,
		     "string_matches_ascii=%d\n",ret));
    
    return ret;
}

static int string_have_ascii_sep0 P_((const struct string *str,
				      const unsigned char *ascii  /* pick one char */,
				      int start_pos,
				      int limit_pos));
static int string_have_ascii_sep0(str,ascii,start_pos,limit_pos)
     const struct string *str;
     const unsigned char *ascii;  /* pick one char */
     int start_pos;
     int limit_pos;
{
    int i;

    for (i = start_pos; i < limit_pos &&
	     i < str->p->len; i++) {
	uint16 c;
	int found;

	c = str->string_type->charset_type->
	    cs_give_unicode_from_it(str,i,&found);

	if (found) {
	    int j;

	    for (j = 0; ascii[j]; j++) {

		if (c == ascii[j]) {
		    DPRINT(Debug,30,(&Debug,
				     "string_have_ascii_sep0=%d (found '%c')\n",
				     i,ascii[j]));
		    return i;
		}
	    }
	}
    }

    DPRINT(Debug,30,(&Debug,"string_have_ascii_sep0=-1\n"));

    return -1;


}

/* Returns -1 or offset */
int string_have_ascii_sep1(str,ascii,start_pos,limit_pos)
     const struct string *str;
     const unsigned char *ascii;  /* pick one char */
     int start_pos;
     int limit_pos;
{
    int r;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep1",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep1",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep1",
	      "Bad map (string type)",0);

    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep1",
	      "Bad magic number (str_private_data)",0);

    if (start_pos < 0)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep1",
	      "Bad start_pos",0);

    if (limit_pos > str->p->len) {
	DPRINT(Debug,30,(&Debug,
			 "string_have_ascii_sep1: limit_pos=%d changing to %d\n",
			 limit_pos,str->p->len));
	limit_pos = str->p->len;
    }

    if (start_pos >= limit_pos) {
	DPRINT(Debug,30,(&Debug,
			 "string_have_ascii_sep1: start_pos = %d >= limit_pos = %d\n",
			 start_pos,limit_pos));
	r = -1;

    } else
	r = string_have_ascii_sep0(str,ascii,start_pos,limit_pos);


    DPRINT(Debug,30,(&Debug,"string_have_ascii_sep1=%d\n",r));

    return r;
}

/* Returns -1 or offset */
int string_have_ascii_sep(str,ascii)
     const struct string *str;
     const unsigned char *ascii;  /* pick one char */
{
    int r;

    if (STRING_magic != str->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep",
	      "Bad magic number",0);

    if (CS_charset_magic != str->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(str->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep",
	      "Bad map (string type)",0);


    if (str->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_have_ascii_sep",
	      "Bad magic number (str_private_data)",0);

    r = string_have_ascii_sep0(str,ascii,0,str->p->len);

    DPRINT(Debug,30,(&Debug,"string_have_ascii_sep=%d\n",r));

    return r;
}

void append_string(res,s,printind)
     struct string ** res; 
     const struct string *s;
     int printind;
{
    if (STRING_magic != s->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"append_string",
	      "Bad magic number",0);

    if (CS_charset_magic != s->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"append_string",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(s->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"append_string",
	      "Bad map (string type)",0);
  
    if (!*res)
	*res = dup_string(s);

    else {
	int ERRORS = 0;

	if (STRING_magic != (*res)->magic)
	    panic("STRING PANIC",__FILE__,__LINE__,"append_string",
		  "Bad magic number (res)",0);

	if (CS_charset_magic != (*res)->string_type->magic) 
	    panic("STRING PANIC",__FILE__,__LINE__,"append_string",
		  "Bad magic number (string type)",0);
    
	if (INVALID_MAP((*res)->string_type))    
	    panic("STRING PANIC",__FILE__,__LINE__,"append_string",
		  "Bad map (string type)",0);



	if ((*res) -> string_type == s -> string_type) {
	    
	    /* Optimize same charset case  -- can do as internal append */	
	    
	    DPRINT(Debug,60,(&Debug,
			     "append_string -- same charset type ('%s'; type=%p '%s')\n",
			     s->string_type->MIME_name ? 
			     s->string_type->MIME_name : "<none>",
			     s->string_type->charset_type,
			     s->string_type->charset_type->type_name));
	    
	    (*res)->string_type->charset_type->cs_add_intdata_to_it(*res,s);
	    
	    if (((*res)->p->lang && ! s->p->lang) ||
		((*res)->p->lang &&
		 0 != strcmp((*res)->p->lang, s->p->lang))) {
		
		/* Not a same language -- reset language information */
		
		free((*res)->p->lang);
		(*res)->p->lang = NULL;
	    }
	
	} else {
	    
	    struct string *t = cat_strings_internal(*res,s,printind,&ERRORS);
	    free_string(res);
	    *res = t;
	}

	DPRINT(Debug,60,(&Debug,
			 "append_string: *res=%p  ('%s'; type=%p '%s'): len=%d ERRORS=%d\n",
			 (*res),
			 (*res)->string_type->MIME_name ? 
			 (*res)->string_type->MIME_name : "<none>",
			 (*res)->string_type->charset_type,
			 (*res)->string_type->charset_type->type_name,
			 (*res)->p->len,
			 ERRORS));	
    }
}

void string_copy_character(res,s,pos,len,errors)
     struct string ** res;
     const struct string *s;
     int *pos; 
     int len;
     int *errors;
{
    int ERRORS = 0;

    if (STRING_magic != s->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"string_copy_character",
	      "Bad magic number",0);

    if (CS_charset_magic != s->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"string_copy_character",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(s->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"string_copy_character",
	      "Bad map (string type)",0);


    if (!*res)
	*res = clip_from_string(s,pos,len);

    else {
	struct string * tmp  = clip_from_string(s,pos,len);
	struct string * tmp2;

	if (STRING_magic != (*res)->magic)
	    panic("STRING PANIC",__FILE__,__LINE__,"string_copy_character",
		  "Bad magic number (res)",0);

	if (CS_charset_magic != (*res)->string_type->magic) 
	    panic("STRING PANIC",__FILE__,__LINE__,"string_copy_character",
		  "Bad magic number (string type)",0);
    
	if (INVALID_MAP((*res)->string_type))    
	    panic("STRING PANIC",__FILE__,__LINE__,"string_copy_character",
		  "Bad map (string type)",0);
		
	tmp2 = cat_strings_internal(*res,tmp,0,&ERRORS);
	
	/* Not a easy to optimize ... */

	free_string(&tmp);
	free_string(res);
	*res = tmp2;

	DPRINT(Debug,60,(&Debug,
			 "string_copy_character: *res=%p  ('%s'; type=%p '%s'): len=%d ERRORS=%d\n",
			 (*res),
			 (*res)->string_type->MIME_name ? 
			 (*res)->string_type->MIME_name : "<none>",
			 (*res)->string_type->charset_type,
			 (*res)->string_type->charset_type->type_name,
			 (*res)->p->len,
			 ERRORS));

    }
    
    if (errors)
	*errors = ERRORS;
}

enum digest_feed_mode cs_digest_feed_from_null(ret,s)
     struct digest_proc * ret;
     const struct string *s;
{

    return digest_feed_fallback_unicode;
}

enum unicode_vector_mode cs_unicode_vector_from_default(ret,s,errcount)
     struct string_unicode_vector *ret;
     const struct string *s;
     int *errcount;
{
    enum unicode_vector_mode res = unicode_vector_fail;
    int prop;
    size_t len;
    int i;
    int FOUNDERR  = 0;
    size_t l = 0;
      
    if (STRING_magic != s->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_unicode_vector_from_default",
	      "Bad magic number",0);

    if (CS_charset_magic != s->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"cs_unicode_vector_from_default",
	      "Bad magic number (string type)",0);
    
    if (s->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_unicode_vector_from_default",
	      "Bad magic number (str_private_data)",0);

    
    if (s->p->len < 1) {
	res = unicode_vector_empty;
	goto done;
    }

    prop = charset_properties(s -> string_type);
    if (0 == (prop & CS_mapping)) {
	res = unicode_vector_fail;
	goto done;
    }

    len = (size_t)s->p->len;

    ret->first_failure = len;
    ret->vector        = safe_array_realloc(ret->vector,len,
					    sizeof (ret->vector[0]));

    for (i = 0; i < s->p->len; i++) {
	int found = 0;
	uint16 c = 
	    s->string_type->charset_type->
	    cs_give_unicode_from_it(s,i,&found);

	if (l >= len) 
	    panic("STRING PANIC",__FILE__,__LINE__,"cs_unicode_vector_from_default",
		  "Overflow",0);

	
	ret->vector[l] = c;

	if (!found) {
	    if (l < ret->first_failure)
		ret->first_failure = l;
	    FOUNDERR++;
	}

	l++;
    }
    ret->vector_len = l;

    res = unicode_vector_done;
    if (errcount)
	*errcount = FOUNDERR;
           
 done:
    if (res <  unicode_vector_done) {
	if (ret->vector) {
	    free(ret->vector);
	    ret->vector = NULL;
	}
	ret->vector_len = 0;
	ret->first_failure = 0;

	if (errcount)
	    *errcount = s->p->len;	
    }

    return res;
}

enum unicode_vector_mode cs_unicode_vector_from_null(ret,s,errcount)
     struct string_unicode_vector *ret;
     const struct string *s;
     int *errcount;
{
    if (STRING_magic != s->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_unicode_vector_from_null",
	      "Bad magic number",0);

    if (CS_charset_magic != s->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"cs_unicode_vector_from_null",
	      "Bad magic number (string type)",0);
    
    if (s->p->magic  != CS_str_magic)
	panic("STRING PANIC",__FILE__,__LINE__,"cs_unicode_vector_from_null",
	      "Bad magic number (str_private_data)",0);


    if (ret->vector) {
	free(ret->vector);
	ret->vector = NULL;
    }
    ret->vector_len = 0;
    ret->first_failure = 0;

    if (errcount)
	*errcount = s->p->len;
    
    if (s->p->len < 1) {
	return unicode_vector_empty;
    }
    
    return  unicode_vector_fail; 
}




void digest_feed_string(ret,s)
     struct digest_proc * ret;
     const struct string *s;
{
    enum digest_feed_mode m;
    unsigned char * buffer = NULL;
    int simple = 1;
    
    if (STRING_magic != s->magic)
	panic("STRING PANIC",__FILE__,__LINE__,"digest_feed_string",
	      "Bad magic number",0);

    if (CS_charset_magic != s->string_type->magic) 
	panic("STRING PANIC",__FILE__,__LINE__,"digest_feed_string",
	      "Bad magic number (string type)",0);
    
    if (INVALID_MAP(s->string_type))    
	panic("STRING PANIC",__FILE__,__LINE__,"digest_feed_string",
	      "Bad map (string type)",0);

    /* Check length */
    s->string_type->charset_type->cs_check_length_it(s);

    if (s->p->len < 1)
	return;
    
    m = s->string_type->charset_type->cs_digest_feed_from_s_it(ret,s);
    
    switch (m) {
	int i;
	
    case digest_feed_fallback_unicode:

	for (i = 0; i < s->p->len; i++) {
	    int found;
	    uint16 res = 
		s->string_type->charset_type->
		cs_give_unicode_from_it(s,i,&found);

	    if (!found)
		goto stream_fallback;

	    if (! res || res > 127) {
		simple = 0;
		break;
	    }
	}
	
	if (simple) {
	    /* Produce same result than digest_feed_str() */
	    
	    buffer = safe_malloc(s->p->len);
	    
	    for (i = 0; i < s->p->len; i++) {
		int found;
		uint16 res =
		    s->string_type->charset_type->
		    cs_give_unicode_from_it(s,i,&found);
		
		
		if (!res || res > 127 || !found)
		    panic("STRING PANIC",__FILE__,__LINE__,"cs_digest_feed_from_utf",
			  "Unexpected data",0);
		
		buffer[i] = res;
	    }
	    
	    feed_digest_proc(ret,buffer,i);
	    
	} else {
	    size_t buffer_len = 2;
	    int buffer_ok = 1;
	    size_t x = 0;
	    /* FALLBACK */
	    
	    buffer_len += 2 * (s->p->len);
	    buffer = safe_malloc(buffer_len);
	    
	    /* Prefix */
	    
	    buffer[x++] = 0;
	    buffer[x++] = 0;

	    /* Data */
	    
	    for (i = 0; i < s->p->len; i++) {
		int found;
		uint16 c = 
		    s->string_type->charset_type->
		    cs_give_unicode_from_it(s,i,&found);
		
		if (x+1 >= buffer_len)
		    panic("STRING PANIC",__FILE__,__LINE__,"digest_feed_string",
			  "Overflow",0);	    
		
		buffer[x++] = c / 256;
		buffer[x++] = c % 256;
		
		if (!found)
		    buffer_ok = 0;
	    }
	    
	    if (buffer_ok) {
		
		feed_digest_proc(ret,buffer,x);
		
	    } else {
		unsigned char * res;
		int reslen;

	    case digest_feed_fallback_stream:
	    stream_fallback:
		/* FALLBACK 2 */
		
		/* Prefix */
		
		if (s->string_type->MIME_name) {
		    size_t l = strlen(s->string_type->MIME_name);
		    
		    feed_digest_proc(ret,s2us(s->string_type->MIME_name),l+1);
		} else {
		    feed_digest_proc(ret,s2us("\1"),2);
		}
		
		/* Data */
		
		res = s->string_type->charset_type->cs_stream_from_it(s,0,NULL,&reslen);
		
		if (res) {
		    feed_digest_proc(ret,res,reslen);
		    
		    free(res);
		} else {
		    DPRINT(Debug,10,(&Debug,"digest_feed_string: Digest not available: %S\n",s));
		}
	    }
	}
	break;
	
    case  digest_feed_done:
	break;
	
    }

    if (buffer)
	free(buffer);
}



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

