static char rcsid[] = "@(#)$Id: mime_decode.c,v 2.40 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.40 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "def_melib.h"
#include "s_me.h"
#include "tagfilter_imp.h"

DEBUG_VAR(Debug,__FILE__,"mime");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

/* Prototype */
static void Xbit_decode	P_((struct in_state *s_in, struct out_state *s_out, 
			    const struct decode_opts *decode_opt,
			    int length, int is_text)); 

static void Xbit_decode (s_in, s_out, decode_opt, length, is_text)
     struct in_state *s_in;
     struct out_state *s_out;
     const struct decode_opts *decode_opt;
     int length;
     int is_text;
{
    /* state_getl adds \n to end of file so do not use it .... */


    int	bytes = 0; /* the total number of bytes read so far */
    int store_ch = 0; /* CRLF conversion */
    int addprefix = 1;

    DPRINT(Debug,10,(&Debug, 
		     "Xbit_decode: length=%d\n",
		     length));
   
    while (1) {
	int ch;
    
	if (bytes >= length)
	    break;
       
	ch = state_getc(s_in);

	if (EOF == ch) {
	    DPRINT(Debug,10,(&Debug, 
			     "Xbit_decode: EOF  bytes=%d\n",
			     bytes));
	    break;
	}
	bytes++;

	if (store_ch && ch != '\n') 
	    state_putc(store_ch,s_out);

	store_ch = 0;

	if (addprefix) {
	    state_add_prefix(s_out,decode_opt);
	    addprefix = 0;
	}
	
	if (is_text && ch == '\r')
	    store_ch = ch;
	else {
	    if (ch == '\n' && is_text && out_state_EOLN_is_CRLF(s_out))
		state_putc('\r',s_out);

	    state_putc(ch,s_out);
	    if (ch == '\n')
		addprefix = 1;
	}      
    }

    /* Make sure to flush anything left in the internal buffer. */
    if (store_ch)  /* for astext */
	state_putc(store_ch,s_out);


}
  

/* ------------------------------------------------------------------------- */

void null_SG_decoder(body,sign,state_in,state_out,decode_opt,micalg, 
		     defcharset, mss, badtype,bodydefcharset)
     mime_t *body; 
     mime_t *sign;
     struct in_state  *state_in;
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     const char *micalg;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t  bodydefcharset;
{
    if (decode_opt->displaying) {	
	struct pager_range *title_range = 
	    state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
					 0);
	
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */
     	
	state_nlputs("\n",state_out);

	/* \n resets this */	
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_nlputs("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, 
			     MeDecodeUnsupportedSig,
			     "Signature %s/%.30s is unsupported, not checked"), 
		     get_major_type_name(sign->TYPE), 
		     get_subtype_name(sign->TYPE));
	state_nlputs(" ]\n",state_out);       

	free_pager_range(&title_range);
    }
    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedSig,
		      "Signature %s/%.30s is unsupported, not checked"),
	      get_major_type_name(sign->TYPE), 
	      get_subtype_name(sign->TYPE));

    mime_decode(body,state_in,state_out, decode_opt, defcharset, mss, badtype,
		bodydefcharset);
}

void null_EC_decoder(init,data,state_in,state_out,decode_opt,defcharset, mss, 
		     badtype,bodydefcharset)
     mime_t *init;
     mime_t *data;
     struct in_state  *state_in;
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
       conversions */
  	
    state_nlputs("\n",state_out);

    /* \n resets this */	
    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
    state_nlputs("[ ",state_out);
    state_printf(state_out,
		 CATGETS(elm_msg_cat, MeSet, 
			 MeDecodeUnsupportedEnc,
			 "Encryption %s/%.30s is unsupported"), 
		 get_major_type_name(init->TYPE), 
		 get_subtype_name(init->TYPE));
    state_nlputs(" ]\n",state_out);
    
    lib_error(CATGETS(elm_msg_cat, MeSet, 
		      MeDecodeUnsupportedEnc,
		      "Encryption %s/%.30s is unsupported"), 
	      get_major_type_name(init->TYPE), 
	      get_subtype_name(init->TYPE));

    free_pager_range(&title_range);    
}

static SG_decoder_t select_SG_decoder P_((const char *protocol));
static SG_decoder_t select_SG_decoder(protocol)
     const char *protocol;
{

    if (use_PGP &&
	0 == istrcmp(protocol,"application/pgp-signature")) {
	DPRINT(Debug,11,(&Debug, 
			 "select_SG_decoder [%s] = pgp_SG_decoder\n",
			 protocol));
	return pgp_SG_decoder;
    }

    DPRINT(Debug,11,(&Debug, 
		     "select_SG_decoder [%s] = null_SG_decoder\n",
		     protocol));
    return null_SG_decoder;
}

/* Prototype */
static void signed_decode P_((mime_t *part, 
			      struct in_state *streamin, 
			      struct out_state *streamout, 
			      const struct decode_opts *decode_opt,
			      charset_t defcharset, 
			      struct header_rec *hdr,
			      type_mismatch_prompt *badtype,
			      charset_t bodydefcharset)); 


/* Returns -1 on failure otherwise same mask than mime_classify_media() */
S_(mime_run_selector signed_selector)
static int signed_selector P_((mime_t *p,
			       struct header_rec * hdr));
static int signed_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = 0;
    mime_t *z;

    int subflag = -1;

    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"signed_selector",
		   "Bad magic number");
    
    if (p->parser_data &&
	mime_parser_subparts(p->parser_data) > 0 &&
	( z = mime_parser_index(p->parser_data,0)) &&
	
	0 <= ( subflag = mime_classify_media(z,hdr,
					     mime_classify_all
					     ) )

	) {

	const char *pv;
	SG_decoder_t sg_decode;
	       
	pv = get_mime_param_compat(p->TYPE_opts,"protocol");

	if (! pv) {
	    DPRINT(Debug,11,(&Debug, 
			     "signed_selector(%p) -- type: %p=%s/%s, flags=%d -- no 'protocol'\n",
			     p,
			     p->TYPE,
			     get_major_type_name(p->TYPE), 
			     get_subtype_name(p->TYPE), 
			     get_type_flags(p->TYPE)));
	    flags |=  NOTPLAIN_need_metamail;
	    goto done;	   	    
	}
	sg_decode = select_SG_decoder(pv);


	if (MIME_selector_magic != p->handler_data->magic)
	    mime_panic(__FILE__,__LINE__,"signed_selector",
		       "Bad magic number (handler_data)");
	
	/* Record selection !! */
	p->handler_data->SG_decoder = sg_decode;

	flags |= subflag;
	
	if (sg_decode != null_SG_decoder) {
	    DPRINT(Debug,11,(&Debug, 
			    "signed_selector(%p) -- type: %p=%s/%s, flags=%d, protocol=%s\n",
			    p,
			    p->TYPE,
			    get_major_type_name(p->TYPE), 
			    get_subtype_name(p->TYPE), 
			    get_type_flags(p->TYPE), 
			    pv));
	    goto done;
	} else if (pagesigned) {
	    DPRINT(Debug,9,(&Debug, 
			    "signed_selector(%p) --  type: %p=%s/%s, flags=%d, protocol=%s, pagesigned=TRUE\n",
			    p,
			    p->TYPE,
			    get_major_type_name(p->TYPE), 
			    get_subtype_name(p->TYPE), 
			    get_type_flags(p->TYPE), 
			    pv));
	    goto done;
	}
    
	
	DPRINT(Debug,11,(&Debug, 
			 "signed_selector(%p) -- type: %p=%s/%s, flags=%d, protocol=%s -- unsupported \n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE), 
			 get_type_flags(p->TYPE), 
			 pv));
	flags |=  NOTPLAIN_need_metamail;
    }
    else
	flags |=  NOTPLAIN_need_metamail;

 done:

    DPRINT(Debug,11,(&Debug, 
		     "signed_selector(%p) = %d %s\n",
		     p,
		     flags,mime_debug_classify_f(flags)));

    return flags;
}

S_(CT_decoder signed_decode)
static void signed_decode P_((mime_t      *part, 
				 struct in_state  *instream, 
				 struct out_state *outstream,
				 const struct decode_opts *decode_opt,
				 charset_t defcharset, 
				 struct header_rec *hdr,
				 type_mismatch_prompt *badtype,
				 charset_t bodydefcharset
				 ));
static void signed_decode (ptr, state_in, state_out, decode_opt,defcharset, 
			   mss, badtype, bodydefcharset)
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    const char *pv;
    const char *micalg;
    SG_decoder_t decode;

    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"signed_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"signed_decode",
		   "signed_decode: unsupported input state");
    }


    if (!ptr->parser_data) {
	const char * T = "multipart/signed";

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			     "Content-Type: %s not parsed"),
		     T);
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			  "Content-Type: %s not parsed"),
		  T);
	goto out;
    }

    
    if (mime_parser_subparts(ptr->parser_data) != 2) {

	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */
	
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeSignedNosubtypes,
			     "Content-Type: multipart/signed, no subtypes"));
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeSignedNosubtypes,
			  "Content-Type: multipart/signed, no subtypes"));
	goto out;
    }
    
    pv = get_mime_param_compat(ptr->TYPE_opts,"protocol");

    if (!pv) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocol,
			     "'protocol' parameter is missing from Multipart/Signed -type!"));
	state_nlputs(" ]\n",state_out);
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocol,
			  "'protocol' parameter is missing from Multipart/Signed -type!"));
	goto out;
    }

    micalg = get_mime_param_compat(ptr->TYPE_opts,"micalg");

    if (!micalg) {
	micalg  = "";
    }

    if (MIME_selector_magic != ptr->handler_data->magic)
	mime_panic(__FILE__,__LINE__,"signed_decode",
		   "Bad magic number (handler_data)");
	
    decode = ptr->handler_data->SG_decoder;

    decode(mime_parser_index(ptr->parser_data,0),
	   mime_parser_index(ptr->parser_data,1),
	   state_in,state_out,
	   decode_opt,
	   micalg, defcharset, 
	   mss, badtype,
	   bodydefcharset
	   );

 out:
    free_pager_range(&title_range);
}


static EC_decoder_t select_EC_decoder P_((const char *protocol));
static EC_decoder_t select_EC_decoder(protocol)
     const char *protocol;
{

    if (use_PGP &&
	0 == istrcmp(protocol,"application/pgp-encrypted")) {
	DPRINT(Debug,11,(&Debug, 
			 "select_EC_decoder [%s] = pgp_EC_decoder\n",
			 protocol));
	return pgp_EC_decoder;
    }

    DPRINT(Debug,11,(&Debug, 
		     "select_EC_decoder [%s] = null_EC_decoder\n",
		     protocol));
    return null_EC_decoder;
}

/* Prototype */
static void encrypted_decode P_((mime_t *part, 
				 struct in_state *streamin, 
				 struct out_state *streamout,
				 const struct decode_opts *decode_opt,
				 charset_t defcharset, 
				 struct header_rec *hdr,
				 type_mismatch_prompt *badtype,
				 charset_t bodydefcharset)); 

/* Returns -1 on failure otherwise same mask than mime_classify_media() */
S_(mime_run_selector encrypted_selector)
static int encrypted_selector P_((mime_t *p,
				  struct header_rec * hdr));
static int encrypted_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = 0;

    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"encrypted_selector",
		   "Bad magic number");
    
    if (p->parser_data &&
	mime_parser_subparts(p->parser_data) > 0) {
	EC_decoder_t ec_decode;
	const char *pv;

	pv = get_mime_param_compat(p->TYPE_opts,"protocol");

	if (!pv) {
	    DPRINT(Debug,11,(&Debug, 
			     "encrypted_selector(%p) -- type: %p=%s/%s, flags=%d -- no 'protocol'\n",
			     p,
			     p->TYPE,
			     get_major_type_name(p->TYPE), 
			     get_subtype_name(p->TYPE), 
			     get_type_flags(p->TYPE)));
	    flags |= NOTPLAIN_need_metamail;
	    goto out;
	}
	ec_decode = select_EC_decoder(pv);

	if (MIME_selector_magic != p->handler_data->magic)
	    mime_panic(__FILE__,__LINE__,"encrypted_selector",
		       "Bad magic number (handler_data)");
	
	/* Record selection !! */
	p->handler_data->EC_decoder = ec_decode;

	if (ec_decode != null_EC_decoder) {
	    DPRINT(Debug,11,(&Debug, 
			     "encrypted_selector(%p) -- type: %p=%s/%s, flags=%d, protocol=%s\n",
			     p,
			     p->TYPE,
			     get_major_type_name(p->TYPE), 
			     get_subtype_name(p->TYPE), 
			     get_type_flags(p->TYPE), 
			     pv));

	    /* TODO:            We really not know on here can be show contents
	                        because we must decode it first to see ....
	    */
	   
	    flags = 0;
	    goto out;
	    
	} 
	
	DPRINT(Debug,11,(&Debug, 
			"encrypted_selector(%p) -- type: %p=%s/%s, flags=%d, protocol=%s\n",
			p,
			p->TYPE,
			get_major_type_name(p->TYPE), 
			get_subtype_name(p->TYPE), 
			get_type_flags(p->TYPE), 
			pv));
	flags |= NOTPLAIN_need_metamail;
    } else
	flags |= NOTPLAIN_need_metamail;

 out:

    DPRINT(Debug,11,(&Debug, 
		     "encrypted_selector(%p)=%d %s\n",
		     p,flags,mime_debug_classify_f(flags)));
    
    return flags;
}

S_(CT_decoder encrypted_decode)
static void encrypted_decode P_((mime_t      *part, 
				 struct in_state  *instream, 
				 struct out_state *outstream,
				 const struct decode_opts *decode_opt,
				 charset_t defcharset, 
				 struct header_rec *hdr,
				 type_mismatch_prompt *badtype,
				 charset_t bodydefcharset
				 ));
static void encrypted_decode (ptr, state_in, state_out, decode_opt,
			      defcharset, mss, 
			      badtype,bodydefcharset)
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    const char *pv;
    EC_decoder_t decode;
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);


    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"encrypted_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"encrypted_decode",
		   "encrypted_decode: unsupported input state");
    }
    
    if (!ptr->parser_data) {
 	const char * T = "multipart/encrypted";

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			     "Content-Type: %s not parsed"),
		     T);
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			  "Content-Type: %s not parsed"),
		  T);
	goto out;
    }

    if (mime_parser_subparts(ptr->parser_data) != 2) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeEncryptedNosubtypes,
			     "Content-Type: multipart/encrypted, no subtypes"));
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeEncryptedNosubtypes,
			  "Content-Type: multipart/encrypted, no subtypes"));
	goto out;
    }
    
    pv = get_mime_param_compat(ptr->TYPE_opts,"protocol");

    if (!pv) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocolEnc,
			     "'protocol' parameter is missing from multipart/encrypted -type!"));
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoProtocolEnc,
			  "'protocol' parameter is missing from multipart/encrypted -type!"));
	goto out;
    }

    if (MIME_selector_magic != ptr->handler_data->magic)
	mime_panic(__FILE__,__LINE__,"encrypted_selector",
		   "Bad magic number (handler_data)");

    decode = ptr->handler_data->EC_decoder;

    decode(mime_parser_index(ptr->parser_data,0),
	   mime_parser_index(ptr->parser_data,1),
	   state_in,state_out,decode_opt,
	   defcharset, mss, badtype, bodydefcharset);

 out:
    free_pager_range(&title_range);    
}


/* related ------------------------------------------------------------------------------ */

/* Prototype */
static void related_decode P_((mime_t *part, 
			       struct in_state *streamin, 
			       struct out_state *streamout, 
			       const struct decode_opts *decode_opt,
			       charset_t defcharset,
			       struct header_rec *hdr,
			       type_mismatch_prompt *badtype,
			       charset_t bodydefcharset)); 

/* Returns -1 on failure otherwise same mask than mime_classify_media() */
S_(mime_run_selector related_selector)
static int related_selector P_((mime_t *p,
				    struct header_rec * hdr));
static int related_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = 0;
    mime_t * z = NULL;
    
    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"related_selector",
		   "Bad magic number");
    
    if (p->parser_data) {
	int parts = mime_parser_subparts(p->parser_data);
		
	if (parts > 0 &&
	    0 <= ( flags = mime_classify_start_page(p,&z,hdr))) {

	    enum pagerelated_v pr = give_dt_enumerate_as_int(&pagerelated);
	    
	    if (MIME_selector_magic != p->handler_data->magic)
		mime_panic(__FILE__,__LINE__,"related_selector",
			   "Bad magic number (handler_data)");

	    /* Record start part */
	    p->handler_data->start_part = z;

	    DPRINT(Debug,11,(&Debug, 
			     "related_selector: pagerelated=%d",pr));
	    switch (pr) {
	    case  pagerelated_no:
		DPRINT(Debug,11,(&Debug, " pagerelated_no"));
		flags |= NOTPLAIN_need_metamail;
		break;
	    case pagerelated_yes:
		DPRINT(Debug,11,(&Debug, 
				 " pagerelated_yes; Allow paging of multpart/related"));
		break;
	    case pagerelated_singlepart:
		DPRINT(Debug,11,(&Debug, " pagerelated_singlepart"));
		if (parts < 2) {
		    DPRINT(Debug,11,(&Debug, 
				     "; %d part, allow paging of multpart/related",
				     parts));		    
		} else {
		    flags |= NOTPLAIN_need_metamail;
		}
		break;
	    case NUM_pagerelated:
		DPRINT(Debug,11,(&Debug, " NUM_pagerelated"));
		break;
	    }
	    DPRINT(Debug,11,(&Debug, "\n"));
	    
	} else {
	    DPRINT(Debug,11,(&Debug, 
			     "related_selector: %d part(s)%s\n",
			     parts, flags < 0 ? ", failure": ""));
	}
	
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "related_selector: No parser data\n"));
	flags |= NOTPLAIN_need_metamail;
    }

    DPRINT(Debug,11,(&Debug, 
		     "related_selector(%p) = %d %s\n",
		     p,
		     flags,mime_debug_classify_f(flags)));
    
    return flags;
}

static void part_info_helper P_((struct out_state *state_out,
				 struct pager_range *title_range,
				 mime_t *att));
static void part_info_helper(state_out,title_range,att)
     struct out_state *state_out;
     struct pager_range *title_range;
     mime_t *att;
{
    char *Encoding = "???";

    if (att->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"part_info_helper",
		   "Bad magic number");
    
    if (att->description &&
	string_len(att->description) > 0) {
	state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
				       MeDecodeDescA,
				       "\"%.60S\" ]\n"),
		     att->description);
	
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
    } else {
	
	enum mime_parameters_v mp = 
	    give_dt_enumerate_as_int(&mime_parameters);
	const struct string *dv;
	const char * dva;
	
	/* 0 == plain
	   1 == encoded
	   2 == plain-and-encoded
	*/
	
	if (mp > mime_parameters_plain &&
	    (dv = get_mime_param(att->DISPOSITION_opts,
				 "filename",
				 mime_rfc1522_filename /* skip compat 
							  paramaters */
				 ))) {
	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					   MeDecodeDescFilenameA,
					   "Filename: %.60S ]\n"),
			 dv);
	    
	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	} else if (mime_rfc1522_filename &&
		   (dv = get_mime_param_rfc1522_hack(att->DISPOSITION_opts,"filename"))) {
	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					   MeDecodeDescFilenameA2,
					   "Filename (=? hack): %.60S ]\n"),
			 dv);
	    
	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	    
	} else if ((dva = get_mime_param_compat(att->DISPOSITION_opts,
						"filename"))) {
	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					   MeDecodeDescFilenameA1,
					   "Filename: %.60s ]\n"),
			 dva);
	    
	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	}		   
    }
    
    
    Encoding = ENCODING(att->encoding);
    
    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
       conversions */
    
    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, MeDecodeDescC,
				   "Type: %.15s/%.30s, Encoding: %s, Size: %d ]\n"),
		 get_major_type_name(att->TYPE), 
		 get_subtype_name(att->TYPE),
		 Encoding, att->length);
    state_nlputs("\n", state_out);
}


S_(CT_decoder related_decode)
static void related_decode P_((mime_t      *part, 
			       struct in_state  *instream, 
			       struct out_state *outstream,
			       const struct decode_opts *decode_opt,
			       charset_t defcharset, 
			       struct header_rec *hdr,
			       type_mismatch_prompt *badtype,
			       charset_t bodydefcharset
			       ));
static void related_decode (ptr, state_in, state_out, decode_opt,
			    defcharset, mss,
			    badtype,bodydefcharset)
     mime_t     *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t  defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    mime_t * start = NULL;
    int count = 0;
    
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"related_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"related_decode",
		   "related__decode: unsupported input state");
    }

    if (!ptr->parser_data) {
	const char * T = "multipart/related";

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			     "Content-Type: %s not parsed"),
		     T);
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			  "Content-Type: %s not parsed"),
		  T);

	goto out;
    }

    if ( (count = mime_parser_subparts(ptr->parser_data)) < 1) {
	const char * T = "multipart/related";
	
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNosubtypes,
			     "Content-Type: %s, no subtypes"),
		     T);
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNosubtypes,
			  "Content-Type: %s, no subtypes"));
	goto out;
    }

    if (MIME_selector_magic != ptr->handler_data->magic)
	mime_panic(__FILE__,__LINE__,"related_decode",
		   "Bad magic number (handler_data)");
    
    start =  ptr->handler_data->start_part;

    if (!start) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ", state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeCtNoStart,
			     "Content-Type: multipart/related, start part not found"));
    
	state_nlputs(" ]\n", state_out);
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeCtNoStart,
			  "Content-Type: multipart/related, start part not found"));	

    }

    if (decode_opt->displaying) {
	int msg = 0;
	int i;

	set_out_state_line_mode(state_out,0,NULL,0);  /* Add newline if not beginning
							 of line 
						      */
	
	/* Treat related parts as attachments */
	
	for (i = 0; i < count; i++) {
	    int nattach = i+1;
	    mime_t *att        = mime_parser_index(ptr->parser_data,i);
	    
	    if (att->magic != MIME_magic)
		mime_panic(__FILE__,__LINE__,"related_decode",
			   "Bad magic number (subpart)");

	    if (att == start) {

		if (count > 1) {
		    /* \n resets this */
		    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		    
		    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
						   MeDecodeStartPart, 
						   "[ Start part #%d/%d: "),
				 nattach,count);
		    
		    part_info_helper(state_out,title_range,att);
		}

		mime_decode (att, state_in, state_out, decode_opt, defcharset, mss, badtype,
			     bodydefcharset);

		set_out_state_line_mode(state_out,0,NULL,0);  /* Add newline if not beginning
								 of line 
							      */
		if (i < count-1)
		    state_nlputs("\n", state_out);
	   		
	    } else {

		if (!msg && start) {
		    		    
		    /* \n resets this */
		    set_out_state_line_mode(state_out,pg_UNDERLINE|pg_BOLD,title_range,
					    1 /* Newline */);
		    
		    state_printf(state_out,
				 CATGETS(elm_msg_cat, MeSet, 
					 MeDecodeRelatedFor,
					 "Related parts for %s/%s:\n"),
				 get_major_type_name(start->TYPE), 
				 get_subtype_name(start->TYPE));

		    msg = 1;
		    state_nlputs("\n", state_out);
		}
		
		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					   MeDecodeRelatedA, 
					       "[ Related #%d/%d: "),
			     nattach,count);
	    
		part_info_helper(state_out,title_range,att);		
	    }
	    
	    
	}
    
    } else if (start) {

	mime_decode (start, state_in, state_out, decode_opt, defcharset, mss, badtype,
		     bodydefcharset);

    } 
    
 out:
    free_pager_range(&title_range);
   
}

/* alternative -------------------------------------------------------------------------- */

/* Prototype */
static void alternative_decode P_((mime_t *part, 
				   struct in_state *streamin, 
				   struct out_state *streamout, 
				   const struct decode_opts *decode_opt,
				   charset_t defcharset,
				   struct header_rec *hdr,
				   type_mismatch_prompt *badtype,
				   charset_t bodydefcharset)); 

/* Returns -1 on failure otherwise same mask than mime_classify_media() */
S_(mime_run_selector alternative_selector)
static int alternative_selector P_((mime_t *p,
				    struct header_rec * hdr));
static int alternative_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = 0;
    mime_t * z = NULL;

    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"alternative_selector",
		   "Bad magic number");
    
    if (p->parser_data &&
	mime_parser_subparts(p->parser_data) > 0 &&
	0 <= ( flags = mime_classify_best_alternative(p,&z,hdr))) {


	if (MIME_selector_magic != p->handler_data->magic)
	    mime_panic(__FILE__,__LINE__,"alternative_selector",
		       "Bad magic number (handler_data)");
	
	/* Record selection !! */
	p->handler_data->selected_alternative = z;

	/* If pagealternative is not set, we must able to handle
	   all parts ... 
	*/
	if (!pagealternative) {
	    DPRINT(Debug,11,(&Debug, 
			     "alternative_selector:  pagealternative is not set, we must able to handle all parts ...\n"));
	    flags = mime_classify_subparts(p,hdr);
	}

	DPRINT(Debug,11,(&Debug, 
			 "alternative_selector(%p) -- type: %p=%s/%s, "
			 "best_alternative: %p=%s/%s\n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE),
			 z->TYPE,
			 get_major_type_name(z->TYPE), 
			 get_subtype_name(z->TYPE)));
	goto out;
    } else	
	flags |= NOTPLAIN_need_metamail;
    
 out:
    
    DPRINT(Debug,11,(&Debug, 
		     "alternative_selector(%p)=%d %s\n",
		     p,flags,mime_debug_classify_f(flags)));
    
    return flags;
}

S_(CT_decoder alternative_decode)
static void alternative_decode P_((mime_t      *part, 
				   struct in_state  *instream, 
				   struct out_state *outstream,
				   const struct decode_opts *decode_opt,
				   charset_t defcharset, 
				   struct header_rec *hdr,
				   type_mismatch_prompt *badtype,
				   charset_t bodydefcharset
				   ));

static void alternative_decode (ptr, state_in, state_out, decode_opt,
				defcharset, mss,
				badtype,bodydefcharset)
     mime_t     *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t  defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    mime_t *best = NULL;
    
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"alternative_decode",
		   "Bad magic number");
    
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"alternative_decode",
		   "alternative_decode: unsupported input state");
    }
    
    if (!ptr->parser_data) {
	const char * T = "multipart/alternative";

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			     "Content-Type: %s not parsed"),
		     T);
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			  "Content-Type: %s not parsed"),
		  T);

	goto out;
    }

    if ( mime_parser_subparts(ptr->parser_data) < 1) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeNosubtypes,
			     "Content-Type: multipart/alternative, no subtypes"));
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeNosubtypes,
			  "Content-Type: multipart/alternative, no subtypes"));
	goto out;
    }


    if (MIME_selector_magic != ptr->handler_data->magic)
	mime_panic(__FILE__,__LINE__,"alternative_decode",
		   "Bad magic number (handler_data)");
    
    best = ptr->handler_data->selected_alternative; 

    if (best) {


	if (best->magic != MIME_magic)
	    mime_panic(__FILE__,__LINE__,"alternative_decode",
		       "Bad magic number (selected alternative)");
	
	DPRINT(Debug,11,(&Debug, 
			 "multipart_alternative: Type: %.15s/%.30s, Encoding: %d, Size: %ld\n",
			 get_major_type_name(best->TYPE), 
			 get_subtype_name(best->TYPE),
			 best->encoding, (long) best->length));
	
	if (in_state_fseek(state_in,best->offset) != 0) {
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeSeekFailed,
				 "multipart_alternative: seek failed"));
	    state_nlputs(" ]\n",state_out);

	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeAlternativeSeekFailed,
			      "multipart_alternative: seek failed"));
	    goto out;
	}


	if (decode_opt->displaying) {
	    char *Encoding = "???";

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    switch (best->disposition) { 
	    case DISP_ATTACH:
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeSelAttachA, 
					       "[ Selecting attach: "));
		break;
	    default:
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeSelPartA,
					       "[ Selecting part: "));
	    }

	    if (best->description &&
		string_len(best->description) > 0) {
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeDescA,
					       "\"%.60S\" ]\n"),
			     best->description);

		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);

		state_puts("[ ",state_out);
	    } else {

		enum mime_parameters_v mp = 
		    give_dt_enumerate_as_int(&mime_parameters);
		const struct string *dv;
		const char * dva;

		/* 0 == plain
		   1 == encoded
		   2 == plain-and-encoded
		*/
		
		
		
		if (mp > mime_parameters_plain &&
		    (dv = get_mime_param(best->DISPOSITION_opts,"filename",
					 mime_rfc1522_filename /* skip compat 
								  paramaters */
					 ))) {
		    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
						   MeDecodeDescFilenameA,
						   "Filename: %.60S ]\n"),
				 dv);

		    /* \n resets this */
		    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		    state_puts("[ ",state_out);
		} else if (mime_rfc1522_filename &&
			   (dv = get_mime_param_rfc1522_hack(best->DISPOSITION_opts,"filename"))) {
		    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
						   MeDecodeDescFilenameA2,
						   "Filename (=? hack): %.60S ]\n"),
				 dv);

		    /* \n resets this */
		    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		    state_puts("[ ",state_out);
		} else if ((dva = get_mime_param_compat(best->DISPOSITION_opts,
							"filename"))) {
		    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
						   MeDecodeDescFilenameA1,
						   "Filename: %.60s ]\n"),
				 dva);

		    /* \n resets this */
		    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		    state_puts("[ ",state_out);
		}
	    }

	    Encoding = ENCODING(best->encoding);
	    
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */

	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, MeDecodeDescC,
					   "Type: %.15s/%.30s, Encoding: %s, Size: %d ]\n"),
			 get_major_type_name(best->TYPE), 
			 get_subtype_name(best->TYPE),
			 Encoding, best->length);
	    state_nlputs("\n", state_out);

	} else {	    
	    if (best->description) {
		/* state_nlputs or state_printf is needed for EOLN_is_CRLF
		   conversions */

		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodeContentDescA,
					       "Content-Description: %.300S\n"),
			     best->description);
		state_nlputs("\n", state_out);
	    } 
	}

	mime_decode (best, state_in, state_out, decode_opt, defcharset, mss, badtype,
		     bodydefcharset);
	
    } else {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ", state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeCtNoAlternative,
			     "Content-Type: multipart/alternative, alternative not found"));
    
	state_nlputs(" ]\n", state_out);
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeCtNoAlternative,
			  "Content-Type: multipart/alternative, alternative not found"));
    }

 out:
    free_pager_range(&title_range);
}


/* multipart ---------------------------------------------------------------------------- */

/* Prototype */
static void multipart_0_decode P_((mime_t *part, 
				   struct in_state *streamin, 
				   struct out_state *streamout,
				   const struct decode_opts *decode_opt,
				   charset_t defcharset, 
				   struct header_rec *hdr,
				   type_mismatch_prompt *badtype,
				   charset_t bodydefcharset)); 

/* Returns -1 on failure otherwise same mask than mime_classify_media() */
S_(mime_run_selector multipart_selector)
static int multipart_selector P_((mime_t *p,
				  struct header_rec * hdr));
static int multipart_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = 0;
    int Flags = 0;
    int subflag = -1;

    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"multipart_selector",
		   "Bad magic number");

    Flags = get_type_flags(p->TYPE);
    
    if ((0 <= ( subflag = mime_classify_subparts(p,hdr))) &&
	((Flags & MIME_MIXED) || 
	 (Flags & MIME_DIGEST)) ) {

	DPRINT(Debug,11,(&Debug, 
			"multipart_selector(%p) -- type: %p=%s/%s, flags=%d\n",
			p,
			p->TYPE,
			get_major_type_name(p->TYPE), 
			get_subtype_name(p->TYPE), 
			get_type_flags(p->TYPE)));

	if (pagemultipart && 0 != ( subflag & NOTPLAIN_need_metamail)) {
	    flags |= subflag & ~NOTPLAIN_need_metamail;

	    DPRINT(Debug,11,(&Debug, 
			     "multipart_selector(%p)   (pagemultipart=TRUE)\n",
			     p));

	} else
	    flags |= subflag;
	goto done;
    } else if (get_major_type_code(p->TYPE) == MIME_TYPE_MULTIPART && 
	       pagemultipart) {
      	DPRINT(Debug,11,(&Debug, 
			 "multipart_selector(%p) -- type: %p=%s/%s "
			 "(pagemultipart=TRUE)\n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE)));
	
	
	if (subflag > 0) {
	    /* Copy	       
	       NOTPLAIN_need_mailcap            0x02
	       NOTPLAIN_canuse_mailcap          0x04
	    */
	    flags |= subflag & ~NOTPLAIN_need_metamail;
	}

	goto done;
    } else {
      	DPRINT(Debug,11,(&Debug, 
			 "multipart_selector(%p) -- type: %p=%s/%s -- need metamail\n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE)));
	
	flags |=  NOTPLAIN_need_metamail;
    }

 done:

    DPRINT(Debug,11,(&Debug, 
		     "multipart_selector(%p) = %d %s\n",
		     p,flags,mime_debug_classify_f(flags)));

    return flags;
}

static int may_group P_((mime_t *ptr));
static int may_group(ptr)
     mime_t *ptr;
{
    const struct string *dv;
    const char * dva;

    if (ptr->disposition != DISP_INLINE)
	return 0;

    if (ptr->description &&
	string_len(ptr->description) > 0)
	return 0;

    dv = get_mime_param(ptr->DISPOSITION_opts,"filename",0);
    if (dv)
	return 0;

    dva = get_mime_param_compat(ptr->DISPOSITION_opts,"filename");
    if (dva)
	return 0;

    if (get_major_type_code(ptr->TYPE) != MIME_TYPE_TEXT ||
	0 != istrcmp(get_subtype_name(ptr->TYPE),"plain"))
	return 0;
    
    if (0 != (ptr->mime_flags & NOTPLAIN_is_fallback) ||
	0 != (ptr->mime_flags & NOTPLAIN_need_metamail))
	return 0;   /* Unsupported charset */
    
    return 1;
}

static void multipart_one_decode  P_((mime_t *att,
				      struct in_state *state_in,
				      struct out_state *state_out,
				      const struct decode_opts *decode_opt,
				      int nattach, int count,
				      charset_t defcharset,
				      struct header_rec *mss,
				      type_mismatch_prompt *badtype,
				      charset_t bodydefcharset));

static void multipart_one_decode(att,state_in,state_out,decode_opt,
				 nattach,count,
				 defcharset,mss,badtype,bodydefcharset)
     mime_t *att;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     int nattach;
     int count;     
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    if (in_state_fseek(state_in,att->offset) != 0) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartSeekFail,
			     "multipart_decode: seek failed"));
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartSeekFail,
			  "multipart_decode: seek failed"));
	goto out;
    }

    if (nattach > 1) {
	set_out_state_line_mode(state_out,0,NULL,0);  /* Add newline if not beginning
							 of line 
						      */

	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */
	
	state_nlputs("\n", state_out);
    }
    
    if (decode_opt->displaying) {	
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	
	switch (att->disposition) { 
	case DISP_ATTACH:
	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					   MeDecodeAttachA, 
					   "[ Attach #%d/%d: "),
			 nattach,count);
	    break;
	default:
	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					   MeDecodePartA,
					   "[ Part #%d/%d: "),
			 nattach,count);
	}

	part_info_helper(state_out,title_range,att);
	
	
    } else {	    
	if (att->description) {
	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */
	    state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					   MeDecodeContentDescA,
					   "Content-Description: %.300S\n"),
			 att->description);
	    state_nlputs("\n", state_out);
	} 
    }
    
    mime_decode (att, state_in, state_out, decode_opt, defcharset, mss, badtype,
		 bodydefcharset);

 out:
    free_pager_range(&title_range);

}


S_(CT_decoder multipart_mixed_decode)
static void multipart_mixed_decode P_((mime_t      *part, 
				       struct in_state  *instream, 
				       struct out_state *outstream,
				       const struct decode_opts *decode_opt,
				       charset_t defcharset, 
				       struct header_rec *hdr,
				       type_mismatch_prompt *badtype,
				       charset_t bodydefcharset
				       ));
static void multipart_mixed_decode P_((mime_t *ptr,
				       struct in_state *state_in,
				       struct out_state *state_out,
				       const struct decode_opts *decode_opt,
				       charset_t defcharset,
				       struct header_rec *mss,
				       type_mismatch_prompt *badtype,
				       charset_t bodydefcharset));
static void multipart_mixed_decode (ptr, state_in, state_out, decode_opt,
				    defcharset, mss,
				    badtype,bodydefcharset)
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    int count;
    int outer, inner = 0;

    /* Group text/plain parts together ... 

       Note that mime parts do not necessary have ending CRLF.
       That also allow changing charset on middle of line.

       RFC 2046 says:

       NOTE:  The CRLF preceding the boundary delimiter line is conceptually
       attached to the boundary so that it is possible to have a part that
       does not end with a CRLF (line  break).  Body parts that must be
       considered to end with line breaks, therefore, must have two CRLFs
       preceding the boundary delimiter line, the first of which is part of
       the preceding body part, and the second of which is part of the
       encapsulation boundary.

    */
    
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);


    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"multipart_mixed_decode",
		   "Bad magic number");

    if (!ptr->parser_data) {
	const char * T = "multipart/mixed";

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			     "Content-Type: %s not parsed"),
		     T);
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			  "Content-Type: %s not parsed"),
		  T);

	goto out;
    }

    if (!in_state_seekable(state_in)) 
	mime_panic(__FILE__,__LINE__,"multipart_mixed_decode",
		   "unsupported input state");
    
    count = mime_parser_subparts(ptr->parser_data);
	
    
    for (outer = 0; outer < count; outer=inner) {
	mime_t *part_outer    = mime_parser_index(ptr->parser_data,outer);

	const int  nattach = outer+1;

	if (! may_group(part_outer)) {

	one_part:

	    if (part_outer->magic != MIME_magic)
		mime_panic(__FILE__,__LINE__,"multipart_mixed_decode",
			   "Bad magic number (subpart)");
	    
	    DPRINT(Debug,11,(&Debug, 
			     "multipart_mixed_decode: [%d]: Type: %p (%s/%s), Encoding: %d, Size: %ld\n",
			     nattach, 
			     part_outer->TYPE,
			     get_major_type_name(part_outer->TYPE),
			     get_subtype_name(part_outer->TYPE),
			     part_outer->encoding, (long) part_outer->length));
	    	    
	    multipart_one_decode(part_outer,state_in,state_out,decode_opt,
				 nattach,count,
				 defcharset, mss, badtype,bodydefcharset);
	    
	    inner = outer+1;
	} else {
	    int scan;
	    int len = 0;

	    for (scan = outer; scan < count; scan++) {
		mime_t *part_scan    = mime_parser_index(ptr->parser_data,scan);
		if (! may_group(part_scan))
		    break;
		
		len +=  part_scan->length;
	    }

	    if (scan == nattach)
		goto one_part;

	    if (nattach > 1) {
		set_out_state_line_mode(state_out,0,NULL,0);  /* Add newline if not beginning
								 of line 
							      */

		/* state_nlputs or state_printf is needed for EOLN_is_CRLF
		   conversions */
		
		state_nlputs("\n", state_out);
	    }
		
	    if (decode_opt->displaying) {
		
		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		state_printf(state_out,CATGETS(elm_msg_cat, MeSet, 
					       MeDecodePartsDesc,
					       "[ Parts #%d-%d/%d: Type: text/plain, Size: %d ]\n"),
			     nattach,scan,count,len);

		state_nlputs("\n", state_out);
	    }


	    for (inner = outer; inner < scan; inner++) {
		mime_t *part_inner    = mime_parser_index(ptr->parser_data,inner);

		if (part_inner->magic != MIME_magic)
		    mime_panic(__FILE__,__LINE__,"multipart_mixed_decode",
			       "Bad magic number (subpart)");
	    
		DPRINT(Debug,11,(&Debug, 
				 "multipart_mixed_decode: [%d]: Type: %p (%s/%s), Encoding: %d, Size: %ld\n",
				 nattach, 
				 part_inner->TYPE,
				 get_major_type_name(part_inner->TYPE),
				 get_subtype_name(part_inner->TYPE),
				 part_inner->encoding, (long) part_inner->length));

		mime_decode (part_inner, state_in, state_out, decode_opt,
			     defcharset, mss, badtype,bodydefcharset);
	    }
	}
    }

 out:
    free_pager_range(&title_range);
}


S_(CT_decoder multipart_0_decode)
static void multipart_0_decode P_((mime_t      *part, 
				   struct in_state  *instream, 
				   struct out_state *outstream,
				   const struct decode_opts *decode_opt,
				   charset_t defcharset, 
				   struct header_rec *hdr,
				   type_mismatch_prompt *badtype,
				   charset_t bodydefcharset
				   ));
static void multipart_0_decode (ptr, state_in, state_out, decode_opt,
				defcharset, mss, badtype,bodydefcharset)
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    int count;
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"multipart_0_decode",
		   "Bad magic number");

    if (!ptr->parser_data) {
	const char * T = "multipart/*";

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			     "Content-Type: %s not parsed"),
		     T);
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			  "Content-Type: %s not parsed"),
		  T);

	goto out;
    }

    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"multipart_0_decode",
		   "unsupported input state");
    }

    count = mime_parser_subparts(ptr->parser_data);
    if (!count) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);	
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartNoparts,
			     "Content-Type: multipart/*, no subparts"));
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeMultipartNoparts,
			  "Content-Type: multipart/*, no subparts"));
    } else {
	int i;

	for (i = 0; i < count; i++) {
	    mime_t *att        = mime_parser_index(ptr->parser_data,i);
	    const int  nattach = i+1;
	    
	    if (att->magic != MIME_magic)
		mime_panic(__FILE__,__LINE__,"multipart_0_decode",
			   "Bad magic number (subpart)");
	    
	    DPRINT(Debug,11,(&Debug, 
			     "multipart_0_decode: [%d]: Type: %p (%s/%s), Encoding: %d, Size: %ld\n",
			     nattach, 
			     att->TYPE,
			     get_major_type_name(att->TYPE),
			     get_subtype_name(att->TYPE),
			     att->encoding, (long) att->length));
	    	    
	    multipart_one_decode(att,state_in,state_out,decode_opt, nattach,count,
				 defcharset, mss, badtype,bodydefcharset);	
	}
    }

 out:
    free_pager_range(&title_range);

}

/* rfc822 -------------------------------------------------------------------------------------- */

/* Prototype */
static void rfc822_decode P_((mime_t *part, 
			      struct in_state *instream, 
			      struct out_state *outstream, 
			      const struct decode_opts *decode_opt,
			      charset_t defcharset,
			      struct header_rec *hdr,
			      type_mismatch_prompt *badtype,
			      charset_t bodydefcharset)); 



int rfc822_header_filter(hdr,flag, have_title)
     header_list_ptr hdr;
     int flag;                  /* elm_filter */
     int have_title;
{
    char buf[80];
    
    strfcpy(buf,give_header_name(hdr->header_name),78);
    strfcat(buf,": ", sizeof buf);
    
    if (flag) {
	if (matches_weedlist (buf, have_title)) 
	    return 0;
	else
	    return 1;
    } else
	return 1;
}

static void nomime_decode P_((mime_t *ptr,
			      struct in_state *state_in,
			      struct out_state *state_out,
			      const struct decode_opts *decode_opt,
			      charset_t bodydefcharset
			      ));
static void nomime_decode (ptr, state_in, state_out, decode_opt,
			   bodydefcharset
			   )
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t bodydefcharset ;
{
    const charset_t * Dv = get_out_state_charset_vector(state_out);
    int j;


    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"nomime_decode",
		   "Bad magic number");
    
    DPRINT(Debug,11,(&Debug, 
		     "nomime_decode -> state: offset=%ld, length=%ld -- ",
		     (long) ptr -> offset, (long) ptr -> length));
    for (j = 0; Dv[j]; j++) {
	const char * MIME_name UNUSED_VAROK = 
	    get_charset_MIME_name(Dv[j]);

	DPRINT(Debug,11,(&Debug, 
			 "'display charset[%d]'=%s ",j,
			 MIME_name ? MIME_name : "<no MIME name>"));
    }
    DPRINT(Debug,11,(&Debug, 
		     "\n"));
        
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"nomime_decode",
		   "nomime_decode: unsupported input state");
    }
    
    
    if (in_state_fseek(state_in,ptr->offset) != 0) {
	struct pager_range *title_range = 
	    state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
					 0);

	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeNoMimeSeekFailed,
			     "nomime_decode: seek failed"));
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoMimeSeekFailed,
			  "nomime_decode: seek failed"));

	free_pager_range(&title_range);
	return;
    }

    if (bodydefcharset)
	set_out_state_filter(state_out,bodydefcharset);
    else
	/* default-nomime-charset gives assumed charset */    
	set_out_state_filter(state_out,default_nomime_charset);

    /* No MIME is just text ... */
    Xbit_decode (state_in, state_out, decode_opt, ptr->length, 1);
    
    DPRINT(Debug,11,(&Debug, 
		     "nomime_decode <- END\n"));


}

/* Returns -1 on failure otherwise same mask than mime_classify_media() */
S_(mime_run_selector rfc822_selector)
static int rfc822_selector P_((mime_t *p,
			       struct header_rec * hdr));
static int rfc822_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = 0;
    int subflag = -1;

    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"rfc822_selector",
		   "Bad magic number");

    if (0 <= ( subflag = mime_classify_subparts(p,
						hdr /* Correct? */
						))) {
	DPRINT(Debug,11,(&Debug, 
			 "rfc822_selector(%p) -- type: %p=%s/%s\n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE)));
	flags |= subflag;
	goto done;
    } else 
	flags |=  NOTPLAIN_need_metamail;


 done:

    DPRINT(Debug,11,(&Debug, 
		     "rfc822_selector(%p) = %d %s\n",
		     p,flags,mime_debug_classify_f(flags)));

    return flags;
}


S_(CT_decoder rfc822_decode)
static void rfc822_decode P_((mime_t      *part, 
			      struct in_state  *instream, 
			      struct out_state *outstream,
			      const struct decode_opts *decode_opt,
			      charset_t defcharset, 
			      struct header_rec *hdr,
			      type_mismatch_prompt *badtype,
			      charset_t bodydefcharset
			      ));
static void rfc822_decode (mt, state_in, state_out, decode_opt, 
			   defcharset, mss, badtype,
			   bodydefcharset)
     mime_t *mt;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    /* This looks a lot like multipart_decode() except that we want to print
     * the headers for the message.  NOTE: "mt" should be a pointer to the
     * RFC822 attachment, NOT its subpart, since we need the offset in order
     * to print out the headers.
     */

    header_list_ptr headers = NULL;
    header_list_ptr From_header, mime_version, osv;
    int decode_headers = 1;
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    int have_title = 0;
    charset_t local_defcharset = NULL;
    charset_t local_bodydefcharset = NULL;
    charset_t local_subjectcharset = NULL;
    
    if (mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"rfc822_decode",
		   "Bad magic number");
  
    if (!in_state_seekable(state_in)) {
	mime_panic(__FILE__,__LINE__,"rfc822_decode",
		   "rfc822_decode: unsupported input state");
    }

  if (in_state_fseek(state_in,mt->offset) != 0) {
      /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	 conversions */

      /* \n resets this */
      set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
      state_puts("[ ",state_out);
      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeRFC822SeekFailed,
			   "rfc822_decode: seek failed"));
      state_nlputs(" ]\n",state_out);
      lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeRFC822SeekFailed,
			"rfc822_decode: seek failed"));
      goto out;
  }

  if (!mt->parser_data) {
      	const char * T = "message/rfc822";

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_puts("[ ",state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			     "Content-Type: %s not parsed"),
		     T);
	state_nlputs(" ]\n",state_out);

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeTypeNotparsed,
			  "Content-Type: %s not parsed"),
		  T);

	goto out;
  }


  headers = state_read_headers(state_in, RHL_MARK_FOLDING);

  mime_version = locate_header_by_name(headers,"MIME-Version");
  osv          = locate_header_by_name(headers,"X-ELM-OSV");

  if (!mime_version && !req_mime_hdrencoding && !req_mime_hdrencoding)
      decode_headers = 0;
  if (osv && osv->body) {
      char value[20];
      if (mime_get_param("no-hdr-encoding",value,osv->body,
			 sizeof value) && atoi(value) > 0)
	  decode_headers = 0;

      if (mime_get_param("hdr-charset",value,osv->body,
			 sizeof value)) {
	  
	  local_defcharset = MIME_name_to_charset(value,CHARSET_create);

	  {
	      const char * MIME_name UNUSED_VAROK = 
		  get_charset_MIME_name(local_defcharset);
	      DPRINT(Debug,9,(&Debug,"-- header-charset=%s\n",MIME_name ? MIME_name : value));
	  }
      }
  }

  if (use_char_set_field_hack) {
      header_list_ptr char_set_chack = locate_header_by_name(headers,"Char-Set");
      char header_buffer[80];      
      const char * body_value;
      
      /* header_trim_whitespace_helper return either original body or pointer to buffer, 
	 may return NULL on overflow; returns also return NULL if body == NULL

	 Also handles \r and \n
      */
      if (char_set_chack &&
	  (body_value = header_trim_whitespace_helper(char_set_chack->body,
						      header_buffer,
						      sizeof header_buffer))) {
	  
	  charset_t X = MIME_name_to_charset(body_value,0);
	  
	  if (X) {
	      
	      const char * MIME_name UNUSED_VAROK = 
		  get_charset_MIME_name(X);
	      DPRINT(Debug,9,(&Debug,"-- Char-Set: %s\n",MIME_name ? MIME_name : body_value));
	      
	      
	      if (!local_defcharset)
		  local_defcharset = X;

	      if (!local_bodydefcharset)
		  local_bodydefcharset = X;
	      
	  } else {
	      DPRINT(Debug,9,(&Debug,"-- Char-Set: %s --- skipped\n",body_value));
	  }
      }
  }

  if (local_defcharset && local_defcharset != defcharset) {
      
      const char * MIME_name UNUSED_VAROK = 
	  get_charset_MIME_name(local_defcharset);
      if (MIME_name) {
	  DPRINT(Debug,9,(&Debug,"-- changing header defcharset to %s\n",MIME_name));
      }

      defcharset = local_defcharset;
  }

  if (local_bodydefcharset && local_bodydefcharset != bodydefcharset) {
      const char * MIME_name UNUSED_VAROK = 
	  get_charset_MIME_name(local_bodydefcharset);
      if (MIME_name) {
	  DPRINT(Debug,9,(&Debug,"-- changing body defcharset to %s\n",MIME_name));
      }

      bodydefcharset = local_bodydefcharset;
  }

  if (use_ct_cs_for_subj_hack && !local_defcharset) {
      header_list_ptr subjecthdr = locate_header_by_name(headers,"Subject");

      if (subjecthdr && subjecthdr->body && check_8bit_str(subjecthdr->body)) {
	  header_list_ptr ct = locate_header_by_name(headers,
						     "Content-Type");
	  char header_buffer[80];
	  
	  const char * body_value;
	  const char *c;

	  
	  /* header_trim_whitespace_helper return either original body or pointer to buffer, 
	     may return NULL on overflow; returns also return NULL if body == NULL
	     
	     Also handles \r and \n
	  */
	  
	  if (ct &&
	      (body_value = header_trim_whitespace_helper(ct->body,
							  header_buffer,
							  sizeof header_buffer)) &&
	      NULL != (c = strchr (body_value, ';'))) {
	      char value [20];
	      c++;
	      
	      if (mime_get_param("charset",value,c,
				 sizeof value)) {
		  		  
		  local_subjectcharset = MIME_name_to_charset(value,0);
		  if (local_subjectcharset) {
		      const char * MIME_name UNUSED_VAROK = 
			  get_charset_MIME_name(local_subjectcharset);
		      DPRINT(Debug,9,(&Debug,"-- 8-bit subject charset %s\n",MIME_name ? MIME_name : value));
		      
		  } else {
		      DPRINT(Debug,9,(&Debug,"-- skipped 8-bit subject charset %s\n",value));
		  }
	      }
	  }
      }
  }
  
  From_header  = locate_header_by_name(headers,"From");

  if (From_header) {
      struct string * buffer = give_decoded_header(From_header,
						   decode_headers,
						   defcharset);

      /* \n resets this */
      set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeStartMail,
			   "-- Start of included mail From: %S\n"),
		   buffer);

      free_string(&buffer);

      have_title = 1; /* Not completely correct */

  }  else {

      /* \n resets this */
      set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeStartMail1,
			   "-- Start of included mail.\n"));
  }

  /* state_nlputs or state_printf is needed for EOLN_is_CRLF
     conversions */

  state_nlputs("\n",state_out);

  state_write_headers(state_out,
		      decode_opt,
		      headers,
		      rfc822_header_filter,
		      elm_filter,
		      decode_headers,
		      defcharset,
		      have_title,
		      local_subjectcharset
		      );


  delete_headers(&headers); 

  /* state_nlputs or state_printf is needed for EOLN_is_CRLF
     conversions */

  state_nlputs("\n",state_out);

  
  if (mime_parser_subparts(mt->parser_data) != 1) {
      /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	 conversions */

      /* \n resets this */
      set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
      state_puts("[ ",state_out);
      state_printf(state_out,
		   CATGETS(elm_msg_cat, MeSet, MeDecodeNoRFC822,
			   "rfc822_decode: no body of RFC 822 mail"));
      state_nlputs(" ]\n",state_out);

      lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeNoRFC822,
			"rfc822_decode: no body of RFC 822 mail"));
      goto out;

  } else {
      mime_t * part = mime_parser_index(mt->parser_data,0);
  
      if (part->magic != MIME_magic)
	  mime_panic(__FILE__,__LINE__,"rfc822_decode",
		     "Bad magic number (parts)");

      /* TODO: Handle Pre-MIME Content-types ... */
      if (mime_version || !req_mime_bodyencoding)
	  mime_decode (part, state_in, state_out, 
		       decode_opt,
		       defcharset,

		       /* TODO: Is it correct to pass parent's message
			  record??? (mss) */
		       NULL,
		       badtype,
		       bodydefcharset);
      else
	  nomime_decode (part, state_in, state_out, decode_opt,
			 bodydefcharset
			 );
  }

  /* \n resets this */
  set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
  state_printf(state_out,
	       CATGETS(elm_msg_cat, MeSet, MeDecodeEndMail,
		       "-- End of included mail.\n"));

 out:
  free_pager_range(&title_range);
  
}

/* ----------------------------------------------------------------------------------- */

 
/* 1 if is text type (true)
 * 0 if not text type
 * -1 if can't be encoded (ie structured) Message/ or Multpart/
 */

int decoder_is_textual(ptr)
     mime_t *ptr;
{
    int is_text;

    /* 7bit and 8bit are only allowed for line orienteed types */
    if (ptr->encoding == ENCODING_NONE || 
	ptr->encoding == ENCODING_7BIT || 
	ptr->encoding == ENCODING_8BIT) {
	DPRINT(Debug,11,(&Debug,
			 "decoder_is_textual: textual encoding (%s)\n",
			 ENCODING(ptr->encoding)));
	is_text = 1;
    } else
	is_text = give_text_type_code(ptr->TYPE);
    
    DPRINT(Debug,11,(&Debug, 
		     "decoder_is_textual=%d; type=%p (%s/%s); encoding=%s (%d)\n",
		     is_text, 
		     ptr->TYPE,
		     get_major_type_name(ptr->TYPE),
		     get_subtype_name(ptr->TYPE),
		     ENCODING(ptr->encoding),
		     ptr->encoding));   
    
    return is_text;
}


int set_filter (ptr, state, decode_opt, default_content_charset,
		override_charset,res_charset) 
     mime_t *ptr;
     struct out_state *state;
     const struct decode_opts *decode_opt;
     charset_t default_content_charset;
     charset_t override_charset;   /* IF set, ignore charset= -paramater */
     charset_t *res_charset;
{
    int code;
    charset_t tmp;
    int res = 0;

    struct pager_range *title_range = 
	state_add_simple_pager_range(state,NULL,PR_MAX_WIDTH,0,
				     0);

    DPRINT(Debug,40,(&Debug, 
		     "set_filter -> state: offset=%d, length=%ld\n",
		     (long) ptr -> offset, (long) ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"set_filter",
		   "Bad magic number");
    
    set_out_state_filter(state,NULL); /* So that messages can be printed ... */
           

    code = mime_get_charset (&tmp, ptr->TYPE_opts,
			     get_out_state_charset_vector(state),
			     default_content_charset
			     );

    if (res_charset)
	*res_charset = tmp;

    if (override_charset && override_charset != tmp) {

	const char * tmp_MIME_name =  get_charset_MIME_name(tmp);
	const char * override_MIME_name = 
	    get_charset_MIME_name(override_charset);

	/* \n resets this */
	set_out_state_line_mode(state,pg_BOLD,title_range,1 /* Newline */);	
	state_printf(state,
		     CATGETS(elm_msg_cat, MeSet, 
			     MeDecodeCharsetOverride,
				 "[ Charset %.15s ignored, treated as %.15s ]\n"), 
		     tmp_MIME_name      ? tmp_MIME_name      : "<none>",
		     override_MIME_name ? override_MIME_name : "<none>");

	state_nlputs("\n",state);

	set_out_state_filter(state,override_charset);

    } else {
	if (0 == code) {
	    const char * tmp_MIME_name = get_charset_MIME_name(tmp);

	    /* Don't show this part (or copy to reply buffer) ... */	  
	    
	    /* \n resets this */
	    set_out_state_line_mode(state,pg_BOLD,title_range,1 /* Newline */);	
	    state_printf(state,
			 CATGETS(elm_msg_cat, MeSet, 
				 MeDecodeCharsetSkipping,
				 "[ Charset %.15s unsupported, skipping... ]\n"), 
			 tmp_MIME_name ? tmp_MIME_name : "<none>");
	    
	    if (decode_opt->displaying) {
		
		/* \n resets this */
		set_out_state_line_mode(state,pg_BOLD,title_range,1 /* Newline */); 
		state_printf(state,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeUseVtoView,
				     "[ Use 'v' to view or save this part. ]\n"));

		/* \n resets this */
		set_out_state_line_mode(state,pg_BOLD,title_range,1 /* Newline */); 
		state_printf(state,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeUseOtoOverride,
				     "[ You can also use 'O' to override charset information. ]\n"));	    
	    }

	    set_out_state_filter(state,tmp);
	    
	    {
		const char * filter_MIME_name                   UNUSED_VAROK = 
		    get_out_state_f_MIME_name(state);
		charset_t c = get_out_state_filter(state);
		const struct charset_type * filter_charset_type UNUSED_VAROK =
		    get_charset_type(c);


		DPRINT(Debug,40,(&Debug, 
				 "set_filter=0 filter=%p '%s' (type=%p)\n",
				 c,
				 filter_MIME_name ? 
				 filter_MIME_name : 
				 "<none>",
				 filter_charset_type));

	    }
	   
	    res = 0;
	    goto out;

	} else {
	    if (code < 0) {
		const char * tmp_MIME_name =  get_charset_MIME_name(tmp);
			
		DPRINT(Debug,40,(&Debug,
				 "   code=%d charset_convert_ok=%d\n",
				 code,charset_convert_ok));
		
		if (code == -3) {   /* Converted without lossage */
		    if (!charset_convert_ok && 
			(!tmp_MIME_name || 
			 0 != istrcmp(tmp_MIME_name,"US-ASCII"))) {
			
			/* \n resets this */
			set_out_state_line_mode(state,pg_BOLD,title_range,1 /* Newline */); 
			state_printf(state,
				     CATGETS(elm_msg_cat, MeSet, 
					     MeDecodeCharsetConvertNoLoss,
					     "[ Charset %.15s converted... ]\n"),
				     tmp_MIME_name ? tmp_MIME_name : "<none>");
		    }

		    
		} else {
		    /* \n resets this */
		    set_out_state_line_mode(state,pg_BOLD,title_range,1 /* Newline */);	
		    state_printf(state,
				 CATGETS(elm_msg_cat, MeSet, 
					 MeDecodeCharsetConvert,
					 "[ Charset %.15s unsupported, converting... ]\n"),
				 tmp_MIME_name ? tmp_MIME_name : "<none>");
		    
		    if (decode_opt->displaying) {	
			/* state_nlputs or state_printf is needed for EOLN_is_CRLF
			   conversions */
		
			/* \n resets this */
			set_out_state_line_mode(state,pg_BOLD,title_range,1 /* Newline */);
			state_printf(state,
				     CATGETS(elm_msg_cat, MeSet, MeDecodeUseAlsoV,
					     "[ You can also use 'v' to view or save this part. ]\n"));	    
			state_nlputs("\n",state);
		    }
		}
	    }

	    set_out_state_filter(state,tmp);
	}
    }
	
    {
	const char * filter_MIME_name                    UNUSED_VAROK = 
	    get_out_state_f_MIME_name(state);
	charset_t c = get_out_state_filter(state);
	const struct  charset_type * filter_charset_type UNUSED_VAROK =
	    get_charset_type(c);

	DPRINT(Debug,40,(&Debug, 
		     "set_filter=1 filter=%p '%s' (type=%p)\n",
		     c,
		     filter_MIME_name ? filter_MIME_name : "<none>",
		     filter_charset_type));	
    }

    res = 1;
 out:
    free_pager_range(&title_range);

    return res;    
}

static void copy_text P_((struct in_state *state_in, struct out_state *state_out,
			  const struct decode_opts *decode_opt));


/* If format= parameter is given, it is unsupported */
S_(text_decoder_f default_text_decoder)
static void default_text_decoder P_((mime_t                   * body,
				     struct in_state          * state_in,
				     struct out_state         * state_out,
				     const struct decode_opts * decode_opt,
				     charset_t                  text_charset));
static void default_text_decoder(body,state_in,state_out,decode_opt,
				 text_charset) 
     mime_t *body;
     struct in_state   *state_in;
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     charset_t text_charset;
{
    const char *pv = get_mime_param_compat(body->TYPE_opts,"format");

    /*  Also text_unsupported_decode  uses this !!! */
    
    DPRINT(Debug,15,(&Debug,"default_text_decoder: "));
    if (pv) {
	DPRINT(Debug,15,(&Debug,"format=%s",pv));
    }
    DPRINT(Debug,15,(&Debug,"\n"));

    if (pv) {
	struct pager_range *title_range = 
	    state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
					 0);


	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MeDecodeUnsupportedTextFormat,
				 "[ %s/%.30s format %.15s is unsupported, treating like fixed format. ]\n"),
		     get_major_type_name(body->TYPE), 
		     get_subtype_name(body->TYPE),
		     pv);

	free_pager_range(&title_range);
    }

    copy_text(state_in,state_out,decode_opt);
}

S_(text_decoder_f fixed_text_decoder)
static void fixed_text_decoder P_((mime_t *body,
				   struct in_state   *state_in,
				   struct out_state  *state_out,
				   const struct decode_opts *decode_opt,
				   charset_t text_charset));
static void fixed_text_decoder(body,state_in,state_out,decode_opt,
			       text_charset) 
     mime_t *body;
     struct in_state   *state_in;
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     charset_t text_charset;
{
    struct decode_opts  DECODE_OPT = *decode_opt;

    struct pager_range * decode_data = 
	state_add_simple_pager_range(state_out,decode_opt->range,PR_PREFORMAT,
				     0,0);

    DPRINT(Debug,15,(&Debug,
		     "fixed_text_decoder: decode_data quote_level=%d range_flags=%d\n",
		     get_pager_range_quote_level(decode_data),
		     get_pager_range_flags(decode_data)));

    DECODE_OPT.range  =  decode_data;

    copy_text(state_in,state_out,&DECODE_OPT);

    free_pager_range(& decode_data);
}

static TXT_decoder_t select_TXT_decoder P_((const char *text_format));
static TXT_decoder_t select_TXT_decoder(text_format)
    const char * text_format;
{
    if (0 == istrcmp(text_format,"fixed")) {
	DPRINT(Debug,11,(&Debug, 
			 "select_TXT_decoder [%s] = fixed_text_decoder\n",
			 text_format));	
	return fixed_text_decoder;
    }

    if (0 == istrcmp(text_format,"flowed")) {

	DPRINT(Debug,11,(&Debug, 
			 "select_TXT_decoder [%s] = flowed_text_decoder\n",
			 text_format));	
	return flowed_text_decoder;
    }


    DPRINT(Debug,11,(&Debug, 
		     "select_TXT_decoder [%s] = default_text_decoder  (unknown format)\n",
		     text_format));	
    
    return default_text_decoder;
}


/* Prototype */
static void text_decode	P_((mime_t *part, 
			    struct in_state *instream, 
			    struct out_state *outstream, 
			    const struct decode_opts *decode_opt,
			    charset_t defcharset, 
			    struct header_rec *hdr,
			    type_mismatch_prompt *badtype,
			    charset_t bodydefcharset)); 

int text_charset_selector(p,hdr)
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = NOTPLAIN_need_metamail;

    int r;
    charset_t disp_vector[2];
    
    /* If mime_get_charsets < 0, we may want call metamail
     * (instead of filtering to US-ASCII)
     */
    charset_t buf1;

    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"text_charset_selector",
		   "Bad magic number");
    
    /* If charset is overrided, do not use metamail ... */
    if (hdr && hdr->override_charset) {
	const char * override_MIME_name UNUSED_VAROK = 
	    get_charset_MIME_name(hdr->override_charset);

       	DPRINT(Debug,11,(&Debug, 
			 "text_charset_selector(%p) -- override charset %s\n",
			 override_MIME_name ? 
			 override_MIME_name :
			 "<no MIME name>"));
	flags = 0;
	goto done_charset; 
    }

    /* XXXX should based to output buffer's display_charset? */
    disp_vector[0] = display_charset;
    disp_vector[1] = NULL;
	
    if ((r=mime_get_charset(&buf1,p->TYPE_opts,disp_vector,NULL)) > 0) {
	const char * MIME_name UNUSED_VAROK = get_charset_MIME_name(buf1);
	
	DPRINT(Debug,11,(&Debug, 
			 "text_charset_selector(%p) -- type: %p=%s/%s; charset=%s\n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE),
			 MIME_name ? MIME_name : "<no MIME name>"));
	flags = 0;
	goto done_charset;
    } 

    if (r < 0 && buf1 == system_charset) {
	const char * MIME_name UNUSED_VAROK = 
	    get_charset_MIME_name(buf1);
	
	DPRINT(Debug,11,(&Debug, 
			 "text_charset_selector(%p) -- type: %p=%s/%s; charset=%s -- is system charset ...\n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE),
			 MIME_name ? MIME_name : "<no MIME name>"));
	flags = 0;
	goto done_charset;
    }

    if (r == -3 && charset_convert_ok) {
	const char * MIME_name UNUSED_VAROK = 
	    get_charset_MIME_name(buf1);

	DPRINT(Debug,11,(&Debug, 
			 "text_charset_selector(%p) -- type: %p=%s/%s; charset=%s -- charset_convert_ok=TRUE\n",
			p,
			p->TYPE,
			get_major_type_name(p->TYPE), 
			get_subtype_name(p->TYPE),
			MIME_name ? MIME_name : "<no MIME name>"));
	flags = 0;
	goto done_charset;
    } 
    

    if (r < 0 && page_known_charsets) {
	const char * MIME_name UNUSED_VAROK = 
	    get_charset_MIME_name(buf1);
	
	DPRINT(Debug,11,(&Debug, 
			 "text_charset_selector(%p) -- type: %p=%s/%s; charset=%s -- page_known_charsets=TRUE\n",
			 p,
			 p->TYPE,
			 get_major_type_name(p->TYPE), 
			 get_subtype_name(p->TYPE),
			 MIME_name ? MIME_name : "<no MIME name>"));
	flags = 0;
	goto done_charset;
    }

    if (-3 == r)                       /* Is fallback -- is conversion */
	flags = NOTPLAIN_is_fallback;
   
 done_charset:

    return flags;
}

/* Returns -1 on failure otherwise same mask than mime_classify_media() */
S_(mime_run_selector text_selector)
static int text_selector P_((mime_t *p,
			     struct header_rec * hdr));
static int text_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = NOTPLAIN_need_metamail;

    /* If mime_get_charsets < 0, we may want call metamail
     * (instead of filtering to US-ASCII)
     */

    const char *pv;
    TXT_decoder_t txt_decoder;

    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"text_selector",
		   "Bad magic number");
   
    flags = text_charset_selector(p,hdr);
    
    pv = get_mime_param_compat(p->TYPE_opts,"format");

    if (pv) {
	txt_decoder = select_TXT_decoder(pv);
	
	if (default_text_decoder != txt_decoder) {
	    DPRINT(Debug,11,(&Debug, 
			     "text_selector(%p) -- format: %s\n",
			     p,pv));
	    goto done_format;
	    
	}
       

#if 0   /* Do not work very well */
	if (0 == flags)
	    flags = NOTPLAIN_is_fallback;
#endif

    } else {
	DPRINT(Debug,11,(&Debug, 
			 "text_selector(%p) -- no format given\n",p));
	txt_decoder = default_text_decoder;
    }

 done_format:
    if (MIME_selector_magic != p->handler_data->magic)
	mime_panic(__FILE__,__LINE__,"text_selector",
		   "Bad magic number (handler_data)");

    p->handler_data->TXT_decoder = txt_decoder;


    DPRINT(Debug,11,(&Debug, 
		     "text_selector(%p) = %d %s\n",
		     p,
		     flags,mime_debug_classify_f(flags)));

    return flags;
}

/* ------------------------------------------------------------------------------------ */

/* Print total 10 error messages */
int print_in_errors(state_in,state_out,decode_opt)
     struct in_state *state_in;
     struct out_state *state_out;   /* may be NULL */
     const struct decode_opts *decode_opt;
{
    int errors;

    struct pager_range *title_range = NULL;

    if (state_out)
	title_range = state_add_simple_pager_range(state_out,NULL,
						   PR_MAX_WIDTH,0,0);


    
    /* Print total 10 error messages */
    for (errors = 0; errors < 10; errors++) {
	const struct string *X =  in_state_error_message(state_in,
							 1  /* Clear error */);

	if (!X)
	    break;
	
	if ( ( decode_opt->displaying ||
	       decode_opt->show_error )
	     &&
	     state_out) {

	    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
	       conversions */
	    state_nlputs("\n",state_out);

	    if (decode_opt->displaying) {
		/* \n resets this */	
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    }
	    state_nlputs("[ ",state_out);

	    state_printf(state_out,FRM("%S"),X);

	    state_nlputs(" ]\n",state_out);
	}
	
	lib_error(FRM("%S"),X);
    }

    if (title_range)
	free_pager_range(&title_range);

    return errors;
}

static void copy_text(state_in,state_out, decode_opt)
     struct in_state *state_in; 
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
{
    int bytes = 0;
    int addprefix = 1;
    int lines = 0;
    int errors = 0;
    int ch = '\0';

    while (1) {
    
	ch = state_getc(state_in);


    retry:

	if (EOF == ch) {
	    break;
	}
	bytes++;

	if (addprefix) {
	    state_add_prefix(state_out,decode_opt);
	    addprefix = 0;
	    lines++;
	}

	state_putc(ch,state_out);
	if (ch == '\n')
	    addprefix = 1;
    }

    errors = print_in_errors(state_in,state_out,decode_opt);


    DPRINT(Debug,10,(&Debug, 
		     "copy_text: %d bytes, %d lines%s, %d errors\n",
		     bytes,lines,
		     addprefix ? "" : ", no newline at end",
		     errors));

    /* Retry after errors */
    if (errors > 0) {
	ch = state_getc(state_in);

	if (EOF != ch) {
	    bytes = 0;
	    lines = 0;
	    
	    goto retry;
	}
    }
    
}

static void text_decode (ptr, state_in, state_out, decode_opt,
			 defcharset, mss,
			 badtype,bodydefcharset) 
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    charset_t   text_charset; /* Selected charset */

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"text_decode",
		   "Bad magic number");
        
    DPRINT(Debug,11,(&Debug, 			
		     "text_decode -> state: offset=%ld, length=%ld \n",
		     (long) ptr -> offset, (long) ptr -> length));
        
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "              (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }
    
    if (set_filter(ptr,state_out,decode_opt,
		   bodydefcharset ? bodydefcharset : default_mimetext_charset,
		   mss ? mss->override_charset : NULL,
		   &text_charset)) {
	
	struct in_state *decoded_state_in = new_in_state(STATE_in_decode);
	
	if (set_in_state_decode(state_in,decoded_state_in,
				ptr->length,
				STATE_DECODE_is_text,
				ptr->encoding,
				NULL)) {
	    
	    TXT_decoder_t decode_text;
	    
	    if (MIME_selector_magic != ptr->handler_data->magic)
		mime_panic(__FILE__,__LINE__,"text_decode",
			   "Bad magic number (handler_data)");
	    
	    decode_text = ptr->handler_data->TXT_decoder;
	    
	    decode_text(ptr,decoded_state_in,state_out,decode_opt,
			text_charset);
	    
	    
	} else {
	    
	    struct pager_range *title_range = 
		state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
					     0);
	    
	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);	
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeEncodingSkipping,
				 "[ Unsupported encoding, skipping... ]\n"));
	    
	    if (decode_opt->displaying) {
		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */); 
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeUseVEncodedSave,
				     "[ Use 'v' to save this part in encoded form. ]\n"));
	    }
	    
	    free_pager_range(&title_range);
	    
	}
	
	free_in_state(&decoded_state_in);
    }

    set_out_state_filter(state_out,NULL);
    
    DPRINT(Debug,11,(&Debug, 
		     "text_decode <- END; \n"));
    
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "            (file); ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }
}

FILE * arrange_binary(ptr,state_in,state_out,newstate2, name)
     mime_t *ptr;
     struct in_state  *state_in;
     struct out_state *state_out;
     struct in_state *newstate2;
     char **name;
{
    FILE * tmpfp = NULL;
    char *fname = NULL;
    const char *tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);

    DPRINT(Debug,11,(&Debug, 
		     "arrange_binary -> state: offset=%ld, length=%ld \n",
		     (long) ptr -> offset, (long) ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"arrange_binary",
		   "Bad magic number");
    
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "                 (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }
      
    if (!tmp)
	return NULL;

    fname = elm_message(FRM("%selmdecode-%d"), 
			tmp, getpid ());
    if (NULL == (tmpfp = safeopen_rdwr(fname,NULL))) {
	/* state_nlputs or state_printf is needed for EOLN_is_CRLF
	   conversions */

	lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
			  "Failed to create file for decoding."));
	if (state_out) {

	    struct pager_range *title_range = 
                state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
					     0);

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeFailedCreate,
				 "Failed to create file for decoding."));
	    state_nlputs(" ]\n",state_out);

	    free_pager_range(&title_range);

	}
    } else { /* Tempfile opened */
	int c = EOF, prev = EOF;
	int is_binary = ptr->encoding == ENCODING_BINARY;
	long pos;

	if (name)
	    *name = fname;
	else
	    unlink(fname);  /* We can unlink it now ... */

	DPRINT(Debug,11,(&Debug, 
			 "                 : is_binary = %d\n",is_binary));

	if (in_state_fseek(state_in,ptr->begin_offset) != 0) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDecodeArrangeSeekFail,
			      "arrange_binary: seek failed"));
	    return NULL;
	}

	while ((pos = in_state_ftell(state_in)) < 
	       ptr -> offset + ptr ->length &&
	       EOF != (c = state_getc(state_in))) {

	    if (pos < ptr -> offset) {   /*  --- HEADER PART -- */
		if ('\n' == c && '\r' != prev) {
		    if (is_binary) {
			DPRINT(Debug,11,(&Debug, 
					 "arrange_binary -- (header) not really binary input!\n")); 
			is_binary = 0;
		    }
		    fputc('\r',tmpfp);
		}		
	    } else if (!is_binary && '\n' == c && '\r' != prev) {
		fputc('\r',tmpfp);
	    }
	    fputc(c,tmpfp);
	    prev = c;
	}
	rewind(tmpfp); /* Rewind it for reading */	    
	set_in_state_file(tmpfp,newstate2);
    }

    if (!name)
	free(fname);
    return tmpfp;
}

/* Prototype */
static void elm_decode	P_((mime_t *part, 
			    struct in_state *instream, 
			    struct out_state *outstream, 
			    const struct decode_opts *decode_opt,
			    charset_t defcharset, struct header_rec *hdr,
			    type_mismatch_prompt *badtype,
			    charset_t defbodycharset)); 

static void elm_decode (ptr, state_in, state_out, decode_opt,defcharset, mss, badtype,
			defbodycharset) 
     mime_t *ptr;
     struct in_state  *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t defbodycharset;
{  
    struct in_state * newstate2 = new_in_state(STATE_in_decode);
    char buffer[LONG_STRING];
  
    int flags = STATE_DECODE_buffer;  /* check_type_magic() requires buffering */
    int errors = 0;

    DPRINT(Debug,11,(&Debug, 
		   "elm_decode -> state: offset=%ld, length=%ld; \n",
		   (long) ptr -> offset, (long) ptr -> length));

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"elm_decode",
		   "Bad magic number");

    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "             (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }

    if (set_in_state_decode_helper(ptr,state_in,newstate2,flags,NULL)) {

	if (check_type_pattern) {

	    if (!check_type_magic(ptr,newstate2, state_out, decode_opt,
				  badtype)) {
		DPRINT(Debug,11,(&Debug, 
				 "elm_decode: mime_type_rejected\n"));
		goto FAILTYPE;
	    }

	    /* Print total 10 error messages */
	    errors = print_in_errors(newstate2,state_out,decode_opt);
	}
	
	if (set_filter(ptr,state_out,decode_opt,
		       defbodycharset ? defbodycharset :
		       NULL /* Not  default_mimetext_charset? */,
		       mss ? mss->override_charset : NULL,
		       NULL)) {
	    int len, crypted = 0;
	    int count = 0;

	    struct pager_range *title_range = 
		state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
					     0);


	    getkey(OFF);
      
	    while((len = state_getl(buffer,sizeof(buffer),newstate2)) > 0) {
		count += len;
	      
		if (!strncmp(buffer, START_ENCODE, strlen(START_ENCODE))) {

		    /* \n resets this */	
		    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		    
		    state_printf(state_out,
				 CATGETS(elm_msg_cat, MeSet, MeDecodeStartElmEncoded,
					 "-- Start of (Elm) encoded section.\n"));
		    crypted = ON;
		    continue;
		} else if (!strncmp(buffer, END_ENCODE, strlen(END_ENCODE))) {
		    crypted = OFF;
		    
		    /* \n resets this */	
		    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		    state_printf(state_out,
				 CATGETS(elm_msg_cat, MeSet, MeDecodeEndElmEncoded,
					 "-- End of (Elm) encoded section.\n"));
		    continue;
		} else if (crypted) {
		    no_ret(buffer);
		    encode(buffer);      
		    if (out_state_EOLN_is_CRLF(state_out))
			strfcat(buffer, "\r\n", sizeof buffer);
		    else
			strfcat(buffer, "\n", sizeof buffer);	    
		}
		state_add_prefix(state_out,decode_opt);
		state_puts(buffer,state_out);
	    }
	  
	    DPRINT(Debug,11,(&Debug, 
			     "elm_decode: Read %d bytes from temp file\n",
			     count));

	    free_pager_range(&title_range);

	}

    FAILTYPE:
	;
    }


    DPRINT(Debug,11,(&Debug, 
		     "elm_decode <- END; \n"));
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 			
			 "           (file); ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }
    
    /* Print total 10 error messages */
    errors += print_in_errors(newstate2,state_out,decode_opt);
    
    DPRINT(Debug,11,(&Debug, 
		     "elm_decode: %d errors\n"));

    free_in_state(&newstate2);
}

/* Prototype */
static void text_unsupported_decode P_((mime_t *part, 
					struct in_state *instream, 
					struct out_state *outsream,
					const struct decode_opts *decode_opt,
					charset_t defcharset, 
					struct header_rec *hdr,
					type_mismatch_prompt *badtype,
					charset_t bodydefcharset
					));

/* Returns -1 on failure otherwise same mask than mime_classify_media() */
S_(mime_run_selector text_unsupported_selector)
static int text_unsupported_selector P_((mime_t *p,
					 struct header_rec * hdr));
static int text_unsupported_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = NOTPLAIN_is_fallback;

    /* Notice: if (decoder == text_unsupported_decode)
     *         we return TRUE, because we want call metamail
     *         if it is available
     */

    if (p->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"text_unsupported_selector",
		   "Bad magic number");
    
    if (MIME_selector_magic != p->handler_data->magic)
	mime_panic(__FILE__,__LINE__,"text_unsupported_selector",
		   "Bad magic number (handler_data)");

    /* text_decode uses this */
    p->handler_data->TXT_decoder =  default_text_decoder;

    DPRINT(Debug,11,(&Debug, 
		     "text_unsupported_selector(%p)=%d %s\n",
		     p,flags,mime_debug_classify_f(flags)));


    return flags;
}

static void text_unsupported_decode (ptr, state_in, state_out, decode_opt,
				     defcharset, 
				     mss, badtype,
				     bodydefcharset) 
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,
				     0);

    DPRINT(Debug,11,(&Debug, 
		     "text_unsupported_decode -> state: offset=%ld, length=%ld; \n",
		     (long) ptr -> offset, (long) ptr -> length));

    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"text_unsupported_decode",
		   "Bad magic number");

    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "                          (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }

    /* state_nlputs or state_printf is needed for EOLN_is_CRLF
       conversions */

    /* \n resets this */
    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
    state_printf(state_out,
		 CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedText,
			 "[ %s/%.30s is unsupported, treating like TEXT/PLAIN ]\n"), 
		 get_major_type_name(ptr->TYPE), 
		 get_subtype_name(ptr->TYPE));
    state_nlputs("\n",state_out);

    text_decode (ptr, state_in, state_out, decode_opt, defcharset, mss, badtype,
		 bodydefcharset);

    DPRINT(Debug,11,(&Debug, 
		     "text_unsupported_decode <- END; \n"));
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "                        (file); ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }

    free_pager_range(&title_range);
}

void null_decode (ptr, state_in, state_out, decode_opt, defcharset, mss) 
     mime_t *ptr;
     struct in_state *state_in;
     struct out_state *state_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec *mss;
{
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,0);

    DPRINT(Debug,11,(&Debug, 
		     "null_decode <-> state: offset=%ld, length=%ld; \n",
		     (long) ptr -> offset, (long) ptr -> length));
    
    if (ptr->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"null_decode",
		   "Bad magic number");
    
    if (in_state_seekable(state_in)) {
	DPRINT(Debug,11,(&Debug, 
			 "               (file): ftell=%ld\n",
			 in_state_ftell(state_in))); 
    }


    /* \n resets this */
    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
    state_printf (state_out,
		  CATGETS(elm_msg_cat, MeSet, MeDecodeUnsupportedSkip,
			  "[ %.15s/%.30s is not supported, skipping... ]\n"),
		  get_major_type_name(ptr->TYPE),
		  get_subtype_name(ptr->TYPE));
 
    if (decode_opt->displaying) {
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MeDecodeUseVtosave,
			     "[ Use 'v' to view or save this part. ]\n"));
    }

    free_pager_range(&title_range);
}


#if __GNUC__ 
#define MTH        struct media_type_handle          
#define PAGER(A)   handle_pager, { pager: & A }

#define REGISTER_MT_HANDLER(A,B,R) \
   register_mt_handler(give_media_type2(A,B,1),& R)
#define REGISTER_MT_DEFHANDLER(A,R) \
   register_mt_defhandler(A,& R)

#else
#define MTH        struct COMPAT_media_type_handle
#define PAGER(A)   handle_pager, (void *) & A

#define REGISTER_MT_HANDLER(A,B,R) \
   register_mt_handler(give_media_type2(A,B,1),(struct media_type_handle *)& R)
#define REGISTER_MT_DEFHANDLER(A,R) \
   register_mt_defhandler(A,(struct media_type_handle *)& R)

#endif

static struct  mt_handle_pager  related_1 = { 
    related_decode, related_selector }; 
static MTH                     related_2 = { PAGER(related_1) };

static struct  mt_handle_pager alternative_1 = { 
    alternative_decode, alternative_selector }; 
static MTH                     alternative_2 = { PAGER(alternative_1) };
static struct  mt_handle_pager signed_1      = { 
    signed_decode, signed_selector }; 
static MTH                     signed_2      = { PAGER(signed_1) };
static struct  mt_handle_pager encrypted_1   = { 
    encrypted_decode, encrypted_selector }; 	

static MTH                     encrypted_2   = { PAGER(encrypted_1) };

static struct  mt_handle_pager multipart_0_1 = { 
    multipart_0_decode, multipart_selector }; 
static MTH                     multipart_0_2 = { PAGER(multipart_0_1) };

static struct  mt_handle_pager multipart_mixed_1 = { 
    multipart_mixed_decode, multipart_selector }; 
static MTH                 multipart_mixed_2 = { PAGER(multipart_mixed_1) };

static struct  mt_handle_pager rfc822_1      = { 
    rfc822_decode, rfc822_selector }; 
static MTH                     rfc822_2      = { PAGER(rfc822_1) };
static struct  mt_handle_pager text_1        = { 
    text_decode, text_selector }; 
static MTH                     text_2        = { PAGER(text_1) };

static struct  mt_handle_pager pgp_1         = { 
    pgp_decode, pgp_selector }; 
static MTH                     pgp_2         = { PAGER(pgp_1) };

static struct  mt_handle_pager text_enriched_1 = { 
    text_enriched_decode, text_enriched_selector 
}; 
static MTH                     text_enriched_2 = { 
    PAGER(text_enriched_1) };

static struct  mt_handle_pager text_html_1 = { 
    text_html_decode, text_html_selector 
}; 
static MTH                     text_html_2 = { 
    PAGER(text_html_1) };

static struct  mt_handle_pager text_unsupported_1 = { 
    text_unsupported_decode, text_unsupported_selector 
}; 
static MTH                     text_unsupported_2 = { 
    PAGER(text_unsupported_1) };
static struct  mt_handle_pager elm_1          = { 
    elm_decode, text_selector }; 
static MTH                     elm_2          = { PAGER(elm_1) };

static struct  mt_handle_pager partial_1          = { 
    partial_decode, partial_selector }; 
static MTH                     partial_2          = { PAGER(partial_1) };


static int melib_register_decoders_called = 0;

void melib_register_decoders()
{
    if (melib_register_decoders_called)
	return;

    /* multipart types */
    REGISTER_MT_HANDLER(MIME_TYPE_MULTIPART,"related",    related_2);
    REGISTER_MT_HANDLER(MIME_TYPE_MULTIPART,"alternative",alternative_2);
    REGISTER_MT_HANDLER(MIME_TYPE_MULTIPART,"signed",     signed_2);
    REGISTER_MT_HANDLER(MIME_TYPE_MULTIPART,"encrypted",  encrypted_2);
    REGISTER_MT_HANDLER(MIME_TYPE_MULTIPART,"mixed",      multipart_mixed_2);
    REGISTER_MT_DEFHANDLER(MIME_TYPE_MULTIPART,           multipart_0_2);

    /* message types */
    REGISTER_MT_HANDLER(MIME_TYPE_MESSAGE,"rfc822",       rfc822_2);
    REGISTER_MT_HANDLER(MIME_TYPE_MESSAGE,"delivery-status", text_2);
    REGISTER_MT_HANDLER(MIME_TYPE_MESSAGE,"disposition-notification", text_2);
    REGISTER_MT_HANDLER(MIME_TYPE_MESSAGE,"partial",      partial_2);

    /* text types */
    REGISTER_MT_HANDLER(MIME_TYPE_TEXT,"plain",           text_2);
    REGISTER_MT_HANDLER(MIME_TYPE_TEXT,"rfc822-headers",  text_2);
    REGISTER_MT_HANDLER(MIME_TYPE_TEXT,"enriched",        text_enriched_2);
    REGISTER_MT_HANDLER(MIME_TYPE_TEXT,"html",            text_html_2);
    
    if (use_PGP) {
	REGISTER_MT_HANDLER(MIME_TYPE_TEXT,"x-pgp",           pgp_2);
    }
    REGISTER_MT_DEFHANDLER(MIME_TYPE_TEXT,                text_unsupported_2);

    /* application types */
    if (use_PGP) {
	REGISTER_MT_HANDLER(MIME_TYPE_APPLICATION,"pgp",      pgp_2);
	REGISTER_MT_HANDLER(MIME_TYPE_APPLICATION,"x-pgp",    pgp_2);
    }
    REGISTER_MT_HANDLER(MIME_TYPE_APPLICATION,"X-ELM-encode", elm_2);

#if 0         /* TODO: is this needed? */
    /* Unknown leaf types */
    REGISTER_MT_DEFHANDLER(MIME_TYPE_LEAF,           XXXX);
#endif

    melib_register_decoders_called++;
}

int is_rfc1522 (s)
     const char *s;
{
    /* Returns TRUE if the string contains RFC 1522 format encoded data 

       Not actually check format
    */
   
    while ((s = strchr (s, '=')) != NULL) {
	s++;
	if (*s == '?')
	    return TRUE;
    }
    return FALSE;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
