static char rcsid[] = "@(#)$Id: mime_parse.c,v 2.22 2022/08/06 08:01:51 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.22 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "def_melib.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mime");

void mime_t_clear (mt)
     mime_t *mt;
{
    DPRINT(Debug,15,(&Debug,
		     "mime_t_clear(%p) --> BEGIN\n",mt));

    if (mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_t_clear",
		   "Bad magic number");
    
    mt->offset = mt->begin_offset = 0;
    mt->length = -1;
    mt->encoding = ENCODING_7BIT;
    mt->TYPE        = give_media_type2(MIME_TYPE_TEXT,"plain",0);
    if (!mt->TYPE)
	mime_panic(__FILE__,__LINE__,"mime_t_clear",
		   "text/plain is not known");

    mt->disposition = DISP_INLINE;
    mt->mime_flags  = 0;
    
    if (mt->parser_data)
	mime_parser_free(&(mt->parser_data));

    if (mt->handler_data)
	mime_selector_free(& (mt->handler_data));    

    mt->metamail_blacklisted = NULL;

    if (mt->description)
	free_string (&(mt->description));
    if (mt->TYPE_opts)
	free_mime_param (&(mt->TYPE_opts));
    if (mt->DISPOSITION_opts)
	free_mime_param (&(mt->DISPOSITION_opts));
    
    if (mt->pathname0) {
	if (mt->unlink) {
	    DPRINT(Debug,15,(&Debug,
			     "mime_t_clear: %s marked for unlink\n",
			     mt->pathname0));
	    
	    if (0 == unlink(mt->pathname0)) {
		DPRINT(Debug,14,(&Debug,
				 "mime_t_clear: %s unlinked\n",
				 mt->pathname0));
	    }
	}

	free (mt->pathname0);

    } else {
	if (mt->unlink) {
	    DPRINT(Debug,3,(&Debug,
			    "mime_t_clear: ERROR: Pathname marked for unlink, but no pathname\n"));
	}
    }

    mt->pathname0 = NULL;
    mt->unlink = 0;

    if (mt->dispname)
	free_string(& (mt->dispname));

    if (mt->content_id)
	free_message_id(& (mt->content_id));
        
    DPRINT(Debug,15,(&Debug,
		     "mime_t_clear(%p) <-- END\n",mt));
    return;
}

void mime_get_disposition (str, mt, def_charset,header_error,header_status)
     char *str;
     mime_t *mt;
     charset_t def_charset;
     struct header_errors **header_error;
     int header_status;

{
    char *c, tmp[VERY_LONG_STRING];
    
    DPRINT(Debug,9,(&Debug,
		    "mime_get_disposition(): str=\"%s\"\n", str));
    
    if (mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_get_disposition",
		   "Bad magic number");
    
    /* Don't harm "str" */
    strfcpy (tmp, str, sizeof(tmp));
    
    rfc822_reap_comments (tmp, NULL, 0);

    if (mt->DISPOSITION_opts) {
	free_mime_param(&(mt->DISPOSITION_opts));
    }
    
    /* Look for the options field */
    if ((c = strchr (tmp, ';')) != NULL) {
	char *d = c;
	while (d > tmp && whitespace(*(d-1)))
	    d--;
	*d = '\0';
	c++;
	while (*c && whitespace(*c))
	    c++;
	mt->DISPOSITION_opts = parse_mime_param("Content-Disposition",
						c, def_charset,
						header_error,
						header_status);

    } else {
	char *d = tmp + strlen(tmp);
	while (d > tmp && whitespace(*(d-1)))
	    d--;
	*d = '\0';
    }
    
    /* All that's left now is the main disposition */
    c = tmp;
    while (*c && whitespace(*c))
	c++;

    /* No Content-Disposition -header     -> DISP_INLINE     (DISP_AUTOATTACH)
     *    Content-Disposition: inline     -> DISP_INLINE
     *    Content-Disposition: attachment -> DISP_ATTACH
     *    Content-Disposition: {unknown}  -> DISP_ATTACH
     * See RFC 1806 (Experimental protocol) for details.
     */   

    if (! valid_mime_token(c,"Content-Disposition",
			   fn_disposition,header_error)) {
	mt->disposition = DISP_ATTACH;
    } else if (istrcmp (c, "inline") != 0)
	mt->disposition = DISP_ATTACH;
    else
	mt->disposition = DISP_INLINE;
    
    DPRINT(Debug,9,(&Debug,
		    "mime_get_disposition(): disposition=\"%s\" (%d)\n",
		    DISPOSITION(mt->disposition), 
		    mt->disposition));
}

/* ASCII assumed, return value
   0   = not special
   1   = is mime tspecial
*/
   
static int is_mime_tspecial P_((char c));
static int is_mime_tspecial(c)
     char c;
{
    uint16 c1;
    int r;
    int ret;

    if (c < 0 ||
	(unsigned char)c > 127) {

	DPRINT(Debug,1,(&Debug, 
			"is_mime_tspecial(%d) invalid",c));
	
	mime_panic(__FILE__,__LINE__,"is_mime_tspecial",
		   "Bad character (not ASCII)");

	return -1;
    }

    c1 = c;

    r = unicode_is_special(c1,TOK_mime);

    ret = r != 0;

    DPRINT(Debug,15,(&Debug, 
		    "is_mime_tspecial(%d '%c')=%d: unicode_is_special=%04x unicode=%04x\n",
		    c,c,ret,r,c1));
    return ret;
}

/* return 0 if not valid, and adds/prints header error adds/prints header error 
   if header_name != NULL
*/

int valid_mime_token(token,header_name,field_name,
		     header_error)
     const char *token;
     const char *header_name;
     enum valid_field field_name;
     struct header_errors ** header_error;
{
    int ret = 1;
    const char *c;

    int non_ascii_count = 0;
    int tspecial_count  = 0;
    int space_count     = 0;
    int control_count   = 0;

    char special = '\0';
    struct string * temp = NULL;

    if (header_name) {
	switch (field_name) {
	case fn_subtype: 
	    temp = format_string(CATGETS(elm_msg_cat, MeSet,
					 MeValidSubType,
					 "subtype"));
	    break;
	case fn_major_type:
	    temp = format_string(CATGETS(elm_msg_cat, MeSet,
					 MeValidMajorType,
					 "major type"));
	    break;
	case fn_disposition:
	    temp = format_string(CATGETS(elm_msg_cat, MeSet,
					 MeValidDisposition,
					 "disposition"));
	    break;
	case fn_name_of_param:
	    temp = format_string(CATGETS(elm_msg_cat, MeSet,
					 MeValidNameOfParam,
					 "name of param"));
	    break;
	default:
	    mime_panic(__FILE__,__LINE__,"valid_mime_token",
		       "Bad fn_subtype");
	    break;
	}
    }

    if (!token[0]) {

	if (header_name)
	    process_header_error(header_error,
				 CATGETS(elm_msg_cat, MeSet,
					 MeValidEmptyToken,
					 "PARSE ERROR: Empty %S on %s header"),
				 temp,header_name);
	ret = 0;
    }

    for (c = token; *c; c++) {
	if (!isascii(*c)) {
	    non_ascii_count++;
	} else if (whitespace(*c)) {	    
	    space_count++;
	} else if (iscntrl(*c)) {
	    control_count++;
	} else if (is_mime_tspecial(*c)) {
	    tspecial_count++;
	    special = *c;
	}
    }

    if (non_ascii_count) {
	if (header_name) {
	    if (1 == non_ascii_count)
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeValidNonAscii1Token,
					     "PARSE ERROR: Non-ascii character on %S of %s header"),
				     temp,header_name);
	    else
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeValidNonAsciiToken,
					     "PARSE ERROR: %d non-ascii characters on %S of %s header"),
				     non_ascii_count,temp,header_name);
	}
	ret = 0;
    } else if (control_count) {
	if (header_name) {
	    if (1 == control_count)
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeValidControlChar1Token,
					     "PARSE ERROR: Control character on %S of %s header"),
				     temp,header_name);
	    
	    else
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeValidControlCharToken,
					     "PARSE ERROR: %d control characters on %S of %s header"),
				     control_count,temp,header_name);
	}
	ret = 0;

    } else if (space_count) {
	if (header_name) {
	    if (1 == space_count)
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeValidSpaceChar1Token,
					     "PARSE ERROR: Space character on %S of %s header"),
				     temp,header_name);
	    
	    else
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeValidSpaceCharToken,
					     "PARSE ERROR: %d space characters on %S of %s header"),
				     space_count,temp,header_name);
	}
	ret = 0;

    } else if (tspecial_count) {
	if (header_name) {
	    if (1 == tspecial_count)
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeValidSpecialChar1Token,
					     "PARSE ERROR: Special character (%c) on %S of %s header"),
				     special,temp,header_name);
	    
	    else
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeValidSpecialCharToken,
					     "PARSE ERROR: %d special characters (for example %c) on %S of %s header"),
				     tspecial_count,special,temp,header_name);
	}
	
	ret = 0;
    }


    DPRINT(Debug,15,(&Debug,
		     "valid_mime_token=%d; token=%Q; %d non ascii characters, %d whitespaces, %d control characters and %d tspecials",
		     ret,token,
		     non_ascii_count,space_count,control_count,tspecial_count));

    if (temp) {
	DPRINT(Debug,15,(&Debug," (header=%s, field=%d %S)",
			 header_name,field_name,temp));
	free_string(&temp);

    }
    DPRINT(Debug,15,(&Debug,"\n"));

    return ret;
}

int mime_get_content1(str,TYPE,TYPE_opts, def_charset,header_error,
		      header_status)
     const char *str;
     media_type_t *TYPE; 
     struct mime_param **TYPE_opts;
     charset_t def_charset;
     struct header_errors **header_error;
     int header_status;
{
    
    char *c;
    char * subtype = TYPE_UNKNOWN_STRING;

    /* Note:  TYPE_UNKNOWN_STRING is not valid mime token ('?' character is not allowed) */

    int ok = 1;
    char *tmp = NULL;

    DPRINT(Debug,9,(&Debug,
		    "mime_get_content1: str=\"%s\"\n", str));

    /* Don't harm "str" */

    tmp = safe_strdup(str);    
    rfc822_reap_comments (tmp, NULL, 0);
    

    /* Look for the options field */
    if ((c = strchr (tmp, ';')) != NULL) {
	char *d = c;
	while (d > tmp && whitespace(*(d-1)))
	    d--;
	*d = '\0';
	c++;
	while (*c && whitespace(*c))
	    c++;
	*TYPE_opts = parse_mime_param("Content-Type",
				      c, def_charset,
				      header_error,
				      header_status);

    } else {
	char *d = tmp + strlen(tmp);
	while (d > tmp && whitespace(*(d-1)))
	    d--;
	*d = '\0';
    }

    if (! tmp[0]) {
	process_header_error(header_error,
			     CATGETS(elm_msg_cat, MeSet,
				     MeEmptyTypeCT,
				     "PARSE ERROR: Empty type on Content-Type header"));

	*TYPE = NULL;
	ok = 0;	
	goto fail;
    }
        
    /* Get the subtype */
    if ((c = strchr (tmp, '/')) != NULL) {
	char *d = c;
	while (d > tmp && whitespace(*(d-1)))
	    d--;
	*d = '\0';
	
	c++;
	while (*c && whitespace(*c))
	    c++;
	subtype = c;

	if (0 == strcmp(TYPE_UNKNOWN_STRING,subtype)) {
	    subtype = TYPE_UNKNOWN_STRING;  /* silent failure */
	    ok = 0;
	} else if (! valid_mime_token(subtype,"Content-Type",
				      fn_subtype,header_error)) {
	    subtype = TYPE_UNKNOWN_STRING;
	    ok = 0;
	}
	
    } else {
	DPRINT(Debug,1,(&Debug,
			"mime_get_content1: No subtype!\n"));

	process_header_error(header_error,
			     CATGETS(elm_msg_cat, MeSet,
				     MeNoSubTypeCT,
				     "PARSE ERROR: No subtype on Content-Type header"));

	ok = 0;
    }

   
    /* All that's left now is the main type */
    c = tmp;
    while (*c && whitespace(*c))
	c++;
    
    if (0 == strcmp(TYPE_UNKNOWN_STRING,c)) {
	c = TYPE_UNKNOWN_STRING;  /* silent failure */
	ok = 0;
    } else if (! valid_mime_token(c,"Content-Type",
				  fn_major_type,header_error)) {
	c = TYPE_UNKNOWN_STRING;
	ok = 0;
    }

    if (c == TYPE_UNKNOWN_STRING &&
	subtype == TYPE_UNKNOWN_STRING) {
	
	DPRINT(Debug,1,(&Debug,
			"mime_get_content1: Both type and subtype bad -- hard failure\n"));

	*TYPE = NULL;
	ok = 0;	
	goto fail;
    }

    /* NOTE: 
       get_major_type_name() returns TYPE_UNKNOWN_STRING for NULL type
       get_subtype_name()    returns TYPE_UNKNOWN_STRING for NULL type
    */

    *TYPE = give_media_type(c,subtype,1);
    
    if (! *TYPE) {
	DPRINT(Debug,1,(&Debug,
			"mime_get_content1: give_media_type for \"%s\" FAILED!\n", 
			str));
	ok = 0;
    }

 fail:
    free(tmp);
    return ok;
}

void mime_get_content (str, mt, def_charset, header_error,header_status)
     char *str;
     mime_t *mt;
     charset_t def_charset;
     struct header_errors **header_error;
     int header_status;
{
    
    if (mt->magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_get_content",
		   "Bad magic number");

    if (mt->TYPE_opts) {
	free_mime_param(&(mt->TYPE_opts));
    }


    if (!mime_get_content1(str, & mt->TYPE, & mt->TYPE_opts, def_charset,
			   header_error,header_status)) {

	DPRINT(Debug,1,(&Debug,
			"mime_get_content(): str=\"%s\" FAILED!\n", str));
	return;	
    }

    DPRINT(Debug,9,(&Debug,
		    "mime_get_content(): %p -- type=\"%s\", subtype=\"%s\"\n",
		    mt->TYPE,
		    get_major_type_name(mt->TYPE), 
		    get_subtype_name(mt->TYPE)));
    
    return;
}

const char* mime_get_boundary (opts)
     struct mime_param *opts;
{

    const char * pv = get_mime_param_compat(opts,"boundary");


    if (!pv) 
	lib_error(CATGETS(elm_msg_cat, MeSet, MeParseNoBoundary,
			  "'boundary' parameter is missing from Multipart -type!"));

    return pv;
}

int mime_get_charset (charset_value, opts, display_charset,
		      default_content_charset)
     charset_t *charset_value;
     struct mime_param *opts; 
     const charset_t * display_charset; /* vector */
     charset_t default_content_charset;
{  
    /* Return > 0  (2)  if charset is displayable with display_charset     
     * Return < 0  (-2) if charset needs translating to display_charset
                    -3  if charset can be translated to display_charset
                        without loss
     */
    int ret = 0,j;
    const char *pv;

    pv = get_mime_param_compat(opts,"charset");


    if (!pv) {
	const char * def_MIME_name = NULL;
		    

	if (default_content_charset && 
	    (def_MIME_name = get_charset_MIME_name(default_content_charset))) {

	    DPRINT(Debug,9,(&Debug,
			    "mime_get_charset: Using content default charset %s\n",
			    def_MIME_name));

	    *charset_value = default_content_charset;

	    /* Charset name is also required on some confitions later */
	    pv = def_MIME_name;

	} else {

	    DPRINT(Debug,9,(&Debug,
			    "mime_get_charset: No charset specified\n"));

	    pv = "US-ASCII";
	    /* Default charset if nothing specified */
	    
	    *charset_value = MIME_name_to_charset(pv,CHARSET_create);
	}
    } else {
	DPRINT(Debug,9,(&Debug,
			"mime_get_charset: found %s\n",pv));

	*charset_value = MIME_name_to_charset(pv,CHARSET_create);
    }

    {
	const char * MIME_name                    UNUSED_VAROK = 
	    get_charset_MIME_name(*charset_value);
	const struct  charset_type * charset_type UNUSED_VAROK =
	    get_charset_type(*charset_value);

	DPRINT(Debug,9,(&Debug,
			"                : charset_value=%p '%s' (type=%p)\n",
			*charset_value, MIME_name, charset_type));

    }
    
    if (display_charset) {
	for (j = 0; display_charset[j]; j++) {
	    
	    const char * MIME_name_d = 
		get_charset_MIME_name(display_charset[j]);
	    const struct charset_type * charset_type_d UNUSED_VAROK =
		get_charset_type(display_charset[j]);

	    DPRINT(Debug,9,(&Debug,
			    "            [%d] : display_charset=%p '%s' (type=%p)\n",
			    j,display_charset[j],
			    MIME_name_d ? MIME_name_d : "<no MIME name>",
			    charset_type_d));
		    
	    if (*charset_value == display_charset[j] ||
		
		/* Accept also charsets with same name although
		   charset is redefined
		*/
		( MIME_name_d && 0 == istrcmp(pv,MIME_name_d))) {
		ret = 2; /* If display charset              */

		break;
	    } else if (charset_superset_of(display_charset[j],*charset_value)) {
		ret = 2; /* if charset is subset of display */
		break;
	    } else if (0 == istrcmp(pv,"US-ASCII") && 
		       charset_ok_p(display_charset[j])) {
		ret = 2; /* If display_charset can show us-ascii? */
		break;
	    }
	}


	if ( 0 == ret) {

	    int prop_disp = charset_properties(*charset_value);

	    if ((CS_printable & prop_disp) &&
		(CS_mapping & prop_disp)) {
		    int prop;

		    if (display_charset[0] &&
			(CS_mapping & (prop = 
				       charset_properties(display_charset[0])))) {

			ret = -2;  /* convert (filter?) to display charset */
		    
			if ( ISO2022_superset_of(display_charset[0],*charset_value)) {
			    
			    const char * MIME_name_d0 UNUSED_VAROK = 
				get_charset_MIME_name(display_charset[0]);

			    
			    DPRINT(Debug,9,(&Debug,
					    "          display charset '%s' --  ISO 2022 superset\n",
					    MIME_name_d0 ?
					    MIME_name_d0 :
					    "<no MIME name>",
					    prop));
			    ret = -3;  /* convert to display charset without loss */
			    


			} else if ((CS_universal_set & prop)) {

			    const char * MIME_name_d0 UNUSED_VAROK = 
				get_charset_MIME_name(display_charset[0]);


			    DPRINT(Debug,9,(&Debug,
					    "          display charset '%s' --  universal charset (flags=%d)\n",
					    MIME_name_d0 ?
					    MIME_name_d0 :
					    "<no MIME name>",
					    prop));
			    ret = -3;  /* convert to display charset without loss */
			}

		    } else {
			DPRINT(Debug,9,(&Debug," --- display charset is not known ...\n"));
		    }

		} else {
		    DPRINT(Debug,9,(&Debug," --- charset is not known ...\n"));

		}
	}
    } else {
	DPRINT(Debug,9,(&Debug," --- display_charset vector not given ... \n"));
    }

    DPRINT(Debug,9,(&Debug,
		    "mime_get_charset=%d\n",ret));

    return ret;
}

void mime_t_zero (ptr)
     mime_t *ptr;
{
    /* This routine should be called whenever a new "mime_t" is created.  It
     * makes sure that the pointers inside are correctly initialized to NULL
     * so that we don't end up calling free() on an uninitialized pointer.
     */
    DPRINT(Debug,15,(&Debug,
		     "mime_t_zero(%p)\n",ptr));

    ptr->parser_data   = NULL;
    ptr->handler_data  = NULL;
    ptr->metamail_blacklisted = NULL;

    ptr->TYPE_opts = ptr->DISPOSITION_opts = NULL;
    ptr->pathname0 = NULL;
    ptr->dispname = NULL;
    ptr->description = NULL;
    ptr->content_id  = NULL;
    
    ptr->magic = MIME_magic;
  

  ptr->offset = ptr->begin_offset = 0;
  ptr->length = -1;
  ptr->encoding = ENCODING_7BIT;
  ptr->unlink = 0;
  ptr->TYPE        = give_media_type2(MIME_TYPE_TEXT,"plain",0);
  if (!ptr->TYPE)
      mime_panic(__FILE__,__LINE__,"mime_t_zero",
		 "text/plain is not known");

  ptr->disposition = DISP_INLINE;
  ptr->mime_flags  = 0;
}

void mime_t_copy(trg, src)
     mime_t *trg, *src;
{
    /* This routines make copy of mime_t structure ... */

    DPRINT(Debug,15,(&Debug,
		     "mime_t_copy(%p,%p) --> BEGIN\n",trg,src));
  
  if (trg->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"mime_t_copy",
	       "Bad magic number (trg)");

  if (src->magic != MIME_magic)
    mime_panic(__FILE__,__LINE__,"mime_t_copy",
	       "Bad magic number (src)");

  mime_t_clear(trg);

  trg->offset =            src->offset; 
  trg->begin_offset  =     src->begin_offset; 
  trg->length =            src->length;
  trg->encoding =          src->encoding;
  trg->unlink = 0;         /* Don't unlink ! */
  trg->TYPE =              src->TYPE;
  trg->disposition =       src->disposition;
  trg->mime_flags =        src->mime_flags;

  

  trg->parser_data = NULL;        
  trg->handler_data = NULL;     

  trg->metamail_blacklisted = src->metamail_blacklisted;


  if (src->parser_data) {
      copy_parser_data(& (trg->parser_data), src->parser_data);
  } else {
      DPRINT(Debug,3,(&Debug,
		       "mime_t_copy(src=%p,trg=%p):  src->parser_data == NULL !\n",trg,src));

  }


  if (src->description) {
      trg->description = dup_string(src->description);
  }

  if (src->content_id) {
      trg->content_id = dup_message_id(src->content_id);
  }
  
  if (src->TYPE_opts) {
      trg->TYPE_opts = copy_mime_param(src->TYPE_opts);
  }

  if (src->DISPOSITION_opts) {
      trg->DISPOSITION_opts = copy_mime_param(src->DISPOSITION_opts);
  }

  DPRINT(Debug,15,(&Debug,
		   "mime_t_copy(%p,%p) <-- END\n",trg,src));
}

mime_t *
mime_t_alloc ()
{
  mime_t *ptr;

  DPRINT(Debug,15,(&Debug,
		   "mime_t_alloc()     --> BEGIN\n"));

  ptr = (mime_t *) safe_malloc (sizeof (mime_t));
  
  /* Make sure to clear the pointers initially so that later we know when
   * to reclaim memory in mime_t_clear().
   */
  mime_t_zero (ptr);
  mime_t_clear (ptr);

    DPRINT(Debug,15,(&Debug,
		     "mime_t_alloc() = %p <-- END\n",ptr));
    return ptr;
}

void parse_mime_headers1 (ptr,headers,part_offset,body_offset,opts, 
			  hdr_charset,header_error,header_status,
			  hdr)
     mime_t *ptr;
     header_list_ptr headers;
     long part_offset;
     long body_offset;
     int opts;
     charset_t hdr_charset;
     struct header_errors   ** header_error;
     int header_status;
     struct header_rec * hdr;
{
    header_list_ptr this_header;
    int no_content_disposition = 0;
    int x;

    /* set some defaults */
    ptr->encoding = ENCODING_7BIT;
    if (opts & MIME_DIGEST) {
	ptr->TYPE = give_media_type2(MIME_TYPE_MESSAGE,"rfc822",0);
	if (!ptr->TYPE)
	    mime_panic(__FILE__,__LINE__,"parse_mime_headers1",
		       "message/rfc822 is not known");
    } else {
	ptr->TYPE = give_media_type2(MIME_TYPE_TEXT,"plain", 0);
	if (!ptr->TYPE)
	    mime_panic(__FILE__,__LINE__,"parse_mime_headers1",
		       "text/plain is not known");
    }

    ptr->disposition = DISP_INLINE;
    ptr->description = NULL;
    
    ptr->begin_offset = part_offset;
    ptr->offset       = body_offset;
    ptr->length = -1;
    
    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-Type")) &&
	NULL != this_header->body) {

	mime_get_content (this_header->body, ptr, hdr_charset,
			  header_error, header_status);

	if (this_header->next_this_header) 
	    process_header_error(header_error,
				 CATGETS(elm_msg_cat, MeSet, 
					 MeParseErrorContent,
					 "PARSE ERROR: Several Content-Type headers!"));
	
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-Type -header\n"));
    }

    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-Disposition")) &&
	NULL != this_header->body) {

	mime_get_disposition (this_header->body, ptr, hdr_charset,
			      header_error,header_status);

	if (this_header->next_this_header)
	    process_header_error(header_error,
				 CATGETS(elm_msg_cat, MeSet, MeParseErrorDisposition,
					 "PARSE ERROR: Several Content-Disposition headers!"));
	
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-Disposition -header\n"));
	no_content_disposition = 1;
    }

    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-Transfer-Encoding")) &&
	NULL != this_header->body) {
	char  * value = this_header->body, *c;
	
	/* This removes comments from buffer this_header->body */
	
	rfc822_reap_comments (value, NULL, 0);
	c = value;
	while (*c && isspace((unsigned char) *c))
	    c++;
	
	ptr->encoding = check_encoding (c);

	if (ENCODING_ILLEGAL == ptr->encoding) 
	    process_header_error(header_error,
				 CATGETS(elm_msg_cat, MeSet, MeContentTransferUnsupported,
					 "PARSE ERROR: %.30s Content-Transfer-Encoding is unsupported"),
				 c);

	if (this_header->next_this_header) 
	    process_header_error(header_error,
				 CATGETS(elm_msg_cat, MeSet, MeParseErrorTransfer,
					 "PARSE ERROR: Several Content-Transfer-Encoding headers!"));
				 
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-Transfer-Encoding -header\n"));
    }
    
    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-Description"))  &&
	NULL != this_header->body) {
	char * c = this_header->body;

	while (*c && whitespace(*c))
	    c++;

	ptr->description = hdr_to_string(HDR_TEXT,c,
					 hdr_charset,
					 1);
					 	
	if (this_header->next_this_header)
	    process_header_error(header_error,
				 CATGETS(elm_msg_cat, MeSet, 
					 MeParseErrorDescription,
					 "PARSE ERROR: Several Content-Description headers!"));
	
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-Description -header\n"));
    }
    
    if (NULL != (this_header = 
		 locate_header_by_name(headers,"Content-ID"))  &&
	NULL != this_header->body) {
	char * c = this_header->body;

	while (*c && whitespace(*c))
	    c++;

	ptr->content_id = parse_header_message_id(give_header_name(this_header->header_name),
						  c,1, hdr_charset,header_error);

	if (this_header->next_this_header)
	    process_header_error(header_error,
				 CATGETS(elm_msg_cat, MeSet, 
					 MeParseErrorXXXName,
					 "PARSE ERROR: Several %s headers!"),
				 give_header_name(this_header->header_name));
    } else {
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: No Content-ID -header\n"));
    }
           
    x =  mime_classify_media(ptr,hdr,
			     DISP_INLINE == ptr->disposition &&
			     !no_content_disposition ? 
			     mime_classify_all :
			     /* Do not ask program on "Mailcap program selection"
				if later part is skipped as attachment 
			     */
			     mime_classify_skip_mailcap
			     );

    if (x) {
	/* mime_classify_media() sets mime_flags */
	
	DPRINT(Debug,9,(&Debug,
			"parse_mime_headers1: 'Not plain': %s\n",
			mime_debug_classify_f(x)
			));
	if (no_content_disposition) {
	    enum auto_attachment_v aa = 
		give_dt_enumerate_as_int(&auto_attachment);

	    DPRINT(Debug,9,(&Debug,
			    "                   : no content-disposition -- auto-attachment = %d\n",
			    aa));

	    if (( get_major_type_code(ptr->TYPE) == MIME_TYPE_APPLICATION && 
		  aa >= auto_attachment_application) 
		||
		(
		 get_major_type_code(ptr->TYPE) != MIME_TYPE_TEXT       && 
		 !(get_type_flags(ptr->TYPE) & MIME_RFC822)             &&
		 get_major_type_code(ptr->TYPE) != MIME_TYPE_MULTIPART  && 
		 aa >= auto_attachment_non_text)
		) {
		
		DPRINT(Debug,9,(&Debug,
				"                   : Set disposition = auto attachment\n"));
		ptr->disposition = DISP_AUTOATTACH;

		DPRINT(Debug,9,(&Debug,
				"                   : was %s/%s\n",
				get_major_type_name(ptr->TYPE), 
				get_subtype_name(ptr->TYPE)));

	    }		
	}
    }    
}

mime_t * parse_mime_headers(headers,part_offset,body_offset,opts,
			    hdr_charset,header_error,header_status,hdr)
     header_list_ptr headers;
     long part_offset;
     long body_offset;
     int opts;
     charset_t hdr_charset;
     struct header_errors **header_error;
     int header_status;
     struct header_rec * hdr;
{
  mime_t *ptr;

  DPRINT(Debug,9,(&Debug,
		  "parse_mime_headers(): part_offset=%ld, body_offset=%ld, opts=%d\n",
		  part_offset,body_offset,opts,hdr));

  ptr = mime_t_alloc ();

  parse_mime_headers1(ptr,headers,part_offset,body_offset,opts,hdr_charset,
		      header_error,header_status,hdr);


  DPRINT(Debug,10,(&Debug,
		   "parse_mime_headers- type=%s/%s; flags=%d;\n",
		   get_major_type_name(ptr->TYPE), 
		   get_subtype_name(ptr->TYPE), 
		   get_type_flags(ptr->TYPE)));
  DPRINT(Debug,10,(&Debug,
		   "parse_mime_headers- begin=%ld, offset=%ld, length=%ld\n",
		   ptr->begin_offset,ptr->offset,ptr->length));

  DPRINT(Debug,9,(&Debug,
		  "parse_mime_headers=%p <-- END\n",(void *)ptr));

  return ptr;
}

mime_t * mime_read_header (fp, opts, defcharset, 
			   header_error,header_status,hdr)
     FILE *fp;
     int opts;
     charset_t defcharset;
     struct header_errors **header_error;
     int header_status;
     struct header_rec * hdr;
{
  mime_t *ptr;
  header_list_ptr headers = NULL;
  long part_offset;
  long body_offset;


  DPRINT(Debug,19,(&Debug,
		   "mime_read_header: opts=%d --> START\n",opts));

  part_offset = ftell (fp);
  headers = file_read_headers(fp,0);
  body_offset = ftell(fp);

  ptr = parse_mime_headers(headers,part_offset,body_offset,opts,
			   defcharset,header_error,header_status,hdr);

  delete_headers(&headers);

  return ptr;
}
 
int mime_warnings(hdr) 
     struct header_rec *hdr;
{
    int ret = 0;

    if (hdr->status & PRE_MIME_CONTENT) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeParsePreMime,
			  "Error: MIME-message has pre-MIME content-type!"));

	ret |= PRE_MIME_CONTENT;
    }

    if (hdr->status & MIME_UNSUPPORTED) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeParseUnsupportedMime,
			  "Warning: Unsupported MIME-Version!"));

	ret |=  MIME_UNSUPPORTED;
    }

    return ret;
}

int  mime_parse_routine1(hdr,fp)
     struct header_rec *hdr;
     FILE *fp;
{
    /* This routine checks to see if the multipart messages specified by
     * "hdr" has been parsed for its subparts, and if not, calls the routine
     * to do so.
     */
    
    if (hdr->mime_rec.magic != MIME_magic)
	mime_panic(__FILE__,__LINE__,"mime_parse_routine1",
		   "Bad magic number (mime_rec)");
    
    if (hdr->mime_rec.parser_data || !(hdr->status & MIME_MESSAGE)) {
	DPRINT(Debug,10,(&Debug,"mime_parse_routine1=1 -- already parsed or no mime message\n"));
	return 1;
    }
    
    if (! hdr->mime_rec.TYPE) {
	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeDecodeBadCT,
			  "Bad content-type"));
	return 1;
    }

    /* read_mailcaps() parses mailcaps only once */
    read_mailcaps();

    /* Copy value */
    if (hdr -> content_length >= 0)
	hdr->mime_rec.length = hdr -> content_length;
    
    if (hdr->mime_rec.begin_offset <= 0) {
	
	fseek(fp,hdr->offset,SEEK_SET);
	hdr->mime_rec.begin_offset = hdr->offset;
	/* Skip mailbox's separator lines ... */
	
	DPRINT(Debug,9,(&Debug,
			"mime_parse_routine1: scanning begin_offset: %ld\n",
			(long) hdr->mime_rec.begin_offset));
	
	hdr->mime_rec.begin_offset = skip_envelope(hdr,fp);
	
	DPRINT(Debug,9,(&Debug,
			"mime_parse_routine1: begin_offset=%ld\n",
			(long) hdr->mime_rec.begin_offset));
	
	if (hdr->mime_rec.begin_offset < 0) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeParseMailError,
			      "Can't parse mail..."));
	    return 0;
	}
    }

    lib_transient(CATGETS(elm_msg_cat, MeSet, MeParsingMime,
			  "Parsing MIME structure..."));
    if (mime_parser_parse(&(hdr->mime_rec),hdr->header_charset,fp,
			  &(hdr->header_error), hdr->status,hdr)) {

	/* mime_classify_media sets hdr->mime_rec.mime_flags  !!! */

	int old = hdr->mime_rec.mime_flags;
	int tmp = 0;

	DPRINT(Debug,9,(&Debug,
			"mime_parse_routine1: disposition=%d (%s), old mime flags %s\n",
			hdr->mime_rec.disposition,
			DISPOSITION(hdr->mime_rec.disposition),
			mime_debug_classify_f(old)));

	
	tmp = mime_classify_media(&(hdr->mime_rec),hdr,
				DISP_INLINE == hdr->mime_rec.disposition ? 
				mime_classify_all :
				/* Do not ask program on "Mailcap program selection"
				   if later part is skipped as attachment 
				*/
				mime_classify_skip_mailcap);

	if (tmp >= 0) {
	    if (tmp != old) {

		DPRINT(Debug,9,(&Debug,
				"mime_parse_routine1: got mime flags %s\n",
				mime_debug_classify_f(tmp)));
		
		/* WARNING:
		   Return value of mime_classify_media do not include
		   NOTPLAIN_is_fallback, that may be set on hdr->mime_rec.mime_flags 
		*/
		
		if (! (tmp & NOTPLAIN_need_metamail) && 
		    ! prompt_metamail) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeParsingMimeNoMetamail,
				      "Parsing MIME structure... metamail not needed"));	    
		}
	    }

	    hdr->mime_parsed            = 1;
	}
    }

    return 1;
}		    

#if ANSI_C
parse_mime_callback mime_parse_routine;
#endif

int mime_parse_routine (folder,hdr, fp)
     struct folder_info *folder;
     struct header_rec *hdr;
     FILE *fp;
{

    return mime_parse_routine1(hdr,fp);

}

int is_pre_mime_content_type (ptr,content_type)
     mime_t *ptr;
     char *content_type;
{
    int result;
    char *cptr = strpbrk(content_type,"/;()");

    DPRINT(Debug,10,(&Debug,
		     "is_pre_mime_content_type(): content_type=%s\n",
		     content_type));
    
    if (!cptr || ';' == *cptr) {
	char *ptr2;
	char *tmp = content_type;
	while (whitespace(*tmp))
	    tmp++;
	
	if (cptr)
	    *cptr = '\0';
	
	ptr2 = strpbrk(tmp," \t");
	if (ptr2)
	    *ptr2 = '\0';
	
	if (istrcmp(tmp,"text")!=0) {
	    char buf[STRING];
	    ptr -> mime_flags = NOTPLAIN_need_metamail;         /* !!! */
	    
	    /* Put some 'intelligent' value */
	    elm_sfprintf(buf,sizeof buf,
			 FRM("X-RFC1049-%.30s"),tmp);
	    ptr->TYPE = give_media_type2(MIME_TYPE_APPLICATION,
					 buf,1);
	} else {
	    ptr -> mime_flags = 0;
	    ptr -> TYPE = give_media_type2(MIME_TYPE_TEXT,"plain",0);
	    if (!ptr->TYPE)
		mime_panic(__FILE__,__LINE__,"is_pre_mime_content_type",
			   "text/plain is not known");
 
	}
	result = 1;
    } else 
	result = 0;
    
    DPRINT(Debug,10,(&Debug,
		     "is_pre_mime_content_type=%d\n",result));
    
    return result;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
