static char rcsid[] = "@(#)$Id: pgp_decode.c,v 2.27 2021/07/07 16:28:00 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.27 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 *****************************************************************************/

#include "def_melib.h"
#include "s_me.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"pgp");

#include <sys/time.h>

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

extern int pgp_keeppassfor; /* 5 minutes */
char pgp_passphrase[PGP_NUM][STRING];
int pgp_expires;

static const char * pgp_names[PGP_NUM] = {
    "*none*",
    "PGP 2",
    "PGP 5",
    "GnuPG"
};

/* if v >= pgp2 returns gives available pgp version */

enum pgp_version have_pgp (v)
     enum pgp_version v;
{
    enum pgp_version return_value = v;
    
    switch(v) {
    case pgp_none:
	return_value = pgp_none;
	;;
    case pgp2: {
	const char * pgp2_path_val = give_dt_estr_as_str(&pgp2_path_e, "pgp2",
							 NULL,NULL);

	if (! pgp2_path_val ||
	    strcmp(pgp2_path_val,"none") == 0 || 
	    pgp2_path_val[0] == '\0') {
	    return_value = pgp_none;
	} else if (pgp2_path_val[0] == '/') {
	    if (-1 == access(pgp2_path_val,EXECUTE_ACCESS)) {
		int err = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantExecute,
				  "Can't execute pgp: %.50s: %.30s"),
			  pgp2_path_val, strerror(err));
	    
		DPRINT(Debug,5,(&Debug,
				"have_pgp: no access %s: %s\n",pgp2_path_val,
				strerror(err)));
		return_value = pgp_none;
	    }
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpPath,
			      "PGP path must start with '/': %.60s"),
		      pgp2_path_val);
	    DPRINT(Debug,5,(&Debug,
			    "have_pgp: bad path: %s\n",pgp2_path_val));
	    return_value = pgp_none;
	}
	if (return_value != pgp_none)
	    break;

	return_value = pgp5;
	/* FALLTROUGH */    
    }

    case pgp5: try_pgp5: {
	/* give_dt_estr_as_str adds / to end */
	const char * pgp5_dir_val  = give_dt_estr_as_str(&pgp5_dir_e, 
							 "pgp5-dir",NULL,NULL);

	if (! pgp5_dir_val ||
	    strcmp(pgp5_dir_val,"none") == 0 || 
	    pgp5_dir_val[0] == '\0') {
	    return_value = pgp_none;
	} else if (pgp5_dir_val[0] == '/') {
	    char * path = elm_message(FRM("%spgpv"),pgp5_dir_val);
	    if (-1 == access(path,EXECUTE_ACCESS)) {
		int err = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantExecute,
				  "Can't execute pgp: %.50s: %.30s"),
			  path, strerror(err));
		
		DPRINT(Debug,5,(&Debug,			           
				"have_pgp: no access %s: %s\n",path,
				strerror(err)));
		return_value = pgp_none;
	    }
	    free(path);
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpPath,
			      "PGP path must start with '/': %.60s"),
		      pgp5_dir_val);
	    DPRINT(Debug,5,(&Debug,			      
			    "have_pgp: bad path: %s\n",pgp5_dir_val));
	    return_value = pgp_none;
	}
    }
	break;
    case gpg: {
	const char * gpg_path_val = give_dt_estr_as_str(&gpg_path_e, "gpg",
							NULL,NULL);

	if (! gpg_path_val ||
	    strcmp(gpg_path_val,"none") == 0 || 
	    gpg_path_val[0] == '\0') {
	    return_value = pgp_none;
	} else if (gpg_path_val[0] == '/') {
	    if (-1 == access(gpg_path_val,EXECUTE_ACCESS)) {
		int err = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantExecute,
				  "Can't execute pgp: %.50s: %.30s"),
			  gpg_path_val, strerror(err));
	    
		DPRINT(Debug,5,(&Debug,
				"have_pgp: no access %s: %s\n",gpg_path_val,
				strerror(err)));
		return_value = pgp_none;
	    }
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpPath,
			      "PGP path must start with '/': %.60s"),
		      gpg_path_val);
	    DPRINT(Debug,5,(&Debug,
			    "have_pgp: bad path: %s\n",gpg_path_val));
	    return_value = pgp_none;
	}
	if (return_value != pgp_none)
	    break;

	return_value = pgp5;
	goto try_pgp5;

    default:
	break;
    }
	break;

    case PGP_NUM:
	break;
    }
    DPRINT(Debug,5,(&Debug,
		    "have_pgp(%d)=%d\n",v,return_value));
    return return_value;
}

int pgp_void_passphrase ()
{
    int i;
    enum pgp_version v;
    int X = 0;
 
    for (v = pgp_none; v < PGP_NUM; v++) {
	if (pgp_passphrase[v][0])
	    X++;
	for (i = 0 ; i < sizeof pgp_passphrase[v] ; i++)
	    pgp_passphrase[v][i] = '\0';
	pgp_expires = 0;
    }
    return X;
}

static int QueryExpirePassphrase P_((void));

static int QueryExpirePassphrase()
{
    struct timeval now;

    /* negative implies never expire */
    if (pgp_keeppassfor < 0)
	return(0);
    gettimeofday(&now, 0);
    if (now.tv_sec < pgp_expires) {
	pgp_expires = now.tv_sec + pgp_keeppassfor;
	return(0);
    }
    pgp_void_passphrase ();
    return(1);
}

static int GetPassphrase P_((enum pgp_version v));

static int GetPassphrase (v)
     enum pgp_version v;
{
    char * buffer;
    struct timeval now;

    DPRINT(Debug,2,(&Debug,
		    "GetPassphrase(%d) -- pgp_names[%d]=%s\n",
		    v,v,pgp_names[v]));

    buffer = lib_prompt(1,
			CATGETS(elm_msg_cat, MeSet, MePgpEnterPassphrase,
				"Please enter your %s passphrase: "),
			pgp_names[v]);
    if (!buffer) 
	return 0;
    
    strfcpy(pgp_passphrase[v],buffer,sizeof pgp_passphrase[v]);
    
    gettimeofday(&now, 0);
    if (pgp_keeppassfor > 0)
	pgp_expires = now.tv_sec + pgp_keeppassfor;

    free(buffer);
    return(pgp_passphrase[v][0] != '\0');
}

int pgp_goodPassphrase(v)
     enum pgp_version v;
{
    if (pgp_passphrase[v][0] == '\0' || QueryExpirePassphrase())
	return(GetPassphrase(v));
    else
	return(1);
}

static void close_them P_((struct run_state *rs));
static void close_them(rs)
     struct run_state *rs;
{
    int *array = rs->ext_init_data;

    close(array[0]);
    close(array[1]);
    if (array[2] != -1)
	close(array[2]);
}

/* opens up a PGP process as a child and returns its stdin and stdout */
int pgp_decrypt_init (fpin, fpout, opts, v, rs)
     FILE **fpin, **fpout;
     int opts;              /* PGP_MESSAGE, PGP_SIGNED_MESSAGE, or
			       PGP_PUBLIC_KEY */
     enum pgp_version v;    /* 'minimum version' */
     struct run_state *rs;
{
    int pgp_child_in[2];
    int pgp_child_out[2];
    int array[3];       
    const char * argv[10];
    char * env[2];
    int passpipe[2];
    int usepass=FALSE;
    int code;

    DPRINT(Debug,2,(&Debug,		      
		    "pgp_descrypt_init() called with opts=%d\n", opts));
    

    rs->save_errno = 0;
    
    if (pipe(pgp_child_in) == -1) {
	rs->save_errno = errno;    
	return(0);
    }
    if (pipe(pgp_child_out) == -1) {
	rs->save_errno = errno;    
	close(pgp_child_in[0]);
	close(pgp_child_in[1]);
	return(0);    
    }
    if ((opts & PGP_MESSAGE) && pgp_keeppass) {
	if (pipe(passpipe) == -1) {
	    rs->save_errno = errno;
    
	    close(pgp_child_in[0]);
	    close(pgp_child_out[0]);
	    close(pgp_child_in[1]);
	    close(pgp_child_out[1]);
	    return(0);
	}
	usepass = TRUE;
    }
    DPRINT(Debug,3,(&Debug,
		    "usepass = %d.\n", usepass));
  
    sr_call_Raw(OFF);
    sr_call_ClearScreen();
    
    /* Tell the user why they are waiting */
    if (opts & PGP_MESSAGE) {
	sr_call_Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpDecryptMes,
				"Running pgp: Decrypting message...\n"));
    } else if (opts & PGP_SIGNED_MESSAGE) {
	sr_call_Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpCheckSig,
				"Running pgp: Checking signature...\n"));
    } else if (opts & PGP_PUBLIC_KEY) {
	sr_call_Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpExtractKeys,
				"Running pgp: Extracting keys...\n"));
    } else {
	sr_call_Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpRun,
				"Running pgp ...\n"));
    }
    
    array[0] = pgp_child_in[1];
    array[1] = pgp_child_out[0];
    array[2] = -1;

    rs->ext_init_data = array;
    rs->ext_init      = close_them;
    
    env[0] = NULL;
    rs->ext_env = env; 

    switch(v) {
    case pgp2:
    case pgp5:
	if (usepass) {
	    static char buffer[20];
	    elm_sfprintf(buffer, sizeof buffer,FRM("PGPPASSFD=%d"),
			 passpipe[0]);
	    env[0] = buffer;
	    array[2] = passpipe[1];
	}
	env[1] = NULL;

    default:
	break;

    }

    switch(v) {
	static char path[1000];
	int n;
    case pgp2:	
	n = 0;
	{
	    const char * pgp2_path_val = give_dt_estr_as_str(&pgp2_path_e, 
							     "pgp2",NULL,NULL);
		
	    if (! pgp2_path_val)
		return 0;
	    argv[n++] = pgp2_path_val;
	}
	argv[n++] = "-f";
	argv[n++] = "+verbose=0";
	argv[n++] = "+KEEPBINARY=OFF";
	if (usepass || opts == PGP_SIGNED_MESSAGE || opts == PGP_PUBLIC_KEY)
	    argv[n++] = "+batchmode";
	if (opts == PGP_PUBLIC_KEY)
	    argv[n++] = "-ka";
	argv[n] = NULL;
	break;
    case pgp5:
	n = 0; 
	{
	    /* give_dt_estr_as_str adds / to end */
	    const char * pgp5_dir_val  = give_dt_estr_as_str(&pgp5_dir_e, 
							     "pgp5-dir",
							     NULL,NULL);
	    if (! pgp5_dir_val)
		return 0;
	    elm_sfprintf(path, sizeof path,FRM("%spgpv"),pgp5_dir_val);

	}
	argv[n++] = path;
	if (opts != PGP_PUBLIC_KEY)
	    argv[n++] = "-f";
	argv[n++] = "+verbose=0";
	if (usepass || opts == PGP_SIGNED_MESSAGE || opts == PGP_PUBLIC_KEY)
	    argv[n++] = "+batchmode";
	argv[n] = NULL;
	break;
    case gpg:	
	n = 0;
	{
	    const char * gpg_path_val = give_dt_estr_as_str(&gpg_path_e, "gpg",
							    NULL,NULL);
	    if (! gpg_path_val)
		return 0;
	    argv[n++] = gpg_path_val;
	}
	if (usepass) {
	    static char buffer[10];
	    argv[n++] = "--passphrase-fd";
	    elm_sfprintf(buffer, sizeof buffer,FRM("%d"),
			 passpipe[0]);
	    argv[n++] = buffer;
	    array[2] = passpipe[1];
	}	
	if (usepass || opts == PGP_SIGNED_MESSAGE || opts == PGP_PUBLIC_KEY)
	    argv[n++] = "--batch";
	if (opts == PGP_PUBLIC_KEY)
	    argv[n++] = "--import";
	else 
	    argv[n++] = "--decrypt";
	argv[n] = NULL;
	break;

    default:
	break;
    }

    code = start_run(rs, SY_RUN_STATE_INIT|SY_RUN_STATE_ENV , argv ,
		     pgp_child_in[0],pgp_child_out[1]);
	  
    close (pgp_child_in[0]);
    close (pgp_child_out[1]);
  
    if (!code)
	return 0;

    /* now send the passphrase if needed */
    if (usepass) {
	int ok = 1;
	DPRINT(Debug,3,(&Debug,			   
			"pgp_decrypt_init: sending pgp passphrase.\n"));
	close (passpipe[0]);

	if (-1 == write (passpipe[1], pgp_passphrase[v], 
			 strlen(pgp_passphrase[v])))
	    ok = 0;
	if (-1 == write (passpipe[1], "\n", 1))
	    ok = 0;

	if (-1 == close (passpipe[1]))
	    ok = 0;

	if (!ok) {
	    int tmp;
	    int err = errno;

	    DPRINT(Debug,3,(&Debug,
			    "pgp_decrypt_init: write failed\n"));

	    kill(rs->pid,SIGTERM);
	    wait_end(rs,&tmp);	    
	    rs->save_errno = err;
	    sr_call_Raw(ON);
	    
	    return 0;
	}
    }
  
    if ((*fpin = fdopen(pgp_child_out[0], "r")) == 0) {
	int tmp;
	int err = errno;

	DPRINT(Debug,3,(&Debug,
			"pgp_decrypt_init: fdopen(%d) failed, errno=%d\n",
			pgp_child_out[0],err));

	kill(rs->pid,SIGTERM);
	wait_end(rs,&tmp);	    
	rs->save_errno = err;
	sr_call_Raw(ON);

	return(0);
    }
    if ((*fpout = fdopen(pgp_child_in[1], "w")) == 0) {
	int err = errno;
	int tmp;

	DPRINT(Debug,3,(&Debug,
			"pgp_decrypt_init: fdopen(%d) failed, errno=%d\n",
			pgp_child_out[0],err));

	kill(rs->pid,SIGTERM);
	wait_end(rs,&tmp);	    
	rs->save_errno = err;
	sr_call_Raw(ON);

	return(0);
    }
  
    return(code);
}

static int pgp_mime_opts P_((struct mime_param *s));

static int pgp_mime_opts (s)
     struct mime_param *s;
{        
    if (s) {
	const char *pv;
	
	pv = get_mime_param_compat(s,"format");
	if (pv) {
	    if (istrcmp (pv, "keys-only") ==  0)
		return PGP_PUBLIC_KEY;
	}

	pv = get_mime_param_compat(s,"x-action");
	if (pv) {
	    if (istrcmp (pv, "encryptsign") == 0)
		return (PGP_MESSAGE | PGP_SIGNED_MESSAGE);
	    else if (istrcmp (pv, "encrypt") == 0)
		return PGP_MESSAGE;
	    else if (istrcmp (pv, "sign") == 0)
		return PGP_SIGNED_MESSAGE;
	    else if (istrcmp (pv, "signclear") == 0)
		return PGP_SIGNED_MESSAGE;
	}
    }
    return PGP_MESSAGE;
}

enum pgp_version decode_pgp_version(c)
     char * c;
{
    int se =  give_dt_enumerate_as_int(&send_pgp_version);

    if (0 == strncmp(c,"2.",2)) {
	DPRINT(Debug,4,(&Debug,
			"decode_pgp_version: Version -- PGP 2\n"));
	return pgp2;
    } else if (0 == strncmp(c,"GnuPG",5)) {
	DPRINT(Debug,4,(&Debug,			   
			"decode_pgp_version: Version -- GnuPG\n"));
	return gpg;
    } else {
	/* give_dt_estr_as_str adds / to end */
	const char * pgp5_dir_val  = give_dt_estr_as_str(&pgp5_dir_e, 
							 "pgp5-dir",
							 NULL,NULL);

	DPRINT(Debug,4,(&Debug,
			"decode_pgp_version: Version -- PGP 5?\n"));
	if (pgp5_dir_val &&
	    '/' == pgp5_dir_val[0])
	    return pgp5;
	else
	    return (enum pgp_version) se;
    }
}


/* Returns -1 on failure otherwise same mask than mime_classify_media() */
#if ANSI_C
mime_run_selector pgp_selector;
#endif
int pgp_selector(p,hdr)     
     mime_t *p;
     struct header_rec * hdr;
{
    int flags = 0;


    DPRINT(Debug,11,(&Debug, 
		     "pgp_selector(%p) -- type: %p=%s/%s\n",
		     p,
		     p->TYPE,
		     get_major_type_name(p->TYPE), 
		     get_subtype_name(p->TYPE)));

    DPRINT(Debug,11,(&Debug, 
		     "pgp_selector(%p) = %d\n",
		     p,
		     flags));


    return flags;
}

#ifdef ANSI_C
CT_decoder pgp_decode;
#endif
void pgp_decode (m, s_in, s_out, decode_opt, defcharset, mss, badtype, defbodycharset)
     mime_t *m;
     struct in_state  *s_in;
     struct out_state *s_out;
     const struct decode_opts *decode_opt;
     charset_t defcharset;
     struct header_rec * mss;
     type_mismatch_prompt *badtype;
     charset_t defbodycharset;
{
    /* This procedure implements the de-facto standard for using PGP with MIME.
     * Content-Type: application/pgp
     * Required-Parameters: none
     * Optional parameters: format, x-action
     *     format = mime | text | keys-only
     *         mime : indicates that the signed/encrypted body contains a MIME
     *                compliant body and should be parsed recursively.
     *         text : [DEFAULT if there is no format option].  This option
     *                means that the encrypted/signed data should be presented
     *                to the user after processing, no additional processing
     *                needed.
     *         keys-only:
     *                The data in the body represents public key data only
     *     x-action = encryptsign | encrypt | sign
     *         This keyword is meant to be helpful to the application, but is
     *         not required, and may not even be necessary to look at.
     *
     *         encryptsign : the application/pgp data is both signed and
     *                       encrypted.
     *         encrypt     : the data is encrypted only
     *         sign        : the data is signed only
     */
    
    char buffer[LONG_STRING];
    FILE *pgpout, *pgpin, *tmpfp=NULL;
    int
	code = 0,
	inbody = FALSE,
	opts,
	len,
	raw,
	stat = -1,
	bytes = 0,
	nested = FALSE; /* PGP output should be parsed as a MIME body */

    int skip_flag = 0;
    mime_t *tmpmt;
    enum pgp_version v = pgp2;
    int armor_header = 0;

    struct run_state RS;
    
    int was_binary = 0;
    struct in_state * newstate2 = new_in_state(STATE_in_decode);
    enum pgp_version version;
    const char *tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    const char *pv;
    
    int flags =  STATE_DECODE_buffer;   /* seek is used */
    int is_text = decoder_is_textual(m);
    
    struct pager_range *title_range = 
	state_add_simple_pager_range(s_out,NULL,PR_MAX_WIDTH,0,0);

    DPRINT(Debug,4,(&Debug,
		    "pgp_decode: is_text = %d\n", is_text));

    if (is_text > 0)
	flags |=  STATE_DECODE_is_text;
    
    if (!tmp)
	goto FAIL3;

    raw = sr_call_RawState ();
    
    if (
	(pv = get_mime_param_compat(m->TYPE_opts,"format")) &&
	(istrcmp (pv, "mime") == 0)) {
	char * tempfile = elm_message(FRM("%selmPT%d"), 
				      tmp, getpid ());
	
	nested = TRUE;
	DPRINT(Debug,3,(&Debug,			   
			"pgp_decode: format=mime\n"));
	
	
	if (NULL == (tmpfp = safeopen_rdwr(tempfile,NULL))) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantCreate,
			      "Failed to create file for decoding."));

	    /* \n resets this */
	    set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",s_out);
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpCantCreate,
				 "Failed to create file for decoding."));      
	    state_puts(" ]\n",s_out);

	    free(tempfile);

	    goto FAIL3;
	}
	
	unlink (tempfile); /* Filename is no longer needed ... */
	free(tempfile);
    }
    else
	tmpfp = NULL;

    opts = pgp_mime_opts(m->TYPE_opts);
    
    buffer[0] = '\0';
    if (opts & PGP_PUBLIC_KEY) {
	state_printf (s_out,
		      CATGETS(elm_msg_cat, MeSet, MePgpPublicKeys,
			      "(** This message contains PGP public key(s) **)\n"));
	state_putc('\n',s_out);
    }

    /* Decode Content-transfer-encoding */
    

    if (set_in_state_decode(s_in,newstate2,m->length,flags,m->encoding,NULL)) {

	/* If STATE_DECODE_buffer was treated this as text, don't consider
	 * about binary PGP files -- that happens when type is text/x-pgp
	 */


	if (check_type_pattern) {
	    if (!check_type_magic(m,newstate2, s_out, decode_opt, badtype)) {
		DPRINT(Debug,11,(&Debug, 
				 "pgp_decode: mime_type_rejected\n"));
		goto FAILTYPE;
	    }
	}
       	
	/* Print text before PGP armor 
	 *
	 */

	len = state_getl (buffer, sizeof (buffer), newstate2);    
	if ( len < 1) {

	    /* \n resets this */
	    set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpNoText1,
				 "[ No text in PGP section. ]\n"));

	} else {
	    int c = (unsigned char) buffer[0],i;
	    
	    if (!pgp_noarmor && opts != PGP_PUBLIC_KEY) {
		was_binary = 1; /* default assumption */
		goto pgp_found;
	    }
	    
	    if (0 == is_text) {
		/* Check if that is binary PGP file */
		if (c & 0x80) {
		    /* In PGP block have type byte which have higgest bit set 
		     * always. so if in first byte have higgest bit set assume
		     * PGP binary file.
		     */
		    DPRINT(Debug,4,(&Debug, 
				    "pgp_decode: first byte = %d -- assume binary PGP file\n",
				c));
		    was_binary = 1; 
		    goto pgp_found;
		}
		/* Another check */
		for (i = 0; i < len; i++) {
		    if (buffer[i] == 0) {
			DPRINT(Debug,4,(&Debug,
					"pgp_decode: byte idx=%d is zero -- binary file?\n",
					i));
			
			/* \n resets this */
			set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
			state_printf(s_out,
				     CATGETS(elm_msg_cat, MeSet, MePgpBinary,
					     "[ Binary file, but does not look like PGP ]\n"));

			was_binary = 1; 
			goto pgp_found;
		    }
		}
	    }
	    if (strncmp(buffer, "-----BEGIN PGP", 14) == 0) 
		goto pgp_found;
	    
	    /* \n resets this */
	    set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpBefore,
				 "[ There is text before PGP section. ]\n"));

	    if (set_filter(m,s_out,decode_opt,
			   defbodycharset ? defbodycharset :
			   NULL /* not default_mimetext_charset? */ ,
			   mss ? mss->override_charset : NULL,
			   NULL)
		) { /* Character set filtering */
		do {
		    state_add_prefix(s_out,decode_opt);

		    /* Take care of CRLF => LF conversion in here because
		     * STATE_DECODE_buffer is treating application/pgp as
		     * binary type (it need to treate it as binary because
		     * it may be binary)
		     */
		    if (0 == is_text &&
			len > 1 &&
			buffer[len - 1] == '\n' &&
			buffer[len - 2] == '\r') {
			buffer[len - 2] = '\n';
			buffer[len - 1] = '\0';
			len--;
		    }

		    state_put(buffer,len,s_out);
		    
		} while ((len = state_getl (buffer, sizeof (buffer), 
					    newstate2)) > 0
			 && strncmp(buffer, "-----BEGIN PGP", 14) != 0);
		
	    } else {
		DPRINT(Debug,4,(&Debug, 
				"pgp_decode: Unsupported character set?\n"));
	    }
	    
	pgp_found:
	    set_out_state_filter(s_out,NULL);
	}

    } else { 
	/* \n resets this */
	set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);

	if (opts & PGP_MESSAGE)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MePgpEncodedCantEncoding,
				 "-- Start of PGP encoded section -- can't decode content-transfer-encoding\n"));
	else if (opts & PGP_SIGNED_MESSAGE)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MePgpSignedCantEncoding,
				 "-- Start of PGP signed section -- can't decode content-transfer-encoding\n"));
	else
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpCantEncoding,
				      "-- Start of PGP section -- can't decode content-transfer-encoding\n"));

	goto FAIL3;
    }

    if ( len < 1) {
      
	/* \n resets this */
	set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	if (opts & PGP_MESSAGE)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpNoTextEncoded,
				    "[ No text in PGP encoded section ]\n"));
	else if (opts & PGP_SIGNED_MESSAGE)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpNoTextSigned,
				    "[ No text in PGP signed section ]\n"));
	else
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpNoText,
				 "[ No text in PGP section ]\n"));

      goto FAIL3;
    }

  
    if (!was_binary) {
      int l = in_state_ftell(newstate2);
      int len1;
      char buf[STRING];

      /* 
       * On PGP 2 messages these is empty line immediately after
       * -----BEGIN PGP SIGNED MESSAGE----
       * 
       * If there is something other such as 
       * Hash: SHA1
       * PGP 2 does not understand message.
       */

      while ((len1 = state_getl (buf, sizeof (buf), newstate2)) > 0) {
	  if ((len1 == 1 && buf[0] == '\n') ||
	      (len1 == 2 && buf[0] == '\r' && buf[1] == '\n'))
	      break;
	    if (opts == PGP_SIGNED_MESSAGE) {
		DPRINT(Debug,4,(&Debug, 
				"pgp_decode: peek: Header on armor -- requires PGP 5 or GnuPG\n" ));
		v = gpg;
		armor_header++;
		break;
	    }
	    if (0 == strncmp("Version: ",buf,9)) {
		char *c = buf+9;
		
		v = decode_pgp_version(c);
		if (armor_header && pgp2 == v) 
		    v = gpg;
	    }
      }

      /* Look also for -----BEGIN PGP SIGNATURE----- ... */
      if (opts == PGP_SIGNED_MESSAGE && len1 > 0) {
	  DPRINT(Debug,4,(&Debug,			     
			  "pgp_decode: Looking for -----BEGIN PGP SIGNATURE\n"));
	  
	  while ((len1 = state_getl (buf, sizeof (buf), newstate2)) > 0) {
	      if (len1 > 24 &&
		  0 == strncmp(buf,
			       "-----BEGIN PGP SIGNATURE",24))
		  break;
	  }
	  
	  while ((len1 = state_getl (buf, sizeof (buf), 
				     newstate2)) > 0) {
	      if ((len1 == 1 && buf[0] == '\n') ||
		  (len1 == 2 && buf[0] == '\r' && buf[1] == '\n'))
		  break;
	      
	      if (0 == strncmp("Version: ",buf,9)) {
		  char *c = buf+9;
		  
		  v = decode_pgp_version(c);
		  if (armor_header && pgp2 == v) 
		      v = gpg;
	      }
	  }
      }
      
      in_state_fseek(newstate2,l);
    }

    version = have_pgp(v);

    if (!version) {

	/* \n resets this */
	set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	if (was_binary) {
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpNotAvailSkipping,
				 "[ PGP not available, skipping... ]\n"));
	    
	    goto FAIL3;
	} else 
	    state_printf (s_out,
			  CATGETS(elm_msg_cat, MeSet, MePgpNotAvailRawdata,
				  "[ PGP not available, raw data follows ]\n"));

	goto fail;
    }

    if ((opts & PGP_MESSAGE) && pgp_keeppass) {
	if (!pgp_goodPassphrase(version)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpBadPassphrase,
			      "Decrypting message... Bad PGP passphrase."));

	    /* \n resets this */
	    set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	    state_putc('[',s_out);
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpBadPassphrase,
				 "Decrypting message... Bad PGP passphrase."));
	    state_puts("]\n",s_out);
	    
	    goto FAIL3;
	}
    }

    if (!pgp_decrypt_init (&pgpout, &pgpin, opts, version, &RS)) {

	/* \n resets this */
	set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	if (was_binary)  {
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpInternalSkipping,
				 "[ Internal error while calling pgp, skipping... ]\n"));
	    goto FAIL3;
	} else {
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpInternalRawdata,
				 "[ Internal error while calling pgp, raw data follows ]\n"));	    
	fail:

	    do {
	      state_add_prefix(s_out, decode_opt);
	
	      if (len > 1 &&
		  buffer[len - 1] == '\n' &&
		  buffer[len - 2] == '\r') {
		  buffer[len - 2] = '\n';
		  buffer[len - 1] = '\0';
		  len--;
	      }
	      
	      state_put(buffer,len,s_out);
	      bytes += len;
	  }  while ((len = state_getl (buffer, sizeof (buffer), 
				       newstate2)) > 0);
	  
	    /* \n resets this */
	    set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpRawEnd,
				 "[ End of raw data. ]\n"));
      }

      goto FAIL3;
    }
 

    do {
	fwrite(buffer,1,len,pgpin);
	bytes += len;
    }  while ((len = state_getl (buffer, sizeof (buffer), 
				 newstate2)) > 0);

    fclose (pgpin);
  
    code = run_already_done(&RS,&stat);
	     
    if (code != 0) {
	/* \n resets this */
	set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	
	if (opts & PGP_MESSAGE)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartEncoded1,
				 "-- Start of PGP encoded section%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						    MePgpFail,
						    ", PGP failed!") : ".");
	else if (opts & PGP_SIGNED_MESSAGE)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartSigned1,
				 "-- Start of PGP signed section%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						    MePgpFail,
						    ", PGP failed!") : ".");
	else if (opts & PGP_PUBLIC_KEY)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartOutput1,
				 "-- Start of PGP output%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						    MePgpFail,
						    ", PGP failed!") : ".");
	else
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStart1,
				 "-- Start of PGP section%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						    MePgpFail,
						    ", PGP failed!") : ".");
	
    } else {
	/* \n resets this */
	set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	
	if (opts & PGP_MESSAGE)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartEncoded,
				 "-- Start of PGP encoded section.\n"));
	else if (opts & PGP_SIGNED_MESSAGE)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartSigned,
				 "-- Start of PGP signed section.\n"));
	else if (opts & PGP_PUBLIC_KEY)
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartOutput,
				 "-- Start of PGP output.\n"));
	else
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStart,
				 "-- Start of PGP section.\n"));
    }
    
    bytes = 0;
    
    if (is_text > 0 && !nested) {

      if (set_filter(m,s_out,decode_opt,NULL,
		     mss ? mss->override_charset : NULL,
		     NULL)) {

	  DPRINT(Debug,4,(&Debug, 
			  "pgp_decode: Filtering according of charset enabled.\n"));
  
      } else {
	  DPRINT(Debug,4,(&Debug, 
			  "pgp_decode: Unsupported character set?\n"));

	  skip_flag++;
      }
      
    }
    
 retry:
    
    while ((len = mail_gets (buffer, sizeof (buffer), pgpout)) > 0) {
	if (nested) {
	    if (buffer[0] == '\n' || (buffer[0] == '\r' && buffer[1] == '\n'))
		inbody = TRUE;
	    fputs (buffer, tmpfp);
	    if (inbody)
		bytes += len;
	} else if (!skip_flag) {
	    state_add_prefix(s_out, decode_opt);
	    state_puts(buffer,s_out);
	}
    }
 
    if (ferror(pgpout) && EINTR == errno) {
	clearerr(pgpout);
	DPRINT(Debug,5,(&Debug,
			"Reading of result interrupted (EINTR) -- retrying\n"));
	
	if (0 == code) {
	    code = run_already_done(&RS,&stat);
	    if (0 != code) {
		DPRINT(Debug,5,(&Debug,
				"now pgp/gpg is completing\n"));
	    }      
	}
	
	goto retry;
    }

    set_out_state_filter(s_out,NULL);
  
    fclose (pgpout); pgpout = NULL;

    if (nested) {
	struct in_state * s2_in = new_in_state(STATE_in_file);
	
	DPRINT(Debug,3,(&Debug,
			"pgp_decode: parsing decrypted data as MIME\n"));
	
	if (EOF == fflush(tmpfp)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpErrorFlush,
			      "Error when flushing temporary file."));

	    /* \n resets this */
	    set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	    state_putc('[',s_out);
	    state_printf(s_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpErrorFlush,
				 "Error when flushing temporary file."));
	    state_puts("]\n",s_out);

      }
      rewind(tmpfp); /* Rewind it for reading */
      
      tmpmt = mime_read_header (tmpfp, 0, defcharset, 
				mss ? & (mss->header_error) : NULL,
				mss ? mss->status: 0,mss);
      tmpmt->length = bytes;

      if (! tmpmt->TYPE) {

	  lib_error(CATGETS(elm_msg_cat, MeSet, 
			    MeParseBadCTEncryptedSubPart,
			    "Bad content-type on encrypted subpart"));

	  /* \n resets this */
	  set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
	  state_puts("[ ",s_out);
	  state_printf(s_out,
		       CATGETS(elm_msg_cat, MeSet, 
			       MeParseBadCTEncryptedSubPart,
			       "Bad content-type on encrypted subpart"));
	  state_puts(" ]\n",s_out);


      } else {
	  /* Pass NULL as message, because this is encrypted section */
	  
	  mime_parser_parse(tmpmt,defcharset,tmpfp,
			    mss ? & (mss->header_error) : NULL,
			    mss ? mss->status : 0,
			    NULL
			    );
	  
	  set_in_state_file(tmpfp,s2_in);
      
	  /* Pass NULL as message, because this is decrypted content */
	  mime_decode (tmpmt, s2_in, s_out, decode_opt, defcharset, NULL, badtype,
		       defbodycharset /* or NULL because this is encrypted section ? */);

      }

      mime_t_clear(tmpmt);
 
      free_in_state(&s2_in);
      
      fclose (tmpfp);
    }
  
    if (0 == code)
	code = wait_end(&RS,&stat);

    DPRINT(Debug,10,(&Debug,"pgp_decode: code=%d, exit status=%d\n",
		     code,stat));

    call_print_status_cooked(&RS,code < 0 ? -code : 0,stat);

    if (raw)
      sr_call_Raw (ON);

    /* \n resets this */
    set_out_state_line_mode(s_out,pg_BOLD,title_range,1 /* Newline */);
    if (opts & PGP_MESSAGE)
	state_printf(s_out,
		     CATGETS(elm_msg_cat, MeSet, MePgpEndEncoded,
			     "-- End of PGP encoded section%s\n"),
		     code < 0 || stat ? catgets(elm_msg_cat, MeSet, MePgpFail,
						", PGP failed!") : ".");
    else if (opts & PGP_SIGNED_MESSAGE)
	state_printf(s_out,
		     CATGETS(elm_msg_cat, MeSet, MePgpEndSigned,
			     "-- End of PGP signed section%s\n"),
		     code < 0 || stat ?  catgets(elm_msg_cat, MeSet, MePgpFail,
						 ", PGP failed!") : ".");
    else if (opts & PGP_PUBLIC_KEY)
	state_printf(s_out,     
		     CATGETS(elm_msg_cat, MeSet, MePgpEndOutput,
			     "-- End of PGP output%s\n"),
		     code < 0 || stat ?  catgets(elm_msg_cat, MeSet, MePgpFail,
						 ", PGP failed!") : ".");
    else
	state_printf(s_out,
		     CATGETS(elm_msg_cat, MeSet, MePgpEnd,
			     "-- End of PGP section%s\n"),
		     code < 0 || stat ? catgets(elm_msg_cat, MeSet, MePgpFail,
						", PGP failed!") : ".");    
    
 FAILTYPE:
    
 FAIL3:
    free_in_state(&newstate2);
    free_pager_range(&title_range);
}

#ifdef ANSI_C
signed_decoder pgp_SG_decoder;
#endif
void pgp_SG_decoder(body,sign,state_in,state_out,
		    decode_opt,micalg, defcharset, mss,
		    badtype,bodydefcharset)
     mime_t *body; 
     mime_t *sign;
     struct in_state   *state_in;
     struct out_state  *state_out;  
     const struct decode_opts *decode_opt;
     const char         *micalg;
     charset_t    defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    int exit_code = -1;
    FILE *binary_fp;
    char *body_name = NULL;
    struct in_state * newstate2 = new_in_state(STATE_in_file);
    enum pgp_version v = pgp2;
    enum pgp_version version;

    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,0);

    if (0 == istrcmp(micalg,"pgp-md5")) {
	DPRINT(Debug,4,(&Debug,
			"pgp_SG_decoder: micalg=%s, PGP 2\n",
			micalg));
	v = pgp2;
    } else { 
	DPRINT(Debug,4,(&Debug, 
			"pgp_SG_decoder: micalg=%s, GPG\n",
			micalg));
	v = gpg;
    }

    if (NULL != (binary_fp = arrange_binary(body,state_in,state_out,newstate2,
					    &body_name))) {	

	struct in_state * newstate3 = new_in_state(STATE_in_decode); 
	const char *sign_name = NULL;   /* reference to STATE_in_decode data */

	int flags = STATE_DECODE_buffer;/* STATE_in_decode unlinks file on 
					   free 
					*/
	int errors = 0;

		
	if (in_state_fseek(state_in,sign->offset) != 0) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpSGSeekFail,
			      "pgp_SG_decoder: seek failed"));

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_putc('[',state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpSGSeekFail,
				 "pgp_SG_decoder: seek failed"));
	    state_puts("]\n",state_out);	    
	}


	if (set_in_state_decode_helper(sign,state_in,newstate3,
				       flags,&sign_name)) {
	    
	    int i = 0;
	    int len1;
	    char buf[STRING];
	    
	    while ((len1 = state_getl (buf, sizeof (buf), newstate3)) > 0) {
		if (i == 0) {
		    if (strncmp(buf, "-----BEGIN PGP", 14) != 0)
			break;
		    i++;
		    continue;
		}
		if ((len1 == 1 && buf[0] == '\n') ||
		    (len1 == 2 && buf[0] == '\r' && buf[1] == '\n'))
		    break;
		if (0 == strncmp("Version: ",buf,9)) {
		    char *c = buf+9;

		    v = decode_pgp_version(c);
		}
	    }
	    in_state_fseek(newstate3,0);
	    
	    /* Print total 10 error messages */
	    errors = print_in_errors(newstate3,state_out,decode_opt);

	    if ((version=have_pgp(v))) {

		struct run_state RS;
		int stat;
		const char * argv[10];
		int raw = sr_call_RawState ();
		
		
		switch(version) {
		    static char path[1000];
		case pgp2:	
		    {
			const char * pgp2_path_val = 
			    give_dt_estr_as_str(&pgp2_path_e, "pgp2",
						NULL,NULL);
			if (! pgp2_path_val)
			    return;
			
			argv[0] = pgp2_path_val;
		    }
		    argv[1] = "+batchmode";
		    argv[2] = "+TEXTMODE=off";
		    argv[3] = "+CHARSET=noconv";
		    argv[4] = sign_name;
		    argv[5] = body_name;
		    argv[6] = NULL;
		    break;
		case pgp5:
		    {
			/* give_dt_estr_as_str adds / to end */
			const char * pgp5_dir_val  = 
			    give_dt_estr_as_str(&pgp5_dir_e,"pgp5-dir",
						NULL,NULL);

			if (! pgp5_dir_val)
			    return;
			elm_sfprintf(path, sizeof path,FRM("%spgpv"),
				     pgp5_dir_val);
		    }
		    argv[0] = path;
		    argv[1] = "+batchmode";
		    argv[2] = "+TEXTMODE=off";
		    argv[3] = "+CHARSET=noconv";
		    argv[4] = sign_name;
		    argv[5] = body_name;
		    argv[6] = NULL;
		    break;
		case gpg: 
		    {
			const char * gpg_path_val = 
			    give_dt_estr_as_str(&gpg_path_e, "gpg",NULL,NULL);

			if (! gpg_path_val)
			    return;

			argv[0] = gpg_path_val;
		    }
		    argv[1] = "--batch";
		    argv[2] = "--verify";
		    argv[3] = sign_name;
		    argv[4] = body_name;
		    argv[5] = NULL;
		    break;

		default:
		    argv[0] = NULL;
		    break;
		}
		    
		sr_call_Raw(OFF);
		sr_call_ClearScreen();
		sr_call_Write_to_screen(CATGETS(elm_msg_cat, MeSet, MePgpCheckSig,
					"Running pgp: Checking signature...\n"));

		stat = start_run(&RS,0,argv,-1,-1);
		if (stat)
		    stat = wait_end(&RS,&exit_code);
		
		DPRINT(Debug,10,(&Debug,"pgp_SG_decoder: stat=%d, exit code=%d\n",
				 stat,exit_code));


		call_print_status_cooked(&RS,stat < 0 ? -stat : 0,exit_code);
		
		if (raw)
		    sr_call_Raw (ON);
		
		if (stat) {
		    if (decode_opt->displaying) {
			state_puts("\n",state_out);		

			/* \n resets this */
			set_out_state_line_mode(state_out,pg_BOLD,title_range,
						1 /* Newline */);
			state_puts("[ ",state_out);		
			if (exit_code == 0)
			    state_printf(state_out,
					 CATGETS(elm_msg_cat, MeSet,MePgpSigOK,
						 "Checking application/pgp-signature: OK"));
			else
			    state_printf(state_out,
					 CATGETS(elm_msg_cat, MeSet,MePgpSigFAILURE,
						 "Checking application/pgp-signature: FAILURE"));		
			state_puts(" ]\n\n",state_out);

		    } else if (exit_code != 0) {
			state_puts("\n",state_out);

			/* \n resets this */
			set_out_state_line_mode(state_out,pg_BOLD,title_range,
						1 /* Newline */);
			state_puts("[ ",state_out);		
			state_printf(state_out,
				     CATGETS(elm_msg_cat, MeSet,MePgpSigFAILURE,
					     "Checking application/pgp-signature: FAILURE"));		
			state_puts(" ]\n",state_out);
		    }
		} else {
		    if (RS.save_errno) {
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailErrno,
					  "Failed: %.30s: %.40s"),
				  argv[0],strerror(RS.save_errno));
		    } else
			lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantStart,
				      "Can't start %.30s"),
				  argv[0]);
		}
	    }

	} else {
	    /* Unsupported encoding */

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpEncodingSkipping,
				 "[ Unsupported encoding on application/pgp-signature, skipping... ]\n"));
	    if (decode_opt->displaying) {	
		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, MeDecodeUseVEncodedSave,
				     "[ Use 'v' to save this part in encoded form. ]\n"));
	    }
	}
	
	/* Print total 10 error messages */
	errors += print_in_errors(newstate3,state_out,decode_opt);

	DPRINT(Debug,11,(&Debug, 
			 "pgp_SG_decoder: %d errors\n"));

	free_in_state(&newstate3);
	
	unlink(body_name);
	free(body_name);

	free_in_state(&newstate2);

	if (binary_fp)        /* in_state_destroy does not close file!! */
	    fclose(binary_fp);
    } else
	free_in_state(&newstate2);

    /* \n resets this */
    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
    state_printf(state_out,
		 CATGETS(elm_msg_cat, MeSet, MePgpStartSigned,
			 "-- Start of PGP signed section.\n"));

    /* Pass mss as message, because this is NOT decrypted (only signed) */
    mime_decode(body,state_in,state_out, decode_opt, defcharset, mss,
		badtype,bodydefcharset);

    /* \n resets this */
    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
    state_printf(state_out,
		 CATGETS(elm_msg_cat, MeSet, MePgpEnd,
			 "-- End of PGP section%s\n"),
		 exit_code ? catgets(elm_msg_cat, MeSet, MePgpFail,
				     ", PGP failed!") : ".");

    free_pager_range(&title_range);
}  

#ifdef ANSI_C
encrypted_decoder pgp_EC_decoder;
#endif
void pgp_EC_decoder(init,data,state_in,state_out, decode_opt,defcharset, mss, 
		    badtype,bodydefcharset)
     mime_t *init;
     mime_t *data;
     struct in_state   *state_in;
     struct out_state  *state_out;
     const struct decode_opts *decode_opt;
     charset_t     defcharset;
     struct header_rec *mss;
     type_mismatch_prompt *badtype;
     charset_t bodydefcharset;
{
    char * tempfile = NULL;
    FILE *tmpfp;
    struct in_state * newstate2 = new_in_state(STATE_in_decode);
    const char *tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    int flags =  STATE_DECODE_buffer;   /* seek is used */

    int errors = 0;
  
    struct pager_range *title_range = 
	state_add_simple_pager_range(state_out,NULL,PR_MAX_WIDTH,0,0);

    if (in_state_fseek(state_in,data->offset) != 0) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MePgpECSeekFail,
			  "pgp_EC_decoder: seek failed"));

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_putc('[',state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MePgpECSeekFail,
			     "pgp_EC_decoder: seek failed"));
	state_puts("]\n",state_out);	    
    }

    tempfile = elm_message(FRM("%selmPT%d"), 
			   tmp, getpid ());
	
    if (NULL == (tmpfp = safeopen_rdwr(tempfile,NULL))) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MePgpCantCreate,
			  "Failed to create file for decoding."));

	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_putc('[',state_out);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MePgpCantCreate,
			     "Failed to create file for decoding."));      
	state_puts("]\n",state_out);

	free(tempfile);
	return;
    }
    unlink (tempfile); /* Filename is no longer needed ... */
    free(tempfile); tempfile = NULL;

    
    if (set_in_state_decode_helper(data,state_in,newstate2,flags,NULL)) {

	int i = 0,len1;
	char buf[STRING];
	enum pgp_version v = pgp2;
	enum pgp_version version
;
	struct run_state RS;
	FILE *pgpout,*pgpin;
	int code;
	mime_t *tmpmt;
	int stat;
	long bytes = 0;
	int inbody = FALSE;
	int raw = sr_call_RawState ();
	int decoded_bytes = 0;
	int pgp_bytes = 0;

	while ((len1 = state_getl (buf, sizeof (buf), newstate2)) > 0) {
	    if (i == 0) {
		if (strncmp(buf, "-----BEGIN PGP", 14) != 0)
		    break;
		i++;
		continue;
	    }
	    if ((len1 == 1 && buf[0] == '\n') ||
		(len1 == 2 && buf[0] == '\r' && buf[1] == '\n'))
		break;
	    if (0 == strncmp("Version: ",buf,9)) {
		char *c = buf+9;
		
		v = decode_pgp_version(c);
	    }
	}
	in_state_fseek(newstate2,0);
      
	/* Print total 10 error messages */
	errors = print_in_errors(newstate2,state_out,decode_opt);

	version = have_pgp(v);
	if (!version) {
	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpNotAvailSkipping,
				 "[ PGP not available, skipping... ]\n"));

	    goto FAILURE;
	}

	if (pgp_keeppass) {
	    if (!pgp_goodPassphrase(version)) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MePgpBadPassphrase,
				  "Decrypting message... Bad PGP passphrase."));
		
		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		state_puts("[ ",state_out);
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, MePgpBadPassphrase,
				     "Decrypting message... Bad PGP passphrase."));
		state_puts(" ]\n",state_out);
		
		goto FAILURE;
	    }
	}

	if (!pgp_decrypt_init (&pgpout, &pgpin, PGP_MESSAGE, version, &RS)) {

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpInternalSkipping,
				 "[ Internal error while calling pgp, skipping... ]\n"));
	    goto FAILURE;
	}

	while ((len1 = state_getl (buf, sizeof (buf), 
				   newstate2)) > 0) {
	    fwrite(buf,1,len1,pgpin);

	    decoded_bytes += len1;
	}  
	fclose (pgpin);
	
	DPRINT(Debug,11,(&Debug,
			 "pgp_EC_decoder: %d bytes after content-transfer-encoding decoded\n",
			 decoded_bytes));


	/* Print total 10 error messages */
	errors += print_in_errors(newstate2,state_out,decode_opt);

	code = run_already_done(&RS,&stat);	
	
	bytes = 0;

    retry:
	while ((len1 = mail_gets (buf, sizeof (buf), pgpout)) > 0) {

	    pgp_bytes += len1;

	  if (buf[0] == '\n' || (buf[0] == '\r' && buf[1] == '\n'))
	      inbody = TRUE;
	  fputs (buf, tmpfp);
	  if (inbody)
	      bytes += len1;
	}

	if (ferror(pgpout) && EINTR == errno) {
	    clearerr(pgpout);
	    DPRINT(Debug,5,(&Debug,
			    "Reading of result interrupted (EINTR) -- retrying\n"));

	    if (0 == code) {
		code = run_already_done(&RS,&stat);
		if (0 != code) {
		    DPRINT(Debug,5,(&Debug,
				    "now pgp/gpg is completing\n"));
		}      
	    }

	    goto retry;
	}
  
	DPRINT(Debug,11,(&Debug,
			 "pgp_EC_decoder: %d bytes after pgp decode.\n",pgp_bytes));

	DPRINT(Debug,11,(&Debug,
			 "pgp_EC_decoder: %d body bytes pgp decoded%s.\n",
			 bytes,inbody ? "" : ", body not seen!"));

	fclose(pgpout);

	if (EOF == fflush(tmpfp)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MePgpErrorFlush,
			      "Error when flushing temporary file."));

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpErrorFlush,
				 "Error when flushing temporary file."));
	    state_puts(" ]\n",state_out);
	}
	rewind(tmpfp); /* Rewind it for reading */
      

	if (!inbody && 0 == code) {

	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);

	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MePgpNestedError,
				 "[ Nested MIME structure inside of pgp encoded message is corrupted. ]\n"));
	    

	}

	tmpmt = mime_read_header (tmpfp, 0, defcharset, 
				  mss ? &(mss->header_error) : NULL,
				  mss ? mss->status : 0,
				  mss);
	tmpmt->length = bytes;

	
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	
	if (code != 0) {
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MePgpStartEncoded1,
				 "-- Start of PGP encoded section%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, 
						    MeSet, 
						    MePgpFail,
						    ", PGP failed!") 
			 : ".");
	} else
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartEncoded,
				 "-- Start of PGP encoded section.\n"));
	
	if (! tmpmt->TYPE) {
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeParseBadCTEncryptedSubPart,
			      "Bad content-type on encrypted subpart"));
	
	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",state_out);
	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, 
			      MeParseBadCTEncryptedSubPart,
				 "Bad content-type on encrypted subpart"));
	    state_puts(" ]\n",state_out);
    
	} else {
	    struct in_state * s2_in = new_in_state(STATE_in_file);

	    /* Pass NULL as message, because this is encrypted section */
	    
	    mime_parser_parse(tmpmt,defcharset,tmpfp,
			      mss ? & (mss->header_error) : NULL, 
			      mss ? mss->status : 0,
			      NULL);
 
	    set_in_state_file(tmpfp,s2_in);
	    	    	    	    
	    /* Pass NULL as message, because this is encrypted section */
	    mime_decode (tmpmt, s2_in, state_out, decode_opt, defcharset, NULL,
			 badtype,
			 bodydefcharset  /* or NULL because this is encrypted section ? */);


	    mime_t_clear(tmpmt);
	    
	    free_in_state(&s2_in);
	}
	
	if (0 == code)
	    code = wait_end(&RS,&stat);

	DPRINT(Debug,10,(&Debug,"pgp_EC_decoder: code=%d, exit code=%d\n",
			 code,stat));


	call_print_status_cooked(&RS,code < 0 ? -code : 0,stat);

	if (raw)
	    sr_call_Raw (ON);
	
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);

	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, MePgpEndEncoded,
			     "-- End of PGP encoded section%s\n"),
		     code < 0 || stat ? catgets(elm_msg_cat, MeSet, MePgpFail,
						", PGP failed!") : ".");	

    FAILURE:
	;

    } else {
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, 
			     MePgpEncodedCantEncoding,
			     "-- Start of PGP encoded section -- can't decode content-transfer-encoding\n"));
	

    }

    /* Print total 10 error messages */
    errors += print_in_errors(newstate2,state_out,decode_opt);
    
    DPRINT(Debug,11,(&Debug, 
		     "pgp_EC_decoder: %d errors\n"));

    free_in_state(&newstate2);

    fclose (tmpfp);

    free_pager_range(&title_range);
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
