static char rcsid[] = "@(#)$Id: tagfilter.c,v 2.4 2024/12/12 18:16:34 hurtta Exp $";

/************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.4 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ************************************************************************/

#include "def_melib.h"
#include "tagfilter_imp.h"
#include "s_me.h"

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

DEBUG_VAR(Debug,__FILE__,"mime");

struct tagfilter_selection tagfilter_text_enriched  = {
    TAGFILTER_SELECTION_magic,
    TAGFLT_MODE_double_smaller |  /* Allow << encode < character */
    TAGFLT_MODE_many_linebreak |  /* Convert n linebreak to space and n-1 linebreak */
    TAGFLT_MODE_enriched_tag   |  /* Tag all in US-ASCII, restricted to the 
				     alphanumeric and hyphen ("-") characters */
    TAGFLT_MODE_lowerascii,       /* Tag lowercase ascii names    */
    NULL,                         /* not allow <!                */
    NULL,                         /* not allow &entity;          */
    &text_enriched_tags,          /* All enriched tags?          */
    give_top_text_enriched,
    give_item_text_enriched,
    locate_item_text_enriched,
    end_tag_text_enriched,
    push_tag_text_enriched,
    close_tag_text_enriched,
    start_atr_text_enriched,
    end_atr_text_enriched,
    start_end_atrval_text_enriched,
    atrvalue_seg_str_text_enriched,
    atrvalue_seg_ent_text_enriched,
    stack_item_clear_text_enriched
};


struct tagfilter_selection tagfilter_text_html      = {
    TAGFILTER_SELECTION_magic,
    TAGFLT_MODE_tag_attributes |   /* Allow <tag attributes>      */
    TAGFLT_MODE_lowerascii |           /* Tag lowercase ascii names    */
    TAGFLT_MODE_convert_cr,            /* Convert CR (outside of CRLF) to LF   */
    "HTML",                           /* <!DOCTYPE HTML ... >         */
    & HTML_entities,                  /* Allow &entity;               */
    & text_html_top,                  /* <html> .... </html>          */
    give_top_text_html,
    give_item_text_html,
    locate_item_text_html,
    end_tag_text_html,
    push_tag_text_html,
    close_tag_text_html,
    start_atr_text_html,
    end_atr_text_html,
    start_end_atrval_text_html,
    atrvalue_seg_str_text_html,
    atrvalue_seg_ent_text_html,
    stack_item_clear_text_html
};

/* On UTF-7 optional direct characters are

               Character   ASCII & Unicode Value (decimal)
                  !           33
                  "           34
                  #           35
                  $           36
                  %           37
                  &           38
                  *           42
                  ;           59
                  <           60
                  =           61
                  >           62
                  @           64
                  [           91
                  ]           93
                  ^           94
                  _           95
                  '           96
                  {           123
                  |           124
                  }           125

    Therefore these can be encoded.  Must operate
    with characters when charset is taken account!
*/

#define TAGFILTER_TOKEN_magic		0xFD0B

enum tag_flag_bits {
    tflag_bit_seen_nl,                  /* Seen newline -- start span      */
    tflag_bin_self_closing,             /* Seen self_closing /             */
    tflag_bin_seen_equals,              /* Seen =                          */
    tflag_bin_num_overflow,             /* Overflow on numeric character reference */
    tflag_bin_unhandled_class,          /* Unhandled token class */
    
    NUM_tag_flag_bits
};

#define TFLAG_seen_nl	       (1 << tflag_bit_seen_nl)      /* Seen newline -- start span      */
#define TFLAG_self_closing     (1 << tflag_bin_self_closing) /* Seen self_closing /             */
#define TFLAG_seen_equals      (1 << tflag_bin_seen_equals)  /* Seen =                          */ 
#define TFLAG_num_overflow     (1 << tflag_bin_num_overflow) /* Overflow on numeric character reference */

#define TFLAG_unhandled_class  (1 <<  tflag_bin_unhandled_class)  /* Unhandled token class */

struct tagfilter_token {
    unsigned short               magic;  	/* TAGFILTER_TOKEN_magic */

    enum tf_token_class {
	tf_doctype_error      = -7,
	tf_tag_atrvalue_error = -6,
	tf_tag_param_error = -5,
	tf_comment_error   = -4,
	tf_bcomment_error  = -3,
	tf_tag_error       = -2,
	tf_entity_error    = -1,
	tf_body = 0,
	tf_start_tag,                    /* <tag  */
	tf_whole_tag,                    /* <tag> */
	tf_selfclosed_tag,               /* <tag/> */
	tf_bcomment_start,               /* <? or <!  or </ */
	tf_bcomment_chunk,               /* <? bogus comment chunk */
	tf_bcomment_end,                 /* end bogus comment > */
	tf_start_endtag,                 /* </tag  */
	tf_whole_endtag,                 /* </tag> */
	tf_entity,
	tf_numeric_entity,               /* Numeric entity */
	tf_double_smaller,               /* << as escaping */
	tf_span_nl,                      /* span of newline (except first)
					  */
	tf_comment_start,                /* <!-- */
	tf_whole_comment,                /* <!----> */
	tf_comment_chunk,                /* <!-- comment chunk  */
	tf_comment_end,                  /* end comment -->     */
	tf_tag_space,                    /* space on attributes */
	tf_tag_atrname,                  /* Got attribute name  */
	tf_tag_selfclosed_end,           /* Got /> */
	tf_tag_end,                      /* Got > */
	tf_tag_atrequal,                 /* Got = */
	tf_tag_atrvalue_start,           /* Start attribute value */
	tf_tag_atrvalue_segment,         /* Part of attribute value */
	tf_tag_atrvalue_end,             /* End attribute value   */
	tf_doctype_start,                /* <!DOCTYPE             */
	tf_doctype_segment,              /* Part of DOCTYPE       */
	tf_doctype_space,                /* Space on doctype      */
	tf_doctype_item,                 /* Collected doctype item */
	tf_doctype_end,                  /* DOCTYPE line ended    */
	
	
    }               token_class;
    
    struct string * tag_name;                   /* Name of start or end tag      */
    struct string * atr_name;                   /* Name of attribute on tag      */
    struct string * atr_value_segment;          /* one part of arribute value    */
    
    struct string * named_reference;            /* Named reference including & ; */
    struct name_entity_walk  * walk_reference;  /* Parsing of named reference    */
    struct name_entity_match * match_reference; /* Result of named reference match */
    uint16          numeric_reference;          /* Numeric reference             */

    struct string * doctype_item;               /* DOCTYPE information splitten  */
    
    
    unsigned int   have_nl         : 1;         /* newline read (and not stored) */
    unsigned int   eof             : 1;       	/* got EOF (or error)            */
    unsigned int   error           : 1;     	/* character error on token      */

    enum tag_state {
	ts_init = 0,
	ts_tag_start,         /* Seen <                          */
	ts_tag_bang,          /* Seen <!                         */
	ts_tag_endmark,       /* Seen </                         */
	ts_tag_params,        /* Seen space after <  or expecting params  */
	ts_tag_ending,       /* Process for /> or  >            */
	ts_tag_bogus_comment, /* Parse to >                      */
	ts_tag_self_closing,  /* Seen <tag/                      */
	ts_tag_comment_start, /* Seen <!--                       */
	ts_tag_comment,       /* Inside of comment               */
	ts_tag_after_atrname, /* Got attribure name, parse =     */
	ts_tag_atrvalue,      /* On attribure value              */
	ts_tag_after_quoted,  /* After quoted value              */
	ts_tag_doctype_start, /* Seen <!DOCTYPE                  */
	ts_tag_doctype_bogus, /* Bogus DOCTYPE line              */
	
    }              tag_state;

    enum entity_state {
	ent_init = 0,
	ent_entity_start,     /* Seen & on tag params or body    */
	ent_decimal_ent,      /* Seen &# on tag params or body   */
	ent_hexdec_ent,       /* Seen &#x or &#X                 */

    }              entity_state;
    
    
    unsigned int   tag_flags       : NUM_tag_flag_bits;
    
    uint16         tag_quote;          /* Seen either ' 0x0027
					  or          " 0x0022 on tag  */

    struct string * tag_lookahead;     /* On some cases we need parse some
					  characters */
    
    struct string        * resubmit;   /* Resubmit as next token / 
					  sbuffer value */
    
    struct string        * sbuffer;    /* All characters which are part of
 					  token inluding quotation
				       */

    /* Process one character on time */
    struct charset_state * rchar;
    charset_t              text_charset;    
};

static void tagfilter_token_init P_((struct tagfilter_token *token,
				     charset_t text_charset));
static void tagfilter_token_init(token,text_charset)
     struct tagfilter_token *token;
     charset_t text_charset;
{

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)token, sizeof (*token));

    token->magic =  TAGFILTER_TOKEN_magic;

    token->token_class     = tf_body;
    token->tag_name        = NULL;         /* Name of start or end tag      */
    token->atr_name        = NULL;         /* Name of attribute on tag      */
    token->atr_value_segment = NULL;       /* one part of arribute value    */
    token->named_reference   = NULL;       /* Named reference including & ; */
    token->walk_reference   = NULL;        /* Parsing of named reference    */
    token->match_reference  = NULL;        /* Result of named reference match */
    token->numeric_reference = UNICODE_BAD_CHAR;    /* Numeric reference             */
    token->doctype_item      = NULL;       /* DOCTYPE information splitten  */

    
    token->have_nl         = 0;         /* newline read (and not stored) */
    token->eof             = 0;       	/* got EOF (or error)              */
    token->error           = 0;     	/* character error on token        */

    token->tag_state       = ts_init;
    token->entity_state    = ent_init;
    token->tag_flags       = 0;
    token->tag_quote       = 0;         /* Seen either ' 0x0027
					   or          " 0x0022 on tag     */

    token->tag_lookahead = NULL;         /* On some cases we need parse some
					   characters */

    token->resubmit      = NULL;   /* Resubmit as next token / sbuffer value */
    
    token->sbuffer         = NULL;
    token->rchar           = 0;
    token->text_charset    = text_charset;

}

static void tagfilter_token_dest P_((struct tagfilter_token *token));
static void tagfilter_token_dest(token)
     struct tagfilter_token *token;
{
    if (TAGFILTER_TOKEN_magic != token->magic)
        mime_panic(__FILE__,__LINE__,"tagfilter_token_dest",
                   "Bad magic number");

    token->token_class     = tf_body;
    if (token->tag_name)                  /* Name of start or end tag      */
	free_string(&(token->tag_name));
    if (token->atr_name)                  /* Name of attribute on tag      */
	free_string(&(token->atr_name));
    if (token->atr_value_segment)         /* one part of arribute value    */
	free_string(&(token->atr_value_segment));
    if (token->named_reference)           /* Named reference including & ; */
	free_string(&(token->named_reference));
    if (token->walk_reference)       /* Parsing of named reference    */
	free_name_entity_walk(&(token->walk_reference));
    if (token->match_reference)        /* Result of named reference match */
	free_name_entity_match(& (token->match_reference));
    token->numeric_reference = UNICODE_BAD_CHAR;    /* Numeric reference   */
    if (token->doctype_item)               /* DOCTYPE information splitten  */
	free_string(&(token->doctype_item));

    token->have_nl         = 0;         /* newline read (and not stored) */
    token->eof             = 0;       	/* got EOF (or error)              */
    token->error           = 0;     	/* character error on token        */

    token->tag_state       = ts_init;
    token->entity_state    = ent_init;
    token->tag_flags       = 0;
    token->tag_quote       = 0;         /* Seen either ' 0x0027
					   or          " 0x0022 on tag     */

    if (token->tag_lookahead)           /* On some cases we need parse some characters */
	free_string(& (token->tag_lookahead));

    if (token->resubmit)   /* Resubmit as next token / sbuffer value */
	free_string(& (token->resubmit));
    
    if (token->sbuffer) 
	free_string(& (token->sbuffer));
    if (token->rchar)
	free_state( & (token->rchar));
}

#if DEBUG
void debug_print_string(a,prefix,prefix2,S)
     struct debug_struct *a;
     const char * prefix;
     const char * prefix2;
     struct string *S;
{
    int L = string_len(S);
    int q = 0;
    int i;
    int next_pos = 0;
    int startline = 1;
    int prefix_len = 0;
    int prefix2_len = 0;

    if (prefix)
	prefix_len = strlen(prefix);
    
    if (prefix2)
	prefix2_len = strlen(prefix2);
    else {
	prefix2     = prefix;
	prefix2_len = prefix_len;
    }

    if (prefix_len) {
	debug_puts(a,prefix_len,prefix);
	startline = 0;
    }
    
    for (i = 0; i < L; i = next_pos) {
	uint16                unicode = 0;
	unsigned char         bytecode = 0;
	int gchar = give_character_from_string(S,i,&unicode,&bytecode);

	if (ison(gchar,GCHAR_unicode) &&
	    unicode_ch(unicode,UOP_printable)) {

	    /* Scan printable string */
	    int scan_pos;
	    int len2;
	    struct string * S1 = NULL;
	    struct string * S2 = NULL;
	    int failcount = 0;
	    
	    for (scan_pos = i; scan_pos < L; scan_pos++) {
		unicode = 0;
                bytecode = 0;
		gchar = give_character_from_string(S,scan_pos,
						   &unicode,&bytecode);

		if (isoff(gchar,GCHAR_unicode))
		    break;
		if (!unicode_ch(unicode,UOP_printable))
		    break;
	    }

	    len2 = scan_pos - i;

	retry:
	    if (len2 < 1)
		goto fail;

	    next_pos = i;
	    failcount = 0;
	    	   	    
	    S1 = clip_from_string(S,&next_pos,len2);
	    S2 = convert_string2(display_charset,S1,&failcount);

	    if (failcount < 1) {
		char * res = NULL;
		int reslen = 0;
	    
		if (!q) {
		    if (startline) {
			if (prefix2_len) debug_puts(a,prefix2_len,prefix2);
			startline = 0;
		    }
		    
		    debug_puts(a,1,"\"");
		    q = 1;		
		}

		bytestream_from_string(S2,&res,&reslen);	    
		debug_puts(a,reslen,res);
		free(res); res = NULL;
	    }
	    
	    free_string(&S2);
	    free_string(&S1);

	    if (failcount > 0 && len2 > 1) {
		len2 /= 2;
		goto retry;
	    }
	    
	    if (failcount > 0)
		goto fail;
	    
	} else {
	    char buffer[16];

	fail:
	    
	    if (q) {
		debug_puts(a,2,"\" ");
		q = 0;
	    } else if (startline) {
		if (prefix2_len) debug_puts(a,prefix2_len,prefix2);
		startline = 0;
	    }

	    if (ison(gchar,GCHAR_unicode)) {
		int len = elm_sfprintf(buffer,sizeof buffer,
				       FRM("%04x "),unicode);
		debug_puts(a,len,buffer);

		startline = 0x000A /* LF  '\n' */ == unicode;
		
	    } else if (ison(gchar,GCHAR_bytecode)) {
		int len = elm_sfprintf(buffer,sizeof buffer,
				       FRM("%02x "),bytecode);
		debug_puts(a,len,buffer);

		startline = '\n' == bytecode;
	    } else {
		debug_puts(a,1,"?");
	    }
		    
	    if (startline) {
		debug_puts(a,1,"\n");
	    }
	    next_pos = i+1;
	}	
    }

    if (q) {
	debug_puts(a,2,"\"\n");
	q = 0;
    } else if (!startline) {
	debug_puts(a,1,"\n");
    }    
}

#endif


/* Mail lines should be under 1000 characters */

#define MAX_TAG_TOKEN 1002
#define MAX_ENRICHED_TOKEN 60
#define MAX_VALUE_SEGMENT  120
#define MAX_DOCTYPE_VALUE  1002

/* Returns -1 if EOF (or error seen) before any characters */
static int get_new_tagfilter_token P_((struct tagfilter_token     * token,
				       struct tagfilter_selection * tagfilter,
				       struct in_state            * state_in));
static int get_new_tagfilter_token(token,tagfilter,state_in)
     struct tagfilter_token     * token;
     struct tagfilter_selection * tagfilter;
     struct in_state            * state_in;
{
    int ret     = 0;
    int gotbyte = 0;
    int len     = 0;           /* token->sbuffer */
    int badstate = 0;

    if (TAGFILTER_TOKEN_magic != token->magic)
        mime_panic(__FILE__,__LINE__,"get_new_tagfilter_token",
                   "Bad magic number");
    
    if (TAGFILTER_SELECTION_magic != tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"get_new_tagfilter_token",
		   "Bad magic number (tagfilter_selection)");

     DPRINT(Debug,22,(&Debug,				
		      "get_new_tagfilter_token: tagflt_mode=%d",
		      tagfilter->tagflt_mode));
     if (tagfilter->tagflt_mode) {
	 enum TAGFLT_MODE_bits d1;

	 for (d1 = 0; d1 < TAGFLT_MODE_bit_count; d1++) {
	     if (ison(tagfilter->tagflt_mode, 1 << d1)) {
		 switch (d1) {
		 case TAGFLT_MODE_bit_tag_attributes:
		     DPRINT(Debug,22,(&Debug," TAGFLT_MODE_tag_attributes"));
		     break;
		 case TAGFLT_MODE_bit_double_smaller:
		     DPRINT(Debug,22,(&Debug," TAGFLT_MODE_double_smaller"));
		     break;
		 case TAGFLT_MODE_bit_many_linebreak:
		     DPRINT(Debug,22,(&Debug," TAGFLT_MODE_many_linebreak"));
		     break;
		 case TAGFLT_MODE_bit_enriched_tag:
		     DPRINT(Debug,22,(&Debug," TAGFLT_MODE_enriched_tag"));
		     break;
		 case TAGFLT_MODE_bit_lowerascii:
		     DPRINT(Debug,22,(&Debug," TAGFLT_MODE_lowerascii"));
		     break;
		 case TAGFLT_MODE_bit_convert_cr:
		     DPRINT(Debug,22,(&Debug," TAGFLT_MODE_convert_cr"));
		     break;
		 case TAGFLT_MODE_bit_count:
		     break;
		 }
	     }
	 }
     }
     if (tagfilter->doctype_name) {
	 DPRINT(Debug,22,(&Debug," doctype_name=%Q",tagfilter->doctype_name));
     }
     DPRINT(Debug,22,(&Debug,"\n"));
    
    if (!token->rchar)
	token->rchar = new_state(token->text_charset);

    if (token->sbuffer)
	free_string(& (token->sbuffer));

    if (token->resubmit) {  /* Resubmit as next token / sbuffer value */
	token->sbuffer = token->resubmit;
	token->resubmit = NULL;
    } else
        token->sbuffer = new_string(token->text_charset);

    token->have_nl         = 0;         /* newline read (and not stored)   */
    token->eof             = 0;       	/* got EOF (or error)              */
    token->error           = 0;     	/* character error on token        */
    token->token_class     = tf_body;
    clearit(token->tag_flags,TFLAG_unhandled_class);
    
    if (token->tag_name)                /* Name of start or end tag        */
	free_string(&(token->tag_name));

    if (ts_tag_params != token->tag_state) {	
	if (token->atr_name)                /* Name of attribute on tag    */
	    free_string(&(token->atr_name));
    }
    if (token->atr_value_segment)       /* one part of arribute value      */
	free_string(&(token->atr_value_segment));
    
    if (token->named_reference)         /* Named reference including & ;   */
	free_string(&(token->named_reference));
    if (token->walk_reference)       /* Parsing of named reference    */
	free_name_entity_walk(&(token->walk_reference));
    if (token->match_reference)        /* Result of named reference match */
	free_name_entity_match(& (token->match_reference));    
    token->numeric_reference = UNICODE_BAD_CHAR;   /* Numeric reference    */
    if (token->doctype_item)               /* DOCTYPE information splitten  */
	free_string(&(token->doctype_item));

    
    do {
	/* NOTE:  It is assumed that CRLF are same than on ascii */
	int ch = EOF;

	if (state_ready(token->rchar)) {

	    /* Have one chararacter collected */
	     
	    uint16 u = give_unicode_from_state(token->rchar);
	    
	    len = string_len(token->sbuffer);

	    DPRINT(Debug,22,(&Debug,				
			     "get_new_tagfilter_token: u=%04u",
			     u));

	    if (state_printable(token->rchar)) {
		DPRINT(Debug,22,(&Debug," '%C'",
				 token->rchar));
	    }	    
	    DPRINT(Debug,22,(&Debug," tag_state=%d",
			     token->tag_state));
	    switch (token->tag_state) {
	    case ts_init:        DPRINT(Debug,22,(&Debug," ts_init"));        break;
	    case ts_tag_start:   DPRINT(Debug,22,(&Debug," ts_tag_start"));   break;
	    case ts_tag_bang:    DPRINT(Debug,22,(&Debug," ts_tag_bang"));    break;
	    case ts_tag_endmark: DPRINT(Debug,22,(&Debug," ts_tag_endmark")); break;
	    case ts_tag_params:  DPRINT(Debug,22,(&Debug," ts_tag_params"));  break;
	    case ts_tag_ending:  DPRINT(Debug,22,(&Debug," ts_tag_ending"));  break;
	    case ts_tag_bogus_comment: DPRINT(Debug,22,(&Debug," ts_tag_bogus_comment"));
		break;
	    case ts_tag_self_closing:  DPRINT(Debug,22,(&Debug," ts_tag_self_closing"));
		break;
	    case ts_tag_comment_start: DPRINT(Debug,22,(&Debug," ts_tag_comment_start"));
		break;
	    case ts_tag_comment: DPRINT(Debug,22,(&Debug," ts_tag_comment")); break;
	    case ts_tag_after_atrname: DPRINT(Debug,22,(&Debug," ts_tag_after_atrname"));
		break;
	    case ts_tag_atrvalue:      DPRINT(Debug,22,(&Debug," ts_tag_atrvalue"));
		break;
	    case ts_tag_after_quoted:  DPRINT(Debug,22,(&Debug," ts_tag_after_quoted"));
		break;
	    case ts_tag_doctype_start: DPRINT(Debug,22,(&Debug," ts_tag_doctype_start"));
		break;
	    case ts_tag_doctype_bogus: DPRINT(Debug,22,(&Debug," ts_tag_doctype_bogus"));
		break;
	    }
	    DPRINT(Debug,22,(&Debug," entity_state=%d",token->entity_state));
	    switch (token->entity_state) {
	    case ent_init:  DPRINT(Debug,22,(&Debug," ent_init")); break;
	    case ent_entity_start: DPRINT(Debug,22,(&Debug," ent_entity_start")); break;
	    case ent_decimal_ent:  DPRINT(Debug,22,(&Debug," ent_decimal_ent"));  break;
	    case ent_hexdec_ent:   DPRINT(Debug,22,(&Debug," ent_hexdec_ent"));   break;
	    }
	    DPRINT(Debug,22,(&Debug," tag_flags=%d",token->tag_flags));
	    if (ison(token->tag_flags, TFLAG_seen_nl)) {
		DPRINT(Debug,22,(&Debug," TFLAG_seen_nl"));
	    }
	    if (ison(token->tag_flags, TFLAG_self_closing)) {
		DPRINT(Debug,22,(&Debug," TFLAG_self_closing"));
	    }
	    if (ison(token->tag_flags, TFLAG_seen_equals)) {
		DPRINT(Debug,22,(&Debug," TFLAG_seen_equals"));
	    }
	    if (ison(token->tag_flags, TFLAG_num_overflow)) {
		DPRINT(Debug,22,(&Debug," TFLAG_num_overflow"));
	    }
	    if (ison(token->tag_flags, TFLAG_unhandled_class)) {
		DPRINT(Debug,22,(&Debug," TFLAG_unhandled_class"));
	    }
	    DPRINT(Debug,22,(&Debug," tag_quote=%d\n",
			     token->tag_quote));
	    
	    if (0x000D /* CR  '\r' */ == u) {
		int peekch = state_getc(state_in);
		
		if ('\n' == peekch) {   /* Got newline */
		    ch = peekch;
		    reset_state(token->rchar,0);
		    goto process_char;
		    
		} else if (EOF != peekch)
		    state_ungetc(peekch,state_in);
		
		if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_convert_cr))
		    u = 0x000A /* LF  '\n' */;		    
	    }  
	    
#define CHECK_SBUFFER_LEN(error_class) do {				\
		len = string_len(token->sbuffer);			\
		if (len >= MAX_TAG_TOKEN) {				\
		    DPRINT(Debug,20,(&Debug,				\
				     "get_new_tagfilter_token: Read %d characters\n", \
				     len));				\
		    ret = len;						\
		    token->token_class = error_class;			\
		    goto out;						\
		}							\
	    } while(0)
	    
#define ADD_SBUFFER_CHAR(error_class) do {				\
		add_state_to_string(token->sbuffer,token->rchar);  /* Returns no status  */ \
		reset_state(token->rchar,0);                       /* Get next character */ \
		CHECK_SBUFFER_LEN(error_class);				\
	    } while(0)

#define ADD_SBUFFER_CHAR_NOCHECK do {					\
		add_state_to_string(token->sbuffer,token->rchar);  /* Returns no status  */ \
		reset_state(token->rchar,0);                       /* Get next character */ \
		len = string_len(token->sbuffer);			\
	    } while (0)

#define EMIT_TOKEN_CLASS(class) do {		\
		ret = len;			\
		token->token_class = class;     \
		goto out;			\
	    } while (0)
	    
	    if (ison(token->tag_flags,TFLAG_seen_nl)) {

		/* Start newline span  */
		
		if (0x000A /* LF  '\n' */ == u) {

		    ADD_SBUFFER_CHAR(tf_span_nl);
		    
		} else {       

		    clearit(token->tag_flags, TFLAG_seen_nl);
		    
		    if (len > 0) {
			/* got newline span */

			EMIT_TOKEN_CLASS(tf_span_nl);
		    }		    
		}


	    } else if (ts_tag_doctype_bogus == token->tag_state) {
		/* Bogus DOCTYPE line */

		if (token->tag_lookahead)
		    free_string(& (token->tag_lookahead));
		
		if (0x003E /* > */                         == u) {
		    
		    token->tag_state    = ts_init;    /* Tag ended */
		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		    EMIT_TOKEN_CLASS(tf_doctype_end);
		    
		} else if ( 0x000A /* LF  '\n' (new line) */       == u) {
		    /* Consume newline */
		    
		    token->have_nl = 1;
		    reset_state(token->rchar,0);
		    
		    EMIT_TOKEN_CLASS(tf_doctype_segment);
		    
		} else if (0x0000 /* NUL */ == u) {
		    uint16 bad_char = UNICODE_BAD_CHAR;
		    
		    DPRINT(Debug,20,(&Debug, 
				     "get_new_tagfilter_token: Found NUL character when parsing doctype line\n"));
		    
		    reset_state(token->rchar,0);  /* Get next character */
		    token->error       = 1;
		    add_unicode_to_string(token->sbuffer,1,&bad_char);
		    
		    CHECK_SBUFFER_LEN(tf_doctype_segment);
		} else {
		    ADD_SBUFFER_CHAR(tf_doctype_segment);	
		}
		
	    } else if (ts_tag_doctype_start == token->tag_state) {

		int lookahead_len = 0;
		
		/* Seen <!DOCTYPE                     */

		if (token->tag_lookahead) 		    
		    lookahead_len = string_len(token->tag_lookahead);
		    
		/* Not correct .... */

		if (0x003E /* > */                         == u) {

		    if (token->tag_quote) {
			
			/* If > inside of quoted value, treate as error */

			if (token->tag_lookahead) 
			    free_string(& (token->tag_lookahead));
			
			token->tag_state = ts_tag_doctype_bogus;
			EMIT_TOKEN_CLASS(tf_doctype_error);
		    }

		    if (token->tag_lookahead) {
			token->doctype_item  = token->tag_lookahead;
			token->tag_lookahead = NULL;
			EMIT_TOKEN_CLASS(tf_doctype_item);
			
		    }

		    token->tag_state    = ts_init;    /* Tag ended */
		    ADD_SBUFFER_CHAR_NOCHECK;

		    EMIT_TOKEN_CLASS(tf_doctype_end);
									
		} else if (0x000A /* LF  '\n' (new line) */       == u) {


		    if (token->tag_quote && token->tag_lookahead) {
			add_state_to_string(token->tag_lookahead,token->rchar);  /* Returns no status  */
		    } 

		    /* Consume newline */

		    token->have_nl = 1;
		    reset_state(token->rchar,0);

		    if (token->tag_quote && token->tag_lookahead) {

			lookahead_len = string_len(token->tag_lookahead);
			if (lookahead_len > MAX_DOCTYPE_VALUE) {

			    token->tag_state = ts_tag_doctype_bogus;
			    /* Forget it */
			    
			    free_string(& (token->tag_lookahead));
			    EMIT_TOKEN_CLASS(tf_doctype_error);
			}

			EMIT_TOKEN_CLASS(tf_doctype_segment);
			
		    } else if (token->tag_lookahead) {
			
			token->doctype_item  = token->tag_lookahead;
			token->tag_lookahead = NULL;
			EMIT_TOKEN_CLASS(tf_doctype_item);
			
		    } else {			
			EMIT_TOKEN_CLASS(tf_doctype_space);
		    }

		} else if (0x0000 /* NUL */ == u) {
		    uint16 bad_char = UNICODE_BAD_CHAR;

		    if (!token->tag_lookahead) {
			len = string_len(token->sbuffer);
			if (len > 0) {
			    DPRINT(Debug,20,(&Debug, 
					     "get_new_tagfilter_token: Found NUL character when parsing doctype line - flushing buffer before it\n"));
			    EMIT_TOKEN_CLASS(tf_doctype_space);
			}
		    }
		    
		    DPRINT(Debug,20,(&Debug, 
				     "get_new_tagfilter_token: Found NUL character when parsing doctype line\n"));
		    
		    reset_state(token->rchar,0);  /* Get next character */
		    token->error       = 1;
		    
		    add_unicode_to_string(token->sbuffer,1,&bad_char);

		    if (!token->tag_lookahead)
			token->tag_lookahead =  new_string(token->text_charset);
		   		    
		    add_unicode_to_string(token->tag_lookahead,1,&bad_char);
		    lookahead_len = string_len(token->tag_lookahead);
		    if (lookahead_len > MAX_DOCTYPE_VALUE) {
			
			token->tag_state = ts_tag_doctype_bogus;
			/* Forget it */
			
			free_string(& (token->tag_lookahead));
			EMIT_TOKEN_CLASS(tf_doctype_error);
		    }
		    		    
		} else 	if (token->tag_quote && token->tag_lookahead) {

		    /* Does not detect if there is no space after quote */
		    
		    if (token->tag_quote == u) {
			token->tag_quote = 0;
			add_state_to_string(token->tag_lookahead,token->rchar);  /* Returns no status  */
			ADD_SBUFFER_CHAR_NOCHECK;

			token->doctype_item  = token->tag_lookahead;
			token->tag_lookahead = NULL;
			EMIT_TOKEN_CLASS(tf_doctype_item);			
		    } else {
			
			add_state_to_string(token->tag_lookahead,token->rchar);  /* Returns no status  */
			lookahead_len = string_len(token->tag_lookahead);

			ADD_SBUFFER_CHAR_NOCHECK;
			
			if (lookahead_len > MAX_DOCTYPE_VALUE) {
			
			    token->tag_state = ts_tag_doctype_bogus;
			    /* Forget it */
			    
			    free_string(& (token->tag_lookahead));
			    EMIT_TOKEN_CLASS(tf_doctype_error);
			}
		    
			CHECK_SBUFFER_LEN(tf_doctype_segment);
		    }
			
		} else {
		    if (0x0009 /* HT  '\t' (horizontal tab) */ == u ||
			0x000C /* FF  '\f' (form feed) */      == u ||
			0x0020 /* SPACE */                     == u) {

			if (token->tag_lookahead) {
			    token->doctype_item  = token->tag_lookahead;
			    token->tag_lookahead = NULL;
			    EMIT_TOKEN_CLASS(tf_doctype_item);
			    
			}

			ADD_SBUFFER_CHAR(tf_doctype_space);

		    } else if (0x0022 /* " */                         == u ||
			       0x0027 /* ' */                         == u) {

			/* Not correct parsing because this does not check 
			   was that part what was required to be quoted 
			*/

			if (token->tag_lookahead) {
			    
			    token->tag_state = ts_tag_doctype_bogus;
			    /* Forget it */
			    
			    free_string(& (token->tag_lookahead));
			    EMIT_TOKEN_CLASS(tf_doctype_error);

			} else {

			    len = string_len(token->sbuffer);
			    if (len > 0) {
				DPRINT(Debug,20,(&Debug, 
						 "get_new_tagfilter_token: Flushing buffer before quoted item when parsing doctype line\n"));
				EMIT_TOKEN_CLASS(tf_doctype_space);
			    }
			    
			    token->tag_lookahead =  new_string(token->text_charset);
			    
			    add_state_to_string(token->tag_lookahead,token->rchar);  /* Returns no status  */
			    token->tag_quote = u;
			}

			ADD_SBUFFER_CHAR_NOCHECK;
			    
		    } else {
			
			/* Enable buffer */
			
			if (!token->tag_lookahead) {
			    len = string_len(token->sbuffer);
			    if (len > 0) {
				EMIT_TOKEN_CLASS(tf_doctype_space);
			    }
			    
			    token->tag_lookahead =  new_string(token->text_charset);
			}

			/* XXXXX -- correct missing? */
			
			add_state_to_string(token->tag_lookahead,token->rchar);  /* Returns no status  */
			lookahead_len = string_len(token->tag_lookahead);

			ADD_SBUFFER_CHAR_NOCHECK;
			
			if (lookahead_len > MAX_DOCTYPE_VALUE) {
			    
			    token->tag_state = ts_tag_doctype_bogus;
			    /* Forget it */
			    
			    free_string(& (token->tag_lookahead));
			    EMIT_TOKEN_CLASS(tf_doctype_error);
			}
			
			CHECK_SBUFFER_LEN(tf_doctype_segment);									
		    }					       
		}
				    
	    } else if (ts_tag_after_quoted == token->tag_state) {
		/* After quoted value              */

		if ( 0x000A /* LF  '\n' (new line) */       == u) {
		    /* Consume newline */
		    
		    token->have_nl = 1;
		    reset_state(token->rchar,0);

		    /* Parse new attribute instead */
		    token->tag_state    =  ts_tag_params;

		    EMIT_TOKEN_CLASS(tf_tag_space);
		} else if (0x0009 /* HT  '\t' (horizontal tab) */ == u ||
			   0x000C /* FF  '\f' (form feed) */      == u ||
			   0x0020 /* SPACE */                     == u) {

		    /* Parse new attribute instead */
		    token->tag_state    =  ts_tag_params;
		    
		} else if (0x003E /* > */                         == u ||
			   0x002F /* / */                         == u) {

		    token->tag_state =  ts_tag_ending;  /* Process for /> or  >  */

		} else {
		    token->tag_state =  ts_tag_params;
		    
		    EMIT_TOKEN_CLASS(tf_tag_error);
		}

	    } else if (ent_hexdec_ent == token->entity_state) {

		uint16 v = 0;
		
		if (0x0030 /* 0 */ <= u && u <= 0x0039 /* 9 */)
		    v = u - 0x0030 /* 0 */;
		else if (0x0041 /* A */ <= u && u <= 0x0046 /* F */)
		    v = u - 0x0041 /* A */ + 10;
		else if (0x0061 /* a */ <= u && u <= 0x0066 /* f */)
		    v = u - 0x0061 /* a */ + 10;
		else if (0x003B /* ; */ == u) {

		    if (isoff(token->tag_flags,TFLAG_num_overflow)) {
			
			token->entity_state    = ent_init;
			
			ADD_SBUFFER_CHAR_NOCHECK;
			
			EMIT_TOKEN_CLASS(tf_numeric_entity);
			
		    } else {
			token->numeric_reference = UNICODE_BAD_CHAR;
			goto end_entity;
		    }		    
		} else {
		    token->numeric_reference = UNICODE_BAD_CHAR;
		    goto end_entity;
		}
		
		/* Hexadecimal numeric character reference */

		if (isoff(token->tag_flags,TFLAG_num_overflow)) {
		    
		    if (token->numeric_reference > 0xFFFF / 16) {
			setit(token->tag_flags,TFLAG_num_overflow);
			token->numeric_reference = UNICODE_BAD_CHAR;
		    } else {
			token->numeric_reference *= 16;
			token->numeric_reference += v;
		    }
		}
				
		ADD_SBUFFER_CHAR(tf_entity_error);
		
	    } else if (ent_decimal_ent == token->entity_state) {

		if (2 == len && ( 0x0078 /* x */ == u ||
				0x0058 /* X */ == u)) {
		    
		    /* Hexadecimal numeric character reference */

		    ADD_SBUFFER_CHAR_NOCHECK;

		    token->entity_state = ent_hexdec_ent;
		    
		} else if (0x0030 /* 0 */ <= u && u <= 0x0039 /* 9 */) {
		    
		    /* Decimal numeric character reference */

		    if (isoff(token->tag_flags,TFLAG_num_overflow)) {
			
			if (token->numeric_reference > 0xFFFF / 10) {
			    setit(token->tag_flags,TFLAG_num_overflow);
			    token->numeric_reference = UNICODE_BAD_CHAR;
			} else {
			    token->numeric_reference *= 10;
			    token->numeric_reference += u - 0x0030 /* 0 */;
			}
		    }
		    
		    ADD_SBUFFER_CHAR(tf_entity_error);
		} else if (0x003B /* ; */ == u) {

		    if (isoff(token->tag_flags,TFLAG_num_overflow)) {
			
			token->entity_state    = ent_init;
			
			ADD_SBUFFER_CHAR_NOCHECK;
			
			EMIT_TOKEN_CLASS(tf_numeric_entity);
			
		    } else {
			token->numeric_reference = UNICODE_BAD_CHAR;
			goto end_entity;
		    }
		} else {
		    token->numeric_reference = UNICODE_BAD_CHAR;
		    goto end_entity;
		}
		
	    } else if (ent_entity_start == token->entity_state) {

		    /* & on body or on attribute value */
	
		if (1 == len && 0x0023 /* # */ == u) {

		    /* Decimal numeric character reference */

		    ADD_SBUFFER_CHAR_NOCHECK;

		    token->entity_state = ent_decimal_ent;
		    token->numeric_reference = 0;    /* Numeric reference             */
		    clearit(token->tag_flags,TFLAG_num_overflow);
		    
		    if (token->named_reference)      /* Named reference including & ; */
			free_string(&(token->named_reference));

		    if (token->walk_reference)       /* Parsing of named reference    */
			free_name_entity_walk(&(token->walk_reference));
		    
		} else if (tagfilter_entity_character(u,NULL)) {

		    /* Named character references */

		    /* Returns no status */
		    add_state_to_string(token->named_reference,token->rchar);

		    ADD_SBUFFER_CHAR(tf_entity_error);		    
		    
		    if (token->walk_reference) {
			/*  free'es struct name_entity_walk, if no match */
			advance_entity_walk(& (token->walk_reference), u);
		    }
		    
		    /* XXXXX --- not correct */
		    
		} else {
		    
		end_entity:
		    if (0x003B /* ; */ == u) {
			
			if (token->named_reference) {          /* Named reference including & ; */
			    /* Returns no status */
			    add_state_to_string(token->named_reference,token->rchar);
			}

			if (token->walk_reference) {
			    /* increment refcount */
			    token->match_reference =
				tagfilter_match_reference(token->walk_reference,
							  tagfilter_match_semicolon);
			    
			    free_name_entity_walk(&(token->walk_reference));
			}
			
			/* End of entity */
			/* XXXX this accepts also no matching character references 
                                even when there is shorter match
			 */
			
			token->entity_state    = ent_init;

			ADD_SBUFFER_CHAR_NOCHECK;

			EMIT_TOKEN_CLASS(tf_entity);
		    } else {
						
			if (0x000A /* LF  '\n' */ == u) {
			    /* Consume newline */
			    
			    token->have_nl = 1;
			    reset_state(token->rchar,0);
			    
			}

			if (token->named_reference) {
			    /* &xxx= on tags are not interpreted as character reference */
			    
			    if (ts_tag_atrvalue == token->tag_state &&
				!token->atr_value_segment &&
				0x003D /* = */ == u) {

				if (token->walk_reference) /* Parsing of named reference    */
				    free_name_entity_walk(&(token->walk_reference));
				
				token->atr_value_segment = token->named_reference;
				token->named_reference   = NULL;
				token->entity_state      = ent_init;
				
				EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
			    }

			    free_string(&(token->named_reference));
			}
		    
		        if (token->walk_reference) {
			    /* is this acceptable without ; 

			       XXXX this is parse error, but it is ignored 
			    */

			    /* increment refcount */
			    token->match_reference =
				tagfilter_match_reference(token->walk_reference,
							  tagfilter_match_prefix);

			    
			    free_name_entity_walk(&(token->walk_reference));
			    
			    if (token->match_reference) {
				token->entity_state    = ent_init;
				EMIT_TOKEN_CLASS(tf_entity);
			    }			    
		        }

			/* XXXX Only handled on body */
			if (token->named_reference && ts_init == token->tag_state) {
			    int reflen,i;

			    reflen = string_len(token->named_reference);
			    /* Perhaps there was shorter match */
			    token->walk_reference =  tagfilter_start_reference(tagfilter->have_entities);
			    
			    for (i = 0; i < reflen; i++) {
				uint16 x = give_unicode_from_string(token->named_reference,i);

				/* Skip */
				if (0x0026 /* & */ == x && 0 == i)
				    continue;

				/* increment refcount */
				token->match_reference =
				    tagfilter_match_reference(token->walk_reference,
							      tagfilter_match_prefix);

				if (token->match_reference)  {
				    /* XXXX this founds shortest prefix only */

				    /* XXXX this is parse error, but it is ignored */
				    
				    struct string * temp = token->named_reference;
				    int POS = 0;

				    token->named_reference = clip_from_string(temp,&POS,i);
				    token->resubmit        = clip_from_string(temp,&POS,reflen);

				    free_string(&temp);
				    free_name_entity_walk(&(token->walk_reference));

				    token->entity_state    = ent_init;
				    EMIT_TOKEN_CLASS(tf_entity);
				}

				/*  free'es struct name_entity_walk, if no match */
				advance_entity_walk(& (token->walk_reference), x);

				if (! token->walk_reference)
				    break;    /* Scan failed */
			    }

			    if (token->walk_reference)  /* Scan was not endded */
				free_name_entity_walk(&(token->walk_reference));
			}
			   		   
			/* Parse error on entity -- do not include character */

			token->entity_state    = ent_init;
			
			EMIT_TOKEN_CLASS(tf_entity_error);
		    }	       		
		}

				
	    } else if (ts_tag_atrvalue == token->tag_state) {

		int seglen = 0;
		
		if (!token->atr_value_segment)       /* one part of arribute value    */
		    token->atr_value_segment = new_string(token->text_charset);

		seglen = string_len(token->atr_value_segment);
		
		if (0x000A /* LF  '\n' (new line) */       == u) {

		    if (token->tag_quote) {

			add_state_to_string(token->atr_value_segment,token->rchar);  /* Returns no status  */
			seglen = string_len(token->atr_value_segment);

			/* Consume newline */

			token->have_nl = 1;
			reset_state(token->rchar,0);
			
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
			
		    } else {
			if (seglen > 0) {
			    EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
			}

			free_string(& token->atr_value_segment);
		    
			/* Consume newline */
			
			if (token->atr_name)           /* Reset name of attribute on tag      */
			    free_string(&(token->atr_name));
		    
			token->have_nl = 1;
			reset_state(token->rchar,0);
			
			/* Parse new attribute instead */
			token->tag_state    =  ts_tag_params;
			
			/* Value ends */
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_end);
		    }
		    
		} else if (! token->tag_quote &&
			   (0x0009 /* HT  '\t' (horizontal tab) */ == u ||
			    0x000C /* FF  '\f' (form feed) */      == u ||
			    0x0020 /* SPACE */                     == u)) {

		    if (seglen > 0) {
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
		    }

		    free_string(& token->atr_value_segment);

		    if (token->atr_name)           /* Reset name of attribute on tag      */
			free_string(&(token->atr_name));
		    		    
		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		     /* Parse new attribute instead */
		    token->tag_state    =  ts_tag_params;

		    /* Value ends */
		    EMIT_TOKEN_CLASS(tf_tag_atrvalue_end);

		} else if (token->tag_quote &&
			   token->tag_quote == u) {

		   if (seglen > 0) {
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
		    }

		    free_string(& token->atr_value_segment);

		    if (token->atr_name)           /* Reset name of attribute on tag      */
			free_string(&(token->atr_name));
		    		    
		    ADD_SBUFFER_CHAR_NOCHECK; 
		    token->tag_quote = 0;
		    token->tag_state = ts_tag_after_quoted;

		    /* Value ends */
		    EMIT_TOKEN_CLASS(tf_tag_atrvalue_end);

		} else if (0x0026 /* &  */ == u  &&
			   tagfilter->have_entities) {

		    if (seglen > 0) {
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
		    }

		    free_string(& token->atr_value_segment);

		    
		    token->entity_state    = ent_entity_start;

		    /* Start & .... */
		    token->named_reference = new_string(token->text_charset);
		    /* Returns no status */
		    add_state_to_string(token->named_reference,token->rchar);
		    token->walk_reference =  tagfilter_start_reference(tagfilter->have_entities);

		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		} else if (! token->tag_quote &&  
			   0x003E /* > */ == u) {

		    if (seglen > 0) {
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
		    }

		    free_string(& token->atr_value_segment);

		    if (token->atr_name)           /* Reset name of attribute on tag      */
			free_string(&(token->atr_name));


		    /* Will emit tag end */
		    token->tag_state    = ts_tag_ending;
		     
		     /* Value ends */
		    EMIT_TOKEN_CLASS(tf_tag_atrvalue_end);


		} else if (0x0000 /* NUL */ == u) {
		    uint16 bad_char = UNICODE_BAD_CHAR;
		    
		    DPRINT(Debug,20,(&Debug, 
				     "get_new_tagfilter_token: Found NUL character when parsing attribute value\n"));
		    
		    reset_state(token->rchar,0);  /* Get next character */
		    
		    token->error       = 1;
		    
		    add_unicode_to_string(token->atr_value_segment,1,&bad_char);
		    seglen = string_len(token->atr_value_segment);
		    
		    add_unicode_to_string(token->sbuffer,1,&bad_char);
		    
		    if (seglen >= MAX_VALUE_SEGMENT) {
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
		    }
		    CHECK_SBUFFER_LEN(tf_tag_atrvalue_error);			

		} else if (! token->tag_quote &&
			   (0x0022 /* " */     == u ||
			    0x0027 /* ' */     == u ||
			    0x003C /* < */     == u ||
			    0x003D /* = */     == u ||
			    0x0060 /* ` */     == u)) {

		    if (seglen > 0) {
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
		    }
		    
		    add_state_to_string(token->atr_value_segment,token->rchar);  /* Returns no status  */
		    ADD_SBUFFER_CHAR_NOCHECK;
		    EMIT_TOKEN_CLASS(tf_tag_atrvalue_error);
		    
		} else {
		    add_state_to_string(token->atr_value_segment,token->rchar);  /* Returns no status  */
		    seglen = string_len(token->atr_value_segment);
		    ADD_SBUFFER_CHAR_NOCHECK;

		    if (seglen >= MAX_VALUE_SEGMENT) {
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_segment);
		    }
		    CHECK_SBUFFER_LEN(tf_tag_atrvalue_error);
		}
		
	    } else if (ts_tag_after_atrname == token->tag_state) {
		/* Got attribute name, parse =     */

		if (0x000A /* LF  '\n' (new line) */       == u) {
		    /* Consume newline */
		    
		    token->have_nl = 1;
		    reset_state(token->rchar,0);

		    EMIT_TOKEN_CLASS(tf_tag_space);

		} else if (0x0009 /* HT  '\t' (horizontal tab) */ == u ||
			   0x000C /* FF  '\f' (form feed) */      == u ||
			   0x0020 /* SPACE */                     == u) {
		    
		    ADD_SBUFFER_CHAR(tf_tag_space);

		} else if (isoff(token->tag_flags,TFLAG_seen_equals) &&
			   0x002F /* / */                         == u) {

		    /* / do not end when parsing unquoted atribute value? */

		    token->tag_state =  ts_tag_ending;  /* Process for /> or  >  */
		    
		    if (len > 0) {
			EMIT_TOKEN_CLASS(tf_tag_space);
		    }

		} else if (0x003E /* > */                         == u) {

		    /* If = seen this is error */

		    token->tag_state =  ts_tag_ending;  /* Process for /> or  >  */
		    
		    if (ison(token->tag_flags,TFLAG_seen_equals)) {
			EMIT_TOKEN_CLASS(tf_tag_error);
		    } else if (len > 0) {
			EMIT_TOKEN_CLASS(tf_tag_space);
		    }
		} else if (isoff(token->tag_flags,TFLAG_seen_equals) &&
			   0x003D /* =  */                        == u) {

		    /* Emit first spaces */

		    if (len > 0) {
			EMIT_TOKEN_CLASS(tf_tag_space);
		    }

		    ADD_SBUFFER_CHAR_NOCHECK;
		    setit(token->tag_flags,TFLAG_seen_equals);
		    EMIT_TOKEN_CLASS(tf_tag_atrequal);

		} else if (ison(token->tag_flags,TFLAG_seen_equals) &&
			   (0x0022 /* " */                         == u ||
			    0x0027 /* ' */                         == u)) {			
		    if (len > 0) {
			EMIT_TOKEN_CLASS(tf_tag_space);
		    }

		    clearit(token->tag_flags,TFLAG_seen_equals); /* Forget = */
		   
		    ADD_SBUFFER_CHAR_NOCHECK;
		    token->tag_quote = u;
		    token->tag_state = ts_tag_atrvalue;
		    EMIT_TOKEN_CLASS(tf_tag_atrvalue_start);
		    
		} else {
		    
		    if (ison(token->tag_flags,TFLAG_seen_equals)) {
			if (len > 0) {
			    EMIT_TOKEN_CLASS(tf_tag_space);
			}

			clearit(token->tag_flags,TFLAG_seen_equals); /* Forget = */
			
			token->tag_quote = 0;
			token->tag_state = ts_tag_atrvalue;
			EMIT_TOKEN_CLASS(tf_tag_atrvalue_start);
		    } else {
			/* Parse new attribute instead */

			if (token->atr_name)           /* Reset name of attribute on tag      */
			    free_string(&(token->atr_name));
			
			clearit(token->tag_flags,TFLAG_seen_equals); /* Forget = */
			
			token->tag_state = ts_tag_params;
			if (len > 0) {
			    EMIT_TOKEN_CLASS(tf_tag_space);
			}
		    }		    
		}
		    		
	    } else if (ts_tag_ending == token->tag_state) {
		/* Process for /> or  >  */

		if (0x002F /* / */                         == u) {

		    if (isoff(token->tag_flags,TFLAG_self_closing)) {
			setit(token->tag_flags,TFLAG_self_closing);

			ADD_SBUFFER_CHAR_NOCHECK;			
		    } else {
			ADD_SBUFFER_CHAR_NOCHECK;

			EMIT_TOKEN_CLASS(tf_tag_param_error);
		    }
		} else if (0x003E /* > */                         == u) {
		    token->tag_state    = ts_init;    /* Tag ended */

		    if (token->tag_name)             /* Clear name of start or end tag      */
			free_string(&(token->tag_name));
		    if (token->atr_name)             /* Clear name of attribute on tag      */
			free_string(&(token->atr_name));

		    
		    if (ison(token->tag_flags,TFLAG_self_closing)) {
			ADD_SBUFFER_CHAR_NOCHECK;

			clearit(token->tag_flags,TFLAG_self_closing); /* Passed already */
			EMIT_TOKEN_CLASS(tf_tag_selfclosed_end);
		    } else {
			ADD_SBUFFER_CHAR_NOCHECK;

			EMIT_TOKEN_CLASS(tf_tag_end);
		    }
		    
		} else {
		    /* Possible / without > */
		    
		    token->tag_state    =  ts_tag_params;

		    EMIT_TOKEN_CLASS(tf_tag_param_error);
		}
				
	    } else if (ts_tag_params == token->tag_state) {
		/* Seen space after <  or expecting params  */
		
		if (token->atr_name) {     /* Name of attribute on tag      */

		    if (0x000A /* LF  '\n' (new line) */       == u) {
			/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);
			
			token->tag_state    = ts_tag_after_atrname; /* Got attribute name, parse =  */
			EMIT_TOKEN_CLASS(tf_tag_atrname);

		    } else if (0x002F /* / */                         == u ||
			       0x003E /* > */                         == u) {


			token->tag_state =  ts_tag_ending;  /* Process for /> or  >  */

			if (len > 0) {
			    EMIT_TOKEN_CLASS(tf_tag_atrname);
			}
						
		    } else if (0x0009 /* HT  '\t' (horizontal tab) */ == u ||
			       0x000C /* FF  '\f' (form feed) */      == u ||
			       0x0020 /* SPACE */                     == u ||
			       0x003D /* = */                         == u) {

			token->tag_state    = ts_tag_after_atrname; /* Got attribute name, parse =  */
			EMIT_TOKEN_CLASS(tf_tag_atrname);

		    } else if (0x0022 /* " */ == u ||
			       0x0027 /* ' */ == u ||
			       0x003C /* < */ == u) {

			/* Report as error, but collect to attribure name */
			
			/* Returns no status */
			add_state_to_string(token->atr_name,token->rchar);

			ADD_SBUFFER_CHAR_NOCHECK;

			/* Same time this is displayed? as error */
			
			EMIT_TOKEN_CLASS(tf_tag_param_error);

		    } else if (0x0000 /* NUL */ == u) {
			uint16 bad_char = UNICODE_BAD_CHAR;
			
			DPRINT(Debug,20,(&Debug, 
					 "get_new_tagfilter_token: Found NUL character when parsing attribute name\n"));
			
			reset_state(token->rchar,0);  /* Get next character */

			token->error       = 1;
			
			add_unicode_to_string(token->atr_name,1,&bad_char);

			add_unicode_to_string(token->sbuffer,1,&bad_char);
			CHECK_SBUFFER_LEN(tf_tag_param_error);			
			
		    } else {

			if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii) &&
			    ((0x0041 /* A */ <= u && u <= 0x005A /* Z */))) {
			    uint16 lower_char = u +  0x0020;
			    add_unicode_to_string(token->atr_name,1,&lower_char);	     
			} else {			
			    /* Returns no status */
			    add_state_to_string(token->atr_name,token->rchar);
			}
			ADD_SBUFFER_CHAR(tf_tag_param_error);
		    }
		    
		} else {
		    clearit(token->tag_flags,TFLAG_seen_equals); /* No = yet */
		    
		    if (0x000A /* LF  '\n' (new line) */       == u) {
						/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);

			EMIT_TOKEN_CLASS(tf_tag_space);
			
		    } else if (0x0009 /* HT  '\t' (horizontal tab) */ == u ||
			       0x000C /* FF  '\f' (form feed) */      == u ||
			       0x0020 /* SPACE */                     == u) {

			ADD_SBUFFER_CHAR(tf_tag_space);
			
		    } else if (0x002F /* / */ == u ||
			       0x003E /* > */ == u) {


			token->tag_state =  ts_tag_ending;  /* Process for /> or  >  */
			
			if (len > 0) {
			    EMIT_TOKEN_CLASS(tf_tag_space);
			}

		    } else if (0x003D /* = */ == u) {

			/* Start tag name and set = as first name of tag name */
			token->atr_name =  new_string(token->text_charset);

			/* Returns no status */
			add_state_to_string(token->atr_name,token->rchar);

			ADD_SBUFFER_CHAR_NOCHECK;

			/* Same time this is displayed? as error */
			
			EMIT_TOKEN_CLASS(tf_tag_param_error);
			
		    } else {
			
			if (len > 0) {
			    EMIT_TOKEN_CLASS(tf_tag_space);
			}
			
			/* Start tag name and reprocess character */
			token->atr_name =  new_string(token->text_charset);
		    }		    
		}
		
	    } else if (ts_tag_comment == token->tag_state) {		
		/* <!-- */
		
		/* May include token->tag_lookahead */

		/* Not correct but this  ends
		   with --> or 
		   --!> (with error)
		*/

		/* Enable buffer */
		
		if (!token->tag_lookahead)
		    token->tag_lookahead =  new_string(token->text_charset);

		if (0x000A /* LF  '\n' */ == u) {
		    /* Consume newline */
		    
		    token->have_nl = 1;
		    reset_state(token->rchar,0);

		    free_string(& token->tag_lookahead);  /* Do not look across lines */

		    EMIT_TOKEN_CLASS(tf_comment_chunk);

		} else if (0x003E /* > */ == u) {

		    int lookahead_len;
		    int r;
		    
		    /* Returns no status */
		    add_state_to_string(token->tag_lookahead,token->rchar);
		    lookahead_len = string_len(token->tag_lookahead);

		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		    if (lookahead_len >= 3 &&
			0 <= (r = string_matches_ascii(token->tag_lookahead,
						       cs2us("-->"),
						       0,SMA_op_find_ascii))) {
			
			free_string(& (token->tag_lookahead));
			token->tag_state    = ts_init;    /* Tag ended */

			EMIT_TOKEN_CLASS(tf_comment_end);
		    } else if (lookahead_len >= 4 &&
			0 <= (r = string_matches_ascii(token->tag_lookahead,
						       cs2us("--!>"),
						       0,SMA_op_find_ascii))) {
			
			free_string(& (token->tag_lookahead));
			token->tag_state    = ts_init;    /* Tag ended */

			EMIT_TOKEN_CLASS(tf_comment_error);
		    }

		    CHECK_SBUFFER_LEN(tf_comment_chunk);
		    
		} else if (0x0000 /* NUL */ == u) {
		    uint16 bad_char = UNICODE_BAD_CHAR;
		    
		    DPRINT(Debug,20,(&Debug, 
				     "get_new_tagfilter_token: Found NUL character when parsing comment text\n"));
		    
		    reset_state(token->rchar,0);  /* Get next character */

		    add_unicode_to_string(token->tag_lookahead,1,&bad_char);
		    
		    token->error       = 1;
		    add_unicode_to_string(token->sbuffer,1,&bad_char);

		    CHECK_SBUFFER_LEN(tf_comment_chunk);
		    
		} else {
		    
		    /* Just append other characters */
		    /* Returns no status */
		    add_state_to_string(token->tag_lookahead,token->rchar);
		    
		    ADD_SBUFFER_CHAR(tf_comment_chunk);
		}
		    		
	    } else if (ts_tag_comment_start == token->tag_state) {

		if (token->tag_lookahead) {
		    if (0x002D /* - */ == u ||
			0x003E /* > */ == u) {
			int lookahead_len;
			
			/* Returns no status */
			add_state_to_string(token->tag_lookahead,token->rchar);
			lookahead_len = string_len(token->tag_lookahead);

			ADD_SBUFFER_CHAR_NOCHECK;

			if (lookahead_len > 3) {			    
			    /* Leaks token->tag_lookahead */
			    
			    token->tag_state       = ts_tag_comment;

			    EMIT_TOKEN_CLASS(tf_comment_start);			    
			} else if (3 == lookahead_len &&
				   string_matches_ascii(token->tag_lookahead,
							cs2us("-->"),
							0,SMA_op_normal)) {
			    free_string(& (token->tag_lookahead));
			    
			    /* <!----> */
			    token->tag_state = ts_init;

			    EMIT_TOKEN_CLASS(tf_whole_comment);
			} else if (2 == lookahead_len &&
				   string_matches_ascii(token->tag_lookahead,
							cs2us("->"),
							0,SMA_op_normal)) {
			    free_string(& (token->tag_lookahead));
			    
			    /* <!---> */
			    token->tag_state       = ts_init;

			    EMIT_TOKEN_CLASS(tf_comment_error);			    
			} 
			
		    } else {

			/* Leaks token->tag_lookahead */
			
			token->tag_state       = ts_tag_comment;

			EMIT_TOKEN_CLASS(tf_comment_start);
		    }

		} else {
		    
		    if (0x000A /* LF  '\n' */ == u) {
			/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);
			
			token->tag_state       = ts_tag_comment;

			EMIT_TOKEN_CLASS(tf_comment_start);			
		    } else if (0x002D /* - */ == u) {
			/* Enable buffer and process again */
			token->tag_lookahead =  new_string(token->text_charset);
		    } else if (0x003E /* > */ == u) {
			/* Closing on empty comment */

			/* <!--> seen */

			token->tag_state       = ts_init;

			EMIT_TOKEN_CLASS(tf_comment_error);
		    } else {
			token->tag_state       = ts_tag_comment;

			EMIT_TOKEN_CLASS(tf_comment_start);
		    }
		}
				
	    } else if (ts_tag_bang == token->tag_state) {

		if (token->tag_lookahead) {
		    if (0x002D /* - */ == u ||
			0x005B /* [ */ == u ||
			(0x0041 /* A */ <= u && u <= 0x005A /* Z */) ||
			(0x0061 /* a */ <= u && u <= 0x007A /* z */)) {
			
			int lookahead_len;
			
			/* Returns no status */
			add_state_to_string(token->tag_lookahead,token->rchar);
			lookahead_len = string_len(token->tag_lookahead);

			ADD_SBUFFER_CHAR_NOCHECK;
					    
			/* DOCTYPE   have 7 characters
			   [CDATA[   have 7 characters
			   --        have 2 characters
			*/
			
			if (lookahead_len > 7) {
			    free_string(& (token->tag_lookahead));
			    
			    /* Bogus <! ... > */
			    token->tag_state = ts_tag_bogus_comment;

			    EMIT_TOKEN_CLASS(tf_bcomment_start);
			} else if (7 == lookahead_len &&
				   string_matches_ascii(token->tag_lookahead,
							cs2us("[CDATA["),
							0,SMA_op_normal)) {
			    
			    free_string(& (token->tag_lookahead));
			    /* CDATA is not now supported .... */

			    /* TODO: Check that is CDATA allowed */

			     token->tag_state = ts_tag_bogus_comment;

			     EMIT_TOKEN_CLASS(tf_bcomment_start);
			    
			    
			} else if (7 == lookahead_len &&
				   string_matches_ascii(token->tag_lookahead,
							cs2us("DOCTYPE"),
							SMA_ignore_case,
							SMA_op_normal)) {

			    /* Seen <!DOCTYPE                  */
			    
			    token->tag_state = ts_tag_doctype_start;
			    token->tag_quote = 0;
			    
			    /* Leaks token->tag_lookahead */

			    EMIT_TOKEN_CLASS(tf_doctype_start);
			    
			} else if (2 == lookahead_len &&
				   string_matches_ascii(token->tag_lookahead,
							cs2us("--"),
							0,SMA_op_normal)) {
			    free_string(& (token->tag_lookahead));
			    
			    /* Seen <!--                       */
			    token->tag_state = ts_tag_comment_start;
			}


			
		    } else {
			free_string(& (token->tag_lookahead));

			/* Bogus <! ... > */
			token->tag_state = ts_tag_bogus_comment;

			EMIT_TOKEN_CLASS(tf_bcomment_start);
		    }

		} else {
		    if (0x002D /* - */ == u ||
			0x0044 /* D */ == u ||
			0x0064 /* d */ == u ||
			0x005B /* [ */ == u) {

			/* Enable buffer and process again */
			token->tag_lookahead =  new_string(token->text_charset);
		    } else {
			/* Bogus <! ... > */

			if (0x000A /* LF  '\n' */ == u) {
			    /* Consume newline */
			    
			    token->have_nl = 1;
			    reset_state(token->rchar,0);
			}
			
			token->tag_state = ts_tag_bogus_comment;

			EMIT_TOKEN_CLASS(tf_bcomment_start);
		    }
		}

	    } else if (ts_tag_endmark == token->tag_state) {

		if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_enriched_tag) &&
		    ((0x0030 /* 0 */ <= u && u <= 0x0039 /* 9 */) ||
		     (0x0041 /* A */ <= u && u <= 0x005A /* Z */) ||
		     (0x0061 /* a */ <= u && u <= 0x007A /* z */) ||
		     0x002D  /* - */ == u)) {
		    
		    /* text/enriched end tag */
		    
		    if (len >= MAX_ENRICHED_TOKEN) {  /* Too long */
			EMIT_TOKEN_CLASS(tf_tag_error);
		    }

		    if (!token->tag_name)
			token->tag_name = new_string(token->text_charset);
		    
		    if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii) &&
			((0x0041 /* A */ <= u && u <= 0x005A /* Z */))) {
			uint16 lower_char = u +  0x0020;
			add_unicode_to_string(token->tag_name,1,&lower_char);
			
		    } else {			
			/* Returns no status */
			add_state_to_string(token->tag_name,token->rchar);
		    }

		    ADD_SBUFFER_CHAR_NOCHECK;

		} else if (len <= 2 &&
			   isoff(tagfilter->tagflt_mode,TAGFLT_MODE_enriched_tag) &&
			   ((0x0041 /* A */ <= u && u <= 0x005A /* Z */) ||
			    (0x0061 /* a */ <= u && u <= 0x007A /* z */))) {
		    
		    /* Start of end tag name */

		    if (!token->tag_name)
			token->tag_name = new_string(token->text_charset);
		    
		    if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii) &&
			((0x0041 /* A */ <= u && u <= 0x005A /* Z */))) {
			uint16 lower_char = u +  0x0020;
			add_unicode_to_string(token->tag_name,1,&lower_char);			    
		    } else {			    
			/* Returns no status */
			add_state_to_string(token->tag_name,token->rchar);
		    }

		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		} else if (0x003E /* > */ == u) {
		    if (len > 2) {

			token->tag_state = ts_init;
			
			ADD_SBUFFER_CHAR_NOCHECK;

			EMIT_TOKEN_CLASS(tf_whole_endtag);
		    } else {
			token->tag_state = ts_init;

			ADD_SBUFFER_CHAR_NOCHECK;

			EMIT_TOKEN_CLASS(tf_tag_error);
		    }
		} else if (len > 2 &&
			   isoff(tagfilter->tagflt_mode,TAGFLT_MODE_enriched_tag)) {

		    /* Parsing name */

		    if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_tag_attributes) &&
			       (0x0009 /* HT  '\t' */ == u ||
				0x000A /* LF  '\n' */ == u ||
				0x000C /* FF  '\f' */ == u ||
				0x0020 /* SPACE */    == u)) {

			if (0x000A /* LF  '\n' */ == u) {
			    /* Consume newline */

			    token->have_nl = 1;
			    reset_state(token->rchar,0);
			}
			
			/* do not include space to name */
			
			token->tag_state = ts_tag_params;

			EMIT_TOKEN_CLASS(tf_start_endtag);

		    } else if (0x0000 /* NUL */ == u) {
			uint16 bad_char = UNICODE_BAD_CHAR;
		
			DPRINT(Debug,20,(&Debug, 
					 "get_new_tagfilter_token: Found NUL character when parsing end tag name\n"));
		
			reset_state(token->rchar,0);  /* Get next character */
			
			token->error       = 1;
			add_unicode_to_string(token->sbuffer,1,&bad_char);

			if (!token->tag_name)
			    token->tag_name = new_string(token->text_charset);
			add_unicode_to_string(token->tag_name,1,&bad_char);

			CHECK_SBUFFER_LEN(tf_tag_error);
		    } else {
			
			/* Accepts eveything as tag name? */
			
			if (!token->tag_name)
			    token->tag_name = new_string(token->text_charset);

			if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii) &&
			    ((0x0041 /* A */ <= u && u <= 0x005A /* Z */))) {
			    uint16 lower_char = u +  0x0020;
			    add_unicode_to_string(token->tag_name,1,&lower_char);			    
			} else {			    
			    /* Returns no status */
			    add_state_to_string(token->tag_name,token->rchar);
			}

			ADD_SBUFFER_CHAR(tf_tag_error);
		    }
			
		} else if (isoff(tagfilter->tagflt_mode,TAGFLT_MODE_enriched_tag)) {

		    /* Bogus </ ... > */

		    token->tag_state = ts_tag_bogus_comment;

		    if (0x000A /* LF  '\n' */ == u) {
			/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);
		    }

		    EMIT_TOKEN_CLASS( tf_bcomment_start);
		    
		} else {

		    if (0x000A /* LF  '\n' */ == u) {
			/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);
		    }
		    
		    /* Parse error on tag -- do not include character */
		    
		    token->tag_state = ts_init;

		    EMIT_TOKEN_CLASS(tf_tag_error);
		}
		    		
	    } else if (ts_tag_bogus_comment == token->tag_state) {

		if (0x003E /* > */ == u) {
		    token->tag_state = ts_init;

		    ADD_SBUFFER_CHAR_NOCHECK;

		    EMIT_TOKEN_CLASS(tf_bcomment_end);
		} else if (0x0000 /* NUL */ == u) {
		    uint16 bad_char = UNICODE_BAD_CHAR;
		    
		    DPRINT(Debug,20,(&Debug, 
				     "get_new_tagfilter_token: Found NUL character when parsing < ... comment\n"));
		    
		    reset_state(token->rchar,0);  /* Get next character */
		    
		    token->error       = 1;
		    add_unicode_to_string(token->sbuffer,1,&bad_char);

		    CHECK_SBUFFER_LEN(tf_bcomment_chunk);
		} else {
		    /* Accepts eveything as  comment? */

		    if (0x000A /* LF  '\n' */ == u) {
			/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);

			EMIT_TOKEN_CLASS(tf_bcomment_chunk);
		    }

		    ADD_SBUFFER_CHAR(tf_bcomment_chunk);
		}

	    } else if (ts_tag_self_closing == token->tag_state) {

		setit(token->tag_flags,TFLAG_self_closing);
			
		if (0x003E /* > */ == u) {
		    		    
		    token->tag_state = ts_init;

		    ADD_SBUFFER_CHAR_NOCHECK;

		    clearit(token->tag_flags,TFLAG_self_closing); /* Passed already */

		    EMIT_TOKEN_CLASS(tf_selfclosed_tag);
		    
		} else if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_tag_attributes)) {
		    
		    /* Do not include to name */

		    if (0x000A /* LF  '\n' */ == u) {
			/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);
		    }
		    
		    token->tag_state = ts_tag_params;

		    EMIT_TOKEN_CLASS(tf_start_tag);		    
		} else {
		    /* Back to parsing on tag name */

		    token->tag_state = ts_tag_start;		    
		    
		    if (0x000A /* LF  '\n' */ == u) {
			/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);

			EMIT_TOKEN_CLASS(tf_tag_error);
		    } else {
			
			if (!token->tag_name)
			    token->tag_name = new_string(token->text_charset);

			if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii) &&
			    ((0x0041 /* A */ <= u && u <= 0x005A /* Z */))) {
			    uint16 lower_char = u +  0x0020;
			    add_unicode_to_string(token->tag_name,1,&lower_char);
			} else {			    
			    /* Returns no status */
			    add_state_to_string(token->tag_name,token->rchar);
			}
		    }
		}
				
	    } else if (ts_tag_start == token->tag_state) {

		clearit(token->tag_flags,TFLAG_self_closing); /* CLear possible bogus flag */
			
		if (1 == len &&
		    ison(tagfilter->tagflt_mode,TAGFLT_MODE_double_smaller) &&
		    0x003C /* <  */ == u) {

		    /* text/enriched escaped < */

		    /* Note that sbuffer includes all charaters 
		       part of token -- not just token value
		    */

		    token->tag_state = ts_init;

		    ADD_SBUFFER_CHAR_NOCHECK;

		    EMIT_TOKEN_CLASS(tf_double_smaller);
		    
		} else if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_enriched_tag) &&
			   ((0x0030 /* 0 */ <= u && u <= 0x0039 /* 9 */) ||
			    (0x0041 /* A */ <= u && u <= 0x005A /* Z */) ||
			    (0x0061 /* a */ <= u && u <= 0x007A /* z */) ||
			    0x002D  /* - */ == u)) {

		    /* text/enriched tag */
		    
		    if (len >= MAX_ENRICHED_TOKEN) {  /* Too long */
			EMIT_TOKEN_CLASS(tf_tag_error);
		    }

		    if (!token->tag_name)
			token->tag_name = new_string(token->text_charset);

		    if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii) &&
			((0x0041 /* A */ <= u && u <= 0x005A /* Z */))) {
			uint16 lower_char = u +  0x0020;
			add_unicode_to_string(token->tag_name,1,&lower_char);
		    } else {
			/* Returns no status */
			add_state_to_string(token->tag_name,token->rchar);
		    }

		    ADD_SBUFFER_CHAR_NOCHECK;

		} else if (1 == len && tagfilter->doctype_name && 0x0021 /* ! */ == u)  {
		    /* Markup declaration */

		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		    token->tag_state = ts_tag_bang;
		} else if (1 == len && 0x002F /* / */ == u) {
		    /* end tag */
		    
		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		    token->tag_state = ts_tag_endmark;
		} else if (1 == len &&
			   isoff(tagfilter->tagflt_mode,TAGFLT_MODE_enriched_tag) &&
			   ((0x0041 /* A */ <= u && u <= 0x005A /* Z */) ||
			    (0x0061 /* a */ <= u && u <= 0x007A /* z */))) {
		    
		    /* Start of tag name */

		    if (!token->tag_name)
			token->tag_name = new_string(token->text_charset);

		    if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii) &&
			((0x0041 /* A */ <= u && u <= 0x005A /* Z */))) {
			uint16 lower_char = u +  0x0020;
			add_unicode_to_string(token->tag_name,1,&lower_char);

		    } else {
			/* Returns no status */
			add_state_to_string(token->tag_name,token->rchar);
		    }

		    ADD_SBUFFER_CHAR_NOCHECK;

		} else if (1 == len &&
			   isoff(tagfilter->tagflt_mode,TAGFLT_MODE_enriched_tag) &&
			   0x003F /* ? */ == u) {

		    /* Bogus <?   > */

		    token->tag_state = ts_tag_bogus_comment;

		    ADD_SBUFFER_CHAR_NOCHECK;

		    EMIT_TOKEN_CLASS(tf_bcomment_start);		    
		} else if (len > 1 && 0x003E /* > */ == u) {
		    
		    token->tag_state = ts_init;

		    ADD_SBUFFER_CHAR_NOCHECK;

		    EMIT_TOKEN_CLASS(tf_whole_tag);		    
		} else if (len > 1 &&
			   isoff(tagfilter->tagflt_mode,TAGFLT_MODE_enriched_tag)) {

		    /* Parsing name */
		    
		    if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_tag_attributes) &&
			       (0x0009 /* HT  '\t' */ == u ||
				0x000A /* LF  '\n' */ == u ||
				0x000C /* FF  '\f' */ == u ||
				0x0020 /* SPACE */    == u)) {

			if (0x000A /* LF  '\n' */ == u) {
			    /* Consume newline */

			    token->have_nl = 1;
			    reset_state(token->rchar,0);
			}
			
			/* do not include space to name */

			token->tag_state = ts_tag_params;

			EMIT_TOKEN_CLASS(tf_start_tag);
		    } else if (0x002F /* / */ == u) {

			token->tag_state = ts_tag_self_closing;
			
			ADD_SBUFFER_CHAR(tf_tag_error);
			
		    } else if (0x0000 /* NUL */ == u) {
			uint16 bad_char = UNICODE_BAD_CHAR;
		
			DPRINT(Debug,20,(&Debug, 
					 "get_new_tagfilter_token: Found NUL character when parsing tag name\n"));
		
			reset_state(token->rchar,0);  /* Get next character */
			
			token->error       = 1;
			add_unicode_to_string(token->sbuffer,1,&bad_char);

			if (!token->tag_name)
			    token->tag_name = new_string(token->text_charset);
			add_unicode_to_string(token->tag_name,1,&bad_char);
			
			CHECK_SBUFFER_LEN(tf_tag_error);
		    } else {
			if (!token->tag_name)
			    token->tag_name = new_string(token->text_charset);

			if (ison(tagfilter->tagflt_mode,TAGFLT_MODE_lowerascii) &&
			    ((0x0041 /* A */ <= u && u <= 0x005A /* Z */))) {
			    uint16 lower_char = u +  0x0020;
			    add_unicode_to_string(token->tag_name,1,&lower_char);
			} else {
			    /* Returns no status */
			    add_state_to_string(token->tag_name,token->rchar);
			}
			    
			/* Accepts eveything as tag name? */

			ADD_SBUFFER_CHAR(tf_tag_error);
		    }
		    
		} else {

		    if (0x000A /* LF  '\n' */ == u) {
			/* Consume newline */
			
			token->have_nl = 1;
			reset_state(token->rchar,0);
		    }
		    
		    /* Parse error on tag -- do not include character */

		    token->tag_state = ts_init;

		    EMIT_TOKEN_CLASS(tf_tag_error);
		}
		
		
	    } else if (ts_init == token->tag_state) {

		if (0x003C /* < */ == u) {

		    if (len > 0) {
			/* Do not eat character -- next character starts tag */
			
			EMIT_TOKEN_CLASS(tf_body);
		    }
		    
		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		    token->tag_state = ts_tag_start;
		    
		} else if (0x0026 /* &  */ == u  &&
			   tagfilter->have_entities) {
		
		    if (len > 0) {
			/* Do not eat character -- next character starts entity */
			
			EMIT_TOKEN_CLASS(tf_body);
		    }
		    		    
		    token->entity_state    = ent_entity_start;
		    
		    /* Start & .... */
		    token->named_reference = new_string(token->text_charset);
		    /* Returns no status */
		    add_state_to_string(token->named_reference,token->rchar);
		    token->walk_reference =  tagfilter_start_reference(tagfilter->have_entities);
		    
		    ADD_SBUFFER_CHAR_NOCHECK;
		    
		} else
		    goto parse_body;
		    
	    } else {

		if (!badstate) {
		    DPRINT(Debug,20,(&Debug, 
				     "get_new_tagfilter_token: bad token->tag_state=%d\n",
				     token->tag_state));
		    badstate = 1;
		}
		

	    parse_body:
		
		/* On body of mail */
		
		if (0x000A /* LF  '\n' */ == u) {
		    token->have_nl = 1;

		    setit(token->tag_flags,TFLAG_seen_nl);
					    
		    reset_state(token->rchar,0);
		    
		    EMIT_TOKEN_CLASS(tf_body);
		}

		ADD_SBUFFER_CHAR(tf_body);
	    }
		
	} else {
	    ch = state_getc(state_in);

	process_char:
	     if (EOF == ch) {

		if (!gotbyte)  {
		    ret = -1;

		    free_string(& (token->sbuffer));
		} else {
		    len = string_len(token->sbuffer);
		    
		    ret = len;
		}
		token->eof             = 1;
		token->token_class     = tf_body;   /* EOF is outside of tag */
		
		goto out;
	    }
	    gotbyte++;

	    if (! add_streambyte_to_state(token->rchar,ch)) {
		uint16 bad_char = UNICODE_BAD_CHAR;
		
		DPRINT(Debug,20,(&Debug, 
				 "get_new_tagfilter_token: Failed to add byte %02x to character when scanning token\n"));
		token->error       = 1;
		reset_state(token->rchar,0);  
		
		add_unicode_to_string(token->sbuffer,1,&bad_char);
		if (token->tag_lookahead)
		    add_unicode_to_string(token->tag_lookahead,1,&bad_char);
		if (token->tag_name)
		    add_unicode_to_string(token->tag_name,1,&bad_char);
		if (token->atr_name)
		    add_unicode_to_string(token->atr_name,1,&bad_char);

	    }
	}

    } while(1);
    
 out:

    DPRINT(Debug,20,(&Debug, 
		     "get_new_tagfilter_token=%d:\n",
		     ret));

    DPRINT(Debug,20,(&Debug, "    token_class  %d",
		     token->token_class));
    switch(token->token_class) {
    case tf_doctype_error:      DPRINT(Debug,20,(&Debug," tf_doctype_error"));       break;
    case tf_tag_atrvalue_error: DPRINT(Debug,20,(&Debug," tf_tag_atrvalue_error"));  break;
    case tf_tag_param_error:    DPRINT(Debug,20,(&Debug," tf_tag_param_error"));     break;
    case tf_comment_error:      DPRINT(Debug,20,(&Debug," tf_comment_error"));       break;
    case tf_bcomment_error:     DPRINT(Debug,20,(&Debug," tf_bcomment_error"));      break;
    case tf_tag_error:          DPRINT(Debug,20,(&Debug," tf_tag_error"));           break;
    case tf_entity_error:       DPRINT(Debug,20,(&Debug," tf_entity_error"));        break;
    case tf_body:               DPRINT(Debug,20,(&Debug," tf_body"));                break;
    case tf_start_tag:          DPRINT(Debug,20,(&Debug," tf_start_tag"));           break;                 /* <tag  */
    case tf_whole_tag:          DPRINT(Debug,20,(&Debug," tf_whole_tag"));           break;                 /* <tag> */
    case tf_selfclosed_tag:     DPRINT(Debug,20,(&Debug," tf_selfclosed_tag"));      break;                 /* <tag/> */
    case tf_bcomment_start:     DPRINT(Debug,20,(&Debug," tf_bcomment_start"));      break;                 /* <? or <!  or </ */
    case tf_bcomment_chunk:     DPRINT(Debug,20,(&Debug," tf_bcomment_chunk"));      break;                 /* <? bogus comment chunk */
    case tf_bcomment_end:       DPRINT(Debug,20,(&Debug," tf_bcomment_end"));        break;                 /* end bogus comment > */
    case tf_start_endtag:       DPRINT(Debug,20,(&Debug," tf_start_endtag"));        break;                 /* </tag  */
    case tf_whole_endtag:       DPRINT(Debug,20,(&Debug," tf_whole_endtag"));        break;                 /* </tag> */
    case tf_entity:             DPRINT(Debug,20,(&Debug," tf_entity"));              break;
    case tf_numeric_entity:     DPRINT(Debug,20,(&Debug," tf_numeric_entity"));      break;                 /* Numeric entity */
    case tf_double_smaller:     DPRINT(Debug,20,(&Debug," tf_double_smaller"));      break;                 /* << as escaping */
    case tf_span_nl:            DPRINT(Debug,20,(&Debug," tf_span_nl"));             break;                 /* span of newline (except first) */
    case tf_comment_start:      DPRINT(Debug,20,(&Debug," tf_comment_start"));       break;                 /* <!-- */
    case tf_whole_comment:      DPRINT(Debug,20,(&Debug," tf_whole_comment"));       break;                 /* <!----> */
    case tf_comment_chunk:      DPRINT(Debug,20,(&Debug," tf_comment_chunk"));       break;                 /* <!-- comment chunk  */
    case tf_comment_end:        DPRINT(Debug,20,(&Debug," tf_comment_end"));         break;                 /* end comment --> */
    case tf_tag_space:          DPRINT(Debug,20,(&Debug," tf_tag_space"));           break;                 /* space on attributes */
    case tf_tag_atrname:        DPRINT(Debug,20,(&Debug," tf_tag_atrname"));         break;                 /* Got attribute name           */
    case tf_tag_selfclosed_end: DPRINT(Debug,20,(&Debug," tf_tag_selfclosed_end"));  break;                 /* Got /> */
    case tf_tag_end:            DPRINT(Debug,20,(&Debug," tf_tag_end"));             break;                 /* Got > */
    case tf_tag_atrequal:       DPRINT(Debug,20,(&Debug," tf_tag_atrequal"));        break;                 /* Got = */
    case tf_tag_atrvalue_start: DPRINT(Debug,20,(&Debug," tf_tag_atrvalue_start"));  break;                 /* Start attribute value */
    case tf_tag_atrvalue_segment: DPRINT(Debug,20,(&Debug," tf_tag_atrvalue_segment")); break;              /* Part of attribute value */
    case tf_tag_atrvalue_end:   DPRINT(Debug,20,(&Debug," tf_tag_atrvalue_end"));    break;                 /* Start attribute value */
    case tf_doctype_start:      DPRINT(Debug,20,(&Debug," tf_doctype_start"));       break;                 /*  <!DOCTYPE            */
    case tf_doctype_segment:    DPRINT(Debug,20,(&Debug," tf_doctype_segment"));     break;                 /* Part of DOCTYPE       */
    case tf_doctype_space:      DPRINT(Debug,20,(&Debug," tf_doctype_space"));       break;                 /* Space on doctype      */
    case tf_doctype_item:       DPRINT(Debug,20,(&Debug," tf_doctype_item"));        break;                 /* Collected doctype item */
    case tf_doctype_end:        DPRINT(Debug,20,(&Debug," tf_doctype_end"));         break;                 /* DOCTYPE line ended    */	
    }
    DPRINT(Debug,20,(&Debug, "\n"));
    if (token->tag_name) {
	DEBUG_PRINT_STRING(Debug,20,
			   "    tag_name     ",
			   "    tag_name >   ",
			   token->tag_name);
    }
    if (token->atr_name) {
	DEBUG_PRINT_STRING(Debug,20,
			   "    atr_name     ",
			   "    atr_name >   ",
			   token->atr_name);
    }
    if (token->atr_value_segment) {
	DEBUG_PRINT_STRING(Debug,20,
			   "    atr_value_segment     ",
			   "    atr_value_segment >   ",
			   token->atr_value_segment);
    }
    if (token->named_reference) {
	DEBUG_PRINT_STRING(Debug,20,
			   "    named_reference     ",
			   "    named_reference >   ",
			   token->named_reference);
    }

    if (UNICODE_BAD_CHAR != token->numeric_reference ||
	tf_numeric_entity == token->token_class ||
	ison(token->tag_flags,TFLAG_num_overflow)) {
	
	DPRINT(Debug,20,(&Debug, "    numeric_reference  %04x\n",
			 token->numeric_reference));	
    }
    if (token->doctype_item) {
	DEBUG_PRINT_STRING(Debug,20,
			   "    doctype_item     ",
			   "    doctype_item >   ",
			   token->doctype_item);
    }
    if (token->have_nl) {
    	DPRINT(Debug,20,(&Debug, "    have_nl\n"));
    }
    if (token->eof) {
    	DPRINT(Debug,20,(&Debug, "    eof\n"));
    }
    if (token->error) {
    	DPRINT(Debug,20,(&Debug, "    error\n"));
    }

    if (token->tag_state   != ts_init ||
	token->token_class != tf_body ||
	ison(token->tag_flags,TFLAG_self_closing) ||
	ison(token->tag_flags,TFLAG_seen_equals) ||
	token->tag_quote ||
	token->tag_lookahead
	) {
	DPRINT(Debug,20,(&Debug, "    tag_state    %d",
			 token->tag_state));

	switch (token->tag_state) {
	case ts_init: 		DPRINT(Debug,20,(&Debug, " ts_init")); 								break;
	case ts_tag_start:  	DPRINT(Debug,20,(&Debug, " ts_tag_start"));        /* Seen <                          */	break;
	case ts_tag_bang:  	DPRINT(Debug,20,(&Debug, " ts_tag_bang"));         /* Seen <!                         */	break;
	case ts_tag_endmark:  	DPRINT(Debug,20,(&Debug, " ts_tag_endmark"));      /* Seen </                         */	break;
	case ts_tag_params:  	DPRINT(Debug,20,(&Debug, " ts_tag_params"));       /* Seen space after <  or expecting params  */ break;
	case ts_tag_ending:  	DPRINT(Debug,20,(&Debug, " ts_tag_ending"));      /* Process for /> or  >            */ 	break;
	case ts_tag_bogus_comment: DPRINT(Debug,20,(&Debug, " ts_tag_bogus_comment")); /* Parse to >                      */ 	break;
	case ts_tag_self_closing:  DPRINT(Debug,20,(&Debug, " ts_tag_self_closing")); /* Seen <tag/                      */ 	break;
	case ts_tag_comment_start: DPRINT(Debug,20,(&Debug, " ts_tag_comment_start")); /* Seen <!--                       */ 	break;
	case ts_tag_comment:	DPRINT(Debug,20,(&Debug, " ts_tag_comment"));      /* Inside of comment               */ 	break;
	case ts_tag_after_atrname: DPRINT(Debug,20,(&Debug, " ts_tag_after_atrname")); /* Got attribure name, parse =     */ 	break;
	case ts_tag_atrvalue:	DPRINT(Debug,20,(&Debug, " ts_tag_atrvalue"));     /* On attribure value              */ 	break;
	case ts_tag_after_quoted:  DPRINT(Debug,20,(&Debug, " ts_tag_after_quoted")); /* After quoted value              */ 	break;
	case ts_tag_doctype_start: DPRINT(Debug,20,(&Debug, " ts_tag_doctype_start")); /* Seen <!DOCTYPE                  */ 	break;
	case ts_tag_doctype_bogus: DPRINT(Debug,20,(&Debug, " ts_tag_doctype_bogus")); /* Bogus DOCTYPE line              */ 	break;
	}
	
	DPRINT(Debug,20,(&Debug, "\n"));

	DPRINT(Debug,20,(&Debug, "    tag_quote    %04x",
			 token->tag_quote));
	switch (token->tag_quote) {
	case 0: DPRINT(Debug,20,(&Debug, " (none)"));    break;
	case 0x0027: DPRINT(Debug,20,(&Debug, " (')"));  break;
	case 0x0022: DPRINT(Debug,20,(&Debug, " (\")")); break;
	}
	DPRINT(Debug,20,(&Debug, "\n"));

	if (token->tag_lookahead) {
	    DEBUG_PRINT_STRING(Debug,20,
			       "    tag_lookahead    ",
			       "    tag_lookahead >  ",
			       token->tag_lookahead);
	}
    }
    
    if (token->tag_flags) {
	DPRINT(Debug,20,(&Debug, "    tag_flags    %d",
			 token->tag_flags));
	
	if (ison(token->tag_flags,TFLAG_seen_nl)) {
	    DPRINT(Debug,20,(&Debug, " TFLAG_seen_nl"));
	}
	if (ison(token->tag_flags,TFLAG_seen_equals)) {
	    DPRINT(Debug,20,(&Debug, " TFLAG_seen_equals"));
	}
	if (ison(token->tag_flags,TFLAG_num_overflow)) {
	    DPRINT(Debug,20,(&Debug, " TFLAG_num_overflow"));
	}
	if (ison(token->tag_flags,TFLAG_unhandled_class)) {
	    DPRINT(Debug,20,(&Debug, " TFLAG_unhandled_class"));
	}
	
	DPRINT(Debug,20,(&Debug, "\n"));
    }

    
    if (token->entity_state != 	ent_init ||
	token->tag_state    != ts_init ||
	token->token_class  == tf_body) {

	DPRINT(Debug,20,(&Debug, "    entity_state %d",
			 token->entity_state));

	switch (token->entity_state) {
	case ent_init:            DPRINT(Debug,20,(&Debug, " ent_init"));         break;
	case ent_entity_start:    DPRINT(Debug,20,(&Debug, " ent_entity_start")); break; /* Seen & on tag params or body    */
	case ent_decimal_ent:     DPRINT(Debug,20,(&Debug, " ent_decimal_ent"));  break; /* Seen &# on tag params or body   */
	case ent_hexdec_ent:      DPRINT(Debug,20,(&Debug, " ent_hexdec_ent"));   break; /* Seen &#x or &#X                 */
	}
	
	DPRINT(Debug,20,(&Debug, "\n"));
    }

    if (token->resubmit) {
	DEBUG_PRINT_STRING(Debug,20,
			   "    resubmit     ",
			   "    resubmit >   ",
			   token->resubmit);
    }

    if (token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "    sbuffer      ",
			   "    sbuffer >    ",
			   token->sbuffer);
    }
        
    return ret;
}

#define TAGFILTER_PARAMS_magic		0xFD12

struct tagfilter_params {
    unsigned short               magic;    /* TAGFILTER_PARAMS_magic */

    mime_t                     * body;
    struct in_state            * state_in;
    struct out_state           * state_out;
    const struct decode_opts   * decode_opt;
    charset_t                    text_charset;
    struct tagfilter_selection * tagfilter;

    struct tagfilter_token      * current_token;
    int                           error;

    struct tagfilter_stack_item **stack;
    size_t                        stack_len;

    struct tagfilter_global     * counter;
};

static struct pager_range *tagfilter_inherited_pager_range
                              P_((struct tagfilter_params *params));
static struct pager_range *tagfilter_inherited_pager_range(params)
     struct tagfilter_params *params;
{
    struct pager_range *inherit = NULL;

    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_inherited_pager_range",
                   "Bad magic number");
    

    if (params->stack && params->stack_len > 0) {
	size_t i = params->stack_len;
	    
	while (i > 0) {
	    i--;

	    if (params->stack[i]) {
		
		if (TAGFILTER_STACK_ITEM_magic != params->stack[i]->magic)
		    mime_panic(__FILE__,__LINE__,
			       "tagfilter_inherited_pager_range",
			       "Bad magic number (tagfilter_stack_item)");

		inherit = params->stack[i]->range;
		if (inherit)
		    break;		
	    }
	}		
    }
   
    return inherit;
}

static int tagfilter_inherited_pg_flags  P_((struct tagfilter_params *params));
static int tagfilter_inherited_pg_flags(params)
       struct tagfilter_params *params;
{
    int inherit = 0;
    int disable_pg_flags = 0;
    
    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_inherited_pg_flags",
                   "Bad magic number");
    
    if (params->stack && params->stack_len > 0) {
	size_t i = params->stack_len;
	    
	while (i > 0) {
	    i--;

	    if (params->stack[i]) {
		int flags;
		int d;
		
		if (TAGFILTER_STACK_ITEM_magic != params->stack[i]->magic)
		    mime_panic(__FILE__,__LINE__,"tagfilter_inherited_pg_flags",
			       "Bad magic number ( tagfilter_stack_item)");

		flags = params->stack[i]->pg_flags;
		
		clearit(flags,disable_pg_flags);
		setit(inherit,flags);
		
		/* No values before this */
		if (params->stack[i]->reset_pg_flags)
		    break;

		d = pg_set_or_disable_flags(NULL,flags);
		setit(disable_pg_flags,d);
	    }
	}
    }

    return inherit;
}

static void tagfilter_dump_token  P_((struct tagfilter_params *params,
				      int pg_flags));
static void tagfilter_dump_token(params,pg_flags)
     struct tagfilter_params *params;
     int pg_flags;
{    
    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_dump_token",
                   "Bad magic number (tagfilter_params)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_dump_token",
                   "Bad magic number (tagfilter_token)");

    if (params->current_token->sbuffer)
	state_putstring(params->current_token->sbuffer,
			params->state_out);

    if (params->current_token->error)
	params->error = 1;
    
    if (params->current_token->have_nl) {
	struct pager_range *inherit
	    = tagfilter_inherited_pager_range(params);
	
	state_putc('\n',params->state_out);
	
	/* Re-set pg_flags */
	set_out_state_line_mode(params->state_out,pg_flags,
				inherit,0 );
    }
}

#define MAX_COLLECTED 1002

enum collect_truncated {
    collect_full_tag,
    collect_truncated_tag,
    collect_nl_tag,
    collect_nl_truncated_tag,
    
};


static void tagfilter_collect_token  P_((struct tagfilter_params *params,
					 struct string ** collected_tag,
					 enum collect_truncated *truncated_tag));
static void tagfilter_collect_token(params,collected_tag,truncated_tag)
     struct tagfilter_params *params;
     struct string ** collected_tag;
     enum collect_truncated *truncated_tag;
{
    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_collect_token",
                   "Bad magic number (tagfilter_params)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_collect_token",
                   "Bad magic number (tagfilter_token)");

    if (params->current_token->error)
	params->error = 1;

    switch (*truncated_tag) {
    case collect_truncated_tag:
    case collect_nl_truncated_tag:
	break;
    case collect_nl_tag:
	if (params->current_token->sbuffer)
	    *truncated_tag = collect_nl_truncated_tag;
	break;
	
    case collect_full_tag:
	if ((* collected_tag) &&
	    string_len(* collected_tag) >= MAX_COLLECTED) {
	    *truncated_tag = collect_truncated_tag;
	} else {
	    if (params->current_token->sbuffer)
		append_string(collected_tag,params->current_token->sbuffer,0);

	     if (params->current_token->have_nl)
		 *truncated_tag = collect_nl_tag;
	}	
    }
}


#if DEBUG
static char * token_class_string P_((enum tf_token_class cl));
static char * token_class_string(cl)
     enum tf_token_class cl;
{
    switch (cl) {
    case tf_doctype_error: 	return "tf_doctype_error";
    case tf_tag_atrvalue_error:	return "tf_tag_atrvalue_error";
    case tf_tag_param_error:	return "tf_tag_param_error";
    case tf_comment_error: 	return "tf_comment_error";
    case tf_bcomment_error:	return "tf_bcomment_error";
    case tf_tag_error: 		return "tf_tag_error";
    case tf_entity_error: 	return "tf_entity_error";
    case tf_body: 		return "tf_body";
    case tf_start_tag:		return "tf_start_tag";                           /* <tag  */
    case tf_whole_tag:		return "tf_whole_tag";                           /* <tag> */
    case tf_selfclosed_tag:	return "tf_selfclosed_tag";                      /* <tag/> */
    case tf_bcomment_start:	return "tf_bcomment_start";                      /* <? or <!  or </ */
    case tf_bcomment_chunk:	return "tf_bcomment_chunk";                      /* <? gogus comment chunk */
    case tf_bcomment_end:	return "tf_bcomment_end";                        /* end bogus comment > */
    case tf_start_endtag:	return "tf_start_endtag";                        /* </tag  */
    case tf_whole_endtag:	return "tf_whole_endtag";                        /* </tag> */
    case tf_entity:		return "tf_entity";
    case tf_numeric_entity:	return "tf_numeric_entity";                      /* Numeric entity */
    case tf_double_smaller:	return "tf_double_smaller";                      /* << as escaping */
    case tf_span_nl:		return "tf_span_nl";                             /* span of newline (except first) */
    case tf_comment_start:	return "tf_comment_start";                       /* <!-- */
    case tf_whole_comment:	return "tf_whole_comment";                       /* <!----> */
    case tf_comment_chunk:	return "tf_comment_chunk";                       /* <!-- comment chunk  */
    case tf_comment_end:	return "tf_comment_end";                         /* end comment --> */
    case tf_tag_space:		return "tf_tag_space";                           /* space on attributes */
    case tf_tag_atrname:	return "tf_tag_atrname";                         /* Got attribute name           */
    case tf_tag_selfclosed_end:	return "tf_tag_selfclosed_end";                  /* Got /> */
    case tf_tag_end:		return "tf_tag_end";                             /* Got > */
    case tf_tag_atrequal:	return "tf_tag_atrequal";                        /* Got = */
    case tf_tag_atrvalue_start:	return "tf_tag_atrvalue_start";                  /* Start attribute value */
    case tf_tag_atrvalue_segment:	return "tf_tag_atrvalue_segment";        /* Part of attribute value */
    case tf_tag_atrvalue_end:	return "tf_tag_atrvalue_end";                    /* Start attribute value */
    case tf_doctype_start:	return "tf_doctype_start";                       /*  <!DOCTYPE            */
    case tf_doctype_segment:	return "tf_doctype_segment";                     /* Part of DOCTYPE       */
    case tf_doctype_space:	return "tf_doctype_space";                       /* Space on doctype      */
    case tf_doctype_item:	return "tf_doctype_item";                        /* Collected doctype item */
    case tf_doctype_end:	return "tf_doctype_end";                         /* DOCTYPE line ended    */	
    }

    return "<unknown>";
}

#endif

/* Returns current_token_len -- */

static int tagfilter_handle_doctype P_((struct tagfilter_params *params,
					int current_token_len));

static int tagfilter_handle_doctype(params,current_token_len)
     struct tagfilter_params * params;
     int current_token_len;
{
    int doctype_ok = 0;
    int idx = 0;
    struct string        * collected_tag  = NULL;
    enum collect_truncated truncated_tag  = collect_full_tag;
    int inherit_pg_flags;
    struct pager_range *inherit;
    
    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_doctype",
                   "Bad magic number");

    if (TAGFILTER_SELECTION_magic != params->tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_doctype",
		   "Bad magic number (tagfilter_selection)");
    
    if (MIME_magic != params->body->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_doctype",
		   "Bad magic number (mime_t)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_doctype",
                   "Bad magic number (tagfilter_token)");

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_doctype: START current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_doctype: current token = ",
			   "tagfilter_handle_doctype: current token > ",
			   params->current_token->sbuffer);
    }
    
    inherit_pg_flags = tagfilter_inherited_pg_flags(params);
    inherit          = tagfilter_inherited_pager_range(params);
       
    while (current_token_len >= 0) {
	if (params->current_token->error) {
	    struct pager_range *title_range = 
		state_add_simple_pager_range(params->state_out,inherit,PR_MAX_WIDTH,0,
					     0);
	    
	    /* \n resets this */        
	    set_out_state_line_mode(params->state_out,pg_BOLD,title_range,
				    1 /* Newline */);
	    state_nlputs("[ ",params->state_out);
	    
	    state_printf(params->state_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MeTagFErrorDoctypeDeclaration,
				 "Character error when parsing DOCTYPE declaration for %s/%s."),
			 get_major_type_name(params->body->TYPE),
			 get_subtype_name(params->body->TYPE));
	    state_nlputs(" ]\n",params->state_out);

	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeTagFErrorDoctypeDeclaration,
			      "Character error when parsing DOCTYPE declaration for %s/%s."),
		      get_major_type_name(params->body->TYPE),
		      get_subtype_name(params->body->TYPE));

	    free_pager_range(&title_range);  
	    
	    goto fail_doctype;
	}


	switch(params->current_token->token_class) {
	case tf_doctype_error:
	case tf_doctype_segment:
	    goto fail_doctype;
	    
	case tf_doctype_start:
	    if (0 != idx)
		goto fail_doctype;
	    break;
	case tf_doctype_space:
	    idx++;
	    break;
	case tf_doctype_item:

	    if (1 == idx &&
		params->tagfilter->doctype_name &&
		params->current_token->doctype_item) {
		
		if (string_matches_ascii(params->current_token->doctype_item,
					 cs2us(params->tagfilter->doctype_name),
					 SMA_ignore_case,SMA_op_normal)) {
		    doctype_ok = 1;
		} else {
		    struct pager_range *title_range = 
			state_add_simple_pager_range(params->state_out,inherit,
						     PR_MAX_WIDTH,0,
						     0);
		   		    
		    /* \n resets this */        
		    set_out_state_line_mode(params->state_out,pg_BOLD,title_range,
					    1 /* Newline */);
		    state_nlputs("[ ",params->state_out);

		    state_printf(params->state_out,
				 CATGETS(elm_msg_cat, MeSet, 
					 MeTagFIdentifierDoctypeDeclaration,
					 "Got identifier %S when parsing DOCTYPE declaration for %s/%s."),
				 params->current_token->doctype_item,
				 get_major_type_name(params->body->TYPE),
				 get_subtype_name(params->body->TYPE));
				 
		    state_nlputs(" ]\n",params->state_out);

		    lib_error(CATGETS(elm_msg_cat, MeSet,
				      MeTagFIdentifierDoctypeDeclaration,
				      "Got identifier %S when parsing DOCTYPE declaration for %s/%s."),
			      params->current_token->doctype_item,
			      get_major_type_name(params->body->TYPE),
			      get_subtype_name(params->body->TYPE));

		    free_pager_range(&title_range);  
	    
		    goto fail_doctype;
		}
	    }
	    break;
	case tf_doctype_end:

	    tagfilter_collect_token(params,&collected_tag,&truncated_tag);
	    
	    current_token_len =
		get_new_tagfilter_token(params->current_token,
					params->tagfilter,
					params->state_in);
	    goto check_doctype;
	default:
	    goto check_doctype;
	}

	tagfilter_collect_token(params,&collected_tag,&truncated_tag);
	
	current_token_len =
	    get_new_tagfilter_token(params->current_token,
				    params->tagfilter,
				    params->state_in);   
	
    }
	   
 check_doctype:
    
    if (!doctype_ok) {
	int pg_flags;
	
	if (params->tagfilter->doctype_name) {

	    struct pager_range *title_range = 
		state_add_simple_pager_range(params->state_out,inherit,
					     PR_MAX_WIDTH,0,
					     0);
	    
	    /* \n resets this */        
	    set_out_state_line_mode(params->state_out,pg_BOLD,title_range,
				    1 /* Newline */);
	    state_nlputs("[ ",params->state_out);

	    state_printf(params->state_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MeTagFNoIdentDoctypeDeclaration,
				 "Identifier %s not found when parsing DOCTYPE declaration for %s/%s."),
			 params->tagfilter->doctype_name,
			 get_major_type_name(params->body->TYPE),
			 get_subtype_name(params->body->TYPE));
				 
	    state_nlputs(" ]\n",params->state_out);

	    lib_error(CATGETS(elm_msg_cat, MeSet,
			      MeTagFNoIdentDoctypeDeclaration,
			      "Identifier %s not found when parsing DOCTYPE declaration for %s/%s."),
		      params->tagfilter->doctype_name,
		      get_major_type_name(params->body->TYPE),
		      get_subtype_name(params->body->TYPE));

	    free_pager_range(&title_range);  

	}
	    
    fail_doctype:
	DPRINT(Debug,20,(&Debug, 
			 "tagfilter_handle_doctype: FAIL current token len = %d, class = %d %s\n",
			 current_token_len,params->current_token->token_class,
			 token_class_string(params->current_token->token_class)));
	if (params->current_token->sbuffer) {
	    DEBUG_PRINT_STRING(Debug,20,
			       "tagfilter_handle_doctype: current token = ",
			       "tagfilter_handle_doctype: current token > ",
			       params->current_token->sbuffer);
	}
	
	pg_flags = pg_UNDERLINE;
	
	set_out_state_line_mode(params->state_out,pg_flags,
				inherit,0 );

	if (collected_tag) 
	    state_putstring(collected_tag,params->state_out);
	
	switch (truncated_tag) {
	case collect_nl_tag:
	    state_putc('\n',params->state_out);
	    
	    set_out_state_line_mode(params->state_out,pg_flags,
				    inherit,0 );
	    break;
	    
	case collect_nl_truncated_tag:
	    state_putc('\n',params->state_out);
	    
	    /* FALLTRU */
	case collect_truncated_tag:
	    
	    set_out_state_line_mode(params->state_out,pg_BOLD|pg_flags,
				    inherit,0 );
	    state_puts(" ... ",params->state_out);

	    
	    /* Does not change if not EOLN */
	    set_out_state_line_pg_flags(params->state_out,pg_flags);
	    break;
	case collect_full_tag:
	    break;
	}
			    
	while (current_token_len >= 0) {
	    
	    switch(params->current_token->token_class) {
		int was_end;
	    case tf_doctype_error:
	    case tf_doctype_start:
	    case tf_doctype_segment:
	    case tf_doctype_space:
	    case tf_doctype_item:
	    case tf_doctype_end:
		
		was_end = tf_doctype_end ==
		    params->current_token->token_class;
		
		tagfilter_dump_token(params,pg_flags);
	       		
		current_token_len =
		    get_new_tagfilter_token(params->current_token,
					    params->tagfilter,
					    params->state_in);
		
		if (was_end)
		    goto  exit_doctype;
		break;
	    default:
		goto  exit_doctype;
	    }
	}

    exit_doctype:

	/* Should not cause newline */
	if (pg_flags || inherit || inherit_pg_flags) {
	    
	    set_out_state_line_mode(params->state_out,inherit_pg_flags,
				    inherit,0 );
	}
	
    }
    
    if (collected_tag)
	free_string(& collected_tag);

	DPRINT(Debug,20,(&Debug, 
			 "tagfilter_handle_doctype: END current token len = %d, class = %d %s\n",
			 current_token_len,params->current_token->token_class,
			 token_class_string(params->current_token->token_class)));
	if (params->current_token->sbuffer) {
	    DEBUG_PRINT_STRING(Debug,20,
			       "tagfilter_handle_doctype: current token = ",
			       "tagfilter_handle_doctype: current token > ",
			       params->current_token->sbuffer);
	}

    
    return current_token_len;
}

/* Returns current_token_len -- */

static int tagfilter_handle_error P_((struct tagfilter_params *params,
				      int current_token_len));

static int tagfilter_handle_error(params,current_token_len)
     struct tagfilter_params * params;
     int current_token_len;
{
    int pg_flags = pg_UNDERLINE;
    struct pager_range *inherit;
    int  inherit_pg_flags ;
    
    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_error",
                   "Bad magic number (tagfilter_params)");

    if (TAGFILTER_SELECTION_magic != params->tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_error",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_error",
                   "Bad magic number (tagfilter_token)");

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_error: START current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_error: current token = ",
			   "tagfilter_handle_error: current token > ",
			   params->current_token->sbuffer);
    }
    
    inherit_pg_flags = tagfilter_inherited_pg_flags(params);
    inherit = tagfilter_inherited_pager_range(params);
        
    set_out_state_line_mode(params->state_out,pg_flags,
			    inherit,0 );
    
    while (current_token_len >= 0 &&
	   params->current_token->token_class < tf_body /* 0 */) {

	tagfilter_dump_token(params,pg_flags);
		
	current_token_len =
	    get_new_tagfilter_token(params->current_token,
				    params->tagfilter,
				    params->state_in);
    }

    /* Should not cause newline */
    if (pg_flags || inherit || inherit_pg_flags) {	
	set_out_state_line_mode(params->state_out,inherit_pg_flags,
				inherit,0 );
    }

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_error: END current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_error: current token = ",
			   "tagfilter_handle_error: current token > ",
			   params->current_token->sbuffer);
    }

    
    return current_token_len;
}

/* Returns current_token_len -- */

static int tagfilter_handle_body P_((struct tagfilter_params *params,
				      int current_token_len));

static int tagfilter_handle_body(params,current_token_len)
     struct tagfilter_params * params;
     int current_token_len;
{
    struct pager_range *inherit;
    int  inherit_pg_flags ;

    int text_visible = 1;

    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_body",
                   "Bad magic number");

    if (TAGFILTER_SELECTION_magic != params->tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_body",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_body",
                   "Bad magic number (tagfilter_token)");

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_body: START current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_body: current token = ",
			   "tagfilter_handle_body: current token > ",
			   params->current_token->sbuffer);
    }
    
    if (params->stack && params->stack_len > 0) {
	size_t idx =  params->stack_len-1;
	
	if (TAGFILTER_STACK_ITEM_magic != params->stack[idx]->magic)
	    mime_panic(__FILE__,__LINE__,"tagfilter_handle_body",
		       "Bad magic number ( tagfilter_stack_item)");
	
	text_visible = params->stack[idx]->text_visible;
    }

    /* body text with entities on body */

    inherit_pg_flags = tagfilter_inherited_pg_flags(params);
    inherit = tagfilter_inherited_pager_range(params);

    DPRINT(Debug,20,(&Debug,
		     "tagfilter_handle_body: Body pg_flags %d %s\n",
		     inherit_pg_flags,
		     give_pg_flags(inherit_pg_flags)));
    
    set_out_state_line_mode(params->state_out,inherit_pg_flags,
			    inherit,0 );
    
    while (current_token_len >= 0) {

	if (tf_body == params->current_token->token_class) {

	    if (params->current_token->sbuffer && text_visible)
		state_putstring(params->current_token->sbuffer,
				params->state_out);

	} else if (tf_double_smaller == params->current_token->token_class) {
	    /* text/enriched escaped < */

	    /* Note that sbuffer includes all charaters 
	       part of token -- not just token value
	    */

	    if (text_visible) 
		state_putc('<',params->state_out);

	} else if ( ts_init == params->current_token->tag_state &&
		    tf_entity_error == params->current_token->token_class ) {

	    if (text_visible) {
		
		set_out_state_line_mode(params->state_out,
					pg_UNDERLINE,
					inherit,0 );
		
		if (params->current_token->sbuffer)
		    state_putstring(params->current_token->sbuffer,
				    params->state_out);
		
		set_out_state_line_mode(params->state_out,
					inherit_pg_flags,
					inherit,0 );
	    }
	    
	} else if ( ts_init == params->current_token->tag_state &&
		    tf_entity == params->current_token->token_class ) {


	    if (params->current_token->sbuffer &&
		params->current_token->match_reference &&
		text_visible) {
		struct out_entity * oe =
		    out_entity_from_match(params->current_token->sbuffer,
					  params->current_token->
					  match_reference,
					  pg_REVERSE);
		
		state_putentity(oe,params->state_out);
		
		free_out_entity(&oe);
	    }
		
	} else if ( ts_init == params->current_token->tag_state &&
		    tf_numeric_entity == params->current_token->token_class) {

	    if (params->current_token->sbuffer &&
		text_visible) {
		struct out_entity * oe =
		    new_out_entity(params->current_token->sbuffer,
				   NULL,
				   params->current_token->
				   numeric_reference,
				   pg_REVERSE);

		state_putentity(oe,params->state_out);
		
		free_out_entity(&oe);
	    }
	    	    	    
	} else
	    break;

	if (params->current_token->error)
	    params->error = 1;
	
	if (params->current_token->have_nl && text_visible) {

	    state_putc('\n',params->state_out);
	    
	    /* Re-set settings after newline */
	    set_out_state_line_mode(params->state_out,
				    inherit_pg_flags,
				    inherit,0 );
	    
	}
	
	
	current_token_len =
	    get_new_tagfilter_token(params->current_token,
				    params->tagfilter,
				    params->state_in);		
    }

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_body: END current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_body: current token = ",
			   "tagfilter_handle_body: current token > ",
			   params->current_token->sbuffer);
    }

    
    return current_token_len;
}

/* Returns current_token_len -- */

static int tagfilter_handle_tag P_((struct tagfilter_params *params,
				      int current_token_len));
static int tagfilter_handle_tag(params,current_token_len)
     struct tagfilter_params * params;
     int current_token_len;
{
    int tag_ok  = 0;
    
    int tag_end_seen = 0;
    int is_start_tag = 0;  
    int is_end_tag   = 0;   
    int force_eoln = 0;

    struct string        * collected_tag  = NULL;
    enum collect_truncated truncated_tag  = collect_full_tag;

    struct pager_range * tag_range      = NULL;    
    int                  tag_pg_flags   = 0;       

    struct string      * tag_name       = NULL;
    struct tagfilter_tag_state * tag_state = NULL;
    size_t               stack_pos      = 0;

    struct  tagfilter_atr_state * attribute_state = NULL;
    
    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_tag",
                   "Bad magic number");

    stack_pos = params->stack_len;    /* Not on stack */
    
    if (TAGFILTER_SELECTION_magic != params->tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_tag",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_tag",
                   "Bad magic number (tagfilter_token)");

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_tag: START current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_tag: current token = ",
			   "tagfilter_handle_tag: current token > ",
			   params->current_token->sbuffer);
    }
    
    switch (params->current_token->token_class) {
    case tf_start_tag:
	is_start_tag = 1;
	
	if (0) {
	case tf_whole_tag:
	    tag_end_seen = 1;
	    is_start_tag = 1;
	}

	if (0) {
	case tf_selfclosed_tag:
	    tag_end_seen = 1;
	    is_start_tag = 1;
	    is_end_tag   = 1;
	}

	if (0) {
	case tf_start_endtag:
	    is_end_tag   = 1;
	}

	if (0) {
	case tf_whole_endtag:
	    tag_end_seen = 1;
	    is_end_tag   = 1;
	}

	if (params->current_token->tag_name) {
	    tag_name = dup_string(params->current_token->tag_name);


	    if (!  tag_end_seen) {
		/* Printed on error */
		tagfilter_collect_token(params,&collected_tag,&truncated_tag);
		
		current_token_len =
		    get_new_tagfilter_token(params->current_token,
					    params->tagfilter,
					    params->state_in);
	    }

	    break;
	}

	/* FALLTHRU */
    default:
	tag_pg_flags = tagfilter_inherited_pg_flags(params);
	tag_range    = tagfilter_inherited_pager_range(params);
	
	goto fail_tag;
    }

    if (is_start_tag) {
	struct tagfilter_stack_item * new_item = NULL;
	struct tagfilter_tags       * search_tags =
	    params->tagfilter->top_tags;
	int reset_range = 0;
	
	if (params->stack && params->stack_len > 0) {
	    size_t idx =  params->stack_len-1;

	    while (idx > 0 && ! params->stack[idx])
		idx--;
	    
	    if (params->stack[idx]) {
		
		if (TAGFILTER_STACK_ITEM_magic != params->stack[idx]->magic)
		    mime_panic(__FILE__,__LINE__,"tagfilter_handle_tag",
			       "Bad magic number ( tagfilter_stack_item)");

		search_tags = params->stack[idx]->nested_tags;
	    }
	}
	
	stack_pos = params->stack_len;
	
	tag_state =
	    params->tagfilter->
	    give_stack_item(&new_item,search_tags,
			    tag_name,
			    params->stack,params->stack_len,
			    &stack_pos,  /* should be params->stack_len */
			    params->body,
			    params->state_in,
			    params->state_out,
			    params->decode_opt,
			    params->text_charset,
			    params->tagfilter,
			    params->counter);

	if (tag_state) {

	    /* Potentially this replaces stack_pos with
	       new tag
	    */
	    
	    if (stack_pos < params->stack_len) {
		size_t i;

		/* Handle case where need close tags from stack */
		
		for (i = params->stack_len -1; i >= stack_pos; i--) {

		    if (params->stack[i]) {
			
			if (params->tagfilter->
			    close_stack_item(tag_state,
					     params->stack,params->stack_len,
					     i,
					     stack_enter_auto,
					     params->body,
					     params->state_in,
					     params->state_out,
					     params->text_charset,
					     params->tagfilter,
					     params->counter)) {
			    
			    tagfilter_free_stack_item(&
						      (params->stack[i]),
						      params->tagfilter);
			    
			    reset_range = 1;
			    
			    /* This left holes to stack if
			       all are not closed
			    */
			    
			    if (params->stack_len == i+1) {
				params->stack_len = i;
			    }
			}
		    }
		}		
	    }
	}
	
	if (new_item) {
	    
	    if (TAGFILTER_STACK_ITEM_magic != new_item->magic)
		mime_panic(__FILE__,__LINE__,"tagfilter_handle_tag",
			   "Bad magic number (tagfilter_stack_item)");
	    
	    params->stack = safe_array_realloc(params->stack,
					       params->stack_len+1,
					       sizeof (params->stack[0]));

	    params->stack[params->stack_len] = NULL;
	    
	    if (params->stack_len > stack_pos) {
		size_t i;
		
		for (i = params->stack_len; i > stack_pos; i--) {
		    params->stack[i] = params->stack[i-1];
		    params->stack[i-1] = NULL;
		}
	    }
	    params->stack_len++;

	    if (NULL != params->stack[stack_pos]) 
		mime_panic(__FILE__,__LINE__,"tagfilter_handle_tag",
			   "stack_pos not NULL");
								   	    
	    params->stack[stack_pos] = new_item;

	    force_eoln = new_item->force_eoln;
	    
	} else if (tag_state) {

	    /* Handle case where need push implicit tags to stack -
	       last item is current stack item
	     */
	    
	    struct tagfilter_stack_item * item2;
	    
	    while (NULL != (item2 =
			    params->tagfilter->
			    push_stack_item(tag_state,
					    params->stack,
					    params->stack_len,
					    params->body,
					    params->state_in,
					    params->state_out,
					    params->decode_opt,
					    params->text_charset,
					    params->tagfilter))) {
		
		if (TAGFILTER_STACK_ITEM_magic != item2->magic)
		    mime_panic(__FILE__,__LINE__,"tagfilter_handle_tag",
			       "Bad magic number ( tagfilter_stack_item)");
		
		params->stack = safe_array_realloc(params->stack,
						   params->stack_len+1,
						   sizeof (params->stack[0]));
		stack_pos = params->stack_len;
		params->stack[params->stack_len++] = item2;
		
		force_eoln = item2->force_eoln;

		if (item2->range) {
		    tag_pg_flags = tagfilter_inherited_pg_flags(params);
		    tag_range    = item2->range;
		    
		    set_out_state_line_mode(params->state_out,tag_pg_flags,
					    tag_range,force_eoln);

		    reset_range = 0;
		} else {
		    reset_range = 1;
		}
		
		new_item = item2;
	    }	
	}

	if (params->stack && params->stack_len > 0) {
	    size_t idx =  params->stack_len-1;

	    while (idx > 0 && ! params->stack[idx])
		idx--;
	    
	    if (params->stack[idx]) {
		
		if (TAGFILTER_STACK_ITEM_magic != params->stack[idx]->magic)
		    mime_panic(__FILE__,__LINE__,"tagfilter_handle_tag",
			       "Bad magic number ( tagfilter_stack_item)");
		force_eoln = params->stack[idx]->force_eoln;
	    }
	}
		
	if (!tag_range || reset_range) {
	    
	    tag_pg_flags = tagfilter_inherited_pg_flags(params);
	    tag_range    = tagfilter_inherited_pager_range(params);
	    	    
	    set_out_state_line_mode(params->state_out,tag_pg_flags,
				    tag_range,force_eoln);
	}
	
    } else if (is_end_tag) {  /* Not both is_start_tag and is_end_tag 
				 which is <tag/>
			       */
	int reset_range = 0;
	
	tag_state =
	    params->tagfilter->
	    locate_stack_item(tag_name,
			      params->stack,params->stack_len,
			      &stack_pos,
			      params->body,
			      params->state_in,
			      params->state_out,
			      params->text_charset,
			      params->tagfilter);

	if (tag_state) {
	    if (stack_pos+1 < params->stack_len) {
		size_t i;
		
		/* Handle case where need close tags from stack */
		
		for (i = params->stack_len -1; i > stack_pos; i--) {

		    if (params->stack[i]) {
		    
			if (params->tagfilter->
			    close_stack_item(tag_state,
					     params->stack,params->stack_len,
					     i,
					     stack_close_auto,
					     params->body,
					     params->state_in,
					     params->state_out,
					     params->text_charset,
					     params->tagfilter,
					     params->counter)) {
			    
			    tagfilter_free_stack_item(&
						      (params->stack[i]),
						      params->tagfilter);

			    reset_range = 1;
			    
			    /* This left holes to stack if
			       all are not closed
			    */
			    
			    if (params->stack_len == i+1) {
				params->stack_len = i;
			    }
			}
		    }
		}

	    }
	}

	if (!tag_range || reset_range) {
	    
	    tag_pg_flags = tagfilter_inherited_pg_flags(params);
	    tag_range    = tagfilter_inherited_pager_range(params);
	    
	    set_out_state_line_mode(params->state_out,tag_pg_flags,
				    tag_range,force_eoln);
	}
			      
    }


    while (current_token_len >= 0) {
	
	switch(params->current_token->token_class) {
	    
	case tf_tag_atrvalue_error:
	case tf_tag_param_error:
	case tf_tag_error:
	case tf_entity_error:

	    /* Fail, if start of stack is seen */
	    
	case tf_start_tag:

	    if (0) {
	    case tf_start_endtag:
		is_end_tag   = 1;
	    }
	   	    
	    goto fail_tag;

	case tf_tag_space:
	    if (attribute_state) {
		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_space calling ->end_attribute(..)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}
				
		params->tagfilter->end_attribute(&attribute_state,
						 tag_state,
						 params->body,
						 params->state_in,
						 params->state_out,
						 params->text_charset,
						 params->tagfilter);
	    } else {
		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_space skipping ->end_attribute(..)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

	    }
		
	    break;
	case tf_tag_atrname:
	    if (attribute_state)  { /* Should not happen */

		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrname calling ->end_attribute(..)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

		
		params->tagfilter->end_attribute(&attribute_state,
						 tag_state,
						 params->body,
						 params->state_in,
						 params->state_out,
						 params->text_charset,
						 params->tagfilter);
	    }
	    
	    if (tag_state && params->current_token->atr_name) {

		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrname calling ->start_attribute(..)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}
		
		attribute_state =
		    params->tagfilter->
		    start_attribute(params->current_token->atr_name,
				    tag_state,
				    params->body,
				    params->state_in,
				    params->state_out,
				    params->text_charset,
				    params->tagfilter);

		if (attribute_state) {
		    DPRINT(Debug,22,(&Debug, 
				     "tagfilter_handle_tag: tf_tag_atrname got attribute_state\n"));
		}
				
	    } else {
		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrname skipping ->start_attribute(..)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

	    }
	    
	    break;
	    
	case tf_tag_atrequal:
	   	    
	    break;
	case tf_tag_atrvalue_start:
	    if (tag_state && attribute_state) {

		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrvalue_start calling ->start_end_atrvalue(...atrvalue_start...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}
		
		params->tagfilter->
		    start_end_atrvalue(attribute_state,
				       atrvalue_start,
				       tag_state,
				       params->body,
				       params->state_in,
				       params->state_out,
				       params->text_charset,
				       params->tagfilter);

	    } else {

		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrvalue_start skipping ->start_end_atrvalue(...atrvalue_start...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

	    }
	    
	    break;
	    
	case tf_tag_atrvalue_end:
	    if (tag_state && attribute_state) {

		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrvalue_end calling ->start_end_atrvalue(...atrvalue_end...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}
		
		params->tagfilter->
		    start_end_atrvalue(attribute_state,
				       atrvalue_end,
				       tag_state,
				       params->body,
				       params->state_in,
				       params->state_out,
				       params->text_charset,
				       params->tagfilter);
	    } else {
		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrvalue_end skipping ->start_end_atrvalue(...atrvalue_end...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

	    }
	    break;
	case tf_tag_atrvalue_segment:
	    if (tag_state && attribute_state &&
		params->current_token->atr_value_segment) {
		
		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrvalue_segment calling ->atrvalue_seg_string(...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: atr_value_segment = ",
				       "tagfilter_handle_tag: atr_value_segment > ",
				       params->current_token->atr_value_segment);
		}
		
		params->tagfilter->
		    atrvalue_seg_string(attribute_state,
					params->current_token->
					atr_value_segment,
					tag_state,
					params->body,
					params->state_in,
					params->state_out,
					params->text_charset,
					params->tagfilter);
	    } else {

		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_tag_atrvalue_segment skipping ->atrvalue_seg_string(...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: atr_value_segment = ",
				       "tagfilter_handle_tag: atr_value_segment > ",
				       params->current_token->atr_value_segment);
		}

	    }
	    
	    break;

	case tf_entity:
	    if (tag_state && attribute_state &&
		params->current_token->sbuffer &&
		params->current_token->match_reference) {
		
		struct out_entity * oe =
		    out_entity_from_match(params->current_token->sbuffer,
					  params->current_token->
					  match_reference,
					  pg_REVERSE);

		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_entity calling ->atrvalue_seg_ent(...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}
		
		params->tagfilter->
		    atrvalue_seg_ent(attribute_state,oe,
				     tag_state,
				     params->body,
				     params->state_in,
				     params->state_out,
				     params->text_charset,
				     params->tagfilter);
		free_out_entity(&oe);		
	    } else {
		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_entity skipping ->atrvalue_seg_ent(...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

	    }
	    break;
	    
	case tf_numeric_entity:
	    if (tag_state && attribute_state &&
		params->current_token->sbuffer) {
		struct out_entity * oe =
		    new_out_entity(params->current_token->sbuffer,
				   NULL,
				   params->current_token->
				   numeric_reference,
				   pg_REVERSE);

		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_numeric_entity calling ->atrvalue_seg_ent(...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

		
		params->tagfilter->
		    atrvalue_seg_ent(attribute_state,oe,
				     tag_state,
				     params->body,
				     params->state_in,
				     params->state_out,
				     params->text_charset,
				     params->tagfilter);
		free_out_entity(&oe);		
	    } else {
		
		DPRINT(Debug,22,(&Debug, 
				 "tagfilter_handle_tag: tf_numeric_entity skipping ->atrvalue_seg_ent(...)\n"));
		if (params->current_token->sbuffer) {
		    DEBUG_PRINT_STRING(Debug,22,
				       "tagfilter_handle_tag: current token = ",
				       "tagfilter_handle_tag: current token > ",
				       params->current_token->sbuffer);
		}

	    }
	    break;

	case tf_whole_tag:
	case tf_tag_end:
	    tag_ok = 1;
	    tag_end_seen = 1;
	    break;

	    
	case tf_whole_endtag:
	case tf_tag_selfclosed_end:
	case tf_selfclosed_tag:
	    is_end_tag   = 1;
	    tag_ok = 1;
	    tag_end_seen = 1;
	    break;
	    
	default:
	    goto fail_tag;
	}

	tagfilter_collect_token(params,&collected_tag,&truncated_tag);

	current_token_len =
	    get_new_tagfilter_token(params->current_token,
				    params->tagfilter,
				    params->state_in);
	
	if (tag_end_seen)
	    goto exit_tag2;

    }
        
    if (!tag_ok) {
	int pg_flags;

	/* Error message? */

    fail_tag:

	DPRINT(Debug,20,(&Debug, 
			 "tagfilter_handle_tag: FAIL current token len = %d, class = %d %s\n",
			 current_token_len,params->current_token->token_class,
			 token_class_string(params->current_token->token_class)));
	if (params->current_token->sbuffer) {
	    DEBUG_PRINT_STRING(Debug,20,
			       "tagfilter_handle_tag: current token = ",
			       "tagfilter_handle_tag: current token > ",
			       params->current_token->sbuffer);
	}
	
	pg_flags = pg_UNDERLINE;
		
	set_out_state_line_mode(params->state_out,pg_flags,
				tag_range,0 );

	if (collected_tag) 
	    state_putstring(collected_tag,params->state_out);
	
	switch (truncated_tag) {
	case collect_nl_tag:
	    state_putc('\n',params->state_out);
	    
	    set_out_state_line_mode(params->state_out,pg_flags,
				    tag_range,0 );
	    break;
	    
	case collect_nl_truncated_tag:
	    state_putc('\n',params->state_out);
	    
	    /* FALLTRU */
	case collect_truncated_tag:
	    
	    set_out_state_line_mode(params->state_out,pg_BOLD|pg_flags,
				    tag_range,0 );
	    state_puts(" ... ",params->state_out);
	    
	    /* Does not change if not EOLN */
	    set_out_state_line_pg_flags(params->state_out,pg_flags);
	    break;
	case collect_full_tag:
	    break;
	}
	
	while (current_token_len >= 0) {

	    switch(params->current_token->token_class) {

	    case tf_tag_atrvalue_error:
	    case tf_tag_param_error:
	    case tf_tag_error:
	    case tf_entity_error:
	    case tf_start_tag:
	    case tf_whole_tag:
	    case tf_selfclosed_tag:
	    case tf_start_endtag:
	    case tf_whole_endtag:
	    case tf_entity:
	    case tf_numeric_entity:
	    case tf_tag_space:
	    case tf_tag_atrname:
	    case tf_tag_selfclosed_end:
	    case tf_tag_end:
	    case tf_tag_atrequal:
	    case tf_tag_atrvalue_start:
	    case tf_tag_atrvalue_segment:
	    case tf_tag_atrvalue_end:

		if (tf_whole_tag      == params->current_token->token_class ||
		    tf_selfclosed_tag == params->current_token->token_class ||
		    tf_whole_endtag   == params->current_token->token_class ||
		    tf_tag_selfclosed_end == params->current_token->token_class ||
		    tf_tag_end        == params->current_token->token_class)
		    tag_end_seen = 1;
		
		tagfilter_dump_token(params,pg_flags);
		
		current_token_len =
		    get_new_tagfilter_token(params->current_token,
					    params->tagfilter,
					    params->state_in);
		
		
		if (tag_end_seen)
		    goto exit_tag;

		break;
	    default:
		goto exit_tag;
	    }

	}

    exit_tag:
	
	/* Should not cause newline */
	if (pg_flags || tag_range || tag_pg_flags) {	    
	    set_out_state_line_mode(params->state_out,tag_pg_flags,
				    tag_range,0 );
	}
	
    }

 exit_tag2:
    if (! is_end_tag && tag_state) {
	if (TAGFILTER_TAG_STATE_magic != tag_state->magic)
	    mime_panic(__FILE__,__LINE__,"tagfilter_free_tag_state_common",
		   "Bad magic number (tagfilter_tag_state)");

	is_end_tag = tag_state->no_context;
    }
    
    
    if (is_end_tag && stack_pos < params->stack_len && tag_state) {

	if (params->stack[stack_pos]) {

	    if (params->tagfilter->
		close_stack_item(tag_state,
				 params->stack,params->stack_len,
				 stack_pos,
				 stack_pos +1 < params->stack_len ?
				 stack_close_middle : stack_close_end,
				 params->body,
				 params->state_in,
				 params->state_out,
				 params->text_charset,
				 params->tagfilter,
				 params->counter)) {

		tagfilter_free_stack_item(& (params->stack[stack_pos]),
					  params->tagfilter
					  );

		/* This left holes to stack if
		   all are not closed
		*/
			    
		if (params->stack_len == stack_pos+1) {
		    params->stack_len = stack_pos;
		}
	    }
	}	
    }

    if (attribute_state)  { /* Should not happen */

	DPRINT(Debug,22,(&Debug, 
			 "tagfilter_handle_tag: calling ->end_attribute(..)\n"));
	if (params->current_token->sbuffer) {
	    DEBUG_PRINT_STRING(Debug,22,
			       "tagfilter_handle_tag: current token = ",
			       "tagfilter_handle_tag: current token > ",
			       params->current_token->sbuffer);
	}
	
	params->tagfilter->end_attribute(&attribute_state,
					 tag_state,
					 params->body,
					 params->state_in,
					 params->state_out,
					 params->text_charset,
					 params->tagfilter);
    }
    
    if (tag_state) {
	
	DPRINT(Debug,22,(&Debug, 
			 "tagfilter_handle_tag: calling ->end_tag_state(..)\n"));
	if (params->current_token->sbuffer) {
	    DEBUG_PRINT_STRING(Debug,22,
			       "tagfilter_handle_tag: current token = ",
			       "tagfilter_handle_tag: current token > ",
			       params->current_token->sbuffer);
	}
	
	params->tagfilter->end_tag_state(&tag_state,
					 params->body,
					 params->state_in,
					 params->state_out,
					 params->text_charset,
					 params->tagfilter,
					 params->counter);

    }
	
    if (tag_name)
	free_string(& tag_name);
    
    if (collected_tag)
	free_string(& collected_tag);


    tag_pg_flags = tagfilter_inherited_pg_flags(params);
    tag_range    = tagfilter_inherited_pager_range(params);

    DPRINT(Debug,20,(&Debug,
		     "tagfilter_handle_tag: After tag pg_flags %d %s\n",
		     tag_pg_flags,
		     give_pg_flags(tag_pg_flags)));
    
    set_out_state_line_mode(params->state_out,tag_pg_flags,
			    tag_range,force_eoln);

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_tag: END current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_tag: current token = ",
			   "tagfilter_handle_tag: current token > ",
			   params->current_token->sbuffer);
    }

    
    return current_token_len;
}


/* Returns current_token_len -- */

static int tagfilter_handle_bcomment P_((struct tagfilter_params *params,
				      int current_token_len));

static int tagfilter_handle_bcomment(params,current_token_len)
     struct tagfilter_params * params;
     int current_token_len;
{
    struct string        * collected_tag  = NULL;
    enum collect_truncated truncated_tag  = collect_full_tag;
    int end_seen = 0;
    int inherit_pg_flags;
    struct pager_range *inherit;
    int idx = 0;

    /* Bocus comment */

    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_bcomment",
                   "Bad magic number");

    if (TAGFILTER_SELECTION_magic != params->tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_bcomment",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_bcomment",
                   "Bad magic number (tagfilter_token)");

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_bcomment: START current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_bcomment: current token = ",
			   "tagfilter_handle_bcomment: current token > ",
			   params->current_token->sbuffer);
    }

    
    inherit_pg_flags = tagfilter_inherited_pg_flags(params);
    inherit          = tagfilter_inherited_pager_range(params);
    
    while (current_token_len >= 0) {
	idx++;
	
	switch(params->current_token->token_class) {
	case tf_bcomment_error:
	    goto fail_bcomment;

	case tf_bcomment_start:
	    if (idx > 1)
		goto fail_bcomment;
	    /* FALLTRU */
	case tf_bcomment_chunk:

	    tagfilter_collect_token(params,&collected_tag,&truncated_tag);
	    
	    current_token_len =
		get_new_tagfilter_token(params->current_token,
					params->tagfilter,
					params->state_in);
	    break;
	    
	case tf_bcomment_end:
	    end_seen = 1;
	    tagfilter_collect_token(params,&collected_tag,&truncated_tag);

	current_token_len =
	    get_new_tagfilter_token(params->current_token,
				    params->tagfilter,
				    params->state_in);
	
	
	    goto exit2_bcomment;
	default:
	     goto fail_bcomment;
	}
    }

    if (!end_seen) {
	int pg_flags;

	/* Error message? */
    fail_bcomment:
	DPRINT(Debug,20,(&Debug, 
			 "tagfilter_handle_bcomment: FAIL current token len = %d, class = %d %s\n",
			 current_token_len,params->current_token->token_class,
			 token_class_string(params->current_token->token_class)));
	if (params->current_token->sbuffer) {
	    DEBUG_PRINT_STRING(Debug,20,
			       "tagfilter_handle_bcomment: current token = ",
			       "tagfilter_handle_bcomment: current token > ",
			   params->current_token->sbuffer);
	}

	pg_flags = pg_UNDERLINE;
	
	set_out_state_line_mode(params->state_out,pg_flags,
				inherit,0 );

	if (collected_tag) 
	    state_putstring(collected_tag,params->state_out);
	
	switch (truncated_tag) {
	case collect_nl_tag:
	    state_putc('\n',params->state_out);
	    
	    set_out_state_line_mode(params->state_out,pg_flags,
				    inherit,0 );
	    break;
	    
	case collect_nl_truncated_tag:
	    state_putc('\n',params->state_out);
	    
	    /* FALLTRU */
	case collect_truncated_tag:
	    
	    set_out_state_line_mode(params->state_out,pg_BOLD|pg_flags,
				    inherit,0 );
	    state_puts(" ... ",params->state_out);
	    	    
	    /* Does not change if not EOLN */
	    set_out_state_line_pg_flags(params->state_out,pg_flags);
	    break;
	case collect_full_tag:
	    break;
	}
	
	while (current_token_len >= 0) {

	    switch(params->current_token->token_class) {
	    case tf_bcomment_end:
		end_seen = 1;
		/* FALLTHRU */
	    case tf_bcomment_error:
	    case tf_bcomment_start:   /* <? or <!  or </        */
	    case tf_bcomment_chunk:   /* <? bogus comment chunk */
		
		tagfilter_dump_token(params,pg_flags);
		
		current_token_len =
		    get_new_tagfilter_token(params->current_token,
					    params->tagfilter,
					    params->state_in);
		
		
		if (end_seen)
		    goto exit_bcomment;

		break;
	    default:
		goto exit_bcomment;
	    }
	}

    exit_bcomment:
	/* Should not cause newline */
	if (pg_flags || inherit || inherit_pg_flags) {	    
	    set_out_state_line_mode(params->state_out,inherit_pg_flags,
				    inherit,0 );
	}
	
    }

 exit2_bcomment:
    if (collected_tag)
	free_string(& collected_tag);

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_bcomment: END current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_bcomment: current token = ",
			   "tagfilter_handle_bcomment: current token > ",
			   params->current_token->sbuffer);
    }

    return current_token_len;
}

static int tagfilter_handle_span_nl P_((struct tagfilter_params *params,
				      int current_token_len));

static int tagfilter_handle_span_nl(params,current_token_len)
     struct tagfilter_params * params;
     int current_token_len;
{
    /* text/enriched */

    int inherit_pg_flags;
    struct pager_range *inherit;

    int text_visible = 1;
    
    /* Bocus comment */

    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_span_nl",
                   "Bad magic number");

    if (TAGFILTER_SELECTION_magic != params->tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_span_nl",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_span_nl",
                   "Bad magic number (tagfilter_token)");

    if (params->stack && params->stack_len > 0) {
	size_t idx =  params->stack_len-1;
	
	if (TAGFILTER_STACK_ITEM_magic != params->stack[idx]->magic)
	    mime_panic(__FILE__,__LINE__,"tagfilter_handle_span_nl",
		       "Bad magic number ( tagfilter_stack_item)");
	
	text_visible = params->stack[idx]->text_visible;
    }
    
    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_span_nl: START current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_span_nl: current token = ",
			   "tagfilter_handle_span_nl: current token > ",
			   params->current_token->sbuffer);
    }
    
    inherit_pg_flags = tagfilter_inherited_pg_flags(params);
    inherit          = tagfilter_inherited_pager_range(params);

    /* Reset for newline chunk */

    set_out_state_line_pager_range(params->state_out,NULL);

    while (current_token_len >= 0) {

	if (tf_span_nl != params->current_token->token_class)
	    goto out;

	if (params->current_token->have_nl
	    && text_visible)   /* Should not be set */
	    state_putc('\n',params->state_out);

	if (params->current_token->sbuffer) { /* Should include only NL */

	    if (text_visible)
		state_putstring(params->current_token->sbuffer,
				params->state_out);
	}
	    
	current_token_len =
	    get_new_tagfilter_token(params->current_token,
				    params->tagfilter,
				    params->state_in);
    }

 out:

    if (inherit || inherit_pg_flags) {	
	set_out_state_line_mode(params->state_out,inherit_pg_flags,
				inherit,0 );
    }

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_span_nl: END current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_span_nl: current token = ",
			   "tagfilter_handle_span_nl: current token > ",
			   params->current_token->sbuffer);
    }
    
    return current_token_len;
}


/* Returns current_token_len -- */

static int tagfilter_handle_comment P_((struct tagfilter_params *params,
				      int current_token_len));

static int tagfilter_handle_comment(params,current_token_len)
     struct tagfilter_params * params;
     int current_token_len;
{
    struct string        * collected_tag  = NULL;
    enum collect_truncated truncated_tag  = collect_full_tag;
    int end_seen = 0;
    int inherit_pg_flags;
    struct pager_range *inherit;
    int idx = 0;
    
    /* Real comment */

    if (TAGFILTER_PARAMS_magic != params->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_comment",
                   "Bad magic number");

    if (TAGFILTER_SELECTION_magic != params->tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_comment",
		   "Bad magic number (tagfilter_selection)");

    if (TAGFILTER_TOKEN_magic != params->current_token->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_handle_comment",
                   "Bad magic number (tagfilter_token)");

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_comment: START current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_comment: current token = ",
			   "tagfilter_handle_comment: current token > ",
			   params->current_token->sbuffer);
    }
   
    inherit_pg_flags = tagfilter_inherited_pg_flags(params);
    inherit          = tagfilter_inherited_pager_range(params);
   
    while (current_token_len >= 0) {
	idx++;
	
	switch(params->current_token->token_class) {
	case tf_comment_error :
	    goto fail_comment;

	case tf_comment_start:
	    if (idx > 1)
		goto fail_comment;
	    /* FALLTRU */	    
	case tf_comment_chunk:
	    tagfilter_collect_token(params,&collected_tag,&truncated_tag);
	    
	    current_token_len =
		get_new_tagfilter_token(params->current_token,
					params->tagfilter,
					params->state_in);
	    break;

	    
	case tf_whole_comment:
	case tf_comment_end:
	    end_seen = 1;
	    tagfilter_collect_token(params,&collected_tag,&truncated_tag);
	    
	    current_token_len =
		get_new_tagfilter_token(params->current_token,
					params->tagfilter,
					params->state_in);
	    
	    goto exit2_comment;
	    
	default:
	    goto fail_comment;
	}
    }

    if (!end_seen) {
	int pg_flags;
	
	/* Error message? */
    fail_comment:
	DPRINT(Debug,20,(&Debug, 
			 "tagfilter_handle_comment: FAIL current token len = %d, class = %d %s\n",
			 current_token_len,params->current_token->token_class,
			 token_class_string(params->current_token->token_class)));
	if (params->current_token->sbuffer) {
	    DEBUG_PRINT_STRING(Debug,20,
			       "tagfilter_handle_comment: current token = ",
			       "tagfilter_handle_comment: current token > ",
			   params->current_token->sbuffer);
	}
	
	pg_flags = pg_UNDERLINE;
	
	set_out_state_line_mode(params->state_out,pg_flags,
				inherit,0 );

	if (collected_tag) 
	    state_putstring(collected_tag,params->state_out);
	
	switch (truncated_tag) {
	case collect_nl_tag:
	    state_putc('\n',params->state_out);
	    
	    set_out_state_line_mode(params->state_out,pg_flags,
				    inherit,0 );
	    break;
	    
	case collect_nl_truncated_tag:
	    state_putc('\n',params->state_out);
	    
	    /* FALLTRU */
	case collect_truncated_tag:
	    
	    set_out_state_line_mode(params->state_out,pg_BOLD|pg_flags,
				    inherit,0 );
	    state_puts(" ... ",params->state_out);
	    
	    /* Does not change if not EOLN */
	    set_out_state_line_pg_flags(params->state_out,pg_flags);
	    break;
	case collect_full_tag:
	    break;
	}

	while (current_token_len >= 0) {

	    switch(params->current_token->token_class) {
	    case tf_whole_comment:
	    case tf_comment_end:
		end_seen = 1;
		/* FALLTHRU */
	    case tf_comment_error:
	    case tf_comment_start:
	    case tf_comment_chunk:		
		tagfilter_dump_token(params,pg_flags);
		
		current_token_len =
		    get_new_tagfilter_token(params->current_token,
					    params->tagfilter,
					    params->state_in);
		
		
		if (end_seen)
		    goto exit_comment;
		
		break;
	    default:
		goto exit_comment;
	    }
	}

    exit_comment:
	/* Should not cause newline */
	if (pg_flags || inherit || inherit_pg_flags) {					   
	    set_out_state_line_mode(params->state_out,inherit_pg_flags,
				    inherit,0 );
	}
	
    }

 exit2_comment:
    if (collected_tag)
	free_string(& collected_tag);

    DPRINT(Debug,20,(&Debug, 
		     "tagfilter_handle_comment: END current token len = %d, class = %d %s\n",
		     current_token_len,params->current_token->token_class,
		     token_class_string(params->current_token->token_class)));
    if (params->current_token->sbuffer) {
	DEBUG_PRINT_STRING(Debug,20,
			   "tagfilter_handle_comment: current token = ",
			   "tagfilter_handle_comment: current token > ",
			   params->current_token->sbuffer);
    }
    
    return current_token_len;
}


void tagfilter_free_stack_item(item, tagfilter)
     struct tagfilter_stack_item **item;
     struct tagfilter_selection * tagfilter;
{    
    if (TAGFILTER_STACK_ITEM_magic != (*item)->magic)
		    mime_panic(__FILE__,__LINE__,"tagfilter_decoder",
			       "Bad magic number ( tagfilter_stack_item)");

    if (TAGFILTER_SELECTION_magic != tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_free_stack_item",
		   "Bad magic number (tagfilter_selection)");


    tagfilter->stack_item_clear(*item,tagfilter);

    if ((*item)->u.magic) {
	DPRINT(Debug,1,(&Debug,
			"tagfilter_free_stack_item: %p: ->stack_item_clear did not clear type specific data %p - magic %04x\n",
			*item,(*item)->u.magic,
			*((*item)->u.magic)));

	(*item)->u.magic  = 0; /* Invalidate */
	free ((*item)->u.magic);
	(*item)->u.magic = NULL;		
    }
    
    if ((*item)->range)
	free_pager_range(& ((*item)->range));

    if ((*item)->current_tag)
	tagfilter_release_cmd(& ((*item)->current_tag));
		
    (*item)->magic = 0;
    free(*item);
    (*item) = NULL;	    
}


void tagfilter_decoder(body,state_in,state_out,decode_opt,text_charset,tagfilter)
     mime_t                     * body;
     struct in_state            * state_in;
     struct out_state           * state_out;
     const struct decode_opts   * decode_opt;
     charset_t                    text_charset;
     struct tagfilter_selection * tagfilter;
{
    struct tagfilter_token current_token;
    int                    current_token_len = -1;  /* -1 on error or failure */

    struct tagfilter_params params;
    struct tagfilter_global counter;

    int errors2 UNUSED_VAROK;
    int errors3;
    int error = 0;

    
    struct tagfilter_stack_item * top_item;

    struct pager_range *filter_message_range = NULL;
    

    /* bzero is defined on hdrs/elm_defs.h */
    bzero(&counter,sizeof counter);
    counter = NULL_counter;
    
    if (MIME_magic != body->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_decoder",
		   "Bad magic number (mime_t)");

    if (TAGFILTER_SELECTION_magic != tagfilter->magic)
	mime_panic(__FILE__,__LINE__,"tagfilter_decoder",
		   "Bad magic number (tagfilter_selection)");

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)&params,sizeof params);
           
    tagfilter_token_init(&current_token,text_charset);

    params.magic         = TAGFILTER_PARAMS_magic;
    params.body          = body;
    params.state_in      = state_in;
    params.state_out     = state_out;
    params.decode_opt    = decode_opt;
    params.text_charset  = text_charset;
    params.tagfilter     = tagfilter;
    params.current_token = &current_token;
    params.error         = 0;
    params.stack         = NULL;
    params.stack_len     = 0;
    params.counter       = &counter;
    
    current_token_len = get_new_tagfilter_token(&current_token,
						tagfilter,state_in);

    if (current_token_len >= 0) {
	 struct pager_range *inherit
	     = tagfilter_inherited_pager_range(&params);

	 filter_message_range =
	     state_add_opcode_pager_range(state_out,
					  pr_opcode_pager_replace
					  /* Replace this on builtin 
					     pager 
					  */,
					  inherit,PR_MAX_WIDTH|PR_WORD_WRAP,
					  0,0);

	 /* \n resets this */
	 set_out_state_line_mode(state_out,pg_BOLD,
				 filter_message_range,1 /* Newline */);
	 
	 state_printf(state_out,
		      CATGETS(elm_msg_cat, MeSet, 
			      MeTagFUnsupTagsFiltered,
			      "[ %s/%s is unsupported, filtering tags. ]\n"),
		      get_major_type_name(body->TYPE),
		      get_subtype_name(body->TYPE));

    }

    
    top_item = tagfilter->give_top_item(body,state_in,state_out,
					decode_opt,
					text_charset,tagfilter);
    if (top_item) {
	params.stack = safe_malloc(sizeof (params.stack[0]));

	params.stack[0]  = top_item;
	params.stack_len = 1;

    }

    if (current_token_len >= 0 &&  tagfilter->doctype_name &&
	tf_doctype_start == current_token.token_class) {
	
	current_token_len =
	    tagfilter_handle_doctype(&params,current_token_len);

    }

    errors3 = print_in_errors(state_in,state_out,decode_opt);
    
    if (errors3 > 0) {
	
	DPRINT(Debug,10,(&Debug, 
			 "tagfilter_decoder: %d errors, current token len = %d, class = %d %s\n",
			 errors3,current_token_len,current_token.token_class,
			 token_class_string(current_token.token_class)));
	
	/* Retry after errors */
	if (current_token_len < 0)
	    current_token_len =
		get_new_tagfilter_token(&current_token,
					tagfilter,state_in);
    }	    
    	                  
    while (current_token_len >= 0) {
	int errors;
	
	DPRINT(Debug,15,(&Debug, 
			 "tagfilter_decoder: current token len = %d, class = %d %s\n",
			 current_token_len,current_token.token_class,
			 token_class_string(current_token.token_class)));
	if (current_token.sbuffer) {
	    DEBUG_PRINT_STRING(Debug,19,
			       "tagfilter_decoder: current token = ",
			       "tagfilter_decoder: current token > ",
			   current_token.sbuffer);
	}
	

	switch (current_token.token_class) {
	default: 
	    if (current_token.token_class < tf_body /* 0 */) {
	    case tf_doctype_error      /* -7 */:
	    case tf_tag_atrvalue_error /* -6 */:
	    case tf_tag_param_error    /* -5 */:
	    case tf_comment_error      /* -4 */:
	    case tf_bcomment_error     /* -3 */:
	    case tf_tag_error          /* -2 */:
	    case tf_entity_error       /* -1 */:
		
		current_token_len =
		    tagfilter_handle_error(&params,current_token_len);
		
	    } else {
		case tf_bcomment_chunk:
		case tf_bcomment_end:
		case tf_comment_chunk:
		case tf_comment_end:
		case tf_tag_space:
		case tf_tag_atrname:
		case tf_tag_selfclosed_end:
		case tf_tag_end:
		case tf_tag_atrequal:
		case tf_tag_atrvalue_start:
		case tf_tag_atrvalue_segment:
		case tf_tag_atrvalue_end:
		case tf_doctype_start:
		case tf_doctype_segment:
		case tf_doctype_space:
		case tf_doctype_end:
		case tf_doctype_item:
		    goto unhandled_token;
			
	    }
	    break;

	case tf_entity:
	case tf_numeric_entity:
	    if (ts_init != current_token.tag_state)
		goto unhandled_token;
	    /* FALLTHRU */
	case tf_double_smaller:
	    /* text/enriched */
	case tf_body:
	    current_token_len =
		tagfilter_handle_body(&params,current_token_len);
	    break;
	    
	case tf_start_tag:
	case tf_whole_tag:
	case tf_selfclosed_tag:
	case tf_start_endtag:
	case tf_whole_endtag:
	    current_token_len =
		tagfilter_handle_tag(&params,current_token_len);
	    break;

	case tf_bcomment_start:
	    /* Bocus comment */

	    current_token_len =
		tagfilter_handle_bcomment(&params,current_token_len);
	    break;

	case tf_span_nl:
	    /* text/enriched */
	    
	    current_token_len =
		tagfilter_handle_span_nl(&params,current_token_len);
	    break;

	case tf_comment_start:
	case tf_whole_comment:
	    /* Real comment */
	    
	    current_token_len =
		tagfilter_handle_comment(&params,current_token_len);
	    break;	    
	}
		   
	
	if (ison(current_token.tag_flags,TFLAG_unhandled_class)) {
	    int inherit_pg_flags;
	    struct pager_range * inherit;

	unhandled_token:
	    
	    inherit_pg_flags         = tagfilter_inherited_pg_flags(&params);
	    inherit = tagfilter_inherited_pager_range(&params);

	    
	    if (current_token.error)
		params.error = 1;
	    	    
	    set_out_state_line_mode(state_out,pg_UNDERLINE,inherit,0 );

	    /* unhandled -- dump it directly */
	    tagfilter_dump_token(&params,0);
	    	    
	    if (current_token.have_nl) 
		state_putc('\n',state_out);
	    
	    /* Should not cause newline */
	    set_out_state_line_mode(state_out,inherit_pg_flags,inherit,0 );
	    
	    current_token_len =
		get_new_tagfilter_token(&current_token,
					tagfilter,state_in);
	    
	} else {
	    setit(current_token.tag_flags,TFLAG_unhandled_class);
	}

	if (!error && params.error) {
	    int inherit_pg_flags         = tagfilter_inherited_pg_flags(&params);
	    struct pager_range *inherit  = tagfilter_inherited_pager_range(&params);
		
	    const char * text_charset_MIME_name = get_charset_MIME_name(text_charset);

	    struct pager_range *title_range = 
		state_add_simple_pager_range(state_out,inherit,PR_MAX_WIDTH,0,
					     0);
	    
	    /* \n resets this */
	    set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);

	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MeTagFErrorCharacterFlag,
				 "[ %/%s format have NUL characters or have some bytes, which can't be interpreted as characters of charset %s. ]\n"),
			 get_major_type_name(body->TYPE),
			 get_subtype_name(body->TYPE),
			 text_charset_MIME_name);

	    free_pager_range(&title_range);
	    error = 1;
	    params.error = 0;
	    
	    /* Should not cause newline */
	    set_out_state_line_mode(state_out,inherit_pg_flags,inherit,0 );
	}

	errors = print_in_errors(state_in,state_out,decode_opt);

	if (errors > 0) {
	    int inherit_pg_flags         = tagfilter_inherited_pg_flags(&params);
	    struct pager_range *inherit  = tagfilter_inherited_pager_range(&params);
	    
	     /* Should not cause newline */
	    set_out_state_line_mode(state_out,inherit_pg_flags,
				    inherit,0 );
	    
	    DPRINT(Debug,10,(&Debug, 
			     "tagfilter_decoder: %d errors, current token len = %d, class = %d %s\n",
			     errors,current_token_len,current_token.token_class,
			     token_class_string(current_token.token_class)));

	    
	    
	    /* Retry after errors */
	    if (current_token_len < 0)
		current_token_len =
		    get_new_tagfilter_token(&current_token,
					    tagfilter,state_in);
	}
    }

    if (current_token_len < 0) {

	if (in_state_ferror(state_in)) {
	    struct pager_range *inherit
		= tagfilter_inherited_pager_range(&params);
	    
	    struct pager_range *title_range = 
		state_add_simple_pager_range(state_out,inherit,PR_MAX_WIDTH,0,
					 0);

	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MeTagFErrorReadCont,
				 "[ Error when reading %s/%s content. ]\n"),
			 get_major_type_name(body->TYPE),
			 get_subtype_name(body->TYPE));

	    free_pager_range(&title_range);
	    error = 1;
	}

	if (! in_state_feof(state_in)) {
	    struct pager_range *inherit
		= tagfilter_inherited_pager_range(&params);
	    
	    struct pager_range *title_range = 
		state_add_simple_pager_range(state_out,inherit,PR_MAX_WIDTH,0,
					     0);

	    state_printf(state_out,
			 CATGETS(elm_msg_cat, MeSet, 
				 MeTagFIncomplteRead,
				 "[ All %s/%s content is not readed. ]\n"),
			 get_major_type_name(body->TYPE),
			 get_subtype_name(body->TYPE));

	    free_pager_range(&title_range);
	}
    }

    if (!error && params.error) {
	const char * text_charset_MIME_name = get_charset_MIME_name(text_charset);
	struct pager_range *inherit
	    = tagfilter_inherited_pager_range(&params);
	
	struct pager_range *title_range = 
	    state_add_simple_pager_range(state_out,inherit,PR_MAX_WIDTH,0,
					 0);
			
	/* \n resets this */
	set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
	
	state_printf(state_out,
		     CATGETS(elm_msg_cat, MeSet, 
			     MeTagFErrorCharacterFlag,
			     "[ %/%s format have NUL characters or have some bytes, which can't be interpreted as characters of charset %s. ]\n"),
		     get_major_type_name(body->TYPE),
		     get_subtype_name(body->TYPE),
		     text_charset_MIME_name);
	
	free_pager_range(&title_range);
	error = 1;
	params.error = 0;
    }

    if (params.stack) {
	size_t i;

	for (i = 0; i < params.stack_len; i++) {
	    if (params.stack[i])
		tagfilter_free_stack_item(& (params.stack[i]),
					  tagfilter);

	}
	free(params.stack);
	params.stack = NULL;
    }
    params.stack_len     = 0;
    
    tagfilter_token_dest(&current_token);

    if (counter.unsupported_tags) {
	struct string ** list =
	    tagfilter_list_tags(counter.unsupported_tags);
	
	if (list) {

	    size_t i;

	    struct string * filter_list
		= new_string(display_charset);

		for (i = 0; list[i]; i++) {
		    
		    if (i > 0) {
			if (counter.have_other || list[i+1]) {
			    elm_append_message(&filter_list,
					       CATGETS(elm_msg_cat, MeSet, 
						       MeTagFiltListComma,
						       ", "));
			    
			} else if (1 == i) {
			    elm_append_message(&filter_list,
					       CATGETS(elm_msg_cat, MeSet, 
						       MeTagFiltListAnd,
						       " and "));
			} else {
			    elm_append_message(&filter_list,
					       CATGETS(elm_msg_cat, MeSet, 
						       MeTagFiltListComAnd,
						       ", and "));
			}
		    }
		    
		    elm_append_message(&filter_list,
				       CATGETS(elm_msg_cat, MeSet, 
					       MeTagFiltListTag,
					       "<%S>"),
				       list[i]);
		    free_string(& list[i]);
		}
		
		if (counter.have_other) {
		    elm_append_message(&filter_list,
				       CATGETS(elm_msg_cat, MeSet, 
					       MeTagFiltListCAndOther,
					       ", and other"));
		}
		
		if (i > 1 || counter.have_other) {
		    elm_append_message(&filter_list,
				       CATGETS(elm_msg_cat, MeSet, 
					       MeTagFiltListTagsUnsupported,
					       " tags are unsupported."));
		} else {
		    elm_append_message(&filter_list,
				       CATGETS(elm_msg_cat, MeSet, 
					       MeTagFiltListTagUnsupported,
					       " tag is unsupported."));
		}
	    

	    
	    if (filter_message_range) {

		struct string * text =
		    format_string(CATGETS(elm_msg_cat, MeSet, 
					  MeTagFUnsupTagsFilterList,
					  "[ %s/%s is unsupported, filtering tags: %S ]\n"),
				  get_major_type_name(body->TYPE),
				  get_subtype_name(body->TYPE),
				  filter_list);

		state_pager_add_param_text(state_out,filter_message_range,
					   pp_opcode_builtin /* show in builtin
								pager */,
					   text);

		free_string(&text);
		
	    } else {

		/* Should not needed */
		
		struct pager_range *inherit
		    = tagfilter_inherited_pager_range(&params);
		
		struct pager_range *title_range = 
		    state_add_simple_pager_range(state_out,inherit,
						 PR_MAX_WIDTH|PR_WORD_WRAP,
						 0,0);
		
		/* \n resets this */
		set_out_state_line_mode(state_out,pg_BOLD,title_range,1 /* Newline */);
		
		state_printf(state_out,
			     CATGETS(elm_msg_cat, MeSet, 
				     MeTagFilteredTags,
				     "[ Tags filtered from %s/%s format: %S ]\n"),
			     get_major_type_name(body->TYPE),
			     get_subtype_name(body->TYPE),
			     filter_list);
		
		free_pager_range(&title_range);	    
	    }

	    free_string(&filter_list);
	    
	    free(list);
	} 
    } else {
	DPRINT(Debug,10,(&Debug, 
			 "tagfilter_decoder: No unsupported tags\n"));
    }
    
    tagfilter_global_dest(&counter);
    
    errors2 = print_in_errors(state_in,state_out,decode_opt);

    if (filter_message_range)
	free_pager_range(& filter_message_range);
    
    DPRINT(Debug,10,(&Debug, 
		     "tagfilter_decoder: %d errors on end\n",errors2));
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
