static char rcsid[] = "@(#)$Id: smtp.c,v 2.50 2022/08/04 05:53:03 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.50 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "elmsmtp.h"

#include "../../hdrs/s_elm.h"

#ifdef I_SYSEXITS
#include <sysexits.h>
#endif

DEBUG_VAR(Debug,__FILE__,"smtp");

#ifdef REMOTE_MBX

static const int cancel_delay_msec = 500; /* 500 milliseconds: 1/2 seconds */

#if ANSI_C
#define S_(x) static x;
#define E_(x) extern x;
#else
#define S_(x)
#define E_(x)
#endif

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

enum server_s { Submission_s,
		SMTP_s ,
		submission_s,
		
		NUM_s
};

static const struct string * Server_s P_((enum server_s idx));

static int smtp_idle_alive_interval = 240;

#define MIN_SMTP_TIMEOUT 5



/* callbacks for connection ----------------------------------------------- */

S_(f_smtp_Server smtp_Server)
static const struct string *smtp_Server  P_((struct smtp_info *I));

S_(ss_action_routine smtp_read_action)
static int smtp_read_action P_((struct streamsched              * ss,
				union ss_action_routine_data      data,
				const struct schedule_timelimit * now));
static int smtp_read_action(ss,data,now) 
     struct streamsched              * ss; 
     union ss_action_routine_data      data;
     const struct schedule_timelimit * now;
{
    int n;
    struct smtp_info *I = data.smtp;
    int linelen;
    int close_it = 0;

    char * ERROR = NULL;
   
    if (!I)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_read_action",
	      "NULL data argument !!!",0);	
    
    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_read_action",
	      "Bad magic number",0);

    if (I->RA.stream != ss) {
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_read_action",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug,
		     "smtp_read_action: mailer info, I=%p\n",I));

    if (smtp_error == I->state) {
        DPRINT(Debug,13,(&Debug,
                         "smtp_read_action=0 (stream need closing)\n"));
        return 0;
    }

    n = ReadFromStream(ss, &(I->read_buffer),-1);

    if (n < 0) {      
	ERROR = RemoveStreamError(ss);
	DPRINT(Debug,13,(&Debug,
			 "smtp_read_stream: read error %s\n",
			 ERROR ? ERROR : "(none)"));
    }

    DPRINT(Debug,50,(&Debug,
		     "Buffer content="));
#ifdef DEBUG
    if (Debug.active >= 50) {
	int i;

	for (i = 0; i < I->read_buffer.read_len; i++) {
	    DPRINT(Debug,50,(&Debug,
			     " %d",
			     I->read_buffer.read_buffer[i]));
	}
	DPRINT(Debug,50,(&Debug,
			 "\n (message?)="));
	for (i = 0; i < I->read_buffer.read_len; i++) {
	    DPRINT(Debug,50,(&Debug,
			     " %c",
			     isascii(I->read_buffer.read_buffer[i]) &&
			     isprint(I->read_buffer.read_buffer[i]) ? 
			     I->read_buffer.read_buffer[i] :
			     '.'));
	}
    }
#endif
    DPRINT(Debug,50,(&Debug,
		     "\n"));

    while (0 < (linelen = find_crlf(&(I->read_buffer),1))) {
	char code[3];
	int final = 1;
	int i;
	char * message = NULL;
	enum smtp_state old_state = I->state;  /* Handle CLOSING case */

	DPRINT(Debug,13,(&Debug,
                         "smtp_read_action: Parsing response line (len =%d)\n",
			 linelen));

	for (i = 0; i < linelen && i < 3; i++)
	    code[i] = I->read_buffer.read_buffer[i];

	if (i < 3) {
	    int z UNUSED_VAROK = i;

	    for (; i < 3; i++)
		code[i] = '\0';
	
	    DPRINT(Debug,1,(&Debug,
			     "smtp_read_action: Line not include full repsonse code: %.3s (only %d chars)\n",
			    code,z));
	}

	if (linelen > 3)
	    final = I->read_buffer.read_buffer[3] != '-';

	if (linelen > 4)
	    message = & (I->read_buffer.read_buffer[4]);

	DPRINT(Debug,14,(&Debug,
                         "smtp_read_action: Parsing response line, code=%.3s final=%d message=%s\n",
			 code,final,
			 message ? message : "<NULL>"));


	if (0 == memcmp(code,"421",3)) {       /* CLOSING */
	    I->state = smtp_closing;
	    if (message)
		lib_transient(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpServerMsg,
				      "%S server: %s"),
			      smtp_Server(I),
			      message);
	    else
		lib_transient(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpServerClosing,
				      "%S server is closing"),
			      smtp_Server(I));
	} 

	/* Parse response ... */
	if (smtp_wait_reply       == old_state ||
	    smtp_reply_incomplete == I->state) {

	    DPRINT(Debug,13,(&Debug,
			    "smtp_read_action: Waited response\n"));

	    /* Clear stalled message */
	    if (I->stalled && I->RA.host) {
		lib_transient(CATGETS(elm_msg_cat, SmtpSet,SmtpResumed,
				      "%S connection to %s resumed."),
			      smtp_Server(I),
			      I->RA.host);
	    }
	    I->stalled = 0;
	    
	    if (smtp_wait_reply       == old_state) {
		if (smtp_wait_reply   == I->state)
		    I->state = smtp_reply_incomplete;
		memcpy(I->response_code,code,3);
		
		if (code[0] < '2' || code[0] > '5') {
		    I->state = smtp_error;
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpBadResponse,
				      "Bad response from %S server: %s"),
			      Server_s((I->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
			      I->read_buffer.read_buffer);
		} 
	    }

	    
	    if (smtp_reply_incomplete == I->state) {
		if (0 != memcmp(I->response_code,code,3)) {
		    I->state = smtp_error;
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpBadResponse,
				      "Bad response from %S server: %s"),
			      Server_s((I->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
			      I->read_buffer.read_buffer);
		}

		if (final) {
		    if ('3' == code[0])
			I->state = smtp_need_data;
		    else
			I->state = smtp_reply_read;

		    if (I->command[0]) {
			DPRINT(Debug,13,(&Debug,
					 "smtp_read_action: command was %.*s\n",
					 sizeof I->command,
					 I->command));				       
		    }
		    
		    DPRINT(Debug,13,(&Debug,
				     "smtp_read_action: %sresponse=%.3s %s\n",
				     smtp_reply_read == I->state ?
				     "command ready, " : "",
				     I->response_code,
				     message ? message : "(no message text)"));
		}
	    }
	    
	    if (message) {
		I->response_text = safe_array_realloc(I->response_text,
						      (I->response_text_len+1),
						      sizeof (*(I->response_text)));
		I->response_text[I->response_text_len++] = 
		    safe_strdup(message);
	    }
	    
	    cut_line(&(I->read_buffer),linelen);
	    message = NULL;          /* No longer valid */
	} else {
	    DPRINT(Debug,16,(&Debug,
			     "smtp_read_action: Quiting parsing loop (%d bytes left, unaccepted lines was %d bytes)\n",
			     I->read_buffer.read_len,
			     linelen));
	    break;
	}
    }

    if (I->read_buffer.read_len) {
	switch(I->state) {
	    int i;
	case smtp_reply_read:
	case smtp_need_data:
	case smtp_data:
	case smtp_idle:
	case smtp_command:
	    DPRINT(Debug,13,(&Debug,
			     "smtp_read_action: %d bytes data left! (bad state):\n",
			     I->read_buffer.read_len));
	    DPRINT(Debug,13,(&Debug,
			     "smtp_read_action: extra data="));
	    for (i =0; i < I->read_buffer.read_len; i++) {
		DPRINT(Debug,13,(&Debug,
				 " %d",
				 I->read_buffer.read_buffer[i]));
	    }
	    DPRINT(Debug,13,(&Debug,
			     "\nsmtp_read_action (message?)="));
	    for (i =0; i < I->read_buffer.read_len; i++) {
		DPRINT(Debug,13,(&Debug,
				 "%c",
				 isascii(I->read_buffer.read_buffer[i]) &&
				 isprint(I->read_buffer.read_buffer[i]) ?
				 I->read_buffer.read_buffer[i] :
				 '.'));
	    }
	    DPRINT(Debug,13,(&Debug,
			     "\n"));
	    
	    lib_error(CATGETS(elm_msg_cat, SmtpSet,
			      SmtpUnexpectedResponse,
			      "Unexpected response from %S server"),
		      Server_s((I->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));
	    close_it = 1;
	    break;

	default:
	    break;
	}
    }

    if (n == 0) {
	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpConnectionClosed,
			  "Connection closed to %S server"),
		  Server_s((I->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));
	close_it = 1;
    }
    if (n < 0 && ERROR) {
	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpErrorReading,
			  "Error reading from %S server: %s"),
		  Server_s((I->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
		  ERROR);
	close_it = 1;
    }
    if (ERROR) {
	free(ERROR);
	ERROR = NULL;
    }
    if (close_it) {
	I->state = smtp_error;
	DPRINT(Debug,13,(&Debug,
			 "smtp_read_action=0 (stream need closing)\n"));
	return 0;
    }
    DPRINT(Debug,13,(&Debug,
		     "smtp_read_action=1\n"));

    return 1;
}

S_(ss_action_routine smtp_write_action)
static int smtp_write_action P_((struct streamsched              * ss,
				 union ss_action_routine_data      data,
				 const struct schedule_timelimit * now));
static int smtp_write_action(ss,data,now) 
     struct streamsched              * ss; 
     union ss_action_routine_data      data;
     const struct schedule_timelimit * now;
{
    char * ERROR = NULL;
    struct smtp_info *I = data.smtp;
    int n;

    if (!I)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_write_action",
	      "NULL data argument !!!",0);	

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_write_action",
	      "Bad magic number",0);

    if (I->RA.stream != ss) {
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_write_action",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug,
		     "smtp_write_action: mailer info, I=%p\n",I));

    if (!I->write_buffer.write_len) {
	DPRINT(Debug,13,(&Debug,
			 "smtp_write_action: NO DATA TO WRITE!\n"));
    }

    if (smtp_error   == I->state ||
	smtp_closing == I->state) {
	DPRINT(Debug,13,(&Debug,
			 "smtp_write_action=0 (stream need closing)\n"));
	return 0;
    }

    if (smtp_command != I->state &&
	smtp_data    != I->state) {

	DPRINT(Debug,1,(&Debug,
			"smtp_write_action: state=%d\n",
			I->state));

	panic("MBX PANIC",__FILE__,__LINE__,"smtp_write_action",
	      "Bad state !!!",0);
    }

    n = WriteToStream(ss, & (I->write_buffer));

    if (n > 0) {
	int i;
	
	DPRINT(Debug,70,(&Debug,
			 "Write content="));
	for (i = 0; i < n; i++) {
	    DPRINT(Debug,70,(&Debug," %d",
			     I->write_buffer.write_buffer[i]));
	}
	DPRINT(Debug,70,(&Debug,"\n    (text?)="));
	for (i = 0; i < n; i++) {
	    DPRINT(Debug,70,(&Debug," %c",
			     isascii(I->write_buffer.write_buffer[i]) 
			     &&
			     isprint(I->write_buffer.write_buffer[i]) 
			     ? 
			     I->write_buffer.write_buffer[i] :
			     '.'));
	}
	DPRINT(Debug,70,(&Debug,"\n"));
	
	cut_Write_Buffer(& (I->write_buffer),n);

    } else if (n < 0) {
	ERROR = RemoveStreamError(ss);
	
	DPRINT(Debug,13,(&Debug,
			 "smtp_write_action: write error %s\n",
			 ERROR ? ERROR : "(none)"));
    }

    if (!I->write_buffer.write_len) {
	if (smtp_command == I->state)
	    I->state = smtp_wait_reply;
	else
	    I->state = smtp_need_data;

	if (ERROR) {
	    free(ERROR);
	    ERROR = NULL;
	}
	       
	DPRINT(Debug,13,(&Debug,
			 "smtp_write_action=0 -- data written\n"));
	return 0;
    }
    
    if (ERROR) {
	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpErrorWriting,
			  "Error writing to %S server: %s"),
		  Server_s((I->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
		  ERROR);
	I->state = smtp_error;
	DPRINT(Debug,13,(&Debug,
			 "smtp_write_action=0 (stream need closing)\n"));

	free(ERROR);
	ERROR = NULL;
        return 0;	
    }

    DPRINT(Debug,13,(&Debug,
		     "smtp_write_action=1\n"));
    return 1;

}

S_(f_smtp_start_command smtp_start_command)
static int smtp_start_command P_((struct smtp_info *I, char *command));

S_(f_smtp_command_ready smtp_command_ready)
static void smtp_command_ready P_((struct smtp_info *I, int data_end));

S_(f_smtp_command_reset smtp_command_reset)
static void smtp_command_reset P_((struct smtp_info *I));


S_(ss_badpid_routine smtp_badpid_action)
static enum badpid_status smtp_badpid_action
  P_((struct streamsched *ss,
      union  ss_action_routine_data data,
      int badpid));
static enum badpid_status smtp_badpid_action(ss,data,badpid)
     struct streamsched *ss;
     union  ss_action_routine_data data;
     int badpid;
{
    struct smtp_info *I = data.smtp;

    if (!I)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_badpid_action",
	      "NULL data argument !!!",0);	

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_badpid_action",
	      "Bad magic number",0);

    if (I->RA.stream != ss) {
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_badpid_action",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug,
		     "smtp_badpid_action: mailer info, I=%p\n",I));


    I->state = smtp_error;
    I->seen_badpid = badpid;
    

    return badpid_remove       /* remove action */;
}

S_(ss_action_routine smtp_timeout_action)
static int smtp_timeout_action P_((struct streamsched              * ss,
				   union ss_action_routine_data      data,
				   const struct schedule_timelimit * now));
static int smtp_timeout_action(ss,data,now) 
     struct streamsched              * ss; 
     union ss_action_routine_data      data;
     const struct schedule_timelimit * now;
{
    struct smtp_info *I = data.smtp;
    int status;

    if (!I)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_timeout_action",
	      "NULL data argument !!!",0);	

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_timeout_action",
	      "Bad magic number",0);

    if (I->RA.stream != ss) {
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_timeout_action",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug,
		     "smtp_timeout_action: mailer info, I=%p\n",I));

    if (smtp_error   == I->state ||
	smtp_closing == I->state) {
	DPRINT(Debug,13,(&Debug,
			 "smtp_timeout_action=0 (stream need closing)\n"));
	return 0;
    }

    if (smtp_reply_read == I->state &&
	0 == strncmp(I->command,"NOOP",
		     sizeof I->command)) {
	
	DPRINT(Debug,13,(&Debug,
			 "smtp_timeout_action: Previous idle command was finished.\n"));
		
	smtp_command_reset(I);
    } else if (smtp_idle != I->state) {

	if (0 == strncmp(I->command,"NOOP",
			 sizeof I->command)) {
	    struct sec_ms interval;
	    int r;

	    if (0 != (r = schedule_time_passed(& (I->NOOP_start),
					       now,
					       &interval))) {

		/* 'now' is time when wait ended, probably not current time */
		
		DPRINT(Debug,13,(&Debug, 
				 "smtp_timeout_action: previous NOOP command executed for %d sec %d msec.\n",
				 interval.timeout_sec, interval.timeout_ms));

		if (I->RA.host) {
		    I->stalled = 1;

                   if (interval.timeout_sec >= 0 &&
                       interval.timeout_sec < 10 &&
                       interval.timeout_ms  <= 999 &&
                       0 != (r & SCHEDULE_TP_HAVE_MSEC)) {
		       
                       lib_transient(CATGETS(elm_msg_cat, SmtpSet,SmtpStalledMS,
                                             "%S connection to %s stalled for %d.%03d seconds."),
				      smtp_Server(I),
				     I->RA.host,
                                     interval.timeout_sec,
                                     interval.timeout_ms);
                   } else if (interval.timeout_sec >= 0) {
                       lib_transient(CATGETS(elm_msg_cat, SmtpSet,SmtpStalledSec,
                                             "%S connection to %s stalled for %d seconds."),
				     smtp_Server(I),
				     I->RA.host,
				     interval.timeout_sec);		       
		   }
		}
	    }
	}
	
	DPRINT(Debug,13,(&Debug,
			 "smtp_timeout_action=1 (stream not idle:  command %.*s)\n",
			 sizeof I->command,
			 I->command));
	return 1;
    }

    status = smtp_start_command(I,"NOOP");

    if (status) 
	smtp_command_ready(I,0);
    /* DO not wait completion of command */
    
    I->NOOP_start = *now;
    
    DPRINT(Debug,13,(&Debug,
                     "smtp_timeout_action=%d\n",
                     status));

    return status;
}



static void free_response_text P_((struct smtp_info *I));
static void free_response_text(I)
     struct smtp_info *I;
{
    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"free_response_text",
	      "Bad magic number",0);

    if (I->response_text) {
	int i;
	for (i = 0; i < I->response_text_len; i++) {
	    if (I->response_text[i]) {
		free(I->response_text[i]);
		I->response_text[i] = NULL;
	    }
	}
	free(I->response_text);
    }            
    I->response_text     = NULL;
    I->response_text_len = 0;
}

static int smtp_start_command_tail P_((struct smtp_info *I,
				       char *command));
static int smtp_start_command_tail(I,command)
     struct smtp_info *I; 
     char *command;
{
    int status = 0;

    if (!I->RA.stream) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command_tail: stream not open\n"));
	status = 0;
	goto clean;
    }
    
    if (I->incomplete_command)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_start_command_tail",
	      "incomplete_command not NUL !",0);
    I->incomplete_command_len = 0;

    I->state = smtp_command;

    DPRINT(Debug,15,(&Debug,
		     "smtp_start_command_tail: Command %.4s...\n",command));

    I->incomplete_command     = safe_strdup(command);
    I->incomplete_command_len = strlen(command);

    strncpy(I->command,command,
	    sizeof I->command);
    
    status = 1;

 clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_start_command_tail=%d\n",status));
    return status;
}

static int smtp_start_command_afterwait P_((struct smtp_info *I));
static int smtp_start_command_afterwait(I)
     struct smtp_info *I;
{
    int status = 0;

    /* DO NOT accpet smtp_need_data here -- it means that
       data (not command) is expected
    */
    if (smtp_error == I->state ||
	smtp_closing == I->state) {
	
	if (I->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_start_command_afterwait: Closing stream"));
	    if (I->seen_badpid) {
		DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 I->seen_badpid));
	    }
	    DPRINT(Debug,12,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(I->RA.stream),
			     I->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}	
	status = 0;
	goto clean;
    }

    if (0 != I->write_buffer.write_len) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command_afterwait: write buffer not empty (%d bytes left)\n",
			 I->write_buffer.write_len));
	status = 0;
	goto clean;
    }       	
    
    free_response_text(I);
    
    status = 1;

clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_start_command_afterwait=%d\n",status));
    return status;
}

static int smtp_start_command_init P_((struct smtp_info *I));
static int smtp_start_command_init(I)
     struct smtp_info *I;
{
    int status = 0;

    if (smtp_error == I->state   ||
	smtp_closing == I->state ||
	!I->RA.stream) {
	if (I->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_start_command_init: Closing stream"));
	    if (I->seen_badpid) {
		DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 I->seen_badpid));
	    }
	    DPRINT(Debug,12,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(I->RA.stream),
			     I->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}	
	status = 0;
	goto clean;
    }
    
    status = 1;

 clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_start_command_init=%d\n",status));
    return status;
}   

static int smtp_start_command(I,command)
     struct smtp_info *I; 
     char *command;
{
    int status = 0;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_start_command",
	      "Bad magic number",0);
    
    DPRINT(Debug,12,(&Debug,
		     "smtp_start_command: mailer info, I=%p\n",
		     I));
    
    if (!smtp_start_command_init(I)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command: failed on init\n"));
	status = 0;
	goto clean;
    }

    if (I->state != smtp_idle) {
	DPRINT(Debug,7,(&Debug,
			"smtp_start_command: wrong state -- perhaps idle timer -- waiting...\n"));
	
	while (smtp_reply_read != I->state &&
	       smtp_need_data    != I->state &&
	       smtp_error        != I->state &&
	       smtp_closing      != I->state &&
	       NULL         != I->RA.stream) {
	    WaitStreamFor(I->RA.stream,SS_read_act);
	    /* Should user to have some way to interrupt progress ? 
	       cancelable version is smtp_start_command_c()
	    */
	}
	DPRINT(Debug,7,(&Debug,
			 "smtp_start_command: ... wait ended\n"));

	if (!smtp_start_command_afterwait(I)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_start_command: failed after wait\n"));
	    status = 0;
	    goto clean;
	}
    }

    if (!smtp_start_command_tail(I,command)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command: failure\n"));
	status = 0;
	goto clean;
    }

    status = 1;

 clean:
    DPRINT(Debug,12,(&Debug,
		     "smtp_start_command=%d\n",
		     status));

    return status;
}

static void smtp_cancel_reset_stream P_((struct smtp_info *I));
static void smtp_cancel_reset_stream(I)
     struct smtp_info *I;
{
    if ((smtp_error != I->state &&
	 smtp_closing != I->state) 
	||
	I->RA.stream) {

	DPRINT(Debug,12,(&Debug,
			 "smtp_cancel_reset_stream: Stream no longer usable because of cancel\n"));

	if (I->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_cancel_reset_stream: Closing stream"));
	     if (I->seen_badpid) {
		DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 I->seen_badpid));
	    }
	     DPRINT(Debug,12,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(I->RA.stream),
			     I->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}
	
	if (smtp_error != I->state &&
	    smtp_closing != I->state) {
	    I->state = smtp_error;

	    DPRINT(Debug,12,(&Debug,
			     "smtp_cancel_reset_stream: Stream on error state\n"));
	}	
    }
}

static int smtp_start_command_c P_((struct smtp_info *I, 
				    char *command,
				    struct cancel_data *cd));
static int smtp_start_command_c(I,command,cd)
     struct smtp_info *I;
     char *command;
     struct cancel_data *cd;
{
    int status = 0;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_start_command",
	      "Bad magic number",0);
    
    DPRINT(Debug,12,(&Debug,
		     "smtp_start_command_c: mailer info, I=%p\n",
		     I));

    if (!smtp_start_command_init(I)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command_c: failed on init\n"));
	status = 0;
	goto clean;
    }

    if (I->state != smtp_idle) {
	DPRINT(Debug,7,(&Debug,
			"smtp_start_command_c: wrong state -- perhaps idle timer -- waiting...\n"));
	while (smtp_reply_read != I->state &&
	       smtp_need_data    != I->state &&
	       smtp_error        != I->state &&
	       smtp_closing      != I->state &&
	       NULL         != I->RA.stream) {

	    if (!WaitStreamFor_c(I->RA.stream,SS_read_act,cd)) {
		
		DPRINT(Debug,7,(&Debug,"smtp_start_command_c: wait not succeed?\n"));
		
		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,7,(&Debug,"smtp_start_command_c: wait canceled!\n"));
		    
		    status = 0;
		    goto clean;
		}
	    }
	}

	DPRINT(Debug,7,(&Debug,
			 "smtp_start_command_c: ... wait ended\n"));

	if (!smtp_start_command_afterwait(I)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_start_command: failed after wait\n"));
	    status = 0;
	    goto clean;
	}
   }

    if (!smtp_start_command_tail(I,command)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command_c: failure\n"));
	status = 0;
	goto clean;
    }

    status = 1;

clean:
    DPRINT(Debug,12,(&Debug,
		     "smtp_start_command_c=%d\n",
		     status));

    return status;

}

static void smtp_push_data_add_dataline P_((struct smtp_info *I,
					   const char *data,
					   int len));
static void smtp_push_data_add_dataline(I,data,len)
     struct smtp_info *I;
     const char *data;
     int len;
{
    
    char * d,* d1;
    int len1;
    union ss_action_routine_data action_data;
    
    /* Dot-Stuff data line */
    if ('.' == data[0]) {
	len1 = len + 1;

	d = safe_malloc(len1);
	d[0] = '.';
	d1 = d + 1;
    } else {
	d = safe_malloc(len);
	d1 = d;
	len1 = len;
    }

    memcpy(d1,data,len);
    add_to_Write_Buffer(& (I->write_buffer),&d,len1);   /* May free d */

    action_data.smtp = I;
    
    ConfigStream2(I->RA.stream,
		  smtp_read_action,smtp_write_action,smtp_timeout_action,
		  ss_nofree_action_rt_data,
		  ss_noinc_action_rt_data_refcount,
		  smtp_badpid_action,
		  smtp_idle_alive_interval > MIN_SMTP_TIMEOUT ?
		  smtp_idle_alive_interval : MIN_SMTP_TIMEOUT,
		  action_data);    
}

static void smtp_push_data_add_command P_((struct smtp_info *I,
					   const char *data,
					   int len));
static void smtp_push_data_add_command(I,data,len)
     struct smtp_info *I;
     const char *data;
     int len;
{
    if (I->incomplete_command)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_push_data_add_command",
	      "incomplete_command not NUL !",0);

    I->incomplete_command = safe_malloc(len);
    memcpy(I->incomplete_command,data,len);
    I->incomplete_command_len = len;  
}

static int smtp_push_data_init P_((struct smtp_info *I));
static int smtp_push_data_init(I)
     struct smtp_info *I;
{
    int status = 0;

    if (smtp_error == I->state   ||
	smtp_closing == I->state ||
	!I->RA.stream) {

	if (I->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_push_data_init: Closing stream"));
	    if (I->seen_badpid) {
		DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 I->seen_badpid));
	    }
	    DPRINT(Debug,12,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(I->RA.stream),
			     I->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}	
	status = 0;
	goto clean;
    }

    if (I->incomplete_command)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_push_data_init",
	      "incomplete_command not NUL !",0);
    I->incomplete_command_len = 0;

    if (smtp_need_data != I->state  &&
	smtp_data != I->state) {
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_push_data_init",
	      "Wrong state !",0);
    }
    I->state = smtp_data;

    status = 1;

 clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_push_data_init=%d\n",status));
    return status;
}

S_(f_smtp_push_data smtp_push_data)
static int smtp_push_data P_((struct smtp_info *I,
			      const char *data,
			      int len,
			      int is_data_line));

static int smtp_push_data(I,data,len,is_data_line)
     struct smtp_info *I; 
     const char *data;
     int len;
     int is_data_line;
{
    int status = 0;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_push_data",
	      "Bad magic number",0);
    
    DPRINT(Debug,12,(&Debug,
		     "smtp_push_data: mailer info, I=%p\n",
		     I));

    if (!smtp_push_data_init(I)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_push_data: failed on init\n"));
	status = 0;
	goto clean;
    }
    
    if (is_data_line) {

	smtp_push_data_add_dataline(I,data,len);

	/* We push large amout of data so start immediately without
	   buffering all of data

	   Cancelable version is smtp_push_data_c()
	*/
	WaitStreamFor(I->RA.stream,SS_write_act);
    } else 
	smtp_push_data_add_command(I,data,len);

    status = 1;

 clean:
    DPRINT(Debug,12,(&Debug,
		     "smtp_push_data=%d\n",status));
    return status;
}

static int smtp_push_data_c P_((struct smtp_info *I,
				const char *data,
				int len,
				int is_data_line,
				struct cancel_data *cd))
    UNUSED_FUNCOK;
static int smtp_push_data_c(I,data,len,is_data_line,cd)
     struct smtp_info *I; 
     const char *data;
     int len;
     int is_data_line;
     struct cancel_data *cd;
{   
    int status = 0;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_push_data_c",
	      "Bad magic number",0);
    
    DPRINT(Debug,12,(&Debug,
		     "smtp_push_data_c: mailer info, I=%p\n",
		     I));

    if (!smtp_push_data_init(I)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_push_data_c: failed on init\n"));
	status = 0;
	goto clean;
    }

    if (is_data_line) {

	smtp_push_data_add_dataline(I,data,len);

	/* We push large amout of data so start immediately without
	   buffering all of data
	*/

	if (!WaitStreamFor_c(I->RA.stream,SS_write_act,cd)) {
	    DPRINT(Debug,7,(&Debug,"smtp_push_data_c: wait not succeed?\n"));

	    if (cd && is_canceled(cd)) {
		DPRINT(Debug,7,(&Debug,"smtp_push_data_c: wait canceled!\n"));
		
		smtp_cancel_reset_stream(I);
		status = 0;
		goto clean;
	    }
	}

    } else 
	smtp_push_data_add_command(I,data,len);

    status = 1;

 clean:
    DPRINT(Debug,12,(&Debug,
		     "smtp_push_data_c=%d\n",status));
    return status;
}


S_(f_smtp_command_arg smtp_command_arg)
static void smtp_command_arg P_((struct smtp_info *I, const char *str));
static void smtp_command_arg(I, str)
     struct smtp_info *I;
     const char *str;
{
    int L = strlen(str);

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_arg",
	      "Bad magic number",0);

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_arg: mailer info, I=%p\n",
		     I));

    if (I->state != smtp_command &&
	I->state != smtp_data) {

	DPRINT(Debug,1,(&Debug,
			"smtp_command_arg: state=%d\n",
			I->state));

	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_arg",
	      "Bad state",0);
    }

    I->incomplete_command = safe_realloc(I->incomplete_command,
					 I->incomplete_command_len+1+L);
    I->incomplete_command[I->incomplete_command_len++] = ' ';

    memcpy(I->incomplete_command+I->incomplete_command_len,str,L);
    I->incomplete_command_len += L;
}

static void smtp_command_ready_init P_((struct smtp_info *I, int data_end));
static void smtp_command_ready_init(I,data_end) 
     struct smtp_info *I;
     int data_end;
{
    char *s;
    union ss_action_routine_data data;
    
    if (I->state != smtp_command &&
	I->state != smtp_data) {
	
	DPRINT(Debug,1,(&Debug,
			"smtp_command_ready_init: state=%d\n",
			I->state));

	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_ready_init",
	      "Bad state",0);
    }
    
    if (data_end && I->state != smtp_data) {
	
	DPRINT(Debug,1,(&Debug,
			"smtp_command_ready_init: state=%d\n",
			I->state));

	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_ready_init",
	      "Bad state -- not smtp data",0);
    }
    
    /* May free I->p.smtp->incomplete_command */
    add_to_Write_Buffer(& (I->write_buffer),
			&(I->incomplete_command),
			I->incomplete_command_len);
    
    s = safe_strdup("\r\n");
    /* May free s */
    add_to_Write_Buffer(& (I->write_buffer),&s,2);

    data.smtp = I;
    
    ConfigStream2(I->RA.stream,
		  smtp_read_action,smtp_write_action,smtp_timeout_action,
		  ss_nofree_action_rt_data,
		  ss_noinc_action_rt_data_refcount,
		  smtp_badpid_action,
		  smtp_idle_alive_interval > MIN_SMTP_TIMEOUT ?
		  smtp_idle_alive_interval : MIN_SMTP_TIMEOUT,		 
		  data);    
}


static void smtp_command_ready(I, data_end)
     struct smtp_info *I;
     int data_end;
{

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_ready",
	      "Bad magic number",0);

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ready: mailer info, I=%p\n",
		     I));
    
    smtp_command_ready_init(I, data_end) ;

    if (data_end) {
	I->state = smtp_command;
	/* We push large amout of data so start immediately without
	   buffering all of data

	   Cancelable version is smtp_command_ready_c()
	*/
	WaitStreamFor(I->RA.stream,SS_write_act);
    }

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ready: done%s\n",
		     smtp_command == I->state ? " (state: smtp_command)" : ""
		     ));
}

static int smtp_command_ready_c P_((struct smtp_info *I,
				     int data_end,
				    struct cancel_data *cd));
static int smtp_command_ready_c(I,data_end,cd)
     struct smtp_info *I;
     int data_end;
     struct cancel_data *cd;
{
    int status = 0;
    
    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_ready_c",
	      "Bad magic number",0);
    
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ready_c: mailer info, I=%p\n",
		     I));
    
    smtp_command_ready_init(I, data_end) ;
    
    if (data_end) {
	I->state = smtp_command;
	/* We push large amout of data so start immediately without
	   buffering all of data
	*/

	if (!WaitStreamFor_c(I->RA.stream,SS_write_act,cd)) {
	    DPRINT(Debug,7,(&Debug,"smtp_command_ready_c: wait not succeed?\n"));

	    if (cd && is_canceled(cd)) {
		DPRINT(Debug,7,(&Debug,"smtp_command_ready_c: wait canceled!\n"));
		
		smtp_cancel_reset_stream(I);
		status = 0;
		goto clean;
	    }
	}
    }

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ready_c: done%s\n",
		     smtp_command == I->state ? " (state: smtp_command)" : ""
		     )); 

    status = 1;

 clean:
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ready_c=%d\n",status));
    return status;

}
				   
static int smtp_command_ok_tail P_((struct smtp_info *I, 
				    char response_code[4]));
static int smtp_command_ok_tail(I,response_code)
     struct smtp_info *I;
     char response_code[4];
{
    int status = 0;

    if (smtp_reply_incomplete == I->state ||
	smtp_reply_read     == I->state ||
	smtp_need_data        == I->state) {
	memcpy(response_code,I->response_code,3);
	
	if ('2' == response_code[0] ||
	    '3' == response_code[0])
	    status = 1;

	response_code[3] = '\0';
    }

    DPRINT(Debug,15,(&Debug,
		     "smtp_command_ok_tail=%d\n",status));
    return status;
}

static int smtp_command_ok_afterwait_r P_((struct smtp_info *I,
					 char response_code[4]));
static int smtp_command_ok_afterwait_r(I,response_code)
     struct smtp_info *I;
     char response_code[4];
{
    int status = 0;

    if (smtp_error == I->state ||
	smtp_closing == I->state) {
	if (I->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_ok_afterwait_r: Closing stream"));
	    if (I->seen_badpid) {
		DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 I->seen_badpid));
	    }
	    DPRINT(Debug,12,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(I->RA.stream),
			     I->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}	
	status = 0;
	goto clean;
    }
    
    if (!I->RA.stream) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_command_ok_afterwait_r: stream not open\n"));
	status = 0;
	goto clean;
    }

    status = 1;
    
 clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_command_ok_afterwait_r=%d\n",status));
    return status;
}


static int smtp_command_ok_afterwait_w P_((struct smtp_info *I,
					 char response_code[4]));
static int smtp_command_ok_afterwait_w(I,response_code)
     struct smtp_info *I;
     char response_code[4];
{
    int status = 0;

    if (smtp_error == I->state ||
	smtp_closing == I->state) {

	if (I->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_ok_afterwait_w: Closing stream"));
	    if (I->seen_badpid) {
		DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 I->seen_badpid));
	    }
	    DPRINT(Debug,12,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(I->RA.stream),
			     I->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}	
	status = 0;
	goto clean;
    }

    status = 1;
    
 clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_command_ok_afterwait_w=%d\n",status));
    return status;
}


static int smtp_command_ok_init P_((struct smtp_info *I, 
				    char response_code[4]));
static int smtp_command_ok_init(I,response_code)
     struct smtp_info *I;
     char response_code[4];
{
    int status = 0;

    response_code[0] = '\0';

    if (I->command[0]) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_command_ok_init: smtp command %.*s\n",
			 sizeof I->command,
			 I->command));
    }
    
    if (smtp_error == I->state   ||
	smtp_closing == I->state ||
	!I->RA.stream) {

	if (I->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_ok_init: Closing stream"));
	    if (I->seen_badpid) {
		DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 I->seen_badpid));
	    }
	    DPRINT(Debug,12,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(I->RA.stream),
			     I->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}	
	status = 0;
	goto clean;
    }

    if (smtp_command          != I->state &&
	smtp_reply_read     != I->state &&
	smtp_need_data        != I->state &&
	smtp_reply_incomplete != I->state &&
	smtp_wait_reply       != I->state) {
	
	DPRINT(Debug,1,(&Debug,
			"smtp_command_ok_init: state=%d\n",
			I->state));
	
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_ok_init",
	      "Bad state",0);
    }

    if (I->incomplete_command)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_ok_init",
	      "incomplete_command not NULL -- smtp_command_ready() not called",0);

    status = 1;

 clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_command_ok_init=%d\n",status));
    return status;
}


S_(f_smtp_command_ok smtp_command_ok)
/* May return command OK before all response lines are returned */
static int smtp_command_ok P_((struct smtp_info *I, 
			       char response_code[4]));
static int smtp_command_ok(I,response_code)
     struct smtp_info *I;
     char response_code[4];
{
    int status = 0;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_ok",
	      "Bad magic number",0);

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ok: mailer info, I=%p\n",
		     I));

    if (!smtp_command_ok_init(I,response_code)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command_init: failed on init\n"));
	status = 0;
	goto clean;
    }
   
    if (smtp_command     == I->state) {
	DPRINT(Debug,7,(&Debug,
			"smtp_command_ok: writing data to server...\n"));
	
	while ( smtp_command  == I->state &&
		NULL         != I->RA.stream) {
	    WaitStreamFor(I->RA.stream,SS_write_act);
	    /* Should user to have some way to interrupt progress ? 
	       Cancelable version is smtp_command_ok_c()
	     */
	}
	DPRINT(Debug,7,(&Debug,
			"smtp_command_ok: ... write ended\n"));

	if (! smtp_command_ok_afterwait_w(I,response_code)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_ok: failure after wait (write)\n"));
	    status = 0;
	    goto clean;
	}
    }

    /* If nothing read, wait data for now */
    if (smtp_wait_reply  == I->state) {
      
	DPRINT(Debug,7,(&Debug,
			"smtp_command_ok: waiting reply from server...\n"));

	while ( smtp_wait_reply  == I->state &&
		NULL         != I->RA.stream) {
	    WaitStreamFor(I->RA.stream,SS_read_act);
	    /* Should user to have some way to interrupt progress ? 
	       Cancelable version is smtp_command_ok_c()
	     */
	}
	DPRINT(Debug,7,(&Debug,
			"smtp_command_ok: ... wait ended\n"));

	if (! smtp_command_ok_afterwait_r(I,response_code)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_ok: failure after wait (reply)\n"));
	    status = 0;
	    goto clean;
	}
    }

    if (! smtp_command_ok_tail(I,response_code)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command_ok: failure\n"));
	status = 0;
	goto clean;
    }

    status = 1;
    
 clean:    
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ok=%d\n",status));
    return status;
}

/* May return command OK before all response lines are returned */
static int smtp_command_ok_c P_((struct smtp_info *I, 
				 char response_code[4],
				 struct cancel_data *cd));
static int smtp_command_ok_c(I,response_code,cd)
     struct smtp_info *I;
     char response_code[4];
     struct cancel_data *cd;
{
    int status = 0;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_ok_c",
	      "Bad magic number",0);
    
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ok_c: mailer info, I=%p\n",
		     I));
    
    if (!smtp_command_ok_init(I,response_code)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command_init_c: failed on init\n"));
	status = 0;
	goto clean;
    }

    if (smtp_command     == I->state) {
	DPRINT(Debug,7,(&Debug,
			"smtp_command_ok_c: writing data to server...\n"));

	while ( smtp_command  == I->state &&
		NULL         != I->RA.stream) {
	
	    if (!WaitStreamFor_c(I->RA.stream,SS_write_act,cd)) {
		DPRINT(Debug,7,(&Debug,"smtp_command_ok_c: wait not succeed?\n"));
		
		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,7,(&Debug,"smtp_command_ok_c: wait canceled!\n"));
		    
		    smtp_cancel_reset_stream(I);
		    status = 0;
		    goto clean;
		}
	    }
	}

	DPRINT(Debug,7,(&Debug,
			"smtp_command_ok_c: ... write ended\n"));

	if (! smtp_command_ok_afterwait_w(I,response_code)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_ok_c: failure after wait (write)\n"));
	    status = 0;
	    goto clean;
	}	
    }
    
    /* If nothing read, wait data for now */
    if (smtp_wait_reply  == I->state) {
      
	DPRINT(Debug,7,(&Debug,
			"smtp_command_ok_c: waiting reply from server...\n"));

	while ( smtp_wait_reply  == I->state &&
		NULL         != I->RA.stream) {

	    if (!WaitStreamFor_c(I->RA.stream,SS_read_act,cd)) {

		DPRINT(Debug,7,(&Debug,"smtp_command_ok_c: wait not succeed?\n"));

		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,7,(&Debug,"smtp_command_ok_c: wait canceled!\n"));
		    
		    smtp_cancel_reset_stream(I);
		    status = 0;
		    goto clean;
		}	       
	    }
	}

	DPRINT(Debug,7,(&Debug,
			"smtp_command_ok_c: ... wait ended\n"));

	if (! smtp_command_ok_afterwait_r(I,response_code)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_start_command_ok_c: failure after wait\n"));
	    status = 0;
	    goto clean;
	}
    }

    if (! smtp_command_ok_tail(I,response_code)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_start_command_ok_c: failure\n"));
	status = 0;
	goto clean;
    }
    
    status = 1;
      
 clean:
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_ok_c=%d\n",status));
    return status;
}

static const char * smtp_command_response_tail P_((struct smtp_info *I,
					     int *idx));
static const char * smtp_command_response_tail(I,idx)
     struct smtp_info *I;
     int *idx;
{
    char * status = NULL;

    if (*idx < I-> response_text_len) {
	status = I-> response_text[(*idx)++];
    }


    DPRINT(Debug,15,(&Debug,
		     "smtp_command_ok_init=%s\n",
		     status ? status : "<NULL>"));

    return status;
}

static int smtp_command_response_afterwait 
   P_((struct smtp_info *I));
static int smtp_command_response_afterwait(I)
     struct smtp_info *I;
{
    int status = 0;

    if (smtp_reply_incomplete != I->state &&
	smtp_reply_read     != I->state &&
	smtp_need_data        != I->state) {

	DPRINT(Debug,12,(&Debug,
			 "smtp_command_response_afterwait: Bad state, state=%d\n",
			 I->state));

	status = 0;
	goto clean;
    }
    status = 1;

 clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_command_response_afterwait=%d\n",status));
    return status;
}

static int smtp_command_response_init P_((struct smtp_info *I,
					  int *idx));
static int smtp_command_response_init(I,idx)
     struct smtp_info *I;
     int *idx;
{
    int status = 0;

    if (*idx < 0) {
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "smtp_command_response_init",
	      "Negative *idx",0);
    }

    if (smtp_reply_incomplete != I->state &&
	smtp_reply_read     != I->state &&
	smtp_need_data        != I->state) {
	
	DPRINT(Debug,12,(&Debug,
			"smtp_command_response_init: bad state, state=%d\n",
			I->state));

	status = 0;
	goto clean;
    }
    
    status = 1;

 clean:
    DPRINT(Debug,15,(&Debug,
		     "smtp_command_response_init=%d\n",status));
    return status;
}   



S_(f_smtp_command_response smtp_command_response)
/* Returns response lines one at time and waits that all is returned */
static const char * smtp_command_response P_((struct smtp_info *I,
					int *idx));

static const char * smtp_command_response(I,idx)
     struct smtp_info *I;
     int *idx;
{
    const char * status = NULL;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_response",
	      "Bad magic number",0);

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_response: mailer info, I=%p, *idx=%d\n",
		     I,*idx));

    if (! smtp_command_response_init(I,idx)) {
	DPRINT(Debug,12,(&Debug,
			     "smtp_command_response: failure on init\n"));
	    status = NULL;
	    goto clean;
    }
        

    if (*idx >= I-> response_text_len &&
	smtp_reply_incomplete == I->state) {
	DPRINT(Debug,7,(&Debug,
			"smtp_command_response: waiting more data from server...\n"));
	while (*idx >= I-> response_text_len &&
	       smtp_reply_incomplete == I->state &&
	       NULL         != I->RA.stream) {
            WaitStreamFor(I->RA.stream,SS_read_act);
            /* Should user to have some way to interrupt progress ? 
	       Cancelable version is smtp_command_response_c()
	     */
        }
	DPRINT(Debug,7,(&Debug,
			"smtp_command_response: ... wait ended\n"));

	if (!smtp_command_response_afterwait(I)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_response: failure after wait\n"));
	    status = NULL;
	    goto clean;
	}
    }

    status = smtp_command_response_tail(I,idx);

 clean:

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_response=%s\n",
		     status ? status : "<NULL>"));

    return status;
}

/* Returns response lines one at time and waits that all is returned */
static const char * smtp_command_response_c P_((struct smtp_info *I,
						int *idx,
						struct cancel_data *cd));
static const char * smtp_command_response_c(I,idx,cd)
     struct smtp_info *I;
     int *idx;
     struct cancel_data *cd;
{
    const char * status = NULL;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_response_c",
	      "Bad magic number",0);

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_response_c: mailer info, I=%p, *idx=%d\n",
		     I,*idx));

    if (! smtp_command_response_init(I,idx)) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_command_response_c: failure on init\n"));
	status = NULL;
	goto clean;
    }

     if (*idx >= I-> response_text_len &&
	smtp_reply_incomplete == I->state) {
	DPRINT(Debug,7,(&Debug,
			"smtp_command_response_c: waiting more data from server...\n"));
	
	while (*idx >= I-> response_text_len &&
	       smtp_reply_incomplete == I->state &&
	       NULL         != I->RA.stream) {

	    if (!WaitStreamFor_c(I->RA.stream,SS_read_act,cd)) {

		DPRINT(Debug,7,(&Debug,"smtp_command_response_c: wait not succeed?\n"));
		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,7,(&Debug,"smtp_command_response_c: wait canceled!\n"));
		    
		    smtp_cancel_reset_stream(I);
		    status = NULL;
		    goto clean;
		}
	    }

	}

	DPRINT(Debug,7,(&Debug,
			"smtp_command_response_c: ... wait ended\n"));

	if (!smtp_command_response_afterwait(I)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_response_c: failure after wait\n"));
	    status = NULL;
	    goto clean;
	}
    }

    status = smtp_command_response_tail(I,idx);

 clean:
    
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_response_c=%s\n",
		     status ? status : "<NULL>"));

    return status;
}

static void smtp_command_reset_tail P_((struct smtp_info *I));
static void smtp_command_reset_tail(I)
     struct smtp_info *I;
{
    if (smtp_error == I->state   ||
	smtp_closing == I->state) {
	if (I->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_command_reset_tail: Closing stream"));
	    if (I->seen_badpid) {
		DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 I->seen_badpid));
	    }
	    DPRINT(Debug,12,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(I->RA.stream),
			     I->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}	
    }

    if (I->command[0]) {
	DPRINT(Debug,12,(&Debug,
			 "smtp_command_reset_tail: resetting command %.*s\n",
			 sizeof I->command,
			 I->command));

	strncpy(I->command,"",sizeof I->command);
    }
    
    free_response_text(I);

    if (smtp_error     != I->state  &&
	smtp_closing   != I->state &&
	smtp_need_data != I->state)
	I->state = smtp_idle;

}

/* Waits that command is completed and resets data then */
static void smtp_command_reset(I)
     struct smtp_info *I;
{

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_reset",
	      "Bad magic number",0);

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_reset: mailer info, I=%p\n",I));
    
    if (smtp_command     == I->state) {
	DPRINT(Debug,7,(&Debug,
			"smtp_command_reset: writing data to server...\n"));
	
	while ( smtp_command  == I->state &&
		NULL         != I->RA.stream) {
	    WaitStreamFor(I->RA.stream,SS_write_act);
	    /* Should user to have some way to interrupt progress ? 
	       Cancelable version is smtp_command_reset_c()
	     */
	}
	DPRINT(Debug,7,(&Debug,
			"smtp_command_reset: ... write ended\n"));
    }

    if (smtp_wait_reply  == I->state ||
	smtp_reply_incomplete == I->state) {

	DPRINT(Debug,7,(&Debug,
			"smtp_command_reset: waiting reply from server...\n"));

	while ( (smtp_wait_reply       == I->state || 
		 smtp_reply_incomplete == I->state) &&
		NULL         != I->RA.stream) {
	    WaitStreamFor(I->RA.stream,SS_read_act);
	    /* Should user to have some way to interrupt progress ? 
	       Cancelable version is smtp_command_reset_c()
	     */
	}
	DPRINT(Debug,7,(&Debug,
			"smtp_command_reset: ... wait ended\n"));
    }

    smtp_command_reset_tail(I);
    
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_reset: done%s\n",
		     smtp_idle == I->state ? " (state: smtp_idle)" : ""));
}

static int smtp_command_reset_c P_((struct smtp_info *I,
				    struct cancel_data *cd));
static int smtp_command_reset_c(I,cd)
     struct smtp_info *I;
     struct cancel_data *cd;
{
    int status = 0;

   if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_command_reset_c",
	      "Bad magic number",0);

    DPRINT(Debug,12,(&Debug,
		     "smtp_command_reset_c: mailer info, I=%p\n",I));

    if (smtp_command     == I->state) {
	DPRINT(Debug,7,(&Debug,
			"smtp_command_reset_c: writing data to server...\n"));

	while ( smtp_command  == I->state &&
		NULL         != I->RA.stream) {

	    if (!WaitStreamFor_c(I->RA.stream,SS_write_act,cd)) {
		DPRINT(Debug,7,(&Debug,"smtp_command_reset_c: wait not succeed?\n"));

		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,7,(&Debug,"smtp_command_reset_c: wait canceled!\n"));

		    smtp_cancel_reset_stream(I);
		    status = 0;
		    goto clean;
		}
	    }
	}

	DPRINT(Debug,7,(&Debug,
			"smtp_command_reset_c: ... write ended\n"));
    }

    if (smtp_wait_reply  == I->state ||
	smtp_reply_incomplete == I->state) {

	DPRINT(Debug,7,(&Debug,
			"smtp_command_reset_c: waiting reply from server...\n"));

	while ( (smtp_wait_reply       == I->state || 
		 smtp_reply_incomplete == I->state) &&
		NULL         != I->RA.stream) {
	
	    if (!WaitStreamFor_c(I->RA.stream,SS_read_act,cd)) {
		DPRINT(Debug,7,(&Debug,"smtp_command_reset_c: wait not succeed?\n"));
		
		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,7,(&Debug,"smtp_command_reset_c: wait canceled!\n"));
		    
		    smtp_cancel_reset_stream(I);
		    status = 0;
		    goto clean;
		}
	    }
	}

	DPRINT(Debug,7,(&Debug,
			"smtp_command_reset_c: ... wait ended\n"));
    }

    smtp_command_reset_tail(I);

    status = 1;
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_reset_: done%s\n",
		     smtp_idle == I->state ? " (state: smtp_idle)" : ""));

clean:
    DPRINT(Debug,12,(&Debug,
		     "smtp_command_reset_c=%d\n",status));
    return status;
}

static const struct string *smtp_Server(I)
     struct smtp_info *I;
{
    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_Server",
	      "Bad magic number",0);

    return Server_s((I->SMTP_flags & SMTP_submission) ? Submission_s : SMTP_s);
}

struct smtp_callbacks SMTP_FUNCTIONS = {
    smtp_start_command,
    smtp_push_data,
    smtp_command_arg,
    smtp_command_ready,
    smtp_command_ok,
    smtp_command_response,
    smtp_command_reset,
    smtp_Server
};

static void free_ESMTP_keywords  P_((struct smtp_info *I));
static void free_ESMTP_keywords(I)
     struct smtp_info *I;
{
    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"free_ESMTP_keywords",
	      "Bad magic number",0);

    if (I->ESMTP_keywords) {
	int i;
	for (i = 0; i < I->ESMTP_keyword_len; i++) {
	    if (I->ESMTP_keywords[i]) {
		free(I->ESMTP_keywords[i]);
		I->ESMTP_keywords[i] = NULL;
	    }
	}
	free(I->ESMTP_keywords);
	I->ESMTP_keywords        = NULL;
    }    
    I->ESMTP_keyword_len     = -1;   /* -1 == EHLO not issued yet */

}

			       

static int smtp_run_ehlo_after_response P_((struct smtp_info *I,
					    struct SMTP_ext_libs *  smtp_ext_libs,
					    int                     smtp_ext_libcount,
					    enum CAPA_phase  phase,
					    int * need_retry_ehlo));
static int smtp_run_ehlo_after_response(I,smtp_ext_libs,smtp_ext_libcount,
					phase,need_retry_ehlo)
     struct smtp_info  * I;
     struct SMTP_ext_libs  * smtp_ext_libs;
     int                     smtp_ext_libcount;
     enum CAPA_phase         phase;
     int                   * need_retry_ehlo;
{
    int ret = 0;

    DPRINT(Debug,13,(&Debug,
		     "%d libraries available\n",smtp_ext_libcount));

    *need_retry_ehlo = 0;

    if (!handle_smtp_ext_libs(I,&smtp_ext_libs,&smtp_ext_libcount,
			      &phase,&SMTP_FUNCTIONS)) {
	/* Something gone wrong ... */
	
	ret = 0;
	goto clean;	
    }

    switch(phase) {
	
    case capa_prelogin_again:
	*need_retry_ehlo = 1;
	break;
	
    case capa_logged_again:
	I->SMTP_flags |= SMTP_logged;
	*need_retry_ehlo = 1;
	break;
	    
    case capa_logged:
	I->SMTP_flags |= SMTP_logged;
	break;
    default:
	break;
    }

    ret = 1;
		
 clean:
    return ret;
}

static void smtp_run_ehlo_handle_line P_((struct smtp_info *I,
					  struct SMTP_ext_libs ** smtp_ext_libs,
					  int                   * smtp_ext_libcount,
					  const char * line));
static void smtp_run_ehlo_handle_line(I,smtp_ext_libs,smtp_ext_libcount,line)
     struct smtp_info *I;
     struct SMTP_ext_libs ** smtp_ext_libs;
     int                   * smtp_ext_libcount;
     const char            * line;
{
    char * ext, *arg = NULL;

    I->ESMTP_keywords = 
	safe_array_realloc(I->ESMTP_keywords,
			   (I->ESMTP_keyword_len+1),
			   sizeof (*(I->ESMTP_keywords)));
    I->ESMTP_keywords[I->ESMTP_keyword_len++] =
	safe_strdup(line);	

    ext = safe_strdup(line);
    if (NULL != (arg = strchr(ext,' '))) {
	*arg = '\0';
	arg++;
    }
    
    DPRINT(Debug,2,(&Debug, 
		    "SMTP server extension: '%s'%s%s\n",
		    ext,
		    arg ? " args: " : "",
		    arg ? arg : ""));

    probe_smtp_ext_lib(smtp_ext_libs,
		       smtp_ext_libcount,
		       ext,arg);
			
    free(ext);
}

static int smtp_run_ehlo P_((struct smtp_info *I));
static int smtp_run_ehlo(I)
     struct smtp_info *I;
{
    int ret = 1;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_run_ehlo",
	      "Bad magic number",0);

    DPRINT(Debug,14,(&Debug,
		     "smtp_run_ehlo: mailer info, I=%p\n",I));

 retry_ehlo:
    if (smtp_start_command(I,"EHLO")) {
	char response_code[4];

	smtp_command_arg(I,hostfullname);
	smtp_command_ready(I,0);

	if (smtp_command_ok(I,response_code)) {

	    if (0 == memcmp(response_code,"250",3)) {
		int i, next_idx = 0;
		const char * line;
		
		struct SMTP_ext_libs *  smtp_ext_libs = NULL;
		int                     smtp_ext_libcount = 0;

		enum CAPA_phase  phase = capa_prelogin;
		int need_retry_ehlo = 0;

		if (I->SMTP_flags & SMTP_logged) {
		    phase = capa_logged;
		    DPRINT(Debug,13,(&Debug,"Already logged in\n"));
		}

		free_ESMTP_keywords(I);

		I->ESMTP_keyword_len = 0;

		for (i = 0; NULL != (line = smtp_command_response(I,&next_idx)); i = next_idx) {
		    if (i > 0) { /* First response line is greeting */
			smtp_run_ehlo_handle_line(I,&smtp_ext_libs,
						  &smtp_ext_libcount,
						  line);			
		    }
		}

		if (smtp_run_ehlo_after_response(I,smtp_ext_libs,
						 smtp_ext_libcount,
						 phase,&need_retry_ehlo)) {

		    if (need_retry_ehlo) {
			DPRINT(Debug,14,(&Debug,
					 "smtp_run_ehlo: post process retries EHLO\n"));
			goto retry_ehlo;
		    }

		} else {
		    DPRINT(Debug,14,(&Debug,
				     "smtp_run_ehlo: post process failed\n"));
		    ret = 0;
		    goto clean;	
		}
	       
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "smtp_run_ehlo: not expected response code: %.3s\n",
				 response_code));
		goto fail;
	    }
	} else {
	fail:
	    /* If EHLO fails try HELO ... (ignoring result) */
	    smtp_command_reset(I);
	    if (smtp_start_command(I,"HELO")) {
		smtp_command_arg(I,hostfullname);
		smtp_command_ready(I,0);

		I->ESMTP_keyword_len = 0;
	    } else 
		goto failure;
	}

    } else {
    failure:
	DPRINT(Debug,14,(&Debug,
			 "smtp_run_ehlo: failure\n"));
	ret = 0;

	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpConnectionFailed,
			  "Connection to %S server failed"),
		  Server_s((I->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));

    }

 clean:
    smtp_command_reset(I);

    DPRINT(Debug,14,(&Debug,
		     "smtp_run_ehlo=%d: done\n",ret));

    return ret;
}

static int smtp_run_ehlo_c P_((struct smtp_info *I,struct cancel_data *cd));
static int smtp_run_ehlo_c(I,cd)
     struct smtp_info *I;
     struct cancel_data *cd;
{
    int ret = 1;
    
    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_run_ehlo_c",
	      "Bad magic number",0);

    DPRINT(Debug,14,(&Debug,
		     "smtp_run_ehlo_c: mailer info, I=%p\n",I));

 retry_ehlo:
    if (smtp_start_command_c(I,"EHLO",cd)) {
	char response_code[4];
	
	smtp_command_arg(I,hostfullname);	
	if (smtp_command_ready_c(I,0,cd)) {
	    DPRINT(Debug,7,(&Debug,"smtp_run_ehlo_c: EHLO done\n"));
	} else
	    goto failure;
      
	if (smtp_command_ok_c(I,response_code,cd)) {

	    if (0 == memcmp(response_code,"250",3)) {
		int i;
		const char * line;
		
		struct SMTP_ext_libs *  smtp_ext_libs = NULL;
		int                     smtp_ext_libcount = 0;

		enum CAPA_phase  phase = capa_prelogin;
		int need_retry_ehlo = 0;

		if (I->SMTP_flags & SMTP_logged) {
		    phase = capa_logged;
		    DPRINT(Debug,13,(&Debug,"Already logged in\n"));
		}

		free_ESMTP_keywords(I);

		I->ESMTP_keyword_len = 0;

		for (i = 0; NULL != (line = smtp_command_response_c(I,&i,cd)); ) {
		    if (i > 0) { /* First response line is greeting */
			smtp_run_ehlo_handle_line(I,&smtp_ext_libs,
						  &smtp_ext_libcount,
						  line);
		    }
		}

		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,7,(&Debug,"smtp_run_ehlo_c: command canceled!\n"));
		    
		    ret = 0;
		    goto clean;
		}

		if (smtp_run_ehlo_after_response(I,smtp_ext_libs,
						 smtp_ext_libcount,
						 phase,&need_retry_ehlo)) {

		    if (need_retry_ehlo) {
			DPRINT(Debug,14,(&Debug,
					 "smtp_run_ehlo_c: post process retries EHLO\n"));
			goto retry_ehlo;
		    }

		} else {
		    DPRINT(Debug,14,(&Debug,
				     "smtp_run_ehlo_c: post process failed\n"));
		    ret = 0;
		    goto clean;	
		}

	    } else {
		DPRINT(Debug,14,(&Debug,
				 "smtp_run_ehlo_c: not expected response code: %.3s\n",
				 response_code));
		goto fail;
	    }

	} else {
	fail:

	    if (cd && is_canceled(cd)) {
		DPRINT(Debug,7,(&Debug,"smtp_run_ehlo_c: command canceled!\n"));
		
		ret = 0;
		goto clean;
	    }

	    if (smtp_command_reset_c(I,cd)) {
		if (smtp_start_command_c(I,"HELO",cd)) {
		    smtp_command_arg(I,hostfullname);

		    if (smtp_command_ready_c(I,0,cd)) {
			DPRINT(Debug,7,(&Debug,"smtp_run_ehlo_c: HELO done\n"));
		    } else 
			goto failure;
		    
		} else 
		    goto failure;
	    } else
		goto failure;
	}

    } else {
    failure:
	DPRINT(Debug,14,(&Debug,
			 "smtp_run_ehlo_c: failure\n"));
	ret = 0;

	if (cd && is_canceled(cd)) {
	    DPRINT(Debug,7,(&Debug,"smtp_run_ehlo_c: command canceled!\n"));
	    
	    goto clean;
	}

	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpConnectionFailed,
			  "Connection to %S server failed"),
		  Server_s((I->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));

    }
   
 clean: 
    if (smtp_command_reset_c(I,cd)) {
	if (ret) {
	    DPRINT(Debug,14,(&Debug,
			     "smtp_run_ehlo_c: done\n"));
	}
    } else {
	DPRINT(Debug,14,(&Debug,
			 "smtp_run_ehlo_c: Command reset failed\n"));
	ret = 0;
    }

    DPRINT(Debug,14,(&Debug,
		     "smtp_run_ehlo_c=%d\n",ret));

    return ret;
}

/* common to submit and sendmail-bs mailer -------------------------------- */


S_(mailer_set_option default_set_option)
static int default_set_option  P_((struct mailer_config *M,
				   struct mailer_option_list *X,
				   char *value));
static int default_set_option(M,X,value) 
     struct mailer_config *M;
     struct mailer_option_list *X;
     char *value;
{
    int r ;

    r = X->value_func(M,X,&value,1);
    return r;
}

static char * default_from P_((void));
static char * default_from(void)
{
    char * ret = NULL;

    const char * maildomain = hostfullname;
    const char * have_mailname = useful_mailname();
    
    if (have_mailname)
	maildomain = have_mailname;
    else {
	DPRINT(Debug,10,(&Debug,
			 "default_from: mailname not set or not useful, using hostfullname=%s\n",
			 maildomain));
    }
	

    if (NULL != strpbrk(username,
                        ":\"\\!#%&()=?',;.:<>")) {
	ret = elm_message(FRM("<%Q@%s>"),
			   username,maildomain);
    } else {
	ret = elm_message(FRM("<%s@%s>"),
			   username,maildomain);
    }

    DPRINT(Debug,13,(&Debug,
		     "default_from=%s\n",ret));

    return ret;
}

static enum msd_addr_status msd_addr_response  P_((char response_code[4]));
static enum msd_addr_status msd_addr_response(response_code)
     char response_code[4];
{
    switch (response_code[0]) {
    case '4':
	return  msd_addr_tmpfail;
    case '5':
	return msd_addr_fail;
    case '2':
	return msd_addr_ok;
    }

    return msd_addrstat_none;  /* Bad response code */
}

static enum msd_data_status msd_data_response  P_((char response_code[4]));
static enum msd_data_status msd_data_response(response_code)
     char response_code[4];
{
    switch (response_code[0]) {
    case '4':
	return msd_mail_tmpfail;
    case '5':
	return msd_mail_fail;
    case '3':
	return msd_data_ok;
    case '2':
	return msd_mail_sent;
    }

    return msd_datastat_none;  /* Bad response code */
}

static void translate_response P_((char response_code[4], int *Exit_stat));
static void translate_response(response_code,Exit_stat)
     char response_code[4];
     int *Exit_stat;
{

#ifdef I_SYSEXITS
    switch (response_code[0]) {
    case '4':
#ifdef  EX_TEMPFAIL
	*Exit_stat = EX_TEMPFAIL;
#endif
	break;
    case '5':
#ifdef  EX_UNAVAILABLE 
	*Exit_stat = EX_UNAVAILABLE;
#endif
	break;
    }

#ifdef  EX_NOUSER   		
    if (0 == strncmp(response_code,"550",3))
	*Exit_stat = EX_NOUSER;   /* Can this to be used for sender? */
#endif
#endif		

}

/* Return 1 if  if mail is moved to ticket */
static int smtp_pass_ticket P_((struct mailer_config *M,
				struct mail_send_state *C,
				struct mailer_cancel_ticket * ticket,
				enum msd_action action));
static int smtp_pass_ticket(M,C,ticket,action)
     struct mailer_config *M;
     struct mail_send_state *C;
     struct mailer_cancel_ticket * ticket;
     enum msd_action action;
{
    int r = 0;

    if (MAILER_CANCEL_TICKET_magic != ticket->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_pass_ticket",
	      "Bad magic number (mailer_cancel_ticket)",0);

    if (C->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_pass_ticket",
	      "Bad magic number (mail_send_state)",0);

    ticket->msd_action = action;

    switch (ticket->msd_action) {
    case msd_act_EOF: 
	DPRINT(Debug,4, (&Debug, 
			 "smtp_pass_ticket: action msd_act_EOF\n"));
	break;
	
    case msd_act_SIG:
	DPRINT(Debug,4, (&Debug, 
			 "smtp_pass_ticket: action msd_act_SIG\n"));
	break;
	
    case msd_act_need_restart:
	DPRINT(Debug,4, (&Debug, 
			 "smtp_pass_ticket: action msd_act_need_restart\n"));
	r = 1;
	break;
	
    case msd_act_default:
	DPRINT(Debug,4, (&Debug, 
			 "smtp_pass_ticket: action msd_act_default\n"));
	break;
	
    case msd_act_cancel:
	DPRINT(Debug,4, (&Debug, 
			 "smtp_pass_ticket action msd_act_cancel\n"));
	r = 1;
	break;
	
    case msd_act_edit:
	DPRINT(Debug,4, (&Debug, 
			 "smtp_pass_ticket: action msd_act_edit\n"));
	r = 1;
	break;
	
    case msd_act_send:
	DPRINT(Debug,4, (&Debug, 
			 "smtp_pass_ticket: action msd_act_send\n"));
	break;
    default:
	DPRINT(Debug,4, (&Debug, 
			 "smtp_pass_ticket: action %d\n",
			 action));
    }

    if (r) {
	
	ticket->fname = C->fname;
	C->fname = NULL;
	
	ticket->OUT   = C->OUT;
	C->OUT        = NULL;
	
	if (C->OUT) 
	    out_state_fseek (C->OUT, 0);
	
	if (! ticket->fname) {
	    DPRINT(Debug,4, (&Debug, 
			     "smtp_pass_ticket: No filename\n"));
	}
	
	if (! ticket->OUT) {
	    DPRINT(Debug,4, (&Debug, 
			     "smtp_pass_ticket: No out_state\n"));
	}
	
	if (! ticket->fname && ! ticket->OUT)   
	    r = 0;
    }

    DPRINT(Debug,8, (&Debug, 
		     "smtp_pass_ticket=%d\n",r));
	
    return r;
    
}

static void smtp_print_response_tail P_((struct smtp_info *I,
					 int *idx));
static void smtp_print_response_tail(I,idx)
     struct smtp_info *I;
     int *idx;
{
    const char *r;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_print_response_tail",
	      "Bad magic number",0);
    
    while (NULL != (r = smtp_command_response(I,idx))) {
	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpServerMsg,
			  "%S server: %s"),
		  smtp_Server(I),
		  r);
    }    
}

static struct string * message_helper P_((struct smtp_info *I,
					  char response_code[4]));
static struct string * message_helper(I,response_code)
     struct smtp_info *I;
     char response_code[4];
{
    int idx = 0;
    const char * r;
    struct string * message;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"message_helper",
	      "Bad magic number",0);

    r = smtp_command_response(I,&idx);

    if (r)
	message = format_string(FRM("%3.3s %s"),
				response_code,r);
    else
	message = format_string(FRM("%3.3s"),
				response_code);

    return message;
}

S_(mailer_backend3_hook smtp_common_mailer_backend3)
static int smtp_common_mailer_backend3 P_((struct mailer_config *M,
					   struct mail_send_state **C,
					   int encoding_top,
					   char * title,
					   sending_message_func *sm,
					   int *exit_status,
					   struct mailer_cancel_ticket * 
					   ticket,
					   long message_size));
static int smtp_common_mailer_backend3(M,C,encoding_top,title,sm,
				       exit_status,ticket,message_size)
     struct mailer_config *M;
     struct mail_send_state **C;
     int encoding_top;
     char * title;
     sending_message_func *sm;
     int *exit_status;   
     struct mailer_cancel_ticket * ticket;
     long message_size;
{
    char buffer [ 32 * 1024 ];
    int l;
    int have_dsn = 0;
    int have_size = 0;
    int Exit_stat = -1;
    int ret = 0; 

    char * from;
    char * f1;
    
    int fail = 0; /* Number of failed addresses */

    struct string * from_message    =  NULL;
    struct string * data_message    =  NULL;
    enum msd_action          action = msd_act_default;

    if ((*C)->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_common_mailer_backend3",
	      "Bad magic number (mail send state)",0);

    f1 = (*C)->mail_from;
    if (!f1)
	f1 = default_from();

    if ('<' == f1[0]) {
	from = elm_message(FRM("FROM:%s"),
			   f1);
    } else {
	from = elm_message(FRM("FROM:<%s>"),
			   f1);
    }
    
    if ((*C)->msd_sd) 
	from_message = new_string2(system_charset,s2us(f1));
	    
    if (f1 != (*C)->mail_from) {
	free(f1);
	f1 = NULL;
    }

    if (SMTP_mailer_info_magic != (*C)->head->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_common_mailer_backend",
	      "Bad magic number (smtp mailer info)",0);

    if ((*C)->msd_sd && 
	(*C)->head->p.smtp->RA.host) {
	
	struct string * message = 
	    new_string2(system_charset,
			s2us((*C)->head->p.smtp->RA.host));

	(*C)->msd_con_upd_cb((*C)->msd_sd,msd_server,message);

	free_string(&message);
    }

    /* With b)ounce command EHLO/HELO is not issued before that */
    if (-1 == (*C)->head->p.smtp->ESMTP_keyword_len) {
	if (!smtp_run_ehlo((*C)->head->p.smtp)) {
	    ret = 0; /* Some extension failed hard */
	    goto fail;
	}
    }

    if ((*C)->msd_sd && (*C)->head->p.smtp->RA.stream) {
	int r = 0;
	struct string * peer_name = NULL;
	StreamInfo((*C)->head->p.smtp->RA.stream,SS_verify,&r,NULL,
		   &peer_name);

	if (r > 0 && peer_name) 
	    (*C)->msd_con_upd_cb((*C)->msd_sd,msd_verified_as,
				 peer_name);

	if (peer_name)
	    free_string(&peer_name);
    }

    if ((*C)->dsn) {
	int i;
	for (i = 0; i < (*C)->head->p.smtp->ESMTP_keyword_len; i++) {
	    if (0 == istrcmp("DSN",(*C)->head->p.smtp->ESMTP_keywords[i])) {
		have_dsn = 1;
	    } 
	}
    }
       
    
    if (message_size >= 0) {
	int i;

	if ((*C)->msd_sd)
	    (*C)->msd_size_upd_cb((*C)->msd_sd,msd_message_size,
				   message_size);


	for (i = 0; i < (*C)->head->p.smtp->ESMTP_keyword_len; i++) {
	    if (0 == istrcmp("SIZE",(*C)->head->p.smtp->ESMTP_keywords[i])) {
		have_size = 1;
	    } else if (0 == strincmp("SIZE ",
				    (*C)->head->p.smtp->ESMTP_keywords[i],
				    5)) {
		char * end = NULL;
		long limit = strtol((*C)->head->p.smtp->ESMTP_keywords[i]+5,
				    &end,10);
		have_size = 1;

		if (end && '\0' == *end) {
		    
		    if ((*C)->msd_sd)
			(*C)->msd_size_upd_cb((*C)->msd_sd,
					      msd_size_limit,
					      limit);		    
		    
		    if (message_size > limit) {
			
			DPRINT(Debug,4,(&Debug, 
					"Message too big: size = %ld > limit = %ld\n",
					message_size,limit));
			
			/* prompt user */
			
			if ((*C)->msd_sd && ticket) {
			    
			    action = 
				(*C)->msd_size_check_cb((*C)->msd_sd,
							message_size,	    
							limit);
			    
			    if (smtp_pass_ticket(M,*C,ticket,action)) {
				DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
				goto cancel_action;
			    } else { 
				switch (action) {
				case msd_act_EOF:
				case msd_act_SIG:
				case msd_act_need_restart:
				case msd_act_cancel:
				case msd_act_edit:
				    
				    DPRINT(Debug,4,(&Debug, " -- cancel request -- no ticket\n"));
				    goto cancel_action;
				case msd_act_default:
				case msd_act_send:
				    break;
				}			    
			    }				    
			} 
		    }		    
		}
	    }
	}
    }

    if (smtp_start_command((*C)->head->p.smtp,"MAIL")) {
	char response_code[4];
	
	if ((*C)->msd_sd)
	    (*C)->msd_addr_upd_cb((*C)->msd_sd,-1,from_message,
				  msd_sending_addr,NULL);

	smtp_command_arg((*C)->head->p.smtp,from);

	if (have_size) {
	    char * arg = elm_message(FRM("SIZE=%ld"),
				     message_size);

	    smtp_command_arg((*C)->head->p.smtp,arg);
	    free(arg);
	}  


	if (encoding_top == ENCODING_8BIT) {
	    int i;
	    for (i = 0; i < (*C)->head->p.smtp->ESMTP_keyword_len; i++)
		if (0 == istrcmp("8BITMIME",
				 (*C)->head->p.smtp->ESMTP_keywords[i])) {
		    smtp_command_arg((*C)->head->p.smtp,"BODY=8BITMIME");
		    break;
		}
	}

#if 0
	if (encoding_top == ENCODING_BINARY) {
	    int i;
	    for (i = 0; i < (*C)->head->p.smtp->ESMTP_keyword_len; i++)
		if (0 == istrcmp("BINARYMIME",
				 (*C)->head->p.smtp->ESMTP_keywords[i])) {
		    smtp_command_arg((*C)->head->p.smtp,"BODY=BINARYMIME");
		    break;
		}
	}
#endif

	if (have_dsn) {
	    if ((*C)->dsn & DSN_FULL) 
		smtp_command_arg((*C)->head->p.smtp,"RET=FULL");
	    else if ((*C)->dsn & DSN_HDRS)
		smtp_command_arg((*C)->head->p.smtp,"RET=HDRS");    
	}

	smtp_command_ready((*C)->head->p.smtp,0);
	if (smtp_command_ok((*C)->head->p.smtp,response_code)) {	    

	    if ((*C)->msd_sd) {

		struct string * message = 
		    message_helper((*C)->head->p.smtp,response_code);

		(*C)->msd_addr_upd_cb((*C)->msd_sd,-1,
				      from_message,
				      msd_addr_response(response_code),
				      message);

		free_string(&message);
	    
	    }

	} else { /* command failed */

	    if (response_code[0]) {
		int idx = 0;
		const char * r = 
		    smtp_command_response((*C)->head->p.smtp,&idx);

		struct string * message = NULL;
		
		if (r) {
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpRejectedMail,
				      "%S server rejected MAIL with %3.3s %s"),
			      smtp_Server((*C)->head->p.smtp),
			      response_code,r);
		    
		    if ((*C)->msd_sd) 
			message = format_string(FRM("%3.3s %s"),
						response_code,r);

		    smtp_print_response_tail((*C)->head->p.smtp,&idx);

		} else {
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpRejectedMail0,
				      "%S server rejected MAIL with code %3.3s"),
			      smtp_Server((*C)->head->p.smtp),
			      response_code);

		    if ((*C)->msd_sd) 
			message = format_string(FRM("%3.3s"),
						response_code);		   
		}
		
		if ((*C)->msd_sd) {
		    if (ticket) 
			action = 
			    (*C)->msd_addr_check_cb((*C)->msd_sd,-1,
						    from_message,
					      msd_addr_response(response_code),
						    message,
						    msd_mail_failed);
		    else
			(*C)->msd_addr_upd_cb((*C)->msd_sd,-1,
					      from_message,
					      msd_addr_response(response_code),
					      message);
		}		

		if (message)
		    free_string(&message);

		translate_response(response_code,&Exit_stat);
	    } else {
		lib_error(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpConnectionFailMail,
				  "Connection to %S server failed after MAIL"),
			  Server_s(((*C)->head->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));
		
		smtp_command_reset((*C)->head->p.smtp);

		goto mail_connection_failure;
	    }

	    if (ticket && 
		smtp_pass_ticket(M,*C,ticket,action)) {
		DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
	    }

	    free(from); from = NULL;
	
	    smtp_command_reset((*C)->head->p.smtp);
	    goto cancel_action;
	} /* end command failed */

    } else {

	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpConnectionFailed,
			  "Connection to %S server failed"),
		  Server_s(((*C)->head->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));

	mail_connection_failure:
#ifdef  EX_TEMPFAIL
	Exit_stat = EX_TEMPFAIL;
#endif
	
	if ((*C)->msd_sd) {
	    /* Not actually data phase */
	    if (ticket)  {
		action = 
		    (*C)->msd_fail_check_cb((*C)->msd_sd,
					    msd_mail_tmpfail,
					    NULL,
					    msd_mail_failed);
			
		if (msd_act_send == action) {
		    /* send after reconnect */
		    
		    DPRINT(Debug,4, (&Debug,
				     "  >> msd_act_send >> msd_act_need_restart\n"));

		    action = msd_act_need_restart;
		}
	    } else
		(*C)->msd_data_upd_cb((*C)->msd_sd,msd_mail_tmpfail,
				      NULL);
	}

	if (ticket && 
	    smtp_pass_ticket(M,*C,ticket,action)) {
	    DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
	}

	free(from); from = NULL;

	goto cancel_action;
    }
    smtp_command_reset((*C)->head->p.smtp);

    free(from); from = NULL;

    if ((*C)->addrs) {
	int i;

	for (i = 0; (*C)->addrs[i]; i++) {
	    char * R = (*C)->addrs[i];

	    struct string * rcpt_message = NULL;

	    if ((*C)->msd_sd)
		rcpt_message = new_string2(system_charset,
					   s2us((*C)->addrs[i]));

	    /* Make sure that addresses are inside of <> */
	    if ('<' != (*C)->addrs[i][0])
		R = kludge_addr( &((*C)->addrs[i]));

	    if (smtp_start_command((*C)->head->p.smtp,"RCPT")) {
		char response_code[4];
		char * rcpt = safe_strdup("TO:");

		if ((*C)->msd_sd)
		    (*C)->msd_addr_upd_cb((*C)->msd_sd,i,rcpt_message,
					  msd_sending_addr,NULL);

		rcpt = strmcat(rcpt,R);
		smtp_command_arg((*C)->head->p.smtp,rcpt);
		free(rcpt); rcpt = NULL;

		if (have_dsn) {
		    if ((*C)->dsn & DSN_NEVER)
			smtp_command_arg((*C)->head->p.smtp,"NOTIFY=NEVER");
		    
		    else if ((*C)->dsn & (DSN_SUCCESS|DSN_FAILURE|DSN_DELAY)) {
			char * A = safe_strdup("NOTIFY=");
			int x = 0;
			if ((*C)->dsn & DSN_SUCCESS) {
			    A = strmcat(A,"SUCCESS"); x++;
			}
			if ((*C)->dsn & DSN_FAILURE) {
			    if (x) A = strmcat(A,",");
			    A = strmcat(A,"FAILURE"); x++;
			}
			if ((*C)->dsn & DSN_DELAY) {
			    if (x) A = strmcat(A,",");
			    A = strmcat(A,"DELAY"); 
			}
			smtp_command_arg((*C)->head->p.smtp,A);
			free(A); A = NULL;
		    }
		}

		smtp_command_ready((*C)->head->p.smtp,0);
		if (smtp_command_ok((*C)->head->p.smtp,response_code)) {

		    if ((*C)->msd_sd) {

			struct string * message = 
			    message_helper((*C)->head->p.smtp,response_code);
			
			(*C)->msd_addr_upd_cb((*C)->msd_sd,i,
					      rcpt_message,
					      msd_addr_response(response_code),
					      message);

			free_string(&message);
			
		    }

		} else { /* command failed */

		    if (response_code[0]) {
			int idx = 0;
			const char * r = 
			    smtp_command_response((*C)->head->p.smtp,&idx);
			
			struct string * message = NULL;

			if (r) {
			    lib_error(CATGETS(elm_msg_cat, SmtpSet,
					      SmtpRejectedCommand,
					      "%S server rejected %s with %3.3s %s"),
				      smtp_Server((*C)->head->p.smtp),
				      (*C)->addrs[i],
				      response_code,r);

			    if ((*C)->msd_sd) 
				message = format_string(FRM("%3.3s %s"),
							response_code,r);

			    smtp_print_response_tail((*C)->head->p.smtp,&idx);
			} else {
			    lib_error(CATGETS(elm_msg_cat, SmtpSet,
					      SmtpRejectedCommand0,
					      
					      "%S server rejected %s with code %3.3s"),
				      smtp_Server((*C)->head->p.smtp),
				      (*C)->addrs[i],
				      response_code);
			    if ((*C)->msd_sd) 
				message = format_string(FRM("%3.3s"),
							response_code);	
			}

			if ((*C)->msd_sd) {
			    if (ticket) {
				enum msd_can_continue can_continue = 
				    msd_address_problem;
				
				if (i == (*C)->addr_count-1 &&
				    fail >= i) {
				    
				    DPRINT(Debug,8,(&Debug, 
						    " -- last address failed"));
				    if (i) {
					DPRINT(Debug,8,(&Debug, 
							"and also previous (%d) addresses are failed",
							fail));
				    }
				    
				    DPRINT(Debug,8,(&Debug,", %d addresses\n",(*C)->addr_count));

				    can_continue = msd_mail_failed;
				}

				action = 
				    (*C)->msd_addr_check_cb((*C)->msd_sd,i,
							    rcpt_message,
					     msd_addr_response(response_code),
							    message,
							    can_continue);
			    } else
				(*C)->msd_addr_upd_cb((*C)->msd_sd,i,
						      rcpt_message,
					     msd_addr_response(response_code),
						      message);
			}

			if (message)
			    free_string(&message);

			translate_response(response_code,&Exit_stat);
		    } else {
			lib_error(CATGETS(elm_msg_cat, SmtpSet,
					  SmtpConnectionFailRCPT,
					  "Connection to %S server failed after RCPT"),
				  Server_s(((*C)->head->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));
			
			smtp_command_reset((*C)->head->p.smtp);			
			goto rcpt_connection_failure;
		    }
		    fail++;

		    if (ticket && 
			smtp_pass_ticket(M,*C,ticket,action)) {
			DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));

			if (rcpt_message)
			    free_string(&rcpt_message);

			smtp_command_reset((*C)->head->p.smtp);
			goto cancel_action;
		    } else if ((*C)->msd_sd) { 
			switch (action) {
			case msd_act_EOF:
			case msd_act_SIG:
			case msd_act_need_restart:
			case msd_act_cancel:
			case msd_act_edit:
			    
			    DPRINT(Debug,4,(&Debug, " -- cancel request -- no ticket\n"));

			    if (rcpt_message)
				free_string(&rcpt_message);

			    smtp_command_reset((*C)->head->p.smtp);
			    goto cancel_action;
			case msd_act_default:
			case msd_act_send:
			    break;
			}
		    }
		    	
		} /* end command failed */

	    } else {
		lib_error(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpConnectionFailed,
				  "Connection to %S server failed"),
			  Server_s(((*C)->head->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));

	    rcpt_connection_failure:
		fail++;

#ifdef  EX_TEMPFAIL
		Exit_stat = EX_TEMPFAIL;
#endif
	
		if ((*C)->msd_sd) {
		    /* Not actually data phase */
		    if (ticket)  {
			action = 
			    (*C)->msd_fail_check_cb((*C)->msd_sd,
						    msd_mail_tmpfail,
						    NULL,
						    msd_mail_failed);
			
			if (msd_act_send == action) {
			    /* send after reconnect */
			    
			    DPRINT(Debug,4, (&Debug,
					     "  >> msd_act_send >> msd_act_need_restart\n"));
			    
			    action = msd_act_need_restart;
			}
		    } else
			(*C)->msd_data_upd_cb((*C)->msd_sd,msd_mail_tmpfail,
					      NULL);
		}
	
		if (ticket && 
		    smtp_pass_ticket(M,*C,ticket,action)) {
		    DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
		}
	
		if (rcpt_message)
		    free_string(&rcpt_message);
		
		free_mail_send_state(C);
		ret = 0; /* FAIL */
		goto fail;       
	    }
	    if (rcpt_message)
		free_string(&rcpt_message);

	    smtp_command_reset((*C)->head->p.smtp);
	}

	if (fail) {
	    lib_error(CATGETS(elm_msg_cat, SmtpSet,
			      SmtpRcptsFailed,
			      "%d recipients from %d failed"),
		      fail,i);

	    if ((*C)->msd_sd)  {
		enum msd_data_status  mail_status =
		    msd_datastat_none;

		if (fail >= i)
		    mail_status = msd_mail_fail;

		data_message = 
		    format_string(CATGETS(elm_msg_cat, SmtpSet,
					  SmtpRcptsFailed,
					  "%d recipients from %d failed"),
				  fail,i);
		
		if (ticket) {
		    
		    enum msd_can_continue can_continue = 
			msd_address_problem;
		    
		    if (fail >= i) {
			DPRINT(Debug,8,(&Debug, 
					" .. all (%d) addresses are failed, %d addresses\n",
					fail,(*C)->addr_count));
			can_continue = msd_mail_failed;
		    }
		    
		    action = 
			(*C)->msd_fail_check_cb((*C)->msd_sd,
						mail_status,
						data_message,
						can_continue);
		} else
		    (*C)->msd_data_upd_cb((*C)->msd_sd,
					  mail_status,
					  data_message);    
	    }

	    if (ticket && 
		smtp_pass_ticket(M,*C,ticket,action)) {
		DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));

		goto cancel_action;
	    } else if ((*C)->msd_sd) { 
		switch (action) {
		case msd_act_EOF:
		case msd_act_SIG:
		case msd_act_need_restart:
		case msd_act_cancel:
		case msd_act_edit:
		    
		    DPRINT(Debug,4,(&Debug, " -- cancel request -- no ticket\n"));		    		    
		    goto cancel_action;
		case msd_act_default:
		case msd_act_send:
		    break;
		}
	    }

	    if (fail >= i) 	       
		goto cancel_action;
	}

    } else {
	DPRINT(Debug,13,(&Debug,
			 "smtp_common_mailer_backend: No recipient addresses...\n"));
	
	goto cancel_action;
    }


    if ((*C)->msd_sd) 
	(*C)->msd_data_upd_cb((*C)->msd_sd,msd_sending_data,
			      data_message);
	
    if (smtp_start_command((*C)->head->p.smtp,"DATA")) {
	char response_code[4];

	if ((*C)->msd_sd)
	    (*C)->msd_data_upd_cb((*C)->msd_sd,
				  msd_datastat_none,
				  data_message);  	    	    
       	
	smtp_command_ready((*C)->head->p.smtp,0);

	if (smtp_command_ok((*C)->head->p.smtp,response_code)) {

	    if ((*C)->msd_sd) {
		struct string * message =
		    data_message ?
		    data_message :
		    message_helper((*C)->head->p.smtp,response_code);

		(*C)->msd_data_upd_cb((*C)->msd_sd,
				      msd_data_response(response_code),
				      message);

		if (message != data_message)
		    free_string(&message);

	    }

	} else { /* command failed */
	    	    
	    if (response_code[0]) {

		int idx = 0;
		const char * r = 
		    smtp_command_response((*C)->head->p.smtp,&idx);
		
		struct string * message = NULL;

		if (r) {
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpRejectedData,
				      "%S server rejected DATA with %3.3s %s"),
			      smtp_Server((*C)->head->p.smtp),
			      response_code,r);

		    if ((*C)->msd_sd) 
			message = format_string(FRM("%3.3s %s"),
						response_code,r);

		    smtp_print_response_tail((*C)->head->p.smtp,&idx);
		} else {
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpRejectedData0,
				      "%s server rejected DATA with code %3.3s"),
			      smtp_Server((*C)->head->p.smtp),
			      response_code);

		    if ((*C)->msd_sd) 
			message = format_string(FRM("%3.3s"),
						response_code);	
		    
		}

		if ((*C)->msd_sd) {
		    if (ticket) 
			action = 
			    (*C)->msd_fail_check_cb((*C)->msd_sd,
						    msd_data_response(response_code),
						    message,
						    msd_mail_failed);
		    else
			(*C)->msd_data_upd_cb((*C)->msd_sd,
					      msd_data_response(response_code),
					      message);
		    
		}

		if (message)
		    free_string(&message);

		translate_response(response_code,&Exit_stat);

	    } else {
		lib_error(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpConnectionFailData,
				  "Connection to %S server failed after DATA"),
			  Server_s(((*C)->head->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s));
		
		smtp_command_reset((*C)->head->p.smtp);

		goto data_connection_failure;		
	    }
	    
	    if (ticket && 
		smtp_pass_ticket(M,*C,ticket,action)) {
		DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
	    }

	    goto cancel_action;
	} /* end command failed */

    } else {

	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpConnectionFailed,
			  "Connection to %S server failed"),
		  smtp_Server((*C)->head->p.smtp));

    data_connection_failure:
	
#ifdef  EX_TEMPFAIL
	if (-1 == Exit_stat)
	    Exit_stat = EX_TEMPFAIL;
#endif
	if ((*C)->msd_sd) {
	    if (ticket)  {
		action = 
		    (*C)->msd_fail_check_cb((*C)->msd_sd,
					    msd_mail_tmpfail,
					    data_message,
					    msd_mail_failed);
		
		if (msd_act_send == action) {
		    /* send after reconnect */
		    
		    DPRINT(Debug,4, (&Debug,
				     "  >> msd_act_send >> msd_act_need_restart\n"));
		    
		    action = msd_act_need_restart;
		}
	    } else
		(*C)->msd_data_upd_cb((*C)->msd_sd,msd_mail_tmpfail,
				      data_message);
	}
		
	if (ticket && 
	    smtp_pass_ticket(M,*C,ticket,action)) {
	    DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
	}

	goto cancel_action;	
    }

    free_response_text((*C)->head->p.smtp);
    out_state_fseek ((*C)->OUT, 0);

    /* may be incorrect ... 
     * (but here should not be other \r or \n than \r\n) 
     */
    
    while (0 < (l = mail_gets(buffer,sizeof buffer,
				  out_state_FILE((*C)->OUT)))) {
	
	if (!smtp_push_data((*C)->head->p.smtp,buffer,l,1)) 
	    goto hard_failure;	
    }
    
    if (smtp_push_data((*C)->head->p.smtp,".",1,0)) {
	char response_code[4];
	smtp_command_ready((*C)->head->p.smtp,1);   /* End data */

	if (smtp_command_ok((*C)->head->p.smtp,response_code)) {	    

	    if ((*C)->msd_sd) {
		struct string * message =
		    data_message ?
		    data_message :
		    message_helper((*C)->head->p.smtp,response_code);

		(*C)->msd_data_upd_cb((*C)->msd_sd,
				      msd_data_response(response_code),
				      message);

		if (message != data_message)
		    free_string(&message);

	    }

	} else {  /* command failed */
	    if (response_code[0]) {
		int idx = 0;
		const char * r = 
		    smtp_command_response((*C)->head->p.smtp,&idx);

		struct string * message = NULL;

		if (r) {
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpRejectedMss,
				      "%S server rejected message with %3.3s %s"),
			      smtp_Server((*C)->head->p.smtp),
			      response_code,r);

		    if ((*C)->msd_sd) 
			message = format_string(FRM("%3.3s %s"),
						response_code,r);

		    smtp_print_response_tail((*C)->head->p.smtp,&idx);
		} else {
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpRejectedMss0,
				      "%S server rejected message with code %3.3s"),
			      smtp_Server((*C)->head->p.smtp),
			      response_code);
		     if ((*C)->msd_sd) 
			message = format_string(FRM("%3.3s"),
						response_code);
		}

		if ((*C)->msd_sd) {
		    if (ticket) 
			action = 
			    (*C)->msd_fail_check_cb((*C)->msd_sd,
						    msd_data_response(response_code),
						    message,
						    msd_mail_failed);
		    else
			(*C)->msd_data_upd_cb((*C)->msd_sd,
					      msd_data_response(response_code),
					      message);
		    
		}

		if (message)
		    free_string(&message);


		translate_response(response_code,&Exit_stat);	      

	    } else {

		lib_error(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpConnectionFailMss,
				  "Connection to %S server failed after message"),
			  smtp_Server((*C)->head->p.smtp));
				
		goto dot_connection_failure;		
	    }

	    if (ticket && 
		smtp_pass_ticket(M,*C,ticket,action)) {
		DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
	    }
	    
	    goto cancel_action;
	} /* end command failed */

	{ 
	    union any_fd FD;
	    struct run_state RS;

	    FD.mail_fd         = *C;
	    RS.save_errno      = 0;

	    /* Fake ... */
	    (*C)->orig_func(FD,title,&RS,1,0);
	}
    } else {
    hard_failure:
	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpConnectionFailData2,
			  "Connection to %S server failed during DATA"),
		  smtp_Server((*C)->head->p.smtp));
	
	if ((*C)->head->p.smtp->RA.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_common_mailer_backend3: Closing stream\n"));
	    
	    FreeStreamStack2(&((*C)->head->p.smtp->RA.stream),
			     1 /* Force clearing of stack even when this is not last reference */);
	}

    dot_connection_failure:

#ifdef  EX_TEMPFAIL
	if (-1 == Exit_stat)
	    Exit_stat = EX_TEMPFAIL;
#endif
	
	if ((*C)->msd_sd) {
	    if (ticket)  {
		action = 
		    (*C)->msd_fail_check_cb((*C)->msd_sd,
					    msd_mail_tmpfail,
					    data_message,
					    msd_mail_failed);
		
		if (msd_act_send == action) {
		    /* send after reconnect */
		    
		    DPRINT(Debug,4, (&Debug,
				     "  >> msd_act_send >> msd_act_need_restart\n"));
		    
		    action = msd_act_need_restart;
		}
	    } else
		(*C)->msd_data_upd_cb((*C)->msd_sd,msd_mail_tmpfail,
				      data_message);
	}				
	
	if (ticket && 
	    smtp_pass_ticket(M,*C,ticket,action)) {
	    DPRINT(Debug,4,(&Debug, " -- passed mail to ticket\n"));
	}
	
	goto cancel_action;     
    } 
       
    if (-1 == Exit_stat && 0 == fail)
	Exit_stat = 0;   /* OK */
    ret = 1;         /* OK */

    if (!ret) {
    cancel_action:
	if (smtp_start_command((*C)->head->p.smtp,"RSET")) {
	    smtp_command_ready((*C)->head->p.smtp,0);
	    smtp_command_reset((*C)->head->p.smtp);
	}
	
	ret = 0; /* backend failed */
    }
    free_mail_send_state(C);

 fail:
    if (data_message)
	free_string(&data_message);
    if (from_message)
	free_string(&from_message);


    if (exit_status && Exit_stat >= 0)
	*exit_status = Exit_stat;

    return ret;
}



S_(mailer_init_hook smtp_common_mailer_init)
static int smtp_common_mailer_init P_((struct mailer_config *M,
				       struct mail_send_state *C,
				       struct mailer_info *I));
static int smtp_common_mailer_init(M,C,I)
     struct mailer_config *M;
     struct mail_send_state *C;
     struct mailer_info *I;
{
    FILE * F;
    int err = 0;
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);

    if (C->magic != MAILER_magic)
    	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_common_mailer_init",
	      "Bad magic number (mail send state)",0);

    if (!I) {
	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpMailerInfo,
			  "SMTP mailer requires info parameter"));
	return 0;
    }
    
    if (!tmp) 
	return 0;
    

    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_common_mailer_init",
	      "Bad magic number (smtp mailer info)",0);

    if (I->p.smtp->SMTP_flags & SMTP_reserve) {
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "smtp_common_mailer_init",
	      "SMTP transaction already on progress",0);
    }
	
    I->p.smtp->SMTP_flags |= SMTP_reserve;


    if (smtp_start_command(I->p.smtp,"NOOP")) {
	smtp_command_ready(I->p.smtp,0);
	smtp_command_reset(I->p.smtp);
    }

    
    C->fname = elm_message(FRM("%selm.sndsm-%d"), 
			   tmp, getpid());

    if (!C->fname) {
	DPRINT(Debug,1,(&Debug,   
			"couldn't make temp file nam! (mail)\n"));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldNotMakeTemp,
			  "Sorry - couldn't make temp file name."));
	return 0;
    }

    F = safeopen_rdwr(C->fname,&err);

    if (!F) {
	int err = errno;
	DPRINT(Debug,1, (&Debug,
			 "Attempt to open file %s for writing failed! (write_header_info)\n",
			 C->fname));
	DPRINT(Debug,1, (&Debug, "** %s **\n\n", strerror(err)));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorTryingToWrite,
			  "Error %s encountered trying to write to %s."), 
		  strerror(err), C->fname);
	
	return 0;
    }

    if (C->OUT)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_common_mailer_init",
	      "out_state already set (mail send state)",0);

    C->OUT = new_out_state(STATE_out_file);
    set_out_state_file(F,C->OUT);

    /* SMTP wants CRLF as eoln */
    set_out_state_EOLN(C->OUT,1);

    return 1;
}


S_(mailer_close_hook smtp_common_mailer_close)
static void smtp_common_mailer_close P_((struct mailer_config *M,
					 struct mail_send_state *C));
static void smtp_common_mailer_close(M,C)
     struct mailer_config *M;
     struct mail_send_state *C;
{

    if (! C->head)
	return;   /* Oh no ... */

    if (SMTP_mailer_info_magic != C->head->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_common_mailer_close",
	      "Bad magic number (smtp mailer info)",0);

    switch (C->head->p.smtp->state) {
    case smtp_need_data:
    case smtp_data:
	DPRINT(Debug,13,(&Debug,
			 "smtp_common_mailer_close: OH NO -- middle of data transfer\n"));
	C->head->p.smtp->state = smtp_error;
	break;
    default:
	break;
    }

    if (smtp_start_command(C->head->p.smtp,"RSET")) {
	smtp_command_ready(C->head->p.smtp,0);
	smtp_command_reset(C->head->p.smtp);
    }

    /* C->fname is free'ed by caller ( free_mail_send_state() ) */

    if (C->OUT) {
	FILE *F = out_state_FILE(C->OUT);
	
	free_out_state(& (C->OUT));
	if (F) {
	    fclose(F);
	}
    }

    C->head->p.smtp->SMTP_flags &= ~SMTP_reserve;
}


/* Result is malloced */
static char ** smtp_mailer_con_name_local P_((size_t                * reslen_p,
					      struct mailer_config  * M,
					      struct mailer_info    * I,
					      struct cancel_data   ** cancel_p));
static char ** smtp_mailer_con_name_local(reslen_p,M,I,cancel_p)
     size_t                * reslen_p;
     struct mailer_config *M;
     struct mailer_info *I;
     struct cancel_data   ** cancel_p;  /* currently not used */
{
    char ** ret = NULL;
    
    if (I->p.smtp->RA.socknamesize && I->p.smtp->RA.socknameaddr.dummy) {
	union SOCKADDR_ptr socknameaddr = I->p.smtp->RA.socknameaddr;
	size_t             socknamesize =  I->p.smtp->RA.socknamesize;


	if (give_name_from_sockaddr(socknameaddr,socknamesize,
				    &ret,reslen_p,cancel_p,
				    no_lookup_message, NULL /* No error message */
				    )) {

	    DPRINT(Debug,12,(&Debug,
			     "smtp_mailer_con_name_local: Succeed\n"));
	} else if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_mailer_con_name_local: Canceled\n"));
	} else {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_mailer_con_name_local: Failed\n"));
	}
    }

    DPRINT(Debug,12,(&Debug,
		     "smtp_mailer_con_name_local="));

    if (ret) {
	size_t i;
	
	DPRINT(Debug,12,(&Debug,"%p",ret));

	for (i = 0; i < *reslen_p; i++) {
	    DPRINT(Debug,12,(&Debug,
			     "%c %Q",
			     i ? ',' : ':',
			     ret[i]));

	}
		    
    } else {
	DPRINT(Debug,12,(&Debug,"NULL"));
    }
    
    DPRINT(Debug,12,(&Debug,"; *reslen_p=%zu\n",
		     *reslen_p));

    return ret;
}

static int  smtp_mailer_is_remote P_((struct mailer_config *M,
				      struct mailer_info *I));
static int  smtp_mailer_is_remote(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{
    int ret = -1;  

    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_mailer_is_remote",
	      "Bad magic number (smtp mailer info)",0);

    if (I->p.smtp->RA.hostname) {
    
	if (0 == istrcmp(I->p.smtp->RA.hostname,"localhost")) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_mailer_is_remote: is local mailer, hostname=%s\n",
			     I->p.smtp->RA.hostname
			     ));
	    
	    ret = 0;
	    
	}
       
    } else {
	DPRINT(Debug,12,(&Debug,
			 "submission_mailer_info_remote_hook: (no hostname)\n"));
    }

    if (I->p.smtp->RA.host) {
    
	if (0 == istrcmp(I->p.smtp->RA.host,"localhost")) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_mailer_is_remote: is local mailer, host=%s\n",
			     I->p.smtp->RA.host
			     ));
	    
	    ret = 0;	    
	}
       
    } else {
	DPRINT(Debug,12,(&Debug,
			 "smtp_mailer_is_remote: (no host)\n"));
    }

    
    if (I->p.smtp->RA.addrsize && I->p.smtp->RA.hostaddr.dummy) {

        int query_flags = INTERFACE_ADDRLIST;
        int interfaces_count = 0;
	
        struct interface_info ** interfaces = 
            get_interfaces(sizeof (struct interface_info),
                           sizeof (struct interface_addr),
                           &interfaces_count,&query_flags);

	char mybuffer2[256];                 
	const char * s2 UNUSED_VAROK
	    =  give_SOCKADDR_ptr_as_string(I->p.smtp->RA.hostaddr,
					   I->p.smtp->RA.addrsize,
					   mybuffer2,sizeof mybuffer2);
	      	
	if (s2) {
	    DPRINT(Debug,12,(&Debug,
			     "smtp_mailer_is_remote: Connection have remote address %s\n",s2));
	}

	if (I->p.smtp->RA.socknamesize && I->p.smtp->RA.socknameaddr.dummy) {
	    char mybuffer3[256];                 
	    const char * s3 UNUSED_VAROK
		= give_SOCKADDR_ptr_as_string(I->p.smtp->RA.socknameaddr,
					      I->p.smtp->RA.socknamesize,
					      mybuffer3,sizeof mybuffer3);
	    
	    if (s3) {
		DPRINT(Debug,12,(&Debug,
				 "smtp_mailer_is_remote: Connection have local address %s\n",s3));
	    }

	    if (same_SOCKADDR_ptr(I->p.smtp->RA.hostaddr,
				  I->p.smtp->RA.addrsize,
				  I->p.smtp->RA.socknameaddr,
				  I->p.smtp->RA.socknamesize,
				  1 /* Ignore port */)) {
		DPRINT(Debug,12,(&Debug,
				 "smtp_mailer_is_remote: remote and local address matches, is local mailer\n"));
		ret = 0;
	    }
	}
	
        if (0 == (query_flags & INTERFACE_ADDRLIST)) {
            DPRINT(Debug,12,(&Debug,
			     "smtp_mailer_is_remote: Failed to get interface address list\n"));
        } 
	
	if (interfaces) {

	    int i_idx;
                
	    for (i_idx = 0; i_idx < interfaces_count; i_idx++) {
		
		if (interfaces[i_idx]) {
		    if (INTERFACE_INFO_magic != interfaces[i_idx]->magic)
			panic("RC PANIC",__FILE__,__LINE__,
			      "smtp_mailer_is_remote",
			      "Bad magic number (interface_info)",0);

		     if (interfaces[i_idx]->addr_list) {
                            int a_idx;
                            
                            for (a_idx = 0; a_idx < interfaces[i_idx]->addr_count;
                                 a_idx++) {
				
                                if (interfaces[i_idx]->addr_list[a_idx]) {
				    char mybuffer[256];                 
				    const char * s UNUSED_VAROK;
					

				    if (INTERFACE_ADDR_magic !=
                                        interfaces[i_idx]->addr_list[a_idx]->magic)
                                        panic("RC PANIC",__FILE__,__LINE__,
                                              "smtp_mailer_is_remote",
                                              "Bad magic number (interface_addr)",
                                              0);

				    s = give_SOCKADDR_ptr_as_string(interfaces[i_idx]->
								    addr_list[a_idx]->address,
								    interfaces[i_idx]->
								    addr_list[a_idx]->addrsize,
								    mybuffer,sizeof mybuffer);

				    
				    if (!  (interfaces[i_idx]->addr_list[a_idx]->valid_address)) {
                                        DPRINT(Debug,12,(&Debug,
                                                         "smtp_mailer_is_remote: interface \"%s\", address #%d %s skipped, not on address list.\n",
                                                         interfaces[i_idx]->name,a_idx,
							 s ? s : ""));
                                        continue;
                                    }


				    
				    if (same_SOCKADDR_ptr(I->p.smtp->RA.hostaddr,
							  I->p.smtp->RA.addrsize,
							  interfaces[i_idx]->addr_list[a_idx]->address,
							  interfaces[i_idx]->addr_list[a_idx]->addrsize,
							  1 /* Ignore port */
							  )) {
					
					DPRINT(Debug,12,(&Debug,
                                                         "smtp_mailer_is_remote: interface \"%s\", address #%d %s matches to connection, is local mailer\n",
							 interfaces[i_idx]->name,a_idx,
							 s ? s : "???"));

					ret = 0;
				    } else {
					DPRINT(Debug,14,(&Debug,
							 "smtp_mailer_is_remote: interface \"%s\", address #%d %s is not connection address %s\n",
							 interfaces[i_idx]->name,a_idx,
							 s ? s : "???",
							 s2 ? s2 : "???"));
				    }
				}
			    }
		     }
		}
	    }

	    if (ret < 0 && interfaces_count > 0) {
		DPRINT(Debug,12,(&Debug,
				 "submission_mailer_info_remote_hook: Assumed remote mailer, address=%s\n",
				 s2 ? s2 : "???"));
		ret = 1;
	    }	    
	}
    }

    if (ret < 0 && I->p.smtp->RA.hostname) {

	DPRINT(Debug,12,(&Debug,
			 "smtp_mailer_is_remote: Assumed remote mailer, hostname=%s\n",
			 I->p.smtp->RA.hostname));
	
	ret = 1;
    }

    if (ret < 0 && I->p.smtp->RA.host) {

	DPRINT(Debug,12,(&Debug,
			 "smtp_mailer_is_remote: Assumed remote mailer, host=%s\n",
			 I->p.smtp->RA.host));
	
	ret = 1;
    }

    if (ret < 0) {
	DPRINT(Debug,12,(&Debug,"smtp_mailer_is_remote: Assumed local mailer\n"));
	ret = 0;
    }
    
    DPRINT(Debug,12,(&Debug,"smtp_mailer_is_remote=%d\n",ret));
    
    return ret;
}

S_(mailer_info_s_query_hook smtp_minfo_s_query)
static char ** smtp_minfo_s_query P_((size_t                * reslen_p,
				     struct mailer_config  * M,
				     struct mailer_info    * I,
				     enum MI_query_string    query,
				     struct cancel_data   ** cancel_p));
static char ** smtp_minfo_s_query(reslen_p,M,I,query,cancel_p)
     size_t                * reslen_p;
     struct mailer_config *M;
     struct mailer_info *I;
     enum MI_query_string    query;
     struct cancel_data   ** cancel_p;
{
    *reslen_p = 0;
    
    switch (query) {
    case MI_con_name_local:
	return smtp_mailer_con_name_local(reslen_p,M,I,cancel_p);
    }
        
    return NULL; /* Not available */
}


S_(mailer_info_query_hook smtp_common_mailer_info_query)
static int smtp_common_mailer_info_query  P_((struct mailer_config *M,
					      struct mailer_info *I,
					      enum MI_query query));
static int smtp_common_mailer_info_query(M,I,query)
     struct mailer_config *M;
     struct mailer_info *I;
     enum MI_query query;
{
    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_common_mailer_info_query",
	      "Bad magic number (smtp mailer info)",0);

    if (-1 == I->p.smtp->ESMTP_keyword_len) {
	if (!smtp_run_ehlo(I->p.smtp)) {
	    DPRINT(Debug,10,(&Debug,"Some extension failed -- ignoring now\n"));
	}
    }

    switch(query) {
	int i;

    case MI_HAVE_8BITMIME:
	for (i = 0; i < I->p.smtp->ESMTP_keyword_len; i++)
	    if (0 == istrcmp("8BITMIME",I->p.smtp->ESMTP_keywords[i]))
		return 1;
	DPRINT(Debug,10,(&Debug,"8BITMIME not found from EHLO response\n"));
	return 0;
    case MI_HAVE_BINARYMIME:
	return 0;       /* This code does not support BINARYMIME */
    case MI_HAVE_DSN:
	for (i = 0; i < I->p.smtp->ESMTP_keyword_len; i++)
	    if (0 == istrcmp("DSN",I->p.smtp->ESMTP_keywords[i]))
		return 1;
	DPRINT(Debug,10,(&Debug,"DSN not found from EHLO response\n"));
	return 0;
    case MI_DONT_ADD_FROM:
	return (M -> mailer_bits & MB_DONT_ADD_FROM) ? 1 : 0;
    case MI_USE_DOMAIN:
	return (M -> mailer_bits & MB_USE_DOMAIN) ? 1 : 0;
    case MI_HAVE_SENDING_DISPLAY:
	return (M -> mailer_bits & MB_SENDING_DISPLAY) ? 1 : 0;
    case MI_REMOTE_MAILER:
	return smtp_mailer_is_remote(M,I);
	
    }

    return 0; /* Not available */
}

static int smtp_need_skip_enchanced P_((struct mailer_info  *I));
static int smtp_need_skip_enchanced(I)
     struct mailer_info  *I;
{
    int i;
    int skip_enchanced = 0;

     if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_need_skip_enchanced",
	      "Bad magic number (smtp mailer info)",0);

    if (-1 == I->p.smtp->ESMTP_keyword_len) {

	if (!smtp_run_ehlo(I->p.smtp)) {
	    /* Some extension failed hard */
	    skip_enchanced = -1;
	    goto done;
	}
    }


     
    for (i = 0; i < I->p.smtp->ESMTP_keyword_len; i++)
	if (0 == istrcmp("ENHANCEDSTATUSCODES",I->p.smtp->ESMTP_keywords[i]))
	    skip_enchanced = 1;

 done:
    DPRINT(Debug,14,(&Debug,"smtp_need_skip_enchanced=%d%s\n",
		     skip_enchanced,
		     skip_enchanced < 0 ? " (failure)" : ""
		     ));
    
    return skip_enchanced;
}

static struct addr_list * smtp_parse_vrfy_response P_((struct mailer_info  *I,
						       int skip_enchanced));
static struct addr_list * smtp_parse_vrfy_response(I,skip_enchanced)
     struct mailer_info  *I;
     int skip_enchanced;
{
    int idx;
    const char * text = NULL;
    const char * buffer = NULL;
    struct addr_list * X = NULL;

    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_parse_vrfy_response",
	      "Bad magic number (smtp mailer info)",0);
    
    for (idx = 0; NULL != (text = smtp_command_response(I->p.smtp,
							&idx));) {
	buffer = text;
    }

    DPRINT(Debug,13,(&Debug," %d verify responses, result=%s\n",
				 idx,buffer ? buffer : "<NULL>"));

    /* If there was exactly one answer we can use it */
    if (1 == idx) {
	char *x;
	
	if (skip_enchanced && buffer[0] == '2' &&
	    (x = strchr(buffer,' '))) {
	    buffer = x+1;
	    DPRINT(Debug,13,(&Debug," strimmed result=%s\n",
			     buffer));
	}
	
	/* FIXME: Posibble wrong charset */			
	X =  parse_header_address(NULL,buffer,
				  0,system_charset,NULL);
    }
    
    return X;
}


S_(mailer_info_verify_addr smtp_tail_verify_addr)
static enum mi_vrfyaddr_mailer_status
    smtp_tail_verify_addr P_((struct mailer_config *M,
			      struct mailer_info  *I, 
			      const char *text,
			      struct addr_verify_result *result));
static enum mi_vrfyaddr_mailer_status smtp_tail_verify_addr(M,I,text,result)
     struct mailer_config *M;
     struct mailer_info  *I;
     const char *text;
     struct addr_verify_result *result;
{
    int skip_enchanced = 0;
    enum mi_vrfyaddr_mailer_status ret = mi_vrfyaddr_default_ok;
   
    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_tail_verify_addr",
	      "Bad magic number (smtp mailer info)",0);

    skip_enchanced =  smtp_need_skip_enchanced(I);
    if (skip_enchanced < 0) {
	DPRINT(Debug,13,(&Debug,
			 "-- Some extensions failed hard -- do not verify address\n"));
	ret = mi_vrfyaddr_test_passwd;   /* caller does job */
	goto done;
    }
    
    if (I->p.smtp->SMTP_flags & SMTP_noverify) {
	DPRINT(Debug,13,(&Debug, "-- Verify temporary disabled\n"));
	ret =  mi_vrfyaddr_test_passwd;       /* Cannot VRFY user */
	goto done;
    }

    
    if (skip_enchanced) {
	DPRINT(Debug,13,(&Debug,
			 "-- Need skip enchanced status codes from responses\n"));
    }

    if (smtp_start_command(I->p.smtp,"VRFY")) {
	enum mi_vrfyaddr_mailer_status succeed = mi_vrfyaddr_failure;
	char response_code[4];

	smtp_command_arg(I->p.smtp,text);
	smtp_command_ready(I->p.smtp,0);

	if (smtp_command_ok(I->p.smtp,response_code)) {
	    if (0 == memcmp(response_code,"252",3)) {
		DPRINT(Debug,13,(&Debug,
				 " verify response %.3s -- cannot verify user\n",
				 response_code));
		succeed = mi_vrfyaddr_test_passwd;       /* Cannot VRFY user (relaying?) */
	    } else if (0 == memcmp(response_code,"250",3)) {
		struct addr_list * X =
		    smtp_parse_vrfy_response(I,
					     skip_enchanced);
		
		if (X) {
		    int len = addr_list_item_count(X);
		    
		    if (1 == len) {
			int group = -1;
			const struct address * address =
			    addr_list_get_item(X,0,&group);
			
			const char          * addr     = 
			    address_get_ascii_addr(address);
			const struct string * fullname = 
			    address_get_phrase(address);
			const struct string * comment  = 
			    address_get_comment(address);
			
			if (addr) {
			    result->addr    = safe_strdup(addr);
			    
			    result->fullname = 
				fullname ? dup_string(fullname) : NULL;
			    result->comment  =
				comment ? dup_string(comment) : NULL;
			    
			    succeed = result->fullname ? mi_vrfyaddr_success : mi_vrfyaddr_default_ok;
			    DPRINT(Debug,13,(&Debug," filled addr=%s\n",
					     result->addr));
			} else
			    succeed = mi_vrfyaddr_default_ok;   /* User OK, but caller should fill data */
		    } else
			succeed = mi_vrfyaddr_default_ok;   /* User OK, but caller should fill data */

		    free_addr_list(&X);
		} else
		    succeed = mi_vrfyaddr_default_ok;   /* User OK, but caller should fill data */
	    }
	    
	} else {
	    /* Failed -- see if response code indicates user unknown */
	    if (0 == memcmp(response_code,"553",3) /* Ambiguous */ ||
		0 == memcmp(response_code,"550",3) /* mailbox unavailable */
		) { 
		succeed = mi_vrfyaddr_failure;		
	    } else if ('4' == response_code[0])
		succeed = mi_vrfyaddr_test_passwd;       /* Cannot VRFY user (temporray error) */
	    else {
		succeed = mi_vrfyaddr_test_passwd;       /* Cannot VRFY user ... */
		DPRINT(Debug,13,(&Debug, "-- Disabling verify...\n"));
		I->p.smtp->SMTP_flags |= SMTP_noverify;
	    }
	}

	smtp_command_reset(I->p.smtp);	
	DPRINT(Debug,13,(&Debug, 
			 "smtp_tail_verify_addr (%s) = %d\n",
			 text,succeed));
	ret = succeed;
	goto done;
    }
    ret = mi_vrfyaddr_test_passwd;   /* caller does job */

 done:
    DPRINT(Debug,12,(&Debug, "smtp_tail_verify_addr=%d",
		     ret));
    switch (ret) {
    case mi_vrfyaddr_test_passwd: DPRINT(Debug,12,(&Debug, " mi_vrfyaddr_test_passwd")); break;
    case mi_vrfyaddr_default_ok:  DPRINT(Debug,12,(&Debug, " mi_vrfyaddr_default_ok"));  break;
    case mi_vrfyaddr_failure:     DPRINT(Debug,12,(&Debug, " mi_vrfyaddr_failure"));     break;
    case mi_vrfyaddr_success:     DPRINT(Debug,12,(&Debug, " mi_vrfyaddr_success"));     break;
    }
    DPRINT(Debug,12,(&Debug, "\n"));

    return ret;
}

static int smtp_set_mail_from P_((struct mailer_config *M,
				  struct mailer_info *I,
				  struct mailer_env_from *X,
				  const char *value,
				  int reset));
static int smtp_set_mail_from(M,I,X,value,reset)
     struct mailer_config *M;
     struct mailer_info   *I;
     struct mailer_env_from *X;
     const char *value;
     int reset;
{
    int verified = 0;
    int skip_enchanced = 0;
    char * rewritten_addr = NULL;
    int ret = 0;

    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"smtp_set_mail_from",
	      "Bad magic number (smtp mailer info)",0);

    
    if (0 == strcmp(value,"<>")) {
	/* Null envelope sender is only valid as sender (not as recipient)
	   so do not verify it
	*/
	goto succeed;
    }


    skip_enchanced =  smtp_need_skip_enchanced(I);
    if (skip_enchanced < 0) {
	DPRINT(Debug,4,(&Debug, "smtp_set_mail_from: some extension failed hard -- ignoring now\n"));
    }

    if (I->p.smtp->SMTP_flags & SMTP_noverify) {
	DPRINT(Debug,4,(&Debug, "smtp_set_mail_from: Verify temporary disabled - assuming ok \n"));
	goto succeed;       /* Cannot VRFY user */		
    }

    if (smtp_start_command(I->p.smtp,"VRFY")) {
	int succeed;
	char response_code[4];

	succeed  = 0;
	smtp_command_arg(I->p.smtp,value);
	smtp_command_ready(I->p.smtp,0);

	if (smtp_command_ok(I->p.smtp,response_code)) {
	    if (0 == memcmp(response_code,"252",3)) {
		/* relayed mail -- can't actually verify mail address */
		DPRINT(Debug,4,(&Debug,
				"smtp_set_mail_from: verify response %.3s -- cannot verify user --  assuming ok \n",
				response_code));
		succeed = 1; /* Cannot VRFY user (relaying?) */
	    } else if (0 == memcmp(response_code,"250",3)) {
		succeed = 1;
		verified = 1;
		
		DPRINT(Debug,4,(&Debug,
				"smtp_set_mail_from: verify response %.3s -- %s OK\n",
				response_code,value));
		
		if (skip_enchanced >= 0) {
		    struct addr_list * X =
			smtp_parse_vrfy_response(I,
						 skip_enchanced);
		    
		    if (X) {
			int len = addr_list_item_count(X);
			
			if (1 == len) {
			    int group = -1;
			    const struct address * address =
				addr_list_get_item(X,0,&group);
			    
			    const char          * addr     = 
				address_get_ascii_addr(address);

			    if (addr) {
				DPRINT(Debug,4,(&Debug,
						"smtp_set_mail_from: Address %s is really %s\n",
						value,addr));
				rewritten_addr = safe_strdup(addr);
			    }
			}

			free_addr_list(&X);
		    }
		}
		
	    } else {
		DPRINT(Debug,4,(&Debug,
				 "smtp_set_mail_from: verify response %.3s -- assuming ok\n",
				 response_code));
		succeed = 1; /* unexpect OK response */
	    }
	} else {
	    /* Failed -- see if response code indicates user unknown */
	    if (0 == memcmp(response_code,"553",3) /* Ambiguous */ ||
		0 == memcmp(response_code,"550",3) /* mailbox unavailable */
		) { 
		DPRINT(Debug,4,(&Debug,
				"smtp_set_mail_from: verify response %.3s -- cannot verify user -- failed\n",
				response_code));
		    goto failed;
	    } else if ('4' == response_code[0])
		DPRINT(Debug,4,(&Debug,
				"smtp_set_mail_from: verify response %.3s -- cannot verify user -- assuming ok \n",
				response_code));
	    else {		
		DPRINT(Debug,13,(&Debug, "smtp_set_mail_from: Disabling verify...\n"));
		I->p.smtp->SMTP_flags |= SMTP_noverify;
		DPRINT(Debug,4,(&Debug,
				"smtp_set_mail_from: verify response %.3s -- cannot verify user -- assuming ok \n",
				response_code));
	    }
	    succeed = 1;
	}

	if (!succeed && 0 == (X->flags & MAILER_ef_verified)) {
	    	DPRINT(Debug,4,(&Debug,
				"smtp_set_mail_from: previous envelope sender %s was not verified either ... ignoring VRFY error\n",
				X->mail_from ? X->mail_from : "(none)"));
	    
	    goto succeed;
	}
	
	if (succeed || reset) {
	succeed:
	    if (rewritten_addr) {		
		X->mail_from = strmcpy(X->mail_from,rewritten_addr);

		free(rewritten_addr);
		rewritten_addr = NULL;
	    } else
		X->mail_from = strmcpy(X->mail_from,value);

	    if (verified) {
		DPRINT(Debug,13,(&Debug, "smtp_set_mail_from: Setting MAILER_ef_verified\n"));
		X->flags  |= MAILER_ef_verified;
		
	    } else {
		DPRINT(Debug,13,(&Debug, "smtp_set_mail_from: Clearing MAILER_ef_verified\n"));
		X->flags &= ~MAILER_ef_verified;
	    }
	    ret = 1;
	} else {
	failed:
	    ret = 0;
	}
	smtp_command_reset(I->p.smtp);	
    } else {
	DPRINT(Debug,4,(&Debug,
			"smtp_set_mail_from failed to run verify command\n"));
	ret = 0;
    }


    DPRINT(Debug,13,(&Debug, "smtp_set_mail_from=%d",
		     ret));

    if (ret && X->mail_from) {
	DPRINT(Debug,13,(&Debug, "; mail_from=%s",
			 X->mail_from));
    }
    
    DPRINT(Debug,13,(&Debug, "\n"));
    return ret;
}


S_(mailer_info_gen_def_ef smtp_common_mailer_info_gen_def_ef)
static void smtp_common_mailer_info_gen_def_ef P_((struct mailer_config *M,
						   struct mailer_info *I,
						   struct mailer_env_from *X)); 
static void smtp_common_mailer_info_gen_def_ef(M,I,X)
     struct mailer_config *M;
     struct mailer_info *I;
     struct mailer_env_from *X; 
{
    smtp_set_mail_from(M,I,X,default_from(),1);
}

/* return 1 if X->mail_from changed */
S_(mailer_info_set_ef smtp_common_mailer_info_set_ef)
static int smtp_common_mailer_info_set_ef
P_((struct mailer_config *M,
    struct mailer_info *I,
    struct mailer_env_from *X,
    const char *value));
static int smtp_common_mailer_info_set_ef(M,I,X,value)
     struct mailer_config *M;
     struct mailer_info *I;
     struct mailer_env_from *X;
     const char *value;
{
    int r = smtp_set_mail_from(M,I,X,
			       value,0);
    
    if (! r) {
	
	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpEnvSenderFailed,
			  "Changing of envelope sender failed: %s"),
		  value);
    }

    return r;
}



/* submission mailer ------------------------------------------------------ */

#include <sys/socket.h>

static int submission_verify_tls_certificate = 0;

static char *submission_server = "localhost";

static char REQUIRE_SERVER_NAME;

static char * submission_peer_name = NULL;

static int  submission_verify_peer  P_((struct smtp_info *I,
					struct cancel_data *cd));
static int  submission_verify_peer(I,cd) 
     struct smtp_info *I;
     struct cancel_data *cd;   
{
    int ret = 0;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"submission_verify_peer",
	      "Bad magic number",0);

    /* May need run ehlo */
    if (-1 == I->ESMTP_keyword_len) {
	if (!smtp_run_ehlo_c(I,cd)) {
	    ret = 0; /* Some extension failed hard 
			or wait canceled
		      */
	    goto failed;
	}
    }

    if (submission_verify_tls_certificate) {
	int r = 0;
	StreamInfo(I->RA.stream,SS_verify,&r,NULL,NULL);

	if (r < 0) {
	    lib_error(CATGETS(elm_msg_cat, SmtpSet,
			      SmtpFailVerifyCert,
			      "Failed to verify %S server %s certificate (required)."),
		      smtp_Server(I),
		      I->RA.host);
	    ret = 0;
	    goto failed;
	} else if (0 == r) {
	    int bits;

	    StreamInfo(I->RA.stream,SS_ssf,&bits,NULL,NULL);

	    if (bits > 0)
		lib_error(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpCertNotVerf,
				  "%S server %s certificate not verified (required)."),
			  smtp_Server(I),
			  I->RA.host);
	    else
		lib_error(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpConnectionNotEnc,
				  "%S server %s connection not encrypted (certificate required)."),
			  smtp_Server(I),
			  I->RA.host);
	    ret = 0;
	    goto failed;			      
	}
    }


    if (submission_peer_name) {
	struct string * require_name = NULL;
	struct string * returned_name = NULL;
	int r;

	if (&REQUIRE_SERVER_NAME == submission_peer_name)
	    require_name = new_string2(system_charset,s2us(submission_server));
	else
	    require_name = new_string2(system_charset,
				       s2us(submission_peer_name));

	if (! (r = StreamVerifyName(I->RA.stream,SS_peer_cn,require_name,
				    & returned_name))) {

	    if (returned_name) {
		lib_error(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpCertCnBad,
				  "%S server %s certificate CN is %S, but %S required."),
			  smtp_Server(I),
			  I->RA.host,
			  returned_name,require_name);
	    } else {
		int bits;
		
		StreamInfo(I->RA.stream,SS_ssf,&bits,NULL,NULL);
		
		if (bits > 0)
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpCertCnReq,
				      "%S server %s certificate CN %S required."),
			      smtp_Server(I),
			      I->RA.host,
			      require_name);
		else
		    lib_error(CATGETS(elm_msg_cat, SmtpSet,
				      SmtpCertNotEncCnReq,
				      "%S server %s connection not encrypted, but certificate CN %S required."),
			      smtp_Server(I),
			      I->RA.host,
			      require_name);
	    }
	}
		
	if (returned_name)			       
	    free_string(&returned_name);
	free_string(&require_name);

	if (!r) {
	    ret = 0;
	    goto failed;
	}
    }

    ret = 1;

 failed:
    DPRINT(Debug,13,(&Debug,"submission_verify_peer=%d\n",
		     ret));

    return ret;
}

/* return 1 if cb success, 0 failure
   return -1 no connection
 */
S_(ra_verify_peer_cb_f submission_verify_peer_cb)
static int submission_verify_peer_cb P_((struct remote_account *ra,
					 union ra_verify_peer_cb_data *data,
					 const struct string * Server,
					 const struct string * server));
static int submission_verify_peer_cb(ra,data,Server,server)
   struct remote_account *ra;
   union ra_verify_peer_cb_data *data;
   const struct string * Server;
   const struct string * server;
{
    int ret = 1;
    struct smtp_info *I = data->smtp;

    /* May need run ehlo */
    if (-1 == I->ESMTP_keyword_len) {
	if (!smtp_run_ehlo(I)) {
	    ret = 0; /* Some extension failed hard */
	}
    }

    DPRINT(Debug,13,(&Debug,"submission_verify_peer_cb=%d\n",
		     ret));

    return ret;
}

static int  submission_verify_peer2  P_((struct smtp_info *I));
static int  submission_verify_peer2(I) 
     struct smtp_info *I;
{
    int status = 1;
    union ra_verify_peer_cb_data cbdata;

    if (SMTP_mailer_info_magic != I->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"submission_verify_peer",
	      "Bad magic number",0);
    cbdata.smtp = I;
    
    if (remote_account_verify_peer(& (I->RA),&cbdata,
				   submission_verify_peer_cb,
				   Server_s((I->SMTP_flags & SMTP_submission) ?
					    Submission_s : SMTP_s),
				   Server_s((I->SMTP_flags & SMTP_submission) ?
					    submission_s : SMTP_s)) < 1) {
	status = 0;   
    }

    DPRINT(Debug,13,(&Debug,"submission_verify_peer2=%d\n",
		     status));

    return status;
}

static int mailer_wait_greeting P_((struct mailer_config *M,
				    struct mailer_info *I,
				    struct cancel_data *cd));
static int mailer_wait_greeting(M,I,cd)
     struct mailer_config *M;
     struct mailer_info *I;
     struct cancel_data *cd;
{
    struct smtp_info *I1 = I->p.smtp;
    union ss_action_routine_data data;

    if (SMTP_mailer_info_magic != I1->magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,"mailer_wait_greeting",
		  "Bad magic (number smtp_info)",0);

    I1->state = smtp_wait_reply;

    data.smtp = I1;
    
    ConfigStream2(I1->RA.stream,
		  smtp_read_action,ss_noaction_routine,smtp_timeout_action,
		  ss_nofree_action_rt_data,
		  ss_noinc_action_rt_data_refcount,
		  smtp_badpid_action,
		  smtp_idle_alive_interval > MIN_SMTP_TIMEOUT ?
		  smtp_idle_alive_interval : MIN_SMTP_TIMEOUT,		 
		  data);
  
    while ( smtp_wait_reply  == I1->state &&
	    NULL         != I1->RA.stream) {

	if (!WaitStreamFor_c(I1->RA.stream,SS_read_act,cd)) {
	    
	    DPRINT(Debug,7,(&Debug,"mailer_wait_greeting: wait not succeed?\n"));

	    if (cd && is_canceled(cd)) {
		DPRINT(Debug,7,(&Debug,"mailer_wait_greeting: wait canceled!\n"));

		return 0;
	    }
	}
    }

    if (smtp_reply_incomplete == I1->state ||
	smtp_reply_read     == I1->state) {
	
	DPRINT(Debug,7,(&Debug,"mailer_wait_greeting: response code %.3s\n",
			I1->response_code));

	if ('2' == I1->response_code[0]) {
	    DPRINT(Debug,7,(&Debug,"mailer_wait_greeting: OK\n"));
	    return 1;
	}
    }

    DPRINT(Debug,7,(&Debug,"mailer_wait_greeting: failed\n"));
    return 0;
}

static struct smtp_info * malloc_smtp_info P_((void));
static struct smtp_info * malloc_smtp_info()
{
    struct smtp_info * ret = safe_malloc(sizeof (* ret));

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)ret,sizeof (* ret));
    
    ret->magic = SMTP_mailer_info_magic;
    
    zero_remote_account(& (ret->RA));

    zero_Read_Buffer(& (ret->read_buffer));
    zero_Write_Buffer(& (ret->write_buffer));

    ret->command[0]        = '\0';
    ret->NOOP_start        = NO_schedule_timelimit;
    ret->state             = smtp_wait_reply;
    ret->seen_badpid       = 0;
    ret->response_code[0]  = '\0';
    ret->response_text     = NULL;
    ret->response_text_len = 0;
    ret->SMTP_flags        = 0;
    ret->incomplete_command     = NULL;
    ret->incomplete_command_len = 0;
    ret->ESMTP_keywords        = NULL;
    ret->ESMTP_keyword_len     = -1;   /* -1 == EHLO not issued yet */
    ret->stalled               = 0;

    
    return ret;
}


S_(mailer_info_init_hook submission_mailer_info_init)
static int submission_mailer_info_init P_((struct mailer_config *M,
					   struct mailer_info *I));
static int submission_mailer_info_init(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{
    struct service_entry      * se         = NULL;
    struct service_entry      * result_se  = NULL;
    struct service_entry      * seX;
    int stflag;

    enum itls_status have_initial_tls = itls_none;

    const struct service_type * got_type = NULL;
    int                         got_port = 0;  /* port number */
    int status = 0;   
    struct cancel_data * cd = NULL;
    struct cancel_data *main_cancel = NULL /* Used if dns lookup was
					      cancelable 
					   */;

    static const int port_masks [] = { STFLAG_is_submission, STFLAG_is_smtp, 0 };  /* Ordering */
    struct enum_service_type ports_2 = NULL_enum_service_type;
    
    DPRINT(Debug,13,(&Debug,
		     "submission_mailer_info_init: mailer info, I=%p\n",I));

    init_enum_service_list(&ports_2,init_tls_default,
			   port_masks,
			   STFLAG_smtp_like); 
    
    se = give_service_entry_can(submission_server, STFLAG_smtp_like,
				itls_ignore,NULL,
				&main_cancel);
    if (!se) {
	
	if (main_cancel && is_canceled(main_cancel)) {
	    DPRINT(Debug,12,(&Debug,
			     "submission_mailer_info_init: DNS lookup canceled; (%s)\n",
			     submission_server));	    
	} else {
	    DPRINT(Debug,12,(&Debug,
			     "submission_mailer_info_init: lookup failed; (%s)\n",
			     submission_server));	    
	}

	DPRINT(Debug,13,(&Debug,
			 "submission_mailer_info_init=0: submission server %s not found. Will not initalize.\n",
			 submission_server));

	goto clean;       
    }

    I->p.smtp = malloc_smtp_info();

    /* There is no connection caching currently */

    I->p.smtp->RA.host = strmcpy(I->p.smtp->RA.host,se->official_name);
    I->p.smtp->NOOP_start        = NO_schedule_timelimit;
    
    I->p.smtp->state                 = smtp_error;
    I->p.smtp->SMTP_flags        = 0;
    
    if (!connect_remote_account_2(&(I->p.smtp->RA),
				  se,main_cancel,

				  &result_se,
				  &got_type,
				  &got_port,
				  
				  &ports_2))
	goto clean;

    DPRINT(Debug,15,(&Debug,
		     "submission_mailer_info_init: %s: connected, got port %d",
		     I->p.smtp->RA.host,got_port));
    if (got_type) {
	DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
			 got_type,
			 service_type_name(got_type),
			 service_type_defport(got_type)));
    }
    DPRINT(Debug,15,(&Debug,"\n"));
    
    switch((have_initial_tls = remote_account_init_tls(&(I->p.smtp->RA)))) {
	
    case itls_unsupported:
    case itls_failed:
	status =  0;
	goto clean;
	
    case itls_none:
	break;
	
    case itls_done:
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_init: Initial TLS done (no STARTTLS)\n"));
    }

    if (result_se) {
	if (SERVICE_ENTRY_magic != result_se->magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,
		  "submission_mailer_info_init",
		  "Bad magic (service_entry)",0);
	
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_init: Got service entry %p %s, type %p %s\n",
			result_se,result_se->official_name,
			result_se->service,service_type_name(result_se->service)));
	
	seX = result_se;
    } else
	seX = se;
    

    switch((stflag = STFLAG_service_type(seX))) {
    case STFLAG_is_submission:
	goto is_submission;
    case STFLAG_is_smtp:
	goto is_smtp;
    default:
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_init: unknown stflag=x%04x (service=%p %s, port=%d)\n",
			stflag,seX->service,service_type_name(seX->service), got_port));
    }

    if (got_type) {
	int flag;

	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_init: got type %p %s\n",
			got_type,service_type_name(got_type)));
	
	switch((flag = service_type_flags(got_type,STFLAG_type_mask))) {
	case STFLAG_is_submission:
	    goto is_submission;
	case STFLAG_is_smtp:
	    goto is_smtp;
	default:
	    DPRINT(Debug,8,(&Debug,
			    "submission_mailer_info_init: unknown flag x%04x (type=%p %s)\n",
			    flag,got_type,service_type_name(got_type)));
	}
    }
    
    switch (got_port) {
    case PORT_submissions:
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_init: Submissions (tls) service ... handled as submission\n"));
	
    case PORT_submission:
    is_submission:
    
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_init: Submission service\n"));
	I->p.smtp->SMTP_flags    = SMTP_submission;
	break;

    case PORT_smtp:
    is_smtp:

	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_init: smtp service\n"));
	    
	break;

    default:
	DPRINT(Debug,1,(&Debug,
			"submission_mailer_info_init: Unknown service!\n"));
    }

    if (I->p.smtp->RA.hostname && 0 != strcmp(seX->official_name,I->p.smtp->RA.hostname))
	cd = new_schedule_cancel(cancel_delay_msec,
				 CATGETS(elm_msg_cat, SmtpSet,
					 SmtpWaitingServer1,
					 "Waiting %S server %s [%s], port %d ..."),
				 Server_s((I->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
				 seX->official_name,
				 I->p.smtp->RA.hostname,
				 got_port);			
    else
	cd = new_schedule_cancel(cancel_delay_msec,
				 CATGETS(elm_msg_cat, SmtpSet,
					 SmtpWaitingServer,
					 "Waiting %S server %s, port %d ..."),
				 Server_s((I->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
				 seX->official_name,
				 got_port);
    
    if (!mailer_wait_greeting(M,I,cd)) {
	
	DPRINT(Debug,12,(&Debug,
			 "submission_mailer_info_init: Failed to get greeting from server\n"));
	
	/* Caller will call submission_mailer_info_close ... */
	status =  0;
	goto clean;
    }
    if (!smtp_command_reset_c(I->p.smtp,cd)) {

	DPRINT(Debug,12,(&Debug,
			 "submission_mailer_info_init: command reset failed\n"));

	status =  0;
	goto clean;
    }
   
    status = 1;

    if (submission_verify_tls_certificate ||
	submission_peer_name)
	status = submission_verify_peer(I->p.smtp,cd);
    else {
	DPRINT(Debug,12,(&Debug,
			 "submission_mailer_info_init: No verify-tls-certificate or require-tls-peer-name (on mailer defination)\n"));
    }
    free_cancel(&cd);

    if (status) 
	status = submission_verify_peer2(I->p.smtp);

 clean:
    if (cd)
	free_cancel(&cd);

    /* 'se' can be NULL */
    free_service_entry(&se);
    free_service_entry(&result_se);

    DPRINT(Debug,12,(&Debug,
		     "submission_mailer_info_init=%d\n",status));
    return status;
}

static void mailer_close2 P_((struct mailer_info *I));
static void mailer_close2(I)
     struct mailer_info *I;
{
    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"mailer_close2",
	      "Bad magic number (smtp mailer info)",0);

    free_Read_Buffer(& (I->p.smtp->read_buffer));
    free_Write_Buffer(& (I->p.smtp->write_buffer));
    
    free_response_text(I->p.smtp);
    
    if (I->p.smtp->incomplete_command)
	free(I->p.smtp->incomplete_command);
    
    I->p.smtp->incomplete_command     = NULL;
    I->p.smtp->incomplete_command_len = 0;
    
    free_ESMTP_keywords(I->p.smtp);
}

/* Return
 -1  if connection lost          (mailer_disconnected)
  0  if OK                       (mailer_not_restarted)
  1  if mailer reinitialized and  query_mailer_info()
        need to be called again  (mailer_reinitialized)
*/
S_(mailer_info_restart_hook submission_mailer_info_rs_hook)
static enum restart_mailer_status
   submission_mailer_info_rs_hook P_((struct mailer_config *M,
				      struct mailer_info *I));
static enum restart_mailer_status submission_mailer_info_rs_hook(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{
    static const int port_masks [] = { STFLAG_is_submission, STFLAG_is_smtp, 0 };  /* Ordering */
    struct enum_service_type ports_2 = NULL_enum_service_type;
        
    int is_ok = 1;
    enum restart_mailer_status status = mailer_not_restarted;
    int stflag;
    struct cancel_data * cd = NULL;

    const struct service_type * got_type = NULL;
    int                         got_port = 0;  /* port number */
    enum itls_status have_initial_tls = itls_none;
    struct service_entry *se = NULL;
    struct service_entry      * result_se  = NULL;
    struct service_entry      * seX;

    DPRINT(Debug,13,(&Debug,
		     "submission_mailer_info_rs_hook: mailer info, I=%p\n",I));
    
    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,
	      "submission_mailer_info_rs_hook",
	      "Bad magic number (smtp mailer info)",0);

    init_enum_service_list(&ports_2,init_tls_default,
			   port_masks,
			   STFLAG_smtp_like); 

    if (I->p.smtp->RA.hostname && 0 != strcmp(I->p.smtp->RA.host,I->p.smtp->RA.hostname)) 
	cd = new_schedule_cancel(cancel_delay_msec,
				 CATGETS(elm_msg_cat, SmtpSet,
					 SmtpWaitingServer3,
					 "Waiting %S server %s [%s] ..."),
				 Server_s((I->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
				 I->p.smtp->RA.host,
				 I->p.smtp->RA.hostname);
    
    else
	cd = new_schedule_cancel(cancel_delay_msec,
				 CATGETS(elm_msg_cat, SmtpSet,
					 SmtpWaitingServer2,
					 "Waiting %S server %s ..."),
				 Server_s((I->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
				 I->p.smtp->RA.host);
    
    if (smtp_start_command_c(I->p.smtp,"NOOP",cd)) {
	char response_code[4];

	if (!smtp_command_ready_c(I->p.smtp,0,cd)) {
	    DPRINT(Debug,12,(&Debug,
			     "submission_mailer_info_rs_hook: NOOP not ready\n"));
	    goto noop_failure;
	}

	is_ok = smtp_command_ok_c(I->p.smtp,response_code,cd);

	if (!smtp_command_reset_c(I->p.smtp,cd)) {
	    DPRINT(Debug,12,(&Debug,
			     "submission_mailer_info_rs_hook: NOOP -- command reset failure\n"));
	    is_ok = 0;
	}
    } else {
    noop_failure:
	is_ok = 0;
    }

    if (is_ok) {
	status = mailer_not_restarted;
	
	DPRINT(Debug,12,(&Debug,
			 "submission_mailer_info_rs_hook:  (OK, mailer not disconnected)\n"));
	
	goto clean;   /* Mailer OK -- need not restart */
    }

    /* Send QUIT */
    if (smtp_idle == I->p.smtp->state) {
	int status = smtp_start_command_c(I->p.smtp,"QUIT",cd);
	
	if (status) {
	    if (!smtp_command_ready_c(I->p.smtp,0,cd)) {
		DPRINT(Debug,12,(&Debug,
				 "submission_mailer_info_rs_hook: QUIT not ready\n"));
		goto quit_failure;
	    }
	    
	    /* Wait for QUIT */	       
	    if (!smtp_command_reset_c(I->p.smtp,cd)) {
		DPRINT(Debug,12,(&Debug,
				 "submission_mailer_info_rs_hook: QUIT -- command reset failure\n"));
		goto quit_failure;
	    }
	}
    }

 quit_failure:
    status = mailer_disconnected;

    /* Actually close connection */
    clear_remote_account(& (I->p.smtp->RA));	
    mailer_close2(I);
    free_cancel(&cd);

    se = give_service_entry_can(submission_server, STFLAG_smtp_like,
				itls_ignore,NULL,&cd);
			    
    if (!se) {
	if (cd && is_canceled(cd)) {
	    DPRINT(Debug,12,(&Debug,
			     "submission_mailer_info_rs_hook: DNS lookup canceled; (%s)\n",
			     submission_server));
	} else {
	    DPRINT(Debug,12,(&Debug,
			     "submission_mailer_info_rs_hook: lookup failed; (%s)\n",
			     submission_server));
	}
	
        status = mailer_disconnected;
	goto clean;
    }

    /* clear_remote_account was cleared this */
    I->p.smtp->RA.host = strmcpy(I->p.smtp->RA.host,se->official_name);

    /* Try reconnect */
    I->p.smtp->state                 = smtp_error;
    I->p.smtp->SMTP_flags            = 0;

    if (!connect_remote_account_2(&(I->p.smtp->RA),
				  se,cd,
				  
				  &result_se,
				  &got_type,
				  &got_port,
				  
				  &ports_2))	
	goto clean;

    DPRINT(Debug,15,(&Debug,
		     "submission_mailer_info_rs_hook: %s: connected, got port %d",
		     I->p.smtp->RA.host,got_port));
    if (got_type) {
	DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
			 got_type,
			 service_type_name(got_type),
			 service_type_defport(got_type)));
    }
    DPRINT(Debug,15,(&Debug,"\n"));

    
    switch((have_initial_tls = remote_account_init_tls(&(I->p.smtp->RA)))) {
	
    case itls_unsupported:
    case itls_failed:
	
	goto clean;
	
    case itls_none:
	break;
	
    case itls_done:
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_rs_hook: Initial TLS done (no STARTTLS)\n"));
    }


    if (result_se) {
	if (SERVICE_ENTRY_magic != result_se->magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,
		  "submission_mailer_info_rs_hook",
		  "Bad magic (service_entry)",0);
	
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_rs_hook: Got service entry %p %s, type %p %s\n",
			result_se,result_se->official_name,
			result_se->service,service_type_name(result_se->service)));
	
	seX = result_se;
    } else
	seX = se;

    

    switch((stflag = STFLAG_service_type(seX))) {
    case STFLAG_is_submission:
	    goto is_submission;
    case STFLAG_is_smtp:
	goto is_smtp;

    default:
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_rs_hook: unknown stflag=x%04x (service=%p %s, port=%d)\n",
			stflag,seX->service,service_type_name(seX->service), got_port));
    }

    if (got_type) {
	int flag;

	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_init: got type %p %s\n",
			got_type,service_type_name(got_type)));
	
	switch((flag = service_type_flags(got_type,STFLAG_type_mask))) {
	case STFLAG_is_submission:
	    goto is_submission;
	case STFLAG_is_smtp:
	    goto is_smtp;
	default:
	    DPRINT(Debug,8,(&Debug,
			    "submission_mailer_info_rs_hook: unknown flag x%04x (type=%p %s)\n",
			    flag,got_type,service_type_name(got_type)));
	}
    }
    
    switch (got_port) {
    case PORT_submissions:
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_rs_hook: Submissions (tls) service ... handled as submission\n"));
		
    case PORT_submission:
    is_submission:
	
	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_rs_hook: Submission service\n"));
	I->p.smtp->SMTP_flags    = SMTP_submission;
	break;

    case PORT_smtp:
    is_smtp:

	DPRINT(Debug,8,(&Debug,
			"submission_mailer_info_rs_hook: smtp service\n"));
	    
	break;
    default:
	DPRINT(Debug,1,(&Debug,
			"submission_mailer_info_rs_hook: Unknown service!\n"));
    }

    if (I->p.smtp->RA.hostname && 0 != strcmp(seX->official_name,I->p.smtp->RA.hostname)) {
	if (cd)
	    set_cancel_message(cd,
			       cancel_delay_msec,
			       CATGETS(elm_msg_cat, SmtpSet,
				       SmtpWaitingServer1,
				       "Waiting %S server %s [%s], port %d ..."),
			       Server_s((I->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
			       seX->official_name,
			       I->p.smtp->RA.hostname,
			       got_port);
	else
	    cd = new_schedule_cancel(cancel_delay_msec,
				     CATGETS(elm_msg_cat, SmtpSet,
					     SmtpWaitingServer1,
					     "Waiting %S server %s [%s], port %d ..."),
				     Server_s((I->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
				     seX->official_name,
				     I->p.smtp->RA.hostname,
				     got_port);			
    } else {
	if (cd)
	    set_cancel_message(cd,  
			       cancel_delay_msec,
			       CATGETS(elm_msg_cat, SmtpSet,
				       SmtpWaitingServer,
				       "Waiting %S server %s, port %d ..."),
			       Server_s((I->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
			       seX->official_name,
			       got_port);
	else
	    cd = new_schedule_cancel(cancel_delay_msec,
				     CATGETS(elm_msg_cat, SmtpSet,
					     SmtpWaitingServer,
					     "Waiting %S server %s, port %d ..."),
				     Server_s((I->p.smtp->SMTP_flags & SMTP_submission) ? submission_s : SMTP_s),
				     seX->official_name,
				     got_port);
    }
    
    if (!mailer_wait_greeting(M,I,cd)) {
	DPRINT(Debug,12,(&Debug,
			 "submission_mailer_info_rs_hook: Failed to get greeting from server\n"));

	/* Caller will call submission_mailer_info_close ... */
	status =  mailer_disconnected;
	goto clean;
    }
    smtp_command_reset(I->p.smtp);
    status = mailer_reinitialized;   /* reconnect succeed */

    if (submission_verify_tls_certificate ||
	submission_peer_name) {
	
	if (! submission_verify_peer(I->p.smtp,cd)) {

	    if (cd && is_canceled(cd)) {
		DPRINT(Debug,7,(&Debug,"submission_mailer_info_rs_hook: wait canceled!\n"));
		
		status = 0;
		goto clean;
	    }

	    status =  mailer_disconnected;

	}
    } else {
	DPRINT(Debug,12,(&Debug,
			 "submission_mailer_info_rs_hook: No verify-tls-certificate or require-tls-peer-name (on mailer defination)\n"));
    }
    free_cancel(&cd);

    if (status > mailer_not_restarted) { 
	if (!submission_verify_peer2(I->p.smtp))
	    status =  mailer_disconnected;
    }

 clean:
    if (cd)
	free_cancel(&cd);

    /* 'se' can be NULL */
    free_service_entry(&se);
    free_service_entry(&result_se);

    DPRINT(Debug,12,(&Debug,
		     "submission_mailer_info_rs_hook=%d",status));
    switch (status) {
    case mailer_disconnected:    DPRINT(Debug,12,(&Debug," mailer_disconnected"));  break;
    case mailer_not_restarted:   DPRINT(Debug,12,(&Debug," mailer_not_restarted")); break;
    case mailer_reinitialized:   DPRINT(Debug,12,(&Debug," mailer_reinitialized"));   break;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return status;
}



S_(mailer_info_close_hook submission_mailer_info_close)
static void submission_mailer_info_close P_((struct mailer_config *M,
					     struct mailer_info *I));
static void submission_mailer_info_close(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{
    DPRINT(Debug,13,(&Debug,
		     "submission_mailer_info_close: mailer info, I=%p\n",I));

    if (I->p.smtp) {
	if (SMTP_mailer_info_magic != I->p.smtp->magic)
	    panic("MAILER PANIC",__FILE__,__LINE__,
		  "submission_mailer_info_close",
		  "Bad magic number (smtp mailer info)",0);

	if (smtp_idle == I->p.smtp->state) {
	    int status = smtp_start_command(I->p.smtp,"QUIT");

	    if (status) {
		smtp_command_ready(I->p.smtp,0);
		
		/* Wait for QUIT */	       
		smtp_command_reset(I->p.smtp);
	    }
	}

	clear_remote_account(& (I->p.smtp->RA));	
	mailer_close2(I);

	I->p.smtp->magic = 0; /* Invalidate */

	free(I->p.smtp);
	I->p.smtp = NULL;
    } else {
	DPRINT(Debug,13,(&Debug,
			 "submission_mailer_info_close: was not initialized or already closed\n"));
    }
}

static int submission_verify_address = 0;


S_(mailer_info_verify_addr submission_mailer_info_verify_addr)
static enum mi_vrfyaddr_mailer_status
    submission_mailer_info_verify_addr P_((struct mailer_config *M,
					   struct mailer_info  *I, 
					   const char *text,
					   struct addr_verify_result 
					   *result));
static enum mi_vrfyaddr_mailer_status submission_mailer_info_verify_addr(M,I,text,result)
     struct mailer_config *M;
     struct mailer_info  *I;
     const char *text;
     struct addr_verify_result *result;
{
    if (!submission_verify_address) {
	enum mi_vrfyaddr_mailer_status ret = mi_vrfyaddr_test_passwd;
	DPRINT(Debug,13,(&Debug, 
			 "submission_mailer_info_verify_addr=%d (Verify disabled)\n",
			 ret));
	return ret;       /* Cannot VRFY user */		
    }

    return smtp_tail_verify_addr(M,I,text,result);
}

S_(MO_value_hook MO_submission_verify)
static int MO_submission_verify P_((struct mailer_config      * M,
			       struct mailer_option_list * L,
			       char **value,int set));
static int MO_submission_verify (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	     submission_verify_address = 1;
	else if (0 == strcmp(*value,no_yes[0]))
	    submission_verify_address = 0;
	else if (0 == strcmp(*value,no_yes[1])) 
	    submission_verify_address = 1;
	else {
	    lib_error(CATGETS(elm_msg_cat, SmtpSet,
			      SmtpBadConfSubmission,
			      "Bad %s=%s for submission"),
		      L->field,*value);
	    return 0;
	}	
	value_set = 1;
    } else {
	if (submission_verify_address) {
	    *value = no_yes[1];
	    return 1;
	}
	*value = no_yes[0]; /* Only show if verify=no or
			       verify=yes is given at least once 
			    */
    }    
    return value_set;
}

S_(MO_value_hook MO_submission_tlsverify)
static int MO_submission_tlsverify P_((struct mailer_config      * M,
			       struct mailer_option_list * L,
			       char **value,int set));
static int MO_submission_tlsverify (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	     submission_verify_tls_certificate = 1;
	else if (0 == strcmp(*value,no_yes[0]))
	    submission_verify_tls_certificate = 0;
	else if (0 == strcmp(*value,no_yes[1])) 
	    submission_verify_tls_certificate = 1;
	else {
	    lib_error(CATGETS(elm_msg_cat, SmtpSet,
			      SmtpBadConfSubmission,
			      "Bad %s=%s for submission"),
		      L->field,*value);
	    return 0;
	}	
	value_set = 1;
    } else {
	if (submission_verify_tls_certificate) {
	    *value = no_yes[1];
	    return 1;
	}
	*value = no_yes[0]; /* Only show if verify-tls-certificate=no or
			       verify-tls-certificate=yes is given at least once 
			    */
    }    
    return value_set;
}
S_(MO_value_hook MO_submission_requirename)
static int MO_submission_requirename P_((struct mailer_config      * M,
			       struct mailer_option_list * L,
			       char **value,int set));
static int MO_submission_requirename (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static int value_set  = 0;
   
     if (set) {

	 if (L->malloced) {
	     free(*(L->value));
	     *(L->value) = NULL;
	 }

	 if (!value || !*value) {
	     *(L->value) = &REQUIRE_SERVER_NAME;
	     L->malloced = 0;
	 } else {
	     *(L->value) = safe_strdup(*value);
	     L->malloced = 1;
	 }
	 value_set = 1;
     } else {
	 if (!*(L->value) || *(L->value) == &REQUIRE_SERVER_NAME)
	     *value = NULL;
	 else {
	     *value = *(L->value);
	     return 1;
	 }
     }

     return value_set;
}

S_(MO_value_hook MO_smtp_sending_display)
static int MO_smtp_sending_display P_((struct mailer_config      * M,
				  struct mailer_option_list * L,
				  char **value,int set));
static int MO_smtp_sending_display (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	    M -> mailer_bits  |= MB_SENDING_DISPLAY;
	else if (0 == strcmp(*value,no_yes[0]))
	    M -> mailer_bits  &= ~MB_SENDING_DISPLAY;
	else if (0 == strcmp(*value,no_yes[1])) 
	    M -> mailer_bits  |= MB_SENDING_DISPLAY;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmBadSendingDisplay,
			      "Bad sending-display=%s for %s"),
		      *value, M->mailer_type);
	    return 0;
	}

	value_set = 1;
    } else {
	
	*value = no_yes[ (M -> mailer_bits & MB_SENDING_DISPLAY) ?
			 1 : 0 ];
       
	if (0 == (M -> mailer_bits & MB_SENDING_DISPLAY))
	    return 1;

	/* Do not show sending-display=yes if it is not set explicity */
    }

    return value_set;
}


static struct mailer_option_list SUBMISSION_options[] = {
    { "server",   &submission_server,                MO_default_value, 0 },
    { "verify",   NULL,                              MO_submission_verify, 0 },
    { "allow-set-sender",NULL,                       MO_allow_set_sender, 0 },
    { "verify-tls-certificate", NULL,                MO_submission_tlsverify, 
      0 },
    { "require-tls-peer-name", & submission_peer_name, 
      MO_submission_requirename, 0 },
    { "sending-display", NULL,                       MO_smtp_sending_display, 
      0 },
    { NULL, NULL,                                    MO_default_value, 0 }
};

/* sendmail-bs mailer ----------------------------------------------------- */

#if defined(BACKGROUD_PROCESSES) 
#define HAVE_SENDMAILBS 1
#endif

#if HAVE_SENDMAILBS

static char * sendmail_mailer_path    = SENDMAIL_MAILER_PATH;



S_(mailer_info_init_hook sendmailbs_mailer_info_init)
static int sendmailbs_mailer_info_init P_((struct mailer_config *M,
					   struct mailer_info *I));
static int sendmailbs_mailer_info_init(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{
    const char * argv[3];
    char response_code[4];

    int options = SY_ENV_SHELL | SY_NOTTY;
    int ret,r;
    int pair[2];

    int r1;
    
    union ss_action_routine_data data;
    
    DPRINT(Debug,13,(&Debug,
		     "sendmailbs_mailer_info_init: mailer info, I=%p\n",I));

    I->p.smtp =  malloc_smtp_info();

    I->p.smtp->state             = smtp_wait_reply;
    I->p.smtp->SMTP_flags        = SMTP_submission;
    
    r = socketpair(AF_UNIX,SOCK_STREAM,0, pair);
    
    if (-1 == r) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, SmtpSet,
			  SmtpSocketPairFail,
			  "socketpair failed: %s"),
		  strerror(err));

	I->p.smtp->magic = 0; /* Invalidate */

	free(I->p.smtp);
	I->p.smtp = NULL;
	return 0;
    }

    argv[0] = sendmail_mailer_path;
    argv[1] = "-bs";
    argv[2] = NULL;
	
    /* Connect standard imput and output to other end of socket ... */

    ret = start_run( & (I->p.smtp->RS),options,argv,pair[1],pair[1]);

    /* Close other end of socket so it is used only by sendmail */
    close(pair[1]);
    pair[1] = -1;

    if (!ret) {
	if (I->p.smtp->RS.save_errno)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailErrno,
			      "Failed: %.30s: %.40s"),
		      argv[0],strerror(I->p.smtp->RS.save_errno));
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantStart,
			      "Can't start %.30s"),
		      argv[0]);

	close(pair[0]); 
	pair[0] = -1;


	I->p.smtp->magic = 0; /* Invalidate */
	free(I->p.smtp);
	I->p.smtp = NULL;
	return 0;
    }

    if (transaction_file) {

	time_t now = 0;
			
	if (((time_t) -1) != time(&now)) {
	    struct tm * zz = localtime(&now);
		    
	    if (!zz)
		goto no_time10;
	
	    fprintf(transaction_file,
		    "%d [%d] %02d:%02d:%02d === CONNECT command %s, pid %d\n",
		    getpid(),pair[0] ,
		    zz->tm_hour,
		    zz->tm_min,
		    zz->tm_sec,
		    sendmail_mailer_path,
		    I->p.smtp->RS.pid);
	} else {
	no_time10:
	    fprintf(transaction_file,
		    "%d [%d] === CONNECT command %s, pid %d\n",
		    getpid(),pair[0] ,
		    sendmail_mailer_path,
		    I->p.smtp->RS.pid);
	    
	}
    }

    if (-1 == (r1 = fcntl(pair[0],F_SETFL,O_NONBLOCK))) {
	int err UNUSED_VAROK = errno;

	DPRINT(Debug,12,(&Debug,
			 "sendmailbs_mailer_info_init: fcntl [F_SETFL O_NONBLOCK] failed: %s\n",
			 strerror(err)));
    } else if (0 == r1) {
	DPRINT(Debug,12,(&Debug,
			 "sendmailbs_mailer_info_init: fcntl(%d,F_SETFL,O_NONBLOCK) done\n",
			 pair[0]));
    }

    if (-1 == (r1 = fcntl(pair[0],F_SETFD,FD_CLOEXEC))) {
	int err UNUSED_VAROK = errno;
	
	DPRINT(Debug,12,(&Debug,
			 "sendmailbs_mailer_info_init: fcntl [F_SETFD FD_CLOEXEC] failed: %s\n",
			 strerror(err)));
    } else if (0 == r1) {
	DPRINT(Debug,12,(&Debug,
			 "sendmailbs_mailer_info_init: fcntl(%d,F_SETFD,FD_CLOEXEC) done\n",
			 pair[0]));
    }
    
    I->p.smtp->RA.stream = returnSimpleStream(pair[0]);

    data.smtp = I->p.smtp;
    
    ConfigStream2(I->p.smtp->RA.stream,
		  smtp_read_action,ss_noaction_routine,smtp_timeout_action,
		  ss_nofree_action_rt_data,
		  ss_noinc_action_rt_data_refcount,
		  smtp_badpid_action,
		  smtp_idle_alive_interval > MIN_SMTP_TIMEOUT ?
		  smtp_idle_alive_interval : MIN_SMTP_TIMEOUT,		 
		  data);
		     
    if (!smtp_command_ok(I->p.smtp,response_code)) {
	DPRINT(Debug,12,(&Debug,
			 "sendmailbs_mailer_info_init: Failed to get greeting from sendmail -bs\n"));

	/* Caller will call sendmailbs_mailer_info_close ... */
	return 0;
    }
    smtp_command_reset(I->p.smtp);

    return 1;
}


/* Return
 -1  if connection lost          (mailer_disconnected)
  0  if OK                       (mailer_not_restarted)
  1  if mailer reinitialized and  query_mailer_info()
        need to be called again  (mailer_reinitialized)
*/
S_(mailer_info_restart_hook sendmailbs_mailer_info_rs_hook)
static enum restart_mailer_status  sendmailbs_mailer_info_rs_hook P_((struct mailer_config *M,
					 struct mailer_info *I));
static enum restart_mailer_status  sendmailbs_mailer_info_rs_hook(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{
    int is_ok = 0;
    enum restart_mailer_status ret;
    
    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"sendmailbs_mailer_info_rs_hook",
	      "Bad magic number (smtp mailer info)",0);
    
    if (smtp_start_command(I->p.smtp,"NOOP")) {
	char response_code[4];

	smtp_command_ready(I->p.smtp,0);

	is_ok = smtp_command_ok(I->p.smtp,response_code);

	smtp_command_reset(I->p.smtp);
    } else
	is_ok = 0;

    if (is_ok) {
	ret = mailer_not_restarted;
	
	DPRINT(Debug,12,(&Debug,
			 "sendmailbs_mailer_info_rs_hook=%d  (OK, mailer not disconnected)\n",
			 ret));
	
	return ret;   /* Mailer OK -- need not restart */
    }

    /* TODO -- implement reconnect */

    ret = mailer_disconnected;
    
    DPRINT(Debug,4,(&Debug,
		    "sendmailbs_mailer_info_rs_hook=%d (reconnect not implemented)\n",
		    ret));

    return ret;  /* reconnect not implemented */
}

S_(mailer_info_s_query_hook sendmailbs_minfo_s_query)
static char ** sendmailbs_minfo_s_query P_((size_t                * reslen_p,
					    struct mailer_config  * M,
					    struct mailer_info    * I,
					    enum MI_query_string    query,
					    struct cancel_data   ** cancel_p));
static char ** sendmailbs_minfo_s_query(reslen_p,M,I,query,cancel_p)
     size_t                * reslen_p;
     struct mailer_config *M;
     struct mailer_info *I;
     enum MI_query_string    query;
     struct cancel_data   ** cancel_p;
{
    *reslen_p = 0;
    
    /* NOt supported */
    
    return NULL;
}


S_(mailer_info_query_hook sendmailbs_mailer_info_query)
static int sendmailbs_mailer_info_query P_((struct mailer_config *M,
					    struct mailer_info *I,
					    enum MI_query query));
static int sendmailbs_mailer_info_query(M,I,query)
     struct mailer_config *M;
     struct mailer_info *I;
     enum MI_query query;
{
   if (SMTP_mailer_info_magic != I->p.smtp->magic)
       panic("MAILER PANIC",__FILE__,__LINE__,"sendmailbs_mailer_info_query",
	      "Bad magic number (smtp mailer info)",0);

   switch(query) {
   case MI_REMOTE_MAILER:
       return 0;  /* Assume local mailer */    
   default:
       return smtp_common_mailer_info_query(M,I,query);
   }   
}


S_(mailer_info_close_hook sendmailbs_mailer_info_close)
static void sendmailbs_mailer_info_close(M,I)
     struct mailer_config *M;
     struct mailer_info *I;
{

    if (SMTP_mailer_info_magic != I->p.smtp->magic)
	panic("MAILER PANIC",__FILE__,__LINE__,"sendmailbs_mailer_info_close",
	      "Bad magic number (smtp mailer info)",0);

    DPRINT(Debug,13,(&Debug,
		     "sendmailbs_mailer_info_close: mailer info, I=%p\n",I));

    if (I->p.smtp) {
	int ret,exit_code;

	ret = run_already_done(& (I->p.smtp->RS),&exit_code);

	if (0 == ret && smtp_idle == I->p.smtp->state) {
	    int status = smtp_start_command(I->p.smtp,"QUIT");

	    if (status) {
		smtp_command_ready(I->p.smtp,0);
		
		/* Wait for QUIT */	       
		smtp_command_reset(I->p.smtp);
	    }
	}

	clear_remote_account(& (I->p.smtp->RA));	
	
	if (0 == ret) 
	    ret = wait_end(&(I->p.smtp->RS),&exit_code);

	if (ret < 0) {
	    DPRINT(Debug,4,(&Debug,
			    "%s died on signal %d\n",
			    sendmail_mailer_path,-ret));

	    if (transaction_file) {

		time_t now = 0;
		
		if (((time_t) -1) != time(&now)) {
		    struct tm * zz = localtime(&now);
		    
		    if (!zz)
			goto no_time11;
		
		    fprintf(transaction_file,
			    "%d [?] %02d:%02d:%02d === DIED command %s, pid %d, signal %d\n",
			    getpid(),
			    zz->tm_hour,
			    zz->tm_min,
			    zz->tm_sec,
			    sendmail_mailer_path,
			    I->p.smtp->RS.pid,
			    -ret);
		} else {
		no_time11:
		    fprintf(transaction_file,
			    "%d [?] === DIED command %s, pid %d, signal %d\n",
			    getpid(),
			    sendmail_mailer_path,
			    I->p.smtp->RS.pid,
			    -ret);
		    
		}
	    }
	} else if (0 == ret) {
	    DPRINT(Debug,4,(&Debug,
			    "%s lost?\n",
			    sendmail_mailer_path));

	    if (transaction_file) {
		time_t now = 0;
		
		if (((time_t) -1) != time(&now)) {
		    struct tm * zz = localtime(&now);
		    
		    if (!zz)
			goto no_time12;
		
		    fprintf(transaction_file,
			    "%d [?] %02d:%02d:%02d === LOST command %s, pid %d\n",
			    getpid(),
			    zz->tm_hour,
			    zz->tm_min,
			    zz->tm_sec,
			    sendmail_mailer_path,
			    I->p.smtp->RS.pid);
		} else {
		no_time12:
		    fprintf(transaction_file,
			    "%d [?] === LOST command %s, pid %d\n",
			    getpid(),
			    sendmail_mailer_path,
			    I->p.smtp->RS.pid);		    
		}
	    }
	} else {
	    DPRINT(Debug,4,(&Debug,
			    "%s exited with %d%s\n",
			    sendmail_mailer_path,
			    exit_code, exit_code == 0 ? " (OK)" : ""));

	    if (transaction_file) {

		time_t now = 0;
		
		if (((time_t) -1) != time(&now)) {
		    struct tm * zz = localtime(&now);
		    
		    if (!zz)
			goto no_time13;

		    fprintf(transaction_file,
			    "%d [?] %02d:%02d:%02d === EXITED command %s, pid %d, exit code %d%s\n",
			    getpid(),
			    zz->tm_hour,
			    zz->tm_min,
			    zz->tm_sec,
			    sendmail_mailer_path,
			    I->p.smtp->RS.pid,
			    exit_code,
			    exit_code == 0 ? " (OK)" : "");
		} else {
		no_time13:

		    fprintf(transaction_file,
			    "%d [?] === EXITED command %s, pid %d, exit code %d%s\n",
			    getpid(),
			    sendmail_mailer_path,
			    I->p.smtp->RS.pid,
			    exit_code,
			    exit_code == 0 ? " (OK)" : "");

		}
	    }
	}

	mailer_close2(I);

	I->p.smtp->magic = 0; /* Invalidate */

	free(I->p.smtp);
	I->p.smtp = NULL;
    } 
}

static int sendmailbs_verify_address = 0;

S_(mailer_info_verify_addr sendmailbs_mailer_info_verify_addr)
static  enum mi_vrfyaddr_mailer_status sendmailbs_mailer_info_verify_addr(M,I,text,result)
     struct mailer_config *M;
     struct mailer_info  *I;
     const char *text;
     struct addr_verify_result *result;
{
    if (!sendmailbs_verify_address) {
	enum mi_vrfyaddr_mailer_status ret = mi_vrfyaddr_test_passwd;
	DPRINT(Debug,13,(&Debug, 
			 "sendmailbs_mailer_info_verify_addr=%d (Verify disabled)\n",
			 ret));
	return ret;       /* Cannot VRFY user */		
    }

    return smtp_tail_verify_addr(M,I,text,result);
}

S_(MO_value_hook MO_sendmailbs_verify)
static int MO_sendmailbs_verify P_((struct mailer_config      * M,
			       struct mailer_option_list * L,
			       char **value,int set));
static int MO_sendmailbs_verify (M,L,value,set)
     struct mailer_config      *M;
     struct mailer_option_list *L;
     char **value;
     int set;
{
    static char *no_yes[] = { "no", "yes" };
    static int value_set  = 0;

    if (set) {
	if (!*value)
	     sendmailbs_verify_address = 1;
	else if (0 == strcmp(*value,no_yes[0]))
	    sendmailbs_verify_address = 0;
	else if (0 == strcmp(*value,no_yes[1])) 
	    sendmailbs_verify_address = 1;
	else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSendmailBadVerify,
			      "Bad verify=%s for sendmail"),
		      *value);
	    return 0;
	}	
	value_set = 1;
    } else {
	if (sendmailbs_verify_address) {
	    *value = no_yes[1];
	    return 1;
	}
	*value = no_yes[0]; /* Only show if verify=no or
			       verify=yes is given at least once 
			    */
    }    
    return value_set;
}


static struct mailer_option_list SENDMAILBS_options[] = {
    { "path",     &sendmail_mailer_path,             MO_default_value, 0 },
    { "verify",   NULL,                              MO_sendmailbs_verify, 0 },
    { "allow-set-sender",NULL,                       MO_allow_set_sender,  0 },
    { "sending-display", NULL,                       MO_smtp_sending_display, 
      0 },
    { NULL, NULL,                                    MO_default_value, 0 }
};

#endif

static struct mailer_config MAILERS[] = {
    { MAILER_CONFIG_magic,
      "submission", &(SUBMISSION_options[0]), default_set_option, 
      smtp_common_mailer_init, smtp_common_mailer_close, 
      smtp_common_mailer_backend3,
      NULL, 
      submission_mailer_info_init, submission_mailer_info_close,
      smtp_common_mailer_info_query, submission_mailer_info_verify_addr,
      smtp_common_mailer_info_gen_def_ef, smtp_common_mailer_info_set_ef,
      submission_mailer_info_rs_hook, smtp_minfo_s_query,
      
      MB_USE_DOMAIN|MB_REQ_DEFAULT_SENDER|MB_SENDING_DISPLAY
    },
#if HAVE_SENDMAILBS
    { MAILER_CONFIG_magic,
      "sendmail-bs", &(SENDMAILBS_options[0]), default_set_option, 
      smtp_common_mailer_init, smtp_common_mailer_close, 
      smtp_common_mailer_backend3,
      &sendmail_mailer_path, 
      sendmailbs_mailer_info_init, sendmailbs_mailer_info_close,
      sendmailbs_mailer_info_query, sendmailbs_mailer_info_verify_addr,
      smtp_common_mailer_info_gen_def_ef, smtp_common_mailer_info_set_ef,
      sendmailbs_mailer_info_rs_hook,  sendmailbs_minfo_s_query,

      MB_USE_DOMAIN|MB_REQ_DEFAULT_SENDER|MB_SENDING_DISPLAY
    },
#endif

};

#if ANSI_C
provides_shared_MCF_f provides_shared_MCF;
#endif
struct mailer_config * provides_shared_MCF P_((int *count, size_t *s_size, 
					       size_t *opt_size,
					       size_t *s_send_state_size, 
					       size_t *s_cancel_size,
					       size_t *s_mailer_info_size, 
					       size_t *s_env_from_size));
struct mailer_config * provides_shared_MCF(count,s_size,opt_size,
					   s_send_state_size,s_cancel_size,
					   s_mailer_info_size,s_env_from_size)
     int *count; 
     size_t *s_size; 
     size_t *opt_size;
     size_t *s_send_state_size;
     size_t *s_cancel_size;
     size_t *s_mailer_info_size;
     size_t *s_env_from_size;
{
    *s_size    = sizeof (MAILERS[0]);
    *count  = (sizeof MAILERS) / sizeof (MAILERS[0]);

    *opt_size           = sizeof (struct mailer_option_list); 
    *s_send_state_size  = sizeof (struct mail_send_state);
    *s_cancel_size      = sizeof (struct mailer_cancel_ticket);
    *s_mailer_info_size = sizeof (struct mailer_info);
    *s_mailer_info_size = sizeof (struct mailer_info);
    *s_env_from_size    = sizeof (struct mailer_env_from);

    if (!remote_account_OK(sizeof (struct remote_account))) {

	DPRINT(Debug,1,(&Debug,			    
			"smtp:provides_shared_MCF: sizeof (struct remote_account) mismatch\n"));

	*count = 0;
    }

    if (! StreamOK(sizeof (struct stream_type))) {
	DPRINT(Debug,1,(&Debug,			    
			"smtp:provides_shared_MCF: sizeof (struct stream_type) mismatch\n"));

	*count = 0;
    }

    return &(MAILERS[0]);
}

static struct string * static_strings [ NUM_s ];
static const struct string * Server_s(idx)
     enum server_s idx;
{
    if (idx < 0 || idx >= NUM_s)
	panic("MAILER PANIC",__FILE__,__LINE__,"Server_s",
	      "Bad idx",0);	
    
    if (static_strings[idx])
	return static_strings[idx];

    switch (idx) {
    case Submission_s:
	static_strings[idx] = 
	    format_string(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpSubmission,
				  "Submission"));
	break;
    case SMTP_s:
	static_strings[idx] = 
	    format_string(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpSMTP,
				  "SMTP"));
	break;
    case submission_s:
	static_strings[idx] = 
	    format_string(CATGETS(elm_msg_cat, SmtpSet,
				  SmtpLcSubmission,
				  "submission"));
	break;
    case NUM_s: /* Not used */
	panic("MAILER PANIC",__FILE__,__LINE__,"Server_s",
	      "Bad idx",0);
	break;
    }

    if (!static_strings[idx])
	panic("MAILER PANIC",__FILE__,__LINE__,"Server_s",
	      "Bad idx?",0);		

    return static_strings[idx];
}




E_(free_shared_cache_f free_shared_cache)
void free_shared_cache P_((void));
void free_shared_cache() 
{
    enum server_s i;

    DPRINT(Debug,7,(&Debug, "free_shared_cache\n"));

    for (i = 0; i < NUM_s; i++)
	if (static_strings[i])
	    free_string(& static_strings[i]);
}


/* elmrc options ---------------------------------------------------------- */

#include "rc_imp.h"
#include "save_opts.h"

static ZZZ_SAVE_TYPE save_info_data[] = {
    {"smtp-idle-alive-interval",
     ZZZ_DT_NUM(&smtp_idle_alive_interval), ZZZ_TAIL },
};


E_(provides_RC_options_f provides_RC_options2)
struct rc_save_info_rec * provides_RC_options2 P_((size_t *count,
						   size_t *s_size));
struct rc_save_info_rec * provides_RC_options2(count,s_size)
     size_t *count; 
     size_t *s_size;
{
    *s_size = sizeof (save_info_data[0]);

    *count = (sizeof save_info_data) / *s_size;
    return (struct rc_save_info_rec *) save_info_data;
}

static void check_minimum_value P_((int *var, const char *varname,
				    const int minimum_value,
				    const char *tag,
				    int *errors));
static void check_minimum_value(var,varname,minimum_value,tag,errors)
     int *var;
     const char *varname;
     const int minimum_value;
     const char * tag;
     int *errors;
{
    if (*var < minimum_value) {
	if (errors)
	    (*errors)++;
	lib_error(CATGETS(elm_msg_cat, SmtpSet, 
			  SmtpChangingMinimumValue,
			  "Changing %s value %d to %d on %s section."),
		  varname,
		  *var, minimum_value,
		  tag ? tag : "<main?>");

	*var = minimum_value;
	mark_local_changed(var,0);	
    }    
}


E_(RC_post_init_f RC_post_init2)
void RC_post_init2  P_((int *errors, int flag, const char * tag));
void RC_post_init2(errors, flag,tag)
     int *errors;
     int flag;
     const char * tag;
{
    DPRINT(Debug,4,(&Debug,"RC_post_init2 ... (smtp)\n"));

    check_minimum_value(&smtp_idle_alive_interval,
			"smtp-idle-alive-interval",
			MIN_SMTP_TIMEOUT,
			tag,errors);
}
#endif

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
