static char rcsid[] = "@(#)$Id: browser_help.c,v 2.7 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.7 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_browser.h"

DEBUG_VAR(Debug,__FILE__,"ui");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

void init_browser_edit(BE,p,aview)
     struct browser_edit *BE;
     struct folder_browser *p;
     struct AliasView    *aview;
{
    BE->magic         = BROWSER_EDIT_magic;
    BE->gb_page       = 0;              /* Page number here */
    BE->gb_line       = 0;              /* line number here */
    BE->gb_flags      = 0;              /* flags       here */
    BE->comment_col   = 30;
    BE->dir_p         = p;
    BE->aview         = aview;
    BE->span_helper   = NULL_span_helper_data;
    BE->pager_helper  = NULL_pager_helper_data;
    BE->wrap_info     = NULL;

    BE->help_text         = NULL;
    BE->help_text_count   = 0;
    BE->self.browser_edit = BE;         /* self linking */

    BE->virtual_line_top  = 0;
    BE->idx_top           = 0;
    BE->buffer_x_top      = 0;
    BE->lineext_top       = NULL;

}

void clear_browser_edit(BE)
     struct browser_edit *BE;
{
    
    if (BROWSER_EDIT_magic != BE->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"clear_browser_edit",
	      "Bad magic number",0);

    BE->dir_p   = NULL;
    BE->aview   = NULL;

    if (BE->wrap_info)
	free_pg_wrap_info(& (BE->wrap_info));

    clear_span_helper_data(& (BE->span_helper));

    if (BE->help_text) {
	int i;

	for (i = 0; i < BE->help_text_count; i++) {
	    if (BE->help_text[i].text)
		free_string(& (BE->help_text[i].text));
	    if (BE->help_text[i].range)
		free_pager_range(& (BE->help_text[i].range));
	}

	free(BE->help_text);
	BE->help_text = NULL;
    }
    BE->help_text_count = 0;
    
    if (BE->lineext_top)
	free_pager_lineext(& (BE->lineext_top));

}

S_(helper_get_param_f br_help_get_param)
static struct pager_param_value * br_help_get_param P_((union pager_help_param   buffer,
							struct pager_range     * range,
							enum pager_param_opcode  opcode,
							int                    * next_pos,
							int                    * found_pos));
static struct pager_param_value * br_help_get_param(buffer,range,opcode,next_pos,found_pos)
     union pager_help_param   buffer;
     struct pager_range     * range;
     enum pager_param_opcode  opcode;
     int                    * next_pos;
     int                    * found_pos;
{

    if (BROWSER_EDIT_magic != buffer.browser_edit->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"br_help_get_param",
	      "Bad magic number",0);

    if (found_pos)
	*found_pos = -1;
    
    return NULL;
}

S_(helper_get_line_f br_help_get_line)
static struct string * br_help_get_line P_((union pager_help_param buffer,
					    int                   idx,
					    int                 * pg_flags,
					    enum range_changed  * range_changed,
					    struct pager_range ** range,
					    struct pager_lineext  ** first_lineext));
static struct string * br_help_get_line(buffer,idx,pg_flags,range_changed,range,
					first_lineext) 
     union pager_help_param   buffer;
     int                      idx;
     int                    * pg_flags;
     enum range_changed     * range_changed;
     struct pager_range    ** range;
     struct pager_lineext  ** first_lineext;
{
    if (range_changed)
	*range_changed = rc_same_range;

    if (first_lineext && *first_lineext)
	free_pager_lineext(first_lineext);

    if (pg_flags)
	*pg_flags = 0;

    if (BROWSER_EDIT_magic != buffer.browser_edit->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"br_help_get_line",
	      "Bad magic number",0);

    if (idx >= buffer.browser_edit->help_text_count) {

	if (range && *range) {
	    free_pager_range(range);
	    if (range_changed)
		*range_changed = rc_newline;
	}

	return NULL;
    }

    if (range) {
	if (*range != buffer.browser_edit->help_text[idx].range) {
	    if (range_changed)
		*range_changed = 
		    get_pager_range_change(*range,
					   buffer.browser_edit->
					   help_text[idx].range);
	    if (*range)
		free_pager_range(range);


	    *range = buffer.browser_edit->help_text[idx].range;
	    if (*range)
		inc_pager_range_refcount(*range);
	}

    }


    if (! buffer.browser_edit->help_text[idx].text)
	return NULL;

    return dup_string( buffer.browser_edit->help_text[idx].text );
}


enum br_help_status browser_draw_help(I)
     struct enter_info *I;
{
    enum br_help_status ret = br_help_none;
    struct string       * buffer = NULL;
    struct pager_range  * pager_range = NULL;

    const int start_line = 3;
    int LINES, COLUMNS;


    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_draw_help",
	      "Bad magic number",0);

    if (! I->browser->wrap_info)
	I->browser->wrap_info = new_pg_wrap_info();

    menu_MoveCursor(I->current_page,start_line,0);    
    menu_CleartoEOS(I->current_page);

    DPRINT(Debug,9,(&Debug, 
		    "browser_draw_help: top virtual=%d idx=%d buffer_x=%d%s\n",
		    I->browser->virtual_line_top,I->browser->idx_top,
		    I->browser->buffer_x_top,
		    I->browser->lineext_top ? ", have lineext" : ""));


    /* Start again from top of help text */
    I->browser->span_helper.idx = I->browser->idx_top;
    
    if (I->browser->span_helper.lineext)
	free_pager_lineext(& (I->browser->span_helper.lineext));
    I->browser->span_helper.lineext = I->browser->lineext_top;
    if (I->browser->span_helper.lineext)
	inc_pager_lineext_refcount(I->browser->span_helper.lineext);
    I->browser->span_helper.buffer_x =
	I->browser->buffer_x_top;
    I->browser->span_helper.virtual_line =
	I->browser->virtual_line_top;
    
    I->browser->span_helper.cur_line = start_line;
    I->browser->span_helper.cur_col  = 0;  /* Number of characters on current line */
    I->browser->span_helper.mayclear = 1;
    I->browser->span_helper.start_of_line = 1;

    I->browser->span_helper.p_width  = pager_paragraph_width;
    I->browser->span_helper.wrap_indicator         = pager_indicate_wrapping ? 1 : 0;

    I->browser->pager_helper.pg_flags   = 0;

    I->browser->pager_helper.search_matches = -1; 
    I->browser->pager_helper.scroll_line    = 0;     /* Not used */

    menu_get_sizes(I->current_page, &LINES, &COLUMNS);

    I->browser->pager_helper.FF_seen = 0;
    I->browser->pager_helper.word_wrapped = 0;
    I->browser->pager_helper.preformatted = 0;
    I->browser->pager_helper.hard_wrapped = 0;

    I->browser->span_helper.joined_line = 0;
    I->browser->span_helper.p_start_COL = 0;
    I->browser->span_helper.cur_col     = 0;    /* On beginning of line */

    while (I->browser->span_helper.idx < I->browser->help_text_count &&
	   I->browser->span_helper.cur_line < LINES-3 &&
	   ! I->browser->pager_helper.FF_seen) {
	
	enum pager_helper_status r;

	if (menu_resized(I->current_page)) {
	    DPRINT(Debug,9,(&Debug, 
			    "browser_draw_help: resized\n"));

	    goto mark_redraw;
	}
	
	if (menu_need_redraw(I->current_page)) {
	    DPRINT(Debug,9,(&Debug, 
			    "browser_draw_help: redraw needed\n"));

	    goto mark_redraw;
	}

	DPRINT(Debug,9,(&Debug, 
			"browser_draw_help  idx = %d (virtual %d) buffer_x=%d start_of_line=%d\n",
			I->browser->span_helper.idx,
			I->browser->span_helper.virtual_line,
			I->browser->span_helper.buffer_x,
			I->browser->span_helper.start_of_line));

	r = pager_helper_process_line(I->current_page,
				      br_help_get_line,
				      I->browser->self,
				      & (I->browser->span_helper),
				      I->browser->wrap_info,
				      &buffer,
				      &pager_range,
				      & (I->browser->pager_helper),
				      NULL /* not supported */,
				      br_help_get_param
				      );

	if (r < 0)  /* EOF */
	      break;
	if (!r)
	      goto failure;   /* FAILURE */

    }

    pager_helper_end_of_page(I->current_page,& (I->browser->span_helper),
			     & (I->browser->pager_helper),LINES-3,0);

 failure:
    if (menu_resized(I->current_page) || 
	menu_need_redraw(I->current_page)) {
    mark_redraw:

	I-> browser->gb_flags |= GB_REDRAW; 
	ret = br_help_redraw;
    }

    if (pager_range)
	free_pager_range(&pager_range);
    if (buffer)
	free_string (&buffer);

    return ret;
}

static void browser_help_next_page P_((struct enter_info *I));
static void browser_help_next_page(I)
     struct enter_info *I;
{
    I->browser->idx_top = I->browser->span_helper.idx;
    
    if (I->browser->lineext_top)
	free_pager_lineext(& (I->browser->lineext_top));
    I->browser->lineext_top = I->browser->span_helper.lineext;
    if (I->browser->lineext_top)	    
	inc_pager_lineext_refcount(I->browser->lineext_top);
    
    I->browser->buffer_x_top = I->browser->span_helper.buffer_x;
    I->browser->virtual_line_top =
	I->browser->span_helper.virtual_line;
}    

void browser_change_help_page(I)     /* Enter changes help page */
     struct enter_info *I;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_change_help_page",
	      "Bad magic number",0);

    /* Start from next page */

    if (I->browser->span_helper.idx < I->browser->help_text_count) {
	
	browser_help_next_page(I);

    } else
	browser_reset_help_page(I);

}

void browser_change_prev_line(I)        
     struct enter_info *I;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_change_prev_line",
	      "Bad magic number",0);

    if (I->browser->virtual_line_top > 0) {
	
	I->browser->virtual_line_top--;

	if (goto_pg_wrap_info(I->browser->wrap_info,
			      I->browser->virtual_line_top,
			      & ( I->browser->idx_top ),
			      & ( I->browser->buffer_x_top),
			      & ( I->browser->lineext_top))) {
	    DPRINT(Debug,10,(&Debug, 
			     "browser_change_prev_line: top line %d ... used wrap info, top index=%d, buffer_x=%d\n",
			     I->browser->virtual_line_top,
			     I->browser->idx_top,I->browser->buffer_x_top));
	    
	} else {
	    DPRINT(Debug,10,(&Debug, 
			     "browser_change_prev_line: top line %d ... no wrap info\n",
			     I->browser->virtual_line_top));

	    browser_reset_help_page(I); 
	}
	
    } else
	browser_reset_help_page(I);
}

void browser_change_next_line(I)     
     struct enter_info *I;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_change_next_line",
	      "Bad magic number",0);

    if (I->browser->span_helper.idx < I->browser->help_text_count) {
	int tmp = I->browser->virtual_line_top;

	I->browser->virtual_line_top++;

	if (goto_pg_wrap_info(I->browser->wrap_info,
			      I->browser->virtual_line_top,
			      & ( I->browser->idx_top ),
			      & ( I->browser->buffer_x_top),
			      & ( I->browser->lineext_top))) {
	    DPRINT(Debug,10,(&Debug, 
			     "browser_change_next_line: top line %d ... used wrap info, top index=%d, buffer_x=%d\n",
			     I->browser->virtual_line_top,
			     I->browser->idx_top,I->browser->buffer_x_top));
	    
	} else {
	    DPRINT(Debug,10,(&Debug, 
			     "browser_change_next_line: top line %d ... no wrap info\n",
			     I->browser->virtual_line_top));

	    I->browser->virtual_line_top = tmp;
	}
    }
}

void browser_change_page_up(I)
     struct enter_info *I;
{
    int LINES, COLUMNS;

    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_reset_page_up",
	      "Bad magic number",0);

    menu_get_sizes(I->current_page, &LINES, &COLUMNS);

    if (I->browser->virtual_line_top > 0) {
	
	I->browser->virtual_line_top -= LINES-3;
	if (I->browser->virtual_line_top < 0)
	    I->browser->virtual_line_top = 0;

	if (goto_pg_wrap_info(I->browser->wrap_info,
			      I->browser->virtual_line_top,
			      & ( I->browser->idx_top ),
			      & ( I->browser->buffer_x_top),
			      & ( I->browser->lineext_top))) {
	    DPRINT(Debug,10,(&Debug, 
			     "browser_change_page_up: top line %d ... used wrap info, top index=%d, buffer_x=%d\n",
			     I->browser->virtual_line_top,
			     I->browser->idx_top,I->browser->buffer_x_top));
	    
	} else {
	    DPRINT(Debug,10,(&Debug, 
			     "browser_change_page_up: top line %d ... no wrap info\n",
			     I->browser->virtual_line_top));

	    browser_reset_help_page(I); 
	}
    }
}

void browser_change_page_down(I)
     struct enter_info *I;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_reset_page_down",
	      "Bad magic number",0);

   if (I->browser->span_helper.idx < I->browser->help_text_count) {	
       browser_help_next_page(I);
   }
}


void browser_reset_help_page(I)     
     struct enter_info *I;
{
    if (BROWSER_EDIT_magic != I->browser->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_reset_help_page",
	      "Bad magic number",0);

    I->browser->span_helper.idx              = 0;
    I->browser->span_helper.lineext          = 0;
    I->browser->span_helper.buffer_x         = 0;
    I->browser->span_helper.force_word_wrap  = 0;
    I->browser->span_helper.p_width          = pager_paragraph_width;
    I->browser->span_helper.wrap_indicator   = pager_indicate_wrapping ? 1 : 0;    

    I->browser->pager_helper.pg_flags        = 0;

    if (! I->browser->wrap_info)
	I->browser->wrap_info = new_pg_wrap_info();
    
    I->browser->virtual_line_top  = 0;
    I->browser->idx_top           = 0;
    I->browser->buffer_x_top      = 0;

    if (I->browser->lineext_top)
	free_pager_lineext(& (I->browser->lineext_top));

}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
