static char rcsid[] = "@(#)$Id: command.c,v 1.21 2020/03/21 08:28:58 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.21 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_command.h"
#include "s_command.h"

DEBUG_VAR(Debug,__FILE__,"command");


static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}


/* ---------------------------------  */

#define COMMAND_magic        0xE701

static struct elm_command {
    unsigned short              magic;     /* COMMAND_magic */

    int   *key;                    /* malloced */
    int    key_count;
    int flags;

    int    refcount;
    
    struct elm_commands *is_prefix;
    
    struct string *desc;
    struct string *keyext;
        
} * malloc_elm_command P_((void));

static struct elm_command * malloc_elm_command() 
{
    struct elm_command * ret = safe_malloc(sizeof (*ret));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));
     
    ret->flags     = 0;

    ret->refcount  = 1;            /* Caller must able to free this */

    ret->key       = NULL;
    ret->key_count = 0;
    ret->is_prefix = NULL;
    ret->desc      = NULL;
    ret->keyext    = NULL;
						

    ret->magic = COMMAND_magic;

    return ret;
}

static void inc_elm_command_refcount P_((struct elm_command *ptr));
static void inc_elm_command_refcount(ptr)
     struct elm_command *ptr;
{
    if (COMMAND_magic != ptr->magic) 
	panic("COMMANDS PANIC",__FILE__,__LINE__,"inc_elm_command_refcount",
	      "Bad magic number",0);
   
    ptr->refcount++;
}

static void free_elm_command P_((struct elm_command **ptr));
static void free_elm_command(ptr)
     struct elm_command **ptr;
{
    if (COMMAND_magic != (*ptr)->magic) 
	panic("COMMANDS PANIC",__FILE__,__LINE__,"free_elm_command",
	      "Bad magic number",0);

    if ((*ptr)->refcount < 1)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"free_elm_command",
	      "Bad refcount",0);

    (*ptr)->refcount--;

    if ((*ptr)->refcount > 0) {   /* Do not free */

	(*ptr) = NULL;           /* Refefence count for this pointer is decrement, */
	                         /* so this must have be reset */
	return;
    }

    if ((*ptr)->key) {
	/* Just a array */
	free((*ptr)->key);
	(*ptr)->key = NULL;
    }
    (*ptr)->key_count = 0;

    if ((*ptr)->is_prefix) {
	DPRINT(Debug,30,(&Debug,
			 "free_elm_command:   %p freeing prefix %p [pointer %p]\n",
			 *ptr,(*ptr)->is_prefix,(& ((*ptr)->is_prefix))));
	free_commands(& ((*ptr)->is_prefix)); /* Just decrements refcount */
    }
    if ((*ptr)->desc)
	free_string(& ((*ptr)->desc));	   
    if ((*ptr)->keyext)
	free_string(& ((*ptr)->keyext));	   


    (*ptr)->magic   =  0;   /* Invalidate */
    free(*ptr);
    (*ptr) = NULL;
}


/* ---------------------------------- */

#define COMMANDS_magic       0xE700


static struct elm_commands  {
    unsigned short              magic;     /* COMMANDS_magic */

    int                         refcount;   /* free_commands() free when this goes to zero */

    struct string *key_title;
    struct string *desc_title;

    struct help_line {
	struct string *line;
	int            pg_flags;
    }	*           title;
    int             title_count;

    struct help_line *trailer;
    int               trailer_count;

    struct command_line {
	struct elm_command *line;
	int                 pg_flags;
    } * command;
    int  command_count;

    int pointer_list_idx;

} * malloc_commands P_((void));         /* Returns refrence count 1 */


static struct elm_commands ***pointers = NULL;
static int                    pointers_count = 0;

static struct elm_commands* malloc_commands() 
{
    struct elm_commands * ret = safe_malloc(sizeof (*ret));

    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)ret,sizeof (*ret));
    
    ret->key_title     = NULL;
    ret->desc_title    = NULL;

    ret->title         =  NULL;
    ret->title_count   =  0;

    ret->trailer       = NULL;
    ret->trailer_count = 0;

    ret->command       = NULL;
    ret->command_count = 0;

    ret->magic     = COMMANDS_magic;
    ret->refcount  = 1;                  /* Caller must able to free this */

    ret->pointer_list_idx  = -1;

    DPRINT(Debug,30,(&Debug,"malloc_commands: %p , > refcount=%d\n",
		     ret,ret->refcount));

    return ret;
}


static void add_commands_key P_((struct elm_commands *cmd,
				 int flags,
				 const int *key, int key_count,
				 struct string *keyext /* shared, free'ed by free_elm_command */,
				 struct elm_commands *is_prefix,
				 struct string *desc /* shared, free'ed by free_elm_command */,
				 int pg_flags
				 ));

static void add_commands_key(cmd,flags,key,key_count,keyext,is_prefix,desc,pg_flags)
     struct elm_commands *cmd;
     int flags;
     const int *key;      /* copy malloced */
     int key_count;
     struct string *keyext /* shared, free'ed by free_elm_command */;
     struct elm_commands *is_prefix;
     struct string *desc; /* shared, free'ed by free_elm_commands */
     int pg_flags;
{
    struct elm_command * ptr = malloc_elm_command();

    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_key",
	      "Bad magic number",0);
    
    if (key_count > 0) {
	int i;

	ptr->key = 
	    safe_calloc(key_count, sizeof(ptr->key[0]));

	for (i = 0; i < key_count; i++)
	    ptr->key[i] = key[i];
	ptr->key_count = key_count;
    } 


    if (is_prefix) {
	int v;

	if (COMMANDS_magic != is_prefix->magic)
	    panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_key",
		  "Bad (is_prefix) magic number",0);
	
	ptr->is_prefix = is_prefix;

	/* Prevent free_commands() to free this */
	v = inc_commands_refcount(ptr->is_prefix);

	DPRINT(Debug,30,(&Debug,
			 "add_commands_key: %p [%d]=%p prefix %p refcount = %d [pointer %p]\n",
			 cmd,cmd->command_count,ptr,
			 ptr->is_prefix,v,
			 & (ptr->is_prefix)));

	if (v < 2) {
	    panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_key",
		  "Bad recount",0);
	}
    }

    ptr->flags     = flags;    
    ptr->desc      = desc;   /* shared, free'ed by free_elm_command */
    ptr->keyext    = keyext; /* shared, free'ed by free_elm_command */

    cmd->command = safe_array_realloc(cmd->command,
				      (cmd->command_count+1),
				      sizeof (cmd->command[0]));
    
    cmd->command[cmd->command_count].line     = ptr;
    cmd->command[cmd->command_count].pg_flags = pg_flags;

    cmd->command_count++;
}


/* ----------------------------------- */


/* (key_title,desc_title) pointers are shared -- 
 *  must be malloced and caller must not free them 
 */
struct elm_commands * new_commands(key_title,desc_title,assign)
     struct string *key_title;
     struct string *desc_title;
     struct elm_commands **assign;
{
    struct elm_commands *ret = malloc_commands();

    ret->key_title = key_title;     /* shared, free'ed by free_commands */
    ret->desc_title = desc_title;   /* shared, free'ed by free_commands */

    if (assign) {
	int i;
	int v UNUSED_VAROK;

	if (*assign) 
	    panic("COMMANDS PANIC",__FILE__,__LINE__,"new_commands",
		  "Already value",0);

	*assign = ret;
	v = inc_commands_refcount(*assign);


	for (i = 0; i < pointers_count; i++) {
	    if (assign == pointers[i]) {

		DPRINT(Debug,30,(&Debug,
				 "new_commands: %p > pointers [%d] %p, refcount %d\n",
				 ret,i,assign,v));


		ret->pointer_list_idx = i;

		goto done;
	    }
	}

	pointers = safe_array_realloc(pointers, 
				      (pointers_count+1), sizeof (pointers[0]));

	DPRINT(Debug,30,(&Debug,
			 "new_commands: %p > pointers [%d] %p, refcount %d\n",
			 ret,pointers_count,assign,v));

	ret->pointer_list_idx = pointers_count; 

	pointers[pointers_count++] = assign;
    }

 done:
    return ret;            /* refcount == 1 */
}

struct elm_commands * new_dummy_prefixed_commands()
{
    struct elm_commands *ret = malloc_commands();

    return ret;  /* refcount == 1 */
}

/* decrement refcount */
void free_commands(cmd) 
     struct elm_commands **cmd;
{
    DPRINT(Debug,30,(&Debug,"free_commands: %p ~ pointer %p\n",
		     *cmd,cmd));


    if (COMMANDS_magic != (*cmd)->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"free_commands",
	      "Bad magic number",0);

    if ((*cmd)->refcount < 1)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"free_commands",
	      "Bad refcount",0);
	
    
    (*cmd)->refcount--;

    DPRINT(Debug,30,(&Debug,"free_commands: %p , > refcount=%d [pointer %p], pointer [%d]\n",
		     *cmd,(*cmd)->refcount,cmd,
		     (*cmd)->pointer_list_idx));

    if ((*cmd)->refcount > 0) {  /* Do not free */

	if ((*cmd)->pointer_list_idx >= 0) {
	    
	    int i = (*cmd)->pointer_list_idx;
	    
	    if (i >= pointers_count || * (pointers[i]) != *cmd) {
		
		panic("COMMANDS PANIC",__FILE__,__LINE__,"free_commands",
		  "Bad pointer_list_idx",0);
	    
	    } 

	    if (pointers[i] == cmd) {
		DPRINT(Debug,30,(&Debug,
				 "free_commands: %p -- this was stored pointer [%d]\n",
				 *cmd,i));

		(*cmd)->pointer_list_idx = -1;				 
	    }

	}


	(*cmd) = NULL;           /* Refefence count for this pointer is decrement, */
	                         /* so this must have be reset */
	return;
    }

    if ((*cmd)->key_title) 
	free_string(& ((*cmd)->key_title));
    if ((*cmd)->desc_title)
	free_string(& ((*cmd)->desc_title));

    if ((*cmd)->title) {
	int i;

	for (i = 0; i < (*cmd)->title_count; i++) {
	    if ((*cmd)->title[i].line)
		free_string(& ((*cmd)->title[i].line));			    
	}

	free((*cmd)->title);
	(*cmd)->title = NULL;
    }
    (*cmd)->title_count = 0;

    if ((*cmd)->trailer) {
	int i;

	for (i = 0; i < (*cmd)->trailer_count; i++) {
	    if ((*cmd)->trailer[i].line)
		free_string(& ((*cmd)->trailer[i].line));
	}

	free((*cmd)->trailer);
	(*cmd)->trailer = NULL;

    } 
    (*cmd)->trailer_count = 0;


    if ((*cmd)->command) {
	int i;

	for (i = 0; i < (*cmd)->command_count; i++) {
	    if ((*cmd)->command[i].line)
		free_elm_command(& ((*cmd)->command[i].line));
	}

	free((*cmd)->command);
	(*cmd)->command = NULL;
    }
    (*cmd)->command_count = 0;
	   
    if ((*cmd)->pointer_list_idx >= 0) {

	int i = (*cmd)->pointer_list_idx;
	
	if (i >= pointers_count || * (pointers[i]) != *cmd) {
	    
	    panic("COMMANDS PANIC",__FILE__,__LINE__,"free_commands",
		  "Bad pointer_list_idx",0);
	    
	} 


	if (pointers[i] == cmd) {
	    DPRINT(Debug,30,(&Debug,
			     "free_commands: %p -- this was stored pointer [%d]\n",
			     *cmd,i));
	    
	    (*cmd)->pointer_list_idx = -1;				 

	} else {

	    panic("COMMANDS PANIC",__FILE__,__LINE__,"free_commands",
		  "Have pointer_list_idx -- bad refcount = 0",0);

	}
    }
    
    DPRINT(Debug,30,(&Debug,"free_commands: %p : free'ed ~ %p reset\n",
		     *cmd,cmd));


    (*cmd)->magic         = 0;   /* Invalidate */
    free(*cmd);
    (*cmd) = NULL;
}        

void free_cached_commands()
{
    if (pointers) {   
	int i;
	
	for (i = 0; i < pointers_count; i++) {

	    if (* (pointers[i])) {

		DPRINT(Debug,30,(&Debug,
				 "free_cached_commands: %p ~ pointers [%d] %p\n",
				 * (pointers[i]),i,pointers[i]));

		free_commands(pointers[i]);

	    }
	}

	free(pointers);
	pointers = NULL;
    }
    pointers_count = 0;
}


int inc_commands_refcount(cmd)
    struct elm_commands *cmd; /* returns refcount */
{
    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"inc_commands_refcount",
	      "Bad magic number",0);

    
    cmd->refcount++;

    DPRINT(Debug,30,(&Debug,"inc_commands_refcount: %p , > refcount=%d\n",
		     cmd,cmd->refcount));

    return cmd->refcount;
}
void add_commands_title_shared(cmd,pg_flags,title)
     struct elm_commands *cmd;
     int pg_flags;
     struct string *title /* shared, free'ed by free_commands */;
{
    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_title_shared",
	      "Bad magic number",0);
    
    cmd->title = safe_array_realloc(cmd->title,
				    (cmd->title_count+1), sizeof (cmd->title[0]));
    
    cmd->title[cmd->title_count].line     = title;
    cmd->title[cmd->title_count].pg_flags = pg_flags;
    
    cmd->title_count++;    
}


void add_commands_title(
#if ANSI_C
			struct elm_commands *cmd,
			int pg_flags,
			const char * format, const char *msg, ...
#else
			cmd, pg_flags,format, msg, va_alist
#endif
			)
#if !ANSI_C
    struct elm_commands *cmd;
    int pg_flags;
    const char * format; 
    const char * msg;
    va_dcl
#endif
{
  va_list vl;
  struct string * store1;

    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_title",
	      "Bad magic number",0);

  DPRINT(Debug,61,(&Debug,   
		   "add_commands_title: format=%s\n",format));
  DPRINT(Debug,61,(&Debug,   
		   "                     msg=%s\n",msg));
  	            
  Va_start(vl, msg);           /* defined in hdrs/elm_defs.h  */

  store1 = elm_smessage(0,format,msg,vl);

  va_end(vl);

  add_commands_title_shared(cmd,pg_flags,store1);
}

void add_commands_trailer(
#if ANSI_C
			  struct elm_commands *cmd, int pg_flags,
			  const char * format, const char *msg, ...
#else
			  cmd, pg_flags, format, msg, va_alist
#endif
			)
#if !ANSI_C
    struct elm_commands *cmd;
    int pg_flags;
    const char * format; 
    const char *msg;
    va_dcl
#endif
{
  va_list vl;
  struct string * store1;

    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_trailer",
	      "Bad magic number",0);


  DPRINT(Debug,61,(&Debug,   
		   "add_commands_trailer: format=%s\n",format));
  DPRINT(Debug,61,(&Debug,   
		   "                     msg=%s\n",msg));
  	            
  Va_start(vl, msg);           /* defined in hdrs/elm_defs.h  */

  store1 = elm_smessage(0,format,msg,vl);

  va_end(vl);

  cmd->trailer = safe_array_realloc(cmd->trailer,
				    (cmd->trailer_count+1), sizeof (cmd->trailer[0]));

  cmd->trailer[cmd->trailer_count].line = store1;
  cmd->trailer[cmd->trailer_count].pg_flags = pg_flags;
  cmd->trailer_count++;
}


void add_commands_help_single(
#if ANSI_C
			      struct elm_commands *cmd,
			      struct elm_commands *is_prefix,
			      int flags, int key,
			      const char * format, const char *msg, ...
#else
			      cmd,is_prefix,flags,key,format,msg,va_alist
#endif
			      )
#if !ANSI_C
    struct elm_commands *cmd;
    struct elm_commands *is_prefix;
    int flags;
    int key;
    const char * format; 
    const char *msg;
    va_dcl
#endif
{
  va_list vl;
  struct string * store1;

  if (COMMANDS_magic != cmd->magic)
      panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_single",
	    "Bad magic number",0);

  if (is_prefix && 
      COMMANDS_magic != is_prefix->magic)
      panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_single",
	    "Bad (is_prefix) magic number",0);

  DPRINT(Debug,61,(&Debug,   
		   "add_commands_help_single: format=%s\n",format));
  DPRINT(Debug,61,(&Debug,   
		   "                           msg=%s\n",msg));
  	            
  Va_start(vl, msg);           /* defined in hdrs/elm_defs.h  */

  store1 = elm_smessage(0,format,msg,vl);

  va_end(vl);

  add_commands_key(cmd,flags,&key,1,NULL,is_prefix,store1,0);
}




/* keyext pointer is shared -- 
 *  must be malloced and caller must not free it
 */
void add_commands_help_single_keyext(
#if ANSI_C
			      struct elm_commands *cmd,
			      struct elm_commands *is_prefix,
			      int flags, int key, struct string *keyext,
			      const char * format, const char *msg, ...
#else
			      cmd,is_prefix,flags,keyext,key,format,msg,va_alist
#endif
			      )
#if !ANSI_C
    struct elm_commands *cmd;
    struct elm_commands *is_prefix;
    int flags;
    int key;
    struct string *keyext;
    const char * format; 
    const char *msg;
    va_dcl
#endif
{
  va_list vl;
  struct string * store1;

  if (COMMANDS_magic != cmd->magic)
      panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_single",
	    "Bad magic number",0);

  if (is_prefix && 
      COMMANDS_magic != is_prefix->magic)
      panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_single",
	    "Bad (is_prefix) magic number",0);

  DPRINT(Debug,61,(&Debug,   
		   "add_commands_help_single: format=%s\n",format));
  DPRINT(Debug,61,(&Debug,   
		   "                           msg=%s\n",msg));
  	            
  Va_start(vl, msg);           /* defined in hdrs/elm_defs.h  */

  store1 = elm_smessage(0,format,msg,vl);

  va_end(vl);

  add_commands_key(cmd,flags,&key,1,keyext,is_prefix,store1,0);
}

void add_commands_help_s_shared(cmd,flags,key,help)
     struct elm_commands *cmd; 
     int flags;
     int key; 
     struct string *help /* shared, free'ed by 
			    free_commands */;
{
    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_s_shared",
	      "Bad magic number",0);

    add_commands_key(cmd,flags,&key,1,NULL,NULL,help,0);
}

void add_commands_help_many(
#if ANSI_C
			    struct elm_commands *cmd,
			    int flags,
			    int *keys_table,  /* 0 terminated */
			    const char * format, const char *msg, ...
#else
			    cmd,flags,keys_table,format,msg,va_alist
#endif
			    )
#if !ANSI_C
    struct elm_commands *cmd;
    int flags;
    int keys_table[];  /* 0 terminated */
    const char * format;
    const char *msg; 
    va_dcl
#endif
{
  va_list vl;
  struct string * store1;
  int count = 0;

  if (COMMANDS_magic != cmd->magic)
      panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_many",
	    "Bad magic number",0);
    
  DPRINT(Debug,61,(&Debug,   
		   "add_commands_help_many: format=%s\n",format));
  DPRINT(Debug,61,(&Debug,   
		   "                           msg=%s\n",msg));
  	            
  Va_start(vl, msg);           /*defined in hdrs/elm_defs.h  */

  store1 = elm_smessage(0,format,msg,vl);

  va_end(vl);

  while(keys_table[count])
      count++;

  add_commands_key(cmd,flags,keys_table,count,NULL,NULL,store1,0);
}


void add_commands_help_many2(
#if ANSI_C
			    struct elm_commands *cmd,
			    int flags,
			    int *keys_table,  /* 0 terminated */
			    int pg_flags,
			    const char * format, const char *msg, ...
#else
			    cmd,flags,keys_table,pg_flags,format,msg,va_alist
#endif
			    )
#if !ANSI_C
    struct elm_commands *cmd;
    int flags;
    int keys_table[];  /* 0 terminated */
    int pg_flags;
    const char * format;
    const char *msg; 
    va_dcl
#endif
{
  va_list vl;
  struct string * store1;
  int count = 0;

  if (COMMANDS_magic != cmd->magic)
      panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_many2",
	    "Bad magic number",0);
    
  DPRINT(Debug,61,(&Debug,   
		   "add_commands_help_many: format=%s\n",format));
  DPRINT(Debug,61,(&Debug,   
		   "                           msg=%s\n",msg));
  	            
  Va_start(vl, msg);           /* defined in hdrs/elm_defs.h  */

  store1 = elm_smessage(0,format,msg,vl);

  va_end(vl);

  while(keys_table[count])
      count++;

  add_commands_key(cmd,flags,keys_table,count,NULL,NULL,store1,pg_flags);
}



void add_commands_help_many_prefix(
#if ANSI_C
			    struct elm_commands *cmd,
			    struct elm_commands *is_prefix,
			    int flags,
			    int *keys_table,  /* 0 terminated */
			    const char * format, const char *msg, ...
#else
			    cmd,is_prefix,flags,keys_table,format,msg,va_alist
#endif
			    )
#if !ANSI_C
    struct elm_commands *cmd;
    struct elm_commands *is_prefix;
    int flags;
    int keys_table[];  /* 0 terminated */
    const char * format;
    const char *msg; 
    va_dcl
#endif
{
  va_list vl;
  struct string * store1;
  int count = 0;

  if (COMMANDS_magic != cmd->magic)
      panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_many",
	    "Bad magic number",0);
    
  DPRINT(Debug,61,(&Debug,   
		   "add_commands_help_many: format=%s\n",format));
  DPRINT(Debug,61,(&Debug,   
		   "                           msg=%s\n",msg));
  	            
  Va_start(vl, msg);           /* defined in hdrs/elm_defs.h */

  store1 = elm_smessage(0,format,msg,vl);

  va_end(vl);

  while(keys_table[count])
      count++;

  add_commands_key(cmd,flags,keys_table,count,NULL,is_prefix,store1,0);
}



static struct string * key_text_one P_((int key));
static struct string * key_text_one(key)
     int key;
{
    int ch = key;
    struct string *result = NULL;

    if (key >= TERMCH_min_char) {
	ch =  report_TERMCH_char(key);

	if (ch != -1)
	    goto letter;
    }
    
    if (key >  REDRAW_MARK) {
	switch (key) {
	case DOWN_MARK: 
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyDown,"<DOWN>"));
	    break;
	case UP_MARK:   
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyUp,"<UP>"));
	    break;
	case LEFT_MARK: 
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyLeft,"<LEFT>"));
	    break;
	case RIGHT_MARK: 
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyRight,"<RIGHT>"));
	    break;
	case HOME_MARK: 
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyHome,"<Home>"));
	    break;
	case DELETE_MARK: 
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyDelete,"<Delete>"));
	    break;
	case INSERT_MARK: 
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyInsert,"<Insert>"));
	    break;
	case PAGEDOWN_MARK:
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyPageDown,"<Page Down>"));
	    break;
	case PAGEUP_MARK:
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyPageUp,"<Page Up>"));
	    break;
	case BACKTAB_MARK:  
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyBackTab,"<BACKTAB>"));
	    break;
	case END_MARK:
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyEnd,"<End>"));
	    break;	
	case HELP_MARK:	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyHelp,"<Help>"));
	    break;
	case FIND_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyFind,"<Find>"));
	    break;

	case  NUMBER_MARK:            /* Only used on help display */
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyNumber,"<number>"));
	    break;

	case F1_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF1,"<F1>"));
	    break;
	case F2_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF2,"<F2>"));
	    break;
	case F3_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF3,"<F3>"));
	    break;
	case F4_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF4,"<F4>"));
	    break;
	case F5_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF5,"<F5>"));
	    break;
	case F6_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF6,"<F6>"));
	    break;
	case F7_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF7,"<F7>"));
	    break;
	case F8_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF8,"<F8>"));
	    break;
	case F9_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF9,"<F9>"));
	    break;
	case F10_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF10,"<F10>"));
	    break;
	case F11_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF11,"<F11>"));
	    break;
	case F12_KEY_MARK: 	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyF12,"<F12>"));
	    break;

	    
	default:
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyUnknown,"<unknown>"));
	    break;
	}
    }

    if (!result &&
	isascii(key)) {
	ch = key;
 letter:
	
	if (ctrl('?') == ch) 
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyDEL,"^? (DEL)"));
	else if (' ' == ch)
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandSpaceKey,
					   "<space>"));
	else if (ch < 32 && ch >= 0) {
	    char c = '@' + ch;
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyCtrl,"^%c (ctrl-%c)"),
				   c,c);
	} else if (
#ifdef ASCII_CTYPE
		   isascii(ch) &&
#endif
		   isprint(ch)) {
	    result = format_string(FRM("%c"),ch);
	} else { /* If TERMCH is not ascii */
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandKeyBadAscii,
					   "{%02x} (unknown)"),
				   ch);
	}    
    }


    return result;
}


static struct string * key_text P_((int keys_table[], int key_count));
static struct string * key_text(keys_table,key_count)
     int keys_table[]; 
     int key_count;
{
    struct string * result = NULL;
    int i;

    for (i = 0; i < key_count; i++) {
	struct string * s = NULL;

	if (i < key_count-1 &&
	    '\n' == keys_table[i] &&
	    '\r' == keys_table[i+1]) {
	    
	    result = format_string(CATGETS(elm_msg_cat, CommandSet,
					   CommandReturnKey,
					   "<return>"));
	    i++;

	} else {

	    /* Ignore keys which do not have text */
	    s = key_text_one(keys_table[i]);
	    
	    if (s) {
		DPRINT(Debug,10,(&Debug,   
				 "key_text: [%d] %d = %S\n",
				 i,keys_table[i],s));
	    }
	}

	if (s) {

	    if (result) {
		add_ascii_to_string(result,s2us(", "));
	    }
	    
	    append_string(&result,s,1);
	    free_string(&s);
	}	
    }

    if (!result)
	result = format_string(CATGETS(elm_msg_cat, CommandSet,
				       CommandKeyNotAvailable,"n/a"));

    DPRINT(Debug,10,(&Debug,   
		     "key_text = %S\n",
		     result));

    return result;
}

static void command_expand_prefix P_((struct out_state    * buffer,
				      int keys[], int prefix_len,
				      struct elm_commands *nested,
				      int L, int W,
				      struct pager_range *cmd_range));
static void command_expand_prefix(buffer,keys,prefix_len,nested,L,W,cmd_range)
     struct out_state    * buffer;
     int keys[]; 
     int prefix_len;
     struct elm_commands *nested;
     int L; 
     int W;
     struct pager_range *cmd_range;
{

    struct string * prefix = NULL;
    int i;

    for (i = 0; i < prefix_len; i++) {
	struct string * s = key_text_one(keys[i]);

	append_string(&prefix,s,1);
	free_string(&s);
    }


    for (i = 0; i < nested->command_count; i++) {

	struct elm_command * ptr = nested->command[i].line;

	if (ptr) {
	    if (COMMAND_magic != ptr->magic) 
		panic("COMMANDS PANIC",__FILE__,__LINE__,
		      "command_expand_prefix",
		      "Bad (elm_command) magic number",0);

	    if (ptr->key_count > 0 &&
		0 == ( FLAG_KEY_no_nested & ptr->flags)) {
		struct string * a = key_text_one(ptr->key[0]);
				
		struct string * b = cat_strings(prefix,a,1);
		int L1;
		
		if (ptr->keyext)
		  append_string(&b,ptr->keyext,1);
	    
		L1 = string_len(b);

		set_out_state_line_pager_range(buffer,cmd_range);
		if (nested->command[i].pg_flags)
		    set_out_state_line_pg_flags(buffer,nested->command[i].pg_flags);

		
		state_printf(buffer,FRM("  %-*S"),L,b);
		if (L1 <= L) 
		    state_puts("  ",buffer);
		else {
		    state_putc('\n',buffer);

		    set_out_state_line_pager_range(buffer,cmd_range);
		    if (nested->command[i].pg_flags)
			set_out_state_line_pg_flags(buffer,nested->command[i].pg_flags);

		    state_printf(buffer,FRM("  %*s  "),L,"");
		}
		state_printf(buffer,FRM("%S\n"),ptr->desc);

		/* Newline resets pg_flags */

		free_string(&b);
		free_string(&a);
	    }

	    if (ptr->is_prefix && ptr->key_count > 0 &&
		prefix_len < 8) {
	    
		int j;
		
		int * array = safe_calloc((prefix_len+1), sizeof (*array));
		
		for (j = 0; j < prefix_len; j++) 
		    array[j]  = keys[j];
		array[prefix_len] = ptr->key[0];
		
		command_expand_prefix(buffer,
				      array,prefix_len+1,
				      ptr->is_prefix,
				      L,W,cmd_range);
		
		free(array);
	    }
	}
    }

    free_string(&prefix);

}



struct stringbuffer * give_commands_help(cmd)
     struct elm_commands *cmd;
{
    struct stringbuffer * bout = create_membuffer();
    struct out_state    * buffer = new_out_state(STATE_out_buffer);


    struct pager_range * title_range = NULL;
    struct pager_range * cmd_range = NULL;
    struct pager_range * desc_range = NULL;
    int i;
    int keylen = 0;
    int L = 0;
    int W = 0;
       
    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"give_commands_help",
	      "Bad magic number",0);

    set_out_state_sbuffer (bout,buffer);

    /* state_add_simple_pager_range requires that stringbuffer is setup
       -- that is: set_out_state_sbuffer is called
    */

    title_range = 
	state_add_simple_pager_range(buffer,NULL,PR_MAX_WIDTH,0,0);
    
    cmd_range = 
	state_add_simple_pager_range(buffer,NULL,
				     PR_MAX_WIDTH|PR_WORD_WRAP,0,0);

    if (cmd->title_count) {

	for (i = 0; i < cmd->title_count; i++) {
	    set_out_state_line_pager_range(buffer,title_range);

	    if (cmd->title[i].pg_flags)
		set_out_state_line_pg_flags(buffer,cmd->title[i].pg_flags);

	    state_printf(buffer,FRM("%S\n"),cmd->title[i].line);

	    /* Newline resets pg_flags */
	}

	state_nlputs("\n",buffer);
    }

    for (i = 0; i < cmd->command_count; i++) {
	int len1 = 0;

	struct elm_command * ptr = cmd->command[i].line;

	if (ptr) {
	    int j;

	    if (COMMAND_magic != ptr->magic) 
		panic("COMMANDS PANIC",__FILE__,__LINE__,
		      "give_commands_help",
		      "Bad (elm_command) magic number",0);

	    for (j = 0; j < ptr->key_count; j++) {
		if (ptr->key[j] >= TERMCH_min_char)
		    len1  += 4;
		else if (ptr->key[j] > REDRAW_MARK)
		    len1  += 13;
		else len1 += 3;		
	    }

	    if (ptr->keyext)
		len1 += string_len(ptr->keyext);

	    if (ptr->is_prefix && keylen < 10)
		keylen = 10;

	    if (keylen < len1)
		keylen = len1;
	}
    }
		     
    if (cmd->key_title && cmd->desc_title) {
	/* Does not count double width characters */
	L = string_len(cmd->key_title);
    }

    if (keylen > 20 && L < 25)
	L = 25;
    else if (keylen > L)
	L = keylen;

    /* Does indent on wrapping */
    desc_range = state_add_simple_pager_range(buffer,
					      cmd_range,PR_SAME_LINE,
					      0,L+4);
					      
	
    if (cmd->key_title && cmd->desc_title) {

	int have_unrderline =
	    set_out_state_line_pg_flags(buffer,pg_UNDERLINE);

	set_out_state_line_pager_range(buffer,title_range);

	W = string_len(cmd->desc_title);
	
	state_printf(buffer,FRM("  %-*S  %S\n"),
		     L,cmd->key_title,cmd->desc_title);

	if (!have_unrderline) { /* Should not be needed */

	    set_out_state_line_pager_range(buffer,title_range);

	    state_puts("  ",buffer);
	    for (i = 0; i < L; i++)
		state_putc('-',buffer);
	    state_puts("  ",buffer);
	    for (i = 0; i < W; i++)
		state_putc('-',buffer);
	}
	state_nlputs("\n",buffer);

	/* Newline resets pg_flags */
    }

    for (i = 0; i < cmd->command_count; i++) {

	struct elm_command * ptr = cmd->command[i].line;

	if (ptr) {
	    struct string *keys;
	    int L1;

	    if (COMMAND_magic != ptr->magic) 
		panic("COMMANDS PANIC",__FILE__,__LINE__,
		      "give_commands_help",
		      "Bad (elm_command) magic number",0);

       
	    keys = key_text(ptr->key,
			    ptr->key_count);
	    
	    if (ptr->keyext)
	        append_string(&keys,ptr->keyext,1);

	    L1 = string_len(keys);
	    
	    set_out_state_line_pager_range(buffer,cmd_range);
	    if (cmd->command[i].pg_flags)
		set_out_state_line_pg_flags(buffer,cmd->command[i].pg_flags);

	    state_printf(buffer,FRM("  %-*S"),L,keys);
	    if (L1 <= L) 
		state_puts("  ",buffer);
	    else {
		state_putc('\n',buffer);

		set_out_state_line_pager_range(buffer,cmd_range);
		if (cmd->command[i].pg_flags)
		    set_out_state_line_pg_flags(buffer,
						cmd->command[i].pg_flags);
		
		state_printf(buffer,FRM("  %*s  "),L,"");
	    }
	    
	    /* Set indent if this is wrapping */
	    set_out_state_line_mode(buffer,cmd->command[i].pg_flags,
				    desc_range,0);
	    state_printf(buffer,FRM("%S\n"),ptr->desc);

	    /* Newline resets pg_flags */

	    if (ptr->is_prefix && ptr->key_count > 0) {
		
		command_expand_prefix(buffer,
				      & (ptr->key[0]),1,
				      ptr->is_prefix,
				      L,W,cmd_range); 
		
	    }

	    if (keys)
		free_string(&keys);
	}
    }

    if (cmd->trailer_count) {
	state_nlputs("\n",buffer);

	for (i = 0; i < cmd->trailer_count; i++) {
	    set_out_state_line_pager_range(buffer,cmd_range);

	    if (cmd->trailer[i].pg_flags)
		set_out_state_line_pg_flags(buffer,
					    cmd->trailer[i].pg_flags);
	    
	    state_printf(buffer,FRM("%S\n"),cmd->trailer[i].line);

	    /* Newline resets pg_flags */
	}
    }
    
    free_out_state(&buffer);
    free_pager_range(&desc_range);
    free_pager_range(&cmd_range);
    free_pager_range(&title_range);

    return bout;
}

const struct elm_command * lookup_elm_command(cmd,key)
     struct elm_commands *cmd;
     int key;
{
    int i;
    int key1 = -1;

    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"lookup_elm_command",
	      "Bad magic number",0);

    if (isascii(key) &&
	isdigit(key))
	key1 = NUMBER_MARK;   /* Only for display */

    for (i = 0; i < cmd->command_count; i++) {

	struct elm_command * ptr = cmd->command[i].line;

	if (ptr) {
	    int j;

	    if (COMMAND_magic != ptr->magic) 
		panic("COMMANDS PANIC",__FILE__,__LINE__,
		      "lookup_elm_command",
		      "Bad (elm_command) magic number",0);
	    
	    for (j = 0; j < ptr->key_count; j++) {
		if (ptr->key[j] == key)
		    return ptr;
		if (ptr->key[j] == key1)
		    return ptr;
	    }
	}
    }

    return NULL;
}

/* Caller must free result */
struct string * elm_command_key_text(C)
     const struct elm_command *C;
{
    struct string * X = NULL;
    if (COMMAND_magic != C->magic) 
	panic("COMMANDS PANIC",__FILE__,__LINE__,"elm_command_key_text",
	      "Bad magic number",0);
   
    X = key_text(C->key,C->key_count);

    if (C->keyext)
      append_string(&X,C->keyext,1);

    return X;
}

/* Shared result, caller must not free result */
const struct string * elm_command_key_desc(C)
     const struct elm_command *C;
{
    if (COMMAND_magic != C->magic) 
	panic("COMMANDS PANIC",__FILE__,__LINE__,"elm_command_key_desc",
	      "Bad magic number",0);

    return C->desc;
}

struct string * elm_command_key1_desc(key)
     int key;
{
    return key_text_one(key);
}

static int classify_key P_((int keys_table[], int key_count, int *c));

static int classify_key(keys_table,key_count,c)
     int keys_table[];
     int key_count;
     int *c;
{
    *c = 0;

    if (key_count > 1 &&
	'\n' == keys_table[0] && '\r' == keys_table[1]) 
	return 0;
    if (key_count > 0 &&
	NUMBER_MARK == keys_table[0])
	return '0';
	
    if (key_count > 0 &&
	' ' <= keys_table[0] && '@' >= keys_table[0])
	return keys_table[0];

    if (key_count > 0 &&
	' ' > keys_table[0]) {
	*c = 4;
	return keys_table[0] + '@';
    }

    if (key_count > 0 &&
	'a' <= keys_table[0] && 'z' >= keys_table[0]) {
	*c = 2;
	
	return (keys_table[0] - 'a') + 'A';
    }

    if (key_count > 0 &&
	'A' <= keys_table[0] && 'Z' >= keys_table[0]) {
	*c = 1;
	
	return keys_table[0];
    }

    if (key_count > 0)
	return keys_table[0];

    return 0;
}

static int compare_command P_((const void *p1, const void *p2));
static int compare_command(p1,p2)
     const void *p1; 
     const void *p2;
{
    const struct command_line  * l1 = p1;
    const struct command_line  * l2 = p2;

    struct elm_command * cmd1 = l1->line;
    struct elm_command * cmd2 = l2->line;

    int k1 = 0, c1 = 0;
    int k2 = 0, c2 = 0;
    
    if (COMMAND_magic != cmd1->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"compare_command",
	      "Bad magic number (elm_command) #1",0);
    if (COMMAND_magic != cmd2->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"compare_command",
	      "Bad magic number (elm_command) #2",0);

    /* ASCII ASSUMED */
    k1 = classify_key(cmd1->key,cmd1->key_count, &c1);
    k2 = classify_key(cmd2->key,cmd2->key_count, &c2);
    
    if (k1 < k2 || 
	(k1 == k2 && c1 < c2))
	return -1;
    if (k2 < k1 ||
	(k2 == k1 && c2 < c1))
	return 1;
    if (l1->pg_flags < l2->pg_flags)
	return -1;
    if (l2->pg_flags < l1->pg_flags)
	return 1;
    return 0;
}

void sort_commands(cmd)
     struct elm_commands *cmd;
{
    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"sort_commands",
	      "Bad magic number (cmd)",0);

    if (cmd->command_count > 1)
	qsort(cmd->command,cmd->command_count,
	      sizeof (cmd->command[0]),compare_command);
}

/* Increments refcount or make duplicate */
static struct elm_command  * filter_command
  P_((struct elm_command   *ptr,
      const struct command_line  *filter_list,
      int                   filter_list_count));

static struct elm_command * filter_command(ptr,filter_list,filter_list_count)
     struct elm_command  * ptr;
     const struct command_line  * filter_list;
     int                   filter_list_count;
{
    struct elm_command * ret = NULL;

    int * key = NULL;
    int   keycount = 0;

    if (COMMAND_magic != ptr->magic) 
	panic("COMMANDS PANIC",__FILE__,__LINE__,"filter_command",
	      "Bad magic number",0);

    if (ptr->key_count > 0) {
	int i;
	key = safe_calloc(ptr->key_count, sizeof(key[0]));

	for (i = 0; i < ptr->key_count; i++) {
	    const int letter = ptr->key[i];
	    int j;
	    
	    for (j = 0; j <  filter_list_count; j++) {

		if (filter_list[j].line) {
		    struct elm_command  * f = filter_list[j].line;
		    int k;
		    
		    if (COMMAND_magic != f->magic) 
			panic("COMMANDS PANIC",__FILE__,__LINE__,"filter_command",
			      "Bad magic number",0);
		    for (k = 0; k < f->key_count; k++)
			if (letter ==  f->key[k])
			    goto duplicate;
		}
	    }
	    
	    key[keycount++] = letter;

	duplicate:
	    ;
	}		
    }
    
    if (keycount == ptr->key_count) {
	ret = ptr;
	inc_elm_command_refcount(ptr);
    } else if (keycount > 0) {
	/* make duplicate */
	
	ret = malloc_elm_command();
	ret->flags= ptr->flags;
	
	ret->key = key;  key = NULL;
	ret->key_count= keycount;
	
	ret->is_prefix = ptr->is_prefix;
	if (ret->is_prefix) {
	    inc_commands_refcount(ret->is_prefix);
	}
	
	if (ptr->desc)
	    ret->desc = dup_string(ptr->desc);
	
	if (ptr->keyext)
	    ret->keyext = dup_string(ptr->keyext);
    }
    
    if (key)
	free(key);
    
    return ret;
}
     
/* Only adds commands, not other texts */
void add_commands_help_merge(cmd,addition,addition_pg_flags)
     struct elm_commands *cmd;
     const struct elm_commands *addition;
     int addition_pg_flags;
{
    struct command_line  * src = NULL;
    int                   src_count = 0;
    int                   alloced;
    int i1, i2;

    if (COMMANDS_magic != cmd->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_merge",
	      "Bad magic number (cmd)",0);

    if (COMMANDS_magic != addition->magic)
	panic("COMMANDS PANIC",__FILE__,__LINE__,"add_commands_help_merge",
	      "Bad magic number (addition)",0);

    alloced = cmd->command_count + addition->command_count;
    if (alloced < 1)
	return;

    src        = cmd->command;
    src_count  = cmd->command_count;

    cmd->command = safe_calloc(alloced, sizeof (cmd->command[0]));
    cmd->command_count = 0;

    for (i1= 0, i2 = 0; 
	 i1 < src_count && i2 < addition->command_count &&
	     cmd->command_count < alloced; ) {
	int k1 = 0, c1 = 0;
	int k2 = 0, c2 = 0;

	/* This merge is incomplete and may produce wrong result on 
	   patalogical case
	*/

	if (! src[i1].line) {
	    i1++;
	    continue;
	}

	if (! addition->command[i2].line) {
	    i2++;
	    continue;
	}

	/* ASCII ASSUMED */
	k1 = classify_key(src[i1].line->key,src[i1].line->key_count, &c1);

	k2 = classify_key(addition->command[i2].line->key,
			  addition->command[i2].line->key_count,&c2);

	if (k1 < k2 || 
	    (k1 == k2 && c1 < c2)) {
	    cmd->command[cmd->command_count] = src[i1];
	    inc_elm_command_refcount(cmd->command[cmd->command_count].line);

	    cmd->command_count++;
	    i1++;
	} else if (k2 < k1 ||
		   (k2 == k1 && c2 < c1)) {

	    /* Increments refcount or make duplicate */
	    struct elm_command * addc = filter_command(addition->command[i2].line,
						       src,src_count);

	    if (addc) {
		cmd->command[cmd->command_count].line = addc;
		cmd->command[cmd->command_count].pg_flags =
		    addition->command[i2].pg_flags |= addition_pg_flags;
						
		cmd->command_count++;
	    }
	    i2++;
			
	} else {  /* Discard duplicate from merge */
	    cmd->command[cmd->command_count] = src[i1];
	    inc_elm_command_refcount(cmd->command[cmd->command_count].line);

	    cmd->command_count++;
	    i1++;
	    i2++;
	}
    }
	 

    while(i1 < src_count && cmd->command_count < alloced) {
	cmd->command[cmd->command_count] = src[i1];
	inc_elm_command_refcount(cmd->command[cmd->command_count].line);
	
	cmd->command_count++;
	i1++;
    }

    while (i2 < addition->command_count &&
	   cmd->command_count < alloced) {

	if (src) {
	    /* Increments refcount or make duplicate */
	    struct elm_command * addc = filter_command(addition->command[i2].line,
						       src,src_count);
	    
	    if (addc) {
		cmd->command[cmd->command_count].line = addc;
		cmd->command[cmd->command_count].pg_flags =
		    addition->command[i2].pg_flags |= addition_pg_flags;
		
		cmd->command_count++;
	    }
	} else {
	    cmd->command[cmd->command_count] = addition->command[i2];
	    cmd->command[cmd->command_count].pg_flags |= addition_pg_flags;
	    
	    inc_elm_command_refcount(cmd->command[cmd->command_count].line);
	    
	    cmd->command_count++;
	}
	i2++;    
    }


    if (src) {
	int i;

	for (i = 0; i < src_count;i++)
	    if (src[i].line)
		free_elm_command(& src[i].line);
	free(src);
    }
}

/* increment refcount -- does not touch original */

struct elm_commands *give_merged_commands(base,append,pg_flags_base,
					  pg_flags_append)
     const struct elm_commands *base;
     const struct elm_commands *append;
     int pg_flags_base,pg_flags_append;
{
    struct elm_commands *ret = malloc_commands();

    if (base->key_title)
	ret->key_title = dup_string(base->key_title);

    if (base->desc_title)
	ret->desc_title = dup_string(base->desc_title);

    if (base->title_count > 0) {
	int i;

	ret->title = safe_calloc(base->title_count,
				 sizeof (ret->title[0]));

	for (i = 0; i < base->title_count; i++) {
	    
	    if (base->title[i].line) {
		ret->title[ret->title_count].line =
		    dup_string(base->title[i].line);
		ret->title[ret->title_count].pg_flags =
		    base->title[i].pg_flags;

		ret->title_count++;
	    }
	}	
    }

    if (base->trailer_count > 0) {
	int i;

	ret->trailer = safe_calloc(base->trailer_count,
				   sizeof (ret->trailer[0]));

	for (i = 0; i < base->trailer_count; i++) {
	    if (base->trailer[i].line) {
		ret->trailer[ret->trailer_count].line =
		    dup_string(base->trailer[i].line);
		ret->trailer[ret->trailer_count].pg_flags = 
		    base->trailer[i].pg_flags;

		ret->trailer_count++;
	    }
	}
    }

    add_commands_help_merge(ret,base,pg_flags_base);
    add_commands_help_merge(ret,append,pg_flags_append);

    return ret;
}        

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
