static char rcsid[] = "@(#)$Id: edithdr.c,v 2.4 2021/07/07 16:28:00 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.4 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_elm.h"
#include "s_elm.h"


DEBUG_VAR(Debug,__FILE__,"header");

enum mode_xx { mode_plain, mode_phrase, mode_comment };

static void print_edit_string P_((enum mode_xx mode,FILE *F,charset_t edit_charset,
				  int encondig_supported,
				  const struct string *s,
				  int *L, int width));
static void print_edit_string(mode,F,edit_charset,encoding_supported,s,L,width)
     enum mode_xx          mode;
     FILE                * F;
     charset_t             edit_charset;
     int                   encoding_supported;
     const struct string * s;
     int                 * L;
     int                   width;
{
    int l = *L;

    int mime_code = encoding_supported &&
	get_string_type(s) != edit_charset;
    const char * MIME_name  = 
	get_string_MIME_name(s);
    int extra = 0;
    int X, nextX;
    int Slen = string_len(s);
    int startX = 0;
    int do_ascii_prefix = 0;
    int is_encoded = 0;

    if (!MIME_name)
	MIME_name = "UNKNOWN-8BIT";

    if (mime_code)
	extra = 5 + strlen(MIME_name);
		

    DPRINT(Debug,25,(&Debug,"print_edit_string: s=%S\n",s));

    switch (mode) {
    case mode_phrase:
	DPRINT(Debug,25,(&Debug,"print_edit_string: mode phrase\n"));
	break;
    case mode_comment: 
	DPRINT(Debug,25,(&Debug,"print_edit_string: mode comment\n"));
	putc('(',F); 
	l++;
	break;
    case mode_plain:
	DPRINT(Debug,25,(&Debug,"print_edit_string: mode plain\n"));
	do_ascii_prefix = 1;
	break;
    default:
	break;
    }

    for (X = 0; X < Slen; X = nextX) {
	int foundX    = -1;
	struct string * s2 = 0;
	char * s1 = NULL;

	if (l+(Slen-startX)+extra < width-5) {
	    foundX = Slen;

	    DPRINT(Debug,25,(&Debug,
			     "print_edit_string: to end of string %d (split not needed)\n",
			     Slen));
	    foundX = Slen;

	} else {
	    int X1;

	    for (X1 = X; X1 < Slen; X1++) {
		uint16 code = give_unicode_from_string(s,X1);
		
		switch (code) {
		case 0x0009 /* HT  '\t' */:
		case 0x0020 /* SPACE    */:
		    
		    if (-1 == foundX ||
			l+(X1-startX)+extra < width-5) {
			foundX    = X1;
		    }
		    break;
		}
		
		if (-1 == foundX && 
		    (startX-X1) > width)
		    break;
	    }
	    
	    if (-1 == foundX) {
		DPRINT(Debug,25,(&Debug,
				 "print_edit_string: to end of string %d (space not found)\n",
				 X1));
		foundX = X1;
	    } else if (foundX == startX) {
		DPRINT(Debug,25,(&Debug,
				 "print_edit_string: found space on beginning %d\n",
				 foundX));
		foundX++;
	    }
	}

	s2 = clip_from_string(s,&startX,
			      foundX-startX);


	DPRINT(Debug,25,(&Debug,"print_edit_string: -> %d: s2=%S\n",
			 startX,s2));


	switch (mode) {
	case mode_phrase:
	    s1 = string_to_hdr(HDR_PHRASE,s2, edit_charset,mime_code, &is_encoded,
			       do_ascii_prefix);
	   
	    break;
	case mode_comment:
	    s1 = string_to_hdr(HDR_COMMENT,s2, edit_charset,mime_code, &is_encoded,
			       do_ascii_prefix);
	    break;

	case mode_plain:
	default:
	    s1 = string_to_hdr(HDR_TEXT,s2, edit_charset,mime_code, &is_encoded,
			       do_ascii_prefix);

	    break;
	}


	/* Estimate */

	if (!mime_code || check_8bit_str(s1))
	    l += string_len(s2);    /* Must not be encoded */
	else
	    l +=  strlen(s1);       /* Assumed to be MIME encoded or originally ascii */


	DPRINT(Debug,25,(&Debug,"print_edit_string: -> %d: s1='%s'\n",
			 startX,s1));

	fputs(s1,F); 
        
	nextX = startX;

	if (startX < Slen) {
	    uint16 code = give_unicode_from_string(s,startX);
	    
	    switch (code) {
	    case 0x0009 /* HT  '\t' */:
		putc('\n',F);
		putc('\t',F);
		l = 8;
		
		nextX = startX+1;
		if (is_encoded)
		    do_ascii_prefix = 0;
		else
		    startX++;
		break;
		
	    case 0x0020 /* SPACE    */:
		putc('\n',F);
		putc(' ',F);
		l = 1;
		
		nextX = startX+1;
		if (is_encoded)
		    do_ascii_prefix = 0;
		else
		    startX++;
		break;
	    default:
		DPRINT(Debug,25,(&Debug,
				 "print_edit_string: Split not ended %d to space, code=%04x\n",
				 startX,code));
		if (mime_code) {
		    putc('\n',F);
		    putc(' ',F);
		    l = 1;
		    
		    if (is_encoded)
			do_ascii_prefix = 0;
		}
		break;
	    }
	}

	free(s1);
	free_string(&s2);
    }


    switch (mode) {
    case mode_comment: putc(')',F); 
	l++;
	break;
    default:                        break;
    }
    
    *L = l;
}


static void  add_addr_header P_((FILE * F,charset_t edit_charset,
				 int encoding_supported,
				 char * name, 
				 struct expanded_address *a,
				 int group_supported,
				 int width));

static void  add_addr_header(F,edit_charset,encoding_supported,name,a,
			     group_supported,
			     width)
     FILE                    * F;
     charset_t                 edit_charset;
     int                       encoding_supported;
     char                    * name; 
     struct expanded_address * a;
     int                       group_supported;
     int                       width;
{
    int l = strlen(name) + 2;
    int widx = 0;

    fprintf(F,"%s: ",name);

    if (a->addrs) {
	int idx;
	int addr_item_count = addr_list_item_count(a->addrs);
	int group_item_count = addr_list_group_count(a->addrs);
	int current_group = -1;
	int last_group = -1;

	for (idx = 0; idx < addr_item_count; idx++) {

	    int group = -1;
	    const struct address * address = 
		addr_list_get_item(a->addrs,idx,&group);
	    
	    const char          * ad   = address_get_ascii_addr(address);
	    const struct string * f    = address_get_phrase(address);
	    const struct string * c    = address_get_comment(address);
	    int sl;

	    if (current_group != group &&
		current_group != -1) {
		putc(';',F);
		current_group = -1;		
		l++;
	    }
	    
	    if (!ad)
		continue;

	    if (idx > 0) {
		putc(',',F);
		l++;

		if ((width > 40) ? (l > width -20) : (l > 60)) {
		    putc('\n',F);
		    l = 0;
		}
		putc(' ',F);
		l++;
	    }

	    if (group >= 0 &&
		current_group == -1) {

		const struct string * groupname1 =
		    addr_list_get_group(a->addrs,group);


		int g;
		
		/* Print empty groups */

		for (g = last_group+1; g < group; g++) {
		    const struct string * groupname =
			 addr_list_get_group(a->addrs,g);

		    print_edit_string (mode_phrase,F,edit_charset,
				       encoding_supported,groupname,
				       &l,width);
		    fputs(":;,",F);
		    l += 3;

		    if (l > 60) {
			putc('\n',F);
			l = 0;
		    }
		    putc(' ',F);
		    l++;
		}

		if (last_group < group)
		    last_group = group;

		print_edit_string (mode_phrase,F,edit_charset,
				   encoding_supported,groupname1,
				   &l,width);

		putc(':',F);
		l ++;
		
		if ((width > 40) ? (l > width -20) : (l > 60)) {
		    putc('\n',F);
		    l = 0;
		}
		putc(' ',F);
		l++;
		current_group = group;			       
	    }

	    if (f && (sl = string_len(f)) > 0) {

		if (idx > widx && l + sl >= width-5 && l > 1) {
		    fputs("\n ",F);
		    l = 1;
		    widx = idx;
		}

		print_edit_string (mode_phrase,F,edit_charset,encoding_supported,f,
				   &l,width);
		putc(' ',F);
		l ++;
	    }

	    sl = strlen(ad);
	    
	    if (idx > widx && l + sl >= width-5 && l > 1) {
		fputs("\n ",F);
		l = 1;
		widx = idx;
	    }
	    putc('<',F);
	    fputs(ad,F);  l += sl + 2;
	    putc('>',F);

	    if (c && (sl = string_len(c)) > 0) {
		if (idx > widx && l + sl >= width-5 && l > 1) {
		    putc('\n',F);
		    l = 0;
		    widx = idx;
		} 
		putc(' ',F);
		l++;
		
		print_edit_string (mode_comment,F,edit_charset,encoding_supported,c,
				   &l,width);
	    }	    
	}

	if (current_group != -1) {
	    putc(';',F);
	    l++;
	}

	/* Print empty groups */

	if (last_group+1 < group_item_count) {
	    int g;

	    if (idx > 0) {
		putc(',',F);
		l++;
		if ((width > 40) ? (l > width -20) : (l > 60)) {
		    putc('\n',F);
		    l = 0;
		}
		putc(' ',F);
		l++;
	    }

	    for (g = last_group+1; g < group_item_count; g++) {
		const struct string * groupname =
		    addr_list_get_group(a->addrs,g);
		
		print_edit_string (mode_phrase,F,edit_charset,
				   encoding_supported,groupname,
				   &l,width);
		fputs(":;",F);
		l += 2;

		if (g < group_item_count-1) {
		    putc(',',F);
		    l++;
		    if ((width > 40) ? (l > width -20) : (l > 60)) {
			putc('\n',F);
			l = 0;
		    }
		    putc(' ',F);
		    l++;
		}
	    }
	}
    }

    putc('\n',F);
}

static void add_string_header P_((FILE * F,charset_t edit_charset,
				  int encoding_supported,char * name, 
				  struct string *s,
				  int width));
static void add_string_header(F,edit_charset,encoding_supported,name,s,width)
     FILE          * F;
     charset_t       edit_charset;
     int             encoding_supported;
     char          * name; 
     struct string * s;
     int             width;
{
    int l = strlen(name) + 1;

    fprintf(F,"%s: ",name);
    print_edit_string (mode_plain,F,edit_charset,encoding_supported,s,
		       &l,width);
    putc('\n',F);

}

static void print_edit_id P_((FILE * F,charset_t edit_charset,
			     int encoding_supported,
			     const struct message_id * id,
			     int *L, int width));
static void print_edit_id(F,edit_charset,encoding_supported,
			 id,L,width)
     FILE * F;
     charset_t edit_charset;
     int encoding_supported;
     const struct message_id * id;
     int *L;
     int width;
{
    int l = *L;
	
    const char * left                = message_id_left(id);
    const char * domain              = message_id_domain(id);
    const struct string * id_comment = message_id_comment(id);
    int sl;

    putc('<',F);
    l++;
    
    if (left) {
	fputs(left,F);
	l += strlen(left);
    }

    if (domain) {
	putc('@',F);
	fputs(domain,F);
	l += strlen(domain) + 1;
    }
    
    putc('>',F);
    l++;
    
    if (id_comment && (sl = string_len(id_comment)) > 0) {
	if (l + sl >= width-5 && l > 1) {
	    putc('\n',F);
	    l = 0;
	} 
	putc(' ',F);
	l++;
	
	print_edit_string (mode_comment,F,edit_charset,encoding_supported,
			   id_comment,
			   &l,width);
    }	
    
    *L = l;
}
			    
static void add_id_header P_((FILE * F,charset_t edit_charset,
			      int encoding_supported,char * name, 
			      struct message_id *id,
			      int width));
static void add_id_header(F,edit_charset,encoding_supported,name,id,width)
     FILE              * F;
     charset_t           edit_charset;
     int                 encoding_supported;
     char              * name; 
     struct message_id * id;
     int                 width;
{
    int l = strlen(name) + 1;
    
    fprintf(F,"%s: ",name);

    if (id) {
	print_edit_id(F,edit_charset,encoding_supported,id,&l,width);
    }

    putc('\n',F);
}

static void print_edit_str P_((FILE *F,const char *str,int *L, int width));
static void print_edit_str(F,str,L,width)
     FILE *F;
     const char *str;
     int *L;
     int width;
{
    int l = *L;
    const char *s;

    for (s = str; *s; s++) {
	if (whitespace(*s)) {
	    if ((width > 40) ? (l > width -20) : (l > 60)) {
		putc('\n',F);
		l = 0;
	    }
	    putc(*s,F);
	} else {
	    putc(*s,F);
	}
	l++;	    	
    }
    
    *L = l;
}

static void add_expires_header P_((FILE * F,
				   char * name, 
				   struct expanded_expires *expires_value,
				   int width));
static void add_expires_header(F,name,expires_value,width)
     FILE                    * F;
     char                    * name; 
     struct expanded_expires * expires_value;
     int                       width;
{
    int l = strlen(name) + 1;

    fprintf(F,"%s: ",name);
    
    if (expires_value) {
	char timebuf[SLEN] = "";

	enum print_time print_time = get_expanded_expires_print_time(expires_value);

	if (days_ahead_hdrval(expires_value,timebuf,sizeof timebuf,
			      print_time)) {

	    print_edit_str(F,timebuf,&l,width);	     
	}
    }
    putc('\n',F);
}

static void add_date_header P_((FILE * F,
				char * name, 
				struct expanded_date *date_value,
			      int width));
static void add_date_header(F,name,date_value,width)
     FILE                 * F;
     char                 * name; 
     struct expanded_date * date_value;
     int                    width;
{
    int l = strlen(name) + 1;
    
    fprintf(F,"%s: ",name);

    if (date_value) {
	char timebuf[SLEN] = "";
	
	 if (date_hdrval(date_value,timebuf,sizeof timebuf,
			 print_date_and_time /* print_time */)) {

	     print_edit_str(F,timebuf,&l,width);	     
	 }
    }
    putc('\n',F);
}


static void add_refs_header P_((FILE * F,charset_t edit_charset,
			      int encoding_supported,char * name, 
			      struct references *refs,
			      int width));
static void add_refs_header(F,edit_charset,encoding_supported,name,refs,width)
     FILE              * F;
     charset_t           edit_charset;
     int                 encoding_supported;
     char              * name; 
     struct references * refs;
     int                 width;
{
    int l = strlen(name) + 1;

    
    fprintf(F,"%s: ",name);

    if (refs) {
	int idx;
	int add_sep = 0;
	int count = references_item_count(refs);
	
	for (idx = 0; idx < count; idx++) {
	    
	    /* message_id also includes comment */
	    const struct message_id * id = references_get_message_id(refs,idx);
	    const struct string * phrase = references_get_phrase(refs,idx);
	    const struct string * tail_comment = references_get_comment(refs,idx);
	    
	    int sl;
	    
	    if (id) {
		if (add_sep) {
		    if ((width > 40) ? (l > width -20) : (l > 60)) {
			putc('\n',F);
			l = 0;
		    }
		    putc(' ',F);
		    l++;
		}
		
		print_edit_id(F,edit_charset,encoding_supported,id,&l,width);
		add_sep = 1;
	    }
	    
	    if (phrase && (sl = string_len(phrase)) > 0) {
		if (add_sep) {
		    if (l + sl >= width-5 && l > 1) {
			putc('\n',F);
			l = 0;
		    }
		    putc(' ',F);
		    l++;
		}
		
		print_edit_string (mode_phrase,F,edit_charset,
				   encoding_supported,phrase,
				   &l,width);
		add_sep = 1;
	    }
	    
	    if (tail_comment && (sl = string_len(tail_comment)) > 0) {
		if (add_sep) {
		    if (l + sl >= width-5 && l > 1) {
			putc('\n',F);
			l = 0;
		    }
		    putc(' ',F);
		    l++;
		}
		
		print_edit_string (mode_comment,F,edit_charset,encoding_supported,
				   tail_comment,
				   &l,width);
		add_sep = 1;
	    }
	}
    }

    putc('\n',F);
}

void  edit_headers_on_editor(headers,editor, encoding_supported,
				    page)
     struct mailing_headers *headers;    
     const char *editor;
     int encoding_supported;
     struct menu_context *page;
{
    charset_t   edit_charset = system_charset;
    FILE * F = NULL; 
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);

    char * filename = elm_message(FRM("%s%sHDR%d"), 
				  tmp ? tmp : "/tmp/", 
				  temp_file, getpid());
    char * buffer = NULL;
    char buffer1[32*1024];
    int stat;
    int c;
    int err;
    int LINES, COLUMNS;

    menu_get_sizes(page, &LINES, &COLUMNS);   

    if (in_string(editor, "%s"))
	buffer = elm_message(FRM(editor), 
			     filename);
    else
	buffer = elm_message(FRM("%s %s"), 
			     editor, filename);


    F = safeopen_rdwr(filename,NULL);
    if (!F) 
	goto fail1;

    PutLineX(LINES-1, 0, CATGETS(elm_msg_cat, ElmSet, ElmInvokeEditor,
				   "Invoking editor..."));
    FlushBuffer();


    /* FIX: Following prints with display_charset and
            not with edit_charset (system_charset)
    */

    {
	const char *MIME_name_e = 
	    get_charset_MIME_name(edit_charset);
	
	elm_fprintf(F,
		    CATGETS(elm_msg_cat, ElmSet, ElmHdrEditComment,
			    "# Headers are automatically converted from %s charset and not need encoded\n"),
		    MIME_name_e ? MIME_name_e : "<no MIME name>");
    }

    if (encoding_supported) {
	elm_fprintf(F,
		    CATGETS(elm_msg_cat, ElmSet, ElmHdrEditCommentEnc,
			    "# MIME encoding of headers is supported\n"));

    } else {
	elm_fprintf(F,
		    CATGETS(elm_msg_cat, ElmSet, ElmHdrEditCommentNo,
			    "# MIME encoding of headers is NOT supported\n"));

    }

    add_addr_header(F,edit_charset,encoding_supported,
		    "From", & headers->from,0,COLUMNS);
    add_addr_header(F,edit_charset,encoding_supported,
		    "To", & headers->to,1,COLUMNS);
    add_addr_header(F,edit_charset,encoding_supported,
		    "Cc", & headers->cc,1,COLUMNS);
    add_addr_header(F,edit_charset,encoding_supported,
		    "Bcc", & headers->bcc,1,COLUMNS);
    add_addr_header(F,edit_charset,encoding_supported,
		    "Reply-To", & headers->reply_to,1,COLUMNS);
    add_string_header(F,edit_charset,encoding_supported,
		      "Subject", headers->subject,COLUMNS);
    add_id_header(F,edit_charset,encoding_supported,
		  "Message-ID",headers->message_id,COLUMNS);
    if (headers->date) {
	switch (get_expanded_date_source(headers->date)) {
	case date_automatic:
	    elm_fprintf(F,
			CATGETS(elm_msg_cat, ElmSet, ElmHdrEditDUpdate,
				"# Date is updated automatically, if Date: is not modified.\n"));
	break;
	case date_explicit:
	    elm_fprintf(F,
			CATGETS(elm_msg_cat, ElmSet, ElmHdrEditDClear,
				"# Date is generated automatically, if Date: is cleared.\n"));
	    break;	    
	}
    } else {
	elm_fprintf(F,
		    CATGETS(elm_msg_cat, ElmSet, ElmHdrEditDEmpty,
			    "# Date is generated automatically, if Date: is left empty.\n"));
    }
    add_date_header(F,"Date",headers->date,COLUMNS);
    add_expires_header(F,"Expires",headers->expires,COLUMNS);
    add_refs_header(F,edit_charset,encoding_supported,
		    "In-reply-to",headers->in_reply_to,COLUMNS);
    add_refs_header(F,edit_charset,encoding_supported,
		    "References",headers->references,COLUMNS);

    if (headers->xmailer) {
	elm_fprintf(F,
		    CATGETS(elm_msg_cat, ElmSet, ElmHdrEditXMEmpty,
			    "# X-Mailer is removed from a message, if X-Mailer: is cleared and left empty.\n"));
	add_string_header(F,edit_charset,encoding_supported,
			  "X-Mailer", headers->xmailer,COLUMNS);

	if (dt_flag_is_set(&program_snd_ident,snd_ident_user_agent)) {
	    elm_fprintf(F,
		       CATGETS(elm_msg_cat, ElmSet, ElmHdrEditUserAgent,
			       "# User-Agent field includes same information and is not editable.\n\
# See program-identification elm.rc option.\n"));
	}
	
    }
    
    fclose(F);

    (void) elm_chown(filename, userid, groupid, NULL);
    
    Raw(OFF);
    /* FIX: SY_DUMPSTATE should perhaps be supported? */
    if ((stat = system_call(buffer, SY_ENAB_SIGHUP,
			    NULL)) == -1) {
	int err UNUSED_VAROK = errno;

	Raw(ON);
	DPRINT(Debug,1,(&Debug,
			"System call failed with stat %d (edit_headers_on_editor)\n", 
			stat));
	DPRINT(Debug,1,(&Debug,
		   "** %s **\n", strerror(err)));
	ClearLine(LINES-2);
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantInvokeEditor,
			  "Can't invoke editor '%s' for composition."), 
		  editor);
	sleep_message();

	goto fail2;
    }
    Raw(ON);

    err = can_open(filename,"r");
    if (err) {
	lib_error(FRM("%s: %s"),filename,err);
	
	goto fail1;
    }

    F = fopen(filename,"r");
    if (!F) {
	/* In than point opening should have succeed */
	int err = errno;
	lib_error(FRM("%s: %s"),filename,err);
	
	goto fail1;
    }

    /* Skip comment block */
    
    while ('#' == (c = getc(F))) {
	
	while ('\n' != (c = getc(F))) {
	    if (EOF == c)
		break;
	}
    }
    
    if (EOF != c)
	ungetc(c,F);


    /* Read headers */
    while (0 < read_header_line(F,buffer1,sizeof buffer1,0)) {

	header_ptr header_name;
	char *k;

	if ('#' == buffer1[0])
	    continue;

	k = strchr(buffer1,':');
	if (!k)
	    break;
	*k = '\0';
	k++;
	
	while (whitespace(*k))
	    k++;
	
	header_name = find_header(buffer1,1);
	
	if (!add_to_mailing_header(headers,header_name,k,
				   encoding_supported,
				   edit_charset,1 /* REPLACE */ )) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmHdrNotSupported,
			      "Editing of %s header is not supported."),
		      buffer1);
	}
    }

    if (!feof(F))
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmHdrGarbage,
			  "Garbage on header file"));

    fclose(F);

 fail2:
    unlink(filename);

 fail1:

    free(filename); filename = NULL;
    free(buffer);   buffer = NULL;

}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
