static char rcsid[] = "@(#)$Id: elm.c,v 2.54 2021/09/22 17:50:59 hurtta Exp $";

/*****************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.54 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 *  Based on Elm 2.4 src/elm.c. That code have following copyright:
 *
 *  The Elm Mail System 
 *
 * This file and all associated files and documentation:
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 *  Incorparated Elm 2.5 code from src/elm.c. 
 *  That code have following copyright:
 *
 *  The Elm Mail System
 *
 * This file and all associated files and documentation:
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/* Main program of the ELM mail system! 
*/

#include "patchlevel.h"

#define INTERN
#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"ELM");


static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

static void do_check_only P_((char *argv[],
			      struct AliasView *aview)); 
static void debug_screen P_((struct MailboxView *mailbox,
			     struct screen_parts *LOC));
static void debug_message P_((struct header_rec *hdr, int current,
			      struct folder_view *index));

static void elm_panic_prepare P_((const int interrupt, 
				  const char * title,
				  const char * ms));
static void elm_panic_prepare(interrupt,title,ms)
     const int interrupt;
     const char * title;
     const char * ms;
{
    int do_cursor = RawState();

    if (do_cursor && !interrupt) {
	lib_error(FRM("%s: %s"),title,ms);
	sleep(1);
    }

    if (do_cursor) {
	/* MoveCursor() is not safe on interrupt */

	if (interrupt) {
	    ClearScreen(1);    /* 1 == is called from interrupt */

	    signal_leave_screen();
	} else {
	    int LINES, COLUMNS;

	    menu_get_sizes(default_context, &LINES, &COLUMNS);   

	    MoveCursor(LINES-1,0);
	    Raw(OFF);
	}
    }
}

static void elm_panic_exit P_((const int interrupt));
static void elm_panic_exit(interrupt)
     const int interrupt;
{
    emergency_exit(interrupt);
}

void forget_passphrase() 
{

    
    if (use_PGP &&
	pgp_void_passphrase())
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmPGPPassphraseForgotten,
			  "PGP passphrase forgotten!"));

#ifdef REMOTE_MBX
    {
	struct cancel_data *cd = NULL;
	
	setup_mbx_cancel_message(&cd,mbx_closing_con);

	DPRINT(Debug,9, (&Debug, "forget_passphrase: Closing cached connections\n"));
	
	if(close_cached_connections(cd))
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
	                      ElmCachedConenctionsClosed,
	                     "Cached connections closed!"));

	if (is_canceled(cd)) {
	    DPRINT(Debug,9, (&Debug, "forget_passphrase: Cancel on close_cached_connections\n"));
        }
	
        free_cancel(&cd);	
    }
#endif

    error_sleep(1);
}

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif         

int mailbox_command(mailbox, ch,  LOC, cmdX, aview, 
		    page, pager_cmds, 
		    is_main_message_loop, menu_cmds)
     struct MailboxView *mailbox;
     int ch;   
     struct screen_parts *LOC;
     showmsg_cmd  cmdX;
     struct AliasView *aview;
     struct menu_context  *page;
     struct elm_commands *pager_cmds;  
     int is_main_message_loop;
     struct elm_commands *menu_cmds;
{
    struct menu_common MENU;
    int LINES, COLUMNS;

    menu_get_sizes(page, &LINES, &COLUMNS);   
    

    set_mcommon_from_mbxview(&MENU,mailbox);

    switch (ch) {

    case '!'    :  
	menu_Writechar(LOC->prompt_page,'!'); 
	FlushBuffer();

	if (ALLOW_subshell)
	    subshell(mailbox, page,LOC->prompt_page);		
	else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoSubshellCommand,
			      "Subshell command isn't configured in this version of ELM."));

	break;


    case '<'    :  {
	/* scan current message for calendar information */

	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, 
				     ElmSet, 
				     ElmScanForCalendar,
				     "Scan message for calendar entries..."));
	FlushBuffer();

	if (ENABLE_calendar) {
	    int current = get_current(mailbox);
	    
	
	    if  (current < 1) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNoMailToScan,
				  "No mail to scan!"));
		
	    } else if (give_message_data(mailbox,current-1,
					 NULL,NULL,NULL,NO_mime_parse)) {
		/* FIX: mime sctructure is not 
		   currently used */
		scan_calendar(mailbox);
	    } else {
		DPRINT(Debug,3,(&Debug, 
				"give_message_data [%d] fails",current-1));
	    }
	} else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmSorryNoCalendar,
			      "Sorry. Calendar function disabled."));
	    
	}
    }
	break;
	
    case 'a'    : {
	int r = alias(mailbox, aview);

	menu_trigger_redraw(page);

	if (r)
	    return r;   /* EOF */

	/* define_softkeys(MAIN); */	
    }
	break;

    case 'b'    :  {
	int current = get_current(mailbox);
	struct header_rec *hdr;
	FILE *F;

	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, 
				     ElmSet, ElmBounceMessage,
				     "Bounce message"));
	FlushBuffer();
	if (current < 1) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToBounce,
			      "No mail to bounce!"));
	    
	    /* Mime parsing is required that remail()
	       can convert content-transfer-econding
	       if required 
	    */

	} else if (give_message_data(mailbox,current-1,
				     &hdr,&F,NULL,
				     mime_parse_routine)) {

	    remail(hdr,F, aview, page, LOC->prompt_page);

	} else {
	    DPRINT(Debug,3,(&Debug, 
			    "give_message_data [%d] fails",current-1));
	}
    }
	break;

    case ctrl('E'): {
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet, ElmExtractPGPKey,
				     "Extract PGP public key"));
	FlushBuffer();
	
	if (use_PGP) {
	    int current = get_current(mailbox);
	    struct header_rec *hdr;
	    FILE *F;
	    	    
	    if (current < 1) 
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoMailInFolder,
				  "There is no mail in this folder!"));
	    
	    else if (give_message_data(mailbox,current-1,
				       &hdr,&F,NULL,NO_mime_parse)) {
		
		if (hdr->pgp & PGP_PUBLIC_KEY) {
		    /* FIXME: mime structure is not currently used */
		    pgp_extract_public_key(hdr,F,page);
		} else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmNoPgpPublicKeys,
				      "This message does not contain PGP public keys!"));
	    } else {
		DPRINT(Debug,3,(&Debug, 
				"give_message_data [%d] fails",current-1));
		
	    }
	} else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoPgpConfigured,
			      "PGP isn't configured in this version of ELM."));

	}
    }
	break;	
	
    case 'E':  {
	int r;
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmExtendedCommand,
				     "Extended command"));
	
	FlushBuffer();
	
	r = extended_command(mailbox, 
			     aview,
			     page,
			     LOC,
			     is_main_message_loop);
	
	if (menu_need_redraw(page)) {
	    DPRINT(Debug,7, (&Debug, 
			     "mailbox_command: pending redraw\n"));
	    
	    showscreen(page);
	}
	
	if (r) {        /* Print prompt again */
	    int lin,col;
	    
	    menu_PutLineX(LOC->prompt_page,0,0,
			  FRM("%S"),mcommon_give_item(&MENU, m_Prompt));
	    menu_GetXYLocation(LOC->prompt_page,&lin,&col);
	    
	    menu_CleartoEOS(LOC->prompt_page);   
	    
	    show_last_error();
	    menu_MoveCursor(LOC->prompt_page,lin,col);
	    
	    return r;
	}
	
    }	
	break;

    case ctrl('F'):
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet, 
				ElmPassphraseForget,
				"Forget passphrase"));
	FlushBuffer();
	forget_passphrase();
	break;

    case 'f'    : {
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet, 
				     ElmForward,
				     "Forward"));
	/* define_softkeys(YESNO); */
	FlushBuffer();
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToForward,
			      "No mail to forward!"));
		  
	else  {
	    forward(current-1,mailbox, aview, 
		    page,
		    LOC->prompt_page,
		    LOC->header_page);
	   
	} 
	/* define_softkeys(MAIN); */
    }
	break;

    case 'g'    :  {
	int current = get_current(mailbox);
	struct header_rec *hdr;

	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet, 
				     ElmGroupReply,
				     "Group reply"));
	FlushBuffer();
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToReply,
			      "No mail to reply to!")); 
	else if (give_message_data(mailbox,current-1,
				   &hdr,NULL,NULL,
				   mime_parse_routine)) {
	    
		/* define_softkeys(YESNO); */
		
		reply_to_everyone(current-1, mailbox, aview,
				  page,
				  LOC->prompt_page,
				  LOC->header_page);	
		/* define_softkeys(MAIN); */

	} else {
	    DPRINT(Debug,3,(&Debug, 
			    "give_message_data [%d] fails",current-1));

	}
	FlushBuffer();
    }
	break;
	
    case 'h'    :  {
	int current = get_current(mailbox);
		
	if (elm_filter)
	    menu_Write_to_screen(LOC->prompt_page,
				 CATGETS(elm_msg_cat, 
					 ElmSet, 
					 ElmMessageWithHeaders,
					 "Message with headers..."));
	else
	    menu_Write_to_screen(LOC->prompt_page,
				 CATGETS(elm_msg_cat, 
					 ElmSet, 
					 ElmDisplayMessage,
					 "Display message"));
	FlushBuffer();

	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToRead,
			      "No mail to read!"));
	else {		
	    int x = show_message_with_headers(mailbox,cmdX,aview,page,
					      LOC->header_page,pager_cmds);
	    
	    if (x != 0)
		return x;   /* "call" parent or EOF */

	    current = get_current(mailbox);
	    
	}
    }
	break;

    case 'I'    : {
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmMailingListInfo,
				     "Mailing List Info"));

	FlushBuffer();
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToViewInfo,
			      "No mail to view list info!"));

	else {
	    list_info(current-1,mailbox, aview, page,LOC->prompt_page);	
	    
	} 
	/* softkeys_on(); */
    }
	break;
	
	case ctrl('K'): {
	    menu_Write_to_screen(LOC->prompt_page,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmMailPGPPublicKey,
					 "Mail PGP public key"));
	    FlushBuffer();
	    
	    if (use_PGP) {
		pgp_mail_public_key(mailbox, aview, 
				    page, 
				    LOC->prompt_page);
	    } else {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNoPgpConfigured,
				  "PGP isn't configured in this version of ELM."));
	    }
	}
	break;

    case 'M'	:  
	if (show_mlists) {
	    lib_transient(CATGETS(elm_msg_cat, ElmSet,
				  ElmMlistOff,
				  "[Mlists Off]"));
			      
	    show_mlists = 0;
	} else {
	    lib_transient(CATGETS(elm_msg_cat, ElmSet,
				  ElmMlistOn,
				  "[Mlists On]"));
			      
	    show_mlists = 1;
	}
	menu_trigger_redraw(LOC->header_page); /* force redraw */
	break;
	
    case 'm'    : {	
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmMail,
				     "Mail"));
	FlushBuffer();
	send_msg_l(current-1, NULL, NULL, NULL,
		   MAIL_EDIT_MSG,
		   mailbox, aview,page, LOC->prompt_page,
		   NULL);
    }
	break;
			   
    case ' '    : 
    case ctrl('J'):
    case ctrl('M'): {
	int current = get_current(mailbox);
		
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmDisplayMessage,
				     "Display message"));  
	FlushBuffer();
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToRead,
			      "No mail to read!"));
	else {
	    int x = show_message_normal(mailbox,cmdX,aview,page,
					LOC->header_page,0,
					pager_cmds);
	    
	    if (x != 0)
		return x;   /* "call" parent or EOF */	    

	    current = get_current(mailbox);
	}
    }
	break;
	
    case 'n'    : {
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmNextMessage,
				     "Next Message"));
	FlushBuffer();
	/* define_softkeys(READ); */
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToRead,
			      "No mail to read!"));
	else {
	    int x = show_next_message(mailbox,cmdX,aview,page,
				      LOC->header_page,
				      pager_cmds);
	    
	    if (x != 0)
		return x;   /* "call" parent or EOF */	    

	    current = get_current(mailbox);
	} 
    }		
	break;

    case ctrl('N'):   {
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmNextUMessage,
				     "Next Unread Message"));
	FlushBuffer();
	
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToRead,
			      "No mail to read!"));
	else {
	    int mc;
	    int i = 0;	    
	    
	    struct pager_page * PP = init_pager_page(&MENU);
	    
	    if(ison_status_message(mailbox,current-1,status_basic,UNREAD)) {
		
		i = mbx_show_msg(mailbox,current-1,PP, pager_cmds);
		
	    } else 
		i = ctrl('N');

	    while (i >0) {
		i = process_showmsg_cmd(i, mailbox, cmdX, aview, 
					PP,0,pager_cmds);
		
		switch (i) {
		case EOF:

		    exit_pager_page(&PP,page);
		    return EOF;/* Read failed, control tty died? */
		    
		case 'X':      /* Quick Exit */
		case 'x':      /* Exit       */
		    
		    exit_pager_page(&PP,page);
		    return i;   /* "call" parent */

		case 0:
		    break;
		    
		default:
		    DPRINT(Debug,3,(&Debug, 
				    "Looping showmsg command %d (%c)\n",
				    i,i));		    
		}
	    }

	    exit_pager_page(&PP,page);
	    current = get_current(mailbox);


	    menu_trigger_redraw(page);
	    if (++current > (mc = get_message_count(mailbox)))
		current = mc;
	    set_current(mailbox,current);
	    
	    get_page(&MENU, LOC->header_page);
	}
    }
	break;
	
    case 'O' : {
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,	  
				     ElmOverrideCharset,
				     "Override charset"));
	FlushBuffer();

	if  (current < 1) 
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToOverride,
			      "No mail to override!"));
	else
	    OverrideCharset(mailbox, page, 
			    LOC->prompt_page,
			    LOC->header_page);

    }
	break;


    case 'P'    : {
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,	  
				     ElmPrintTextMail,
				     "Print text of mail"));
	FlushBuffer();
	
	if (current < 1) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToPrint,
			      "No mail to print!"));
	    
	} else if (give_message_data(mailbox,current-1,
				     NULL,NULL,NULL,
				     mime_parse_routine)) {

	    print_text(TRUE, mailbox, page, 
		       LOC->header_page,LOC->prompt_page);

	} else {
	    DPRINT(Debug,3,(&Debug, 
			    "give_message_data [%d] fails",current-1));
	}
    }
	break;
	
    case 'R'    : {
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmGenericReply,
				     "Generic Reply"));

	FlushBuffer();
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToReplyTo,
			      "No mail to reply to!")); 
	else {
	    generic_reply(current-1,mailbox, aview, 
			  page,LOC->prompt_page);	
	    
	} 
	/* softkeys_on(); */
    }
	break;

    case 'r'    : {
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmReplyToMessage,
				     "Reply to message"));
	FlushBuffer();
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToReplyTo,
			      "No mail to reply to!")); 
	else {

	    reply(current-1,mailbox, aview, 
		  page, 
		  LOC->prompt_page,
		  LOC->header_page);	

	} 
	/* softkeys_on(); */
    }
	break;
	
    case '>'    : /** backwards compatibility **/
	
    case 'C'	:
    case 's'    :  {
	int current = get_current(mailbox);
	
	if (ch != 'C')
	    menu_Write_to_screen(LOC->prompt_page,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmSaveToFolder,
					 "Save to folder"));
	else
	    menu_Write_to_screen(LOC->prompt_page,
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmCopyToFolder,
					 "Copy to folder"));
	FlushBuffer();
	
	if  (current < 1) {
	    if (ch != 'C')
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNoMailToSave,
				  "No mail to save!"));
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNoMailToCopy,
				  "No mail to copy!"));
	    
	} else if (give_message_data(mailbox,current-1,
				     NULL,NULL,NULL,NO_mime_parse)) {
	    
	    if (save(FALSE, (ch != 'C'), 0,mailbox, aview, 
		     page, 
		     LOC->prompt_page,
		     LOC->header_page)
		&& resolve_mode && ch != 'C') {
		int i;

		if((i=next_message(current-1, TRUE, &MENU)) != -1) {
		    
		    int current        = get_current(mailbox);
		    current = i+1;
		    set_current(mailbox,current);
		    
		    get_page(&MENU, LOC->header_page);
		    
		}
	    }

	} else {
	    DPRINT(Debug,3,(&Debug, 
			    "give_message_data [%d] fails",current-1));
	}
	ClearLine(LINES-3);		
    }
	break;

    case 'S'     :   { 
	int current = get_current(mailbox);
	
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmSaveTextToFileorFolder,
				     "Save (copy) text to file or folder"));
	FlushBuffer();
	
	if  (current < 1) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToCopy,
			      "No mail to copy!"));
	    
	} else if (give_message_data(mailbox,current-1,
				     NULL,NULL,NULL,
				     mime_parse_routine)) {

	    save(FALSE, 0, 1, mailbox, aview, 
		 page, 
		 LOC->prompt_page,
		 LOC->header_page);

	} else {
	    DPRINT(Debug,3,(&Debug, 
			    "give_message_data [%d] fails",current-1));
	}
	
	ClearLine(LINES-3);		
    }
	break;	                   

    case 'u' :   /* Unread or undelete */
	if (get_message_count(mailbox) < 1)
	    goto try_motion;
	else {
	    int current = get_current(mailbox);
	    struct header_rec * hdr = give_header(mailbox,current-1);
	    
	    /* Overload u to have it mark unread a read
	     * message which is not deleted.
	     */
	    

	    if (!hdr)
		goto try_motion;
	    if (ison(hdr->status,DELETED))
		goto try_motion;
	}

	/* FALL THROUGH */
    case 'U'    :  {
	if (get_message_count(mailbox) < 1) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmNoItemToMarkUnread,
			      "No %S to mark as unread!"),
		      mcommon_give_item(&MENU,m_items));
	    
	}
	else
	    unread_msg(mailbox, LOC->header_page);
    }
	break;
       	
    case 'v'    :   {	      
	struct header_rec *hdr;
	FILE *F;
	int current = get_current(mailbox);
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMessages,
			      "There are no messages!"));
	else {
	    menu_Write_to_screen(LOC->prompt_page,
				 CATGETS(elm_msg_cat, ElmSet, 
					 ElmViewParts,
					 "View parts of message"));

	     FlushBuffer();

	    if (give_message_data(mailbox,current-1,
				  &hdr,&F,NULL,mime_parse_routine)) {

		if ((hdr->status & MIME_MESSAGE)) {
		 
		    const struct remote_server * remote_server =
			give_message_remote_server(mailbox,current-1);
		       
		    mime_warnings(hdr);
		    
		    attach_menu(F,& (hdr->mime_rec),
				NULL, 
				hdr->header_charset, NULL,
				hdr,
				aview,
				mailbox,
				& (hdr->header_error),
				remote_server);

		    menu_trigger_redraw(page);
		} else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmNoMIMEMessage,
				      "This is not a MIME message!"));
	    } else {
		DPRINT(Debug,3,(&Debug, 
				"give_message_data [%d] fails",current-1));
	    }
	}
    }	   	
	break;

    case 'V': {
	int current = get_current(mailbox);

	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmViewDigest,
				     "View digest as mailbox"));

	FlushBuffer();
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToView,
			      "No mail to view!")); 
	else if (give_message_data(mailbox,current-1,
				   NULL,NULL,NULL,mime_parse_routine)) {
	    
	    ViewDigest(mailbox,aview,page);

	    current = get_current(mailbox);
	    get_page(&MENU, LOC->header_page);

	} else {
	    DPRINT(Debug,3,(&Debug, 
			    "give_message_data [%d] fails",current-1));

	}
    }
	break;

    case EOF : 
	return EOF;/* Read failed, control tty died? */
     
    case '@'    : 
	debug_screen(mailbox,LOC);  
	menu_trigger_redraw(page);	
	break;

    case '#'    : {
	int current = get_current(mailbox);
	menu_Write_to_screen(LOC->prompt_page,
			     CATGETS(elm_msg_cat, ElmSet, 
				     ElmCheckMessage,
				     "Check message"));
	FlushBuffer();
	
	if (current < 1)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmNoMailToCheck,
			      "No mail to check."));
	else {
	    struct header_rec * hdr = give_header(mailbox,
						  current-1);
	    if (hdr) {
		struct folder_view index;
		
		give_index_number(mailbox,current-1,
				  &index);
		
		debug_message(hdr, current, &index); 
		menu_trigger_redraw(page);	
	    }
	} 
    }
	break;

    default:
    try_motion: 
	/* Maybe motion command */

	ch = motion(ch,&MENU,LOC, page, menu_cmds);
    
	/* motion() may already set, menu_trigger_redraw(page),
           we do not examine it */

	return ch;
		
    }

    return 0;   /* Command OK */
}


void new_mail_check (mailbox, page, LOC,
		     exit_hint_p)
     struct MailboxView *mailbox;
     struct menu_context  *page;
     struct screen_parts *LOC;
     const char ** exit_hint_p;
{
    int idx, mbxcount;
    struct menu_common MENU;
    int   LINES, COLUMNS;	
    int update_selected = 0;

    static const int cancel_delay_msec = 500; /* 500 milliseconds: 1/2 seconds */

    struct cancel_data *cd = NULL;
	

    
    menu_get_sizes(page,&LINES, &COLUMNS);

    set_mcommon_from_mbxview(&MENU,mailbox);
    
    mbxcount = get_storage_count(mailbox);

    for (idx = 0 ; idx < mbxcount; idx++) {
	long bytes = 0;
	int newmails = 0;
	enum new_mail_stat r;
	int err = 0;
	int flush_r;
		
	struct current_storage * storage = get_storage(mailbox, idx);
	    
	if (!storage ||
	    !storage->current_folder)
	    continue;

	if (!cd)
	    cd = new_schedule_cancel(cancel_delay_msec,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmCheckingNewMail,
					     "Checking new mail..."));
	
	flush_r = flush_folder(storage->current_folder,&err,cd);
	if (flush_r) {	
	    r = new_mail_on_folder(storage->current_folder,
				   &bytes,&newmails,&err,
				   cd);
	} else {
	    DPRINT(Debug,12, (&Debug,
			      "new_mail_check: flush_folder failed\n"));
	    r = failed_mbx_ign_new_mail;
	}

	if (is_canceled(cd)) {
	    const char * ftype = folder_type(storage->current_folder);
	    
	    DPRINT(Debug,12, (&Debug,
			      "new_mail_check: Check canceled\n"));

	    /* Canceled message is immediately removed, show
	       use error message instead 
	    */

	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmCheckNewMailCanceled,
			      "Checking new mail canceled (%s)"),
		      ftype);
	    
	    break;
	}
	
	switch (r) {
	case new_mail_check_failed: {
	    const char * ftype = folder_type(storage->current_folder);

	    DPRINT(Debug,12, (&Debug,
			      "new_mail_check: folder failed\n"));
	    	    
	    if (exit_hint_p && *exit_hint_p == ftype)
		break;
	    
	    if (err)
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmFailedNoNewMailS,
				  "%s failed! No new mail: %s"),
			  ftype,strerror(err));
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmFailedNoNewMail,
				  "%s failed! No new mail..."),
			  ftype);
	    if (exit_hint_p)
		*exit_hint_p = ftype;
	}
	    break;
	case failed_mbx_ign_new_mail: {
	    const char * ftype = folder_type(storage->current_folder);
	    DPRINT(Debug,12, (&Debug,
			      "new_mail_check: folder failed, but ignoring new mail check anyway\n"));

	    if (exit_hint_p && *exit_hint_p == ftype)
		break;

	    
	    if (err)
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmFailedIqnS,
				      "%s failed: %s"),
			      ftype,strerror(err));
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmFailedIgn,
				      "%s failed..."),
			      ftype);
	    if (exit_hint_p)
		*exit_hint_p = ftype;
	}
	    break;
	case have_new_mail: {
	    int last_in_folder = storage->message_count;
	    int selected       = get_selected(mailbox);
	    int ok;
	    
	    DPRINT(Debug,2,(&Debug, 
			    "new_mail_check: Just received %ld bytes more mail, storage idx %d\n", 
			    bytes, idx));

	    if (newmails > 0) { /* IMAP mailbox, count is available */
		DPRINT(Debug,2,(&Debug, 
				"new_mail_check: Got %d new mails\n",
				newmails));

	    }

	    free_cancel(&cd);
	    
	    if (newmails > 1) {  /* IMAP mailbox, count is available */
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmXNewMailsHangOn,
				  "%d new mails have arrived! Hang on..."),
			  newmails);
	    } else {	    
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNewMailHangOn,
				  "New mail has arrived! Hang on..."));
	    }
		
	    ok = read_new_mails(storage, page);

	    if (ok)
		clear_error();
		  
	    if (selected) {
		DPRINT(Debug,12,(&Debug, 
				 "new_mail_check: Was %d selected",
				 selected
				 ));
		
		/* update count of selected messages */
		selected += storage->message_count - last_in_folder;

		DPRINT(Debug,12,(&Debug, " => %d selected now.\n",
				 selected));
		
		set_selected(mailbox,selected);
	    }		  

	    if (!update_selected) {
		menu_trigger_redraw(LOC->header_page);	
		menu_trigger_redraw(LOC->title_page);	
	    }
		
	    ClearLine(LINES-2);	     /* remove reading message... */
	    
	    if (ok) {
		if ((storage->message_count - last_in_folder) == 1)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNewMessageRecv,
				      "1 new message received."));
		else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmNewMessageRecvPlural,
				      "%d new messages received."), 
			      storage->message_count - last_in_folder);
	    } else
		show_last_error();
	}
	    break;
	case no_new_mail:
	    if (bytes < 0) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmALERTMailfileShrinken,
				  "ALERT: Mailfile shrinken %ld bytes!"),
			  -bytes);
		DPRINT(Debug,1,(&Debug, 
				"ALERT: Mailfile shrinken %ld bytes!",
				-bytes));
		sleep_message();
	    }
	    break;
	case new_mail_reconnect: {
	    const char * ftype = folder_type(storage->current_folder);
	    int last_in_folder = storage->message_count;
	    int selected       = get_selected(mailbox);
	    int ok;
	    	    
	    DPRINT(Debug,2,(&Debug, 
			    "new_mail_check: folder reconnected, storage idx %d\n",
			    idx));

	    free_cancel(&cd);
	    
	    lib_transient(CATGETS(elm_msg_cat, ElmSet, ElmReconnectedHangOn,
				  "%s reconnected! Hang on..."),
			  ftype);

	    ok = reconnect_mails(storage,page);
	    
	    
	    if (ok) 
		clear_error();

	    if (selected) {
		DPRINT(Debug,12,(&Debug, 
				 "new_mail_check: was %d selected, need recalculate\n",
				 selected));
		update_selected = 1;
		
	    } else {		
		menu_trigger_redraw(LOC->header_page);	
		menu_trigger_redraw(LOC->title_page);	
	    }
		
	    ClearLine(LINES-2);	     /* remove reading message... */
	    
	    if (ok) {
		if ((storage->message_count - last_in_folder) == 1)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNewMessageRecv,
				      "1 new message received."));
		else if (storage->message_count > last_in_folder)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmNewMessageRecvPlural,
				      "%d new messages received."), 
			      storage->message_count - last_in_folder);		
	    }
	}
	    break;
	}
    }

    if (cd)
	free_cancel(&cd);
    
    if (update_selected) {
	update_selected_counter(mailbox);
		
	menu_trigger_redraw(LOC->header_page);	
	menu_trigger_redraw(LOC->title_page);	
    }
}

void mailbox_screen_common(page,LOC, LIST)
     struct menu_context  *page;
     struct screen_parts *LOC;
     struct menu_param  *LIST;
{    
    int   LINES, COLUMNS;	
    int  headers_per_page;

    menu_get_sizes(page,&LINES, &COLUMNS);

    /* 3) prompt part */

    if (LOC->prompt_page)
	menu_subpage_relocate(LOC->prompt_page,page,LINES-4,4);
    else 
	LOC->prompt_page = new_menu_subpage(page,LINES-4,4,
					    subpage_simple_noredraw,LIST);


    headers_per_page = LOC->menu_page ? LINES-13 : LINES-9;
    if (headers_per_page < 1) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmWindowSizeTooSmall,
			  "The window is too small to run Elm.  It must have at least %d rows"),
		  LINES - headers_per_page);
	
	headers_per_page = 1;
    }

    /* 4) headers of screen */
    
    if (! LOC->header_page)
	LOC->header_page = new_menu_header(page,4,
					   headers_per_page,
					   hdr_show_header,
					   hdr_show_current,
					   null_header_param_changed,
					   hdr_show_status,
					   hdr_line_separator_index,
					   hdr_show_separator,
					   hdr_init_separators,
					   hdr_setup_init,
					   hdr_setup_line,
					   LIST);
    else 
	menu_header_relocate(LOC->header_page,page,
			     4,headers_per_page);

}

void free_mailbox_screen(LOC)
     struct screen_parts *LOC;
{
    erase_menu_context (&(LOC->title_page));

    if (LOC->menu_page)
	erase_menu_context (&(LOC->menu_page));

    erase_menu_context (&(LOC->prompt_page));

    if (LOC->header_page)
	erase_menu_context (&(LOC->header_page));
}


#if ANSI_C
static subpage_simple_redraw sb_update_sendtitle;
#endif
static int sb_update_sendtitle(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    menu_ClearScreen(ptr);

    menu_StartXX(ptr,pg_BOLD);
    menu_print_format_center(ptr,1,
			     CATGETS(elm_msg_cat, ElmSet, ElmSendOnlyMode,
				     "Send only mode [ELM %s]"), 
			     version_buff);
    menu_EndXX(ptr,pg_BOLD);

    return 1;   /* Redraw completely done */
}

static void set_sendmode_screen P_((struct menu_context  *page, 
				   struct screen_parts *LOC,
				   struct menu_param  *LIST));
static void set_sendmode_screen(page,LOC, LIST)
     struct menu_context  *page;
     struct screen_parts *LOC;
     struct menu_param  *LIST;
{    
    int   LINES, COLUMNS;	

    menu_get_sizes(page,&LINES, &COLUMNS);

    /* 1)  Title part of screen */
    
    if (! LOC->title_page)
	LOC->title_page = new_menu_subpage(page,0,4,sb_update_sendtitle,LIST);
    else
	menu_subpage_relocate(LOC->title_page,page,0,4);

    /* 2) menu part */
    LOC->menu_page = NULL;

    /* 3) prompt part */

    if (LOC->prompt_page)
	menu_subpage_relocate(LOC->prompt_page,page,LINES-4,4);
    else 
	LOC->prompt_page = new_menu_subpage(page,LINES-4,4,
					    subpage_simple_noredraw,LIST);

    /* 4) headers of screen */
    LOC->header_page = NULL;
}


int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    int exit_status = 0;
    char **req_mfile_vector = NULL;    /* Malloced */
    char **to_whom = NULL;

    struct menu_context  *page = NULL;

    struct menu_param  PARAM[elm_mp_COUNT+1] = { 
	{ mp_menu_common, { 0 } },
	{ mp_integer,     { 0 } },
	{ mp_integer,     { 0 } },
	{ mp_integer,     { 0 } },
	{ mp_integer,     { 0 } },
	{ mp_END,         { 0 } }
    };

    struct AliasView     * cur_aliaslist = NULL;
    char                 * url_mode = NULL;
    struct text_block   * body          = NULL;
    const char          * included_file = NULL;
    const char          * subject       = NULL;


    set_panic_prepare(elm_panic_prepare);
    set_panic_exit(elm_panic_exit);

#if DEBUG
    init_debugfile("ELM");
#endif

    /* USER may want enable debug output for locale_init() */
    set_early_options(argc, argv);

    locale_init();
	
    req_mfile_vector = parse_arguments(argc, argv, &to_whom,
				       &url_mode,&included_file,
				       &subject);
    initialize(&cur_aliaslist, &page);

    if (included_file) {
	body = block_from_filename(included_file,
				   system_charset /* Is correct? */,
				   0);
	if (!body) {
	    DPRINT(Debug,4,(&Debug,
			    "parse_arguments: block_from_filename() failed to include %s\n",
			    included_file));
	    goto fail_1;
	}

    } else if (OPMODE_IS_BATCHMODE(opmode)) {
	if (url_mode) {
	    DPRINT(Debug,4,(&Debug, 
			    "main: url mode, will not read standard input yet.\n"));
	} else if (OPMODE_IS_SENDMODE(opmode)) {
	    body = block_from_stdin();
	    
	    if (!body) {
		DPRINT(Debug,4,(&Debug,
				"main block_from_stdin() failed to include standard input.\n"));
		goto fail_1;
	    }
	} else {
	     DPRINT(Debug,4,(&Debug, 
			     "main: not in send mode will not read standard input.\n"));
	}	
    }


    if (url_mode) {
	struct string * s   = new_string2(system_charset,s2us(url_mode));
	struct url    * url = url_from_raw(s,NULL,NULL);
	enum url_type T;
	
	DPRINT(Debug,2,(&Debug, 
			"Elm: Url-mode\n"));
	
	/* Be sure that errors are visible */
	error_wait();
	
	if (!url) {	    
	    free_string(&s);
	    goto fail_1;
	}
	
	T = get_url_type(url);
	
	if (mail_files_used() !=  no_mail_files_given &&
	    url_mailing != T) {
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmArgsBadMailFileOptO,
			      "Option --mail-file is incompatible with -O option."));

	    goto fail_1;
	}
	
	switch (T) {
	    int tmp;
	    int r;	    
	    
	case url_unknown:
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmUnknownUrl,
			      "Unsupported URL type. Can not open %S"),
		      s);
	    free_url(&url);
	    free_string(&s);
	    
	    goto fail_1;	
	    
	case url_mailing: 
	    DPRINT(Debug,2,(&Debug, 
			    "Mail-only: mailing to URL\n"));
	    
	    tmp = opmode & OPMODE_MASK;

	    if (tmp && ! OPMODE_IS_SENDMODE(opmode)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUrlBadMailing,
				  "Mailing Url given with -O option is incompatible with other options."));

		free_url(&url);
		free_string(&s);

		goto fail_1;	
	    }

	    opmode &= ~OPMODE_MASK;
	    opmode |=  OPMODE_SEND;
	    

	    if(! OPMODE_IS_BATCHMODE(opmode)) {
		struct screen_parts  LOC  = { NULL, NULL, NULL, NULL };
		
		set_sendmode_screen(page,&LOC,NULL);  
		sb_update_sendtitle(LOC.title_page,NULL);

		r = send_msg_url(url, subject, 
				 MAIL_EDIT_MSG | 
				 (confirm_url ? MAIL_ASK_SEND : 0),
				 cur_aliaslist, page, 
				 LOC.prompt_page,
				 &exit_status,body);

		free_mailbox_screen(&LOC);

		if (!r) {
		    free_url(&url);
		    free_string(&s);

		    goto fail_1;
		}

	    } else {

		if (confirm_url) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmConfirmUrlBatch,
				      "Can't confirm url on batch mode."));
		    free_url(&url);
		    free_string(&s);
		    
		    goto fail_1;

		}

		r = send_msg_url(url, subject, MAIL_EDIT_MSG,
				 cur_aliaslist, page, 
				 NULL, &exit_status,body);
		
		if (!r) {
		    free_url(&url);
		    free_string(&s);

		    goto fail_1;
		}

	    }
	    break;

	case url_mailbox_list:
	    DPRINT(Debug,2,(&Debug, 
			    "Selecting mailbox from URL\n"));

	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmMailboxListUnsUrl,
			      "Mailbox list is unsupported. Can not open %S"),
		      s);
	    free_url(&url);
	    free_string(&s);

	    goto fail_1;	


	case url_mailbox:
	    DPRINT(Debug,2,(&Debug, 
			    "Viewing mailbox from URL\n"));
	    
	    tmp = opmode & OPMODE_MASK;

	    if (tmp && ! OPMODE_IS_READMODE(opmode)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmUrlBadMailbox,
				  "Mailbox Url given with -O option is incompatible with other options."));

		free_url(&url);
		free_string(&s);

		goto fail_1;	
	    }

	    opmode &= ~OPMODE_MASK;
	    opmode |=  OPMODE_READ;

	    if (OPMODE_IS_BATCHMODE(opmode)) {

		DPRINT(Debug,2,(&Debug, 
				"Batch-only\n"));
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmBatchOnlyMailbox,
				  "Can't browse mailbox on batch-only mode!"));
		goto fail_1;
	    }

	    r = url_messages_menu(page,url,cur_aliaslist);

	    if (EOF == r) {
		free_url(&url);
		free_string(&s);

		goto fail_EOF;
	    }
	    
	    if (!r) {
		free_url(&url);
		free_string(&s);
		
		goto fail_1;
	    }
	    break;		 
	}

	
	free_url(&url);
	free_string(&s);
	
    } else if (OPMODE_IS_SENDMODE(opmode)) {

	DPRINT(Debug,2,(&Debug, 
			"Mail-only: mailing to\n"));
	
	if (! OPMODE_IS_BATCHMODE(opmode)) {
	    struct screen_parts  LOC  = { NULL, NULL, NULL, NULL };
	    
	    int r;

	    set_sendmode_screen(page,&LOC,PARAM);  
	    sb_update_sendtitle(LOC.title_page,PARAM);

	    r = send_msg_argv(to_whom, subject, MAIL_EDIT_MSG,
			      cur_aliaslist, page, 
			      LOC.prompt_page, &exit_status,body);

	    free_mailbox_screen(&LOC);

	    if (EOF == r)
		goto fail_EOF;
	    
	    if (!r)
		goto fail_1;

	} else {
	    int r = send_msg_argv(to_whom, subject, MAIL_EDIT_MSG,
				  cur_aliaslist, page, 
				  NULL, &exit_status,body);

	    if (EOF == r)
		goto fail_EOF;
	    
	    if (!r)
		goto fail_1;
	}

    } else if (OPMODE_IS_CHECKMODE(opmode)) {

	DPRINT(Debug,2,(&Debug, 
			"Elm: Check-only\n"));

	do_check_only(to_whom, cur_aliaslist);

    } else if (OPMODE_IS_READMODE(opmode)) {

	DPRINT(Debug,2,(&Debug, 
			"Elm: Read-mode\n"));


	if (OPMODE_IS_BATCHMODE(opmode)) {   

	    DPRINT(Debug,2,(&Debug, 
			    "Elm: Batch-only\n"));
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmBatchOnlyMailbox,
			      "Can't browse mailbox on batch-only mode!"));
	    goto fail_1;


	} else {
	    int r;
	    
	    DPRINT(Debug,2,(&Debug, 
			    "Elm: Normal mode\n"));
	    
	    r = main_messages_menu(page,req_mfile_vector,
				   cur_aliaslist);
	    
	    if (req_mfile_vector) {
		int i;
		
		for (i = 0; req_mfile_vector[i]; i++) {
		    free(req_mfile_vector[i]);
		    req_mfile_vector[i] = NULL;
		}
		
		free(req_mfile_vector);
		req_mfile_vector = NULL;
	    }

	    if (EOF == r)
		goto fail_EOF;

	    if (!r)
		goto fail_1;
	    
	}

    } else {

	DPRINT(Debug,2,(&Debug, 
			"Elm: Unknown operation mode!\n"));

    }
    
    if (cur_aliaslist) {
	DPRINT(Debug,9, (&Debug, 
			 "main: freeing cur_aliaslist\n"));

	free_aliasview(&cur_aliaslist);
    }

    erase_menu_context(&page);

    if (body)
	free_block(&body);

    leave(0,exit_status);
    
    return exit_status;

 fail_1:
 fail_EOF: 
    if (0 == exit_status)
	exit_status = 1;

    if (cur_aliaslist) {
	DPRINT(Debug,9, (&Debug, 
			 "main: freeing cur_aliaslist\n"));
	
	free_aliasview(&cur_aliaslist);
    }

    if (body)
	free_block(&body);
    
    erase_menu_context(&page);
    
#ifdef REMOTE_MBX
    {
	 struct cancel_data *cd = NULL;
	 
	 setup_mbx_cancel_message(&cd,mbx_closing_con);

	 DPRINT(Debug,9, (&Debug, "main: Closing cached connections\n"));
	 
	 
	 if (close_cached_connections(cd)) {
	     DPRINT(Debug,9, (&Debug, "main: Cached connections closed\n"));
	 }

	 if (is_canceled(cd)) {
	     DPRINT(Debug,9, (&Debug, "main: Cancel on close_cached_connections\n"));
	 }

	 free_cancel(&cd);	
    }
#endif

    Raw_OFF(1); /* Raw(OFF) and reprint error message */
    
    free_opts_menu();
    free_cached_commands();
#ifdef USE_DLOPEN
    free_shared_caches();
#endif
    free_addr_util();
    free_mail_file_list();
    close_messages_log();
    free_rc_options();
        
    return exit_status;

}

static void debug_screen(mailbox,LOC)
     struct MailboxView *mailbox;
     struct screen_parts *LOC;
{
	/**** spit out all the current variable settings and the table
	      entries for the current 'n' items displayed. ****/

	int i, j;
	int LINES, COLUMNS;
	int li,co;
	struct menu_context *page = new_menu_context();

redraw:
	if (menu_resized(page) ||
	    menu_need_redraw(page))
	/* Nothing */;
        
	menu_get_sizes(page, &LINES, &COLUMNS);   

	menu_ClearScreen(page);

	{ 

	    struct folder_info * f = get_main_folder(mailbox);
	    if (f) {
		menu_PutLineX(page,
			      4,0,CATGETS(elm_msg_cat, ElmSet, 
					  ElmDbxCurrMailFile,
					  "\nCurrent mail file is %S.\n\r\n"), 
			      f->cur_folder_disp);
	    }
	}

	i = menu_header_get(LOC->header_page,header_top_line);	/* starting header */
	menu_get_sizes(LOC->header_page, &li, &co);   

	if ((j = i + li-1) >= get_message_count(mailbox)) 
	    j = get_message_count(mailbox)-1;

	/* FIXME -- selected */

	menu_Write_to_screen(page,
			     CATGETS(elm_msg_cat, ElmSet, ElmDebugHeader,
				     "Num      From                    Subject                Lines   Offset  Content\n\r\n\r"));
	
	for (;i <= j; i++) {	  
	    struct header_rec * hdr = give_header(mailbox,i);
	    struct string * x    = NULL;
	    struct string * temp = NULL;
	    

	    if (! hdr)
		continue;
	    
	    x = hdr->subject;
	    if (!x) {
		temp = new_string2(ASCII_SET,s2us("<-- NO SUBJECT -->"));
		x = temp;
	    }

	    if (hdr->from && 1 == addr_list_item_count(hdr->from)) {
		int group = -1;

		const struct address * address = 
		    addr_list_get_item(hdr->from,0,&group);

		const char          * addr     = address_get_ascii_addr(address);
		const struct string * fullname = address_get_phrase(address);
		
		if (fullname && string_len(fullname))
		    menu_Write_to_screen(page,
					 FRM("%3d %-16.16S %-35.35S %4d %8d %8d\n\r"),
					 i+1,
					 fullname, 
					 x,
					 hdr->lines,
					 hdr->offset,
					 hdr->content_length);
		else if (addr)
		    menu_Write_to_screen(page,
					 FRM("%3d %-16.16s %-35.35S %4d %8d %8d\n\r"),
					 i+1,
					 addr,
					 x,
					 hdr->lines,
					 hdr->offset,
					 hdr->content_length);
		else 
		    goto no_addr;
	    } else {
	    no_addr:
		
		menu_Write_to_screen(page,
				     FRM("%3d %-16.16s %-35.35S %4d %8d %8d\n\r"),
				     i+1,
				     hdr->env_from,
				     x,
				     hdr->lines,
				     hdr->offset,
				     hdr->content_length);
	    }


	    if (temp)
		free_string(&temp);

	}
	
	menu_PutLineX(page,
		      LINES-1,0,CATGETS(elm_msg_cat, ElmSet, 
					ElmDbxPressAnyKey,"Press any key to return."));
	if (REDRAW_MARK == menu_ReadCh(page, REDRAW_MARK))
	    goto redraw;

	erase_menu_context(&page);
	return;
	
}

static void debug_message(current_header,current,index)
     struct header_rec *current_header;
     int current;
     struct folder_view *index;
{
	/**** Spit out the current message record.  Include EVERYTHING
	      in the record structure. **/
	
	time_t header_time;
	int LINES, COLUMNS;
	struct menu_context *page = new_menu_context();
	struct tm *A;
	
redraw:
	if (menu_resized(page) ||
	    menu_need_redraw(page))
	    /* Nothing */;
        
	menu_get_sizes(page, &LINES, &COLUMNS);   

	menu_ClearScreen(page);

	menu_Write_to_screen(page,
			     FRM("\t\t\t----- Message %d -----\n\r\n\r\n\r\n\r"), 
			     current);
	
	menu_Write_to_screen(page,
			     FRM("Lines: %-17dStatus: A  C  D  E  N  P  T  U  V  O  R  M  P  U  N\n\r"), 
			     current_header->lines);
	menu_Write_to_screen(page,
			     FRM("Content-Length: %-16dc  o  e  x  e  r  a  r  i  l  e  i  r  s  H\n\r"), 
			     current_header->content_length);
	menu_Write_to_screen(page,
			     FRM("Binary: %-24dt  n  l  p  w  i  g  g  s  d  p  m  e  u  d\n\r"), 
			     current_header->binary);
	menu_Write_to_screen(page,
			     FRM("'From ' on body: %-15dn  f  d  d     v  d  n  i     l  e  M  p  r\n\r"),
			     current_header->have_from);
	
	menu_Write_to_screen(page,
			     FRM("\n\rOffset: %-22ld%3d%3d%3d%3d%3d%3d%3d%3d%3d%3d%3d%3d%3d%3d%3d\n"),
			     current_header->offset,
			     (current_header->status & ACTION) != 0,
			     (current_header->status & CONFIDENTIAL) != 0,
			     (current_header->status & DELETED) != 0,
			     (current_header->status & EXPIRED) != 0,
			     (current_header->status & NEW) != 0,
			     (current_header->status & PRIVATE_MAIL) != 0,
			     (current_header->status & TAGGED) != 0,
			     (current_header->status & URGENT) != 0,
			     (current_header->status & VISIBLE) != 0,
			     (current_header->status & UNREAD) != 0,
			     (current_header->status & REPLIED_TO) != 0,
			     (current_header->status & MIME_MESSAGE) != 0,
			     (current_header->status & PRE_MIME_CONTENT) != 0,
			     (current_header->status & MIME_UNSUPPORTED) != 0,
			     (current_header->status & NOHDRENCODING) != 0);

	
	A = localtime(&current_header->received_time);
	/* asctime() includes \n */
	if (A) {
	    char * t = asctime(A);

	    if (t)
		menu_Write_to_screen(page,
				     FRM("\n\rReceived on: %s\r"),
				     t);
	}
	    
	header_time = current_header->time_sent + current_header->tz_offset;

	A = gmtime(&header_time);
	/* asctime() includes \n */
	if (A) {
	    char * t = asctime(A);
	    
	    menu_Write_to_screen(page,
				 FRM("Message sent on: %s\r"),
				 t);
	}
	
	if (current_header->time_menu_year > 0)
	    menu_Write_to_screen(page,
				 FRM("Message sent on: %d (year)\n\r"),
				 current_header->time_menu_year);

	menu_Write_to_screen(page,FRM("From timezone: %s (%d)\n\r"),
			     current_header->time_zone,
			     current_header->tz_offset);

	
	menu_Write_to_screen(page,
			     FRM("(Env) From: %s\n\r"), 
			     current_header->env_from);

	

	if (current_header->from) {
	    
	    int idx, addr_item_count = addr_list_item_count(current_header->from);
	    
	    for (idx = 0; idx < addr_item_count; idx++) {
		int group = -1;   /* groups are not allowed on From: header field */
		
		
		const struct address * address = 
		    addr_list_get_item(current_header->from,idx,&group);
		
		const char          * addr     = address_get_ascii_addr(address);
		const struct string * fullname = address_get_phrase(address);
		const struct string * comment  = address_get_comment(address);
		
		
		if (0 == idx)
		    menu_Write_to_screen(page,
					 FRM(     "From: "));
		else
		    menu_Write_to_screen(page,
					 FRM(",\r\n      "));

		if (fullname)
		    menu_Write_to_screen(page,
					 FRM("%-30S "), fullname);
		if (addr)
		    menu_Write_to_screen(page,
					 FRM("<%s>"),addr);
		
		if (comment && string_len(comment)) 
		    menu_Write_to_screen(page,
					 FRM(" (%S)"),comment);
	    }
	    menu_Write_to_screen(page,
				 FRM("\r\n"));
	}
	
	
	if (current_header->subject) {       
	    menu_Write_to_screen(page,
				 FRM("Subject: %S\n\rInternal Index Reference Number = %d/%d (hdr index %d)\n\r"),
				 current_header->subject,
				 index->mailbox_number,
				 index->index,
				 current_header->index_number_X);
	} else {
	    menu_Write_to_screen(page,
				 FRM("Subject: <none>\n\rInternal Index Reference Number = %d/%d (hdr index %d)\n\r"),
				 index->mailbox_number,
				 index->index,
				 current_header->index_number_X);
	}
	
	menu_Write_to_screen(page,FRM("Thread number = %d\r\n"),
			     index->thread_number);

	if (current_header->to) {
	    
	    int idx, addr_item_count = addr_list_item_count(current_header->to);
	    
	    for (idx = 0; idx < addr_item_count; idx++) {
		int group = -1;   
				
		const struct address * address = 
		    addr_list_get_item(current_header->to,idx,&group);
		
		const char          * addr     = address_get_ascii_addr(address);
		const struct string * fullname = address_get_phrase(address);
		const struct string * comment  = address_get_comment(address);

		if (0 == idx)
		    menu_Write_to_screen(page,
					 FRM(     "To: "));
		else
		    menu_Write_to_screen(page,
					 FRM(",\r\n    "));

		if (fullname)
		    menu_Write_to_screen(page,
					 FRM("%-30S "), fullname);
		if (addr)
		    menu_Write_to_screen(page,
					 FRM("<%s>"),addr);
		
		if (comment && string_len(comment)) 
		    menu_Write_to_screen(page,
					 FRM(" (%S)"),comment);
	    }
	    menu_Write_to_screen(page,
				 FRM("\r\n"));
	}

	if (current_header->cc) {

	    int idx, addr_item_count = addr_list_item_count(current_header->cc);
	    
	    for (idx = 0; idx < addr_item_count; idx++) {
		int group = -1;   
				
		const struct address * address = 
		    addr_list_get_item(current_header->cc,idx,&group);
		
		const char          * addr     = address_get_ascii_addr(address);
		const struct string * fullname = address_get_phrase(address);
		const struct string * comment  = address_get_comment(address);

		if (0 == idx)
		    menu_Write_to_screen(page,
					 FRM(     "CC: "));
		else
		    menu_Write_to_screen(page,
					 FRM(",\r\n    "));

		if (0 == idx)
		    menu_Write_to_screen(page,
					 FRM(     "To: "));
		else
		    menu_Write_to_screen(page,
					 FRM(",\r\n    "));

		if (fullname)
		    menu_Write_to_screen(page,
					 FRM("%-30S "), fullname);
		if (addr)
		    menu_Write_to_screen(page,
					 FRM("<%s>"),addr);
		
		if (comment && string_len(comment)) 
		    menu_Write_to_screen(page,
					 FRM(" (%S)"),comment);
	    }

	    menu_Write_to_screen(page,
				 FRM("\r\n"));
	}


	if (current_header->message_id) {
	    char * XX = get_message_id_ascii_brackets(current_header->message_id);

	    if (!XX)
		goto msgid_none;

	    menu_Write_to_screen(page,
				 FRM("Message-ID: %s\n\r"), 
			     XX);
	    
	    free(XX);
	    
	} else {
	msgid_none:

	    menu_Write_to_screen(page,
				 FRM("Message-ID: <none>\n\r"));


	}

	menu_Write_to_screen(page,
			     FRM("Status: %s\n\r"),  
			     current_header->mailx_status);

	menu_Write_to_screen(page,
			     FRM("Content-Transfer-Encoding: %s\n\r"),
			     ENCODING(current_header->mime_rec.encoding));

	menu_PutLineX(page,
		      LINES-1,0,CATGETS(elm_msg_cat, ElmSet, 
					ElmDbxPressAnyKey,
					"Please Press any key to return."));
	if (REDRAW_MARK == menu_ReadCh(page, REDRAW_MARK))
	  goto redraw;

	erase_menu_context(&page);
	return;

}

static void do_check_only(to_whom,aview)
     char *to_whom[];
     struct AliasView *aview;
{
    struct expanded_address A;
    
    zero_expanded_address(&A);
    
    dump_expanded_address(3,"Check-only: checking",A);

    if (to_whom) {
	struct mailer_info *mailer_info = get_mailer_info();
	if (argv_to_expanded(&A,to_whom,mailer_info,aview)) {
	    struct string * addr_string = hdr_to_expval(A);

	    clear_error();
	    if (addr_string) {
		
		elm_fprintf(stdout,
			    CATGETS(elm_msg_cat, ElmSet, ElmExpandsTo,
				    "Expands to: %S\n"),
			    addr_string);
		free_string(&addr_string);
	    }
	}
	free_expanded_address(&A);
	free_mailer_info(&mailer_info);
    }
}

void check_range(mc,LOC)
     struct menu_common *mc;
     struct screen_parts *LOC;
{
    int i;
    int current  = mcommon_get_current(mc);
    int selected = mcommon_get_selected(mc);

    i = compute_visible(current, mc);

    if ((current < 1) || (selected && i < 1)) {
	if (mcommon_get_count(mc) > 0) {
	    /* We are out of range! Get to first message! */
	    if (selected)
		current = compute_visible(1,mc);
	    else
		current = 1;
	}
	else
	    current = 0;
    }
    else if ((current > mcommon_get_count(mc))
	      || (selected && i > selected)) {

	if (mcommon_get_count(mc) > 0) {
	    /* We are out of range! Get to last (visible) message! */
	    if (selected)
		current = visible_to_index(selected,mc)+1;
	    else
		current = mcommon_get_count(mc);

	} else
	    current = 0;
    }
    
    mcommon_set_current(mc,current);
    copy_current(mc,LOC->header_page);
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
