static char rcsid[] = "@(#)$Id: fileio.c,v 2.27 2022/11/19 06:50:54 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.27 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/fileio.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 *  Incorparated Elm 2.5 code from src/fileio.c. 
 *  That code was following copyright:
 *
 *  The Elm Mail System
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** File I/O routines, including deletion from the folder! 

**/

#include "def_elm.h"
#include "s_elm.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mbox");
DEBUG_VAR(PgpDebug,__FILE__,"pgp");
DEBUG_VAR(MimeDebug,__FILE__,"mime");

/* SHRT_MAX */
#include <limits.h>

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2cus P_((const char *str));
static const unsigned char *cs2cus(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

static const unsigned char * csUs P_((const char *str));
static const unsigned char * csUs(str)
     const char *str;
{
    return (const unsigned char *)str;
}

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif         

/* Prototype */
static int copy_pgp P_((struct out_state *,int, struct header_rec *, 
			FILE *));

static int copy_pgp(dest_file,cm_options,current_header, infile)
     struct out_state *dest_file;
     int cm_options;
     struct header_rec *current_header;
     FILE * infile;
{
    int body_bytes = 0;
    FILE *fpin = NULL, *fpout = NULL;
    char buffer[VERY_LONG_STRING];
    int buf_len=0,err,code = -1;
    struct run_state RS;
    int stat;
    int pgp_seen = !pgp_noarmor && current_header->pgp != PGP_PUBLIC_KEY;
    enum pgp_version v = pgp2;
    enum pgp_version version;
    int armor_header = 0;
    int ret = 1;

    struct pager_range * decode_data = NULL;
    struct pager_range * title_range = NULL;
    int raw = 0;
    
    DPRINT(PgpDebug,5,(&PgpDebug,
		       "copy_pgp called: Need read %d bytes\n",
		       current_header->content_length));

    /* Allow case where content_length == -1 */
    if (current_header->content_length < 1) {
	DPRINT(Debug,1,(&Debug, 
			"copy_pgp=1: Nothing to do\n"));
	return 1;
    }

    title_range = 
	state_add_simple_pager_range(dest_file,NULL,PR_MAX_WIDTH,0,0);
    raw = sr_call_RawState ();
    
    if (0 != (cm_options & CM_QUOTE_L) || 0 != (cm_options & CM_PREFIX)) {
	int flags = 0;
	
	if (0 != (cm_options & CM_QUOTE_L)) 
	    flags |= PR_QUOTE_L;
	if (0 != (cm_options & CM_PREFIX))
	    flags |= PR_REPLY_QUOTE;
	
	if (flags)
	    decode_data =  state_add_simple_pager_range(dest_file,NULL,
							flags,0,0);
    }
    

    if (decode_data) {
	DPRINT(MimeDebug,15,(&MimeDebug,
			     "copy_pgp: decode_data quote_level=%d range_flags=%d\n",
			     get_pager_range_quote_level(decode_data),
			     get_pager_range_flags(decode_data)));

    }

    if (current_header->pgp & PGP_PUBLIC_KEY) {
	/* \n resets this */
	set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	state_printf(dest_file,CATGETS(elm_msg_cat, MeSet, MePgpPublicKeys,
				       "(** This message contains PGP public key(s) **)\n"));
	state_nlputs("\n",dest_file);
    }
    
    /* default-nomime-charset gives assumed charset */

    if (current_header->override_charset) {
	const char *MIME_name_o UNUSED_VAROK = 
	    get_charset_MIME_name(current_header->override_charset);

	set_out_state_filter(dest_file,current_header->override_charset);

	DPRINT(PgpDebug,5,(&PgpDebug,
			   "Override charset %s given\n",
			   MIME_name_o ? MIME_name_o : "<none>"));
    } else
	set_out_state_filter(dest_file,default_nomime_charset);

    if(pgp_seen) {
	buf_len = mail_gets(buffer, VERY_LONG_STRING, infile);
    } else {
	while (body_bytes < current_header->content_length) {      
	    if (! (buf_len = mail_gets(buffer, sizeof buffer, infile)))
		break;
	    
	    if (strncmp(buffer, "-----BEGIN PGP", 14) == 0) {
		pgp_seen = 1;
		break;
	    }
	    /* text before PGP section */
	    if (0 == body_bytes) {
		/* \n resets this */
		set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
		state_printf(dest_file,
			     CATGETS(elm_msg_cat, MeSet, MePgpBefore,
				     "[ There is text before PGP section. ]\n"));
	    }	    

	    body_bytes += buf_len;

	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(buffer,&buf_len,sizeof buffer,dest_file);

	    if (decode_data)
		set_out_state_line_pager_range(dest_file,decode_data);

	    err = state_put(buffer, buf_len, dest_file);
	    if (err != buf_len) {
		DPRINT(PgpDebug,1,(&PgpDebug,
				   "copy_pgp fails (err=%d != %d=buf_len)\n",
				   err,buf_len));
		goto fail;
	    }	
	}
    }
   
    set_out_state_filter(dest_file,NULL);
    
    if (buf_len <= 0 || body_bytes >= current_header->content_length) {
	ret = 1;
	goto okay;
    } else {
	int l = ftell(infile);
	int len1;
	char buf[STRING];
	
	/* 
	 * On PGP 2 messages these is empty line immediately after
	 * -----BEGIN PGP SIGNED MESSAGE----
	 * 
	 * If there is something other such as 
	 * Hash: SHA1
	 * PGP 2 does not understand message.
	 */
	
	while (0 < (len1 = 
		    mail_gets (buf, sizeof (buf), infile))) {
	    if ((len1 == 1 && buf[0] == '\n') ||
		(len1 == 2 && buf[0] == '\r' && buf[1] == '\n'))
		break;
	    if (current_header->pgp == PGP_SIGNED_MESSAGE) {
		DPRINT(PgpDebug,4,(&PgpDebug,			     
				   "copy_pgp: peek: Header on armor -- requires PGP 5 or GnuPG\n" ));
		v = gpg;
		armor_header++;
		break;
	    }
	    if (0 == strncmp("Version: ",buf,9)) {
		char *c = buf+9;

		v = decode_pgp_version(c);
		if (armor_header && pgp2 == v) 
		    v = gpg;
	    }
	}

	/* Look also for -----BEGIN PGP SIGNATURE----- ... */
	if (current_header->pgp == PGP_SIGNED_MESSAGE && len1 > 0) {
	    DPRINT(PgpDebug,4,(&PgpDebug,
			       "copy_pgp: Looking for -----BEGIN PGP SIGNATURE\n"));
	    
	    while ((len1 = mail_gets (buf, sizeof (buf), infile)) > 0) {
		if (len1 > 24 &&
		    0 == strncmp(buf,
				 "-----BEGIN PGP SIGNATURE",24))
		    break;
	    }
	  
	    while ((len1 = mail_gets (buf, sizeof (buf), infile)) > 0) {
		if ((len1 == 1 && buf[0] == '\n') ||
		    (len1 == 2 && buf[0] == '\r' && buf[1] == '\n'))
		    break;
	      
		if (0 == strncmp("Version: ",buf,9)) {
		    char *c = buf+9;
		    
		    v = decode_pgp_version(c);
		    if (armor_header && pgp2 == v) 
			v = gpg;
		}
	    }
	}
	
	fseek(infile,l,SEEK_SET);
    }
    
    version = have_pgp(v);
    
    if (!version) {
	/* \n resets this */
	set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	state_printf(dest_file,
		     CATGETS(elm_msg_cat, MeSet, MePgpNotAvailRawdata,
			     "[ PGP not available, raw data follows ]\n"));
	goto raw;
    }

    if ((current_header->pgp & PGP_MESSAGE) && pgp_keeppass) {
	if (!pgp_goodPassphrase(version)) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDecryptBadPGP,
			      "Decrypting message... Bad PGP passphrase."));

	    /* \n resets this */
	    set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	    state_puts("[ ",dest_file);
	    state_printf(dest_file,CATGETS(elm_msg_cat, ElmSet, 
					   ElmDecryptBadPGP,
					   "Decrypting message... Bad PGP passphrase."));
	    state_nlputs(" ]\n",dest_file);

	    ret = 1;
	    goto okay;
	}
    }
    
    if (!pgp_decrypt_init(&fpin, &fpout, current_header->pgp,
			  version,&RS)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmDecryptFailInitPGP,
			  "Decrypting message... Failed to init PGP."));

	/* \n resets this */
	set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	state_printf(dest_file,
		     CATGETS(elm_msg_cat, ElmSet, ElmDecryptFailInitPGPRaw,
			     "[ Decrypting message... Failed to init PGP. Raw data follows. ]\n"));
    raw:

	while (body_bytes < current_header->content_length) {      
	    body_bytes += buf_len;

	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(buffer,&buf_len,sizeof buffer,dest_file);

	    if (decode_data)
		set_out_state_line_pager_range(dest_file,decode_data);
	    err = state_put(buffer, buf_len, dest_file);
	    if (err != buf_len) {
		DPRINT(PgpDebug,4,(&PgpDebug, 
				   "copy_pgp fails (err=%d != %d=buf_len)\n",
				   err,buf_len));
		goto fail;
	    }	

	    if (! (buf_len = mail_gets(buffer, VERY_LONG_STRING, infile)))
		break;
	}

	/* \n resets this */
	set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	state_printf(dest_file,
		     CATGETS(elm_msg_cat, ElmSet, ElmDecryptEndRaw,
			     "[ Decrypting message... End of raw data. ]\n"));
	ret = 1;
	goto okay;
    }

    /* Pass PGP section to pgp */   
    while (body_bytes < current_header->content_length) {      
	if (EOF == fputs(buffer, fpout)) {
	    DPRINT(PgpDebug,4,(&PgpDebug,
			       "copy_pgp fails\n"));
	    goto fail;
	}
	body_bytes += buf_len;
	if (! (buf_len = mail_gets(buffer, VERY_LONG_STRING, infile)))
	    break;
    }
    
    if (fclose(fpout) == EOF) {
	fpout = NULL;
	DPRINT(PgpDebug,1,(&PgpDebug, 
			   "copy_pgp fails\n"));
	goto fail;
    }
    fpout = NULL;

    DPRINT(PgpDebug,5,(&PgpDebug,
		       "copy_pgp: Passed %d bytes to PGP.\n",
		       body_bytes));


    if (body_bytes != current_header->content_length) {
	DPRINT(PgpDebug,1,(&PgpDebug,
			   "copy_pgp: ERROR: read bytes %d != content-length %d\n",
			   body_bytes,current_header->content_length));
	ret = 0;
    }

    body_bytes = 0;

    code = run_already_done(&RS,&stat);
	     
    if (code != 0) {
	/* \n resets this */
	set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	
	if (current_header->pgp & PGP_MESSAGE)
	    state_printf(dest_file,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartEncoded1,
				 "-- Start of PGP encoded section%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						    MePgpFail,
						    ", PGP failed!") : ".");
	else if (current_header->pgp & PGP_SIGNED_MESSAGE)
	    state_printf(dest_file,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartSigned1,
				 "-- Start of PGP signed section%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						    MePgpFail,
						    ", PGP failed!") : ".");
	else if (current_header->pgp & PGP_PUBLIC_KEY)
	    state_printf(dest_file,
			 CATGETS(elm_msg_cat, MeSet, MePgpStartOutput1,
				 "-- Start of PGP output%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						    MePgpFail,
						    ", PGP failed!") : ".");
	else
	    state_printf(dest_file,
			 CATGETS(elm_msg_cat, MeSet, MePgpStart1,
				 "-- Start of PGP section%s\n"),
			 code < 0 || stat ? catgets(elm_msg_cat, MeSet, 
						    MePgpFail,
						    ", PGP failed!") : ".");
	
    } else {
	/* \n resets this */	
	set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	if (current_header->pgp & PGP_MESSAGE)
	    state_printf(dest_file,CATGETS(elm_msg_cat, MeSet, MePgpStartEncoded,
					   "-- Start of PGP encoded section.\n"));
	else if (current_header->pgp & PGP_SIGNED_MESSAGE)
	    state_printf(dest_file,CATGETS(elm_msg_cat, MeSet, MePgpStartSigned,
					   "-- Start of PGP signed section.\n"));
	else if (current_header->pgp & PGP_PUBLIC_KEY)
	    state_printf(dest_file,CATGETS(elm_msg_cat, MeSet, MePgpStartOutput,
					   "-- Start of PGP output.\n"));
	else 
	    state_printf(dest_file,CATGETS(elm_msg_cat, MeSet, MePgpStart,
					   "-- Start of PGP section.\n"));
    }

 retry:  
    while (0 < (buf_len = mail_gets(buffer, VERY_LONG_STRING, fpin))) {
	body_bytes += buf_len;
	
	/* Take care of CRLF => LF conversion or
	   LF -> CRLF conversion 
	*/
	state_convert_EOLN(buffer,&buf_len,sizeof buffer,dest_file);

	if (decode_data)
	    set_out_state_line_pager_range(dest_file,decode_data);
	err = state_put(buffer, buf_len, dest_file);
	if (err != buf_len) {
	    DPRINT(PgpDebug,1,(&PgpDebug, 
			       "copy_pgp fails (err=%d != %d=buf_len)\n",
			       err,buf_len));
	    goto fail;
	}
    }


    if (ferror(fpin) && EINTR == errno) {
	clearerr(fpin);
	DPRINT(PgpDebug,5,(&PgpDebug,
			   "Reading of result interrupted (EINTR) -- retrying\n"));
	
	if (0 == code) {
	    code = run_already_done(&RS,&stat);
	    if (0 != code) {
		DPRINT(PgpDebug,5,(&PgpDebug,
				   "now pgp/gpg is completing\n"));
	    }      
	}
	
      goto retry;
    }

    fclose(fpin); fpin = NULL;

    if (0 == code)
	code = wait_end(&RS,&stat);

    DPRINT(PgpDebug,10,(&PgpDebug,"copy_pgp: code=%d, exit status=%d\n",
		     code,stat));

    call_print_status_cooked(&RS,code < 0 ? -code : 0,stat);

    if (raw)
	sr_call_Raw (ON);

    /* \n resets this */	
    set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
    if (current_header->pgp & PGP_MESSAGE)
	state_printf(dest_file,
		     CATGETS(elm_msg_cat, MeSet, MePgpEndEncoded,
			     "-- End of PGP encoded section%s\n"),
		     code < 0 || stat ? catgets(elm_msg_cat, MeSet, MePgpFail,
						", PGP failed!") : ".");
    else if (current_header->pgp & PGP_SIGNED_MESSAGE)
	state_printf(dest_file,
		     CATGETS(elm_msg_cat, MeSet, MePgpEndSigned,
			     "-- End of PGP signed section%s\n"),
		     code < 0 || stat ?  catgets(elm_msg_cat, MeSet, MePgpFail,
						 ", PGP failed!") : ".");
    else if (current_header->pgp & PGP_PUBLIC_KEY)
	state_printf(dest_file,
		     CATGETS(elm_msg_cat, MeSet, MePgpEndOutput,
			     "-- End of PGP output%s\n"),
		     code < 0 || stat ?  catgets(elm_msg_cat, MeSet, MePgpFail,
						 ", PGP failed!") : ".");
    else
	state_printf(dest_file,
		     CATGETS(elm_msg_cat, MeSet, MePgpEnd,
			     "-- End of PGP section%s\n"),
		     code < 0 || stat ? catgets(elm_msg_cat, MeSet, MePgpFail,
						", PGP failed!") : ".");
  
    DPRINT(PgpDebug,5,(&PgpDebug,
		       "copy_pgp=%d: Read %d bytes from PGP.\n",
		       ret,body_bytes));
    

 okay:
    if (decode_data)
	free_pager_range(& decode_data);
    free_pager_range(&title_range);

    return ret;

 fail:
    if (fpout)
	fclose(fpout);
    if (fpin)
	fclose(fpin);    
    if (0 == code)
	code = wait_end(&RS,&stat);
    
    if (decode_data)
	free_pager_range(& decode_data);
    free_pager_range(&title_range);

    return 0;
}

/* Prototype */
int copy_mime P_((struct out_state *dest_file,
		  int cm_options,
		  struct header_rec *entry,
		  FILE *infile));

int copy_mime(dest_file,cm_options,current_header, infile)
     struct out_state *dest_file;
     int cm_options;
     struct header_rec *current_header;
     FILE *infile;
{
    struct in_state   * state_in   = NULL;
    struct decode_opts  DECODE_OPT = NULL_decode_opt;
    struct pager_range * decode_data = NULL;

    DPRINT(MimeDebug,5,(&MimeDebug,
			"copy_mime called: Need read %d bytes\n",
			current_header->content_length));

    /* Allow case where content_length == -1 */
    if (current_header->content_length < 1) {
	DPRINT(Debug,1,(&Debug, 
			"copy_mime=1: Nothing to do\n"));
	return 1;
    }

    state_in   = new_in_state(STATE_in_file);
    
    if (!current_header->mime_parsed) {
	DPRINT(MimeDebug,1,(&MimeDebug,"mime_parse_routine was not called\n"));
	mime_parse_routine(NULL,current_header,infile);
    }

    DECODE_OPT.displaying = ison(cm_options,CM_DISPLAYING);
    DECODE_OPT.show_error = ison(cm_options,CM_SHOW_ERROR);
    
    if (ison(cm_options,CM_QUOTE_L) || ison(cm_options,CM_PREFIX)) {
	int flags = 0;

	if (0 != (cm_options & CM_QUOTE_L)) 
	    flags |= PR_QUOTE_L;
	if (0 != (cm_options & CM_PREFIX))
	    flags |= PR_REPLY_QUOTE;
		
	if (flags)
	    decode_data =  state_add_simple_pager_range(dest_file,NULL,
							flags,0,0);
    }

    if (decode_data) {
	DPRINT(MimeDebug,15,(&MimeDebug,
			     "copy_mime: decode_data quote_level=%d range_flags=%d\n",
			     get_pager_range_quote_level(decode_data),
			     get_pager_range_flags(decode_data)));
    }

    DECODE_OPT.range  =  decode_data;

    set_in_state_file(infile,state_in);

    mime_decode(&(current_header->mime_rec), state_in, 
		dest_file,&DECODE_OPT,
		current_header->header_charset,
		current_header, mime_signature_mismatch,
		current_header->default_body_charset);
  
    DPRINT(MimeDebug,5,(&MimeDebug,
			"copy_mime: mail decoded\n"));

    free_in_state(&state_in); 

    if (decode_data)
	free_pager_range(& decode_data);

    return 1;
}

/* Prototype */
static int copy_encrypted P_((struct out_state *,int, struct header_rec *, 
			       FILE *));

static int copy_encrypted(dest_file,cm_options,current_header,infile)    
     struct out_state *dest_file;
     int cm_options;
     struct header_rec *current_header;
     FILE *infile;
{
    char buffer[VERY_LONG_STRING];
    long body_bytes = 0;
    int crypted = OFF;
    int buf_len;
    
    struct pager_range * title_range = NULL;
    struct pager_range * decode_data = NULL;

    DPRINT(Debug,5,(&Debug,
		    "copy_encrypted called: Need read %d bytes\n",
		    current_header->content_length));

    /* Allow case where content_length == -1 */
    if (current_header->content_length < 1) {
	DPRINT(Debug,1,(&Debug, 
			"copy_encrypted=1: Nothing to do\n"));
	return 1;
    }

    title_range = 
	state_add_simple_pager_range(dest_file,NULL,PR_MAX_WIDTH,0,0);


    if (0 != (cm_options & CM_QUOTE_L) || 0 != (cm_options & CM_PREFIX)) {
	int flags = 0;
	
	if (0 != (cm_options & CM_QUOTE_L)) 
	    flags |= PR_QUOTE_L;
	if (0 != (cm_options & CM_PREFIX))
	    flags |= PR_REPLY_QUOTE;
	
	if (flags)
	    decode_data =  state_add_simple_pager_range(dest_file,NULL,
							flags,0,0);
    }
    


    

    if (decode_data) {
	DPRINT(MimeDebug,15,(&MimeDebug,
			     "copy_encrypted: decode_data quote_level=%d range_flags=%d\n",
			     get_pager_range_quote_level(decode_data),
			     get_pager_range_flags(decode_data)));

    }

    getkey(OFF);

    /* default-nomime-charset gives assumed charset */
    if (current_header->override_charset) {
	const char *MIME_name_o UNUSED_VAROK = 
	    get_charset_MIME_name(current_header->override_charset);

	set_out_state_filter(dest_file,current_header->override_charset);

	DPRINT(Debug,5,(&Debug,
			"Override charset %s given\n",
			MIME_name_o ? MIME_name_o : "<none>"));
    } else
	set_out_state_filter(dest_file,default_nomime_charset);
  
    while (body_bytes < current_header->content_length) {      
	int err;

	if (! (buf_len = mail_gets(buffer, sizeof buffer, infile)))
	    break;
    

	if (body_bytes + buf_len >= current_header->content_length) {

	    DPRINT(Debug,5,(&Debug,
			    "copy_encrypted: Readed past of end of body\n"));

	    if (buf_len > 0 &&  
		'\n' == buffer[buf_len-1]  && 
		body_bytes + buf_len -1 == current_header->content_length) {
		DPRINT(Debug,5,(&Debug,
				"    ... ignoring NL\n"));
		buf_len -= 1;
	    } else if (buf_len > 1 &&  
		       '\r' == buffer[buf_len-2] &&
		       '\n' == buffer[buf_len-1] &&
		       body_bytes + buf_len -2 == 
		       current_header->content_length
		       ) {

		DPRINT(Debug,5,(&Debug,
				"    ... ignoring CR NL\n"));
		buf_len -= 2;
	    }
	}



	body_bytes += buf_len;

	/* Take care of CRLF => LF conversion or
	   LF -> CRLF conversion 
	*/
	state_convert_EOLN(buffer,&buf_len,sizeof buffer,dest_file);
    
	if (mime_body_keywords && !strncmp(buffer, START_ENCODE, strlen(START_ENCODE))) {
	    crypted = ON;

	    /* \n resets this */	
	    set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(dest_file,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeStartElmEncoded,
				 "-- Start of (Elm) encoded section.\n"));
	    continue;
	} else if (mime_body_keywords && !strncmp(buffer, END_ENCODE, strlen(END_ENCODE))) {
	    crypted = OFF;
	    /* \n resets this */	
	    set_out_state_line_mode(dest_file,pg_BOLD,title_range,1 /* Newline */);
	    state_printf(dest_file,
			 CATGETS(elm_msg_cat, MeSet, MeDecodeEndElmEncoded,
				 "-- End of (Elm) encoded section.\n"));
	    continue;
	} else if (crypted) {
	    no_ret(buffer);
	    encode(buffer);      
	    if (out_state_EOLN_is_CRLF(dest_file))
		strfcat(buffer, "\r\n", sizeof buffer);
	    else
		strfcat(buffer, "\n", sizeof buffer);
	}

	if (decode_data)
	    set_out_state_line_pager_range(dest_file,decode_data);

	err = state_puts(buffer,dest_file);
	if (err == EOF) {
	    DPRINT(Debug,1,(&Debug,
			    "copy_encrypted fails\n"));
	    goto fail;
	}
    }

    DPRINT(Debug,5,(&Debug,
		    "copy_encrypted: Read %d bytes from body\n",body_bytes));

    if (body_bytes != current_header->content_length) {
	DPRINT(Debug,1,(&Debug,
			"copy_encrypted: ERROR: read bytes %d != content-length %d\n",
			body_bytes,current_header->content_length));
	goto fail;
    }

    if (decode_data)
	free_pager_range(& decode_data);
    free_pager_range(&title_range);

    return 1;

 fail:

    if (decode_data)
	free_pager_range(& decode_data);
    free_pager_range(&title_range);

    return 0;
}

/* Prototype */
int copy_plain P_((struct out_state *,int, struct header_rec *, FILE *));

int copy_plain(dest_file,cm_options,current_header, infile) 
     struct out_state *dest_file;
     int cm_options;
     struct header_rec *current_header;
     FILE *infile;
{
    char buffer[VERY_LONG_STRING];
    int body_bytes = 0;
    int buf_len, err;
    
    struct pager_range * decode_data = NULL;

    DPRINT(Debug,5, (&Debug, 
		     "copy_plain called: Need read %d bytes\n",
		     current_header->content_length));;

    /* Allow case where content_length == -1 */
    if (current_header->content_length < 1) {
	DPRINT(Debug,1,(&Debug, 
			"copy_plain=1: Nothing to do\n"));
	return 1;
    }

    if (0 != (cm_options & CM_QUOTE_L) || 0 != (cm_options & CM_PREFIX)) {
	int flags = 0;
	
	if (0 != (cm_options & CM_QUOTE_L)) 
	    flags |= PR_QUOTE_L;
	if (0 != (cm_options & CM_PREFIX))
	    flags |= PR_REPLY_QUOTE;
	
	if (flags)
	    decode_data =  state_add_simple_pager_range(dest_file,NULL,
							flags,0,0);
    }
    
    if (decode_data) {
	DPRINT(MimeDebug,15,(&MimeDebug,
			     "copy_plain: decode_data quote_level=%d range_flags=%d\n",
			     get_pager_range_quote_level(decode_data),
			     get_pager_range_flags(decode_data)));

    }

    /* default-nomime-charset gives assumed charset */
    if (current_header->override_charset) {
	const char *MIME_name_o UNUSED_VAROK = 
	    get_charset_MIME_name(current_header->override_charset);
	
	set_out_state_filter(dest_file,current_header->override_charset);

	DPRINT(Debug,5,(&Debug,
			"Override charset %s given\n",
			MIME_name_o ? MIME_name_o : "<none>"));
    } else
	set_out_state_filter(dest_file,default_nomime_charset);
    
    while (body_bytes < current_header->content_length) {      
	if (! (buf_len = mail_gets(buffer, VERY_LONG_STRING, infile)))
	    break;

	if (body_bytes + buf_len >= current_header->content_length) {

	    DPRINT(Debug,5,(&Debug,
			    "copy_plain: Readed past of end of body\n"));

	    if (buf_len > 0 &&  
		'\n' == buffer[buf_len-1]  && 
		body_bytes + buf_len -1 == current_header->content_length) {
		DPRINT(Debug,5,(&Debug,
				"    ... ignoring NL\n"));
		buf_len -= 1;
	    } else if (buf_len > 1 &&  
		       '\r' == buffer[buf_len-2] &&
		       '\n' == buffer[buf_len-1] &&
		       body_bytes + buf_len -2 == 
		       current_header->content_length
		       ) {

		DPRINT(Debug,5,(&Debug,
				"    ... ignoring CR NL\n"));
		buf_len -= 2;
	    }
	}

	
	body_bytes += buf_len;

	/* Take care of CRLF => LF conversion or
	   LF -> CRLF conversion 
	*/
	state_convert_EOLN(buffer,&buf_len,sizeof buffer,dest_file);
	
	if (decode_data)
	    set_out_state_line_pager_range(dest_file,decode_data);
	err = state_put(buffer, buf_len, dest_file);
	if (err != buf_len) {
	    DPRINT(Debug,1,(&Debug, 
			    "copy_plain fails (err=%d != %d=buf_len)\n",
			    err,buf_len));
	    goto fail;
	}
    }
    DPRINT(Debug,5,(&Debug,
		    "copy_plain: Read %d bytes from body\n",
		    body_bytes));
    
    if (body_bytes != current_header->content_length) {
	DPRINT(Debug,1,(&Debug,
			"copy_plain: ERROR: read bytes %d != content-length %d\n",
			body_bytes,current_header->content_length));
	goto fail;
    }

    if (decode_data)
	free_pager_range(& decode_data);
    return 1;

 fail:
    if (decode_data)
	free_pager_range(& decode_data);
    return 0;
}

/* Prototype */
int copy_binary P_((struct out_state *,int, struct header_rec *, FILE *));

int copy_binary(dest_file,cm_options,current_header, infile) 
     struct out_state *dest_file;
     int cm_options;
     struct header_rec *current_header;
     FILE *infile;
{
    char buffer[VERY_LONG_STRING];
    int body_bytes = 0;
    int buf_len, err;
    
    struct string * prefixS = NULL;

    DPRINT(Debug,5, (&Debug, 
		     "copy_binary called: Need read %d bytes\n",
		     current_header->content_length));;

        /* Allow case where content_length == -1 */
    if (current_header->content_length < 1) {
	DPRINT(Debug,1,(&Debug, 
			"copy_binary=1: Nothing to do\n"));
	return 1;
    }

    prefixS = ((cm_options & CM_PREFIX) ? quote_prefix : NULL);


    /* No filtering or charset handling ... */
    set_out_state_filter(dest_file,NULL);

    while (body_bytes < current_header->content_length) {      
	if (! (buf_len = mail_gets(buffer, VERY_LONG_STRING, infile)))
	    break;

	if (body_bytes + buf_len >= current_header->content_length) {

	    DPRINT(Debug,5,(&Debug,
			    "copy_binary: Readed past of end of body\n"));

	    if (buf_len > 0 &&  
		'\n' == buffer[buf_len-1]  && 
		body_bytes + buf_len -1 == current_header->content_length) {
		DPRINT(Debug,5,(&Debug,
				"    ... ignoring NL\n"));
		buf_len -= 1;
	    } else if (buf_len > 1 &&  
		       '\r' == buffer[buf_len-2] &&
		       '\n' == buffer[buf_len-1] &&
		       body_bytes + buf_len -2 == 
		       current_header->content_length
		       ) {

		DPRINT(Debug,5,(&Debug,
				"    ... ignoring CR NL\n"));
		buf_len -= 2;
	    }
	}


	body_bytes += buf_len;

	/* NO CONVERSIONS! */
    
	DPRINT(Debug,15,(&Debug,
		  "copy_binary: -> %.*s",buf_len,buffer));
	if (buf_len < 1 || buffer[buf_len-1] != '\n') {
	    DPRINT(Debug,15,(&Debug,
			     "\ncopy_binary <-- NO NEWLINE\n"));
	}

	err = 0;
	if (prefixS)
	    err = state_printf(dest_file,FRM("%S"),prefixS);
	if (err != EOF)
	    err = state_put(buffer, buf_len, dest_file);
	if (err != buf_len) {
	    DPRINT(Debug,1,(&Debug,
			    "copy_binary fails (err=%d != %d=buf_len)\n",
			    err,buf_len));
	    goto fail;
	}
    }
    DPRINT(Debug,5,(&Debug,
		    "copy_binary: Read %d bytes from body\n",body_bytes));

    if (body_bytes != current_header->content_length) {
	DPRINT(Debug,1,(&Debug,
			"copy_binary: ERROR: read bytes %d != content-length %d\n",
			body_bytes,current_header->content_length));
	goto fail;
    }
    return 1;

 fail:

    return 0;
}


/* Prototype */
int copy_cooked P_((struct out_state *,int, struct header_rec *, FILE *));

int copy_cooked(dest_file,cm_options,current_header, infile) 
     struct out_state *dest_file;
     int cm_options;
     struct header_rec *current_header;
     FILE *infile;
{
    char buffer[VERY_LONG_STRING];
    int body_bytes = 0;
    int buf_len, err;

    struct string * prefixS = NULL;

    DPRINT(Debug,5, (&Debug, 
		     "copy_cooked called: Need read %d bytes\n",
		     current_header->content_length));;

    /* Allow case where content_length == -1 */
    if (current_header->content_length < 1) {
	DPRINT(Debug,1,(&Debug, 
			"copy_cooked=1: Nothing to do\n"));
	return 1;
    }

    prefixS = 
	((cm_options & CM_PREFIX) ? quote_prefix : NULL);


    /* No filtering or charset handling ... */
    set_out_state_filter(dest_file,NULL);

    while (body_bytes < current_header->content_length) {      
	if (! (buf_len = mail_gets(buffer, sizeof buffer, infile)))
	    break;


	if (body_bytes + buf_len >= current_header->content_length) {

	    DPRINT(Debug,5,(&Debug,
			    "copy_cooked: Readed past of end of body\n"));

	    if (buf_len > 0 &&  
		'\n' == buffer[buf_len-1]  && 
		body_bytes + buf_len -1 == current_header->content_length) {
		DPRINT(Debug,5,(&Debug,
				"    ... ignoring NL\n"));
		buf_len -= 1;
	    } else if (buf_len > 1 &&  
		       '\r' == buffer[buf_len-2] &&
		       '\n' == buffer[buf_len-1] &&
		       body_bytes + buf_len -2 == 
		       current_header->content_length
		       ) {

		DPRINT(Debug,5,(&Debug,
				"    ... ignoring CR NL\n"));
		buf_len -= 2;
	    }
	}



	body_bytes += buf_len;
	
	DPRINT(Debug,15,(&Debug,
			 "copy_cooked: -> %.*s",buf_len,buffer));
	if (buf_len < 1 || buffer[buf_len-1] != '\n') {
	    DPRINT(Debug,15,(&Debug,
			     "\ncopy_cooked <-- NO NEWLINE\n"));
	} else {
	    /* Take care of CRLF => LF conversion or
	       LF -> CRLF conversion 
	    */
	    state_convert_EOLN(buffer,&buf_len,sizeof buffer,dest_file);
	}
	
	err = 0;
	if (prefixS)
	    err = state_printf(dest_file,FRM("%S"),prefixS);
	if (err != EOF)
	    err = state_put(buffer, buf_len, dest_file);
	if (err != buf_len) {
	    DPRINT(Debug,1,(&Debug, 
			    "copy_cooked fails (err=%d != %d=buf_len)\n",
			    err,buf_len));
	    goto fail;
	}
    }
    DPRINT(Debug,5,(&Debug,
		    "copy_cooked: Read %d bytes from body\n",body_bytes));

    if (body_bytes != current_header->content_length) {
	DPRINT(Debug,1,(&Debug,
			"copy_cooked: ERROR: read bytes %d != content-length %d\n",
			body_bytes,current_header->content_length));
	goto fail;
    }
    return 1;

 fail:

    return 0;
}


copy_decoder_t select_copy_decoder (current_header) 
     struct header_rec * current_header;
{
    if (current_header->status & MIME_MESSAGE) 
	return copy_mime; 

    else if (use_PGP &&
	     0 != (current_header->pgp & (PGP_MESSAGE|PGP_SIGNED_MESSAGE|PGP_PUBLIC_KEY)))
	return copy_pgp;

    else if (current_header -> encrypted)
	return copy_encrypted;
    else
	return copy_plain;
}

int copy_message_d(infolder,current_header,dir,dest,cm_options,
		   file_set,sender)
     struct folder_info *infolder;
     struct header_rec *current_header;
     struct folder_browser *dir; 
     WRITE_STATE dest; 
     int cm_options;
     charset_t file_set;
     const char *sender;
{
    FILE * infile;
    int remove_envelope = cm_options & CM_REMOVE_ENVELOPE;
    int decode = cm_options & CM_DECODE;

    int ret = 0;
    int env_flags = 0;
    struct out_state * buffer = NULL;
    charset_t charset_vector[2];

    if (decode && !file_set)
	panic("FILE PANIC",__FILE__,__LINE__,
	      "copy_message_d",
	      "decode set but not file_set",0);


    /* Needed by STATE_out_dir */
    charset_vector[0] = decode ? file_set : RAW_BUFFER;
    charset_vector[1] = NULL;

    if (decode) {
	const char *MIME_name_0 UNUSED_VAROK = 
	    get_charset_MIME_name(charset_vector[0]);

	DPRINT(MimeDebug,5, 
	       (&MimeDebug, "Decoding message to %s charset\n",
		MIME_name_0 ? MIME_name_0 : "<no MIME name>"));
    }


    if (!prepare_message_access(infolder,
				current_header,
				parse_header_routine,
				parse_body_routine,
				NULL,
				decode ? mime_parse_routine : NO_mime_parse)) {
	DPRINT(Debug,5, (&Debug,
			 "copy_message_d: prepare_message_access failed\n"));
	return 0;
    }

    infile = folder_to_fd(infolder,current_header->offset);
    if (!infile) {
	DPRINT(Debug,1, 
	       (&Debug, 
		"ERROR: Attempt to seek %d bytes into file failed (%s)",
		current_header->offset, "copy_message_d"));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSeekFailed,
			  "ELM [seek] failed trying to read %d bytes into file."),
		  current_header->offset);
	return 0;
    }
    
    if (!write_envelope_start(dir,dest,!remove_envelope,
			      current_header,&env_flags))
	goto fail;
    
    buffer = new_out_state(STATE_out_dir);

    set_out_state_cs(buffer,NULL,charset_vector);
    set_out_state_dir(dir,dest,buffer);
            
    ret = copy_message_2(infile,current_header,buffer,cm_options,
			 env_flags,sender);

    free_out_state(&buffer);

    if (!write_envelope_end(dir,dest,!remove_envelope,
				    current_header))
	ret = 0;

 fail:
    if (!ret) {
	DPRINT(Debug,1,(&Debug, 
			"copy_message_d fails\n"));
    }
    return ret;
}

int copy_message_df(infile,current_header,dir,dest,cm_options,
		   file_set,sender)
     FILE *infile;
     struct header_rec *current_header;
     struct folder_browser *dir; 
     WRITE_STATE dest; 
     int cm_options;
     charset_t file_set;
     const char *sender;
{
    int remove_envelope = cm_options & CM_REMOVE_ENVELOPE;
    int decode = cm_options & CM_DECODE;
    
    int ret = 0;
    int env_flags = 0;
    struct out_state * buffer = NULL;
    charset_t charset_vector[2];
    
    if (decode && !file_set)
	panic("FILE PANIC",__FILE__,__LINE__,
	      "copy_message_df",
	      "decode set but not file_set",0);

    /* Needed by STATE_out_dir */
    charset_vector[0] = decode ? file_set : RAW_BUFFER;
    charset_vector[1] = NULL;

    if (decode) {
	const char *MIME_name_0 UNUSED_VAROK = 
	    get_charset_MIME_name(charset_vector[0]);
	
	DPRINT(MimeDebug,5, 
	       (&MimeDebug, "Decoding message to %s charset\n",
		MIME_name_0 ? MIME_name_0 : "<no MIME name>"));
    }

    /* FILE * should be already seeked to correct position, but
       this is extra assurance in case of it is readed meanwhile
    */
    if (0 !=  fseek(infile,current_header->offset,SEEK_SET)) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeFailedSeekEnvelope,
			  "Failed to seek beginning of mail envelope (%ld)"),
		  current_header->offset);
	goto fail;
    }

    if (!write_envelope_start(dir,dest,!remove_envelope,
			      current_header,&env_flags))
	goto fail;
    
    buffer = new_out_state(STATE_out_dir);
    set_out_state_cs(buffer,NULL,charset_vector);
    set_out_state_dir(dir,dest,buffer);
            
    ret = copy_message_2(infile,current_header,buffer,cm_options,
			 env_flags,sender);

    free_out_state(&buffer);

    if (!write_envelope_end(dir,dest,!remove_envelope,
				    current_header))
	ret = 0;

 fail:
    if (!ret) {
	DPRINT(Debug,1,(&Debug, 
			"copy_message_df fails\n"));
    }
    return ret;
}

int copy_message_f(infile, current_header, dest_file, cm_options,
		   file_set,sender)
     FILE *infile;
     struct header_rec *current_header;
     FILE *dest_file;
     int cm_options;
     charset_t file_set;
     const char *sender;
{
    int ret = 0;
    struct folder_browser *dir  = NULL;
    WRITE_STATE            dest = NULL;

    start_fd_write_state(dest_file,&dir,&dest);

    ret = copy_message_df(infile,current_header,dir,dest,cm_options,
			  file_set,sender);

    end_fd_write_state(&dir,&dest);

    /* Since fprintf is buffered, its return value is only useful for
     * writes which exceed the blocksize.  Do a fflush to ensure that
     * the message has, in fact, been written.
     */
    
    if (fflush(dest_file) == EOF) {
	DPRINT(Debug,1, (&Debug, 
			 "copy_message_f: Final fflush failed!\n")); 
	ret = 0;
    }

    if (!ret) {
	DPRINT(Debug,1,(&Debug, 
			"copy_message_f fails\n"));
    }

    return ret;
}

/* Assumes that headers are already copied ------------------- */
int copy_body(infile,current_header,dest_file,cm_options)
     FILE *infile;
     struct header_rec *current_header;
     struct out_state *dest_file; 
     int cm_options;
{
    int ret = 0;
    int decode = cm_options & CM_DECODE;
    int j;
    const charset_t * Dv = get_out_state_charset_vector(dest_file);
    const char * filter_name  UNUSED_VAROK = 
	get_out_state_f_MIME_name(dest_file);


    DPRINT(Debug,5, (&Debug, 
		     "copy_body: decode=%d\n",decode));
    DPRINT(Debug,5, (&Debug, 
		     "         : filter=%s\n",
		     filter_name ? filter_name : "<not set or no MIME name>"));
    for (j = 0; Dv[j]; j++) {
	const char * MIME_name_d UNUSED_VAROK = get_charset_MIME_name(Dv[j]);

	DPRINT(Debug,5, (&Debug, 
			 "         : display charset[%d]=%s\n",
			 j,
			 MIME_name_d ? MIME_name_d : "<no MIME name>"));
    }

    if (decode) {   
	copy_decoder_t decoder = select_copy_decoder(current_header);
	ret = decoder(dest_file,cm_options,current_header,infile);
       
	set_out_state_filter(dest_file,NULL);
    } else if (cm_options & CM_CRLF ||
	       cm_options & CM_LF) {
	ret = copy_cooked(dest_file,cm_options,current_header,infile);
    } else    
	ret = copy_binary(dest_file,cm_options,current_header,infile);

    if (!ret) {
	DPRINT(Debug,1,(&Debug, 
			"copy_body fails\n"));
    }
    return ret;
}


static void makeAttString P_((const struct string *attribution	/* attribution string to expand	*/,
			      int sel_field	/* field to select in "%[...]" list	*/,
			      const struct header_rec *messageHeader /* current message header info	*/,
			      struct out_state *dest_file,
			      header_list_ptr all_headers));

static void makeAttString(attribution, sel_field, messageHeader, dest_file, all_headers)
     const struct string *attribution;	/* attribution string to expand		*/
     int sel_field;			/* field to select in "%[...]" list	*/
     const struct header_rec *messageHeader; /* current message header info	*/
     struct out_state *dest_file;
     header_list_ptr all_headers;
{

    int in_selection;		/* currently doing %[...] list?		*/
    int curr_field = 0;		/* field number of current %[...] list	*/

    /* Convert attribution to struct strincg so that multi-byte strings work */

    int len              = string_len(attribution);
    int pos;

    in_selection = FALSE;

    /*
     * Process the attribution string.
     */
    for ( pos = 0; pos < len; ) {

	uint16 unicode = give_unicode_from_string(attribution,pos);

	struct string * expval = NULL;

	/*
	 * Handle the character if it is not a %-expansion.
	 */
	switch (unicode) {

	case 0x007C  /*  '|' */:		/* next choice of "%[sel0|sel1|...]" list */
	    if (in_selection) {
		++curr_field;
		pos++;           /* skip '|' */
	    } else {
		expval = clip_from_string(attribution,&pos,1);
	    }
	    break;

	case 0x005D  /* ']'  */:		/* end of "%[sel0|sel1|...]" list */
	    if (in_selection) {
		in_selection = FALSE;
		pos++;
	    } else {
		expval = clip_from_string(attribution,&pos,1);
	    }
	    break;

	case 0x005C /* '\\' */:		/* backslash-quoting */
	    pos++;
	    if (pos < len) {
		unicode = give_unicode_from_string(attribution,pos);

		switch (unicode) {
		case 0x0074 /* 't' */:
		    expval = new_string(system_charset);
		    add_ascii_to_string(expval,s2us("\t"));
		    break;
		case 0x006E  /* 'n' */:
		    expval = new_string(system_charset);
		    add_ascii_to_string(expval,s2us("\n"));
		    break;
		default:
		    expval = clip_from_string(attribution,&pos,1);
		    break;
		}
	    } else {
		pos--;
		expval = clip_from_string(attribution,&pos,1);
	    }
	    break;


	case 0x0025  /* '%' */:		/* special %-expansion */

	    pos++;
	    if (pos < len) {
		unicode = give_unicode_from_string(attribution,pos);
		

		switch (unicode) {
		    header_list_ptr X;
		    int l;

		case 0x0073 /* 's' */:  /* backward compatibility with 2.4 */
		case 0x0046 /* 'F' */:  /* expand from */
		    
		    X = locate_header_by_name(all_headers,"From");

		    if (X)
			expval = give_decoded_header(X, 
						     !(messageHeader->status & NOHDRENCODING),
						     messageHeader->header_charset);
		    if (!expval) 
			expval = format_string(CATGETS(elm_msg_cat, ElmSet, 
						       ElmEnvFromParen,
						       "(env-from %s)"),
					       messageHeader->env_from);
		
		    pos++;
		    break;
		    
		case 0x0044 /* 'D' */: /* expand date */

		    if (time_OK(messageHeader->time_sent)) {
			char *c = ctime(&messageHeader->time_sent);
			if (c) {
			    l = strlen(c)-1;
			    c[l] = '\0';
			    
			    expval = new_string(system_charset);
			    add_ascii_to_string(expval,s2us(c));
			    
			} else 
			    goto fail_ctime;
			
		    } else {
		    fail_ctime:
			if (messageHeader->time_menu_year > 0 &&
			    messageHeader->time_menu_year < SHRT_MAX) {
			    expval = format_string(CATGETS(elm_msg_cat, ElmSet, 
							   ElmDateYearParen,
							   "(year %d)"),
						   messageHeader->time_menu_year);
			} else {
			    expval = format_string(CATGETS(elm_msg_cat, ElmSet, 
							   ElmDateBadParen,
							   "(bad date)"));
			}
			

		    }
		    pos++;
		    
		    break;

		case 0x0049  /* 'I' */: /* expand message ID */
		    
		    expval = new_string(system_charset);

		    if (messageHeader->message_id) { 
			char * XX = get_message_id_ascii_brackets(messageHeader->message_id);

			if (XX) {
			    add_ascii_to_string(expval,
						cs2cus(XX));
			    free(XX);
			}

		    }
		    
		    pos++;
		    break;

		case 0x0053  /* 'S' */: /* expand subject */
		    expval = dup_string(messageHeader->subject);
		    pos++;
		    break;

		case 0x005B  /* '[' */: /* %[sel0|sel1|...] */
		    in_selection = TRUE;
		    curr_field = 0;
		    expval = NULL;
		    pos++;
		    break;
		  
		case 0x0029  /* ')' */: /* special case for %)F - from name */
		    /*FALLTHROUGH*/
		case 0x003E  /* '>' */: /* special case for %>F - from address */

		    pos++;
		    if (pos < len) {
			if (give_unicode_from_string(attribution,pos) 
			    != 0x0046 /* 'F' */) {
			    pos -= 2;
			    expval = clip_from_string(attribution,&pos,3);
									    
			} else {
			  
			    switch (unicode) { 

			    case 0x0029  /* ')' */:		/* from name */

				expval = gen_From_buffer(messageHeader);
						   
				break;

			    case 0x003E  /* '>' */:             /* from addr */
				
				if (messageHeader->from) {
				    
				    int idx;
				    int addr_item_count = 
					addr_list_item_count(messageHeader->from);
				    
				    for (idx = 0; idx < addr_item_count; idx++) {
					int group = -1;
					const struct address * address = 
					    addr_list_get_item(messageHeader->from,idx,&group);
					
					const char   * addr = address_get_ascii_addr(address);
					if (!addr)
					    continue;
					
					if (expval)
					    add_ascii_to_string(expval,s2us(", "));
					else
					    expval = new_string(display_charset);
					add_ascii_to_string(expval,csUs(addr));
				    }
				}
				
				break;
			    }


			    if (!expval)   
				expval = format_string(CATGETS(elm_msg_cat, ElmSet, 
							       ElmEnvFromParen,
							       "(env-from %s)"),
						       messageHeader->env_from);

			    pos++;

			}
			
		    } else {
			pos -= 2;
			expval = clip_from_string(attribution,&pos,3);
		    }
		    break;

		case 0x0025 /* '%' */: /* add a % and skip on past... */

		    expval = clip_from_string(attribution,&pos,1);
		    break;

		    
		default:
		    
		    pos--;
		    expval = clip_from_string(attribution,&pos,2);
		    break;

		}
	    } else {
		pos--;
		expval = clip_from_string(attribution,&pos,1);
	    }
	    break;
	    
	default:			/* just a regular char */
	    expval = clip_from_string(attribution,&pos,1);
	}

	/*
	 * print the expansion value 
	 */
	if (expval != NULL && (!in_selection || curr_field == sel_field)) {

	    state_printf(dest_file,FRM("%S"),expval);
	}

	if (expval)
	    free_string(&expval);
	
    }
    
    state_nlputs("\n", dest_file);           
}


int copy_message_2(infile,current_header,dest_file,cm_options,
		   env_flags,sender)
     FILE *infile;
     struct header_rec *current_header;
     struct out_state *dest_file; 
     int cm_options;
     int env_flags;
     const char *sender;
{
    /** Copy current message to destination file, with optional 'prefix' 
	as the prefix for each line.  If remove_header is true, it will 
	skip lines in the message until it finds the end of header line...
	then it will start copying into the file... If remote is true
	then it will append "remote from <hostname>" at the end of the
	very first line of the file (for remailing) 
	
	If "filter_header" is true then it'll do some nice things to
	ensure that the forwarded message looks pleasant; e.g. remove
	stuff like ">From " lines and "Received:" lines.
	
	If "update_status" is true then it will write a new Status:
	line at the end of the headers.  It never copies an existing one.
	
	If "decode" decode MIME, PGP and elm's (unsafe) own decoding.
    **/
    
    int first_line = TRUE;
    int remove_header      = cm_options & CM_REMOVE_HEADER;
    int remove_envelope    = cm_options & CM_REMOVE_ENVELOPE;
    int update_status      = cm_options & CM_UPDATE_STATUS;
    int remail             = cm_options & CM_REMAIL;
    int decode             = cm_options & CM_DECODE;
    int filter_headers     = cm_options & CM_FILT_HDR;

    int make_attribution   = cm_options & CM_ATTRIBUTION;   
    int forwarding         = cm_options & CM_FORWARDING;

    int bytes_seen             UNUSED_VAROK = 0;
    int content_length_seen = FALSE;
    int return_path_seen = FALSE;
    long CL_pos = -1L, BODY_pos = -1L, END_pos = -1L;
    int was_binary = current_header -> binary && !decode;

    long R1, R2;
    header_list_ptr all_headers = NULL, next_hdr;
    
    int buf_len;
    struct pager_range * decode_data = NULL;

    DPRINT(Debug,5, (&Debug, 
		     "copy_message_2: cm_options=(%d)%s%s%s%s%s%s%s%s%s%s%s \n",
		     cm_options,
		     remove_header   ? " CM_REMOVE_HEADER"   : "",
		     remove_envelope ? " CM_REMOVE_ENVELOPE" : "",
		     update_status   ? " CM_UPDATE_STATUS"   : "",
		     remail          ? " CM_REMAIL"          : "",
		     decode          ? " CM_DECODE"          : "",
		     filter_headers  ? " CM_FILT_HDR"        : "",
		     (cm_options & CM_DISPLAYING) ? " CM_DISPLAYING" : "",
		     (cm_options & CM_CRLF) ? " CM_CRLF" : "",
		     (cm_options & CM_LF) ? " CM_LF" : "",
		     (cm_options & CM_PREFIX) ? " CM_PREFIX" : "",
		     (cm_options & CM_QUOTE_L) ? " CM_QUOTE_L" : ""));
    
    DPRINT(Debug,5, (&Debug, 
		     "copy_message_2: env_flags=(%d)%s%s\n",
		     env_flags,
		     (env_flags & WE_ADD_RETURN_PATH) ? " WE_ADD_RETURN_PATH" : "",
		     (env_flags & WE_USE_CRLF) ? " WE_USE_CRLF" : ""));


    if (was_binary && (cm_options & CM_LF) &&
	current_header->mime_rec.encoding != ENCODING_BINARY) {
	DPRINT(Debug,5,(&Debug,
			"copy_message_2: cm_options: CM_LF set, encoding not binary (%s), resetting binary flag\n",
			ENCODING(current_header->mime_rec.encoding)
			));
	was_binary = 0;
    }


    if (was_binary && ! out_state_EOLN_is_CRLF(dest_file)) {
	set_out_state_EOLN(dest_file,1);

	DPRINT(Debug,5,(&Debug,
			"copy_message_2: binary: Setting EOLN_is_CRLF\n"));
    }


    if (was_binary && (cm_options & CM_LF)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmBinaryMailNoConvert,
			  "Binary mail -- can not convert CRLF -> LF"));
	
	DPRINT(Debug,5,(&Debug,
			"copy_message_2: binary: Resetting CM_LF\n"));
	cm_options &= ~CM_LF;
    }


	
    if ((env_flags & WE_USE_CRLF) && ! (cm_options & CM_CRLF)) {
	cm_options |= CM_CRLF;
	DPRINT(Debug,5,(&Debug,
			"copy_message_2: env_flags: Setting cm_options CM_CRLF\n"));
    }

    if ((cm_options & CM_CRLF) && ! out_state_EOLN_is_CRLF(dest_file)) {
	set_out_state_EOLN(dest_file,1);	

	DPRINT(Debug,5,(&Debug,
			"copy_message_2: cm_options: Setting EOLN_is_CRLF\n"));
    }

    if ((cm_options & CM_CRLF) && (cm_options & CM_LF)) {
	DPRINT(Debug,1,(&Debug,
                        "copy_message_2: cm_options: Conflicting CM_CRLF and CM_LF -- resetting CM_LF\n"));
	cm_options &= ~CM_LF;
    }

    if ((cm_options & CM_LF) && out_state_EOLN_is_CRLF(dest_file)) {
	set_out_state_EOLN(dest_file,1);

	DPRINT(Debug,1,(&Debug,
			"copy_message_2: cm_options: CM_LF conflict with EOLN_is_CRLF -- resetting CM_LF\n"));
	cm_options &= ~CM_LF;
    }

    if (was_binary && (cm_options & CM_CRLF)) {
	cm_options &= ~CM_CRLF;
	DPRINT(Debug,5,(&Debug,
			"copy_message_2: binary: Resetting cm_options CM_CRLF flag (no conversion!)\n"));
    }

    /** get to the first line of the message desired **/
    

    /* No filter on here ... */
    set_out_state_filter(dest_file,NULL);
    
    if (decode) {
	const charset_t * Dv = get_out_state_charset_vector(dest_file);
	/* HACK:
	   Required by 
	   state_write_header so
	   that 8bit headers are
	   printed correctly
	*/
	set_out_state_filter(dest_file,Dv[0]);
    }

    /* now while not EOF & still in message... copy it! */
    
    DPRINT(Debug,5,(&Debug,
		    "copy_message_2: [%ld] start mailbox separator section\n",
		    ftell(infile)));
  
    while (1) {      
	char buffer[1024]; 

	long last_pos = ftell(infile);
	if (last_pos < 0) {
	    DPRINT(Debug,5,(&Debug,
			    "copy_message_2: ftell(infile) failed!\n"));
	    break;
	}
	
	if (! (buf_len = mail_gets(buffer, sizeof(buffer), infile)))
	    break;
    
	if (have_MMDF &&
	    strcmp(buffer, MSG_SEPARATOR) == 0)
	    continue; /* MSG SEPRATOR is already written */
    
	if(buffer[buf_len - 1] == '\n') {
	    no_ret(buffer);
	    if (first_word(buffer, "From ")) {
		
		if (! real_from(buffer,NULL)) {
		    DPRINT(Debug,15,(&Debug,
				     "copy_message_2: Not From separator: %s\n",
				     buffer));
		    break;
		}

		first_line = FALSE;
		continue;
	    }
	    
	    if (!first_line && first_word_nc(buffer, ">From")) {

		if (! real_from(buffer+1,NULL)) {
		    DPRINT(Debug,15,(&Debug,
				     "copy_message_2: Not escaped >From separator: %s\n",
				     buffer));
		    break;
		}


#if 0
		if (!filter_headers && !remove_header && !remove_envelope) {
		    if (state_printf(dest_file, 
				     FRM("%S%s\n"), 
				     quote_prefix, buffer) == EOF) {
			DPRINT(Debug,1,(&Debug, 
					"copy_message_2 fails\n"));
			goto fail;
		    }
		}
#endif
		continue;	
	    }
	    /* fall thru */
	}
	DPRINT(Debug,5,(&Debug,
			"copy_message_2: Not a mailbox line -- seeking back!\n"));
	DPRINT(Debug,5,(&Debug,
			"copy_message_2- Line was: %s\n",buffer));
	
	if (0 != fseek(infile,last_pos,SEEK_SET)) {
	    DPRINT(Debug,5,(&Debug,
			    "copy_message_2: seek failed!\n"));
	    DPRINT(Debug,1,(&Debug, 
			    "copy_message_2 fails\n"));
	    goto fail;
	}
	break; /* Go out of loop */
    }
  
    R1 = ftell(infile);
    DPRINT(Debug,5,(&Debug,
		    "copy_message_2: [%ld] start header section\n",
		    R1));

    all_headers = file_read_headers(infile,RHL_CHECK_HEADER|RHL_MARK_FOLDING);

    R2 = ftell(infile);
    bytes_seen = R2 - R1;
    DPRINT(Debug,5,(&Debug,
		    "copy_message_2: [%ld] end of headers. Read ~ %d bytes.\n",
		    R2, bytes_seen));

    /* emit the opening attribution string */
    if (make_attribution) {  

	if (forwarding) {
	    if (fwdattribution_s) {

		makeAttString(fwdattribution_s, 0, current_header, 
			      dest_file, all_headers);
	    } else {

		struct string * From_buffer = gen_From_buffer(current_header);

		if (From_buffer) {
		    state_printf(dest_file,
				 CATGETS(elm_msg_cat, ElmSet, ElmForwarded1,
					 "----- Forwarded message from %S -----\n"), 
				 From_buffer);
		    free_string(& From_buffer);
		} else
		    state_printf(dest_file, 
				 CATGETS(elm_msg_cat, ElmSet, 
					 ElmForwarded2,
					 "----- Forwarded message (env-from %s) -----\n"), 
				 current_header->env_from);		
		state_putc('\n',dest_file);
	    }
	} else if (attribution_s) {
	    makeAttString(attribution_s, 0, current_header, dest_file, 
			  all_headers);
	}
    }


    if (!remove_header) {
	
	struct decode_opts  DECODE_OPT = NULL_decode_opt;

	DECODE_OPT.displaying = ison(cm_options,CM_DISPLAYING);
	DECODE_OPT.show_error = ison(cm_options,CM_SHOW_ERROR);

	if (ison(cm_options,CM_QUOTE_L) || ison(cm_options,CM_PREFIX)) {
	    int flags = 0;
	    
	    if (0 != (cm_options & CM_QUOTE_L)) 
		flags |= PR_QUOTE_L;
	    if (0 != (cm_options & CM_PREFIX))
		flags |= PR_REPLY_QUOTE;
	    
	    if (flags)
		decode_data =  state_add_simple_pager_range(dest_file,NULL,
							    flags,0,0);
	}

	if (decode_data) {
	    DPRINT(MimeDebug,15,(&MimeDebug,
				 "copy_message_2: decode_data quote_level=%d range_flags=%d\n",
				 get_pager_range_quote_level(decode_data),
				 get_pager_range_flags(decode_data)));
	    
	}

	DECODE_OPT.range  =  decode_data;
	
	if (remail) {

	    if (!sender) {
		DPRINT(Debug,1,(&Debug,
				"copy_message_2: sender is not set, using %s\n",
				username));
		sender = username;
	    }

	    if (decode_data)
		set_out_state_line_pager_range(dest_file,decode_data);
	    if (state_printf (dest_file, FRM("Sender: %s\n"),
			      sender) == EOF) {
		DPRINT(Debug,1,(&Debug, "copy_message_2 fails\n"));
		goto fail;
	    }
	}

	for (next_hdr = all_headers; 
	     next_hdr; 
	     next_hdr = next_hdr -> next_header) {
	    const char * hdr_name = give_header_name(next_hdr->header_name);


	    if ((remail) && 0 == istrcmp("Sender",hdr_name)) {
		continue;
	    }

	    if ((forwarding||remail) && 0 == istrcmp("Bcc",hdr_name)) {
		continue;
	    }


	    if(0 == istrcmp(hdr_name, "Content-Length")) {
		/* make correct Content-Length later */
		content_length_seen = TRUE;
		continue;
	    }

	    if (0 == istrcmp(hdr_name, "Return-Path")) 
		return_path_seen = TRUE;

	   
	    if (!filter_headers) {
		if (0 == istrcmp(hdr_name,"Status"))
		    continue;   /* we will output a new Status: line later, if desired. */
	    } else { /* filter_headers */
		if (0 == istrcmp(hdr_name, "Received") ||
		    0 == istrcmp(hdr_name, "Status") ||
		    0 == istrcmp(hdr_name, "Return-Path") ||
		    0 == istrcmp("X-UIDL",hdr_name)   || /* Gen. by some POP deamons  */
		    0 == istrcmp("X-UID",hdr_name)    || /* Gen. by some IMAP deamons */
		    0 == istrcmp("X-Status",hdr_name))   /* Gen. by some IMAP deamons */
		    continue;
		if (remail && 0 == istrcmp(hdr_name, "To"))
		    hdr_name = "Orig-To";  /* Works only if NOT decode */
	    }

	    /* These headers are incorrect after decoding ... */
	    if (decode) { 

		if ((current_header->status & MIME_MESSAGE) &&
		    (0 == istrcmp(hdr_name, "MIME-Version") ||
		     0 == istrcmp(hdr_name, "Content-Type") ||
		     0 == istrcmp(hdr_name, "Content-Transfer-Encoding")))
		    continue;
		
		state_write_header(dest_file,
				   & DECODE_OPT,
				   next_hdr,
				   !(current_header -> status & NOHDRENCODING),
				   current_header -> header_charset);

	    } else {
		/* NOT decode */
		
		state_write_raw_header(dest_file,& DECODE_OPT, next_hdr);
	    }
	}
	     
	/* Make artificial Return-Path header */
	if ((remove_envelope || 0 != (env_flags & WE_ADD_RETURN_PATH))
	    && !return_path_seen && !filter_headers) {

	    if (decode_data)
		set_out_state_line_pager_range(dest_file,decode_data);

	    if (state_printf (dest_file, 
			      FRM("Return-Path: <%s>\n"),
			      current_header->env_from) == EOF) {
		DPRINT(Debug,1,(&Debug, "copy_message_2 fails\n"));
		goto fail;
	    }
	}

	if (out_state_seekable(dest_file)) { 
	    if ((!filter_headers && content_length_seen) ||
		(!remove_envelope && (current_header->have_from || decode))) {
		
		if (state_printf (dest_file, 
				  FRM("Content-Length: ")) == EOF) {
		    DPRINT(Debug,1,(&Debug, "copy_message_2 fails\n"));
		    goto fail;
		}
		CL_pos = out_state_ftell(dest_file);
		
		DPRINT(Debug,5,(&Debug,
				"copy_message_2: Content-Length value offset=[%ld]\n",CL_pos));
		
		
		if (state_printf (dest_file, 
				  FRM("%5d\n"),
				  current_header->content_length) == EOF) {
		    DPRINT(Debug,1,(&Debug, "copy_message_2 fails\n"));
		    goto fail;
		}

	    } else {
		DPRINT(Debug,15,(&Debug,
				 "copy_message_2: Need not write Content-Length\n"));
	    }
	} else {
	    DPRINT(Debug,15,(&Debug,
			     "copy_message_2: Can't write Content-Length (output is not seekable)\n"));
	}

	if (update_status) {
	    char buffer[WLEN+10];   /* Enough space for status letters */

	    if (decode_data)
		set_out_state_line_pager_range(dest_file,decode_data);
	    
	    if (state_printf (dest_file, 
			      FRM("Status: ")) == EOF) {
		goto fail;
	    }
	
	    if (status_2_mailbox(current_header,buffer,sizeof buffer) > 0) {

		DPRINT(Debug,15,(&Debug,
				 "copy_message_2: updating status: %s\n",
				 buffer));

		if (state_printf (dest_file, 
				  FRM("%s"),
				  buffer) == EOF) {

		    goto fail;
		}      
	    }

	    /* Must use state_printf so EOLN_is_CRLF conversion occurs! */
	    if (state_printf(dest_file, 
			     FRM("\n")) == EOF) {
		goto fail;
	    }      
	}	
    
	/*
	 * Add empty line between headers and body (that was not copied
	 *  in above)
	 */
	
	/* Must use state_printf so EOLN_is_CRLF conversion occurs! */
	if (state_printf(dest_file, 
			 FRM("\n")) == EOF) {
	    goto fail;
	}
    }

    set_out_state_filter(dest_file,NULL);   /* End HACK */
   
    DPRINT(Debug,5,(&Debug,
		    "copy_message_2: [%ld] Starting reading of body: %d bytes expected\n",
		    ftell(infile),
		    current_header->content_length));

    
    if (out_state_seekable(dest_file)) {
	BODY_pos = out_state_ftell(dest_file);
	DPRINT(Debug,5,(&Debug,
			"copy_message_2: output offset=[%ld]\n",BODY_pos));
    }
     
    if (!copy_body(infile,current_header,dest_file,cm_options)) {
	DPRINT(Debug,1,(&Debug, "copy_message_2 fails (copy_body fails)\n"));
	goto fail;
    }
    
    /* emit the closing attribution */

    if (make_attribution && forwarding) {
	if (fwdattribution_s) {
	    makeAttString(fwdattribution_s, 1, current_header, dest_file, 
			  all_headers);
	} else {
	    struct string * From_buffer = gen_From_buffer(current_header);
	    if (From_buffer) {
		state_printf(dest_file, 
			    CATGETS(elm_msg_cat, ElmSet, ElmForwarded3,
				    "----- End of forwarded message from %S -----\n"),
			    From_buffer);

		free_string(&From_buffer);
	    } else {
		state_printf(dest_file, 
			     CATGETS(elm_msg_cat, ElmSet, 
				     ElmForwarded4,
				     "----- End of forwarded message (env-from %s) -----\n"),
			     current_header->env_from);
	    }
	}
    }

    DPRINT(Debug,5,(&Debug,
		    "copy_message_2: [%ld] Body read.\n",
	      ftell(infile)));

  if (out_state_seekable(dest_file)) {
      FILE * f1 = out_state_FILE(dest_file);
      END_pos = out_state_ftell(dest_file);

      DPRINT(Debug,5,(&Debug,
		      "copy_message_2: output offset=[%ld]\n",END_pos));

      if (CL_pos > 0 && BODY_pos > 0 && END_pos >= BODY_pos) {
	  /* Actually written content length is good if conversions are
	   * not done and there was no errors ... 
	   */
	  if (f1)
	      clearerr(f1);

	  /* Notice that these tests indicates failure */
	  if (0   != out_state_fseek(dest_file,CL_pos) ||
	      out_state_ftell(dest_file) != CL_pos ||
	      EOF == state_printf (dest_file, 
				   FRM("%5d"), (int) (END_pos - BODY_pos)) ||
	      0   != out_state_fseek(dest_file,END_pos)) {
	      DPRINT(Debug,5,(&Debug,
			"copy_message_2: Writing Content-length -- writing or seeking failed.\n"));
	      goto fail;
	  } else {
	      DPRINT(Debug,1,(&Debug,
			"copy_message_2: Content-length fixed: %d bytes.\n",
			(int) (END_pos - BODY_pos)));
	  }      
      }

  }

  if (all_headers)
      delete_headers(&all_headers);


  if (out_state_ferror(dest_file)) {
      DPRINT(Debug,1,(&Debug, 
		      "copy_message_2: out_state_ferror report error\n"));
      return 0;
  }

  if (decode_data)
	free_pager_range(& decode_data);

  return 1;

 fail:

  if (all_headers)
      delete_headers(&all_headers);

  if (decode_data)
      free_pager_range(& decode_data);

  DPRINT(Debug,1,(&Debug, "copy_message_2 fails\n"));
  return 0;

}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
