static char rcsid[] = "@(#)$Id: mailmsg1.c,v 2.29 2022/07/14 14:16:01 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.29 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>  
 *                       (was hurtta+elm@posti.FMI.FI), hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/mailmsg1.c. That code was following copyright:
 *
 *  The Elm Mail System
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 ****************************************************************************
 *  Incorparated Elm 2.5 code from src/sndmsg.c. 
 *  That code was following copyright:
 *
 *  The Elm Mail System
 *
 * This file and all associated files and documentation:
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** Interface to allow mail to be sent to users.  Part of ELM  **/

#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"mail");

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}



static enum copy_msg_mode { cm_no_copy = 0, cm_get_copy,  cm_canceled_mail,
			    cm_mimeforward, cm_CANCEL
} copy_the_msg P_((struct mailing_headers *headers,
		   int options,
		   struct menu_context *page,
		   struct menu_context *prompt_area,
		   struct header_rec *current_header  /* for forwarding */));

static const unsigned char * csUs P_((const char *str));
static const unsigned char * csUs(str)
     const char *str;
{
    return (const unsigned char *)str;
}

static void output_to_tail P_((struct string *newaddress,
			       struct menu_context *page,
			       int to_line, int to_col));
static void output_to_tail(newaddress,page,to_line,to_col)
     struct string *newaddress;
     struct menu_context *page;
     int to_line; 
     int to_col;
{
    int wide = 0 == to_col;    /* mail only */
    int li, co;	

    menu_get_sizes(page,&li, &co);

    if (wide)
	if (string_len(newaddress) > 80) 
	    menu_PutLineX(page,to_line, to_col, 
			  CATGETS(elm_msg_cat, ElmSet, ElmToParen, 
				  "To: (%.*S...)"), 
			  60,newaddress);
	else
	    menu_PutLineX(page,to_line, to_col, 
			  CATGETS(elm_msg_cat, ElmSet, ElmToNoParen, 
				  "To: %S"), 
			  newaddress);

    /* FIXME: Not completely correct ... */
    else if (string_len(newaddress) + to_col +7 < co ) {
	int LI, CO;
	int newpos = co - string_len(newaddress) -7;

	menu_GetXYLocation(page,&LI,&CO);
		    
	if (LI == to_line && CO <= newpos) {
	    menu_CleartoEOLN(page);
	} else {
	    menu_MoveCursor(page,to_line,to_col);
	    menu_CleartoEOLN(page);
	}
	
	menu_PutLineX(page,to_line, newpos, 
		      CATGETS(elm_msg_cat, ElmSet, ElmToNoParen, 
			      "To: %S"), 
		      newaddress);
	   
    } else if (string_len(newaddress) > 50) 
	menu_PutLineX(page,to_line, to_col, 
		      CATGETS(elm_msg_cat, ElmSet, ElmToParen, 
			      "To: (%.*S...)"), 
		      40,newaddress);
    else {
	if (string_len(newaddress) > 30)
	    menu_PutLineX(page,to_line, to_col, 
			  CATGETS(elm_msg_cat, ElmSet, ElmToNoParen, 
				  "To: %S"), 
			  newaddress);
	   else
	       menu_PutLineX(page,to_line, to_col, 
			     CATGETS(elm_msg_cat, ElmSet, ElmToNoParen2, 
				     "          To: %S"), 
			     newaddress);
	menu_CleartoEOLN(page);
    }   
}

static void output_abbreviated_to P_((struct addr_list *address_list,
				      struct menu_context *page,
				      int to_line, int to_col
				      ));

static void output_abbreviated_to (address_list, page, to_line, to_col)
     struct addr_list *address_list;
     struct menu_context *page;
     int to_line;
     int to_col;
{
    struct string *newaddress = NULL;

    if (address_list) {
	
	int idx, count = addr_list_item_count(address_list);
	
	for (idx = 0; idx < count; idx++) {
	    int group = -1;
	    const struct address * address = 
		addr_list_get_item(address_list,idx,&group);
	    
	    const char          * addr     = address_get_ascii_addr(address);
	    const struct string * fullname = address_get_phrase(address);
	    
	    
	    if (newaddress)
		add_ascii_to_string(newaddress,s2us(", "));
	    else
		newaddress = new_string(display_charset);
	    
	    if (fullname && string_len(fullname)) {
		append_string(&newaddress,fullname,1);
	    } else if (addr) {
		add_ascii_to_string(newaddress,csUs(addr));
	    }
	}
    }

    if (!newaddress)
	return;

    output_to_tail(newaddress,page,to_line,to_col);

    free_string(&newaddress);

    return;    
}

void display_to(address,page,prompt_area)
     struct expanded_address address;
     struct menu_context  *page;
     struct menu_context *prompt_area;  /* NULL if mail only mode */
{
    /** Simple routine to display the "To:" line according to the
	current configuration (etc) 			      
    **/
    struct string *addr_string;

    struct menu_context *p;
    int to_line, to_col;  
    int li, co;	

    if (prompt_area) {
	menu_get_sizes(prompt_area,&li, &co);

	to_line = 0;
	to_col  = co -50;
	p = prompt_area;
    } else {   /* Mail only */
	menu_get_sizes(page,&li, &co);
	to_line = 3;
	to_col  = 0;
	p = page;
    }

    if (names_only) {
	output_abbreviated_to(address.addrs,p,to_line,to_col);
	return;
    }

    addr_string = hdr_to_expval(address);
    if (!addr_string)
	return;

    output_to_tail(addr_string,p,to_line,to_col);

    free_string(&addr_string);
}

int get_to(to,mailer_info,aview, page, cancel_view, cancel_selection,
	   prompt_area)
     struct expanded_address *to;
     struct mailer_info  *mailer_info;
     struct AliasView    *aview;
     struct menu_context *page;
     struct MailboxView  *cancel_view;
     int                 *cancel_selection;
     struct menu_context     *prompt_area; 
{

    int li, co;
    int delay_redraw = 0;
    int can_flag  = 0;
    enum user_level_v ul = give_dt_enumerate_as_int(&user_level);

    int L UNUSED_VAROK = 0;
    int res = 0;
    struct string * buffer = NULL;
    struct string           * title = NULL; /* Title on full page mode */
    
    menu_get_sizes(prompt_area,&li, &co);

    if (cancel_selection)
	*cancel_selection = -1;

    /** prompt for the "To:" field, expanding into address if possible.
	This routine returns ZERO if errored, or non-zero if okay **/

    if (to->surface_len == 0) {
	int canceled_count = 0;
	struct expanded_address work;

	int code = 0;
	
	if (cancel_view)
	    canceled_count = get_message_count(cancel_view);
	
	DPRINT(Debug,7,(&Debug,
			"get_to: canceled_count=%d\n",
			canceled_count));
		
	zero_expanded_address(&work);
	
	/* Does nothing */
	copy_expanded_address(&work,to);
	
	title = format_string(CATGETS(elm_msg_cat, ElmSet, 
				      ElmGetToTitle,
				      "To: address selection"));
	
	zero_expanded_address(&work);
	
	/* Does nothing */
	copy_expanded_address(&work,to);
	
	do {
	    int line = 1;
	    int flag = OE_TABACTION;  
	    
	    if (code) {
		DPRINT(Debug,10,(&Debug, "get_to: looping code=%d",code));
		switch (code) {
		case REDRAW_MARK: DPRINT(Debug,10,(&Debug, " REDRAW_MARK")); break;
		}
		if (buffer) {
		    DPRINT(Debug,10,(&Debug, ", buffer=%S",buffer));
		}		    
		DPRINT(Debug,10,(&Debug, "\n"));
	    }
	    
	    if (canceled_count > 1 && cancel_selection) {
		menu_print_format_center(prompt_area,line,
					 CATGETS(elm_msg_cat, ElmSet, 
						 ElmIsCanceledX,
						 "%d canceled mails. Use / to select one of them."),
					 canceled_count); 
		line++;
		flag |= OE_ALT_SOLIDUS;
	    } else if (1 == canceled_count && cancel_selection) {
		menu_print_format_center(prompt_area,line,
					 CATGETS(elm_msg_cat, ElmSet, 
						 ElmIsCanceled1,
						 "1 canceled mail. Use / to select it.")); 
		line++;
		flag |= OE_ALT_SOLIDUS;
	    }
	    
	    /* On redrawing / should not go to cancel view */
	    
	    if (delay_redraw &&
		((buffer && string_len(buffer)) > 0
		 || work.surface_len > 0))
		flag = OE_APPEND_CURRENT;
	    
	    if (ul < user_level_expert) {
		code = prompt_expanded_address(page,&work,mailer_info,&buffer,
					       aview,title,
					       prompt_area,line,0,
					       OE_REDRAW_MARK|flag|
					       OE_SIG_CHAR /* Ctrl-C */, 
					       CATGETS(elm_msg_cat, ElmSet, 
						       ElmSendTheMessageTo,
						       "Send the message to: "));
		
	    } else {
		code = prompt_expanded_address(page,&work,mailer_info,&buffer,
					       aview,title,
					       prompt_area,line,0,
					       OE_REDRAW_MARK|flag|
					       OE_SIG_CHAR /* Ctrl-C */,
					       CATGETS(elm_msg_cat, ElmSet, ElmTo, 
						       "To: "));	  
	    }
	    
	    if (REDRAW_MARK == code) {
		menu_ClearScreen(page);   /* Clear possible redraw mark */
		
		/* Call refresh routines of children */
		menu_redraw_children(page);
		
		if (menu_need_redraw(prompt_area))
		    menu_ClearScreen(prompt_area);   /* but clear redraw mark from prompt_area*/
		
		/* NOTICE: using menu_trigger_redraw(page) on here
		   may cause redraw loop!
		*/
		delay_redraw++;
		
		continue;
	    }	    
	    
	} while (REDRAW_MARK == code);
	
	if (OE_ALT_SOLIDUS == code) { 
	    can_flag = view_canceled_mails(cancel_view,
					   cancel_selection,aview);
	    
	    menu_ClearScreen(page);   
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    if (menu_need_redraw(prompt_area))		    
		menu_ClearScreen(prompt_area);   /* Clear redraw mark from prompt_area*/
	    
	    delay_redraw = 1;
	}
	
	if (0 != code || can_flag) {
	    free_expanded_address(&work);
	    
	    if (-1 == code) {   /* Ctrl-C */
		menu_ClearLine(prompt_area,1);	
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailNotSent, 
				  "Mail not sent."));
	    }
	    
	    if (can_flag > 0) {
		DPRINT(Debug,4,(&Debug, "get_to: Canceled mail selected\n"));
		res =  1;               /*  Canceled mail selected */
		goto out;
	    }
	    
	    res =  0;
	    goto out;
	}
	
	if (0 == work.surface_len) {
	    free_expanded_address(&work);
	    free_expanded_address(to);
	    
	    menu_ClearLine(prompt_area,1);	
	    
	    res =  0;
	    goto out;
	}
	
	DPRINT(Debug,10,(&Debug,
			 "get_to: surface len %d",
			 work.surface_len));
	if (buffer) {
	    DPRINT(Debug,10,(&Debug, ", buffer=%S",buffer));
	}		    
	DPRINT(Debug,10,(&Debug,"\n"));
	
	copy_expanded_address(to,&work);
	free_expanded_address(&work);
	    
	
    } else {
	DPRINT(Debug,10,(&Debug,
			 "get_to: Address already given, surface len %d\n",
			 to->surface_len
			 ));
    }
	
    build_address_l(to,mailer_info,aview);
    
    if (! to->addrs || 
	(L = addr_list_item_count(to->addrs)) == 0) {	/* bad address!  Removed!! */
	menu_ClearLine(prompt_area,1);	
    
	res =  0;
	goto out;
    }

    DPRINT(Debug,10,(&Debug,
		     "get_to: OK, %d address(es)\n",
		     L));
    
    res = 1; /* everything is okay... */
    
 out:
    if (title) 
	free_string(& title);    
    
    if (buffer) {
	DPRINT(Debug,12,(&Debug,
			 "get_to: free'ing buffer=%S\n",buffer));
	
	free_string(&buffer);
    }
	
    if (delay_redraw) {
	DPRINT(Debug,10,(&Debug,
			 "get_to: delay_redraw redrawing\n"));
	
	menu_trigger_redraw(page);
    }
    
    DPRINT(Debug,10,(&Debug,
		     "get_to=%d\n",res));
    
    return res;
}

static int get_copies P_((struct expanded_address * cc,
			  int copy_message,    
			  struct mailer_info *mailer_info,
			  struct AliasView *aview,
			  struct menu_context *page,
			  struct menu_context *prompt_area));
static int get_subject P_((struct string **field,
			   struct menu_context *page,
			   struct menu_context *prompt_area));

int send_msg_middle2 P_((struct mailing_headers *headers,
			 int index,
			 int options,
			 struct mailer_info  *mailer_info,
			 struct MailboxView *mailbox,
			 struct AliasView *aview,
			 struct menu_context *page,
			 struct menu_context *prompt_area,
			 struct text_block *body,
			 int *exit_status,
			 int pgp_status,
			 struct header_rec *current_header
			 /* for forwarding */));

int send_msg_middle2(headers,index,options,mailer_info,
		     mailbox,aview,page,prompt_area,body,
		     exit_status,pgp_status,current_header)
     struct mailing_headers *headers;
     int index;
     int options; 
     struct mailer_info  *mailer_info;
     struct MailboxView *mailbox;
     struct AliasView *aview;
     struct menu_context *page;
     struct menu_context *prompt_area;
     struct text_block *body;
     int *exit_status;               /* Only used on send only mode */
     int pgp_status;
     struct header_rec *current_header  /* for forwarding */;
{
    enum copy_msg_mode copy_msg = cm_no_copy; 
    char *p;

    struct MailboxView *cancel_view = NULL;
    int cancel_selection = -1;
    
    int forwarding = 0 != (options & MAIL_FORWARDING);
    int replying   = 0 != (options & MAIL_REPLYING);
    int ask_send   = 0 != (options & MAIL_ASK_SEND);
    int r = 0;
    
    if ((p = getenv("REPLYTO")) != NULL) {
	struct string * X = new_string2(system_charset,
					s2us(p));

	update_expanded_from_edit_buffer(&(headers->reply_to),X,
					 mailer_info,aview);

	free_string(&X);
    }

    /* auto bcc line */
    if ((p = getenv("BCC")) != NULL) {
	struct string * X = new_string2(system_charset,
					s2us(p));

	update_expanded_from_edit_buffer(&(headers->bcc),X,mailer_info,
					 aview);

	free_string(&X);
    }

    if (prompt_area) {   /* Not a batch mode */
	
	DPRINT(Debug,7,(&Debug,
			"prompt_area=%p  prompt questions -- not in  batch mode\n",
			prompt_area));

	/* copy msg into edit buffer? */
	copy_msg = copy_the_msg(headers,
				options, page, prompt_area,
				current_header);

	switch (copy_msg) {
	case cm_CANCEL:      
	    goto free_it;

	case cm_get_copy:    
	    options |= MAIL_COPY_MSG;  
	    break;
	case cm_mimeforward: 
	    options |= MAIL_MIME_ATTCH; 
	    options |= MAIL_COPY_MSG;
	    break;   
	default:
	    
	    if (!forwarding &&
		!replying)
		cancel_view = give_canceled_mail();

	    DPRINT(Debug,7,(&Debug,"cancel_view=%p%s\n",
			    cancel_view,
			    cancel_view ? "" : " (NULL)"));

	    break;
	}
 
	/* get the To: address and expand --
	 */
	if (! get_to(&(headers->to),mailer_info,aview, page,cancel_view,
		     &cancel_selection,prompt_area)) {
	    goto free_it;
	}
     
	if (cancel_selection >= 0) {
	    struct header_rec * H = give_header(cancel_view,cancel_selection);

	    if (H) {
		options |= MAIL_COPY_SELECTION | MAIL_DELETE_CANCEL;

		/* Copy some addresses from header struct 
		 *
		 * Actually these addresses will be overwritten when 
		 * canceled mail is readed and parsed
		 */
		
		if (! headers->subject && H->subject)
		    headers->subject = dup_string(H->subject);
		
		expanded_address_from_list (&(headers->from), H->from);
		expanded_address_from_list (&(headers->to), H->to);
		expanded_address_from_list (&(headers->cc), H->cc);

		/* NOTE: add_to_mailing_header() do not currently handle
		         In-Reply-To: or References:  do they must
		         updated on here

                   See header_types[] on lib/addr/headers.c
		*/


		if (! headers->in_reply_to && H->in_reply_to)
		    headers->in_reply_to = dup_references(H->in_reply_to);
		
		if (! headers->references && H->references)
		    headers->references = dup_references(H->references);		
		
	    }
	}
	
    } else {   /* batch mode */

	/* expand the To: address */
	if (headers->to.surface_len)
	    build_address_l(&(headers->to),mailer_info,aview);

    }


    /* expand the Cc: address */
    if (headers->cc.surface_len)
	build_address_l(&(headers->cc),mailer_info,aview);

    /* expand the Reply-To: address */
    if (headers->reply_to.surface_len)
	build_address_l(&(headers->reply_to),mailer_info,aview);

    /* expand the bcc address */
    if (headers->bcc.surface_len)
	build_address_l(&(headers->bcc),mailer_info,aview);

    DPRINT(Debug,4,(&Debug, "send_msg_middle2 -- options (%d): %s%s%s%s%s%s%s\n",
		    options,
		    options & MAIL_COPY_MSG   ? " MAIL_COPY_MSG"  : "",
		    options & MAIL_EDIT_MSG   ? " MAIL_EDIT_MSG"  : "",
		    options & MAIL_REPLYING   ? " MAIL_REPLYING"  : "",
		    options & MAIL_FORWARDING ? " MAIL_FORWARDING" : "",
		    options & MAIL_MIME_ATTCH ? " MAIL_MIME_ATTCH" : "",
		    options & MAIL_COPY_SELECTION ? " MAIL_COPY_SELECTION" : "",
		    options & MAIL_DELETE_CANCEL ? " MAIL_DELETE_CANCEL" : ""));

    /** if we're batchmailing, let's send it and GET OUTTA HERE! 
     **
     ** If MAIL_COPY_SELECTION is set data for headers will be readed
     **                    from canceled mail, do not prompt anything
     **/

    dump_expanded_address(4,"send_msg_middle2 -- from ",headers->from);
    dump_expanded_address(4,"send_msg_middle2 -- to   ",headers->to);
    dump_expanded_address(4,"send_msg_middle2 -- cc   ",headers->cc);
    
    if (!prompt_area || 
	(options & MAIL_COPY_SELECTION)) {

	r = mail(index, options, headers,mailer_info,
		 mailbox, aview, page,
		 cancel_view,cancel_selection,body,
		 exit_status,0);       

	/* mail does menu_trigger_redraw(), not need to be examine */

    } else {
	int redraw1 = 0;
       
	struct menu_context *use_prompt_area = 
	    OPMODE_IS_SENDMODE(opmode) ? NULL : prompt_area;

    again1:
	display_to(headers->to, page, use_prompt_area);

	/* URL code may want verify sending ... */
	if (ask_send) {   
	    struct string * addr_string = hdr_to_expval(headers->to);
	    int X1;

	    struct menu_context *p;
	    int prompt_line;

	    if (!use_prompt_area) { /* Mail only */
		p           = page;
		prompt_line =  4;
	    } else {
		p           = use_prompt_area;
		prompt_line = 1;
	    }

	    if (!addr_string) {
		goto free_it;
	    }

	    X1 = prompt_letter(prompt_line,"",*def_ans_no,
			       PROMPT_yesno|PROMPT_cancel|
			       PROMPT_redraw_mark|PROMPT_ctrlL,
			       p,			    
			       CATGETS(elm_msg_cat, ElmSet, ElmAskSendMailTo,
				       "Send mail to %.50S%s ? (%c/%c) "), 
			       addr_string,
			       string_len(addr_string) > 50 ? "..." : "",
			       *def_ans_yes, *def_ans_no);

	    free_string(&addr_string);

	    if (TERMCH_interrupt_char == X1 ||
		EOF == X1) {
		goto free_it;
	    }

	    if (X1 == ('L'&31) || X1 == REDRAW_MARK) {
		menu_ClearScreen(page);   /* Clear possible redraw mark */

		/* Call refresh routines of children */
		menu_redraw_children(page);
	     
		if (menu_need_redraw(prompt_area))
		    menu_ClearScreen(prompt_area);   /* but clear redraw mark from prompt_area*/
	     
	     
		/* NOTICE: using menu_trigger_redraw(page) on here
		   may cause redraw loop!
		*/
		redraw1 = 1;
		goto again1;
	    }


	    if (X1 != *def_ans_yes)
		goto free_it;
	}

	if (redraw1)
	    menu_trigger_redraw(page); 


	/* get the Subject: field */
	if (get_subject(&(headers->subject), page, 
			use_prompt_area) == 0) {	    

	    goto free_it;
	}

	if (prompt_for_cc) {
	    if (get_copies(&(headers->cc), copy_msg, mailer_info, 
			   aview, page, 
			   use_prompt_area) == 0) {	       
		goto free_it;
	    }
	}

	menu_MoveCursor(prompt_area,3,0);	/* so you know you've hit <return> ! */
	FlushBuffer();

	/** generate the In-Reply-To: header...
	    and the References: header
	*/
	
	if ( 0 != (options & MAIL_REPLYING)) {
	    struct header_rec *current_header =  give_header(mailbox,index);
	    if (current_header) {
		generate_reply_to(current_header,headers);
		generate_references(current_header,headers);
	    }
	}

	/* and mail that puppy outta here! */
	
	DPRINT(Debug,3,(&Debug, 
			"\nsend_msg_middle2() ready to mail...\n"));
	dump_expanded_address(3,"to",headers->to);
	if (headers->subject) {
	    DPRINT(Debug,3,
		   (&Debug,"subject=\"%S\"\n",
		    headers->subject));
	}
	dump_expanded_address(5,"cc",headers->cc);
	dump_expanded_address(5,"bcc",headers->bcc);

	r = mail(index,options, headers,
		 mailer_info, 
		 mailbox, aview, page,
		 cancel_view,cancel_selection,body,
		 exit_status,
		 pgp_status);

	/* mail does menu_trigger_redraw(), not need to be examine */

    }

 free_it:    

    if (cancel_view)
	sync_canceled_mails(cancel_view);

    if (prompt_area)
	menu_trigger_redraw(prompt_area);

    return r;
}
    
int send_msg_middle(index, given_to, given_cc, given_subject, options, 
		    mailer_info, 
		    mailbox,aview, page, prompt_area,body,exit_status,
		    pgp_status,
		    current_header)
     int index;
     struct expanded_address *given_to, *given_cc;
     const char *given_subject;
     int   options;
     struct mailer_info  *mailer_info;
     struct MailboxView *mailbox;
     struct AliasView *aview;
     struct menu_context *page;
     struct menu_context *prompt_area;   /* NULL in batch mode */ 
     struct text_block *body;
     int *exit_status;                   /* Only used on send only mode */
     int pgp_status;
     struct header_rec *current_header  /* for forwarding */;
{
    struct mailing_headers headers;
    int r = 0;

    zero_mailing_headers(&headers);

    if (!mailer_info)
	return 0;

    if (given_subject)
	headers.subject = new_string2(display_charset,cs2us(given_subject));
    if (given_to)
	copy_expanded_address(&headers.to,given_to);
    if (given_cc)
	copy_expanded_address(&headers.cc,given_cc);

    r = send_msg_middle2(&headers,index,options,mailer_info,
			 mailbox, aview,page,prompt_area,body,
			 exit_status,pgp_status,
			 current_header);
    
    free_mailing_headers(&headers);
    
    return r;
}

int send_msg_l(index,
	       given_to, given_cc, given_subject, options, 
	       mailbox, aview, page, prompt_area,
	       current_header
	       )     
     int index;
     struct addr_list *given_to, *given_cc;
     char *given_subject;
     int   options;
     struct MailboxView *mailbox;
     struct AliasView *aview;
     struct menu_context *page;
     struct menu_context *prompt_area;
     struct header_rec *current_header  /* for forwarding */;
{
    int r;

    /* If redraw is needed use
               menu_trigger_redraw(page)
    */
    struct expanded_address A, B;
    struct mailer_info *mailer_info   = get_mailer_info();

    DPRINT(Debug,12,(&Debug,"send_msg_l: index=%d mailbox=%p\n",
		     index,mailbox));

    if (!mailer_info)
	return 0;

    zero_expanded_address(&A);
    zero_expanded_address(&B);

    addr_to_expanded(&A,given_to,mailer_info,aview);
    addr_to_expanded(&B,given_cc,mailer_info,aview);

    r = send_msg_middle(index,&A,&B,
			given_subject,options,
			mailer_info, 
			mailbox, aview, page,
			prompt_area,NULL,NULL,0,
			current_header);

    /* send_msg_middle does menu_trigger_redraw(page),
       do not need examine */

    free_expanded_address(&A);
    free_expanded_address(&B);


    free_mailer_info(&mailer_info);
   
    return r;
}

int send_msg_argv(argv, given_subject, options,  aview,
		  page, prompt_area,exit_status,body)
     char *argv[];
     const char *given_subject;
     int   options;
     struct AliasView *aview;
     struct menu_context *page;
     struct menu_context *prompt_area;   /* NULL in patch mode */
     int *exit_status;
     struct text_block  *body;

{
    struct expanded_address A, B;
    struct mailer_info *mailer_info   = get_mailer_info();
    
    int ret = 0;

    if (!mailer_info)
	return 0;

    zero_expanded_address(&A);
    zero_expanded_address(&B);

    if (argv)
	argv_to_expanded(&A,argv,mailer_info,aview);

    ret = send_msg_middle(-1,&A,&B,
			  given_subject,options,mailer_info,
			  NULL, aview, page, prompt_area,
			  body,exit_status,0,
			  NULL);

    /* send_msg_middle uses menu_trigger_redraw(),
        need not examine 
    */

    free_expanded_address(&A);
    free_expanded_address(&B);

    free_mailer_info(&mailer_info);

    return ret;
}


int send_msg_url(url,given_subject,options,aview,page,prompt_area, 
		 exit_status,body)
     const struct url *url;
     const char          * given_subject;
     int                   options; 
     struct AliasView    * aview;
     struct menu_context * page;
     struct menu_context * prompt_area;  /* NULL if on batch mode ? */
     int                 * exit_status;
     struct text_block   * body;
{
   struct mailer_info *mailer_info   = get_mailer_info();
   struct mailing_headers headers;
   int r = 0;
   struct text_block *body1 = NULL;
   int errors = 0;

   zero_mailing_headers(&headers);
   
   if (!mailer_info) {
       DPRINT(Debug,5,(&Debug, 
		       "send_msg_url: No mailer_info\n"));
       return 0;
   }
   
   body1 = give_text_body_from_url(url,mime_body_keywords,&errors);

   if (errors)
       goto fail;

   if (body1) {
       if (body) {
	   const char * included_filename = 
	       get_block_filename(body);

	   lib_error(CATGETS(elm_msg_cat, ElmSet, 
			     ElmUrlBodyandInclude,
			     "Both -i %s and body on -O given."),
		     included_filename ? included_filename : "<no filename>");
	   goto fail;
       }

       body = body1;

   } else if (!prompt_area && !body) {  
       /* main() was not readed standard input on url mode */
       body = block_from_stdin();
       if (!body)
	   goto fail;
   }
   
   r = set_mailing_headers_from_url(&headers,url,mailer_info);
   if (!r) {
       DPRINT(Debug,5,(&Debug, 
		       "send_msg_url: set_mailing_headers_from_url failed!\n"));
       goto fail;
   }
   if (given_subject && !headers.subject)
       headers.subject = new_string2(display_charset,cs2us(given_subject));
   else if (given_subject && headers.subject) {
       lib_error(CATGETS(elm_msg_cat, ElmSet, 
			 ElmUrlSubjectandUrl,
			 "Both -s %s and subject on -O given."),
		 given_subject);
       goto fail;
   }
       

   r = send_msg_middle2(&headers,-1,options,mailer_info,NULL,aview,
			page,prompt_area,body,exit_status,0,
			NULL);

 fail:
   if (body1)
       free_block(&body1);

   free_mailing_headers(&headers);
   free_mailer_info(&mailer_info);

   DPRINT(Debug,5,(&Debug, 
		   "send_msg_url=%d\n",r));

   return r;
}


static int get_subject(subject_field, page, prompt_area)
     struct string **subject_field;
     struct menu_context *page;
     struct menu_context *prompt_area;  /* NULL if mail only mode */
{
    int code;
    charset_t do_utf7;
    int prompt_line;
    int delay_redraw = 0;
#if UNUSED
    struct menu_context *p;
#endif
    enum user_level_v ul = 
	give_dt_enumerate_as_int(&user_level);

redraw:
    if (!prompt_area) { /* Mail only */
#if UNUSED
	p           = page;
#endif
	prompt_line =  4;
    } else {
	/* FIXME --optionally_enter*  should use prompt_area */
#if UNUSED
	p           =  prompt_area;
#endif
	prompt_line = menu_GetAbsLine(prompt_area,1);
    }

    if (ul == user_level_beginner) {
	code = optionally_enter2(page,
				 subject_field, prompt_line, 0,
				 OE_APPEND_CURRENT|OE_REDRAW_MARK
				 /* Allow user paste mime encoded
				    words to buffer */
				 |OE_ALLOW_MIMEENC
				 |OE_SIG_CHAR /* Ctrl-C */,
				 CATGETS(elm_msg_cat, ElmSet, 
					 ElmSubjectOfMessage,
					 "Subject of message: "));
    } else
	code = optionally_enter2(page,
				 subject_field, prompt_line, 0,
				 OE_APPEND_CURRENT|OE_REDRAW_MARK
				 /* Allow user paste mime encoded
				    words to buffer */
				 |OE_ALLOW_MIMEENC
				 |OE_SIG_CHAR /* Ctrl-C */,
				 CATGETS(elm_msg_cat, ElmSet, ElmSubject, 
					 "Subject: "));
    
    if (REDRAW_MARK == code) {
	/* NOTICE: using menu_trigger_redraw(page) on here
	   may cause redraw loop!
	*/

	menu_ClearScreen(page);   /* Clear possible redraw mark */
	
	/* Call refresh routines of children */
	menu_redraw_children(page);
	
	delay_redraw++;
	if (prompt_area && menu_need_redraw(prompt_area))
	    menu_ClearScreen(prompt_area); /* but Clear redraw mark from prompt_area */
	goto redraw;
    }

    if(code==-1){
    cancel:
	/** User hit  Ctrl-C key! **/
	MoveCursor(prompt_line,0); 	
	CleartoEOLN();
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailNotSent, 
			  "Mail not sent."));

	if (delay_redraw)
	    menu_trigger_redraw(page); 
	return(0);
    }

    if (!*subject_field ||
	string_len(*subject_field) == 0) {	/* zero length subject?? */

	int answer;

    redraw2:
	if (prompt_area) 
	    answer = prompt_letter(1,"",*def_ans_no,
				   PROMPT_yesno|PROMPT_redraw_mark|
				   PROMPT_ctrlL|PROMPT_cancel,
				   prompt_area,
				   CATGETS(elm_msg_cat, ElmSet, 
					   ElmNoSubjectContinue,
					   "No subject - Continue with message? (%c/%c) "),
				   *def_ans_yes, *def_ans_no);
	else
	    answer = prompt_letter(4,"",*def_ans_no,
				   PROMPT_yesno|PROMPT_redraw_mark|
				   PROMPT_ctrlL|PROMPT_cancel,
				   page,
				   CATGETS(elm_msg_cat, ElmSet, 
					   ElmNoSubjectContinue,
					   "No subject - Continue with message? (%c/%c) "),
				   *def_ans_yes, *def_ans_no);
	    

		
	if (('L'&31)    == answer ||
	    REDRAW_MARK == answer) {
	    menu_ClearScreen(page);   /* Reset possible redraw flag */

	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    if (prompt_area && menu_need_redraw(prompt_area))
		menu_ClearScreen(prompt_area);
	    
	    delay_redraw++;   /* Can't trigger redraw yet... */
	    goto redraw2;
	}
	
	if (TERMCH_interrupt_char == answer ||
	    EOF == answer) 
	    goto cancel;

	if (answer != *def_ans_yes) {	/* user says no! */
	    if (sleepmsg > 0)
		sleep((sleepmsg + 1) / 2);
	    ClearLine(prompt_line);
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailNotSend, 
			      "Mail not sent."));
	    
	    if (delay_redraw)
		menu_trigger_redraw(page); 
	    return(0);

	} else {
	    PutLineX(prompt_line,0,
		     CATGETS(elm_msg_cat, ElmSet, 
			     ElmSubjectNone, 
			     "Subject: <none>"));
	    CleartoEOLN();
	    if (!*subject_field) 
		*subject_field = new_string(display_charset);
	}
    }
    
    do_utf7 = want_convert_to_utf7(*subject_field);
    if (do_utf7) {
	
	struct string * XX = convert_string(do_utf7,*subject_field,1);
	free_string(subject_field);

	*subject_field = XX;
    }

    if (delay_redraw)
	menu_trigger_redraw(page); 
    return(1);		/** everything is cruising along okay **/
}

static int get_copies P_((struct expanded_address *cc,
			  int copy_message,    
			  struct mailer_info *mailer_info,
			  struct AliasView *aview,
			  struct menu_context *page,
			  struct menu_context *prompt_area));

static int get_copies(cc,copy_message,mailer_info,aview, page,
		      prompt_area)
     struct expanded_address * cc; 
     int copy_message;
     struct mailer_info *mailer_info;
     struct AliasView *aview;
     struct menu_context *page;
     struct menu_context *prompt_area;   /* NULL if mail only */
{
    /* Get the list of people that should be cc'd, returning ZERO if
     * any problems arise.  
     *
     * If copy-message, that means that we're going to have to invoke
     * a screen editor, so we'll need to delay after displaying the
     * possibly rewritten Cc: line...
     */

    struct string *buffer = NULL;
    int delay_redraw = 0;
    struct menu_context *p = NULL;
    struct string       * title = NULL; /* Title on full page mode */
    int p_line = 5;    
    int flag = 0;
    int res = 0;
    struct expanded_address work;
    int code = 0;
    
    title = format_string(CATGETS(elm_msg_cat, ElmSet, 
				  ElmGetCCTitle,
				  "Cc: address selection"));
    
    flag = OE_TABACTION;
    
    zero_expanded_address(&work);
    
    /* Does nothing */
    copy_expanded_address(&work,cc);
    
    do {
	if (code) {
	    DPRINT(Debug,10,(&Debug, "get_copies: looping code=%d",code));
	    switch (code) {
	    case REDRAW_MARK: DPRINT(Debug,10,(&Debug, " REDRAW_MARK")); break;
	    }
	    if (buffer) {
		DPRINT(Debug,10,(&Debug, ", buffer=%S",buffer));
	    }		    
	    DPRINT(Debug,10,(&Debug, "\n"));
	}
	
	if (!prompt_area) { /* Mail only */
	    p           = page;
	    p_line      = 5;
	} else {
	    p           =  prompt_area;
	    p_line      = 2;
	}
	
	if (delay_redraw &&
	    ((buffer && string_len(buffer)) > 0
	     || work.surface_len > 0))
	    flag = OE_APPEND_CURRENT;
	
	code = prompt_expanded_address(page,&work,mailer_info,&buffer,
				       aview,title,
				       p,p_line,0,
				       OE_REDRAW_MARK|flag|
				       OE_SIG_CHAR /* Ctrl-C */,
				       CATGETS(elm_msg_cat, ElmSet, ElmCopiesTo, 
					       "Copies to: "));
	
	if (REDRAW_MARK == code) {
	    /* NOTICE: using menu_trigger_redraw(page) on here
	       may cause redraw loop!
	    */
	    
	    menu_ClearScreen(page);   /* Clear possible redraw mark */
	    
	    /* Call refresh routines of children */
	    menu_redraw_children(page);
	    
	    delay_redraw++;
	    if (prompt_area && menu_need_redraw(prompt_area))
		menu_ClearScreen(prompt_area); /* Clear redraw mark from prompt_area */
	    continue;
	}
	
    } while (REDRAW_MARK == code);
    
    if (code == -1) {      /* Ctrl-C */
	menu_ClearLine(p,p_line-1);
	menu_ClearLine(p,p_line);
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailNotSend, 
			  "Mail not sent."));
	
	res =  0;
	goto out;
    }
    
    DPRINT(Debug,10,(&Debug,
		     "get_copies: surface len %d",
		     work.surface_len));
    if (buffer) {
	DPRINT(Debug,10,(&Debug, ", buffer=%S",buffer));
    }		    
    DPRINT(Debug,10,(&Debug,"\n"));
    
    copy_expanded_address(cc,&work);
    free_expanded_address(&work);
		         
    if (build_address_l(cc,mailer_info, aview)) {

	/* Display expanded aliases, if not builtin editor is used */
	
	struct string * B = hdr_to_expval(*cc);
	if (B) {
	    unsigned int editor_keyword = 0, editor_mask = 0;
	    const char * editor_val = give_dt_estr_as_str(&editor_e,"editor",
							  &editor_keyword,
							  &editor_mask);
		
	    menu_PutLineX(p,p_line, 0,
			  CATGETS(elm_msg_cat,
				  ElmSet, ElmCopiesToExp, 
				  "Copies to: %S"), B);
	    
	    if (!editor_val) 
		error_wait();  /* Pause if there was error */    
	    else if (editor_kw_builtin != editor_keyword
		     || copy_message)
		sleep_message();

	    free_string(&B);
	}
    }

    res = 1; /* everything looks okay! */

 out:
    if (title)
	free_string(&title);
    
    if (buffer) {
	DPRINT(Debug,12,(&Debug,
			 "get_copies: free'ing buffer=%S\n",buffer));
	
	free_string(&buffer);
    }

    
    if (delay_redraw) {
	DPRINT(Debug,10,(&Debug,
			 "get_copies: delay_redraw redrawing\n"));
	
	menu_trigger_redraw(page);
    }
    
    DPRINT(Debug,10,(&Debug,
		     "get_copies=%d\n",res));
    
    return res;		
}
	
static enum copy_msg_mode copy_the_msg(headers, options, page, prompt_area,
				       current_header)
     struct mailing_headers * headers;
     int                      options;
     struct menu_context    * page;
     struct menu_context    * prompt_area;
     struct header_rec      * current_header  /* for forwarding */;
{
    int forwarding = 0 != (options & MAIL_FORWARDING);
    int replying   = 0 != (options & MAIL_REPLYING);

    /** Returns True iff the user wants to copy the message being
	replied to into the edit buffer before invoking the editor! 
    **/
    enum copy_msg_mode answer = cm_no_copy;
    int redraw = 0;
        
 again:
    
    if (forwarding) {
	enum mimeforward_v mf = give_dt_enumerate_as_int(&mimeforward);

	switch (mf) {
	case browser_wildcard_no:  answer = cm_get_copy;     break;
	case browser_wildcard_yes: answer = cm_mimeforward;  break;
	case browser_wildcard_file_browser:
	    if (current_header &&
		0 != (current_header->status & MIME_MESSAGE)) {
		charset_t X = display_charset;
		int v UNUSED_VAROK;

		/* Is this correct? */
		charset_t disp_vector[2];
		disp_vector[0] = display_charset;  
		disp_vector[1] = NULL;
		
		if (current_header->mime_rec.disposition  != DISP_INLINE) {
		    DPRINT(Debug,7,(&Debug,"copy_the_msg: disposition not inline on forwarded message\n"));
		    answer = cm_mimeforward;
		} else if (get_major_type_code(current_header->mime_rec.TYPE)
			 != MIME_TYPE_TEXT ||
			 0 != istrcmp(get_subtype_name(current_header->
						       mime_rec.TYPE),
				      "plain")) {
		    DPRINT(Debug,7,(&Debug,"copy_the_msg: type not text/plain on forwarded message\n"));
		    answer = cm_mimeforward;
		} else if ((v = mime_get_charset(&X,
						 current_header->
						 mime_rec.TYPE_opts,
						 disp_vector,NULL)) > 0) {
		    DPRINT(Debug,7,(&Debug,"copy_the_msg: charset ok (mime_get_charset=%d) on forwarded message\n",
				    v));
		    answer = cm_get_copy;
		} else if (-3 == v) {
			DPRINT(Debug,7,(&Debug,
					"copy_the_msg: charset convertable (mime_get_charset=%d) on forwarded message\n",
					v));
			answer = cm_get_copy;
		} else {
		    DPRINT(Debug,7,(&Debug,"copy_the_msg: charset problem (mime_get_charset=%d) on forwarded message\n",
				    v));

		    answer = cm_mimeforward;
		}
	    } else
		answer = cm_get_copy;		
	    break;
	case NUM_browser_wildcards: break;
	}
	    	
	if (askmimeforward) { 
	    int def = cm_mimeforward == answer ? 'p' : 't';
	    int ret;

	    /* NOTICE: prompt_letter may return EOF */
	    
	    ret = prompt_letter(0,"tp",def,
				PROMPT_redraw_mark|PROMPT_cancel|
				PROMPT_ctrlL,
				prompt_area,			    
				CATGETS(elm_msg_cat, ElmSet, ElmMimeFWMessage,
					"Forward t)ext only; or whole message as separate p)art ? (t/p) "));
				
	    
	    if (ret == ('L'&31) || REDRAW_MARK == ret) {
		menu_ClearScreen(page);   /* Clear possible redraw mark */

		/* Call refresh routines of children */
		menu_redraw_children(page);
		
		if (menu_need_redraw(prompt_area))
		    menu_ClearScreen(prompt_area);   /* Clear redraw mark from prompt_area*/

		/* NOTICE: using menu_trigger_redraw(page) on here
		   may cause redraw loop!
		*/
		redraw = 1;
		goto again;
	    }

	    if (EOF == ret)
		goto out;
	    
	    if (TERMCH_interrupt_char == ret) {
		answer = cm_CANCEL;
		goto out;
	    }

	    switch (ret) {
	    case 't' :
		answer = cm_get_copy;
		break;
	    case 'p':
		answer = cm_mimeforward;
		break;
	    }

	}
      


    } else if (replying) {  

	answer = reply_copy ? cm_get_copy : cm_no_copy;
	/* predefined 'to' line! */
	
	if (ask_reply_copy) {
	    
	    int ret;
	    enum user_level_v ul = 
		give_dt_enumerate_as_int(&user_level);

	    /* NOTICE: prompt_letter may return EOF */

	    if (ul < user_level_expert)	      
		ret = prompt_letter(0,"",
				    reply_copy ? *def_ans_yes : *def_ans_no, 
				    PROMPT_yesno|PROMPT_redraw_mark
				    |PROMPT_cancel|PROMPT_ctrlL,
				    prompt_area,
				    CATGETS(elm_msg_cat, ElmSet, 
					    ElmCopyMessageIntoReplyYN,
					    "Copy message into reply? (%c/%c) "),
				    *def_ans_yes, *def_ans_no);
	    else 	      
		ret = prompt_letter(0,"",
				    reply_copy ? *def_ans_yes : *def_ans_no, 
				    PROMPT_yesno|PROMPT_redraw_mark
				    |PROMPT_cancel|PROMPT_ctrlL,
				    prompt_area,
				    CATGETS(elm_msg_cat, ElmSet, 
					    ElmCopyMessageYN,
					    "Copy message? (%c/%c) "), 
				    *def_ans_yes, *def_ans_no);
	    
	    if (ret == ('L'&31) || REDRAW_MARK == ret) {
		menu_ClearScreen(page);   /* Clear possible redraw mark */

		/* Call refresh routines of children */
		menu_redraw_children(page);

		if (menu_need_redraw(prompt_area))
		    menu_ClearScreen(prompt_area);   /* Clear redraw mark from prompt_area*/

		/* NOTICE: using menu_trigger_redraw(page) on here
		   may cause redraw loop!
		*/
		redraw = 1;
		goto again;
	    }

	    if (EOF == ret)
		goto out;

	    if (TERMCH_interrupt_char == ret) {
		answer = cm_CANCEL;
		goto out;
	    }

	    if (ret == *def_ans_yes)
		answer = cm_get_copy;
	    else if (ret == *def_ans_no)
		answer = cm_no_copy;	   
	}

    }

 out:
    if (redraw)
	menu_trigger_redraw(page); 
    
    DPRINT(Debug,7,(&Debug,"copy_the_msg=%d",answer));
    switch (answer) {
    case cm_no_copy:       DPRINT(Debug,7,(&Debug," (cm_no_copy)"));       break;
    case cm_get_copy:      DPRINT(Debug,7,(&Debug," (cm_get_copy)"));       break;
    case cm_canceled_mail: DPRINT(Debug,7,(&Debug," (cm_canceled_mail)")); break;
    case cm_mimeforward:   DPRINT(Debug,7,(&Debug," (cm_mimeforward)"));   break;
    case cm_CANCEL:        DPRINT(Debug,7,(&Debug," (cm_CANCEL)"));        break;
    }
    DPRINT(Debug,7,(&Debug,"\n"));
    return(answer);
}

void a_sendmsg(edit_message,  mailbox, aview, 
	       page, LOC)
     int   edit_message;
     struct MailboxView *mailbox;
     struct AliasView *aview;
     struct menu_context *page;
     struct screen_parts *LOC;
{
    /** Prompt for fields and then call mail() to send the specified
	message.  If 'edit_message' is true then by defualt go to
	editor. 
	
	If redraw is needed use menu_trigger_redraw(page)
    **/

    register int tagged = 0, i;
   
    struct mailing_headers headers;
    struct mailer_info *mailer_info   = get_mailer_info();

    zero_mailing_headers(&headers);

    tagged = aliases_to_expanded(&headers.to, aview);
    
    DPRINT(Debug,4, (&Debug, "%d aliases tagged for mailing (a_sndmsg)\n",
		     tagged));

    /******* And now the real stuff! *******/
    
    /* build the To: address and expand */
    if (build_address_l(&headers.to,mailer_info, aview) == 0) {
	free_mailer_info(&mailer_info);
	return;
    }
    display_to(headers.to,page,LOC->prompt_page);	/* display the To: field on screen... */

    /* get the Subject: field */

    if (get_subject(&headers.subject,page,
		    LOC->prompt_page) == 0) {	
	goto free_it;
    }

    if (prompt_for_cc) {
	if (get_copies(&headers.cc, FALSE,mailer_info, aview,
		       page, LOC->prompt_page) == 0) {	    
	    goto free_it;
	}
    }
	
    menu_MoveCursor(LOC->prompt_page,3,0);	/* so you know you've hit <return> ! */
    FlushBuffer();

    /* and mail that puppy outta here! */
    
    DPRINT(Debug,3,(&Debug, 
		    "\na_sendmsg() ready to mail...\n"));
    dump_expanded_address(3,"to",headers.to);
    if (headers.subject) {
	DPRINT(Debug,4, (&Debug,	   
			 "subject=\"%S\"\n",
			 headers.subject));
    }
    dump_expanded_address(5,"cc",headers.cc);
    dump_expanded_address(5,"bcc",headers.bcc);
	
    mail(-1, edit_message ? MAIL_EDIT_MSG : 0, 
	 &headers,mailer_info, 
	 mailbox, aview, page,
	 NULL,-1,NULL,NULL,0);
    
    /*
     *	Since we got this far, it must be okay to clear the tags.
     */
    i = 0;
    while (tagged) {
	struct aliasview_record * a = give_alias(aview,i);
	
	if (a &&
	     aliasview_have_status_mask(a, TAGGED)) {	    
	    struct menu_common MENU;
	    int vis;

	    aliasview_clearf_status(a,TAGGED);
	    
	    set_mcommon_from_aliasview(&MENU,aview);
	    
	    vis = compute_visible(i+1, &MENU);
	    menu_header_status_update(LOC->header_page,vis-1);
	
	    tagged--;
	}
	i++;
    }
    
 free_it:    
    free_mailing_headers(&headers);
    free_mailer_info(&mailer_info);


    /* mail() uses menu_trigger_redraw(page), do not need examine */
 
    return;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
