static char rcsid[] = "@(#)$Id: mailmsg2.c,v 2.73 2022/11/19 06:50:54 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.73 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/mailmsg2.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 *  Incorparated Elm 2.5 code from src/sndmsg.c. 
 *  That code was following copyright:
 *
 *  The Elm Mail System
 *
 * This file and all associated files and documentation:
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** Interface to allow mail to be sent to users.  Part of ELM  **/

#include "def_sndmail.h"
#include "s_elm.h"
#include "s_me.h"

#ifdef I_SYSEXITS
#include <sysexits.h>
#endif

DEBUG_VAR(Debug,__FILE__,"mail");

#include "menu2.h"

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2cus P_((const char *str));
static const unsigned char *cs2cus(str) 
     const char *str;
{
    return (const unsigned char *)str;
}

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

extern char version_buff[];

int  gotten_key;


struct classify_domain_result const NULL_classify_domain_result = {
    0,
    NULL,
    NULL,
    verify_domain_not_available,
    0,
    0,
    0
};

void rp_classify_domain_message(env_from,result,cdr)
     struct mailer_env_from   * env_from;
     enum rp_classify_domain    result;
     struct classify_domain_result *cdr;
{
    const char * env_from_addr   =  mailer_env_from_value(env_from,NULL);
    const char * env_from_domain =  mailer_env_from_domain(env_from);
    
    switch (result) {
    case env_from_mailer_verified:
	break;
	
    case env_from_reset:
	lib_transient(CATGETS(elm_msg_cat, ElmSet,
			      ElmEnvSendChanged,
			      "Envelope sender address changed to %s."),
		      env_from_addr);
	break;
    case env_from_no_domain:
	break;
    case env_from_bad_domain:
	if (verify_domain_bad_syntax == cdr-> domain_result && env_from_addr && env_from_domain) {
	    
	    lib_error(CATGETS(elm_msg_cat, ElmSet,
			      ElmEnvSendBadDomain,
			      "Envelope sender address %s have invalid domain %s."),
		      env_from_addr,env_from_domain);
	}
	break;
    case env_from_ok_domain:
	break;

    case env_from_unknown_domain:
	lib_error(CATGETS(elm_msg_cat, ElmSet,
			  ElmEnvSendUnknownDomain,
			  "Domain %s is unknown on envelope sender address %s."),
		  env_from_domain,env_from_addr);
	break;

    case env_from_no_mail_domain:
	lib_error(CATGETS(elm_msg_cat, ElmSet,
			  ElmEnvSendNoMailDomain,
			  "Domain %s accepts no mail on envelope sender address %s."),
		  env_from_domain,env_from_addr);
	break;
    case env_from_dns_verified:
	break;
	
    }
}


/* Return 1 if succeed */
static int rewrite_env_from_domain P_((struct mailer_env_from *env_from,
					const char * new_domain));
static int rewrite_env_from_domain(env_from,new_domain)
     struct mailer_env_from *env_from;
     const char * new_domain;
{
    int ret = 0;
    const char * env_from_domain =  mailer_env_from_domain(env_from);
    const char * env_from_addr   =  mailer_env_from_value(env_from,NULL);

    
    DPRINT(Debug,8,(&Debug,
		    "rewrite_env_from_domain: Rewriting domain %s on address %s to %s\n",
		    env_from_domain ? env_from_domain : "(no domain)",
		    env_from_addr ?  env_from_addr : "(no address)",
		    new_domain));

    if (! env_from_addr ||
	! env_from_addr[0] ||
	0 == strcmp(env_from_addr,"<>")) {
	DPRINT(Debug,8,(&Debug,
			"rewrite_env_from_domain: No address, domain %s not used, rewriting canceled\n",
			new_domain));
	ret = 0;

	goto fail;
    }

    if (env_from_domain &&
	0 == strcmp(env_from_domain,new_domain)) {
	
	DPRINT(Debug,8,(&Debug,
			"rewrite_env_from_domain: Rewriting not needed, canceled\n"));
			
	ret = 0;
	
	goto fail;		
    }
    
    if (env_from_addr) {
	const char * value1 = env_from_addr;
	char * buffer1 = NULL;
	char * new_address = NULL;
	
	const char * p;
	
	if ('<' == env_from_addr[0]) {
	    int l = strlen(env_from_addr);

	    if (l < 2 || '>' != env_from_addr[l-1]) {
		DPRINT(Debug,8,(&Debug,
			"rewrite_env_from_domain: Bad address, domain %s not used, rewriting canceled\n",
				new_domain));
		ret = 0;
		goto fail2;
	    }
	    buffer1 = safe_malloc(l);
	    if (l > 2) {
		memcpy(buffer1,env_from_addr+1,l-2);
	    }
	    buffer1[l-2] = '\0';
	    DPRINT(Debug,8,(&Debug,
			    "rewrite_env_from_domain: stripping address %s => %s\n",
			    env_from_addr,buffer1));
	    value1 = buffer1;
	}

	p = qstrpbrk_c(value1,"@");
	if (p) {
	    if (p > value1) {
		int len = p - value1;

		new_address = 
		    elm_message(FRM("%.*s@%s"),
				len,value1,new_domain);

		DPRINT(Debug,8,(&Debug,
				"rewrite_env_from_domain: setting domain, address %s => %s\n",
				value1,new_address));	    	    

	    } else  {
		/* If first character is @ then this is routed address,
		   do not change 
		*/
		DPRINT(Debug,8,(&Debug,
				"rewrite_env_from_domain: Unsupported address, domain %s not used, rewriting canceled\n",
				new_domain));
		ret = 0;
		goto fail2;		
	    }

	} else {
	    new_address = 
		elm_message(FRM("%s@%s"),
			    value1,new_domain);
	    
	    DPRINT(Debug,8,(&Debug,
			    "rewrite_env_from_domain: adding domain, address %s => %s\n",
			    value1,new_address));	 
	}

	if (new_address) {
	     DPRINT(Debug,8,(&Debug,
			     "rewrite_env_from_domain: changing address %s => %s\n",
			     env_from_addr,new_address));

	     ret = mailer_env_from_change(env_from,new_address);
	     if (ret) {
		 DPRINT(Debug,8,(&Debug,
				 "rewrite_env_from_domain: return path address changed\n"));
	     }
	}
	
    fail2:
	if (buffer1) {
	    free(buffer1);
	    buffer1 = NULL;
	}

	if (new_address) {
	    free(new_address);
	    new_address = NULL;
	}
    }
	
 fail:

    env_from_addr   =  mailer_env_from_value(env_from,NULL);	
    DPRINT(Debug,8,(&Debug,
		    "rewrite_env_from_domain=%d",		    
		    ret));
    if (env_from_addr) {
	DPRINT(Debug,8,(&Debug,"; return path address=%s",
			env_from_addr));
    }
    DPRINT(Debug,8,(&Debug,"\n"));
    
    return ret;    
}

static enum verify_trans_result 
  verify_transmission P_((char *filename,
			  int already_has_text, 
			  struct copy_file *copy_file,
			  int force_cmd, int options, 
			  int *dsn, 
			  struct mailing_headers *headers,
			  struct Attachments *attachments,
			  charset_t file_set,
			  struct mailer_info  *mailer_info,
			  mime_send_t *mime_info,
			  struct MailboxView *mailbox,			  
			  struct AliasView *aview,
			  struct menu_context *page,
			  struct menu_context *prompt_area,
			  struct header_rec *parent_message,
			  const struct remote_server * remote_server,
			  int *PGP_status,
			  struct elm_commands *snd_cmds,
			  enum rp_classify_domain * env_from_domain_class_p,
			  struct classify_domain_result * env_from_cdr
			  ));  

static void mail_sent P_((union any_fd fd, char * title, struct run_state *rs,
 			  int ret, int exit_stat));

static int append_sig P_((FILE *file,struct mailing_headers *headers,
			  struct menu_context *page));


static void mailing_message P_((int background));
static void mailing_message(background)
     int background;
{
    if (!background)
	lib_transient(CATGETS(elm_msg_cat, ElmSet, ElmSendingMail,
			      "Sending mail..."));
    else
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmSendingMailBackground,
			  "Sending mail... in background"));
    
    FlushBuffer();
}

static const unsigned char * csUs P_((const char *str));
static const unsigned char * csUs(str)
     const char *str;
{
    return (const unsigned char *)str;
}

struct string * gen_From_buffer(current_header)
     const struct header_rec * current_header;
{
    struct string * From_buffer = NULL;
    
    if (current_header->from) {

	int idx;
	int addr_item_count = 
	    addr_list_item_count(current_header->from);

	for (idx = 0; idx < addr_item_count; idx++) {

	    int group = -1;   /* Group phrases are not allowed on
				 From: -header field              */
	    const struct address * address = 
		addr_list_get_item(current_header->from,idx,&group);

	    const char          * addr     = address_get_ascii_addr(address);
	    const struct string * fullname = address_get_phrase(address);

	    if (From_buffer)
		add_ascii_to_string(From_buffer,s2us(", "));
	    else
		From_buffer = new_string(display_charset);

	    if (fullname && string_len(fullname)) {

		append_string(&From_buffer,fullname,1);

	    } else if (addr) {

		add_ascii_to_string(From_buffer,csUs(addr));

	    }
	}
    }

    return From_buffer;
}


enum mailer_capab get_mailer_level(mailer_info)
     struct mailer_info *mailer_info;
{
    enum allow_no_encoding_v an = 
	give_dt_enumerate_as_int(&allow_no_encoding);

    enum mailer_capab result = mailer_7bit;

    if (query_mailer_info(mailer_info,MI_HAVE_8BITMIME)) {
	DPRINT(Debug,8,(&Debug,"mailer supports 8BITMIME\n"));
	result        = mailer_8bit;
	
	if (query_mailer_info(mailer_info,MI_HAVE_BINARYMIME)) {
	    DPRINT(Debug,8,(&Debug,"mailer supports BINARYMIME\n"));
	    result        = mailer_binary;
	}
    }

    if (an >= allow_pass_binary) {

	/* Just send BINARY anyway */
	result     = mailer_binary;

	DPRINT(Debug,8,(&Debug,"just send binary anyway\n"));

    }  else if (an >= allow_pass_8bit &&
		result < mailer_8bit) {
	
	/* Just send 8BIT anyway */
	result     = mailer_8bit;

	DPRINT(Debug,8,(&Debug,"just 8 bit anyway\n"));
    }


    DPRINT(Debug,8,(&Debug,"get_mailer_level=%d\n",result));

    return result;
	
}

static void handle_mailer_options  P_((struct mailer_info *mailer_info,
				       mime_send_t *mime_info,
				       int *dsn,
				       struct Attachments *attachments,
				       int *reask));

static void handle_mailer_options(mailer_info,mime_info,dsn, attachments,
				  reask)
     struct mailer_info *mailer_info;
     mime_send_t *mime_info;
     int *dsn;
     struct Attachments *attachments;
     int *reask;  /* on reask loop */
{
    int i;
    int changed  = 0;

    if (query_mailer_info(mailer_info,MI_HAVE_DSN)) {
	DPRINT(Debug,8,(&Debug,"mailer supports DSN\n"));
	if(DSN_success && !reask) {
	    (*dsn) |= DSN_SUCCESS|DSN_FAILURE|DSN_DELAY;
	    DPRINT(Debug,8,(&Debug," dsn-success is set .. enabling\n"));
	}
    } else if (*dsn) {
	*dsn = 0;
	
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmNoDSN,
			  "Mailer do not support DSN, disabling it"));

	if (reask)
	    *reask = 1;

    }

    mime_info->raw_level    = get_mailer_level(mailer_info);
    
    for (i = 0; i < attachments->attachment_count; i++) {
	mime_t *part = & (attachments->attach_files[i]);
	
	if (ENCODING_BINARY == part->encoding &&
	    mime_info->raw_level < mailer_binary) {
	    part->encoding = ENCODING_BASE64;
	    changed++;
	}
		
	if (ENCODING_8BIT == part->encoding &&
	    mime_info->raw_level < mailer_8bit) {
	    part->encoding = ENCODING_BASE64;
	    changed++;
	}
    }

    if (changed) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmChangedAttachEnc,
			  "Changed encoding of %d attachments"),
		  changed);
	
	if (reask)
	    *reask = 1;
    }
}

static int mmsg_copy_part P_((FILE * mailbox_file,mime_t *mime_rec,
			      FILE *outfile,int quote_l, int *need_enc,
			      char **fname));

static int mmsg_copy_part(mailbox_file,mime_rec,outfile,quote_l,
			     need_enc, fname)
     FILE * mailbox_file;
     mime_t *mime_rec;
     FILE *outfile;   /* MAY be NULL */
     int quote_l;
     int *need_enc;/* MAY be NULL */
     char **fname;    
{
    struct in_state * state_in = new_in_state(STATE_in_file);
    struct in_state *decoded_state_in = NULL;
    int is_text = 0;
    int flags;
    int ret = 0;

    const char *nameref = NULL;

    if (fname)
	*fname = NULL;

    if (0 != fseek(mailbox_file,mime_rec->offset,SEEK_SET)) {
	DPRINT(Debug,3,(&Debug,
			"mmsg_copy_part: Seek failed to %ld\n",
			mime_rec->offset));
	
	goto FAIL0;
    }

    if (quote_l) {
	DPRINT(Debug,11,(&Debug,
			     "mmsg_copy_part: quote_l set, assuming textual encoding (%s)\n",
			     ENCODING(mime_rec->encoding)));

	is_text = 1;

    } else 
	is_text = decoder_is_textual(mime_rec);

    decoded_state_in = new_in_state(STATE_in_decode);

    set_in_state_file(mailbox_file,state_in);

    flags = STATE_DECODE_buffer;

    if (fname)
	flags |= STATE_DECODE_keep_file;

    if (is_text > 0)
	flags |=  STATE_DECODE_is_text;


    if (set_in_state_decode(state_in,decoded_state_in,
			    mime_rec->length,
			    flags,
			    mime_rec->encoding,
			    fname ? &nameref : NULL)) {
	int errors;

	if (outfile) {
	    int count = 0;
	    int ch;
	    int nl = 1;	    	   

	    while (EOF != (ch  = state_getc(decoded_state_in))) {

		if (nl && '[' == ch && quote_l)
		    putc('[',outfile);
		
		putc(ch,outfile);
		nl = '\n' == ch;

		count++;
	    }

	    DPRINT(Debug,11,(&Debug,
			     "mmsg_copy_part: %d characters copied to outfile\n",count));
	}

	ret = 1;
	
	/* Print total 10 error messages */
	for (errors = 0; errors < 10; errors++) {
	    const struct string *X =  
		in_state_error_message(decoded_state_in,
				       1  /* Clear error */);
	    
	    if (!X)
		break;
	    
	    lib_error(FRM("%S"),X);

	    ret = 0;
	}

	if (!ret)
	    goto FAIL1; 

	if (need_enc) {
	    if (0 != in_state_fseek(decoded_state_in,0L)) {
		 const struct string *X =  in_state_error_message(decoded_state_in,
								  1  /* Clear error */);

		DPRINT(Debug,3,(&Debug,
				"mmsg_copy_part: Seek failed to 0 (STATE_in_decode)\n"));
		
		if (X)
		    lib_error(FRM("%S"),X);

		goto FAIL1;
	    }
	    
	    *need_enc = in_state_needs_encoding(decoded_state_in,NULL,0);

	    
	    DPRINT(Debug,11,(&Debug,
			     "mmsg_copy_part: *need_enc=%d\n",*need_enc));
	}

	if (fname)
	    *fname = safe_strdup(nameref);

	/* nameref is free'ed on free_in_state(&decoded_state_in) */

    } else {

    FAIL1:
	ret = 0;
	
	/* FAILURE */
	
	if (fname) {
	    if (0 == unlink(nameref)) {
		DPRINT(Debug,11,(&Debug,
				 "mmsg_copy_part: %s unlinked\n",nameref));
	    }
	}
    }
 
    free_in_state(&decoded_state_in);

 FAIL0:
    free_in_state(&state_in);

    if (fname && *fname) {
	DPRINT(Debug,11,(&Debug,
			 "mmsg_copy_part: *fname=%s\n",*fname));
    }

    DPRINT(Debug,11,(&Debug,
			"mmsg_copy_part=%d\n",ret));
    

    return ret;
}


static int convert_attach_message P_((FILE *infile,
				      struct header_rec *current_header,
				      enum mailer_capab  mailer_level,
				      FILE * outfile,
				      enum encoding      * encoding
				      ));
static int convert_attach_message(infile,current_header,mailer_level,outfile,
				  encoding)
     FILE *infile;
     struct header_rec *current_header;
     enum mailer_capab  mailer_level;
     FILE                   * outfile;
     enum encoding          * encoding;
{
    int r;
    struct out_state  * target = new_out_state(STATE_out_file);

    set_out_state_file(outfile,target);

    r = convert_remail_message_1(infile,current_header,mailer_level,
				 target,encoding,NULL);

    /* Does not close file */
    free_out_state(&target);

    DPRINT(Debug,8,(&Debug, 
		    "convert_attach_message=%d\n",
		    r)); 

    return r;
}

int attach_rfc822(attachments,hdr,mailbox_file,title,
		  mailer_level)
     struct Attachments *attachments;
     struct header_rec  *hdr;
     FILE               *mailbox_file;
     struct string      *title;
     enum mailer_capab  mailer_level;
{
    mime_t attach;
    static int count = 0;
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    char * filename;
    FILE * tmpfp;
    enum encoding encoding = ENCODING_7BIT;

    if (!tmp)
	return 0;
    
    filename = elm_message(FRM("%selmmsg.%d-%d"), 
			   tmp, getpid (), count++);
    
    tmpfp = safeopen_rdwr(filename,NULL);
    
    if (!tmpfp) {
	free(filename);
	return 0;
    }
     
    if (hdr->status  & MIME_MESSAGE) {

	enum encoding encoding1 = ENCODING_NONE;

	if (0 != mime_warnings(hdr)) {
	    
	    DPRINT(Debug,4,(&Debug, 
			    "attach_rfc822: Bad-MIME message .. remailing as is\n"));
	    goto as_is;
	}

	DPRINT(Debug,4,(&Debug, 
			"attach_rfc822: Doing possible mime conversions\n"));
	
	/* Do possible conversions   */

	if (!convert_attach_message(mailbox_file,hdr,
				    mailer_level,tmpfp,
				    &encoding1)) {

	    unlink(filename);
	    free(filename);
	    fclose(tmpfp);
	    
	    return 0;
	}

	/* Pick up the encoding from the message. */
	update_encoding (&encoding, encoding1);

    } else {
   		
	DPRINT(Debug,4,(&Debug, 
			"attach_rfc822: Non-MIME message .. remailing as is\n"));
    as_is:
       
	if (!copy_message_f(mailbox_file,hdr,tmpfp,
			    CM_REMOVE_ENVELOPE,NULL,
			    NULL)) {
	    
	    unlink(filename);
	    free(filename);
	    fclose(tmpfp);
	    
	    return 0;
	}

	/* Pick up the encoding from the message. */
	update_encoding (&encoding,
			 hdr->mime_rec.encoding);
    }


    mime_t_zero(&attach);
    attach.pathname0 = filename;
    attach.dispname  = new_string2(system_charset,s2us(filename));
		    
    attach.length = fsize(tmpfp);
    fclose (tmpfp);
    
    attach.unlink = 2; /* mark for later deletion */
    attach.TYPE = give_media_type2(MIME_TYPE_MESSAGE,"rfc822", 0);
    if (!attach.TYPE)
	mime_panic(__FILE__,__LINE__,"attach_rfc822",
		   "message/rfc822 is not known");
 
    if (title)
	attach.description = dup_string(title);

    attach.encoding = encoding;
    add_Attachments(attachments,&attach);

    return 1;
}

int attach_body(attachments,hdr,mailbox_file,mailer_level)
     struct Attachments *attachments;
     struct header_rec  *hdr;
     FILE               *mailbox_file;
     enum mailer_capab  mailer_level;
{

    int need_enc = 0;
    mime_t attach;
    
    mime_t_zero(&attach);

    if (! mmsg_copy_part(mailbox_file,& (hdr->mime_rec),
			 NULL,0,&need_enc,
			 &attach.pathname0)) {
	
	DPRINT(Debug,4, (&Debug, 
			 "Failed to copy message \n"));
	return 0;
    }

    attach.dispname = new_string2(display_charset,
				  s2us(attach.pathname0));
    
    attach.unlink = 2; /* mark for later deletion */
    attach.TYPE = hdr->mime_rec.TYPE;
    attach.encoding     = hdr->mime_rec.encoding;
    attach.disposition  = hdr->mime_rec.disposition;
    
    if (attach.encoding >= ENCODING_NONE &&
	attach.encoding <= ENCODING_BINARY) {
	enum encoding  UNUSED_VAROK old = attach.encoding;

	if (need_enc & HAVE_BINARY) {
	    if (mailer_level >= mailer_binary)
		attach.encoding = ENCODING_BINARY;
	    else
		/* no -BBINARYMIME option */
		attach.encoding = ENCODING_QUOTED;
	}
	else if (need_enc & HAVE_8BIT) {
	    if (mailer_level >= mailer_8bit)
		/* Just send 8BIT anyway */
		attach.encoding = ENCODING_8BIT;
	    else
		/* no -B8BITMIME option */
		attach.encoding = ENCODING_QUOTED;
	}

	DPRINT(Debug,12,(&Debug,
			 "attach_body: %s need_enc=%d%s%s fixup encoding %d (%s) => %d (%s)\n",
			 attach.pathname0,
			 need_enc,
			 (need_enc & HAVE_BINARY) ? " BINARY" : "",
			 (need_enc & HAVE_8BIT)   ? " 8BIT"   : "",
			 old,ENCODING(old),
			 attach.encoding, ENCODING(attach.encoding)));	
    }



    if (hdr->mime_rec.description) {
	attach.description = 
	    dup_string(hdr->mime_rec.description);
    }
    
    if (hdr->mime_rec.TYPE_opts) {
	attach.TYPE_opts = 
	    copy_mime_param(hdr->mime_rec.TYPE_opts);
    }
    
    if (hdr->mime_rec.DISPOSITION_opts) {
	attach.DISPOSITION_opts = 
	    copy_mime_param(hdr->mime_rec.DISPOSITION_opts);
    }
    
    add_Attachments(attachments,&attach);
    
    return 1;
}

static int attach_part P_((struct Attachments *attachments,
			   struct header_rec  *hdr,
			   FILE               *mailbox_file,
			   int x,
			   enum mailer_capab  mailer_level));
static int attach_part(attachments,hdr,mailbox_file,x,mailer_level)
     struct Attachments *attachments;
     struct header_rec  *hdr;
     FILE               *mailbox_file;
     int x;
     enum mailer_capab  mailer_level;
{    
    mime_t *z = NULL;
    mime_t attach;
    int need_enc = 0;

    z = mime_parser_index(hdr->mime_rec.parser_data,x);
    if (!z) {
	DPRINT(Debug,4, (&Debug, "Failed to get part %d\n",x));
	return 0;
    }

    if (!z->TYPE) {
	DPRINT(Debug,4, (&Debug, "No type on part %d\n",x));
	return 0;
    }

    mime_t_zero(&attach);


    if (! mmsg_copy_part(mailbox_file,z,NULL,0,
			 &need_enc, &attach.pathname0)) {

	DPRINT(Debug,4, (&Debug, 
			 "Failed to copy part %d/%d\n",x));
	return 0;
    }

    attach.dispname = new_string2(display_charset,s2us(attach.pathname0));

    attach.unlink = 2; /* mark for later deletion */
    attach.TYPE = z->TYPE;
    attach.encoding     = z->encoding;
    attach.disposition  = z->disposition;

    if (attach.encoding >= ENCODING_NONE &&
	attach.encoding <= ENCODING_BINARY) {
	enum encoding  UNUSED_VAROK old = attach.encoding;
	
	if (need_enc & HAVE_BINARY) {
	    if (mailer_level >= mailer_binary)
		attach.encoding = ENCODING_BINARY;
	    else
		/* no -BBINARYMIME option */
		attach.encoding = ENCODING_QUOTED;
	}
	else if (need_enc & HAVE_8BIT) {
	    if (mailer_level >= mailer_8bit)
		/* Just send 8BIT anyway */
		attach.encoding = ENCODING_8BIT;
	    else
		/* no -B8BITMIME option */
		attach.encoding = ENCODING_QUOTED;
	}
	
	DPRINT(Debug,12,(&Debug,
			 "attach_part: %s need_enc=%d%s%s fixup encoding %d (%s) => %d (%s)\n",
			 attach.pathname0,
			 need_enc,
			 (need_enc & HAVE_BINARY) ? " BINARY" : "",
			 (need_enc & HAVE_8BIT)   ? " 8BIT"   : "",
			 old,ENCODING(old),
			 attach.encoding, ENCODING(attach.encoding)));	
    }
			
    if (z->description) {
	attach.description = dup_string(z->description);
    }

    if (z->TYPE_opts) {
	attach.TYPE_opts = copy_mime_param(z->TYPE_opts);
    }

    if (z->DISPOSITION_opts) {
	attach.DISPOSITION_opts = copy_mime_param(z->DISPOSITION_opts);
    }

    add_Attachments(attachments,&attach);

    return 1;
}


/* Return 1 if mail is saved to canceled mail */

static int cancel_ticket_to_canceled_mail 
  P_((struct mailer_cancel_ticket *ticket,
     struct mailing_headers * headers));
static int cancel_ticket_to_canceled_mail(ticket,headers)
     struct mailer_cancel_ticket *ticket;
     struct mailing_headers * headers;
{
    struct MailboxView *cm = give_canceled_mail();

    char * lbuf = NULL;
    FILE *F = NULL;
    const char * s_fname = have_cancel_filename(ticket);
    FILE *       s_F     =  have_cancel_FILE(ticket);
    int ok = 0;
    int lines = -1;

    if (s_F) {
	int c;
	lines = 0;

	while (EOF != (c = getc(s_F))) {
	    if ('\n' == c)
		lines++;
	}
	rewind(s_F);

	DPRINT(Debug,10,(&Debug, "Canceled mail have %d lines.\n",
			 lines));
    }
 
    if (cm) {
	time_t now = 0;
	    
	if (((time_t) -1) != time(&now)) {

	    F = add_canceled_mail(&lbuf,headers,cm,now,lines,NULL,
				  -1L);
	} else {
	    int err UNUSED_VAROK = errno;
	    
	    DPRINT(Debug,4,(&Debug,
			    "cancel_ticket_to_canceled_mail: Date: not given, time: %s\n",
			    strerror(err)));
	}
    }
	
    if (!F) {
	if (lbuf)
	    free(lbuf);
	
	lbuf = elm_message(FRM("%s/%s"), 
			   home, dead_letter);
	
	F = safeopen_rdwr(lbuf,NULL);
    } else 
	ok = 1;

    if (NULL != F) {	
	DPRINT(Debug,4,(&Debug, "Writing canceled mail to %s\n",
			lbuf));

	/* rename replaces just created file */

	if (s_fname && 0 == rename(s_fname,lbuf)) {
	    DPRINT(Debug,4,(&Debug, 
			    "Renamed %s => %s\n",
			    s_fname,lbuf));
	} else if (s_F) {
	    int c;

	    while (EOF != (c = getc(s_F))) {
		putc(c,F);
	    }
	    
	    if (0 == fflush(F) &&
		!ferror(s_F) && 
		!ferror(F)) {
		DPRINT(Debug,4,(&Debug, 
				"Copied %s => %s\n",
				s_fname,lbuf));
	    } else
		ok = 0;
	} else
	    ok = 0;
	
    }

    if (F)
	fclose(F);

    if (ok) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmVfyMessageKept1,
			  "Message kept.  Can be restored at next m)ail command."));
	
	if (lbuf) {
	    
	    if (last_canceled_mail)
		free(last_canceled_mail);
	    last_canceled_mail = lbuf;	    
	}
    } else if (lbuf) {
	free(lbuf);
    }
   
    return ok;
}

static FILE * generate_edit_buffer P_((struct text_block *body,
				       int * Exstat,
				       int * already_has_text,
				       charset_t * cur_editcharset));
static FILE * generate_edit_buffer(body,Exstat,already_has_text,cur_editcharset)
     struct text_block *body;
     int * Exstat;
     int * already_has_text;
     charset_t * cur_editcharset;
{
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    FILE * reply = NULL;
    int err = 0;
    
    if (!tmp)
	return NULL;

    /** first generate the temporary filename **/
    
    elm_sfprintf(cur_editfile,sizeof cur_editfile,
		 FRM("%s%s%d"), 
		 tmp, temp_file, getpid());
    
    /** if we're not retransmitting, create the file.. **/

    if ((reply = safeopen_rdwr(cur_editfile,&err)) == NULL) {
	DPRINT(Debug,1,(&Debug,  
			"Attempt to write to temp file %s failed with error %s (mail)\n",
			cur_editfile, strerror(err)));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldNotCreateFile,
			  "Could not create file %s (%s)."),
		  cur_editfile, strerror(err));

#ifdef  EX_CANTCREAT
	*Exstat = EX_CANTCREAT;  /* Give better error status */
#endif
	
	goto fail;	
    }
   
    (void) elm_chown(cur_editfile, userid, groupid, NULL);

    if (body) {
	if (! add_block_to_mail(body,
				already_has_text,
				reply,
				cur_editcharset)) {

	    DPRINT(Debug,1,(&Debug,  
			    "Failed to add body block to editfile\n"));
	    
	    goto fail;
	}
    }

    if (!reply) {
    fail:

	if (reply) {
	    fclose(reply);
	    reply = NULL;

	    unlink(cur_editfile);
	}
    }

    DPRINT(Debug,8,(&Debug,"generate_edit_buffer %s, filename %s\n",
		    reply ? "succeed" : "failed",
		    cur_editfile));

    return reply;
}

/* Returns 0 if copying of message failed */
int copy_message_to_buffer P_((FILE               * reply,
			       mime_send_t        * MIME_info,
			       charset_t          * cur_editcharset,
			       int                * already_has_text,
			       struct Attachments * attachments,
			       struct mailing_headers * headers,
			       struct MailboxView * W,
			       int                  Wi,
			       int options));

/* Returns 0 if copying of message failed */
int copy_message_to_buffer(reply,MIME_info,cur_editcharset,
			   already_has_text,attachments,
			   headers,
			   W,Wi,options)
     FILE               * reply;
     mime_send_t        * MIME_info;
     charset_t          * cur_editcharset;
     int                * already_has_text;
     struct Attachments * attachments;
     struct mailing_headers * headers;
     struct MailboxView * W;
     int                  Wi;
     int options;
{
    int forwarding     = 0 != (options & MAIL_FORWARDING);
    int mime_attach    = 0 != (options & MAIL_MIME_ATTCH);
    int copy_msg       = 0 != (options & MAIL_COPY_MSG);
    int edit_message   = 0 != (options & MAIL_EDIT_MSG);
    int delete_cancel  = 0 != (options & MAIL_DELETE_CANCEL);
    int copy_selection UNUSED_VAROK = 0 != (options & MAIL_COPY_SELECTION); 


    struct header_rec *  hdr = NULL;
    FILE              *  mailbox_file = NULL;              
    int r = 0;

    if (!give_message_data(W,Wi,
			   &hdr,&mailbox_file,NULL,
			   mime_parse_routine)) {
	
	DPRINT(Debug,4,(&Debug, 
			"Failed to retrieve original mail for replying/forwarding/canceled mail\n"));
	
	return 0;
    }

    if (forwarding || mime_attach || copy_msg) { 

	if (!*cur_editcharset)
	    *cur_editcharset = display_charset;

	if (!mime_attach) {
	    
	    if (edit_message && mailbox_file) {

		int NOHDR = forwarding ? noheaderfwd : noheader;
		int NOQUOTE = forwarding && !quote_forward; 
		int FORW = forwarding;

		r = copy_message_f(mailbox_file,hdr,
				   reply,
				   ( NOQUOTE ? 0 : CM_PREFIX ) |
				   ( NOHDR ? CM_REMOVE_HEADER : 0 ) |
				   CM_REMOVE_ENVELOPE | CM_DECODE |
				   ( FORW ? CM_FORWARDING : 0 ) |
				   ( mime_body_keywords ? CM_QUOTE_L : 0 ) |
				   CM_ATTRIBUTION | CM_FILT_HDR |
				   CM_SHOW_ERROR, 
				   *cur_editcharset,
				   NULL);

	    } else if (mailbox_file) {
		
		int NOHDR = forwarding ? noheaderfwd : noheader;
		int FORW = forwarding;

		r = copy_message_f(mailbox_file,hdr,
				   reply,
				   ( NOHDR ? CM_REMOVE_HEADER : 0 ) |
				   CM_REMOVE_ENVELOPE | CM_DECODE |
				   ( FORW ? CM_FORWARDING : 0 ) |
				   ( mime_body_keywords ? CM_QUOTE_L : 0 ) |
				   CM_ATTRIBUTION, *cur_editcharset,
				   NULL);
	    }

	    if (r)
		*already_has_text = 1;

	} else { /* mime_attach */

	    struct string * title = NULL;
	    struct string * From_buffer = gen_From_buffer(hdr);	    

	    if (forwarding) {
		if (From_buffer)
		    title = format_string(CATGETS(elm_msg_cat, ElmSet, 
						  ElmForwardedMesg1,
						  "Forwarded message from %S"), 
					  From_buffer);
		else
		    title =	format_string(CATGETS(elm_msg_cat, ElmSet, 
						      ElmForwardedMesg2,
						      "Forwarded message from %s"), 
					      hdr->env_from);
	    }

	    r = attach_rfc822(attachments, hdr,mailbox_file,title,
			      MIME_info->raw_level);

	    if (title)
		free_string(&title);
	    
	    if (From_buffer)
		free_string(&From_buffer);	       	    

	    if (!r) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailFileForward,
				  "Failed to create file for forwarding"));

	    }

	}

    } else {  /* not forwarding, mime_attach, copy_msg */

	/* make referenced message as edited message */
	header_list_ptr all_headers,walk;
	int Flags, major;
	const char *subtype;

	DPRINT(Debug,4, (&Debug,
			 "Not forwarding, copying message normally or mime attaching, making message as edited -- copy_selection=%d\n",
			 copy_selection));
	
	if (skip_envelope(hdr,mailbox_file) == -1) {
	    DPRINT(Debug,4, (&Debug, "Can't skip envelope of message \n"));
	    
	    return 0;
	}

	all_headers = file_read_headers(mailbox_file,RHL_CHECK_HEADER);

	for (walk = all_headers; walk; walk = walk -> next_header) {
	    const char * hdr_name = give_header_name(walk->header_name);
	    
	    if (0 == strincmp(hdr_name,"Content-",8) ||
		0 == istrcmp(hdr_name,"MIME-Version") ||
		    0 == istrcmp(hdr_name,"Received") ||
		0 == istrcmp(hdr_name,"Sender"))
		continue;
	    
	    if (!add_to_mailing_header(headers,walk->header_name,
				       walk->body,
				       !(hdr->status & NOHDRENCODING),
				       hdr->header_charset,1)) {
		DPRINT(Debug,4, (&Debug, "Header %s not copied\n",
				 hdr_name));
	    }
	}
	
	dump_expanded_address(4,"after copy -- from ",headers->from);
	dump_expanded_address(4,"after copy -- to   ",headers->to);
	dump_expanded_address(4,"after copy -- cc   ",headers->cc);
	dump_expanded_address(4,"after copy -- bcc  ",headers->bcc);

	if (! hdr->mime_rec.TYPE) {
	    DPRINT(Debug,4, (&Debug, "no mime type\n"));
	    goto fail1;		
	}

	Flags   =  get_type_flags(hdr->mime_rec.TYPE);
	major   =  get_major_type_code(hdr->mime_rec.TYPE);
	subtype =  get_subtype_name(hdr->mime_rec.TYPE);

	/* 1) Includes only one text part -- put it to edit buffer */
	if (MIME_TYPE_TEXT == major && 0 == istrcmp(subtype,"Plain") &&
	    DISP_INLINE == hdr->mime_rec.disposition) {
	    
	    const char *cs = NULL;
	    
	    if (! mmsg_copy_part(mailbox_file,& (hdr->mime_rec),
				 reply,mime_body_keywords,NULL,NULL)) {
		
		DPRINT(Debug,4, (&Debug, "Failed to copy part\n"));
		
		goto fail1;
	    }
	    
	    cs = get_mime_param_compat(hdr->mime_rec.TYPE_opts,
				       "charset");
	    if (!cs) {
		DPRINT(Debug,4, (&Debug, "No charset parameter\n"));

		/* Not necessary correct */
		*cur_editcharset = display_charset;

	    } else {

		*cur_editcharset = MIME_name_to_charset(cs,0);
		if (!*cur_editcharset) {
		    DPRINT(Debug,4, (&Debug, "charset parameter %s unknown?\n",
				     cs));
		    goto fail1;
		}
		
		if (charset_superset_of(display_charset,*cur_editcharset)) {
		    const char *MIME_name_c UNUSED_VAROK = 
			get_charset_MIME_name(*cur_editcharset);
		    const char *MIME_name_d UNUSED_VAROK = 
			get_charset_MIME_name(display_charset);
		    
		    DPRINT(Debug,4, (&Debug, 
				     "charset parameter %s (%s) is subset of display charset %s\n",
				     MIME_name_c ? 
				     MIME_name_c : 
				     "<no MIME name>",
				     cs,
				     MIME_name_d ?
				     MIME_name_d :
				     "<no MIME name>"));
		    *cur_editcharset = display_charset;
		}
	    }
	    
	    *already_has_text = 1;
	    r = 1;
	    DPRINT(Debug,4, (&Debug, "Added whole mail to editing buffer\n"));
	    
	    /* 2) multipart MIXED */
	} else if ((Flags & MIME_MIXED) &&
		   DISP_INLINE == hdr->mime_rec.disposition &&
		   hdr->mime_rec.parser_data) {
	    
	    int s = 0;
	    int c = mime_parser_subparts(hdr->mime_rec.parser_data);
	    mime_t *z = NULL;
	    int x;
	    
	    /* Pick first parts to editing buffer */
	    if (c > 0 &&
		(z = mime_parser_index(hdr->mime_rec.parser_data,0)) &&
		z->TYPE) {
		int major           =  get_major_type_code(z->TYPE);
		const char *subtype =  get_subtype_name(z->TYPE);
		
		const char *cs = NULL;
		
		if (MIME_TYPE_TEXT == major && 0 == istrcmp(subtype,"Plain") &&
		    DISP_INLINE == z->disposition) {
		    
		    if (! mmsg_copy_part(mailbox_file,z,reply,
					 mime_body_keywords,
					 NULL,NULL)) {
			DPRINT(Debug,4, (&Debug, "failed to copy part\n"));
			goto fail1;
		    }
		    
		    cs = get_mime_param_compat(z->TYPE_opts,
					       "charset");
		    if (!cs) {
			DPRINT(Debug,4, (&Debug, "No charset parameter?\n"));
			goto fail1;
		    }
		    
		    *cur_editcharset = MIME_name_to_charset(cs,0);
		    if (!*cur_editcharset) {
			DPRINT(Debug,4, (&Debug, "charset parameter %s unknown?\n",
					 cs));
			goto fail1;
		    }
		    
		    if (charset_superset_of(display_charset,*cur_editcharset)) {
			
			const char *MIME_name_c UNUSED_VAROK = 
			    get_charset_MIME_name(*cur_editcharset);
			const char *MIME_name_d UNUSED_VAROK = 
			    get_charset_MIME_name(display_charset);
						    
			DPRINT(Debug,4, (&Debug, 
					 "charset parameter %s (%s) is subset of display charset %s\n",
					 MIME_name_c ?
					 MIME_name_c :
					 "<no MIME name>",
					 cs,
					 MIME_name_d ?
					 MIME_name_d :
					 "<no MIME name>"));
			*cur_editcharset = display_charset;
		    }
			
		    DPRINT(Debug,4, (&Debug, 
				     "Added first part of mail to editing buffer\n"));
		    
		    /* Start attachments from second part */
		    s = 1;		    
		}       	
	    }

	    r = 1;			
	    
	    for (x = s; x < c; x++) {
		
		if (! attach_part(attachments, hdr,mailbox_file,x,
				  MIME_info->raw_level)) {
		    DPRINT(Debug,4, (&Debug, 
				     "Failed to copy part %d/%d\n",
				     x,c));
		    continue;
		}
		
		
		DPRINT(Debug,4, (&Debug, 
				 "Added part %d/%d\n",
				 x,c));
		
	    }
	    
	    
	    /* 3) Add just whole mail as attachment */
	} else {
	    
	    if (! attach_body(attachments, hdr,mailbox_file,
			      MIME_info->raw_level)) {
		DPRINT(Debug,4, (&Debug, 
				 "Failed to copy messge \n"));
		goto fail1;
	    }
	    
	    
	    DPRINT(Debug,4, (&Debug, 
			     "Added message\n"));
	    
	    r = 1;
	}
	
	if (delete_cancel)
	    delete_current_cancel(W,mailbox_file);

	
    fail1:

	if (all_headers)
	    delete_headers(&all_headers);

    }

    if (!r) {
	DPRINT(Debug,4,(&Debug, 
			"Failed to process original mail for replying/forwarding/canceled mail\n"));
    }

    return r;
}			       

static void add_self_addr P_((struct expanded_address * cc_or_bcc,
			      struct address          * usraddr,
			      struct mailer_info      * mailer_info,
			      struct expanded_address * from,
			      const char              * header_name));
static void add_self_addr(cc_or_bcc,usraddr,mailer_info,from,header_name)
     struct expanded_address * cc_or_bcc;
     struct address          * usraddr;
     struct mailer_info      * mailer_info;
     struct expanded_address * from;
     const char              * header_name;
{
    const char * addrliteral = NULL;
    char       * Q           = NULL;
    int pos                  = -1;
    struct string * surface  = NULL; 
    
    if (usraddr)
	addrliteral = address_get_ascii_addr(usraddr);
    else if(from &&  have_expanded_address(from))  {

	DPRINT(Debug,8,(&Debug,"add_self_addr: will use from header\n"));

	/* Do not erase previous content */
	append_expanded_address(cc_or_bcc,from);
    } else {
	char * from_domain = NULL;
	               /* Result is malloced */

	int from_ok = 1;
	Q           = from_addr_literal(mailer_info,
					&from_domain);

	DPRINT(Debug,8,(&Debug,"add_self_addr: from addr %s",
			Q));
	if (from_domain) {
	    int          is_literal;
	    const char * whitelisted_name = NULL;
	    const char * reserved_name    = NULL;
	    
	    DPRINT(Debug,8,(&Debug,"; from domain %s\n",
			    from_domain));

	    from_ok = build_address_classify_domain(from_domain,
						    &is_literal,
						    &whitelisted_name,
						    &reserved_name);

	    if (!from_ok) {
		build_address_classify_message(Q,from_domain,
					       is_literal,
					       whitelisted_name,
					       reserved_name);

		if (header_name)
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmSelfFailNotAdd,
				      "Domain %s for self address %s not valid; Not added to %s -header."),
			      from_domain,Q,header_name);
	    }
	    
	    free(from_domain);
	    from_domain = NULL;
	} else {
	    DPRINT(Debug,8,(&Debug,"\n"));
	}
	
	if (from_ok) {
	    addrliteral = Q;
	} else {
	    DPRINT(Debug,8,(&Debug,"add_self_addr: Address %s ignored\n",
			    Q));
	}
    }

    if (addrliteral) {
	
	if (cc_or_bcc->addrs) {
	    int count = addr_list_item_count(cc_or_bcc->addrs);
	    
	    int i;
	    
	    for (i = 0; i < count; i++) {
		int g;
		const struct address * adr = addr_list_get_item(cc_or_bcc->addrs,i,&g);
		const char * ascaddr = address_get_ascii_addr(adr);
		
		if (ascaddr &&
		    0 == strcmp(addrliteral,ascaddr)) {
		    DPRINT(Debug,8,(&Debug,"add_self_addr: Address %s found already\n",
				    addrliteral));
		    goto done;
		}
	    }
	}
	
	DPRINT(Debug,8,(&Debug,"add_self_addr: Needed add address %s\n",
			addrliteral));
	
	if (usraddr) {
	    pos = add_expanded_addr0_(cc_or_bcc,usraddr,-1);
	    surface = make_surface_addr(usraddr);
	    
	} else if (Q) {
	    pos = add_expanded_addr_(cc_or_bcc,Q,full_username_s,NULL);
	    /* Use username as surface addr */
	    surface =  username_to_surface_addr();
	}
	
	if (pos >= 0 && surface)
	    add_textual_addr_(cc_or_bcc,surface,pos,1);
    }
	
 done:
    if (Q)
	free(Q);
    if (surface)
	free_string(& surface);
}


enum rp_classify_domain env_from_classify_domain(X,cdr,mailer_info)
     struct mailer_env_from * X;
     struct classify_domain_result * cdr;
     struct mailer_info      * mailer_info;
{
    enum rp_classify_domain ret =  env_from_no_domain;

    /* Check return path domain */	

    if (cdr)
	*cdr = NULL_classify_domain_result;
    else {
	static struct classify_domain_result temp;

	temp = NULL_classify_domain_result;
	cdr = &temp;
    }
	
    if (mailer_env_from_verified(X)) {
	DPRINT(Debug,8,(&Debug,
			"env_from_classify_domain: Mailer verified return path as local address, domain verification not needed\n"));
	ret = env_from_mailer_verified;
    } else {
	const char * env_from_domain =  mailer_env_from_domain(X);
        
	if (env_from_domain) {
	    int ok_domain;
	    
	    DPRINT(Debug,8,(&Debug,
			    "env_from_classify_domain: Return path domain is %s\n",
			    env_from_domain));
	    
	    ok_domain = build_address_classify_domain(env_from_domain,
						      & (cdr->is_literal),
						      & (cdr->whitelisted_name),
						      & (cdr->reserved_name));
	    
	    
	    if (ok_domain) {
		
		ret = env_from_ok_domain;

		if (build_address_is_local(env_from_domain)) {
		    DPRINT(Debug,12,(&Debug,
				     "env_from_classify_domain: %s is local\n",
				     env_from_domain));
		    cdr->is_local = 1;
		}

		if (!(cdr->whitelisted_name) && !(cdr->is_literal)) {

		    int val = give_dt_enumerate_as_int(&verify_domain);

		    if (val) {

			DPRINT(Debug,12,(&Debug,
					 "env_from_classify_domain: verify-domain =  %d%s\n",
					 val,
					 val  < 0 ? ", uses shared library" : ""));

			if (! cdr->is_local ||
			    need_verify_ldomain(mailer_info,
						mailer_env_from_value(X,
								      NULL))) {
			     struct cancel_data  * main_cancel =
				 build_address_cancel_mode(env_from_domain);
			    char *rewrite = NULL;
			    
			    cdr->need_verify = 1;
			    
			    cdr->domain_result =
				verify_mail_domain_util(env_from_domain,
							&rewrite,
							&main_cancel);

			    switch (cdr->domain_result) {
			    case verify_domain_not_found:
				ret = env_from_unknown_domain;
				break;
			    case verify_domain_no_mail:
				/* RFC 7505:  "Null MX" No Service Resource Record 
				   for Domains That Accept No Mail
				*/
				ret =  env_from_no_mail_domain;
				break;
			    case verify_domain_bad_syntax:
				ret = env_from_bad_domain;
				break;
			    case verify_domain_ok:
				ret =  env_from_dns_verified;

				if (rewrite_env_from_domain(X,rewrite)) {
				    /* Be sure that pointer is valid */

				    cdr->domain_rewritten = 1;
				    
				    env_from_domain =  mailer_env_from_domain(X);
				    
				    DPRINT(Debug,12,(&Debug,
						     "env_from_classify_domain: Domain rewritten"));

				    if (env_from_domain) {
					DPRINT(Debug,12,(&Debug,", domain=%s",
							 env_from_domain));
				    }

				    DPRINT(Debug,12,(&Debug,"\n"));
				}
				break;
			    case verify_domain_not_available:
			    case verify_domain_fallback:
			    case verify_domain_failure:
				break;
			    }

			    if (main_cancel)
				free_cancel(&main_cancel);
			    if (rewrite) {
				free(rewrite);
				rewrite = NULL;
			    }
			    
			} else {
			    DPRINT(Debug,12,(&Debug,
					     "env_from_classify_domain: Do not verify local domain %s\n",
					     env_from_domain));
			}				  
		    }
		} else {
		    DPRINT(Debug,12,(&Debug,
				     "env_from_classify_domain: domain not verified",
				     env_from_domain));

		    if (cdr->is_literal) {
			DPRINT(Debug,12,(&Debug, ", literal %s",
					 env_from_domain));
		    }

		    if (cdr->whitelisted_name) {
			DPRINT(Debug,12,(&Debug, ", whitelisted %s",
					 cdr->whitelisted_name));
		    }
		    
		    DPRINT(Debug,12,(&Debug,"\n"));
		}
		
	    } else {
		DPRINT(Debug,8,(&Debug,
				"env_from_classify_domain: Domain %s not OK.\n",
				env_from_domain));
		
		ret = env_from_bad_domain;
	    }
	}

    }       

    DPRINT(Debug,8,(&Debug,
		    "env_from_classify_domain=%d",
		    ret));
    switch (ret) {
    case env_from_reset:           DPRINT(Debug,8,(&Debug," env_from_reset"));           break;
    case env_from_no_domain:       DPRINT(Debug,8,(&Debug," env_from_no_domain"));       break;
    case env_from_bad_domain:      DPRINT(Debug,8,(&Debug," env_from_bad_domain"));      break;
    case env_from_ok_domain:       DPRINT(Debug,8,(&Debug," env_from_ok_domain"));       break;
    case env_from_mailer_verified: DPRINT(Debug,8,(&Debug," env_from_mailer_verified")); break;
    case env_from_unknown_domain:  DPRINT(Debug,8,(&Debug," env_from_unknown_domain"));  break;
    case env_from_no_mail_domain:  DPRINT(Debug,8,(&Debug," env_from_no_mail_domain"));  break;
    case env_from_dns_verified:    DPRINT(Debug,8,(&Debug," env_from_dns_verified"));    break;
    }
    DPRINT(Debug,8,(&Debug,"\n"));
    
    return ret;
    
}   

/* Clear page and make 4 line prompt area */

struct menu_context *prompt_mailmsg_page_interactive(page)
     struct menu_context *page;
{
    struct menu_context *prompt_area = NULL;
    int LINES, COLUMNS;
    
    menu_ClearScreen(page);
    
    menu_get_sizes(page,&LINES, &COLUMNS);
    
    prompt_area = 
	new_menu_subpage(page,LINES-4,4,
			 subpage_simple_noredraw,NULL);

    return prompt_area;
}

void mail_env_from_presend(env_from_domain_class_p,
			   headers,env_from_cdr_p)
     enum rp_classify_domain       * env_from_domain_class_p;
     struct mailing_headers        * headers;
     struct classify_domain_result *  env_from_cdr_p;
{

    switch (*env_from_domain_class_p) {
	    
    case env_from_reset:
	/* Dummy */
	break;
	
    case env_from_no_domain:
    case env_from_ok_domain:
    case env_from_mailer_verified:
    case env_from_dns_verified:
	
	/* OK */
	break;
	
    case env_from_unknown_domain:   /*  Only give message, but send mail anyway */
	rp_classify_domain_message(headers->env_from,
				   *env_from_domain_class_p,
				   env_from_cdr_p);
	DPRINT(Debug,8,(&Debug,
			"mail_env_from_presend: domain of env_from is unknown -- trying send mail anyway\n"));
	break;
	    
    case env_from_no_mail_domain:
	rp_classify_domain_message(headers->env_from,
				   *env_from_domain_class_p,
				   env_from_cdr_p);
	DPRINT(Debug,8,(&Debug,
			"mail_env_from_presend: domain of env_from does not accept mail -- setting env_from to <>\n"));
	
	if (mailer_env_from_change(headers->env_from,"<>")) {
	    *env_from_domain_class_p = env_from_reset;
	}
	break;	    
	    
    case env_from_bad_domain:
	rp_classify_domain_message(headers->env_from,
				   *env_from_domain_class_p,
				   env_from_cdr_p);
	
	build_address_classify_message(mailer_env_from_value(headers->env_from,
							     NULL),
				 mailer_env_from_domain(headers->env_from),
				       env_from_cdr_p->is_literal,
				       env_from_cdr_p->whitelisted_name,
				       env_from_cdr_p->reserved_name);
	
	DPRINT(Debug,8,(&Debug,
			"mail_env_from_presend: bad env_from -- setting env_from to <>\n"));
	    
	if (mailer_env_from_change(headers->env_from,"<>")) {
	    *env_from_domain_class_p = env_from_reset;
	}
	break;	    
    }
}

static void mail_redraw_presend_screen P_((struct menu_context    * page,
					   struct mailing_headers * headers,
					   charset_t                hdr_charset,
					   int                      encode_hdr,
					   enum rp_classify_domain  env_from_domain_class
					   ));
static void mail_redraw_presend_screen(page,headers,hdr_charset,encode_hdr,
				       env_from_domain_class)
     struct menu_context    * page;
     struct mailing_headers * headers;
     charset_t                hdr_charset;
     int                      encode_hdr;
     enum rp_classify_domain env_from_domain_class;
{
    menu_ClearScreen(page);
    menu_StartXX(page,pg_BOLD);
    
    menu_print_format_center(page,0, 
			     CATGETS(elm_msg_cat, ElmSet,
				     ElmMailScreenTitle, 
				     "Mail Pre-Send Screen"));
    
    menu_EndXX(page,pg_BOLD);
    
    show_last_error();
    
    show_presend_headers(headers,hdr_charset, page,
			 show_prehdr_normal);
    
    
    if (headers->env_from)
	mail_show_env_from(headers,env_from_domain_class,page);
    
    
    menu_redraw_children(page);
  
}

static enum from_action {
    fa_fail     = -2  /* make_from_fail */,
    fa_EOF      = EOF /* EOF seen */,
    fa_none     = 0,
    fa_forget         /* forget sending mail */,
    fa_continue       /* continue sending / previewing */,
    fa_transmission   /* goto verify_transmission loop */,

} mail_check_default_from P_((struct mailing_headers * headers,
			      struct mailer_info     * mailer_info,
			      enum make_from_status    from_ok,
			      const char             * from_domain,
			      const char             * from_addr,
			      int                    * command_ch,
			      enum verify_trans_result code,
			      struct menu_context    * page,
			      struct menu_context    * prompt_area,
			      struct elm_commands    * snd_cmds,
			      mime_send_t            * MIME_info,
			      enum rp_classify_domain  env_from_domain_class,
			      char                  ** update_domain_p
			      ));
static enum from_action mail_check_default_from(headers,mailer_info,
						from_ok,from_domain,
						from_addr,command_ch,
						code,page,prompt_area,
						snd_cmds,
						MIME_info,
						env_from_domain_class,
						update_domain_p)
     
     struct mailing_headers * headers;
     struct mailer_info     * mailer_info;
     enum make_from_status    from_ok;
     const char             * from_domain;
     const char             * from_addr;
     int                    * command_ch;
     enum verify_trans_result code;
     struct menu_context    * page;
     struct menu_context    * prompt_area;
     struct elm_commands    * snd_cmds;
     mime_send_t            * MIME_info;
     enum rp_classify_domain  env_from_domain_class;
     char                  ** update_domain_p;
{
    enum from_action fa_reply = fa_continue;

    if (command_ch)
	*command_ch =  '\0';

    if (update_domain_p && *update_domain_p) {
	free(*update_domain_p);
	*update_domain_p = NULL;
    }
    
    switch (from_ok) {
    case make_from_localaddr:
	fa_reply = fa_continue;
	    
	if (from_addr) {
	    DPRINT(Debug,12,(&Debug,"From address %s is local (no domain)\n",
			     from_addr));
	}
	break;
	
    case make_from_normal:
	fa_reply = fa_continue;
	    
	if (from_addr) {
	    DPRINT(Debug,12,(&Debug,"From address %s is normal (domain %s).\n",
			     from_addr,
			     from_domain ? from_domain : "(?)"
			     ));
	}
	
	if (ft_forget == code ||
	    ft_forget_empty == code) {
	    
	    DPRINT(Debug,12,(&Debug,
			     "mail_check_default_from: %s: .. forget .. From address verify skipped.\n",
				     from_addr ? from_addr : "(?)"));
	} else if (from_domain &&
		   need_verify_ldomain(mailer_info,from_addr)) {
	    
	    int val = give_dt_enumerate_as_int(&verify_domain);
	    
	    DPRINT(Debug,12,(&Debug, 
			     "mail_check_default_from: %s: verify-domain =  %d%s, ",
			     from_addr ? from_addr : "(?)",
			     val,
			     val  < 0 ? ", uses shared library" : ""));
	    
	    if (val) {
		struct cancel_data  *main_cancel;
		enum verify_domain_result r;
		char *rewrite  = NULL;
		int def_cmd = 'c'; 

		struct elm_commands *cmds = NULL;
		struct elm_commands *all_cmds = NULL;
		int do_redraw = 0;
		int LINES, COLUMNS;
		int printed_prompt = 0;
		
		menu_get_sizes(page,&LINES, &COLUMNS);
    		
		DPRINT(Debug,12,(&Debug," verifying domain %s\n",
				 from_domain));
		main_cancel =
		    build_address_cancel_mode(from_domain);
		
		if (main_cancel) {
		    DPRINT(Debug,12,(&Debug, "  ... have cancel mode\n"));
		}

		r = verify_mail_domain_util(from_domain,&rewrite,&main_cancel);

		do {
		    menu_set_default(page);
		    
		    if (0) {
		    redraw:
			do_redraw = 1;
		    }
	
		resize_mark:
		    if (menu_resized(page)) {
			menu_get_sizes(page,&LINES, &COLUMNS);
			menu_subpage_relocate(prompt_area,page,LINES-4,4);
			do_redraw = 1;
			
		    } else if (menu_need_redraw(page))
			do_redraw = 1;
		    
		    if (do_redraw) {
			DPRINT(Debug,8,(&Debug,"mail_check_default_from: need redraw\n"));
			
			do_redraw = 0;
			
			mail_redraw_presend_screen(page,headers,
						   MIME_info->hdr_charset,
						   MIME_info->encode_hdr,
						   env_from_domain_class);
		    }
		    
		    switch (r) {
		    case verify_domain_not_found:
			if (from_addr) 
			    lib_error(CATGETS(elm_msg_cat, ElmSet, 
					      ElmMailAddrFNotFound,
					      "Domain %s of From address %s is unknown."),
				      from_domain,from_addr);
			def_cmd = 'c';
			goto prompt_it;
		    case verify_domain_no_mail:
			if (from_addr) 
			    lib_error(CATGETS(elm_msg_cat, ElmSet, 
					      ElmMailAddrFNotMail,
					      "Domain %s of From address %s accepts no mail."),
				      rewrite ? rewrite : from_domain,from_addr);
			def_cmd = 'V';
			goto prompt_it;
		    case verify_domain_bad_syntax:
			if (from_addr) 
			    lib_error(CATGETS(elm_msg_cat, ElmSet, 
					      ElmMailAddrFInValidDom,
					      "Invalid domain %s on From address %s."),
				      from_domain,from_addr);
			def_cmd = 'V';
			
		    prompt_it:
			
			if (prompt_area) {
			    int lin, col;
			    int cmd;

#ifdef BACKGROUD_PROCESSES      
			    if (handle_sigchld)
				sigchld_handler();
#endif
			    
			    if (menu_resized(prompt_area) ||
				menu_need_redraw(prompt_area)) {
				
				DPRINT(Debug,8,(&Debug,"mail_check_default_from:  clear or resize of prompt area\n"));
			    }
			    
			    if (!cmds)
				cmds = give_checkfrom_commands();
			    if (!all_cmds)
				all_cmds = give_merged_commands(cmds,snd_cmds,0,0);
			    menu_ClearScreen(prompt_area);
			    
			    if (ft_send == code)
				menu_print_format_center(prompt_area,1,
							 CATGETS(elm_msg_cat, ElmSet, 
								 ElmMcdfPromptLine1,
								 "Domain of default From: address may be inoperative. Confirm sending mail."));
			    else
				menu_print_format_center(prompt_area,1,
							 CATGETS(elm_msg_cat, ElmSet, 
							     ElmMcdfPromptLine1a,
								 "Domain of default From: address may be inoperative. Confirm action."));
			    
			    menu_print_format_center(prompt_area,2,
						     CATGETS(elm_msg_cat, ElmSet, 
							     ElmMcdfPromptLine2,
							     "'%c' or 'c' confirms (continues action). '%c' or 'V' cancels. ? = help."),
						     *def_ans_yes,*def_ans_no);
			    
			    menu_PutLineX(prompt_area,
					  0, 0, CATGETS(elm_msg_cat, ElmSet, 
							ElmMcdfPromptText,
							"Continue send or preview message? (%c/%c/f) %c"),
					  *def_ans_yes,*def_ans_no,def_cmd);
			    
			    menu_GetXYLocation(prompt_area,&lin,&col);
			    col--; 
			    
			    menu_CleartoEOLN(prompt_area);
			    show_last_error();
			    
			    FlushBuffer();
			    
			    menu_MoveCursor(prompt_area,lin, col);
			    cmd = menu_ReadCh(prompt_area,REDRAW_MARK|READCH_resize);
			    
			    fa_reply = fa_none;
			    
			    if (isascii(cmd) && 
				isprint(cmd)) {
				DPRINT(Debug,8,(&Debug,"mail_check_default_from: ... command: '%c'\n",
						cmd));
			    } else {
				DPRINT(Debug,8,(&Debug,"mail_check_default_from: ... command: %d\n",
						cmd));
			    }
			    
			    if (cmd == REDRAW_MARK) {
				DPRINT(Debug,4, (&Debug,"mail_check_default_from:       .... redrawing\n"));
				menu_ClearScreen(page);   /* Clear possible redraw mark */
				
				/* Call refresh routines of children */
				menu_redraw_children(page);
				
				if (menu_need_redraw(prompt_area))		    
				    menu_ClearScreen(prompt_area);   /* Clear redraw mark from prompt_area*/
				
				goto redraw;
			    }
			    
			    if (cmd == RESIZE_MARK) {
				
				DPRINT(Debug,10, (&Debug,"mail_check_default_from:       .... resizing\n"));
				goto resize_mark;
			    }
			    
			    if (cmd == EOF) {
				fa_reply = fa_EOF;
				goto quit_prompt;
			    }
			    
			    clear_error();
			    printed_prompt = 1;
			    
			    /* Put cursor pack to correct place */
			    menu_MoveCursor(prompt_area,lin,col);

			got_command:
			    if (cmd == *def_ans_yes) {
				if (printed_prompt) {
				    menu_Write_to_screen(prompt_area,
							 CATGETS(elm_msg_cat, ElmSet, ElmYesWord, 
								 "Yes."));
				    FlushBuffer();
				}
				
				fa_reply = fa_continue;
				goto quit_prompt;
			    }
			    if (cmd == *def_ans_no) {
				if (printed_prompt) {
				    menu_Write_to_screen(prompt_area,
							 CATGETS(elm_msg_cat, ElmSet, ElmNoWord, 
								 "No."));
				    FlushBuffer();
				}
				fa_reply = fa_transmission;
				goto quit_prompt;
			    }

			    switch (cmd) {
			    case '\n':
			    case '\r':
				cmd = def_cmd;
				
				if (isascii(cmd) && 
				    isprint(cmd)) {
				    DPRINT(Debug,10,(&Debug,"mail_check_default_from: ... default command: '%c'\n",
						     cmd));
				} else {
				    DPRINT(Debug,10,(&Debug,"mail_check_default_from: ... default command: %d\n",
						     cmd));
				}
			    }

			    switch (cmd) {

			    case '?':
				if (printed_prompt) {
				    menu_Write_to_screen(prompt_area,
							 FRM("?"));
				    FlushBuffer();
				}

				cmd = help_generic(all_cmds,TRUE,page,prompt_area);

				if (menu_need_redraw(page)) {
				    menu_ClearScreen(page);
				    
				    do_redraw = 1;
				}
				
				if (cmd == EOF) {
				    fa_reply = fa_EOF;
				    goto quit_prompt;
				}
				
				if (cmd) {
				    printed_prompt = 0;
				    goto got_command;
				}
				break;
				
			    case 'c':
				if (printed_prompt) {
				    menu_Write_to_screen(prompt_area,
							 CATGETS(elm_msg_cat, ElmSet,
								 ElmMcdfContinue,
								 "Continue"));
				    FlushBuffer();
				}
				
				fa_reply = fa_continue;
				goto quit_prompt;
			    case 'f':
				if (printed_prompt) {
				    menu_Write_to_screen(prompt_area,
							 CATGETS(elm_msg_cat, ElmSet,
								 ElmMcdfForget,
								 "Forget message"));
				    
				    FlushBuffer();
				}
				fa_reply = fa_forget;
				goto quit_prompt;

			    case ctrl('L'):
				goto redraw;
				
			    case 'V': 
				if (printed_prompt) {
				    menu_Write_to_screen(prompt_area,
							 CATGETS(elm_msg_cat, ElmSet, 
								 ElmMcdfPreSendScreen,
								 "Go to Mail Pre-Send Screen"));
				    FlushBuffer();
				}
				
				if (command_ch)
				    *command_ch =  '\0';
				fa_reply = fa_transmission ;
				goto quit_prompt;

			    default: {
				
				const struct elm_command * snd_cmd =
				    lookup_elm_command(snd_cmds,cmd);
				
				if (snd_cmd && command_ch) {
				    if (printed_prompt) {
					const struct string * S =
					    elm_command_key_desc(snd_cmd);
					
					if (S) {
					    /* uses description as prompt text! */
					    menu_Write_to_screen(prompt_area,
								 FRM("%S"),
								 S);
					    FlushBuffer();
					}
				    }				    
				    *command_ch = cmd;
				    fa_reply = fa_transmission ;
				    goto quit_prompt;
				} else {

				    DPRINT(Debug,10,(&Debug,
						     "mail_check_default_from: command %d not found\n",
						     cmd));
				    
				    menu_Write_to_screen(prompt_area,
							 FRM("%c??"), 07);
				    FlushBuffer();
				    if (sleepmsg > 0)
					error_sleep((sleepmsg + 1) / 2);

				}
				
			    }
				break;
				
			    }
			}
			break;
			
		    case verify_domain_ok:

			if (update_domain_p && rewrite) {

			    DPRINT(Debug,10,(&Debug,
					     "mail_check_default_from: rewrite domain %s to %s\n",
					     from_domain,rewrite));
			    *update_domain_p = rewrite;
			    rewrite = NULL;
			}

			break;
		    case verify_domain_not_available:
		    case verify_domain_fallback:
		    case verify_domain_failure:
			break;
		    }
		    
		} while (fa_none == fa_reply);

	    quit_prompt:
		if (main_cancel)
		    free_cancel(&main_cancel);
		
		if (rewrite) {
		    free (rewrite);
		    rewrite = NULL;
		}

		if (all_cmds)
		    free_commands(&all_cmds);
		if (cmds)
		    free_commands(&cmds);
	    } else {
		DPRINT(Debug,12,(&Debug," verify skipped\n"));
	    }
	    
	} else {
	    DPRINT(Debug,12,(&Debug, 
			     "mail_check_default_from: %s: From address verify not used\n",
			     from_addr ? from_addr : "(?)"));
	}
	break;
	
    case make_from_literal:
	fa_reply = fa_continue;
	    
	if (from_addr) {
	    DPRINT(Debug,12,(&Debug,"mail_check_default_from: From address %s have literal domain (domain %s),\n",
			     from_addr,
			     from_domain ? from_domain : "(?)"
			     ));
	}
	break;

    case make_from_whitelisted:
	fa_reply = fa_continue;
	
	if (from_addr) {
	    DPRINT(Debug,12,(&Debug,"From address %s have whitelisted domain (domain %s).\n",
			     from_addr,
			     from_domain ? from_domain : "(?)"
			     ));
	}
	break;
	
    case make_from_fail:
	fa_reply = fa_fail;

	DPRINT(Debug,12,(&Debug,"mail_check_default_from: Failed to add From address."));
	
	if (from_addr) {
	    DPRINT(Debug,12,(&Debug," %s",
			     from_addr));
	}
	DPRINT(Debug,12,(&Debug,".\n"));
	
	if (OPMODE_IS_INTERACTIVE(opmode) &&
	    (ft_forget != code &&
	     ft_forget_empty != code)) {		    
	    
	    if (from_domain)
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmMailAddrFailDomain,
				  "Failed to add From address (domain %s); Use f)orget."),
			  from_domain);
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmMailAddrFail1,
				  "Failed to add From address; Use f)orget."));
	}

	break;
    }


    DPRINT(Debug,12,(&Debug,"mail_check_default_from=%d",
		     fa_reply));
    switch (fa_reply) {
    case fa_fail:         DPRINT(Debug,12,(&Debug," fa_fail"));         break;
    case fa_EOF:          DPRINT(Debug,12,(&Debug," fa_EOF"));          break;
    case fa_none:         DPRINT(Debug,12,(&Debug," fa_none"));         break;
    case fa_forget:       DPRINT(Debug,12,(&Debug," fa_forget"));       break;
    case fa_continue:     DPRINT(Debug,12,(&Debug," fa_continue"));     break;
    case fa_transmission: DPRINT(Debug,12,(&Debug," fa_transmission")); break;
    }
    if (command_ch && *command_ch) {
	DPRINT(Debug,8,(&Debug,"; *command_ch=%d",
			*command_ch));
    }
    if (update_domain_p && *update_domain_p) {
	DPRINT(Debug,8,(&Debug,"; *update_domain_p=%s",
			*update_domain_p));
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return fa_reply;
}


static void digest_feed_opts P_((struct digest_proc * ret, const struct mime_param *opts));
static void digest_feed_opts(ret,opts)
     struct digest_proc * ret;
     const struct mime_param  *opts;
{
    if (opts) {
	char * X = encode_mime_params(opts);
	if (X) {
	    digest_feed_str(ret,"; ");
	    digest_feed_str(ret,X);
	    free(X);
	}
    }    
}

static struct digest_proc * count_message_hash P_((FILE * converted_buffer,
						   mime_send_t *mime_info,
						   struct mailing_headers * headers,
						   const struct digest_proc_type * hash_type));
static struct digest_proc * count_message_hash(converted_buffer,mime_info,headers,hash_type)
    FILE * converted_buffer;
    mime_send_t *mime_info;
    struct mailing_headers * headers;
    const struct digest_proc_type * hash_type;
{
    int i;

    size_t res_len;
    
    struct digest_proc * ret = malloc_digest_proc(hash_type);

    mime_info -> encode_hdr = !allow_no_hdrencoding;
    
    feed_common_mailing_headers(ret,headers, mime_info -> encode_hdr,mime_info->hdr_charset);

  

    /* MIME-Version not counted (constant header) */
    
    if (mime_info->msg_is_multipart) {
	
	digest_feed_str(ret,"Content-Type: multipart/mixed");
	digest_feed_opts(ret,mime_info->TYPE_opts_top);
	digest_feed_str(ret,"\n");

	digest_feed_str(ret,"Content-Transfer-Encoding: ");
	digest_feed_str(ret,ENCODING(mime_info->encoding_top));
	digest_feed_str(ret,"\n");	
    }
    
    if (mime_info->msg_is_multipart) {
	digest_feed_str(ret,"\n");
    }
    
    for (i = 0; i < mime_info->top_parts_count; i++) {
	struct mime_send_part * part = &(mime_info->top_parts[i]);
	long pos1;
	int line_len;
	char buffer[SLEN];			/* file reading buffer */

	if (mime_info->msg_is_multipart) {
	    digest_feed_str(ret,"\n--");
	    digest_feed_str(ret,mime_info->mime_boundary);
	    digest_feed_str(ret,"\n");
	}
	
	if (part->encoding_part < ENCODING_EXPERIMENTAL &&
	    part->encoding_part > ENCODING_NONE) {
	    digest_feed_str(ret,"Content-Transfer-Encoding: ");
	    digest_feed_str(ret,ENCODING(part->encoding_part));
	    digest_feed_str(ret,"\n");
	} else if (part->encoding_part_text) {
	    digest_feed_str(ret,"Content-Transfer-Encoding: ");
	    digest_feed_str(ret,part->encoding_part_text);
	    digest_feed_str(ret,"\n");
	}

	digest_feed_str(ret,"Content-Type: ");
	digest_feed_str(ret,get_major_type_name(part->TYPE));
	digest_feed_str(ret,"/");
	digest_feed_str(ret,get_subtype_name(part->TYPE));
	digest_feed_opts(ret,part->TYPE_opts_part);
	digest_feed_str(ret,"\n");

	if (part->disposition != DISP_INLINE ||
	    part->DISPOSITION_opts) {

	    digest_feed_str(ret,"Content-Disposition: ");
	    digest_feed_str(ret,DISPOSITION(part->disposition));
	    digest_feed_opts(ret,part->DISPOSITION_opts);
	    digest_feed_str(ret,"\n");
	}

	if (part->description) {
	    digest_feed_str(ret,"Content-Description: ");
	    digest_feed_string(ret,part->description);
	    digest_feed_str(ret,"\n");
	}
	digest_feed_str(ret,"\n");
	
	if (0 != fseek(converted_buffer,part->start_loc,SEEK_SET)) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSeekFailed,
			      "ELM [seek] failed trying to read %d bytes into file."),
		      (int) part->start_loc);
	    
	    break;
	}

	while ((pos1 = ftell(converted_buffer)) < part->end_loc &&
	       (line_len = mail_gets(buffer, sizeof buffer -1, converted_buffer))) {
	    
	    /* Text part necessary do not end to \n so there is possibility to
	       get part of next part 
	    */
	    
	    if (pos1 + line_len > part->end_loc) {
		int old = line_len;
		
		line_len = part->end_loc - pos1;
		
		DPRINT(Debug,14,(&Debug,
				 "count_message_hash: Got past of end of attachment -- changing line_len %d -> %d\n",
				 old,line_len));
		if (line_len < 0 || line_len > old)
		    mime_panic(__FILE__,__LINE__,
			       "count_message_hash",
			       "line_len calculation error");
		
		if (0 == line_len)
		    break;
	    }

	    feed_digest_proc(ret,s2us(buffer),line_len);	    
	}

	if (! mime_info->msg_is_multipart)
	    break;
    }

    if (mime_info->msg_is_multipart) {
	digest_feed_str(ret,"\n--");
	digest_feed_str(ret,mime_info->mime_boundary);
	digest_feed_str(ret,"--\n");
    }
    
    res_len = len_digest_proc(ret);

    if (res_len) {
	unsigned char *res = safe_malloc(res_len);

	res_len = end_digest_proc(ret,res,res_len);

	if (res_len) {
	    size_t x;
	    
	    DPRINT(Debug,14,(&Debug,"count_message_hash: length %zu [",
			     res_len));
	    
	    for (x = 0; x < res_len; x++) {
		DPRINT(Debug,14,(&Debug,"%02x",res[x]));
	    }
	    DPRINT(Debug,14,(&Debug,"]\n"));
	}
		
	free(res);
    }
        
    DPRINT(Debug,14,(&Debug,"count_message_hash=%p\n",ret));

    return ret;
}

#if ANSI_C
static default_xmailer_cb  default_xmailer;
#endif
static struct string * default_xmailer P_((void));
static struct string * default_xmailer() 
{
    if (dt_flag_is_set(&program_snd_ident,snd_ident_version))	
	return format_string(FRM("ELM [version %s]"), version_buff);
    else
	return format_string(FRM("ELM"));
}


/* Returns 0 if reading of body failed */
int mail(mail_index,options, headers,mailer_info,
	 mailbox,aview, parent_page,
	 selection_view,selection_idx,
	 body,
	 exit_status,pgp_status)
     int mail_index;
     int options;
     struct mailing_headers * headers;
     struct mailer_info *mailer_info;     
     struct MailboxView *mailbox /* for mail_index */;
     struct AliasView *aview;
     struct menu_context  *parent_page;
     struct MailboxView *selection_view;
     int selection_idx;
     struct text_block *body;
     int *exit_status;               /* Only used on send only mode */
     int pgp_status;
{
    static const time_t DATE_REGENERATION_TIME = 5;

    int replying      = 0 != (options & MAIL_REPLYING);
    int copy_msg      = 0 != (options & MAIL_COPY_MSG);
    int forwarding    = 0 != (options & MAIL_FORWARDING);
#if 0
    int mime_attach   = 0 != (options & MAIL_MIME_ATTCH);
#endif
    int copy_selection = 0 != (options & MAIL_COPY_SELECTION); 
    int delete_cancel  = 0 != (options & MAIL_DELETE_CANCEL);

    struct Attachments attachments = NULL_Attachments;

    /** Given the addresses and various other miscellany (specifically, 
	'copy-msg' indicates whether a copy of the current message should 
	be included, (options & MAIL_EDIT_MSG) indicates whether the message should 
	be edited) this routine will invoke an editor for the user and 
	then actually mail off the message. 

	In redraw is needed use 
	       use menu_trigger_redraw(parent_page)
    **/

    FILE *reply = NULL;
    int command_ch = '\0';

    int  already_has_text = 0;		/* we need an ADDRESS */
    int	 signature_done   = 0;
    int	 err;
    int reask_verify = 0;

    int dsn = 0;
    mime_send_t MIME_info;
    
    FILE * converted_buffer = NULL;      /* Temp file which hold converted
					    body parts -- now include mime
					    headers however ...
					 */
    static int conv_count = 0;
    char * temp1 = NULL;

    char title[80];
    enum verify_trans_result code = ft_none;
    struct copy_file COPY_FILE;

    charset_t   cur_editcharset = NULL;
    
    struct menu_context *page = NULL;
    struct menu_context *prompt_area = NULL;
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
    int ret = 0;

    const struct remote_server  * remote_server = NULL;
    struct mailer_cancel_ticket * cancel_ticket = NULL;
    struct sending_display      * sending_display = NULL;

    int Exstat = 0;   /* OK */
    int restart_loop_count = 0;
    struct elm_commands         * snd_cmds = NULL;
    struct address              * usraddr = NULL;
    int mail_files_included = 0;
    int                           usraddr_flags = 0;

    enum rp_classify_domain env_from_domain_class   = env_from_no_domain;
    struct classify_domain_result    env_from_cdr   = NULL_classify_domain_result;
    int have_reconnect = 0;

    int local_options = 0;
 
    if (!tmp)
	return 0;

    /* Fail if no mailer available */
    if (!mailer_info)
	return 0;

    /* Initialize structure */
    clear_mime_send_info(&MIME_info);

    /* free_mailing_headers will free this */
    if (!headers->env_from)
	headers->env_from = mailer_get_env_from(mailer_info);

    if (dt_flag_is_set(&program_snd_ident,snd_ident_x_mailer) &&
	!headers->xmailer) {

	headers->xmailer = default_xmailer();
    }
   
    cancel_ticket = new_mailer_cancel_ticket();
    snd_cmds      =  give_send_commands();

    zero_copy_file(&COPY_FILE);

    dump_expanded_address(4,"Mailing to",headers->to);

    if (0 != (options & MAIL_EDIT_MSG)) {
	
	command_ch = 'e';
	DPRINT(Debug,4, (&Debug,"   (with editing)"));
    }
    DPRINT(Debug,4, (&Debug,"\n"));

    if (OPMODE_IS_SENDMODE(opmode)) {
	mail_files_included = inline_parts_from_mail_files(&attachments,
							   mailer_info);
    } else if (exit_status) {
	/* Maybe url_mode ? */
	mail_files_included = inline_parts_from_mail_files(&attachments,
							   mailer_info);
    }
        
    gotten_key = 0;		/* ignore previously gotten encryption key */
       
    /* verify_transmission() displays new screen,
       we create context for it on here
    */
       
    page = new_menu_context();
    if (OPMODE_IS_INTERACTIVE(opmode))
	/* Clear page and make 4 line prompt area */
	prompt_area = prompt_mailmsg_page_interactive(page);
    

    reply = generate_edit_buffer(body,&Exstat,
				 &already_has_text,
				 &cur_editcharset);
    if (!reply)
	goto fail_label;


    handle_mailer_options(mailer_info,&MIME_info,&dsn, 
			  &attachments, NULL);

    if (mailbox) {
	remote_server = give_message_remote_server(mailbox,mail_index);

	DPRINT(Debug,8,(&Debug,
			" ... have mailbox (%s remote server), mail index #%d\n",
			remote_server ? "have" : "no",
			mail_index));
    }

    if (remote_server) {
	
	usraddr = give_remote_server_useraddr(remote_server,&usraddr_flags);

	DPRINT(Debug,8,(&Debug,
			" ... have remote server (%s user address)\n",
			usraddr ? "have" : "no"));

    }

    if ((mailbox && mail_index >= 0 &&
	 (copy_msg || forwarding)) 
	|| 
	copy_selection) {
       
	struct MailboxView *  W  = mailbox;
	int                   Wi = mail_index;

	if (copy_selection) {
	    W  = selection_view;
	    Wi = selection_idx;
	}

	if (! copy_message_to_buffer(reply,&MIME_info,
				     &cur_editcharset,
				     &already_has_text,
				     &attachments,
				     headers,
				     W,Wi,options)) {
	    goto fail_label;
	}				            			     			      
    }
    
    if (!cur_editcharset)
	cur_editcharset = display_charset;

    if (attach_files.attachment_count > 0) {
	int i;

	for (i = 0; i < attach_files.attachment_count; i++) {
	    add_Attachments(& attachments,&attach_files.attach_files[i]);
	    /* add_Attachments zeros attach_files.attach_files[i],
	       so array can be free'ed later 
	    */
	}
	free_Attachments(&attach_files); /* Actually only frees array */
    }

    /* append signature now if we are going to use an external editor */
    /* Don't worry about the remote/local determination too much */
    
    {
	unsigned int editor_keyword = 0, editor_mask = 0;
	const char * editor_val = give_dt_estr_as_str(&editor_e,"editor",
						      &editor_keyword,
						      &editor_mask);

	if (already_has_text || 
	    (editor_val  &&
	     editor_kw_NO == editor_keyword)) {
	    signature_done = TRUE;

	    already_has_text |= append_sig(reply, headers, page);
	}
    }

    /** Edit the message **/

    /* calculate default save_file name */
    if (COPY_FILE.copy_file)
	clear_copy_file(&COPY_FILE);        /* signals to not save a copy */
    
    if(auto_copy_sent) {
	if(save_by_name || save_by_alias) {
	    if(force_name) {
		/* signals save by 'to' logname */
		COPY_FILE.copy_file = format_string(FRM("="));
	    } else {
		/* conditional save by 'to' logname */
		COPY_FILE.copy_file = format_string(FRM("=?"));
	    }
	} else {
	    /* signals save to sentmail */
	    COPY_FILE.copy_file = format_string(FRM("<"));
	}
    } 
    
    if (use_PGP &&
	mailbox && mail_index >= 0) {
	struct header_rec *hdr =  give_header(mailbox,mail_index);
	if (hdr &&
	    hdr->pgp & PGP_MESSAGE)
	    options |= MAIL_HAVE_PGP_ENCODED;
    }

    /* Now .mailheaders is read before mail is composed! */
    if (!copy_selection) {
	import_mailheaders(headers,import_mh_normal);

	dump_expanded_address(8,"after import -- from ",headers->from);
	dump_expanded_address(8,"after import -- to   ",headers->to);
	dump_expanded_address(8,"after import -- cc   ",headers->cc);
	dump_expanded_address(8,"after import -- bcc  ",headers->bcc);
	dump_expanded_address(8,"after import -- reply-to ",headers->reply_to);

    }

    if (headers->env_from)  {
	env_from_domain_class =
	    env_from_classify_domain(headers->env_from,
				     &env_from_cdr,
				     mailer_info);

    } else {
	DPRINT(Debug,8,(&Debug," ... no envelope sender (Return-Path)\n"));
    }
	   
    if (usraddr_flags) {
	if (0 != (usraddr_flags & USERADDR_SELF_CC)) {
	    DPRINT(Debug,8,(&Debug,
			    " ... self-cc \n"));
	    add_self_addr(& (headers->cc),usraddr,mailer_info,
			  & (headers->from),
			  "Cc");
	}
	
	if (0 != (usraddr_flags & USERADDR_SELF_BCC)) {
	    DPRINT(Debug,8,(&Debug,
			    " ... self-bcc \n"));
	    add_self_addr(& (headers->bcc),usraddr,mailer_info,
			  & (headers->from),
			  "Bcc");
	}
	
	dump_expanded_address(8,"after useraddr flags -- cc   ",headers->cc);
	dump_expanded_address(8,"after useraddr flags -- bcc  ",headers->bcc);
    }

    
    if (mail_files_included) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmSendingMailFilesIncluded,
			  "Sending mail... %d mail files from command line included as separate parts."),
		  mail_files_included);	
    }

    
    if (0) {
    restart_verify_transmission:
	++restart_loop_count;
	DPRINT(Debug,8,(&Debug,"... restart verify_transmission, count=%d\n",
			restart_loop_count));
	if (restart_loop_count > 50) {

#ifdef  EX_UNAVAILABLE
	    Exstat = EX_UNAVAILABLE;  /* Give better error status */
#endif
	    goto fail_label;	
	    
	}
    }

 verify_trans_menu:
    
    do {      /* verify_transmission loop */
	
	if (sending_display) {
	    free_sending_display(&sending_display);
	    menu_trigger_redraw(page);
	}

	if (reply != NULL) {
	    int r = fclose(reply);	
	    reply = NULL;

	    if (EOF == r) {
		int err = errno;
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmWriteFailedCloseFile,
				  "Write failed when closing file %s: %s"),
			  cur_editfile, strerror(err));
			  
#ifdef  EX_IOERR
		Exstat = EX_IOERR;  /* Give better error status */
#endif
			  
		goto fail_label;
	    }
	}

	code = ft_none;
	reask_verify = 0;
	
	/* ask the user to confirm transmission of the message */
	if (OPMODE_IS_INTERACTIVE(opmode)) {
	    
	    struct header_rec *parent_message = NULL;
	    
	    menu_set_default(page);
	    
	    if (replying && mailbox && mail_index >= 0) 
		parent_message =  give_header(mailbox,mail_index);
	    
	    if (command_ch) {
		if (isascii(command_ch) && 
		    isprint(command_ch)) {
		    DPRINT(Debug,8,(&Debug,"... command: '%c'\n",
				    command_ch));
		} else {
		    DPRINT(Debug,8,(&Debug,"... command: %d\n",
				    command_ch));
		}
	    }

	    code = verify_transmission(cur_editfile, 
				       already_has_text, &COPY_FILE, 
				       command_ch,
				       options | local_options,
				       &dsn,
				       headers, &attachments,
				       cur_editcharset,
				       mailer_info,
				       &MIME_info,
				       mailbox,
				       aview,
				       page,
				       prompt_area,
				       parent_message,
				       remote_server,
				       &pgp_status,
				       snd_cmds,
				       &env_from_domain_class,
				       &env_from_cdr);	    
	    command_ch = '\0';  /* Reset command anyway */
	    
	    DPRINT(Debug,8,(&Debug,
			    " ... verify_transmission result = %d, attachment_count=%d\n",
			    code,attachments.attachment_count));
	    
	    switch(code) {
	    case ft_forget_empty:
		DPRINT(Debug,8,(&Debug," ... ft_forget_empty\n"));
		/* Empty ... */
		if (!attachments.attachment_count) 
		    goto fail_label;
		/* If attachments, then need still save message */
		break;
	    case ft_EOF:
		DPRINT(Debug,8,(&Debug," ... ft_EOF\n"));
		goto fail_label;
	    case ft_forget:   /* Need still save message */
	    case ft_none:		
	    case ft_send:
	    case ft_preview:
	    case ft_preview_menu:
	    case ft_preview_mime:
	    case ft_preview_hdrs:
		break;
	    }
	} else {
	    DPRINT(Debug,8,(&Debug,"Non interactive -- assuming 'send' \n"));

	    code = ft_send;
	}

	local_options = 0;  /* No MAIL_FORGET */
	
	err = can_open(cur_editfile,"r+");
	if (err) {
	    DPRINT(Debug,1,(&Debug,  
			    "Attempt to open file %s for reading failed with error %s (mail) -- can_open\n",
			    cur_editfile, strerror(err)));
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldNotOpenReply,
			      "Could not open reply file (%s)."), 
		      strerror(err));

	    goto fail_label;
	}

	if ((reply = fopen(cur_editfile,"r+")) == NULL) {
	    err = errno;
	    DPRINT(Debug,1,(&Debug,  
			    "Attempt to open file %s for reading failed with error %s (mail)\n",
			    cur_editfile, strerror(err)));
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldNotOpenReply,
			      "Could not open reply file (%s)."), 
		      strerror(err));

	    goto fail_label;
	}

	/* Append signature if not done earlier, so that need_eccoding
	 * takes account of 8-bit data on signature
	 */
	
	if (!signature_done) {
	    /* Go to the end of the file! */
	    fseek (reply, 0, 2);
	    append_sig(reply, headers, page);
	    rewind(reply);
	    signature_done = TRUE;
	}
	
	if (code >= ft_send) { /* If not canceled */
	    enum restart_mailer_status r = mailer_restarted(mailer_info);

	    switch (r) {
	    case mailer_reinitialized:
		have_reconnect = 1;
		handle_mailer_options(mailer_info,&MIME_info,&dsn, 
				      &attachments, &reask_verify);
		break;

	    case mailer_disconnected:
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailerDisconnected,
				  "Mailer disconnected! Can't send message. Use f)orget !"));
		local_options = MAIL_FORGET;
				
		reask_verify = TRUE; /* Go to verify_transmission again. */
		break;
		
	    case mailer_not_restarted:
		break;
	    }
	}

	MIME_info.encoding_top  = ENCODING_7BIT;/* Encoding for Multipart/ */
	  	 
#if 0	  
	if (allow_no_hdrencoding && headers->subject)
	    MIME_info.need_enc |= check_8bit_string (headers->subject);
#endif
	
	if (check_for_multipart(reply, &MIME_info,mailer_info,
				cur_editcharset, remote_server) < 0) { 
	    /* Error in [include ...] or other [keyword] */
	    if (code < ft_send)
		goto fail_label;

	    if (OPMODE_IS_INTERACTIVE(opmode)) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFixIncludeKeyword,
				  "Please fix [include ...] and other [keyword] lines!"));
		reask_verify = TRUE; /* Go to verify_transmission again. */
		continue;
	    }
	}

	if (converted_buffer)
	    fclose(converted_buffer);
	temp1 = elm_message(FRM("%selmcv-%d-%d"),
			    tmp, getpid (),
			    conv_count++);
	converted_buffer = safeopen_rdwr(temp1,NULL);
	if (!converted_buffer) {

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmOpenFailedCopy,
			      "Temp file open failed to in copy"));


	    free(temp1);
	    temp1 = NULL;

#ifdef  EX_CANTCREAT
	    Exstat = EX_CANTCREAT; /* Give better error status */
#endif

	    if (code < ft_send ||
		! OPMODE_IS_INTERACTIVE(opmode))
		goto fail_label;
	    
	    reask_verify = TRUE; /* Go to verify_transmission again. */
	    continue;
	}
	DPRINT(Debug,2,(&Debug,   "mail: using temp file (%s)\n",
			temp1));
	unlink(temp1);
	free(temp1);
	temp1 = NULL;

	if (!convert_text(reply,converted_buffer,
			  &MIME_info,
			  cur_editcharset,text_charset,
			  pgp_status,
			  &attachments,
			  mailer_info,
			  page,prompt_area,
			  remote_server)) {

#ifdef  EX_DATAERR
	    Exstat = EX_DATAERR;  /* Give better error status */  
#endif

	    if (code < ft_send ||
		! OPMODE_IS_INTERACTIVE(opmode))
		goto fail_label;

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedConvertMessage,
			      "Failed convert message, fix errors and retry or f)orget it."));
	    reask_verify = TRUE; /* Go to verify_transmission again. */
	} else
	    MIME_info.msg_is_multipart = MIME_info.top_parts_count > 1;

	
	DPRINT(Debug,8,(&Debug,
			" ... top_parts_count=%d, msg_is_multipart=%d\n",
			MIME_info.top_parts_count,
			MIME_info.msg_is_multipart));


	/* End of check_for_multipart failure loop */
    } while (OPMODE_IS_INTERACTIVE(opmode) && reask_verify);

       
    if (headers->env_from)  
	mail_env_from_presend(&env_from_domain_class,
			      headers,&env_from_cdr);
    
    
    if (MIME_info.msg_is_multipart &&
	!MIME_info.mime_boundary[0]) {
	(void) mime_generate_boundary (MIME_info.mime_boundary,
				       sizeof MIME_info.mime_boundary);
	mime_params_add_compat(&(MIME_info.TYPE_opts_top), 
			       "boundary",
			       MIME_info.mime_boundary);

	DPRINT(Debug,8,(&Debug,
			" ... mime boundary: %s\n",MIME_info.mime_boundary));
    }

    MIME_info.encode_hdr = !allow_no_hdrencoding;

    sndhelper_update_user_agent(headers,MIME_info.hdr_charset,MIME_info.encode_hdr,default_xmailer);
    
    if (headers->sender) {
	DPRINT(Debug,6,(&Debug,
			"Got Sender address: %s\n",
			headers->sender));

	if (! headers->from.addrs) {
	    DPRINT(Debug,8,(&Debug," ... resetting Sender: %s\n",
			    headers->sender));

	    free(headers->sender);
	    headers->sender = NULL;
	    goto no_sender;
	}
	
    } else {

    	enum add_sender_v add_sender_h;
	char * sender_addr;
	char * sender_domain;
	int    sender_domain_verify;
	int from_addr_len;
	int sender_ok;
	enum sndhelper_status helperstatus;
	
    no_sender:
	/* No headers->sender set yet */

    	add_sender_h =
	    give_dt_enumerate_as_int(&add_sender_header);
	sender_addr    = NULL;
	sender_domain  = NULL;
	sender_domain_verify   = 0;
	from_addr_len      = 0;
	sender_ok = 1;

	DPRINT(Debug,12,(&Debug,"add-sender = %d\n",
			 add_sender_h));
	
	if (headers->from.addrs &&
	    (from_addr_len = addr_list_item_count(headers->from.addrs)) > 0) {
	    
	    DPRINT(Debug,6,(&Debug,
			    "From address already specified by user (len %d)\n",
			    from_addr_len));
	    
	    dump_expanded_address(6,"From address",headers->from);


	    sndhelper_get_sender(add_sender_h,mailer_info,&sender_domain,
				 &sender_domain_verify,&sender_addr,
				 from_addr_len);
				   	   	    
	    if (usraddr) {
		DPRINT(Debug,6,(&Debug,
				"From address from hashmark ignored!\n"));
	    }
	    
	} else if (usraddr) {
	    
	    DPRINT(Debug,6,(&Debug,
			    "From address from hashmark (by user or system\n"));
	    
	    ADD_EXPANDED0((headers->from),usraddr);

	    if (headers->from.addrs)
		from_addr_len = addr_list_item_count(headers->from.addrs);
	    else {
		DPRINT(Debug,12,(&Debug,"No From addresses\n"));
	    }
	    	    
	    dump_expanded_address(6,"From address",headers->from);

	    switch (add_sender_h) {
	    case add_sender_auto:
		sender_domain_verify = query_mailer_info(mailer_info,
						  MI_REMOTE_MAILER);

		if (sender_domain_verify) {
		    DPRINT(Debug,9,(&Debug,
				    "mail: remote mailer, Sender address domain need to be verified before adding\n"));
		}
		
		/* FALLTHRU */
	    case add_sender_yes:

		/* Result is malloced */		
		sender_addr = from_addr_literal(mailer_info,
						    &sender_domain);
		break;
		
	    case  add_sender_no:
	    case NUM_add_sender: /* not used */ break;
	    }
	
	} else if (query_mailer_info(mailer_info,MI_DONT_ADD_FROM)) {
	    
	    DPRINT(Debug,12,(&Debug,
			    "mail: From address not needed: MI_DONT_ADD_FROM\n"));
	} else {
	    char * from_domain = NULL;
	    char * copy_addr   = NULL;
	    char * update_domain = NULL;
	    
	    enum make_from_status
		from_ok = make_from_addr(&(headers->from),mailer_info,&from_domain,
					 &copy_addr);

	    enum from_action fa_reply = fa_continue;

	    if (headers->from.addrs)
		from_addr_len = addr_list_item_count(headers->from.addrs);
	    else {
		DPRINT(Debug,12,(&Debug,"No From addresses\n"));
	    }

	    fa_reply =  mail_check_default_from(headers,
						mailer_info,
						from_ok,
						from_domain,
						copy_addr,
						&command_ch,
						code,
						page,
						prompt_area,
						snd_cmds,
						&MIME_info,
						env_from_domain_class,
						&update_domain);
	   
	    if (update_domain) {
		if (from_domain &&
		    0 == strcmp(from_domain,
				update_domain)) {
		    DPRINT(Debug,12,(&Debug,
				     "mail: Rewritten From address domain %s not needed, canceled.\n",
				     from_domain));
		} else {
		    force_from_addr(&(headers->from),update_domain);

		    if (from_domain) {
			DPRINT(Debug,12,(&Debug,
					 "Rewritten From address domain %s to %s\n",
					 from_domain,
					 update_domain));

		    } else {		    
			DPRINT(Debug,12,(&Debug,
					 "Rewritten From address domain to %s\n",
					 update_domain));
		    }
		}
		free(update_domain);
		update_domain = NULL;
	    } else if (from_domain) {
		DPRINT(Debug,12,(&Debug,"From address domain %s\n",
				 from_domain));       
	    }

	    if (from_domain) {
		free(from_domain);
		from_domain = NULL;
	    }

	    
	    if (copy_addr) {
		free(copy_addr);
		copy_addr = NULL;
	    }

	    switch (fa_reply) {
	    case fa_none:
		DPRINT(Debug,8,(&Debug," ... fa_none\n"));
		break;
	    case fa_forget:
		DPRINT(Debug,8,(&Debug," ... fa_forget\n"));
		goto forget_message;
	    case fa_continue :
		DPRINT(Debug,8,(&Debug," ... fa_continue\n"));
		break;
	    case fa_transmission:
		DPRINT(Debug,8,(&Debug," ... fa_transmission"));
		if (command_ch) {
		    DPRINT(Debug,8,(&Debug,", command_ch=%d",
				    command_ch));
		} else {
		    DPRINT(Debug,8,(&Debug,", no command"));
		}
		DPRINT(Debug,8,(&Debug,"\n"));
	    	
		goto verify_trans_menu;
	    case fa_EOF:
		DPRINT(Debug,8,(&Debug," ... fa_EOF\n"));
		
		code = ft_EOF;  /* LEAVE */
		goto fail_label;

	    case fa_fail:
		DPRINT(Debug,8,(&Debug," ... fa_fail\n"));

		if (ft_forget == code || ft_forget_empty == code) 
		    goto forget_message;
		
		
		if (OPMODE_IS_INTERACTIVE(opmode)) {
		    local_options = MAIL_FORGET;				    
		    goto restart_verify_transmission;
		}
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmMailAddrFail2,
				  "Failed to add From address."));
		goto fail_label;		
	    }	    	    
	}
	
	helperstatus = sndhelper_handle_sender(add_sender_h,mailer_info,&sender_domain,
					       &sender_domain_verify,&sender_addr,
					       from_addr_len,&sender_ok,headers);


	switch (helperstatus) {
	case sndhelper_domain_failure:
	    if (!sender_ok && sender_addr && sender_domain) {
		
		if (ft_forget != code &&
		    ft_forget_empty != code) {		    
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmSenderFailDomain,
				      "Domain %s for Sender address %s not valid; Use f)orget."),
			      sender_domain,sender_domain);
		    local_options = MAIL_FORGET;
		}
	    }
	    break;
	case sndhelper_none:
	    break;
	}
	
	if (sender_domain) {
	    free(sender_domain);
	    sender_domain = NULL;
	}

	if (sender_addr) {
	    free(sender_addr);
	    sender_addr = NULL;
	}

	if (!sender_ok) {

	    /* If Sender: -header is only omitted,
	       when domain is on special-use-domains-blacklist
	       elmrc option, user can prevent addition
	       of Sender: header with just editing
	       on special-use-domains-blacklist on
	       ~/.elm/elmrc
	    */

	    if (ft_forget == code ||
		ft_forget_empty == code)
		goto forget_message;
	    
	    
	    if (OPMODE_IS_INTERACTIVE(opmode))
		goto restart_verify_transmission;

	    goto fail_label;
	}

    }

    if (headers->date) {
	time_t now = 0;
	time_t XX  = 0;
	
	switch (get_expanded_date_source(headers->date)) {
	case date_automatic:

	    XX = get_expanded_date_cached_time(headers->date);

	    if (((time_t) -1) == XX || 0 == XX) {

		DPRINT(Debug,14,(&Debug,
				 "mail: Removing generated date.\n"));
		free_expanded_date(& (headers->date));
				
	    } else if (((time_t) -1) != time(&now)) {

		if (now < DATE_REGENERATION_TIME || XX < now - DATE_REGENERATION_TIME || XX > now) {

		    lib_error(CATGETS(elm_msg_cat, ElmSet,
				      ElmDateWillRegenerated,
				      "Date will be regenerated."));
		    
		    DPRINT(Debug,14,(&Debug,
				     "mail: Removing generated date.\n"));
		    free_expanded_date(& (headers->date));

		    error_sleep(1);

		} else {
		    int a UNUSED_VAROK = now-XX;
		    
		    DPRINT(Debug,14,(&Debug,
				     "mail: Preserving generated date (generated %d seconds ago).\n",
				     a));

		}
	    } else {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,4,(&Debug,
				"mail: Date: not given, time: %s\n",
				strerror(err)));
		
	    }
	    break;
	case date_explicit:
	    DPRINT(Debug,14,(&Debug,
			     "mail: Preserving explicit date.\n"));
	    break;

	}
    }
    
    {
	enum message_id_status  msgidstat = msid_stat_none;
	struct digest_proc   * new_hash = NULL;
	struct cancel_data  * main_cancel = NULL;
	const struct digest_proc_type * need_hash = NULL;

	
	if (! headers->message_id) {	    
	    DPRINT(Debug,14,(&Debug,
			     "mail: No message id, checking"));
	    if (have_reconnect) {
		have_reconnect = 0;
		
		DPRINT(Debug,14,(&Debug,
				 ", mailer reinitialized"));
	    }	    
	    DPRINT(Debug,14,(&Debug,"\n"));
	    
	    msgidstat = update_message_id(& (headers->message_id),
					  headers->message_digest,
					  NULL,mailer_info,
					  &need_hash,
					  &main_cancel);

	    switch (msgidstat) {
	    case msid_stat_need_hash:

		if (main_cancel) {
		    if (is_canceled(main_cancel)) {       
			DPRINT(Debug,7,(&Debug, "mail: message id check canceled\n"));
			break;
		    }
		    free_cancel(&main_cancel);
		}
				
		DPRINT(Debug,14,(&Debug,
				 "mail: Message id needs hash, calculating\n"));
		
		new_hash = count_message_hash(converted_buffer,&MIME_info,headers,need_hash);
		break;
	    case msid_stat_need_restart:
	    case msid_stat_none:
	    case msid_stat_updated:
	    case msid_stat_no_changes:
		break;
	    }
	} else if (headers->message_digest) {
	    enum digest_proc_comp A;

	    need_hash = digest_type(headers->message_digest);
	    
	    DPRINT(Debug,14,(&Debug,
			     "mail: calculating new hash\n"));
		
	    new_hash  = count_message_hash(converted_buffer,&MIME_info,headers,need_hash);

	    if (new_hash) {

		A = digest_proc_equal(headers->message_digest,new_hash);
		switch(A) {
		case digest_proc_incomplete:
		    DPRINT(Debug,14,(&Debug,
				     "mail: incomplete hash?\n"));
		    break;
		case digest_proc_differ:
		    DPRINT(Debug,14,(&Debug,
				     "mail: hash changed, need new message id\n"));
		    msgidstat = msid_stat_need_hash;
		    break;
		case digest_proc_same:
		    DPRINT(Debug,14,(&Debug,
				     "mail: hash not changed\n"));
		    break;
		}

	    } else {
		DPRINT(Debug,14,(&Debug,
				 "mail: no hash?\n"));
	    }
	} 
       				
	switch (msgidstat) {
	case msid_stat_need_hash:
	    if (new_hash) {
		DPRINT(Debug,14,(&Debug,
				 "mail: new hash, recalculating message id\n"));
		
		msgidstat = update_message_id(& (headers->message_id),
					      headers->message_digest,
					      new_hash,mailer_info,
					      &need_hash,&main_cancel);
		break;
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "mail: new hash not available?\n"));
		msgidstat = msid_stat_none;
	    }

	    /* FALLTHRU */
	case msid_stat_none:
	    if (have_reconnect) {
		DPRINT(Debug,14,(&Debug,
				 "mail: mailer reinitialized, recalculating message id\n"));
		
		msgidstat = update_message_id(& (headers->message_id),
					      headers->message_digest,
					      new_hash,mailer_info,
					      &need_hash,
					      &main_cancel);
		
	    }
	    break;
	    
	case msid_stat_need_restart:
	case msid_stat_updated:
	case msid_stat_no_changes:


	    
	    break;
	}

	switch (msgidstat) {
	case msid_stat_need_hash:  
	    DPRINT(Debug,14,(&Debug,
			     "mail: message id needs still new hash?\n"));
	    break;
	case msid_stat_none:
	    if (! headers->message_id) {
		DPRINT(Debug,14,(&Debug,
				 "mail: message id not generated\n"));
	    }
	    break;
	case msid_stat_updated:
	    if (headers->message_id) {
		if (headers->message_digest)
		    free_digest_proc(& headers->message_digest);
		
		DPRINT(Debug,14,(&Debug,
				 "mail: message id updated"));
		
		if (new_hash) {
		    headers->message_digest = new_hash;
		    new_hash = NULL;
		    DPRINT(Debug,14,(&Debug,", message hash changed\n"));
		}
		DPRINT(Debug,14,(&Debug,"\n"));
	    }
	    break;
	case msid_stat_no_changes:
	    if (headers->message_id) {
		DPRINT(Debug,14,(&Debug,
				 "mail: message id preserved\n"));
	    }
	    break;
	case msid_stat_need_restart: {
	    enum restart_mailer_status r;
	    
	    DPRINT(Debug,14,(&Debug,
			     "mail: mailer need restart\n"));

	    if (new_hash)
		free_digest_proc(& new_hash);

	    if (main_cancel)
		free_cancel(&main_cancel);

	    
	    r = mailer_restarted(mailer_info);
	    
	    switch (r) {
	    case mailer_reinitialized:
		have_reconnect = 1;
		
		handle_mailer_options(mailer_info,&MIME_info,&dsn, 
				      &attachments, &reask_verify);
		

		break;

	    case mailer_disconnected:
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailerDisconnected,
				  "Mailer disconnected! Can't send message. Use f)orget !"));
		local_options = MAIL_FORGET;
		
		if (OPMODE_IS_INTERACTIVE(opmode)) 
		    goto restart_verify_transmission;
		
		goto fail_label;
		
	    case mailer_not_restarted:
		break;
	    }
	    
	}
	    goto restart_verify_transmission;		

	}
	if (new_hash)
	    free_digest_proc(& new_hash);
	if (main_cancel)
	    free_cancel(&main_cancel);	
    }
        	 
    switch(code) {

    case ft_preview: 
    case ft_preview_hdrs:
    case ft_preview_menu: 
    case ft_preview_mime: {

	enum snd_preview_result pvr = 

	    preview_message(converted_buffer,&command_ch,
			    page,snd_cmds,code,
			    &MIME_info,headers,
			    aview,mailbox,remote_server);
	switch(pvr) {
	case sp_none: 
	    lib_error(CATGETS(elm_msg_cat, ElmSet, 
			      ElmFailedPreviewMess,
			      "Failed to preview message. Use f)orget !"));
	    local_options = MAIL_FORGET;
	    
	    if (OPMODE_IS_INTERACTIVE(opmode))
		goto restart_verify_transmission;

	    break;

	case sp_send: goto send_message;
	case sp_verify_trans: 
	    DPRINT(Debug,8,(&Debug," ... sp_verify_trans: "));

	    if (command_ch) {
		DPRINT(Debug,8,(&Debug,"command_ch=%d",
				command_ch));
	    } else {
		DPRINT(Debug,8,(&Debug,"no command"));
	    }
	    DPRINT(Debug,8,(&Debug,"\n"));
	    	    
	    goto verify_trans_menu; 
	case sp_forget: 
	    if (reply && fsize(reply) == 0) {		
		DPRINT(Debug,8,(&Debug," ... sp_forget: empty\n"));
		/* Empty ... */
		if (!attachments.attachment_count) 
		    break;
	    }
	    goto forget_message;
	case sp_EOF:
	    DPRINT(Debug,8,(&Debug," ... sp_EOF\n"));
	    break;
	}
    }
	break;
	
    case ft_forget:
    case ft_forget_empty: 
    forget_message: { /* Forget */	

	struct MailboxView *cm = give_canceled_mail();

	char * lbuf = NULL;
	struct out_state  * fp_copy = new_out_state(STATE_out_file);
	FILE *F = NULL;

	if (cm) {
	    time_t now = 0;

	    if (((time_t) -1) != time(&now)) {
		    	    
		F = add_canceled_mail(&lbuf,headers,cm,now,-1,NULL,
				      -1L);

	    } else {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,4,(&Debug,
				"mail: Date: not given, time: %s\n",
				strerror(err)));
	    }
	}
		
	if (!F) {
	    if (lbuf)
		free(lbuf);

	    lbuf = elm_message(FRM("%s/%s"), 
			       home, dead_letter);

	    F = safeopen_rdwr(lbuf,NULL);
	} else {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmVfyMessageKept1,
			      "Message kept.  Can be restored at next m)ail command."));
	}
	
	if (NULL != F) {
	    
	    DPRINT(Debug,4,(&Debug, "mail: Writing canceled mail to %s\n",
			    lbuf));
	    
	    set_out_state_file(F,fp_copy);
	    write_header_info(fp_copy,headers,cm_across_copy,
			      &MIME_info);

	    copy_message_across(&MIME_info,fp_copy,
				cm_across_copy,converted_buffer);

	}
	free_out_state(&fp_copy);
	if (F)
	    fclose(F);

	if (lbuf) {

	    if (last_canceled_mail)
		free(last_canceled_mail);
	    last_canceled_mail = lbuf;	    
	}
    }
	break;
	
    case ft_send:
    send_message: {
	int r UNUSED_VAROK;
	struct mail_send_state * mailer_snd_buffer = NULL;
	int address_count = 0;
	char **argvt = argv_from_headers(headers);

	if (!argvt) {
	    DPRINT(Debug,4,(&Debug,"mail: No mail addresses\n"));
	    
	    if (OPMODE_IS_INTERACTIVE(opmode))
		goto restart_verify_transmission;
	    
	    goto fail_label;
	}

	mailer_snd_buffer = mailer_init(argvt,dsn,sendmail_verbose,mailer_info,
					headers->env_from);

	address_count = 0;
	while (argvt[address_count])
	    address_count++;
	
	free_argv(&argvt);	    	    	 
	
	if (mailer_snd_buffer) {
	    DPRINT(Debug,14,(&Debug,
			     "mail: mailer snd buffer %p, %d addresses\n",
			     mailer_snd_buffer,address_count));
	    
	} else {
	    DPRINT(Debug,14,(&Debug,
			     "mail: no mailer snd buffer, %d addresses\n",
			     address_count));
		   
	    if (OPMODE_IS_INTERACTIVE(opmode))
		goto restart_verify_transmission;
	    
	    goto fail_label;
	}

	rewind(converted_buffer);
	if (COPY_FILE.copy_file) /* i.e. if copy_file contains a name */
	    save_copy(headers,&COPY_FILE, &MIME_info, 
		      converted_buffer, 
		      page,aview,
		      prompt_area);  /* Need access for all open 
				       mailboxes/folders */

	/** write all header information into whole_msg_file **/

	/** try to write headers to new temp file **/
	
	
	dump_expanded_address(2,"From",headers->from);
	dump_expanded_address(2,"To",headers->to);
	dump_expanded_address(2,"CC",headers->cc);
	dump_expanded_address(2,"Bcc",headers->bcc);

	if (sending_display)
	    free_sending_display(&sending_display);
	    
	if (query_mailer_info(mailer_info,
			      MI_HAVE_SENDING_DISPLAY)) {

	    DPRINT(Debug,4,(&Debug,"mail: Mailer have sending display\n"));

	    sending_display = 
		new_sending_display(address_count,
				    OPMODE_IS_INTERACTIVE(opmode));
	    
	}

	
	
	if (!mailer_snd_buffer) {

	    if (OPMODE_IS_INTERACTIVE(opmode))
		goto restart_verify_transmission;
	    
	    goto fail_label;
	} else if (sending_display) {

	    if (! mailer_set_sending_display(mailer_snd_buffer,
					     sending_display,
					     free_sending_display,
					     inc_sending_display_refcount,
					     update_addr_sd,
					     check_addr_sd,
					     update_data_sd,
					     check_fail_sd,
					     update_con_sd,
					     size_check_sd,
					     update_size_cb)) {
		DPRINT(Debug,4,(&Debug,
				"mail: Mailer did not supported sending display ( mailer_set_sending_display failed)\n"));
	    }
	}
	
	/* Let state routines be aware of encoding */
	if (MIME_info.encoding_top == ENCODING_BINARY)
	    set_out_state_EOLN(get_mail_outfd(mailer_snd_buffer),1);

	write_header_info(get_mail_outfd(mailer_snd_buffer),headers,
			  cm_across_send, &MIME_info);

	rewind(converted_buffer);

	if (!copy_message_across(&MIME_info,get_mail_outfd(mailer_snd_buffer),
				 cm_across_send,converted_buffer)) {

	    if (OPMODE_IS_INTERACTIVE(opmode))
		goto restart_verify_transmission;
	    
	    goto fail_label;
	}
	
	if (headers->to.addrs) {
	    int cnt;
	    
	    if ((cnt = addr_list_item_count(headers->to.addrs)) == 1) {
		int group;
		const struct address * X = 
		    addr_list_get_item(headers->to.addrs,0,&group);
		const char * addr = address_get_ascii_addr(X);
		
		elm_sfprintf(title, sizeof title,
			     CATGETS(elm_msg_cat, ElmSet, ElmMailTo,
				     "Mail to %.50s..."),
			     addr ? addr : "(someone)");
		
	    } else
		elm_sfprintf(title, sizeof title,
			     CATGETS(elm_msg_cat, ElmSet, ElmMailToXRecipients,
				     "Mail to %d recipients (To:)"),
			     cnt);


	} else
	    elm_sfprintf(title, sizeof title,
			 CATGETS(elm_msg_cat, ElmSet, ElmMailToSomeone,
				 "Mail to someone..."));


	/* May free mailer_snd_buffer */
	
	r = mail_backend3(&mailer_snd_buffer,mail_sent, 
			  MIME_info.encoding_top, title,mailing_message,
			  &Exstat,
			  cancel_ticket,
			  /* message size */
			  MIME_info.cl_end);
	
	DPRINT(Debug,4,(&Debug,"mail_backend2 returned %d, exit status = %d\n",
			r,Exstat));

	if (cancel_ticket) {
	    enum msd_action action = have_msd_action(cancel_ticket);
	    switch (action) {
	    case msd_act_EOF: 
		DPRINT(Debug,4,(&Debug,
				"  cancel_ticket: action=msd_act_EOF (%d)\n",
				action));
		    
		code = ft_EOF;  /* LEAVE */
		goto fail_label;
		
	    case msd_act_SIG:
		DPRINT(Debug,4,(&Debug,
				"  cancel_ticket: action=msd_act_SIG (%d)\n",
				action));
		goto fail_label;
		
	    case msd_act_need_restart: {
		 enum restart_mailer_status r;
		
		DPRINT(Debug,4,(&Debug,
				"  cancel_ticket: action=msd_act_need_restart (%d)\n",
				action));

		r = mailer_restarted(mailer_info);

		switch (r) {
		case mailer_reinitialized:
		    have_reconnect = 1;
		    
		    handle_mailer_options(mailer_info,&MIME_info,&dsn, 
					  &attachments, &reask_verify);
		    
		    /* Need rebuild mail */
		    break;
		case mailer_disconnected:
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailerDisconnected,
				      "Mailer disconnected! Can't send message. Use f)orget !"));
		    local_options = MAIL_FORGET;
		    		    
		    if (OPMODE_IS_INTERACTIVE(opmode))
			goto restart_verify_transmission;
		    
		    cancel_ticket_to_canceled_mail(cancel_ticket,headers);
		    goto fail_label;
		case mailer_not_restarted:
		    break;
		}
		
		goto restart_verify_transmission;		
	    }
		break;
	    case msd_act_default:
		DPRINT(Debug,4,(&Debug,
				"  cancel_ticket: action=msd_act_default (%d)\n",
				action));
		
		break;
	    case msd_act_cancel:
		DPRINT(Debug,4,(&Debug,
				"  cancel_ticket: action=msd_act_cancel (%d)\n",
				action));
		
		cancel_ticket_to_canceled_mail(cancel_ticket,headers);
		
		goto fail_label;

	    case msd_act_edit:
		DPRINT(Debug,4,(&Debug,
				"  cancel_ticket: action=msd_act_edit (%d)\n",
				action));

		if (OPMODE_IS_INTERACTIVE(opmode))
		    goto restart_verify_transmission;

		cancel_ticket_to_canceled_mail(cancel_ticket,headers);
		goto fail_label;

	    case msd_act_send:
		DPRINT(Debug,4,(&Debug,
				"  cancel_ticket: action=msd_act_send (%d)\n",
				action));
		break;
	    }	  	    
	}

	   

 	/* 0   may be:   start failed
 	 *               mailing continued on backend
 	 *
  	 *  1            command completed, perhaps with failure 
 	 */
	
 	ret = 1;


	/* NOTE: succees failure, backgrouded status currently ignored */

	/* mark the "replied" status of the message */
	if (replying && mailbox && mail_index >= 0 && ret) {

	    struct header_rec *hdr =  give_header(mailbox,mail_index);
	    if (hdr && 	0 == (hdr->status & REPLIED_TO)) {
		
		DPRINT(Debug,4,(&Debug,
				"Adding replied mark to current message\n"));
		
		hdr->status |= REPLIED_TO;
		hdr->status_chgd = TRUE;
	    } else {
		DPRINT(Debug,8,(&Debug,
				"Not adding replied mark %s\n",
				hdr ? "" : " -- getting header failed"));
	    }

	} else {
	    DPRINT(Debug,8,(&Debug,"Not adding replied mark %s%s%s\n",
			    replying ? "" : " -- not replying ",
			    mailbox ? "" : " -- no mailbox",
			    ret ? "" : " -- no succeed"));
	}
    }
	break;

    case ft_none:
    case ft_EOF:
	break;

    }
	
 fail_label:
				      
    if (reply)
	fclose(reply);

    if (sending_display)
	free_sending_display(&sending_display);

    if (delete_cancel)
	sync_canceled_mails(selection_view);

    if (usraddr)
	free_address(& usraddr);

    if (snd_cmds)
	free_commands(&snd_cmds);

    if (cancel_ticket)
	free_mailer_cancel_ticket(&cancel_ticket);
    
    free_Attachments(&attachments);

    if (converted_buffer)
	fclose(converted_buffer);
    converted_buffer = NULL;

    /* Unlink temp file now.
     * This is a precaution in case the message was encrypted.
     * I.e. even though this file is readable by the owner only,
     * encryption is supposed to hide things even from someone
     * with root privileges. The best we can do is not let this
     * file just hang after we're finished with it.
     */

    if (cur_editfile[0]) {
	if (0 ==unlink (cur_editfile)) {
	    DPRINT(Debug,8,(&Debug,"Removing %s\n",
			    cur_editfile));
	} 
    }
    
    free_mime_send_info(&MIME_info);
    clear_copy_file(&COPY_FILE);


    if (prompt_area)
	erase_menu_context(&prompt_area);


    if (page)
	erase_menu_context(&page);

    if (parent_page)
	menu_trigger_redraw(parent_page);

    if (ft_EOF == code) {
	if (0 == Exstat) {
	    Exstat = 
#ifdef  EX_UNAVAILABLE
		EX_UNAVAILABLE  /* Give better error status */
#else
		1
#endif
		;	    
	}

	DPRINT(Debug,8,(&Debug,
			"mail: Exiting with exit status %d\n",
			Exstat));

	leave(0,Exstat);
    }

    if (0 == Exstat && !ret)
	Exstat = 1;

    if (exit_status) {
	*exit_status = Exstat;

	DPRINT(Debug,8,(&Debug,
			"mail: Setting exit status to %d\n",Exstat));
    }

    DPRINT(Debug,8,(&Debug,"mail=%d\n",ret));

    return ret;
}

/* Caller must call free_string() */
static struct string * exit_status_text P_((int exit_stat));
static struct string * exit_status_text(exit_stat)
     int exit_stat;
{
    struct string * ret = NULL;

#ifdef I_SYSEXITS

    /* Codes from sysexits.h */

    switch(exit_stat) {

#ifdef  EX_USAGE 
    case EX_USAGE    : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_USAGE,
				    "Usage error"));         break;
#endif
#ifdef  EX_DATAERR
    case EX_DATAERR  : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_DATAERR,
				    "Data format error"));   break;
#endif
#ifdef  EX_NOINPUT  
    case EX_NOINPUT  : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_NOINPUT,
				    "Cannot open input"));   break;
#endif
#ifdef  EX_NOUSER   
    case EX_NOUSER   : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_NOUSER,
				    "Address or user unknown")); break;
#endif
#ifdef  EX_NOHOST   
    case EX_NOHOST   : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_NOHOST,
				    "Host name unknown"));   break;
#endif
#ifdef  EX_UNAVAILABLE 
    case EX_UNAVAILABLE : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_UNAVAILABLE,
				    "Service unavailable")); break;
#endif
#ifdef  EX_SOFTWARE 
    case EX_SOFTWARE : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_SOFTWARE,
				    "Internal software error")); break;
#endif
#ifdef  EX_OSERR    
    case EX_OSERR    : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_OSERR,
				    "System error"));        break;
#endif
#ifdef  EX_OSFILE   
    case EX_OSFILE   : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_OSFILE   ,
				    "OS file missing"));     break;
#endif
#ifdef  EX_CANTCREAT
    case EX_CANTCREAT: 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_CANTCREAT,
				    "Can't create output file")); break;
#endif
#ifdef  EX_IOERR    
    case EX_IOERR    : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_IOERR,
				    "Input/output error"));  break;
#endif
#ifdef  EX_TEMPFAIL 
    case EX_TEMPFAIL : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_TEMPFAIL,
				    "Temporary failure"));   break;
#endif
#ifdef  EX_PROTOCOL 
    case EX_PROTOCOL : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_PROTOCOL,
				    "Protocol error"));      break;
#endif
#ifdef  EX_NOPERM   
    case EX_NOPERM   : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_NOPERM,
				    "Permission denied"));   break;
#endif
#ifdef  EX_CONFIG   
    case EX_CONFIG   : 
	ret = format_string(CATGETS(elm_msg_cat, ElmSet,
				    ElmEX_CONFIG,
				    "Configuration error")); break;
#endif
    }    
#endif
    return ret;
}


static void mail_sent (fd,title,rs,ret,exit_code)
     union any_fd fd; 
     char * title;
     struct run_state *rs;
     int ret; 
     int exit_code;
{    
    lower_prompt(title);
    
    if (ret < 0) 
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailSignal,
			  "%.30s fail: Signal?"),
		  get_mailer_path(fd.mail_fd));
    else if (ret > 0) {
	if (rs->save_errno)
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailErrno,
			      "Failed: %.30s: %.40s"),
		      get_mailer_path(fd.mail_fd),
		      strerror(rs->save_errno));
	else if (exit_code) {
	    struct string * sys_exit = exit_status_text(exit_code);
	    
	    if (sys_exit) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmMailerReturnedErrorT,
				  "mailer returned error status %d (%S)"), 
			  exit_code,sys_exit);
		free_string(&sys_exit);
	    } else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmMailerReturnedError,
				  "mailer returned error status %d"), 
			  exit_code);
	} else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmMailSent, "Mail sent!"));
    } else {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmLostErrno,
			  "%.30s lost: %.40s"),
		  get_mailer_path(fd.mail_fd),
		  strerror(rs->save_errno));
    }
}

static void dsn_menu P_((int *));
static void dsn_menu(dsn)
     int * dsn;
{
    static int hdr_only = 0;
    static int failure  = 1;
    static int delay    = 1;
    static int success  = 0;
    static int DSN_off  = 0;

#define BOL 1
    static struct menu_item dsn_items[] = {
      { "Return H)eaders only on FAILURE:", 'h', 3,  BOL, 
	{ (char *) &hdr_only},
	sizeof hdr_only },
      { "Return DSN on F)AILURE         :", 'f', 5,  BOL, 
	{ (char *) &failure },
	sizeof failure },
      { "              D)ELAY           :", 'd', 6,  BOL, 
	{ (char *) &delay },
	sizeof delay },
      { "              S)UCCESS         :", 's', 7,  BOL, 
	{ (char *) &success },
	sizeof success },
      { "U)se defaults (DSN off)        :", 'u', 12, BOL, 
	{ (char *) &DSN_off },
	sizeof DSN_off }
    };
    
    if (0 == *dsn)
	DSN_off = 1;
    else {
	if (DSN_FULL    & *dsn) hdr_only = 0;
	if (DSN_HDRS    & *dsn) hdr_only = 1;
	if (DSN_SUCCESS & *dsn) success  = 1;
	if (DSN_FAILURE & *dsn) failure  = 1;
	if (DSN_DELAY   & *dsn) delay    = 1;
	if (DSN_NEVER   & *dsn) { success = 0; failure = 0; delay = 0; }
    }
  
    if (generic_menu(dsn_items,sizeof dsn_items / sizeof (struct menu_item),
		     "DSN (Delivery Status Notification) Configuration",
		     "DSN: ")) {

	if (DSN_off) *dsn = 0;
	else if (!success && !failure && !delay) *dsn = DSN_NEVER;
	else {
	    int val = 0;
	    if (hdr_only) val |= DSN_HDRS;
	    else          val |= DSN_FULL;
	    if (success)  val |= DSN_SUCCESS;
	    if (failure)  val |= DSN_FAILURE;
	    if (delay)    val |= DSN_DELAY;
	    
	    *dsn = val;
	}
    }
}

int mail_edit_return_path(headers, page)
     struct mailing_headers *headers;
     struct menu_context *page;
{

    if (headers->env_from) {
	int ret;
	int can_edit = 0;
	const char *t = mailer_env_from_value(headers->env_from,&can_edit);
	struct string *buffer = NULL;
	int changed = 0;
	
	int LINES, COLUMNS;

	menu_get_sizes(page,&LINES, &COLUMNS);

	if (!can_edit) {
	    DPRINT(Debug,8, (&Debug,
			     "mail_edit_return_path: Can not edit return path\n"));
	    return 0; /* bad command */
	}

	menu_MoveCursor(page,LINES-4,0);
	menu_CleartoEOS(page);
	       
	menu_print_format_center(page,
				 LINES-4, 
				 CATGETS(elm_msg_cat, ElmSet,
					 ElmEnterEnvFrom, 
					 "Enter value for envelope sender address."));

	if (t)
	    buffer = new_string2(display_charset,cs2cus(t));

	/* NOTE: Currently alias expansion is not possble */

	ret = optionally_enter2(page,
				&buffer,LINES-2, 0, 
				OE_REDRAW_MARK|
				OE_SIG_CHAR /* Ctrl-C */,
				CATGETS(elm_msg_cat, ElmSet, 
					ElmPreSEnvFromPrompt,
					"Return-path: "));
	
	if (ret == 0 && buffer) {
	    char * s = us2s(stream_from_string(buffer,0,NULL));

	    changed = mailer_env_from_change(headers->env_from,s);

	    free(s);
	}
	
	if (buffer)
	    free_string(&buffer);
	
	if (ret == REDRAW_MARK)
	    return REDRAW_MARK;

	if (0 != ret)       /* Error or Ctrl-C */
	    return 1;

	menu_ClearLine(page,12);

	t = mailer_env_from_value(headers->env_from,&can_edit);

	if (t && '<' != t[0]) {
	    const char * a = mailer_env_from_angleaddr(headers->env_from);
	    if (a) {
		DPRINT(Debug,12, (&Debug,
				  "mail_edit_return_path: env_from=%s displaying angleaddr=%s\n",
				  t,a));
		t = a;
	    }
	}

	if (can_edit) {
	    if (t)
		menu_PutLineX(page,
			      12,0,CATGETS(elm_msg_cat, ElmSet, ElmPreSEnvFrom,
					   "R)eturn-path: %s"),
			      t);
	    else
		menu_PutLineX(page,
			      12,0,CATGETS(elm_msg_cat, ElmSet, ElmPreSEnvFromDef,
				     "R)eturn-path: (default)"));
	} else if (t)
	    menu_PutLineX(page,
			  12,0,CATGETS(elm_msg_cat, ElmSet, ElmPreSEnvFromNoEdit,
				       "Return-path: %s"),
		     t);

	DPRINT(Debug,8, (&Debug,
			 "mail_edit_return_path=%d: env_from %s\n",
			 changed,
			 changed ? "changed" : "not changed"));
	
	return changed;
    }

    DPRINT(Debug,8, (&Debug,
		     "mail_edit_return_path: env_from is not set\n"));
    return 0; /* bad command */
}

static enum pgp_confirm_loop {   /* jump targets */
    pcf_no   = 0,    /* No */
    pcf_out,         /* Yes */
    pcf_redraw,
    pcf_resize,
    pcf_EOF = -1
} pgp_send_confirm_loop P_((int replying, int was_pgp_encoded,
			    int *PGP_status, 
			    struct menu_context *page,
			    struct menu_context *prompt_area));
static enum pgp_confirm_loop pgp_send_confirm_loop(replying,was_pgp_encoded,
						   PGP_status,page,prompt_area)
     int replying; 
     int was_pgp_encoded;
     int *PGP_status;
     struct menu_context *page;
     struct menu_context *prompt_area;
{
    if (use_PGP &&
	PGP_status && replying && was_pgp_encoded &&
	! ((*PGP_status) & PGP_MESSAGE)) {

	menu_ClearScreen(prompt_area);

	menu_PutLineX(prompt_area,1, 0, 
		      CATGETS(elm_msg_cat, ElmSet, ElmPgpRcvSure, 
			      "The recv'd message was PGP encoded, are you sure? "));
	for (;;) {
	    int cmd = menu_ReadCh(prompt_area,REDRAW_MARK|READCH_resize);
	    
	    if (cmd == REDRAW_MARK) {
		DPRINT(Debug,4, (&Debug,"       .... redrawing\n"));
		menu_ClearScreen(page);   /* Clear possible redraw mark */
		
		/* Call refresh routines of children */
		menu_redraw_children(page);
		
		if (menu_need_redraw(prompt_area))		    
		    menu_ClearScreen(prompt_area);   /* Clear redraw mark from prompt_area*/
		
		return pcf_redraw;
	    }

	    if (cmd == RESIZE_MARK) {
		
		DPRINT(Debug,4, (&Debug,"       .... resizing\n"));
		return pcf_resize;
	    }

	    if (cmd == *def_ans_yes) {
		menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, ElmYesWord, 
					     "Yes."));
		FlushBuffer();
		return pcf_out;
	    }
	    if (cmd == *def_ans_no) {
		menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, ElmNoWord, 
					     "No."));
		FlushBuffer();		    
		return pcf_no;
	    }

	    if (cmd == EOF)
		return pcf_EOF;
	    
	}
    }
    
    return pcf_out;
}

static const char * have_ispell P_((void));
static const char * have_ispell() 
{
    const char * return_value   = ISPELL_path;

    if (! have_ISPELL || 
	! ISPELL_path[0]) {
	return_value = NULL;
    } else if (ISPELL_path[0] == '/') {
	if (-1 == access(ISPELL_path,EXECUTE_ACCESS)) {
            int err = errno;
            lib_error(CATGETS(elm_msg_cat, ElmSet,ElmCantIspell,
                              "Can't execute ispell: %s: %s"),
                      ISPELL_path,
                      strerror(err));
            DPRINT(Debug,6, (&Debug, 
                             "have_ispell: no access %s: %s\n",
                             ISPELL_path,
                             strerror(err)));
            sleep_message();
            return_value = NULL;
        }	
    }

    DPRINT(Debug,5,(&Debug,   
                    "have_ispell=%s\n",
                    return_value ? return_value : "<NULL>"));
    return return_value;

 }

void mail_show_env_from(headers,env_from_domain_class,page)
     struct mailing_headers * headers;
     enum rp_classify_domain  env_from_domain_class;
     struct menu_context    * page;
{
    enum user_level_v ul = 
	give_dt_enumerate_as_int(&user_level);

    menu_ClearLine(page,12);
	
    switch (env_from_domain_class) {
	
    case env_from_no_domain:
    case env_from_ok_domain:
    case env_from_mailer_verified:
    case env_from_dns_verified:
	if ((ul > 0) ||
	    mailer_env_from_changed(headers->env_from)) {
	    int can_edit;
	    const char *t;
	    
	case env_from_reset:
	case env_from_bad_domain:
	case env_from_unknown_domain:
	case env_from_no_mail_domain:
	    
	    t = mailer_env_from_value(headers->env_from,&can_edit);

	    if (t && '<' != t[0]) {
		const char * a = mailer_env_from_angleaddr(headers->env_from);
		if (a) {
		    DPRINT(Debug,12, (&Debug,
				      " mail_show_env_from: env_from=%s displaying angleaddr=%s\n",
				      t,a));
		    t = a;
		}
	    }
	    
	    if (can_edit) {
		if (t)
		    PutLineX(12,0,CATGETS(elm_msg_cat, ElmSet, ElmPreSEnvFrom,
					  "R)eturn-path: %s"),
			     t);
		else
		    PutLineX(12,0,CATGETS(elm_msg_cat, ElmSet, ElmPreSEnvFromDef,
					  "R)eturn-path: (default)"));
	    } else if (t)
		PutLineX(12,0,CATGETS(elm_msg_cat, ElmSet, ElmPreSEnvFromNoEdit,
				      "Return-path: %s"),
			 t);				     
	}
	
	break;
    }
}

int mail_env_from_def_edit(env_from_domain_class,headers,env_from_cdr)
     enum rp_classify_domain         env_from_domain_class;
     struct mailing_headers        * headers;
     struct classify_domain_result * env_from_cdr;
{
    int ret = 0;

    switch (env_from_domain_class) {
	int is_editable;
	const char * ef_value;
	const char * ef_domain;
	
    case env_from_reset:
	/* Dummy */
	break;
	
    case env_from_no_domain:
    case env_from_ok_domain:
    case env_from_mailer_verified:
    case env_from_dns_verified:
	
	/* OK */
	break;
	
    case env_from_unknown_domain:  
    case env_from_no_mail_domain:
    case env_from_bad_domain:
	is_editable = 0;
	
	ef_value = mailer_env_from_value(headers->env_from,
					 &is_editable);
	ef_domain = mailer_env_from_domain(headers->env_from);
	
	DPRINT(Debug,10,(&Debug,
			 "env from %s (%s) domain %s bad\n",
			 ef_value ? ef_value : "(?)",
			 is_editable ? "editable" : "can't change",
			 ef_domain ? ef_domain : "(?)"));

	rp_classify_domain_message(headers->env_from,
				   env_from_domain_class,
				   env_from_cdr);
	
	build_address_classify_message(ef_value,
				       ef_domain,
				       env_from_cdr->is_literal,
				       env_from_cdr->whitelisted_name,
				       env_from_cdr->reserved_name);
	
	if (is_editable)
	    ret = 1;
	break;
    }

    
    DPRINT(Debug,10,(&Debug,"mail_env_from_def_edit=%d\n",
		     ret));
    return ret;
}

/*
 * verify_transmission() - Ask the user to confirm transmission of the
 * message.  Returns 1 to send it,      -1 to forget it,
 *                   2 to preview,      -2 to forget empty file,
 *                   3 to preview menu, -3 to EOF on user input
 *                   4 to preview mime
 */
static enum verify_trans_result verify_transmission(filename, 
						    already_has_text, 
						    copy_file, force_cmd, 
						    options,
						    dsn, headers, attachments,
						    file_set,
						    mailer_info,
						    mime_info,
						    mailbox,
						    aview,page,
						    prompt_area,
						    parent_message,
						    remote_server,
						    PGP_status,
						    snd_cmds,
						    env_from_domain_class_p,
						    env_from_cdr)
     char *filename;	    /* pathname to mail mssg composition file        */
     int already_has_text;  /* indicates there is already text in the mssg   */
     struct copy_file *copy_file; /* pointer to buffer holding copy file name   */
     int force_cmd;	    /* command to do, '\0' to prompt user for cmd    */
     int options;
     int *dsn;
     struct mailing_headers *headers;   
     struct Attachments *attachments;
     charset_t file_set;      /* charset of file (filename) */
     struct mailer_info  *mailer_info;
     mime_send_t *mime_info;
     struct MailboxView *mailbox;   /* needed for system_call() */
     struct AliasView *aview;
     struct menu_context *page;    /* new menu context assumed! */
     struct menu_context  *prompt_area;
     struct header_rec *parent_message;   /* Replying .... */
     const struct remote_server * remote_server;
     int * PGP_status;
     struct elm_commands *snd_cmds;
     enum rp_classify_domain * env_from_domain_class_p;
     struct classify_domain_result *env_from_cdr;
{
    struct string *prompt_mssg_s = NULL;	/* message to display prompting for cmd	*/
    struct string *prompt_menu1_s = NULL;       /* menu of available commands		*/
    struct string *prompt_menu2_s = NULL;
    int bad_cmd = 0;		/* set TRUE to bitch about user's entry	*/
    int did_prompt;		/* TRUE if cmd prompted for and entered	*/
    int cmd;			/* command to perform			*/
    int x_coord = 0, y_coord = 0;
    int replying          = 0 != (options & MAIL_REPLYING);
    int forget            = 0 != (options & MAIL_FORGET);
    
    int was_pgp_encoded   = use_PGP &&
	0 != (options & MAIL_HAVE_PGP_ENCODED);

    int def_cmd = *def_ans_yes;
    int do_redraw = 1;
    int LINES, COLUMNS;
    enum user_level_v ul = 
	give_dt_enumerate_as_int(&user_level);
    
    enum verify_trans_result ret = ft_none;  
    int menu_build = 0;
    int env_from_message = 1;

    struct hdrmenu_context redraw_context = NULL_hdrmenu_context;
    
    menu_get_sizes(page,&LINES, &COLUMNS);

    
    
    if (use_PGP &&
	PGP_status && replying && was_pgp_encoded &&
	! ((*PGP_status) & PGP_MESSAGE)) 
	def_cmd = 'P';
    
    for (;;) {
	menu_set_default(page);

	if (0) {
	redraw:
	    do_redraw = 1;
	}

    resize_mark:
	if (menu_resized(page)) {
	    menu_get_sizes(page,&LINES, &COLUMNS);
	    menu_subpage_relocate(prompt_area,page,LINES-4,4);
	    do_redraw = 1;

	} else if (menu_need_redraw(page))
	    do_redraw = 1;
	
	if (do_redraw) {
	    do_redraw = 0;

	    mail_redraw_presend_screen(page,headers,
				       mime_info->hdr_charset,
				       mime_info->encode_hdr,
				       *env_from_domain_class_p
				       );
	}

	/* build up prompt and menu strings */
	if (menu_build) {
	    ; /* not changed - no need to rebuild the strings */
	} else {

	    switch(ul) {
		
	    case user_level_beginner: {
		
		if (prompt_mssg_s)
		    free_string(&prompt_mssg_s);
		prompt_mssg_s = format_string(CATGETS(elm_msg_cat, ElmSet,
						      ElmVfyPromptSendTheMsg,
						      "Send the message now? y"));
		
		if (prompt_menu1_s)
		    free_string(&prompt_menu1_s);
		prompt_menu1_s = 
		    format_string(CATGETS(elm_msg_cat, ElmSet,
					  ElmVfyMenu1User0,
					  "Select letter of header to edit, 'e' to edit the message,"));

		if (prompt_menu2_s)
		    free_string(&prompt_menu2_s);
		prompt_menu2_s = 
		    format_string(CATGETS(elm_msg_cat, ElmSet,
					  ElmVfyMenu2User0V,
					  "'a' to make attachments, '%c' to send message, or '%c' to cancel"),
				  *def_ans_yes,*def_ans_no);
		
		/* In some conditions add these also to menu in user_level == 0 */
		
		if (query_mailer_info(mailer_info,MI_HAVE_DSN)) {
		    DPRINT(Debug,10,(&Debug,"mailer supports DSN\n"));
		    
		    if (dsn && *dsn != 0) {
			elm_append_message(&prompt_menu2_s,
					   CATGETS(elm_msg_cat, ElmSet,
						   ElmVfyMenuDSN0,
						   ", D)SN"));

		    }
		}

		if (use_PGP &&
		    replying && was_pgp_encoded) {

		    elm_append_message(&prompt_menu2_s,
				       CATGETS(elm_msg_cat, ElmSet,
					       ElmVfyMenuPGP0,
					       ", P)gp"));
		}

		elm_append_message(&prompt_menu2_s,FRM("."));
		
	    }
		break;
		
	    default: {
		if (prompt_mssg_s)
		    free_string(&prompt_mssg_s);
		prompt_mssg_s = format_string(CATGETS(elm_msg_cat, ElmSet,
						      ElmVfyPromptSendMessage,
						      "Send message now? y"));
		
		if (prompt_menu1_s)
		    free_string(&prompt_menu1_s);		
		prompt_menu1_s = 
		    format_string(CATGETS(elm_msg_cat, ElmSet,
					  ElmVfySelectLetter,
					  "Select letter of header to edit, "));
		
		elm_append_message(&prompt_menu1_s,
				   CATGETS(elm_msg_cat, ElmSet,
					   ElmVfyMenuEditMsg, 
					   "e)dit message, "));
		
		elm_append_message(&prompt_menu1_s,
				   CATGETS(elm_msg_cat, ElmSet,
					   ElmVfyMenuHeaders1, 
					   "all h)eaders, "));

		if (prompt_menu2_s)
		    free_string(&prompt_menu2_s);
		prompt_menu2_s = 
		    format_string(CATGETS(elm_msg_cat, ElmSet,
					  ElmVfyMenuVfyCpy,
					  "a)ttachments, co(p)y, "));

		if (have_ISPELL)
		    elm_append_message(&prompt_menu2_s,
				       CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuIspell,
					       "i)spell, "));

		if (ALLOW_subshell)
		    elm_append_message(&prompt_menu2_s,
				       CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuShell,
					       "!)shell, "));
		
		if (query_mailer_info(mailer_info,MI_HAVE_DSN)) {
		    DPRINT(Debug,10,(&Debug,"mailer supports DSN\n"));
		    
		    if (dsn)
			elm_append_message(&prompt_menu2_s,
					   CATGETS(elm_msg_cat, ElmSet,
						   ElmVfyMenuDSN,
						   "D)SN, "));
		}


		if (use_PGP) 
		    elm_append_message(&prompt_menu2_s,
				       CATGETS(elm_msg_cat, ElmSet,
					       ElmVfyMenuPGP,
					       "P)gp, "));

		if (mailbox) {
		    elm_append_message(&prompt_menu2_s,
				       CATGETS(elm_msg_cat, ElmSet,
					       ElmVfyMenuFolder,
					       "view) folder, "));
		}
		
		elm_append_message(&prompt_menu2_s,
				   CATGETS(elm_msg_cat, ElmSet,
					   ElmVfyMenuForget1,
					   "or f)orget"));		
	    }	   
		break;
	    }
	  
	    elm_append_message(&prompt_menu2_s,
			       CATGETS(elm_msg_cat, ElmSet,
				       ElmVfyMenuHelp,
				       " ? = help"));

    	    menu_build = 1;
	}

	if (forget) {
	    def_cmd = 'f';
	} else if (bytes(filename) <= 0 &&
	    0 == attachments->attachment_count) {

	    if (prompt_mssg_s)
		free_string(&prompt_mssg_s);
	    prompt_mssg_s = format_string(CATGETS(elm_msg_cat, ElmSet,
						  ElmVfyPromptEmptySndMsg,
						  "Send empty message now? e"));

	    def_cmd = 'e';

	    menu_build = 0;  /* Rebuild menu */
	} else {
	    def_cmd = *def_ans_yes;

	    if (use_PGP &&
		PGP_status && replying && was_pgp_encoded &&
		! ((*PGP_status) & PGP_MESSAGE)) 
		def_cmd = 'P';
	}

	if (env_from_message && headers->env_from) {
	    if ('\0' == force_cmd)
		env_from_message = 0;

	    if (mail_env_from_def_edit(*env_from_domain_class_p,
				      headers,env_from_cdr))
		def_cmd = 'R';	    
	}

	/* complain if last entry was bad */
	if (bad_cmd) {
	    menu_Write_to_screen(prompt_area,
				 FRM("%c??"), 07);
	    FlushBuffer();

	    if (sleepmsg > 0)
		error_sleep((sleepmsg + 1) / 2);
	    bad_cmd = 0;
	}

#ifdef BACKGROUD_PROCESSES      
	if (handle_sigchld)
	    sigchld_handler();
#endif
	
	if (menu_resized(prompt_area) ||
	    menu_need_redraw(prompt_area)) {
	
	    menu_ClearScreen(prompt_area);
	 
	    menu_PutLineX(prompt_area,
			  0, 0, FRM("%S"),prompt_mssg_s);

	    menu_print_center(prompt_area,1,prompt_menu1_s);
	    menu_print_center(prompt_area,2,prompt_menu2_s);

	    show_last_error();
	}

	/* if we don't have a cmd, display prompt and get response from user */
	if (force_cmd != '\0' &&
	    isascii(force_cmd)) {

	    cmd = force_cmd;     /* Removed tolower() from here */
		
	    force_cmd = '\0';
	    did_prompt = FALSE;

	    if (isascii(cmd) && 
		isprint(cmd)) {
		DPRINT(Debug,8,(&Debug,"verify_transmission: ... forced command: '%c'\n",
				cmd));
	    } else {
		DPRINT(Debug,8,(&Debug,"verify_transmission: ... forced command: %d\n",
				cmd));
	    }

	} else {
	    clear_hdrmenu_context(&redraw_context);
	    
	    menu_PutLineX(prompt_area,
			  0, 0, FRM("%S"),prompt_mssg_s);

	    menu_GetXYLocation(prompt_area,
			       &x_coord, &y_coord);
	    y_coord--; /* backspace over default answer */

	    menu_CleartoEOLN(prompt_area);
	    show_last_error();
	    
	    menu_PutLineX(prompt_area,
			  0, y_coord, FRM("%c"),def_cmd);
	    FlushBuffer();

	    menu_MoveCursor(prompt_area,
			    x_coord, y_coord);
	    cmd = menu_ReadCh(prompt_area,REDRAW_MARK|READCH_resize);

	    if (isascii(cmd) && 
		isprint(cmd)) {
		DPRINT(Debug,8,(&Debug,"verify_transmission: ... command: '%c'\n",
				cmd));
	    } else {
		DPRINT(Debug,8,(&Debug,"verify_transmission: ... command: %d\n",
				cmd));
	    }
	   
	    if (cmd == REDRAW_MARK) {
		DPRINT(Debug,4, (&Debug,"verify_transmission:       .... redrawing\n"));
		menu_ClearScreen(page);   /* Clear possible redraw mark */
		
		/* Call refresh routines of children */
		menu_redraw_children(page);

		if (menu_need_redraw(prompt_area))		    
		    menu_ClearScreen(prompt_area);   /* Clear redraw mark from prompt_area*/

		goto redraw;
	    }

	    if (cmd == RESIZE_MARK) {

		DPRINT(Debug,4, (&Debug,"verify_transmission:       .... resizing\n"));
		goto resize_mark;
	    }

	    if (cmd == EOF)
		goto handle_EOF;

	    did_prompt = TRUE;
	    clear_error();

	    /* Put cursor pack to correct place */
	    menu_MoveCursor(prompt_area,
			    x_coord, y_coord);
	}

	switch (cmd) {
	case '\n':
	case '\r':
	    cmd = def_cmd;

	    if (isascii(cmd) && 
		isprint(cmd)) {
		DPRINT(Debug,8,(&Debug,"verify_transmission: ... default command: '%c'\n",
				cmd));
	    } else {
		DPRINT(Debug,8,(&Debug,"verify_transmission: ... default command: %d\n",
				cmd));
	    }
	}

	if (cmd == *def_ans_yes) {
	    if (did_prompt) {
	        menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuSend,
					     "Send"));
		/* Call FlushBuffer() after writing of "Send" so that
		 * user get immediately feedback (it may take time
		 * to encode big attachments)
		 */
		FlushBuffer();
	    }

	    switch(pgp_send_confirm_loop(replying,was_pgp_encoded,
					 PGP_status,page,prompt_area)) {
	    case pcf_no: break;
	    case pcf_out: 
		ret = ft_send;
		goto out;
	    case pcf_redraw: goto redraw;
	    case pcf_resize: goto resize_mark;
	    case pcf_EOF:   goto handle_EOF;
	    }


	} else if (cmd == *def_ans_no )
	    goto do_no;

	/* handle command */
	switch (cmd) {
	
	case ' ':
	    if (did_prompt) {
	        menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuPreview,
					     "Preview message"));
		FlushBuffer();
	    }

	    switch(pgp_send_confirm_loop(replying,was_pgp_encoded,
					 PGP_status,page,prompt_area)) {
	    case pcf_no: break;
	    case pcf_out: 
		ret = ft_preview;
		goto out;
	    case pcf_redraw: goto redraw;
	    case pcf_resize: goto resize_mark;
	    case pcf_EOF:   goto handle_EOF;
	    }
	    break;

	case '!':
	    menu_ClearLine(prompt_area,0);
	    if (ALLOW_subshell)
		subshell(NULL, page, prompt_area);
	    else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNoSubshellCommand,
				  "Subshell command isn't configured in this version of ELM."));
	      
	    if (menu_need_redraw(page)) {
		menu_ClearScreen(page);

		do_redraw = 1;
	    }
	    break;

	 case '?':
	     if (did_prompt) {
		 menu_Write_to_screen(prompt_area,
				      FRM("?"));
		 FlushBuffer();
	     }
	    
	    {		
		int ch = help_generic(snd_cmds,TRUE,page,prompt_area);
				
		if (menu_need_redraw(page)) {
		    menu_ClearScreen(page);
		    
		    do_redraw = 1;
		}
		
		if (EOF == ch) 
		    goto handle_EOF;
	    }
	    break;

	case 'a':
	    if (did_prompt) {
	        menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuAtttach,
					     "Attachments"));
		FlushBuffer();
	    }

	    attach_menu(NULL,NULL,attachments,display_charset,mailer_info,
			NULL, aview,mailbox,NULL,remote_server);

	    do_redraw = 1;
	    break;

        case 'D':
	    if (dsn) {
		
		if (did_prompt) {
		    menu_Write_to_screen(prompt_area,
					 CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuDsn,
						 "Dsn"));
		    FlushBuffer();
		}
		
		if (query_mailer_info(mailer_info,MI_HAVE_DSN)) {
		    DPRINT(Debug,8,(&Debug,"mailer supports DSN\n"));
		    
		    dsn_menu(dsn);
		    do_redraw = 1;
		} else {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmVfyMenuDsnNotSupported,
				      "Mailer does not support delivery notification settings."));
		    
		}
	    } 
	    break;

	case ctrl('D') :
	    if (dsn) {

		if (did_prompt) {
		    menu_Write_to_screen(prompt_area,
					 CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuDsnToggle,
						 "Dsn toggle"));
		    FlushBuffer();
		}
		
		if (query_mailer_info(mailer_info,MI_HAVE_DSN)) {

		    if (0 == *dsn || 
			0 != (*dsn & DSN_NEVER) ||
			0 == (*dsn & DSN_SUCCESS)) {

			*dsn &=  ~DSN_NEVER;
			*dsn |=   DSN_SUCCESS|DSN_FAILURE|DSN_DELAY;
		
			lib_transient(CATGETS(elm_msg_cat, ElmSet, 
					      ElmVfyMenuDsnEnabled,
					      "Delivery notifications enabled.")); 

		    } else if (0 != (*dsn & DSN_SUCCESS)) {
			*dsn &=   ~DSN_SUCCESS;

			if (0 == *dsn) {
			    *dsn = DSN_NEVER;
			    lib_transient(CATGETS(elm_msg_cat, ElmSet, 
						  ElmVfyMenuDsnDisabled,
						  "Delivery notifications disabled."));
			
			} else
			    lib_transient(CATGETS(elm_msg_cat, ElmSet, 
						  ElmVfyMenuDsnSucceesDisabled,
						  "Delivery notification for SUCCESS disabled."));
			
		    } 
		    
		} else {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, 
				      ElmVfyMenuDsnNotSupported,
				      "Mailer does not support Dsn notification settings."));
		    
		}		    
	    }
	    break;

	case 'e': 
	    if (did_prompt) {
		menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmVfyMenuEdit, "Edit"));
		FlushBuffer();
	    }

	    /* can trigger redraw 
	       return 0 on OK */
	    if (0 != edit_the_message(filename, already_has_text,
				 headers,
				 NULL /* use "editor" or "alteditor" */,
				 file_set,
				 mailer_info, mailbox,
				 aview, page)) {
		ret = ft_none;
		goto out;
	    }
	    break;

	case ctrl('F'):
	    if (did_prompt) {
		menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet,
					     ElmPassphraseForget,
					     "Forget passphrase"));
		FlushBuffer();
	    }

	    forget_passphrase();	      
	    break;

	  case 'f':
	    do_no:
	      if (did_prompt) {
		  menu_Write_to_screen(prompt_area,
				       CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuForget, 
					       "Forget"));
		  FlushBuffer();
	      }

	      if (PGP_status)
		  *PGP_status= 0; /* make sure to reset! */
	      
	      if (bytes(filename) <= 0 &&
		  0 == attachments->attachment_count) {
		  ; /* forget about empty files */
		  ret = ft_forget_empty;
		  goto out;
	      } 
	      ret = ft_forget;
	      goto out;
	      /*NOTREACHED*/

	case 'H':
	    if (did_prompt) {
	        menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuPreviewHdrs,
					     "Preview message with headers"));
		FlushBuffer();
	    }

	    switch(pgp_send_confirm_loop(replying,was_pgp_encoded,
					 PGP_status,page,prompt_area)) {
	    case pcf_no: break;
	    case pcf_out: 
		ret = ft_preview_hdrs;
		goto out;
	    case pcf_redraw: goto redraw;
	    case pcf_resize: goto resize_mark;
	    case pcf_EOF:   goto handle_EOF;
	    }
	    break;

	case 'h':
	    if (did_prompt) {
		menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmVfyMenuHeaders,"Headers"));
		FlushBuffer();
	    }

	    edit_headers(headers,mailer_info,
			 mime_info->hdr_charset,
			 mime_info->encode_hdr,
			 aview,parent_message);
	    
	    do_redraw = 1;
	    break;

	case 'i': 
	    if (did_prompt) {
		menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmVfyMenuIspell2,"Ispell"));
		FlushBuffer();
	    }


	    if (have_ISPELL) {

		const char * have_it = have_ispell();

		if (have_it) {
		    char * lbuf = elm_message(FRM("%s %s %s"),
					      have_it, ISPELL_options, filename);
		    Raw(OFF);
		    system_call(lbuf, SY_ENAB_SIGHUP, NULL);
		    
		    do_redraw = 1;
		    Raw(ON);

		    free(lbuf);
		}
	    } else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNoIspellCommand,
				  "Ispell command isn't configured in this version of ELM."));
	
	    break;

	case ctrl('L'):
	    goto redraw;

	case 'p':
	    if (did_prompt) {
		menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmVfyMenuCopyFile, 
					     "Copy file"));
		FlushBuffer();
	    }
	    name_copy_file(copy_file, aview, mailbox,page);
	    menu_trigger_redraw(prompt_area);   
	    break;

        case 'P':
	    if (did_prompt) {
	        menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, 
					     ElmVfyMenuPgp,"Pgp"));
		FlushBuffer();
	    }
	    
	    if (use_PGP) {
		if (PGP_status && !*PGP_status) {
		    *PGP_status = pgp_menu (filename,headers, page);
		    if (*PGP_status)
			def_cmd = *def_ans_yes;
		    else
			def_cmd = 'P';
		    do_redraw = 1;

		} else
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmPgpAlreadyEncSig,
				      "This message is already encrypted and/or signed!"));
	    } else
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmNoPgpConfigured,
				  "PGP isn't configured in this version of ELM."));
            break;

	case 'R':
	    {
		int c = mail_edit_return_path(headers, page);

		DPRINT(Debug,8,(&Debug,"verify_transmission: mail_edit_return_path returned %d\n",c));

		if (headers->env_from)  {
		    *env_from_domain_class_p =
			env_from_classify_domain(headers->env_from,
						 env_from_cdr,
						 mailer_info);
		    env_from_message = 1;
		    if (env_from_cdr->domain_rewritten) {
			mail_show_env_from(headers,*env_from_domain_class_p,page);
			env_from_cdr->domain_rewritten = 0;
		    }
		} else
		    *env_from_domain_class_p = env_from_no_domain;
		
		if (0 == c)
		    bad_cmd = TRUE;
		else if (REDRAW_MARK == c) 
		    goto redraw;		

		menu_trigger_redraw(prompt_area);
	    }
	    break;

	case 'v': 
	    if (did_prompt) {
	        menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuPrevMime,
					     "Preview parts of message"));
		FlushBuffer();
	    }

	    switch(pgp_send_confirm_loop(replying,was_pgp_encoded,
					 PGP_status,page,prompt_area)) {
	    case pcf_no: break;
	    case pcf_out: 
		ret = ft_preview_mime;
		goto out;
	    case pcf_redraw: goto redraw;
	    case pcf_resize: goto resize_mark;
	    case pcf_EOF:   goto handle_EOF;
	    }

	    break;
		

	case 'V':
	    if (did_prompt) {
	        menu_Write_to_screen(prompt_area,
				     CATGETS(elm_msg_cat, ElmSet, ElmVfyMenuPrevMenu,
					     "Preview menu"));
		FlushBuffer();
	    }
	    switch(pgp_send_confirm_loop(replying,was_pgp_encoded,
					 PGP_status,page,prompt_area)) {
	    case pcf_no: break;
	    case pcf_out: 
		ret = ft_preview_menu;
		goto out;
	    case pcf_redraw: goto redraw;
	    case pcf_resize: goto resize_mark;
	    case pcf_EOF:   goto handle_EOF;
	    }

	    break;

	case 'w':
	    if (mailbox) {
		int ch;

		if (did_prompt) {
		    menu_Write_to_screen(prompt_area,
					 CATGETS(elm_msg_cat, ElmSet, 
						 ElmVfyMenuViewFolder,
						 "View folder"));
		    FlushBuffer();
		}

		ch = send_view_folder(mailbox, attachments, page, aview,
				 mailer_info);
		
		if (menu_need_redraw(page)) {
		    menu_ClearScreen(page);

		    do_redraw = 1;
		}

		if (EOF == ch) 
		    goto handle_EOF;

	    } else
		bad_cmd = TRUE;
	    break;
    
	default:

	    {
		int c = presend_action(headers,mailer_info,cmd,
				       &redraw_context,
				       mime_info->hdr_charset,
				       aview, page, prompt_area,
				       parent_message, show_prehdr_normal);

		if (0 == c) {   /* Accept also uppercase letters */
		    int old = cmd;
#ifdef ASCII_CTYPE
		    if (isascii(cmd))
#endif
			cmd = tolower((unsigned char)cmd);
		    
		    if (old != cmd)
			c = presend_action(headers,mailer_info,cmd,
					   &redraw_context,
					   mime_info->hdr_charset,
					   aview, page, prompt_area, parent_message,
					   show_prehdr_normal);

		    if (0 == c)
			bad_cmd = TRUE;
		    else if (REDRAW_MARK == c) {
			force_cmd = cmd;  /* repreat command after redraw */
			goto redraw;		    
		    }
		    clear_hdrmenu_context(&redraw_context);
		    
		} else if (REDRAW_MARK == c)
		    goto redraw;		 

		menu_trigger_redraw(prompt_area);   
	    }
	    
	    break;
	}
    }

    if (0) {	
    handle_EOF:
	DPRINT(Debug,2,(&Debug,"verify_transmission: EOF on verify_transmission\n"));
	
	ret = ft_EOF;
    }

 out:

    clear_hdrmenu_context(&redraw_context);
    if (prompt_menu2_s)
	free_string(&prompt_menu2_s);
    if (prompt_menu1_s)
	free_string(&prompt_menu1_s);
    if (prompt_mssg_s)
	free_string(&prompt_mssg_s);

    DPRINT(Debug,10,(&Debug,"verify_transmission="));
    switch (ret) {
    case ft_none: DPRINT(Debug,10,(&Debug,"ft_none")); break;
    case ft_send: DPRINT(Debug,10,(&Debug,"ft_send")); break;
    case ft_preview: DPRINT(Debug,10,(&Debug,"ft_preview")); break;
    case ft_preview_menu: DPRINT(Debug,10,(&Debug,"ft_preview_menu")); break;
    case ft_preview_mime: DPRINT(Debug,10,(&Debug,"ft_preview_mime")); break;
    case ft_preview_hdrs: DPRINT(Debug,10,(&Debug,"ft_preview_hdrs")); break;
    case ft_forget: DPRINT(Debug,10,(&Debug,"ft_forget")); break;
    case ft_forget_empty: DPRINT(Debug,10,(&Debug,"ft_forget_empty")); break;
    case ft_EOF: DPRINT(Debug,10,(&Debug,"ft_EOF")); break;
    }
    DPRINT(Debug,10,(&Debug," (%d)\n",ret));

    return ret;
}

void write_header_info(mailer, headers,
		       copy, mime_info)
     struct out_state *mailer;
     struct mailing_headers * headers;  
     enum cm_across_mode   copy;
     mime_send_t *mime_info;
{
    enum write_header_mode wmode = write_header_send;


    DPRINT(Debug,14, (&Debug,
		      "write_header_info: copy=%d",
		      copy));
    switch(copy) {
    case cm_across_send:
	DPRINT(Debug,14, (&Debug," cm_across_send"));
	wmode = write_header_send;
	break;
    case cm_across_copy:
	DPRINT(Debug,14, (&Debug," cm_across_copy"));
	wmode = write_header_copy;
	break;
    case cm_across_preview:
	DPRINT(Debug,14, (&Debug," cm_across_preview"));
	wmode = write_header_preview;
	break;	
    }
    DPRINT(Debug,14, (&Debug,"\n"));

    
    /**
       
        Added the ability
	to have backquoted stuff in the users .elmheaders file!
	If copy is TRUE, then treat this as the saved copy of outbound
	mail.
    **/
    

    mime_info -> encode_hdr = !allow_no_hdrencoding;


    
    write_common_mailing_headers(mailer,headers,
				 mime_info->encoding_top,
				 mime_info -> encode_hdr,
				 mime_info->hdr_charset,
				 wmode);
				 
    mime_write_top_headers (mailer, mime_info);

    mime_info->cl_offset = 0;
    if (copy == cm_across_copy) {
	const char * t = NULL;
	int x;

	if (headers->env_from)
	    t = mailer_env_from_value(headers->env_from,&x);

	if (t)
	    write_text_header(mailer,"Return-path",t,
			      mime_info->encoding_top);
	
	/* Only write content-length to copy */
	state_puts("Content-Length: ",mailer);
	mime_info->cl_offset = out_state_ftell(mailer);
	state_puts("          ",mailer); /* Print Blanks as Placeholders */
	print_EOLN(mailer,mime_info->encoding_top);
	state_puts("Status: RO",mailer);
	print_EOLN(mailer,mime_info->encoding_top);
    }
    print_EOLN(mailer,mime_info->encoding_top);
}

static int copy_part P_((mime_send_t *mime_info,struct out_state *dest,
			 FILE *conv_file,struct mime_send_part * X,
			 int copy));
static int copy_part(mime_info,dest,conv_file,X,copy)
     mime_send_t *mime_info;
     struct out_state *dest;
     FILE *conv_file;
     struct mime_send_part * X;
     int copy;
{
    int line_len;
    char buffer[SLEN];			/* file reading buffer */
    long a UNUSED_VAROK = out_state_ftell(dest);
    long pos1;

    if (0 != fseek(conv_file,X->start_loc,SEEK_SET)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSeekFailed,
			  "ELM [seek] failed trying to read %d bytes into file."),
		  (int) X->start_loc);
	return 0;
    }
       
    while ((pos1 = ftell(conv_file)) < X->end_loc &&
	   (line_len = mail_gets(buffer, sizeof buffer -1, conv_file))) {

	/* Text part necessary do not end to \n so there is possibility to
	   get part of next part 
	*/

	if (pos1 + line_len > X->end_loc) {
	    int old = line_len;

	    line_len = X->end_loc - pos1;

	    DPRINT(Debug,4,(&Debug,
			    "copy_part: Got past of end of attachment -- changing line_len %d -> %d\n",
			    old,line_len));
	    if (line_len < 0 || line_len > old)
		mime_panic(__FILE__,__LINE__,
			   "copy_part",
			   "line_len calculation error");

	    if (0 == line_len)
		break;

	}


	if (X->encoding_part == ENCODING_BINARY) {
	    /* \n -> \r\n conversion is done for that part already */
	    state_put(buffer, line_len, dest);
	} else if (mime_info->encoding_top == ENCODING_BINARY) {
	    /* It is better perhaps use canonical eol (CRLF) when mail have
	     * content transfer encoding BINARY somewhere (see notes about 
	     * BINARYMIME)
	     */
	    
	    /* THIS assumes that 
	       dest->EOLN_is_CRLF is set
	    */
	    state_convert_EOLN(buffer,&line_len,sizeof buffer,dest);

	    /* We don't do escaping because line does not end either 
	     * to plain \n now...
	     */
	    state_put(buffer, line_len, dest);
#ifndef DONT_ESCAPE_MESSAGES
	} else if (copy && (strncmp(buffer, "From ", 5) == 0)) {
	    /* Add in the > to a From on our copy */
	    state_puts(">",dest);
	    state_put(buffer, line_len, dest);
#endif
#ifdef NEED_LONE_PERIOD_ESCAPE
	} else if (!copy && strcmp(buffer, ".\n") == 0) {
	    /* Because some mail transport agents take a lone period to
	     * mean EOF, we add a blank space on outbound message.
	     */
	    state_puts(". \n", dest);
#endif /* NEED_LONE_PERIOD_ESCAPE */
	} else {
	    /* SMTP uses \r\n */
	    if (out_state_EOLN_is_CRLF(dest))
		state_convert_EOLN(buffer,&line_len,sizeof buffer,dest);

	    state_put(buffer, line_len, dest);
	}
    }

    if (out_state_ferror(dest) || ferror(conv_file))
	return 0;

    DPRINT(Debug,4, (&Debug,
		     "Preparing mail for %s: part %d -- %d bytes written to temp%s\n",
		     copy ? "copy" : "sending",
		     X-mime_info->top_parts,
		     out_state_ftell(dest)-a,
		     feof(conv_file) ? ", got EOF" : ""));

    return 1;
}
 
int copy_message_across(mime_info, dest, copy, conv_file)
     mime_send_t *mime_info;
     struct out_state *dest;
     enum cm_across_mode copy;
     FILE *conv_file;
{
    char * mode UNUSED_VAROK     = "????";

    /** If copy is TRUE, treat as a saved copy of outbound mail. **/
    
    mime_info -> cl_start = out_state_ftell(dest);

    DPRINT(Debug,14, (&Debug,
		      "copy_message_across: copy=%d",
		      copy));
    switch(copy) {
    case cm_across_send:
	DPRINT(Debug,14, (&Debug," cm_across_send"));
	mode  = "sending";
	break;
    case cm_across_copy:
	DPRINT(Debug,14, (&Debug," cm_across_copy"));
	mode  = "copy";
	break;
    case cm_across_preview:
	DPRINT(Debug,14, (&Debug," cm_across_preview"));
	mode  = "preview";
	break;	
    }
    DPRINT(Debug,14, (&Debug,"\n"));
    
    if (mime_info->msg_is_multipart) {
	int i;

	DPRINT(Debug,4, (&Debug,
			 "Preparing mail for %s: %d parts (multipart)\n",
			 mode,
			 mime_info->top_parts_count));

	for (i = 0; i < mime_info->top_parts_count; i++) {
	    if (cm_across_copy != copy || 
		mime_info->top_parts[i].save_it_on_copy) {

		print_EOLN(dest,mime_info->encoding_top);
		state_printf(dest,FRM("--%s"),mime_info->mime_boundary);
		print_EOLN(dest,mime_info->encoding_top);

		mime_write_part_headers(dest,mime_info,
					&(mime_info->top_parts[i]));
		print_EOLN(dest,mime_info->encoding_top);

		if (!copy_part(mime_info,dest,conv_file,
			       &(mime_info->top_parts[i]),copy)) {
		    return 0;
		}
	    }
	}
	print_EOLN(dest,mime_info->encoding_top);
	state_printf(dest,FRM("--%s--"), mime_info->mime_boundary);
	print_EOLN(dest,mime_info->encoding_top);

    } else if (1 == mime_info->top_parts_count) {
	DPRINT(Debug,4,(&Debug,  
		  "Preparing mail for %s: one part (not multipart)\n",
		  mode));

	if (cm_across_copy != copy || 
	    mime_info->top_parts[0].save_it_on_copy) {
	    if (!copy_part(mime_info,dest,conv_file,
			   &(mime_info->top_parts[0]),copy)) {
		return 0;
	    }
	}
    } else {
	DPRINT(Debug,4,(&Debug,  
			"Preparing mail for %s: ODD: %d parts (NOT multipart)\n",
			mode,mime_info->top_parts_count));
    }

    mime_info->cl_end = out_state_ftell(dest) ;
    if (mime_info->cl_offset > 0) {
	/* go fixup the content length header */
	out_state_fseek(dest, mime_info->cl_offset);
	state_printf(dest, FRM("%ld"), 
		     (long)(mime_info->cl_end - mime_info->cl_start));
	  
	/* Return to the end of the file! */
	out_state_fseek (dest, mime_info->cl_end);
    }

    DPRINT(Debug,4,(&Debug,  
		    "Preparing mail for %s: total %d bytes body, ~%d bytes headers\n",
		    mode,
		    mime_info->cl_end - mime_info -> cl_start,
		    mime_info -> cl_start));
    
    /* !! */
    if (EOF == fflush(out_state_FILE(dest)) || ferror(out_state_FILE(dest))) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmWriteFailedCopy,
			  "Write failed to temp file in copy"));
	return 0;
    }

    return 1;
}

static int have_remote_address P_((struct addr_list * addr_list));
static int have_remote_address(addr_list)
     struct addr_list * addr_list;
{
    int idx;
    int len = addr_list_item_count(addr_list);
    
    /* check To: list */
    
    for (idx = 0; idx < len; idx++) {
	
	int group = -1;
	const struct address * address = 
	    addr_list_get_item(addr_list,idx,&group);
	int is_local = 0;
	
	const char  * ad  = address_get_ascii_addr(address);
	
	const char *ptr;

	if (!ad)
	    continue;
	
	ptr = qstrpbrk_c(ad,":@");
	
	/* Have route, so address is remote */
	if (ptr && ':' == *ptr)
	    return 1;

	/* Address do not have '@' character, 
	   so this address is local         */
	if (!ptr)
	    continue;

	ptr++;

	is_local = build_address_is_local(ptr);

	if (! is_local)
	    return 1;
	
    }

    return 0;
}

static int append_sig(file, headers, page)
     FILE *file;
     struct mailing_headers *headers;
     struct menu_context *page;
{
    /* Append the correct signature file to file.  Return TRUE if
       we append anything.  */
    
    /* Look at the to and cc list to determine which one to use */
    
    /* We could check the bcc list too, but we don't want people to
       know about bcc, even indirectly */
    
    /* Some people claim that  user@anything.same_domain should be 
       considered local.  Since it's not the same machine, better be 
       safe and use the remote sig (presumably it has more complete
       information).  You can't necessarily finger someone in the
       same domain. */

    const char * lsig = give_dt_estr_as_str(&local_signature_e, 
					    "localsignature",NULL,NULL);
    const char * rsig = give_dt_estr_as_str(&remote_signature_e,
					    "remotesignature",NULL,NULL);


    if (!lsig || !rsig)
	error_wait();

    /* FIXME -- is this correct? */
    if (! OPMODE_IS_BATCHMODE(opmode) && (lsig || rsig)) {

	char filename2[SLEN];
	const char *sig = "";

	/* check each @ for @thissite.domain */
	/* if any one is different than this, then use remote sig */

	sig = lsig;

	if (headers->to.addrs &&
	    have_remote_address(headers->to.addrs))
	    sig = rsig;	   	    
	

	/* still local? */         
	if (sig == lsig &&
	    headers->cc.addrs &&
	    have_remote_address(headers->cc.addrs))
	    sig = rsig;
    
	
	if (sig && sig[0]) {  /* if there is a signature file */
	    if (sig[0] != '/')
		elm_sfprintf(filename2, sizeof filename2,
			     FRM("%s/%s"), 
			     home, sig);
	    else
		strfcpy(filename2, sig, sizeof filename2);

	    /* append the file - with a news 2.11 compatible */
	    /* seperator if "sig_dashes" is enabled */
	    (void) append(file, filename2, (sig_dashes ? "\n-- \n" : NULL),
			  page);
	    
	    return TRUE;
	}    
    }
    return FALSE;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
