static char rcsid[] = "@(#)$Id: metapager.c,v 2.38 2023/12/13 16:55:32 hurtta Exp $";

/*******************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.38 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, 
 *                            hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *
 *  Initially written by: Michael Elkins <elkins@aero.org>, 1995
 ******************************************************************/

#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"ui");

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

/* Returns '\0' or EOF */
int PressAnyKeyToContinue(void)
{
    int r = 0;
    struct menu_context *cpage;
    int     old_raw;
    int LINES, COLUMNS;

    if (( old_raw = RawState()) == ON) {
	DPRINT(Debug,9, (&Debug, 
			 "PressAnyKeyToContinue: Called from raw mode\n"));
	Raw(OFF);
    }
    cpage = Raw(ON | NO_TITE);
    menu_get_sizes(cpage,&LINES, &COLUMNS);

 redraw:
    menu_PutLineX(cpage,
		  LINES-1, 0, 
		  CATGETS(elm_msg_cat, ElmSet, ElmMetaPagePressAnyKey,
			  "Press any key to continue..."));
    r = menu_ReadCh(cpage, REDRAW_MARK);
    if (r == REDRAW_MARK)
	goto redraw;

    Raw(OFF | NO_TITE);
    printf("\n\r");

    if (EOF != r)
	r = '\0';

    if (old_raw == ON)
	Raw(ON);

    return r;
}

static void this_message P_((struct out_state *buffer, int *inf));

static void this_message(buffer,inf)
     struct out_state *buffer; 
     int *inf;
{

    if (!*inf)
	state_printf(buffer,CATGETS(elm_msg_cat, ElmSet, 
				     ElmThisMessage,
				     "(** This message "));
    else
	state_printf(buffer,CATGETS(elm_msg_cat, ElmSet, 
				     ElmCommaAnd,
				     ", and "));

    (*inf)++;
}

struct string * title_text(hdr,current,message_count,width,cs)
     struct header_rec *hdr;
     int current, message_count;
     int width;
     charset_t cs;
{
    struct string * res = new_string(cs);
    struct string * buf2 = NULL;
    struct string *t;
    int l1,len;
    enum pager_time_display_v time_display
	= give_dt_enumerate_as_int(& pager_time_display);
    
    if (hdr->status & DELETED)
	t = format_string(CATGETS(elm_msg_cat, ElmSet, 
				  ElmPagerStatDeleted,
				  "[Deleted] %d/%d "),
			  current,message_count);
    else
	t = format_string(CATGETS(elm_msg_cat, ElmSet, 
				  ElmPagerStatMessage,
				  "Message %d/%d "),
			  current,message_count);
    append_string(&res,t,1);
    free_string(&t);
    l1 = string_len(res);

    buf2 = elm_date_text(hdr,time_display,elm_date_show_zone);
    if (!buf2)
	buf2 = new_string(cs);
    
    len = width - 1 - l1 - string_len(buf2);

    if (hdr->from) {
	struct string * buf4;
	int buf4_visible_len = 0;
	int filllen;
	int idx;
	int addr_len = addr_list_item_count(hdr->from);

#define ADDRLEN   15
	
	DPRINT(Debug,9, (&Debug, "title_text: addr_len=%d (item count)\n",
			  addr_len));

	if (addr_len < 1) 
	    goto no_from;

	buf4 = new_string(cs);

	for (idx = 0; idx < addr_len; idx++) {
	    int group = -1;
	    const struct address * address = 
		addr_list_get_item(hdr->from,idx,&group);
	    
	    const char * addr               = address_get_ascii_addr(address);
	    const struct string * fullname  = address_get_phrase(address);
	    
	    int left = len- buf4_visible_len -4;
	    int total = 0;
	    int reserve = addr ? strlen(addr) : 0;
	    

	    if (fullname) {
		DPRINT(Debug,12, (&Debug, "title_text: %d: fullname=%S\n",
				  idx,fullname));

	    }

	    if (reserve > ADDRLEN)
		reserve = ADDRLEN;
	    
	    if (left < 4) {

		DPRINT(Debug,12, (&Debug, "title_text: %d: left=%d (no space)\n",
				  idx,left));

		break;
	    }
	    
	    if (string_len(buf4) > 0) {
		add_ascii_to_string(buf4,s2us(", "));
		buf4_visible_len += 2;
		
		left = len - buf4_visible_len-4;
	    }

	    if (reserve > left) {
		add_ascii_to_string(buf4,s2us("..."));
		buf4_visible_len += 3;
		break;
	    }

	    DPRINT(Debug,12, (&Debug, "title_text: %d: left=%d, reserve=%d\n",
			      idx,left,reserve));


	    if (left > reserve+8 && fullname &&
		(0 < (total = string_len(fullname)))) {
		struct string * Sbase;
		struct string * s1 = NULL;
		int X = 0;
		int visible_len;		    
		int Old;
		
		/* Use convert_string instead of dup_string so 
		 * that text with unknown charset is printed
		 * as [?charset?]
		 */
		Sbase = convert_string(cs,fullname,1);
		
		Old = string_len(Sbase);
		
		s1 = curses_printable_clip(Sbase,&X,len+total,&visible_len,
					   left-reserve-8);
			
		if (s1) {

		    DPRINT(Debug,12, (&Debug, "title_text: %d: s1=%S, visible_len=%d\n",
				      idx,s1,visible_len));


		    append_string(&buf4,s1,1);
		    
		    buf4_visible_len += visible_len;   /* Assumed */
		    free_string(&s1);
		}		    

		if (X < Old) {
		    /* Look if rest of name fits on place of '...' */

		    left = len - buf4_visible_len-4;

		    s1 = curses_printable_clip(Sbase,&X,len+total,&visible_len,
					       left-reserve-4);
			    
		    if (X == Old && s1) {

			append_string(&buf4,s1,1);
				
			buf4_visible_len += visible_len;   /* Assumed */
			
		    } else {
			
			add_ascii_to_string(buf4,s2us("..."));
			buf4_visible_len += 3;
		    }
		    if (s1)
			free_string(&s1);
		}
			
		free_string(&Sbase);

		
	    } else if (total > 0) {
		add_ascii_to_string(buf4,s2us("..."));
		buf4_visible_len += 3;
	    }
	    
	    /* And include address */
	    left = len - buf4_visible_len -4;
	    if (left > 4 && addr) {
		const char *T;
		
		struct string * s0 = NULL;
		struct string * s2 = NULL;
		int X = 0;
		int visible_len;		    
		
		T = addr;
		if ((total=strlen(addr)) > left-3) {
		    int clip = total-left +6;
		    
		    if (clip > total)
			clip=total;
		    
		    T = addr + clip;
		}			
		
		s0 = format_string(FRM(" <%s%s>"),
				   T > addr ? "..." : "",
				   T);
		
		s2 = curses_printable_clip(s0,&X,len+total,&visible_len,
					   left);
		
		free_string(&s0);
		if (s2) {
		    append_string(&buf4,s2,1);
			    
		    buf4_visible_len += visible_len;   /* Assumed */
			    
		    free_string(&s2);
		}		    
	    }
	}

	filllen  = len - buf4_visible_len;
	if (filllen < 0)
	    filllen = 0;
	
	t = format_string(FRM("%S%*s %S"),			    
			  buf4,
			  filllen,"",
			  buf2);
	free_string(&buf4);
    } else {
    no_from:
	if (len > 10) 
	    t = format_string(FRM("(env) %-*.*s %S"),
			      len-7,
			      len-7,
			      hdr->env_from,
			      buf2);
    }
    free_string(& buf2);
    
    append_string(&res,t,1);
    free_string(&t);

    DPRINT(Debug,9, (&Debug, "title_text=%S\n",res));


    
    return res;
}


struct pager_page * init_pager_page(mptr) 
     struct menu_common *mptr;
{
    struct pager_page *p = safe_malloc(sizeof (*p));

    /* bzero is defined on hdrs/defs.h */
    bzero((void *)p, sizeof (*p));

    p->root   = new_menu_context();

    p->border_line  = NULL;
    p->prompt_area  = NULL;

    if (mptr) {
	p->PARAM[0].t = mp_menu_common;
	p->PARAM[1].t = mp_END;
	
	mp_list_set_mcommon(p->PARAM,elm_mp_menu,mptr);
    } else
	p->PARAM[0].t = mp_END;
    

    return p;
}

void exit_pager_page(pager_page,other)
     struct pager_page  ** pager_page;
     struct menu_context * other;
{ 
    if ((*pager_page)->border_line)
	erase_menu_context( &((*pager_page)->border_line));

    if ((*pager_page)->prompt_area)
	erase_menu_context( &((*pager_page)->prompt_area));

    if ((*pager_page)->root)
	erase_menu_context( &((*pager_page)->root));

    menu_trigger_redraw(other);

    free(*pager_page);
    (*pager_page) = NULL;
}


#if ANSI_C
type_mismatch_prompt mime_signature_mismatch;
#endif

/* Return 1 to decode */
int mime_signature_mismatch(ptr, displaying)
     mime_t *ptr;
     int displaying;
{
    int ret = 0;
    struct menu_context  * page = new_menu_context();
    int update = 1;

    int LINES, COLUMNS;

    struct string * filename = NULL;
    
    enum mime_parameters_v mp = 
	give_dt_enumerate_as_int(&mime_parameters);
    
    const struct string *pv;
    const char * pva;
    
    if ( mp > mime_parameters_plain &&
	 (pv = get_mime_param(ptr->DISPOSITION_opts,"filename",
			      mime_rfc1522_filename /* skip compat 
						       paramaters */))) {
	
	filename = dup_string(pv);

    } else if (mime_rfc1522_filename &&
	       (pv = get_mime_param_rfc1522_hack(ptr->DISPOSITION_opts,
						 "filename"))) {
	
	filename = dup_string(pv);

    } else if ( (pva = get_mime_param_compat(ptr->DISPOSITION_opts,
					     "filename"))) {
	
	filename = format_string(FRM("./%s"),pva);
	
    }

    menu_get_sizes(page, &LINES, &COLUMNS);   
    
    while (1) {
	int res;

	if (menu_resized(page)) {
	    menu_get_sizes(page, &LINES, &COLUMNS);  
	    
	    update = 1;
	}
	
	if (menu_need_redraw(page)) {
	    update = 1;
	}
	
	if (update) {
	    menu_ClearScreen(page);

	    menu_StartXX(page,pg_BOLD);
	    menu_print_format_center(page,
				     1, CATGETS(elm_msg_cat, ElmSet, 
						ElmMiscMatchTitle,
						"Mime type check"));
	    menu_EndXX(page,pg_BOLD);

					    
	    menu_PutLineX(page,4,0,
			  CATGETS(elm_msg_cat, ElmSet, 
				  ElmMiscMatchType,"Content-Type: %s/%s"),
			  get_major_type_name(ptr->TYPE), 
			  get_subtype_name(ptr->TYPE));

	    if (filename)
		menu_PutLineX(page,6,0,
			      CATGETS(elm_msg_cat, ElmSet, 
				      ElmMiscMatchFilename,
				      "Filename: %S"),
			      filename);

	    menu_PutLineX(page,9,0,
			  CATGETS(elm_msg_cat, ElmSet, 
				  ElmMiscMatchLine1,
				  "Content of mail part do not match to signature (magic number) of"));

	    menu_PutLineX(page,10,0,
			  CATGETS(elm_msg_cat, ElmSet, 
				  ElmMiscMatchLine2,
				  "Mime content-type. Content-type may be misleading and incorrect."));
	    menu_PutLineX(page,11,0,
			  CATGETS(elm_msg_cat, ElmSet, 
				  ElmMiscMatchLine3,
				  "Viewing it according of content-type may result to incorrect result."));


	    update = 0;
	}


	if (displaying)
	    res = prompt_letter(LINES-3,"",*def_ans_no,
				PROMPT_yesno|PROMPT_cancel|
				PROMPT_redraw_mark|PROMPT_ctrlL,
				page,
				CATGETS(elm_msg_cat, ElmSet, ElmMiscMatchAsk,
					"View %s/%s anyway? (%c/%c) "), 
				get_major_type_name(ptr->TYPE), 
				get_subtype_name(ptr->TYPE),
				*def_ans_yes, *def_ans_no);
	else
	    res = prompt_letter(LINES-3,"",*def_ans_no,
				PROMPT_yesno|PROMPT_cancel|
				PROMPT_redraw_mark|PROMPT_ctrlL,
				page,
				CATGETS(elm_msg_cat, ElmSet, ElmMiscMatchAsk2,
					"Process %s/%s anyway? (%c/%c) "), 
				get_major_type_name(ptr->TYPE), 
				get_subtype_name(ptr->TYPE),
				*def_ans_yes, *def_ans_no);

	    
	if (TERMCH_interrupt_char == res ||
	    EOF == res) {

	    ret = 0;
	    break;
	}

	if (res == ('L'&31) || res == REDRAW_MARK) {
	    menu_ClearScreen(page);   /* Clear possible redraw mark */
	    update = 1;
	}

	if (res == *def_ans_yes) {
	    ret = 1;
	    break;
	}

	if (res == *def_ans_no) {
	    ret = 0;
	    break;
	}
    }

    if (filename)
	free_string(&filename);
    
    erase_menu_context(&page);

    return ret;
}


int metapager (fp, hdr, metapager_flags, current, message_count, pager_page,
	       pager_cmds)
     FILE *fp;
     struct header_rec *hdr;
     int metapager_flags;
     int current;
     int message_count;
     struct pager_page *pager_page;
     struct elm_commands *pager_cmds;
{

    int wait_ret, fork_ret, builtin = 0, status, len;
    int ch = 0, copy_ret;
    FILE                * fpout = NULL;   /* Buffer for external pager */
    struct stringbuffer * bout  = NULL;   /* Buffer for internal pager */
    struct out_state    * buffer = NULL;  /* General handle for both   */

    int do_headers   = 0 != (metapager_flags & METAPAGER_do_headers);
    
    int err;				/* place holder for errno */
    char tempfile[STRING];
    charset_t  charset_vector[255];
    int LINES, COLUMNS;
    unsigned int pager_keyword = 0, pager_mask = 0;
    const char * exp_pager = give_dt_estr_as_str(&pager_e,"pager",
						 &pager_keyword,&pager_mask);
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);

    enum pager_prompt have_index = prompt_return_index;

    if (0 != (metapager_flags & METAPAGER_no_index))
	have_index = prompt_quit_pager;
    else if (0 != (metapager_flags & METAPAGER_index_and_quit))
	have_index = prompt_quit_or_return_index;

    DPRINT(Debug,9, (&Debug, 
		     "metapager: flags %d%s, current %d/%d, have_index=%d",
		     metapager_flags,
		     do_headers ? " (parse headers)" : "",		     
		     current,message_count,have_index));
    switch (have_index) {
    case prompt_quit_pager:           DPRINT(Debug,9, (&Debug, " prompt_quit_pager"));   break;
    case prompt_return_index:         DPRINT(Debug,9, (&Debug, " prompt_return_index")); break;
    case prompt_quit_or_return_index: DPRINT(Debug,9, (&Debug, " prompt_quit_or_return_index"));
	break;
    }
    DPRINT(Debug,9, (&Debug, "\n"));
    
    menu_get_sizes(pager_page->root,&LINES, &COLUMNS);

    /* check to see if we want the internal pager */

    error_wait();  /* Pause if there was error */    

    if (!exp_pager && !pager_keyword) {  /* "none" or error */
	builtin++;	
    } else { 
	switch(pager_keyword) {
	case pager_kw_builtin:
	    builtin++;
	    break;

	case pager_kw_NO:
	    if (builtin_lines < 0 
		? hdr->lines < LINES-1 + builtin_lines 
		: hdr->lines < builtin_lines) {
		
		builtin++;
	    }
	    break;

	default:
	    DPRINT(Debug,9, (&Debug, 
			     "metapager(): bad pager keyword #%d\n",
			     pager_keyword));
	    builtin++;
	    break;
	}
    }
        
    if (!tmp)
	return 0;

    if (!builtin) {
	

	elm_sfprintf (tempfile, sizeof tempfile,
		      FRM("%selm.%d"), tmp, getpid());
	
	unlink(tempfile);
	fpout = safeopen_rdwr(tempfile,NULL);
	unlink(tempfile); /* We can now unlink tempfile ... So nobody
			   * sees data easily (and O_EXCL should prevent
			   * evil playings with symlinks) ...
			   */
	if (!fpout) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmPagerFailedTmpFile,
			      "Failed to create temporary file!"));
	    return 0;
	}
	
	buffer = new_out_state(STATE_out_file);
	set_out_state_file(fpout,buffer);


	/* For external pager we use system charset for display */
	charset_vector[0] = system_charset; 
	charset_vector[1] = NULL;

	set_out_state_cs(buffer,NULL,charset_vector);
						  	
	DPRINT(Debug,9, (&Debug, 
			 "metapager(): using tempfile %s\n", tempfile));
    } else {
	if (hdr -> content_length > 8000) {
	    bout = create_filebuffer();
	    DPRINT(Debug,1,(&Debug,   
			    "metapager(): using stringbuffer (possibly file based)\n"));
	} else {
	    bout = create_membuffer();
	    DPRINT(Debug,1,(&Debug,   
			    "metapager(): using stringbuffer\n"));
	}

	buffer = new_out_state(STATE_out_buffer);

	/* Internal pager uses display_charset ... */
	set_out_state_cs(buffer,NULL,
			 give_display_charsets(charset_vector,
					       sizeof charset_vector / 
					       sizeof (charset_vector[0]),
					       0 /* not signal */
					       ));

	set_out_state_sbuffer (bout,buffer);	
    }
        
    if (fseek (fp, hdr->offset, SEEK_SET) == -1) {
	err = errno;
	DPRINT(Debug,1,(&Debug,   
			"Error: seek %d bytes into file, errno %s (show_message)\n",
			hdr->offset, strerror(err)));
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSeekFailedFile,
			  "ELM [seek] couldn't read %d bytes into file (%s)."),
		  hdr->offset, strerror(err));	
	
	free_out_state(&buffer);

	if (!builtin)
	    fclose(fpout);
	else
	    free_stringbuffer(&bout);
	
	return (0);
    }
    
    /* this is only true if metapager() is not called from ViewAttachment() */
    if (do_headers) {
	header_list_ptr all_hdrs = NULL, last_hdr = NULL;
	int in_envelope = 1;
	char header_buffer[MAX_UNFOLDED_HDR+1];
	int have_title = 0;
    
	if (title_messages && current > 0) {
	    int addempty = 0;
	    const charset_t *Dc =  get_out_state_charset_vector(buffer);

	    struct string *T = title_text(hdr,current,message_count,
					  COLUMNS,
					  Dc[0]);

	    struct pager_range *title_range = 
		state_add_simple_pager_range(buffer,NULL,PR_MAX_WIDTH,0,0);

	    /* first print a title line */
	    int  inf = 0;
	   
	    have_title = 1;

	    /* Only supported by internal pager */
	    set_out_state_line_mode(buffer,pg_UNDERLINE,title_range,1 /* Newline */);

	    state_printf(buffer,FRM("%S\n"),T);
	    free_string(&T);

	    free_pager_range(&title_range);

	    /** Print the subject line centered if there is enough room **/
	    if (hdr->subject &&
		(len = string_len(hdr->subject)) > 0 && 
		matches_weedlist("subject:",have_title)) {
		len = (COLUMNS-len)/2;
		if (len < 0)
		    len = 0;

		state_printf(buffer,
			     FRM("%*.*s%S\n"),
			     len,len,"",hdr->subject);
		addempty++;
	    } else
		len = 0;
            
	    /* now some information about this message */
      
	    if (hdr->status & EXPIRED) {
		this_message(buffer,&inf);
		state_printf(buffer,CATGETS(elm_msg_cat, ElmSet, 
					     ElmHasExpired,
					     "has EXPIRED"));
	    }
	    if (hdr->status & CONFIDENTIAL) {
		this_message(buffer,&inf);
		state_printf(buffer,CATGETS(elm_msg_cat, ElmSet, 
					     ElmTaggedConfidential,
					     "is tagged CONFIDENTIAL"));
	    }      
	    if (hdr->status & URGENT) {
		this_message(buffer,&inf);
		state_printf(buffer,CATGETS(elm_msg_cat, ElmSet, 
					     ElmTaggedUrgent,
					     "is tagged URGENT"));
	    }
	    if (hdr->status & PRIVATE_MAIL) {
		this_message(buffer,&inf);
		state_printf(buffer,CATGETS(elm_msg_cat, ElmSet, 
					     ElmTaggedPrivate,
					     "is tagged PRIVATE"));
	    }
	    if (inf) {
		state_puts(" **)\n", buffer);
		addempty++;
	    }
	    
	    if (addempty)
		state_putc('\n',buffer);
	}

	/** Now do the headers. **/

	/* state_write_header() changes itself buffer.filter according
	 *  of current header charset and buffer.display_charset
	 */


	/* read_header_line with flag = 1 (RHL_MARK_FOLDING)
	 * terminates header with \n and marks folding with \n
	 */
	while (0 < read_header_line (fp, header_buffer, 
				     sizeof header_buffer,
				     RHL_MARK_FOLDING)) {

	    /* read_header_line with flag = 1 (RHL_MARK_FOLDING)
	     * returns \n on EOF
	     */
	    if (0 == strcmp(header_buffer,"\n"))
		break;

	    if (in_envelope) {

		if (have_MMDF &&
		    strcmp (header_buffer, MSG_SEPARATOR) == 0)
		    continue;

		if (0 == strncmp(header_buffer,"From ",5)) {
		    
		    if (! real_from(header_buffer,NULL)) {
			DPRINT(Debug,15,(&Debug,
					 "metapager: Not From separator: %s\n",
					 header_buffer));
			goto badenv;
		    }


		    in_envelope = 2;
		} else if (2 == in_envelope && 
			   first_word_nc(header_buffer, ">From")) {

		    if (! real_from(header_buffer+1,NULL)) {
			DPRINT(Debug,15,(&Debug,
					 "metapager: Not escaped >From separator: %s\n",
					 header_buffer));
			goto badenv;			
		    }

		    /* OK */
		} else {
		badenv:

		    DPRINT(Debug,12,(&Debug,   
				     " .... Leaving envelope part: %s\n",
				     header_buffer));
		    in_envelope = 0;
		    goto not_in_envelope;
		}
		
		if (elm_filter && matches_weedlist(header_buffer,
						   have_title))
		    continue;
		
		state_puts(header_buffer, buffer);

	    }
	    else { 
		char *ptr; 

	    not_in_envelope:
		ptr  = strchr(header_buffer,':');
		if (ptr) {	
		    /* Strip whitespace before ':' */
		    char * k1 = ptr;
		    while(k1 > header_buffer && 
			  whitespace(*(k1-1))) {
			k1 --;
			*k1 = '\0';
		    }
	
		    *ptr = '\0';
		    ptr++;
		    
		    /* Only skip first whitespace for display purposes ... */
		    if (whitespace(*ptr))
			ptr++;		    
		    update_header_list(&all_hdrs, &last_hdr, 
				       header_buffer, ptr);
		} else {
		    DPRINT(Debug,12,(&Debug,   
				     " .... not a header: '%s' (update_header_list not called\n",
				     header_buffer));
		}
	    }
	}

	state_write_headers(buffer,&NULL_decode_opt,
			    all_hdrs,
			    rfc822_header_filter,
			    elm_filter,			    
			    !(hdr -> status & NOHDRENCODING),
			    hdr->header_charset,
			    have_title,

			    /* Assumes that hdr->subject is not converted
			       to another charset */
			    use_ct_cs_for_subj_hack && hdr->subject ?
			    get_string_type(hdr->subject) : NULL
			    );
	
	set_out_state_filter(buffer,NULL);   /* End HACK */
	state_putc('\n',buffer);
	
	delete_headers(&all_hdrs);


	if (hdr->list_info)
	    metapager_list_note(buffer,hdr->list_info);
    }

    

    /* Now this does decoding of MIME and PGP --
       copy_body assumes that headers are already read
    */
    copy_ret = copy_body(fp,hdr,
			 buffer,CM_REMOVE_HEADER|CM_DECODE|CM_DISPLAYING);
    
    free_out_state(&buffer);

    if (!copy_ret) {
	error_wait();

	DPRINT(Debug,1, (&Debug,"metapager: Failed to copy message\n"));

	ch = PressAnyKeyToContinue();
	clear_error();

	goto out;
    }

    if (!builtin)
	rewind(fpout);
    clear_error();
    
    /** Now run the pager! **/
    
    if (builtin) {
	ch = builtinplusplus (bout, pager_page, 0, pager_cmds,
			      have_index);
	free_stringbuffer(&bout);

	goto out;
	
    } else {               /** The rest of the code is for an external pager. **/

	struct menu_context *cpage;
	       
	cpage = Raw(OFF); /* Raw(OFF) must do in parent.... 
			   * Or otherwise in Raw(ON) does not
			   * have effect (because Raw (ON /OFF) 
			   * does nothing if it thinks that mode is
			   * already correct)
			   */


	menu_ClearScreen(cpage);   /* Clear screen before calling external pager 
				      this needs to be come AFTER Raw(OFF) so
				      correct screen is cleared (after ti/te)
				   */
    
	if ((fork_ret = fork()) == -1) {
	    err = errno;
	    DPRINT(Debug,1,(&Debug,    
			    "Error: fork failed, errno %s (metapager)\n",
			    strerror(err)));
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmPreparePagerFork,
			      "Could not prepare for external pager(fork()-%s)."),
		      strerror(err));	
	    ch = PressAnyKeyToContinue();
	    Raw (ON);
	
	    fclose(fpout);

	    goto out;
	} else if (fork_ret == 0) {
	    /* child fork */
	    int p =  getpid();
	    
	    DPRINT(Debug,12,(&Debug,
			     "metapager: on CHILD (pid %d)\n",
			     p));

	    clear_other_actions(p);
	    
	    /* Direct our temporary file to standard input of child.
	     * Because we immediately unlinked it (for security reasons) 
	     * after creating we don't have name for it and
	     * we can't use < in system_call
	     */
	    
	    if (dup2 (fileno(fpout),0) == -1) {
		err = errno;
		DPRINT(Debug,1,(&Debug,   
				"Error: dup failed, errno %s (metapager)\n",
				strerror(err)));	
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmPreparePagerDup,
				  "Could not prepare for external pager(dup()-%s)."),
			  strerror(err));	
		_exit(err);	
	    }
	    
	    fclose(fpout);
	    clear_error();
	    menu_ClearScreen(cpage);
	
	    /* now execute pager and exit */
	    
	    /* system_call() will return user to user's normal permissions. */
	    _exit(system_call(exp_pager, SY_ENAB_SIGINT, NULL));
	}
	
	fclose (fpout);
	
	while (((wait_ret = my_wait (fork_ret,&status)) != fork_ret && wait_ret != -1)
	       /* Handle possible signals ... */	 
	       || (wait_ret == -1 && errno == EINTR))
	    ;
	
	/* turn raw on **after** child terminates in case child
	 * doesn't put us back to cooked mode after we return ourselves to
	 * raw. In that point we don't want output ti yet...
	 */
	cpage = Raw(ON | NO_TITE);
    
	if (prompt_after_pager) {
	    menu_StartXX(cpage,pg_STANDOUT);
	    ch = 0;
	    
	    switch (have_index) {
	    case prompt_return_index:
		menu_PutLineX(cpage,
			      LINES-1, 0, 
			      CATGETS(elm_msg_cat, ElmSet, 
				      ElmCommandIToReturn,
				      " Command ('i' to return to index): "));
		menu_EndXX(cpage,pg_STANDOUT);
		FlushBuffer();
		ch = menu_ReadCh(cpage, 'i' | READCH_CURSOR);
		break;
	    case prompt_quit_pager:
		menu_PutLineX(cpage,
			      LINES-1, 0, 
			      CATGETS(elm_msg_cat, ElmSet, 
				      ElmCommandQToQuitPager,
				      " Command ('q' to quit pager): "));
		menu_EndXX(cpage,pg_STANDOUT);
		FlushBuffer();
		ch = menu_ReadCh(cpage, 'q' | READCH_CURSOR);
		break;
	    case prompt_quit_or_return_index:
		menu_PutLineX(cpage,
			      LINES-1, 0, 
			      CATGETS(elm_msg_cat, ElmSet,
				      ElmCommandQIToQuitPager,
				      " Command ('q' to quit pager, 'i' to return to index): "));
		FlushBuffer();
		ch = menu_ReadCh(cpage, 'q' | READCH_CURSOR);
		break;
	    }
	    menu_EndXX(cpage,pg_STANDOUT);
	}
	else
	    ch = 0;
	
	/* This is necessary so that ti is outputted
	 * after te that was caused on Raw(OFF) before pager.
	 */
	Raw(OFF | NO_TITE);
	Raw(ON);
    }

    if (0 != (metapager_flags & METAPAGER_no_index) &&
	'i' == ch) {

	DPRINT(Debug,9, (&Debug,"metapager: ch='%c' reset\n",
			  ch));
	
	ch = 0;
    } else if (ch == 'q') {
	DPRINT(Debug,9, (&Debug,"metapager: ch='%c' reset\n",
			  ch));
	
	ch = 0;
    }

 out:
    DPRINT(Debug,9, (&Debug,"metapager=%d",ch));
    if (isascii(ch) && isprint(ch)) {
	DPRINT(Debug,9, (&Debug," '%c'",ch));
    }
    DPRINT(Debug,9, (&Debug,"\n"));
    
    return (ch);
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
