static char rcsid[] = "@(#)$Id: quit.c,v 2.14 2020/05/17 06:02:36 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.14 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/quit.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** quit: leave the current folder and quit the program.
  
**/

#include "def_elm.h"
#include "s_elm.h"


DEBUG_VAR(Debug,__FILE__,"ELM");

#include <errno.h>
#ifndef ANSI_C
extern int errno;		/* system error number on failure */
#endif       


/* returns:

   0 .. do not do exit
   1 .. do normal exit
   EOF
*/

int quit(prompt, mailbox, page,prompt_area)
     int prompt;
     struct MailboxView *mailbox;
     struct menu_context  *page;
     struct menu_context  *prompt_area;
{
    int redraw_flag = 0;
    int ret = 0;

 requit:
    if (mailbox) {
	enum sync_mbox_stat status =
	    sync_mbox(FALSE, TRUE, prompt, NULL, mailbox,
		      page,prompt_area);
	switch (status) {
	case sync_mbox_cancel:
	    /* Perhaps Ctrl-C given */
	    lib_transient(CATGETS(elm_msg_cat, ElmSet, 
				  ElmQuitCancelled1,
				  "Quit canceled..."));

	    ret = 0;
	    goto leave_it;
	    
	case sync_mbox_reconnect:
	    ret = 0;
	    goto leave_it;
	    
	case sync_mbox_newmail:	    
	    /* new mail - leave not done - can't change to another file yet
	     * check for change in mailfile_size in main() will do the work
	     * of calling newmbox to add in the new messages to the current
	     * file and fix the sorting sequence that leave_mbox may have
	     * changed for its own purposes */
	    ret = 0;
	    goto leave_it;

	case sync_mbox_failure:
	case sync_mbox_nochange:
	case sync_mbox_changed:
	    break;
	    
	case sync_mbox_EOF:
	    ret = EOF;
	    goto leave_it;
	    /* Read failed, control tty died? */		
	    
	}	

	if (status < sync_mbox_nochange) {
	    int ans;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedLeaveFolder,
			      "Failed to leave folder!"));
	    sleep_message();
	    
	    if (!prompt || !prompt_area) { 
		ret = 0;
		goto leave_it;
	    }
	    
	again1:
	    
	    /* NOTICE: prompt_letter may return EOF */
	    ans = prompt_letter(3,"P",*def_ans_no,
				PROMPT_center|PROMPT_yesno|
				PROMPT_redraw_mark|PROMPT_ctrlL,
				prompt_area,
				CATGETS(elm_msg_cat, ElmSet, ElmFailedLeaveFolderQuit,
					"Failed to leave folder: Try to quit again ?  ([P]anic/%c/%c) "),
				*def_ans_yes, *def_ans_no);
	    
	    if (ans == ('L'&31) || ans == REDRAW_MARK) {
		menu_ClearScreen(page);      

		/* Call refresh routines of children */
		menu_redraw_children(page);
		
		if (menu_need_redraw(prompt_area))
		    menu_ClearScreen(prompt_area);   /* but clear redraw mark from prompt_area*/
		
		redraw_flag = 1;

		goto again1;
	    }

	    if (ans == *def_ans_no) {
		ret = 0;
		goto leave_it;
	    }
	    if (ans == *def_ans_yes) {
		clear_error();				    
		goto requit;
	    }
	  
	    if (ans == 'P')
		emergency_exit(0);

	    if (ans == EOF) {
		ret = EOF;
		goto leave_it;
	    }

	    ret = 0;
	    goto leave_it;
	}
    }
  

    /* May remove mailbox */
    close_cleanup_mbox(mailbox);
    ret = 1;

 leave_it:
    if (redraw_flag)
	menu_trigger_redraw(page);


    DPRINT(Debug,10,(&Debug, "quit: ret=%d",ret));

    return ret;
}

void resync(mailbox, page, LOC)
     struct MailboxView **mailbox;
     struct menu_context  *page;
     struct screen_parts *LOC;
{
    /** Resync on the current folder. Leave current and read it back in.

        If redraw is required, uses 
           menu_trigger_redraw(page)
	  
    **/

    if (*mailbox) {
	int sc,i;	
	int  err = 0;
	int reconnect = 0;
	
	enum sync_mbox_stat
	    result = sync_mbox(TRUE, FALSE, TRUE, NULL, *mailbox, 
			       page,LOC->prompt_page);

	switch (result) {
	case sync_mbox_cancel:

	    /* Perhaps Ctrl-C given */
	    lib_transient(CATGETS(elm_msg_cat, ElmSet, 
				  ElmResyncCancelled1,
				  "Resync canceled..."));
	    return;
	    
	case sync_mbox_reconnect:
	    reconnect = 1;

	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmReConnectedReread,
			      "Folder reconnected; not resynced. Rereading mails!"));
	    
	    sleep_message();
	    
	    goto handle_reconnect;
	case sync_mbox_newmail:
	    /* new mail - leave not done - can't change to another file yet
	     * check for change in mailfile_size in main() will do the work
	     * of calling newmbox to add in the new messages to the current
	     * file and fix the sorting sequence that leave_mbox may have
	     * changed for its own purposes */

	    return;
	    
	case sync_mbox_failure:
	case sync_mbox_nochange:
	case sync_mbox_changed:
	    break;

	case sync_mbox_EOF:
	    /* Read failed, control tty died? */		
	    emergency_exit(0);
	    break;
	    
	}
	
	if (result < 0) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedResyncFolder,
			      "Failed to resync folder!"));
	    sleep_message();

	    return;
	}

    handle_reconnect:
	sc = get_storage_count(*mailbox);
	for (i = 0; i < sc; i++) {
	    struct current_storage *storage = get_storage(*mailbox,i);
	    enum sessionlock_status ret;
	    

	    RECONNECT_MODE reconnect_mode = NULL;
	    
	    if (!storage ||
		! storage->current_folder || !storage->headers) 
		continue;

	    ret = sessionlock_folder(storage->current_folder,
				     SESSIONLOCK_CHECK,&err,
				     &reconnect_mode); 
	    switch (ret) {
	    case sessionlock_fail:

		DPRINT(Debug,1,(&Debug, 
				"resync: given file %s as folder - unreadable",
				storage->current_folder->cur_folder_sys));

		if (err) {
		    DPRINT(Debug,1,(&Debug, " (%s), errno=%d!\n", 
				    strerror(err),err));
		    		
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantOpenFolderRead,
				      "Can't open folder '%S' for reading: %s"), 
			      storage->current_folder->cur_folder_disp,
			      strerror(err));
		} else {
		    DPRINT(Debug,1,(&Debug, "!\n", 
				    strerror(err),err));

		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCantOpenFolderRd0,
				      "Can't open folder '%S' for reading."), 
			      storage->current_folder->cur_folder_disp);
		}
		
		Raw_OFF(1);  /* Will reprint error */

		leave(0,1);
		break;
	    case sessionlock_open:
		 DPRINT(Debug,10,(&Debug,  
				  "resync: %s can be opened.\n",
				  storage->current_folder-> cur_folder_sys));
		 
		newmbox_1(storage->current_folder, mailbox,
			  1 /* append */, 
			  page,reconnect,
			  &reconnect_mode,
			  /* struct quota_display * */ NULL);
		break;

	    case  sessionlock_reconnect:
		 DPRINT(Debug,10,(&Debug,  
				  "resync: %s can be opened, was reconnected.\n",
				  storage->current_folder-> cur_folder_sys));

		newmbox_1(storage->current_folder, mailbox,
			  1 /* append */, 
			  page,
			  1 /* reconnect*/,
			  &reconnect_mode,
			  /* struct quota_display * */ NULL);
		break;
	    }

	    if (reconnect_mode)
		free_reconnect_mode( &reconnect_mode);
	}

    }


    {
	struct menu_common MENU;
	set_mcommon_from_mbxview(&MENU,*mailbox);

	get_page(&MENU, LOC->header_page);  /* resort_mailbox no longer call get_page() */
    }

    menu_trigger_redraw(LOC->header_page);
    menu_trigger_redraw(LOC->title_page);
    
    return;
}

struct MailboxView * change_file(mailbox, aview, parent_page,prompt_area,ret_letter,cmds)
     struct MailboxView  * mailbox;
     struct AliasView    * aview;
     struct menu_context * parent_page;
     struct menu_context * prompt_area;
     int                 * ret_letter;
     struct elm_commands * cmds /* may be NULL */;
{
    /* Prompt user for name of folder to change to.
     * If all okay with that folder, leave the current folder.
     * If leave goes okay (i.e. no new messages in current folder),
     * change to the folder that the user specified.
     *
     * If redraw is needed use menu_trigger_redraw(parent_page)
     */

    int redraw = FALSE;    
    struct folder_info *new_folder = NULL;
    struct folder_browser * XXX = new_browser(selection_folder);
    struct string         * buffer = NULL;
    struct MailboxView    * new_mailbox = NULL;
    enum sessionlock_status sessionlock_ret = sessionlock_fail;
    struct cancel_data *cd = NULL;

        
    struct fbrowser_call * Fcall = 
	alloc_fbrowser_call_f(FB_EXIST|FB_READ,
			     CATGETS(elm_msg_cat, ElmSet, 
				     ElmChangeToWhichFolderFB,
				     "Folder changing selection"));
    int file_changed = 0;

    int current = get_current(mailbox);
    
    const struct remote_server * remote_server =
	give_message_remote_server(mailbox,current-1);
    
    if (remote_server)
	browser_set_remote_server(XXX,remote_server);

    if (ret_letter)
	*ret_letter = '\0';
    
    /* For some reason 0 is OK on input routine */

 ask_again:
    while(1) {
	int mbc;
	int i;
	int err = 0;
	
	new_folder = folder_browser(parent_page,
				    XXX,&buffer,
				    Fcall,
				    aview,
				    mailbox,
				    CATGETS(elm_msg_cat, ElmSet, 
					    ElmChangeToWhichFolder,
					    "Change to which folder: "));
	if (!new_folder)
	    break;
	
	
	DPRINT(Debug,3,(&Debug,  
			"change_file(%p): new folder = %p\n",
			mailbox,new_folder));

	mbc = get_storage_count(mailbox);
	
	for (i = 0; i < mbc; i++) {
	    struct current_storage *storage = get_storage(mailbox,i);

	    if (!storage)
		continue;

	    /* don not accept the same file as the current */
	    if (storage->current_folder && 
		selection_is_folder(XXX,storage->current_folder)) {
		int r;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmAlreadyReadingThatFolder,
				  "Already reading that folder!"));
	    
		/* Closing of IMAP folder may clear that error message
		 * se sleep on here ...
		 */

		FlushBuffer();
		sleep_message();

		setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
		
		DPRINT(Debug,9, (&Debug, 
				 "change_file: freeing new_folder\n"));
		
		r = leave_old_folder(&new_folder,CLOSE_NORMAL,cd);

		if (!r) {
		    DPRINT(Debug,9, (&Debug, 
				     "change_file: leave_old_folder failed\n"));
		}
		if (is_canceled(cd)) {
		    DPRINT(Debug,9, (&Debug, 
				     "change_file: Cancel on leave_old_folder\n"));
		}
		free_cancel(&cd);
		
		goto ask_again;	/* prompt again */
	    }	
	}
	
	/* Make sure this is a file the user can open, unless it is 
	 * the default mailfile, which is openable even if empty 
	 */

	sessionlock_ret = sessionlock_folder(new_folder,SESSIONLOCK_CHECK,
					     &err,NULL);

	switch (sessionlock_ret) {
	    int r;
	case sessionlock_fail:
	    DPRINT(Debug,10,(&Debug,  
			     "change_file: %s can not be opened",
			     new_folder-> cur_folder_sys));
	    if (err) {
		DPRINT(Debug,10,(&Debug, ": %s (errno=%d)",
				 strerror(err),err));
	    }
	    DPRINT(Debug,10,(&Debug, "\n"));

	    setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);

	    DPRINT(Debug,9, (&Debug, 
			     "change_file: freeing new_folder\n"));
	    
	    r = leave_old_folder(&new_folder,CLOSE_NORMAL,cd);
	    
	    if (!r) {
		DPRINT(Debug,9, (&Debug, 
				 "change_file: leave_old_folder failed\n"));
	    }
	    if (is_canceled(cd)) {
		DPRINT(Debug,9, (&Debug, 
				 "change_file: Cancel on leave_old_folder\n"));
	    }
	    free_cancel(&cd);
		
	    continue; 	/* prompt again */
	case sessionlock_open:
	    DPRINT(Debug,10,(&Debug,  
			     "change_file: %s can be opened.\n",
			     new_folder-> cur_folder_sys));
	    break;
	case sessionlock_reconnect:
	    DPRINT(Debug,10,(&Debug,  
			     "change_file: %s can be opened, was reconnected.\n",
			     new_folder-> cur_folder_sys));

	    break;
	}
	
	file_changed = 1;
	break;	/* exit loop - we got the name of a good file */
	
    }
	    
    if (file_changed) {
	int LINES, COLUMNS;
    rechange:

	menu_get_sizes(parent_page, &LINES, &COLUMNS);   

	/* All's clear with the new file to go ahead and leave the current. */
	MoveCursor(LINES-4, 30);
	CleartoEOS();
	
	
	if (mailbox) {
	    enum sync_mbox_stat  result;

	    if (redraw) {
		/* Show parent_page what we are leaving... */
		menu_ClearScreen(parent_page);      

		/* Call refresh routines of children */
		menu_redraw_children(parent_page);
	    }

	    result = sync_mbox(FALSE, FALSE, TRUE, &new_folder, mailbox,
			       parent_page,prompt_area);
	    switch (result) {
	    case sync_mbox_cancel:
		/* Perhaps Ctrl-C given */

		lib_transient(CATGETS(elm_msg_cat, ElmSet,
				  ElmChangeCanceled1,
				  "Folder change canceled..."));
		goto fail;
		
	    case sync_mbox_newmail:
		/* new mail - leave not done - can not change to another file 
		 * yet check for change in mailfile_size in main() will do the
		 * work of calling newmbox to add in the new messages to the 
		 * current file and fix the sorting sequence that leave_mbox 
		 * may have changed for its own purposes */
		
		goto fail;
	    case sync_mbox_reconnect:
		/* new_mail_check() should detect that folder was
		   reconnected, and reread it
		*/
		lib_error(CATGETS(elm_msg_cat, ElmSet,
				  ElmReconnectChangeCanceled,
				  "Folder reconnected; Change canceled..."));
		
		sleep_message();
		goto fail;
	    case sync_mbox_failure:
	    case sync_mbox_nochange:
	    case sync_mbox_changed:
		break;
		
	    case sync_mbox_EOF:

		/* Read failed, control tty died? */		
		
		if (ret_letter) {
		    *ret_letter = EOF;		    
		    goto fail;
		}
		emergency_exit(0);

		break;
	    }	   

	    if (result < sync_mbox_nochange) {
		int ans;
	    
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedLeaveFolder,
				  "Failed to leave folder!"));
		sleep_message();
		
		if (! prompt_area)
		    goto fail;


	    again1:

		/* NOTICE: prompt_letter may return EOF */
		ans = prompt_letter(3,"P",*def_ans_no,
				    PROMPT_center|PROMPT_yesno|
				    PROMPT_redraw_mark|PROMPT_ctrlL,
				    prompt_area,
				    CATGETS(elm_msg_cat, ElmSet, ElmFailedLeaveChange,
					    "Failed to leave folder: Try to change again ? ([P]anic/%c/%c) "),
				    *def_ans_yes, *def_ans_no);

		
		if (ans == ('L'&31) || ans == REDRAW_MARK) {
		    redraw = 1;
		    menu_ClearScreen(parent_page);      
		    goto again1;
		}
		if (ans == *def_ans_no) {
		    goto fail;
		}
		if (ans == *def_ans_yes) {
		    clear_error();
		    goto rechange;
		}
		if (ans == 'P')
		    emergency_exit(0);
		
		if (ans == EOF) {
		    /* Read failed, control tty died? */		

		    if (ret_letter) {
			*ret_letter = EOF;		    
			goto fail;
		    }
		    emergency_exit(0);

		}
		
		goto fail;
	    }
	
	    /* sync_mbox may have closed new_folder ... */
	    
	    if (!new_folder) {
		int err = 0;
		
		DPRINT(Debug,3,(&Debug,  
				"--> Need reopen new folder...\n"));

		new_folder = folder_from_dir_item(XXX,TreatAsSpooled);		

		if (!new_folder)  		    
		    goto fail;   
		

		sessionlock_ret = sessionlock_folder(new_folder,SESSIONLOCK_NORMAL,
						     &err,NULL);
		switch (sessionlock_ret) {
		case sessionlock_fail:
		    DPRINT(Debug,10,(&Debug,  
				     "change_file: %s open failed",
				     new_folder-> cur_folder_sys));
		    if (err) {
			DPRINT(Debug,10,(&Debug, ": %s (errno=%d)",
					 strerror(err),err));
		    }
		    DPRINT(Debug,10,(&Debug, "\n"));
		    		    
		    goto fail;
		    
		case sessionlock_open:
		    DPRINT(Debug,10,(&Debug,  
				     "change_file: %s opened.\n",
				     new_folder-> cur_folder_sys));
		    break;
		case sessionlock_reconnect:
		    DPRINT(Debug,10,(&Debug,  
				     "change_file: %s opened, was reconnected.\n",
				     new_folder-> cur_folder_sys));	       
		    break;
		}
	    }				    
	}    

	/* Always true */
	if (new_folder) {	    
	    struct quota_display * quotad = NULL;
	    struct mail_quota    * mquota = NULL;
	    
	    enum show_mail_quota_v  smq =
		give_dt_enumerate_as_int(&show_mail_quota_e); 
	    
	    DPRINT(Debug,20, (&Debug, 
			      "change_file: show-mail-quota=%d",smq));
	    switch (smq) {
	    case show_mquota_OFF:     DPRINT(Debug,20, (&Debug, " show_mquota_OFF"));     break;
	    case show_mquota_ON_open: DPRINT(Debug,20, (&Debug, " show_mquota_ON_open")); break;
	    case NUM_show_mail_quota: DPRINT(Debug,20, (&Debug, " NUM_show_mail_quota")); break;
	    }
	    DPRINT(Debug,20, (&Debug, "\n"));
	    
	    if (set_window_title || with_title)
		set_folder_window_title(new_folder,
					wanted_title_string,
					wanted_icon_string);

	    switch (smq) {
	    case show_mquota_OFF:
		DPRINT(Debug,20, (&Debug, 
				  "change_file: mail quota display disabled\n"));
		break;
	    case show_mquota_ON_open: 		
		mquota = have_folder_quota(new_folder);
		
		DPRINT(Debug,20, (&Debug, 
				  "change_file: mail quota %s\n",
				  mquota ? "supported" : "not available"));
		break;
	    case NUM_show_mail_quota:  break;
	    }

       	
	    if (mquota) {

		setup_mbx_cancel_message(&cd,mbx_checking_mailbox);
		
		quotad = new_quota_display(mquota,parent_page,cd);

		
		if (is_canceled(cd)) {
		    DPRINT(Debug,9, (&Debug, 
				     "change_file: Cancel on new_quota_display\n"));
		    goto fail;	    
		}
		
		free_cancel(&cd);
	    }
	
	    /* folder is set to NULL, if it is not needed
	       to free
	       
	       read in the folder! 
	    */
	    new_mailbox = enter_mailbox(&new_folder,parent_page,quotad);

	    if (ret_letter && quotad)
		*ret_letter = quota_display_show(quotad,cmds);
	    
	    if (mquota)
		free_mail_quota(&mquota);
	    if (quotad)
		free_quota_display(&quotad,parent_page);
	}
	
	/* No redraws before data is updated ... */
    }   

fail:

    if (new_folder) {
	int r;

	setup_mbx_cancel_message(&cd,mbx_leaving_mailbox);
	
	DPRINT(Debug,9, (&Debug, 
			 "change_file: freeing new_folder\n"));
	
	r = leave_old_folder(&new_folder,CLOSE_NORMAL,cd);
	if (!r) {
	    DPRINT(Debug,9, (&Debug, 
			     "change_file: leave_old_folder failed\n"));
	}
	if (is_canceled(cd)) {
	    DPRINT(Debug,9, (&Debug, 
				     "change_file: Cancel on leave_old_folder\n"));
	}
    }

    if (cd)
	free_cancel(&cd);

    if (buffer)
	free_string(&buffer);

    free_dir(&XXX);
    free_fbrowser_call(&Fcall);

    if (redraw) {
	/* Force default return to parent page ... */
	menu_set_default(parent_page); 

	menu_trigger_redraw(parent_page);
    }
    
    if (!new_mailbox) {
	DPRINT(Debug,3,(&Debug,  
			"change_file(%p) = NULL",mailbox));	
    } else {
	DPRINT(Debug,3,(&Debug,  
			"change_file(%p) = %p",mailbox,new_mailbox));
    }
    if (ret_letter) {
	DPRINT(Debug,9, (&Debug, "; *ret_letter=%d",
			 *ret_letter));
	if (isascii(*ret_letter)) {
	    DPRINT(Debug,9, (&Debug, " (%c)",
			     *ret_letter));
	} else if (EOF ==  *ret_letter) {
	    DPRINT(Debug,9, (&Debug, " EOF"));
	}
    }    
    DPRINT(Debug,3,(&Debug, "\n"));
    
    return new_mailbox;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
