static char rcsid[] = "@(#)$Id: quotadisp.c,v 2.7 2024/10/13 10:57:10 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.7 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org
 *****************************************************************************/

#include "def_elm.h"
#include "s_me.h"
#include "pager.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

struct menu_anon_param {
    struct quota_display *qd;
};

enum { qd_disp_self,
       qd_disp_quotaroot_width     /*  "quotaroot"   = 9 */,
       qd_disp_resource_width      /*  "resource"    = 8 */,
       qd_disp_usage_width         /*  "usage"       = 5 */,
       qd_disp_limit_width         /*  "limit"       = 5 */,
       qd_disp_percent_width       /*  "prct"        = 4 */,
       
       qd_disp_resource_col        /* 20 */,
       qd_disp_usage_col           /* 40 */,
       qd_disp_percent_col         /* 60 */,
       
       qd_disp_COUNT };

static const int default_quotaroot_width = 9;
static const int default_resource_width  = 8;
static const int default_usage_width     = 5;
static const int default_limit_width     = 5;
static const int default_percent_width   = 4;

static const int max_column_step         = 25;


/* -------------------------------------------------------------

Quotaroot    Resource     usage/limit    xxx%


 */


#define QUOTA_DISPLAY_magic	0xF529

struct quota_display {
    unsigned short magic; 

    int           refcount;

    struct menu_context  *current_page;

    struct  menu_context *quota_page;
    
    struct mail_quota_display {
	struct mail_quota  *mquota;

	/* Drawing/Printing algorithm */
	
	int virtual_mq_start_line;    /*  real visible line: *_start_line + top_line - top_offset */
	
	struct mail_quota_list_display {   
	    struct mail_quotaroot_list *quota_list;

	    int virtual_ql_start_line;      /*  real visible line: *_start_line + top_line - top_offset */

	    int virtual_tb_start_line;      /*  real visible line: *_start_line + top_line - top_offset */
	    
	    struct string     ** folder_disp_name_list;
	    size_t               folder_disp_name_count;
	} *    mail_quotarootl_list;
	size_t mail_quotarootl_count;

    } *     mail_quota_list;
    size_t  mail_quota_count;

    /* self reference */
    struct menu_param PARAM[qd_disp_COUNT+1];
    struct menu_anon_param self;


    /* Drawing algorithm */
    
    size_t top_mail_quota;
    size_t top_quotarootl;

    int top_line;      /* Start line after title */
    int top_offset;

    unsigned int is_end:1;
}; 

enum qd_step_mode { qd_step_prev = -1, qd_step_this = 0, qd_step_next };

static void  qd_draw_eoln P_((struct menu_context  * print_area,
			      int                  * current_line_p,
			      int                  * current_col_p,
			      int                  * mayclear_p,
			      int                  * start_of_line_p,
			      int                  * printing_p,
			      int                    empty_line));
static void  qd_draw_eoln(print_area,current_line_p,current_col_p,
			  mayclear_p,start_of_line_p,printing_p,empty_line)
     struct menu_context  * print_area;
     int                  * current_line_p;
     int                  * current_col_p;
     int                  * mayclear_p;
     int                  * start_of_line_p;   
     int                  * printing_p;
     int                    empty_line;
{
    int  L_current_line  = * current_line_p;
    int  L_current_col   = * current_col_p;
    int  L_mayclear      = * mayclear_p;
    int  L_start_of_line = * start_of_line_p;
    int  L_printing      = * printing_p;
    
    int LINES, COLUMNS;
    
    menu_get_sizes(print_area, &LINES, &COLUMNS);

    if (L_current_col < COLUMNS) {

	if (L_current_line <  LINES) {
	
	    menu_MoveCursor(print_area,L_current_line,L_current_col);

	    if (L_mayclear) {
		menu_CleartoEOLN(print_area);

	    }
	    	    
	} else {
	    if (L_printing) {
		DPRINT(Debug,12,(&Debug,"qd_draw_eoln: Gone out of screen, LINES=%d\n",
				 LINES));
		L_printing = 0;
	    }
	}
	    
	if (!L_start_of_line) {

	    if (L_printing) {
		menu_Writechar(print_area,'\r');                   
		menu_Writechar(print_area,'\n');  
	    }
	    
	    L_current_line++;
	}
	
    } else {
	DPRINT(Debug,12,(&Debug,"qd_draw_eoln:  L_current_col = %d >= COLUMNS = %d\n",
			 L_current_col,COLUMNS));
    }

    L_mayclear      = 0;
    L_start_of_line = 1;
    L_current_col   = 0;

    while (empty_line > 0) {
	empty_line--;

	if (L_current_line < LINES) {
	    menu_ClearLine(print_area,L_current_line);
	} else {
				 
	    if (L_printing) {
		DPRINT(Debug,12,(&Debug,"qd_draw_eoln: Gone out of screen, LINES=%d\n",
				 LINES));
		L_printing = 0;
	    }
	}
	
	L_current_line ++;
    }


    DPRINT(Debug,12,(&Debug,"qd_draw_eoln: "));

    if (* current_line_p  !=  L_current_line) {
	DPRINT(Debug,12,(&Debug, " * current_line_p = %d => %d",
			 * current_line_p,L_current_line));
	
	* current_line_p  =  L_current_line;
    }

    if (* current_col_p  != L_current_col) {
	DPRINT(Debug,12,(&Debug, " * current_col_p = %d => %d",
			 * current_col_p,L_current_col));
	
	* current_col_p  = L_current_col;
    }

    if (* mayclear_p      !=  L_mayclear) {
	DPRINT(Debug,12,(&Debug, " * mayclear_p = %d => %d",
			 * mayclear_p,L_mayclear));
	
	* mayclear_p      =  L_mayclear; 
    }

    if (* start_of_line_p !=  L_start_of_line) {
	DPRINT(Debug,12,(&Debug, " * start_of_line_p = %d => %d",
			 * start_of_line_p,L_start_of_line));

	* start_of_line_p =  L_start_of_line;
    }

    if (* printing_p      !=  L_printing) {
	DPRINT(Debug,12,(&Debug," * printing_p = %d => %d",
			 * printing_p,L_printing));
	
	* printing_p      =  L_printing;
    }

    DPRINT(Debug,12,(&Debug,"\n"));
}


static int qd_exceed_or_printing P_((struct menu_context  * print_area,
				     int                  * exceed_p,
				     int                  * current_line_p,
				     int                  * printing_p,
				     const int   top_line));
static int qd_exceed_or_printing(print_area,
				 exceed_p,current_line_p,printing_p,
				 top_line)
     struct menu_context  * print_area;
     int                  * exceed_p;
     int                  * current_line_p;
     int                  * printing_p;
     const int   top_line;
{
    int ret = 0;
    int L_exceed         = * exceed_p;
    int L_current_line   = * current_line_p;
    int L_printing       = * printing_p;

    int LINES, COLUMNS;

    menu_get_sizes(print_area,&LINES, &COLUMNS);

    if (L_exceed)   
	L_exceed--;
    else if (L_current_line >= top_line &&
	     L_current_line < LINES) {

	ret = 1;

    } else if (L_current_line >= LINES) {
	if (L_printing) {
	    DPRINT(Debug,12,(&Debug,"qd_exceed_or_printing: Gone out of screen, LINES=%d\n",
			     LINES));
	    L_printing = 0;
	}
    } else {
	L_current_line++;
    }	 

    DPRINT(Debug,12,(&Debug,"qd_exceed_or_printing=%d",ret));

    if (* exceed_p  != L_exceed) {
	DPRINT(Debug,12,(&Debug," * exceed_p = %d => %d",
			 * exceed_p,L_exceed));
	* exceed_p  = L_exceed;
    }
	
    if (* current_line_p  != L_current_line) {
	DPRINT(Debug,12,(&Debug," * current_line_p = %d => %d",
			 * current_line_p,L_current_line));
	* current_line_p  = L_current_line;
    }

    if (* printing_p      !=  L_printing) {
	DPRINT(Debug,12,(&Debug," * printing_p = %d => %d",
			 * printing_p,L_printing));
	
	* printing_p      =  L_printing;
    }

    
    DPRINT(Debug,12,(&Debug,"\n"));
    return ret;
}


static int qd_draw_text P_((struct menu_context  * print_area,
			    int                  * current_line_p,
			    int                  * current_col_p,
			    int                  * mayclear_p,
			    int                  * joined_line_p,
			    int                  * start_of_line_p,
			    const struct string  * s1));
static int qd_draw_text(print_area,
			current_line_p,current_col_p,
			mayclear_p,joined_line_p,
			start_of_line_p,s1)
     struct menu_context  * print_area;
     int                  * current_line_p;
     int                  * current_col_p;
     int                  * mayclear_p;
     int                  * joined_line_p;
     int                  * start_of_line_p;
     const struct string  * s1;
{
    int ret = 0;
    
    int  L_current_line  = * current_line_p;
    int  L_current_col   = * current_col_p;
    int  L_mayclear      = * mayclear_p;
    int  L_joined_line   = * joined_line_p;
    int  L_start_of_line = * start_of_line_p;

    int pos = 0;
    
    enum span_result span_res = span_ok;

    int s1_len = string_len(s1);

    int LINES, COLUMNS;

    menu_get_sizes(print_area,&LINES, &COLUMNS);

    DPRINT(Debug,12,(&Debug, "qd_draw_text: s1=%S\n",s1));
    
    while (span_res >= span_ok && span_EOS != span_res &&
	   pos < s1_len
	   ) {
	int new_line = 0;
	int old_pos UNUSED_VAROK =  pos;
	

	span_res =  span_words(print_area,
			       L_current_line,&L_current_col,s1,
			       &pos,&L_mayclear,
			       0  /* Flags */,
			       COLUMNS /* max width */,
			       s1_len  /* max_len */,
			       L_joined_line,
			       0   /* p_start_COL */,
			       0   /* p_width -- no limit */,
			       1   /* wrap_indicator */,
			       NULL /* No collapse of space */);

	DPRINT(Debug,12,(&Debug,
			 "qd_draw_text: pos=%d => %d span_res=%d ",
			 old_pos,pos,span_res));
	    
	switch (span_res) {
	case span_clear_alt_data:
	    DPRINT(Debug,12,(&Debug," (span_clear_alt_data)"));

	    /* FALLTHRU */
	    
	case span_failure:  DPRINT(Debug,12,(&Debug,"failure"));
	    if (L_start_of_line) {
		DPRINT(Debug,12,(&Debug,"\n"));
		goto fail;
	    }
	    new_line = 1;   break;
	case span_ok:       DPRINT(Debug,12,(&Debug,"ok"));          break;
	case span_FF_seen:  DPRINT(Debug,12,(&Debug,"span_FF_seen"));
	    new_line = 1;   break;
	case span_EOS:      DPRINT(Debug,12,(&Debug,"span_EOS"));    break;
	case span_NL_seen:  DPRINT(Debug,12,(&Debug,"span_NL_seen"));
	    new_line = 1;   break;
	case span_wrapping: DPRINT(Debug,12,(&Debug,"span_wrapping"));
	    new_line = 1;   break;
	case span_wordwrapped: DPRINT(Debug,12,(&Debug,"span_wordwrapped"));
	    new_line = 1;   break;
	case span_space_collapsed: DPRINT(Debug,12,(&Debug,"span_space_collapsed"));
	    break;
	}
	DPRINT(Debug,12,(&Debug,"\n"));

	L_start_of_line = 0;
	L_joined_line   = 1;

	if (L_mayclear) {
	    menu_CleartoEOLN(print_area);
	    L_mayclear = 0;
	}

	if (new_line) {
	    if (L_current_line >= LINES-1)
		goto fail;
	    
	    menu_Writechar(print_area,'\r');                   
	    menu_Writechar(print_area,'\n');     
			      
	    L_current_line++;
	    L_current_col = 0;
	    L_mayclear = 1;
	    L_start_of_line = 1;			      
	}
    }

    ret = pos >= s1_len;
    
 fail:

    DPRINT(Debug,12,(&Debug,"qd_draw_text=%d: ",ret));

    if (* current_line_p  !=  L_current_line) {
	DPRINT(Debug,12,(&Debug, " * current_line_p = %d => %d",
			 * current_line_p,L_current_line));
	
	* current_line_p  =  L_current_line;
    }

    if (* current_col_p   !=  L_current_col) {
	DPRINT(Debug,12,(&Debug, " * current_col_p = %d => %d",
			 * current_col_p,L_current_col));

	* current_col_p   =  L_current_col;
    }

    if (* mayclear_p      !=  L_mayclear) {
	DPRINT(Debug,12,(&Debug, " * mayclear_p = %d => %d",
			 * mayclear_p,L_mayclear));

	* mayclear_p      =  L_mayclear; 
    }

    if (* joined_line_p   !=  L_joined_line) {
	DPRINT(Debug,12,(&Debug, " * joined_line_p = %d => %d",
			 * joined_line_p,L_joined_line));

	* joined_line_p   =  L_joined_line;	
    }

    if (* start_of_line_p !=  L_start_of_line) {
	DPRINT(Debug,12,(&Debug, " * start_of_line_p = %d => %d",
			 * start_of_line_p,L_start_of_line));

	* start_of_line_p =  L_start_of_line;
    }

    DPRINT(Debug,12,(&Debug,"\n"));

    return ret;
}

static void gd_top_reset_start_line P_((int *virtual_start_line,
				       int top_offset,int add_to_top,
				       int add));
static void gd_top_reset_start_line(virtual_start_line,top_offset,
				   add_to_top,add)
     int *virtual_start_line;
     int top_offset;
     int add_to_top;
     int add;
{
    int L_virtual_start_line = *virtual_start_line;
    
    /* If  block is before current top_quotarootl,
       then it's virtual_start_line < current top_offset 
    */

    DPRINT(Debug,12,(&Debug,"gd_top_reset_start_line: "));
    
    if (L_virtual_start_line >= top_offset) {

	DPRINT(Debug,12,(&Debug," top_offset=%d",top_offset));
	
	/* Reset data */
			
	L_virtual_start_line = 0;
    
    }  else if (add_to_top) {

	DPRINT(Debug,12,(&Debug," add=%d",add));
		
	/* This may cause that 
	   
	   L_virtual_start_line >= qd->top_offset
	   
	   but 
	   
	   top_offset is incremented later
	*/
	L_virtual_start_line += add;			
    }
    
    if (*virtual_start_line != L_virtual_start_line) {
	DPRINT(Debug,12,(&Debug," *virtual_start_line = %d  => %d",
			 *virtual_start_line,L_virtual_start_line));
	
	*virtual_start_line = L_virtual_start_line;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
}

static void qd_top_reset_quotarootl P_((struct mail_quota_list_display *mqld,
				       int top_offset,int add_to_top,
				       int add));
static void qd_top_reset_quotarootl(mqld,top_offset,add_to_top,add)
     struct mail_quota_list_display *mqld;
     int top_offset;
     int add_to_top;
     int add;
{
    /* If struct mail_quota_list_display is before current top_quotarootl,
       then it's virtual_ql_start_line < current top_offset 
    

       This may cause that 
       
       qd-> mail_quota_list[i].mail_quotarootl_list[j].virtual_ql_start_line 
       >= qd->top_offset
       
       but 
       
       qd->top_offset is incremented later
    */

    gd_top_reset_start_line(& (mqld->virtual_ql_start_line),
			    top_offset,add_to_top,add);
    
    gd_top_reset_start_line(& (mqld->virtual_tb_start_line),
			    top_offset,add_to_top,add);
    
    
}

static void qd_top_reset_quota P_((struct mail_quota_display *mqd,
				   int top_offset,
				   int add_to_top,
				   int add));

static void qd_top_reset_quota(mqd,top_offset,add_to_top,add)
     struct mail_quota_display *mqd;
     int top_offset;
     int add_to_top;
     int add;
{
    /* If struct mail_quota_display is before current top_mail_quota,
       then it's virtual_mq_start_line < current top_offset 
    */
    
    if (mqd->virtual_mq_start_line >= top_offset) {
	
	/* Reset data */

	mqd->virtual_mq_start_line  = 0;
		
    } else if (add_to_top) {
	/* This may cause that 

	   qd-> mail_quota_list[i].virtual_mq_start_line >= qd->top_offset
	   
	   but 
	   
	   qd->top_offset is incremented later
	*/

	mqd->virtual_mq_start_line += add;		   
    }

    if (mqd->mail_quotarootl_list) {
	size_t j;
	
	for (j = 0; j < mqd->mail_quotarootl_count; j++) {

	    qd_top_reset_quotarootl(& (mqd->mail_quotarootl_list[j]),
				    top_offset,add_to_top,add);
	}
    }

}
 
static void qd_draw_to_start_line P_((struct menu_context  * print_area,
				      int * current_line_p,
				      int * printing_p,
				      const int   start_line,
				      const int   top_line));
static void qd_draw_to_start_line(print_area,
				  current_line_p,printing_p,
				  start_line,top_line)
     struct menu_context  * print_area;
     int * current_line_p;
     int * printing_p;
     const int   start_line;
     const int   top_line;
{
    int L_current_line   = * current_line_p;
    int L_printing       = * printing_p;

    int LINES, COLUMNS;

    menu_get_sizes(print_area,&LINES, &COLUMNS);

    DPRINT(Debug,12,(&Debug,
		     "qd_draw_to_start_line: L_current_line=%d L_printing=%d start_line=%d top_line=%d\n",
		     L_current_line,L_printing,start_line,top_line));
    
    while (L_current_line < start_line &&
	   L_current_line < LINES) {
		 		 
	if (L_current_line >= top_line &&
	    L_current_line < LINES) {

	    if (!L_printing) {
		DPRINT(Debug,12,(&Debug,"qd_draw_to_start_line: Gone to printing area, top_line=%d\n",
				 top_line));
		L_printing = 1;
	    }
			    
	    menu_ClearLine(print_area,L_current_line);

	}
	
	L_current_line++;
    }

    if (L_current_line >= LINES &&
	L_current_line < start_line) {

	if (L_printing) {
	    DPRINT(Debug,12,(&Debug,"qd_draw_to_start_line: Gone out of screen, LINES=%d\n",
			     LINES));
	    L_printing = 0;
	}
	
	L_current_line = start_line;	
    }
    
    DPRINT(Debug,12,(&Debug,"qd_draw_to_start_line:"));
    
    if (* current_line_p  != L_current_line) {
	DPRINT(Debug,12,(&Debug," * current_line_p = %d => %d",
			 * current_line_p,L_current_line));
	* current_line_p  = L_current_line;
    }

    if (* printing_p      !=  L_printing) {
	DPRINT(Debug,12,(&Debug," * printing_p = %d => %d",
			 * printing_p,L_printing));
	
	* printing_p      =  L_printing;
    }
    
    DPRINT(Debug,12,(&Debug,"\n"));
}

static int virtual_start_set P_((int current_line,int top_line,
				 int *virtual_start_line, int top_offset));

static int virtual_start_set(current_line,top_line,virtual_start_line,top_offset)
     int current_line;
     int top_line;
     int *virtual_start_line;
     int top_offset;
{
    int L_virtual_start_line = *virtual_start_line;
    int ret;

    DPRINT(Debug,12,(&Debug,
		     "virtual_start_set: current_line=%d top_line=%d L_virtual_start_line=%d top_offset=%d\n",
		     current_line,top_line,L_virtual_start_line,top_offset));
    
    if (L_virtual_start_line > 0) {
	ret = 1;
    } else {
	L_virtual_start_line = top_offset + current_line - top_line;

	ret = 0;
    }

    DPRINT(Debug,12,(&Debug,"virtual_start_set=%d:",ret));
    if (*virtual_start_line != L_virtual_start_line) {
	DPRINT(Debug,12,(&Debug," *virtual_start_line = %d => %d",
			 *virtual_start_line,L_virtual_start_line));

	*virtual_start_line = L_virtual_start_line;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
}	 

static int qd_start_line P_((int current_line,int top_line,
			     int *virtual_start_line, int top_offset,
			     int add));
static int qd_start_line(current_line,top_line,virtual_start_line,
			 top_offset,add)
     int current_line;
     int top_line;
     int *virtual_start_line;
     int top_offset;
     int add;
{
    int L_virtual_start_line = *virtual_start_line;
    int start_line;

    DPRINT(Debug,12,(&Debug,
		     "qd_start_line: current_line=%d top_line=%d L_virtual_start_line=%d top_offset=%d add=%d\n",
		     current_line,top_line,L_virtual_start_line,top_offset,add));

    if (current_line < top_line + L_virtual_start_line - top_offset &&
	add > 0) {
	
	L_virtual_start_line += add;
	
    }

    start_line = top_line + L_virtual_start_line - top_offset;
    
    DPRINT(Debug,12,(&Debug,"qd_start_line=%d: ",start_line));
    
    if (*virtual_start_line != L_virtual_start_line) {
	DPRINT(Debug,12,(&Debug," *virtual_start_line = %d => %d",
			 *virtual_start_line,L_virtual_start_line));

	*virtual_start_line = L_virtual_start_line;
    }
    DPRINT(Debug,12,(&Debug,"\n"));

    return start_line;
}

static void qd_update_start_line P_((int current_line,
				     int * start_line,
				     int * virtual_start_line,
				     int * add));
static void qd_update_start_line(current_line,start_line,virtual_start_line,add)
     int current_line;
     int * start_line;
     int * virtual_start_line;
     int * add;
{
    int L_start_line         = * start_line;
    int L_virtual_start_line = * virtual_start_line;
    int L_add                = * add;

    DPRINT(Debug,12,(&Debug, "qd_update_start_line: current_line=%d",
		     current_line));
    
    if (current_line > L_start_line) {

	int diff = current_line - L_start_line;

	DPRINT(Debug,12,(&Debug, " diff=%d",diff));
	
	if (L_virtual_start_line > 0) {
	    	    
	    L_add += diff;
			 
	}

	L_virtual_start_line += diff;

	L_start_line += diff;
    }

    if (* start_line        != L_start_line) {
	DPRINT(Debug,12,(&Debug, " * start_line = %d => %d",
			 * start_line,L_start_line));
	
	* start_line         = L_start_line;
    }

    if (* virtual_start_line != L_virtual_start_line) {
	DPRINT(Debug,12,(&Debug, " * virtual_start_line = %d => %d",
			 * virtual_start_line,L_virtual_start_line));
	
	* virtual_start_line = L_virtual_start_line;
    }

    if (* add                !=  L_add) {
	DPRINT(Debug,12,(&Debug, " * add = %d => %d",
			 * add,L_add));
	
	* add                =  L_add;
    }
    
    DPRINT(Debug,12,(&Debug, "\n"));
}

static int top_start_reached P_((const int top_offset,
				 const int            add,
				 int                  * before_top_p,
				 const int              add_to_top,
				 int                  * virtual_start_line_p,
				 int                  * exceed_p
				 ));
static int top_start_reached(top_offset,add,before_top_p,
			     add_to_top,virtual_start_line_p,exceed_p)
     const int top_offset;
     const int              add;
     int                  * before_top_p;
     const int              add_to_top;
     int                  * virtual_start_line_p;
     int                  * exceed_p;
{
     int L_before_top   = * before_top_p;
     int L_virtual_start_line = * virtual_start_line_p;
     int L_exceed       = * exceed_p;
     int ret = 0;

     int the_add = 0;

     if (add_to_top && add > 0)
	 the_add = add;
     
     if (L_before_top) {

	 if (L_virtual_start_line > 0) {

	     if (L_virtual_start_line + the_add >= top_offset) {
		 DPRINT(Debug,12,(&Debug,"top_start_reached: L_virtual_start_line=%d (add %d) reached\n",
				  L_virtual_start_line,the_add));
		 
		 L_before_top = 0;
		 L_exceed = 0;
		 ret = 1;	     
		 
	     } else {
		 
		 L_virtual_start_line += the_add;
		 L_exceed = top_offset - L_virtual_start_line;
		 
	     }
	 } else  {
	     DPRINT(Debug,12,(&Debug,"top_start_reached: L_virtual_start_line not set, assuming that it is reached\n"));
	     
	     L_before_top = 0;
	     L_exceed = 0;
	     ret = 1;	     
	 }
	     
     } else {
	 ret      = 1;
	 L_exceed = 0;
     }

     DPRINT(Debug,12,(&Debug,"top_start_reached=%d",ret));

     if (* before_top_p       !=  L_before_top) {
	 DPRINT(Debug,12,(&Debug,"  * before_top_p=%d => %d",
			  * before_top_p,L_before_top));
	 
	 * before_top_p       =  L_before_top;
     }

     if (* virtual_start_line_p !=  L_virtual_start_line) {
	 DPRINT(Debug,12,(&Debug,"* virtual_start_line_p=%d => %d",
			  * virtual_start_line_p,L_virtual_start_line));
	 
	 * virtual_start_line_p =  L_virtual_start_line;
     }

     if (* exceed_p           !=  L_exceed) {
	 DPRINT(Debug,12,(&Debug,"  * exceed_p=%d => %d",
			  * exceed_p,L_exceed));
	 
	 * exceed_p           =  L_exceed;
     }

     DPRINT(Debug,12,(&Debug,"\n"));
     
     return ret;
}

enum qd_column {
    qd_quotaroot,
    qd_resource,
    qd_usage,
    qd_limit,
    qd_percent };

static struct resize_columns {

    int quotaroot_width;
    int resource_width;
    int usage_width;
    int limit_width;
    int percent_width;
    
} NULL_resize_columns = {
    0,0,0,0,0
};

enum qd_setup_mode { qd_setup_change = 0, qd_setup_initial = 1 };

static void qd_setup_columns P_((struct menu_context  *ptr,
				 struct menu_param *list,
				 enum qd_setup_mode initial,
				 struct resize_columns * resize
				 ));


enum qd_align { qd_align_left, qd_align_center, qd_align_rigth };


static void qd_reach_col P_((struct menu_context   * print_area,
			     int                   * current_line_p,
			     const int               col,
			     int                   * start_of_line_p
			     ));
static void qd_reach_col(print_area,current_line_p,col,start_of_line_p)
     struct menu_context   * print_area;
     int                   * current_line_p;
     const int               col;
     int                   * start_of_line_p;
{
    int L_current_col   =  * current_line_p;
    int L_start_of_line =  * start_of_line_p;

    int LINES, COLUMNS;
    
    menu_get_sizes(print_area,&LINES, &COLUMNS);
    
    if ( L_current_col < col && L_current_col < COLUMNS) {

	L_start_of_line = 0;

	while (L_current_col < col && L_current_col < COLUMNS) {
	    menu_Writechar(print_area,' ');
	    L_current_col++;
	}
		 

    }
    
    DPRINT(Debug,12,(&Debug,"qd_reach_col:"));

    
    if (* current_line_p  != L_current_col) {
	DPRINT(Debug,12,(&Debug,
			 " * current_line_p = %d => %d",
			 * current_line_p,L_current_col));
	
	* current_line_p  = L_current_col;
    }

    if (* start_of_line_p != L_start_of_line) {
	DPRINT(Debug,12,(&Debug,
			 " * start_of_line_p = %d => %d",
			 * start_of_line_p,L_start_of_line));
	* start_of_line_p = L_start_of_line;
    }

    DPRINT(Debug,12,(&Debug,"\n"));
}

			    
static void qd_draw_cell P_((struct menu_context   * print_area,
			     const enum qd_column    column,
			     int                   * current_line_p,
			     int                   * current_col_p,
			     int                   * redrawing_p,
			     struct string         * text,
			     struct resize_columns * resize,
			     struct menu_param     * list,
			     const int               text_flags,
			     const enum qd_align     align,
			     int                   * mayclear_p,
			     int                   * start_of_line_p
			     ));
static void qd_draw_cell(print_area,column,current_line_p,current_col_p,
			 redrawing_p,text,resize,list,text_flags,align,
			 mayclear_p,start_of_line_p)
     struct menu_context   * print_area;
     const enum qd_column    column;
     int                   * current_line_p;
     int                   * current_col_p;
     int                   * redrawing_p;
     struct string         * text;
     struct resize_columns * resize;
     struct menu_param     * list;
     const int               text_flags;
     const enum qd_align     align;
     int                   * mayclear_p;
     int                   * start_of_line_p;
{
     int L_current_line  = * current_line_p;
     int L_current_col   = * current_col_p;
     int L_redrawing     = * redrawing_p;
     int L_mayclear      = * mayclear_p;
     int L_start_of_line = * start_of_line_p;
     
     int LINES, COLUMNS;

     menu_get_sizes(print_area,&LINES, &COLUMNS);

     if (L_current_line < LINES && L_current_col < COLUMNS) {
	 int width = COLUMNS - L_current_col;
	 int col   = L_current_col;
	 const char * col_name UNUSED_VAROK = "??";
	 	 
	 menu_MoveCursor(print_area, L_current_line,L_current_col);

	 switch (column) {
	 case qd_quotaroot:
	     col_name = "quotaroot";
	     col      = 0;
	     width    = mp_lookup_integer(list,qd_disp_quotaroot_width);
	     break;
	 case qd_resource:
	     col_name = "resource";
	     col      = mp_lookup_integer(list,qd_disp_resource_col);
	     width    = mp_lookup_integer(list,qd_disp_resource_width);
	     break;
	 case qd_usage:
	     col_name = "usage";
	     col      = mp_lookup_integer(list,qd_disp_usage_col);
	     width    = mp_lookup_integer(list,qd_disp_usage_width);
	     break;
	 case qd_limit:
	     col_name = "limit";
	     col      = mp_lookup_integer(list,qd_disp_usage_col) +
		 mp_lookup_integer(list,qd_disp_usage_width) + 1;
	     width    =  mp_lookup_integer(list,qd_disp_limit_width);
	     break;
	 case qd_percent:
	     col_name = "percent";
	     col      = mp_lookup_integer(list,qd_disp_percent_col);
	     width    = mp_lookup_integer(list,qd_disp_percent_width);
	     break;
	 }

	 if (L_current_col <= col) {
	     int L = string_len(text);
	     int visible_len = 0;
	     int tail_len    = 0;
	     int temp UNUSED_VAROK;

	     DPRINT(Debug,12,(&Debug,
			      "qd_draw_cell: column=%d %s col=%d width=%d, text len=%d text=%S\n",
			      column,col_name,col,width,L,text));

	     if (L_current_col < col && L_current_col < COLUMNS) {

		 qd_reach_col(print_area,&L_current_col,col,&L_start_of_line);
		 
	     }
	     
	     if ((temp = L_current_col + width) > COLUMNS) {
		  DPRINT(Debug,12,(&Debug,
				   "qd_draw_cell: Overflow L_current_col %d + width %d = %d > COLUMNS %d -- width=%d",
				   L_current_col,width,temp,COLUMNS,width));

		  width = COLUMNS - L_current_col;
		  DPRINT(Debug,12,(&Debug," => %d\n",
				   width));
	     }
	     
	     menu_StartXX(print_area,text_flags);
	     
	     if (L > 0 && width > 0) {
		 int pos = 0;
		 char * tail = "";
		 
		 
		 struct string * S =
		     curses_printable_clip(text,&pos,L,
					   &visible_len,width);

		 if (S) {
		 
		     int need = visible_len + ( L - pos );  /* Not correct actually */
		 
		     if (pos < L && width > 3) {
			 free_string(&S);
			 pos = 0;
			 
			 S =
			     curses_printable_clip(text,&pos,L,
						   &visible_len,width);
			 
			 tail = "...";
			 tail_len = 3;
		     } else if (pos == L && visible_len < width) {
			 int fill = 0;
			 int x;

			 DPRINT(Debug,12,(&Debug,
					  "qd_draw_cell: visible_len=%d < width=%d align=%d",
					  visible_len,width,align));
			 switch (align) {
			 case qd_align_left:
			     DPRINT(Debug,12,(&Debug," qd_align_left"));
			     fill = 0;
			     break;
			 case qd_align_center:
			     DPRINT(Debug,12,(&Debug," qd_align_center"));
			     fill = (width - visible_len) /2;
			     break;
			 case qd_align_rigth:
			     DPRINT(Debug,12,(&Debug," qd_align_rigth"));
			     fill = width - visible_len;
			     break;
			 }

			 DPRINT(Debug,12,(&Debug," fill=%d",fill));

			 if (fill > 0) {
			     DPRINT(Debug,12,(&Debug,
					      " L_current_col=%d",
					      L_current_col));
			     for (x = 0; x < fill; x++) {
				 menu_Writechar(print_area,' ');
				 L_current_col++;
			     }
		 
			     DPRINT(Debug,12,(&Debug," => %d",L_current_col));
			 }

			 DPRINT(Debug,12,(&Debug,"\n"));
		     }

		     if (visible_len > 0) {
			 menu_Write_to_screen(print_area,
					      FRM("%S"),S);

			 DPRINT(Debug,12,(&Debug,
					  "qd_draw_cell: L_current_col=%d",
					  L_current_col));

			 L_start_of_line = 0;
			 L_current_col += visible_len;

			 DPRINT(Debug,12,(&Debug," => %d  column=%d %s text=%S\n",
					  L_current_col,column,col_name,S));

		     }

		     free_string(&S);

		     if (need > width) {
			 DPRINT(Debug,12,(&Debug,
					  "qd_draw_cell: need=%d > width=%d",
					  need,width));

			 switch (column) {
			 case qd_quotaroot:
			     if (resize) {
				 DPRINT(Debug,12,(&Debug,
						  ", resize quotaroot_width=%d",resize->quotaroot_width));
				 if (need > resize->quotaroot_width) {
				     resize->quotaroot_width = need;
				     DPRINT(Debug,12,(&Debug," => %d",resize->quotaroot_width));
				     L_redrawing = 1;
				 }
			     }
			     break;
			 case qd_resource:
			      if (resize) {
				  DPRINT(Debug,12,(&Debug,
						   ", resize resource_width=%d",resize->resource_width));
				  if (need > resize->resource_width) {
				      resize->resource_width = need;
				      DPRINT(Debug,12,(&Debug," => %d",resize->resource_width));
				      L_redrawing = 1;
				  }
			      }
			      break;
			 case qd_usage:
			     if (resize) {
				 DPRINT(Debug,12,(&Debug,
						  ", resize usage_width=%d",resize->usage_width));
				 if (need > resize->usage_width) {
				     resize->usage_width = need;
				     DPRINT(Debug,12,(&Debug," => %d",resize->usage_width));
				     L_redrawing = 1;
				 }
			     }
			     break;
			 case qd_limit:
			     if (resize) {
				 DPRINT(Debug,12,(&Debug,
						  ", resize limit_width=%d",resize->limit_width));
				 if (need > resize->limit_width) {
				     resize->limit_width = need;
				     DPRINT(Debug,12,(&Debug," => %d",resize->limit_width));
				 }
			     }
			     break;
			 case qd_percent:
			     break;
			 }

			 DPRINT(Debug,12,(&Debug,"\n"));
		     }


		 } else if (width >= 3) {
		     tail = "-?-";
		     tail_len = 3;
		 }

		 if (tail && tail_len > 0 && width >=  visible_len + tail_len) {
		     
		     menu_PutLine0(print_area,L_current_line,L_current_col,tail);

		      DPRINT(Debug,12,(&Debug,
				       "qd_draw_cell: L_current_col=%d",
				       L_current_col));

		      L_start_of_line = 0;
		      L_current_col += tail_len;
		     
		      DPRINT(Debug,12,(&Debug," => %d  column=%d %s tail=%s\n",
				       L_current_col,column,col_name,tail));
		     
		 }


		 if (L_current_col < col + width) {

		     L_start_of_line = 0;
		     
		     DPRINT(Debug,12,(&Debug,
				       "qd_draw_cell: L_current_col=%d",
				      L_current_col));

		     while (L_current_col < col + width) {
			 menu_Writechar(print_area,' ');
			 L_current_col++;
		     }
		 
		     DPRINT(Debug,12,(&Debug," => %d\n",L_current_col));		     		     
		 }
		 
	     } else if (L_current_col >= COLUMNS) {
		  DPRINT(Debug,12,(&Debug,
				   "qd_draw_cell: Overflow L_current_col=%d >= COLUMNS=%d  column=%d %s\n",
				   L_current_col,COLUMNS,column,col_name));
	     }

	     	     
	     menu_EndXX(print_area,text_flags);

	     
	 } else {
	     DPRINT(Debug,12,(&Debug,
			      "qd_draw_cell: Overflow L_current_col=%d > col=%d column=%d %s\n",
			      L_current_col,col,column,col_name));	     
	 }
	 
     } else {

	 DPRINT(Debug,12,(&Debug,
			  "qd_draw_cell: Overflow L_current_line=%d %s LINES=%d, L_current_col=%d %s COLUMNS=%d\n",
			  L_current_line,
			  L_current_line < LINES ? "<" : ">=",
			  LINES,
			  L_current_col,
			  L_current_col < COLUMNS ? "<" : ">",
			  COLUMNS));
	 
     }
     
     DPRINT(Debug,12,(&Debug,"qd_draw_cell:"));
     if (* current_line_p !=  L_current_line) {
	 DPRINT(Debug,12,(&Debug," * current_line_p = %d => %d",
			  * current_line_p,L_current_line));
	 * current_line_p =  L_current_line;
     }
     if (* current_col_p  != L_current_col) {
	 DPRINT(Debug,12,(&Debug," * current_col_p  = %d => %d",
			  * current_col_p,L_current_col));	 
	 * current_col_p  = L_current_col;
     }
     if (* redrawing_p    != L_redrawing) {
	 DPRINT(Debug,12,(&Debug,"  * redrawing_p    = %d => %d",
			  * redrawing_p,L_redrawing));
	 
	 * redrawing_p    = L_redrawing;
     }
     if (* mayclear_p      !=  L_mayclear) {
	 DPRINT(Debug,12,(&Debug, " * mayclear_p = %d => %d",
			  * mayclear_p,L_mayclear));
	 
	 * mayclear_p      =  L_mayclear; 
     }
     if (* start_of_line_p !=  L_start_of_line) {
	 DPRINT(Debug,12,(&Debug, " * start_of_line_p = %d => %d",
			  * start_of_line_p,L_start_of_line));
	 
	 * start_of_line_p =  L_start_of_line;
     }
     DPRINT(Debug,12,(&Debug,"\n"));


}

static void qd_draw_quotaroot_name P_((struct menu_context   * print_area,
				       struct mail_quotaroot * qr,
				       int                   * current_line_p,
				       int                   * current_col_p,
				       int                   * redrawing_p,
				       struct resize_columns * resize,
				       struct menu_param     * list,
				       int                   * mayclear_p,
				       int                   * start_of_line_p));
static void qd_draw_quotaroot_name(print_area,qr,current_line_p,current_col_p,
				   redrawing_p,resize,list,mayclear_p,start_of_line_p)
     struct menu_context   * print_area;
     struct mail_quotaroot * qr;
     int                   * current_line_p;
     int                   * current_col_p;
     int                   * redrawing_p;
     struct resize_columns * resize;
     struct menu_param     * list;
     int                   * mayclear_p;
     int                   * start_of_line_p;
{

    enum quota_reserved_quotaroot qr_name_token = qr_root_none;
    struct string               * qr_name /* May be NULL */ =
	quotaroot_name(qr,&qr_name_token);
    
    int L_current_line  = * current_line_p;
    int L_current_col   = * current_col_p;
    int L_redrawing     = * redrawing_p;
    int L_mayclear      = * mayclear_p;
    int L_start_of_line = * start_of_line_p;
    int printed = 0;
    
    struct string * qr_NA =
	format_string(CATGETS(elm_msg_cat, MeSet,
			      MeMQNA,
			      "n/a"));


    DPRINT(Debug,12,(&Debug,"qd_draw_quotaroot_name: qr_name_token=%d",
		     qr_name_token));
    switch (qr_name_token) {
    case  qr_root_none: DPRINT(Debug,12,(&Debug," qr_root_none")); break;
    }        
    if (qr_name) {
	DPRINT(Debug,12,(&Debug," qr_name=%S",qr_name));
    }    
    DPRINT(Debug,12,(&Debug,"\n"));

    switch (qr_name_token) {
    case  qr_root_none:
	qd_draw_cell(print_area,qd_quotaroot,&L_current_line,
		     &L_current_col,&L_redrawing,
		     qr_name ? qr_name : qr_NA,
		     resize,list,0,qd_align_left,
		     &L_mayclear,&L_start_of_line);
	printed = 1;
	break;
    }

    
    if (! printed) {
	DPRINT(Debug,12,(&Debug,"qd_draw_quotaroot_name: Bad qr_name_token\n"));
	
	qd_draw_cell(print_area,qd_quotaroot,&L_current_line,
		     &L_current_col,&L_redrawing,
		     qr_name ? qr_name : qr_NA,
		     resize,list,0,qd_align_left,
		     &L_mayclear,&L_start_of_line);	
    }

    if (qr_name)
	free_string(&qr_name);

    free_string(&qr_NA);
    
    DPRINT(Debug,12,(&Debug,"qd_draw_quotaroot_name:"));
     if (* current_line_p !=  L_current_line) {
	 DPRINT(Debug,12,(&Debug," * current_line_p = %d => %d",
			  * current_line_p,L_current_line));
	 * current_line_p =  L_current_line;
     }
     if (* current_col_p  != L_current_col) {
	 DPRINT(Debug,12,(&Debug," * current_col_p  = %d => %d",
			  * current_col_p,L_current_col));	 
	 * current_col_p  = L_current_col;
     }
     if (* redrawing_p    != L_redrawing) {
	 DPRINT(Debug,12,(&Debug,"  * redrawing_p    = %d => %d",
			  * redrawing_p,L_redrawing));
	 
	 * redrawing_p    = L_redrawing;
     }
     if (* mayclear_p      !=  L_mayclear) {
	 DPRINT(Debug,12,(&Debug, " * mayclear_p = %d => %d",
			  * mayclear_p,L_mayclear));
	 
	 * mayclear_p      =  L_mayclear; 
     }
     if (* start_of_line_p !=  L_start_of_line) {
	 DPRINT(Debug,12,(&Debug, " * start_of_line_p = %d => %d",
			  * start_of_line_p,L_start_of_line));
	 
	 * start_of_line_p =  L_start_of_line;
     }
     DPRINT(Debug,12,(&Debug,"\n"));

}

static void qd_draw_resource_name  P_((struct menu_context    * print_area,
				       struct mail_quota_item * qr_item,
				       int                    * current_line_p,
				       int                    * current_col_p,
				       int                    * redrawing_p,
				       struct resize_columns  * resize,
				       struct menu_param      * list,
				       int                    * mayclear_p,
				       int                    * start_of_line_p));

static void qd_draw_resource_name(print_area,qr_item,current_line_p,current_col_p,
				  redrawing_p,resize,list,mayclear_p,start_of_line_p)
     struct menu_context    * print_area;
     struct mail_quota_item * qr_item;
     int                    * current_line_p;
     int                    * current_col_p;
     int                    * redrawing_p;
     struct resize_columns  * resize;
     struct menu_param      * list;
     int                   * mayclear_p;
     int                   * start_of_line_p;
{
    enum quota_reserved_resource qr_item_token = qr_res_none;
    struct string              * qr_item_name  /* May be NULL */ =
	quota_item_name(qr_item,&qr_item_token);

    int L_current_line  = * current_line_p;
    int L_current_col   = * current_col_p;
    int L_redrawing     = * redrawing_p;
    int L_mayclear      = * mayclear_p;
    int L_start_of_line = * start_of_line_p;
    int printed = 0;
    
    struct string * qr_NA =
	format_string(CATGETS(elm_msg_cat, MeSet,
			      MeMQNA,
			      "n/a"));

    DPRINT(Debug,12,(&Debug,"qd_draw_resource_name: qr_item_token=%d",
		     qr_item_token));
    switch (qr_item_token) {
    case qr_res_none: DPRINT(Debug,12,(&Debug," qr_res_none")); break;
    }
    if (qr_item_name) {
	DPRINT(Debug,12,(&Debug," qr_item_name=%S",qr_item_name));
    }    
    DPRINT(Debug,12,(&Debug,"\n"));

    
    switch (qr_item_token) {
    case qr_res_none:
	qd_draw_cell(print_area,qd_resource,&L_current_line,
		     &L_current_col,&L_redrawing,
		     qr_item_name ? qr_item_name : qr_NA,
		     resize,list,0,qd_align_left,
		     &L_mayclear,&L_start_of_line);
	printed = 1;
	break;
    }

    if (! printed) {
	DPRINT(Debug,12,(&Debug,"qd_draw_resource_name: Bad qr_item_token\n"));
	
	qd_draw_cell(print_area,qd_resource,&L_current_line,
		     &L_current_col,&L_redrawing,
		     qr_item_name ? qr_item_name : qr_NA,
		     resize,list,0,qd_align_left,
		     &L_mayclear,&L_start_of_line);
    }
    
    if (qr_item_name)
	free_string(&qr_item_name);

    free_string(&qr_NA);
    
    DPRINT(Debug,12,(&Debug,"qd_draw_resource_name:"));
    if (* current_line_p !=  L_current_line) {
	DPRINT(Debug,12,(&Debug," * current_line_p = %d => %d",
			 * current_line_p,L_current_line));
	* current_line_p =  L_current_line;
    }
    if (* current_col_p  != L_current_col) {
	DPRINT(Debug,12,(&Debug," * current_col_p  = %d => %d",
			 * current_col_p,L_current_col));	 
	* current_col_p  = L_current_col;
    }
    if (* redrawing_p    != L_redrawing) {
	DPRINT(Debug,12,(&Debug,"  * redrawing_p    = %d => %d",
			 * redrawing_p,L_redrawing));
	 
	* redrawing_p    = L_redrawing;
    }
    if (* mayclear_p      !=  L_mayclear) {
	DPRINT(Debug,12,(&Debug, " * mayclear_p = %d => %d",
			 * mayclear_p,L_mayclear));
	
	* mayclear_p      =  L_mayclear; 
    }
    if (* start_of_line_p !=  L_start_of_line) {
	DPRINT(Debug,12,(&Debug, " * start_of_line_p = %d => %d",
			 * start_of_line_p,L_start_of_line));
	
	* start_of_line_p =  L_start_of_line;
    }
    DPRINT(Debug,12,(&Debug,"\n"));    
}

static struct scale {
    char         * unit;
    unsigned long  factor;
    unsigned long  limit;

} scales[] = {
    { "",       1,        0 },
    { "k",     1000,    10000 },
    { "M",  1000000, 10000000 }
};

static void qd_draw_resource_values P_((struct menu_context    * print_area,
					struct mail_quota_item * qr_item,
					int                    * current_line_p,
					int                    * current_col_p,
					int                    * redrawing_p,
					struct resize_columns  * resize,
					struct menu_param      * list,
					int                   * mayclear_p,
					int                   * start_of_line_p));
					       
static void qd_draw_resource_values(print_area,qr_item,current_line_p,current_col_p,
				  redrawing_p,resize,list,mayclear_p,start_of_line_p)
     struct menu_context    * print_area;
     struct mail_quota_item * qr_item;
     int                    * current_line_p;
     int                    * current_col_p;
     int                    * redrawing_p;
     struct resize_columns  * resize;
     struct menu_param      * list;
     int                   * mayclear_p;
     int                   * start_of_line_p;
{
    
    int L_current_line  = * current_line_p;
    int L_current_col   = * current_col_p;
    int L_redrawing     = * redrawing_p;
    int L_mayclear      = * mayclear_p;
    int L_start_of_line = * start_of_line_p;

    uint32 qr_item_usage = 0;
    uint32 qr_item_limit = 0;
    int qr_item_flag     =
	quota_item_values(qr_item,&qr_item_usage,&qr_item_limit);

    int scale_index = (sizeof scales) / sizeof (scales[0]) - 1;

    struct string * usage_s = NULL;
    struct string * limit_s = NULL;
    struct string * prct_s = NULL;

    int LINES, COLUMNS;
    menu_get_sizes(print_area,&LINES, &COLUMNS);
    
    /* ;; Unsigned 32-bit integer
       ;; (0 <= n < 4,294,967,296)
    */

    while (scale_index > 0) {
	if (ison(QUOTA_ITEM_have_usage,qr_item_flag)) {

	    if (qr_item_usage < scales[scale_index].limit) {
		scale_index--;
		continue;
	    }

	}

	if (ison(QUOTA_ITEM_have_limit,qr_item_flag)) {
	    if (qr_item_limit < scales[scale_index].limit) {
		scale_index--;
		continue;
	    }	    
	}
       	
	break;
    }
    

    DPRINT(Debug,12,(&Debug,"qd_draw_resource_values: scale_index=%d unit=%s factor=%lu\n",
		     scale_index,scales[scale_index].unit,scales[scale_index].factor));

    if (ison(QUOTA_ITEM_have_usage,qr_item_flag)) {
	unsigned long ul = qr_item_usage;

	usage_s =
	    format_string(FRM("%lu%s"),
			  ul / scales[scale_index].factor,
			  scales[scale_index].unit);
	
    } else {

	usage_s = format_string(FRM("-"));

    }

    qd_draw_cell(print_area,qd_usage,&L_current_line,
		 &L_current_col,&L_redrawing,usage_s,
		 resize,list,0,qd_align_rigth,
		 &L_mayclear,&L_start_of_line);
    
    if (L_current_line < LINES && L_current_col < COLUMNS) {
	menu_Writechar(print_area,'/');
	L_current_col++;
    }
        
    if (ison(QUOTA_ITEM_have_limit,qr_item_flag)) {
	unsigned long ul = qr_item_limit;

	limit_s =
	    format_string(FRM("%lu%s"),
			  ul / scales[scale_index].factor,
			  scales[scale_index].unit);

	
    } else {
	limit_s =
	    format_string(FRM("-"));

    }

    qd_draw_cell(print_area,qd_limit,&L_current_line,
		 &L_current_col,&L_redrawing,limit_s,
		 resize,list,0,qd_align_left,
		 &L_mayclear,&L_start_of_line);

    if (ison(QUOTA_ITEM_have_usage,qr_item_flag) && 0 == qr_item_usage) {
	prct_s =
	    format_string(FRM("0%%"));
    } else if (ison(QUOTA_ITEM_have_usage,qr_item_flag) &&
	       ison(QUOTA_ITEM_have_limit,qr_item_flag) &&
	       qr_item_limit > 0) {

	if (qr_item_usage == qr_item_limit) {
	    prct_s =
		format_string(FRM("100%%"));

	} else {
	    float f = 100.0 * qr_item_usage / qr_item_limit;

	    if (f < 999.0) {
		int v = (int)f;

		if ((float)v < f && v < 100)
		    v++;

		prct_s =
		    format_string(FRM("%d%%"),
				  v);
				
	    } else {
		prct_s =
		    format_string(FRM("-"));
	    }	    
	}
	
    } else {
	prct_s =
	    format_string(FRM("-"));	
    }

    qd_draw_cell(print_area,qd_percent,&L_current_line,
		 &L_current_col,&L_redrawing,prct_s,
		 resize,list,0,qd_align_rigth,
		 &L_mayclear,&L_start_of_line);

    free_string(&prct_s);
    free_string(&limit_s);
    free_string(&usage_s);
    
    
    DPRINT(Debug,12,(&Debug,"qd_draw_resource_values:"));
    if (* current_line_p !=  L_current_line) {
	DPRINT(Debug,12,(&Debug," * current_line_p = %d => %d",
			 * current_line_p,L_current_line));
	* current_line_p =  L_current_line;
    }
    if (* current_col_p  != L_current_col) {
	DPRINT(Debug,12,(&Debug," * current_col_p  = %d => %d",
			 * current_col_p,L_current_col));	 
	* current_col_p  = L_current_col;
    }
    if (* redrawing_p    != L_redrawing) {
	DPRINT(Debug,12,(&Debug,"  * redrawing_p    = %d => %d",
			 * redrawing_p,L_redrawing));
	 
	* redrawing_p    = L_redrawing;
    }
    if (* mayclear_p      !=  L_mayclear) {
	DPRINT(Debug,12,(&Debug, " * mayclear_p = %d => %d",
			 * mayclear_p,L_mayclear));
	
	* mayclear_p      =  L_mayclear; 
    }
    if (* start_of_line_p !=  L_start_of_line) {
	DPRINT(Debug,12,(&Debug, " * start_of_line_p = %d => %d",
			 * start_of_line_p,L_start_of_line));
	
	* start_of_line_p =  L_start_of_line;
    }
    DPRINT(Debug,12,(&Debug,"\n"));    

}


static void qd_draw_mail_quota  P_((struct menu_context  * print_area,
				    struct quota_display * qd,
				    const size_t           mq_ptr,
				    const size_t           start_quotaroot,
				   
				    int                  * add_p,
				    int                  * current_line_p,
				    int                  * printing_p,
				    int                  * redrawing_p,
				    int                  * before_top_p,
				    const int              add_to_top,
				    struct cancel_data   * cd,
				    struct resize_columns * resize,
				    struct menu_param     * list,
				    int                   * add_line_p
				    ));
static void qd_draw_mail_quota(print_area,qd,mq_ptr,start_quotaroot,
			       add_p,current_line_p,
			       printing_p,redrawing_p,before_top_p,
			       add_to_top,cd,resize,list,add_line_p)
     struct menu_context   * print_area;
     struct quota_display  * qd;
     const size_t            mq_ptr;
     const size_t            start_quotaroot;
     int                   * add_p;
     int                   * current_line_p;
     int                   * printing_p;
     int                   * redrawing_p;
     int                   * before_top_p;
     const int               add_to_top;
     struct cancel_data    * cd;
     struct resize_columns * resize;
     struct menu_param     * list;
     int                   * add_line_p;
{
     int L_add =          * add_p;
     int L_current_line = * current_line_p;
     int L_printing     = * printing_p;
     int L_redrawing    = * redrawing_p;
     int L_before_top   = * before_top_p;
     int L_add_line     = * add_line_p;
     
     int LINES, COLUMNS;
     int exceed = 0;
     
     menu_get_sizes(print_area,&LINES, &COLUMNS);

     if (QUOTA_DISPLAY_magic != qd->magic)
	 panic("MBX PANIC",__FILE__,__LINE__,
	       "qd_draw_mail_quota",
	       "Bad magic number (quota_diplay)",0);

     if (mq_ptr > qd->mail_quota_count)
	 panic("MBX PANIC",__FILE__,__LINE__,
	       "Bad mq_ptr",
	       "Bad magic number (quota_diplay)",0);

	 
     if (top_start_reached(qd->top_offset,L_add,&L_before_top,add_to_top,
			   & (qd-> mail_quota_list[mq_ptr].virtual_mq_start_line),
			   &exceed)) {
     
	 DPRINT(Debug,12,(&Debug,
			  "qd_draw_mail_quota: L_current_line = %d top_line=%d mail_quota_list[%zu].virtual_mq_start_line=%d top_offset=%d L_add=%d\n",
			  L_current_line,qd->top_line,
			  mq_ptr,qd-> mail_quota_list[mq_ptr].virtual_mq_start_line,
			  qd->top_offset,L_add));

	 if (virtual_start_set(L_current_line,qd->top_line,
			       & (qd-> mail_quota_list[mq_ptr].virtual_mq_start_line),
			       qd->top_offset)) {
	 
	     int mq_start_line =
		 qd_start_line(L_current_line,qd->top_line,
			       & (qd-> mail_quota_list[mq_ptr].virtual_mq_start_line),
			       qd->top_offset,L_add);
	 
	     qd_draw_to_start_line(print_area,&L_current_line,
			       &L_printing,mq_start_line,qd->top_line);
	 
	     if (L_current_line > mq_start_line) {
		 
		 int diff = L_current_line - mq_start_line;
		 
		 if (qd-> mail_quota_list[mq_ptr].virtual_mq_start_line > 0) {
		     
		     DPRINT(Debug,12,(&Debug,
				      "qd_draw_mail_quota: L_add = %d",
				      L_add));
		     
		     L_add += diff;
		     
		     DPRINT(Debug,10,(&Debug," => %d\n",
				      L_add));
		 }
		 
		 DPRINT(Debug,10,(&Debug,
				  "qd_draw_mail_quota: mail_quota_list[%zu].virtual_mq_start_line = %d",
				  mq_ptr,qd-> mail_quota_list[mq_ptr].virtual_mq_start_line));
	     
		 qd-> mail_quota_list[mq_ptr].virtual_mq_start_line += diff;
	     
		 DPRINT(Debug,12,(&Debug," => %d\n",
				  qd-> mail_quota_list[mq_ptr].virtual_mq_start_line));
	     
		 mq_start_line += diff;
	     }
	     
	     DPRINT(Debug,12,(&Debug,
			      "qd_draw_mail_quota: mail_quota_list[%zu]: L_current_line = %d %s mq_start_line=%d\n",
			      mq_ptr,L_current_line,
			      (L_current_line == mq_start_line) ? "==" : "<>",
			      mq_start_line));
	 } else {
	     DPRINT(Debug,12,(&Debug,
			      "qd_draw_mail_quota: mail_quota_list[%zu].virtual_mq_start_line = %d -- was not set\n",
			      mq_ptr,qd-> mail_quota_list[mq_ptr].virtual_mq_start_line));
	 }
	 
     } else {
	 DPRINT(Debug,12,(&Debug,
			  "qd_draw_mail_quota: mail_quota_list[%zu].virtual_mq_start_line = %d -- %d before top\n",
			  mq_ptr,qd-> mail_quota_list[mq_ptr].virtual_mq_start_line,
			  exceed));
     }
	    
     if (qd-> mail_quota_list[mq_ptr].mail_quotarootl_list) {
	 size_t j;
	 	 
	 for (j = qd->top_quotarootl; j < qd-> mail_quota_list[mq_ptr].mail_quotarootl_count; j++) {

	     if (L_add_line) {

		 if (L_current_line >= qd->top_line &&
		     L_current_line < LINES) {

		     if (!L_printing) {
			 DPRINT(Debug,12,(&Debug,"qd_draw_mail_quota: Gone to printing area, top_line=%d\n",
					  qd->top_line));
			 L_printing = 1;
		     }
		     
		     menu_ClearLine(print_area,L_current_line);
		 }

		 L_current_line ++;

		 if (L_current_line >= LINES) {
		     if (L_printing) {
			 DPRINT(Debug,12,(&Debug,"qd_draw_mail_quota: Gone out of screen, LINES=%d\n",
					  LINES));
			 L_printing = 0;
		     }
		 }		 
	     }
	     
	     if (top_start_reached(qd->top_offset,L_add,&L_before_top,add_to_top,
				   &( qd->mail_quota_list[mq_ptr].
				      mail_quotarootl_list[j].
				      virtual_ql_start_line ),
				   &exceed)) {
		 
		 DPRINT(Debug,12,(&Debug,
				  "qd_draw_mail_quota: L_current_line = %d top_line=%d qd->mail_quota_list[%zu].mail_quotarootl_list[%zu].virtual_ql_start_line=%d top_offset=%d L_add=%d\n",
				  L_current_line,qd->top_line,
				  mq_ptr,j,qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				  virtual_ql_start_line,
				  qd->top_offset,L_add));

		 if (virtual_start_set(L_current_line,qd->top_line,
				       &( qd->mail_quota_list[mq_ptr].
					  mail_quotarootl_list[j].
					  virtual_ql_start_line ),
				       qd->top_offset)) {
		 
		     int ql_start_line = qd_start_line(L_current_line,qd->top_line,
						       &( qd->mail_quota_list[mq_ptr].
							  mail_quotarootl_list[j].
							  virtual_ql_start_line ),
						       qd->top_offset,L_add);
		 
		 
		     qd_draw_to_start_line(print_area,&L_current_line,&L_printing,
					   ql_start_line,qd->top_line);
		     
		     qd_update_start_line(L_current_line,&ql_start_line,
					  & (qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
					     virtual_ql_start_line),&L_add);
		 } else {
		     DPRINT(Debug,12,(&Debug,
				      "qd_draw_mail_quota: mail_quota_list[%zu].mail_quotarootl_list[%zu].virtual_ql_start_line -- was not set\n",
				      mq_ptr,j,
				      qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				      virtual_ql_start_line));
		 }
		 
		 /* Draw folder names */
		 
		 if (L_current_line >= qd->top_line &&
		     L_current_line < LINES) {
		     
		     int current_col = 0;
		     
		     int mayclear = 1;
		     int joined_line = 0;
		     int start_of_line = 1;
		     
		     struct string * s1 = format_string(CATGETS(elm_msg_cat, MeSet,
								MeMailQuota,
								"Mail quota"));
		     
		     if (!L_printing) {
			 DPRINT(Debug,12,(&Debug,"qd_draw_mail_quota: Gone to printing area, top_line=%d\n",
					  qd->top_line));
			 L_printing = 1;
		     }
		     
		     menu_MoveCursor(print_area,L_current_line,current_col);
		     
		     if (!qd_draw_text(print_area,&L_current_line,&current_col,
				       &mayclear,&joined_line,&start_of_line,
				       s1))
			 goto fail;
		     
		     if (qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].folder_disp_name_list) {
			 size_t k;
			 
			 struct string * s2 = format_string(CATGETS(elm_msg_cat, MeSet,
								    MeQuotaFor,
								    " for "));
			 
			 struct string * s3 = format_string(FRM(", "));
			 struct string * delim = s2;
			 
			 for (k = 0; k < qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				  folder_disp_name_count ; k++) {
			     
			     if (qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				 folder_disp_name_list[k]) {
				 if (!qd_draw_text(print_area,&L_current_line,&current_col,
						   &mayclear,&joined_line,&start_of_line,
						   delim))
				     goto fail2;
				 
				 if (!qd_draw_text(print_area,&L_current_line,&current_col,
						   &mayclear,&joined_line,&start_of_line,
						   qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
						   folder_disp_name_list[k]))
				     goto fail2;
				 
				 delim = s3;
			     }
			 }
			 
			 
			 if ( qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].quota_list) {
			     struct string * s4 = format_string(FRM(": "));
			     
			     
			     if (!qd_draw_text(print_area,&L_current_line,&current_col,
					       &mayclear,&joined_line,&start_of_line,
					       s4))
				 goto fail3;

			     qd_draw_eoln(print_area,&L_current_line,&current_col,
					  &mayclear,&start_of_line,&L_printing,1);
			     
			 fail3:
			     free_string(&s4);
			 }
			 
		     fail2:
			 free_string(&s3);
			 free_string(&s2);
			 
		     }
		     
		 fail:
		     free_string(&s1);
		     
		 } else if (L_current_line >= LINES) {
		     if (L_printing) {
			 DPRINT(Debug,12,(&Debug,"qd_draw_mail_quota: Gone out of screen, LINES=%d\n",
					  LINES));
			 L_printing = 0;
		     }
		 } else {
		     L_current_line++;
		 }
		 		 		
	     } else {
		 DPRINT(Debug,12,(&Debug,
				  "qd_draw_mail_quota: mail_quota_list[%zu].mail_quotarootl_list[%zu].virtual_ql_start_line = %d -- %d before top\n",
				  mq_ptr,j,
				  qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				  virtual_ql_start_line,
				  exceed));
	     }
		 
		   		    		   
	     if (top_start_reached(qd->top_offset,L_add,&L_before_top,add_to_top,
				   &( qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				      virtual_tb_start_line),
				   &exceed)) {
		     
		 DPRINT(Debug,12,(&Debug,
				  "qd_draw_mail_quota: L_current_line = %d top_line=%d qd->mail_quota_list[%zu].mail_quotarootl_list[%zu].virtual_tb_start_line=%d  top_offset=%d L_add=%d\n",
				  L_current_line,qd->top_line,
				  mq_ptr,j,qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				  virtual_tb_start_line,
				  qd->top_offset,L_add));

		 if (virtual_start_set(L_current_line,qd->top_line,
				       &( qd->mail_quota_list[mq_ptr].
					  mail_quotarootl_list[j].
					  virtual_tb_start_line),
				       qd->top_offset)) {
		     int tb_start_line = qd_start_line(L_current_line,qd->top_line,
						       &( qd->mail_quota_list[mq_ptr].
							  mail_quotarootl_list[j].
							  virtual_tb_start_line),
						       qd->top_offset,L_add);
		     
		     qd_draw_to_start_line(print_area,&L_current_line,&L_printing,
					   tb_start_line,qd->top_line);
		     
		     qd_update_start_line(L_current_line,&tb_start_line,
					  &( qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
					     virtual_tb_start_line),&L_add);
		 } else {
		     DPRINT(Debug,12,(&Debug,
				      "qd_draw_mail_quota: mail_quota_list[%zu].mail_quotarootl_list[%zu].virtual_tb_start_line = %d -- was not set\n",
				      mq_ptr,j,
				      qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				      virtual_tb_start_line));
		 }

		 L_add_line = 1;
	     } else {
		 DPRINT(Debug,12,(&Debug,
				  "qd_draw_mail_quota: mail_quota_list[%zu].mail_quotarootl_list[%zu].virtual_tb_start_line = %d -- %d before top\n",
				  mq_ptr,j,
				  qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].
				  virtual_tb_start_line,
				  exceed));
	     }
	
	     
	     if (qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].quota_list) {

		 struct mail_quotaroot_list * const ql
		     = qd->mail_quota_list[mq_ptr].mail_quotarootl_list[j].quota_list;
		 
		 const size_t quota_list_len =
		     quotaroot_list_len(ql);

		 struct string * qr_NA =
		     format_string(CATGETS(elm_msg_cat, MeSet,
					   MeMQNA,
					   "n/a"));

		 
		 if (quota_list_len) {
		     size_t k;
		     
		     /* Show title */

		     /* Should title always printed? */

		     if (qd_exceed_or_printing(print_area,
					       &exceed,&L_current_line,&L_printing,qd->top_line)) {

			 int current_col = 0;
			 int mayclear = 1;
			 int start_of_line = 1;

			 struct string * s_qr = format_string(CATGETS(elm_msg_cat, MeSet,
								      MeMQQuotaRootCol,
								      "Quotaroot"));

			 struct string * s_res = format_string(CATGETS(elm_msg_cat, MeSet,
								      MeMQResourceCol,
								      "Resource"));


			 struct string * s_usa = format_string(CATGETS(elm_msg_cat, MeSet,
								      MeMQUsageCol,
								      "Usage"));
			 
			 struct string * s_lim = format_string(CATGETS(elm_msg_cat, MeSet,
								       MeMQLimitCol,
								       "Limit"));

			 struct string * s_prcnt = format_string(CATGETS(elm_msg_cat, MeSet,
								       MeMQLimitPrcnt,
								       "Prct"));
			 
			 
			 qd_draw_cell(print_area,qd_quotaroot,&L_current_line,
				      &current_col,&L_redrawing,s_qr,
				      resize,list,pg_UNDERLINE,qd_align_center,
				      &mayclear,&start_of_line);


			 qd_draw_cell(print_area,qd_resource,&L_current_line,
				      &current_col,&L_redrawing,s_res,
				      resize,list,pg_UNDERLINE,qd_align_center,
				      &mayclear,&start_of_line);

			 qd_draw_cell(print_area,qd_usage,&L_current_line,
				      &current_col,&L_redrawing,s_usa,
				      resize,list,pg_UNDERLINE,qd_align_rigth,
				      &mayclear,&start_of_line);

			 if (L_current_line < LINES && current_col < COLUMNS) {
			     menu_Writechar(print_area,'/');
			     current_col++;
			 }
			 
			 qd_draw_cell(print_area,qd_limit,&L_current_line,
				      &current_col,&L_redrawing,s_lim,
				      resize,list,pg_UNDERLINE,qd_align_left,
				      &mayclear,&start_of_line);

			 qd_draw_cell(print_area,qd_percent,&L_current_line,
				      &current_col,&L_redrawing,s_prcnt,
				      resize,list,pg_UNDERLINE,qd_align_rigth,
				      &mayclear,&start_of_line);
			 
			 qd_draw_eoln(print_area,&L_current_line,&current_col,
				      &mayclear,&start_of_line,&L_printing,0);

			 free_string(&s_prcnt);
			 free_string(&s_lim);
			 free_string(&s_usa);
			 free_string(&s_res);
			 free_string(&s_qr);
		     } 
		     
		     for (k = 0; k < quota_list_len; k++) {
			 struct mail_quotaroot * qr =
			     quotaroot_list_item(ql,k,cd);
			 
			 if (qr) {
			     enum quota_reserved_quotaroot qr_name_token = qr_root_none;
			     struct string * qr_name /* May be NULL */ =
				 quotaroot_name(qr,&qr_name_token);
			     size_t qr_item_count         =  quotaroot_item_count(qr);
			     enum quota_quotaroot_status qr_status =
				 quotaroot_status(qr);
			     struct string * status_text = NULL;
			     int   show_empty = 0;

			     size_t m;

			     DPRINT(Debug,12,(&Debug,
					      "qd_draw_mail_quota: mail_quota_list[%zu].mail_quotarootl_list[%zu]: quotaroot %zu/%zu: qr_status=%d",
					      mq_ptr,j,
					      k,quota_list_len,qr_status));
			     
			     switch (qr_status) {
			     case qr_stat_failed: 	 DPRINT(Debug,12,(&Debug," qr_stat_failed"));  break;
			     case qr_stat_none:          DPRINT(Debug,12,(&Debug," qr_stat_none"));    break;
			     case qr_stat_empty:         DPRINT(Debug,12,(&Debug," qr_stat_empty"));   break;
			     case qr_stat_normal:        DPRINT(Debug,12,(&Debug," qr_stat_normal"));  break;
			     }
			     if (qr_name) {
				 DPRINT(Debug,12,(&Debug," qr_name=%S",qr_name));
			     }
			     DPRINT(Debug,12,(&Debug," qr_item_count=%zu\n",qr_item_count));

			     switch (qr_status) {
			     case qr_stat_failed:
				 status_text =
				     format_string(CATGETS(elm_msg_cat, MeSet,
							   MeMQFailedParse,
							   "Failed to parse quota root resource limits"));
				 break;
			     case qr_stat_none:
				 show_empty = 1;
				 break;
			     case qr_stat_empty:
				 status_text =
				     format_string(CATGETS(elm_msg_cat, MeSet,
							   MeMQNoAdminsLimits,
							   "No administrative resource limits in the quota root"));
				 break;
			     case qr_stat_normal:
				 break;
			     }

			     for (m = 0; m < qr_item_count; m++) {
				 struct mail_quota_item * qr_item =
				     quotaroot_item(qr,m);


				 if (qr_item) {

				     if (qd_exceed_or_printing(print_area,
							       &exceed,&L_current_line,&L_printing,
							       qd->top_line)) {					 
					 int current_col = 0;
					 int mayclear = 1;
					 int start_of_line = 1;

					 qd_draw_quotaroot_name(print_area,qr,
								&L_current_line,
								&current_col,&L_redrawing,
								resize,list,
								&mayclear,&start_of_line);
					 
					 qd_draw_resource_name(print_area,qr_item,
								&L_current_line,
								&current_col,&L_redrawing,
								resize,list,
								&mayclear,&start_of_line);

					 qd_draw_resource_values(print_area,qr_item,
								&L_current_line,
								&current_col,&L_redrawing,
								resize,list,
								&mayclear,&start_of_line);
					 
					 qd_draw_eoln(print_area,&L_current_line,&current_col,
						      &mayclear,&start_of_line,&L_printing,0);
					 
				     }
					 
				     free_mail_quota_item(&qr_item);
				 }

			     }

			     if (show_empty && qr_name) {   /* ? */

				 if (qd_exceed_or_printing(print_area,
							   &exceed,&L_current_line,&L_printing,
							   qd->top_line)) {
				     
				     int current_col = 0;
				     int mayclear = 1;
				     int start_of_line = 1;

				     qd_draw_quotaroot_name(print_area,qr,
							    &L_current_line,
							    &current_col,&L_redrawing,
							    resize,list,
							    &mayclear,&start_of_line);
					 
				     qd_draw_eoln(print_area,&L_current_line,&current_col,
						  &mayclear,&start_of_line,&L_printing,0);
				     
				 }

			     }
			    			     
			     if (status_text) {
				 
				 if (qd_exceed_or_printing(print_area,
							   &exceed,&L_current_line,&L_printing,
							   qd->top_line)) {
				     
				     int current_col = 0;
				     int mayclear = 1;
				     int start_of_line = 1;
				     int joined_line = 0;

				     int  col      = mp_lookup_integer(list,qd_disp_resource_col);

				     qd_draw_quotaroot_name(print_area,qr,
							    &L_current_line,
							    &current_col,&L_redrawing,
							    resize,list,
							    &mayclear,&start_of_line);
				     
				     if (current_col <= col) {

					 qd_reach_col(print_area,&current_col,col,&start_of_line);
					 
				     }

				     
				     if (!qd_draw_text(print_area,&L_current_line,&current_col,
						       &mayclear,&joined_line,&start_of_line,
						       status_text))
					 goto fail6;
				     
				     
				     qd_draw_eoln(print_area,&L_current_line,&current_col,
						  &mayclear,&start_of_line,&L_printing,0);
				     
				 }

			     fail6:
				 free_string(&status_text);
			     }
				 
			     if (qr_name)
				 free_string(&qr_name);			     
			     free_mail_quotaroot(&qr);
			 } else if (cd && is_canceled(cd)) {
			     DPRINT(Debug,12,(&Debug,
					      "qd_draw_mail_quota: Update canceled\n"));
			     break;
			 }
		     }

		 } else {

		     if (qd_exceed_or_printing(print_area,
					       &exceed,&L_current_line,&L_printing,qd->top_line)) {

			 struct string * s5 = format_string(CATGETS(elm_msg_cat, MeSet,
								    MeQuotaNotAvailable,
								    "Mail quota not available"));
			 int current_col = 0;
		     
			 int mayclear = 1;
			 int joined_line = 0;
			 int start_of_line = 1;

			 if (!qd_draw_text(print_area,&L_current_line,&current_col,
					   &mayclear,&joined_line,&start_of_line,
					   s5))
			      goto fail5;
			 
			 qd_draw_eoln(print_area,&L_current_line,&current_col,
				      &mayclear,&start_of_line,&L_printing,1);

		     fail5:
			 free_string(&s5);
			 
		     } 
		 }

		 free_string(&qr_NA);
		 L_add_line = 1;
	     }
	     
	     
	     if (cd && is_canceled(cd)) {
		 DPRINT(Debug,12,(&Debug,
				  "qd_draw_mail_quota: Update canceled\n"));
		 break;
	     }	     
	 }
     }


     /* ----- */

     DPRINT(Debug,12,(&Debug,"qd_draw_mail_quota:"));

     if (* add_p          != L_add) {
	 DPRINT(Debug,12,(&Debug,"  * add_p=%d => %d",
			  * add_p,L_add));
	 
	 * add_p          = L_add;
     }

     if (* current_line_p != L_current_line) {
	 DPRINT(Debug,12,(&Debug,"  * current_line_p=%p => %d",
			  * current_line_p,L_current_line));
	 
	 * current_line_p = L_current_line   ;
     }

     if (* printing_p     !=  L_printing) {
	 DPRINT(Debug,12,(&Debug,"  * printing_p=%d => %d",
			  * printing_p,L_printing));
	 
	 * printing_p     =  L_printing       ;
     }

     if (* redrawing_p    !=  L_redrawing) {
	 DPRINT(Debug,12,(&Debug,"  * redrawing_p=%d => %d",
			  * redrawing_p,L_redrawing));
	 
	 * redrawing_p    =  L_redrawing      ;
     }

     if (* before_top_p   !=  L_before_top) {
	 DPRINT(Debug,12,(&Debug,"  * before_top_p=%d => %d",
			  * before_top_p,L_before_top));
	 
	 * before_top_p   =  L_before_top     ;
     }

     if ( * add_line_p    != L_add_line  )  {
	 DPRINT(Debug,12,(&Debug,"  * add_line_p=%d => %d",
			  * add_line_p,L_add_line));
	 
	 * add_line_p    = L_add_line; 
     }
     
     DPRINT(Debug,12,(&Debug,"\n"));
}

static void qd_set_top_quotarootl P_((struct quota_display * qd));
static void qd_set_top_quotarootl(qd)
     struct quota_display * qd;
{
    
    if (QUOTA_DISPLAY_magic != qd->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
	      "qd_set_top_quotarootl",
              "Bad magic number (quota_diplay)",0);
       
    qd->top_quotarootl = 0;
    
    if (qd->top_mail_quota < qd->mail_quota_count) {
	
	size_t i;
	
	for (i = 0;
	     i < qd->mail_quota_list[qd->top_mail_quota].mail_quotarootl_count;
	     i++) {

	    if (0 == qd->mail_quota_list[qd->top_mail_quota].
		mail_quotarootl_list[i].
		virtual_ql_start_line)
		break;
	    
	    if (qd->top_offset >=
		qd->mail_quota_list[qd->top_mail_quota].
		mail_quotarootl_list[i].
		virtual_ql_start_line) {
		
		qd->top_quotarootl = i;
		
	    } else
		break;	    
	}
    }
}

static void qd_change_page P_((struct quota_display * qd,
			       enum qd_step_mode      qd_step,
			       int                    lines));
static void qd_change_page(qd,qd_step,lines)
     struct quota_display * qd;
     enum qd_step_mode      qd_step;
     int lines;
{
    
    if (QUOTA_DISPLAY_magic != qd->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
	      "qd_change_page",
              "Bad magic number (quota_diplay)",0);
    
    switch (qd_step) {
    case qd_step_prev:
	if (qd->top_offset > 0) {
	    
	    DPRINT(Debug,10,(&Debug,
			     "qd_change_page: top_offset=%d",qd->top_offset));
	    
	    qd->top_offset -= lines;
	    if (qd->top_offset < 0)
		qd->top_offset = 0;
	    
	    DPRINT(Debug,10,(&Debug," => %d   lines=%d\n",qd->top_offset,lines));
	}
	
	if (qd->top_mail_quota > 0) {
	    
	    if (/* 1)   Out of range                       */
		qd->top_mail_quota >= qd->mail_quota_count ||
		/* 2) virtual_mq_start_line not set        */
		0 == qd->mail_quota_list[qd->top_mail_quota].virtual_mq_start_line ||
		/* 3) Top offset is before start of this */
		qd->top_offset < qd->mail_quota_list[qd->top_mail_quota].virtual_mq_start_line) {
		
		DPRINT(Debug,10,(&Debug,
				 "qd_change_page: top_mail_quota=%zu",qd->top_mail_quota));
		
		while (qd->top_mail_quota > 0 &&
		       (qd->top_mail_quota >= qd->mail_quota_count ||
			0 == qd->mail_quota_list[qd->top_mail_quota].virtual_mq_start_line ||
			qd->top_offset < qd->mail_quota_list[qd->top_mail_quota].virtual_mq_start_line)) {
		    qd->top_mail_quota--;
		}
		
		DPRINT(Debug,10,(&Debug," => %zu",qd->top_mail_quota));
		
		qd_set_top_quotarootl(qd);
		
		DPRINT(Debug,10,(&Debug,", top_quotarootl=%zu\n",qd->top_quotarootl));
		
	    } else
		goto check_current_1;
	    
	} else {
	check_current_1:
	    
	    if (qd->top_mail_quota < qd->mail_quota_count) {
		
		if (qd->top_quotarootl > 0) {
		    
		    if (/* 1)   Out of range                       */
			qd->top_quotarootl >= qd->mail_quota_list[qd->top_mail_quota].mail_quotarootl_count ||
			 /* virtual_ql_start_line not set           */
			0 == qd->mail_quota_list[qd->top_mail_quota].
			mail_quotarootl_list[qd->top_quotarootl].virtual_ql_start_line ||
			/* 3) Top offset is before start of this */
			qd->top_offset <  qd->mail_quota_list[qd->top_mail_quota].
			mail_quotarootl_list[qd->top_quotarootl].virtual_ql_start_line) {
			
			DPRINT(Debug,10,(&Debug,
					 "qd_change_page: top_mail_quota=%zu,  top_quotarootl=%zu",
					 qd->top_mail_quota,qd->top_quotarootl));
			
			while (qd->top_quotarootl > 0 &&
			       (qd->top_quotarootl >= qd->mail_quota_list[qd->top_mail_quota].mail_quotarootl_count ||
				0 == qd->mail_quota_list[qd->top_mail_quota].
				mail_quotarootl_list[qd->top_quotarootl].virtual_ql_start_line ||
				qd->top_offset <=  qd->mail_quota_list[qd->top_mail_quota].
				mail_quotarootl_list[qd->top_quotarootl].virtual_ql_start_line)) {
			    qd->top_quotarootl--;
			}
			
			DPRINT(Debug,10,(&Debug," => %zu\n",qd->top_quotarootl));			 
			
		    }
		}
		
	    } else if (qd->top_quotarootl > 0) {
		DPRINT(Debug,10,(&Debug,
				 "qd_change_page: top_mail_quota=%zu,  top_quotarootl=%zu",
				 qd->top_mail_quota,qd->top_quotarootl));
		qd->top_quotarootl = 0;
		DPRINT(Debug,10,(&Debug," => %zu\n",qd->top_quotarootl));
	    }	     
	}
	
	
	break;
    case qd_step_this: break;
	
    case qd_step_next:
	if (!qd->is_end) {
	    
	    DPRINT(Debug,10,(&Debug,
			     "qd_change_page: top_offset=%d",qd->top_offset));	     
	    qd->top_offset += lines;
	    DPRINT(Debug,10,(&Debug," => %d   LINES=%d\n",qd->top_offset,lines));
	}
	
	if (qd->top_mail_quota < qd->mail_quota_count-1) {
	    
	    size_t mq;
	    size_t new_top_mq = qd->top_mail_quota;
	    
	    for (mq = qd->top_mail_quota + 1; mq < qd->mail_quota_count; mq++) {
		if (0 == qd->mail_quota_list[mq].virtual_mq_start_line)
		    break;
		if (qd->mail_quota_list[mq].virtual_mq_start_line <= qd->top_offset)
		    new_top_mq = mq;
		else
		    break;
	    }
	    
	    if (new_top_mq > qd->top_mail_quota) {
		
		DPRINT(Debug,10,(&Debug,
				 "qd_change_page: top_mail_quota=%zu",qd->top_mail_quota));
		qd->top_mail_quota = new_top_mq;
		
		DPRINT(Debug,10,(&Debug," => %zu",qd->top_mail_quota));
		
		qd_set_top_quotarootl(qd);
		
		DPRINT(Debug,10,(&Debug,", top_quotarootl=%zu\n",qd->top_quotarootl));
	    } else		 
		goto check_current_2;
	} else {
	    
	    
	check_current_2:
	    
	    if (qd->top_mail_quota < qd->mail_quota_count) { 
		if (qd->top_quotarootl <  qd->mail_quota_list[qd->top_mail_quota].mail_quotarootl_count-1) {
		    size_t qrl;
		    size_t new_top_qrl = qd->top_quotarootl;
		    
		    for (qrl = qd->top_quotarootl +1; qrl < qd->mail_quota_list[qd->top_mail_quota].mail_quotarootl_count; qrl++) {
			if (0 == qd->mail_quota_list[qd->top_mail_quota].mail_quotarootl_list[qrl].virtual_ql_start_line)
			    break;
			if (qd->mail_quota_list[qd->top_mail_quota].mail_quotarootl_list[qrl].virtual_ql_start_line <=
			    qd->top_offset)
			    new_top_qrl = qrl;
			else
			    break;
		    }
		    
		    if (new_top_qrl > qd->top_quotarootl) {
			DPRINT(Debug,10,(&Debug,
					 "qd_chnage_page: top_mail_quota=%zu, top_quotarootl=%zu",
					 qd->top_mail_quota,qd->top_quotarootl));
			qd->top_quotarootl = new_top_qrl;
			DPRINT(Debug,10,(&Debug," => %zu\n",qd->top_quotarootl));			
		    }
		}
	    } else if (qd->top_quotarootl > 0) {
		DPRINT(Debug,10,(&Debug,
				 "qd_change_page: top_mail_quota=%zu, top_quotarootl=%zu",
				 qd->top_mail_quota,qd->top_quotarootl));
		qd->top_quotarootl = 0;
		DPRINT(Debug,10,(&Debug," => %zu\n",qd->top_quotarootl));
	    }	     	     
	}
	
	break;
	
    }    
}


static void qd_draw_forward P_((struct menu_context  * print_area,
				struct quota_display * qd,
				enum qd_step_mode      qd_step,
				int                    add,
				int                    current_line,
				struct cancel_data   * cd,
				struct menu_param    * list));
static void qd_draw_forward(print_area,qd, qd_step, add, current_line,cd,list)
     struct menu_context  * print_area;
     struct quota_display * qd;
     enum qd_step_mode      qd_step;
     int                    add;
     int                    current_line;
     struct cancel_data   * cd;
     struct menu_param    * list;
{
    int add_to_top = 0;
    int printing   = 0;
    
    int LINES, COLUMNS;

    int redrawing = 0;
    int redraw_current_line = current_line;

    struct resize_columns resize_cols = NULL_resize_columns;
    
    menu_get_sizes(print_area,&LINES, &COLUMNS);

    if (QUOTA_DISPLAY_magic != qd->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
	      "qd_draw_forward",
              "Bad magic number (quota_diplay)",0);

    DPRINT(Debug,10,(&Debug,
		     "qd_draw_forward: add = %d\n",
		     add));
    
    if (current_line <= qd->top_line &&
	add > 0) {
	add_to_top = 1;
    }
    
    /* Increment virtual_mq_start_line and virtual_ql_start_line
       for display which is before top
    */
    
    if (qd->mail_quota_list) {
	size_t i;
	
	for (i = 0; i < qd-> mail_quota_count && i < qd->top_mail_quota; i++) {	    

	    qd_top_reset_quota(& (qd-> mail_quota_list[i]),
			       qd->top_offset,add_to_top,add);
	}

	if (qd->top_mail_quota < qd-> mail_quota_count) {

	    const size_t i = qd->top_mail_quota;

	    if (qd-> mail_quota_list[i].mail_quotarootl_list) {
		size_t j;
		
		for (j = 0; j < qd-> mail_quota_list[i].mail_quotarootl_count &&
			 j < qd->top_quotarootl; j++) {

		    qd_top_reset_quotarootl(& (qd-> mail_quota_list[i].mail_quotarootl_list[j]),
					    qd->top_offset,add_to_top,add);
		}		
	    }	       
	}		
    }

    if (add_to_top) {
	qd->top_offset += add;
	printing = 1;
    }

    if (current_line < 0) {
	DPRINT(Debug,10,(&Debug,
			 "qd_draw_forward: current_line=%d  RESETTING\n",
			 current_line));

	current_line = 0;
    }

    DPRINT(Debug,10,(&Debug,
		     "qd_draw_forward: qd_step = %d",qd_step));
    switch (qd_step) {
    case qd_step_prev: DPRINT(Debug,10,(&Debug," qd_step_prev"));    break;
    case qd_step_this: DPRINT(Debug,10,(&Debug," qd_step_this"));    break;
    case qd_step_next: DPRINT(Debug,10,(&Debug," qd_step_this"));    break;
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    qd_change_page(qd,qd_step,LINES);

    qd->is_end = 0;
     
    do {
	int before_top = 0; 
	int add_line = 0;
	
	if (redrawing) {
	    redrawing = 0;
	    printing  = 0;
	    
	    current_line = redraw_current_line;
	    add = 0;

	    if (current_line <= qd->top_line && current_line < LINES)
		printing = 1;
	    
	    DPRINT(Debug,10,(&Debug,
			     "qd_draw_forward: current_line=%d  printing=%d REDRAWING\n",
			     current_line,printing));

	    qd_setup_columns(print_area,list,qd_setup_change,&resize_cols);

	} else {
	    redraw_current_line = current_line;

	}

	if (qd->mail_quota_list) {
	    if (qd->top_mail_quota < qd-> mail_quota_count) {
		size_t i;
		
		qd_draw_mail_quota(print_area,qd,qd->top_mail_quota,qd->top_quotarootl,
				   &add,&current_line,&printing,&redrawing,&before_top,
				   add_to_top,cd,&resize_cols,list,&add_line);

		if (cd && is_canceled(cd)) {
		    DPRINT(Debug,12,(&Debug,
				     "qd_draw_forward: Update canceled\n"));
		    break;
		}
		
		for (i = qd->top_mail_quota+1; i < qd-> mail_quota_count; i++) {
		    qd_draw_mail_quota(print_area,qd,i,0,
				       &add,&current_line,&printing,&redrawing,&before_top,
				       add_to_top,cd,&resize_cols,list,&add_line);

		    if (cd && is_canceled(cd)) {
			DPRINT(Debug,12,(&Debug,
					 "qd_draw_forward: Update canceled\n"));
			break;
		    }
		}		
	    }	
	}

	if (cd && is_canceled(cd)) {
	    DPRINT(Debug,12,(&Debug,
			     "qd_draw_forward: Update canceled\n"));
	    break;
	}
	
    } while (redrawing);

    if (current_line < LINES && printing && !qd->is_end) {
	
	DPRINT(Debug,12,(&Debug,"qd_draw_forward: End of a mail quota screen\n"));

	qd->is_end = 1;
	
    }
    
    while (current_line < LINES) {
	if (current_line >= qd->top_line) {
	    if (!printing) {
		DPRINT(Debug,12,(&Debug,"qd_draw_forward: Gone to printing area on end of page, top_line=%d\n",
				 qd->top_line));
		printing = 1;
	    }
	    
	    menu_ClearLine(print_area,current_line);
	}
	
	current_line++;
    }

}


static int qd_step_len P_((int width, int step));
static int qd_step_len(width,step)
     int width;
     int step;
{
    int a = width+1;

    if (a < step)
	a = step;

    DPRINT(Debug,12,(&Debug,
		     "qd_step_len=%d: width=%d step=%d\n",
		     a,width,step));

    return a;
}

static void qd_setup_columns(ptr,list,initial,resize)
     struct menu_context  *ptr;
     struct menu_param *list;
     enum qd_setup_mode initial;
     struct resize_columns * resize;
{
    int LINES, COLUMNS;

    int step;

    menu_get_sizes(ptr,&LINES, &COLUMNS);

    step = COLUMNS/4;
    
    if (step < 5) {
	step = 5;

	DPRINT(Debug,12,(&Debug,
			 "qd_setup_columns: COLUMNS=%d step=%d\n",
			 COLUMNS,step));
    } else if (step > max_column_step) {
	step = max_column_step;

	DPRINT(Debug,12,(&Debug,
			 "qd_setup_columns: COLUMNS=%d step=%d\n",
			 COLUMNS,step));
    }
    
    switch (initial) {
	int x;
	
    case qd_setup_initial:

	/* Quotaroot  column */

	if (resize && resize->quotaroot_width > 0) {

	    mp_list_set_integer(list,qd_disp_quotaroot_width,resize->quotaroot_width);

	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_quotaroot_width: %d\n",
			     resize->quotaroot_width));
	    
	    x = qd_step_len(resize->quotaroot_width,step);

	    
	} else {
	
	    mp_list_set_integer(list,qd_disp_quotaroot_width,default_quotaroot_width);
	
	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_quotaroot_width: %d\n",
			     default_quotaroot_width));

	    x = qd_step_len(default_quotaroot_width,step);

	}
	
	/* Resource column */

	mp_list_set_integer(list,qd_disp_resource_col,x);
	
	if (resize && resize->resource_width > 0) {

	    mp_list_set_integer(list,qd_disp_resource_width,resize->resource_width);

	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_resource_width: %d\n",
			     resize->resource_width));
	    
	    x += qd_step_len(resize->resource_width,step);
	    
	} else {
	
	    mp_list_set_integer(list,qd_disp_resource_width,default_resource_width);

	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_resource_width: %d\n",
			     default_resource_width));
	    
	    x += qd_step_len(default_resource_width,step);

	}

	/* usage/limit columns */

	mp_list_set_integer(list,qd_disp_usage_col,x);

	if (resize) {
	    int usage_width = default_usage_width;
	    int limit_width = default_limit_width;

	    if (resize->usage_width > 0)
		usage_width = resize->usage_width;
	    if (resize->limit_width > 0)
		limit_width = resize->limit_width;
	    
	    mp_list_set_integer(list,qd_disp_usage_width,usage_width);

	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_usage_width: %d\n",
			     usage_width));
	    
	    mp_list_set_integer(list,qd_disp_limit_width,limit_width);

	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_limit_width: %d\n",
			     limit_width));
	    
	    x += qd_step_len(usage_width+limit_width+1,step);
			    
	} else {
	    mp_list_set_integer(list,qd_disp_usage_width,default_usage_width);

	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_usage_width: %d\n",
			     default_usage_width));
	    
	    mp_list_set_integer(list,qd_disp_limit_width,default_limit_width);
	    
	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_limit_width: %d\n",
			     default_limit_width));
	    
	    x += qd_step_len(default_usage_width+default_limit_width+1,step);
        }
	    
	/* percent column */

	mp_list_set_integer(list,qd_disp_percent_col,x);
	
	if (resize && resize->percent_width) {
	    mp_list_set_integer(list,qd_disp_percent_width,resize->percent_width);

	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_percent_width: %d\n",
			     resize->percent_width));
	    
	    x += resize->percent_width;
	} else {
	    mp_list_set_integer(list,qd_disp_percent_width,default_percent_width);

	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: qd_disp_percent_width: %d\n",
			     default_percent_width));
			     
	    x += default_percent_width;
	}


	if (x >= COLUMNS) {
	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: last column = %d >= COLUMNS = %d\n",
			     x,COLUMNS));
	}
	
	break;

    case qd_setup_change: {
	
	int quotaroot_width = mp_lookup_integer(list,qd_disp_quotaroot_width);
	int resource_width  = mp_lookup_integer(list,qd_disp_resource_width);
	int usage_width     = mp_lookup_integer(list,qd_disp_usage_width);
	int limit_width     = mp_lookup_integer(list,qd_disp_limit_width);
	int percent_width   = mp_lookup_integer(list,qd_disp_percent_width);
	
	int resource_col    = mp_lookup_integer(list,qd_disp_resource_col);
	int usage_col       = mp_lookup_integer(list,qd_disp_usage_col);
	int percent_col     = mp_lookup_integer(list,qd_disp_percent_col);

	int sum = quotaroot_width + resource_width + usage_width + limit_width + percent_width +4;
	
	int overflow = 0;
	int reduce = 0;

	if (sum >= COLUMNS) {
	    reduce = sum-COLUMNS;
	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: COLUMNS=%d total width = %d overflow, reduce=%d\n",
			     COLUMNS,sum,reduce));
	    overflow = 1;
	}
	
	if (percent_col + percent_width >= COLUMNS) {
	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: COLUMNS=%d percent_col=%d percent_width=%d overflow\n",
			     COLUMNS,percent_col,percent_width));
	    overflow = 1;			     
	}

	if (usage_col+usage_width+1+limit_width+1 >= percent_col) {
	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: percent_col=%d usage_col=%d usage_width=%d limit_width=%d overflow\n",
			     percent_col,usage_col,usage_width,limit_width));
	    overflow = 1;
	}
	
	if (resource_col + resource_width +1 >= usage_col) {
	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: usage_col=%d resource_col=%d resource_width=%d overflow\n",
			     usage_col,resource_col,resource_width));
	    overflow = 1;
	}

	if (quotaroot_width+1 >= resource_col) {
	    DPRINT(Debug,12,(&Debug,
			     "qd_setup_columns: resource_col=%d quotaroot_width=%d overflow\n",
			     quotaroot_width,resource_col));

	    overflow = 1;
	}

	if (resize && resize->quotaroot_width > 0) {
	    DPRINT(Debug,12,(&Debug,"qd_setup_columns: wanted quotaroot_width=%d => %d\n",
			     quotaroot_width, resize->quotaroot_width));

	    overflow = 1;  
	}

	if (resize && resize->resource_width > 0) {
	    DPRINT(Debug,12,(&Debug,"qd_setup_columns: wanted resource_width=%d => %d\n",
			     resource_width,resize->resource_width));

	    overflow = 1;
	}

	if (resize && resize->usage_width > 0) {
	    DPRINT(Debug,12,(&Debug,"qd_setup_columns: wanted usage_width=%d => %d\n",
			     usage_width,resize->usage_width));

	    overflow = 1;
	}

	if (resize && resize->limit_width > 0) {
	    DPRINT(Debug,12,(&Debug,"qd_setup_columns: wanted limit_width=%d => %d\n",
			     limit_width,resize->limit_width));
	    overflow = 1;
	}

	if (resize && resize->percent_width > 0) {
	    DPRINT(Debug,12,(&Debug,"qd_setup_columns: wanted percent_width=%d => %d\n",
			     percent_width,resize->percent_width));
	    overflow = 1;
	}

	if (overflow) {

	    int can_reduce = 0;

	    if (reduce > 0) {
	    
		DPRINT(Debug,12,(&Debug,"qd_setup_columns: can reduce"));

		if (quotaroot_width > default_quotaroot_width) {

		    DPRINT(Debug,12,(&Debug," quotaroot_width"));
		    
		    can_reduce++;
		}

		if (resource_width > default_resource_width) {
		    DPRINT(Debug,12,(&Debug," resource_width"));
		    
		    can_reduce++;
		}

		if (usage_width > default_usage_width) {
		    DPRINT(Debug,12,(&Debug," usage_width"));

		    can_reduce++;
		}

		if (limit_width > default_limit_width) {
		    DPRINT(Debug,12,(&Debug," limit_width"));

		    can_reduce++;
		}

		if (percent_width > default_percent_width) {
		    DPRINT(Debug,12,(&Debug," percent_width"));

		    can_reduce++;
		}

		DPRINT(Debug,12,(&Debug,", count %d\n",
				 can_reduce));
	    }
	    
	    
	    /* Quotaroot  column */

	    if (reduce > 0 && quotaroot_width > default_quotaroot_width) {
		int a = reduce;
		
		if (can_reduce > 0)
		    a = reduce / can_reduce;
		
		if (quotaroot_width <  default_quotaroot_width+a)
		    a = quotaroot_width - default_quotaroot_width;
		
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: quotaroot_width=%d",quotaroot_width));
		
		quotaroot_width -= a;
		
		DPRINT(Debug,12,(&Debug," => %d\n",quotaroot_width));
		mp_list_set_integer(list,qd_disp_quotaroot_width,quotaroot_width);

		reduce -= a;
		
		if (can_reduce > 0)
		    can_reduce --;
	    } else if (resize && resize->quotaroot_width > 0) {
		int a = 0;

		if (resize->quotaroot_width <  quotaroot_width)
		    a =  quotaroot_width - resize->quotaroot_width;
		if (a > reduce)
		    a = reduce;

		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: quotaroot_width=%d",quotaroot_width));
		
		quotaroot_width = resize->quotaroot_width;

		DPRINT(Debug,12,(&Debug," => %d\n",quotaroot_width));
		mp_list_set_integer(list,qd_disp_quotaroot_width,quotaroot_width);

		reduce -= a;
		resize->quotaroot_width = 0;
	    }

	    x = qd_step_len(quotaroot_width,step);

	    /* Resource column */
	    
	    if (resource_col > x) {
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: resource_col=%d",resource_col));
		resource_col = x;

		DPRINT(Debug,12,(&Debug," => %d\n",resource_col));
		mp_list_set_integer(list,qd_disp_resource_col,resource_col);
		
	    } else {
		if (resource_col < quotaroot_width+1) {
		    DPRINT(Debug,12,(&Debug,
				     "qd_setup_columns: resource_col=%d",resource_col));
		
		    resource_col = quotaroot_width+1;

		    DPRINT(Debug,12,(&Debug," => %d\n",resource_col));
		    mp_list_set_integer(list,qd_disp_resource_col,resource_col);
		}
		
		x = resource_col;
	    }

	    if (reduce > 0 && resource_width > default_resource_width) {
		int a = reduce;

		if (can_reduce > 0)
		    a = reduce / can_reduce;
		
		if (resource_width < default_resource_width+a)
		    a = resource_width - default_resource_width;
		
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: resource_width=%d",resource_width));
		
		resource_width -= a;

		DPRINT(Debug,12,(&Debug," => %d\n",resource_width));
		mp_list_set_integer(list,qd_disp_resource_width,resource_width);

		reduce -= a;
		if (can_reduce > 0)
		    can_reduce --;
	    } else if (resize && resize->resource_width > 0) {
		int a = 0;

		if (resize->resource_width < resource_width)
		    a = resource_width - resize->resource_width;
		if (a > reduce)
		    a = reduce;

		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: resource_width=%d",resource_width));

		resource_width =  resize->resource_width;
		
		DPRINT(Debug,12,(&Debug," => %d\n",resource_width));
		mp_list_set_integer(list,qd_disp_resource_width,resource_width);

		reduce -= a;
		resize->resource_width = 0;
	    }

	    x += qd_step_len(resource_width,step);
	    
	    /* usage/limit columns */

	    if (usage_col > x) {
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: usage_col=%d",usage_col));
		usage_col = x;
		
		DPRINT(Debug,12,(&Debug," => %d\n",usage_col));
		mp_list_set_integer(list,qd_disp_usage_col,usage_col);
	    } else {
		if (usage_col < resource_col+resource_width+1) {
		    DPRINT(Debug,12,(&Debug,
				     "qd_setup_columns: usage_col=%d",usage_col));

		    usage_col = resource_col+resource_width+1;
		    
		    DPRINT(Debug,12,(&Debug," => %d\n",usage_col));
		    mp_list_set_integer(list,qd_disp_usage_col,usage_col);
		}
		
		x = usage_col;
	    }

	    if (reduce > 0 && usage_width > default_usage_width) {
		int a = reduce;

		if (can_reduce > 0)
		    a = reduce / can_reduce;
		
		if (usage_width < default_usage_width+a)
		    a = usage_width - default_usage_width;

		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: usage_width=%d",usage_width));

		usage_width -= a;

		DPRINT(Debug,12,(&Debug," => %d\n",usage_width));
		mp_list_set_integer(list,qd_disp_usage_width,usage_width);

		reduce -= a;
		if (can_reduce > 0)
		    can_reduce--;
	    } else if (resize && resize->usage_width > 0) {
		int a = 0;

		if (resize->usage_width < usage_width)
		    a = usage_width - resize->usage_width;
		if (a > reduce)
		    a = reduce;
		
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: usage_width=%d",usage_width));
		
		usage_width =  resize->usage_width;
		
		DPRINT(Debug,12,(&Debug," => %d\n",usage_width));
		mp_list_set_integer(list,qd_disp_usage_width,usage_width);

		reduce -= a;
		resize->usage_width = 0;
	    }

	    if (reduce > 0 && limit_width > default_limit_width) {
		int a = reduce;

		if (can_reduce > 0)
		    a = reduce / can_reduce;
		
		if (limit_width < default_limit_width+a)
		    a = limit_width - default_limit_width;

		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: limit_width=%d",limit_width));
		
		limit_width -= a;

		DPRINT(Debug,12,(&Debug," => %d\n",limit_width));
		mp_list_set_integer(list,qd_disp_limit_width,limit_width);

		reduce -= a;
		if (can_reduce > 0)
		    can_reduce--;

	    } else if (resize && resize->limit_width > 0) {
		int a= 0;

		if (resize->limit_width < limit_width)
		    a = limit_width - resize->limit_width;
		if (a > reduce)
		    a = reduce;
		
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: limit_width=%d",limit_width));

		limit_width = resize->limit_width;
		
		DPRINT(Debug,12,(&Debug," => %d\n",limit_width));
		mp_list_set_integer(list,qd_disp_limit_width,limit_width);
		
		reduce -= a;
		resize->limit_width = 0;
	    }

	    x += qd_step_len(usage_width+limit_width+1,step);

	    /* percent column */

	    if (percent_col > x) {
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: percent_col=%d",percent_col));
		percent_col = x;
		
		DPRINT(Debug,12,(&Debug," => %d\n",percent_col));
		mp_list_set_integer(list,qd_disp_percent_col,percent_col);
	    } else {
		if (percent_col < usage_col+usage_width+1+limit_width+1) {
		    DPRINT(Debug,12,(&Debug,
				     "qd_setup_columns: percent_col=%d",percent_col));

		    percent_col = usage_col+usage_width+1+limit_width+1;
		    mp_list_set_integer(list,qd_disp_percent_col,percent_col);
		}

		x = percent_col;
	    }

	    if (reduce > 0 && percent_width > default_percent_width) {

		int a = reduce;

		if (percent_width < default_percent_width+a)
		    a = percent_width - default_percent_width;

		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: percent_width=%d",percent_width));

		percent_width -= a;
		DPRINT(Debug,12,(&Debug," => %d\n",percent_width));
		mp_list_set_integer(list,qd_disp_percent_width,percent_width);

		reduce -= a;
		if (can_reduce > 0)
		    can_reduce--;

	    }  else if (resize && resize->percent_width > 0) {
		int a = 0;

		if (resize->percent_width < percent_width)
		    a = percent_width - resize->percent_width;
		if (a > reduce)
		    a = reduce;
		
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: percent_width=%d",percent_width));

		percent_width = resize->percent_width;
		DPRINT(Debug,12,(&Debug," => %d\n",percent_width));
		mp_list_set_integer(list,qd_disp_percent_width,percent_width);

		reduce -= a;
		resize->percent_width = 0;
	    }

	    
	    if (x >= COLUMNS) {
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: last column = %d >= COLUMNS = %d\n",
				 x,COLUMNS));
	    }

	    if (reduce > 0) {
		DPRINT(Debug,12,(&Debug,
				 "qd_setup_columns: reduce = %d not available\n",
				 reduce));
	    }	    
	}	    
    }
	break;
    }
}

static void qd_quota_step_page P_((struct quota_display * qd,
				   struct cancel_data  *  cd,
				   enum qd_step_mode      qd_step));
static void qd_quota_step_page(qd,cd,qd_step)
     struct quota_display * qd;
     struct cancel_data   * cd;
     enum qd_step_mode      qd_step;
{
    int current_line = 2;
    int add          = 0;

    if (QUOTA_DISPLAY_magic != qd->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "qd_quota_step_page",
              "Bad magic number (quota_diplay)",0);

    menu_MoveCursor(qd->quota_page,current_line,0);
    menu_CleartoEOS(qd->quota_page);
    
    qd_draw_forward(qd->quota_page,qd,qd_step,add,
		    current_line,cd,
		    qd->PARAM);

}

static void qd_quota_page_update P_((struct quota_display * qd,
				     struct menu_context  *ptr,
				     struct menu_param *list,
				     struct cancel_data  *  cd));

static void qd_quota_page_update(qd,ptr,list,cd)
     struct quota_display * qd;
     struct menu_context  * ptr;
     struct menu_param    * list;
     struct cancel_data   * cd;
{
    int current_line = 1;  /* start from second line */
    int add          = 0;

    if (QUOTA_DISPLAY_magic != qd->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "qd_quota_page_update",
              "Bad magic number (quota_diplay)",0);
    
    menu_ClearScreen(ptr); /* Clears also need update flag */
    
    menu_StartXX(ptr,pg_BOLD);
    menu_print_format_center(ptr,current_line,
                             CATGETS(elm_msg_cat, MeSet,
                                     MeQuotaScreenTitle, 
                                     "Mail Quota Screen [ELM %s]"),
			     version_buff
			     );
    menu_EndXX(ptr,pg_BOLD);
    
    current_line += 2;
    
    if (current_line > qd->top_line) {
	add = current_line - qd->top_line;
	
	qd->top_line = current_line;
    }
    
    
    qd_draw_forward(ptr,qd,qd_step_this,add,current_line,cd,list);

}
    

S_(subpage_simple_redraw qd_update_quota_page)
static int qd_update_quota_page P_((struct menu_context  *ptr,
				    struct menu_param *list));
static int qd_update_quota_page(ptr,list)
     struct menu_context  *ptr;
     struct menu_param *list;
{
    struct menu_anon_param * anon_disp_self  = 
	mp_lookup_anon(list, qd_disp_self);

    struct quota_display * qd = anon_disp_self->qd;
    struct cancel_data  *  cd = NULL;
    
    
    if (QUOTA_DISPLAY_magic != qd->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "qd_update_quota_page",
              "Bad magic number (quota_diplay)",0);

    DPRINT(Debug,25, (&Debug,
                      "qd_update_quota_page: ptr=%p\n",
                      ptr));

    setup_mbx_cancel_message(&cd,mbx_checking_mailbox);

    qd_quota_page_update(qd,ptr,list,cd);
    

    free_cancel(&cd);
		
    return 1;
}

static void qdisp_set_page P_((struct quota_display *qd));
static void qdisp_set_page(qd)
     struct quota_display *qd;
{
    int LINES, COLUMNS;

    if (QUOTA_DISPLAY_magic != qd->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "qdisp_set_page",
              "Bad magic number (quota_diplay)",0);

    menu_get_sizes(qd->current_page,&LINES, &COLUMNS);

    if (LINES > 5) {
	if (!qd->quota_page)
	    qd->quota_page = 
		new_menu_subpage(qd->current_page,0,LINES-4,
				 qd_update_quota_page,
				 qd->PARAM);
	else
	    menu_subpage_relocate(qd->quota_page,
				  qd->current_page,0,LINES-4);		
    } else if (qd->quota_page)
	erase_menu_context(& (qd->quota_page));
    
}

struct quota_display * new_quota_display(mquota,parent_page,cd)
     struct mail_quota    * mquota;
     struct menu_context  * parent_page;
     struct cancel_data   * cd;
{
    int have_screen = RawState();
    
    
    struct quota_display *ret =
	safe_zero_alloc(sizeof (*ret));

    ret->magic = QUOTA_DISPLAY_magic;
    
    ret->refcount = 1;

    ret->mail_quota_list  = NULL;
    ret->mail_quota_count = 0;

    ret->current_page     = new_menu_context();
    ret->quota_page       = NULL;

    ret->top_mail_quota   = 0;
    ret->top_quotarootl   = 0;
    
    ret->top_line         = 2;  /* Line 0 is title ?? */

    /* just build self reference */

    ret->self.qd   = ret;

    ret->PARAM[qd_disp_self].t   = mp_anon_param;
    
    ret->PARAM[qd_disp_quotaroot_width].t = mp_integer;
    ret->PARAM[qd_disp_resource_width].t  = mp_integer;
    ret->PARAM[qd_disp_usage_width].t     = mp_integer;
    ret->PARAM[qd_disp_limit_width].t     = mp_integer;
    ret->PARAM[qd_disp_percent_width].t   = mp_integer;

    
    ret->PARAM[qd_disp_resource_col].t    = mp_integer;
    ret->PARAM[qd_disp_usage_col].t       = mp_integer;
    ret->PARAM[qd_disp_percent_col].t     = mp_integer;
    
    ret->PARAM[qd_disp_COUNT].t  = mp_END;

    mp_list_set_anon(ret->PARAM,qd_disp_self,& (ret->self ));

    qd_setup_columns(ret->current_page,ret->PARAM,qd_setup_initial,NULL);
    
    
    if (have_screen) {
	
	qdisp_set_page(ret);

	if (ret->quota_page)
	    qd_quota_page_update(ret,ret->quota_page,
				 ret->PARAM,cd);
    }
    
    return ret;
}

struct menu_context *  quota_display_current_page(quotad)
     struct quota_display * quotad;
{
    
    if (QUOTA_DISPLAY_magic != quotad->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "quota_display_current_page",
              "Bad magic number (quota_diplay)",0);

    return quotad->current_page;
}

enum qd_create_flag { qd_search = 0, qd_create = 1 };

/* Return 1 on succeed */
static int quota_display_quota_index P_((struct quota_display * quotad,
					 struct mail_quota    * mquota,
					 size_t               * index_ref,
					 enum qd_create_flag    create));
static int quota_display_quota_index(quotad,mquota,index_ref,create)
     struct quota_display * quotad;
     struct mail_quota    * mquota;
     size_t               * index_ref;
     enum qd_create_flag    create;
{
    size_t i;

    if (QUOTA_DISPLAY_magic != quotad->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "quota_display_quota_index",
              "Bad magic number (quota_diplay)",0);

    if (index_ref)
	*index_ref = quotad-> mail_quota_count;
    
    for (i = 0; i < quotad-> mail_quota_count; i++) {
	if (quotad-> mail_quota_list[i].mquota == mquota) {
	    
	    if (index_ref)
		*index_ref = i;
	    
	    return 1;
	}
    }

    if (i != quotad-> mail_quota_count)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quota_display_quota_index",		  
	      "Bad index",0);
    
    switch (create) {
	size_t newcount;
	
    case qd_search:
	break;

    case qd_create:
	newcount = quotad-> mail_quota_count+1;
	quotad-> mail_quota_list
	    = safe_array_realloc(quotad-> mail_quota_list,
				 newcount,
				 sizeof(quotad-> mail_quota_list[0]));

	/* bzero is defined hdrs/elm_defs.h */
	bzero (& (quotad-> mail_quota_list[i]),
	       sizeof (quotad-> mail_quota_list[i]));
	
	       	
	quotad-> mail_quota_list[i].mquota = mquota;
	inc_mail_quota_refcount(quotad-> mail_quota_list[i].mquota);
	quotad-> mail_quota_count = newcount;

	/* Drawing algorithm */

	quotad-> mail_quota_list[i].virtual_mq_start_line = 0;
	
	if (index_ref)
	    *index_ref = i;
	
	return 1;	
    }
    
    return 0;
}

void quota_display_add_quota(quotad,mquota,cd)
     struct quota_display * quotad;
     struct mail_quota    * mquota;
     struct cancel_data   * cd;

{
    size_t idx;
    
    if (QUOTA_DISPLAY_magic != quotad->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "quota_display_add_quota",
              "Bad magic number (quota_diplay)",0);

    /* Should always succeed */
    if (quota_display_quota_index(quotad,mquota,&idx,qd_create)) {
	
	int have_screen = RawState();

	DPRINT(Debug,12,(&Debug,
			 "quota_display_add_quota: quota added idx=%d, have_screen=%d\n",
			 idx,have_screen));

	
	if (have_screen &&
	    quotad->current_page &&
	    quotad->quota_page) {

	    DPRINT(Debug,12,(&Debug,
			     "quota_display_add_quota: Redrawing\n"));

	    switch (menu_redraw_done(quotad->quota_page,
				     menu_disable_redraw_callback  /* Avoid calling qd_update_quota_page() */)) {
	    case menu_redraw_need:
		DPRINT(Debug,12,(&Debug,
				 "quota_display_add_quota: Need full\n"));
		qd_quota_page_update(quotad,
				     quotad->quota_page,
				     quotad->PARAM,
				     cd);
		break;
	    case menu_redraw_none:
		qd_quota_step_page(quotad,cd,qd_step_this);
		break;
	    case menu_done_redraw:
		DPRINT(Debug,12,(&Debug,
				 "quota_display_add_quota: Was already done\n"));
		break;
		
	    }
	}
    }
}

/* Return 1 on succeed , if create set also deletes previous folder_disp_name */
static int quota_display_qlist_index P_((struct quota_display       * quotad,
					 size_t                       mquota_idx,
					 struct mail_quotaroot_list * qlist,
					 struct string              * folder_disp_name,
					 size_t                     * index_ref,
					 enum qd_create_flag          create));
static int quota_display_qlist_index(quotad,mquota_idx,qlist,folder_disp_name,
				     index_ref,create)
     struct quota_display       * quotad;
     size_t                       mquota_idx;
     struct mail_quotaroot_list * qlist;
     struct string              * folder_disp_name;
     size_t                     * index_ref;
     enum qd_create_flag          create;
{
    size_t i;
    
    int index_set = 0;
    size_t  res_index = 0;

    int found_it = 0;
    
    if (QUOTA_DISPLAY_magic != quotad->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "quota_display_qlist_index",
              "Bad magic number (quota_diplay)",0);
    
    if ( mquota_idx >= quotad-> mail_quota_count ||
	 ! quotad-> mail_quota_list)
        panic("MBX PANIC",__FILE__,__LINE__,
              "quota_display_qlist_index",
              "Bad mquota_idx (quota_diplay)",0);
	
    if (index_ref)
	*index_ref = quotad-> mail_quota_list[mquota_idx].mail_quotarootl_count;
    
    for (i = 0; i < quotad-> mail_quota_list[mquota_idx].mail_quotarootl_count; i++) {
	size_t j;
	
	int found_qlist =
	    ( quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i].quota_list
	      == qlist );

	if (found_qlist) {
	    res_index = i;
	    index_set = 1;
	}
	
	for (j = 0;
	     j < quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i].
		 folder_disp_name_count; j++) {

	    if (quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i].
		folder_disp_name_list[j] &&

		0 == string_cmp(quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i].
				folder_disp_name_list[j],
				folder_disp_name,
				-999 /* not comparable */)) {

		if (found_qlist) {
		    if (index_ref)
			*index_ref = i;
		    
		    found_it = 1;
		} else {
		    switch (create) {
		    case qd_search:   /* Ignore bad qlist */
			break;
		    case qd_create:
			free_string(& (quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i].
				       folder_disp_name_list[j]));
			break;
		    }
		}

	    }
	}
    }

    if (i !=  quotad-> mail_quota_list[mquota_idx].mail_quotarootl_count)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quota_display_qlist_index",		  
	      "Bad index (1)",0);

    if (index_set) {
	
	if (res_index >= quotad-> mail_quota_list[mquota_idx].mail_quotarootl_count)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quota_display_qlist_index",		  
		  "Bad index (2)",0);

	if (quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].quota_list
	    != qlist)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quota_display_qlist_index",		  
		  "Bad index (3)",0);
    }

    if (found_it) {
	if (index_ref) {
	    if (*index_ref >= quotad-> mail_quota_list[mquota_idx].mail_quotarootl_count)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "quota_display_qlist_index",		  
		      "Bad index (4)",0);
	    
	    if (quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[*index_ref].quota_list
		!= qlist)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "quota_display_qlist_index",		  
		      "Bad index (5)",0);
	}
	return 1;
    }
    
    switch (create) {
	size_t j;
	size_t new_namecount;
	
    case qd_search:
	break;


    case qd_create:
	if (!index_set) {
	    size_t newcount =  quotad-> mail_quota_list[mquota_idx].mail_quotarootl_count +1;

	    quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list =
		safe_array_realloc(quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list,
				   newcount,
				   sizeof(quotad-> mail_quota_list[mquota_idx].
					  mail_quotarootl_list[0]));

	    /* bzero is defined hdrs/elm_defs.h */
	    
	    bzero(& (quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i]),
		  sizeof (quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i]));
	    
	    quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i].
		quota_list = qlist;
	    inc_mail_quotaroot_list_refcount(quotad-> mail_quota_list[mquota_idx].
					     mail_quotarootl_list[i].quota_list);



	    
	    quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i]. virtual_ql_start_line = 0;
	    quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i]. virtual_tb_start_line = 0;

	    
	    quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i].folder_disp_name_list =
		NULL;
	    quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[i].folder_disp_name_count =
		0;

	    
	    quotad-> mail_quota_list[mquota_idx].mail_quotarootl_count = newcount;
	    res_index = i;
	}

	for (j = 0;
	     j < quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
		 folder_disp_name_count; j++) {

	    if (quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
		folder_disp_name_list[j]) {

		if ( 0 == string_cmp(quotad-> mail_quota_list[mquota_idx].
				     mail_quotarootl_list[res_index].
				     folder_disp_name_list[j],
				     folder_disp_name,
				     -999 /* not comparable */)) {

		    panic("MBX PANIC",__FILE__,__LINE__,
			  "quota_display_qlist_index",
			  "folder_disp_name already on list",0);
		    
		}

	    } else {
		goto slot_found;
	    }
	}

	if (j != quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
	    folder_disp_name_count) {
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quota_display_qlist_index",		  
		  "Bad index (6)",0);
	}
	
	new_namecount = quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
	    folder_disp_name_count + 1;
	quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
	    folder_disp_name_list =
	    safe_array_realloc(quotad-> mail_quota_list[mquota_idx].
			       mail_quotarootl_list[res_index].folder_disp_name_list,
			       new_namecount,
			       sizeof(quotad-> mail_quota_list[mquota_idx].
				      mail_quotarootl_list[res_index].folder_disp_name_list[0]));
	quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
	    folder_disp_name_list[j] = NULL;
	quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
	    folder_disp_name_count = new_namecount;
	
    slot_found:
	if (quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
	    folder_disp_name_list[j]) {
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quota_display_qlist_index",		  
		  "Slot not empty",0);
	}

	quotad-> mail_quota_list[mquota_idx].mail_quotarootl_list[res_index].
	    folder_disp_name_list[j] =
	    dup_string(folder_disp_name);

	if (index_ref)
	    *index_ref = res_index;

	return 1;
    }

    return 0;
}

void quota_display_add_qlist(quotad,qlist,folder_disp_name,cd)
     struct quota_display       * quotad;
     struct mail_quotaroot_list * qlist;
     struct string              * folder_disp_name;
     struct cancel_data         * cd;
{
    struct mail_quota    * mquota;
    size_t                 mquota_idx;
    
    if (QUOTA_DISPLAY_magic != quotad->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "quota_display_add_qlist",
              "Bad magic number (quota_diplay)",0);

    /* Increments refcount! */
    mquota = quotaroot_list_mquota(qlist);

    if (!mquota) {
	DPRINT(Debug,12,(&Debug,
			 "quota_display_add_qlist: No mail quota on quota_list\n"));

	return;
    }
    
    /* Should always succeed */
    if (quota_display_quota_index(quotad,mquota,&mquota_idx,qd_create)) {

	size_t qlist_idx;

	/* Should always succeed */
	if (quota_display_qlist_index(quotad,mquota_idx,qlist,folder_disp_name,
				      &qlist_idx,qd_create)) {


	    int have_screen = RawState();

	    DPRINT(Debug,12,(&Debug,
			     "quota_display_add_qlist: quota list added, mquota_idx=%d qlist_idx=%d folder_disp_name=%S have_screen=%d\n",

			     mquota_idx,qlist_idx,folder_disp_name,have_screen));
	    
	    if (have_screen &&
		quotad->current_page &&
		quotad->quota_page
		) {
		
		DPRINT(Debug,12,(&Debug,
				 "quota_display_add_qlist: Redrawing\n"));
		
		switch (menu_redraw_done(quotad->quota_page,
					 menu_disable_redraw_callback  /* Avoid calling qd_update_quota_page() */)) {
		case menu_redraw_need:
		    DPRINT(Debug,12,(&Debug,
				     "quota_display_add_qlist: Need full\n"));
		    qd_quota_page_update(quotad,
					 quotad->quota_page,
					 quotad->PARAM,
					 cd);
		    break;
		case menu_redraw_none:
		    qd_quota_step_page(quotad,cd,qd_step_this);
		    break;
		case menu_done_redraw:
		    DPRINT(Debug,12,(&Debug,
				     "quota_display_add_qlist: Was already done\n"));
		    break;	       
		}
	    }
	}
    }
	
    free_mail_quota(&mquota);
}

/* Return '\0', EOF, or letter from cmds (may be NULL) */

int quota_display_show(quotad,cmds)
     struct quota_display       * quotad;
     struct elm_commands        * cmds; /* may be NULL */
{
    int have_screen = RawState();
    int retch = '\0';

    if (QUOTA_DISPLAY_magic != quotad->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "quota_display_show",
              "Bad magic number (quota_diplay)",0);
    
    if (have_screen &&
	quotad->current_page) {

	struct menu_context * prompt_area = NULL;
	
	struct elm_commands * disp_cmds = give_quotadisp_commands();
	struct cancel_data  * cd = NULL;
	struct elm_commands * all_cmds = NULL;
	
	int do_redraw = 0;

	int LINES, COLUMNS;

	setup_mbx_cancel_message(&cd,mbx_checking_mailbox);
	
	menu_get_sizes(quotad->current_page,&LINES, &COLUMNS);
	
	while (!retch && !is_canceled(cd)) {
	    int cmd = '\0';
	    int lin,col;
	    
	    menu_set_default(quotad->current_page);

#ifdef BACKGROUD_PROCESSES      
	    if (handle_sigchld)
		sigchld_handler();
#endif

	    if (0) {
	    redraw:
		do_redraw = 1;
	    }
	    
	resize_mark:
	    if (menu_resized(quotad->current_page)) {
		DPRINT(Debug,12,(&Debug,"quota_display_show: resized\n"));
		
		menu_get_sizes(quotad->current_page,&LINES, &COLUMNS);

		do_redraw = 1;

		if (LINES > 3) {
		    if (prompt_area)
			menu_subpage_relocate(prompt_area,quotad->current_page,LINES-4,4);
		    
		    else 
			new_menu_subpage(quotad->current_page,LINES-4,4,
					 subpage_simple_noredraw,NULL);
		    
		} else if (prompt_area)
		    erase_menu_context(&prompt_area);

		qdisp_set_page(quotad);
	    } else {
		switch (menu_redraw_done(quotad->current_page, menu_disable_redraw_callback)) {
		case menu_redraw_none:         break;
		case menu_done_redraw:
		    DPRINT(Debug,12,(&Debug,"quota_display_show: current page done redraw\n"));
		    break;
		case menu_redraw_need:
		    DPRINT(Debug,12,(&Debug,"quota_display_show: current page need redraw\n"));
		    do_redraw = 1;
		    break;
		}
	    }
		
	    if (LINES > 4 && !prompt_area) {
		DPRINT(Debug,12,(&Debug,"Creating prompt area\n"));
		
		prompt_area = 
		    new_menu_subpage(quotad->current_page,LINES-4,4,
				     subpage_simple_noredraw,NULL);

	    }
		
	    if (do_redraw) {
		DPRINT(Debug,12,(&Debug,"quota_display_show: Need redraw\n"));
		
		menu_ClearScreen(quotad->current_page);
	    }

	    if (quotad->quota_page) {
		switch(menu_redraw_done(quotad->quota_page,menu_disable_redraw_callback)) {
		case menu_redraw_none:         break;
		case menu_done_redraw:
		    DPRINT(Debug,12,(&Debug,"quota_display_show: quota page done redraw\n"));
		    break;
		case menu_redraw_need:
		    DPRINT(Debug,12,(&Debug,"quota_display_show: quota page need redraw\n"));
		    goto quota_page_redraw;
		}
	           	
		if (menu_resized(quotad->quota_page) ||
		    do_redraw) {
		    
		quota_page_redraw:
		    DPRINT(Debug,12,(&Debug,"quota_display_show: Redrawing quota page\n"));
		    
		    qd_quota_page_update(quotad,
					 quotad->quota_page,
					 quotad->PARAM,
					 cd);		    
		}
	    } else {
		DPRINT(Debug,12,(&Debug,"quota_display_show: No quota page\n"));
		break;
	    }
	    
	    do_redraw = 0;

	    if (is_canceled(cd)) {
		DPRINT(Debug,12,(&Debug,"quota_display_show: Prompt canceled\n"));
		break;
	    }

	    if (prompt_area) {

		if (menu_need_redraw(prompt_area) ||
		    menu_resized(prompt_area)) {
		    DPRINT(Debug,12,(&Debug,"quota_display_show: Redrawing or resizing prompt area\n"));

		    menu_ClearScreen(prompt_area); /* Clears also need update flag */
		}
		
		menu_StartXX(prompt_area,pg_STANDOUT);
		
		if (quotad->is_end) 
		    menu_PutLineX(prompt_area,0, 0,
				  CATGETS(elm_msg_cat, MeSet,
					  MeMQPromptEnd,
					  "Press 'q' or <return> to quit a quota screen ('?' for help):"));
		else			   
		    menu_PutLineX(prompt_area,0, 0,
				  CATGETS(elm_msg_cat, MeSet,
					  MeMQPrompt,
					  "Press 'q' to quit a quota screen, or <return> to continue ('?' for help):"));
	    
		menu_EndXX(prompt_area,pg_STANDOUT);

		menu_GetXYLocation(prompt_area,&lin,&col);
		menu_CleartoEOLN(prompt_area);

		show_last_error();
			    
		FlushBuffer();
			    
		menu_MoveCursor(prompt_area,lin, col);

		/* Accept cursor keys and specially pageup / pagedown keys */
		
		cmd = menu_ReadCh(prompt_area,REDRAW_MARK|READCH_resize|READCH_CURSOR);

		if (isascii(cmd) && 
		    isprint(cmd)) {
		    DPRINT(Debug,8,(&Debug,"quota_display_show: ... command: '%c'\n",
				    cmd));
		} else {
		    DPRINT(Debug,8,(&Debug,"quota_display_show: ... command: %d\n",
				    cmd));
		}
		
		if (cmd == REDRAW_MARK) {
		    
		    DPRINT(Debug,4, (&Debug,"quota_display_show:       .... redrawing\n"));
		    
		    do_redraw = 1;
		    continue;
		}
		if (cmd == RESIZE_MARK) {
		    
		    DPRINT(Debug,10, (&Debug,"quota_display_show:       .... resizing\n"));
		    goto resize_mark;
		}
			    
		if (cmd == EOF) {
		    DPRINT(Debug,10, (&Debug,"quota_display_show:       .... EOF\n"));
		    retch = EOF;
		    break;
		}

		clear_error();
	     
		/* Put cursor pack to correct place */
		menu_MoveCursor(prompt_area,lin,col);

		switch (cmd) {
		    
		case ctrl('L'):
		    goto redraw;
		    
		case '\n':
		case '\r':
		    if (quotad->is_end) {
			DPRINT(Debug,10,(&Debug,"quota_display_show: End of quota listing reached.\n"));
			goto out;
		    }
		    /* FALLTHRU */
		case ' ':
		case '+':
		case RIGHT_MARK:
		case PAGEDOWN_MARK:
		    
		    qd_quota_step_page(quotad,cd,qd_step_next);
		    break;

		case '-':
		case LEFT_MARK:
		case PAGEUP_MARK:
		    
		    qd_quota_step_page(quotad,cd,qd_step_prev);
		
		break;

		    
		case '?':
		case  HELP_MARK:
		    
		    menu_Writechar(prompt_area,cmd);
		    
		    FlushBuffer();
		    
		    if (cmds) {
			if (!all_cmds)
			    all_cmds = give_merged_commands(disp_cmds,cmds,pg_BOLD,0);
			
			retch = help_generic(all_cmds,0,quotad->current_page,prompt_area);
		    } else
			retch = help_generic(disp_cmds,0,quotad->current_page,prompt_area);
		    break;
		    
		case 'q':
		    
		    menu_Writechar(prompt_area,cmd);
		    FlushBuffer();
		    goto out;
		    
		default:
		    if (cmds) {
			const struct elm_command * ret_cmd =
			    lookup_elm_command(cmds,cmd);
			
			if (ret_cmd) {
			    menu_Writechar(prompt_area,cmd);
			    FlushBuffer();
			    retch = cmd;
			    goto out;
			}			
		    }
		    
		    DPRINT(Debug,10,(&Debug,
				     "quota_display_show: command %d not found\n",
				     cmd));
		    
		    menu_Write_to_screen(prompt_area,
					 FRM("%c??"), 07);
		    FlushBuffer();
		    if (sleepmsg > 0)
			error_sleep((sleepmsg + 1) / 2);
		    
		    break;
		}

	    } else {
		DPRINT(Debug,10, (&Debug,"quota_display_show: Too few lines: LINES=%d\n",
				  LINES));
		goto out;
	    }
	}
	
    out:
	free_cancel(&cd);
	if (all_cmds)
	    free_commands(&all_cmds);
	free_commands(&disp_cmds);
	
	if (prompt_area)
	    erase_menu_context(&prompt_area);
	
	
    }

    DPRINT(Debug,12,(&Debug,"quota_display_show=%d",retch));

    switch (retch) {
    case EOF: DPRINT(Debug,12,(&Debug," EOF")); break;
    default:
	if (isascii(retch) && isprint(retch)) {
	    DPRINT(Debug,12,(&Debug," '%c'",retch));
	}
	break;
    }        
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return retch;
}
     
void free_quota_display(quotad, parent_page)
     struct quota_display ** quotad;
     struct menu_context   * parent_page;
{
    if (QUOTA_DISPLAY_magic != (*quotad)->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "free_quota_display",
              "Bad magic number (quota_diplay)",0);
    
    if ((*quotad)->refcount < 1)
        panic("MBX PANIC",__FILE__,__LINE__,
	      "free_quota_diplay",
              "Bad refcount",0);
    
    (*quotad)->refcount--;
    if ((*quotad)->refcount > 0) {
        /* Just reset this reference */
        
        *quotad = NULL;
        return;
    }

    if ((*quotad)->mail_quota_list) {
	size_t i;

	for (i = 0; i < (*quotad)-> mail_quota_count; i++) {	    
	    if ((*quotad)-> mail_quota_list[i].mquota)
		free_mail_quota(&((*quotad)-> mail_quota_list[i].mquota));

	    if ((*quotad)-> mail_quota_list[i].mail_quotarootl_list) {
		size_t j;

		for (j = 0; j < (*quotad)-> mail_quota_list[i].mail_quotarootl_count; j++) {

		    if ((*quotad)-> mail_quota_list[i].mail_quotarootl_list[j].quota_list) 
			free_mail_quotaroot_list(& ((*quotad)-> mail_quota_list[i].
						    mail_quotarootl_list[j].quota_list));
		    
		    if ((*quotad)-> mail_quota_list[i].mail_quotarootl_list[j].
			folder_disp_name_list) {
			size_t k;

			for (k = 0; k < (*quotad)-> mail_quota_list[i].mail_quotarootl_list[j].
				 folder_disp_name_count; k++) {

			    if ((*quotad)-> mail_quota_list[i].mail_quotarootl_list[j].
				folder_disp_name_list[k]) {
				free_string(& ((*quotad)-> mail_quota_list[i].
					       mail_quotarootl_list[j].
					       folder_disp_name_list[k]));
			    }
			}

			free((*quotad)-> mail_quota_list[i].mail_quotarootl_list[j].
			     folder_disp_name_list);
			(*quotad)->mail_quota_list[i].mail_quotarootl_list[j].
			    folder_disp_name_list = NULL;
		    }
		    (*quotad)->mail_quota_list[i].mail_quotarootl_list[j].
			folder_disp_name_count = 0;
		}
		

		free((*quotad)-> mail_quota_list[i].mail_quotarootl_list);
		(*quotad)-> mail_quota_list[i].mail_quotarootl_list = NULL;
	    }
	    (*quotad)-> mail_quota_list[i].mail_quotarootl_count = 0;
	}

	free((*quotad)->mail_quota_list);
	(*quotad)->mail_quota_list = NULL;
    }
    (*quotad)->mail_quota_count = 0;


    if ((*quotad)->quota_page)
	erase_menu_context(& ((*quotad)->quota_page));
    
    if ((*quotad)->current_page)
	erase_menu_context(&  ((*quotad)->current_page));

    if (parent_page) 
	menu_set_default(parent_page);
    
    (*quotad)->magic = 0;  /* Invalidate */
    
    free(*quotad);
    *quotad = NULL;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
