static char rcsid[] = "@(#)$Id: signals.c,v 2.11 2020/09/17 16:31:38 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.11 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 src/signals.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This set of routines traps various signals and informs the
    user of the error, leaving the program in a nice, graceful
    manner.

**/

#include "def_elm.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"signal");

volatile int pipe_abort;		/* set to TRUE if receive SIGPIPE */

SIGHAND_TYPE
quit_signal(sig)
     int sig;
{
    int unsafe_interrupt = !wait_can_signal;
    wait_can_signal = 0;
    

    SIGDPRINT(Debug,1, (&Debug, 	      
			"\n\n** Received SIGQUIT **\n\n\n\n"));

    InGetPrompt = 0;    /*  If we are leaving we are not longer in prompt */

    leave(unsafe_interrupt,1);
}

SIGHAND_TYPE
hup_signal(sig)
     int sig;
{
    int unsafe_interrupt = !wait_can_signal;
    wait_can_signal = 0;    

    SIGDPRINT(Debug,1, (&Debug, 	      
			"\n\n** Received SIGHUP **\n\n\n\n"));

    InGetPrompt = 0;    /*  If we are leaving we are not longer in prompt */


    leave(unsafe_interrupt,1);
}

SIGHAND_TYPE
term_signal(sig) 
     int sig;
{
    int unsafe_interrupt = !wait_can_signal;
    wait_can_signal = 0;    

    SIGDPRINT(Debug,1, (&Debug, 	      
			"\n\n** Received SIGTERM **\n\n\n\n"));

    InGetPrompt = 0;    /*  If we are leaving we are not longer in prompt */

    leave(unsafe_interrupt,0 /* OK? */);
}

char * Ill_Text = "\n\nIllegal Instruction signal!\n\n";

SIGHAND_TYPE
ill_signal(sig)
     int sig;
{
    int unsafe_interrupt = !wait_can_signal;
    wait_can_signal = 0;    

    SIGDPRINT(Debug,1, (&Debug, 	       
			"\n\n** Received SIGILL **\n\n\n\n"));
    
    panic("SIGNAL PANIC",
	  __FILE__,__LINE__,"ill_signal",
	  Ill_Text,
	  unsafe_interrupt);
}

char * Fpe_Text = "\n\nFloating Point Exception signal!\n\n";

SIGHAND_TYPE
fpe_signal(sig)  
     int sig;
{   
    int unsafe_interrupt = !wait_can_signal;
    wait_can_signal = 0;    

    SIGDPRINT(Debug,1, (&Debug, "\n\n** Received SIGFPE **\n\n\n\n"));
    
    panic("SIGNAL PANIC",
	  __FILE__,__LINE__,"fpe_signal",
	  Fpe_Text,
	  unsafe_interrupt);
}

char * Bus_Text = "\n\nBus Error signal!\n\n";

#ifdef SIGBUS
SIGHAND_TYPE
bus_signal(sig)
     int sig;
{
    int unsafe_interrupt = !wait_can_signal;
    wait_can_signal = 0;    

    SIGDPRINT(Debug,1, (&Debug, "\n\n** Received SIGBUS **\n\n\n\n"));
 
    panic("SIGNAL PANIC",
	  __FILE__,__LINE__,"bus_signal",
	  Bus_Text,
	  unsafe_interrupt);
}
#endif

char * Segv_Text = "\n\nSegment Violation signal!\n\n";

SIGHAND_TYPE
segv_signal(sig)
     int sig;
{    
    int unsafe_interrupt = !wait_can_signal;
    wait_can_signal = 0;    

    SIGDPRINT(Debug,1, (&Debug, "\n\n** Received SIGSEGV **\n\n\n\n"));
    
    panic("SIGNAL PANIC",
	  __FILE__,__LINE__,"segv_signal",
	  Segv_Text,
	  unsafe_interrupt);
}

SIGHAND_TYPE
alarm_signal(sig)
     int sig;
{	
    signal(SIGALRM, alarm_signal);

    SIGDPRINT(Debug,4, (&Debug, "\n SIGALRM signal\n"));
    
    /** silently process alarm signal for timeouts... **/
    if (InGetPrompt && wait_can_signal) {

	wait_can_signal = 0;
	InGetPrompt = 0;    /*  No longer */
	
#ifdef HASSIGHOLD
	sigrelse(SIGALRM);
	
#endif
	LONGJMP(GetPromptBuf, 1);
    }
}

SIGHAND_TYPE
pipe_signal(sig)
     int sig;
{
    /** silently process pipe signal... **/
    SIGDPRINT(Debug,2, (&Debug, "*** received SIGPIPE ***\n\n"));
	
    pipe_abort = TRUE;	/* internal signal ... wheeee!  */
    
    signal(SIGPIPE, pipe_signal);
}

char * Stopped_Text = "\nSTOPPED\n";   /* Initialized by init.c */
char * Back_Text = "\nBack in ELM\n";  /* Initialized by init.c */

#ifdef SIGTSTP
int was_in_raw_state;

SIGHAND_TYPE
sig_user_stop(sig)
     int sig;
{
    SIGDPRINT(Debug,1, (&Debug,  "\n\n** Received SIGTSTP **\n\n\n\n"));
    /* This is called when the user presses a ^Z to stop the
       process within BSD 
    */
    
    menu_context_redraw();
    
    signal(SIGTSTP, SIG_DFL);
    
    was_in_raw_state = RawState();
    signal_leave_screen();	    /* turn it off regardless */
    
    switch_title(0);

    WriteRaw(Stopped_Text);
    
    kill(getpid(), SIGSTOP);
}

SIGHAND_TYPE
sig_return_from_user_stop(sig)
     int sig;
{  
    /** this is called when returning from a ^Z stop **/
    SIGDPRINT(Debug,1, (&Debug, "\n\n** Received SIGCONT **\n\n\n\n"));
    
    menu_context_redraw();
    
    signal(SIGCONT, sig_return_from_user_stop);
    signal(SIGTSTP, sig_user_stop);
    
    CarriageReturn();  /* Reset column counter? ... */
    
    WriteRaw(Back_Text);
    
    if (was_in_raw_state)
	signal_enter_screen();

    if (InGetPrompt && wait_can_signal) {

	wait_can_signal = 0;
	InGetPrompt = 0;    /*  no longer */
    
#ifdef HASSIGHOLD
	sigrelse(SIGTSTP);
	sigrelse(SIGCONT);
#endif
	LONGJMP(GetPromptBuf, 1);
    }
}
#endif

#ifdef SIGWINCH
SIGHAND_TYPE
winch_signal(sig)
     int sig;
{
    signal(SIGWINCH, winch_signal);
    
    SIGDPRINT(Debug,2, (&Debug, "*** received SIGWINCH ***\n\n"));

    menu_context_resize();

    if (InGetPrompt && wait_can_signal) {

	wait_can_signal = 0;
	InGetPrompt = 0;    /*  no longer */
   
#ifdef HASSIGHOLD
	sigrelse(SIGWINCH);
#endif
	LONGJMP(GetPromptBuf, 1);
    }
}
#endif

SIGHAND_TYPE
usr1_signal(sig)
     int sig;
{
    struct MailboxView *m;
    int idx2 = 0;
    int is_ok = 1;
    
    int unsafe_interrupt = !wait_can_signal;
    
    SIGDPRINT(Debug,1, (&Debug, "\n\n** Received SIGUSR1 **\n\n\n\n"));
    if (!InGetPrompt || unsafe_interrupt) {
	SIGDPRINT(Debug,1, (&Debug, "-- can't leave folder\n"));
	return;
    }
    wait_can_signal = 0;    


    while (NULL != (m = give_next_open_mailbox(&idx2,!InGetPrompt))) {
	int retry = 1;

    doit:
	while (m && retry) {	    
	    enum sync_mbox_stat result =
		sync_mbox(TRUE, FALSE, TRUE, NULL, m,
			  default_context,NULL);

	    retry = 0;

	    switch (result) {
	    case sync_mbox_newmail: {
		int idx;
		int mbxcount = get_storage_count(m);

		SIGDPRINT(Debug,1, (&Debug, " ** new mail arrived ** \n"));
		
		for (idx = 0 ; idx < mbxcount; idx++) {
		    
		    struct current_storage * storage = get_storage(m,idx);
		    
		    if (!storage ||
			!storage->current_folder)
			continue;

		    retry = 1;
		    read_new_mails(storage, default_context);
		}
	    }
		goto doit;

	    case sync_mbox_cancel:
	    case sync_mbox_reconnect:
	    case sync_mbox_failure:
	    case sync_mbox_nochange:
	    case sync_mbox_changed:
		
		break;

	    case sync_mbox_EOF:
		emergency_exit(unsafe_interrupt);
		break;

	    }
	    
	    if (result < sync_mbox_nochange) {
		SIGDPRINT(Debug,1, (&Debug, "sync_mbox result=%d  (can't leave)\n",result));

		is_ok = 0;
		break;
	    }
	    
	}
    }

    if (is_ok) {
	InGetPrompt = 0;    /*  If we are leaving we are not longer in prompt */
	
	leave(unsafe_interrupt,0 /* OK? */);  
    }

}

SIGHAND_TYPE
usr2_signal(sig)
     int sig;
{
    struct MailboxView *m;
    int idx2 = 0;
    int is_ok = 1;
    
    int unsafe_interrupt = !wait_can_signal;


    SIGDPRINT(Debug,1, (&Debug, "\n\n** Received SIGUSR2 **\n\n\n\n"));
    
    if (!InGetPrompt || unsafe_interrupt) {
	SIGDPRINT(Debug,1, (&Debug,  "-- can't leave folder\n"));
	return;
    }
    wait_can_signal = 0;    

    while (NULL != (m = give_next_open_mailbox(&idx2,!InGetPrompt))) {
	int retry = 1;

    doit:
	while (m && retry) {
	    
	    enum sync_mbox_stat result =
	       sync_mbox(FALSE, TRUE, FALSE, NULL,
			 m,
			 default_context,NULL);

	    retry = 0;

	    
	    switch (result) {
	    case sync_mbox_newmail: {
		int idx;
		int mbxcount = get_storage_count(m);

		SIGDPRINT(Debug,1, (&Debug, " ** new mail arrived ** \n"));
		
		for (idx = 0 ; idx < mbxcount; idx++) {
		    
		    struct current_storage * storage = get_storage(m,idx);
		    
		    if (!storage ||
			!storage->current_folder)
			continue;
		
		    read_new_mails(storage, default_context);

		    retry = 1;
		}

	    }
		goto doit;

	    case sync_mbox_cancel:
	    case sync_mbox_reconnect:
	    case sync_mbox_failure:
	    case sync_mbox_nochange:
	    case sync_mbox_changed:
		break;
	    case sync_mbox_EOF:
		emergency_exit(unsafe_interrupt);
		break;		
	    }
	    
	    if (result < sync_mbox_nochange) {
		SIGDPRINT(Debug,1, (&Debug, "sync_mbox result=%d  (can't leave)\n",result));
				    
		is_ok = 0;
		break;
	    }
	}
    }
    
    if (is_ok) {
	InGetPrompt = 0;    /*  If we are leaving we are not longer in prompt */

	leave(unsafe_interrupt,0 /* OK? */);
    }
}

void init_signals()
{

    DPRINT(Debug,4, (&Debug, "Initializing terminating signals\n"));

    signal(SIGINT,  SIG_IGN);
    signal(SIGQUIT, quit_signal);		/* Quit signal 	            */
    signal(SIGTERM, term_signal); 	/* Terminate signal         */
    signal(SIGILL,  ill_signal);		/* Illegal instruction      */
    signal(SIGFPE,  fpe_signal);		/* Floating point exception */
#ifdef SIGBUS
    signal(SIGBUS,  bus_signal);		/* Bus error  		    */
#endif
    signal(SIGSEGV, segv_signal);		/* Segmentation Violation   */
    signal(SIGHUP,  hup_signal);		/* HangUp (line dropped)    */
    signal(SIGUSR1, usr1_signal);		/* User request 1	    */
    signal(SIGUSR2, usr2_signal);		/* User request 2	    */
}

void init_signals1() {
#if defined(SIGVEC) & defined(SV_INTERRUPT)
    struct sigvec alarm_vec;
#endif /* defined(SIGVEC) & defined(SV_INTERRUPT) */

    DPRINT(Debug,4, (&Debug, "Initializing operating signals\n"));

#if !defined(POSIX_SIGNALS) & defined(SIGVEC) & defined(SV_INTERRUPT)
    bzero((char *) &alarm_vec, sizeof(alarm_vec));
    alarm_vec.sv_handler = alarm_signal;
    alarm_vec.sv_flags = SV_INTERRUPT;
    sigvec (SIGALRM, &alarm_vec, (struct sigvec *)0);	/* Process Timer Alarm	    */
#else /* defined(SIGVEC) & defined(SV_INTERRUPT) */
    signal(SIGALRM, alarm_signal);		/* Process Timer Alarm      */
#endif /* defined(SIGVEC) & defined(SV_INTERRUPT) */
    signal(SIGPIPE, pipe_signal);		/* Illegal Pipe Operation   */
#ifdef SIGTSTP
    signal(SIGTSTP, sig_user_stop);		/* Suspend signal from tty  */
    signal(SIGCONT, sig_return_from_user_stop);	/* Continue Process */
#endif
#ifdef SIGWINCH
    signal(SIGWINCH, winch_signal);		/* change screen size */
#endif

}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
