static char rcsid[] = "@(#)$Id: elmalias.c,v 2.13 2018/12/21 11:21:43 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.13 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 utils/elmalias.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/*
 * elmalias - Elm alias database access.
 *
 * This utility will display information from the Elm user and system
 * alias databases.  It will expand alias names specified on the command
 * line and display the alias value.  If an item on the command line is
 * not a valid alias, then it's value is returned.  If no names are
 * specified on the command line then all alises are displayed.  Output
 * is formatted by either a user-defined format specifier or one of a
 * number of predefined specifications.  The default output format
 * displays just the alias value.
 *
 * The format specifiers are:
 *
 *	%a	alias (the alias name)
 *	%c	comment
 *
 *    person alias
 *	%l	last name
 *      %f      first name
 *      %A      address (the alias value)
 *
 *    address list
 *      %L      address list (the alias value)
 *
 *    group
 *      %P      phrase
 *      %V      vector (the alias value)
 *
 *    general
 *	%n	name
 *	%v	address (the alias value)
 *	%t	type (Person, Group, or Unknown)
 *
 * In the case that a command line argument does not correspond to
 * an alias, then %a and %v will evaluate to the argument value and
 * all other items will be empty.
 *
 * Printf-like field widths may be used, e.g. "%-20.20a".  Conditionals
 * are bracketed by question marks.  For example "?n(%n)?" means format
 * "(%n)" if the "name" is defined, otherwise produce no output.  Some
 * backslash sequences (e.g. "\t" == tab) are recognized.  Any other
 * characters are displayed as is.
 *
 * Example:
 *	$ elmalias -f "alias %a is address \"%v\" ?n(%n)?" chip
 *	alias chip is address "chip@chinacat.unicom.com" (Chip Rosenthal)
 *
 * Synopsis:
 *	elmalias [-adensuvV] [-f format] [alias ...]
 *
 * Options:
 *	-a	Display alias name.  Equivalent to -f "%-20.20a %v".
 *	-d	Turn debugging on. Only useful if DEBUG is defined.
 *	-e	Fully expand alias values.
 *	-f fmt	User-specified output format.
 *	-n	Display name.  Equivalent to -f "%v?n (%n)?".
 *	-r	Complain about arguments that are not valid aliases.
 *	-s	Consult system-wide alias file.
 *	-u	Consult user-specific alias file.
 *	-v	Verbose output.  Equivalent to -f "%-20.20a %v?n (%n)?".
 *	-V	Babble about *everything* we known about.
 *
 * If neither -s or -u are specified, then the default is to search
 * both alias files.
 */

#include "def_utils.h"

#include "s_elmalias.h"
#include <pwd.h>

#include "addrlib.h"
#include "aliaslib.h"

DEBUG_VAR(Debug,__FILE__,"util");

/*
 * Maximum number of alias files we can consult.
 */
#define MAXDB		2	/* user and system alias files		*/

/*
 * These are used by the "dbz" routines.
 */

char *Progname;

/*
 * "aliasdb" library routines.
 */

/*
 * Local procedures.
 */

static struct address_alias *make_dummy_rec P_((const struct string *key));
static void exp_print_alias P_((struct aliases_map *dblist[], 
				const char *fmt,  
				const struct address_alias *ar,
				const struct string *key));
static void print_alias P_((const char *fmt, 
			    const struct address_alias *ar,
			    const struct string *key,
			    int nogroup));
static struct string * sel_alias_mem P_((const struct address_alias *ar,
					 int sel,
					 const struct string *key,
					 int nogroup));


static void usage_error P_((void));
static void usage_error()
{
    lib_error(CATGETS(elm_msg_cat, ElmaliasSet, ElmaliasUsage,
		      "usage: %s [-adenrsuvV] [-f format] [alias ...]\n"), 
	      Progname);
    exit(1);
    /*NOTREACHED*/
}


static void malloc_fail_handler P_((char *proc, size_t size));


/*ARGSUSED*/
static void malloc_fail_handler(proc, size)
     char *proc;
     size_t size;
{
   /* NOTE:
       Can't use elm_fprintf (or routines of lib/output.c)
       because here also malloc may fail, therefore
       can not use CATGETS macro;
       And can't use catgets because it may have given 
       incorrent format string from catalog ...
    */

      fprintf(stderr,"%s: out of memory [could not allocate %lu bytes]\n", 
	      Progname, (unsigned long)size);
      exit(1);
    /*NOTREACHED*/
}

static char version_buff[NLEN];


static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    const char *out_fmt;	/* output printing format		*/
    int do_user_alias;		/* TRUE to examine user alias file	*/
    int do_system_alias;	/* TRUE to examine system alias file	*/
    int do_expand;		/* TRUE to recursively expand aliases	*/
    int do_complain;		/* TRUE to insist args are valid aliases*/
    int numdb;			/* number of alias files to consult	*/
    struct aliases_map * dblist[MAXDB+1];	/* NULL terminated list of files	*/

    const int read_flags = 0;
    
    int d, i;
    extern int optind;
    extern char *optarg;

    /*
     * Initialize.
     */

#if DEBUG
    init_debugfile("ELMALIAS");

    /* HACK .... 
       USER may want enable debug output for locale_init()
       
       This gives wrong result if some options value start
       with "-d"
    */

    for (i = 1; i < argc; i++) {
	if ('-' == argv[i][0] &&
	    'd' == argv[i][1] &&
	    argv[i][2]) {
	    set_debugging(& (argv[i][2]));
	}
    }
#endif
    locale_init();

    safe_malloc_fail_handler =	/* install procedure to trap errors in	*/
	malloc_fail_handler;	/*   the safe_malloc() routines		*/
    Progname = argv[0];		/* program name for diag messages	*/
    do_user_alias = FALSE;	/* indicate the user hasn't selected	*/
    do_system_alias = FALSE;	/*   any alias files (yet)		*/
    do_expand = FALSE;		/* do not recursively expand groups	*/
    do_complain = FALSE;	/* allow non-aliases on cmd line	*/
    out_fmt = "%v";		/* default is to just show alias value	*/
    numdb = 0;			/* no alias files opened yet		*/

    user_init();
    init_addrlib(read_flags);
    init_aliaslib(read_flags);
    init_defaults(read_flags);

    /*
     * Crack command line options.
     */
    while ((i = getopt(argc, argv, "ad:ef:nrsuvVc:")) != EOF) {
	switch (i) {
	case 'a':			/* show alias name and value	*/
	    out_fmt = "%-20.20a %v";
	    break;
	
	case 'c':
	    set_user_rc_location(optarg); 
	    break;

	case 'd' : 
#if DEBUG
	  set_debugging(optarg);	  
#else
 	  lib_error(CATGETS(elm_msg_cat, ElmaliasSet, ElmaliasArgsIngoringDebug,
			    "Warning: system created without debugging enabled - request ignored\n"));
#endif
	  break;
	case 'e':			/* recursively expand aliases	*/
	    do_expand = TRUE;
	    break;
	case 'f':			/* user-specified format	*/
	    out_fmt = optarg;
	    break;
	case 'n':			/* show alias value and fullname*/
	    out_fmt = "%v?n (%n)?";
	    break;
	case 'r':			/* insist args are valid aliases*/
	    do_complain = TRUE;
	    break;
	case 's':			/* use system-wide alias file	*/
	    do_system_alias = TRUE;
	    break;
	case 'u':			/* use per-user alias file	*/
	    do_user_alias = TRUE;
	    break;
	case 'v':			/* verbose output format	*/
	    out_fmt = "%-20.20a %v?n (%n)?";
	    break;
	case 'V':			/* show the user's life story	*/
	    out_fmt = "\
Alias:\t\t%a\n\
  Address:\t%v\n\
  Type:\t\t%t\n\
?n  Name:\t\t%n\n?\
?l  Last Name:\t%l\n?\
?c  Comment:\t%c\n?\
";
	    break;
	default:
	    usage_error();
	    /*NOTREACHED*/
	}
    }

    read_rc_file(read_flags);

    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);

#ifdef DEBUG
    { 
	int d = panic_dprint("\n\
======================================================\n\
Debug output of the ELMALIAS program (version %s).\n",
			     version_buff);

	if (d >= 50) {
#if 0	
	    panic_dprint("WARNING: Edit manually out sensitive information from that file!\n");
    
	    lower_prompt("WARNING: Debug file may include passwords -- edit it!");
	    sleep(5+sleepmsg);	    
#endif
	}
    }
#endif


    /*
     * If user didn't request specific alias files then use them all.
     */
    if (!do_system_alias && !do_user_alias)
	do_system_alias = do_user_alias = TRUE;


    /*
     * Open up the alias files we need to access, in the order of priority.
     */
    if (do_user_alias && (dblist[numdb] = user_aliases_map) != NULL)
	++numdb;
    if (do_system_alias && (dblist[numdb] = system_aliases_map) != NULL)
	++numdb;
    dblist[numdb] = NULL;

   
    /*
     * If no names specified on command line then dump all alias files..
     */
    if (optind == argc) {
	if (do_expand) {
	  lib_error(CATGETS(elm_msg_cat, ElmaliasSet,
			    ElmaliasCannotSpecifyExpand,
			    "%s: cannot specify \"-e\" when dumping all aliases\n"),
		    Progname);
	  exit(1);
	}
	for (d = 0 ; dblist[d] != NULL ; ++d) {
	    int count = aliases_map_item_count(dblist[d]);
	    int i;

	    for ( i = 0; i < count; i++) {
		const struct string *key = NULL;
		const struct address_alias * ar =
		    aliases_map_get_alias(dblist[d],i,&key);

		print_alias(out_fmt, ar,key,0);

	    }
	}
	exit(0);
    }


    /*
     * Expand each name on the command line.
     */
    for (i = optind ; i < argc ; ++i) {

	const struct address_alias * ar = NULL;

	struct string * key = new_string2(system_charset,
					  s2us(argv[i]));


	/*
	 * Try each of the alias files for a match.
	 */

	for (d = 0 ; dblist[d] != NULL ; ++d) {
	    int idx;

	    if ((ar = aliases_map_lookup_alias(dblist[d], 
					       key, &idx)) != NULL)
		break;
	}


	/*
	 * Print the result.
	 */
	if (ar == NULL) {
	    if (do_complain) {
		lib_error(CATGETS(elm_msg_cat, ElmaliasSet,
				  ElmaliasUnknownAlias, 
				  "%s: \"%S\" is not a known alias\n"),
			  Progname, key);
		exit(1);
	    }

	    {
		struct address_alias * ar0 = make_dummy_rec(key);
		print_alias(out_fmt, ar0, key, -1);
		free_address_alias(&ar0);
	    }

	} else if (do_expand) {
	    exp_print_alias(dblist, out_fmt, ar,key);
	} else {
	    print_alias(out_fmt, ar,key,0);
	}


	free_string(&key);
    }

    return 0;
}


/*
 * Cobble up an alias record structure to hold some address info.
 */
static struct address_alias *make_dummy_rec(key)
     const struct string *key;
{
    struct address_alias *ret = NULL;
    struct address * address  = new_address_string (key,NULL,NULL);

    ret = new_person_address_alias(NULL,NULL,NULL,address);

    free_address(&address);

    return ret;
}


/*
 * Recursively expand out a list of addresses and print the expansions.
 */
static void exp_print_alias(dblist, fmt, ar, key)
     struct aliases_map *dblist[];
     const char *fmt;
     const struct address_alias *ar;
     const struct string *key;
{

    const struct string *firstn   = NULL;
    const struct string *lastn    = NULL;
    const struct address *address = NULL;

    const struct string *phrase       = NULL;
    const struct alias_vector *vector = NULL;

    /* Alias cab be same time person, list and group alias */

    if (address_alias_get_person(ar,&firstn,&lastn,&address) ||
	address_alias_get_list(ar))
	print_alias(fmt, ar, key,1);
	
    
    if (address_alias_get_group(ar,&phrase,&vector)) {
	int count = alias_vector_item_count(vector);

	int i;

	for (i = 0; i < count; i++) {
	    const struct string * key1 = 
		alias_vector_get_alias (vector,i);

	    const struct address_alias * ar0 = NULL;
	    int d;

	    for (d = 0 ; dblist[d] != NULL ; ++d) {
		int idx;
		
		if ((ar0 = aliases_map_lookup_alias(dblist[d], 
						   key1, &idx)) != NULL)
		    break;
	    }

	    if (ar0 == NULL) {
		struct address_alias * ar1 = make_dummy_rec(key1);
		print_alias(fmt, ar1, key1, -1);
		free_address_alias(&ar1);
	    } else {
		exp_print_alias(dblist, fmt, ar0,key1);
	    }
	}	
    }

}


/*
 * Print out alias information according to a format specification.
 */
void print_alias(fmt, ar, key, nogroup)
     const char *fmt;
     const struct address_alias *ar;
     const struct string *key;
     int nogroup;
{
    int in_conditional;		/* TRUE if in middle of cond expression	*/
    int print_enab;		/* TRUE if OK to print output		*/
    int c;

    print_enab = TRUE;
    in_conditional = FALSE;

#define INC(p) { (p)++; if (!*(p)) break; }

    while (*fmt != '\0') {

	switch (*fmt) {

	/*
	 * Formatted output.
	 */
	case '%': {
	    struct string * value = NULL;
	    struct  format_elem  E;

	    /* bzero is defined hdrs/elm_defs.h */
	    bzero((void *)&E,sizeof E);

	    E.fill   = ' ';
            E.left   = 0;
            E.val1   = -1;
            E.val2   = -1;



	    /*
	     * Extract the "%m.n" portion of the format.
	     */

	    INC(fmt);
	    
	    if ('0' == *fmt)      { E.fill = '0'; INC(fmt); }
	    else if ('-' == *fmt) { E.left = 1;   INC(fmt); }

	    if (*fmt >= '1' && *fmt <= '9') {
		E.val1 = 0;
		while (*fmt >= '0' && *fmt <= '9') {
		    E.val1 = E.val1 * 10 + *fmt - '0';
		    INC(fmt);
		}	    
	    }

	    if ('.' == *fmt) {
		INC(fmt);

		if (*fmt >= '1' && *fmt <= '9') {
		    E.val2 = 0;
		    while (*fmt >= '0' && *fmt <= '9') {
			E.val2 = E.val2 * 10 + *fmt - '0';
			INC(fmt);
		    }
		}
	    }



	    /*
	     * Determine what we are printing.
	     */
	    if ((value = sel_alias_mem(ar, *fmt, key, nogroup)) == NULL) {
		value = format_string(CATGETS(elm_msg_cat, ElmaliasSet, 
					      ElmaliasIllegalFmtChar,
					      "<illegal format char>"));
	    }

	    /*
	     * Print out the formatted field.
	     */
	    if (print_enab) {
		int len = 0;
		struct string * formatted = 
		    format_S_helper(value,&E,-1,&len);

		elm_fprintf(stdout,FRM("%S"),formatted);

		free_string(&formatted);
	    }

	    free_string(&value);

	}
	    break;

	/*
	 * Conditional printing.
	 */
	case '?':
	    if (in_conditional) {
		in_conditional = FALSE;
		print_enab = TRUE;
	    } else {
		struct string * value = sel_alias_mem(ar, *++fmt,key,nogroup);
		in_conditional = TRUE;
		
		if (value) {
		    print_enab = string_len(value) > 0;
		    free_string(&value);
		} else
		    print_enab = 0;
	    }
	    break;
	
	/*
	 * Backslash escapes.
	 */
	case '\\':
	    switch (*++fmt) {
		case 'b':  c = '\b'; break;
		case 'f':  c = '\f'; break;
		case 'n':  c = '\n'; break;
		case 'r':  c = '\r'; break;
		case 't':  c = '\t'; break;
		default:   c = *fmt; break;
	    }
	    if (print_enab && c != '\0')
		putchar(c);
	    break;

	/*
	 * Non-special character to print.
	 */
	default:
	    if (print_enab)
		putchar(*fmt);
	    break;

	}

	if (*fmt != '\0')
	    ++fmt;

    }

    putchar('\n');
}


/*
 * Select a member of the alias record structure.
 */
static struct string *sel_alias_mem(ar, sel,key,nogroup)
     const struct address_alias *ar;
     int sel;
     const struct string *key;
     int nogroup;
{
    /* nogroup:
          0 == show group
          1 == exclude group
	 -1 == unknown alias
    */

    const struct string *firstn   = NULL;
    const struct string *lastn    = NULL;
    const struct address *address = NULL;

    const struct string *phrase       = NULL;
    const struct alias_vector *vector = NULL;

    int is_person = address_alias_get_person(ar,&firstn,&lastn,&address);

    const struct addr_list * addr_list = address_alias_get_list(ar);
    
    int is_group = 0;

    const struct string * comment = address_alias_get_comment(ar);

    if (nogroup < 1)
	is_group = address_alias_get_group(ar,&phrase,&vector);

    switch (sel) {
    case 'a':
	return dup_string(key);

	/* Person alias */

    case 'l':
	if (lastn)
	    return dup_string(lastn);
	break;
    case 'f':
        if (firstn)
	    return dup_string(firstn);
        break;
    case 'A':
	if (address)
	    return address_to_string(address);
	break;

	/* Address List */
    case 'L':
	if (addr_list)
	    return addr_list_to_string(addr_list);
	break;

	/* Group */
    case 'P':
        if (phrase)
	    return dup_string(phrase);
	break;
    case 'V':
	if (vector) {
	    struct string * X = new_string(system_charset);
	    int count = alias_vector_item_count(vector);
	    int i;

	    for (i = 0; i < count; i++) {
		if (i)
		    add_ascii_to_string(X,s2us(", "));
		append_string(&X,alias_vector_get_alias(vector,i),1);
	    }

	    return X;
	}
	break;

	/* General */
    case 'n': {
	struct string * X = new_string(system_charset);

	if (firstn)
	    append_string(&X,firstn,1);

	if (lastn) {
	    if (firstn)
		add_ascii_to_string(X,s2us(" "));
	    append_string(&X,lastn,1);
	}

	if (phrase) {
	    if (firstn || lastn)
		add_ascii_to_string(X,s2us(", "));
	    append_string(&X,phrase,1);
	}

	return X;
    }
	break;

    case 'c':
	if (comment)
	    return dup_string(comment);
	break;

    case 'v': {  /* Simulate old address value */
	struct string * X = new_string(system_charset);

	if (address) {
	    struct string * X1 = address_to_string(address);

	    append_string(&X,X1,1);
	    free_string(&X1);
	}

	if (addr_list) {
	    struct string * X1 = addr_list_to_string(addr_list);

	    if (address)
		add_ascii_to_string(X,s2us(", "));

	    append_string(&X,X1,1);
	    free_string(&X1);
	}

	if (vector) {
	    int count = alias_vector_item_count(vector);
	    int i;

	     if (address || addr_list)
		 add_ascii_to_string(X,s2us(", "));

	     for (i = 0; i < count; i++) {
		 if (i)
		     add_ascii_to_string(X,s2us(", "));
		 append_string(&X,alias_vector_get_alias(vector,i),1);
	     }	     
	}

	return X;
    }
	break;

    case 't':

	if (nogroup < 0)

	    return format_string(CATGETS(elm_msg_cat, ElmaliasSet,
					 ElmaliasTypeUnknown, "Unknown"));
	
	else if ((is_person && is_group)  
		 ||
		 (is_person && addr_list) 
		 ||
		 (is_group  && addr_list)
		 ) 
	    return format_string(CATGETS(elm_msg_cat, ElmaliasSet,
					 ElmaliasTypeMixed, "Mixed"));

	else if (is_person)
	    
	    return format_string(CATGETS(elm_msg_cat, ElmaliasSet,
					 ElmaliasTypePerson, "Person"));

	else if (addr_list)
	    
	    return format_string(CATGETS(elm_msg_cat, ElmaliasSet,
					 ElmaliasTypeAddrList, "Address list"));
	    
	else if (is_group)
	    
	    return format_string(CATGETS(elm_msg_cat, ElmaliasSet,
					 ElmaliasTypeGroup, "Group"));
	else
	    return format_string(CATGETS(elm_msg_cat, ElmaliasSet,
					 ElmaliasTypeUnknown, "Unknown"));

    default:
	return NULL;
    }

   /* Return empty string indicating non-existang field */
   return new_string(system_charset);
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
