static char rcsid[] = "@(#)$Id: elmhashmarks.c,v 2.2 2018/12/21 11:21:43 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.2 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_utils.h"
#include "reghelper.h"
#include "mboxlib.h"
#include "rc_imp.h"
#include "s_me.h"
#include "s_elm.h"

extern char *optarg;		
extern int   optind;		

DEBUG_VAR(Debug,__FILE__,"util");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


const char * program_name = "elmhashmarks";

int    register_fd  = -1;
char * register_module = NULL;

static char version_buff[NLEN];

static char ** copydirs = NULL;
static int     copydir_count = 0;

S_(usermap_copy_callback usermap_copy)
static FILE *usermap_copy P_((const char *name,
			      const char *pathname));
static FILE *usermap_copy(name,pathname)
     const char *name; 
     const char *pathname;
{
    int i;
    FILE *F = NULL;
    FILE *res = NULL;
    int fd,c;

    for (i = 0; i < copydir_count; i++) {
	char * n = elm_message(FRM("%s/%s"),copydirs[i],name);
	
	F = fopen(n,"r");

	if (F) {
	    lib_transient(CATGETS(elm_msg_cat, MeSet,MeCopyingMapfile,
				  "Copying mapfile %s: %s => %s"),
			  name,n,pathname);
	    log_config((char *)pathname);
	}
	free(n);
	if (F)
	    break;
    }

    if (!F) {
	lib_transient(CATGETS(elm_msg_cat, MeSet,
			      MeNoCopyingMapfile,
			      "*** Copying mapfile %s => %s -- file %s not found?"),
		      name,pathname,name);
	return NULL;
    }

    fd = open(pathname, O_RDWR | O_CREAT | O_EXCL, 0444);
    if (fd < 0) {
	int errcode;
	
    error1:
	errcode = errno;

	lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			  "File %.50s is not writeable: %s"),
		  pathname, strerror(errcode));

	fclose(F);
	if (fd >= 0)
	    close(fd);
	return NULL;
    }

    res = fdopen(fd,"w+");
    if (!res)
	goto error1;

    while (EOF != (c = getc(F))) {
	if (putc(c,res) == EOF) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorTryingToWrite,
			      "Error %s encountered trying to write to %s."),
		      strerror(err),pathname);
	    fclose(F);
	    fclose(res);
	    return NULL;
	}
    }
	   
    if (ferror(res)) {
	lib_error(CATGETS(elm_msg_cat, ElmSet, ElmErrorReading,
			  "Error reading from %.50s"),
		  name);
	fclose(F);
	fclose(res);
	return NULL;
    }

    fclose(F);

    rewind(res);
    return res;
}

static const char OPTION_LETTERS[] = "Gd:cC:w:S:r:";

int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    int err = 0;
    int c;
    int global = 0;   
    struct hash_marks_conf  * MAP = NULL;
    charset_t                cs = NULL;
    struct editor_propline * pl = NULL;

    char *targetfile = NULL;
    FILE * commentfile = NULL;
    int config_merge = 0;

    int read_flags = 0;
    
#if DEBUG
    init_debugfile("ELMHASHMARKS");
#endif
    
    while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	switch(c) {
	case 'G':
	    global++;
	    break;
	case 'd':
#if DEBUG
	    set_debugging(optarg);
#endif
	    /* Error is printed later */
	    break;
	}
    }
    optind = 1;     /* Reset scanning */

    REGHELPER_INIT(argv[0]);

    read_flags = global ? READ_FLAG_GLOBAL_ONLY : 0;
	
    user_init();

    init_mboxlib(read_flags);
    init_defaults(read_flags);
    
    while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	switch(c) {

	case 'G':
	    global++;
	    break;
	case 'c':
	    config_merge++;
	    break;

	case 'C':
	    if (0 != access(optarg,READ_ACCESS)) {
		int errcode = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotReadable,
				  "File %.50s is not readable: %s"),
			  optarg, strerror(errcode));
		err = 1;
		goto fail;
	    }

	    commentfile = fopen(optarg,"r");
	    if (!commentfile) {
		int errcode = errno;
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotReadable,
				  "File %.50s is not readable: %s"),
			  optarg, strerror(errcode));
		err = 1;
		goto fail;
		
	    }
	    break;

	case 'S':
	    if (0 == access(optarg,ACCESS_EXISTS)) {
		copydirs = safe_array_realloc(copydirs,(copydir_count+1), 
					      sizeof (*copydirs));
		copydirs[copydir_count++] = optarg;
	    } else {
		int errcode = errno;
		
		lib_error(CATGETS(elm_msg_cat, MeSet,
				  MeDirNotExists,
				  "Directory %s does not exists: %s"),
			  optarg,strerror(errcode));		
		err = 1;
	    }
	    break;

	case 'd' : 
#if DEBUG
	    set_debugging(optarg);	 
#else
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmArgsIngoringDebug,
			      "Warning: system created without debugging enabled - request ignored\n"));
#endif
	    break;

	case 'r':
	    set_user_rc_location(optarg); 
	    break;

	case 'w' : 
	    targetfile = optarg;
	    if (0 != access(targetfile,WRITE_ACCESS)) {
		int errcode = errno;
		
		if (errcode != ENOENT) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
				      "File %.50s is not writeable: %s"),
			      targetfile, strerror(errcode));
		    err = 1;
		    goto fail;
		}
	    }
	    break;
	case '?':
	    err = 1;
	    goto fail;
	}
    }

    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);
    
#ifdef DEBUG
    { 
	int d UNUSED_VAROK = panic_dprint("\n\
======================================================\n\
Debug output of the ELMHASHMARKS program (version %s).\n",
					  version_buff);

#if 0	
	if (d >= 50) {
	    panic_dprint("WARNING: Edit manually out sensitive information from that file!\n");
    
	    lower_prompt("WARNING: Debug file may include passwords -- edit it!");
	    sleep(5+sleepmsg);	    
	}
#endif
    }
#endif

    if (copydir_count)
	set_usermap_copy_callback(usermap_copy);

    if (!global)
	read_rc_file(read_flags);
    else
	post_init_check(read_flags);

    if (optind < argc) {
	int errcount = 0;
	MAP = parse_hash_mark_entries(argv[optind],&errcount,&cs,&pl,
				      global ? SYSTEM_RC : LOCAL_RC);

	if (!MAP || errcount) {
	    err = 1;
	    goto fail;
	}
    }

    if (config_merge) {

	if (!MAP) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeConfigMergeMap,
			      "Config merge (-c) requires map as argument"));
	    err = 1;
	    goto fail;
	}

	if (targetfile) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeConfigMergeNoW,
			      "Config merge (-c) Can not used with -w"));
	    err = 1;
	    goto fail;
	}

	if (pl)
	    free_editor_propline(&pl);

	if (global) {


	    change_hash_marks_conf(&system_hash_marks_conf,MAP);
	    free_hash_marks_conf(&MAP);

	    MAP = system_hash_marks_conf;
	    targetfile = system_hash_marks;
	    cs  = system_hash_marks_cs;
	    pl  = system_hash_marks_pl;

	} else {

	    change_hash_marks_conf(&user_hash_marks_conf,MAP);
	    free_hash_marks_conf(&MAP);

	    MAP = user_hash_marks_conf;
	    targetfile = user_hash_marks;
	    cs  = user_hash_marks_cs;
	    pl  = user_hash_marks_pl;

	}

	if (0 != access(targetfile,WRITE_ACCESS)) {
	    int errcode = errno;
	    
	    if (errcode != ENOENT) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
				  "File %.50s is not writeable: %s"),
			  targetfile, strerror(errcode));
		err = 1;
		goto fail;
	    }
	}
    }
	
    if (!MAP) {
	if (global) {
	    MAP = system_hash_marks_conf;
	    cs  = system_hash_marks_cs;
	    pl  = system_hash_marks_pl;
	} else {
	    MAP = user_hash_marks_conf;
	    cs  = user_hash_marks_cs;
	    pl  = user_hash_marks_pl;
	}
    }
    
    if (targetfile) {
	char * tmp = elm_message(FRM("%s.N"),targetfile);
	int errcode = can_open(tmp,"w");
	FILE *f;
	
	if (errcode) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err = 1;
	    free(tmp);
	    goto fail;
	}
	f = fopen(tmp,"w");
	if (!f) {
	    int errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err = 1;
	    free(tmp);
	    goto fail;
	}

	if (! dump_hash_mark_entries(f,MAP,commentfile,
				     "ELMHASHMARKS",
				     version_buff,
				     cs,pl,
				     global ? SYSTEM_RC : LOCAL_RC)) {
	    err = 1;
	    fclose(f);
	    free(tmp);
	    goto fail;	
	}

	if (EOF == fclose(f)) {
	    int errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotWriteable,
			      "File %.50s is not writeable: %s"),
		      tmp, strerror(errcode));
	    
	    err = 1;
	    free(tmp);
	    goto fail;
	}
	if (0 != rename(tmp,targetfile)) {
	    int errcode = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileNotRenamed,
			      "Failed to rename temporary file to %.50s: %.30s"),
		      targetfile, strerror(errcode));
	    
	    err = 1;
	    free(tmp);
	    goto fail;
	}
	log_config(targetfile);
	free(tmp);

    } else {
	if (!dump_hash_mark_entries(stdout,
				    MAP,commentfile,
				    "ELMHASHMARKS",
				    version_buff,
				    system_charset,pl,
				    global ? SYSTEM_RC : LOCAL_RC)) {
	    err = 1;
	    goto fail;
	}
    }

 fail:
    if (commentfile)
	fclose(commentfile);

    if (MAP &&
	MAP != system_hash_marks_conf &&
	MAP != user_hash_marks_conf)
	free_hash_marks_conf(&MAP);

    if (err)
	lib_error(CATGETS(elm_msg_cat, MeSet, MeProgFailed,
			  "%s failed; exit code=%d"),
		  argv[0],err);


    return err;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
