static char rcsid[] = "@(#)$Id: fastmail.c,v 2.41 2022/11/02 16:26:29 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.41 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 utils/fastmail.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This program is specifically written for group mailing lists and
    such batch type mail processing.  It does NOT use aliases at all,
    it does NOT read the /etc/password file to find the From: name
    of the user and does NOT expand any addresses.  It is meant 
    purely as a front-end for either /bin/mail or /usr/lib/sendmail
    (according to what is available on the current system).

         **** This program should be used with CAUTION *****

**/


/** The calling sequence for this program is:

	fastmail {args}  [ filename | - ] full-email-address 

   where args could be any (or all) of;

	   -b bcc-list		(Blind carbon copies to)
	   -c cc-list		(carbon copies to)
	   -C comment-line      (Comments:)
	   -d ...		(debug on)
	   -f from 		(from name)
	   -F from-addr		(the actual address to be put in the From: line)
	   -i msg-id            (In-Reply-To: msgid)
	   -r reply-to-address 	(Reply-To:)
	   -R references        (References:)
	   -P precedence        (Precedence:)
	   -s subject 		(subject of message)
**/

#include "def_utils.h"
#include "addrlib.h"
#include "schedule_time.h"
#include "mailerlib.h"
#include "misclib.h"
#include "s_fastmail.h"

#ifdef I_SYSEXITS
#include <sysexits.h>
#endif


DEBUG_VAR(Debug,__FILE__,"util");

#define  temphome	"/tmp/fastmail."


/* Returns exitcode */
static int usage P_((void));

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

/* Caller must call free_string() */
static struct string * exit_status_text P_((int exit_stat));
static struct string * exit_status_text(exit_stat)
     int exit_stat;
{
    struct string * ret = NULL;

#ifdef I_SYSEXITS

    /* Codes from sysexits.h */

    switch(exit_stat) {

#ifdef  EX_USAGE 
    case EX_USAGE    : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_USAGE,
				    "Usage error"));         break;
#endif
#ifdef  EX_DATAERR
    case EX_DATAERR  : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_DATAERR,
				    "Data format error"));   break;
#endif
#ifdef  EX_NOINPUT  
    case EX_NOINPUT  : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_NOINPUT,
				    "Cannot open input"));   break;
#endif
#ifdef  EX_NOUSER   
    case EX_NOUSER   : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_NOUSER,
				    "Address or user unknown")); break;
#endif
#ifdef  EX_NOHOST   
    case EX_NOHOST   : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_NOHOST,
				    "Host name unknown"));   break;
#endif
#ifdef  EX_UNAVAILABLE 
    case EX_UNAVAILABLE : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_UNAVAILABLE,
				    "Service unavailable")); break;
#endif
#ifdef  EX_SOFTWARE 
    case EX_SOFTWARE : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_SOFTWARE,
				    "Internal software error")); break;
#endif
#ifdef  EX_OSERR    
    case EX_OSERR    : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_OSERR,
				    "System error"));        break;
#endif
#ifdef  EX_OSFILE   
    case EX_OSFILE   : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_OSFILE   ,
				    "OS file missing"));     break;
#endif
#ifdef  EX_CANTCREAT
    case EX_CANTCREAT: 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_CANTCREAT,
				    "Can't create output file")); break;
#endif
#ifdef  EX_IOERR    
    case EX_IOERR    : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_IOERR,
				    "Input/output error"));  break;
#endif
#ifdef  EX_TEMPFAIL 
    case EX_TEMPFAIL : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_TEMPFAIL,
				    "Temporary failure"));   break;
#endif
#ifdef  EX_PROTOCOL 
    case EX_PROTOCOL : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_PROTOCOL,
				    "Protocol error"));      break;
#endif
#ifdef  EX_NOPERM   
    case EX_NOPERM   : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_NOPERM,
				    "Permission denied"));   break;
#endif
#ifdef  EX_CONFIG   
    case EX_CONFIG   : 
	ret = format_string(CATGETS(elm_msg_cat, FastmailSet,
				    FastmailEX_CONFIG,
				    "Configuration error")); break;
#endif
    }    
#endif
    return ret;
}

static void mail_sent P_((union any_fd fd, char * title, struct run_state *rs,
 			  int ret, int exit_stat));

static void mail_sent (fd,title,rs,ret,exit_code)
     union any_fd fd; 
     char * title;
     struct run_state *rs;
     int ret; 
     int exit_code;
{    
    printf("%s\n",title);
    fflush(stdout);

    if (ret < 0)  {
	lib_error(CATGETS(elm_msg_cat, FastmailSet, FastmailFailSignal,
			  "%.30s fail: Signal?"),
		  get_mailer_path(fd.mail_fd));
    } else if (ret > 0) {
	if (rs->save_errno) {
	    lib_error(CATGETS(elm_msg_cat, FastmailSet, FastmailFailErrno,
			      "Failed: %.30s: %.40s"),
		      get_mailer_path(fd.mail_fd),
		      strerror(rs->save_errno));
	} else if (exit_code) {
	    struct string * sys_exit = exit_status_text(exit_code);
	    
	    if (sys_exit) {
		lib_error(
			  CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailMailerReturnedErrorT,
				  "mailer returned error status %d (%S)"), 
		      exit_code,sys_exit);
		free_string(&sys_exit);
	    } else {
		lib_error(
			  CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailMailerReturnedError,
				  "mailer returned error status %d"), 
			  exit_code);
	    }
	} else {
	    lib_error(CATGETS(elm_msg_cat, FastmailSet, FastmailMailSent, 
			      "Mail sent!"));
	}
    } else {
	lib_error(CATGETS(elm_msg_cat, FastmailSet, FastmailLostErrno,
			  "%.30s lost: %.40s"),
		  get_mailer_path(fd.mail_fd),
		  strerror(rs->save_errno));
    }
}

static void mailing_message P_((int background));
static void mailing_message(background)
     int background;
{
    if (!background)
	lib_transient(CATGETS(elm_msg_cat, FastmailSet, FastmailSendingMail,
			      "Sending mail..."));
    else
	lib_error(CATGETS(elm_msg_cat, FastmailSet, 
			  FastmailSendingMailBackground,
			  "Sending mail... in background"));    
}

static void simple_add_addr P_((struct expanded_address *expanded,
				struct mailer_info *mailer_info,
				char * addr,
				charset_t defcharset,
				int demime,
				int *exitstat));
static void simple_add_addr(expanded,mailer_info,addr,defcharset,demime,
			    exitstat)
     struct expanded_address *expanded;
     struct mailer_info *mailer_info;
     char * addr;
     charset_t defcharset;
     int demime;
     int *exitstat;
{
    struct addr_list * TMP = parse_header_address(NULL,addr,demime,defcharset,
						  NULL);

    if (TMP) {
	int len = addr_list_item_count(TMP);

	int idx;

	for (idx = 0; idx < len; idx++) {
	    int group = -1;
	    const struct address * address = 
		addr_list_get_item(TMP,idx,&group);
	    
	    const char          * addr     = 
		address_get_ascii_addr(address);
	    const struct string * fullname = 
		address_get_phrase(address);
	    const struct string * comment  = 
		address_get_comment(address);
	    
	    const struct string * groupname = NULL;

	    int pos;
	    struct string *s;
	    const char * sep = NULL;

	    if (group >= 0) 
		groupname =  addr_list_get_group(TMP,group);
	    
	    if (!addr)
		continue;

	    sep = qstrpbrk_c(addr,"!:@");

	    if (sep) {
		DPRINT(Debug,9,
		       (&Debug, 
			"simple_add_addr: addr %s separator %c\n",
			addr, *sep));
	    }
	    
	    if (sep && '@' == *sep && sep > addr) {
		
		if ('[' == *(sep+1)) {   

		    /* Address literal must be one token */
		    
		    int len = rfc822_toklen(sep+1);
		    
		    if (len < 2 || *(sep + len + 1) ||
			']' != *(sep + len)) {
			
			lib_error(CATGETS(elm_msg_cat, 
					 FastmailSet, 
					  FastmailVerifyBadAddressLiteral,
					  "Invalid literal %s on address %s"),
				  sep+1,addr);
			continue;
		    }

		    DPRINT(Debug,12,(&Debug,
				     "simple_add_addr: %s is literal\n",sep+1));

		} else {

		    const char * whitelisted_name = 
			is_whitelisted_valid_domain(sep+1);
		    
		    if (whitelisted_name) {
			DPRINT(Debug,12,(&Debug,
					 "simple_add_addr: Domain %s is whitelisted on address %s\n",
					 whitelisted_name,
					 addr));
			
		    } else {
			
			const char * reserved_name = 
			    is_special_use_domain(sep+1);
			
			if (reserved_name) {
			    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
					      FastmailVerifyReservedDomain,
					      "Reserved domain %s is unsupported on address %s"),
				      reserved_name,addr);
			    
			    DPRINT(Debug,9,
				   (&Debug, 
				    "simple_add_addr: %s: special use domain %s\n",
				    addr,reserved_name));

#ifdef EX_NOHOST
			    if (exitstat && !*exitstat)
				*exitstat = EX_NOHOST;
#endif
			    continue;
			}
		    }
		}

	    } else if (!sep &&
		mailer_info &&
		query_mailer_info(mailer_info,MI_USE_DOMAIN)) {

		char * addr1              = safe_strdup(addr);
		struct address * address1 = NULL;
		
		const char * maildomain = hostfullname;

		const char * have_mailname = useful_mailname();
		
		if (have_mailname)
		    maildomain = have_mailname;
		else {
		    DPRINT(Debug,10,(&Debug,
				    "simple_add_addr: mailname not set or not useful, using hostfullname=%s\n",
				     maildomain));
		}

		DPRINT(Debug,10,(&Debug,
				 "simple_add_addr: Adding domain %s to address %s\n",
				 maildomain, addr));
				
		addr1    = strmcat(addr1,"@");
		addr1    = strmcat(addr1, maildomain);  
		
		address1 = new_address(addr1,fullname,comment);
		
		addr_list_replace_item(TMP,
				       idx,address1,
				       group);   
		
		free_address(&address1);
		free(addr1); addr1 = NULL;
		
		/* Replace invalidates pointers, so reload new pointers */
		
		address   = addr_list_get_item(TMP,idx,&group);
		addr      = address_get_ascii_addr(address);
		fullname  = address_get_phrase(address);
		comment   = address_get_comment(address);	    
	    }
	    
	    pos = add_expanded_addr_g(expanded,address,groupname);
	    s = make_surface_addr(address);
	    add_textual_addr_(expanded,s,pos,1);
	    free_string(&s);
	}

	free_addr_list(&TMP);
    }
}

static char *mime_encode_names[] = {
    "none", /* Not used. */
    "7bit",
    "8bit",
    "binary",
    "quoted-printable",
    "base64"
};

enum args_option_code {
    ARGSOPT_IPv4 = ipv4_option /* '4' */,
    ARGSOPT_IPv6 = ipv6_option /* '6' */,
    ARGSOPT_bcc         = 'b',
    ARGSOPT_cc          = 'c',
    ARGSOPT_comments    = 'C',
    ARGSOPT_Debug       = 'd',
    ARGSOPT_fromName    = 'f',
    ARGSOPT_fromAddr    = 'F',
    ARGSOPT_ArgsHelp    = 'h',
    ARGSOPT_ReturnPath  = 'H',
    ARGSOPT_InReplyTo   = 'i',
    ARGSOPT_Precedence  = 'P',
    ARGSOPT_ReplyTo     = 'r',
    ARGSOPT_References  = 'R',
    ARGSOPT_Subject     = 's',
    ARGSOPT_Type        = 't',
    ARGSOPT_transaction = 'X',

    ARGSOPT_getopt_error= '?'  /* getopt() returns '?' on error */
};


static void write_mailing_headers P_((struct out_state              * fh,
				      struct mailing_headers        * headers,
				      struct string                 * comments,
				      enum encoding                   message_encoding,
				      int                             is_text,
				      media_type_t                    content_type,
				      char                         ** ctype_params,
				      int                             ctype_param_count,
				      charset_t                       cs));
static void write_mailing_headers(fh,headers,comments,message_encoding,is_text,
				  content_type,ctype_params,ctype_param_count,cs)
     struct out_state              * fh;
     struct mailing_headers        * headers;
     struct string                 * comments;
     enum encoding                   message_encoding;
     int                             is_text;
     media_type_t                    content_type;
     char                         ** ctype_params;
     int                             ctype_param_count;
     charset_t                       cs;
{

    write_common_mailing_headers(fh,headers,message_encoding,
				 !allow_no_hdrencoding,cs,
				 write_header_send);
  
    if (comments)
	write_string_header(fh,"Comments",
			    comments,
			    message_encoding,
			    !allow_no_hdrencoding,
			    cs);

    write_text_header(fh,"Mime-Version","1.0",message_encoding);
 
    if (message_encoding <  sizeof mime_encode_names / sizeof (mime_encode_names[0])
 	&&
	message_encoding > ENCODING_NONE) {
 	state_printf(fh,FRM("Content-Transfer-Encoding: %s"),
		     mime_encode_names[message_encoding]);
	print_EOLN(fh,message_encoding);
    }

    if (content_type) {    
 	state_printf(fh,FRM("Content-Type: %s/%s"),
 		     get_major_type_name(content_type),
 		     get_subtype_name(content_type));
  	
 	if (ctype_params) {
 	    int i;
	    
 	    state_putc(';',fh);
	    
 	    if (strlen(get_major_type_name(content_type)) +
 		strlen(get_subtype_name(content_type)) > 70)
 		print_EOLN(fh,message_encoding);
	     	    
 	    for (i = 0; i < ctype_param_count; i++) {
 		
 		state_putc(' ',fh);
 		state_puts(ctype_params[i],fh);
 		
 		if (i < ctype_param_count-1) {
 		    state_putc(';',fh);
		    print_EOLN(fh,message_encoding);
 		}
 	    }	    
 	}
 	
 	print_EOLN(fh,message_encoding);
    }

}



static struct digest_proc * calculate_message_hash P_((FILE *F,
						       struct mailing_headers * headers,
						       const struct digest_proc_type * hash_type,
						       struct string *comments,
						       enum encoding message_encoding,
						       int is_text,
						       media_type_t content_type,
						       char **      ctype_params,
						       int ctype_param_count));


static struct digest_proc * calculate_message_hash(F,headers,hash_type,comments,
						   message_encoding,
						   is_text,
						   content_type,ctype_params,
						   ctype_param_count)
     FILE                          * F;
     struct mailing_headers        * headers;
     const struct digest_proc_type * hash_type;
     struct string                 * comments;
     enum encoding                   message_encoding;
     int                             is_text;
     media_type_t                    content_type;
     char                         ** ctype_params;
     int                             ctype_param_count;
{
    struct digest_proc * ret = malloc_digest_proc(hash_type);
    size_t res_len;
    
    feed_common_mailing_headers(ret,headers,!allow_no_hdrencoding,
				display_charset            /* What is correct charset? */);



    if (comments) {
	digest_feed_str(ret,"Comments: ");
	digest_feed_string(ret,comments);
	digest_feed_str(ret,"\n");
    }
    
    /* MIME-Version not counted (constant header) */

    if (message_encoding <  sizeof mime_encode_names / sizeof (mime_encode_names[0])
 	&&
	message_encoding > ENCODING_NONE) {
	digest_feed_str(ret,"Content-Transfer-Encoding: ");
	digest_feed_str(ret,mime_encode_names[message_encoding]);
	digest_feed_str(ret,"\n");
    }

    if (content_type) {    
	digest_feed_str(ret,"Content-Type: ");
	digest_feed_str(ret, get_major_type_name(content_type));
	digest_feed_str(ret, get_subtype_name(content_type));

	if (ctype_params) {
	    int i;
	    
	    digest_feed_str(ret,";");

	    for (i = 0; i < ctype_param_count; i++) {
		digest_feed_str(ret," ");
		digest_feed_str(ret,ctype_params[i]);
		
		if (i < ctype_param_count-1) {
		    digest_feed_str(ret,";");
		}
	    }
	}
	digest_feed_str(ret,"\n");
    }

    if (message_encoding <= ENCODING_BINARY &&
	message_encoding >= ENCODING_NONE) {
	
	if (is_text > 0 || message_encoding < ENCODING_BINARY) {

	    char buffer[VERY_LONG_STRING];
	    int len;
		
	    while ((len = mail_gets (buffer, sizeof buffer, F)) > 0) {

		feed_digest_proc(ret,s2us(buffer),len);	    
	    }	    
			    
	} else
	    goto count_binary;
    } else {

	char buffer[VERY_LONG_STRING];
	int len;
	
    count_binary:

	while (1) {
            if ((len = fread(buffer, 1,sizeof(buffer)-1, F)) <= 0)
                break;
	    
	    feed_digest_proc(ret,s2us(buffer),len);
	}
    }

    rewind(F);

    res_len = len_digest_proc(ret);

    if (res_len) {
	unsigned char *res = safe_malloc(res_len);

	res_len = end_digest_proc(ret,res,res_len);

	if (res_len) {
	    size_t x;
	    
	    DPRINT(Debug,14,(&Debug,"calculate_message_hash: length %zu [",
			     res_len));
	    
	    for (x = 0; x < res_len; x++) {
		DPRINT(Debug,14,(&Debug,"%02x",res[x]));
	    }
	    DPRINT(Debug,14,(&Debug,"]\n"));
	}
		
	free(res);
    }
        
    DPRINT(Debug,14,(&Debug,"calculate_message_hash=%p\n",ret));

    return ret;

    
    return ret;
}

static char version_buff[NLEN];

#if ANSI_C
static default_xmailer_cb  default_xmailer;
#endif
static struct string * default_xmailer P_((void));
static struct string * default_xmailer()
{
    if (dt_flag_is_set(&program_snd_ident,snd_ident_version))	
	return format_string(FRM("fastmail [version %s]"), version_buff);
    else
	return format_string(FRM("fastmail"));
}

static const char OPTION_LETTERS[] = "46b:c:C:d:f:F:hi:r:R:P:s:t:X:H";

int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    extern char *optarg;
    extern int optind;

    struct mailing_headers mailing_headers;
    
    char *p;
    int  c;

    struct mail_send_state * M = NULL;
    struct out_state *  mailer = NULL;
    FILE                  * F =NULL;
    char ** addrs3 = NULL;

    int exitcode = 1;
    struct mailer_info *mailer_info = NULL;

    struct string * comments     = NULL;
    struct string * from_string  = NULL;

    const char * from_addr_opt_seen = NULL;
    
    char * filename = NULL;
    char * tempname = NULL;

    char title[STRING];

    struct scan_list * scanlist = NULL;
    char *ext = NULL;
    
    struct mime_types_item * type_item = NULL;
    
    int need_enc = 0;
    enum encoding message_encoding = ENCODING_NONE;
    enum allow_no_encoding_v an = 
	give_dt_enumerate_as_int(&allow_no_encoding);
    
    media_type_t content_type = NULL;
    char **      ctype_params = NULL;
    int          ctype_param_count = 0;
    charset_t    ctype_charset = NULL;
    
    int is_text = 1;  /* -1 structured, 0 binary, 1 text */
    int test_utf8 = 0;

    const int read_flags = 0;

    enum import_mh_mode  import_mode = import_mh_normal;

    int file_errcode = 0;
    
#if DEBUG
    init_debugfile("FASTMAIL");
#endif
    locale_init();

#if DEBUG
    /* So that is is possible to initialize debugging before all other stuff */
    if (argc > 1 && 0 == strncmp("-d",argv[1],2) &&
	argv[1][2]) {
	set_debugging(argv[1]+2);	   
    }
#endif    

    zero_mailing_headers(&mailing_headers);
    
    user_init();
    init_misclib(read_flags);
    init_addrlib(read_flags);
    init_mailerlib(read_flags);

    init_defaults(read_flags);

    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);
        
    while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	enum args_option_code code = c;  /* Get warning abount missing enum
					    values on switch */
	switch (code) {
	case ARGSOPT_IPv4:
	case ARGSOPT_IPv6:
	      if (add_ipv_option(code,ipv4_option_printerr) <=  ipv_opt_unsupported) {
		  exit(1);
	      }	      
	      break;

	case ARGSOPT_Debug: 	
#if DEBUG
	    set_debugging(optarg);	  
#endif
	    /* Error is printed later */
	    break;

	case ARGSOPT_transaction : 
#ifdef REMOTE_MBX	    
	    if (!set_transaction_file(optarg))
		exit(1);
#endif
	    break;

	case ARGSOPT_ReturnPath:
	    /* This needs to be set early */
	    import_mode = import_mh_returnpath;
	    break;

	case ARGSOPT_ArgsHelp:
	case ARGSOPT_getopt_error:  /* getopt() returns '?' on error */
	    /* early check, before mailer is connected */
	    
	    exitcode = usage();
	    goto FAILURE;

	case ARGSOPT_bcc:
	case ARGSOPT_cc:
	case ARGSOPT_comments:
	case ARGSOPT_fromName:
	case ARGSOPT_fromAddr:
	case ARGSOPT_InReplyTo:
	case ARGSOPT_Precedence:
	case ARGSOPT_ReplyTo:
	case ARGSOPT_References:
	case ARGSOPT_Subject:
	case ARGSOPT_Type:
	    break;
	}

    }

    /* early check, before mailer is connected */

    if (optind < argc) {	
	filename = argv[optind];

	if (0 != strcmp(filename, "-")) {
	    int r = access(filename,READ_ACCESS);

	    if (-1 == r) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailCantFind,
				  "Error: can't find file %s: %s\n"), 
			  filename,strerror(err));
#ifdef EX_NOINPUT
		exitcode = EX_NOINPUT; /* Better exit code */
#else		    
		exitcode = 1;
#endif
		goto FAILURE;
	    }
	}	    		
    } else {
	exitcode = usage();
	goto FAILURE;
    }
    
    
    optind = 1;     /* Reset scanning */
    
    read_rc_file(read_flags);

    if (dt_flag_is_set(&program_snd_ident,snd_ident_x_mailer) &&
	!mailing_headers.xmailer) {
	mailing_headers.xmailer = default_xmailer();
    }
    
    mailer_info   = get_mailer_info();

    ctype_charset = display_charset;

    if (!mailer_info) {
#ifdef EX_UNAVAILABLE
	exitcode = EX_UNAVAILABLE;
#else
	exitcode = 1;
#endif	
	goto FAILURE;
    }
    
    /* free_mailing_headers will free this */
    if (!mailing_headers.env_from)
	mailing_headers.env_from = mailer_get_env_from(mailer_info);

    /* Read .mailheaders for Return-Path (for envelope sender actually) 
       and user defined headers
    */
    import_mailheaders(&mailing_headers,
		       import_mode);

    dump_expanded_address(8,"after import -- from ",mailing_headers.from);
    dump_expanded_address(8,"after import -- to   ",mailing_headers.to);
    dump_expanded_address(8,"after import -- cc   ",mailing_headers.cc);
    dump_expanded_address(8,"after import -- bcc  ",mailing_headers.bcc);
    dump_expanded_address(8,"after import -- reply-to ",mailing_headers.reply_to);
    
    /* FIXME: what is correct character set? */
    if ((p = getenv("REPLYTO")) != NULL)
	simple_add_addr(&mailing_headers.reply_to,
			mailer_info,
			p,
			display_charset,
			!allow_no_hdrencoding,
			NULL); 
    
    /* auto bcc line */
    if ((p = getenv("BCC")) != NULL)   
	simple_add_addr(&mailing_headers.bcc,
			mailer_info,
			p,
			display_charset,
			!allow_no_hdrencoding,
			NULL); 
    
    while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	enum args_option_code code = c;  /* Get warning abount missing enum
					    values on switch */
	switch (code) {
	case ARGSOPT_IPv4:
	case ARGSOPT_IPv6:

	    /* handled earlier */
	      
	    break;
	    
	case ARGSOPT_bcc: 
	    
	    simple_add_addr(&mailing_headers.bcc,mailer_info,
			    optarg,system_charset,
			    !allow_no_hdrencoding,
			    NULL); 
	    break;

	case ARGSOPT_cc: 
	    simple_add_addr(&mailing_headers.cc,mailer_info,
			    optarg,system_charset,
			    !allow_no_hdrencoding,
			    NULL); 
	    break;

	case ARGSOPT_comments: 
	    
	    if ( comments ) {
		exitcode = usage();
		goto FAILURE;
	    }
		
	    comments = hdr_to_string(HDR_TEXT,optarg,system_charset,
				     !allow_no_hdrencoding);
	    break;

	case  ARGSOPT_Debug: 
#if DEBUG
	    set_debugging(optarg);	   
#else
	    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
			      FastmailIngoringDebug,
			      "Warning: system created without debugging enabled - request ignored\n"));
#endif
	    break;
	case ARGSOPT_fromName :
	    if (from_string) {
		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailFromNameDuplicate,
				  "Duplicate from name -f %s not allowed. Already given -f %S option."),
			  optarg,from_string);

		exitcode = usage();
		goto FAILURE;		
	    }

	    
	    from_string = hdr_to_string(HDR_TEXT,optarg,system_charset,
					!allow_no_hdrencoding);


	    if (from_addr_opt_seen) {
		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailFromNameAndAddr,
				  "Do not use both from name -f %S and from address -F %s options together."),
			  from_string,from_addr_opt_seen);

		exitcode = usage();
		goto FAILURE;
	    }
			     	
	    break;
	case ARGSOPT_fromAddr:

	    if (! from_addr_opt_seen && mailing_headers.from.addrs) {
		struct string * From_field_string =
		    addr_list_to_string(mailing_headers.from.addrs);

		if (From_field_string) {
		    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				      FastmailFromGivenDiscard,
				      "From address -F %s option given. Discarding From %S"),
			      optarg,From_field_string);
		    
		    free_string(&From_field_string);
		}
		
		free_expanded_address(& (mailing_headers.from));
	    }
	    
	    simple_add_addr(&mailing_headers.from,mailer_info,
			    optarg,system_charset,
			    !allow_no_hdrencoding,NULL);

	    if (from_string) {
		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailFromNameAndAddr,
				  "Do not use both from name -f %S and from address -F %s options together."),
			  from_string,optarg);

		exitcode = usage();
		goto FAILURE;				  
	    }

	    from_addr_opt_seen = optarg;
	    
	    break;
	case ARGSOPT_InReplyTo:
	    if (mailing_headers.in_reply_to) {
		exitcode = usage();
		goto FAILURE;
	    }
		
	    mailing_headers.in_reply_to =
		parse_header_references(NULL,optarg,!allow_no_hdrencoding,
					system_charset,NULL);

	    if (!add_irt_phrase && header_references_have_phrase(mailing_headers.in_reply_to)) {

		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailHdrPhraseIRT,
				  "In-reply-to header should include only message-id"));
	    
	    }

					
	    
	    break;

	case ARGSOPT_ReplyTo:
	    simple_add_addr(&mailing_headers.reply_to,mailer_info,
			    optarg,system_charset,
			    !allow_no_hdrencoding,
			    NULL); 
	    break;
	case ARGSOPT_References:
	    if (mailing_headers.references)
		usage();

	    mailing_headers.references =
		parse_header_references(NULL,optarg,!allow_no_hdrencoding,
					system_charset,NULL);
	    
	    if (header_references_have_phrase(mailing_headers.references)) {

		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailHdrPhraseReferences,
				  "References header should include only one or more message-ids"));

	    }


	    break;
	case ARGSOPT_Precedence:
	    if (mailing_headers.precedence) {
		exitcode = usage();
		goto FAILURE;
	    }
	    
	    if (check_8bit_str(optarg)) {
		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailNo8bitonP,
				  "8-bit data not allowed on -P: %s"),
			  optarg);
		
		goto FAILURE;
	    } 

	    mailing_headers.precedence = safe_strdup(optarg);
	    break;

	case ARGSOPT_Subject: 
	    if (mailing_headers.subject) {
		exitcode = usage();
		goto FAILURE;
	    }
	    
	    mailing_headers.subject = 
		hdr_to_string(HDR_TEXT,optarg,system_charset,
			      !allow_no_hdrencoding);
	    
	    break;

 	case ARGSOPT_Type: {
 	    char * X;
 	    char * walk = NULL;
 	    char * type;
	    
 	    if (check_8bit_str(optarg)) {
 		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
 				  FastmailNo8bitont,
 				  "8-bit data not allowed on -t: %s"),
 			  optarg);
		
 		goto FAILURE;
 	    } 
	    
 	    X = safe_strdup(optarg);
 	    type = mime_parse_content_opts(X,&walk);
	    
 	    if (type) {
 		char * A;
 		char * p;
 		char * subtype;
		
 		rfc822_reap_comments (type, NULL, 0);
		
 		/* Parse type */
		
 		while (*type && whitespace(*type))
 		    type++;
		
 		if (!*type)
 		    goto params_only;
 		
 		if (content_type) {
		    exitcode = usage();
		    goto FAILURE;
		}
		
 		p = strchr(type,'/');
		if (!p) {
		    
 		no_type:
 		    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
 				      FastmailBadTypeont,
 				      "Bad type on on -t: %s"),
 			      optarg);
 		
 		    free(X);
 		    goto FAILURE;
 		}
 
 		subtype = p+1;
 
 		while (p > type && whitespace(*(p-1)))
 		    p--;
 		*p = '\0';
 		
 		if (!*type)
 		    goto no_type;
 
 		while (*subtype && whitespace(*subtype))
 		    subtype++;
 
 		if (!*subtype)
 		    goto no_type;
 
 		p = subtype + strlen(subtype);
 		while (p > subtype && whitespace(*(p-1)))
 		    p--;
 		*p = '\0';
 
 		content_type = give_media_type(type,subtype,1);
 		
 		if (!content_type)
 		    goto no_type;
 
 	    params_only:
 		while(NULL != (A =  mime_parse_content_opts(NULL,&walk))) {
 		    int len = strlen(A);
 
 		    if (0 == strncmp(A,"charset=",8) && len > 8) {
 			char * X = dequote_opt(A+8,len-8);
 
 			ctype_charset = MIME_name_to_charset(X,1);

 			free(X);
 
 		    } else {
 			ctype_params = safe_array_realloc(ctype_params,
							  ( ctype_param_count+1),
							  sizeof( ctype_params[0]));
 			
 			ctype_params[ctype_param_count++] = safe_strdup(A);
 		    }
 		}
 	    }
 
 	    free(X);
 	}
	    break;

	case ARGSOPT_ReturnPath:
	case ARGSOPT_transaction:
	    /* Already processed */
	    break;

	case ARGSOPT_ArgsHelp:
	case ARGSOPT_getopt_error:  /* getopt() returns '?' on error */
	    exitcode = usage();
	    goto FAILURE;
	}
    }	
    

    if (from_string) {
	int n = 0;
	
	if (! mailing_headers.from.addrs ||
	    0 == (n = addr_list_item_count(mailing_headers.from.addrs))) {
	    
	    struct string * comment = new_string(display_charset);
	    struct string * s1 = NULL;
	    
	    char *s = safe_strdup(username); /* Use username as surface addr */
	    char *x;
	    
	    char * Q = from_addr_literal(mailer_info,
					 NULL);
	    
	    int pos = add_expanded_addr_(&mailing_headers.from,
					 Q,from_string,comment);
	    free(Q); Q = NULL;
	    
	    /* No fancy quotation stuff on textual form for editing */
	    for (x = s; *x; x++) {
		switch(*x) {
		case '"': case '\\': case '(': case ')': case ':':
		case '<': case '>':
		    *x = '_';
		    break;
		}
	    }
	    
	    s1 = new_string2(display_charset,s2us(s));
	    
	    add_textual_addr_(&mailing_headers.from,s1,pos,1);
	    free(s); s = NULL;
	    free_string(&s1);
	    
	    free_string(&comment);
	} else if (1 == n) {

	    struct string * From_field_string =
		addr_list_to_string(mailing_headers.from.addrs);
	    struct string * From2_field_string = NULL;
	    
	    int group = -1;
	    
	    const struct address * address = 
		addr_list_get_item(mailing_headers.from.addrs,0,&group);

	    const struct string * old_comment =
		address_get_comment(address);
	    const char * old_ascii_addr = address_get_ascii_addr(address);

	    struct address * new_addr =
		new_address(old_ascii_addr,from_string,old_comment);

	    struct string * message =
		format_string(CATGETS(elm_msg_cat, FastmailSet, 
				      FastmailFromNameOptChange,
				      "From name -f %S changed From"),
			      from_string);
	    
	    addr_list_replace_item(mailing_headers.from.addrs,0,
				   new_addr,group);
	    
	    free_address(&new_addr);

	    From2_field_string = addr_list_to_string(mailing_headers.from.addrs);
	    
	    
	    if (From_field_string) {

		elm_append_message(&message,
				   CATGETS(elm_msg_cat, FastmailSet, 
					   FastmailFromNameSource,
					   " value %S"),
				   From_field_string);
						
		free_string(& From_field_string);
	    }

	    if (From2_field_string) {
		elm_append_message(&message,
				   CATGETS(elm_msg_cat, FastmailSet, 
					   FastmailFromNameResult,
					   " to value %S"),
				   From2_field_string);

		free_string(& From2_field_string);
	    }

	    lib_error(FRM("%S"),message);
	    
	    free_string(& message);
	} else {
	    struct string * From_field_string =
		addr_list_to_string(mailing_headers.from.addrs);

	    if (From_field_string) {
		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailFromNameIgn,
				  "From name -f %S ignored, From is %S"),
			  from_string,From_field_string);

		free_string(& From_field_string);
	    }
	}
    }
        
#ifdef DEBUG
    { 
	int d = panic_dprint("\n\
======================================================\n\
Debug output of the FASTMAIL program (version %s).\n",
			     version_buff);

	if (d >= 50) {
#if 0	
	    panic_dprint("WARNING: Edit manually out sensitive information from that file!\n");
    
	    lower_prompt("WARNING: Debug file may include passwords -- edit it!");
	    sleep(5+sleepmsg);	    
#endif
	}
    }
#endif

    if (optind >= argc) {
	exitcode = usage();
	goto FAILURE;
    }

    if (!mailing_headers.subject)
	mailing_headers.subject = 
	    hdr_to_string(HDR_TEXT,DEFAULT_BATCH_SUBJECT,
			  system_charset,
			  !allow_no_hdrencoding);

    sndhelper_update_user_agent(&mailing_headers,system_charset,!allow_no_hdrencoding,default_xmailer);
    
    filename = argv[optind++];

    if (optind >= argc) {
	exitcode = usage();
	goto FAILURE;
    }

    if (0 == strcmp(filename, "-")) {
	const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);
	if (!tmp)
	    tmp = "/tmp/";

	filename = tempname = elm_message(FRM("%selmfstmail%d"),tmp,getpid());

	F = safeopen_rdwr(filename,&file_errcode);

	if (F) {
	    int ch;

	    if (-1 == unlink(filename)) {
		int err = errno;

		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailUnlinkTmpFileError,
				  "Error to unlink temporary file %s: %s\n"),
			  filename,strerror(err));
	    }

	    while (EOF != (ch = getchar())) {
		
		if (EOF == putc(ch,F)) {
		    int err = errno;

		    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				      FastmailTmpFileError,
				      "Error to write temporary file %s: %s\n"),
			      filename,strerror(err));

#ifdef EX_IOERR
		    exitcode = EX_IOERR; /* Better exit code */
#else		    
		    exitcode = 1;
#endif
		    goto FAILURE;
		}
	    }
	    
	    if (ferror(stdin)) {
		int err = errno;

		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailStdInputError,
				  "Error when reading from standard input: %s\n"),
			  filename,strerror(err));
		
#ifdef EX_IOERR
		exitcode = EX_IOERR; /* Better exit code */
#else		    
		exitcode = 1;
#endif
		goto FAILURE;

	    }

	    if (EOF == fflush(F)) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailTmpFileError,
				  "Error to write temporary file %s: %s\n"),
			  filename,strerror(err));
#ifdef EX_IOERR
		exitcode = EX_IOERR; /* Better exit code */
#else		    	
		exitcode = 1;
#endif
		goto FAILURE;
	    }			

	    rewind(F);
	}
	

    } else {
	F = fopen(filename,"r");

	if (!F)
	    file_errcode = errno;
	
	ext = strrchr (filename, '.');
	if (ext)
	    ext++;

    }

    if (!F) {
	lib_error(CATGETS(elm_msg_cat, FastmailSet, 
			  FastmailCantFind,
			  "Error: can't find file %s: %s\n"), 
		  filename,strerror(file_errcode));
#ifdef EX_NOINPUT
	exitcode = EX_NOINPUT; /* Better exit code */
#else		    
	exitcode = 1;
#endif
	goto FAILURE;
    }

    if (user_mimetypes_map)
	scanlist = get_scan_list(user_mimetypes_map,ext);

    if (system_mimetypes_map) {
	struct scan_list * X = get_scan_list(system_mimetypes_map,ext);
	
	if (X) {
	    if (scanlist) {
		append_scanlist(scanlist,X);
		free_scan_list(&X);
	    } else
		scanlist = X;
	}
    }
    
    
    { struct scan_list * Y = get_scan_list(builtin_mimetypes_map,ext);
    
	if (Y) {
	    if (scanlist) {
		append_scanlist(scanlist,Y);
		free_scan_list(&Y);
	    } else
		scanlist = Y;
	}
    }

    test_utf8 = is_utf8_charset(ctype_charset);

    need_enc = needs_encoding(F,scanlist,test_utf8);

    if (0 != (need_enc & HAVE_BINARY)) {

	if (query_mailer_info(mailer_info,MI_HAVE_BINARYMIME)) {
	    
	    DPRINT(Debug,4,(&Debug,"HAVE_BINARY: Mailer supports BINARYMIME\n"));
	    message_encoding = ENCODING_BINARY;
	} else if (an >= allow_pass_binary) {
	    DPRINT(Debug,4,(&Debug,"HAVE_BINARY: Just send binary anyway\n"));
	    message_encoding = ENCODING_BINARY;
	} else {
	    DPRINT(Debug,4,(&Debug,"HAVE_BINARY: Use Base64\n"));
	    message_encoding = ENCODING_BASE64;
	}

    } else if (0 != (need_enc & HAVE_8BIT)) {

	if (query_mailer_info(mailer_info,MI_HAVE_8BITMIME)) {
	    DPRINT(Debug,4,(&Debug,"HAVE_8BIT: Mailer supports 8BITMIME\n"));
	    message_encoding = ENCODING_8BIT;
	} else if (an >= allow_pass_8bit) {
	    DPRINT(Debug,4,(&Debug,"HAVE_8BIT: Just send 8bit anyway\n"));
	    message_encoding = ENCODING_8BIT;
	} else {
	    DPRINT(Debug,4,(&Debug,"HAVE_8BIT: Use quoted-printable\n"));
	    message_encoding = ENCODING_QUOTED;
	}
	    
    }


    type_item = loc_mime_type_from_scan(scanlist);

    if (type_item) {
	const char * params = mime_type_to_params(type_item,NULL);

	content_type = mime_type_to_media_type(type_item);

	DPRINT(Debug,4,(&Debug,"Found type %s/%s\n",
			get_major_type_name(content_type),
			get_subtype_name(content_type)));

	if (params) {
	    char * X = safe_strdup(params);
	    char * walk = NULL;
	    char *A;
	    
	    for (A = mime_parse_content_opts(X,&walk); A;
		 A = mime_parse_content_opts(NULL,&walk)) {

		if (! check_8bit_str(A)) {
		    ctype_params = safe_array_realloc(ctype_params,
						      ( ctype_param_count+1),
						      sizeof( ctype_params[0]));

		    ctype_params[ctype_param_count++] = safe_strdup(A);
		} else {
		    DPRINT(Debug,4,(&Debug,"Ignoring parameter %s\n",A));
		}
	    }
		     
	    free(X);
	}
    }

    if (! content_type &&
	0 != (need_enc & HAVE_BINARY) &&
	0 != (need_enc & HAVE_CTRL)) {

	/* If HAVE_BINARY is set, but HAVE_CTRL is not set, file 
	   have just long lines. So it still can be text.
	*/

	content_type = give_media_type2(MIME_TYPE_APPLICATION,
					"octet-stream", 1);

	if (content_type) {
	    DPRINT(Debug,4,(&Debug,"Binary data - setting type %s/%s\n",
			    get_major_type_name(content_type),
			    get_subtype_name(content_type)));
	}

    }

    /* Add charset paramater ? */
    if ((need_enc || ! charset_ok_p(ctype_charset)) 
	&&
	( ! content_type  ||
	  get_major_type_code(content_type) == MIME_TYPE_TEXT)) {

	if (! content_type) {
	    content_type = give_media_type2(MIME_TYPE_TEXT,"plain",0);

	    if (content_type) {
		DPRINT(Debug,4,(&Debug,"Setting type %s/%s\n",
				get_major_type_name(content_type),
				get_subtype_name(content_type)));
	    }
	}

	{
	    const char *ctype_MIME_name = get_charset_MIME_name(ctype_charset);

	    if (ctype_MIME_name) {
		if (istrcmp(ctype_MIME_name,"US-ASCII") == 0 &&
		    (need_enc & HAVE_8BIT)) {
		    
		    lib_error(CATGETS(elm_msg_cat, FastmailSet, FastmailUsingUNKNOWN8BIT,
				      "Text has 8BIT data and charset=US-ASCII, using charset=UNKNOWN-8BIT instead."));
		    ctype_params = safe_array_realloc(ctype_params,
						      ( ctype_param_count+1),
						      sizeof( ctype_params[0]));
		    
		    ctype_params[ctype_param_count++] = safe_strdup("charset=UNKNOWN-8BIT");
		    
		} else if (test_utf8 && (need_enc & HAVE_8BIT) &&
			   ! (need_enc & HAVE_UTF8)) {
		    
		    lib_error(CATGETS(elm_msg_cat, FastmailSet, FastmailNotUTF8,
				      "Text is not UTF-8 text, using charset=UNKNOWN-8BIT instead."));
		    ctype_params = safe_array_realloc(ctype_params,
						      ( ctype_param_count+1),
						      sizeof( ctype_params[0]));
		    
		    ctype_params[ctype_param_count++] = safe_strdup("charset=UNKNOWN-8BIT");
		    
		    
		} else {
		    ctype_params = safe_array_realloc(ctype_params,
						      ( ctype_param_count+1),
						      sizeof( ctype_params[0]));
		    
		    ctype_params[ctype_param_count++] = 
			elm_message(FRM("charset=%Q"),
				    ctype_MIME_name);
		    
		    DPRINT(Debug,4,(&Debug,"Setting charset %s\n",
				    ctype_MIME_name));
		
		}	
	    }
	}
    }


    if (content_type) {
	is_text = give_text_type_code(content_type);

	/* Content-Transfer-Encoding 7bit or 8bit implies text type */

	if (0 == is_text && message_encoding < ENCODING_BINARY) {

	    /* DO not use base64 for structured data (ie is_text < 0) */

	    if (query_mailer_info(mailer_info,MI_HAVE_BINARYMIME)) {
		
		DPRINT(Debug,4,(&Debug,
				"Non-text content type: Mailer supports BINARYMIME\n"));
		message_encoding = ENCODING_BINARY;
	    } else if (an >= 2) {
		DPRINT(Debug,4,(&Debug,"Non-text content type: Just send binary anyway\n"));
		message_encoding = ENCODING_BINARY;
	    } else if (need_enc) {
		DPRINT(Debug,4,(&Debug,"Non-text content type: Use Base64\n"));
		message_encoding = ENCODING_BASE64;
	    } else {
		DPRINT(Debug,4,(&Debug,"Non-text content type: Use quoted-printable\n"));
		message_encoding = ENCODING_QUOTED;
	    }
	}
    }

    if (optind < argc) {
	int Exitcode = 0;
	
	while (optind < argc) {
	    simple_add_addr(&mailing_headers.to,mailer_info,
			    argv[optind++],system_charset,
			    !allow_no_hdrencoding,&Exitcode); 
	}

	/* Require To: */
	if (! mailing_headers.to.addrs ||
	    0 == addr_list_item_count(mailing_headers.to.addrs)) {

	    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
			      FastmailNoToAddressAdded,
			      "No to address added."));

	    if (Exitcode)
		exitcode = Exitcode;
	    else
		exitcode = 1;
	    goto FAILURE;
	}
    }
	
    if (mailing_headers.sender) {   /* Should not happen? */
	DPRINT(Debug,6,(&Debug,
			"Got Sender address: %s\n",
			mailing_headers.sender));

	if (! mailing_headers.from.addrs) {
	    DPRINT(Debug,8,(&Debug," ... resetting Sender: %s\n",
			    mailing_headers.sender));

	    free(mailing_headers.sender);
	    mailing_headers.sender = NULL;
	    goto no_sender;
	}
	
    } else {

    	enum add_sender_v add_sender_h UNUSED_VAROK;
	char * sender_addr             UNUSED_VAROK;
	char * sender_domain           UNUSED_VAROK;
	int    sender_domain_verify    UNUSED_VAROK;
	int from_addr_len              UNUSED_VAROK;
	int sender_ok                   UNUSED_VAROK;
	enum sndhelper_status helperstatus;
	
    no_sender:
	/* No mailing_headers.sender set yet */
	
	add_sender_h =
	    give_dt_enumerate_as_int(&add_sender_header);
	sender_addr    = NULL;
	sender_domain  = NULL;
	sender_domain_verify   = 0;
	from_addr_len      = 0;
	sender_ok = 1;

	if (mailing_headers.from.addrs &&
	    (from_addr_len = addr_list_item_count(mailing_headers.from.addrs)) > 0) {
	    
	    DPRINT(Debug,6,(&Debug,
			    "From address already specified by user (len %d)\n",
			    from_addr_len));
	    
	    sndhelper_get_sender(add_sender_h,mailer_info,&sender_domain,
				 &sender_domain_verify,&sender_addr,from_addr_len);

	} else if (query_mailer_info(mailer_info,MI_DONT_ADD_FROM)) {
	    
	    DPRINT(Debug,12,(&Debug,
			     "fastmail: From address not needed: MI_DONT_ADD_FROM\n"));
	} else {
	    char * from_domain = NULL;
	    char * copy_addr   = NULL;

	    enum make_from_status
		from_ok = make_from_addr(&(mailing_headers.from),mailer_info,&from_domain,
					 &copy_addr);
	    
	    if (mailing_headers.from.addrs)
		from_addr_len = addr_list_item_count(mailing_headers.from.addrs);
	    else {
		DPRINT(Debug,12,(&Debug,"fastmail: No From addresses\n"));
	    }
		    

	    switch (from_ok) {

	    case make_from_fail:

		DPRINT(Debug,4,(&Debug,
				"fastmail: Bad from address\n"));

		if (copy_addr) {
		    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				      FastmailFromAddrFail,
				      "From address %s fails."),
			      copy_addr);
		}
		
		exitcode = 1;
		goto FAILURE;

	    case make_from_normal:           /* normal domain      */
	    case make_from_localaddr:        /* no domain          */
	    case make_from_literal:          /* [literal] domain   */
	    case make_from_whitelisted:     /* whitelisted domain */

		DPRINT(Debug,6,(&Debug,
				"From address set"));
		if (copy_addr) {
		    DPRINT(Debug,6,(&Debug,": %s",copy_addr));
		}
		DPRINT(Debug,6,(&Debug,"\n"));
		
		break;
	    }
	    
	    if (from_domain) {
		free(from_domain);
		from_domain = NULL;
	    }

	    
	    if (copy_addr) {
		free(copy_addr);
		copy_addr = NULL;
	    }

	}
	
	helperstatus =
	    sndhelper_handle_sender(add_sender_h,mailer_info,&sender_domain,
				    &sender_domain_verify,&sender_addr,
				    from_addr_len,&sender_ok,&mailing_headers);
	
	switch (helperstatus) {
	case sndhelper_domain_failure:
	    if (!sender_ok && sender_addr && sender_domain) {
		
		lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				  FastmailSenderFailDomain,
				  "Domain %s for Sender address %s not valid."),
			  sender_domain,sender_domain);
		
	    }
	    break;
	case sndhelper_none:
	    break;
	}

	if (sender_domain) {
	    free(sender_domain);
	    sender_domain = NULL;
	}

	if (sender_addr) {
	    free(sender_addr);
	    sender_addr = NULL;
	}

	if (!sender_ok) {

	    /* If Sender: -header is only omitted,
	       when domain is on special-use-domains-blacklist
	       elmrc option, user can prevent addition
	       of Sender: header with just editing
	       on special-use-domains-blacklist on
	       ~/.elm/elmrc
	    */

	    DPRINT(Debug,4,(&Debug,
			    "fastmail: Bad sender\n"));
	    exitcode = 1;
	    goto FAILURE;
	}
    }
    
    if (!mailing_headers.message_id) {
	enum message_id_status          msgidstat  = msid_stat_none;
	struct digest_proc            * new_hash   = NULL;
	const struct digest_proc_type * need_hash  = NULL;
	
	/* Need generate message-id */
	
	DPRINT(Debug,4,(&Debug,
			"fastmail: No message id, checking\n"));
	
	msgidstat = update_message_id(& (mailing_headers.message_id),
				      mailing_headers.message_digest,
				      NULL,mailer_info,
				      &need_hash,NULL);
	
	switch (msgidstat) {
	case msid_stat_need_hash:
	    DPRINT(Debug,4,(&Debug,
			    "fastmail: Message id needs hash, calculating\n"));

	    new_hash = calculate_message_hash(F,&mailing_headers,need_hash,
					      comments,message_encoding,
					      is_text,content_type,
					      ctype_params,ctype_param_count);

	    if (new_hash) {
		DPRINT(Debug,4,(&Debug,
				"fastmail: new hash, recalculating message id\n"));
		
		msgidstat = update_message_id(& (mailing_headers.message_id),
					      mailing_headers.message_digest,
					      new_hash,mailer_info,
					      &need_hash,NULL);
	    }
		
	    break;
	case msid_stat_need_restart:
	case msid_stat_none:
	case msid_stat_updated:
	case msid_stat_no_changes:
	    break;
	}

	switch (msgidstat) {
	case msid_stat_need_hash:  
	    DPRINT(Debug,4,(&Debug,
			    "fastmail: message id needs still new hash?\n"));
	    break;
	case msid_stat_none:
	    if (! mailing_headers.message_id) {
		DPRINT(Debug,4,(&Debug,
				"fastmail: message id not generated\n"));
	    }
	    break;
	case msid_stat_updated:
	    if (mailing_headers.message_id) {
		if (mailing_headers.message_digest)
		    free_digest_proc(& mailing_headers.message_digest);
		
		DPRINT(Debug,4,(&Debug,
				"fastmail: message id updated"));
		
		if (new_hash) {
		    mailing_headers.message_digest = new_hash;
		    new_hash = NULL;
		    DPRINT(Debug,4,(&Debug,", message hash changed\n"));
		}
		DPRINT(Debug,14,(&Debug,"\n"));
	    }
	    break;

	case msid_stat_no_changes:
	    if (mailing_headers.message_id) {  
		DPRINT(Debug,4,(&Debug,
				 "fastmail: message id preserved\n"));
	    }
	    break;
	case msid_stat_need_restart: 

	    DPRINT(Debug,4,(&Debug,
			     "fastmail: mailer need restart\n"));

	    if (new_hash)
		free_digest_proc(& new_hash);
	    exitcode = 1;
	    goto FAILURE;
	}

	if (new_hash)
	    free_digest_proc(& new_hash);
    }

    dump_expanded_address(8,"before argv_from_headers -- from ",mailing_headers.from);
    dump_expanded_address(8,"before argv_from_headers  -- to   ",mailing_headers.to);
    dump_expanded_address(8,"before argv_from_headers  -- cc   ",mailing_headers.cc);
    dump_expanded_address(8,"before argv_from_headers  -- bcc  ",mailing_headers.bcc);
    dump_expanded_address(8,"before argv_from_headers  -- reply-to ",mailing_headers.reply_to);
    
    addrs3 = argv_from_headers(&mailing_headers);

    if (! addrs3) {
	DPRINT(Debug,4,(&Debug,"fastmail: No mail addresses\n"));
	exitcode = 1;
	goto FAILURE;
    }

    M = mailer_init(addrs3,0,0,mailer_info,
		    mailing_headers.env_from );

    free_argv(&addrs3);

    if (!M) {

	exitcode = 1;
	goto FAILURE;
    }
 
    mailer = get_mail_outfd(M);

    if (ENCODING_BINARY == message_encoding)
	set_out_state_EOLN(mailer,1);

    write_mailing_headers(mailer,
			  &mailing_headers,
			  comments,message_encoding,
			  is_text,content_type,
			  ctype_params,ctype_param_count,
			  display_charset            /* What is correct charset? */
			  );
    


    


    
    if (mailing_headers.to.addrs) {
	int cnt;

	if (1 == (cnt = addr_list_item_count(mailing_headers.to.addrs))) {

	    int group;
	    const struct address * X = 
		addr_list_get_item(mailing_headers.to.addrs,0,&group);
	    const char * addr = address_get_ascii_addr(X);

	    elm_sfprintf(title, sizeof title,
			 CATGETS(elm_msg_cat, FastmailSet, FastmailMailTo,
				 "Mail to %.50s..."),
			 addr ? addr : "(someone)");
	} else
	    elm_sfprintf(title, sizeof title,
			 CATGETS(elm_msg_cat, FastmailSet, FastmailMailToXRecipients,
				 "Mail to %d recipients (To:)"),
			 cnt);

    } else
	elm_sfprintf(title, sizeof title,
		     CATGETS(elm_msg_cat, FastmailSet, FastmailMailToSomeone,
			     "Mail to someone..."));
	                			       	
    /* Empty line separates bydy and headers */
    print_EOLN(mailer,message_encoding);
    
    {
 	int r UNUSED_VAROK;
		
 	if (message_encoding <= ENCODING_BINARY &&
 	    message_encoding >= ENCODING_NONE) {
  
	    if (is_text > 0 || message_encoding < ENCODING_BINARY) {
		
		char buffer[VERY_LONG_STRING];
		int len;
		
		while ((len = mail_gets (buffer, sizeof buffer, F)) > 0) {
		    
		    state_convert_EOLN(buffer,&len,sizeof buffer,mailer);
		    
		    state_put(buffer, len, mailer);
		}
	    } else {
		int ch;
		
		if (is_text < 0) {
		    
		    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
				      FastmailBinaryEncodingNotSupported,
				      "%s: Binary encoding is not supported for structured types."),
			      filename);
		    
		}
		
		while (EOF != (ch = getc(F))) {
		    state_putc(ch,mailer);
		    
		}
	    }
	    
	} else if (ENCODING_QUOTED == message_encoding) {
 	    quoted_printable_encode(F,mailer,is_text, message_encoding,0);
 	} else if (ENCODING_BASE64 == message_encoding) {
 	    base64_encode(F,mailer,is_text, message_encoding,0);
 	} else {
 	    DPRINT(Debug,4,(&Debug,"bad encoding %d\n",message_encoding));
 	    goto FAILURE;
 	}
	
 	if(ferror(F)) {
 	    int err = errno;
	    
 	    lib_error(CATGETS(elm_msg_cat, FastmailSet, 
 			      FastmailFileReadError,
 			      "Error when reading from file %s: %s\n"), 
 		      filename,strerror(err));
	    
#ifdef EX_IOERR
	    exitcode = EX_IOERR; /* Better exit code */
#else		    
 	    exitcode = 1;
#endif
 	    goto FAILURE;
 	}
	    	

 	exitcode = 0;

	r = mail_backend3(&M,mail_sent,
			  0,title,mailing_message,&exitcode,
			  NULL,out_state_ftell(mailer));
  
 	DPRINT(Debug,4,(&Debug,"mail_backend2 returned %d\n",r));

 	/* 0   may be:    start failed
 	 *                mailing contined on backend
 	 *
  	 * 1              command completed, perhaps with failure 
 	 */
	
    } 

 FAILURE:
    if (mailer_info)
	free_mailer_info(&mailer_info);

    free_mailing_headers(&mailing_headers);

    if (F)
 	fclose(F);
    
    if (tempname)
 	free(tempname);
    
    if (scanlist)
 	free_scan_list(&scanlist);
    
    if (ctype_params) {
 	int i;
	
 	for (i = 0; i < ctype_param_count; i++) 
 	    free(ctype_params[i]);
 	free(ctype_params);
    }
    
    DPRINT(Debug,4,(&Debug,"exitcode %d\n",exitcode));
    
    return exitcode;
}

static int usage()
{
    int x;
 
    lib_error(CATGETS(elm_msg_cat, FastmailSet, FastmailUsage1,
		      "Usage: fastmail {args} [ filename | - ] address(es)\n\
   where {args} can be:\n"));

    for (x = 0; OPTION_LETTERS[x]; x++) {

        int have_arg = ':' == OPTION_LETTERS[x+1];

	enum args_option_code code = OPTION_LETTERS[x];

	if (isascii(code) && isprint (code)) {
            switch (code) {
	    case ARGSOPT_IPv4:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOptIPv4,
				  "\t -%c                 \t\tUse IPv4 connection (for mailer=submission)\n"),
			  code);
		break;
	    case ARGSOPT_IPv6:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOptIPv6,
				  "\t -%c                 \t\tUse IPv6 connection (for mailer=submission)\n"),
			  code);
		break;
	    case ARGSOPT_bcc:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOptbcc,
				  "\t -%c bcc-list        \t\tSet blind-carbon copies addresses.\n"),
			  code);
		break;
	    case ARGSOPT_cc: 
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOptcc,
				  "\t -%c cc-list          \t\tSet carbon copies addresses.\n"),
			  code);
		break;
	    case ARGSOPT_comments:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOpComments,
				  "\t -%c comments       \t\tSet Comments: header field.\n"),
			  code);
		break;

	    case ARGSOPT_Debug:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOpDebug,
				  "\t -%cclass=debugfile:n\t\tDebug - set debug level to 'n' for 'class'\n"),
			  code);
		break;
	    case ARGSOPT_fromName:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOpFromName,
				  "\t -%c from-name       \t\tThis overrides the users name in the From: header field.\n"),
			  code);
		break;
	    case ARGSOPT_fromAddr:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOpFromAddr,
				  "\t -%c from-addr       \t\tThis specifies From: header field or adds new address to From: header field.\n"),
			  code);
		break;
	    case ARGSOPT_transaction:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOpTransaction,
				  "\t -%cx                \t\tLog remote protocol transaction to file 'x'.\n\
\t                     \t\tWarning: A transaction (transcript) file will include passwords!\n"),
			  code);
		break;
	    case ARGSOPT_ArgsHelp:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOpArgsHelp,
				  "\t -%c                \t\tHelp - give this list of options\n"),
			  code);
		break;
	    case ARGSOPT_ReturnPath:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOpReturnPath,
				  "\t -%c                \t\tSet only Return-Path from %s.\n"),
			  code,MAILHEADERS);
		break;
	    case ARGSOPT_InReplyTo:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsOpInReplyTo,
				  "\t -%c msg-id         \t\tSet message-id to In-Reply-To: header filed.\n"),
			  code);
		break;
	    case ARGSOPT_ReplyTo:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsReplyTo,
				  "\t -%c reply-to       \t\tThis specifies Reply-To: header field or adds new address to Reply-To: header field.\n"),
			  code);
		break;
	    case ARGSOPT_References:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsReferences,
				  "\t -%c references      \t\tSet References: header field.\n"),
			  code);
		break;
	    case ARGSOPT_Precedence:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsPrecedence,
				  "\t -%c precedence     \t\tSet Precedence: header field.\n"),
			  code);
		break;
	    case ARGSOPT_Subject:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsSubject,
				  "\t -%c subject        \t\tSet Subject: header field.\n"),
			  code);
		break;
	    case ARGSOPT_Type:
		lib_error(CATGETS(elm_msg_cat, FastmailSet,
				  FastmailArgsType,
				  "\t -%c type/subtype   \t\tSet MIME content-type of message.\n\
\t -%c ;charset=character-set\tSet MIME character-set of message.\n"),
			  code,code);
		break;
	    case ARGSOPT_getopt_error:
		break;
	    }
	}

	if (have_arg) {
	    /* Skip argument */	    
	    x++;
	}
    }

#ifdef EX_USAGE
    return EX_USAGE;
#else
    return 1;
#endif
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
