static char rcsid[] = "@(#)$Id: newmail.c,v 2.23 2022/11/02 16:26:29 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.23 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                       (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 *  Based on Elm 2.4 utils/newmail.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 * 			Copyright (c) 1988-1992 USENET Community Trust
 * 			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This is actually two programs folded into one - 'newmail()' and
    'wnewmail()'.  They perform essentially the same function, to
    monitor the mail arriving in a set of/a mailbox or folder, but
    newmail is designed to run in background on a terminal, and
    wnewmail is designed to have a window of its own to run in.

    The main difference is that wnewmail checks for mail more often.

    The usage parameters are:

	-i <interval>  		how often to check for mail
				(default: 60 secs if newmail,
					  10 secs if wnewmail)

	<filename>		name of a folder to monitor
				(can prefix with '+'/'=', or can
			 	default to the incoming mailbox)

	<filename>=prefix	file to monitor, output with specified
				prefix when mail arrives.

    If we're monitoring more than one mailbox the program will prefix
    each line output (if 'newmail') or each cluster of mail (if 'wnewmail')
    with the basename of the folder the mail has arrived in.  In the 
    interest of exhaustive functionality, you can also use the "=prefix"
    suffix (eh?) to specify your own strings to prefix messages with.

    The output format is either:

	  newmail:
	     >> New mail from <user> - <subject>
	     >> Priority mail from <user> - <subject>

	     >> <folder>: from <user> - <subject>
	     >> <folder>: Priority from <user> - <subject>

	  wnewmail:
	     <user> - <subject>
	     Priority: <user> - <subject>

	     <folder>: <user> - <subject>
	     <folder>: Priority: <user> - <subject>\fR

**/

#include "def_utils.h"
#include "addrlib.h"
#include "mboxlib.h"
#include "s_newmail.h"

DEBUG_VAR(Debug,__FILE__,"util");

#ifdef PWDINSYS
#  include <sys/pwd.h>
#else
#  include <pwd.h>
#endif


/**********
   Since a number of machines don't seem to bother to define the utimbuf
   structure for some *very* obscure reason.... 

   Suprise, though, BSD has a different utime() entirely...*sigh*
**********/

#define LINEFEED		(char) 10
#define BEGINNING		0			/* seek fseek(3S) */
#define DEFAULT_INTERVAL	60

#define MAX_FOLDERS		25		/* max we can keep track of */



/*
 * The "read_headers()" and "show_header()" routines use a (struct header_rec)
 * to hold the header information.  This structure does not have a flag to
 * save the message priority status.  We don't need the "encrypted" flag in
 * the structure, so we will use that instead.
 */
#define priority encrypted

#define metachar(c)	(c == '+' || c == '=' || c == '%')

struct folder_struct {
    struct folder_info *F;
    char	prefix[NLEN];	
    int		access_error;
} folder_list[MAX_FOLDERS];

static int read_headers P_((register struct folder_struct *cur_folder,
			    struct read_folder_state * read_state_ptr));

static void add_folder P_((char *name));
static void add_default_folder P_((void));
static void show_header P_((struct header_rec *hdr,
			    struct folder_struct *cur_folder));
static void usage P_((char *name));
static void pad_prefixes P_((void));

static int TreatAsSpool;
static int  interval_time,		/* how long to sleep between checks */
     in_window = 0,		/* are we running as 'wnewmail'?    */
     total_folders = 0,		/* # of folders we're monitoring    */
     print_prefix = 0,		/* force printing of prefix	    */
     current_folder = 0;	/* struct pointer for looping       */

#ifdef PIDCHECK
int  parent_pid;		/* See if sucide should be attempt  */
#endif /* PIDCHECK */

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif       

struct stat utime_stat_buffer;
int have_stat = 0;

static char	*priority_to,	/* pointer to Priority to text	*/
		*priority_text,	/* pointer to priority text	*/
		*To_text,	/* pointer To to text		*/
		*priority_mail,	/* pointer to priority mail	*/
		*mail_text,	/* pointer to mail text		*/
		*to_text,	/* pointer to to text		*/
		*from_text;	/* pointer to from text		*/

static void expand_filename P_((char *, char *, int)); /* Prototype */

#ifndef	ANSI_C
extern struct passwd *getpwuid();
#endif

static void save_acc P_((char *name));

static char version_buff[NLEN];

const char OPTION_LETTERS[] = "46d:i:Mr:wX:";

enum newmail_option_code {
    NMAILOPT_IPv4        = ipv4_option /* '4' */,
    NMAILOPT_IPv6        = ipv6_option /* '6' */,
    NMAILOPT_Debug       = 'd',
    NMAILOPT_Interval    = 'i',
    NMAILOPT_AsSpool     = 'M',
    NMAILOPT_RCFile      = 'r',
    NMAILOPT_Window      = 'w',
    NMAILOPT_transaction = 'X',

    NMAILOPT_getopt_error = '?'  /* getopt() returns '?' on error */
};


int main P_((int argc, char *argv[]));
int main(argc, argv)
     int argc;
     char *argv[];
{
    int ret = 0;

	extern char *optarg;
	extern int   optind;
	char *ptr;
	int c, i;
	struct folder_struct *cur_folder;
	const int read_flags = 0;

	
#if DEBUG
	init_debugfile("NEWMAIL");
#endif

	while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {
	    switch(c) {
	    case 'd':
#if DEBUG
		set_debugging(optarg);
#endif
		/* Error is printed later */
		break;
	    }
	}
	optind = 1;     /* Reset scanning */

	locale_init();

	user_init();
	
	init_addrlib(read_flags);	
	init_mboxlib(read_flags);
	init_defaults(read_flags);

	/* Get the No subject string */

	priority_to = catgets(elm_msg_cat, NewmailSet,
	   NewmailInWinPriorityTo, "Priority to ");
	priority_text = catgets(elm_msg_cat, NewmailSet,
	      NewmailInWinPriority, "Priority ");
	To_text = catgets(elm_msg_cat, NewmailSet, NewmailInWinTo, "To ");
	priority_mail = catgets(elm_msg_cat, NewmailSet,
	   NewmailPriorityMail, "Priority mail ");
	mail_text = catgets(elm_msg_cat, NewmailSet, NewmailMail, "Mail ");
	to_text = catgets(elm_msg_cat, NewmailSet, NewmailTo, "to ");
	from_text = catgets(elm_msg_cat, NewmailSet, NewmailFrom, "from ");

#ifdef PIDCHECK				/* This will get the pid that         */
	parent_pid = getppid();		/* started the program, ie: /bin/sh   */
					/* If it dies for some reason (logout)*/
#endif /* PIDCHECK */			/* Then exit the program if PIDCHECK  */

	interval_time = DEFAULT_INTERVAL;

	/** let's see if the first character of the basename of the
	    command invoked is a 'w' (e.g. have we been called as
	    'wnewmail' rather than just 'newmail'?)
	**/

	for (i=0, ptr=(argv[0] + strlen(argv[0])-1); !i && ptr > argv[0]; ptr--)
	  if (*ptr == '/') {
	    in_window = (*(ptr+1) == 'w');
	    i++;
	  }

	if (ptr == argv[0] && i == 0 && argv[0][0] == 'w')
	  in_window = 1;

	while ((c = ELM_GETOPT(argc, argv, OPTION_LETTERS)) != EOF) {

	    enum newmail_option_code code = c;
	    
	  switch (code) {
	  case NMAILOPT_IPv4:
	  case NMAILOPT_IPv6:
	      if (add_ipv_option(c,ipv4_option_printerr) <=  ipv_opt_unsupported) {
		  exit(1);
	      }
	      
	      break;
	      
       	  case NMAILOPT_Debug: 
#if DEBUG
	      set_debugging(optarg);	   
#else
	      lib_error(CATGETS(elm_msg_cat, NewmailSet, 
				NewmailArgsIngoringDebug,
				"Warning: system created without debugging enabled - request ignored\n"));
#endif
	      break;
	  case NMAILOPT_Interval: interval_time = atoi(optarg);	break;
	  case NMAILOPT_AsSpool : TreatAsSpool++;               break;
	  case NMAILOPT_RCFile  : set_user_rc_location(optarg); break;
	  case NMAILOPT_Window  : in_window = 1;		break;
	  case NMAILOPT_transaction : 
#ifdef REMOTE_MBX	    
	      if (!set_transaction_file(optarg))
		  exit(1);
#endif
	      break;

	  case NMAILOPT_getopt_error:  /* getopt() returns '?' on error */
	      usage(argv[0]);				exit(1);
	      break;
	  }
	}

	read_rc_file(read_flags);

    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);

#ifdef DEBUG	  	 
	{ 
	    int d = panic_dprint("\n\
======================================================\n\
Debug output of the NEWMAIL program (version %s).\n",
			     version_buff);
	    
	    if (d >= 50) {
		printf("WARNING: Debug file may include passwords -- edit it!\n");
		panic_dprint("WARNING: Edit manually out sensitive information from that file!");
	    }
	}
#endif
	
	if (interval_time < 10) {
	    if (interval_time == 1)
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailShort,
"Warning: interval set to 1 second.  I hope you know what you're doing!\n"));
	    else
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailShortPlur,
"Warning: interval set to %d seconds.  I hope you know what you're doing!\n"),
			  interval_time);
	}

	/* now let's parse the foldernames, if any are given */

	if (optind >= argc) /* get default */
	    add_default_folder();
	else {
	  while (optind < argc)
	      add_folder(argv[optind++]);
	  pad_prefixes();			/* for nice output...*/
	}
	if (total_folders > 1)
		print_prefix = 1;

#ifdef AUTO_BACKGROUND
	if (! in_window) {
	    if (fork())	    /* automatically puts this task in background! */
		exit(0);

	    (void) signal(SIGINT, SIG_IGN);
	    (void) signal(SIGQUIT, SIG_IGN);
	}
#endif
	(void) signal(SIGHUP, SIG_DFL);

	if (in_window)
	     elm_fprintf(stdout,
			 CATGETS(elm_msg_cat, NewmailSet, NewmailIncommingMail,
				 "Incoming mail:\n"));

	while (1) {
	    long bytes = 0;  /* This is size change delta */

#ifdef PIDCHECK
	    if ( kill(parent_pid,0))
		exit(0);
#else
#ifndef AUTO_BACKGROUND		/* won't work if we're nested this deep! */
	    if (getppid() == 1) 	/* we've lost our shell! */
		exit(0);
#endif /* AUTO_BACKGROUND */
#endif /* PIDCHECK */

	    if (! isatty(1))  /* we're not sending output to a tty any more */
		goto clean_exit;

	    DPRINT(Debug,1,(&Debug,			    
			    "\n----\n"));

	    for (i = 0; i < total_folders; i++) {
		enum new_mail_stat r1;
		int err = 0;
		
		cur_folder = &folder_list[i];
		DPRINT(Debug,1,(&Debug,			    
				"[checking folder #%d: %s]\n",
				i, cur_folder->F->cur_folder_sys));

		if (0 != (get_folder_mode(cur_folder->F) & FOLDER_FILE)) {
		    save_acc(cur_folder->F->cur_folder_sys);
		}

		r1 = new_mail_on_folder(cur_folder->F,&bytes,NULL,&err,
					NULL);
		
		switch (r1) {
		case new_mail_check_failed: {
                    DPRINT(Debug,10,(&Debug,  
				     "newmail: %s check failed",
				     cur_folder->F-> cur_folder_sys));
		    if (err) {
			DPRINT(Debug,10,(&Debug, ": %s (errno=%d)",
					 strerror(err),err));
		    }
		    DPRINT(Debug,10,(&Debug, "\n"));
		    
		    if (EACCES == err) {
			cur_folder->access_error++;
			if (cur_folder->access_error > 5) {
			    lib_error(CATGETS(elm_msg_cat, 
					      NewmailSet,
					      NewmailErrNoPerm,
					      "Permission to monitor \"%S\" denied!"),
				      cur_folder->F->cur_folder_disp);
			    
			    if (POLL_method) 
				wait_for_timeout(5);
			    else
				sleep(5);
			    
			    ret = 1;
			    goto clean_exit;
			}
		    }
		}
		    break;
		case have_new_mail: {
		    
		    struct read_folder_state * read_state_ptr = NULL;
		    enum sessionlock_status sessionlock_ret = sessionlock_fail;
		    int err = 0;

		    sessionlock_ret =
			sessionlock_folder(cur_folder->F,SESSIONLOCK_NONE,
					   &err,NULL);

		    switch (sessionlock_ret) {
		    case sessionlock_fail:
			DPRINT(Debug,10,(&Debug,  
					 "newmail: %s open failed",
					 cur_folder->F-> cur_folder_sys));
			if (err) {
			    DPRINT(Debug,10,(&Debug, ": %s (errno=%d)",
					     strerror(err),err));
			}
			DPRINT(Debug,10,(&Debug, "\n"));

			if (EACCES == err) {
			    cur_folder->access_error++;
			    if (cur_folder->access_error > 5) {
				lib_error(CATGETS(elm_msg_cat, 
						  NewmailSet,
						  NewmailErrNoPerm,
						  "Permission to monitor \"%S\" denied!"),
					  cur_folder->F->cur_folder_disp);
				
				if (POLL_method) 
				    wait_for_timeout(5);
				else
				    sleep(5);
				
				ret = 1;
				goto clean_exit;
			    }
			}
			continue;
		    case sessionlock_open:
			DPRINT(Debug,10,(&Debug,  
					 "newmail: %s opened.\n",
					 cur_folder->F -> cur_folder_sys));
			break;
		    case sessionlock_reconnect:
			DPRINT(Debug,10,(&Debug,  
					 "newmail: %s opened, was reconnected.\n",
					 cur_folder->F -> cur_folder_sys));
			
			break;						
		    }

		    DPRINT(Debug,1,(&Debug,			    
				    "\tnew mail has arrived!  %ld bytes more mail\n",
				    bytes));

		    /* skip what we've read already... */
		    
		    if (!prepare_read_folder(cur_folder->F,
					     PREPARE_NEW_ONLY_NOLOCK,
					     &read_state_ptr,NULL)) {
			goto clean;
		    }

		    /* read and display new mail! */
		    if (read_headers(cur_folder,read_state_ptr) && ! in_window)
			printf("\n\r");

		    end_read_folder(cur_folder->F,&read_state_ptr,NULL,1);

		clean:
		    /* try to set the file access times back, ignore
		       failures */

		    if (0 != (get_folder_mode(cur_folder->F) & FOLDER_FILE)) {
			int r = close_folder(cur_folder->F,CLOSE_NORMAL,
					     /* struct cancel_data  * cd */ NULL);

			if (!r) {
			    DPRINT(Debug,9, (&Debug, 
					     "newmail: close_folder failed\n"));
			}

			if (have_stat)
			    change_file_utime(cur_folder->F->cur_folder_sys,
					      & utime_stat_buffer, NULL);
		    }

		}
		    break;
		case failed_mbx_ign_new_mail: 
                    DPRINT(Debug,10,(&Debug,  
				     "newmail: %s check failed",
				     cur_folder->F-> cur_folder_sys));
		    if (err) {
			DPRINT(Debug,10,(&Debug, ": %s (errno=%d)",
					 strerror(err),err));
		    }
		    DPRINT(Debug,10,(&Debug, "; new mail is not available anyway (POP?)\n"));
		    
		    ret = 1;
		    goto clean_exit;
		case no_new_mail:      
		    if (bytes < 0) {	/* file SHRUNK! */
			long last = bytes;
			
			int done  = 0; 
			int count = 0;
			
			enum sessionlock_status sessionlock_ret = sessionlock_fail;
			int err = 0;
			
			DPRINT(Debug,1,(&Debug,			    
					"\tfile is shrunk  %ld bytes\n",
					-bytes));
			
			while (! done) {
			    long temp = 0;  /* This is size change delta 
					       compared to last 
					       prepare_read_folder() */
			    
			    if (POLL_method) 
				wait_for_timeout(1);
			    else
				sleep(1);	/* wait for the size to stabilize */
			    
			    new_mail_on_folder(cur_folder->F,&temp,NULL,NULL,NULL);
			    
			    if (temp != last)
				last = temp;
			    else {
				DPRINT(Debug,1,(&Debug,"\t (%d) file size change stablized to %ld bytes\n",
						count,last));
				done++;
			    }
			    
			    if (++count > 1000) {
				DPRINT(Debug,1,(&Debug,"\tsize does not  stabilize\n"));
				
				ret = 1;
				goto clean_exit;
			    }
			}

			if (0) {
			case new_mail_reconnect:
			    DPRINT(Debug,10,(&Debug,  
					     "newmail: %s was reconnected.\n",
					     cur_folder->F -> cur_folder_sys));
			    
			}
			
			sessionlock_ret =
			    sessionlock_folder(cur_folder->F,SESSIONLOCK_NONE,
					       &err,NULL);
			
			switch (sessionlock_ret) {
			case sessionlock_fail:
			    DPRINT(Debug,10,(&Debug,  
					     "newmail: %s open failed",
					     cur_folder->F-> cur_folder_sys));
			    if (err) {
				DPRINT(Debug,10,(&Debug, ": %s (errno=%d)",
						 strerror(err),err));
			    }
			    DPRINT(Debug,10,(&Debug, "\n"));
			    break;
			    
			case sessionlock_open: 
			    
			    DPRINT(Debug,10,(&Debug,  
					     "newmail: %s opened.\n",
					     cur_folder->F -> cur_folder_sys));
			    if (0) {
			    case sessionlock_reconnect:
				DPRINT(Debug,10,(&Debug,  
						 "newmail: %s opened, was reconnected.\n",
						 cur_folder->F -> cur_folder_sys));
			    }

			    { 
				struct read_folder_state * read_state_ptr = NULL;
				
				/* Updates folder reference size for new_mail_on_folder() */
				if (prepare_read_folder(cur_folder->F,PREPARE_NOLOCK,
							&read_state_ptr,NULL)) {
				    end_read_folder(cur_folder->F,&read_state_ptr,NULL,1);
				}
			    }
			    break;
			}
		    }
		    break;
		}
	    }
	
	    if (POLL_method) 
		wait_for_timeout(interval_time);
	    else
		sleep(interval_time);

	}
 clean_exit:
	for (i = 0; i < total_folders; i++) {
	    int r = close_folder(folder_list[i].F,CLOSE_NORMAL,
				 /* struct cancel_data  * cd */ NULL
				 );
	    if (!r) {
		DPRINT(Debug,9, (&Debug, 
				 "newmail: close_folder failed\n"));
	    }	    
	}
	return ret;
}

static int parse_header_routine P_((struct folder_info *folder,
				    READ_STATE read_state_ptr,
				    struct header_rec *entry,
				    header_list_ptr parsed_headers ));
static int parse_header_routine(folder,read_state_ptr,entry,parsed_headers)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *entry;
     header_list_ptr parsed_headers;
{
    header_list_ptr tmphdr;

    /* copy_envelope_folder() may have set entry->content_length */
    /* copy_envelope_folder() have set entry->header_charset */


    if (NULL != (tmphdr = locate_header_by_name(parsed_headers,
						"From"))) {

	if (entry->from)
	    free_addr_list(&(entry->from));
	entry->from = 
	    parse_header_address("From",tmphdr->body, 
				 !(entry->status & NOHDRENCODING),
				 entry->header_charset,
				 & (entry-> header_error));	    
    }
        
    if (NULL != (tmphdr = locate_header_by_name(parsed_headers,
						"To"))) {	    
	if (entry->to)
	    free_addr_list(&(entry->to));
	entry->to = 
	   parse_header_address("To",
				tmphdr->body, 
				!(entry->status & NOHDRENCODING),
				entry->header_charset,
				& (entry-> header_error));	    
    }
        
    if (NULL != (tmphdr = locate_header_by_name(parsed_headers,
						"Cc"))) {	    
	if (entry->cc)
	    free_addr_list(&(entry->cc));
	entry->cc = 
	    parse_header_address("Cc",tmphdr->body, 
				 !(entry->status & NOHDRENCODING),
				 entry->header_charset,
				 & (entry-> header_error));	    
    }
        
    if (NULL != (tmphdr = locate_header_by_name(parsed_headers,
						"Status"))) {	    
	if (index(tmphdr->body, 'R') != NULL)
	    entry->status &= ~(NEW | UNREAD);
	else if (index(tmphdr->body,'O') != NULL) {
	    entry->status &= ~NEW;
	    entry->status |= UNREAD;
	}
	if (index(tmphdr->body, 'r') != NULL)
	    entry->status |= REPLIED_TO;
    }
    
    if (NULL != (tmphdr = locate_header_by_name(parsed_headers,
						"Subject"))) {	    
	
	if (entry->subject)
	    free_string(&(entry->subject));
	
	entry->subject = 
	    hdr_to_string(HDR_TEXT,
			  tmphdr->body,
			  entry->header_charset,
			  !(entry->status & NOHDRENCODING));	  
    } else if (!entry->subject)
	entry->subject = 
	    format_string(CATGETS(elm_msg_cat, NewmailSet, 
				  NewmailNoSubject,
				  "(No Subject Specified)"));
    
    if (NULL != (tmphdr = locate_header_by_name(parsed_headers,
						"Importance"))) {	    
	if (atoi(tmphdr->body) >= 2)
	    entry->priority = TRUE;
    }

    if (NULL != (tmphdr = locate_header_by_name(parsed_headers,
						"Priority"))) {	    
	
	if (strincmp(tmphdr->body, "normal", 6) != 0 &&
	    strincmp(tmphdr->body, "non-urgent", 10) != 0)
	    entry->priority = TRUE;		
    }
    return 1;
}


static int parse_body_routine P_((struct folder_info *folder,
			   READ_STATE read_state_ptr,
			   struct header_rec *entry,
			   header_list_ptr parsed_headers,
			   struct counter_data *counter
			   ));
static int parse_body_routine(folder,read_state_ptr,entry,parsed_headers,
			      counter)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *entry;
     header_list_ptr parsed_headers;
     struct counter_data *counter;
{
    int ret = 0;
    char * buffer;
    int len;
    long content_remaining                = -1L;
    enum copy_env_end_status endstat = copy_env_end_failure;
    long newbytes = 0;
    int newmails  = 0;

    content_remaining = entry->content_length;

    DPRINT(Debug,10,(&Debug,
		     "parse_body_routine: reading body"));
    if (content_remaining >= 0L) {
	DPRINT(Debug,10,(&Debug,", content_remaining %ld",
			 content_remaining));
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
 reset_body:

    while (copy_body_folder(folder,read_state_ptr,
			    &buffer,&len,&content_remaining)) {
	if (buffer)
	    free(buffer);
    }

    endstat = copy_envelope_end_folder(folder,read_state_ptr,
				       &newbytes,
				       &newmails);
    switch (endstat) {
    case copy_env_end_mismatch:
	DPRINT(Debug,10,(&Debug,
			 "parse_body_routine: copy ended not on end of folder"));
	if (content_remaining >= 0L) {
	    DPRINT(Debug,10,(&Debug,", content_remaining %ld",
			     content_remaining));
	}
	DPRINT(Debug,10,(&Debug,"\n"));
	
	if (entry->content_length >= 0L) {
	    entry->content_length = -1L;
	    
	    if (copy_envelope_reset_body(folder,read_state_ptr,
					 &content_remaining))
		goto reset_body;
	}
	/* FALLTHRU */
    case copy_env_end_failure:
	
	DPRINT(Debug,10,(&Debug,			    
			 "-- Message parsing FAILED.\n"));
	ret = 0;
	goto clean;

    case  copy_env_end_newmail:
	DPRINT(Debug,10,(&Debug,
			 "parse_body_routine: Got new mail"));
	if (newbytes  >= 0L) {
	    DPRINT(Debug,10,(&Debug,", %ld new bytes",
			     newbytes));
	}
	if (newmails >= 0) {
	    DPRINT(Debug,10,(&Debug,", %d new mails",
			     newmails));
	}	
	DPRINT(Debug,10,(&Debug,"\n"));

	/* FALLTHRU */
    case copy_env_end_match:
	DPRINT(Debug,10,(&Debug,			    
			 "-- Message parsed.\n"));
	
	entry->body_parsed = 1;
	ret = 1;
	break;
    }
	
 clean:
    DPRINT(Debug,10,(&Debug,
		     "parse_body_routine=%d\n",ret));
    return ret;
}

static int read_headers(cur_folder,read_state_ptr)
     struct folder_struct *cur_folder;
     struct read_folder_state * read_state_ptr;
{
    /** read the headers, output as found given current_folder,
	the prefix of that folder, and whether we're in a window
	or not.
    **/
    enum copy_env_status status = copy_env_eof;

    struct header_rec hdr;		/* holds header info on curr mssg */
    int count = 0;			/* count of messages done	  */
    int read_folder = 1;

    
    bzero((void *)&hdr, sizeof hdr);
    hdr.mbx_info    = NULL;
    hdr.binary      = 0;
    /*
     * Go through every line of the mailbox.
     */


    while(read_folder) {
	status = 
	    copy_envelope_folder(cur_folder->F,read_state_ptr,
				 &hdr,parse_header_routine,parse_body_routine,
				 NULL);
	
	switch (status) {
	case copy_env_ok: 
        		
	    if (hdr.status & NEW) {
		show_header(&hdr, cur_folder);
	    } else {
		DPRINT(Debug,10,(&Debug,			    
				 "-- Message %d is not new...\n",count));
	    }

	case copy_env_no_data:
	    DPRINT(Debug,10,(&Debug,			    
			     "-- Message %d is not handled...\n",count));
	    break;
	default:
	    read_folder = 0;
		break;
	}

	free_rec_mbx_info(&hdr);
	if (hdr.header_error)
	    free_header_errors(& (hdr.header_error));
	bzero((void *)&hdr, sizeof hdr);   /* memory leak ... */
	hdr.binary      = 0;

	if (read_folder) {
	    count++;
	} else
	    break;

    }
    
    return(count);
}

static void add_folder(name)
     char *name;
{
    /* add the specified folder to the list of folders...ignore any
       problems we may having finding it (user could be monitoring
       a mailbox that doesn't currently exist, for example)
    */

    char *cp, buf[SLEN];
    
    struct read_folder_state * read_state_ptr = NULL;
    enum sessionlock_status sessionlock_ret = sessionlock_fail;
    int err = 0;

    if (current_folder >= MAX_FOLDERS) {
	lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailErrMaxFolders,
			   "Sorry, but I can only keep track of %d folders.\n"), 
		   MAX_FOLDERS);
	exit(1);
    }

    /* now let's rip off the suffix "=<string>" if it's there... */

    for (cp = name + strlen(name); cp > name+1 && *cp != '=' ; cp--)
	/* just keep stepping backwards */ ;

    /* if *cp isn't pointing to the first character we'e got something! */

    if (cp > name+1) {

	*cp++ = '\0';		/* null terminate the filename & get prefix */

	if (metachar(*cp)) cp++;
	
	strfcpy(folder_list[current_folder].prefix, cp,
		sizeof folder_list[current_folder].prefix);
	print_prefix = 1;
    }
    else {			/* nope, let's get the basename of the file */
	for (cp = name + strlen(name); cp > name && *cp != '/'; cp--)
	    /* backing up a bit... */ ;
	
	if (metachar(*cp)) cp++;
	if (*cp == '/') cp++;

	strfcpy(folder_list[current_folder].prefix, cp,
		sizeof folder_list[current_folder].prefix);
    }

    /* and next let's see what kind of weird prefix chars this user
       might be testing us with.  We can have '+'|'='|'%' to expand
       or a file located in the incoming mail dir...
    */

    if (metachar(name[0])) {
	expand_filename(name, buf, sizeof buf);
	folder_list[current_folder].F = 
	    enter_new_folder(buf,
			     TreatAsSpool,0,0);
    }
    else if (access(name, ACCESS_EXISTS) == -1) {
	const char * mailhome_val = give_dt_estr_as_str(&mailhome_dir_e,
						  "mailhome-dir",
						  NULL,NULL);

	if (mailhome_val) {

	    /* let's try it in the mail home directory */
	    elm_sfprintf(buf, sizeof buf,
			 FRM("%s%s"), 
			 mailhome_val, name);
	    if (access(buf,ACCESS_EXISTS) != -1) 		/* aha! */
		folder_list[current_folder].F = enter_new_folder(buf,
								 TreatAsSpool,
								 0,0);
	    else
		folder_list[current_folder].F = enter_new_folder(name,
								 TreatAsSpool,
								 0,0);
	} else
		folder_list[current_folder].F = enter_new_folder(name,
								 TreatAsSpool,
								 0,0);
    } else
	folder_list[current_folder].F = enter_new_folder(name,TreatAsSpool,
							 0,0);
    
    if (!folder_list[current_folder].F) {
	DPRINT(Debug,1,(&Debug,			    
			"failed to add folder %s\n",name));
	return;
    }
    
    /* now let's try to actually open the file descriptor and grab
       a size... */

     sessionlock_ret =
	 sessionlock_folder(folder_list[current_folder].F,SESSIONLOCK_NONE,&err,
			    NULL);
     
     switch (sessionlock_ret) {
     case sessionlock_fail:
	 DPRINT(Debug,10,(&Debug,  
			  "add_folder: %s open failed",
			  folder_list[current_folder].F-> cur_folder_sys));
	 if (err) {
	     DPRINT(Debug,10,(&Debug, ": %s (errno=%d)",
			      strerror(err),err));
	 }
	 DPRINT(Debug,10,(&Debug, "\n"));
	 
	 if (err == EACCES) {
	     lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailErrNoPerm,
			       "Permission to monitor \"%S\" denied!"),
		       folder_list[current_folder].F->cur_folder_disp);
	     exit(1);
	 }
	 goto clean;
     case sessionlock_open:
	 DPRINT(Debug,10,(&Debug,  
			  "add_folder: %s opened.\n",
			  folder_list[current_folder].F -> cur_folder_sys));
	 break;
     case sessionlock_reconnect:
	 DPRINT(Debug,10,(&Debug,  
			  "add_folder: %s opened, was reconnected.\n",
			  folder_list[current_folder].F -> cur_folder_sys));
	 
	 break;		 
    }

    if (!prepare_read_folder(folder_list[current_folder].F,PREPARE_NOLOCK,
			     &read_state_ptr,NULL)) {
			     
	goto clean;
    }

    end_read_folder(folder_list[current_folder].F,&read_state_ptr,NULL,1);

/* and increment current-folder please! */
 clean:

    DPRINT(Debug,1,(&Debug,			    
		    "folder %d: \"%s\" <%s>\n",
		    current_folder,
		    folder_list[current_folder].F->cur_folder_sys,
		    folder_list[current_folder].prefix));

    current_folder++;
    total_folders++;
}

static void add_default_folder()
{
    unsigned int defaultfile_keyword = 0;
    unsigned int defaultfile_cmask   = 0;

    struct read_folder_state * read_state_ptr;
    enum sessionlock_status sessionlock_ret = sessionlock_fail;
    int err = 0;
    
    const char * default_val = 
	give_dt_estr_as_str(&defaultfile_e,
			    "incoming-mailbox",
			    &defaultfile_keyword,
			    &defaultfile_cmask);

    if (!default_val)
	return;

    folder_list[0].F = enter_new_folder(default_val,TreatAsSpool,
					defaultfile_keyword,
					defaultfile_cmask);

    if (!folder_list[0].F) {
	DPRINT(Debug,1,(&Debug,			    
			"failed to add folder %s\n",
			default_val));
	return;
    }

    sessionlock_ret =
	sessionlock_folder(folder_list[0].F,SESSIONLOCK_NONE,
			   &err,NULL);
    
    
    switch (sessionlock_ret) {
    case sessionlock_fail:
	DPRINT(Debug,10,(&Debug,  
			 "add_default_folder: %s open failed",
			 folder_list[0].F-> cur_folder_sys));
	if (err) {
	    DPRINT(Debug,10,(&Debug, ": %s (errno=%d)",
			     strerror(err),err));
	}
	DPRINT(Debug,10,(&Debug, "\n"));
	
	if (err == EACCES) {
	    lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailErrNoPerm,
			      "Permission to monitor \"%S\" denied!"),
		      folder_list[0].F->cur_folder_disp);
	    exit(1);
	}
	goto clean;
    case sessionlock_open:
	DPRINT(Debug,10,(&Debug,  
			  "add_default_folder: %s opened.\n",
			  folder_list[0].F -> cur_folder_sys));
	 break;
     case sessionlock_reconnect:
	 DPRINT(Debug,10,(&Debug,  
			  "add_default_folder: %s opened, was reconnected.\n",
			  folder_list[0].F -> cur_folder_sys));
	 
	 break;	
	 
    }
    
    if (!prepare_read_folder(folder_list[0].F,PREPARE_NOLOCK,
			     &read_state_ptr,NULL)) {
	
	goto clean;
    }
    
    end_read_folder(folder_list[0].F,&read_state_ptr,NULL,1);

 clean:

    DPRINT(Debug,1,(&Debug,			    
		    "default folder: \"%s\" <%s>\n",
		    folder_list[0].F->cur_folder_sys,
		    folder_list[0].prefix));
    total_folders = 1;
    
}

static void show_header(hdr, cur_folder)
     struct header_rec *hdr;
     struct folder_struct *cur_folder;
{
    struct string * from_line = NULL;
    char prefix[SLEN];
    int used_to_line;
    
    const struct remote_server * remote_server =
	give_folder_server(cur_folder->F);
    struct address              * usraddr = NULL;
    
    if (remote_server)
	usraddr = give_remote_server_useraddr(remote_server,
					      NULL);

    used_to_line = DisplayAddress(hdr,&from_line,usraddr);

    prefix[0] = '\0';

    if (! in_window)
	strfcat(prefix, ">> ", sizeof prefix);

    if (print_prefix) {
	strfcat(prefix, cur_folder->prefix, sizeof prefix);
	strfcat(prefix, ": ", sizeof prefix);
    }

    if (in_window) {
	if (hdr->priority && used_to_line)
	    strfcat(prefix, priority_to, sizeof prefix);
	else if (hdr->priority)
	    strfcat(prefix, priority_text, sizeof prefix);
	else if (used_to_line)
	    strfcat(prefix, To_text, sizeof prefix);
	elm_fprintf(stdout,FRM("\007%s%S -- %S\n"), 
		    prefix, from_line, hdr->subject);
    } else {
	if (hdr->priority)
	    strfcat(prefix, priority_mail, sizeof prefix);
	else
	    strfcat(prefix, mail_text, sizeof prefix);
	if (used_to_line)
	    strfcat(prefix, to_text, sizeof prefix);
	else
	    strfcat(prefix, from_text, sizeof prefix);
	elm_fprintf(stdout,FRM("\n\r%s%S - %S\n\r"), 
		    prefix, from_line, hdr->subject);
    }

    if (usraddr)
	free_address(& usraddr);
}

static void save_acc(name)
     char *name;
{
    int r = stat(name, &utime_stat_buffer);

    switch (r) {
    case -1:
	if (errno != ENOENT) {
	    lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailErrFstat,
			      "Error %d attempting fstat on %s"), errno, name);
	    exit(1);
	}
	break;
	
    case 0:
	have_stat = 1;
	break;
    }

}

static void usage(name)
     char *name;
{
    int x;
    /* print a nice friendly usage message */

    lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailArgsHelp1a,
		      "\nUsage: %s [-d] [-i interval] [-w] {folders}\n\
\targ\t\t\tMeaning\n"),
	      name);

    for (x = 0; OPTION_LETTERS[x]; x++) {

	int have_arg = ':' == OPTION_LETTERS[x+1];
	
	enum newmail_option_code code =  OPTION_LETTERS[x];
	
	if (isascii(code) && isprint (code)) {
	    switch (code) {
	    case NMAILOPT_IPv4:
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailOptIPv4,
				  "\t -%c \t\tUse IPv4 connection.\n"),
			  code);
		break;
	    case NMAILOPT_IPv6:
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailOptIPv6,
				  "\t -%c \t\tUse IPv6 connection.\n"),
			  code);
		break;
	    case NMAILOPT_Debug:
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailOptDebug,
				  "\t -%cclass=debugfile:n\t\tDebug - set debug level to 'n' for 'class'.\n"),
			  code);
		break;
	    case NMAILOPT_Interval:
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailOptInterval,
				  "\t -%c D\t\tSets the interval checking time to 'D' seconds.\n"),
			  code);
		break;
	    case NMAILOPT_AsSpool:
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailOptAsSpool,
				  "\t -%c \t\tMagic mode - treat all folders as spool files.\n"),
			  code);
		 break;
	    case NMAILOPT_RCFile:
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailOptRCFile,
				  "\t -%cx\t\tRcfile - Use 'x' as the elmrc instead of the default.\n"),
			  code);
		break;
	    case NMAILOPT_Window:
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailOptWindow,
				  "\t -%c \t\tForces 'window'-style output, and bypasses auto-background\n"),
			  code);
		break;
	    case NMAILOPT_transaction:
		lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailOptTransaction,
				  "\t -%cx\t\tLog remote protocol transaction to file 'x'.\n\
\t    \t\tWarning: A transaction (transcript) file will include passwords!\n"),
			  code);
		break;

	    case NMAILOPT_getopt_error:  /* getopt() returns '?' on error */
		break;
	    }
	}
	if (have_arg) {
	    /* Skip argument */	    
	    x++;
	}
    }

    lib_error(FRM("\n"));
    
    lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailArgsHelp2,
"Folders can be specified by relative or absolute path names, can be the name\n\
of a mailbox in the incoming mail directory to check, or can have one of the\n\
standard Elm mail directory prefix chars (e.g. '+', '=' or '%%').\n\
Furthermore, any folder can have '=string' as a suffix to indicate a folder\n\
identifier other than the basename of the file.\n\n"));

}

static void expand_filename(name, store_space,size)
     char *name, *store_space;
     int size;
{
    strfcpy(store_space, name, size);
    if (expand(store_space, size) == 0) {
	lib_error(CATGETS(elm_msg_cat, NewmailSet, NewmailErrExpand,
			  "Sorry, but I couldn't expand \"%s\"\n"),name);
	exit(1);
    }
}

static void pad_prefixes()
{
	/** This simple routine is to ensure that we have a nice
	    output format.  What it does is whip through the different
	    prefix strings we've been given, figures out the maximum
	    length, then space pads the other prefixes to match.
	**/

	register int i, j, len = 0;

	for (i=0; i < total_folders; i++)
	  if (len < (j=strlen(folder_list[i].prefix)))
	    len = j;
	
	for (i=0; i < total_folders; i++)
	  for (j = strlen(folder_list[i].prefix); j < len; j++)
	    strfcat(folder_list[i].prefix, " ", sizeof folder_list[i].prefix);
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
