static char rcsid[] = "@(#)$Id: readmsg.c,v 2.22 2023/12/13 16:55:33 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.22 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                       (was hurtta+elm@posti.FMI.FI, 
 *                            hurtta+elm@ozone.FMI.FI)
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 *  Based on Elm 2.4 utils/readmsg.c. That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 *  Incorparated Elm 2.5 code from utils/readmsg.c. 
 *  That code was following copyright:
 *
 *  The Elm Mail System
 *
 *                      Copyright (c) 1988-1995 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************/

/** This program extracts messages from a mail folder.  It is particularly
    useful when the user is composting a mail response in an external
    editor.  He or she can call this program to pull a copy of the original
    message (or any other message in the folder) into the editor buffer.

    One of the first things we do is look for a folder state file.
    If we are running as a subprocess to Elm this file should tell us
    what folder is currently being read, the seek offsets to all the
    messages in the folder, and what message(s) in the folder is/are
    selected.  We will load in all that info and use it for defaults,
    as applicable.

    If a state file does not exist, then the default is to show whatever
    messages the user specifies on the command line.  Unless specified
    otherwise, this would be from the user's incoming mail folder.

    Even if a state file exists, the user can override the defaults
    and select a different set of messages to extract, or select an
    entirely different folder.

    Messages can be selected on the command line as follows:

	readmsg [-options] *

	    Selects all messages in the folder.

	readmsg [-options] pattern ...

	    Selects messsage(s) which match "pattern ...".  The selected
	    message will contain the "pattern ..." somewhere within
	    the header or body, and it must be an exact (case sensitive)
	    match.  Normally selects only the first match.  The "-a"
	    selects all matches.

	readmsg [-options] sel ...

	    where:  sel == a number -- selects that message number
		    sel == $ -- selects last message in folder
		    sel == 0 -- selects last message in folder

    The undocumented "-I" option is a kludge to deal with an Elm race
    condition.  The problem is that Elm does piping/printing/etc. by
    running "readmsg|command" and placing the mail message selection
    into a folder state file.  However, if the "command" portion of
    the pipeline craps out, Elm might regain control before "readmsg"
    completes.  The first thing Elm does is unlink the folder state
    file.  Thus "readmsg" can't figure out what to do -- there is no
    state file or command line args to select a message.  In this
    case, "readmsg" normally gives a usage diagnostic message.  The
    "-I" option says to ignore this condition and silently terminate.

**/

#include "def_readmsg.h"
#include "s_readmsg.h"
#include "readmsg.h"

DEBUG_VAR(Debug,__FILE__,"readmsg");

#define metachar(c)	(c == '=' || c == '+' || c == '%')


/* program name for diagnostics */
char *prog;

static void malloc_fail_handler P_((char *proc,
				    size_t size));
static void malloc_fail_handler(proc, size)
     char *proc;
     size_t size;
{
    /* NOTE:
       Can't use elm_fprintf (or routines of lib/output.c)
       because here also malloc may fail, therefore
       can not use CATGETS macro;
       And can't use catgets because it may have given 
       incorrent format string from catalog ...
    */

    fprintf(stderr,
	    "%s: Out of memory [malloc of %lu bytes failed].\n", 
	    prog,(unsigned long)size);
    exit(1);
}

volatile int pipe_abort;		/* set to TRUE if receive SIGPIPE */

static SIGHAND_TYPE pipe_signal P_((int sig));
static SIGHAND_TYPE pipe_signal(sig)
     int sig;
{
    SIGDPRINT(Debug,2, (&Debug, "*** received SIGPIPE ***\n\n"));

    pipe_abort = TRUE;	/* internal signal ... wheeee!  */
    
    signal(SIGPIPE, pipe_signal);
}

static const char OPTION_LETTERS[] = "46ad:nhf:prIw:c:";

enum readmsg_option_code {
    RMSGOPT_IPv4        = ipv4_option /* '4' */,
    RMSGOPT_IPv6        = ipv6_option /* '6' */,
    RMSGOPT_AllMatch    = 'a',
    RMSGOPT_RCFile      = 'c',
    RMSGOPT_Debug       = 'd',
    RMSGOPT_Folder      = 'f',
    RMSGOPT_DispAll     = 'h',
    RMSGOPT_IgnNullRq   = 'I',
    RMSGOPT_DispNone    = 'n',
    RMSGOPT_PageBreak   = 'p',
    RMSGOPT_Raw         = 'r',
    RMSGOPT_Weed        = 'w',

    RMSGOPT_getopt_error= '?'  /* getopt() returns '?' on error */
};

/* FIXME  -- re-add weedlist */
static void usage_error P_((void));
static void usage_error()
{
    int x;
    
    lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgUsage1,
		      "Usage: %s {args} {MessageNum ... | pattern | *}\n\
   where {args} can be:\n"),
	      prog);

    for (x = 0; OPTION_LETTERS[x]; x++) {

        int have_arg = ':' == OPTION_LETTERS[x+1];

	enum readmsg_option_code code = OPTION_LETTERS[x];

	if (isascii(code) && isprint (code)) {
            switch (code) {
	    case RMSGOPT_IPv4:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptIPv4,
				  "\t -%c \t\tUse IPv4 connection.\n"),
			  code);
		break;
		
	    case RMSGOPT_IPv6:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptIPv6,
				  "\t -%c \t\tUse IPv6 connection.\n"),
			  code);
		break;
	    case RMSGOPT_AllMatch:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptAllMatch,
				  "\t -%c \t\tPrint all matching messages.\n"),
			  code);
		break;
	    case RMSGOPT_RCFile:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptRCFile,
				  "\t -%cx \t\tUse 'x' as the elmrc instead of the default\n"),
			  code);
		break;
	    case RMSGOPT_Debug:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptDebug,
				  "\t -%cclass=debugfile:n\t\tDebug - set debug level to 'n' for 'class'.\n"),
			  code);
		break;
	    case RMSGOPT_Folder:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptFolder,
				  "\t -%cfolder\tRead folder 'folder'.\n"),
			  code);
		break;		
	    case RMSGOPT_DispAll:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptDispAll,
				  "\t -%c \t\tDisplay all headers.\n"),
			  code);
		break;
	    case RMSGOPT_IgnNullRq:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptIgnNullR,
				  "\t -%c \t\tIgnore empty request.\n"),
			  code);
		break;
	    case RMSGOPT_DispNone:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptDispNone,
				  "\t -%c \t\tDo not display headers.\n"),
			  code);
		break;
	    case RMSGOPT_PageBreak:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptPageBreak,
				  "\t -%c \t\tPrint page break (form feed) between messages.\n"),
			  code);
		break;
	    case RMSGOPT_Raw:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptRaw,
				  "\t -%c \t\tPrint mail source without MIME decoding it.\n"),
			  code);
		break;
	    case RMSGOPT_Weed:
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgOptWeed,
				  "\t -%cweed-list\tDisplay headers matching 'weed-list'.\n"),
			  code);
		break;

	    case RMSGOPT_getopt_error:
		break;
	    }
	}

	if (have_arg) {
	    /* Skip argument */	    
	    x++;
	}
    }
	
	
    exit(1);
}

/*
 * Header weeding information.
 */

#define DFLT_WEED_LIST	"Subject: From: To: Cc: Apparently- Date:"

struct weed_header_info {
    char *hdrname;	/* string to compare against header	*/
    int hdrlen;		/* characters in "hdrname"		*/
    int hdrskip;	/* skip (instead of accept) on match?	*/
    struct weed_header_info *next;
};

/* head of linked list of weeded headers */
struct weed_header_info Weedlist;

/* is default to accept headers NOT in weed list? */
int weed_dflt_accept = FALSE;


static void setup_weed_info P_((const char *sel_weed_str));

/*
 * Process interesting mail headers.
 *
 * When in WEED mode, this routine should be called with each header line
 * in sequence, and with a NULL at the end of the header.  If the header
 * is interesting then we will print it out.  Continued header lines (i.e.
 * ones starting with whitespace) are also handled.  When the end of the
 * header is reached, if either the From: or Date: was missing then we
 * will generate replacements from the From_ header.
 *
 * Strict intepretation of RFC822 says { '\\' '\n' } at the end of a header
 * line should (at least in some cases) continue to the next line, but we
 * don't handle that.
 */

#if ANSI_C
static header_filter weed_header_filter;
#endif
static int weed_header_filter P_((header_list_ptr hdr,
				  int flag,
				  int have_title));

static int weed_header_filter(hdr,flag,have_title)
     header_list_ptr hdr;
     int flag;
     int have_title;
{
    enum hdr_disp_level code = flag;   /* Check switch values */
    const char * hdr_name = give_header_name(hdr->header_name);

    switch (code) {

    case NONE:
	return 0;

    case WEED: {
	struct weed_header_info *w;
	int accept_header = weed_dflt_accept;

	for (w = Weedlist.next ; w != NULL ; w = w->next) {  

	    int x = w->hdrlen;

	    if (x > 0 && ':' == w->hdrname[x-1]) {
		x--;

		if (strincmp(hdr_name, w->hdrname, x) == 0 &&
		    x == strlen(hdr_name)) {
		    accept_header = !w->hdrskip;
		    break;
		}

	    } else {
		if (strincmp(hdr_name, w->hdrname, x) == 0) {
		    accept_header = !w->hdrskip;
		    break;
		}
	    }
	}

	/* output accepted headers */
	if (accept_header)
	    return 1;

	return 0;	
    }

    case ALL:
    case BADLEV:
	return 1;
    }

    return 1;
}

/* 
 * Initialize the header weeding info.
 *
 * The header weeding info is a linked list of (struct weed_header_info).
 */
void setup_weed_info(sel_weed_str)
     const char *sel_weed_str;
{
    struct weed_header_info *w;
    char *weed_str, *fld, *s;

    /* make a copy we can scribble on safely */
    weed_str = safe_strdup(sel_weed_str);

    w = &Weedlist;
    while ((fld = strtok(weed_str, " \t\n,")) != NULL) {
	weed_str = NULL;

	/* convert "_" to " " */
	for (s = fld ; *s != '\0' ; ++s) {
	    if (*s == '_')
		*s = ' ';
	}

	/* add weeding info to end of list */
	w->next = (struct weed_header_info *)
	    safe_malloc(sizeof(struct weed_header_info));
	w = w->next;
	if (
	    (w->hdrskip = (*fld == '!'))
	    )
	    ++fld;
	w->hdrname = fld;
	w->hdrlen = strlen(fld);
	w->next = NULL;

	/*
	 * The default weed action is the opposite of the last list entry.
	 * That is, if the list ends in "foo" (an accept action) then the
	 * default is to reject things not in the weed list.  If the list
	 * ends in "!foo" (a reject action) then the default is to accept
	 * things not in the weed list.
	 *
	 * If this doesn't make sense to you, consider the following
	 * two example commands...and what you'd like them to do:
	 *
	 *	readmsg -w "From: Date: Subject:"
	 *	readmsg -w "!From_ !Received: !Status:"
	 *
	 */
	weed_dflt_accept = w->hdrskip;

    }  
}



#if ANSI_C
static type_mismatch_prompt mime_mismatch;
#endif
static int mime_mismatch P_((mime_t *ptr, int displaying));
static int mime_mismatch(ptr,displaying)
     mime_t *ptr; 
     int displaying;
{

    return 0;
}

/* Return 1 on success */
static int print_index_message P_((struct folder_handler  * folder,
				   int idx,
				   enum hdr_disp_level hdr_disp_level,
				   int do_raw_output,
				   int print_separator));

static int print_index_message(folder,idx,hdr_disp_level,do_raw_output,
			       print_separator)
     struct folder_handler  * folder;
     int idx;
     enum hdr_disp_level hdr_disp_level;
     int do_raw_output;
     int print_separator;
{
    struct header_rec *entry = NULL;
    long content_length;
    long left;
    enum message_error err = error_none;
    header_list_ptr hdrs = NULL;
    charset_t local_subjectcharset = NULL;

    FILE *F = give_message_from_folder(folder,idx, &content_length,
				       &err,print_separator,NULL,
				       &entry);

    int buf_len;
    int ret = 0;

    char buf[SLEN];

    struct in_state  * state_in = NULL;

    struct out_state * state_out = NULL;
    charset_t charset_vector[2];

    switch (err) {
    case error_seek:
	lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgCannotSeek0,
			  "%s: Cannot seek or retrieve to selected message.\n"),
		  prog);	
	break;
    case error_start:
	lib_error(CATGETS(elm_msg_cat, ReadmsgSet,
			  ReadmsgCannotFindStart0,
			  "%s: Cannot find start of selected message.\n"),
		  prog);
	break;
    case error_none:
	/* Not used */
	break;
    }

    if (!F) {
	ret = 0;
	goto fail;
    }

    state_in = new_in_state(STATE_in_file);
    state_out = new_out_state(STATE_out_file);

    set_in_state_file(F,state_in);
    set_out_state_file(stdout,state_out);
    
    charset_vector[0] = system_charset;
    charset_vector[1] = NULL;
    
    set_out_state_cs(state_out,NULL,charset_vector);

    hdrs = file_read_headers(F,RHL_MARK_FOLDING);

    if (! do_raw_output &&
	use_ct_cs_for_subj_hack &&
	! entry->header_charset_set) {

	header_list_ptr subjecthdr = locate_header_by_name(hdrs,"Subject");
	
	if (subjecthdr && subjecthdr->body && check_8bit_str(subjecthdr->body)) {
	    header_list_ptr ct = locate_header_by_name(hdrs,
						       "Content-Type");
	    char header_buffer[80];
	    
	    const char * body_value;
	    const char *c;
	    	    
	    /* header_trim_whitespace_helper return either original body or pointer to buffer, 
	       may return NULL on overflow; returns also return NULL if body == NULL
	       
	       Also handles \r and \n
	    */
	  
	    if (ct &&
		(body_value = header_trim_whitespace_helper(ct->body,
							    header_buffer,
							    sizeof header_buffer)) &&
		NULL != (c = strchr (body_value, ';'))) {
		char value [20];
		c++;
		
		if (mime_get_param("charset",value,c,
				   sizeof value)) {
		    
		    local_subjectcharset = MIME_name_to_charset(value,0);
		    if (local_subjectcharset) {
			const char * MIME_name UNUSED_VAROK = 
			    get_charset_MIME_name(local_subjectcharset);
			DPRINT(Debug,9,(&Debug,"-- 8-bit subject charset %s\n",MIME_name ? MIME_name : value));

		    } else {
			DPRINT(Debug,9,(&Debug,"-- skipped 8-bit subject charset %s\n",value));
		    }
		}
	    }
	}
    }
    
    if (hdr_disp_level != NONE) {

	if (do_raw_output)
	    state_write_raw_headers(state_out,&NULL_decode_opt,
				    hdrs,
				    weed_header_filter,
				    hdr_disp_level,0);
	else
	    state_write_headers(state_out,&NULL_decode_opt,
				hdrs,
				weed_header_filter,
				hdr_disp_level,			    
				!(entry -> status & NOHDRENCODING),
				entry->header_charset,
				0,
				local_subjectcharset
				);	    
	
	state_printf(state_out, FRM("\n"));

    }

    delete_headers(&hdrs);


    if (do_raw_output ||
	0 == (entry->status & MIME_MESSAGE)) {
	left = content_length;
	
	/* print body 
	   it is assumed that content_length matches to whole lines ..
	   no partial lines on end ...
	*/
	
	while (left > 0 &&
	       (buf_len = mail_gets(buf, SLEN, F)) > 0) {
	    
	    fwrite(buf, 1, buf_len, stdout);
	    left -= buf_len;
	}

    } else {

	if (!entry->mime_parsed) {
	    DPRINT(Debug,5,(&Debug,"mime_parse_routine was not called\n"));
	    mime_parse_routine(NULL,entry,F);
	}
	
	mime_decode(&(entry->mime_rec), state_in, 
		    state_out,&NULL_decode_opt,
		    entry->header_charset,
		    entry, mime_mismatch,
		    entry->default_body_charset);
    }
    ret = 1;

 fail:

    if (state_in)
	free_in_state(&state_in); 
    if (state_out)
	free_out_state(&state_out); 

    return ret;
}



/* Return 1 on success */
static int print_matching_messages P_((struct folder_handler  * folder,
				       const char * match,
				       int do_all_matches,
				       enum hdr_disp_level hdr_disp_level, 
				       int do_page_breaks,
				       int do_raw_output));
static int print_matching_messages (folder,match,do_all_matches,
				    hdr_disp_level,
				    do_page_breaks,
				    do_raw_output)
     struct folder_handler  * folder;
     const char * match;
     int do_all_matches;
     enum hdr_disp_level hdr_disp_level;
     int do_page_breaks;
     int do_raw_output;
{
    int ret = 1;
    int i;   
    int found_count = 0;
    int print_separator = 0;

    if (do_raw_output)
	print_separator = 1;

    for (i = 0; i < folder->num_messages; i++) {
	enum message_error err = error_none;
	long content_length, left;
	int buf_len;
	char buf[SLEN];
	FILE *F;
	char * env_buffer = NULL;

	if (pipe_abort) {
	    ret = 0;
	    break;
	}

	F = give_message_from_folder(folder,i, &content_length,
				     &err,0,&env_buffer,NULL);

	if (!F) {
	    
	    goto clean;
	}

	if (env_buffer && strstr(env_buffer, match) != NULL) {

	    goto do_print;
	}

	/* Headers */
	while ((buf_len = mail_gets(buf, SLEN, F)) > 0) {
	    if ((1 == buf_len && buf[0] == '\n') 
		||
		(2 == buf_len && buf[0] == '\r' && buf[1] == '\n')
		) {
		break;
	    }

	    if (strstr(buf, match) != NULL) {
		goto do_print;
	    }
	}

	/* Body */
	left = content_length;
	while (left > 0 &&
	       (buf_len = mail_gets(buf, SLEN, F)) > 0) {
	    
	    if (strstr(buf, match) != NULL) {
		goto do_print;
	    }
	    left -= buf_len;
	}


	if (0) {
	do_print:
	    
	    if (print_separator) {
		/* OK */

	    } else if (found_count++ == 0)
		; /* no seperator before first message */
	    else if (do_page_breaks)
		putchar(FORMFEED);
	    else
		puts("\n------------------------------------------------------------------------------\n");

	    if (! print_index_message(folder,i,hdr_disp_level, do_raw_output,
				      print_separator))
		ret = 0;
		
	    if (!do_all_matches)
		break;

	}

    clean:
	if (env_buffer)
	    free(env_buffer);

    }

    if (!found_count) {
	lib_error(CATGETS(elm_msg_cat, ReadmsgSet,
			  ReadmsgPatternNotFound,
			  "%s: Search pattern not found: %s"), 
		  prog,match);

	ret = 0;
    }

    return ret;
}

/* Return 1 on success */
int print_number_message (folder,num,
			  hdr_disp_level,
			  do_page_breaks,
			  do_raw_output)
     struct folder_handler  * folder;
     int num;
     enum hdr_disp_level hdr_disp_level;
     int do_page_breaks;
     int do_raw_output;
{
    static int num_mssgs_listed = 0;

    int print_separator = 0;

    if (do_raw_output)
	print_separator = 1;


    if (print_separator) {

	/* OK */

    } else if (num_mssgs_listed++ == 0)
	; /* no seperator before first message */
    else if (do_page_breaks)
	putchar(FORMFEED);
    else
	puts("\n------------------------------------------------------------------------------\n");
   
    return print_index_message(folder,num-1,hdr_disp_level, do_raw_output,
			       print_separator);
}

char version_buff[NLEN];


int main P_((int argc, char *argv[]));					
int main(argc, argv)
     int argc;
     char *argv[];
{
    int is_fstate;

    int do_all_matches  = 0;	/* true to show all mssgs which match pat*/
    enum hdr_disp_level 
	 hdr_disp_level = BADLEV; /* amount of headers to show		 */
    int do_page_breaks  = 0;	/* true to FORMFEED between messages	*/
    int ign_no_request  = 0;    /* terminate if no actions requested	*/
    char * folder_name = NULL;	/* pathname to the mail folder		*/
    int i;
    int exit_status = 0;
    int do_raw_output   = 0;    /* do raw output                        */
    char *sel_weed_str =  DFLT_WEED_LIST;   /* header weeding specification		*/

    const int read_flags = 0;

    
    /* install trap for safe_malloc() failure */
    safe_malloc_fail_handler = malloc_fail_handler;

#if DEBUG
    init_debugfile("READMSG");
#endif
    locale_init();

    prog = argv[0];

    user_init();
    init_addrlib(read_flags);
    init_mboxlib(read_flags);
    init_melib(read_flags);
    init_defaults(read_flags);

    signal(SIGPIPE, pipe_signal);		


    is_fstate = have_fstate();

    DPRINT(Debug,10,(&Debug,"is_fstate=%d\n",is_fstate));

    if (is_fstate < 0) {
	lib_error(CATGETS(elm_msg_cat, ReadmsgSet,
			  ReadmsgStateFileCorrupt,
			  "%s: Elm folder state file appears to be corrupt!\n"), 
		  prog);
	exit(1);
    }
    /* crack the command line */
    while ((i = getopt(argc, argv,OPTION_LETTERS)) != EOF) {

	enum readmsg_option_code code = i; /* Get warning abount missing enum
					       values on switch */
	switch (code) {
	case RMSGOPT_IPv4:
	case RMSGOPT_IPv6:
	      if (add_ipv_option(i,ipv4_option_printerr) <=  ipv_opt_unsupported) {
		  exit(1);
	      }	      
	      break;
	    
	case RMSGOPT_AllMatch:
	    do_all_matches = 1;
	    break;
	case RMSGOPT_Debug:
#if DEBUG
	    set_debugging(optarg);	   
#else
	    lib_error(CATGETS(elm_msg_cat, ReadmsgSet, 
			      ReadmsgArgsIngoringDebug,
			      "Warning: system created without debugging enabled - request ignored\n"));
#endif
	    break;
	case RMSGOPT_DispNone:
	    hdr_disp_level = NONE;
	    break;
	case RMSGOPT_DispAll:
	    hdr_disp_level = ALL;
	    break;
	case RMSGOPT_Weed:
	    hdr_disp_level = WEED;
	    sel_weed_str = optarg;
	    break;
	case RMSGOPT_Folder:
	    folder_name = optarg;
	    if (metachar(folder_name[0])) {
		static char buffer[LONG_STRING];
		
		strfcpy(buffer,optarg, sizeof buffer);
		
		if (!expand(buffer, sizeof buffer)) {
		    lib_error(CATGETS(elm_msg_cat, ReadmsgSet,
				      ReadmsgCannotExpandFolderName,
				      "%s: Cannot expand folder name \"%s\".\n"),
			      prog, folder_name);
		    exit(1);
		}
		folder_name = buffer;
	    }
	    break;
	case RMSGOPT_PageBreak:
	    do_page_breaks = 1;
	    break;
	case RMSGOPT_Raw:
	    do_raw_output  = 1;
	    break;
	case RMSGOPT_IgnNullRq :
	    ign_no_request = 1;
	    break;
	case RMSGOPT_RCFile:
	    set_user_rc_location(optarg); 
	    break;
	case RMSGOPT_getopt_error:  /* getopt() returns '?' on error */
	    usage_error();
	    break;
	}
    }

    read_rc_file(read_flags);


    elm_sfprintf(version_buff, sizeof version_buff,
		 FRM("%s PL%s"), VERSION, PATCHLEVEL);

#ifdef DEBUG	  	 
    { 
	    int d = panic_dprint("\n\
======================================================\n\
Debug output of the READMSG program (version %s).\n",
			     version_buff);

	    if (d >= 50) {
#if 0	    
		printf("WARNING: Debug file may include passwords -- edit it!\n");
		panic_dprint("WARNING: Edit manually out sensitive information from that file!");
#endif
	    }
    }
#endif

    
    if (do_raw_output &&  do_page_breaks)
	usage_error();


    if (BADLEV == hdr_disp_level) {
	if (do_raw_output)
	    hdr_disp_level = ALL;
        else 
	    hdr_disp_level = WEED;
    }

    /* process the header weeding specification */
    if (hdr_disp_level == WEED)
	setup_weed_info(sel_weed_str);

    if (do_raw_output && hdr_disp_level != ALL)
	usage_error();

    if (optind == argc) {

	if (folder_name)
	    usage_error();

	if (is_fstate) {
	    if (!process_fstate_print(hdr_disp_level, do_page_breaks, do_raw_output))
		exit_status = 1;  /* Failure */

	} else {
	    if (!ign_no_request)
		usage_error();

	}

    } else if (optind < argc) {

	struct folder_handler  * folder = NULL;


	if (folder_name) {
	    /* 1) given file */
	    
	    folder = open_normal_folder(folder_name,0,0);
	    
	    if (!folder) {
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgNoPermRead,
				  "%s: You have no permission to read %s!\n\r"),
			  prog, folder_name);
		exit (1);
	    }
	    
	    
	} else if (is_fstate) {
	    /* 2) fstate file */
	    

	    folder = open_fstate();
	    if (!folder) {
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgNoPermReadFS,
				  "%s: You have no permission to read folder state.\n\r"),
			  prog);
		exit (1);
	    }
	    
	} else { 
	    /* 3) default folder/mailbox */
	    unsigned int defaultfile_keyword = 0;
	    unsigned int defaultfile_cmask   = 0;

	    const char * default_val = give_dt_estr_as_str(&defaultfile_e,
							   "incoming-mailbox",
							   &  defaultfile_keyword,
							   & defaultfile_cmask);

	    if (!default_val) {
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet,
				  ReadmsgCannotGetIncomingName,
				  "%s: Cannot figure out name of your incoming mail folder.\n"),
			  prog);
		exit(1);
	    }

	    folder = open_normal_folder(default_val,
					defaultfile_keyword,
					defaultfile_cmask);
	    if (!folder) {
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgNoPermReadDefault,
				  "%s: You have no permission to read default folder.\n\r"),
			  prog);
		exit (1);
	    }
	    
	}
	
	if (!parse_folder(folder)) {
	    if (folder_name)
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgFailedParse,
				  "%s: Failed to parse folder \"%s\".\n"), 
			  prog, folder_name);
	    else
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgFailedParse0,
				  "%s: Failed to parse folder.\n"), 
			  prog);
	    
	    exit_status = 0;
	    goto failure;

	}


	if (folder->num_messages < 1) {
	    if (folder_name)
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgFolderEmpty,
				  "%s: Folder \"%s\" is empty.\n"), 
			  prog, folder_name);
	    else
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet, ReadmsgFolderEmpty0,
				  "%s: Folder is empty.\n"), 
			  prog);
	    
	    exit_status = 0;
	    goto failure;
	}
	

	/* see if we are trying to match a pattern */
	if (index("0123456789$*", argv[optind][0]) == NULL) {

	    char * buffer = safe_strdup(argv[optind]);
	    
	    for (optind++ ; optind < argc ; ++optind) {
		buffer = strmcat(buffer," ");
		buffer = strmcat(buffer,argv[optind]);
	    }

	    if (!print_matching_messages(folder,buffer, do_all_matches,
					 hdr_disp_level, do_page_breaks,
					 do_raw_output))
		exit_status = 1;

	    free(buffer);

	    /* see if all messages should be shown */
	} else if (argc-optind == 1 && strcmp(argv[optind], "*") == 0) {

	    int i;

	    int print_separator = 0;

	    if (do_page_breaks)
		print_separator = 1;

	    for (i = 0; i < folder->num_messages; i++) {
		
		if (pipe_abort) {
		    exit_status = 1;
		    break;
		}


		if (i > 0) {
		    if (print_separator) {
			/* OK */
		    } else if (do_page_breaks)
			putchar(FORMFEED);
		    else 
			puts("\n------------------------------------------------------------------------------\n");		    
		}

		if (!print_index_message(folder,i, 
					 hdr_disp_level, do_raw_output,
					 print_separator))
		    exit_status = 1;
	    }


	    /* print out all the messages specified on the command line */
	} else for ( ; optind < argc ; ++optind) {

	    int num;
	    
	    if (strcmp(argv[optind], "$") == 0 || 
		strcmp(argv[optind], "0") == 0) 
		num = folder->num_messages;
	    else if ((num = atoi(argv[optind])) == 0) {
		lib_error(CATGETS(elm_msg_cat, ReadmsgSet,
				  ReadmsgIDontUnderstand,
				  "%s: I don't understand what \"%s\" means.\n"),
			  prog, argv[optind]);

		exit_status = 1;
		goto failure;
	    }

	    if (pipe_abort) {
		exit_status = 1;
		break;
	    }
	    
	    if (!print_number_message(folder,num, 
				      hdr_disp_level, do_page_breaks,
				      do_raw_output))
		exit_status = 1;

	}

    failure:
	if (folder)
	    free_folder_handler(&folder);
    } else {
	panic("READMSG PANIC",__FILE__,__LINE__,"main",
	      "Bad optind",0);

    }


    return exit_status;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
