static char rcsid[] = "@(#)$Id: getid.c,v 2.12 2020/05/22 14:56:37 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.12 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>  (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/


#include "def_addr.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"addr");

static struct message_id * parse_tokenized_message_id P_((const char  *headername,
							  char ** tokenized,
							  int demime,
							  charset_t defcharset,
							  struct header_errors **header_error));


static struct message_id * parse_tokenized_message_id(headername,tokenized,
						      demime,defcharset,header_error)
     const char  *headername;
     char ** tokenized;
     int demime;
     charset_t defcharset;
     struct header_errors **header_error;
{
    struct message_id * result    = NULL;

    char              * left      = NULL;
    char              * domain    = NULL;

    int walk = 0, next_point;

    char **scanned = NULL;
    struct string *comments = NULL;

    int error = 0;

    look_special_tokens(tokenized,"<@>",walk,&next_point,demime,
			defcharset,&comments,&scanned);


    /* Get '<' */

    if (tokenized[next_point] &&
	'<' == tokenized[next_point][0]) {

	if (scanned) {

	    DPRINT(Debug,25,(&Debug, 
			     "parse_tokenized_message_id: parse error, phrase before <\n"));
	    
	    if (headername)
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MePhraseNotAllowed,
					     "PARSE ERROR: Phrase before < on %s header not allowed"),
				     headername);
	    
	    free_rfc822tokenized(scanned);   
	    scanned = NULL;
	}


	/* Parse left part*/

	walk = next_point+1;
	look_special_tokens(tokenized,"<@>",walk,&next_point,demime,
			    defcharset,&comments,&scanned);
	
	if (scanned) {
	    left = scanned_to_str(scanned);
	    free_rfc822tokenized(scanned); 
	    scanned = NULL;
	} else {
	    DPRINT(Debug,25,(&Debug, 
			     "parse_tokenized_message_id: parse error, no local part before '@'\n"));

	    if (headername)
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeMissingLocalPart,
					     "PARSE ERROR: Local part before @ missing on %s header"),
				     headername);	    	    
	}

	if (tokenized[next_point] &&
	    '@' == tokenized[next_point][0]) {

	    /* Parse right part */

	    walk = next_point+1;
	    look_special_tokens(tokenized,"<@>",walk,&next_point,demime,
				defcharset,&comments,&scanned);
	    
	    if (scanned) {
		domain = scanned_to_str(scanned);
		free_rfc822tokenized(scanned); 
		scanned = NULL;
	    } else {

		DPRINT(Debug,25,(&Debug, 
				 "parse_tokenized_message_id: parse error, no domain part after '@'\n"));
		
		if (headername)
		    process_header_error(header_error,
					 CATGETS(elm_msg_cat, MeSet,
						 MeMissingDomainPart,
						 "PARSE ERROR: Domain part after @ missing on %s header"),
					 headername);	  	    

		error = 1;
	    }

	    if (tokenized[next_point] &&
		'>' == tokenized[next_point][0]) {
		
		/* Parse comment part */

		walk = next_point+1;
		look_special_tokens(tokenized,"<@>",walk,&next_point,demime,
				    defcharset,&comments,&scanned);

		if (scanned && scanned[0]) {

		    DPRINT(Debug,25,(&Debug, 
				     "parse_tokenized_message_id: parse error, extra data after '>'\n"));
			
		    if (headername)
			    process_header_error(header_error,
						 CATGETS(elm_msg_cat, MeSet,
							 MeExtradataAfterGreaterThan,
							 "PARSE ERROR: Extra data after > on %s header"),
						 headername);	  	    

		    error = 1;
		}

	    } else if (tokenized[next_point]) { 

		DPRINT(Debug,25,(&Debug, 
				 "parse_tokenized_message_id: parse error, next token '@'\n"));

		if (headername)
		    process_header_error(header_error,
					 CATGETS(elm_msg_cat, MeSet,
						 MeExtraToken,
						 "PARSE ERROR: Extra %s on %s header"),
					 tokenized[next_point],headername);

		error = 1;

	    } else {

		DPRINT(Debug,25,(&Debug, 
				 "parse_tokenized_message_id: parse error, next token not '>'\n"));
		
		if (headername)
		    process_header_error(header_error,
					 CATGETS(elm_msg_cat, MeSet,
						 MeMissingGreaterThan,
						 "PARSE ERROR: Missing > on %s header"),
					 headername);
		error = 1;

	    }
	    
	} else {

	    DPRINT(Debug,25,(&Debug, 
			     "parse_tokenized_message_id: parse error, next token not '@'\n"));

	    if (headername) 
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeMissingAt,
					     "PARSE ERROR: @ missing on %s header"),
				     headername);

	    error = 1;
	}

    } else {

	DPRINT(Debug,25,(&Debug, 
			 "parse_tokenized_message_id: parse error, no < seen\n"));
	
	if (headername)
	    process_header_error(header_error,
				 CATGETS(elm_msg_cat, MeSet,
					 MeMessageIdStart,
					 "PARSE ERROR: ID must start with < on header %s"),
				 headername);

	error = 1;
	   
    }

    if (scanned) {
	free_rfc822tokenized(scanned);  
	scanned = NULL;
    }

    if (left && domain && !error) {
	result = new_message_id(left,domain,comments);

	DPRINT(Debug,25,(&Debug, 
			 "parse_tokenized_message_id: Message-ID parsed\n"));

	
    }

    if (left) {
	DPRINT(Debug,25,(&Debug, 
			 "parse_tokenized_message_id: left=%s\n",left));
	free(left); left = NULL;
    }

    if (domain) {
	DPRINT(Debug,25,(&Debug, 
			 "parse_tokenized_message_id: domain=%s\n",domain));
	free(domain); domain = NULL;
    }

    if (comments) {
	DPRINT(Debug,25,(&Debug, 
			 "parse_tokenized_message_id: comments=%S\n",comments));
	free_string(&comments);
    }

    return result;
}



struct message_id * parse_header_message_id(headername,buffer,demime,
					    defcharset,header_error)
     const char *headername;
     const char *buffer;
     int demime;
     charset_t defcharset;
     struct header_errors **header_error;
{
    char             ** tokenized = rfc822_tokenize(buffer);
    struct message_id * result    = NULL;



    DPRINT(Debug,9,(&Debug, 
		    "parse_header_message_id: headername=%s buffer=%s\n",
		    headername ? headername : "<not given>",
		    buffer));

    /* if buffer is empty, tokenized[0] == NULL
       tokenized is not NULL
    */
    
    if (!tokenized)
	return NULL;

    result = parse_tokenized_message_id(headername,tokenized,demime,
					defcharset,header_error);

    free_rfc822tokenized(tokenized);   
    tokenized = NULL;

    if (result) {	 
	DPRINT(Debug,11,(&Debug, 
			 "parse_header_message_id=%p\n",
			 result));
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "parse_header_message_id=NULL\n"));

    }

    return result;
}


struct references * parse_header_references(headername,buffer,demime,
					    defcharset,header_error)
     const char *headername;
     const char *buffer;
     int demime;
     charset_t defcharset;
     struct header_errors **header_error;
{
    char             ** tokenized = rfc822_tokenize(buffer);
    struct references * result    = NULL;

    int walk = 0, next;

    DPRINT(Debug,9,(&Debug, 
		    "parse_header_references: headername=%s buffer=%s\n",
		    headername ? headername : "<not given>",
		    buffer));

    /* if buffer is empty, tokenized[0] == NULL
       tokenized is not NULL
    */
    
    if (!tokenized)
	return NULL;
    
    for (walk = 0; tokenized[walk]; walk = next) {
	char **scanned = NULL;

	struct message_id   * message_id = NULL;
	struct string       * phrase     = NULL;
	struct string       * comment   = NULL;
    
	while (tokenized[walk] && 
	       whitespace(tokenized[walk][0]))
	    walk++;

	/* 1) get message-id */

	if (tokenized[walk] &&
	    '<' == tokenized[walk][0]) {
	    int len,i;

	    for (next = walk+1; tokenized[next]; next++) {
		if ('>' == tokenized[next][0]) {
		    next++;
		    break;
		}
	    }

	    while (tokenized[next] && 
		   ( '(' == tokenized[next][0] ||
		     whitespace(tokenized[next][0])))
		next++;
	    
	    len = next-walk;

	    scanned = safe_calloc((len+1), sizeof (scanned[0]));

	    for (i = 0; i < len; i++) {
		scanned[i] = safe_strdup(tokenized[walk+i]);
	    }
	    scanned[i] = NULL;


	    message_id = 
		parse_tokenized_message_id(headername,scanned,demime,
					   defcharset,header_error);
	    free_rfc822tokenized(scanned);
	    scanned = NULL;

	    walk = next;
	}

	
	/* 2) get phrase */

	if (tokenized[walk]) {

	    look_special_tokens(tokenized,"<@>",walk,&next,demime,
				defcharset,&comment,&scanned);


	    if (scanned) {

		phrase = scanned_to_phrase (scanned,demime,defcharset);
		
		free_rfc822tokenized(scanned);
		scanned = NULL;

	    }

	    walk = next;

	}

	if (phrase || message_id || comment) {
	    
	    if (!result)
		result = new_references(message_id,phrase,comment);
	    else
		references_add_item(result,message_id,phrase,comment);
	}



	if (tokenized[walk] &&
	    '@' == tokenized[walk][0]) {

	    DPRINT(Debug,25,(&Debug, 
			     "parse_header_references: parse error, @ on phrase\n"));
	    
	    if (headername)
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeAtNotAllowedPhrase,
					     "PARSE ERROR: Unquoted @ not allowed on phrase (on header %s)"),
				     headername);
	    next = walk+1;
	}


	if (tokenized[walk] &&
	    '>' == tokenized[walk][0]) {

	    DPRINT(Debug,25,(&Debug, 
			     "parse_header_references: parse error, > on phrase\n"));
	    
	    if (headername)
		process_header_error(header_error,
				     CATGETS(elm_msg_cat, MeSet,
					     MeGreaterThanNotAllowedPhrase,
					     "PARSE ERROR: Unquoted > not allowed on phrase (on header %s)"),
				     headername);
	    next = walk+1;
	}


	if (message_id)
	    free_message_id(& message_id);
	if (phrase)
	    free_string(&phrase);
	if (comment)
	    free_string(&comment);
   
    }

    free_rfc822tokenized(tokenized);


    if (result) {	 
	DPRINT(Debug,11,(&Debug, 
			 "parse_header_references=%p\n",
			 result));
    } else {
	DPRINT(Debug,11,(&Debug, 
			 "parse_header_references=NULL\n"));

    }

    return result;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
