static char rcsid[] = "@(#)$Id: outheaders.c,v 2.21 2022/05/18 18:28:31 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.21 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_addr.h"
#include "hdr_imp.h"
#include "s_me.h"

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif         

DEBUG_VAR(Debug,__FILE__,"header");


static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}




int add_group_phrase_(x,GRP) 
     struct expanded_address *x;
     const struct string *GRP;
{
    if (x->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"add_group_phrase_",
	      "Bad magic number",0);
    
    if (!x->addrs)
	x->addrs = new_empty_addr_list(0);

    return add_group_to_list(x->addrs, GRP);
}

int add_expanded_addr0_ (x,ADR,g)
     struct expanded_address *x;
     const struct address *ADR;
     int g;
{
    if (x->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"add_expanded_addr0_",
	      "Bad magic number",0);

    if (!x->addrs)
	x->addrs = new_empty_addr_list(1);

    return add_address_to_list(x->addrs,ADR,g);
}

extern int add_expanded_addr_g(x,ADR,GRP)
     struct expanded_address *x;
     const struct address *ADR;
     const struct string *GRP;
{
    int g = -1;

    if (x->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"add_expanded_addr_g",
	      "Bad magic number",0);

    if (!x->addrs)
	x->addrs = new_empty_addr_list(1);

    if (GRP) 
	g = addr_list_lookup_group(x->addrs,GRP,1);
    
    return add_address_to_list(x->addrs,ADR,g);
}


int add_expanded_addr_ (x,ADR,FN,COM)
     struct expanded_address *x;
     const char *ADR;
     const struct string *FN;
     const struct string *COM;
{
    int X;

    struct address * a0 = new_address(ADR,FN,COM);

    X = add_expanded_addr0_(x,a0,-1);

    free_address(&a0);

    return X;
}

int add_expanded_addr_4(x,ADR,FN,COM,group)
     struct expanded_address *x;
     const char *ADR;
     const struct string *FN;
     const struct string *COM;
     const struct string *group;
{
    int X;

    struct address * a0 = new_address(ADR,FN,COM);

    X = add_expanded_addr_g(x,a0,group);

    free_address(&a0);

    return X;
}

int add_textual_addr_ (x,TEXTUAL,POS,LEN)
     struct expanded_address *x;
     const struct string *TEXTUAL;
     int POS, LEN;
{
    if (x->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"add_textual_addr_",
	      "Bad magic number",0);

    x->surface = safe_array_realloc(x->surface,(x->surface_len + 2),
				    sizeof (struct textual));
    if (!TEXTUAL)
	x->surface[x->surface_len].Textual     = NULL;
    else
	x->surface[x->surface_len].Textual     = dup_string(TEXTUAL);
    x->surface[x->surface_len].pos         = POS;
    x->surface[x->surface_len].len         = LEN;
    x->surface[x->surface_len+1].Textual   = NULL;
    x->surface[x->surface_len+1].pos       = POS+LEN;
    x->surface[x->surface_len+1].len       = 0;
    return x->surface_len++;
}

void zero_expanded_address (x)
     struct expanded_address *x;
{
    /* bzero is defined hdrs/elm_defs.h */
    bzero((void *)x,sizeof (*x));
    x->magic        = EXP_ADDR_magic;

    x->addrs        = NULL;
    x->surface      = NULL;
    x->surface_len  = 0;
}


void append_expanded_address (result,source) 
     struct expanded_address       * result;
     const struct expanded_address * source;
{
    int group_offset = 0;
    
    if (source->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"append_expanded_address",
	      "Bad magic number (source)",0);

    if (source->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"append_expanded_address",
	      "Bad magic number (result)",0);

     if (result->addrs)
	 group_offset = addr_list_group_count(result->addrs);
     
    if (source->addrs) {
	int g, glen = addr_list_group_count(source->addrs);
	
	if (!result->addrs)
	    result->addrs = 
		new_empty_addr_list(addr_list_item_count(source->addrs));
	
	for (g = 0; g < glen; g++)
	    add_group_phrase_(result,addr_list_get_group(source->addrs,g));
    }

    if (source->surface) {
	int i;

	for (i = 0; i < source->surface_len; i++) {
	    int count = 0;
	    int pos = result->addrs ? addr_list_item_count(result->addrs) : 0;
	    int j;
	    for (j = 0; j < source->surface[i].len; j++) {
		int g;
		
		const struct address * adr =
		    addr_list_get_item(source->addrs,
				       source->surface[i].pos+j,
				       &g);
		
		int p;
		
		if (g >= 0)
		    g += group_offset;
		
		p = add_expanded_addr0_(result,adr,g);
	    
		if (0 == count)
		    pos = p;
		count++;
	    }
	    add_textual_addr_(result,source->surface[i].Textual,pos,count);      
	}
    }
}

int have_expanded_address(source)
     const struct expanded_address * source;
{
    int addrcount UNUSED_VAROK;
	
    if (source->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"have_expanded_address",
	      "Bad magic number (source)",0);

    if (source->surface_len > 0 &&
	source->surface &&
	source->addrs &&
	(addrcount = addr_list_item_count(source->addrs)) > 0) {
	
	DPRINT(Debug,8,(&Debug,"have_expanded_address=1; surface len %d, address count %d\n",
			source->surface_len,addrcount));

	return 1;
    }
    
    return 0;    
}

void copy_expanded_address (result,source) 
     struct expanded_address       * result;
     const struct expanded_address * source;
{    
    free_expanded_address(result);

    append_expanded_address(result,source);
    
}

void free_expanded_address(x)
     struct expanded_address *x;
{
    int i;
    
    if (x->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"free_expanded_address",
	      "Bad magic number",0);

    if (x->addrs) 
	free_addr_list(& x->addrs);
    
    if (x->surface) {
	for (i = 0; i < x->surface_len; i++) {
	    if (x->surface[i].Textual) {
		free_string (&(x->surface[i].Textual));	     
	    }
	}
	free(x->surface);
	x->surface = NULL;
    }
    x->surface_len = 0;
}

struct string * make_surface_addr(address)
     const struct address * address;
{
    struct string * result = new_string(display_charset);
    int fl = 0, cl = 0;

    const char          * addr     = 
	address_get_ascii_addr(address);
    const struct string * fullname = address_get_phrase(address);
    const struct string * comment  = address_get_comment(address);

    if (fullname &&
	(fl = string_len(fullname))) {

	int i;
	
	add_ascii_to_string(result,s2us("\""));
	
	for (i = 0; i < fl; i++) {
	    uint16 code = give_unicode_from_string(fullname,i);

	    /* No fancy quotation stuff on textual form for editing */
	    
	    switch(code) {
	    case 0x0022  /* '"'  */: 
	    case 0x005C  /* '\\' */: 
		code = 0x005F /* '_' */;
		break;
	    }
	    add_unicode_to_string(result,1,&code);
	}

       	add_ascii_to_string(result,s2us("\" "));
    }

    add_ascii_to_string(result,s2us("<"));
    if (addr)
	add_ascii_to_string(result,cs2us(addr));
    add_ascii_to_string(result,s2us(">"));
    
    if (comment && 
	(cl = string_len(comment))) {
	int i;

	add_ascii_to_string(result,s2us(" ("));

	for (i = 0; i < cl; i++) {
	    uint16 code = give_unicode_from_string(comment,i);

	    /* No fancy quotation stuff on textual form for editing */

	    switch(code) {
	    case 0x0022 /* '"'  */: 
	    case 0x005C /* '\\' */: 
	    case 0x0028 /* '('  */: 
	    case 0x0029 /* ')'  */: 
	    case 0x003A /* ':'  */:
	    case 0x003C /* '<'  */: 
	    case 0x003E /* '>'  */:
		code = '_';
		break;
	    }
	    add_unicode_to_string(result,1,&code);
	}

	add_ascii_to_string(result,s2us(")"));

    }

    return result;
}

void expanded_address_from_list(result,addr_list)
     struct expanded_address *result;
     const struct addr_list *addr_list;
{
    int j;
    int len,glen;
    int group_offset = 0;

    if (!addr_list)
	return;

    len =  addr_list_item_count(addr_list);
    glen = addr_list_group_count(addr_list);

    if (result->addrs)
	group_offset = addr_list_group_count(result->addrs);

    for (j = 0; j < glen; j++) {
	add_group_phrase_(result,
			  addr_list_get_group(addr_list,j));
    }

    for (j = 0; j < len; j++) {
	int g;
	int pos;
	struct string *s;

	const struct address * address =
	    addr_list_get_item(addr_list,j,&g);

	if (g >= 0)
	    g += group_offset;

	pos = add_expanded_addr0_(result,address,g);

	s = make_surface_addr(address); 

	add_textual_addr_(result,s,pos,1);
	free_string(&s);

    }
}

static int hdr_addrs_are_allowed P_((struct addr_list * list,
				     const char * header_name));
static int hdr_addrs_are_allowed(list,header_name)
     struct addr_list * list;
     const char * header_name;
{
    int ret = 1;
    int len =  addr_list_item_count(list);
    int j;
    
    for (j = 0; j < len; j++) {
	int g;

	const struct address * address =
	    addr_list_get_item(list,j,&g);
	const char          * addr     = 
	    address_get_ascii_addr(address);

	if (addr) {
	    const char * sep =  qstrpbrk_c(addr,"!:@");

	    if (sep) {
		DPRINT(Debug,14, (&Debug, 
				  "hdr_addrs_are_allowed: #%d: addr %s separator %c\n",
				  j, addr, *sep));
		
		if ('@' == *sep && sep > addr) {
		    const char * domain = sep+1;
		    
		    DPRINT(Debug,14, (&Debug, 
				      "hdr_addrs_are_allowed: #%d: addr %s domain %s\n",
				      j, addr,domain));
		    
		    if (! domain[0]) {
			DPRINT(Debug,14, (&Debug, 
					  "hdr_addrs_are_allowed: #%d: addr %s, empty domain\n",
					  j,addr));
			

			if (ret && header_name) {
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeHeaderAddressEmptyDomain,
					      "Empty domain on address %s on header field %s."),
				      addr,header_name);
			}
			
			ret = 0;
		    } else if ('[' == domain[0]) {

			int len = rfc822_toklen(domain);
	 
			if (len < 2 || domain[len] || domain[len-1] != ']') {
			    DPRINT(Debug,14,
				   (&Debug, 
				    "hdr_addrs_are_allowed: #%d: addr %s, domain %s  failed to parse as literal\n",
				    j,addr,domain));

			    if (ret && header_name) {
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeHeaderBadAddressLiteral,
						  "Invalid literal %s on address %s on header field %s."),
					  domain,addr,header_name);
			    }

			    ret = 0;
			} else {
			    DPRINT(Debug,14,
				   (&Debug, 
				    "hdr_addrs_are_allowed: #%d: addr %s, domain %s is literal\n",
				    j,addr,domain));
			}


		    } else {
			const char * whitelisted_name =
			    is_whitelisted_valid_domain(domain);
			
			if (whitelisted_name) {
			    DPRINT(Debug,14,
				   (&Debug, 
				    "hdr_addrs_are_allowed: #%d: addr %s: domain %s is whitelisted on %s\n",
				    j,addr, whitelisted_name,domain));
		         } else {
			    const char * reserved_name =
				is_special_use_domain(domain);

			    if (reserved_name) {
				
				DPRINT(Debug,14,
				       (&Debug, 
					"hdr_addrs_are_allowed: #%d: addr %s: domain %s is reserved on %s\n",
					j,addr, reserved_name,domain));
				
				if (ret && header_name) {
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeHeaderAddressReservedDomain,
						      "Reserved domain %s is unsupported on address %s on header field %s."),
					      reserved_name,addr,header_name);
				}

				ret = 0;
			    }
		        }
		    }
		}

	    }

	}

    }

    DPRINT(Debug,14, (&Debug, "hdr_addrs_are_allowed=%d\n",ret));
    return ret;
}


static int add_expanded_address_hdr P_((struct expanded_address *eaddr,
					header_ptr X,
					const char *value,
					int demime,
					charset_t defcharset,
					int replace));
static int add_expanded_address_hdr(eaddr,X,value,demime,defcharset,replace)
     struct expanded_address *eaddr;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret  = 0;
    const char * hdr_name = give_header_name(X);

    /* list is != NULL also when value is empty */
    
    struct addr_list * list = 
	parse_header_address(hdr_name,value,demime,defcharset,NULL);
    
    if (eaddr->magic   != EXP_ADDR_magic) 
    	panic("HEADERS PANIC",__FILE__,__LINE__,"add_expanded_address_hdr",
	      "Bad magic number (expanded_address)",0);
    
    if (list) {

	if (hdr_addrs_are_allowed(list,hdr_name)) {
	    
	    if (replace)
		free_expanded_address(eaddr);
	    
	    expanded_address_from_list(eaddr, list);
	    ret = 1;
	}
    
	free_addr_list(&list);
	
	DPRINT(Debug,14, (&Debug,
			  "add_expanded_address_hdr=%d: header name %s, value=%s\n",
			  ret,hdr_name,value));	    

    } else {
	
	ret = 0;
	
	DPRINT(Debug,14, (&Debug,
			  "add_expanded_address_hdr=%d: No address, header name %s, value=%s\n",
			  ret,hdr_name,value));	    
    }
    
    return ret;
}



#ifdef ANSI_C
hdr_add_to_mailing_hdr add_from_hdr;
#endif
int add_from_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret  = add_expanded_address_hdr(&(headers->from),
					X,value,demime,
					defcharset,replace);

    DPRINT(Debug,14, (&Debug, "add_from_hdr=%d; value=%s\n",
		      ret,value));
    
    return ret;
}

static int not_whitespace P_((const char *s));

#ifdef ANSI_C
hdr_add_to_mailing_hdr add_date_hdr;
#endif
int add_date_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret = 0;
    const char * hdr_name = give_header_name(X);

    if (not_whitespace(value)) {

	static char time_zone[SLEN] = "";
	
	int month, day, year, hours, mins, secs, tz, wday = 0;
	
	enum parse_date_time_result r =
	    parse_parts_date_time(value,
				  time_zone,sizeof time_zone,
				  &month, &day, &year, &hours, &mins, &secs, &tz,
				  &wday /* 0 - 6 or -1 if not given */);
	
	switch (r) {
	    struct tm temp;
	    
	case parse_dt_unix:
	case parse_dt_mail:
	    
	    /* bzero is defined hdrs/elm_defs.h */
	    bzero((void *)&temp,sizeof temp);
	    
	    if (wday < -1 || wday > 6)
		goto failure;
	    
	    if (month < 1 || month > 12)
		goto failure;
	    if (day < 1 || day > 31)
		goto failure;
	    if (year < 1900)
		goto failure;
	    if (hours < 0 || hours > 23)
		goto failure;
	    if (mins < 0 || mins > 59)
		goto failure;
	    if (secs < 0 || secs > 60)
		goto failure;
	    if (tz < -13*60 || tz > 13*60)
		goto failure;

	    if (-1 == wday) 
		wday = wday_from_year_month_day(year,month,day);
		
	    temp.tm_wday = wday;
	    temp.tm_sec  = secs;
	    temp.tm_min  = mins;
	    temp.tm_hour = hours;
	    temp.tm_mday = day;
	    temp.tm_mon  = month-1;
	    temp.tm_year = year-1900;
	    temp.tm_isdst = -1;   /* Not available */
	
#ifdef TZNAME_USE_TM_ZONE
	    temp.tm_gmtoff = tz*60;
	    temp.tm_zone   = time_zone;
#endif	
	    if (headers->date) {
		const struct tm *  old_tm =
		    get_expanded_date_value(headers->date);

		if (old_tm) {
		    
#define compare(field)  (old_tm->field == temp.field)
		
		    if (compare(tm_wday) &&
			compare(tm_sec)  &&
			compare(tm_min)  &&
			compare(tm_hour)  &&
			compare(tm_mday)  &&
			compare(tm_mon)   &&
			compare(tm_year)) {
			ret = 1;

			DPRINT(Debug,14, (&Debug,
					  "add_date_hdr=%d: No change, header name %s, value=%s\n",
					  ret,hdr_name,value));	    

#undef compare	    
			goto no_change;
		    } else {
			char timebuf[SLEN] = "";
						
			if (expanded_time_hdrval(old_tm,get_expanded_date_cached_time(headers->date),
						 timebuf,sizeof timebuf,
						 print_date_and_time /* print_time */,
						 NULL, 0)) {		    
			    DPRINT(Debug,14, (&Debug,
					      "add_date_hdr: Changed, header name %s, old value=%s\n",
					      hdr_name,timebuf));
			}

			timebuf[0] = '\0';
			if (expanded_time_hdrval(&temp,0,
						 timebuf,sizeof timebuf,
						 print_date_and_time /* print_time */,
						 time_zone,tz*60)) {		    
			    DPRINT(Debug,14, (&Debug,
					      "add_date_hdr: Changed, header name %s, new value=%s\n",
					      hdr_name,timebuf));
			}

		    }
		}
				
		free_expanded_date(&  (headers->date));
	    }
	    headers->date =
		new_expanded_date(&temp,
				  0,
				  date_explicit);

	    ret = 1;
	    DPRINT(Debug,14, (&Debug,
			      "add_date_hdr=%d: Updated, header name %s, value=%s\n",
			      ret,hdr_name,value));	    
	    
	    break;
	    	    
	case parse_dt_fail:
	case parse_dt_dateonly:
	    ret = 0;
	    DPRINT(Debug,14, (&Debug,
			      "add_date_hdr=%d: Bad value, header name %s, value=%s\n",
			      ret,hdr_name,value));	    
	    
	    break;
	}
	    
    } else {
	ret = 1;

	if (replace && headers->date) 
	    free_expanded_date(&  (headers->date));

	DPRINT(Debug,14, (&Debug,
			  "add_date_hdr=%d: No date (whitespace only), header name %s, value=%s\n",
			  ret,hdr_name,value));	    

    }

    if (0) {
    failure:
	ret = 0;

	DPRINT(Debug,14, (&Debug,
			  "add_date_hdr=%d: Failure, header name %s, value=%s\n",
			  ret,hdr_name,value));	    

    }
    
 no_change:	
    return ret;
}


#ifdef ANSI_C
hdr_add_to_mailing_hdr add_to_hdr;
#endif
int add_to_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret =  add_expanded_address_hdr(&(headers->to),
					X,value,demime,
					defcharset,replace);

    DPRINT(Debug,14, (&Debug, "add_to_hdr=%d; value=%s\n",
		      ret,value));
    
    return ret;
}


#ifdef ANSI_C
hdr_add_to_mailing_hdr add_cc_hdr;
#endif
int add_cc_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret = add_expanded_address_hdr(&(headers->cc),
				       X,value,demime,
				       defcharset,replace);
    
    DPRINT(Debug,14, (&Debug, "add_cc_hdr=%d; value=%s\n",
		      ret,value));
    
    return ret;
}


#ifdef ANSI_C
hdr_add_to_mailing_hdr add_bcc_hdr;
#endif
int add_bcc_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret = add_expanded_address_hdr(&(headers->bcc),
				       X,value,demime,
				       defcharset,replace);
    
    DPRINT(Debug,14, (&Debug, "add_bcc_hdr=%d; value=%s\n",
		      ret,value));
    return ret;
}



#ifdef ANSI_C
hdr_add_to_mailing_hdr add_replyto_hdr;
#endif
int add_replyto_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret = add_expanded_address_hdr(&(headers->reply_to),
				       X,value,demime,
				       defcharset,replace);
    
    DPRINT(Debug,14, (&Debug, "add_replyto_hdr=%d; value=%s\n",
		      ret,value));
    return ret;
}

#if ANSI_C
static env_from_change_hook dummy_env_from_change;
#endif
static int dummy_env_from_change P_((struct mailer_env_from *X,
				     const char * value));

static int dummy_env_from_change(X,value)
     struct mailer_env_from *X;
     const char * value;
{
    return 0;

}

static env_from_change_hook  * ef_hook = &dummy_env_from_change;

void set_env_from_change_hook(hook)
     env_from_change_hook *hook;
{
    ef_hook = hook;
}

#ifdef ANSI_C
hdr_add_to_mailing_hdr add_env_from_hdr;
#endif
int add_env_from_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret = 0;
    
    if (0 == strcmp(value,"<>")) {
	ret = ef_hook(headers->env_from,value);
    } else {

	/* TMP is != NULL also when value is empty */

	struct addr_list * TMP = 
	    parse_header_address(give_header_name(X),
				 value,demime,defcharset,NULL);
	
	if (TMP) {
	    int len = addr_list_item_count(TMP);

	    if (1 == len) {
		int group = -1;
		const struct address * address =
		    addr_list_get_item(TMP,0,&group);
		const char *addr = 
		    address_get_ascii_addr(address);

		if (addr && headers->env_from) {
		    char * t = elm_message(FRM("<%s>"),addr);
		    ret = ef_hook(headers->env_from,t);	    
		    free(t);
		}
	    }
	    
	    free_addr_list(&TMP);
	}
    }

    return ret;
}


/* This replaces always */
#ifdef ANSI_C
hdr_add_to_mailing_hdr add_subject_hdr;
#endif

int add_subject_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    if (headers->subject)
	free_string( & (headers->subject)  );

    headers->subject = 
	hdr_to_string(HDR_TEXT,value,defcharset,demime);

    return 1;
}

#ifdef ANSI_C
hdr_add_to_mailing_hdr add_xmailer_hdr;
#endif

int add_xmailer_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    if (headers->xmailer)
	free_string( & (headers->xmailer)  );

    if (not_whitespace(value)) {
	headers->xmailer = 
	    hdr_to_string(HDR_TEXT,value,defcharset,demime);
    }
    
    return 1;
}

#ifdef ANSI_C
hdr_add_to_mailing_hdr add_user_hdr;
#endif

int add_user_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    if (replace) {

	/* Only replace header with same name */

	int i;
	int count = 0;

	for (i = 0; i < headers->user_header_count; i++) {

	    if (X == headers->user_header[i].name) {

		if (headers->user_header[i].value)
		    free_string(&(headers->user_header[i].value));
		headers->user_header[i].name = NULL;
	    } else {
		headers->user_header[count] = headers->user_header[i];

		if (count != i) {
		    headers->user_header[i].value = NULL;
		    headers->user_header[i].name = NULL;
		}
		count++;
	    }

	}
	headers->user_header_count = count;
    }

    headers->user_header = safe_array_realloc(headers->user_header,
					      (headers->user_header_count +1),
					      sizeof (headers->user_header[0]));
    headers->user_header[headers->user_header_count].name = X;
    headers->user_header[headers->user_header_count].value = 
	hdr_to_string(HDR_TEXT,value,defcharset,demime);
    headers->user_header_count++;

    return 1;
}

/* return (0 = Sunday, 1 = Monday, ... 6 = Saturday),
   input: month (m= 1 to 12), day (d= 1 to 31), and year
*/
int wday_from_year_month_day(year,month,day)
     int year;     
     int month;
     int day;
{
    /* https://en.wikipedia.org/wiki/Determination_of_the_day_of_the_week#Implementation-dependent_methods */
    int y = year;
    int m = month;
    int d = day;
    
    int wday = (d+=m<3?y--:y-2,23*m/9+d+4+y/4-y/100+y/400)%7;
    
    /* http://www.cadaeic.net/calendar.htm
       Mike Keith
    */
    
    DPRINT(Debug,16,(&Debug,
		     "wday_from_year_month_day=%d (0-6): year=%d month=%d (1-12) day=%d (1-31)\n",
		     wday,year,month,day));
    
    return wday;
}

#ifdef ANSI_C
hdr_add_to_mailing_hdr add_expires_hdr;
#endif
int add_expires_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret = 0;
    const char * hdr_name = give_header_name(X);
    
    if (not_whitespace(value)) {

	static char time_zone[SLEN] = "";
	enum print_time print_time = print_date_and_time;
	
	int month, day, year, hours, mins, secs, tz, wday = 0;
	
	enum parse_date_time_result r =
	    parse_parts_date_time(value,
				  time_zone,sizeof time_zone,
				  &month, &day, &year, &hours, &mins, &secs, &tz,
				  &wday /* 0 - 6 or -1 if not given */);
	

	switch (r) {
	    struct tm temp;
	    long l;
	    char *ptr;
	    
	case parse_dt_dateonly:
	    print_time = print_date_only;
	    /* FALLTHRU */
	    
	case parse_dt_unix:
	case parse_dt_mail:
	    
	    /* bzero is defined hdrs/elm_defs.h */
	    bzero((void *)&temp,sizeof temp);
	    
	    if (wday < -1 || wday > 6)
		goto failure;
	    
	    if (month < 1 || month > 12)
		goto failure;
	    if (day < 1 || day > 31)
		goto failure;
	    if (year < 1900)
		goto failure;
	    if (hours < 0 || hours > 23)
		goto failure;
	    if (mins < 0 || mins > 59)
		goto failure;
	    if (secs < 0 || secs > 60)
		goto failure;
	    if (tz < -13*60 || tz > 13*60)
		goto failure;
	    
	    if (-1 == wday) 
		wday = wday_from_year_month_day(year,month,day);
	    
	    temp.tm_wday = wday;
	    temp.tm_sec  = secs;
	    temp.tm_min  = mins;
	    temp.tm_hour = hours;
	    temp.tm_mday = day;
	    temp.tm_mon  = month-1;
	    temp.tm_year = year-1900;
	    temp.tm_isdst = -1;   /* Not available */
	    
#ifdef TZNAME_USE_TM_ZONE
	    temp.tm_gmtoff = tz*60;
	    temp.tm_zone   = time_zone;
#endif	
	    
	    if (headers->expires)
		free_expanded_expires(& (headers->expires));
	    headers->expires =
		new_expanded_expires(NULL,&temp,-1,
				     print_time);
	    
	    ret =  1;

	    DPRINT(Debug,14, (&Debug,
			      "add_expires_hdr=%d: Updated, header name %s, value=%s\n",
			      ret,hdr_name,value));	    
	    
	    break;
	case parse_dt_fail:

	    ptr = NULL;
	    
	    if (0 < (l = strtol(value,&ptr,10))) {

		while (whitespace(*ptr))
		    ptr++;

		if ('d' == *ptr &&
		    ('\0' == ptr[1] ||
		     0 == strcmp(ptr,"days"))) {

		    DPRINT(Debug,14, (&Debug,
				      "add_expires_hdr: %ld days\n",l));

		    if (l < 8*7) {
			if (headers->expires)
			    free_expanded_expires(& (headers->expires));
			headers->expires =
			    new_expanded_expires(NULL,NULL,l,
						 Expires_have_time ?  print_date_and_time : print_date_only);
			
			ret =  1;
			
			DPRINT(Debug,14, (&Debug,
					  "add_expires_hdr=%d: Updated, header name %s, value=%s\n",
					  ret,hdr_name,value));	    
	    
			break;
		    }		    		    
		}		    	       
	    }
	    
	    ret = 0;
	    
	    DPRINT(Debug,14, (&Debug,
			      "add_expires_hdr=%d: Bad value, header name %s, value=%s\n",
			      ret,hdr_name,value));	    


	    break;
	}
	
    } else {
	ret = 1;

	if (replace && headers->expires)
	    free_expanded_expires(& (headers->expires));

	DPRINT(Debug,14, (&Debug,
			  "add_expires_hdr=%d: No expires (whitespace only), header name %s, value=%s\n",
			  ret,hdr_name,value));	    
	
    }

    if (0) {
    failure:
	ret = 0;

	DPRINT(Debug,14, (&Debug,
			  "add_expires_hdr=%d: Failure, header name %s, value=%s\n",
			  ret,hdr_name,value));	    

    }

    return ret;
}

static int not_whitespace(s)
     const char *s;
{
    const char *c;

    for (c = s; *c; c++) {
	if (!whitespace(*c))
	    return 1;
    }

    return 0;
}


#ifdef ANSI_C
hdr_add_to_mailing_hdr add_message_id_hdr;
#endif
int add_message_id_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret  = 0;
    const char * hdr_name = give_header_name(X);

    if (not_whitespace(value)) {
    
	struct message_id * newid =
	    parse_header_message_id(hdr_name,value,demime,defcharset,NULL);
	
	if (newid) {
	    
	    const char * domain = message_id_domain(newid);
	    
	    if (domain) {
		int ok = 0;
		
		if ('[' == domain[0]) {
		    
		    int len = rfc822_toklen(domain);
		    
		    if (len < 2 || domain[len] || domain[len-1] != ']') {
			DPRINT(Debug,14,
			       (&Debug, 
				"add_message_id_hdr: %s, domain %s  failed to parse as literal\n",
				value,domain));
			
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeHeaderBadIDLiteral,
					  "Invalid literal %s on id %s on header field %s."),
				  domain,value,hdr_name);
			
			
			ret = 0;
		    } else {
			DPRINT(Debug,14,
			       (&Debug, 
				"add_message_id_hdr: %s, domain %s is literal\n",
				value,domain));
			ok = 1;
		    }
	    	    
		} else {
		    const char * whitelisted_name =
			is_whitelisted_msgid_domain(domain);
		    
		    if (whitelisted_name) {
			DPRINT(Debug,14,
			       (&Debug, 
				"add_message_id_hdr: %s, domain %s is whitelisted on %s\n",
				value, whitelisted_name,domain));
			ok = 1;
		    } else {
			const char * reserved_name =
			    is_special_use_domain(domain);
			
			if (reserved_name) {
			    
			    DPRINT(Debug,14,
				   (&Debug, 
				    "add_message_id_hdr: %s, domain %s is reserved on %s\n",
				    value, reserved_name,domain));
			    
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeHeaderIDReservedDomain,
					      "Reserved domain %s is unsupported on id %s on header field %s."),
				      reserved_name,value,hdr_name);
			    
			    ret = 0;
			} else
			    ok = 1;
		    }
		}
		
		if (ok) {
		    
		    if (headers->message_id)
			free_message_id(& (headers->message_id));
		    
		    headers->message_id = newid;
		    newid = NULL;
		    ret = 1;
		}	    
	    }
	    
	    if (newid)
		free_message_id(& newid);
	    
	} else {
	    ret = 0;
	
	    DPRINT(Debug,14, (&Debug,
			      "add_message_id_hdr: No id, header name %s, value=%s\n",
			      hdr_name,value));	    
	}
    } else {
	ret = 1;
	
	if (replace && headers->message_id)
	    free_message_id(& (headers->message_id));

	DPRINT(Debug,14, (&Debug,
			  "add_message_id_hd: No id (whitespace only), header name %s, value=%s\n",
			  hdr_name,value));	    
    }
    
    DPRINT(Debug,14, (&Debug,
		      "add_message_id_hdr=%d: header name %s, value=%s\n",
		      ret,hdr_name,value));	    
    
    return ret;
}

static int add_common_references_hdr P_((struct references ** ref_p,
					 header_ptr X,
					 const char *value,
					 int demime,
					 charset_t defcharset,
					 int replace));
static int add_common_references_hdr(ref_p,X,value,demime,defcharset,replace)
     struct references ** ref_p;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret  = 0;
    const char * hdr_name = give_header_name(X);

    if (not_whitespace(value)) {

	/* parse_header_references() may return NULL if no message-id's found */

	struct references * newref = parse_header_references(hdr_name,value,demime,defcharset,NULL);

	if (newref) {

	    /* Append does not make sense, so always replace */
	    
	    if (*ref_p) 
		free_references(ref_p);
	    *ref_p = newref; newref = NULL;
	    ret = 1;
		
	} else {
	    ret = 0;
	    
	    DPRINT(Debug,14, (&Debug,
			      "add_common_references_hdr: No ids, header name %s, value=%s\n",
			      hdr_name,value));	    

	}		
    } else {
	ret = 1;

	if (replace && *ref_p) 
	    free_references(ref_p);
	
	DPRINT(Debug,14, (&Debug,
			  "add_common_references_hdr: No ids (whitespace only), header name %s, value=%s\n",
			  hdr_name,value));	    

    }

    DPRINT(Debug,14, (&Debug,
		      "add_common_references_hdr=%d: header name %s, value=%s\n",
		      ret,hdr_name,value));	    

    
    return ret;
}



#ifdef ANSI_C
hdr_add_to_mailing_hdr add_in_reply_to_hdr;
#endif
int add_in_reply_to_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret = add_common_references_hdr(&(headers->in_reply_to),X,
					value,demime,defcharset,replace);
    
    DPRINT(Debug,14, (&Debug, "add_in_reply_to_hdr=%d; value=%s\n",
		      ret,value));
    
    return ret;
}

#ifdef ANSI_C
hdr_add_to_mailing_hdr add_references_hdr;
#endif
int add_references_hdr(headers,X,value,demime,defcharset,replace)
     struct mailing_headers * headers;
     header_ptr X;
     const char *value;
     int demime;
     charset_t defcharset;
     int replace;
{
    int ret = add_common_references_hdr(&(headers->references),X,
					value,demime,defcharset,replace);
    
    DPRINT(Debug,14, (&Debug, "add_references_hdr=%d; value=%s\n",
		      ret,value));
    
    return ret;
}
     

void dump_expanded_address(debuglevel,text,expanded)
     int debuglevel; 
     const char *text;
     struct expanded_address expanded;

{
    struct textual *ptr;
    int l UNUSED_VAROK = strlen(text);
    int LASTPOS = -1;
    int alen = expanded.addrs ?
	addr_list_item_count(expanded.addrs) : 0;
    int glen = expanded.addrs ?
	addr_list_group_count(expanded.addrs) : 0;
 
    DPRINT(Debug,debuglevel,
	   (&Debug, 
	    "%s: %d surfaces, %d addresses, %d groups\n",
	    text, expanded.surface_len, alen, glen));
 
    for (ptr = expanded.surface; 
	 ptr < expanded.surface + expanded.surface_len; 
	 ptr++) {
	int j;
	if (!ptr->Textual) {
	    DPRINT(Debug,debuglevel,
		   (&Debug, 
		    "%*s [%d] surface NULL! \t(%d addresses)\n",
		    l,"",ptr-expanded.surface,ptr->len));
	} else {
	    DPRINT(Debug,debuglevel,
		   (&Debug, 
		    "%*s [%d] surface=%S \t(%d addresses)\n",
		    l,"",ptr-expanded.surface,ptr->Textual,
		    ptr->len));
	}
	if (ptr->pos != LASTPOS + 1) {     
	    DPRINT(Debug,debuglevel,
		   (&Debug, 
		    "%*s [%d] ERROR!  address pos = %d != %d (lastpost +1)\n",
		    l,"",ptr-expanded.surface, ptr->pos, LASTPOS+1));			
	}
	for (j = 0; j < ptr->len; j++) {
	    if (ptr->pos+j < 0 || ptr->pos+j >= alen) {
		DPRINT(Debug,debuglevel,
		       (&Debug, 
			"%*s    %d [%d] ERROR, not range 0 - %d\n",
			l,"",j,ptr->pos,
			alen-1));
	    } else {

		int group = -1;
		const struct address * address = 
		    addr_list_get_item(expanded.addrs,ptr->pos+j,&group);
		
		const char          * addr     UNUSED_VAROK = 
		    address_get_ascii_addr(address);
		const struct string * fullname = address_get_phrase(address);
		const struct string * comment  = address_get_comment(address);

		DPRINT(Debug,debuglevel,
		       (&Debug, 			    
			"%*s    %d [%d] addr=%s",
			l,"",j,ptr->pos+j,
			addr ? addr : "<NULL>"));

		if (fullname) {
		    DPRINT(Debug,debuglevel,
			   (&Debug,"\tfullname=%S",
			    fullname));
		} else {
		    DPRINT(Debug,debuglevel,
			   (&Debug,"\tfullname=<NULL>"));

		}

		if (comment) {
		    DPRINT(Debug,debuglevel,
			   (&Debug,"\tcomment=%S\n",
			    comment));
		} else {
		    DPRINT(Debug,debuglevel,
			   (&Debug,"\tcomment=<NULL>\n",
			    comment));
		}

		if (fullname || comment) {

		    DPRINT(Debug,debuglevel,
			   (&Debug, 			    
			    "%*s         ",
			    l,""));
			    
		    if (fullname) {
			const char *A UNUSED_VAROK = 
			    get_string_MIME_name(fullname);
			const char *B UNUSED_VAROK =
			    get_string_lang(fullname);
			
			DPRINT(Debug,debuglevel,
			       (&Debug, 			    
				"fullname cs=%s lang=%s",
				A ? A : "<none>",
				B ? B : "<none>"));
		    }
		    
		    if (fullname && comment) {
			DPRINT(Debug,debuglevel,
			       (&Debug,", "));
		    }

		    
		    if (comment) {
			const char *A UNUSED_VAROK = get_string_MIME_name(comment);
			const char *B UNUSED_VAROK = get_string_lang(comment);
			
			DPRINT(Debug,debuglevel,
			       (&Debug,"comment cs=%s lang=%s",
				A ? A : "<none>",
				B ? B : "<none>"));
		    }
		    
		    DPRINT(Debug,debuglevel,
			   (&Debug,"\n"));  		
		}
		
		DPRINT(Debug,debuglevel,
		       (&Debug, 			    
			"%*s         group=%d",l,"",group));
		if (group >= glen) {
		    DPRINT(Debug,debuglevel,(&Debug," ERROR not range\n"));
		} else if (group >= 0) {
		    const struct string * G UNUSED_VAROK = 
			addr_list_get_group(expanded.addrs,group);
		
		    DPRINT(Debug,debuglevel,(&Debug,"=%S\n",G));
		} else {
		    DPRINT(Debug,debuglevel,(&Debug,"\n"));
		}

		
		LASTPOS = ptr->pos+j;
	    }
	}
    }
    DPRINT(Debug,debuglevel,
	   (&Debug, 
	    "%*s === end\n",l,""));
}


/* Returns either backquote replaced string or original
   if return value != buffer, value is malloced 
*/

static char * eval_backquote P_((char *buffer));
static char * eval_backquote(buffer)
     char *buffer;
{
    char *ptr ;
    char *result  = NULL;
    char *x1      = buffer;
    char *cptr    = NULL;
    int quitflag  = 0;

    DPRINT(Debug,8,(&Debug, "eval_backquote: %s\n",
		    buffer));

    for (ptr = buffer; (*ptr || result) && !quitflag; ptr++) {

	if ('`' == *ptr || !*ptr) {
	    
	    char * start = cptr ? cptr : x1;
	    int len;
	    char * clip;

	    quitflag = !*ptr;

	    if (!start)
		panic("HEADERS PANIC",__FILE__,__LINE__,"eval_backquote",
		      "No start pointer",0);
	    
	    len = ptr - start;
	    if (len < 0)
		panic("HEADERS PANIC",__FILE__,__LINE__,"eval_backquote",
		      "Bad start pointer",0);
		
	    clip = safe_malloc(len+1);
	    memcpy(clip,start,len);
	    clip[len] = '\0';

	    DPRINT(Debug,11,(&Debug, "eval_backquote: clip(len=%d)=%s\n",
			     len,clip));

	    if (!cptr) {
		/* Got start of command */

		result = strmcat(result,clip);
		cptr = ptr +1;
		x1 = NULL;
		free(clip);
		continue;
	    }

	    if (len > 0) {
		struct run_state RS;
		const char * argv[4];
		int retcode;

		char local_buffer[80];
		const char * sh = give_dt_estr_as_str(&shell_e,"shell",
						      NULL,NULL);
		int n;

		int res;

		if (!sh)
		    sh = "/bin/sh";

		DPRINT(Debug,5,(&Debug, "eval_backquote: command: %s\n",
				clip));

		argv[0] = sh;
		argv[1] = "-c";
		argv[2] = clip;
		argv[3] = NULL;
		
		res = start_run(&RS,
				SY_ENAB_SIGHUP|SY_ENAB_SIGINT|
				SY_RUN_STATE_OPIPE,argv,-1,-1);
		if (0 == res) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeBackquoteCmdFailed,
				      "Backquoted command \"%s\" in elmheaders failed."),
			      clip);
		    free(clip);
		    free(result);

		    DPRINT(Debug,5,(&Debug, 
				    "eval_backquote: start_run failed\n"));
		    return NULL;
		}

		while (0 < (n = 
			    mail_gets(local_buffer, sizeof local_buffer, 
				      RS.pfd))) {	
		    DPRINT(Debug,5,(&Debug, 
				    "eval_backquote: output: %s",
				    local_buffer));
		    if (local_buffer[n-1] != '\n') {
			DPRINT(Debug,5,(&Debug, 
				    "<-- no NEWLINE\n"));
		    }

		    result = strmcat(result,local_buffer);
		    
		}

		if (result && 0 < (n = strlen(result)) &&
		    '\n' == result[n-1]) {
		    result[n-1] = '\0';
		    DPRINT(Debug,5,(&Debug, 
				    "... Removed newline from end of result\n"));
		}

		res = wait_end(&RS,&retcode);
		if (res < 0 || retcode != 0) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeBackquoteCmdFailed,
				      "Backquoted command \"%s\" in elmheaders failed."),
			      clip);
		    free(clip);
		    free(result);
		    
		    DPRINT(Debug,5,(&Debug, 
				    "eval_backquote: command failed: res=%d, retcode=%d\n",
				    res,retcode));
		    return NULL;

		}
	    }

	    x1 = ptr +1;
	    cptr = NULL;
	    free(clip);
	}
    }

    if (result) {
	DPRINT(Debug,8,(&Debug, "eval_backquote: result=%s\n",
			result));
    }
	
    return result ? result : buffer;
}


void import_mailheaders(headers,import_mode)
     struct mailing_headers * headers;
     enum import_mh_mode      import_mode;
{
    int err;
    FILE *F;
    char buffer[32*1024];
    int done = 0;
    
    if (0 != (err = can_open(user_mailheaders,"r"))) {
	DPRINT(Debug,5,(&Debug, "%s not available: %s (errno=%d)\n",
			user_mailheaders,strerror(err),err));
	return;
    }

    F = fopen(user_mailheaders,"r");
    if (!F) {
	/* In than point opening should have succeed */
	int err = errno;
	lib_error(FRM("%s: %s"),user_mailheaders,err);
	
	return;
    }

    while (0 < read_header_line(F,buffer,sizeof buffer,0)) {

	char *k;
	char * value1 = NULL;

	if ('#' == buffer[0]) {
	    DPRINT(Debug,8,(&Debug,
			    "import_mailheaders: Skipping %s\n",
			    buffer));
	    continue;
	}
	
	value1 = eval_backquote(buffer);

	if (!value1) {
	    DPRINT(Debug,8,(&Debug,
			    "import_mailheaders: Failed %s\n",
			    buffer));
	    continue;
	}
	    
	k = strchr(value1,':');
	if (k) {
	    header_ptr header_name;
	    
	    /* Strip whitespace before ':' */
	    char * k1 = k;
	    
	    while(k1 > value1 && 
		  whitespace(*(k1-1))) {
		k1 --;
		*k1 = '\0';
	    }
	    
	    *k = '\0';
	    k++;
	    
	    /* Skip whitespace after ':' */
	    while (whitespace(*k))
		k++;
	
	    header_name = find_header(value1,1);
	    
	    if (header_name) {
		switch (import_mode) {
		    
		case import_mh_returnpath:
		    
		    if (0 != istrcmp(value1,
				     "Return-Path")) {
			
			DPRINT(Debug,8,(&Debug,
					"import_mailheaders: Skipping %s\n",
					value1));
			break;
		    }
		    
		    /* FALLTHRU */
		    
		case import_mh_normal:
		    if (!add_to_mailing_header(headers,header_name,k,
					       !allow_no_hdrencoding,
					       system_charset,0)) {
			
			if (strlen(k) < 20)
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeFailImportElmHeader,
					      "Failed to import \"%s: %s\" from elmheaders."),
				      value1,k);
			else
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeFailImportElmHeader1,
					      "Failed to import %s from elmheaders."),
				      value1);
			
		    }
		    break;	    	    
		}
		
	    } else {
		DPRINT(Debug,8,(&Debug,
				"import_mailheaders: find_header %s failed\n",
				value1));
	    }

	} else {
	    DPRINT(Debug,8,(&Debug,
			    "import_mailheaders: End of headers: %s\n",
			    value1));
	    
	    done = 1;
	}

	if (value1 != buffer)
	    free(value1);
	if (done) {
	     DPRINT(Debug,8,(&Debug,
			     "import_mailheaders: Quiting on %s\n",
			     buffer));
	     break;
	}
    }

    fclose(F);
}

static const char * is_whitelisted_name P_((const char *domain_name,
					    const char ** whitelist));

static const char * is_whitelisted_name(domain_name,whitelist)
     const char   * domain_name;
     const char ** whitelist;
{
    const char * p;
    int i;

    if (!whitelist)
	return NULL;

    if ('.' == domain_name[0])
	return NULL;
    
    /* These mail domains are valid without verifying */

    for (i = 0; whitelist[i]; i++)
	if (0 == istrcmp(domain_name,whitelist[i]))
	    return whitelist[i];

   /* look names under these domains */

    for (p = domain_name; *p; p++) {
	if ('.' == *p) {

	    for (i = 0; whitelist[i]; i++) {
		
		/* name on whitelist must start with '.' */
		
		if (0 == istrcmp(p,whitelist[i]))
		    return whitelist[i];
	    }
	}
    }

    return NULL;


}

/* Returns domain name if matches to valid-msg-id-domains */
const char * is_whitelisted_msgid_domain(msgid_domain_name)
     const char * msgid_domain_name;
{
    /* Pointer to internal list, do NOT free */
    const char ** whitelist = 
	give_dt_path_as_elems(&valid_msgid_domains,
			      "valid-msg-id-domains");
    

    const char *  ret =
	is_whitelisted_name(msgid_domain_name,
			    whitelist);

    DPRINT(Debug,15,(&Debug,"is_whitelisted_msgid_domain(%Q)=",
		     msgid_domain_name));
    if (ret) {
	DPRINT(Debug,15,(&Debug,"%Q",ret));
    } else {
	DPRINT(Debug,15,(&Debug,"NULL"));
    }
    DPRINT(Debug,15,(&Debug,"\n"));
    
    return ret;
}
    
/* Returns domain name if matches to valid-domains */

const char * is_whitelisted_valid_domain(mail_domain_name)
     const char * mail_domain_name;
{
    /* Pointer to internal list, do NOT free */
    const char ** whitelist = 
	give_dt_path_as_elems(&valid_domains,
			      "valid-domains");

    const char *  ret =
	is_whitelisted_name(mail_domain_name,
			    whitelist);

    DPRINT(Debug,15,(&Debug,"is_whitelisted_valid_domain(%Q)=",
		     mail_domain_name));
    if (ret) {
	DPRINT(Debug,15,(&Debug,"%Q",ret));
    } else {
	DPRINT(Debug,15,(&Debug,"NULL"));
    }
    DPRINT(Debug,15,(&Debug,"\n"));
    
    return ret;
}

 
/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
