static char rcsid[] = "@(#)$Id: address_alias.c,v 1.8 2021/07/13 07:58:36 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.8 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/
#include "def_alias.h"
#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"addr");

#define ALIASES_person_magic	0xF603
#define ALIASES_group_magic	0xF604
#define ADDRESS_ALIAS_magic     0xF605 

struct address_alias {
    unsigned short magic;      /* ADDRESS_ALIAS_magic */

    struct string *comment;

    struct aliases_person {
	unsigned short magic;        /*  ALIASES_person_magic */
	
	struct string *firstn;
	struct string *lastn;

	struct address *address;

    }  * person;

    struct addr_list *list;
    

    struct aliases_group {
	unsigned short magic;     /* ALIASES_group_magic */
	
	struct string *phrase;
	struct alias_vector * aliases;

    }      * group;

    unsigned int    edited :1;  /* Editing routines called */
    
};


static void free_aliases_person P_((struct aliases_person **person));
static void free_aliases_person(person)
     struct aliases_person **person;
{
 
    if (ALIASES_person_magic != (*person)->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"free_aliases_person",
	      "Bad magic number",0);

    if ((*person)->firstn)
	free_string(& ((*person)->firstn));

    if ((*person)->lastn)
	free_string(& ((*person)->lastn));

    if ((*person)->address)
	free_address(& ((*person)->address));

    (*person)->magic = 0;      /* Invalidate */
    free(*person);
    *person = NULL;
}

static void free_aliases_group P_((struct aliases_group **group));
static void free_aliases_group(group)
     struct aliases_group **group;
{
    if (ALIASES_group_magic != (*group)->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"free_aliases_group",
	      "Bad magic number",0);

    if ((*group)->phrase)
	free_string(& ((*group)->phrase));

    if ((*group)->aliases)
	free_alias_vector(& ((*group)->aliases));

    (*group)->magic = 0;   /* Invalidate */
    free(*group);
    *group = NULL;
}

void free_address_alias(alias)
     struct address_alias **alias;
{

    if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"free_address_alias",
	      "Bad magic number",0);

    if ((*alias)->comment)
	free_string(& ((*alias)->comment));
    
    if ((*alias)->person)
	free_aliases_person(& ((*alias)->person));

    if ((*alias)->list)
	free_addr_list(& ((*alias)->list));

    if ((*alias)->group)
	free_aliases_group(& ((*alias)->group));

    (*alias)->magic = 0;      /* Invalidate */
    free(*alias);
    *alias = NULL;
}

static struct address_alias * new_address_alias_1 P_((void));
static struct address_alias * new_address_alias_1()
{
    struct address_alias *ret = safe_zero_alloc(sizeof(*ret));

    ret->magic   = ADDRESS_ALIAS_magic;
    ret->comment = NULL;
    ret->person  = NULL;
    ret->list    = NULL;
    ret->group   = NULL; 

    ret->edited  = 0;
    
    return ret;
}

static struct aliases_person * new_aliases_person P_((void));
static struct aliases_person * new_aliases_person()
{
    struct aliases_person *ret = safe_zero_alloc(sizeof(*ret));

    ret->magic   = ALIASES_person_magic;
    ret->firstn  = NULL;
    ret->lastn   = NULL;
    ret->address = NULL;

    return ret;
}

static struct aliases_group * new_aliases_group P_((void));
static struct aliases_group * new_aliases_group()
{
    struct aliases_group *ret = safe_zero_alloc(sizeof(*ret));

    ret->magic   =  ALIASES_group_magic;
    ret->phrase  = NULL;
    ret->aliases = NULL;

    return ret;
}

static struct aliases_person * dup_aliases_person P_((struct aliases_person *person));
static struct aliases_person * dup_aliases_person(person)
     struct aliases_person *person;
{
    struct aliases_person *ret = new_aliases_person();

    if (ALIASES_person_magic != person->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"dup_aliases_person",
	      "Bad magic number",0);

    if (person->firstn)
	ret->firstn = dup_string(person->firstn);

    if (person->lastn)
	ret->lastn  = dup_string(person->lastn);

    if (person->address)
	ret->address = dup_address(person->address);

    return ret;
}

static struct aliases_group * dup_aliases_group P_((struct aliases_group *group));
static struct aliases_group * dup_aliases_group(group)
     struct aliases_group *group;
{
    struct aliases_group * ret = new_aliases_group();

    if (ALIASES_group_magic != group->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"dup_aliases_group",
	      "Bad magic number",0);

    if (group->phrase)
	ret->phrase = dup_string(group->phrase);

    if (group->aliases)
	ret->aliases = dup_alias_vector(group->aliases);

    return ret;
}

struct address_alias * dup_address_alias(alias)
     const struct address_alias *alias;
{
    struct address_alias *ret = new_address_alias_1();

    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"dup_address_alias",
	      "Bad magic number",0);

    if (alias->comment)
	ret->comment = dup_string(alias->comment);

    if (alias->person)
	ret->person  = dup_aliases_person(alias->person);

    if (alias->list)
	ret->list    = dup_addr_list(alias->list);

    if (alias->group)
	ret->group   = dup_aliases_group(alias->group);

    return ret;
}


struct address_alias * new_person_address_alias(firstn,lastn,comment,address)
     const struct string *firstn;
     const struct string *lastn;
     const struct string *comment;
     const struct address *address;
{
    struct address_alias *ret = new_address_alias_1();

    ret->person = new_aliases_person();

    if (firstn)
	ret->person->firstn = dup_string(firstn);
    if (lastn)
	ret->person->lastn  = dup_string(lastn);

    ret->person->address = dup_address(address);

    if (comment)
	ret->comment = dup_string(comment);

    return ret;
}

struct address_alias * new_list_address_alias(comment,list)
     const struct string * comment;
     const struct addr_list *list;
{
    struct address_alias *ret = new_address_alias_1();
	
    if (comment)
	ret->comment = dup_string(comment);
    
    ret->list = dup_addr_list(list);

    return ret;
}

struct address_alias * new_group_address_alias(phrase,comment,aliases)
     const struct string *phrase;
     const struct string *comment;
     const struct alias_vector * aliases;
{
    struct address_alias *ret = new_address_alias_1();

    ret->group = new_aliases_group();

    if (comment)
	ret->comment = dup_string(comment);

    if (phrase)
	ret->group->phrase = dup_string(phrase);

    ret->group->aliases = dup_alias_vector(aliases);

    return ret;
}

struct address_alias * new_address_alias(firstn,lastn,comment,phrase,
					 list,aliases)
     const struct string *firstn;
     const struct string *lastn;
     const struct string *comment;
     const struct string *phrase;
     const struct addr_list *list;
     const struct alias_vector * aliases;
{
    struct address_alias *ret = new_address_alias_1();

    if (firstn || lastn) {
	ret->person = new_aliases_person();

	if (firstn)
	    ret->person->firstn = dup_string(firstn);
	if (lastn)
	    ret->person->lastn  = dup_string(lastn);

	if (list && 1 == addr_list_item_count(list)) {
	    int group;

	    const struct address *addr =
		addr_list_get_item(list,0,&group);
	    
	    ret->person->address = dup_address(addr);
	}
    }

    if (comment)
	ret->comment = dup_string(comment);
    
    if (list && 
	(addr_list_item_count(list) > 1 ||
	 addr_list_group_count(list) > 0 ||
	 ! ret->person || !ret->person->address))
	ret->list = dup_addr_list(list);

    if (phrase || aliases) {
	ret->group = new_aliases_group();

	if (phrase)
	    ret->group->phrase = dup_string(phrase);
	
	if (aliases)
	    ret->group->aliases = dup_alias_vector(aliases);

    }

    return ret;
}


/* Person alias is also address list alias */
/* Returns 1 if person alias */

int address_alias_get_person(alias,firstn,lastn,address)
     const struct address_alias *alias;
     const struct string **firstn;
     const struct string **lastn;
     const struct address **address;
{
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_get_person",
	      "Bad magic number",0);
   
    *firstn  = NULL;
    *lastn   = NULL;
    *address = NULL;

    if (alias->person) {

	if (ALIASES_person_magic != alias->person->magic)
	    panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_get_person",
		  "Bad person magic number",0);
	    
	*firstn  = alias->person->firstn;
	*lastn   = alias->person->lastn;
	*address = alias->person->address;

	return 1;
    }

    return 0;
}

const struct string * address_alias_get_comment(alias)
     const struct address_alias *alias;
{
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_get_comment",
	      "Bad magic number",0);
   
    return alias->comment;
}

const struct addr_list * address_alias_get_list(alias) 
     const struct address_alias *alias;
{
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_get_list",
	      "Bad magic number",0);
   
    return alias->list;
}


/* Returns 1 if group alias */
int address_alias_get_group(alias,phrase,vector)
     const struct address_alias *alias;
     const struct string **phrase;
     const struct alias_vector **vector;
{
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_get_group",
	      "Bad magic number",0);

    *phrase = NULL;
    *vector = NULL;

    if (alias->group) {
	if (ALIASES_group_magic  != alias->group->magic)
	    panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_get_group",
		  "Bad group magic number",0);

	*phrase = alias->group->phrase;
	*vector = alias->group->aliases;
	return 1;
    }

    return 0;
}

/* Editing routines --------------------------------------------- */

/* Removes old person alias */
void set_person_address_alias(alias,firstn,lastn,address)
     struct address_alias *alias;
     const struct string *firstn;
     const struct string *lastn;
     const struct address *address;
{
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"set_person_address_alias",
	      "Bad magic number",0);
    
    if (! alias->person)
	alias->person =  new_aliases_person();
    else if (ALIASES_person_magic != alias->person->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"set_person_address_alias",
	      "Bad (person) magic number",0);

    if (alias->person->firstn)
	free_string(& (alias->person->firstn));
    if (alias->person->lastn)
	free_string(& (alias->person->lastn));
    if (alias->person->address)
	free_address(& (alias->person->address));
    
    if (firstn)
	alias->person->firstn = dup_string(firstn);
    if (lastn)
	alias->person->lastn  = dup_string(lastn);
    
    alias->person->address = dup_address(address);

    alias->edited = 1;
}

/* Removes old list */
void set_list_address_alias(alias,list)
     struct address_alias *alias;
     const struct addr_list *list;
{
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"set_list_address_alias",
	      "Bad magic number",0);

    if (alias->list)
	free_addr_list(& (alias->list));

    alias->list = dup_addr_list(list);

    alias->edited = 1;
}

void set_group_address_alias(alias,phrase,aliases)
     struct address_alias *alias;
     const struct string *phrase;
     const struct alias_vector * aliases;
{
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"set_group_address_alias",
	      "Bad magic number",0);
    
    if (! alias->group)
	alias->group = new_aliases_group();
    else if (ALIASES_group_magic != alias->group->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"set_group_address_alias",
	      "Bad (group) magic number",0);

    if (alias->group->phrase)
	free_string(& alias->group->phrase);
    if (alias->group->aliases)
	free_alias_vector(& (alias->group->aliases));

    if (phrase)
	alias->group->phrase = dup_string(phrase);
    
    alias->group->aliases = dup_alias_vector(aliases);

    alias->edited = 1;
}


void set_alias_comment(alias,comment)
     struct address_alias *alias;
     const struct string *comment;
{
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"set_alias_comment",
	      "Bad magic number",0);

    if (alias->comment)
	free_string(& alias->comment);

    alias->comment = dup_string(comment);

    alias->edited = 1;
}

/* Parser helper routines --------------------------------------- */

int edited_address_alias(alias,reset)
     struct address_alias *alias;
     int reset /* reset edited flag */;
{
    int ret;
    
    if (ADDRESS_ALIAS_magic  != alias->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_comment",
	      "Bad magic number",0);

    ret = alias->edited;

    if (reset)
	alias->edited = 0;
    
    return ret;
}

void address_alias_add_comment(alias,comment,have_error)
     struct address_alias **alias;
     const struct string *comment;
     struct string **have_error;
{
    if (!*alias)
	*alias = new_address_alias_1();
    else if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_comment",
	      "Bad magic number",0);

    *have_error = NULL;

    if ((*alias)->comment) {
	*have_error = format_string(CATGETS(elm_msg_cat, MeSet, 
					    MeAliasDupComment,
					    "Duplicate comment"));
	return;
    }

    
    (*alias)->comment = dup_string(comment);
}

void address_alias_add_firstname(alias,firstname,have_error)
     struct address_alias **alias;
     const struct string *firstname;
     struct string **have_error;
{
    if (!*alias)
	*alias = new_address_alias_1();
    else if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_firstname",
	      "Bad magic number",0);

    if (! (*alias)->person)
	(*alias)->person = new_aliases_person();
    else if (ALIASES_person_magic != (*alias)->person->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_firstname",
	      "Bad person magic number",0);


    *have_error = NULL;

    if ((*alias)->person->firstn) {
	*have_error = format_string(CATGETS(elm_msg_cat, MeSet, 
					    MeAliasDupFirstn,
					    "Duplicate firstname"));
	return;
    }

    (*alias)->person->firstn = dup_string(firstname);
}

void address_alias_add_lastname(alias,lastname,have_error)
     struct address_alias **alias;
     const struct string *lastname;
     struct string **have_error;
{
    if (!*alias)
	*alias = new_address_alias_1();
    else if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_lastname",
	      "Bad magic number",0);

    if (! (*alias)->person)
	(*alias)->person = new_aliases_person();
    else if (ALIASES_person_magic != (*alias)->person->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_lastname",
	      "Bad person magic number",0);


    *have_error = NULL;

    if ((*alias)->person->lastn) {
	*have_error = format_string(CATGETS(elm_msg_cat, MeSet, 
					    MeAliasDupLastn,
					    "Duplicate lastname"));
	return;
    }

    (*alias)->person->lastn = dup_string(lastname);
}

void address_alias_add_address(alias,addr,have_error)
     struct address_alias **alias;
     const struct address *addr;
     struct string **have_error;
{
    if (!*alias)
	*alias = new_address_alias_1();
    else if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_address",
	      "Bad magic number",0);

    if (! (*alias)->person)
	(*alias)->person = new_aliases_person();
    else if (ALIASES_person_magic != (*alias)->person->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_address",
	      "Bad person magic number",0);


    *have_error = NULL;

    if ((*alias)->person->address) {
	*have_error = format_string(CATGETS(elm_msg_cat, MeSet, 
					    MeAliasDupAddr,
					    "Duplicate address"));
	return;
    }

    (*alias)->person->address = dup_address(addr);

}

void address_alias_add_addr_list(alias,addr_list,have_error)
     struct address_alias **alias;
     const struct addr_list *addr_list;
     struct string **have_error;
{
    if (!*alias)
	*alias = new_address_alias_1();
    else if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_addr_list",
	      "Bad magic number",0);
    
    if ((*alias)->list) {
	*have_error = format_string(CATGETS(elm_msg_cat, MeSet, 
					    MeAliasDupAddrList,
					    "Duplicate address list"));
	return;
    }

    (*alias)->list = dup_addr_list(addr_list);
}

void address_alias_add_phrase(alias,phrase,have_error)
     struct address_alias **alias;
     const struct string *phrase;
     struct string **have_error;
{
   if (!*alias)
	*alias = new_address_alias_1();
    else if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_phrase",
	      "Bad magic number",0);

   if (! (*alias)->group)
       (*alias)->group = new_aliases_group();
   else if (ALIASES_group_magic != (*alias)->group->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"dup_aliases_group",
	      "address_alias_add_phrase",0);

   if ((*alias)->group->phrase) {
	*have_error = format_string(CATGETS(elm_msg_cat, MeSet, 
					    MeAliasDupPhrase,
					    "Duplicate phrase"));
	return;
   }

   (*alias)->group->phrase = dup_string(phrase);
}

void address_alias_add_group_start(alias,have_error)
     struct address_alias **alias;
     struct string **have_error;
{
   if (!*alias)
	*alias = new_address_alias_1();
    else if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_group_start",
	      "Bad magic number",0);

   if (! (*alias)->group)
       (*alias)->group = new_aliases_group();
   else if (ALIASES_group_magic != (*alias)->group->magic)
	panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_group_start",
	      "address_alias_add_phrase",0);

   if ((*alias)->group->aliases) {
	*have_error = format_string(CATGETS(elm_msg_cat, MeSet, 
					    MeAliasDupgroup,
					    "Duplicate group"));
	return;

   }

   (*alias)->group->aliases = new_alias_vector();
}

void address_alias_add_group_item(alias,alias_item,have_error)
     struct address_alias **alias;
     const struct string *alias_item;
     struct string **have_error;
{
    /* Assume that address_alias_add_group_start() is called */

     if (ADDRESS_ALIAS_magic  != (*alias)->magic)
	 panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_group_item",	       
	      "Bad magic number",0);

     if (ALIASES_group_magic != (*alias)->group->magic)
	 panic("ALIAS PANIC",__FILE__,__LINE__,"address_alias_add_group_item",
	       "address_alias_add_phrase",0);

     alias_vector_add_alias((*alias)->group->aliases,
			    alias_item);
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
