static char rcsid[] = "@(#)$Id: aliases.c,v 1.21 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 1.21 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/


#include "def_alias.h"
#include "s_me.h"
#include "rc_imp.h"

#include "commentfile.h"
#include "filelock.h"

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
extern int errno;
#endif

DEBUG_VAR(Debug,__FILE__,"alias");

#define ALIASES_NODE_DEF_magic	0xF611

struct aliases_node_def {
    unsigned short magic;	      /* ALIASES_NODE_DEF_magic */

    unsigned int     old : 1;          /* == old entry on disk */    
};

#define ALIASES_map_magic	0xF602
#define ALIASES_node_magic      0xF606

enum give_aliases_mode {
    ga_mode_search = 0,
    ga_mode_create,
    ga_mode_append,
    ga_mode_remove
};


static struct aliases_map_node {
    unsigned short            magic;    /* ALIASES_node_magic */

    int                      refcount;  
    
    struct string_sort       * alias_name;
    struct string            * alias_key;  /* copy of alias_name */;
    struct address_alias     * alias;

    int idx;                   /* vector_idx */

    unsigned int changed : 1;          /* == status not on disk */
    unsigned int     old : 1;          /* == old entry on disk  */
    unsigned int  need_on_disk : 1;    /* == entry need to be disk on current read */
    unsigned int  delete : 1;          /* Delete alias when updating  */
    
    /* Increments refcount */
} * give_aliases_map_node P_((struct sortlist     ** aliases_map,
			      struct string        * alias_name /* shared */,
			      enum give_aliases_mode create_flag,
			      size_t               * idx /* for append mode */,
			      int                  * append_need_rewrite,
			      const struct aliases_node_def * def1));


static void free_aliases_map_node P_((struct aliases_map_node **node));

static struct aliases_map_node * malloc_aliases_map_node P_((struct string_sort  * alias_name,
							     const struct aliases_node_def * def));
struct aliases_map_node * malloc_aliases_map_node(alias_name,def)
     struct string_sort            * alias_name;
     const struct aliases_node_def * def;
{
    struct aliases_map_node * ret = safe_zero_alloc(sizeof(*ret));

    
    ret->alias_name = alias_name;
    inc_string_sort_refcount(ret->alias_name);

    /* Increments refcount */
    ret->alias_key  = give_string_from_string_sort(ret->alias_name);
    
    ret->alias      = NULL;
    ret->idx        = -1;

    ret->changed = 0;
    ret->old     = 0;
    ret->delete  = 0;
    ret->need_on_disk = 0;

    if (def) {
	if (ALIASES_NODE_DEF_magic != def->magic)
	    panic("ALIASES PANIC",__FILE__,__LINE__,
		  "malloc_aliases_map_node",
		  "Bad magic number (aliases_node_def)",0);

	ret->old    = def->old;
    }
    
    ret->magic = ALIASES_node_magic;

    ret->refcount = 1;

    
    return ret;
}

static void  inc_aliases_map_node_refcount P_((struct aliases_map_node *node));
static void  inc_aliases_map_node_refcount(node)
     struct aliases_map_node *node;
{
    if (ALIASES_node_magic != node->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,
	      "inc_aliases_map_node_refcount",
              "Bad magic number",0);

    node->refcount++;
}


S_(alloc_sort_item_f alloc_aliases_item)
static void alloc_aliases_item P_((union sort_item      * res,
				const union sort_key   key,
				const union sort_item_default def
				));
static void alloc_aliases_item(res,key,def)
     union sort_item      * res;
     const union sort_key   key;
     const union sort_item_default def;  
{
    res->aliases = malloc_aliases_map_node(key.string_sort,def.aliases);
}


S_(free_sort_item_f free_aliases_item)
static void free_aliases_item P_((union sort_item      * ptr)); /* Decrements refcount */
static void free_aliases_item(ptr)
     union sort_item      * ptr; /* Decrements refcount */
{    
    free_aliases_map_node( & ( ptr->aliases ));
}

S_(inc_sort_item_refcount_f inc_aliases_item_refcount)
static void inc_aliases_item_refcount P_((union sort_item item));
static void inc_aliases_item_refcount(item)
     union sort_item item;
{
    inc_aliases_map_node_refcount(item.aliases);
}

S_(sort_item_debug_name_f aliases_item_name)
static struct string * aliases_item_name   P_((const union sort_item item));
static struct string * aliases_item_name (item)
     const union sort_item item;
{

    struct aliases_map_node * node = item.aliases;

    if (ALIASES_node_magic != node->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_item_name",
              "Bad magic number",0);


    /* give_string_from_string_sort() returns original string with incremented refcount 

       That is more efficient, than
       dup_string(node->alias_key)
    */
    
    return give_string_from_string_sort(node->alias_name);
}

S_(compare_sort_key_to_item_f aliases_key_cmp_item)
static int aliases_key_cmp_item P_((const union sort_key key,
				    const union sort_item item));
static int aliases_key_cmp_item(key,item)
     const union sort_key key;
     const union sort_item item;
{
    struct aliases_map_node * node = item.aliases;

    if (ALIASES_node_magic != node->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_key_cmp_item",
              "Bad magic number",0);
	   
    return string_sort_cmp(key.string_sort,node->alias_name);
}

S_(sort_key_debug_name_f aliases_key_name)
static struct string * aliases_key_name P_((const union sort_key key));
static struct string * aliases_key_name(key)
     const union sort_key key;
{

    /* give_string_from_string_sort() returns original string with incremented refcount */
    
    return give_string_from_string_sort(key.string_sort);
}


static struct sort_operation aliases_map_operation = {
    SORT_OPERATION_magic,
    alloc_aliases_item,
    free_aliases_item,
    inc_aliases_item_refcount,
    aliases_item_name,
    aliases_key_cmp_item,

    /* KEY operations */
    aliases_key_name
};



/* Increments refcount */
static struct aliases_map_node * give_aliases_map_node(aliases_map,alias_name,create_flag,
						       idx,append_need_rewrite,def1)
     struct sortlist              ** aliases_map;
     struct string                 * alias_name /* shared */;
     enum give_aliases_mode          create_flag;
     size_t                        * idx;
     int                           * append_need_rewrite;
     const struct aliases_node_def * def1;

{
    struct aliases_map_node * ret = NULL;
    union sort_key            key;
    union sort_item           res;
    union sort_item_default   def;
    size_t                    res_idx = 0;      
    enum sort_list_search_op  op =  sort_list_search_normal;
    struct string_sort      * search_key = NULL;

    DPRINT(Debug,14,(&Debug,
		     "give_aliases_map_node: alias_name=\"%S\"\n",
		     alias_name));

    switch (create_flag) {
    case ga_mode_search:
	op =  sort_list_search_normal;
	break;
    case ga_mode_remove:
	op = sort_list_search_remove;
	break;
    case ga_mode_append:
	if (idx) {
	    res_idx = *idx;
	    op =   sort_list_insert_hint;
	    DPRINT(Debug,14,(&Debug,
			     "give_aliases_map_node: insert_hint %zu\n",
			     res_idx));
	    break;
	}
	/* FALLTHRU */
    case ga_mode_create:
	op = sort_list_search_create;
	break;
    }


    if (idx)
	*idx = 0;

    if (!*aliases_map) {

	if (ga_mode_search == create_flag ||
	    ga_mode_remove == create_flag
	    ) {
	    DPRINT(Debug,9,(&Debug,
			    "give_aliases_map_node: \"%S\" not found (no aliases_map)\n",
			    alias_name));

	    return NULL;
	}
	
	DPRINT(Debug,14,(&Debug,
			 "give_aliases_map_node: Creating aliases_map\n"));

	*aliases_map = alloc_sort_list(&aliases_map_operation,
					1 /* prealloc one item */);
    }

    /* Shares alias_name and increments refcount */
    search_key      = new_string_sort(alias_name);
    
    key.string_sort = search_key;
    def.aliases     = def1;
    res.dummy       = NULL;

    /* Increments refcount, returns 1 if found or created */
    if (search_sort_list_item(*aliases_map,op,
			      key,def,&res,&res_idx,
			      append_need_rewrite)) {
	
	DPRINT(Debug,14,(&Debug,
			 "give_aliases_map_node: search_sort_list_item found or created item #%zu\n",
			 res_idx));

	if (idx)
	    *idx  = res_idx;

	ret = res.aliases;

    } else {
	switch (create_flag) {
	case ga_mode_search:
	case ga_mode_remove:
	    DPRINT(Debug,9,(&Debug,
			    "give_aliases_map_node: \"%S\" not found\n",
			    alias_name));
	    break;
	case ga_mode_create:
	case ga_mode_append:
	    
	    DPRINT(Debug,14,(&Debug,
			     "give_aliases_map_node: search_sort_list_item  failed to create entry\n"));
	    ret =  malloc_aliases_map_node(search_key,def1);
	    break;
	}
    }

    /* Decrement refcount */
    free_string_sort(&search_key);
    
    return ret;
}

static void free_aliases_map_node(node)
     struct aliases_map_node **node;
{
    if ((*node)->magic != ALIASES_node_magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"free_aliases_map_node",
              "Bad magic number",0);

    if ((*node)->refcount < 1)
	panic("ALIASES PANIC",__FILE__,__LINE__,"free_aliases_map_node",
              "Bad refcount",0);

    (*node)->refcount--;

    if ((*node)->refcount > 0) {

	*node = NULL;
	return;
    }

    if (-1 != (*node)->idx)
	panic("ALIASES PANIC",__FILE__,__LINE__,"free_aliases_map_node",
              "Idx not reset",0);
	
    if ((*node)->alias_name)
	free_string_sort(& ((*node)->alias_name));

    if ((*node)->alias_key)
	free_string(& ((*node)->alias_key));
    
    if ((*node)->alias)
	free_address_alias(& ((*node)->alias));
    
    (*node)->magic = 0; /* Invalidate */
    free(*node);
    *node = NULL;
    
}

struct aliases_map {
    unsigned short magic;        /*  ALIASES_map_magic */

    struct sortlist    * root;

    /* XXXX Is this needed ? */
    struct aliases_map_node ** vector;
    int                        vector_len;
};


void free_aliases_map(map)
     struct aliases_map **map;
{

    if (ALIASES_map_magic != (*map)->magic)
	panic("URL PANIC",__FILE__,__LINE__,"free_aliases_map",
	      "Bad magic number",0);

    if ((*map)->root)
	free_sort_list(& ((*map)->root));

    
    if ((*map)->vector) {
	int i;
	
	for (i = 0; i < (*map)->vector_len; i++) {
	    if ((*map)->vector[i]) {
		if ((*map)->vector[i]->magic != ALIASES_node_magic)
		    panic("ALIASES PANIC",__FILE__,__LINE__,"free_aliases_map",
			  "Bad magic number aliases_map_node()",0);

		if (i != (*map)->vector[i]->idx)
		    panic("ALIASES PANIC",__FILE__,__LINE__,"free_aliases_map",
			  "Bad idx",0);
		(*map)->vector[i]->idx = -1;
		free_aliases_map_node(& ((*map)->vector[i]));
	    }	    
	}
	
        free((*map)->vector);
        (*map)->vector = NULL;
    }
    (*map)->vector_len = 0;

    (*map)->magic = 0;      /* Invalidate */
    free(*map);
    *map = NULL;
}


struct aliases_map * new_aliases_map()
{
    struct aliases_map *ret = safe_zero_alloc(sizeof(*ret));

    ret->magic         = ALIASES_map_magic;

    ret->root          = NULL;

    /* XXXX Is this needed ? */
    ret->vector     = NULL;
    ret->vector_len = 0;

    return ret;
}


/*  Format:
 *
 *  [alias]         Alias name
 *  Comment:        Comment text
 *
 *  Firstname:     name
 *  Lastname:      name
 *  Address:       One address
 *
 *  List:          addr-list

 *  Phrase:        phrase text
 *  Group:         alias-list (',' separated)
 *
 */


#define ALIAS_TOKEN   	"[alias]"
#define COMMENT_TOKEN	"Comment"
#define FIRSTNAME_TOKEN	"Firstname"
#define LASTNAME_TOKEN  "Lastname"
#define ADDRESS_TOKEN 	"Address"
#define LIST_TOKEN	"List"
#define PHRASE_TOKEN    "Phrase"
#define GROUP_TOKEN     "Group"


static int read_aliases_map  P_((const char              * filename,
				 FILE                    * f,
				 struct sortlist        ** aliases_map,
				 struct aliases_node_def * def,
				 int                     * errors,
				 charset_t               * fileset,
				 struct editor_propline ** propline,
				 int                     * suggest_rewrite,
				 int                     * need_update));
static int read_aliases_map(filename,f,aliases_map,def,
			    errors,fileset,propline,suggest_rewrite,
			    need_update)
     const char              * filename;
     FILE                    * f;
     struct sortlist        ** aliases_map;
     struct aliases_node_def * def;
     int                     * errors;
     charset_t               * fileset;
     struct editor_propline ** propline;
     int                     * suggest_rewrite;
     int                     * need_update  /* sets 1 if * aliases_map was updated */;
{
    charset_t  cs  = system_charset;    /* Charset of file */
    int lineno = 0;

    struct aliases_map_node * current_node = NULL;
    int alias_count = 0;

    char * buf = NULL;
    char *ptr;
    
    size_t idx = 0;  /* append mode position */
    size_t map_len = 0;
    
    if (fileset && *fileset) {
	const char * cs_name = get_charset_MIME_name(*fileset);
	
	cs = *fileset;

	if (cs_name) {
	    DPRINT(Debug,10,(&Debug, 
			    "read_aliases_map: %s: default charset: %s\n",
			    filename,cs_name));
	}
    }

    /* Look for editor property line */

    if (propline)  {
	enum editor_propline_v propline_mode = 
	    give_dt_enumerate_as_int(&editor_al_propline);
	
	/* Nothing detected yet */
	
	if (*propline)
	    free_editor_propline(propline);

	if (propline_mode != editor_propline_ignore) {
	    int i;
	    
	    for (i = 0; i < 3 && !*propline; i++) {
		int l1;

		int c = fgetc(f);

		if (EOF == c)
		    goto propline_failure;
		if ('\n' == c)
		    continue;

		ungetc(c,f);

		if ('#' != c)
		    break;

		l1 = malloc_gets(&buf,LONG_STRING,f);
		if (l1 > 1) {

		    if ('#' != buf[0])
			goto propline_failure;
		    
		    *propline =
			detect_editor_propline(propline_mode,
					       buf,l1,filename,i,&cs);
					       
		} else { /* something wrong? */
		propline_failure:

		    rewind(f);
		    break;
		}
	    }
	}	
    }

    if (*aliases_map) {
	size_t i;
	size_t old_count = 0;

	map_len = sort_list_len(*aliases_map);

	
	DPRINT(Debug,14,(&Debug,
			 "read_aliases_map: %zu items on aliases_map already\n",
			 map_len));

	for (i = 0; i < map_len; i++) {

	    union sort_item res;

	    
	    res.aliases = NULL;
	    
	    /* Increments refcount */
	    get_sort_list_item(*aliases_map,sort_list_get_normal,i,&res);
	    
	    if (res.aliases) {
		
		if (res.aliases->magic != ALIASES_node_magic)
		    panic("ALIASES PANIC",__FILE__,__LINE__,"read_aliases_map",
			  "Bad magic number (aliases_map_node)",0);

		DPRINT(Debug,14,(&Debug,
				 "read_aliases_map: entry %zu = \"%S\"",
				 i,res.aliases->alias_key));

		if (res.aliases->delete) {
		    DPRINT(Debug,14,(&Debug,", marked for delete"));
		} else if (res.aliases->old) {
		    res.aliases->need_on_disk = 1;
		    DPRINT(Debug,14,(&Debug,", old -- marked need_on_disk"));
		    old_count++;
		} else if (res.aliases->need_on_disk) {
		    res.aliases->need_on_disk = 0;
		    DPRINT(Debug,14,(&Debug,", clearing need_on_disk"));
		}

		if (res.aliases->changed) {
		    DPRINT(Debug,14,(&Debug,", changed"));
		}

		DPRINT(Debug,14,(&Debug,"\n"));

		free_aliases_map_node(& res.aliases);

	    }
	}

	if (old_count) {
	    DPRINT(Debug,14,(&Debug,
			     "read_aliases_map: %zu old items marked need_on_disk\n",
			     old_count));
	}

	if (need_update && * need_update) {
	    DPRINT(Debug,14,(&Debug,
			     "read_aliases_map: need_update is set\n"));
	}
	
    } else {
	size_t count = 0;
	size_t pos = ftell(f);
	int c;
	int last_c = '\n';
	enum syscall_status r;

	if (need_update) {
	    if (* need_update) {
		DPRINT(Debug,14,(&Debug,
				 "read_aliases_map: need_update is set\n"));
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "read_aliases_map: Setting need_update\n"));
		* need_update = 1;
	    }
	}
	    
	DPRINT(Debug,14,(&Debug,
			 "read_aliases_map: Creating aliases_map, counting tag lines from offset %ld at %s\n",
			 pos,filename));
	
	while (EOF != (c = fgetc(f))) {
	    
	    /* Rough guess about [alias] lines */
	    
	    if (last_c == '\n' &&
		c == '[')
		count++;
	    last_c = c;
	}

        r = fseek(f,pos,SEEK_SET);
	switch (r) {
	    int err UNUSED_VAROK;
	    
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			 "read_aliases_map: Reading again from offset %ld at %s\n",
			 pos,filename));
	break;
	case syscall_error:
	    err = errno;

	    DPRINT(Debug,14,(&Debug,
			     "read_aliases_map: Failed to seek to offset %ld at %s: %s\n",
			     pos,filename,strerror(err)));
		   
	   	    
	    return 0;	
	}

	DPRINT(Debug,14,(&Debug,
			 "read_aliases_map: Creating aliases_map,  count=%zu tags on %s\n",
			 count,filename));
	
	*aliases_map =  alloc_sort_list(&aliases_map_operation,count);
    }

    while(!feof(f) && !ferror(f)) {
        int l1 = malloc_gets(&buf,32000,f);
	char * c = buf;
	
	if (-1 == l1) {
            lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLineNo,
                              "%s: %d: Too long line: %.30s..."),
                      filename,lineno,buf ? buf : "???");
            (*errors) ++;
            goto OUT;
	    
        } else if (-2 == l1) {
            int err = errno;
	    
            lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
			      "%.30s: Reading error: %s"),
                      filename,strerror(err));
            (*errors) ++;
	    
            goto OUT;
        } 

	lineno++;

	if (0 == l1) {
	    /* There is empty line before different kind alias types */
            continue;
	}

	l1 = trim_whitespace_from_end(buf,l1);
	
	if (read_charset_tag(buf,l1,filename,lineno,errors,&cs,NULL))
	    continue;
	
        while (*c && whitespace ( *c)) /* skip leading whitespace */
            c++;
        if (*c == '#') /* Skip comments */
            continue;
        if (! *c) {
	    /* Empty line exists between different parts of alias defination ...*/
	    
            continue;
	}

	/* Tag line */
	if ('[' == buf[0]) {
	    
	    if (0 == strncmp(ALIAS_TOKEN,buf,strlen(ALIAS_TOKEN))) {
		char * x = buf + strlen(ALIAS_TOKEN);
		struct string * alias_name = NULL;
		
		while (*x && whitespace ( *x)) /* skip leading whitespace */
		    x++;
		
		if (!*x) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadLineNo,
				      "%s: %d: Bad line: %.30s..."),
			      filename,lineno,buf);
		    (*errors) ++;
		    continue;
		}
		
		if (current_node)
		    free_aliases_map_node(& current_node);
		
				/* Allow mime encoding used on here .... */
		alias_name = hdr_to_string(HDR_PHRASE,x,cs,1);

		/* Increments refcount */
		current_node =
		    give_aliases_map_node(aliases_map,alias_name,
					  ga_mode_append,&idx,
					  suggest_rewrite,def);

		
		if (ALIASES_node_magic != current_node->magic)
		    panic("ALIASES PANIC",__FILE__,__LINE__,"read_aliases_map",
			  "Bad magic number (aliases_map_node)",0);

		
		DPRINT(Debug,14,(&Debug,
				 "read_aliases_map:  Alias %S idx #%zu%s",
				 alias_name,
				 idx,
				 current_node->changed ? ", changed" : ""));

		if (current_node->need_on_disk) {
		    DPRINT(Debug,14,(&Debug,", seen (clearing need_on_disk)"));
		    current_node->need_on_disk = 0;
		} 
		DPRINT(Debug,14,(&Debug,"\n"));
		
		
		if (! current_node->changed) {
		    /* Remove current data */

		    if (current_node->alias) {

			if (edited_address_alias(current_node->alias,0)) {
			    DPRINT(Debug,14,(&Debug,
					     "read_aliases_map:  Alias %S idx #%zu marking changed\n",
					     alias_name,
					     idx));
					     
			    current_node->changed = 1;
			} else {
			    free_address_alias(& (current_node->alias));

			    /* Assuming that that alias data was changed */
			    
			    if (need_update && ! * need_update) {
				DPRINT(Debug,14,(&Debug,
						 "read_aliases_map: Setting need_update\n"));
				* need_update = 1;
			    } 
			}
			
		    } else {

			/* Is new alias data  */
			if (need_update && ! * need_update) {
			    DPRINT(Debug,14,(&Debug,
					     "read_aliases_map: Setting need_update\n"));
			    * need_update = 1;
			}
			
		    }
		}
		    
		free_string(&alias_name);

		alias_count++;
		
		
		idx++;   /* Increment append mode position */
		
	    } else {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasBadTokenNo,
				  "%s: %d: Bad token on line: %.100s..."),
			  filename,lineno,buf);
		
		(*errors) ++;
	    }

	    continue;
	}

	ptr = strchr(buf,':');
	if (!ptr) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasBadLineNo,
			      "%s: %d: Bad line: %.100s..."),
		      filename,lineno,buf);
		
	    (*errors) ++;   
	    continue;
	}
	*ptr = '\0';
	ptr++;
	
	while (whitespace(*ptr))
	    ptr++;

	if (!current_node) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasNameExpectedNo,
			      "%s: %d: Alias name expected: %s: %.100s..."),
		      filename,lineno,buf,ptr);
	    (*errors) ++;
	    continue;
	}

	if (0 == strcmp(buf,COMMENT_TOKEN)) {
	    	    
	    /* Allow mime encoding used on here .... */
	    struct string * comment = hdr_to_string(HDR_PHRASE,ptr,cs,1);

	    if (! current_node->changed) {
	    
		struct string * err = NULL;
	    
		address_alias_add_comment(& current_node->alias, comment, &err);
		
		if (err) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasErrorNo,
				      "%s: %d: %S: %s: %.100s..."),
			      filename,lineno,errno,err,buf,ptr);
		    
		    (*errors) ++;
		    free_string(&err);
		}

	    }
	    
	    free_string(&comment);

	} else if (0 == strcmp(buf,FIRSTNAME_TOKEN)) {
	    
	    /* Allow mime encoding used on here .... */
	    struct string * firstname = hdr_to_string(HDR_PHRASE,ptr,cs,1);

	    if (! current_node->changed) {

		struct string * err = NULL;
		
		address_alias_add_firstname(& current_node->alias, firstname, &err);
		
		if (err) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasErrorNo,
				      "%s: %d: %S: %s: %.100s..."),
			      filename,lineno,err,buf,ptr);
		    
		    (*errors) ++;
		    free_string(&err);
		}

	    }
				  
	    free_string(&firstname);

	} else if (0 == strcmp(buf,LASTNAME_TOKEN)) {
	    
	    /* Allow mime encoding used on here .... */
	    struct string * lastname = hdr_to_string(HDR_PHRASE,ptr,cs,1);

	    if (! current_node->changed) {
	    
		struct string * err = NULL;
		
		address_alias_add_lastname(& current_node->alias, lastname, &err);
		
		if (err) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasErrorNo,
				      "%s: %d: %S: %s: %.100s..."),
			      filename,lineno,err,buf,ptr);
		    
		    (*errors) ++;
		    free_string(&err);
		}

	    }
	    
	    free_string(&lastname);

	} else if (0 == strcmp(buf,ADDRESS_TOKEN)) {
	    
	    /* Allow mime encoding used on here .... */
	    struct address * addr = parse_one_address(ptr,1,cs);
	    
	    if (!addr) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasAddressErrorNo,
				  "%s: %d: Failed to parse address: %s: %s"),
			  filename,lineno,buf,ptr);
		(*errors) ++;
		continue;
	    }

	    if (! current_node->changed) {
		
		struct string * err = NULL;
	    		
		address_alias_add_address(& current_node->alias, addr, &err);
		
		if (err) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasErrorNo,
				      "%s: %d: %S: %s: %.100s..."),
			      filename,lineno,err,buf,ptr);
		    
		    (*errors) ++;
		    free_string(&err);
		}

	    }
				  
	    free_address(&addr);

	} else if (0 == strcmp(buf,LIST_TOKEN)) {
	    
	    /* Allow mime encoding used on here .... */
	    struct addr_list * list = 
		parse_header_address(NULL,ptr,1,cs,NULL);

	    if (!list) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasAddressListErrorNo,
				  "%s: %d: Failed to parse address list: %s: %s"),
			  filename,lineno,buf,ptr);
		(*errors) ++;
		continue;
	    }

	    if (! current_node->changed) {
	    
		struct string * err = NULL;
	    
		address_alias_add_addr_list(& current_node->alias, list, &err);

		if (err) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasErrorNo,
				      "%s: %d: %S: %s: %.100s..."),
			      filename,lineno,err,buf,ptr);
		    
		    (*errors) ++;
		    free_string(&err);
		}

	    }
				  
	    free_addr_list(&list);

	} else if (0 == strcmp(buf,PHRASE_TOKEN)) {
	    
	    /* Allow mime encoding used on here .... */
	    struct string * phrase = hdr_to_string(HDR_PHRASE,ptr,cs,1);

	    if (! current_node->changed) {
	    
		struct string * err = NULL;
	    
		address_alias_add_phrase(& current_node->alias, phrase, &err);
		
		if (err) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasErrorNo,
				      "%s: %d: %S: %s: %.100s..."),
			      filename,lineno,err,buf,ptr);
		    
		    (*errors) ++;
		    free_string(&err);
		}

	    }
				  
	    free_string(&phrase);

	} else if (0 == strcmp(buf,GROUP_TOKEN)) {

	    if (! current_node->changed) {
			    
		char *ptr2, *next = NULL;
		char * temp = safe_strdup(ptr);
		struct string * err = NULL;
	    
		address_alias_add_group_start(& current_node->alias,&err);
		
		if (err) {
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasErrorNo,
				      "%s: %d: %S: %s: %.100s..."),
			      filename,lineno,err,buf,ptr);
		    
		    (*errors) ++;
		    free_string(&err);
		}
		
		
		for (ptr2 = temp; ptr2 && *ptr2; ptr2 = next) {
		    struct string * alias = NULL;
		    struct string * err = NULL;
		    
		    next = qstrpbrk(ptr2,",");
		    if (next) {
			int l2 = next-ptr2;
			*next = '\0';
			next++;
			while (whitespace(*next))
			    next++;
			
			while (l2-- > 0 && whitespace(ptr2[l2]))
			    ptr2[l2] = '\0';
		    } 
		    
		    /* Allow mime encoding used on here .... */
		    alias = hdr_to_string(HDR_PHRASE,ptr2,cs,1);
		    
		    address_alias_add_group_item(& current_node->alias,
						 alias, &err);
		    
		    if (err) {
			lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasError2No,
					  "%s: %d: %S: %S: %s: %.100s..."),
				  filename,lineno,err,alias,buf,ptr);
			
			(*errors) ++;
			free_string(&err);
		    }
		    
		    free_string(&alias);
		}
		
		free(temp);
	    }

	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeAliasBadTokenLineNo,
			      "%s: %d: Bad token line: %s: %.100s..."),
		      filename,lineno,buf,ptr);
		
	    (*errors) ++;   
	}

    }

    if (ferror(f)) {
        int err = errno;
	
        lib_error(CATGETS(elm_msg_cat, MeSet, MeReadError,
                          "%s: Reading error: %s"),
                  filename,strerror(err));
        (*errors) ++;
    }

 OUT:

    if (suggest_rewrite && *suggest_rewrite) {

	DPRINT(Debug,5,(&Debug,
			"read_aliases_map:  Suggesting rewrite of %s\n",
			filename));
	
    }
    
    if (current_node)
	free_aliases_map_node(& current_node);

    if (fileset)
	*fileset = cs;
    
    if (buf) {
	free(buf);
    }

    map_len =  sort_list_len(*aliases_map);
    
    if (feof(f)) {
	size_t i;
	
	for (i = 0; i < map_len; i++) {

	    union sort_item res;
	    
	    res.aliases = NULL;
	    
	    /* Increments refcount */
	    get_sort_list_item(*aliases_map,sort_list_get_normal,i,&res);
	    
	    if (res.aliases) {
		
		if (res.aliases->magic != ALIASES_node_magic)
		    panic("ALIASES PANIC",__FILE__,__LINE__,"read_aliases_map",
			  "Bad magic number (aliases_map_node)",0);

		DPRINT(Debug,14,(&Debug,
				 "read_aliases_map: entry %zu = \"%S\"",
				 i,res.aliases->alias_key));

		if (res.aliases->old) {
		    DPRINT(Debug,14,(&Debug,", old"));
		}

		if (res.aliases->changed) {
		    DPRINT(Debug,14,(&Debug,", changed"));
		}
		
		if (res.aliases->delete) {
		    DPRINT(Debug,14,(&Debug,", marked for delete"));

		}

		if (res.aliases->need_on_disk) {
		    DPRINT(Debug,14,(&Debug,", not seen on disk"));
		    if (! res.aliases->changed &&
			! res.aliases->delete) {
			DPRINT(Debug,14,(&Debug," (marking for delete)"));
			res.aliases->delete = 1;
		    }
		}
		
		DPRINT(Debug,14,(&Debug,"\n"));
		
		free_aliases_map_node(& res.aliases);
	    }
	}
		
    } else {
        DPRINT(Debug,3,(&Debug,
                        "read_aliases_map: %s, All results not readed.\n",
                        filename));
    }


    
    DPRINT(Debug,7,(&Debug,
		    "read_aliases_map: %d aliases, aliases_map len %zu\n",
		    alias_count,map_len));

    
    
    
    return 1;    
}






struct aliases_map * load_aliases_map(filename,errors,
				      fileset, propline,
				      suggest_rewrite,
				      file_information)
     const char              * filename;
     int                     * errors;
     charset_t               * fileset;
     struct editor_propline ** propline;
     int                     * suggest_rewrite;
     struct file_changes     * file_information;
{
    struct aliases_map *ret = NULL;


    FILE * f;
    struct sortlist         * aliases_map = NULL;
    size_t map_len = 0;
    
    int file_fd = -1;
    int file_fd_locked = 0;
    int loop_count = 0;
    int loop_message_printed = 0;

    struct stat file_fd_stat;
    struct stat filename_stat;
    
    int have_file_fd_stat = 0;
    int have_filename_stat = 0;
    
    static struct aliases_node_def item_default = {
	ALIASES_NODE_DEF_magic,

	1 /* old entry on disk */ 
    };


    /* Similar than am_write_changed()  but opens current
       aliases file for read only and uses shared lock */
    while (-1 == file_fd) {
	enum syscall_status  r;
	enum FLOCKING_status r1;
	
	int err = can_open(filename,"r");

	file_fd_locked = 0;
	
	if (err) {
	    DPRINT(Debug,2,(&Debug,"load_aliases_map: %s: %s (can_open)\n",
			    filename,strerror(err)));
	    
	    if (EINTR == err) {
		continue;
	    }
	    
	    if (loop_message_printed) {
		lib_transient(CATGETS(elm_msg_cat, MeSet,
				      MeAliasesWaitingFail,
				      "Waiting aliases file %s to be updated... Fail: %s"),
			      filename,strerror(err));
	    }			      

	    return NULL;
	}

	file_fd = open(filename,O_RDONLY);
	
	if (-1 == file_fd) {
	    err = errno;
	    
	    DPRINT(Debug,10,(&Debug,"load_aliases_map: %s: %s\n",
			     filename,strerror(err)));

	    if (EINTR == err) {
		continue;
	    }
	   
	    if (loop_message_printed) {
		lib_transient(CATGETS(elm_msg_cat, MeSet,
				      MeAliasesWaitingFail,
				      "Waiting aliases file %s to be updated... Fail: %s"),
			      filename,strerror(err));
	    }			      
	    
	    return NULL;
	}

	r = fstat(file_fd,&file_fd_stat);
	switch (r) {
	    char *X;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: fstat %d: %s succeed: dev %lu ino %lu size %ld modified %ld",
			     file_fd,filename,
			     (unsigned long)file_fd_stat.st_dev,
			     (unsigned long)file_fd_stat.st_ino,
			     (long)file_fd_stat.st_size,
			     (long)file_fd_stat.st_mtime));

	    X = ctime(& (file_fd_stat.st_mtime));
	    if (X) { /* ctime() includes newline */
		DPRINT(Debug,14,(&Debug," -- %s",X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }

	    have_file_fd_stat = 1;
	    
	    break;
	case syscall_error:
	    err = errno;

	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: fstat %d: %s: %s\n",
			     file_fd,filename,strerror(err)));

	    if (EINTR == err) {
		goto oops;
	    }
	    
	    /* Just read old file ? */
	    goto quit_locking_loop;
	}

	/* When write_conf() and  am_write_changed()
	   merges changes from
	   <conf_file> and writes <conf_file>.N,
	   it locks  <conf_file> with exclusive (write
	   lock) althouh new file <conf_file>.N
	   is written and then renamed to <conf_file>

	   so try lock <conf_file> and after that 
	   check if file was renamed !!
	*/
	
	r1 = filelock_fd(file_fd,FLOCKING_shared,
			 &conf_merge_locking,
			 filename,
			 FLOCKING_non_block,
			 &err);

	switch (r1) {
	case FLOCKING_FAIL:
	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: filelock_fd %d: %s locking failed: %s\n",
			     file_fd,filename,strerror(err)));
		
		/* Just read old file ? */
		goto quit_locking_loop;
	case FLOCKING_OK:
	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: filelock_fd %d: %s locked\n",
			     file_fd,filename));
	    file_fd_locked = 1;
	    break;
	case FLOCKING_RETRY:
	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: filelock_fd %d: %s locking failed (retry)",
			     file_fd,filename));
	    if (err) {
		DPRINT(Debug,14,(&Debug,": %s",
				 strerror(err)));
	    }
	    DPRINT(Debug,14,(&Debug,"\n"));
	    
	    if (EINTR == err) {
		goto oops;
	    }
	    	    
	    goto wait_to_change;
	case FLOCKING_UNSUPPORTED:
	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: filelock_fd %d: %s locking not supported\n",
			     file_fd,filename));
	    /* Just read old file ? */
	    goto quit_locking_loop;
	}
		       
	r = stat(filename,&filename_stat);
	switch (r) {
	    char *X;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: stat %s succeed: dev %lu ino %lu size %ld modified %ld",
			     filename,
			     (unsigned long)filename_stat.st_dev,
			     (unsigned long)filename_stat.st_ino,
			     (long)filename_stat.st_size,
			     (long)filename_stat.st_mtime));
	    X = ctime(& (filename_stat.st_mtime));
	    if (X) { /* ctime() includes newline */
		DPRINT(Debug,14,(&Debug," -- %s",X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }

	    have_filename_stat = 1;
	    
	    break;
	case syscall_error:
	    err = errno;

	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: stat %s: %s\n",
			     filename,strerror(err)));

	    if (EINTR == err) {
		goto oops;
	    }
	    
	    /* Just read old file ? */
	    goto quit_locking_loop;
	}

	if (!have_filename_stat || !have_file_fd_stat) {

	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: file %s - no stat?\n",
			     filename));

	    goto quit_locking_loop;
	    
	} else if (filename_stat.st_dev   == file_fd_stat.st_dev  &&
		   filename_stat.st_ino   == file_fd_stat.st_ino  &&
		   filename_stat.st_size  == file_fd_stat.st_size &&
		   filename_stat.st_mtime == file_fd_stat.st_mtime) {

	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: file %s not changed (since open)\n",
			     filename));

	    if (loop_message_printed) {
		lib_transient(CATGETS(elm_msg_cat, MeSet,
				      MeAliasesWaitingOK,
				      "Waiting aliases file %s to be updated... OK"),
			      filename);
		loop_message_printed = 0;
	    }
	    
	    goto quit_locking_loop;
			
	} else {
	    int wait_it;
	    
	    DPRINT(Debug,14,(&Debug,
			     "load_aliases_map: file %s changed after open\n",
			     filename));

	oops:
	    wait_it = 0;
	    
	    if (0) {
	    wait_to_change:
		wait_it = 1;
	    }
	    
	    if (loop_count++ > 10) {
		DPRINT(Debug,14,(&Debug,
				 "load_aliases_map: file %s - try #%d -- quiting\n",
				 filename,loop_count));
		goto quit_locking_loop;
	    }
	    	    
	    if (wait_it) {
		DPRINT(Debug,14,(&Debug,
				 "load_aliases_map: file %s - try #%d -- waiting\n",
				 filename,loop_count));
	    
	    
		if (!loop_message_printed) {
		    lib_transient(CATGETS(elm_msg_cat, MeSet,
					  MeAliasesWaiting,
					  "Waiting aliases file %s to be updated..."),
				  filename);
		    loop_message_printed = 1;
		    
		}
		
		if (POLL_method)
		    wait_for_timeout(5);
		else
		    sleep(5);

	    } else {
		DPRINT(Debug,14,(&Debug,
				 "load_aliases_map: file %s - try #%d -- looping\n",
				 filename,loop_count));
	    }
	    
	    if (file_fd_locked) { /* Just ignore error --
				     close() should release this anyway
				  */
		filelock_fd(file_fd,FLOCKING_release,&conf_merge_locking,
			    filename,FLOCKING_non_block,NULL);
	    }
	    
	    close(file_fd);  /* Result ignored */
	    file_fd = -1;	    
	}
    }

	    
 quit_locking_loop:

    if (loop_message_printed) {
	lib_transient(CATGETS(elm_msg_cat, MeSet,
			      MeAliasesWaitingError,
			      "Waiting aliases file %s to be updated... Error"),
		      filename);
    }			      

    if (file_information) {
	if (FILE_CHANGES_magic != file_information->magic)
	    panic("ALIASES PANIC",__FILE__,__LINE__,"load_aliases_map",
		  "Bad magic number (file_changes)",0);

	if (sizeof (*file_information) != file_information->self_size) {
	    
	    DPRINT(Debug,1,(&Debug,
			    "load_aliases_map: size mismatch file_changes size %zu != self size %zu\n",
			    sizeof (*file_information),file_information->self_size));
	    	    
	} else {
	    stat_to_file_changes(have_file_fd_stat ? &file_fd_stat : NULL,
				 file_information);
	}
    }


    
    f = fdopen(file_fd,"r");
    if (!f) {
 	int err UNUSED_VAROK = errno;
	DPRINT(Debug,10,(&Debug,"load_aliases_map: fdopen %d: %s: %s\n",
			 file_fd,filename,strerror(err)));

	if (file_fd_locked) { /* Just ignore error --
				 close() should release this anyway
			      */
	    filelock_fd(file_fd,FLOCKING_release,&conf_merge_locking,
			filename,FLOCKING_non_block,NULL);
	}
	
	close(file_fd);  /* Result ignored */
	return NULL;
    }

	
    
    if (! read_aliases_map(filename,f,
			   &aliases_map,
			   &item_default,
			   errors,
			   fileset,
			   propline,
			   suggest_rewrite,
			   NULL)) {

	DPRINT(Debug,10,(&Debug,
			 "load_aliases_map: %s: read_aliases_map failed\n",
			 filename));

	if (aliases_map)
	    free_sort_list(&aliases_map);


	if (file_fd_locked) { /* Just ignore error --
				 close() should release this anyway
			      */
	    filelock_fd(file_fd,FLOCKING_release,&conf_merge_locking,
			filename,FLOCKING_non_block,NULL);
	}

	fclose(f);

	return NULL;
    }

    if (file_fd_locked) { /* Just ignore error --
			     close() should release this anyway
			  */
	filelock_fd(file_fd,FLOCKING_release,&conf_merge_locking,
		    filename,FLOCKING_non_block,NULL);
    }
    
    fclose(f);		
    
    ret = new_aliases_map();

    ret->root        = aliases_map;

    map_len =  sort_list_len(aliases_map);
    if (map_len > 0) {
	int x = 0;
	size_t i;
	int max_len;
       
       DPRINT(Debug,7,(&Debug,
		       "load_aliases_map: aliases_map len %zu\n",
		       map_len));

       ret->vector = safe_calloc(map_len,sizeof (ret->vector[0]));
       max_len = map_len;
       
       for (i = 0; i < map_len; i++) {
	   union sort_item item;

	   item.aliases = NULL;

	   /* increments refcount */
	   get_sort_list_item(aliases_map,sort_list_get_normal,
			      i,&item);
	   
	   if (item.aliases) {

	       if (ALIASES_node_magic != item.aliases->magic)
		   panic("ALIASES PANIC",__FILE__,__LINE__,"load_aliases_map",
			  "Bad magic number (aliases_map_node)",0);

	       
	       if (-1 != item.aliases->idx)
		   panic("ALIASES PANIC",__FILE__,__LINE__,
			 "load_aliases_map",
			 "Bad idx (should be -1)",0);

	       if (x >= max_len)
		   panic("ALIASES PANIC",__FILE__,__LINE__,
			 "load_aliases_map",
			 "Overflow",0);

	       DPRINT(Debug,14,(&Debug,
				"load_aliases_map: %s -- added \"%S\" idx #%zu as map idx #%d\n", 
				filename, item.aliases->alias_key,i,x));

	       /* Refcount incremented on get_sort_list_item() */
	       ret->vector[x] = item.aliases;
	       ret->vector[x]->idx = x;

	       x++;
	   }
       }
       ret->vector_len = x;       
    }

    return ret;
}

static int need_mime_encoding P_((const struct string *S, charset_t fileset));
static int need_mime_encoding(S,fileset)
     const struct string *S; 
     charset_t fileset;
{
    charset_t cs = get_string_type(S);
    int ret = 0;

    if (fileset != cs) 
	ret = 1;
    else {
	uint16 prev = 0x0000;
	int len = string_len(S);
	int x;

	for (x = 0; x < len; x++) {
            uint16 u = give_unicode_from_string(S,x);

	    if ((0x003D /* = */ == prev &&
		 0x003F /* ? */ == u)    ||
		(0x003F /* ? */ == prev &&
		 0x003D /* = */ == u)) {
		ret = 1;
                break;
	    }
	}
    }
    
    return ret;
}

int merge_aliases_map(filename,f,map,errors,fileset,propline,suggest_rewrite,need_update,
		      file_information)
		      const char              * filename;
		      FILE                    * f;
		      struct aliases_map      * map;
		      int                     * errors;
		      charset_t               * fileset;
		      struct editor_propline ** propline;
		      int                     * suggest_rewrite;
		      int                     * need_update  /* sets 1 if map was updated */;
		      struct file_changes     * file_information;
{

    int r = 0;
    size_t map_len = 0;
    
    static struct aliases_node_def item_default = {
	ALIASES_NODE_DEF_magic,

	1 /* old entry on disk */ 
    };

    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,"merge_aliases_map",
	      "Bad magic number",0);

    
    r = read_aliases_map(filename,f,&(map->root),&item_default,
			 errors,fileset,propline,suggest_rewrite,
			 need_update
			 );
    

    
    if (!r) {
	DPRINT(Debug,10,(&Debug,
			 "merge_aliases_map: %s: read_aliases_map failed\n",
			 filename));
    }

    if (file_information) {
	int have_file_fd_stat = 0;
	struct stat file_fd_stat;
	int file_fd = fileno(f);
	
	enum syscall_status  r1 = fstat(file_fd,&file_fd_stat);
	switch (r1) {
	    char *X;
	    int err;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "merge_aliases_map: fstat %d: %s succeed: dev %lu ino %lu size %ld modified %ld",
			     file_fd,filename,
			     (unsigned long)file_fd_stat.st_dev,
			     (unsigned long)file_fd_stat.st_ino,
			     (long)file_fd_stat.st_size,
			     (long)file_fd_stat.st_mtime));
	    
	    X = ctime(& (file_fd_stat.st_mtime));
	    if (X) { /* ctime() includes newline */
		DPRINT(Debug,14,(&Debug," -- %s",X));
	    } else {
		DPRINT(Debug,14,(&Debug,"\n"));
	    }
	    
	    have_file_fd_stat = 1;
	    break;
	case syscall_error:
	    err = errno;

	    DPRINT(Debug,14,(&Debug,
			     "merge_aliases_map: fstat %d: %s: %s\n",
			     file_fd,filename,strerror(err)));
	    break;
	}

	if (FILE_CHANGES_magic != file_information->magic)
	    panic("ALIASES PANIC",__FILE__,__LINE__,"merge_aliases_map",
		  "Bad magic number (file_changes)",0);
	
	if (sizeof (*file_information) != file_information->self_size) {
	    
	    DPRINT(Debug,1,(&Debug,
			    "merge_aliases_map: size mismatch file_changes size %zu != self size %zu\n",
			    sizeof (*file_information),file_information->self_size));
	    
	} else {
	    
	    stat_to_file_changes(have_file_fd_stat ? &file_fd_stat : NULL,
				 file_information);
	}
    }
	
    
    map_len =  sort_list_len(map->root);
    if (map_len > 0) {
	int x       = map->vector_len;
	int max_len = map->vector_len;

	size_t i;
	
	DPRINT(Debug,7,(&Debug,
			"merge_aliases_map: aliases_map len %zu\n",
			map_len));

	if (map->vector_len < map_len) {
	    map->vector = safe_array_realloc(map->vector,
					     map_len,sizeof (map->vector[0]));
	    max_len = map_len;
	}
	    
	
	for (i = 0; i < map_len; i++) {
	    union sort_item item;
	    
	    item.aliases = NULL;
	    
	    /* increments refcount */
	    get_sort_list_item(map->root,sort_list_get_normal,
			       i,&item);
	    
	    if (item.aliases) {

		if (ALIASES_node_magic != item.aliases->magic)
		    panic("ALIASES PANIC",__FILE__,__LINE__,"merge_aliases_map",
			  "Bad magic number (aliases_map_node)",0);
		
		if (item.aliases->idx >= 0) {
	
		    if (item.aliases->idx >= map->vector_len)
			panic("ALIASES PANIC",__FILE__,__LINE__,"merge_aliases_map",
			      "bad index",0); 
		    
		    if (item.aliases != map->vector[item.aliases->idx])
			panic("ALIASES PANIC",__FILE__,__LINE__,"merge_aliases_map",
			      "bad vector",0);        

		    DPRINT(Debug,14,(&Debug,
				     "merge_aliases_map: %s -- \"%S\" idx #%zu is map idx #%d\n",
				     filename, item.aliases->alias_key,i,
				     item.aliases->idx));

		} else {
		    
		    if (-1 != item.aliases->idx)
			panic("ALIASES PANIC",__FILE__,__LINE__,
			      "merge_aliases_map",
			      "Bad idx (should be -1)",0);
		    
		    if (x >= max_len)
			panic("ALIASES PANIC",__FILE__,__LINE__,
			      "merge_aliases_map",
			      "Overflow",0);

		    DPRINT(Debug,14,(&Debug,
				     "merge_aliases_map: %s -- added \"%S\" idx #%zu as map idx #%d\n",				
				     filename, item.aliases->alias_key,i,x));

		    
		    map->vector[x] = item.aliases;
		    map->vector[x]->idx = x;

		    /* count extra reference */
		    inc_aliases_map_node_refcount(map->vector[x]);

		    x++;


		    /* Should not needed */
		    if (need_update && ! * need_update) {
			DPRINT(Debug,14,(&Debug,
					 "merge_aliases_map: Setting need_update\n"));
			* need_update = 1;
		    }		    
		}


		 /* Decrement refcount */
		free_aliases_map_node(& item.aliases);
	    }
	    
	}

	map->vector_len = x;  
    }
    
    return r;

}

int dump_aliases_map(f,map,commentfile,actor,version_buff,
		     fileset,propline,file_information,errno_res)
     FILE                         * f; 
     struct aliases_map           * map;
     FILE                         * commentfile;
     const char                   * actor;
     char                         * version_buff;
     charset_t                      fileset;
     const struct editor_propline * propline;
     struct file_changes          * file_information;
     int                          * errno_res;
{
    int ret = 0;
        
    enum editor_propline_v propline_mode = 
	give_dt_enumerate_as_int(&editor_al_propline);
    const char * sysnam = NULL;

    int have_file_fd_stat = 0;
    struct stat file_fd_stat;

    int r;

    if (errno_res)
	*errno_res = 0;
    
    if (!map)
	return 0;

    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,"dump_aliases_map",
	      "Bad magic number",0);

    if (!fileset) {
	DPRINT(Debug,10,(&Debug,
			 "dump_aliases_map: File charset is not set, using system charset\n"));
	fileset = system_charset;    
    }
    
    sysnam = get_charset_MIME_name(fileset);

    /* Must have first line */
    if (propline_mode != editor_propline_ignore)
	write_editor_propline(propline,propline_mode,
			      f,"# ","\n",fileset);

    if (sysnam) {
	/* Store used character set */

	elm_fprintf(f, 
		    CATGETS(elm_msg_cat,MeSet, MeCharsetExplain,
			    "# Character set used on this file\n"));

	fprintf(f,"@charset = %s\n",sysnam);
    }

    insert_commentfile(f,ELMALIASES_INFO,commentfile,
		       actor,version_buff);

    if (map->root) {
	size_t i;
	size_t len;
	int need_delete = 0;
	
	/* Dump on sort list order !!! */
	
	len = sort_list_len(map->root);

	DPRINT(Debug,14,(&Debug,
			 "dump_aliases_map: %zu aliases\n",
			 len));
	
	
	for (i = 0; i < len; i++) {
	    
	    union sort_item item;
	    struct aliases_map_node * node = NULL;
	    
	    item.aliases = NULL;
	    
	    /* increments refcount */
	    get_sort_list_item(map->root,sort_list_get_normal,i,
			       &item);
	    
	    
	    node = item.aliases;
	    
	    if (!node)
		continue;
	    
	    if (node->magic != ALIASES_node_magic)
		panic("ALIASES PANIC",__FILE__,__LINE__,"dump_aliases_map",
		      "Bad node magic number",0);
	    
	    if (node->delete) {
		DPRINT(Debug,14,(&Debug,
				 "dump_aliases_map: \"%S\" idx #%zu -- marked for deletion -- skipping\n",
				 node->alias_key,i));
		need_delete++;
		
	    } else if (node->alias) {
		
		char * s1 = string_to_hdr(HDR_PHRASE,
					  node->alias_key,
					  fileset, need_mime_encoding(node->alias_key,fileset),
					  NULL,1);
		
		const struct string * comment = 
		    address_alias_get_comment(node->alias);
		
		const struct string *firstn   = NULL;
		const struct string *lastn    = NULL;
		const struct address *address = NULL;
		const struct addr_list *list  = 
		    address_alias_get_list(node->alias);
		const struct string * phrase = NULL;
		const struct alias_vector * group = NULL;
		
		node->changed = 0;   /* Is on disk now */
		node->old     = 1;   /* Need to be on disk */
		
		fprintf(f,"%s\t%s\n",ALIAS_TOKEN,s1); 
		free(s1); s1=NULL;
	       
		if (comment) {
		    
		    char * s1 = string_to_hdr(HDR_PHRASE,comment,
					      fileset, 
					      need_mime_encoding(comment,fileset),
					      NULL,0);
		    
		    fprintf(f,"%s:\t%s\n",
			    COMMENT_TOKEN, s1 ? s1 : "");
		    free(s1); s1=NULL;
		}
		
		if (address_alias_get_person(node->alias,&firstn,&lastn,
					     &address)) {
		    
		    putc('\n',f);
		    
		    if (firstn) {		  
			char * s1 = string_to_hdr(HDR_PHRASE,firstn,
						  fileset,
						  need_mime_encoding(firstn,fileset),
						  NULL,1);
			
			fprintf(f,"%s:\t%s\n",
				FIRSTNAME_TOKEN, s1 ? s1 : "");
			if (s1)
			    free(s1); 
			s1=NULL;
		    }
		    
		    if (lastn) {
			char * s1 = string_to_hdr(HDR_PHRASE,lastn,
						  fileset,
						  need_mime_encoding(lastn,fileset), 
						  NULL,1);
			
			
			fprintf(f,"%s:\t%s\n",
				LASTNAME_TOKEN,s1 ? s1 : "");
			if (s1)
			    free(s1); 
			s1=NULL;
		    }
		    
		    if (address) {
			char * s1 = address_to_str(address,1,fileset);
			
			fprintf(f,"%s:\t%s\n",
				ADDRESS_TOKEN,s1 ? s1 : "");
			if (s1)
			    free(s1); 
			s1=NULL;
		    }		    
		}
		
		if (list) {
		    char * s1 = list_to_str(list,1,fileset);
		    
		    putc('\n',f);
		    fprintf(f,"%s:\t%s\n",
			    LIST_TOKEN,s1 ? s1 : "");
		    if (s1)
			free(s1); 
		    s1=NULL;
		    
		}
		
		if (address_alias_get_group(node->alias,&phrase,&group)) {
		    putc('\n',f);
		    
		    if (phrase) {
			char * s1 = string_to_hdr(HDR_PHRASE,phrase,
						  fileset,
						  need_mime_encoding(phrase,fileset), 
						  NULL,1);
			
			fprintf(f,"%s:\t%s\n",
				PHRASE_TOKEN,s1 ? s1 : "");
			if (s1)
			    free(s1); 
			s1=NULL;
		    }
		    
		    if (group) {
			int len = alias_vector_item_count(group);
			int i;
			
			fprintf(f,"%s:\t",GROUP_TOKEN);
			
			for (i = 0; i < len; i++) {
			    const struct string * alias =
				alias_vector_get_alias(group,i);
			    
			    char * s1 = string_to_hdr(HDR_PHRASE,alias,
						      fileset,
						      need_mime_encoding(alias,fileset),
						      NULL,1);
			    
			    if (i > 0) {
				putc(',',f);
			    }
			    
			    if (s1) {
				fputs(s1,f);
				free(s1); s1=NULL;
			    }
			}		   
			
			putc('\n',f);			
		    }
		}
		
		putc('\n',f);
	    } else {
		DPRINT(Debug,14,(&Debug,
				 "dump_aliases_map: \"%S\" idx #%zu -- no alias data\n",
				 node->alias_key,i));
	    }
	    
	    /* Decrement refcount */
	    free_aliases_map_node(& node);
	}
   
	if (need_delete) {
	    DPRINT(Debug,14,(&Debug,
			     "dump_aliases_map: %d aliases need to be delete (total %zu aliases)\n",
			     need_delete,len));
	    
	    if (len > 0) {
		size_t i = len -1;
		size_t deleted = 0;
		
		while (1) {
		    union sort_item res;
		    
		    res.aliases = NULL;
		    
		    /* Increments refcount */
		    get_sort_list_item(map->root,sort_list_get_normal,i,&res);
		    
		    if (res.aliases) {
			
			if (res.aliases->magic != ALIASES_node_magic)
			    panic("ALIASES PANIC",__FILE__,__LINE__,"dump_aliases_map",
				  "Bad magic number (aliases_map_node)",0);
			
			DPRINT(Debug,14,(&Debug,
					 "dump_aliases_map: entry %zu = \"%S\"",
					 i,res.aliases->alias_key));
			
			if (res.aliases->delete) {
			    DPRINT(Debug,14,(&Debug," -- deleting\n"));
			    
			    free_aliases_map_node(& (res.aliases));
			    
			    /* Increments refcount */
			    get_sort_list_item(map->root,sort_list_get_remove,i,&res);
			    
			    if (res.aliases) {
				
				if (res.aliases->magic != ALIASES_node_magic)
				    panic("ALIASES PANIC",__FILE__,__LINE__,"dump_aliases_map",
					  "Bad magic number (aliases_map_node)",0);
				
				DPRINT(Debug,14,(&Debug,
						 "dump_aliases_map: entry %zu = \"%S\" deleted",
						 i,res.aliases->alias_key));
				
				
			    }
			    
			}
			DPRINT(Debug,14,(&Debug,"\n"));

			if (res.aliases)
			    free_aliases_map_node(& (res.aliases));
		    }
		    if (i > 0)
			i--;
		    else
			break;
		}
		
		if (deleted) {
		    len = sort_list_len(map->root);
		    
		    DPRINT(Debug,14,(&Debug,
				     "dump_aliases_map: %zu entries deleted, %zu entries left\n",deleted,len));
		    
		}	    
	    }		
	}
    }
        
    r = fflush(f);

    switch (r) {
	int err;

    case 0:

	ret = 1;
	
	if (file_information) {
	    int file_fd = fileno(f);
	    
	    enum syscall_status  r1 = fstat(file_fd,&file_fd_stat);
	    switch (r1) {
		char *X;
	    case syscall_success:
		DPRINT(Debug,14,(&Debug,
				 "dump_aliases_map: fstat %d: succeed: dev %lu ino %lu size %ld modified %ld",
				 file_fd,
				 (unsigned long)file_fd_stat.st_dev,
				 (unsigned long)file_fd_stat.st_ino,
				 (long)file_fd_stat.st_size,
				 (long)file_fd_stat.st_mtime));
		
		X = ctime(& (file_fd_stat.st_mtime));
		if (X) { /* ctime() includes newline */
		    DPRINT(Debug,14,(&Debug," -- %s",X));
		} else {
		    DPRINT(Debug,14,(&Debug,"\n"));
		}
		
		have_file_fd_stat = 1;
		break;
	    case syscall_error:
		err = errno;
		
		DPRINT(Debug,14,(&Debug,
				 "dump_aliases_map: fstat %d: %s\n",
				 file_fd,strerror(err)));
		break;
	    }
	}
	break;
    case EOF:
	err = errno;
		
	DPRINT(Debug,14,(&Debug,
			 "dump_aliases_map: fflush: %s\n",
			 strerror(err)));

	if (errno_res)
	    *errno_res = err;

	break;	
    }

	
    if (file_information) {

	if (FILE_CHANGES_magic != file_information->magic)
	    panic("ALIASES PANIC",__FILE__,__LINE__,"dump_aliases_map",
		  "Bad magic number (file_changes)",0);
	
	if (sizeof (*file_information) != file_information->self_size) {	    
	    DPRINT(Debug,1,(&Debug,
			    "dump_aliases_map: size mismatch file_changes size %zu != self size %zu\n",
			    sizeof (*file_information),file_information->self_size));
	} else {
	    stat_to_file_changes(have_file_fd_stat ? &file_fd_stat : NULL,
				 file_information);
	}
    }
    
    DPRINT(Debug,14,(&Debug,
		     "dump_aliases_map=%d",ret));

    if (errno_res && *errno_res) {
	DPRINT(Debug,14,(&Debug,"; error %s",
			 strerror(*errno_res)));
    }
    DPRINT(Debug,14,(&Debug,"\n"));
    
    return ret;
}


const struct address_alias * aliases_map_get_alias(map,idx,key)
     const struct aliases_map *map;
     int idx;
     const struct string **key;
{
    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_get_alias",
	      "Bad magic number",0);

    if (idx < 0 || idx >= map->vector_len)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_get_alias",
	      "Bad index",0);

    if (! map->vector[idx])
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_get_alias",
	      "Bad vector",0);

    if (ALIASES_node_magic != map->vector[idx]->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_get_alias",
	      "Bad node magic number",0);

    if (idx != map->vector[idx]->idx)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_get_alias",
	      "Bad node index number",0);
    
    *key = map->vector[idx]->alias_key;

    return map->vector[idx]->alias;
}

const struct address_alias *  aliases_map_lookup_alias(map,key,idx)
     struct aliases_map *map;
     const struct string *key;
     int *idx;
{
    struct aliases_map_node * node = NULL;
    struct string * alias_name = NULL;

    const struct address_alias * ret = NULL;
    
    
    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_lookup_alias",
	      "Bad magic number",0);

    if (idx)
	*idx = -1;

    if (!map->root)
	return NULL;    /* Not found */	

    alias_name = dup_string(key);  /* Need to be not const -- 
				      possible shared
				   */

    /* Increments refcount */
    node = give_aliases_map_node(&(map->root),alias_name,
				 ga_mode_search,NULL,NULL,NULL);


    free_string(&alias_name);

    if (!node) 
	return NULL;    /* Not found */

    if (node->idx >= 0) {
	
	if (node->idx >= map->vector_len)
	    panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_lookup_alias",
		  "bad index",0); 
			       
	if (node != map->vector[node->idx])
	    panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_lookup_alias",
		  "bad vector",0);        

	if (idx)	
	    *idx = node->idx;

	ret = map->vector[node->idx]->alias;
	
    }

    free_aliases_map_node(&node);
    
    return ret;
}


/* Replaces existing aliases */
void aliases_map_add_alias(map,key,alias)
     struct aliases_map *map;
     const struct string *key;
     const struct address_alias *alias;
{
    struct aliases_map_node * node = NULL;
    struct string * alias_name = dup_string(key);  /* shared */

    static struct aliases_node_def new_item_default = {
	ALIASES_NODE_DEF_magic,

	0 /* Not on disk */ 
    };

    
    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_add_alias",
	      "Bad magic number",0);

    /* Increments refcount */
    node = give_aliases_map_node(& (map->root),alias_name,
				 ga_mode_create,NULL,NULL,
				 &new_item_default);

    if (node->idx < 0) {    /* Need add new alias to list */

	map->vector = safe_array_realloc(map->vector,
					 (map->vector_len+1),
					 sizeof(map->vector[0]));

	map->vector[map->vector_len] = node;
	inc_aliases_map_node_refcount(map->vector[map->vector_len]);
	
	node->idx = map->vector_len;

	DPRINT(Debug,9,(&Debug,
			"aliases_map_add_alias: New alias \"%S\" index %d\n",
			key,node->idx));
	
	map->vector_len++;
	

    } else {

	if (node->idx >= map->vector_len)
	    panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_add_alias",
		  "bad index",0); 
	
	if (node != map->vector[node->idx])
	    panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_add_alias",
		  "bad vector",0);        

	DPRINT(Debug,15,(&Debug,
			"aliases_map_add_alias: Updating alias \"%S\" index %d\n",
			key,node->idx));

    }

    if (node->alias)
	free_address_alias( & (node->alias));

    node->alias = dup_address_alias(alias);
    node->changed = 1;

    if (node->delete) {
	DPRINT(Debug,9,(&Debug,
			"aliases_map_add_alias: Clearing delete flag for alias \"%S\".\n",
			alias_name));
	node->delete = 0;
    }
    
    free_string(&alias_name);
    free_aliases_map_node(&node);
    
    return;
}

int aliases_map_remove_alias(map,key) 
     struct aliases_map *map;
     const struct string *key;
{
    struct aliases_map_node * node = NULL;
    struct string * alias_name = dup_string(key);  /* shared */
    int r = 0;
    size_t sortlist_idx = 0;
    
    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
	      "Bad magic number",0);

    /* Increments refcount */
    node = give_aliases_map_node(& (map->root),
				 alias_name,
				 ga_mode_search  /* do not remove yet */,
				 &sortlist_idx,
				 NULL,NULL);

    free_string(&alias_name);
    
    if (!node) {
	
	return 0;    /* Not found */
    }

    if (ALIASES_node_magic != node->magic)
	panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
	      "Bad magic number (aliases_map_node)",0);

    
    node->delete = 1;  /* mark for deletion */
    
    if (node->alias) {
	if (node->idx < 0)
	    panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
		  "No index for alias",0); 

	free_address_alias( & (node->alias));
    }

    if (node->idx >= 0) {
	int i;

	if (node->idx >= map->vector_len)
	    panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
		  "bad index",0); 
	
	if (node != map->vector[node->idx])
	    panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",		  
		  "bad vector",0);      

	/* Decrement refcount */
	free_aliases_map_node(& (map->vector[node->idx]));
	
	for (i = node->idx+1; i < map->vector_len; i++) {

	    if (ALIASES_node_magic != map->vector[i]->magic)
		panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
		      "Bad node magic number",0);
	    
	    if (i != map->vector[i]->idx)
		panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
		      "Bad node index number",0);	    

	    if (map->vector[i-1])
		panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
		      "Vector index not free",0);
	    
	    map->vector[i-1] = map->vector[i];
	    map->vector[i-1]->idx = i-1;
	    map->vector[i] = NULL;
	}
	
	map->vector_len--;
	node->idx = -1;                /* Node not in use */

	r = 1;
    }

    free_aliases_map_node(& node);
    
    return r;
}

int aliases_map_deleted_alias(map,idx)
     struct aliases_map *map;
      int idx;
{
    struct aliases_map_node * node = NULL;

    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_deleted_alias",
	      "Bad magic number",0);
    
    if (idx < 0 || idx >= map->vector_len)
	panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_deleted_alias",
	      "bad index",0); 
    
    node = map->vector[idx];

    if (ALIASES_node_magic != node->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_deleted_alias",
	      "Bad node magic number",0);
    
    
    if (idx != node->idx) 
	panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_deleted_alias",
	      "bad vector or index",0);    

    return node->delete;
}

void aliases_map_remove_index(map,idx)  
      struct aliases_map *map;
      int idx;
{
    int i;
    struct aliases_map_node * node = NULL;

    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_index",
	      "Bad magic number",0);
    
    if (idx < 0 || idx >= map->vector_len)
	panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_remove_index",
	      "bad index",0); 
    
    node = map->vector[idx];

    if (ALIASES_node_magic != node->magic)
	panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_index",
	      "Bad node magic number",0);
    
    
    if (idx != node->idx) 
	panic("ALIASES PANIC",__FILE__,__LINE__,"aliases_map_remove_index",
	      "bad vector or index",0);    
        
    node->idx = -1;                /* Node not in use */
    node->delete = 1;              /* mark for deletion */
    free_address_alias( & (node->alias));
    
    /* Decrement refcount */
    free_aliases_map_node(& (map->vector[idx]));
    node = NULL;   /* Pointer probably no longer valid */
    
    for (i = idx+1; i < map->vector_len; i++) { 
	
	if (ALIASES_node_magic != map->vector[i]->magic)
	    panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
		  "Bad node magic number",0);
	
	if (i != map->vector[i]->idx)
	    panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_index",
		  "Bad node index number",0);	    

	if (map->vector[i-1])
	    panic("URL PANIC",__FILE__,__LINE__,"aliases_map_remove_alias",
		  "Vector index not free",0);
	
	map->vector[i-1] = map->vector[i];
	map->vector[i-1]->idx = i-1;
	map->vector[i] = NULL;
    }
    map->vector_len--;

}


int aliases_map_item_count(map)
     struct aliases_map *map;
{
    if (ALIASES_map_magic != map->magic)
	panic("URL PANIC",__FILE__,__LINE__,
	      "aliases_map_item_count",
	      "Bad magic number",0);

    return map->vector_len;
} 

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
