static char rcsid[] = "@(#)$Id: con_imap.c,v 2.25 2022/02/20 13:46:55 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.25 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *                  (was hurtta+elm@posti.FMI.FI)  
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"

DEBUG_VAR(Debug,__FILE__,"imap");

#ifdef REMOTE_MBX
#include "ss_imp.h"
#include "mbximap_imp.h"

#include "s_me.h"

/* For INT_MAX, LONG_MAX */
#include <limits.h>

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}

static char *us2s P_((unsigned char *str));
static char *us2s(str)
     unsigned char *str;
{
    return (char *)str;
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

S_(cache_zero_connection cache_zero_imap)
static void  cache_zero_imap P_((struct connection_cache *c));

S_(cache_free_connection cache_free_imap)
static void  cache_free_imap P_((struct connection_cache *c));

S_(cache_open_connection cache_open_imap)
static int  cache_open_imap P_((struct connection_cache *c));

S_(cache_login_connection cache_login_imap)
static int  cache_login_imap P_((struct connection_cache *c,
				 const struct string *password /* May be NULL */));

S_(cache_close_connection cache_close_imap)
static int  cache_close_imap P_((struct connection_cache *c,
				 struct cancel_data  * cd));

S_(cache_folder_from_connection cache_folder_from_imap)
static void cache_folder_from_imap P_((struct connection_cache *c,
				       struct folder_info *f));

S_(cache_browser_from_connection cache_browser_from_imap)
static void cache_browser_from_imap P_((struct connection_cache *c,
					struct folder_browser *d));

S_(cache_verify_server_const cache_verify_IMAP_server)
static  struct string * cache_verify_IMAP_server  P_((void));

S_(cache_verify_cb_connection cache_verify_cb_imap)
static  int cache_verify_cb_imap P_((struct connection_cache *c,
				     struct remote_account *ra,
				     const struct string * Server,
				     const struct string * server));

S_(cache_open_connection_hm cache_open_hm_imap)
static int cache_open_hm_imap P_((struct connection_cache *c,
				  struct browser_passhm *passhm,
				  int  * use_passhm));

struct connection_type IMAP_connection = { 
    CONNECTION_TYPE_magic,
    "IMAP",
    cache_zero_imap,
    cache_free_imap,
    cache_open_imap,
    cache_login_imap,
    cache_close_imap,     
    cache_folder_from_imap,
    cache_browser_from_imap,
    cache_verify_IMAP_server,
    cache_verify_IMAP_server,
    cache_verify_cb_imap,
    cache_open_hm_imap
    
};

static  struct string * cache_verify_IMAP_server()
{
    return format_string(CATGETS(elm_msg_cat, MeSet,MeIMAPserver,
				 "IMAP server"));
}

void set_imap_connection_state(Ch,st)
     struct connection_cache  *Ch; 
     imap_states st; 
{
    Ch->a.imap_con->imap_state = st;
}

/* ------------------------------------------------------------------------ */

static void add_token P_((struct Imap_Token_Buffer *X, struct imap_token T));
static void add_token(X,T)
     struct Imap_Token_Buffer *X; 
     struct imap_token T;
{
    X->tokens = safe_array_realloc(X->tokens,  
				   (X->token_count+1),
				   sizeof (struct imap_token));
    X->tokens[X->token_count++] = T;
}

static void free_tokens P_((struct Imap_Token_Buffer *X));
static void free_tokens(X)
     struct Imap_Token_Buffer *X;
{

    if (X -> tokens) {
	int i;
	
	for (i = 0; i < X -> token_count; i++) {
	    if (X -> tokens[i].str) {
		free (X -> tokens[i].str);
		X -> tokens[i].str = NULL;
	    }
	    X -> tokens[i].imap_token = imap_zero;
	}
	
	free(X -> tokens);
	X -> tokens = NULL;
	X -> token_count = 0;
    }
}



static struct kwd {
    char         *   kw;
    enum token_type  t;
} keywords[] = {
    { "OK",       imap_status_keyword },
    { "NO",       imap_status_keyword },
    { "BAD",      imap_status_keyword },
    { "PREAUTH",  imap_status_keyword },
    { "BYE",      imap_status_keyword }
};

static struct capab {
    char         *    capa;
    long              mask;
} capabilities[] = {
    { "IMAP4rev1",   CAPA_IMAP4rev1 },
    { "UNSELECT",    CAPA_UNSELECT  },
    { "QUOTA",	     CAPA_QUOTA     }
};

struct imap_flag IMAP_flags[] = {
    { "\\Seen",        0,           0,          IMAP_Seen     },
    { "\\Answered",    REPLIED_TO,  0,          IMAP_Answered },
    { "\\Flagged",     0,           S1_FLAGGED, IMAP_Flagged  },
    { "\\Deleted",     DELETED,     0,          IMAP_Deleted  },
    { "\\Draft",       0,           0,          IMAP_Draft  },
    { "\\Recent",      NEW,         0,          IMAP_Recent  },
} ;
int IMAP_flags_count = sizeof IMAP_flags / sizeof (struct imap_flag);

static struct browser_flag {
    char         *   flag;
    int              browser_flag;
} BROWSER_flags[] = {
    { "\\Marked",      BROWSER_MARKED   },
    { "\\Noinferiors", BROWSER_NODIR    },
    { "\\Noselect",    BROWSER_NOFOLDER },
    { "\\Unmarked",    0 },
} ; 

static void add_to_write_buffer P_((struct IMAP_CON * M,
				    char **str));
static void add_to_write_buffer(M,str)
     struct IMAP_CON * M;
     char **str;
{
    int l = strlen(*str);
    
    add_to_Write_Buffer(& (M->write_buffer), str, l);
}

/* rfc1730:
   number          ::= 1*digit
                       ;; Unsigned 32-bit integer
                       ;; (0 <= n < 4,294,967,296)
*/

static uint32 parse_imap_number P_((char *p, int left, char **end, int *error));  
static uint32 parse_imap_number(p,left,end,error)
     char *p;
     int left;
     char **end;
     int *error;
{
    /* strtol and strtoul accepts extra characters */

    uint32 result = 0;
    const uint32 max = 
#ifdef UINT32_MAX
	UINT32_MAX
#else
	4294967295
#endif
	;
    char * p1;

    if (left < 0)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "parse_imap_number",
	      "Bad left value < 0", 0);
    
    if (error)
	*error = 0;
    if (end)
	*end = NULL;

    for (p1 = p; p1 < p + left; p1++) {
	int c = *p1;
	uint32 new_val;
	uint32 char_val;
	if (!c)
	    break;
	if (c < '0' || c > '9')
	    break;
	char_val = c - '0';

	if (char_val > 9)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "parse_imap_number",
		  "Bad char_val value > 9", 0);
	           
	if (result > max / 10) {
	    /* Assume overflow */
	    if (error)
		*error = 1;
	    break;
	}
	new_val = result * 10;

	if (max - char_val < new_val) {
	    /* Assume overflow */
	    if (error)
		*error = 1;
	    break;
	}

	result = new_val + char_val; 
    }

    if (end)
	*end = p1;
    if (p == p1)
	*error = 1;

    return result;
}

static int imap_gen_writable_data P_((struct IMAP_CON * M));
static int imap_gen_writable_data(M)
     struct IMAP_CON * M;
{
    char * t;

    if (M->write_buffer.write_len > 0) {
	/* Have incomplete data ... */
	return 1;
    }

    free_Write_Buffer(& (M->write_buffer));

    if (M->imap_state != IMAP_command)
	return 0;    /* No data available */

    if (M->token_index >= M->write_tokens.token_count)
	return 0;    /* No data available */

    switch(M->token_index) {
    case -2 :        /* NOOP command */
	M->write_buffer.write_buffer = elm_message (FRM("%s NOOP\r\n"),
				       M->current_NOOP_tag);
	M->write_buffer.write_len    = strlen(M->write_buffer.write_buffer);
	return 1;

    case -1:         /* Normal command */
	M->write_buffer.write_buffer = elm_message(FRM("%s "),
				      M->current_tag);
	M->write_buffer.write_len    = strlen(M->write_buffer.write_buffer);
	M->literal_mode = no_literal;
	M->token_index++;
    }

    while (M->token_index < M->write_tokens.token_count) {
	char *s1 = NULL;
	char * space = "";
	unsigned long value;
	
	if (M->token_index > 0 &&
	    M->write_tokens.tokens[M->token_index-1].imap_token != 
	    imap_list_begin)
	    space = " ";
	
	switch(M->write_tokens.tokens[M->token_index].imap_token) {
	case imap_literal:
	    switch(M->literal_mode) {
		char *s;
		int len;		
	    case continue_literal:
		if (M->write_buffer.write_buffer != NULL)
 		    panic("MBX PANIC",__FILE__,__LINE__,
			  "imap_gen_writable_data",
			  "write_buffer not NULL",0);
		M->write_buffer.write_buffer = 
		    M->write_tokens.tokens[M->token_index].str;
		
		if (M->write_tokens.tokens[M->token_index].str_len > (size_t)INT_MAX) {
		    M->write_buffer.write_len   = INT_MAX;
		    
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "imap_gen_writable_data",
			  "string too long",0);
		} else
		    M->write_buffer.write_len    = 
			M->write_tokens.tokens[M->token_index].str_len;

		/* So that it is not free()ed twice */
		M->write_tokens.tokens[M->token_index].str = NULL; 

		M->literal_mode = no_literal;
		M->token_index++;
		continue;       /* Write literal */            
	    case waiting_literal_continue:
		DPRINT(Debug,7,(&Debug, 
				"imap_gen_writable_data=0:  Waiting literal continue\n"));
		return 0;
	    case no_literal:

		if (M->write_tokens.tokens[M->token_index].str_len > (size_t)INT_MAX) {
		    len = INT_MAX;
		    
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "imap_gen_writable_data",
			  "string too long",0);
		} else
		    len = M->write_tokens.tokens[M->token_index].str_len;
		
		s = elm_message(FRM("%s{%d}\r\n"),
				space,
				len);
		add_to_write_buffer(M,&s);  /* May free() s */
		M->literal_mode = waiting_literal_continue;
		return 1;
	    }
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "imap_gen_writable_data",
		  "Bad literal mode",0);
	case imap_number:
	    value = M->write_tokens.tokens[M->token_index].value;
	    
	    s1 = elm_message(FRM("%s%lu"),space,
			     value);
	    M->token_index++;
	    break;
	case imap_list_begin:
	    s1 = elm_message(FRM("%s("),space);
	    M->token_index++;
	    break;
	case imap_list_end:
	    s1 = elm_message(FRM(")"));
	    M->token_index++;
	    break;
	case imap_atom:
	    s1 = elm_message(FRM("%s%s"),space,
			     M->write_tokens.tokens[M->token_index].str);
	    M->token_index++;
	    break;
	case imap_string:
	    /* NOTE:
	     *
	     * 1) We must NOT convert date-time to literal:
	     *        16-Sep-2000 22:52:00 +0300
	     *
	     * 2) Also it is no effective to convert % to literal
	     *
	     */

	    if (strspn(M->write_tokens.tokens[M->token_index].str,
		       " 1234567890%:+-.!*,()[]#_<>qwertyuiopasdfghjklzxcvbnmQWERTYUIOPASDFGHJKLZXCVBNM") < 
		M->write_tokens.tokens[M->token_index].str_len) {
		DPRINT(Debug,50,(&Debug, 
				"imap_gen_writable_data: %d string \"%s\" converting to literal...\n",
				M->token_index,
				M->write_tokens.tokens [M->token_index].str));
		M->write_tokens.tokens[M->token_index].imap_token =
		    imap_literal;
		break;   /* RETRY */
	    }
	    s1 = elm_message(FRM("%s\"%s\""),space,
			     M->write_tokens.tokens[M->token_index].str);
	    M->token_index++;
	    break;
	default:
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "imap_gen_writable_data",
		  "Bad token type",0);
	}

	if (s1)
	    add_to_write_buffer(M,&s1);  /* May free() s1 */
    }

    /* Write end of command (CRLF) */
    t = safe_strdup("\r\n");
    add_to_write_buffer(M,&t);  /* May free() t */

    return 1;  /* have writable data */
}

/* -1 == failure
   0  == continue   (read more data)
   1  == ready
*/
static int tokenize P_((struct IMAP_CON * M));
static int tokenize(M)
     struct IMAP_CON * M;
{
    int idx;
    int len;
    int ret;

    if (M->wanna_literal) {
	struct imap_token T;
	size_t w_literal_len;
	
    try_literal:
	if (M->tokenizer_state != state_list)
	    panic("MBX PANIC",__FILE__,__LINE__,"tokenize",
		  "Bad tokenizer_state (IMAP)",0);

	if ((unsigned long)(M->wanted_literal_len) >= (unsigned long)size_MAX) {
	    DPRINT(Debug,1,(&Debug, 
			    "tokenize=-1 (wanted_literal_len=%lu >= size_MAX=%lz)\n",
			    (unsigned long)(M->wanted_literal_len),size_MAX
			    ));
	    return -1;
	}
	w_literal_len = M->wanted_literal_len;
	
	
	if ((unsigned long)M->read_buffer.read_len < w_literal_len) {
	    DPRINT(Debug,7,(&Debug, 
			    "tokenize=0 (incomplete literal, read_len=%zu < literal len=%zu)\n",
			    M->read_buffer.read_len, w_literal_len
			    ));
	    return 0;
	}
		
	T.imap_token = imap_literal;
	T.str = malloc(w_literal_len+1);
	if (! T.str) {
	    int err = errno;
	    DPRINT(Debug,1,(&Debug, 
			    "tokenize=-1 (literal_len=%zu): can't allocate: %s (errno=%d)\n",
			    w_literal_len,strerror(err),err));

	    return -1;
	}
	
	memcpy(T.str,M->read_buffer.read_buffer,w_literal_len);
	T.str[w_literal_len] = '\0';
	T.str_len = w_literal_len;
	T.value   = 0;
	
	add_token(&(M->read_tokens),T);
	
	cut_line(&(M->read_buffer),w_literal_len);   
	M->wanna_literal = 0;
    }

    idx = 0;
    ret = 1;
    len = find_crlf(&(M->read_buffer),1);
    if (!len) {
	DPRINT(Debug,7,(&Debug, "tokenize=0 (no data)\n"));
	return 0;
    }

    while (idx < len) {	
	struct imap_token T;
	char * p = M->read_buffer.read_buffer +idx;
	int left = len - idx;
	
	if (!*p) {
	    DPRINT(Debug,20,(&Debug, "tokenize: idx=%d   END\n",idx));
	    break;
	}

	T.str_len      = 0;
	T.str          = NULL;
	T.value        = 0;
	T.imap_token   = imap_zero;

	switch (M->tokenizer_state) {
	    char  *n;
	    int i;
	case state_tag:
	    n = memchr(p,' ',left);
	    
	    if (NULL == n || n == p) {
	    fail:
		lib_error(CATGETS(elm_msg_cat, MeSet,MeUnparseableResponse,
				  "Unparseable response from IMAP server"));

		free_Read_Buffer(&(M->read_buffer));		
		DPRINT(Debug,7,(&Debug, "tokenize=-1 (failure)\n"));
		return -1;
	    }
	    *n = '\0';

	    DPRINT(Debug,20,(&Debug, 
			"tokenize: idx=%d  TAG   '%.*s'\n",
			idx,n-p,p));

	    M->tokenizer_state = state_response;	    
	    if (n-p == 1 && p[0] == '+') {
		T.imap_token = imap_continue;
		M->tokenizer_state = state_text;
	    } else if (n-p == 1 && p[0] == '*') {
		T.imap_token = imap_notag;
	    } else {
		T.imap_token = imap_atom;
		T.str = safe_malloc(n-p+1);
		memcpy(T.str,p,n-p+1);
		T.str_len = n-p; 
	    }
	    add_token(&(M->read_tokens),T);

	    idx = (n - M->read_buffer.read_buffer) +1;
	    break;  
	case state_response:
	    if (p[0] >= '0' && p[0] <= '9') {
		char *c;
		int error = 0;
		
		T.value        =  parse_imap_number(p,left,&c,&error);
		if (error) {
		    
		    DPRINT(Debug,10,(&Debug, 
				     "tokenize: idx=%d failed to parse number '%.*s'",
				     idx,c-p,p));

		    if (c < p +left && *c && isascii(*c) && isprint(*c)) {
			DPRINT(Debug,10,(&Debug, ", next character '%c'",
					 *c));
		    }
		    DPRINT(Debug,10,(&Debug, "\n"));
		    goto fail;
		    
		} else if (*c == ' ') {

		    DPRINT(Debug,20,(&Debug, 
				     "tokenize: idx=%d  RESPONSE (number) '%.*s'\n",
				     idx,c-p,p));
		
		    T.imap_token = imap_number;
		    add_token(&(M->read_tokens),T);
		    idx = (c - M->read_buffer.read_buffer) +1;
		    break;
		} else {
		    DPRINT(Debug,20,(&Debug, 
				     "tokenize: idx=%d  parsing number '%.*s' -- no space seen",
				     idx,c-p,p));
		    if (c < p +left && *c && isascii(*c) && isprint(*c)) {
			DPRINT(Debug,10,(&Debug, ", next character '%c'",
					 *c));
		    }
		    DPRINT(Debug,10,(&Debug, "\n"));		    
		}
	    }
	    n = memchr(p,' ',left);
	    if (n == p) 
		goto fail;
	    
	    if (NULL == n)
		n = p + left;
	    else 
		*n = '\0';
		
	    DPRINT(Debug,20,(&Debug, 
			     "tokenize: idx=%d  RESPONSE (keyword) '%.*s'\n",
			     idx,n-p,p));

	    for (i = 0; i < sizeof keywords / sizeof (struct kwd); i++)
		if (0 == istrcmp(keywords[i].kw,p))
		    break;
	    
	    if (i < sizeof keywords / sizeof (struct kwd))
		T.imap_token = keywords[i].t;
	    else
		T.imap_token = imap_other_keyword;
	    
	    if (imap_status_keyword == T.imap_token)
		M->tokenizer_state = state_text;
	    else if (imap_other_keyword == T.imap_token)
		M->tokenizer_state = state_list;

	    T.str = safe_malloc(n-p+1);
	    memcpy(T.str,p,n-p+1);
	    T.str_len = n-p;
	    add_token(&(M->read_tokens),T);

	    idx = (n - M->read_buffer.read_buffer) +1;
	    break;
	case state_text:
	    if ('[' == p[0]) {

		DPRINT(Debug,20,(&Debug, 
			    "tokenize: idx=%d  TEXT   [\n",
			    idx));
		
		T.imap_token = imap_code_begin;
		M->tokenizer_state = state_text_code;
		idx++;
		add_token(&(M->read_tokens),T);
		break;
	    }	    

	    DPRINT(Debug,20,(&Debug, 
			     "tokenize: idx=%d  TEXT   '%.*s'\n",
			     idx,left,p));

	    T.imap_token = imap_string;
	    T.str = safe_malloc(left);
	    memcpy(T.str,p,left);
	    if (left >= 2)
		T.str_len = left-2;   /* Not find_crlf() */
	    else {
		DPRINT(Debug,20,(&Debug, 
				 "tokenize: left=%d < 2 (find_crlf includes CRLF)\n",
				 left));
	    }
	    	    
	    add_token(&(M->read_tokens),T);

	    idx += left;
	    break;
	case state_text_code:
	    if (']' == p[0]) {

		DPRINT(Debug,20,(&Debug, 
			    "tokenize: idx=%d  CODE   ]\n",
			    idx));

		T.imap_token = imap_code_end;
		M->tokenizer_state = state_text;
		idx++;
		if (' ' == p[1])   /* Skip separated space between code */
		    idx++;         /* argument and textual code         */
		add_token(&(M->read_tokens),T);
		break;
	    }	    
	    /* FALLTHRU */
	case state_list:
	    if ('(' == p[0]) {

		DPRINT(Debug,20,(&Debug, 
			    "tokenize: idx=%d  LIST   (\n",
			    idx));
		
		T.imap_token = imap_list_begin;
		idx++;
		add_token(&(M->read_tokens),T);
		break;
	    }	    
	    if (')' == p[0]) {

		DPRINT(Debug,20,(&Debug, 
			    "tokenize: idx=%d  LIST   )\n",
			    idx));

		T.imap_token = imap_list_end;
		idx++;
		if (' ' == p[1])   /* Skip separated space between list */
		    idx++;         /* argument and next argument        */
		add_token(&(M->read_tokens),T);
		break;
	    }	    
	    if (p[0] >= '0' && p[0] <= '9') {
		char *c;
		int error = 0;

		T.value     =  parse_imap_number(p,left,&c,&error);
		if (error) {
		    
		    DPRINT(Debug,10,(&Debug, 
				     "tokenize: idx=%d failed to parse number '%.*s'",
				     idx,c-p,p));
		    
		    if (c < p +left && *c && isascii(*c) && isprint(*c)) {
			DPRINT(Debug,10,(&Debug, ", next character '%c'",
					 *c));
		    }
		    DPRINT(Debug,10,(&Debug, "\n"));
		    goto fail;
		    
		} else if (*c == ' ' || *c == ')' || *c == ']') {
		    
		    DPRINT(Debug,20,(&Debug, 
				     "tokenize: idx=%d  LIST (number)   '%.*s'\n",
				     idx,c-p,p));
		    
		    T.imap_token = imap_number;
		    add_token(&(M->read_tokens),T);
		    if (*c == ' ')
			idx = (c - M->read_buffer.read_buffer) +1;
		    else
			idx = (c - M->read_buffer.read_buffer);
		    break;
		}
	    }

	    if (M->tokenizer_state != state_text_code) {
		if ('"' == p[0]) {
		    char * x, *y;
		    T.imap_token = imap_string;
		    T.str = safe_malloc(left);
		    y = T.str;

		    for (x = p+1; x < p + left && '\0' != *x && '"' != *x; 
			 x++) {
			if ('\\' == *x)
			    x++;
			if (y < T.str + left -1)
			    *y++ = *x;
		    }
		    if (*x != '"')
			goto fail;
		    x++;
		    *y = '\0';

		    DPRINT(Debug,20,(&Debug, 
				     "tokenize: idx=%d  LIST (string)   '%.*s'\n",
				     idx,y-T.str,T.str));

		    T.str_len = y-T.str; 
		    add_token(&(M->read_tokens),T);
		    
		    if (' ' == *x) /* Skip separated space between string */
			x++;       /* argument and next argument          */
		    idx = x - M->read_buffer.read_buffer;		
		    break;
		}
		if (p[0] == '{' && left > 4 && p[left-3] == '}' &&
		    M->tokenizer_state == state_list) {
		    char *c;
		    int error = 0;

		    M->wanted_literal_len = parse_imap_number(p+1,left-4,&c,&error);
		    

		    if (error) {
		    
			DPRINT(Debug,10,(&Debug, 
					 "tokenize: idx=%d failed to parse literal len '%.*s'",
					 idx,c-p-1,p+1));
			
			if (c < p +left && *c && isascii(*c) && isprint(*c)) {
			    DPRINT(Debug,10,(&Debug, ", next character '%c'",
					     *c));
			}
			DPRINT(Debug,10,(&Debug, "\n"));
			goto fail;
			
		    } else if (c == &(p[left-3])) {   /* Start parsing literal */

			DPRINT(Debug,20,(&Debug, 
					 "tokenize: idx=%d  LIST (literal)   len=%lu\n",
					 idx,
					 (unsigned long)(M->wanted_literal_len)));
			
			M->wanna_literal = 1;
			idx += left;
			ret = 0;               /* Continue reading ... */
			break;
		    }
		}
	    }

	    if (' ' == p[0]) {
		/* Communigate Pro 3.3.1 produces extra space between
		   CAPABILITY response tokens before auth capabilities
		*/
		DPRINT(Debug,3,(&Debug, 
				"* Format error -- extra space between tokens on IMAP response:\n"));
		DPRINT(Debug,3,(&Debug, 
				">>> %s\n",
				p));
		idx++;
		continue;
	    }


	    T.imap_token = imap_atom;
	    if (M->tokenizer_state == state_text_code)
		i = strcspn(p," ()]");
	    else
		i = strcspn(p,"(){ *%\"");
	    if (i == 0) {
		DPRINT(Debug,9,(&Debug, 
				"tokenize: Possible internal error:'%s'\n",
				p));
		goto fail;
	    }

	    DPRINT(Debug,20,(&Debug, 
			     "tokenize: idx=%d  LIST (atom)   '%.*s'\n",
			     idx,i,p));
	    
	    T.imap_token = imap_atom;
	    T.str = safe_malloc(i+1);
	    memcpy(T.str,p,i);
	    T.str[i]    = '\0';
	    T.str_len = i;
	    
	    idx += i;
	    if (' ' == M->read_buffer.read_buffer[idx])
		idx++;
	    add_token(&(M->read_tokens),T);
	    break;
	default:
	    panic("MBX PANIC",__FILE__,__LINE__,"tokenize",
		  "Bad tokenizer_state (IMAP)",0);
	}
    }
    
    if (!M->wanna_literal) {
	/* Again beginning of line */
	M->tokenizer_state = state_tag;
    }
    cut_line(&(M->read_buffer),len);

    if (M->wanna_literal && M->read_buffer.read_len)
	goto try_literal;

    DPRINT(Debug,7,(&Debug, "tokenize=%d (token count %d)\n",
		    ret,M->read_tokens.token_count));

    return ret;   /* Response parsed? */
}

/* return code pos or -1 */
static int pick_text P_((struct IMAP_CON * M, struct string **Text,
			  char **code));
static int pick_text(M,Text,code)
     struct IMAP_CON * M;
     struct string **Text;
     char **code;
{
    int code_pos = -1;
    int i;
    int q = 0;
    
    *Text = NULL;
    *code = NULL;

    for (i = 2; i < M->read_tokens.token_count; i++) {
	if (imap_code_begin == M->read_tokens.tokens[i].imap_token)
	    q++;
	else if (imap_code_end == M->read_tokens.tokens[i].imap_token)
	    q--;
	else if (q && imap_atom == M->read_tokens.tokens[i].imap_token &&
		 !*code) {
	    *code = M->read_tokens.tokens[i].str;
	    code_pos = i;
	} else if (!q && imap_string == M->read_tokens.tokens[i].imap_token &&
		 !*Text)
	    /* User readable code may use MIME encoded words on message */
	    *Text = hdr_to_string(HDR_TEXT,
				  M->read_tokens.tokens[i].str,
				  imap_charset,1);	    
    }

    DPRINT(Debug,21,(&Debug,
		     "pick_text=%d",code_pos));    
    if (*code) {
	DPRINT(Debug,21,(&Debug," *code=%s",
			 *code));
    }
    if (*Text) {
	DPRINT(Debug,21,(&Debug," *Text=%S",
			 *Text));
    }    
    DPRINT(Debug,21,(&Debug,"\n"));
    
    return code_pos;
}

static int scan_list P_((struct imap_token ** ptr,
			 int                * left,
			 struct imap_token ** found_list,
			 int                * list_len));
static int scan_list(ptr,left,found_list,list_len)
     struct imap_token ** ptr;
     int                * left;
     struct imap_token ** found_list;
     int                * list_len;
{
    struct imap_token * p = *ptr;
    int                 l = *left;
    int i,q=0;

    *found_list = NULL;
    *list_len   = 0;

    if (l < 2 || imap_list_begin != p[0].imap_token)
	return 0;

    *found_list = p+1;
    
    for (i = 0; i < l; i++) {
	if (imap_list_begin == p[i].imap_token)
	    q++;
	else if (imap_list_end == p[i].imap_token) {
	    q--;
	    if (0 == q) {
		*list_len = i-1;
		*ptr      = p+  (i +1);
		*left     = l - (i +1);
		return 1;
	    }
	}
    }
    return 0;
}

void imap_free_reference(item)
     struct imap_reference *item;
{
    item->uid_number        = -1;
    item->imap_flags        = 0;
    item->rfc822_size       = 0;
    item->From_size         = 0;

    if (item->From_separator) {
	free(item->From_separator);
	item->From_separator = NULL;
    }
    
    if (item->internaldate) {
	free(item->internaldate);
	item->internaldate  = NULL;
    }

    if (item->header) {
	free(item->header);
	item->header      = NULL;
	item->header_len = 0;	    
    }
    if (item->body) {
	free(item->body);
	item->body      = NULL;
	item->body_len  = 0;    
    }
}

static void zero_reference P_((struct imap_reference *item));
static void zero_reference(item)
     struct imap_reference *item;
{
    item->uid_number        = -1;
    item->imap_flags        = 0;
    item->rfc822_size       = 0;
    item->From_size         = 0;
    item->From_separator    = NULL;
    item->internaldate      = NULL;
    item->header_len        = 0;
    item->header            = 0;
    item->body_len          = 0;
    item->body              = 00;
}


static int got_header P_((struct imap_reference * item,
			  struct imap_token * lst));
static int got_header(item,lst)
     struct imap_reference * item;
     struct imap_token * lst;
{
    /* Originally malloced by tokenize() */
    if (item->header) {
	free(item->header);
	item->header = NULL;
    }

    switch(lst->imap_token) {
    case imap_atom:
	if (0 == istrcmp(lst->str,"NIL")) {
	    item->header     = NULL;
	    item->header_len = 0;
	    return 1;       /* OK */
	}
	return 0;           /* Bad */
    case imap_string:
    case imap_literal:
	/* Move malloced data to struct imap_reference */
	item->header     = lst->str;
	/* Maybe SIZE_MAX is 18446744073709551615UL,
	   but INT_MAX is 2147483647 ?
	   Is it possible to alloc more than  INT_MAX bytes memory once ?

	   For literal maximum length is 4294967295 bytes, but
	   imap_read_stream() closes stream if literal length is INT_MAX bytes
	   or more.
	   
	*/
	if ((unsigned long)(lst->str_len) >= (unsigned long)(INT_MAX)) {
	    DPRINT(Debug,7,(&Debug,
			    "got_header: str_len=%lu >= INT_MAX (%d)\n",
			    (unsigned long)(lst->str_len),INT_MAX));
	    item->header_len = INT_MAX;
	} else {
	    item->header_len = lst->str_len;
	}
	lst->str         = NULL;  /* So that free_tokens() does not try free 
				   * it  
				   */
	return 1;           /* OK */
    default:
	return 0;   /* Bad */
    }
}

static int got_body P_((struct imap_reference * item,
			struct imap_token * lst));
static int got_body(item,lst)
     struct imap_reference * item;
     struct imap_token * lst;
{
    /* Originally malloced by tokenize() */
    if (item->body) {
	free(item->body);
	item->body = NULL;
    }

    switch(lst->imap_token) {
    case imap_atom:
	if (0 == istrcmp(lst->str,"NIL")) {
	    item->body     = NULL;
	    item->body_len = 0;
	    return 1;       /* OK */
	}
	return 0;           /* Bad */
    case imap_string:
    case imap_literal:
	/* Move malloced data to struct imap_reference */
	item->body     = lst->str;
	/* Maybe SIZE_MAX is 18446744073709551615UL,
	   but INT_MAX is 2147483647 ?
	   Is it possible to alloc more than  INT_MAX bytes memory once ?

	   For literal maximum length is 4294967295 bytes, but
	   imap_read_stream() closes stream if literal length is INT_MAX bytes
	   or more.
	   
	*/
	if ((unsigned long)(lst->str_len) >= (unsigned long)(INT_MAX)) {
	    DPRINT(Debug,7,(&Debug,
			    "got_body: str_len=%lu >= INT_MAX (%d)\n",
			    (unsigned long)(lst->str_len),INT_MAX));
	    item->body_len = INT_MAX;
	} else {
	    item->body_len = lst->str_len;
	}
	lst->str       = NULL;  /* So that free_tokens() does not try free 
				 * it  
				 */
	return 1;           /* OK */
    default:
	return 0;   /* Bad */
    }

}

static void process_list_flags P_((struct imap_dir_entry * item,
				   struct imap_token * lst,
				   int                 lst_len));
static void process_list_flags(item,lst,lst_len)
     struct imap_dir_entry * item;
     struct imap_token     * lst;
     int                     lst_len;
{
    DPRINT(Debug,9,(&Debug, 
		    "process_list_flags: len=%d",
		    lst_len));
    while (lst_len > 0) {
	struct imap_token * kwd = lst;
	lst_len--;
	lst++;
	if (imap_atom == kwd->imap_token) {
	    int j;
	    int found = 0;
	    for (j = 0; 
		 j < sizeof BROWSER_flags / 
		     sizeof (struct browser_flag);
		 j++) {
		if (0 == istrcmp(kwd->str,
				 BROWSER_flags[j].flag)) {
		    item->flags |= BROWSER_flags[j].browser_flag;
		    found++;
		    break;
		}
	    }
	    if (!found) {
		DPRINT(Debug,9,(&Debug, 
				" [Unknown flag %s] ",
				kwd->str));
	    }
	    DPRINT(Debug,10,(&Debug, 

			     " [known flags:"));
	    for (j = 0;
		 j < sizeof BROWSER_flags / sizeof (struct browser_flag);
		 j++) {
		if (BROWSER_flags[j].browser_flag &&
		    (item->flags & BROWSER_flags[j].browser_flag)
		    == BROWSER_flags[j].browser_flag)
		    DPRINT(Debug,10,(&Debug, 
				     " %s",BROWSER_flags[j].flag));
	    }
	    DPRINT(Debug,9,(&Debug, 
			    "]"));
	} else {
	    DPRINT(Debug,9,(&Debug, 
			    " ???\n"));
	    DPRINT(Debug,9,(&Debug, 
			    "process_list_flags: List response  (..) flag item have token type %d\n",
			    kwd->imap_token));
	    break;	/* Quit processing */	
	}
    }
    DPRINT(Debug,9,(&Debug, "\n"));
}


static void process_fetch_items P_((struct imap_reference * item,
				    struct imap_token * lst,
				    int                 lst_len));
static void process_fetch_items(item,lst,lst_len)
     struct imap_reference * item;
     struct imap_token     * lst;
     int                     lst_len;
{			
    DPRINT(Debug,7,(&Debug,  
		    "process_fetch_items: len=%d",
		    lst_len));
    while (lst_len > 1) {
	struct imap_token * kwd = lst;
	lst_len--;
	lst++;
	
	if (imap_atom == kwd->imap_token) {
	    if (0 == istrcmp(kwd->str,"UID")) {
		if (imap_number == lst->imap_token) {
		    /* Perhaps LONG_MAX is 2147483647L,
		       maximum imap_token value is 4294967295
		    */
		    if ((unsigned long)lst->value > (unsigned long)LONG_MAX) {
			DPRINT(Debug,9,(&Debug, 
					" UID %lu unsupported",
					(unsigned long)lst->value));
		    } else {
			item->uid_number = lst->value;

			DPRINT(Debug,9,(&Debug, 
					" UID %ld",item->uid_number));
		    }
			
		    lst_len--;
		    lst++;
		} else {
		    DPRINT(Debug,9,(&Debug, " UID ???\n"));
		    DPRINT(Debug,1,(&Debug, 
				    "process_fetch_items: Fetch response  (..) after UID number expected, got type %d\n",
			       lst->imap_token));
		    break; /* Quit processing */
		}
	    } else if (0 == istrcmp(kwd->str,"INTERNALDATE")) {
		if (imap_string == lst->imap_token) {
		    DPRINT(Debug,9,(&Debug,  " INTERNALDATE %s (len %zu)",
				    lst->str,lst->str_len));
		    if (lst->str_len != 26) {
			DPRINT(Debug,9,(&Debug, "\n"));
			DPRINT(Debug,1,(&Debug, 
				   "process_fetch_items: Fetch response  (..) after INTERNALDATE 26 char string expected, got %zu chars\n",
					lst->str_len));
		    } else {
			if (item->internaldate)
			    free(item->internaldate);
			item->internaldate = lst->str;
			lst->str          = NULL;   /* Avoid double free() */
		    }
		    lst_len--;
		    lst++;
		} else {
		    DPRINT(Debug,9,(&Debug, " INTERNALDATE ????\n"));
		    DPRINT(Debug,1,(&Debug, 
			       "process_fetch_items: Fetch response  (..) after INTERNALDATE string expected, got type %d\n",
			       lst->imap_token));
		    break; /* Quit processing */
		}
	    } else if (0 == istrcmp(kwd->str,"RFC822.SIZE")) {
		if (imap_number == lst->imap_token) {
		    
		    if ((unsigned long)lst->value > (unsigned long)INT_MAX) {
			DPRINT(Debug,9,(&Debug, " RFC822.SIZE %lu unsupported",
					(unsigned long)lst->value));

			item->rfc822_size = INT_MAX;
		    } else {
		       			
			item->rfc822_size = lst->value;
			
			DPRINT(Debug,9,(&Debug, " RFC822.SIZE %d",
					item->rfc822_size));
		    }
		    
		    lst_len--;
		    lst++;
		} else {
		    DPRINT(Debug,9,(&Debug, " RFC822.SIZE ????\n"));
		    DPRINT(Debug,9,(&Debug, 
				    "process_fetch_items: Fetch response  (..) after RFC822.SIZE number expected, got type %d\n",
				    lst->imap_token));
		    break; /* Quit processing */
		}
	    } else if (0 == istrcmp(kwd->str,"RFC822.HEADER")) {
		if (got_header(item,lst)) {		    
		    DPRINT(Debug,9,(&Debug, " RFC822.HEADER ..."));
		    lst_len--;
		    lst++;
		} else {
		    DPRINT(Debug,9,(&Debug, " RFC822.HEADER ???\n"));
		    DPRINT(Debug,1,(&Debug, 
				    "process_fetch_items: Fetch response  (..) after RFC822.HEADER nstring expected, got type %d\n",
			       lst->imap_token));
		    break; /* Quit processing */
		}
	    } else if (0 == istrcmp(kwd->str,"RFC822.TEXT")) {
		if (got_body(item,lst)) {		    
		    DPRINT(Debug,9,(&Debug, " RFC822.TEXT ..."));
		    lst_len--;
		    lst++;
		} else {
		    DPRINT(Debug,9,(&Debug, " RFC822.TEXT ???\n"));
		    DPRINT(Debug,1,(&Debug, 
				    "process_fetch_items: Fetch response  (..) after RFC822.BODY nstring expected, got type %d\n",
				    lst->imap_token));
		    break; /* Quit processing */
		}
	    } else if (0 == istrcmp(kwd->str,"BODY[HEADER]")) {
		if (got_header(item,lst)) {		    
		    DPRINT(Debug,9,(&Debug, " BODY[HEADER] ..."));
		    lst_len--;
		    lst++;
		} else {
		    DPRINT(Debug,9,(&Debug, " BODY[HEADER] ???\n"));
		    DPRINT(Debug,1,(&Debug, 
				    "process_fetch_items: Fetch response  (..) after BODY[HEADER] nstring expected, got type %d\n",
				    lst->imap_token));
		    break; /* Quit processing */
		}
	    } else if (0 == istrcmp(kwd->str,"BODY[TEXT]")) {
		if (got_body(item,lst)) {		    
		    DPRINT(Debug,9,(&Debug, " BODY[TEXT] ..."));
		    lst_len--;
		    lst++;
		} else {
		    DPRINT(Debug,9,(&Debug, " BODY[TEXT] ???\n"));
		    DPRINT(Debug,1,(&Debug, 
				    "process_fetch_items: Fetch response  (..) after BODY[TEXT] nstring expected, got type %d\n",
			       lst->imap_token));
		    break; /* Quit processing */
		}
	    } else if (0 == istrcmp(kwd->str,"FLAGS")) {
		struct imap_token * flg;
		int                 flg_len;
		
		item->imap_flags = 0;
		
		DPRINT(Debug,9,(&Debug, " FLAGS "));
		if (scan_list(&lst,&lst_len,&flg,&flg_len)) {
		    int i, j;
		    
		    DPRINT(Debug,9,(&Debug, "(listlen=%d)",flg_len));
		    for (i = 0; i < flg_len; i++) {
			if (imap_atom == flg[i].imap_token) {
			    int found = 0;
			    for (j = 0; 
				 j < sizeof IMAP_flags / 
				     sizeof (struct imap_flag);
				 j++) {
				if (0 == istrcmp(flg[i].str,
						 IMAP_flags[j].flag)) {
				    item->imap_flags |= 
					IMAP_flags[j].mask_imap;
				    found++;
				    break;
				}
			    }
			    if (!found) {
				DPRINT(Debug,9,(&Debug, 
						" [Unknown flag %s] ",
						flg[i].str));
			    }
			}			
		    }
		    DPRINT(Debug,10,(&Debug, 
				     " [known flags:"));
		    for (j = 0;
			 j < sizeof IMAP_flags / sizeof (struct imap_flag);
			 j++) {
			if ((item->imap_flags & IMAP_flags[j].mask_imap)
			    == IMAP_flags[j].mask_imap)
			    DPRINT(Debug,10,(&Debug, " %s",IMAP_flags[j].flag));
		    }
		    DPRINT(Debug,10,(&Debug, "]"));
		} else {
		    DPRINT(Debug,9,(&Debug, " ???\n"));
		    DPRINT(Debug,1,(&Debug, 
				    "process_fetch_items: Fetch response  (..) after FLAGS (list) expected, got type %d\n",
				    lst->imap_token));
		    break; /* Quit processing */
		}
	    } else {
		DPRINT(Debug,9,(&Debug, " ???\n"));
		DPRINT(Debug,1,(&Debug, 
				"process_fetch_items: Fetch response  (..) list item %s unsupported\n",
				kwd->str));
		break; /* Quit processing */
	    }				    
	} else {
	    DPRINT(Debug,9,(&Debug, " ???\n"));
	    DPRINT(Debug,1,(&Debug, 
			    "process_fetch_items: Fetch response  (..) list item have token type %d\n",
			    kwd->imap_token));
	    break;	/* Quit processing */	
	}
    }    
    DPRINT(Debug,7,(&Debug, "\n"));
}

S_(ss_action_routine imap_read_stream)
static int imap_read_stream P_((struct streamsched              * ss,
				union ss_action_routine_data      data,
				const struct schedule_timelimit * now));

S_(ss_badpid_routine imap_badpid_stream)
static  enum badpid_status imap_badpid_stream
   P_((struct streamsched *ss,
       union  ss_action_routine_data data,
       int badpid));


static int imap_clear_cmd_head P_((struct connection_cache *con));
static int imap_clear_cmd_head(con)
     struct connection_cache *con;
{
    int status = 0;
    struct IMAP_CON * const M = con->a.imap_con;
    
    if (!M) {
	DPRINT(Debug,7,(&Debug,
			"imap_clear_cmd_head: NO IMAP_CON\n!"));
	goto clean;
    }

    if (IMAP_idle == M->imap_state) {
	DPRINT(Debug,7,(&Debug, 
		    "imap_clear_cmd_head: Already clear!\n"));
    } else if (IMAP_error  == M->imap_state) {
		DPRINT(Debug,7,(&Debug, 
				"imap_clear_cmd_head: On ERROR state\n"));
    } else if (IMAP_closing == M->imap_state) {
	DPRINT(Debug,7,(&Debug, 
			"imap_clear_cmd_head: On closing state!\n"));
    } else if (IMAP_command_ready_NO  != M->imap_state &&
	       IMAP_command_ready_BAD != M->imap_state &&
	       IMAP_command_ready_OK  != M->imap_state) {
	DPRINT(Debug,7,(&Debug, 
			"imap_clear_cmd_head: WRONG STATE!\n"));
	M->imap_state = IMAP_error;
    } else
	M->imap_state = IMAP_idle;

    status = 1;
    
 clean:
    DPRINT(Debug,13,(&Debug,
		     "imap_clear_cmd_head=%d\n",
		     status));
    return status;
}

static void imap_clear_cmd_tail P_((struct connection_cache *con));
static void imap_clear_cmd_tail(con)
     struct connection_cache *con;
{
    struct IMAP_CON * const M = con->a.imap_con;
    
    /* Reset possible incomplete send literal */
    free_tokens(& (M->write_tokens));
    M->literal_mode = no_literal;
    M->token_index  = -1;

    free_Write_Buffer(& (M->write_buffer));
    
    if (M->current_tag) {
	free(M->current_tag);
	M->current_tag = NULL;
    }

    if (M->command_result_code) {
	free(M->command_result_code);
	M->command_result_code = NULL;
    }

    if (CON_error == con->state ||
	IMAP_error == M->imap_state) {
	DPRINT(Debug,3,(&Debug, 
			"imap_clear_cmd_tail: Have error\n"));
    }
    
}
    
int imap_clear_command_c(con,cd)
     struct connection_cache *con;
     struct cancel_data  * cd    /* Allow cancelation (Ctrl-C)
				    check of remote mailbox
				 */;
{
    int status = 0;
    struct IMAP_CON * const M = con->a.imap_con;

    DPRINT(Debug,7,(&Debug, 
		    "imap_clear_command_c: con=%p (%s@%s)\n",
		    con,
		    con->C.username ? con->C.username : "<NULL>",
		    con->C.host ? con->C.host : "<NULL>"));

    if (!imap_clear_cmd_head(con)) {
	goto clean;
    }

    if (M->current_NOOP_tag &&
	CON_error != con->state &&
	M->imap_state != IMAP_error) {

	int count = 0;

	if (cd && is_canceled(cd)) {
	    DPRINT(Debug,3,(&Debug, 
			    "imap_clear_command_c: idle NOOP in proggress, but waiting already canceled\n"));
	    goto clean;
	}

	
	DPRINT(Debug,3,(&Debug, 
			"imap_clear_command_c: Waiting for idle NOOP complete...\n"));

	while(M->current_NOOP_tag && 
	      M->imap_state != IMAP_error &&
	      CON_error != con->state &&
	      con->C.stream) {

	    count++;
	    DPRINT(Debug,13,(&Debug,
			     "imap_clear_command_c: waiting #%d\n",
			     count));

	    WaitStreamFor_c(con->C.stream,SS_read_act,cd);

	    if (cd && is_canceled(cd)) {
		DPRINT(Debug,3,(&Debug,
				"imap_clear_command_c: Wait canceled\n"));
		
		goto clean;
	    }

	}

	if (CON_error == con->state ||
	     IMAP_error == con->a.imap_con->imap_state) {
	     DPRINT(Debug,3,(&Debug, 
			     "imap_clear_command_c: idle NOOP wait ended, have error\n"));

	 } else {	    
	     DPRINT(Debug,3,(&Debug, 
			     "imap_clear_command_c: idle NOOP wait completed\n"));
	 }
    }

    status = 1;
    
    imap_clear_cmd_tail(con);
	
 clean:
    DPRINT(Debug,7,(&Debug, 
		    "imap_clear_command_c=%d\n",
		    status));
    return status;
}

void imap_clear_command(con)
     struct connection_cache *con;
{
    struct IMAP_CON * const M = con->a.imap_con;

    DPRINT(Debug,7,(&Debug, 
		"imap_clear_command: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));


    if (!imap_clear_cmd_head(con)) {
	return;
    }
        
    if (M->current_NOOP_tag &&
	CON_error != con->state &&
	M->imap_state != IMAP_error) {

	int count = 0;
	
	DPRINT(Debug,3,(&Debug, 
		    "imap_clear_command: Waiting for idle NOOP complete...\n"));

	
	
	while(M->current_NOOP_tag && 
	      M->imap_state != IMAP_error &&
	      CON_error != con->state &&
	      con->C.stream) {

	    count++;
	    DPRINT(Debug,13,(&Debug,
			     "imap_clear_command: waiting #%d\n",
			     count));


	    WaitStreamFor(con->C.stream,SS_read_act);
	    /* Should user to have some way to interrupt progress ? 
	       
	       imap_clear_command_c() uses interruple wait
	     */
	}

	 if (CON_error == con->state ||
	     IMAP_error == con->a.imap_con->imap_state) {
	     DPRINT(Debug,3,(&Debug, 
			     "imap_clear_command: idle NOOP wait ended, have error\n"));

	 } else {	    
	     DPRINT(Debug,3,(&Debug, 
			     "imap_clear_command: idle NOOP wait completed\n"));
	 }
    }

    imap_clear_cmd_tail(con);
    
}    

struct string *conv_from_imap_name(name)
     const char *name;
{
    struct string *ret = NULL;
    int L;

    /* Try IMAP encoding and if it fails, fall back to imap_charset.
       Do not return IMAP encoding in string's charset, because
       there is danger that it can leak out. Instead return string
       with well known encoding of UNICODE (ie. UTF-7)
    */
       

    if (IMAP_name_convention && name[0] && 0 < (L = strlen(name))) {
	struct charset_state * P = new_state(IMAP_ENCODING);
	uint16 * vector = safe_calloc(L,sizeof (vector[0]));
	const char *s;
	int X = 0;
	int ok = 1;
	charset_t utf7 = NULL;

	DPRINT(Debug,12,(&Debug,  "conv_from_imap_name: %s =",name));
	for (s = name; *s && X < L; s++) {
	    if (!add_streambyte_to_state(P, (unsigned char)*s)) {
		DPRINT(Debug,12,(&Debug,  " FAIL"));
		ok = 0;
		break;
	    }
	    if (state_ready(P)) {
		vector[X++] = give_unicode_from_state(P);
		DPRINT(Debug,12,(&Debug,  " %04X",vector[X-1]));
		reset_state(P,0);
	    }
	}
	DPRINT(Debug,12,(&Debug,  "\n"));

	if (!ok) {
	    DPRINT(Debug,7,(&Debug, 
			    "conv_from_imap_name: Not IMAP encoded: %s\n",
			    name));
	} else if (NULL != (utf7 = MIME_name_to_charset("UTF-7",0))) {
	    ret = new_string(utf7);
	    add_unicode_to_string(ret,X,vector);
	    DPRINT(Debug,7,(&Debug, 
			    "conv_from_imap_name: IMAP encoded, len=%d: %s (%S)\n",
			    X,name,ret));
	}

	free_state(&P);
	free(vector);
    }


    if (!ret)
	ret = new_string2(imap_charset,cs2us(name));

    {
	const char * MIME_name UNUSED_VAROK = get_string_MIME_name(ret);
        const charset_t cs     UNUSED_VAROK = get_string_type(ret);

	DPRINT(Debug,7,(&Debug, 
			"conv_from_imap_name=%S [type=%p '%s']   (name=%s) \n",
			ret,
                        cs,
                        MIME_name ? MIME_name : "<no MIME name>",
                        name));

    }
    return ret;
}


/* Special return value 2 indicates that caller should re-enable 
 * imap_write_stream()
 */
static int parse_response P_((struct connection_cache *con));
static int parse_response(con)
     struct connection_cache *con;

{
    struct IMAP_CON     * M_a = con->a.imap_con;
    struct IMAP_MBX     * M_b = NULL;
    struct IMAP_BROWSER * M_c = NULL;
    struct private_data * M_p  = NULL;

    if (con->f) {
	if (con->f->folder_type != &imap_mbx) 
	    panic("MBX PANIC",__FILE__,__LINE__,"parse_response",
		  "Bad folder type attached to connection",0);
	else {	    
	    M_p = con->f->p;
	    
	    if (PRIVATE_DATA_magic != M_p->magic) 
		panic("MBX PANIC",__FILE__,__LINE__,"parse_response",
		      "Bad magic number (private_data)",0);

	    M_b = &(M_p->a.imap_mbx);

	}
    }

    if (con->d) {
	if (con->d->type != &imap_browser)
	    panic("MBX PANIC",__FILE__,__LINE__,"parse_response",
		  "Bad browser type attached to connection",0);
	else {
	    M_c = con->d->a.imap_browser;
    
	    if (IMAP_BROWSER_magic != M_c->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"parse_response",
		      "Bad magic type",0);
	}
    }    

    if (!M_a) {
	panic("MBX PANIC",__FILE__,__LINE__,"parse_response",
	      "No imap connection",0);
    }

    if (M_a->read_tokens.token_count < 2) {
	DPRINT(Debug,1,(&Debug, "parse_response: token_count=%d\n",
		   M_a->read_tokens.token_count));
	lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
			  "Bad response from IMAP server"));
	return 0;
    }

    /* Clear stalled message */
    if (M_a->stalled &&
	con->C.host) {

	lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPresumed,
			      "IMAP connection to %s resumed."),
		      con->C.host);
    }
    M_a->stalled = 0;

    
    switch (M_a->read_tokens.tokens[0].imap_token) {
    case imap_notag:
	switch (M_a->read_tokens.tokens[1].imap_token) {
	    struct string *Text;
	    char *code;   /* Do not free() this -- this is just pointer
			     to read_tokens area
			  */
	    int code_pos;
	    
	case imap_status_keyword:
	    Text = NULL;
	    code = NULL;
	    code_pos = pick_text(M_a,&Text,&code);
	    if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"BYE")) {
		M_a->imap_state = IMAP_closing;
		if (Text)
		    lib_transient(CATGETS(elm_msg_cat, MeSet,MeImapBye,
					  "IMAP server: %S"),
				  Text);		
	    } else if (code && Text && 0 == istrcmp(code,"ALERT")) {
		lib_error(CATGETS(elm_msg_cat, MeSet,MeImapAlert,
				  "IMAP alert: %S"),
			  Text);		
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"OK")) {
		if (IMAP_show_greeting && !code)
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeImapServer,
				      "IMAP server: %S"),
			      Text);

		if (code && 0 == istrcmp(code,"UIDVALIDITY")) {
		    if (code_pos < 0 || code_pos >= M_a->read_tokens.token_count ||
			0 != istrcmp(M_a->read_tokens.tokens[code_pos].str,
				     "UIDVALIDITY"))
			panic("MBX PANIC",__FILE__,__LINE__,"parse_response",
			      "Bad code_pos",0);
		
		    if (code_pos+1 < M_a->read_tokens.token_count &&
			imap_number == M_a->read_tokens.tokens[code_pos+1].imap_token) {
			if (M_b) {
			    if ((unsigned long)M_a->read_tokens.tokens[code_pos+1].value
				> (unsigned long)INT_MAX) {
				
				DPRINT(Debug,12,(&Debug, 
						 "parse_response: UIDVALIDITY %lu unsupported\n",
						 (unsigned long)M_a->read_tokens.tokens[code_pos+1].value));


			    } else {
				M_b->uidvalidity = 
				    M_a->read_tokens.tokens[code_pos+1].value;
				DPRINT(Debug,12,(&Debug, 
						 "parse_response: UIDVALIDITY %d\n",
						 M_b->uidvalidity));
			    }
			}
		    }					
		}
		
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"NO")) {
		if (IMAP_show_greeting)
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeImapWarning,
				      "IMAP warning: %S"),
			      Text);		
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"BAD")) {
		if (IMAP_show_error)
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeImapProtoError,
				      "IMAP error: %S"),
			      Text);			    
	    }
	    if (Text)
		free_string(&Text);
	    break;
	case imap_other_keyword:

	    if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"CAPABILITY")) {
		int i,j;
		M_a->capability_bits = 0;
		
#ifdef USE_DLOPEN
		free_only_imap_capa_libs( & (M_a->capability_libs),
					  & (M_a->capability_lib_count));
#endif		

		for (i = 2; i < M_a->read_tokens.token_count; i++) {
		    if (imap_atom == M_a->read_tokens.tokens[i].imap_token) {
			
			DPRINT(Debug,2,(&Debug,  
					"IMAP server capacity: '%s'\n",
					M_a->read_tokens.tokens[i].str));

#ifdef USE_DLOPEN
		    probe_imap_capa_lib( & (M_a->capability_libs),
					 & (M_a->capability_lib_count),
					 M_a->read_tokens.tokens[i].str);
#endif

			for (j = 0; 
			     j < sizeof capabilities / sizeof (struct capab);
			     j++) {
			    if (0 == istrcmp(M_a->read_tokens.tokens[i].str,
					     capabilities[j].capa)) 
				M_a->capability_bits |= capabilities[j].mask;
			}
		    }
		}
		DPRINT(Debug,9,(&Debug, 
				"parse_response: (known) Capabilities:"));
		for (j = 0;
		     j < sizeof capabilities / sizeof (struct capab);
		     j++) {
		    if ((M_a->capability_bits & capabilities[j].mask)
			== capabilities[j].mask)
			DPRINT(Debug,9,(&Debug, " %s",capabilities[j].capa));
		}
		DPRINT(Debug,9,(&Debug, "\n"));

	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"LIST")) {
		if (M_c) {
		    struct imap_token * ptr  = 
			M_a->read_tokens.tokens+2;
		    int                left = 
			M_a->read_tokens.token_count-2;
		    struct imap_token * lst;
		    int                 lst_len;


		    M_c->dir_entries = 
			safe_array_realloc(M_c->dir_entries,
					   (M_c->dir_entry_count +1),
					   sizeof (struct imap_dir_entry));

		    M_c->dir_entries[M_c->dir_entry_count].flags = 0;
		    if (!scan_list(&ptr,&left,&lst,&lst_len) ||
			left != 2) {
			DPRINT(Debug,1,(&Debug, 
				   "parse_response: scan_list failed to parse LIST response list, left=%d\n",
				   left));
			lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
					  "Bad response from IMAP server"));
			return 0;
		    } else {
			process_list_flags(M_c->dir_entries+
					   M_c->dir_entry_count,
					   lst,lst_len);
			if (imap_atom == ptr[0].imap_token &&
			    0 == istrcmp(ptr[0].str,"nil"))
			    M_c->dir_entries[M_c->dir_entry_count].sep ='\0';
			else if (imap_string == ptr[0].imap_token &&
				 1 == ptr[0].str_len)
			    M_c->dir_entries[M_c->dir_entry_count].sep = 
				ptr[0].str[0];
			else {
			    DPRINT(Debug,1,(&Debug, 
				       "parse_response: LIST: Bad separator, type=%d\n",
				       ptr[0].imap_token));
			    lib_error(CATGETS(elm_msg_cat, MeSet,
					      MeUnexpextedResponseImap,
					      "Bad response from IMAP server"));
			    return 0;			    
			}
			
			if (imap_string == ptr[1].imap_token)
			    M_c->dir_entries[M_c->dir_entry_count].
				imap_name = safe_strdup(ptr[1].str);
			else if (imap_literal == ptr[1].imap_token)
			    M_c->dir_entries[M_c->dir_entry_count].
				imap_name = safe_strdup(ptr[1].str);
			else if (imap_atom == ptr[1].imap_token)
			    M_c->dir_entries[M_c->dir_entry_count].
				imap_name = safe_strdup(ptr[1].str);
			else {
			    DPRINT(Debug,1,(&Debug, 
				       "parse_response: LIST: Bad mailbox, type=%d\n",
				       ptr[0].imap_token));
			    lib_error(CATGETS(elm_msg_cat, MeSet,
					      MeUnexpextedResponseImap,
					      "Bad response from IMAP server"));
			    return 0;			    
			}
			M_c->dir_entries[M_c->dir_entry_count].
			    translated_name = 
			    conv_from_imap_name(M_c->dir_entries
						[M_c->dir_entry_count].
						imap_name);
			DPRINT(Debug,30,(&Debug, 
					 "parse_response: %d: list item '%s' separator=%c\n",
					 M_c->dir_entry_count,
					 M_c->dir_entries[M_c->dir_entry_count].
					 imap_name ,
					 M_c->dir_entries[M_c->dir_entry_count].
					 sep));
				  
			DPRINT(Debug,30,(&Debug, 
					 "parse_response: %d: %p  '%S'\n",
					 M_c->dir_entry_count,
					 M_c->dir_entries[M_c->dir_entry_count].
					 translated_name,
					 M_c->dir_entries[M_c->dir_entry_count].
					 translated_name));
			       
			M_c->dir_entry_count++;
		    }
		} else {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Unexpected LIST response\n"));
		}
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"LSUB")) {
		/* TODO: Handle LSUB */
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"STATUS")) {
		/* TODO: Handle STATUS */
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"SEARCH")) {
		/* TODO: Handle SEARCH */
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"FLAGS")) {
		int i,j;

		if (M_b) {
		    M_b->flag_bits = 0;

		    for (i = 2; i < M_a->read_tokens.token_count; i++) {
			int found = 0;
			if (imap_atom == M_a->
			    read_tokens.tokens[i].imap_token) {
			    for (j = 0; 
				 j < sizeof IMAP_flags / 
				     sizeof (struct imap_flag);
				 j++) {
				if (0 == istrcmp(M_a->
						 read_tokens.tokens[i].str,
						 IMAP_flags[j].flag)) {
				    M_b->flag_bits |= IMAP_flags[j].mask_imap;
				    found++;
				    break;
				}			
			    }
			    if (!found) {
				DPRINT(Debug,9,(&Debug, 
						"parse_response: Unknown flag %s\n",
						M_a->read_tokens.tokens[i].str));
			    }
			}
		    }
		    DPRINT(Debug,9,(&Debug, 
				"parse_response: (known) Flags:"));
		    for (j = 0;
			 j < sizeof IMAP_flags / sizeof (struct imap_flag);
			 j++) {
			if ((M_b->flag_bits & IMAP_flags[j].mask_imap)
			    == IMAP_flags[j].mask_imap)
			    DPRINT(Debug,9,(&Debug, " %s",IMAP_flags[j].flag));
		    }
		    DPRINT(Debug,9,(&Debug, "\n"));
		} else {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Unexpected FLAGS response\n"));
		}
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"QUOTAROOT")) {
		
		if (M_p && ! (M_p->quota)) {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Got QUOTAROOT response -- setting quota for mailbox\n"));

		    init_imap_quota_mbx(con);		    		    		    
		}
		
		if (con->quota) {

		    if (M_a->read_tokens.token_count > 2) {
			parse_imap_quotaroot(con->quota,&(M_a->read_tokens.tokens[2]),
					     M_a->read_tokens.token_count-2);
		    } else {
			 DPRINT(Debug,1,(&Debug, 
					 "parse_response: Empty QUOTAROOT response\n"));
		    }

		} else {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Unexpected QUOTAROOT response\n"));

		}
		    

	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"QUOTA")) {

		if (M_p && ! (M_p->quota)) {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Got QUOTA response -- setting quota for mailbox\n"));

		    init_imap_quota_mbx(con);		    		    		    
		}


		if (con->quota) {

		    if (M_a->read_tokens.token_count > 2) {
			parse_imap_quota(con->quota,&(M_a->read_tokens.tokens[2]),
					 M_a->read_tokens.token_count-2);
		    } else {
			 DPRINT(Debug,1,(&Debug, 
					 "parse_response: Empty QUOTA response\n"));
		    }
		    
		} else {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Unexpected QUOTA response\n"));

		}		
	    } else {
		if ((unsigned long)M_a->read_tokens.tokens[1].str_len > (unsigned long)INT_MAX) {
		    DPRINT(Debug,20,(&Debug, 
				     "parse_response: unexpected %.20s... response (unsupported imap_other_keyword)\n",
				     M_a->read_tokens.tokens[1].str));
		    
		} else {		
		    int len = M_a->read_tokens.tokens[1].str_len;
		    char * A = "";
		    
		    if (len > 20) {
			len=20;
			A = "...";
		    }
		    DPRINT(Debug,20,(&Debug, 
				     "parse_response: unexpected %.*s%s response (unsupported imap_other_keyword)\n",
				     len,
				     M_a->read_tokens.tokens[1].str,A));
		}
	    }
	    break;
	case imap_number:
	    if (imap_other_keyword == M_a->read_tokens.tokens[2].imap_token) {

		if (0 == istrcmp(M_a->read_tokens.tokens[2].str,"RECENT")) {
		    if (M_b) {
			if ((unsigned long)M_a->read_tokens.tokens[1].value >
			    (unsigned long)INT_MAX) {

			    DPRINT(Debug,20,(&Debug, 
					     "parse_response: %lu RECENT unsupported\n",
					     (unsigned long)M_a->read_tokens.tokens[1].value));

			    M_b->num_recent = INT_MAX;
			} else {			
			    M_b->num_recent = 
				M_a->read_tokens.tokens[1].value;
			}

		    } else {
			DPRINT(Debug,1,(&Debug, 
					"parse_response: Unexpected RECENT response\n"));
		    }
		} else  if (0 == istrcmp(M_a->read_tokens.tokens[2].str,
					 "EXISTS")) {

		    if ((unsigned long)M_a->read_tokens.tokens[1].value >=
			(unsigned long)INT_MAX) {

			 DPRINT(Debug,20,(&Debug, 
					  "parse_response: %lu EXISTS unsupported\n",
					  (unsigned long)M_a->read_tokens.tokens[1].value));

		    } else {
			
			int val = M_a->read_tokens.tokens[1].value ;

			DPRINT(Debug,9,(&Debug, 
					"parse_response: (mailbox index %d) EXISTS\n",
					val));			
			if (M_b) {
			    /*
			      M_b->reference_count == number of messages + 1
			      M_b->references [0]     is not used
			    */
			    
			    if (val+1 < M_b->reference_count) {
				DPRINT(Debug,1,(&Debug, 
						"parse_response: Bad EXISTS value: %d\n",
						val));
			    } else if (val+1 > M_b->reference_count) {
				size_t new_size = 0;

				if (safe_array_size(&new_size,
						    val +1,
						    sizeof (M_b->references[0]))) {

				    struct imap_reference * new_array =
					realloc(M_b->references,new_size);

				    if (new_array) {
				    
					DPRINT(Debug,10,(&Debug,						 
							 "parse_response: %d mails, references array %zu bytes, array %p\n",
							 val,new_size,new_array));
					M_b->references = new_array;
					
					while (M_b->reference_count < val+1) {
					    struct imap_reference *item = 
						&(M_b->references[M_b->reference_count]);
					    zero_reference (item);
					    M_b->reference_count++;
					}
				    } else {
					int err = errno;
					
					DPRINT(Debug,1,(&Debug,						 
							"parse_response: %d mails, references array %zu bytes, can't allocate: %s (errno=%d)\n",
							val,new_size,strerror(err),err));
				    }
				} else {
				    DPRINT(Debug,1,(&Debug,						 
						    "parse_response: %d mails, bad size\n",
						    val));
				}
			    }
			} else {
			    DPRINT(Debug,1,(&Debug, 
					    "parse_response: Unexpected RECENT response\n"));
			}
		    }
		    
		} else  if (0 == istrcmp(M_a->read_tokens.tokens[2].str,
					 "EXPUNGE")) {

		    if ((unsigned long)M_a->read_tokens.tokens[1].value >
			(unsigned long)INT_MAX) {

			DPRINT(Debug,20,(&Debug, 
					 "parse_response: %lu EXPUNGE unsupported\n",
					 (unsigned long)M_a->read_tokens.tokens[1].value));

		    } else {
			int val = M_a->read_tokens.tokens[1].value ;

			DPRINT(Debug,9,(&Debug, 
					"parse_response: (mailbox index %d) EXPUNGE\n",
					val));

			if (M_b) {
			    
			    /* M_b->references[0] is not used ...
			       M_b->references[ 1 .. num of messages ] is on use
			       
			       M_b->reference_count == num of messages +1
			    */
			    
			    M_b -> got_EXPUNGE = 1;
			    
			    if (val < 0 || val >= M_b->reference_count) {
				DPRINT(Debug,1,(&Debug, 
						"parse_response: Bad EXPUNGE value: %d\n",
						val));
			    } else {
				int i;
				imap_free_reference (& (M_b->references[val]));
				
				for (i = val+1; i < M_b->reference_count; i++) {
				    M_b->references[i-1] = M_b->references[i];
				}
				M_b->reference_count--;			
			    }
			} else {
			    DPRINT(Debug,1,(&Debug, 		   
					    "parse_response: Unexpected EXPUNGE response\n"));
			}
		    }
		    
		} else  if (0 == istrcmp(M_a->read_tokens.tokens[2].str,
					 "FETCH")) {


		    if ((unsigned long)M_a->read_tokens.tokens[1].value >
			(unsigned long)INT_MAX) {

			DPRINT(Debug,20,(&Debug, 
					 "parse_response: %lu FETCH unsupported\n",
					 (unsigned long)M_a->read_tokens.tokens[1].value));

		    } else {
			int val = M_a->read_tokens.tokens[1].value ;
			
			DPRINT(Debug,9,(&Debug, 
					"parse_response: (mailbox index %d) FETCH\n",val));
			
			if (M_b) {
			    /* M_b->references[0] is not used ...
			       M_b->references[ 1 .. num of messages ] is on use
			       
			       M_b->reference_count == num of messages +1
			    */
			    
			    if (val < 0 || val >= M_b->reference_count) {
				DPRINT(Debug,1,(&Debug, "parse_response: Bad FETCH value: %d\n",
						val));
			    } else {
				struct imap_token * ptr  = 
				    M_a->read_tokens.tokens+3;
				int                left = 
				    M_a->read_tokens.token_count-3;
				struct imap_token * lst;
				int                 lst_len;
				
				if (!scan_list(&ptr,&left,&lst,&lst_len) ||
				    left > 0) {
				    DPRINT(Debug,1,(&Debug, 
						    "parse_response: scan_list failed to parse FETCH response list\n"));
				    lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
						      "Bad response from IMAP server"));
				    return 0;
				} else {
				    process_fetch_items(M_b->references+val,
							lst,lst_len);
				}
			    }

			} else {
			    DPRINT(Debug,1,(&Debug, 
					"parse_response: Unexpected FETCH response\n"));
			}
		    }
		} else {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Unsupported data %s\n",
				    M_a->read_tokens.tokens[2].str));
		}
	    } else {
		DPRINT(Debug,1,(&Debug, 
				"parse_response: token[2] type =%d\n",
				M_a->read_tokens.tokens[2].imap_token));
		lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
				  "Bad response from IMAP server"));
		return 0;
	    }
	    break;
	default:
	    DPRINT(Debug,1,(&Debug, 
			    "parse_response: token[1] type =%d\n",
			    M_a->read_tokens.tokens[1].imap_token));
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
			      "Bad response from IMAP server"));
	    return 0;
	}
	break;
    case imap_atom:
	if (M_a->current_NOOP_tag &&
	    0 == strcmp(M_a->current_NOOP_tag,
			M_a->read_tokens.tokens[0].str)) {
	    free(M_a->current_NOOP_tag);
	    M_a->current_NOOP_tag = NULL;
	    M_a->NOOP_start       = NO_schedule_timelimit;
	    
	    DPRINT(Debug,9,(&Debug, 
			    "parse_response: Response for idle timer NOOP command -- ignored.\n"));
	    return 1;
	}
	if (! M_a->current_tag  ||
	    0 != strcmp(M_a->current_tag,
			M_a->read_tokens.tokens[0].str)) {
	    DPRINT(Debug,1,(&Debug, "parse_response: bad tag %s\n",
			    M_a->read_tokens.tokens[0].str));
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
			      "Bad response from IMAP server"));
	    return 0;
	}

	free(M_a->current_tag);
	M_a->current_tag = NULL;

	if (M_a->command_result_code) {
	    free(M_a->command_result_code);
	    M_a->command_result_code = NULL;
	}

	switch (M_a->read_tokens.tokens[1].imap_token) {
	    struct string *Text;
	    char *code;   /* Do not free() this -- this is just pointer
			     to read_tokens area
			  */	    
	case imap_status_keyword:
	    Text = NULL;
	    code = NULL;
	    pick_text(M_a,&Text,&code);

	    if (code)
		M_a->command_result_code = safe_strdup(code);

	    if (code && Text && 0 == istrcmp(code,"ALERT")) {
		lib_transient(CATGETS(elm_msg_cat, MeSet,MeImapAlert,
				      "IMAP alert: %S"),
			      Text);		
	    } else if (code && 0 == istrcmp(code,"READ-ONLY")) {
		if (M_b) {
		    M_b->folder_status &= ~IMAP_writable;
		    DPRINT(Debug,5,(&Debug, 
				    "parse_response: Folder is read-only\n"));
		} else {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Unexpected READ-ONLY response\n"));
		}
	    } else if (code && 0 == istrcmp(code,"READ-WRITE")) {
		if (M_b) {
		    M_b->folder_status |= IMAP_writable;
		    DPRINT(Debug,5,(&Debug, 
				    "parse_response: Folder is read-write\n"));
		} else {
		    DPRINT(Debug,1,(&Debug, 
				    "parse_response: Unexpected READ-WRITE response\n"));
		}
	    }

	    if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"NO")) {
		M_a->imap_state = IMAP_command_ready_NO;
		if (Text)
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeImapError,
				      "IMAP server: %S"),
			      Text);		
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"BAD")) {
		M_a->imap_state = IMAP_command_ready_BAD;	    
		if (Text)
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeImapError,
				      "IMAP server: %S"),
			      Text);
	    } else if (0 == istrcmp(M_a->read_tokens.tokens[1].str,"OK")) {
		/* Don't forget closing state ... */
		if (IMAP_closing != M_a->imap_state)
		    M_a->imap_state = IMAP_command_ready_OK;	    
	    } else {
		DPRINT(Debug,1,(&Debug, "parse_response: status=%s\n",
				M_a->read_tokens.tokens[1].str));
		lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
				  "Bad response from IMAP server"));
		if (Text)
		    free_string(&Text);
		return 0;
	    }
	    if (Text)
		free_string(&Text);
	    break;
	default:
	    DPRINT(Debug,1,(&Debug, "parse_response: token[1] type =%d\n",
			    M_a->read_tokens.tokens[1].imap_token));
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
			      "Bad response from IMAP server"));
	    return 0;
	}
	break;

    case imap_continue:
	/* TODO: Handle AUTHENTICATE */

	if (M_a->literal_mode != waiting_literal_continue) {
	    DPRINT(Debug,1,(&Debug, 
			    "parse_response: Unexpected continuation request\n"));
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
			      "Bad response from IMAP server"));
	    return 0;
	}
	M_a->literal_mode = continue_literal;

	if (!imap_gen_writable_data(M_a)) {
	    panic("MBX PANIC",__FILE__,__LINE__,"parse_response",
		  "Not literal data for writing (IMAP)",0);
	}

	DPRINT(Debug,9,(&Debug, 
			"parse_response=2 (continue literal sending)\n")); 
	return 2;
    default:
	DPRINT(Debug,1,(&Debug, "parse_response: token[0] type =%d\n",
			M_a->read_tokens.tokens[0].imap_token));
	lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponseImap,
			  "Bad response from IMAP server"));
	return 0;
    }

    return 1;
}

S_(ss_action_routine  imap_write_stream)
static int imap_write_stream P_((struct streamsched               * ss,
				 union ss_action_routine_data      data,
				 const struct schedule_timelimit * now));

S_(ss_action_routine  imap_idle_timer)
static int imap_idle_timer P_((struct streamsched              * ss,
			       union ss_action_routine_data      data,
			       const struct schedule_timelimit * now));

static int imap_read_stream (ss,data,now)
     struct streamsched *ss; 
     union ss_action_routine_data data;
     const struct schedule_timelimit * now;
{
    struct connection_cache *con = data.con;
    int n;
    int close_it = 0;
    int tok_ret;
    char * ERROR = NULL;

    if (!con || 
	con->type != &IMAP_connection ||
	con->C.stream != ss) {
	
	panic("MBX PANIC",__FILE__,__LINE__,"imap_read_stream",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug, 
		     "imap_read_stream: con=%p (%s@%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>"));

    if (! con->a.imap_con) {
	DPRINT(Debug,13,(&Debug,
                         "imap_read_stream=0 (NO IMAP CONNECTION!)\n"));
	return 0;
    }

    if (CON_error == con->state ||
	IMAP_error == con->a.imap_con->imap_state) {
	DPRINT(Debug,13,(&Debug, 
			 "imap_read_stream=0 (stream need closing)\n"));
	return 0;
    }

    if (con->a.imap_con->wanna_literal) {

	if ( (unsigned long)con->a.imap_con->wanted_literal_len >
	     (unsigned long)con->a.imap_con->read_buffer.read_len) {

	    int wanted_read = 0;

	    if ((unsigned long)con->a.imap_con->wanted_literal_len >= INT_MAX) {

		DPRINT(Debug,13,(&Debug, 
				 "imap_read_stream=0: closing stream -- wanted_literal_len=%ld >= INT_MAX %d\n",
				 (unsigned long)con->a.imap_con->wanted_literal_len,
				 INT_MAX));
				 
		return 0;
	    }

	    wanted_read =
		(unsigned long)con->a.imap_con->wanted_literal_len -
		(unsigned long)con->a.imap_con->read_buffer.read_len;
		
	    
	    n = ReadFromStream(ss, &(con->a.imap_con->read_buffer), wanted_read);
	    
	} else {
	    DPRINT(Debug,13,(&Debug, 
			     "imap_read_stream: wanted_literal_len=%ld <= read_len=%zu\n",
			     (unsigned long)con->a.imap_con->wanted_literal_len,
			     con->a.imap_con->read_buffer.read_len));
	    
	    n = ReadFromStream(ss, &(con->a.imap_con->read_buffer),-1);	    	    
	}
	
    } else
	n = ReadFromStream(ss, &(con->a.imap_con->read_buffer),-1);

    DPRINT(Debug,50,(&Debug, "Buffer content="));
#ifdef DEBUG
    if (Debug.active >= 50) {
	int i;

	for (i = 0; i < con->a.imap_con->read_buffer.read_len; i++) {
	    DPRINT(Debug,50,(&Debug, " %d",
			     con->a.imap_con->read_buffer.read_buffer[i]));
	}
	DPRINT(Debug,50,(&Debug, "\n (message?)="));
	for (i = 0; i < con->a.imap_con->read_buffer.read_len; i++) {
	    DPRINT(Debug,50,(&Debug, " %c",
			     isascii(con->a.imap_con->read_buffer.
				     read_buffer[i]) &&
			     isprint(con->a.imap_con->read_buffer.
				     read_buffer[i]) ? 
			     con->a.imap_con->read_buffer.read_buffer[i] :
			     '.'));
	}
    }
#endif
    DPRINT(Debug,50,(&Debug, "\n"));
    
    /* Parse response ... */
    do {
	tok_ret = tokenize(con->a.imap_con);
	if (tok_ret < 0) {
	    DPRINT(Debug,7,(&Debug, 
			    "imap_read_stream: Failed to tokenize response ...\n"));
	    close_it = 1;
	} else if (tok_ret > 0) {  /* Tokenized */
	    int X;
	    if (!(X = parse_response(con))) {
		DPRINT(Debug,7,(&Debug, 
				"imap_read_stream: Failed to parse response ...\n"));
		close_it = 1;
	    } else if (2 == X) {		
		/* Restart writing of literal ... */

		ConfigStream2(con->C.stream,
			      imap_read_stream,imap_write_stream,
			      imap_idle_timer,
			      ss_nofree_action_rt_data,
			      ss_noinc_action_rt_data_refcount,
			      imap_badpid_stream,
			      imap_idle_alive_interval > MIN_IMAP_TIMEOUT ?
			      imap_idle_alive_interval : MIN_IMAP_TIMEOUT,
			      data);
		DPRINT(Debug,13,(&Debug, 
				 "imap_read_stream: Continue literal sending...\n"));
	    }
	    
	    free_tokens(& (con->a.imap_con->read_tokens));
	}	
    } while (0  < tok_ret && !close_it &&
	     con->a.imap_con->read_buffer.read_len > 0);
    
    if (n == 0) {
	if (con->a.imap_con->imap_state != IMAP_closing)
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeConnectionClosedImap,
			      "Connection closed to IMAP server"));
	close_it = 1;
    }

    if (n < 0) {
	ERROR = RemoveStreamError(ss);
	if (ERROR) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeErrorReadingImap,
			      "Error reading from IMAP server: %s"),
		      ERROR);
	    close_it = 1;
	}
    }

    if (ERROR) {
	free(ERROR);
	ERROR = NULL;
    }

    if (close_it) {
	con->a.imap_con->imap_state = IMAP_error;
	DPRINT(Debug,9,(&Debug, 
			"imap_read_stream=0 (stream need closing)\n"));
	return 0;
    }
    DPRINT(Debug,13,(&Debug, 
		     "imap_read_stream=1\n"));
    return 1;
}

static  enum badpid_status imap_badpid_stream(ss,data,badpid)
     struct streamsched *ss;
     union  ss_action_routine_data data;
     int badpid;
{
    struct connection_cache *con = data.con;
    if (!con || 
	con->type != &IMAP_connection ||
	con->C.stream != ss) {
	
	panic("MBX PANIC",__FILE__,__LINE__,"imap_badpid_stream",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug, 
		     "imap_badpid_stream: con=%p (%s@%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>"));
    
    con->state        = CON_error;
    con->seen_badpid  = badpid;

    if (con->a.imap_con) {
	con->a.imap_con->imap_state = IMAP_error;
    } else {
	DPRINT(Debug,13,(&Debug,
			 "imap_badpid_stream: NO IMAP CONNECTION!\n"));
    }
	     
    
    return badpid_remove       /* remove action */;
}

static int imap_write_stream (ss,data,now)
     struct streamsched *ss;
     union ss_action_routine_data data;
     const struct schedule_timelimit * now;
{
    struct connection_cache *con = data.con;
    char * ERROR = NULL;
    int n;


    if (!con || 
	con->type != &IMAP_connection ||
	con->C.stream != ss) {
	
	panic("MBX PANIC",__FILE__,__LINE__,"imap_write_stream",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug, 
		"imap_write_stream: folder=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));

    if (!con->a.imap_con) {
	DPRINT(Debug,7,(&Debug,
                        "imap_write_stream=0 (NO imap connection)\n"));
	return 0;
    }


    if (!imap_gen_writable_data(con->a.imap_con)) {
	DPRINT(Debug,7,(&Debug, 
			"imap_write_stream: NO DATA TO WRITE!\n"));
    }

    if (CON_error == con->state ||
	IMAP_error == con->a.imap_con->imap_state) {
	DPRINT(Debug,7,(&Debug, 
			"imap_write_stream=0 (stream need closing)\n"));
	return 0;
    }

    n = WriteToStream(ss,& (con->a.imap_con->write_buffer));


    if (n > 0) {
	int i;
	
	DPRINT(Debug,70,(&Debug, "Write content="));
	for (i = 0; i < n; i++) {
	    DPRINT(Debug,70,(&Debug, " %d",
			con->a.imap_con->write_buffer.write_buffer[i]));
	}
	DPRINT(Debug,70,(&Debug, "\n    (text?)="));
	for (i = 0; i < n; i++) {
	    DPRINT(Debug,70,(&Debug, " %c",
			     isascii(con->a.imap_con->write_buffer.write_buffer[i]) 
			     &&
			     isprint(con->a.imap_con->write_buffer.write_buffer[i]) 
			     ? 
			     con->a.imap_con->write_buffer.write_buffer[i] :
			     '.'));
	}
	DPRINT(Debug,70,(&Debug, "\n"));

	cut_Write_Buffer(& (con->a.imap_con->write_buffer),n);
	
    } else if (n < 0) {
	ERROR = RemoveStreamError(ss);

	DPRINT(Debug,7,(&Debug, 
			"imap_write_stream: write error %s\n",
			ERROR ? ERROR : "(none)"));
    }

    if (!con->a.imap_con->write_buffer.write_len) {
	if (!imap_gen_writable_data(con->a.imap_con)) {
	    if (ERROR) {
		free(ERROR);
		ERROR = NULL;
	    }
	    DPRINT(Debug,13,(&Debug, 
			     "imap_write_stream=0 -- data written\n"));

	    return 0;
	}
    }

    if (ERROR) {
	lib_error(CATGETS(elm_msg_cat, MeSet,MeErrorWritingImap,
			  "Error writing to IMAP server: %s"),
		  ERROR);
	con->a.imap_con->imap_state = IMAP_error;
	DPRINT(Debug,7,(&Debug, 
			"imap_write_stream=0 (stream need closing)\n"));
	free(ERROR);
	ERROR = NULL;
        return 0;	
    }

    DPRINT(Debug,13,(&Debug, "imap_write_stream=1\n"));
    return 1;
}

#define MAX_TAG 99999

static char * tag_generator P_((void));
static char * tag_generator()
{
    /* NOTE: There is no need to have per imap connection tag 
       generating sequence
    */ 
    static char tag_letter = 'A';
    static int  tag_number = 0;

    tag_number++;
    if (tag_number > MAX_TAG) {
	tag_number = 1;
	tag_letter++;
	if ('Z' == tag_letter) {
	    tag_letter = 'A';
	    DPRINT(Debug,13,(&Debug, 
			     "tag_generator: Tag wrapping around\n"));
	}
    }

    return elm_message(FRM("%c%05d"),tag_letter,tag_number);
}

static int imap_idle_timer (ss,data,now)
     struct streamsched              * ss;
     union ss_action_routine_data      data;
     const struct schedule_timelimit * now;
{
    struct connection_cache *con = data.con;

    if (!con || 
	con->type != &IMAP_connection ||
	con->C.stream != ss) {

	panic("MBX PANIC",__FILE__,__LINE__,"imap_idle_timer",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug, 
		"imap_idle_timer: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));

    if (CON_error == con->state ||
	IMAP_error == con->a.imap_con->imap_state) {
	DPRINT(Debug,7,(&Debug, 
			"imap_idle_timer=0 (stream need closing)\n"));
	return 0;
    }

    if (con->a.imap_con->current_NOOP_tag) {
	struct sec_ms interval;
	int r;
	
	if (0 != (r = schedule_time_passed(& (con->a.imap_con->NOOP_start),
					   now,
					   &interval))) {


	    
	    /* 'now' is time when wait ended, probably not current time */
	    
	    DPRINT(Debug,13,(&Debug, 
			     "imap_idle_timer: previous NOOP command executed for %d sec %d msec.\n",
			     interval.timeout_sec, interval.timeout_ms));
	    
	    if (con->C.host) {
		con->a.imap_con->stalled = 1;
		
		if (interval.timeout_sec >= 0 &&
		    interval.timeout_sec < 10 &&
		    interval.timeout_ms  <= 999 &&
		    0 != (r & SCHEDULE_TP_HAVE_MSEC)) {

		    lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPstalledMS,
					  "IMAP connection to %s stalled for %d.%03d seconds."),
				  con->C.host,
				  interval.timeout_sec,
				  interval.timeout_ms);
		} else if (interval.timeout_sec >= 0) {
		    lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPstalledSec,
					  "IMAP connection to %s stalled for %d seconds."),
				  con->C.host,
				  interval.timeout_sec);		    
		} 
	    }
	}
	
	DPRINT(Debug,13,(&Debug, 
			 "imap_idle_timer=1 (previous NOOP command is not finished)\n"));
	return 1;
    }

    if (con->a.imap_con->token_index >= 0 || 
	con->a.imap_con->imap_state != IMAP_idle) {
	DPRINT(Debug,13,(&Debug, 
			 "imap_idle_timer=1 (NOT idle)\n"));
	return 1;	
    }
    con->a.imap_con->current_NOOP_tag = tag_generator();
    con->a.imap_con->token_index      = -2;    /* Produce NOOP command */
    con->a.imap_con->imap_state       = IMAP_command;   /* For imap_gen_writable_data */

    if (!imap_gen_writable_data(con->a.imap_con)) {
	panic("MBX PANIC",__FILE__,__LINE__,"imap_idle_timer",
	      "No NOOP command for writing (IMAP)",0);
    }

    ConfigStream2(con->C.stream,imap_read_stream,imap_write_stream,
		  imap_idle_timer,
		  ss_nofree_action_rt_data,
		  ss_noinc_action_rt_data_refcount,
		  imap_badpid_stream,
		  imap_idle_alive_interval > MIN_IMAP_TIMEOUT ?
		  imap_idle_alive_interval : MIN_IMAP_TIMEOUT,			     
		  data);

    /* Mark it again IDLE ... */
    con->a.imap_con->imap_state = IMAP_idle;
    con->a.imap_con->NOOP_start = *now;
    
    return 1;
}

static void imap_wait_c P_((struct connection_cache *con,
			    struct cancel_data  * cd
			    /* Allow cancelation (Ctrl-C)
			       check of remote mailbox
			    */));
static void imap_wait_c(con,cd)
     struct connection_cache *con;
     struct cancel_data  * cd
     /* Allow cancelation (Ctrl-C)
	check of remote mailbox
     */;
{
    int count = 0;
    
    if (!con->a.imap_con) {
	DPRINT(Debug,7,(&Debug,"imap_wait_c: No imap connection\n"));
	return;
    }

    while (CON_error              != con->state &&
	   IMAP_command_ready_OK  != con->a.imap_con->imap_state &&
	   IMAP_command_ready_NO  != con->a.imap_con->imap_state &&
	   IMAP_command_ready_BAD != con->a.imap_con->imap_state &&
	   IMAP_closing           != con->a.imap_con->imap_state &&
	   IMAP_error             != con->a.imap_con->imap_state &&
           NULL                   != con->C.stream) {

	count++;
	DPRINT(Debug,13,(&Debug,
			 "imap_wait_c: waiting #%d\n",
			 count));
	
	WaitStreamFor_c(con->C.stream,SS_read_act,cd);
	
	if (cd && is_canceled(cd)) {
	    DPRINT(Debug,12,(&Debug,
			     "imap_wait_c: Wait canceled\n"));
		
	    return;
	}
    }

    if (CON_error == con->state ||
	IMAP_error  == con->a.imap_con->imap_state) {
	DPRINT(Debug,7,(&Debug,"imap_wait_c: Have error\n"));
    } else {
	DPRINT(Debug,13,(&Debug,
			 "imap_wait_c: Wait done ok\n"));
    }
}

void imap_wait(con)
     struct connection_cache *con;
{
    int count = 0;
    
    if (!con->a.imap_con) {
	DPRINT(Debug,7,(&Debug,"imap_wait: No imap connection\n"));
	return;
    }

    while (CON_error              != con->state &&
	   IMAP_command_ready_OK  != con->a.imap_con->imap_state &&
	   IMAP_command_ready_NO  != con->a.imap_con->imap_state &&
	   IMAP_command_ready_BAD != con->a.imap_con->imap_state &&
	   IMAP_closing           != con->a.imap_con->imap_state &&
	   IMAP_error             != con->a.imap_con->imap_state &&
           NULL                   != con->C.stream) {

	count++;
	DPRINT(Debug,13,(&Debug,
			 "imap_wait: waiting #%d\n",
			 count));

	
	WaitStreamFor(con->C.stream,SS_read_act);
	/* Should user to have some way to interrupt progress ? 

	    imap_wait_c() uses interruptible wait
	 */
    }

    if (CON_error == con->state ||
	IMAP_error  == con->a.imap_con->imap_state) {
	DPRINT(Debug,7,(&Debug,"imap_wait: Have error\n"));
    } else {
	DPRINT(Debug,13,(&Debug,
			 "imap_wait: Wait done ok\n"));
    }
}

void imapcon_idle_wait(con)
     struct connection_cache *con;
{
   struct IMAP_CON * Ma;

    if (con->type != &IMAP_connection) {
	panic("MBX PANIC",__FILE__,__LINE__,"imapcon_idle_wait",
	      "Wrong type connection attached to folder",0);
    }
   
    Ma = con->a.imap_con;

    if (!Ma)
	panic("MBX PANIC",__FILE__,__LINE__,"imapcon_idle_wait",
              "No imap connection",0);

    if (Ma->current_NOOP_tag &&
	CON_error != con->state &&
	Ma->imap_state != IMAP_closing &&
	Ma->imap_state != IMAP_error) {
	DPRINT(Debug,3,(&Debug, 
			"imapcon_idle_wait: Waiting for idle NOOP complete...\n"));
	while(Ma->current_NOOP_tag && 
	      Ma->imap_state != IMAP_closing &&
	      Ma->imap_state != IMAP_error &&
	      CON_error != con->state &&
	      con->C.stream) {
	    WaitStreamFor(con->C.stream,SS_read_act);
	    /* TODO:    Should user to have some way to interrupt progress ? */
	}
	
	if (CON_error == con->state ||
	    IMAP_error == con->a.imap_con->imap_state) {
	    DPRINT(Debug,3,(&Debug, 
			    "imapcon_idle_wait: idle NOOP wait ended, have error\n"));
	    
	} else {
	    DPRINT(Debug,3,(&Debug, 
			    "imapcon_idle_wait: idle NOOP wait completed\n"));
	}
    }
    if (IMAP_idle != Ma->imap_state) {
	DPRINT(Debug,3,(&Debug, 
		    "imapcon_idle_wait: Waiting for command complete...\n"));

	imap_wait(con);
	DPRINT(Debug,3,(&Debug, 
			"imapcon_idle_wait: command wait completed\n"));
    }

    if (CON_error == con->state ||
	IMAP_error == con->a.imap_con->imap_state) {
	DPRINT(Debug,3,(&Debug, 
			"imapcon_idle_wait: Have error\n"));
    }    
}

static int imap_cmd_ok_head P_((struct connection_cache *con,
				imap_states *cmd_result,
				char        **extra_code));
static int imap_cmd_ok_head(con,cmd_result,extra_code)
     struct connection_cache *con;
     imap_states *cmd_result;
     char        **extra_code;
{
    int status = 0;

    if (extra_code) 
	*extra_code = NULL;

    if (IMAP_idle == con->a.imap_con->imap_state) {
	DPRINT(Debug,7,(&Debug, 
			"imap_cmd_ok_head: WRONG STATE\n"));
	status = 0;
	*cmd_result = IMAP_error;
	goto clean;
    }
    status = 1;
    
 clean:
    DPRINT(Debug,13,(&Debug, 
		     "imap_cmd_ok_head=%d\n",
		     status));
    return status;
}

static int imap_cmd_ok_tail P_((struct connection_cache *con,
				imap_states *cmd_result,
				char        **extra_code));
static int imap_cmd_ok_tail(con,cmd_result,extra_code)
     struct connection_cache *con;
     imap_states *cmd_result;
     char        **extra_code;
{
    int status = 0;

    status = IMAP_command_ready_OK == con->a.imap_con->imap_state;
    *cmd_result = con->a.imap_con->imap_state;

    if (IMAP_command_ready_OK == con->a.imap_con->imap_state ||
	IMAP_command_ready_NO == con->a.imap_con->imap_state ||
	IMAP_command_ready_BAD == con->a.imap_con->imap_state) {
	
	if (extra_code) {
	    *extra_code = con->a.imap_con->command_result_code;
	    con->a.imap_con->command_result_code = NULL;	    
	}
    }

    if (CON_error == con->state ||
	IMAP_error == con->a.imap_con->imap_state) {
	if (con->C.stream) {
	    DPRINT(Debug,7,(&Debug, 
			    "imap_cmd_ok_tail: Closing stream"));

	    if (con->seen_badpid) {
		DPRINT(Debug,10,(&Debug,", not our stream, owner pid %d",
				 con->seen_badpid));	    
	    }    
	    DPRINT(Debug,10,(&Debug,"\n"));
	    
	    FreeStreamStack0(&(con->C.stream),
			     con->seen_badpid,
			     1 /* Force clearing of stack even when this is not last reference */);
	}	
	status = 0;
	goto clean;
    }
    
 clean:
    DPRINT(Debug,13,(&Debug, 
		     "imap_cmd_ok_tail=%d\n",
		     status));
    return status;
}

int imap_command_ok_c(con,cmd_result,extra_code,cd)
     struct connection_cache *con;
     imap_states *cmd_result;
     char        **extra_code;
     struct cancel_data  * cd
     /* Allow cancelation (Ctrl-C)
	check of remote mailbox
     */;
{
    int status = 0;
    
    DPRINT(Debug,12,(&Debug, 
		"imap_command_ok_c: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));


    if (!imap_cmd_ok_head(con,cmd_result,extra_code)) {
	status = 0;
	goto clean;
    }
    
    imap_wait_c(con,cd);
    
    if (cd && is_canceled(cd)) {
	DPRINT(Debug,12,(&Debug,
			 "imap_command_ok_c: Wait canceled\n"));
	
	status = 0;
	goto clean;
    }
	        
    if (!imap_cmd_ok_tail(con,cmd_result,extra_code)) {
	status = 0;
	goto clean;
    }
	
    status = 1;
    
 clean:
    DPRINT(Debug,12,(&Debug, 
		     "imap_command_ok_c=%d\n",
		     status));
    return status;
}

int imap_command_ok(con,cmd_result,extra_code) 
     struct connection_cache *con;
     imap_states *cmd_result;
     char **extra_code;
{
    int status = 0;
    
    DPRINT(Debug,12,(&Debug, 
		"imap_command_ok: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));


    if (!imap_cmd_ok_head(con,cmd_result,extra_code)) {
	status = 0;
	goto clean;
    }
    imap_wait(con);

    if (!imap_cmd_ok_tail(con,cmd_result,extra_code)) {
	status = 0;
	goto clean;
    }
	
    status = 1;
    
 clean:
    DPRINT(Debug,12,(&Debug, 
		     "imap_command_ok=%d\n",
		     status));
    return status;
}


static void start_imap_cmd_fail P_((struct connection_cache *con));
static void start_imap_cmd_fail(con)
     struct connection_cache *con;
{
    if (con->C.stream) {
	
	DPRINT(Debug,12,(&Debug, 
			 "start_imap_cmd_fail: Closing stream"));
	if (con->seen_badpid) {
	    DPRINT(Debug,10,(&Debug,", not our stream, owner pid %d",
			     con->seen_badpid));	    
	}    
	DPRINT(Debug,10,(&Debug,"\n"));
	
	FreeStreamStack0(& (con->C.stream),
			 con->seen_badpid,
			 1 /* Force clearing of stack even when this is not last reference */);
    }	
}

static int start_imap_cmd_head P_((struct connection_cache *con));
static int start_imap_cmd_head(con)
     struct connection_cache *con;
{
    int status = 0;
    
    if (IMAP_closing == con->a.imap_con->imap_state) {
	DPRINT(Debug,12,(&Debug, 
			 "start_imap_cmd_head: imap_state == IMAP_closing\n"));
	goto failure;
    }

    if (CON_error == con->state ||
	IMAP_error == con->a.imap_con->imap_state) {
	if (IMAP_error == con->a.imap_con->imap_state) {
	    DPRINT(Debug,12,(&Debug, 
			     "start_imap_cmd_head: imap_state == IMAP_error\n"));
	}
	
	if (CON_error == con->state) {
	    DPRINT(Debug,12,(&Debug, 
			     "start_imap_cmd_head: connection state = CON_error\n"));
	}
	    
    failure:
	start_imap_cmd_fail(con);
	status = 0;
	goto clean;
    }

    status = 1;

 clean:
    DPRINT(Debug,13,(&Debug, 
		     "start_imap_cmd_head=%d\n",
		     status));
    
    return status;
}

static int start_imap_cmd_tail P_((struct connection_cache *con,
				   char * verb));
static int start_imap_cmd_tail(con,verb)
     struct connection_cache *con;
     char * verb;
{
    int status = 0;


    if (con->a.imap_con->current_tag) {
	DPRINT(Debug,12,(&Debug, 
			 "start_imap_cmd_tail: Tag %s still active\n",
			 con->a.imap_con->current_tag));
	status = 0;
	goto clean;	
    }

    con->a.imap_con->imap_state  = IMAP_command;
    free_tokens(& (con->a.imap_con->write_tokens));
    con->a.imap_con->token_index  = -1;
    con->a.imap_con->literal_mode = no_literal;

    if (NULL == con->C.stream) {
	DPRINT(Debug,12,(&Debug, 
			 "start_imap_cmd_tail: stream not open\n"));
	status = 0;
	goto clean;
    }

    con->a.imap_con->current_tag = tag_generator();
    imap_command_push_atom(con,verb);
    status          = 1; 

    DPRINT(Debug,15,(&Debug, 
		     "start_imap_cmd_tail: Command %s ...\n",verb));

 clean:
    DPRINT(Debug,13,(&Debug, 
		     "start_imap_cmd_tail=%d\n",
		     status));
    
    return status;
}

int start_imap_command_c(con,verb,cd)
     struct connection_cache *con;
     char * verb;
     struct cancel_data  * cd
     /* Allow cancelation (Ctrl-C)
	check of remote mailbox
     */;
{
    int status = 0;

    DPRINT(Debug,12,(&Debug, 
		     "start_imap_command_c: con=%p (%s@%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>"));
    if (! start_imap_cmd_head(con)) {
	status = 0;
	goto clean;
    }

    if (IMAP_idle != con->a.imap_con->imap_state) {
	DPRINT(Debug,7,(&Debug, 
			"start_imap_command_c: wrong state  ... waiting...\n"));
	
	imap_wait_c(con,cd);
	
	if (cd && is_canceled(cd)) {
	    DPRINT(Debug,7,(&Debug, 
			    "start_imap_command_c: ... wait ended (canceled)\n"));
	    status = 0;
	    goto clean;
	} else {
	    DPRINT(Debug,7,(&Debug, 
			    "start_imap_command_c: ... wait ended\n"));
	}

	if (IMAP_error == con->a.imap_con->imap_state) {
	    DPRINT(Debug,12,(&Debug, 
			     "start_imap_command_c: imap_state == IMAP_error\n"));

	    goto failure;	
	}

	if (IMAP_closing == con->a.imap_con->imap_state) {
	    DPRINT(Debug,12,(&Debug, 
			     "start_imap_command_c: imap_state == IMAP_closing\n"));

	failure:
	    start_imap_cmd_fail(con);
	    status = 0;
	    goto clean;
	}
    }

    if (! start_imap_cmd_tail(con,verb)) {
	status = 0;
	goto clean;
    }
    
    status          = 1;
    
 clean:
    DPRINT(Debug,12,(&Debug, 
		"start_imap_command_c=%d\n",
		status));
    return status;    
}

int start_imap_command(con,verb)
     struct connection_cache *con;
     char * verb;
{
    int status = 0;

    DPRINT(Debug,12,(&Debug, 
		"start_imap_command: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));

    if (! start_imap_cmd_head(con)) {
	status = 0;
	goto clean;
    }
    
    if (IMAP_idle != con->a.imap_con->imap_state) {
	DPRINT(Debug,7,(&Debug, 
		    "start_imap_command: wrong state  ... waiting...\n"));
	
	imap_wait(con);
	DPRINT(Debug,7,(&Debug, 
		    "start_imap_command: ... wait ended\n"));
	if (IMAP_error == con->a.imap_con->imap_state) {
	    DPRINT(Debug,12,(&Debug, 
			     "start_imap_command: imap_state == IMAP_error\n"));

	    goto failure;	
	}

	if (IMAP_closing == con->a.imap_con->imap_state) {
	    DPRINT(Debug,12,(&Debug, 
			     "start_imap_command: imap_state == IMAP_closing\n"));
	failure:
	    start_imap_cmd_fail(con);
	    status = 0;
	    goto clean;
	}
    }

    if (! start_imap_cmd_tail(con,verb)) {
	status = 0;
	goto clean;
    }
    
    status          = 1;
    
 clean:
    DPRINT(Debug,12,(&Debug, 
		"start_imap_command=%d\n",
		status));
    return status;    
}


void imap_command_push_atom(con,st)
     struct connection_cache *con;
     const char *st;
{
    struct imap_token T;

    T.imap_token    = imap_atom;
    T.str           = safe_strdup(st);
    T.str_len       = strlen(st);
    T.value         = 0;
    add_token(&(con->a.imap_con->write_tokens),T);
}

void imap_command_push_literal(con,len,st)
     struct connection_cache *con;
     int len;
     const char *st;
{
    struct imap_token T;

    if (len < 0) {
	panic("MBX PANIC",__FILE__,__LINE__,
	      "imap_command_push_literal","negative len",0);
    }

    T.imap_token    = imap_literal;
    T.str           = safe_malloc(len+1);
    memcpy(T.str,st,len);
    T.str[len]      = '\0';
    T.str_len       = len;
    T.value         = 0;
    add_token(&(con->a.imap_con->write_tokens),T);
}


void imap_command_push_string(con,st)
     struct connection_cache *con;
     const char *st;
{
    struct imap_token T;

    T.imap_token    = imap_string;
    T.str           = safe_strdup(st);
    T.str_len       = strlen(st);
    T.value         = 0;
    add_token(&(con->a.imap_con->write_tokens),T);
}

void imap_command_push_astring(con,st)
     struct connection_cache *con;
     const char *st;
{
    if ('\0' == st[0] ||
	strspn(st,"ABCDEFGHIJKLMNOPQRSTUVXYZabcdefghijklmnopqrstuvxyz0123456789")
	!= strlen(st))
	imap_command_push_string(con,st);
    else 
	imap_command_push_atom(con,st);
}


void imap_command_push_aliteral(con,st)
     struct connection_cache *con;
     const char *st;
{
    int len = strlen(st);

    if ('\0' == st[0] ||
	strspn(st,"ABCDEFGHIJKLMNOPQRSTUVXYZabcdefghijklmnopqrstuvxyz0123456789")
	!= len)
	imap_command_push_literal(con,len,st);
    else 
	imap_command_push_atom(con,st);
}


void imap_command_push_range(con,A,B)
     struct connection_cache *con;
     int A,B;
{
    char * st = elm_message(FRM("%d:%d"),A,B);
    imap_command_push_atom(con,st);
    free(st);
}


void imap_command_push_number(con,A)
     struct connection_cache *con;
     int A;
{
    char * st = elm_message(FRM("%d"),A);
    imap_command_push_atom(con,st);
    free(st);
}

void imap_command_push_list(con,l)
     struct connection_cache *con;
     const char **l;
{
    int i;
    struct imap_token T;

    T.imap_token    = imap_list_begin;
    T.str           = NULL;
    T.str_len       = 0;
    T.value         = 0;
    add_token(&(con->a.imap_con->write_tokens),T);

    for (i = 0; l[i]; l++)
	imap_command_push_atom(con,l[i]);

    T.imap_token    = imap_list_end;
    T.str           = NULL;
    T.str_len       = 0;
    T.value         = 0;
    add_token(&(con->a.imap_con->write_tokens),T);
}

void end_imap_command(con)
     struct connection_cache *con;
{
    union ss_action_routine_data data;
    
    DPRINT(Debug,12,(&Debug, 
		"end_imap_command: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));

    if (CON_error == con->state ||
	IMAP_error == con->a.imap_con->imap_state) {
	DPRINT(Debug,12,(&Debug, 
			 "end_imap_command:  On error state\n"));
	return;
    }

    if (IMAP_closing == con->a.imap_con->imap_state) {
	DPRINT(Debug,12,(&Debug, 
			 "end_imap_command:  Is closing\n"));
	return;
    }

    if (con->a.imap_con->imap_state != IMAP_command)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "end_imap_command","wrong state",0);

    if (!imap_gen_writable_data(con->a.imap_con)) {
	panic("MBX PANIC",__FILE__,__LINE__,"end_imap_command",
	      "No command for writing",0);
    }

    data.con = con;
    
    ConfigStream2(con->C.stream,imap_read_stream,imap_write_stream,
		  imap_idle_timer,
		  ss_nofree_action_rt_data,
		  ss_noinc_action_rt_data_refcount,
		  imap_badpid_stream,
		  imap_idle_alive_interval > MIN_IMAP_TIMEOUT ?
		  imap_idle_alive_interval : MIN_IMAP_TIMEOUT,			     
		  data);
}

/* ----------------------------------------------------------------------- */

/* IMAP connection */

static void  cache_zero_imap(con)
     struct connection_cache *con;
{
    DPRINT(Debug,11,(&Debug, "cache_zero_imap: con=%p\n",con));

    con->a.imap_con = safe_zero_alloc (sizeof (* (con->a.imap_con)));

    con->a.imap_con->capability_bits      = 0;
#ifdef USE_DLOPEN
    con->a.imap_con->capability_libs      = NULL;
    con->a.imap_con->capability_lib_count = 0;
#endif

    zero_Read_Buffer(&(con->a.imap_con->read_buffer));
    con->a.imap_con->wanna_literal      = 0;
    con->a.imap_con->wanted_literal_len = 0;

    con->a.imap_con->read_tokens.tokens          = NULL;
    con->a.imap_con->read_tokens.token_count     = 0;
    con->a.imap_con->tokenizer_state             = state_tag;

    con->a.imap_con->current_tag         = NULL;
    con->a.imap_con->imap_state          = IMAP_idle;
    con->a.imap_con->command_result_code = NULL;

    con->a.imap_con->write_tokens.tokens      = NULL;
    con->a.imap_con->write_tokens.token_count = 0;
    con->a.imap_con->token_index  = -1;

    zero_Write_Buffer(& (con->a.imap_con->write_buffer));
    con->a.imap_con->literal_mode = no_literal;

    con->a.imap_con->current_NOOP_tag = NULL;
    con->a.imap_con->NOOP_start       = NO_schedule_timelimit;
    con->a.imap_con->stalled      = 0;
}

static void cache_imap_clear_buffers P_((struct connection_cache *con));
static void cache_imap_clear_buffers(con)
     struct connection_cache *con;
{
    DPRINT(Debug,12,(&Debug, 
		"cache_imap_clear_buffers: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));
        
    con->a.imap_con->capability_bits = 0;
#ifdef USE_DLOPEN
    free_only_imap_capa_libs( & (con->a.imap_con->capability_libs),
			      & (con->a.imap_con->capability_lib_count));
#endif

    free_Read_Buffer(&(con->a.imap_con->read_buffer));

    con->a.imap_con->wanna_literal      = 0;
    con->a.imap_con->wanted_literal_len = 0;
    
    free_tokens(&(con->a.imap_con->read_tokens));

    con->a.imap_con->tokenizer_state = state_tag;

    if (con->a.imap_con->current_tag) {
	free(con->a.imap_con->current_tag);
	con->a.imap_con->current_tag = NULL;
    }
    con->a.imap_con->imap_state = IMAP_error;
    if (con->a.imap_con->command_result_code) {
	free(con->a.imap_con->command_result_code);
	con->a.imap_con->command_result_code = NULL;
    }

    free_tokens(&(con->a.imap_con->write_tokens));
    con->a.imap_con->token_index = -1;
    
    
    free_Write_Buffer ( &(con->a.imap_con->write_buffer));
    con->a.imap_con->literal_mode = no_literal;

    if (con->a.imap_con->current_NOOP_tag) {
	free(con->a.imap_con->current_NOOP_tag);
	con->a.imap_con->current_NOOP_tag = NULL;
    }
    con->a.imap_con->NOOP_start       = NO_schedule_timelimit;
    con->a.imap_con->stalled = 0;
}

#ifdef USE_DLOPEN
static struct imap_callbacks imap_commands = {
    imap_wait,
    imapcon_idle_wait,
    imap_command_ok,
    start_imap_command,
    imap_command_push_atom,
    imap_command_push_literal,
    imap_command_push_string,
    imap_command_push_astring,
    imap_command_push_aliteral,
    imap_command_push_range,
    imap_command_push_number,
    imap_command_push_list,
    end_imap_command,
    imap_clear_command
};
#endif

static void  cache_free_imap(con)
     struct connection_cache *con;
{    
    DPRINT(Debug,11,(&Debug, 
		     "cache_free_imap: con=%p (%s@%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>"));
    
    if (con->a.imap_con) {
	cache_imap_clear_buffers(con);

	bzero((void *)con->a.imap_con, sizeof (* (con->a.imap_con)));

	free(con->a.imap_con);
	con->a.imap_con = NULL;
    }

}

int imap_handle_greeting(con)
     struct connection_cache *con;
{
    int status = 0;
    imap_states res;

    union ss_action_routine_data data;
    
    if (con->type != &IMAP_connection) 
	panic("MBX PANIC",__FILE__,__LINE__,"imap_handle_greeting",
	      "Bad connection type",0);

    data.con = con;
    
    if (CON_greeting == con->state) {

	DPRINT(Debug,7,(&Debug, 
			"imap_handle_greeting: Waiting for server greeting...\n"));

	/* We want server's initial response to connection ...
	   .. maybe untagged BYE?
	*/
	con->a.imap_con->imap_state = IMAP_idle;
	ConfigStream2(con->C.stream,
		      imap_read_stream,ss_noaction_routine,imap_idle_timer,
		      ss_nofree_action_rt_data,
		      ss_noinc_action_rt_data_refcount,
		      imap_badpid_stream,
		      imap_idle_alive_interval > MIN_IMAP_TIMEOUT ?
		      imap_idle_alive_interval : MIN_IMAP_TIMEOUT,			     
		      data);
	WaitStreamFor(con->C.stream,SS_read_act);
	if (NULL ==  con->C.stream || 
	    CON_error == con->state ||
	    IMAP_error   == con->a.imap_con->imap_state ||
	    IMAP_closing == con->a.imap_con->imap_state) {
	    DPRINT(Debug,7,(&Debug, 
			    "imap_handle_greeting: Server does not accept connection\n"));
	    con->state = CON_error;
	    goto clean;
	}
	con->state = CON_open;
    }
    /* Assume that everything OK */
    status = 1;
    con->a.imap_con->imap_state = IMAP_idle;
  
#ifdef USE_DLOPEN
 retry_prelogin_capa:
#endif

    if (!start_imap_command(con,"CAPABILITY")) 
	goto clean;
    end_imap_command(con);
    
    if (!imap_command_ok(con,&res,NULL))
	goto clean;

#ifdef USE_DLOPEN
    else {
	enum CAPA_phase phase = capa_prelogin;
	
	DPRINT(Debug,13,(&Debug, "%d libraries available\n",
		    con->a.imap_con->capability_lib_count));
	if (!handle_imap_capa_libs(con,
				   & (con->a.imap_con->capability_libs),
				   & (con->a.imap_con->capability_lib_count),
				   &phase, &imap_commands)) {
	    status = 0;
	    con->state = CON_error;
	}

	switch(phase) {
	case capa_prelogin_again:
	    goto retry_prelogin_capa;
	case capa_logged:
	    con->state = CON_logged;
	    break;
	default:
	    break;
	}

    }
#endif

 clean:
    imap_clear_command(con);

    DPRINT(Debug,11,(&Debug, 
		     "imap_handle_greeting=%d\n",status)); 

    return status;
}

static int cache_open_common_imap P_((struct connection_cache *con));
static int cache_open_common_imap(con)
     struct connection_cache *con;
{
    int status = 0;

    struct cancel_data *main_cancel = NULL /* Used if dns lookup was
					      cancelable */;
    
    struct service_entry *se =
	give_service_entry_can(con->C.host,STFLAG_is_imap,
			       itls_ignore,
			       NULL,&main_cancel);
    PORTS force_port = PORT_end;
    enum itls_status have_initial_tls = itls_none;
    struct service_entry *result_se = NULL;

    int got_port = 0;

    if (!se) {
	if (main_cancel && is_canceled(main_cancel)) {
	    DPRINT(Debug,12,(&Debug,
			     "cache_open_common_imap: DNS lookup canceled\n"));
	}
	
	status = 0;
	con->state = CON_error;
	goto clean;
    }
    
    con->C.host = strmcpy(con->C.host,
			  se->official_name);
    
    if (con->port) {
	static const PORTS IMAP_ports[] = { PORT_imap4, PORT_end };
	
	force_port = con->port;
	DPRINT(Debug,7,(&Debug, 
			"cache_open_common_imap: force_port = %d (no initial tls)\n",
			force_port));
    

	if (!connect_remote_account(&(con->C),
				    &got_port,se,
				    IMAP_ports,
				    force_port,
				    main_cancel)) {
	    status = 0;
	    con->state = CON_error;
	    goto clean;
	}
	
    } else {
	const struct service_type * got_type = NULL;

	struct enum_service_type ports_imaponly_2 = NULL_enum_service_type;
	
	init_enum_service_type(&ports_imaponly_2,init_tls_default,
			       STFLAG_is_imap,STFLAG_mbox);


	if (!connect_remote_account_2(&(con->C),
				      se,main_cancel,
				      &result_se,
				      &got_type,
				      &got_port,
				      &ports_imaponly_2)) {
	    status = 0;
	    con->state = CON_error;
	    goto clean;

	}

	DPRINT(Debug,15,(&Debug,
			 "cache_open_common_imap: %s: connected, got port %d",
			 con->C.host,got_port));
	if (got_type) {
	    DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
			     got_type,
			     service_type_name(got_type),
			     service_type_defport(got_type)));
	}
	DPRINT(Debug,15,(&Debug,"\n"));

    }
	
    
    switch((have_initial_tls = remote_account_init_tls(&(con->C)))) {
	
    case itls_unsupported:
    case itls_failed:
	
	status = 0;
	con->state = CON_error;
	goto clean;
	
    case itls_none:
	break;
	
    case itls_done:
	DPRINT(Debug,8,(&Debug,
			"cache_open_common_imap: Initial TLS done (no STARTTLS)\n"));
    }
    
    con->state = CON_greeting;
    status = 1;
        
 clean:
    if (main_cancel)
	free_cancel(& (main_cancel));

    /* 'se' can be NULL */
    free_service_entry(&se);
    free_service_entry(&result_se);

    DPRINT(Debug,15,(&Debug,
		     "cache_open_common_imap=%d\n",status));

    return status;
}

static int  cache_open_imap(con)
     struct connection_cache *con;
{
    int status = 0;

    DPRINT(Debug,11,(&Debug, 
		     "cache_open_imap: con=%p (%s@%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>"));
    
    /* Clear buffer from old data ... */
    cache_imap_clear_buffers(con);
    
    if (NULL == con->C.stream || 
	con->state == CON_error) {

	status = cache_open_common_imap(con);

	if (!status)
	    goto clean;   	
    }
    
    status = imap_handle_greeting(con);

 clean:
    imap_clear_command(con);

    DPRINT(Debug,11,(&Debug, 
		     "cache_open_imap=%d\n",status)); 
    return status;
}

static int cache_open_hm_imap(con,passhm,use_passhm) 
     struct connection_cache *con;
     struct browser_passhm *passhm;
     int  * use_passhm;
{

   int status = 0;

    DPRINT(Debug,11,(&Debug, 
		     "cache_open_hm_imap: con=%p (%s@%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>"));
    
    /* Clear buffer from old data ... */
    cache_imap_clear_buffers(con);
    
    if (NULL == con->C.stream || 
	con->state == CON_error) {

	if (passhm) {

	    if (browser_passhm_check_type(passhm, hmcon_imap)) {
		static const PORTS IMAP_ports[] = { PORT_imap4, PORT_end };
		PORTS force_port = PORT_end;
		
		if (con->port) {
		    force_port = con->port;
		    DPRINT(Debug,7,(&Debug, 
				    "cache_open_hm_imap: force_port = %d\n",
				    force_port));
		}
		
		if (use_passhm)
		    *use_passhm = 1;

		/* -1 == fallback */
		status = browser_passhm_open_ra(passhm,&(con->C),
						IMAP_ports,
						STFLAG_is_imap,
						force_port);
		
		if (-1 == status)
		    status = cache_open_common_imap(con);
		else if (!status) 
		    con->state = CON_error;
		else {
		    enum itls_status have_initial_tls = itls_none;
		    
		    switch((have_initial_tls = remote_account_init_tls(&(con->C)))) {
			
		    case itls_unsupported:
		    case itls_failed:
			
			status = 0;
			con->state = CON_error;
			goto clean;
			
		    case itls_none:
			break;
			
		    case itls_done:
			DPRINT(Debug,8,(&Debug,
					"cache_open_hm_imap: Initial TLS done (no STARTTLS)\n"));
		    }
		    
		    con->state = CON_greeting;	   
		}
	    } else {
		DPRINT(Debug,8,(&Debug,
				"cache_open_hm_imap: browser_passhm discarded\n"));

		if (use_passhm)
		    *use_passhm = 0;

		status = cache_open_common_imap(con);
	    }

	} else
	    status = cache_open_common_imap(con);

	if (!status)
	    goto clean;

    } else {

	if (passhm && use_passhm)
	    *use_passhm = browser_passhm_check_type(passhm, hmcon_imap);
    }

    status = imap_handle_greeting(con);

 clean:
    imap_clear_command(con);

    DPRINT(Debug,11,(&Debug, 
		     "cache_open_hm_imap=%d",status)); 

    if (use_passhm) {
	DPRINT(Debug,11,(&Debug, ": *use_passhm=%d",
			 *use_passhm));
    }

    DPRINT(Debug,11,(&Debug, "\n"));
    return status;
}


static int  cache_login_imap(con,password)
     struct connection_cache *con;
     const struct string *password; /* May be NULL */
{
    char * data1 = NULL;
    imap_states res;
    int status = 0;

    DPRINT(Debug,11,(&Debug, 
		     "cache_login_imap: con=%p (%s@%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>"));

    if (con->state != CON_open ||
	NULL == con->C.stream) {
	DPRINT(Debug,11,(&Debug, 
			 "cache_login_imap: state not open\n")); 
	con->state = CON_error;
	status = 0;
	goto clean;
    }

    if (!start_imap_command(con,"LOGIN")) 
	goto clean;

    imap_command_push_astring(con,con->C.username);

    mbx_remote_login_msg(&(con->C),rlogin_IMAP);

    

    if (password) {
	struct string * pass1;

	if (!  can_ascii_string(password)) {
	    DPRINT(Debug,1,(&Debug, 
			    "cache_login_imap: Non ascii password are not supported\n"));
	}

	/* NOTE: Original string is kept, if it can not converted
	   to US-ASCII -- then this produces garbage
	*/
	pass1 = ascify_string(password);

	data1 =  us2s(stream_from_string(pass1,0,NULL));

	free_string(&pass1);

    } else
	data1 = lib_prompt(1,CATGETS(elm_msg_cat, MeSet,MeLoginPasswd,
				     "Password for %s@%s:"),
			   con->C.username,
			   con->C.host);
    if (!data1) {
	status = 0;
	goto clean;	    
    }

    /* Put is as literal */

    imap_command_push_aliteral(con,data1);
    end_imap_command(con);

    if (!imap_command_ok(con,&res,NULL)) {

	/* Show possible login error */

	if (sleepmsg > 0) {  /* HACK */

	    /* POLL_METHOD is defined because REMOTE_MBX is defined */

	    wait_for_timeout(sleepmsg);

	}
	goto clean;
    }
    con->state = CON_logged;
    status = 1;

    lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPLoginOK,
			  "IMAP login to %s as %s ... OK"),
		  con->C.host,
		  con->C.username);
    
 clean:
    imap_clear_command(con);

    if (data1) {
	free(data1);
	data1 = NULL;
    }
    DPRINT(Debug,11,(&Debug, 
		     "cache_login_imap=%d\n",status)); 
    
    return status;
}

static int  cache_close_imap(con,cd)
     struct connection_cache *con;
     struct cancel_data  * cd;
{
    int ret = 1;
    
    DPRINT(Debug,12,(&Debug, 
		     "cache_close_imap: con=%p (%s@%s)\n",
		     con,
		     con->C.username ? con->C.username : "<NULL>",
		     con->C.host ? con->C.host : "<NULL>"));
    
    if (con->state != CON_error) {
	DPRINT(Debug,12,(&Debug, 
			 "cache_close_imap: LOGOUT\n"));
	
	if (start_imap_command_c(con,"LOGOUT",cd)) {
	    imap_states res;
	    int r;
	    
	    end_imap_command(con);
	    ret = imap_command_ok_c(con,&res,NULL,cd);
	    if (ret) {
		DPRINT(Debug,12,(&Debug, 
				 "cache_close_imap: LOGOUT succeed\n"));
	    }
	    
	    r = imap_clear_command_c(con,cd);
	    if (!r) {
		DPRINT(Debug,12,(&Debug, 
				 "cache_close_imap: clear command failed\n"));
		ret = 0;
	    }
	} else {
	    DPRINT(Debug,12,(&Debug, 
			     "cache_close_imap: start_imap_command failed\n"));
	    ret = 0;
	}
    }

    if (con->C.stream) {
	DPRINT(Debug,12,(&Debug, 
			 "cache_close_imap: Closing stream"));
	if (con->seen_badpid) {
	    DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
			     con->seen_badpid));	    
	    }    
	DPRINT(Debug,12,(&Debug,"\n"));
	
	FreeStreamStack0(&(con->C.stream),
			 con->seen_badpid,
			 1 /* Force clearing of stack even when this is not last reference */);
    }	

    con->state = CON_error;
    
    DPRINT(Debug,12,(&Debug, 
		     "cache_close_imap=%d\n",
		     ret));
    
    return ret;
}


static void cache_folder_from_imap(con,folder)
     struct connection_cache *con;
     struct folder_info *folder;
{
    DPRINT(Debug,12,(&Debug, 
		"cache_folder_from_imap: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));
    
    if (folder->folder_type != &imap_mbx) 
	panic("MBX PANIC",__FILE__,__LINE__,"cache_folder_from_imap",
	      "Can't attach folder type to connection",0);

    if (folder->p->a.imap_mbx.Ch &&
	folder->p->a.imap_mbx.Ch != con) {
	int r;
	
	DPRINT(Debug,12,(&Debug, 
		    "cache_folder_from_imap: Freeing old connection: %p\n",
		    folder->p->a.imap_mbx.Ch));
	r = free_connection(&(folder->p->a.imap_mbx.Ch),NULL);
	if (!r) {
	    DPRINT(Debug,12,(&Debug, "cache_folder_from_imap: free_connection failed\n"));
	}			 
    }

    folder->p->a.imap_mbx.Ch            = con;
    folder->p->a.imap_mbx.folder_status = 0;

    /* TODO :  Check necessary actions */
    
}

static void cache_browser_from_imap(con,dir)
     struct connection_cache *con;
     struct folder_browser   *dir;
{
    DPRINT(Debug,12,(&Debug, 
		"cache_browser_from_imap: con=%p (%s@%s)\n",
		con,
		con->C.username ? con->C.username : "<NULL>",
		con->C.host ? con->C.host : "<NULL>"));
    
    if (dir->type != &imap_browser) 
	panic("MBX PANIC",__FILE__,__LINE__,"cache_browser_from_imap",
	      "Can't attach browser type to connection",0);

    DPRINT(Debug,12,(&Debug, " ... caching %s\n",
		     IMAP_connection_cache ? "enabled" : "disabled"));

    if (IMAP_BROWSER_magic != dir->a.imap_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"cache_browser_from_imap",
	      "Bad magic type",0);


    if (dir->a.imap_browser->Ch &&
	dir->a.imap_browser->Ch != con) {

	if (dir->a.imap_browser->Ch->state != CON_error &&
	    IMAP_connection_cache) {
	    DPRINT(Debug,12,(&Debug, 
			"cache_browser_from_imap: Caching old connection: %p\n",
			dir->a.imap_browser->Ch));
	    cache_connection(dir->a.imap_browser->Ch);
	} else {
	    int r;
	    
	    DPRINT(Debug,12,(&Debug, 
			     "cache_browser_from_imap:  freeing old connection: %p\n",
			     dir->a.imap_browser->Ch));
	    r = free_connection(&(dir->a.imap_browser->Ch),NULL);
	    if (!r) {
		DPRINT(Debug,12,(&Debug, 
				 "cache_browser_from_imap: free_connection failed\n"));
	    }
	}
    }

    dir->a.imap_browser->Ch            = con;

    /* TODO :  Check necessary actions */
    
}

/* return 1 if cb success, 0 failure
   return -1 no connection
 */
static  int cache_verify_cb_imap(c,ra,Server,server)
     struct connection_cache *c;
     struct remote_account *ra;
     const struct string * Server;
     const struct string * server;
{
    /* cache_open_imap calls CAPABILITY -- nothing need to be done */

    return 1;
}




#endif /* REMOTE_MBX */



/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
