static char rcsid[] = "@(#)$Id: hashmark.c,v 2.9 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.9 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *       or Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "hashmark.h"
#include "rc_imp.h"
#include "s_me.h"
#include "s_elm.h"
#include "hashmark_imp.h"
#include "commentfile.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}


/****************************************************************************

Userfile      ~/.elm/hash.marks
Global file:  {lib}/elm.hashmarks

Syntax:

[hashmark]
hashtype:   ( remote-mailbox, local-folder-directory, local-directory )
description: (printable text)

== For remote-mailbox

hostname:   hostname
ipaddr:     X.X.X.X.
servicetype:  (imap, pop, imaps, pops )
verify-tls-certificate
require-tls-peer-name
require-tls-peer-name: name
userid:     username                                            (only on userfile)
from-address: aaa bcc <local-part@addr>                          (only on userfile)
usermap:    local-userid    username   aaa bbbb <local-part@addr>    (only on global file)
usermap-file: xxx                                               (only on global file)

== For local-folder-directory and local-directory 

directory:  /abc

Values can be "quoted", inside of quotes \ escapes next charater.
On mail address only local-part can be quoted.

Usermap have space or tab separated values (spaces inside of
"quotes" does not separate values.

== usermap file

hashmark local-userid    username   aaa bbbb <local-part@addr>
hashmark local-userid    username   aaa bbbb <local-part@addr>   
hashmark "local-userid"  "username" "phrase" <local-part@addr>
hashmark local-userid    username
hashmark "local-userid"  "username"
hashmark local-userid    :
hashmark local-userid    :          aaa bbbb <local-part@addr>
hashmark "local-userid"  :          "phrase" <local-part@addr>
:        local-userid    username   aaa bbbb <local-part@addr>

****************************************************************************/
						
static char *hashkeyword_names[NUM_hashkeyword] = {
    "bad",
    "hashtype",
    "description",
    "hostname",
    "ipaddr",
    "servicetype",
    "verify-tls-certificate",
    "require-tls-peer-name",
    "userid",
    "from-address",
    "usermap",
    "usermap-file",
    "directory",
    "on-default-menu",
    "hide",
    "self-cc",
    "self-bcc"   

};

const char * hashmark_Keyword (kw)
     enum hashkeyword_v kw;
{
    if (hashkeyword_bad <= kw && kw < NUM_hashkeyword)
	return hashkeyword_names[kw];

    return "?";
}

static struct hashtype_desc {
    char * name;
    enum selection_type sel;
} hashtypes[NUM_hashtype] UNUSED_VAROK = {
    { "none",                   NUM_selection_type   /* not used */ },
    { "remote-mailbox",         selection_folder },
    { "local-folder-directory", selection_folder },
    { "local-directory",        selection_file },
};

const char * hashmark_Hashtype(t)
     enum hashtype_v t;
{
    if (hashtype_none <= t && t < NUM_hashtype)
	return hashtypes[t].name;
    return "?";
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

S_(hashtype_init_item_f hashtype_init_item_none)
static void hashtype_init_item_none P_((struct hashmark_item *item));
S_(hashtype_free_item_f  hashtype_free_item_none)
static void hashtype_free_item_none P_((struct hashmark_item *item));
S_(hashtype_parse_kw_value_f hashtype_parse_kw_value_none)
static int hashtype_parse_kw_value_none P_((const char *filename,int lineno,
					    struct string * rest_S,
					    struct hashmark_item *item,
					    enum hashkeyword_v kw,
					    struct string * value,
					    enum record_mode rc,
					    const struct schedule_timelimit * now));

S_(hashtype_parse_kw_f hashtype_parse_kw_none)
static int hashtype_parse_kw_none P_((const char *filename,int lineno,
				      struct string * rest_S,
				      struct hashmark_item *item,
				      enum hashkeyword_v kw,
				      enum record_mode rc,
				      const struct schedule_timelimit * now));

S_(hashtype_parse_check_f hashtype_parse_check_none);
static int hashtype_parse_check_none P_((const char *filename,int lineno,
					 struct hashmark_item *item,
					 enum record_mode rc,
					 const struct schedule_timelimit * now));

S_(hashtype_dump_kw_f hashtype_dump_kw_none)
static int hashtype_dump_kw_none P_((FILE *F,
				   struct hashmark_item *item,
				   enum hashkeyword_v kw,
				   charset_t fileset,
				   const char * sysnam,
				   enum record_mode rc));

S_(hashtype_merge_f hashtype_merge_none)
static void hashtype_merge_none P_((struct hashmark_item *item,
				    const struct hashmark_item * base,
				    const struct hashmark_item * overwrite));

S_(hashtype_browser_flags_f hashtype_brflags_none)
static int  hashtype_brflags_none P_((const struct hashmark_item *item));

S_(hashtype_init_data_f hashtype_initd_none)
static void hashtype_initd_none P_((const struct hashmark_item *item,
				    union hashmark_data      * hashmark_data));

S_(hashtype_free_data_f hashtype_freed_none)
static void hashtype_freed_none P_((const struct hashmark_item *item,
				    union hashmark_data      * hashmark_data));

S_(hashtype_browser_changedir_f hashtype_brchdir_none)
static int hashtype_brchdir_none P_((struct hashmark_item       * item,
				     struct folder_browser      * dir,
				     union hashmark_data        * hashmark_data,
				     const struct string        * path_tail,
				     struct string             ** dispname));

S_(hashtype_passhm_verify_ra_con_f hashtype_verify_racon_none)
static int hashtype_verify_racon_none P_(( const struct hashmark_item * item,
					   struct browser_passhm *passhm,
					   struct remote_account *C));

S_(hashtype_passhm_open_ra_f hashtype_passhm_open_ra_none)
static int hashtype_passhm_open_ra_none P_((const struct hashmark_item * item,
					    struct browser_passhm      * passhm,
					    struct remote_account      * ra,
					    const PORTS                  default_portlist[],
					    int                          give_service_entry_flag,
					    PORTS                        force_port));

S_( hashtype_passhm_open_ra2_f hashtype_passhm_open_ra2_none)
static int hashtype_passhm_open_ra2_none P_((const struct hashmark_item * item,
					     struct browser_passhm      * passhm,
					     struct remote_account      * ra,
					     int                          give_service_entry_flag,
					     
					     /* For enumerate_service_type */
					     struct enum_service_type   * est
					     ));

S_(hashtype_selectbr_item_f hashtype_selectbr_item_none)
static int hashtype_selectbr_item_none P_((struct hashmark_item  * item,
					   struct folder_browser * dir,
					   union hashmark_data   * hashmark_data,
					   const struct string   * path_tail,
					   struct string        ** dispname,
					   int                   * newpos));

S_(hashtype_folder_from_browser_f hashtype_folder_from_none)
static struct folder_info * hashtype_folder_from_none
    P_((struct hashmark_item  * item,
	struct folder_browser * dir,
	union hashmark_data   * hashmark_data,
	int                     treat_as_spooled));

/* -1 == use fallback
    0 == lookup failed
    1 == ok

    caller must free *se, *username, *hostname, *addr
*/
S_(hashtype_remote_lookup_f hashtype_remote_lookup_none)
static int hashtype_remote_lookup_none P_((const struct hashmark_item  * item,
					   struct service_entry       ** se,
					   int                           lookup_flags,
					   char                      ** username,
					   char                      ** hostname,
					   struct address            ** addr,
					   int                        * useraddr_flags,
					   struct cancel_data         ** cancel_p
					   /* May be NULL, Used if dns lookup was cancelable */
					   ));


S_(hashtype_selection_is_folder_f hashtype_selection_is_none)
static int hashtype_selection_is_none P_((struct hashmark_item  * item,
					 struct folder_browser * dir,
					 union hashmark_data   * hashmark_data,
					 struct folder_info    * folder));

S_(hashtype_valid_on_user_f hashtype_valid_on_user_none)
static enum haskmark_valid_v hashtype_valid_on_user_none P_((const struct hashmark_item  * 
							     item));

S_(hashtype_prepare_write_item_f hashtype_prepare_write_none)
static int hashtype_prepare_write_none P_((struct hashmark_item  * item,
					  struct folder_browser * dir,
					  union hashmark_data   * hashmark_data,
					  WRITE_STATE             ptr));


/* Return directory prefix and l_item,
   changes browser type
*/
S_(hashtype_cat_hashmark_item_f hashtype_cat_hashmark_none)
static  struct string * hashtype_cat_hashmark_none
    P_((struct hashmark_item  * h_item,
	struct folder_browser * dir,
	union hashmark_data   * hashmark_data,
	const struct string   * l_item));


static struct hashtype_actions hashtype_none_action = 
    { /*  hashtype_none */
	HASHTYPE_ACTIONS_magic,
	hashtype_init_item_none,
	hashtype_free_item_none,
	hashtype_parse_kw_value_none,
	hashtype_parse_kw_none,
	hashtype_parse_check_none,
	hashtype_dump_kw_none,
	hashtype_merge_none,
	hashtype_brflags_none,
	hashtype_initd_none,
	hashtype_freed_none,
	hashtype_brchdir_none,
	hashtype_verify_racon_none,
	hashtype_passhm_open_ra_none,
	hashtype_selectbr_item_none,
	hashtype_folder_from_none,
	hashtype_remote_lookup_none,
	hashtype_selection_is_none,
	hashtype_valid_on_user_none,
	hashtype_prepare_write_none,
	hashtype_cat_hashmark_none,	
	hashtype_passhm_open_ra2_none
    };
 
static struct hashtype_actions * hashtype_actions[NUM_hashtype] = {

    &hashtype_none_action,

#if defined(I_NETINET_IN) && defined(I_ARPA_INET) && defined(REMOTE_MBX)
    &hashtype_remote_mbx_action
#else
    &hashtype_none_action
#endif
    ,

    &hashtype_folder_dir_action,
    &hashtype_local_dir_action

};

static struct  hashmark_item * malloc_hashmark_item P_((enum hashtype_v type,
							const struct string * name  
							/* may be NULL */));


static  struct  hashmark_item * malloc_hashmark_item(type,name)
     enum hashtype_v type;
     const struct string * name;  /* may be NULL */
{
    struct  hashmark_item * ret = safe_zero_alloc(sizeof(*ret));
    
    ret->description = NULL;
    ret->hashmark_name = name ? dup_string(name) : NULL;
    ret->hashtype    = type;
    ret->refcount    = 1;
    ret->magic       = HASHMARK_ITEM_magic;

    ret->on_default_menu = 0;
    ret->hide_hashmark   = 0;

    if (ret->hashtype < hashtype_none ||
	ret->hashtype >= NUM_hashtype ||
	! hashtype_actions[ret->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "malloc_hashmark_item",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic != 
	hashtype_actions[ret->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "malloc_hashmark_item",
              "Bad magic number (hashtype action)",0);
    
    hashtype_actions[ret->hashtype]->init_it(ret);
    
    return ret;
}


static void hashtype_init_item_none(item)
     struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_init_item_none",
              "Bad magic number",0);

    item->u.dummy = NULL;
}

static void hashtype_free_item_none(item)
     struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_free_item_none",
              "Bad magic number",0);
  
    if (item->u.dummy) {

	free(item->u.dummy);
	item->u.dummy = NULL;
    }
}

void inc_hashmark_item_refcount(item) 
     struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "inc_hashmark_item_refcount",
              "Bad magic number",0);

    item->refcount++;
}

const struct string * hashmark_item_name(item)
     const struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_item_name",
              "Bad magic number",0);
    
    return item->hashmark_name;
}


void free_hashmark_item(item) 
     struct hashmark_item **item;
{
    if (HASHMARK_ITEM_magic != (*item)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_hashmark_item",
              "Bad magic number",0);

    if ((*item)->refcount < 1) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_hashmark_item",
              "Bad recount",0);

    (*item)->refcount--;

    if ((*item)->refcount > 0) {  /* Do not free */

        (*item) = NULL;           /* Refefence count for this pointer is decrement, */
                                  /* so this must have be reset */
        return;	
    }

    if ((*item)->hashmark_name)
	free_string(& ((*item)->hashmark_name));
    if ((*item)->description)
	free_string(& ((*item)->description));

    if ((*item)->hashtype < hashtype_none ||
	(*item)->hashtype >= NUM_hashtype ||
	! hashtype_actions[(*item)->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_hashmark_item",
              "Bad hashtype",0);

    if (HASHTYPE_ACTIONS_magic != 
	hashtype_actions[(*item)->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_hashmark_item",
              "Bad magic number (hashtype action)",0);

    hashtype_actions[(*item)->hashtype]->
	free_it(*item);

    (*item)->magic = 0;
    free(*item);
    *item = NULL;
}

#define HASH_MARKS_CONF_magic	0xF51A

static struct hash_marks_conf {
    unsigned short magic;   /* HASH_MARKS_CONF_magic */

    struct hashmark {
	struct string * hashmark_text;

	/* hashmarks are per browser type */

	struct hashmark_item * items[NUM_selection_type];
	       
    } * hashmarks;
    int hashmark_count;

} * malloc_hash_marks_conf P_((void));


static struct hash_marks_conf * malloc_hash_marks_conf()
{
    struct hash_marks_conf * ret = safe_zero_alloc(sizeof(*ret));

    ret->hashmarks      = NULL;
    ret->hashmark_count = 0;
    ret->magic          =  HASH_MARKS_CONF_magic;

    return ret;
}

E_(unq_can_print_error_f hashmark_is_printableln)
int hashmark_is_printableln(filename,lineno,rest_S)
     const char *filename;
     int lineno;
     const struct string * rest_S;
{
    int L = string_len(rest_S);
    int idx;
    
    for (idx = 0; idx < L; idx++) {
	uint16        ucode = 0;
	unsigned char bcode = 0;
	int gchar_flags = give_character_from_string(rest_S,idx,
						     &ucode,&bcode);

	if (0 != (gchar_flags & GCHAR_unicode) &&
	    0x0009 /* HT  '\t' (horizontal tab) */ == ucode) {
	     
	    /* Allow tabulator */

	} else if (0 == (gchar_flags & GCHAR_unicode ) ||
		   0 == unicode_ch(ucode,UOP_printable)) {
	    char ubuffer[8], * ub = "";
	    char bbuffer[5], * bb = "";
	    
	    if (0 != (gchar_flags & GCHAR_unicode)) {
		elm_sfprintf(ubuffer,sizeof ubuffer,
			     FRM(" u%04X"),
			     ucode);
		ub = ubuffer;
	    }
	    if (0 !=  (gchar_flags & GCHAR_bytecode)) {
		elm_sfprintf(bbuffer,sizeof bbuffer,
			     FRM(" x%02X"),
			     bcode);
		bb = bbuffer;
	    }

	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeNonPrintableCharOnLine,					  
			      "%s: %d: Non-printable character%s%s on line"),
		      filename,lineno,ub,bb);
	    return 0;
	}
    }

    return 1;
}

static int hashtype_parse_kw_value_none(filename,lineno,
					rest_S,item,kw,
					value,rc,now)
     const char *filename;
     int lineno;
     struct string * rest_S;
     struct hashmark_item *item;
     enum hashkeyword_v kw;
     struct string * value;
     enum record_mode rc;
     const struct schedule_timelimit * now;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_kw_value_none",
	      "Bad magic (hashmark_item)",0);	
    
    if (hashmark_is_printableln(filename,lineno,rest_S))
	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeHTypeDontSupoKwOnLIne,
			  "%s: %d: Hashtype %s do not support keyword %s on line %S"),
		  filename,lineno,
		  hashmark_Hashtype(item->hashtype),
		  hashmark_Keyword(kw),
		  rest_S);
    
    return 0;
}

static int hashtype_parse_kw_none(filename,lineno,rest_S,item,kw,rc,now)
     const char *filename;
     int lineno;
     struct string * rest_S;
     struct hashmark_item *item;
     enum hashkeyword_v kw;
     enum record_mode rc;
     const struct schedule_timelimit * now;
{
   if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_kw_none",
	      "Bad magic (hashmark_item)",0);	
    
    if (hashmark_is_printableln(filename,lineno,rest_S))
	lib_error(CATGETS(elm_msg_cat, MeSet, 
			  MeHTypeDontSupoKwOnLIne,
			  "%s: %d: Hashtype %s do not support keyword %s on line %S"),
		  filename,lineno,
		  hashmark_Hashtype(item->hashtype),
		  hashmark_Keyword(kw),
		  rest_S);
    
    return 0;
}

static int hashtype_parse_check_none(filename,lineno,item,rc,now)
     const char *filename;
     int lineno;
     struct hashmark_item *item;
     enum record_mode rc;
     const struct schedule_timelimit * now;
{
     if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_check_none",
	      "Bad magic (hashmark_item)",0);	

     return 1; /* Nothing to check */
}

static int hashmark_parse_check  P_((const char *filename,int lineno,
				     struct hashmark_item *item,
				     enum record_mode rc,
				     const struct schedule_timelimit * now));
static int hashmark_parse_check(filename,lineno,item,rc,now)
     const char          *  filename;
     int                    lineno;
     struct hashmark_item * item;
     enum record_mode       rc;
     const struct schedule_timelimit *                 now;
{
    int r = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_parse_check",
	      "Bad magic (hashmark_item)",0);	
    
    if (item->hashtype < hashtype_none ||
	item->hashtype >= NUM_hashtype ||
	! hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_parse_check",
	      "Bad hashtype",0);

    if (HASHTYPE_ACTIONS_magic != 
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_parse_check",
	      "Bad magic number (hashtype action)",0);
   
    r = hashtype_actions[item->hashtype]->
	parse_check(filename,lineno,item,rc,
		    now);

    DPRINT(Debug,10,(&Debug, "hashmark_parse_check=%d (%s, file %s: %d)\n",
		     r,
		     r ? "OK" : "failed",
		     filename,lineno));

    return r;
}

const unsigned char hashmark_RESERVED_SEPARATOR[] = " \t()\":[]\\@#%!/$&";


struct hash_marks_conf * parse_hash_mark_entries(filename,errors,fileset,
						 propline,rc)
     const char *filename;
     int        *errors;
     charset_t *fileset;
     struct editor_propline **propline;
     enum record_mode rc;  /* SYSTEM_RC = 0, LOCAL_RC = 1 */
{
    int max_count = 0;
    FILE * f;
    char * buf = NULL;
    int c;
    charset_t     cs = system_charset;
    const char * csn = get_charset_MIME_name(cs);
    int lineno = 0;
    int last_c = 0;
    int err = can_open(filename,"r");

    struct hash_marks_conf * ret = NULL;

    struct  hashmark_item  * current_item = NULL;
    struct string          * current_hash = NULL; 
    int                      current_index = -1;

    struct schedule_timelimit now = NO_schedule_timelimit;

    /* Should succeed -- error == ((time_t) -1) ignored */
    schedule_set_current_time(&now);
    
    if (err) {
	DPRINT(Debug,2,(&Debug, 
			"parse_hash_mark_entries=NULL: %s: %s (can_open)\n",
			filename,strerror(err)));
	return NULL;
    }

    f = fopen(filename,"r");
    if (!f) {
	int err UNUSED_VAROK = errno;
	DPRINT(Debug,2,(&Debug, 
			"parse_hash_mark_entries=NULL: %s: %s\n",
			filename,strerror(err)));
	return NULL;
    }

    if (fileset && *fileset) {
	csn = get_charset_MIME_name(*fileset);
	
	cs = *fileset;

	if (csn) {
	    DPRINT(Debug,2,(&Debug, 
			    "parse_hash_mark_entries: %s: default charset: %s\n",
			    filename,csn));
	}
    }

    /* Look for editor property line */

    if (propline)  {
	enum editor_propline_v propline_mode = 
	    give_dt_enumerate_as_int(&editor_hm_propline);

	/* Nothing detected yet */
	
	if (*propline)
	    free_editor_propline(propline);

	if (propline_mode != editor_propline_ignore) {
	    int i;
	    
	    for (i = 0; i < 3 && !*propline; i++) {
		int l1;

		c = fgetc(f);

		if (EOF == c)
		    goto propline_failure;
		if ('\n' == c)
		    continue;

		ungetc(c,f);

		if ('#' != c)
		    break;

		l1 = malloc_gets(&buf,LONG_STRING,f);
		if (l1 > 1) {

		    if ('#' != buf[0])
			goto propline_failure;
		    
		    *propline =
			detect_editor_propline(propline_mode,
					       buf,l1,filename,i,&cs);
	
		    csn = get_charset_MIME_name(cs);
				       
		} else { /* something wrong? */
		propline_failure:

		    rewind(f);
		    break;
		}
	    }
	}	
    }

    while(EOF != (c = fgetc(f))) {
	if ('\n' == c)
	    max_count++;
	last_c = c;
    }

    if (last_c && '\n' != last_c) {
	DPRINT(Debug,9,(&Debug, 
			"parse_hash_mark_entries: %s, no newline on end of file\n",
			filename));
	max_count++;	
    }

    ret = malloc_hash_marks_conf();

    if (!max_count) {
	fclose(f);
	return ret;
    }
    rewind(f);

    ret->hashmarks =
	safe_calloc(max_count,sizeof(ret->hashmarks[0]));

    while (ret->hashmark_count < max_count) {

	int l1 = malloc_gets(&buf,LONG_STRING,f);
	struct string * rest_S = NULL;
	char * cx = buf;
	int restlen = 0;
	int ERRORS = 0;
	int rs;
	int L;

	if (-1 == l1) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeTooLongLineNo,
			      "%s: %d: Too long line: %.30s..."),
		      filename,lineno+1,buf);
	    (*errors) ++;

	    DPRINT(Debug,9,(&Debug, 
			    "parse_hash_mark_entries: %s: %d: failed to parse line, too long line\n",
			    filename,lineno+1));


	    break;
	} else if (l1 < 0) {
	    DPRINT(Debug,9,(&Debug, 
			    "parse_hash_mark_entries: %s: %d: read error or EOF\n",
			    filename,lineno+1));
	    break;
	}

	lineno++;

	if (l1 == 0)
	    continue;

	l1 = trim_whitespace_from_end(buf,l1);

	if (read_charset_tag(buf,l1,
			     filename,lineno,
			     errors,&cs,&csn))
	     continue;
			    
	cx = buf;
	while (*cx && whitespace (*cx)) /* skip leading whitespace */
	    cx++;
	if ('#' == *cx)
	    continue;

	if (!*cx) {

	    if (cx < buf + l1) {
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeNULcharacterOnLine,
				  "%s: %d: NUL character on line."),
			  filename,lineno);
		(*errors) ++;

		DPRINT(Debug,9,(&Debug, 
				"parse_hash_mark_entries: %s: %d: failed to parse line (NUL character)\n",
				filename,lineno));


	    }

	    /* Empty line start new hash specification */
	    
	    if (current_item) {

		if (! hashmark_parse_check(filename,lineno-1,
					   current_item,rc,
					   &now)) {
		    
		    DPRINT(Debug,9,(&Debug, 
				    "parse_hash_mark_entries: %s: %d: Hashmark check failed\n",
				    filename,lineno));

		    (*errors) ++; 
		}

		free_hashmark_item(& current_item);

	    }
	    if (current_hash)
		free_string(& current_hash);
	    current_index = -1;

	    continue;
	}


	rest_S = new_string(cs);
	restlen = (buf + l1) - cx;
	rs =  add_streambytes_to_string(rest_S,restlen,
					cs2us(cx),&ERRORS);
	if (rs < restlen) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedAsCharset,
			      "Failed to parse line %d as charset %s in \"%s\" file"),
		      lineno, csn ? csn : "<no MIME name>",filename);

	    (*errors) ++;

	    DPRINT(Debug,9,(&Debug, 
			    "parse_hash_mark_entries: %s: %d: failed to parse line\n",
			    filename,lineno));

	    goto fail;
	}
	
	if (ERRORS) {
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFileAsCharsetErrors,
			      "There is %d errors when parsing line %d as charset %s in \"%s\" file"),
		      ERRORS,lineno,
		      csn ? csn : "<no MIME name>",filename);

	    (*errors) ++;

	    DPRINT(Debug,9,(&Debug, 
			    "parse_hash_mark_entries: %s: %d: failed to parse line (%d character errors)\n",
			    filename,lineno, ERRORS));


	    goto fail;
	}

	L = string_len(rest_S);
	if (L > 0) {
	    uint16 code = give_unicode_from_string(rest_S,0);

	    if (0x005B /* [ */ == code) {
		int X = 1;
		int idx;
		enum selection_type st;
		int L1;

		if (current_item) {
		    
		    if (! hashmark_parse_check(filename,lineno-1,
					       current_item,rc,
					       &now)) {
			(*errors) ++; 

			DPRINT(Debug,9,(&Debug, 
					"parse_hash_mark_entries: %s: %d: Hashmark check failed\n",
					filename,lineno));
		    }
		    
		    free_hashmark_item(& current_item);
		}
		if (current_hash)
		    free_string(& current_hash);
		current_index = -1;
		
		if (L < 3 || 
		    0x005D /* ] */ != give_unicode_from_string(rest_S,L-1)) {
		    
		    if (hashmark_is_printableln(filename,lineno,rest_S))
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeUnsupportedLineS,
					  "%s: %d: Unsupported line: %S"),
			      filename,lineno,rest_S);
		    (*errors) ++;

		    DPRINT(Debug,9,(&Debug, 
				    "parse_hash_mark_entries: %s: %d: failed to parse line\n",
				    filename,lineno));
		    
		    goto fail;
		    
		}

		current_hash = clip_from_string(rest_S,&X,L-2);

		L1 = string_len(current_hash);
		for (idx = 0; idx < L1; idx++) {
		    uint16        ucode = 0;
		    unsigned char bcode = 0;
		    int gchar_flags = give_character_from_string(current_hash,idx,
								 &ucode,&bcode);
			
		    if (0 == (gchar_flags & GCHAR_unicode ) ||
			0 == unicode_ch(ucode,UOP_printable)) {
			char ubuffer[8], * ub = "";
			char bbuffer[5], * bb = "";
			
			if (0 != (gchar_flags & GCHAR_unicode)) {
			    elm_sfprintf(ubuffer,sizeof ubuffer,
					 FRM(" u%04X"),
					 ucode);
			    ub = ubuffer;
			}
			if (0 !=  (gchar_flags & GCHAR_bytecode)) {
			    elm_sfprintf(bbuffer,sizeof bbuffer,
					 FRM(" x%02X"),
					 bcode);
			    bb = bbuffer;
			}

			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeNonPrintableCharOnHash,					  
					  "%s: %d: Non-printable character%s%s on hash"),
				  filename,lineno,ub,bb);
			(*errors) ++;

			DPRINT(Debug,9,(&Debug, 
					"parse_hash_mark_entries: %s: %d: failed to parse line\n",
					filename,lineno));

			goto fail;
		    }
		}

		for (idx = 0; idx < L1; idx++) {
		    uint16        ucode = 0;
		    unsigned char bcode = 0;
		    int gchar_flags = give_character_from_string(current_hash,idx,
								 &ucode,&bcode);

		    if ( (0 != (gchar_flags & GCHAR_unicode ) && unicode_ch(code,UOP_space)) 
			 ||
			 (0 != (gchar_flags & GCHAR_bytecode) && isascii(bcode) && isspace(bcode))) {

			char ubuffer[8], * ub = "";
			char bbuffer[5], * bb = "";
			
			if (0 != (gchar_flags & GCHAR_unicode)) {
			    elm_sfprintf(ubuffer,sizeof ubuffer,
					 FRM(" u%04X"),
					 ucode);
			    ub = ubuffer;
			}
			if (0 !=  (gchar_flags & GCHAR_bytecode)) {
			    elm_sfprintf(bbuffer,sizeof bbuffer,
					 FRM(" x%02X"),
					 bcode);
			    bb = bbuffer;
			}

			if (hashmark_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeSpaceCharOnHash,
					      "%s: %d: Space character%s%s on hash %S on line: %S"),
				      filename,lineno,ub,bb,current_hash,rest_S);	 
			(*errors) ++;

			DPRINT(Debug,9,(&Debug, 
					"parse_hash_mark_entries: %s: %d: failed to parse line\n",
					filename,lineno));

			goto fail;
		    }
		}

		idx = string_have_ascii_sep(current_hash,
					    hashmark_RESERVED_SEPARATOR);
		if (idx >= 0) {
		    uint16        ucode = 0;
		    unsigned char bcode = 0;
		    int gchar_flags = give_character_from_string(current_hash,idx,
								 &ucode,&bcode);
		    char ubuffer[8], * ub = "";
		    char bbuffer[5], * bb = "";
		    
		    if (0 != (gchar_flags & GCHAR_unicode)) {
			elm_sfprintf(ubuffer,sizeof ubuffer,
				     FRM(" u%04X"),
				     ucode);
			ub = ubuffer;
		    }
		    if (0 !=  (gchar_flags & GCHAR_bytecode)) {
			elm_sfprintf(bbuffer,sizeof bbuffer,
				     FRM(" x%02X"),
				     bcode);
			bb = bbuffer;
		    }
		    
		    if (hashmark_is_printableln(filename,lineno,rest_S))
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeReservedCharOnHash,
					  "%s: %d: Character%s%s is reserved (%s) on hash %S on line: %S"),
				  filename,lineno,ub,bb,hashmark_RESERVED_SEPARATOR,
				  current_hash,rest_S);
		    (*errors) ++;

		    DPRINT(Debug,9,(&Debug, 
				    "parse_hash_mark_entries: %s: %d: failed to parse line\n",
				    filename,lineno));

		    goto fail;
		}


		for (current_index = 0; 
		     current_index < ret->hashmark_count; 
		     current_index++) {
		    if (0 == string_cmp(ret->hashmarks[current_index].hashmark_text,
					current_hash, 
					-99 /* can't compare */ ))
			goto found;

		}

		if (current_index != ret->hashmark_count)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "parse_hash_mark_entries",
			  "Bad index",0);

		if (current_index >= max_count)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "parse_hash_mark_entries",
			  "Overflow",0);

		ret->hashmarks[current_index].hashmark_text
		    = dup_string(current_hash);

		for (st = 0; st < NUM_selection_type; st++)
		    ret->hashmarks[current_index].items[st] = NULL;
		ret->hashmark_count++;

	    found:

		DPRINT(Debug,9,(&Debug, 
				"parse_hash_mark_entries: %s: %d: hash %S, index %d\n",
				filename,lineno,
				current_hash,current_index));

	    } else {  /* Not [ ] */

		enum hashkeyword_v kw_index;
		

		int have_colon = 0;

		enum hashkeyword_v found_kw = hashkeyword_bad;
		int                found_len = 0;

		for (kw_index = hashkeyword_bad; 
		     kw_index < NUM_hashkeyword; 
		     kw_index++) {
		    int match_len = 0;

		    DPRINT(Debug,25,(&Debug, 
				     "parse_hash_mark_entries: keyword [%d]=%s\n",
				     kw_index,hashkeyword_names[kw_index]));

		    match_len = 
			string_matches_ascii(rest_S,cs2us(hashkeyword_names[kw_index]),
					     SMA_return_len,SMA_op_match_prefix);

		    if (match_len > found_len) {
			DPRINT(Debug,9,(&Debug, 
					"parse_hash_mark_entries: %s: %d: match len %d: %s\n",
					filename,lineno,match_len,
					hashkeyword_names[kw_index]));
							 
			found_kw  = kw_index;
			found_len = match_len;
			
		    } else if (match_len) {
			DPRINT(Debug,9,(&Debug, 
					"parse_hash_mark_entries: %s: %d: match len %d (ignored): %s\n",
					filename,lineno,match_len,
					hashkeyword_names[kw_index]));
		    }
		}

		if (hashkeyword_bad == found_kw ||
		    NUM_hashkeyword == found_kw) {
		    if (hashmark_is_printableln(filename,lineno,rest_S))
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeUnsupportedLineS,
					  "%s: %d: Unsupported line: %S"),
				  filename,lineno,rest_S);
		    (*errors) ++;

		    DPRINT(Debug,9,(&Debug, 
				    "parse_hash_mark_entries: %s: %d: failed to parse line\n",
				    filename,lineno));
		    

		    goto badline;
		}

		if (found_len < L) {
		    int X = found_len;		    
		    code = give_unicode_from_string(rest_S,found_len);

		    have_colon = (0x003A /* : */ == code);
		    if (have_colon)
			X++;

		    while (X < L) {
			code = give_unicode_from_string(rest_S,X);

			if (0x0020 /* SPACE  */ == code ||
			    0x0009 /* HT */     == code)
			    X++;
			else
			    break;
		    }

		    if (have_colon) {
			struct string * value = 
			    clip_from_string(rest_S,&X,L);
			
			int L2 = string_len(value);
			int idx;
			struct string * unquoted_value = NULL;
			int                value_error = 0;


			for (idx = 0; idx < L2; idx++) {
			    uint16        ucode = 0;
			    unsigned char bcode = 0;
			    int gchar_flags = give_character_from_string(value,idx,
									 &ucode,&bcode);
			    
			    if (0 != (gchar_flags & GCHAR_unicode) &&
				0x0009 /* HT  '\t' (horizontal tab) */ == ucode) {
	     
				/* Allow tabulator */
				
			    } else if (0 == (gchar_flags & GCHAR_unicode ) ||
				0 == unicode_ch(ucode,UOP_printable)) {
				char ubuffer[8], * ub = "";
				char bbuffer[5], * bb = "";
	    
				if (0 != (gchar_flags & GCHAR_unicode)) {
				    elm_sfprintf(ubuffer,sizeof ubuffer,
						 FRM(" u%04X"),
						 ucode);
				    ub = ubuffer;
				}
				if (0 !=  (gchar_flags & GCHAR_bytecode)) {
				    elm_sfprintf(bbuffer,sizeof bbuffer,
						 FRM(" x%02X"),
						 bcode);
				    bb = bbuffer;
				}
				
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeNonPrintableCharOnValueOf,					  
						  "%s: %d: Non-printable character%s%s on value of keyword %s"),
					  filename,lineno,ub,bb,
					  hashkeyword_names[found_kw]);
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));


				goto skip_value;
			    }
			}
			
			switch(found_kw) {
			    enum hashtype_v t;
			    enum selection_type sel;

			case hashkeyword_hashtype:
			    
			    if (current_item) {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeDuplicateKwOnLine,
						      "%s: %d: Duplicate keyword %s on line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],rest_S);
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));


				goto fail;
			    }

			    unquoted_value = unquote_string(value,hashmark_is_printableln,
							    filename,lineno,&value_error);
			    if (value_error) {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeKwdBADvalueOnLine,
						      "%s: %d: Keyword %s have bad value on line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],rest_S);

				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));
			    }

			    if (unquoted_value) {
				for (t = hashtype_none; t < NUM_hashtype; t++) {
				    if (string_matches_ascii(unquoted_value,
							     cs2us(hashtypes[t].name),0,
							     SMA_op_normal))
					goto type_found;
				}
			    }

			    current_item = 
				malloc_hashmark_item(hashtype_none,current_hash);
			    
			    if (hashmark_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeUnsupportedLineS,
						  "%s: %d: Unsupported line: %S"),
					  filename,lineno,rest_S);
			    (*errors) ++;

			    DPRINT(Debug,9,(&Debug, 
					    "parse_hash_mark_entries: %s: %d: failed to parse line\n",
					    filename,lineno));

			    break;

			type_found:
			    if (current_index < 0) {
				current_item = 
				    malloc_hashmark_item(t,current_hash);

				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeHashNotGivenBefore,
						      "%s: %d: Hash not given before line %S"),
					      filename,lineno,rest_S);
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse line (hash not given)\n",
						filename,lineno));


				break;
			    }

			    if (current_index >= ret->hashmark_count ||
				! ret->hashmarks[current_index].hashmark_text)
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "parse_hash_mark_entries",
				      "Bad index",0);

			    sel = hashtypes[t].sel;
			    if (NUM_selection_type == sel) {   /*  hashtype_none */
				current_item = 
				    malloc_hashmark_item(t,current_hash);
				
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeUnsupportedLineS,
						      "%s: %d: Unsupported line: %S"),
					      filename,lineno,rest_S);
				(*errors) ++;
				
				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse line\n",
						filename,lineno));
				
				break;
			    }

			    if (sel < 0 || sel >= NUM_selection_type)
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "parse_hash_mark_entries",
				      "Bad selection type",0);	
			    
			    if (ret->hashmarks[current_index].items[sel]) {
				
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeDuplicateHashWithHashtype,
						      "%s: %d: Duplicate hash %S; hashtype %s on line %S"),
					      filename,lineno,
					      ret->hashmarks[current_index].hashmark_text,
					      hashtypes[t].name,rest_S);
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse line (duplicate hash)\n",
						filename,lineno));


				if (HASHMARK_ITEM_magic != 
				    ret->hashmarks[current_index].items[sel]->magic)
				    panic("CONNECTION PANIC",__FILE__,__LINE__,
					  "parse_hash_mark_entries",
					  "Bad magic (hashmark_item)",0);	

				if (t == ret->hashmarks[current_index].items[sel]->hashtype) {
				    current_item = ret->hashmarks[current_index].items[sel];
				    inc_hashmark_item_refcount(current_item);
				} else {
				    current_item = 
					malloc_hashmark_item(t,current_hash);
				}

			    } else {
				current_item = 
				    malloc_hashmark_item(t,current_hash);
				ret->hashmarks[current_index].items[sel]
				    = current_item;
				inc_hashmark_item_refcount(ret->hashmarks[current_index].items[sel]);
			    }

			    break;

			case hashkeyword_description:
			    if (!current_item) {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeKwdIgnOnLine,
						      "%s: %d: Keyword %s ignored on line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],rest_S);
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));

				break;
			    }

			    if (HASHMARK_ITEM_magic != current_item->magic)
				 panic("CONNECTION PANIC",__FILE__,__LINE__,
					  "parse_hash_mark_entries",
				       "Bad magic (hashmark_item)",0);	

			    if (current_item->description) {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeDuplicateKwOnLine,
						      "%s: %d: Duplicate keyword %s on line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],rest_S);
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));

				break;
			    }
			    
			    current_item->description = 
				unquote_string(value,hashmark_is_printableln,
					       filename,lineno,&value_error);
			    if (value_error) {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeKwdBADvalueOnLine,
						      "%s: %d: Keyword %s have bad value on line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],rest_S);

				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));
			    }
			    break;

			case hashkeyword_default_menu:
			case hashkeyword_hide:
			    if (hashmark_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeKwdNOvalueOnLine,
						  "%s: %d: Keyword %s have extra value on line: %S"),
					  filename,lineno,
					  hashkeyword_names[found_kw],rest_S);
			    (*errors) ++;
			    
			    DPRINT(Debug,9,(&Debug, 
					    "parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
					    filename,lineno,hashkeyword_names[found_kw]));

			    break;

			default:
			    if (!current_item) {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeKwdIgnOnLine,
						      "%s: %d: Keyword %s ignored on line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],rest_S);
				(*errors) ++;
				
				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));
				
				break;
			    }

			    if (HASHMARK_ITEM_magic != current_item->magic)
				 panic("CONNECTION PANIC",__FILE__,__LINE__,
					  "parse_hash_mark_entries",
				       "Bad magic (hashmark_item)",0);	

			    if (current_item->hashtype < hashtype_none ||
				current_item->hashtype >= NUM_hashtype ||
				! hashtype_actions[current_item->hashtype])
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "parse_hash_mark_entries",
				      "Bad hashtype",0);
			    
			    if (HASHTYPE_ACTIONS_magic != 
				hashtype_actions[current_item->hashtype]->magic)
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "parse_hash_mark_entries",
				      "Bad magic number (hashtype action)",0);

			    if (! hashtype_actions[current_item->hashtype]->
				parse_kw_value(filename,lineno,
					       rest_S,current_item,
					       found_kw,value,rc,&now)) {
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));

			    }
			    break;
			}
			
		    skip_value:
			if (unquoted_value)
			    free_string(&unquoted_value);
			if (value)
			    free_string(&value);


		    }  else if (X < found_len) {
			if (hashmark_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeUnsupportedLineS,
					      "%s: %d: Unsupported line: %S"),
				      filename,lineno,rest_S);
			(*errors) ++;
			
			DPRINT(Debug,9,(&Debug, 
					"parse_hash_mark_entries: %s: %d: failed to parse line\n",
					filename,lineno));

		    } else
			goto no_value;

		} else {
		no_value:

		    switch(found_kw) {
		    case hashkeyword_hashtype:
			
			if (current_item) {
			    if (hashmark_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeDuplicateKwOnLine,
						  "%s: %d: Duplicate keyword %s on line: %S"),
					  filename,lineno,
					  hashkeyword_names[found_kw],rest_S);
			    (*errors) ++;

			    DPRINT(Debug,9,(&Debug, 
					    "parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
					    filename,lineno,hashkeyword_names[found_kw]));

			    goto fail;
			}

			current_item = 
			    malloc_hashmark_item(hashtype_none,current_hash);

			if (hashmark_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeUnsupportedLineS,
					      "%s: %d: Unsupported line: %S"),
				      filename,lineno,rest_S);
			(*errors) ++;

			DPRINT(Debug,9,(&Debug, 
					"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
					filename,lineno,hashkeyword_names[found_kw]));

			break;

		    case hashkeyword_description:
			if (hashmark_is_printableln(filename,lineno,rest_S))
			    lib_error(CATGETS(elm_msg_cat, MeSet, 
					      MeUnsupportedLineS,
					      "%s: %d: Unsupported line: %S"),
				      filename,lineno,rest_S);
			(*errors) ++;

			DPRINT(Debug,9,(&Debug, 
					"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
					filename,lineno,hashkeyword_names[found_kw]));


			break;

		    case hashkeyword_default_menu:

			if (current_item) {
			    if (HASHMARK_ITEM_magic != current_item->magic)
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "parse_hash_mark_entries",
				      "Bad magic (hashmark_item)",0);	

			    if (current_item->hide_hashmark) {
				
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeHashtypeBothGivenUsing,
						      "%s: %d: Both %s and %s -keywords given. Using keyword %s. Line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],
					      hashkeyword_names[hashkeyword_hide],
					      hashkeyword_names[found_kw],
					      rest_S);

				(*errors) ++;
				current_item->hide_hashmark = 0;
			    }

			    if (current_item->on_default_menu) {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeDuplicateKwOnLine,
						      "%s: %d: Duplicate keyword %s on line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],rest_S);
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));

			    } else
				current_item->on_default_menu = 1;

			} else {
			    if (hashmark_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeKwdIgnOnLine,
						  "%s: %d: Keyword %s ignored on line: %S"),
					  filename,lineno,
					  hashkeyword_names[found_kw],rest_S);
			    (*errors) ++;

			    DPRINT(Debug,9,(&Debug, 
					    "parse_hash_mark_entries: %s: %d: keyword %s ignored\n",
					    filename,lineno,hashkeyword_names[found_kw]));


			}
			break;

		    case hashkeyword_hide:
			
			if (current_item) {
			    if (HASHMARK_ITEM_magic != current_item->magic)
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "parse_hash_mark_entries",
				      "Bad magic (hashmark_item)",0);	

			    if (current_item->on_default_menu) {

				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeHashtypeBothGivenUsing,
						      "%s: %d: Both %s and %s -keywords given. Using keyword %s. Line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],
					      hashkeyword_names[hashkeyword_default_menu],
					      hashkeyword_names[hashkeyword_default_menu],
					      rest_S);
				(*errors) ++;

			    } else if (current_item->hide_hashmark) {
				if (hashmark_is_printableln(filename,lineno,rest_S))
				    lib_error(CATGETS(elm_msg_cat, MeSet, 
						      MeDuplicateKwOnLine,
						      "%s: %d: Duplicate keyword %s on line: %S"),
					      filename,lineno,
					      hashkeyword_names[found_kw],rest_S);
				(*errors) ++;

				DPRINT(Debug,9,(&Debug, 
						"parse_hash_mark_entries: %s: %d: failed to parse keyword %s\n",
						filename,lineno,hashkeyword_names[found_kw]));

			    } else
				current_item->hide_hashmark = 1;

			} else {
			    if (hashmark_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeKwdIgnOnLine,
						  "%s: %d: Keyword %s ignored on line: %S"),
					  filename,lineno,
					  hashkeyword_names[found_kw],rest_S);
			    (*errors) ++;

			    DPRINT(Debug,9,(&Debug, 
					    "parse_hash_mark_entries: %s: %d: keyword %s ignored\n",
					    filename,lineno,hashkeyword_names[found_kw]));


			}
			break;

		    default:

			if (!current_item) {
			    if (hashmark_is_printableln(filename,lineno,rest_S))
				lib_error(CATGETS(elm_msg_cat, MeSet, 
						  MeKwdIgnOnLine,
						  "%s: %d: Keyword %s ignored on line: %S"),
					  filename,lineno,
					  hashkeyword_names[found_kw],rest_S);
			    (*errors) ++;

			    DPRINT(Debug,9,(&Debug, 
					    "parse_hash_mark_entries: %s: %d: keyword %s ignored\n",
					    filename,lineno,hashkeyword_names[found_kw]));

			    break;
			}
			
			if (HASHMARK_ITEM_magic != current_item->magic)
			    panic("CONNECTION PANIC",__FILE__,__LINE__,
				  "parse_hash_mark_entries",
				  "Bad magic (hashmark_item)",0);	
			
			if (current_item->hashtype < hashtype_none ||
			    current_item->hashtype >= NUM_hashtype ||
			    ! hashtype_actions[current_item->hashtype])
			    panic("CONNECTION PANIC",__FILE__,__LINE__,
				  "parse_hash_mark_entries",
				  "Bad hashtype",0);
			
			if (HASHTYPE_ACTIONS_magic != 
			    hashtype_actions[current_item->hashtype]->magic)
			    panic("CONNECTION PANIC",__FILE__,__LINE__,
				  "parse_hash_mark_entries",
				  "Bad magic number (hashtype action)",0);
			
			if (! hashtype_actions[current_item->hashtype]->
			    parse_kw(filename,lineno,rest_S,
				     current_item,found_kw,rc,
				     &now)) {
			    (*errors) ++;
			    
			    DPRINT(Debug,9,(&Debug, 
					    "parse_hash_mark_entries: %s: %d: Parsing of keyword %s failed\n",
					    filename,lineno,hashkeyword_names[found_kw]));


			}

			break;
		    }		   
		}
	    }


	} else {
	    /* Empty line start new hash specification */
	fail:
	    
	    if (current_item) {
		
		if (! hashmark_parse_check(filename,lineno-1,
					   current_item,rc,
					   &now)) {
		    (*errors) ++; 
		    
		    DPRINT(Debug,9,(&Debug, 
				    "parse_hash_mark_entries: %s: %d: Hashmark check failed\n",
				    filename,lineno));		    
		}
		
		free_hashmark_item(& current_item);
	    }
	    if (current_hash)
		free_string(& current_hash);
	    current_index = -1;
	}

    badline:

	free_string(&rest_S);

    }

    if (current_item) {
	
	if (! hashmark_parse_check(filename,lineno-1,
				   current_item,rc,
				   &now)) {
	    (*errors) ++; 

	    DPRINT(Debug,9,(&Debug, 
			    "parse_hash_mark_entries: %s: %d: Hashmark check failed\n",
			    filename,lineno));
	}

	free_hashmark_item(& current_item);
    }
    if (current_hash)
	free_string(& current_hash);

    if (buf)
	free(buf);

    fclose(f);
    
    DPRINT(Debug,9,(&Debug, 
		    "parse_hash_mark_entries: %s parsed, %d hashmarks, %d errors\n",
		    filename, ret->hashmark_count, *errors));
    
    return ret;

}

void free_hash_marks_conf(conf)
     struct hash_marks_conf **conf;
{

    if (HASH_MARKS_CONF_magic != (*conf)->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "free_hash_marks_conf",
              "Bad magic number",0);

    if ((*conf)->hashmarks) {
	int i;

	for (i = 0; i < (*conf)->hashmark_count; i++) {
	    enum selection_type T;

	    if ((*conf)->hashmarks[i].hashmark_text)
		free_string(& ((*conf)->hashmarks[i].hashmark_text));

	    for (T = 0; T < NUM_selection_type; T++) {
		if ((*conf)->hashmarks[i].items[T]) 
		    free_hashmark_item(& ((*conf)->hashmarks[i].items[T]));
	    }
	}

	free((*conf)->hashmarks);
	(*conf)->hashmarks = NULL;
    }
    (*conf)->hashmark_count = 0;

    free(*conf);
    *conf = NULL;
}

static int hashtype_dump_kw_none(F,item,kw,fileset,sysnam,rc)
     FILE                 * F;
     struct hashmark_item * item;
     enum hashkeyword_v     kw;
     charset_t              fileset;
     const char           * sysnam;
     enum record_mode       rc;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_dump_kw_none",
              "Bad magic number",0);


    return 1; /* Nothing to do */
}

static void hashtype_merge_none(item,base,overwrite)
     struct hashmark_item *item;
     const struct hashmark_item * base;
     const struct hashmark_item * overwrite;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_none",
              "Bad magic number",0);

    /* Nothing to do */
}

static int  hashtype_brflags_none(item) 
     const struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_brflags_none",
              "Bad magic number",0);
    
    return BROWSER_NODIR|BROWSER_NOFOLDER;
}

static void hashtype_initd_none(item,hashmark_data)
     const struct hashmark_item *item;
     union hashmark_data      * hashmark_data;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_initd_none",
              "Bad magic number",0);

    hashmark_data->dummy = NULL;
}

static void hashtype_freed_none(item,hashmark_data)
     const struct hashmark_item *item;
     union hashmark_data      * hashmark_data;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_freed_none",
              "Bad magic number",0);

    if (hashmark_data->dummy) {
	free(hashmark_data->dummy);
	hashmark_data->dummy = NULL;
    }
}

static int hashtype_brchdir_none(item,dir,hashmark_data,path_tail,dispname)
     struct hashmark_item       * item;
     struct folder_browser      * dir;
     union hashmark_data        * hashmark_data;
     const struct string        * path_tail;
     struct string             ** dispname;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_brchdir_none",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static int hashtype_verify_racon_none(item,passhm,C)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account *C;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_verify_racon_none",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static int hashtype_passhm_open_ra_none(item,passhm,ra,default_portlist,
					give_service_entry_flag,force_port)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account      * ra;
     const PORTS                  default_portlist[];
     int                          give_service_entry_flag;
     PORTS                        force_port;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra_none",
              "Bad magic number",0);

    return 0; /* Not supported */
}


static int hashtype_passhm_open_ra2_none(item,passhm,ra,give_service_entry_flag,est)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account      * ra;
     int                          give_service_entry_flag;
					     
     /* For enumerate_service_type */
     struct enum_service_type   * est;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra2_none",
              "Bad magic number",0);

    return 0; /* Not supported */
}


static int hashtype_selectbr_item_none(item,dir,hashmark_data,path_tail,dispname,newpos)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * path_tail;
     struct string        ** dispname;
     int                   * newpos;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selectbr_item_none",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static struct folder_info * hashtype_folder_from_none(item,dir,hashmark_data,treat_as_spooled)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     int                     treat_as_spooled;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_folder_from_none",
              "Bad magic number",0);

    return NULL; /* Not supported */

}

/* -1 == use fallback
    0 == lookup failed
    1 == ok

    caller must free *se, *username, *hostname, *addr
*/
static int hashtype_remote_lookup_none(item,se,lookup_flags,username,
				       hostname,addr,useraddr_flags,cancel_p)
     const struct hashmark_item * item;
     struct service_entry      ** se;
     int                          lookup_flags;     
     char                      ** username;
     char                      ** hostname;
     struct address            ** addr;
     int                        * useraddr_flags;
     struct cancel_data        ** cancel_p /* May be NULL,
					       Used if dns lookup was
					       cancelable */;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_remote_lookup_none",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static int hashtype_selection_is_none(item,dir,hashmark_data,folder)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     struct folder_info    * folder;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selection_is_none",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static enum haskmark_valid_v hashtype_valid_on_user_none(item)
    const struct hashmark_item  * item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_valid_on_user_none",
              "Bad magic number",0);

    return hashmark_unusable;
}

static int hashtype_prepare_write_none(item,dir,hashmark_data,ptr)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     WRITE_STATE             ptr;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_prepare_write_none",
              "Bad magic number",0);

    DPRINT(Debug,15,(&Debug,"hashtype_prepare_write_none=0 (not supported)\n"));
    
    return 0; /* Not supported */
}

/* Return directory prefix and l_item,
   changes browser type
*/
static  struct string * hashtype_cat_hashmark_none(h_item,dir,
						  hashmark_data,l_item)
     struct hashmark_item  * h_item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * l_item;
{
    struct string * ret = NULL;
    
    if (HASHMARK_ITEM_magic != h_item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_cat_hashmark_none",
              "Bad magic number",0);

    if (h_item->hashmark_name) {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_none: Using hashmark name %S\n",
			 h_item->hashmark_name));
	
	if (l_item)
	    ret = format_string(FRM("#%S:%S"),
				h_item->hashmark_name,
				l_item);
	else
	    ret = format_string(FRM("#%S"),
				h_item->hashmark_name);
    }

    if (ret) {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_none=%S\n",
			 ret));
    } else {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_none=NULL\n",
			 ret));

    }
        
    return ret;
}
    
/* ------------------------------------------------ */


int dump_string_quote_helper(f,fileset,sysnam,kw,value,x)
     FILE                * f;
     charset_t             fileset;
     const char          * sysnam;
     enum hashkeyword_v    kw;
     const struct string * value;
     int x;
{
    int ok   = 1;
    int err2 = 0;
    char *value2;   
    const struct string * value3 = value;
    struct string       * value4 = NULL;

    if (string_need_quote(value)) {
	value4 = string_quote_phrase(value);
	value3 = value4;
	
	DPRINT(Debug,9,(&Debug, 
			"dump_string_quote_helper: Quoting %S => %S\n",
			value,value3));	
    }

    value2 = convert_to_fileset2(value3,fileset,&err2);
				
    if (err2) {
	if (x >= 0) {
	    DPRINT(Debug,8,(&Debug, 
			    "dump_string_quote_helper: Failed to convert hashmark #%d ",
			    x));
	} else {
	    DPRINT(Debug,8,(&Debug, 
			    "dump_string_quote_helper: Failed to convert hashmark "));

	}

	DPRINT(Debug,8,(&Debug,
			"kw %s: \"%S\" to %s, %d errors\n",
			hashmark_Keyword(kw),		
			value3,
			sysnam ? sysnam : "<no NIME name>",
			err2));
	
	ok = 0;
    }		
		    
    fputs(hashmark_Keyword(kw),f); putc(':', f); putc(' ', f);
    fputs(value2,f); putc('\n', f);
    
    free(value2);
    
    if (value4)
	free_string(&value4);

    return ok;
}

int dump_hash_mark_entries(f,conf,commentfile,
			   actor,version_buff,fileset,
			   propline,rc)
     FILE *f;
     struct hash_marks_conf *conf;
     FILE *commentfile;
     const char *actor;
     char *version_buff;
     charset_t fileset;
     const struct editor_propline *propline;
     enum record_mode rc  /* SYSTEM_RC = 0, LOCAL_RC = 1 */;
{
    int fail = 0;
    enum editor_propline_v propline_mode = 
	give_dt_enumerate_as_int(&editor_hm_propline);
    const char * sysnam = NULL;
    int x;

    if (!conf)
	return 1;

    if (HASH_MARKS_CONF_magic != conf->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "dump_hash_mark_entries",
              "Bad magic number",0);

    
    if ( conf->hashmark_count < 1 &&
	!fileset && (! propline || propline_mode == editor_propline_ignore))
	return 1;

    if (!fileset) {
	DPRINT(Debug,10,(&Debug,
			 "dump_hash_mark_entries: File charset is not set, using system charset\n"));
	fileset = system_charset;    
    }

    sysnam = get_charset_MIME_name(fileset);

    /* Must have first line */
    if (propline_mode != editor_propline_ignore)
	write_editor_propline(propline,propline_mode,
			      f,"# ","\n",fileset);

    if (sysnam) {
	/* Store used character set */

	elm_fprintf(f, 
		    CATGETS(elm_msg_cat,MeSet, MeCharsetExplain,
			    "# Character set used on this file\n"));

	fprintf(f,"@charset = %s\n",sysnam);
    }

    insert_commentfile(f,ELMHASHMARKS_INFO,commentfile,actor,version_buff);

    for (x = 0; x < conf->hashmark_count; x++) {
       
	if (conf->hashmarks[x].hashmark_text) {
	    enum selection_type T;
	    int err = 0;	      
	    char *value = convert_to_fileset2(conf->hashmarks[x].hashmark_text,
					      fileset,&err);
	    
	    if (err) {
		DPRINT(Debug,8,(&Debug, 
				"dump_hash_mark_entries: Failed to convert hashmark #%d: \"%S\" to %s, %d errors\n",
				x,conf->hashmarks[x].hashmark_text,
				sysnam ? sysnam : "<no NIME name>",
				err));
	    }

	    
	    for (T = 0; T < NUM_selection_type; T++) {
		if (conf->hashmarks[x].items[T]) {
		    enum hashkeyword_v kw;
		    int ok3 = 1;

		    putc('\n', f);
		    putc('[', f);  fputs(value,f);  putc(']', f); putc('\n', f);
		    
		    if (HASHMARK_ITEM_magic != 
			conf->hashmarks[x].items[T]->magic)
			panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "dump_hash_mark_entries",
			      "Bad magic number (hashmark_item)",0);
		    
		    
		    for (kw =  hashkeyword_bad; kw < NUM_hashkeyword; kw++) {
			switch(kw) {
			case hashkeyword_bad: 
			    break;
			case hashkeyword_hashtype:
			    fputs(hashmark_Keyword(kw),f); putc(':', f); putc(' ', f);
			    fputs(hashmark_Hashtype(conf->hashmarks[x].items[T]->hashtype),f); putc('\n', f);
			    break;
			case hashkeyword_description:
			    if (conf->hashmarks[x].items[T]->description) {
				ok3 = dump_string_quote_helper(f,fileset,sysnam,kw,
							       conf->hashmarks[x].items[T]->description,
							       x);
			    }
			    break;
			    
			case hashkeyword_default_menu:
			    if (conf->hashmarks[x].items[T]->on_default_menu) {
				fputs(hashmark_Keyword(kw),f);  putc('\n', f);
			    }
			    break;
			    
			case hashkeyword_hide:
			    if (conf->hashmarks[x].items[T]->hide_hashmark) {
				fputs(hashmark_Keyword(kw),f);  putc('\n', f);
			    }
			    break;
			    			    
			default:
			    if (conf->hashmarks[x].items[T]->hashtype  < hashtype_none ||
				conf->hashmarks[x].items[T]->hashtype >= NUM_hashtype ||
				!  hashtype_actions[conf->hashmarks[x].items[T]->hashtype])
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "dump_hash_mark_entries",
				      "Bad hashtype",0);
			    
			    if (HASHTYPE_ACTIONS_magic != 
				hashtype_actions[conf->hashmarks[x].items[T]->
						 hashtype]->magic)
				panic("CONNECTION PANIC",__FILE__,__LINE__,
				      "dump_hash_mark_entries",
				      "Bad magic number (hashtype action)",0);
			    
			    ok3 = hashtype_actions[conf->hashmarks[x].items[T]->
						   hashtype]->
				dump_kw(f,conf->hashmarks[x].items[T],
					kw,fileset,sysnam,rc);
			    break;
			    			    
			}
		    }

		    if (err || !ok3)
			fail++;
		}
	    }

	    free(value);
	}
    }

    if (fail) {
	elm_fprintf(f, 
		    CATGETS(elm_msg_cat,MeSet, MeFailedConvertEntries,
			    "# Failed to convert %d entries to %s -charset\n"),
		    fail,sysnam ? sysnam : "<no MIME name>");
    }

    fflush(f);

    return !fail;
}

/* increments refcount */
static struct hashmark_item *  search_hm_name_from_conf 
    P_((const  struct hash_marks_conf    * hash_marks_conf,
	const struct string *name,
	enum selection_type sel_type));
static struct hashmark_item *  search_hm_name_from_conf(hash_marks_conf,
							name,sel_type)
     const  struct hash_marks_conf    * hash_marks_conf;
     const struct string              * name;
     enum selection_type                sel_type;
{
    int idx;

    if (HASH_MARKS_CONF_magic != hash_marks_conf->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "search_hm_name_from_conf",
              "Bad magic number",0);

    if (sel_type < 0 || sel_type >= NUM_selection_type) {
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "search_hm_name_from_conf",
              "Bad selection type",0);
    }

    for (idx = 0; 
	 idx < hash_marks_conf->hashmark_count; 
	 idx++) {

	if (0 == string_cmp(hash_marks_conf->hashmarks[idx].hashmark_text,
			    name, 
			    -99 /* can't compare */ )) {
	    
	    DPRINT(Debug,9,(&Debug, 
			    "search_hm_name_from_conf: hash %S, index %d\n",
			    name,idx));
	    
	    if (hash_marks_conf->hashmarks[idx].items[sel_type]) {
		DPRINT(Debug,9,(&Debug, 
				"search_hm_name_from_conf: found selection type %d\n",
				sel_type));

		inc_hashmark_item_refcount(hash_marks_conf->
					   hashmarks[idx].items[sel_type]);
		return hash_marks_conf->
		    hashmarks[idx].items[sel_type];		
	    }
	}
    }

    DPRINT(Debug,9,(&Debug, 
		    "search_hm_name_from_conf: hash %S, not found; %d hashmarks\n",
		    name,hash_marks_conf->hashmark_count));

    return NULL;
    
}

/* creates new hashmark_item */
static struct hashmark_item * merge_hashmark_item P_((const struct hashmark_item * base,
						      const struct hashmark_item * overwrite));
static struct hashmark_item * merge_hashmark_item(base,overwrite)
     const struct hashmark_item * base;
     const struct hashmark_item * overwrite;
{
    struct hashmark_item *ret = NULL;
    const struct string * name = NULL;

    if (HASHMARK_ITEM_magic != base->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge_hashmark_item",
              "Bad magic number (base)",0);

    if (HASHMARK_ITEM_magic != overwrite->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge_hashmark_item",
              "Bad magic number (overwrite)",0);

    if (base->hashmark_name &&
	overwrite->hashmark_name) {

	if (0 != string_cmp(base->hashmark_name,
			    overwrite->hashmark_name,
			    -99)) 
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "merge_hashmark_item",
		  "Name mismatch",0);
	name = base->hashmark_name;
    } else if (base->hashmark_name)
	name = base->hashmark_name;
    else
	name = overwrite->hashmark_name;

    if (base->hashtype != overwrite->hashtype)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge_hashmark_item",
              "Type mismatch",0);

    if (base->hashtype < hashtype_none ||
	base->hashtype >= NUM_hashtype ||
	!  hashtype_actions[base->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge_hashmark_item",
	      "Bad hashtype",0);

    if (HASHTYPE_ACTIONS_magic != hashtype_actions[base->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "merge_hashmark_item",
	      "Bad magic number (hashtype action)",0);


    ret = malloc_hashmark_item(base->hashtype,name);

    if (overwrite->description)
	ret->description = dup_string(overwrite->description);
    else if (base->description)
	ret->description = dup_string(base->description);

    if ((base->on_default_menu && !overwrite->hide_hashmark) || 
	overwrite->on_default_menu)
	ret->on_default_menu = 1;
    if ((base->hide_hashmark && !overwrite->on_default_menu) ||
	overwrite->hide_hashmark)
	ret->hide_hashmark = 1;
	    
    /* hashtype specific part of merge */

    hashtype_actions[base->hashtype]->merge(ret,base,overwrite);

    return ret;
}


/* increments refcount */
struct hashmark_item * search_hashmark_name(name,sel_type)
     const struct string *name;
     enum selection_type sel_type;
{
    struct hashmark_item *ret = NULL;

    if (system_hash_marks_conf) {
	ret = search_hm_name_from_conf(system_hash_marks_conf,
				       name,sel_type);

    }

    if (ret) {
	if (user_hash_marks_conf) {
	    struct hashmark_item * usr_item =  
		search_hm_name_from_conf(user_hash_marks_conf,
					 name,sel_type);

	    if (usr_item) {
		
		if (usr_item->hashtype != ret->hashtype) {
		    
		    /* Type mismatch, no merge */

		    free_hashmark_item(&ret);
		    ret = usr_item;
		    
		} else {
		    struct hashmark_item * temp =
			merge_hashmark_item(ret,usr_item);
		    
		    free_hashmark_item(&usr_item);
		    free_hashmark_item(&ret);
		    ret = temp;
		}
	    }
	}

    } else if (user_hash_marks_conf) {
	ret = search_hm_name_from_conf(user_hash_marks_conf,
				       name,sel_type);	
    }

    DPRINT(Debug,8,(&Debug, "search_hashmark_name=%p; name=%S, sel_type=%d\n",
		    ret,name,sel_type));

    return ret;
}

struct hashmark_item ** hashmark_dir(len,sel_type)
     int *len;
     enum selection_type sel_type;
{
    int idx = 0;
    int i;
    struct hashmark_item **vector = NULL;
    int maxlen = 0;

    if (user_hash_marks_conf) {
	if (HASH_MARKS_CONF_magic != user_hash_marks_conf->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "hashmark_dir",
		  "Bad magic (hash_marks_conf, user_hash_marks)",0);	

	if (user_hash_marks_conf->hashmark_count < 0)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "hashmark_dir",
		  "Bad hashmark_count (user_hash_marks)",0);	

	maxlen += user_hash_marks_conf->hashmark_count;
	    
    }

    if (system_hash_marks_conf) {
	if (HASH_MARKS_CONF_magic != system_hash_marks_conf->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "hashmark_dir",
		  "Bad magic (hash_marks_conf, user_hash_marks)",0);	

	if (system_hash_marks_conf->hashmark_count < 0)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "hashmark_dir",
		  "Bad hashmark_count (user_hash_marks)",0);	

	maxlen += system_hash_marks_conf->hashmark_count;
	    
    }

    if (maxlen < 1) {
	len = 0;
	return NULL;
    }

    if (sel_type < 0 || sel_type >= NUM_selection_type) {
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_dir",
              "Bad selection type",0);
    }

    vector = safe_calloc(maxlen,sizeof (vector[0]));

    for (i = 0; i < maxlen; i++)
	vector[i] = NULL;

    if (system_hash_marks_conf) {
	for (i = 0; i < system_hash_marks_conf->hashmark_count &&
		 idx < maxlen; i++) {
	    
	    if (system_hash_marks_conf->hashmarks[i].items[sel_type]) {

		vector[idx] = system_hash_marks_conf->hashmarks[i].items[sel_type];
		inc_hashmark_item_refcount(vector[idx]);

		idx++;
	    }
	}
    }

    if (user_hash_marks_conf) {
	for (i = 0; i < user_hash_marks_conf->hashmark_count &&
		 idx < maxlen; i++) {

	    int prev = -1;
	    int j;
	    
	    for (j =  0; j < idx; j++) {
		if (HASHMARK_ITEM_magic != vector[j]->magic)
		    panic("CONNECTION PANIC",__FILE__,__LINE__,
			  "hashmark_dir",
			  "Bad magic (hashmark_item)",0);

		if (vector[j]->hashmark_name &&
		    0 == string_cmp(vector[j]->hashmark_name,
				    user_hash_marks_conf->hashmarks[i].hashmark_text,
				    -99 /* can't compare */ )) {
		    prev = j;
		    break;
		}
	    }

	    if (user_hash_marks_conf->hashmarks[i].items[sel_type]) {

		if (prev >= 0) {

		    if (vector[prev]->hashtype ==
			user_hash_marks_conf->hashmarks[i].items[sel_type]->hashtype) {
			struct  hashmark_item * tmp =
			    merge_hashmark_item(vector[prev],
						user_hash_marks_conf->hashmarks[i].items[sel_type]);
			
			free_hashmark_item(& (vector[prev]));
			vector[prev] = tmp;
		    } else {
			/* Type mismatch, no merge
			   user's hashmark hides system's hashmark
			 */

			free_hashmark_item(& (vector[prev]));

			vector[prev] = user_hash_marks_conf->hashmarks[i].items[sel_type];
			inc_hashmark_item_refcount(vector[prev]);
		    }

		} else {
		    vector[idx] = user_hash_marks_conf->hashmarks[i].items[sel_type];
		    inc_hashmark_item_refcount(vector[idx]);

		    idx++;
		}
	    }
	}
    }

    *len = idx;
    return vector;
}

/* 'flags' value for add_dir_vector() */
int hashmark_browser_flags(item)
    const struct hashmark_item *item;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_browser_flags",
              "Bad magic number",0);

    /*
      BROWSER_NODIR                        Not a directory 
      BROWSER_NOFOLDER                     Not a folder  
      BROWSER_MARKED                       New mails ?   
      BROWSER_NEEDSTAT                     Internal only ... 
      BROWSER_HAVELOCKFILE                 Have correspond .lock file which is not shown 
      BROWSER_SESSIONLOCK                  Have corresponding sessionlock file 

      Operations for selection
      BROWSER_MAILFILE
      BROWSER_SELECTED
      BROWSER_EXIST
      BROWSER_DIRPREFIX                    Selection is dir prefix
    */

    if (item->hashtype < hashtype_none ||
	item->hashtype >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_browser_flags",
	      "Bad hashtype",0);

    if (HASHTYPE_ACTIONS_magic != hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_browser_flags",
	      "Bad magic number (hashtype action)",0);

    ret =  hashtype_actions[item->hashtype]->browser_flags(item);

    DPRINT(Debug,15,(&Debug, "hashmark_browser_flags=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));

    return ret;
}

void  hashmark_init_data(item,hashmark_data)
     const struct hashmark_item *item;
     union hashmark_data      * hashmark_data;
{
   if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_init_data",
              "Bad magic number",0);

    if (item->hashtype < hashtype_none ||
	item->hashtype >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_init_data",
	      "Bad hashtype",0);

    if (HASHTYPE_ACTIONS_magic != hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_init_data",
	      "Bad magic number (hashtype action)",0);

    
    hashtype_actions[item->hashtype]->init_data(item,hashmark_data);
}

void  hashmark_free_data(item,hashmark_data)
     const struct hashmark_item * item;
     union hashmark_data        * hashmark_data;
{
   if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_free_data",
              "Bad magic number",0);

    if (item->hashtype < hashtype_none ||
	item->hashtype >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_free_data",
	      "Bad hashtype",0);

    if (HASHTYPE_ACTIONS_magic != hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_free_data",
	      "Bad magic number (hashtype action)",0);

    hashtype_actions[item->hashtype]->free_data(item,hashmark_data);
}

/* Return 1 if change done */

int hashmark_changebr_dir(item,dir,hashmark_data,path_tail,dispname)
     struct hashmark_item       * item;
     struct folder_browser      * dir;
     union hashmark_data        * hashmark_data;
     const struct string        * path_tail; /* NULL if to root dir of hashmark */
     struct string             ** dispname;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_changebr_dir",
              "Bad magic number",0);
    
    if (item->hashtype < hashtype_none ||
	item->hashtype >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_changebr_dir",
	      "Bad hashtype",0);

    if (HASHTYPE_ACTIONS_magic != hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_changebr_dir",
	      "Bad magic number (hashtype action)",0);

    ret =  hashtype_actions[item->hashtype]->browser_changedir(item,dir,
							       hashmark_data,
							       path_tail,
							       dispname);

    DPRINT(Debug,15,(&Debug, "hashmark_changebr_dir=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));

    return ret;
}

/* Return 1 if succeed */

int hashmark_passhm_verify_ra_con(item,passhm,C)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account *C;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_verify_ra_con",
	      "Bad magic number (hashmark_item)",0);
    
    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_verify_ra_con",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_verify_ra_con",
	      "Bad magic number (hashtype action)",0);
        
    ret = hashtype_actions[item->hashtype]
	->passhm_verify_ra_con(item,passhm,C);

    DPRINT(Debug,15,(&Debug, "hashmark_passhm_verify_ra_con=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));

    return ret;
}

/*  Return 1 if succeed, 
    caller does remote_account_init_tls()
    -1 == use fallback
*/
int hashmark_passhm_open_ra(item,passhm,ra,default_portlist,
			    give_service_entry_flag,force_port)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account      * ra;
     const PORTS                  default_portlist[];
     int                          give_service_entry_flag;
     PORTS                        force_port;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_open_ra",
	      "Bad magic number (hashmark_item)",0);
    
    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_open_ra",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_open_ra",
	      "Bad magic number (hashtype action)",0);


    ret = hashtype_actions[item->hashtype]->
	passhm_open_ra(item,passhm,ra,default_portlist,
		       give_service_entry_flag,force_port);

    DPRINT(Debug,15,(&Debug, "hashmark_passhm_open_ra=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));

    return ret;
}

/*  Return 1 if succeed, 
    caller does remote_account_init_tls()
    -1 == use fallback
*/
int hashmark_passhm_open_ra2(item,passhm,ra,give_service_entry_flag,est)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account      * ra;
     int                          give_service_entry_flag;
					
     /* For enumerate_service_type */
     struct enum_service_type   * est;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_open_ra2",
	      "Bad magic number (hashmark_item)",0);
    
    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_open_ra2",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_passhm_open_ra2",
	      "Bad magic number (hashtype action)",0);

    ret = hashtype_actions[item->hashtype]->
	passhm_open_ra2(item,passhm,ra,give_service_entry_flag,est);

    DPRINT(Debug,15,(&Debug, "hashmark_passhm_open_ra2=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));

    return ret;

}

/* Return 1 if succeed */
int hashmark_selectbr_item(item,dir,hashmark_data,path_tail,dispname,newpos)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * path_tail;
     struct string        ** dispname;
     int                   * newpos;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_selectbr_item",
	      "Bad magic number (hashmark_item)",0);
    
    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_selectbr_item",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_selectbr_item",
	      "Bad magic number (hashtype action)",0);

    ret = hashtype_actions[item->hashtype]->
	browser_select_item(item,dir,hashmark_data,path_tail,dispname,newpos);

    DPRINT(Debug,15,(&Debug, "hashmark_selectbr_item=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));

    return ret;
}

struct folder_info * hashmark_folder_from_browser(item,dir,hashmark_data,treat_as_spooled)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     int treat_as_spooled;
{

    struct folder_info * ret = NULL;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_folder_from_browser",
	      "Bad magic number (hashmark_item)",0);
    
    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_folder_from_browser",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_folder_from_browser",
	      "Bad magic number (hashtype action)",0);

    ret = hashtype_actions[item->hashtype]->
	folder_from_browser(item,dir,hashmark_data,treat_as_spooled);

    if (ret) {
	DPRINT(Debug,15,(&Debug, "hashmark_folder_from_browser=%p",
			 ret));
    } else {
	DPRINT(Debug,15,(&Debug, "hashmark_folder_from_browser=NULL"));
    }
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));


    return ret;
}

/* -1 == use fallback 
   caller must free *se, *username, *hostname, *addr
*/
int hashmark_remote_lookup(item,se,lookup_flags,username,
			   hostname,addr,useraddr_flags,
			   cancel_p)
     const struct hashmark_item * item;
     struct service_entry      ** se;
     int                          lookup_flags;
     char                      ** username;
     char                      ** hostname;
     struct address            ** addr;
     int                        * useraddr_flags;
     struct cancel_data        ** cancel_p
     /* May be NULL, Used if dns lookup was cancelable */;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_remote_lookup",
	      "Bad magic number (hashmark_item)",0);
    
    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_remote_lookup",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_remote_lookup",
	      "Bad magic number (hashtype action)",0);

    ret = hashtype_actions[item->hashtype]->
	remote_lookup(item,se,lookup_flags,
		      username,hostname,addr,
		      useraddr_flags,cancel_p);

    DPRINT(Debug,15,(&Debug, "hashmark_remote_lookup=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));

    return ret;
}

int hashmark_selection_is_folder(item,dir,hashmark_data,folder)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     struct folder_info    * folder;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_selection_is_folder",
	      "Bad magic number (hashmark_item)",0);
    
    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_selection_is_folder",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_selection_is_folder",
	      "Bad magic number (hashtype action)",0);

    ret = hashtype_actions[item->hashtype]->
	selection_is_folder(item,dir,hashmark_data,folder);

    DPRINT(Debug,15,(&Debug, "hashmark_selection_is_folder=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));


    return ret;
}

/* Return 1 if succeed, changes browser type */
int hashmark_prepare_write(item,dir,hashmark_data,ptr)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     WRITE_STATE             ptr;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_prepare_write",
	      "Bad magic number (hashmark_item)",0);

    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_prepare_write",
	      "Bad hashtype",0);

    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_prepare_write",
	      "Bad magic number (hashtype action)",0);

    ret = hashtype_actions[item->hashtype]->
	prepare_write(item,dir,hashmark_data,ptr);

    DPRINT(Debug,15,(&Debug, "hashmark_prepare_write=%d",ret));
    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));
    
    return ret;
}

/* Return directory prefix and l_item,
   changes browser type
*/
struct string * hashmark_cat_item(h_item,dir,hashmark_data,l_item)
     struct hashmark_item  * h_item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * l_item;
{
    struct string * ret = NULL;
    
    if (HASHMARK_ITEM_magic != h_item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_cat_item",
	      "Bad magic number (hashmark_item)",0);

    if (h_item->hashtype < hashtype_none ||
	h_item->hashtype >= NUM_hashtype ||
	!  hashtype_actions[h_item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_cat_item",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[h_item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_cat_item",
	      "Bad magic number (hashtype action)",0);

    ret = hashtype_actions[h_item->hashtype]->
	cat_hashmark(h_item,dir,hashmark_data,l_item);

    if (ret) {
	DPRINT(Debug,15,(&Debug, "hashmark_cat_item=%S",ret));
    } else {
	DPRINT(Debug,15,(&Debug, "hashmark_cat_item=NULL"));
    }

    if (h_item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",h_item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));
    
    return ret;
}

enum hashtype_v hashmark_type(item)
    const struct hashmark_item * item;
{
    
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_type",
	      "Bad magic number (hashmark_item)",0);

    return item->hashtype;
}

int hashmark_visible(item)
     const struct hashmark_item  * item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_valid_user",
	      "Bad magic number (hashmark_item)",0);
    
    return ! (item->hide_hashmark);
}


enum haskmark_valid_v hashmark_valid_on_user(item)
     const struct hashmark_item * item;
{
    enum haskmark_valid_v ret = hashmark_unusable;

   if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_valid_user",
	      "Bad magic number (hashmark_item)",0);

    if (item->hashtype < hashtype_none ||
	item->hashtype  >= NUM_hashtype ||
	!  hashtype_actions[item->hashtype])
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_valid_on_user",
	      "Bad hashtype",0);
    
    if (HASHTYPE_ACTIONS_magic !=
	hashtype_actions[item->hashtype]->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_valid_on_user",
	      "Bad magic number ((hashtype action)",0);

    ret = hashtype_actions[item->hashtype]->
	valid_on_user(item);

    if (hashmark_valid == ret && item->on_default_menu) {
	DPRINT(Debug,15,(&Debug, "hashmark_valid_on_user: Showing on default menu\n"));
	ret = hashmark_default_menu;
    } else if (hashmark_valid == ret && item->hide_hashmark) {
	DPRINT(Debug,15,(&Debug, "hashmark_valid_on_user: Hiding hashmark\n"));
	ret = hashmark_hidden;
    } 
        
    DPRINT(Debug,15,(&Debug, "hashmark_valid_on_user=%d",ret));

    switch(ret) {
    case hashmark_unusable: DPRINT(Debug,15,(&Debug," hashmark_unusable")); break;
    case hashmark_valid:    DPRINT(Debug,15,(&Debug," hashmark_valid")); break;
    case hashmark_default_menu: DPRINT(Debug,15,(&Debug," hashmark_default_menu")); break;
    case hashmark_hidden:   DPRINT(Debug,15,(&Debug," hashmark_hidden")); break;
    }

    if (item->hashmark_name) {
	DPRINT(Debug,15,(&Debug, " (name=%S)",item->hashmark_name));
    }
    DPRINT(Debug,15,(&Debug, "\n"));

    return ret;
}

const struct string * hashmark_item_description(item)
     const struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashmark_item_description",
	      "Bad magic number (hashmark_item)",0);

    return item->description;
}

void change_hash_marks_conf(conf,new)
     struct hash_marks_conf **conf;
     const struct hash_marks_conf *new;
{
    int x;
    int max_count;

    if (*conf) {
	if (HASH_MARKS_CONF_magic != (*conf)->magic)
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  "change_hash_marks_conf",
		  "Bad magic number (*conf, hash_marks_conf) ",0);
    } else 
	*conf = malloc_hash_marks_conf();

    if (HASH_MARKS_CONF_magic != new->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "change_hash_marks_conf",
	      "Bad magic number (new, hash_marks_conf) ",0);

    if (new->hashmark_count < 1)
	return;

    max_count = new->hashmark_count + (*conf)->hashmark_count;

    (*conf)->hashmarks =
	safe_array_realloc((*conf)->hashmarks,
			   max_count, sizeof( (*conf)->hashmarks[0]));

    for (x = 0; x < new->hashmark_count; x++) {
	
	if (new->hashmarks[x].hashmark_text) {

	    int current_index;
	    enum selection_type st;
			

	    for (current_index = 0; 
		 current_index < (*conf)->hashmark_count; 
		 current_index++) {
		if (0 == string_cmp((*conf)->hashmarks[current_index].hashmark_text,
				    new->hashmarks[x].hashmark_text,
				    -99 /* can't compare */ ))
		    goto found;		
	    }

	    if (current_index != (*conf)->hashmark_count)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "change_hash_marks_conf",
		      "Bad index",0);

	    if (current_index >= max_count)
		panic("CONNECTION PANIC",__FILE__,__LINE__,
		      "change_hash_marks_conf",
		      "Overflow",0);

	    (*conf)->hashmarks[current_index].hashmark_text
		= dup_string(new->hashmarks[x].hashmark_text);

	    for (st = 0; st < NUM_selection_type; st++)
		(*conf)->hashmarks[current_index].items[st] = NULL;
	    (*conf)->hashmark_count++;
	    
	found:
	    
	    for (st = 0; st < NUM_selection_type; st++) {

		if (new->hashmarks[x].items[st]) {

		    if (HASHMARK_ITEM_magic != new->hashmarks[x].items[st]->magic)
			panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "hashmark_dir",
			      "Bad magic (new, hashmark_item)",0);
		    
		    if ((*conf)->hashmarks[current_index].items[st]) {
			if (HASHMARK_ITEM_magic != 
			    (*conf)->hashmarks[current_index].items[st]->magic)
			    panic("CONNECTION PANIC",__FILE__,__LINE__,
			      "hashmark_dir",
				  "Bad magic (*conf, hashmark_item)",0);
			
			if ((*conf)->hashmarks[current_index].items[st]->hashtype ==
			    new->hashmarks[x].items[st]->hashtype) {
			    
			    struct  hashmark_item * tmp =
				merge_hashmark_item((*conf)->hashmarks[current_index].items[st],
						    new->hashmarks[x].items[st]);

			    free_hashmark_item(& ((*conf)->hashmarks[current_index].items[st]));
			    (*conf)->hashmarks[current_index].items[st] = tmp;

			} else {
			    /* Type mismatch, no merge */

			    free_hashmark_item(& ((*conf)->hashmarks[current_index].items[st]));
			    (*conf)->hashmarks[current_index].items[st] =
				new->hashmarks[x].items[st];
			    inc_hashmark_item_refcount((*conf)->hashmarks[current_index].items[st]);
			}

		    } else {

			(*conf)->hashmarks[current_index].items[st] =
			    new->hashmarks[x].items[st];
			inc_hashmark_item_refcount((*conf)->hashmarks[current_index].items[st]);

		    }
		}
	    }
	}
    }
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
