static char rcsid[] = "@(#)$Id: hashmark_local.c,v 2.7 2024/06/10 18:05:34 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.7 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *       or Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "hashmark.h"
#include "hashmark_imp.h"
#include "s_me.h"
#include "rc_imp.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif



static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}


static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}



S_(hashtype_init_item_f hashtype_init_item_dir)
static void hashtype_init_item_dir P_((struct hashmark_item *item));
S_(hashtype_free_item_f  hashtype_free_item_dir)
static void hashtype_free_item_dir P_((struct hashmark_item *item));
S_(hashtype_parse_kw_value_f hashtype_parse_kw_value_dir)
static int hashtype_parse_kw_value_dir P_((const char *filename,int lineno,
					   struct string * rest_S,
					   struct hashmark_item *item,
					   enum hashkeyword_v kw,
					   struct string * value,
					   enum record_mode rc,
					   const struct schedule_timelimit * now));

S_(hashtype_parse_kw_f hashtype_parse_kw_dir)
static int hashtype_parse_kw_dir P_((const char *filename,int lineno,
				     struct string * rest_S,
				     struct hashmark_item *item,
				     enum hashkeyword_v kw,
				     enum record_mode rc,
				     const struct schedule_timelimit * now));

S_(hashtype_parse_check_f hashtype_parse_check_dir);
static int hashtype_parse_check_dir P_((const char *filename,int lineno,
					struct hashmark_item *item,
					enum record_mode rc,
					const struct schedule_timelimit * now));

S_(hashtype_dump_kw_f hashtype_dump_kw_dir)
static int hashtype_dump_kw_dir P_((FILE *F,
				    struct hashmark_item *item,
				    enum hashkeyword_v kw,
				    charset_t fileset,
				    const char * sysnam,
				    enum record_mode rc));

S_(hashtype_merge_f hashtype_merge_dir)
static void hashtype_merge_dir P_((struct hashmark_item *item,
				   const struct hashmark_item * base,
				   const struct hashmark_item * overwrite));

S_(hashtype_browser_flags_f hashtype_brflags_dir)
static int  hashtype_brflags_dir P_((const struct hashmark_item *item));

S_(hashtype_init_data_f hashtype_initd_dir)
static void hashtype_initd_dir P_((const struct hashmark_item *item,
				    union hashmark_data      * hashmark_data));

S_(hashtype_free_data_f hashtype_freed_dir)
static void hashtype_freed_dir P_((const struct hashmark_item *item,
				    union hashmark_data      * hashmark_data));

S_(hashtype_browser_changedir_f hashtype_brchdir_dir)
static int hashtype_brchdir_dir P_((struct hashmark_item       * item,
				    struct folder_browser      * dir,
				    union hashmark_data        * hashmark_data,
				    const struct string        * path_tail,
				    struct string             ** dispname));

S_(hashtype_passhm_verify_ra_con_f hashtype_verify_racon_dir)
static int hashtype_verify_racon_dir P_(( const struct hashmark_item * item,
					  struct browser_passhm *passhm,
					  struct remote_account *C));

S_(hashtype_passhm_open_ra_f hashtype_passhm_open_ra_dir)
static int hashtype_passhm_open_ra_dir P_((const struct hashmark_item * item,
					   struct browser_passhm      * passhm,
					   struct remote_account      * ra,
					   const PORTS                  default_portlist[],
					   int                          give_service_entry_flag,
					   PORTS                        force_port));

S_( hashtype_passhm_open_ra2_f hashtype_passhm_open_ra2_dir)
static int hashtype_passhm_open_ra2_dir P_((const struct hashmark_item * item,
					     struct browser_passhm      * passhm,
					     struct remote_account      * ra,
					     int                          give_service_entry_flag,
					     
					     /* For enumerate_service_type */
					     struct enum_service_type   * est
					     ));


S_(hashtype_selectbr_item_f hashtype_selectbr_item_filedir)
static int hashtype_selectbr_item_filedir P_((struct hashmark_item  * item,
					  struct folder_browser * dir,
					  union hashmark_data   * hashmark_data,
					  const struct string   * path_tail,
					  struct string        ** dispname,
					  int                   * newpos));

S_(hashtype_selectbr_item_f hashtype_selectbr_item_folderdir)
static int hashtype_selectbr_item_folderdir P_((struct hashmark_item  * item,
					  struct folder_browser * dir,
					  union hashmark_data   * hashmark_data,
					  const struct string   * path_tail,
					  struct string        ** dispname,
					  int                   * newpos));


S_(hashtype_folder_from_browser_f hashtype_folder_from_folderdir)
static struct folder_info * hashtype_folder_from_folderdir
    P_((struct hashmark_item  * item,
	struct folder_browser * dir,
	union hashmark_data   * hashmark_data,
	int                     treat_as_spooled));

S_(hashtype_folder_from_browser_f hashtype_folder_from_filed)
static struct folder_info * hashtype_folder_from_filed
    P_((struct hashmark_item  * item,
	struct folder_browser * dir,
	union hashmark_data   * hashmark_data,
	int                     treat_as_spooled));

/* -1 == use fallback
    0 == lookup failed
    1 == ok

    caller must free *se, *username, *hostname, *addr
*/
S_(hashtype_remote_lookup_f hashtype_remote_lookup_dir)
static int hashtype_remote_lookup_dir P_((const struct hashmark_item * item,
					  struct service_entry      ** se,
					  int                          lookup_flags,
					  char                      ** username,
					  char                      ** hostname,
					  struct address            ** addr,
					  int                        * useraddr_flags,
					  struct cancel_data         ** cancel_p
					  /* May be NULL, Used if dns lookup was
					     cancelable 
					  */
					  ));

S_(hashtype_selection_is_folder_f hashtype_selection_is_folderdir)
static int hashtype_selection_is_folderdir P_((struct hashmark_item  * item,
					  struct folder_browser * dir,
					  union hashmark_data   * hashmark_data,
					  struct folder_info    * folder));

S_(hashtype_selection_is_folder_f hashtype_selection_is_filedir)
static int hashtype_selection_is_filedir P_((struct hashmark_item  * item,
					  struct folder_browser * dir,
					  union hashmark_data   * hashmark_data,
					  struct folder_info    * folder));

S_(hashtype_valid_on_user_f hashtype_valid_on_user_dir)
static enum haskmark_valid_v hashtype_valid_on_user_dir P_((const struct hashmark_item  * 
							    item));

S_(hashtype_prepare_write_item_f hashtype_prepare_write_dir)
static int hashtype_prepare_write_dir P_((struct hashmark_item  * item,
					  struct folder_browser * dir,
					  union hashmark_data   * hashmark_data,
					  WRITE_STATE             ptr));


/* Return directory prefix and l_item,
   changes browser type
*/
S_(hashtype_cat_hashmark_item_f hashtype_cat_hashmark_dir)
static  struct string * hashtype_cat_hashmark_dir
    P_((struct hashmark_item  * h_item,
	struct folder_browser * dir,
	union hashmark_data   * hashmark_data,
	const struct string   * l_item));

struct hashtype_actions hashtype_folder_dir_action = 
    { /* hashtype_folder_dir */
	HASHTYPE_ACTIONS_magic,
	hashtype_init_item_dir,
	hashtype_free_item_dir,
	hashtype_parse_kw_value_dir,
	hashtype_parse_kw_dir,
	hashtype_parse_check_dir,
	hashtype_dump_kw_dir,
	hashtype_merge_dir,
	hashtype_brflags_dir,
	hashtype_initd_dir,
	hashtype_freed_dir,
	hashtype_brchdir_dir,
	hashtype_verify_racon_dir,
	hashtype_passhm_open_ra_dir,
	hashtype_selectbr_item_folderdir,
	hashtype_folder_from_folderdir,
	hashtype_remote_lookup_dir,
	hashtype_selection_is_folderdir,
	hashtype_valid_on_user_dir,
	hashtype_prepare_write_dir,
	hashtype_cat_hashmark_dir,
	hashtype_passhm_open_ra2_dir,
    };

struct hashtype_actions hashtype_local_dir_action = 
    { /* hashtype_dir */
	HASHTYPE_ACTIONS_magic,
	hashtype_init_item_dir,
	hashtype_free_item_dir,
	hashtype_parse_kw_value_dir,
	hashtype_parse_kw_dir,
	hashtype_parse_check_dir,
	hashtype_dump_kw_dir,
	hashtype_merge_dir,
	hashtype_brflags_dir,
	hashtype_initd_dir,
	hashtype_freed_dir,
	hashtype_brchdir_dir,
	hashtype_verify_racon_dir,
	hashtype_passhm_open_ra_dir,
	hashtype_selectbr_item_filedir,
	hashtype_folder_from_filed,
	hashtype_remote_lookup_dir,
	hashtype_selection_is_filedir,
	hashtype_valid_on_user_dir,
	hashtype_prepare_write_dir,
	hashtype_cat_hashmark_dir,
	hashtype_passhm_open_ra2_dir
    };


#define HASHMARK_DIR_magic	0xF51D

struct hashmark_dir {
    unsigned short magic; 	/* HASHMARK_DIR_magic */

    struct string      * directory;
    char               * expanded_dir /* local-fs-charset */;           
};


static void hashtype_init_item_dir(item)
     struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_init_item_dir",
              "Bad magic number",0);

    item->u.dir = safe_zero_alloc(sizeof(* (item->u.dir)));
    item->u.dir->magic     = HASHMARK_DIR_magic;
    item->u.dir->directory    = NULL;
    item->u.dir->expanded_dir = NULL;
}

static void hashtype_free_item_dir(item)
     struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_free_item_dir",
              "Bad magic number",0);
  
    if (HASHMARK_DIR_magic != item->u.dir->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_free_item_dir",
              "Bad magic number (hashmark_dir)",0);

    if (item->u.dir->directory) {
	free_string(&(item->u.dir->directory));
    }

    if (item->u.dir->expanded_dir) {
	free(item->u.dir->expanded_dir);
	item->u.dir->expanded_dir = NULL;
    }

    free(item->u.dir);
    item->u.dir = NULL;
}

static int hashtype_parse_kw_value_dir(filename,lineno,
				       rest_S,item,kw,
				       value,rc,now)
     const char *filename;
     int lineno;
     struct string * rest_S;
     struct hashmark_item *item;
     enum hashkeyword_v kw;
     struct string * value;
     enum record_mode rc;
     const struct schedule_timelimit * now;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_kw_value_dir",
	      "Bad magic (hashmark_item)",0);	

    switch (kw) {
    case hashkeyword_dir: 

	if (HASHMARK_DIR_magic != item->u.dir->magic) 
	    panic("CONNECTION PANIC",__FILE__,__LINE__,
		  " hashtype_parse_kw_value_dir",
		  "Bad magic number (hashmark_dir)",0);

	if (item->u.dir->directory) {
	    if (hashmark_is_printableln(filename,lineno,rest_S))
		lib_error(CATGETS(elm_msg_cat, MeSet, 
				  MeDuplicateKwOnLine,
				  "%s: %d: Duplicate keyword %s on line: %S"),
			  filename,lineno,
			  hashmark_Keyword(kw),rest_S);
	    ret = 0;

	} else {
	    int             value_error = 0;
	    
	    item->u.dir->directory =  unquote_string(value,hashmark_is_printableln,
						     filename,lineno,&value_error);
	    
	    if (value_error) {
		if (hashmark_is_printableln(filename,lineno,rest_S))
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeKwdBADvalueOnLine,
				      "%s: %d: Keyword %s have bad value on line: %S"),
			      filename,lineno,
			      hashmark_Keyword(kw),rest_S);
	    }

	    if (item->u.dir->directory) {
		char            expand_space[LONG_STRING];
		unsigned char * str;
		int             failcount = 0;
		struct string * Lstr = convert_string2(local_fs_charset,item->u.dir->directory,&failcount);

		if (failcount) {
		    const char *x = get_charset_MIME_name(local_fs_charset);

		    if (hashmark_is_printableln(filename,lineno,rest_S))
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeNameNotValidAsLocalOnLine,
					  "%s: %d: Name %S not valid as local, %d errors with charset %s on line: %S"),
				  filename,lineno,
				  item->u.dir->directory,failcount,
				  x ? x : raw_local_fs_charset,
				  rest_S);				     
		}

		str = stream_from_string(Lstr,0,NULL);

		if (expand_meta(expand_space,us2s(str),sizeof expand_space) >= 0) {
		    if (0 == failcount)
			ret =!value_error;

		} else {
		    if (hashmark_is_printableln(filename,lineno,rest_S))
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeBadVariableOnKWValOnLine,
					  "%s: %d: Bad variable on keyword %s value %S on line %S"),
				  filename,lineno,hashmark_Keyword(kw),
				  Lstr,rest_S);	
		}
		
		item->u.dir->expanded_dir = 
		    strmcpy(item->u.dir->expanded_dir,expand_space);
	    
		free_string(&Lstr);
		free(str);
	    }
	}

	break;
    default:
	if (hashmark_is_printableln(filename,lineno,rest_S))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeHTypeDontSupoKwOnLIne,
			      "%s: %d: Hashtype %s do not support keyword %s on line %S"),
		      filename,lineno,
		      hashmark_Hashtype(item->hashtype),
		      hashmark_Keyword(kw),
		      rest_S);
	ret = 0;
	break;
    }
    
    return ret;
}


static int hashtype_parse_kw_dir(filename,lineno,rest_S,item,kw,rc,now)
     const char *filename;
     int lineno;
     struct string * rest_S;
     struct hashmark_item *item;
     enum hashkeyword_v kw;
     enum record_mode rc;
     const struct schedule_timelimit * now;
{
   if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_kw_dir",
	      "Bad magic (hashmark_item)",0);	

    switch (kw) {
    case hashkeyword_dir: 
	if (hashmark_is_printableln(filename,lineno,rest_S))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeHTypeKwREqValOnLine,
			      "%s: %d: Hashtype %s keyword %s requires value on line %S"),
		      filename,lineno,
		      hashmark_Hashtype(item->hashtype),
		      hashmark_Keyword(kw),
		      rest_S);
	break;

    default:
	if (hashmark_is_printableln(filename,lineno,rest_S))
	    lib_error(CATGETS(elm_msg_cat, MeSet, 
			      MeHTypeDontSupoKwOnLIne,
			      "%s: %d: Hashtype %s do not support keyword %s on line %S"),
		      filename,lineno,
		      hashmark_Hashtype(item->hashtype),
		      hashmark_Keyword(kw),
		      rest_S);
	break;
    }
    return 0;
}

static int hashtype_parse_check_dir(filename,lineno,item,rc,now)
     const char *filename;
     int lineno;
     struct hashmark_item *item;
     enum record_mode rc;
     const struct schedule_timelimit * now;
{
    struct hashmark_dir * item_d;

     if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_parse_check_dir",
	      "Bad magic (hashmark_item)",0);	

     if (HASHMARK_DIR_magic != item->u.dir->magic) 
	 panic("CONNECTION PANIC",__FILE__,__LINE__,
	       "hashtype_parse_check_dir",
	       "Bad magic number (hashmark_dir)",0);
     item_d = item->u.dir;

     if (! item_d->directory) {
	 switch (rc) {
	 case SYSTEM_RC:
	 case RC_MODE_COUNT:
	     item_d->directory =
		 format_string(FRM("/"));
	     item_d->expanded_dir = 
		 strmcpy(item_d->expanded_dir,
			 "/");
	     break;
	 case LOCAL_RC:
	     item_d->directory =
		 format_string(FRM("$HOME"));
	     item_d->expanded_dir = 
		 strmcpy(item_d->expanded_dir,
			 home);
	     break;
	 }

	 lib_error(CATGETS(elm_msg_cat, MeSet, 
			   MeHTypeReqKeywordValue,
			   "%s: %d: Hashtype %s requires value for keyword %s; Using value %S"),
		   filename,lineno,
		   hashmark_Hashtype(item->hashtype),
		   hashmark_Keyword(hashkeyword_dir),
		   item_d->directory);

	 return 0;
     }

     return 1; 
}

static int hashtype_dump_kw_dir(F,item,kw,
				 fileset,sysnam,rc)
     FILE                 * F;
     struct hashmark_item * item;
     enum hashkeyword_v     kw;
     charset_t              fileset;
     const char           * sysnam;
     enum record_mode       rc;
{
    struct hashmark_dir * item_d;
    int ok = 1;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_dump_kw_dir",
              "Bad magic number",0);

    if (HASHMARK_DIR_magic != item->u.dir->magic) 
	 panic("CONNECTION PANIC",__FILE__,__LINE__,
	       "hashtype_dump_kw_dir",
	       "Bad magic number (hashmark_dir)",0);
     item_d = item->u.dir;

    switch (kw) {
    case hashkeyword_dir: 
	if (item_d->directory) 
	    ok = dump_string_quote_helper(F,fileset,sysnam,kw,
					  item_d->directory,
					  -1);	    
	break;
    default:
	break;
    }

    return ok;
}

static void hashtype_merge_dir(item,base,overwrite)
     struct hashmark_item *item;
     const struct hashmark_item * base;
     const struct hashmark_item * overwrite;
{
    struct hashmark_dir * item_d;
    struct hashmark_dir * base_d;
    struct hashmark_dir * overwrite_d;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_dir",
              "Bad magic number (item)",0);

    if (HASHMARK_ITEM_magic != base->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_dir",
              "Bad magic number (base)",0);

    if (HASHMARK_ITEM_magic != overwrite->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_dir",
              "Bad magic number (overwrite)",0);

    if (HASHMARK_DIR_magic != item->u.dir->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_dir",
              "Bad magic number (hashmark_dir, item)",0);
    item_d = item->u.dir;

    if (HASHMARK_DIR_magic != base->u.dir->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_dir",
              "Bad magic number (hashmark_dir, base)",0);
    base_d = base->u.dir;

    if (HASHMARK_DIR_magic != overwrite->u.dir->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_merge_dir",
              "Bad magic number (hashmark_dir, overwrite)",0);
    overwrite_d =  overwrite->u.dir; 

    if (overwrite_d->directory) {
	item_d->directory =
	    dup_string(overwrite_d->directory);

	if (overwrite_d->expanded_dir)
	    item_d->expanded_dir =
		safe_strdup(overwrite_d->expanded_dir);
	
    } else if (base_d->directory) {
	item_d->directory =
	    dup_string(base_d->directory);

	if (base_d->expanded_dir)
	    item_d->expanded_dir =
		safe_strdup(base_d->expanded_dir);
    }
}

static int  hashtype_brflags_dir(item) 
     const struct hashmark_item *item;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_brflags_dir",
              "Bad magic number",0);
    
    return BROWSER_NOFOLDER;
}

static void hashtype_initd_dir(item,hashmark_data)
     const struct hashmark_item *item;
     union hashmark_data      * hashmark_data;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_initd_dir",
              "Bad magic number",0);

    hashmark_data->dummy = NULL;
}

static void hashtype_freed_dir(item,hashmark_data)
     const struct hashmark_item *item;
     union hashmark_data      * hashmark_data;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_freed_dir",
              "Bad magic number",0);

    if (hashmark_data->dummy) {
	free(hashmark_data->dummy);
	hashmark_data->dummy = NULL;
    }
}


static int hashtype_br_change_local P_((struct hashmark_item       * item,
					struct folder_browser      * dir,
					const struct string        * path_tail,
					struct string             ** new_buffer,
					struct string             ** dispname));
static int hashtype_br_change_local(item,dir,path_tail,new_buffer,dispname)
     struct hashmark_item       * item;
     struct folder_browser      * dir;
     const struct string        * path_tail; /* NULL if to root dir of hashmark */
     struct string             ** new_buffer;
     struct string             ** dispname;
{
    struct hashmark_dir * item_d;
    int ret = 0;
    struct browser_type *old_type = dir->type;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_br_change_local",
              "Bad magic number",0);

    if (HASHMARK_DIR_magic != item->u.dir->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_br_change_local",
              "Bad magic number (hashmark_dir, item)",0);
    item_d = item->u.dir;
  
    if (item_d->expanded_dir && item_d->directory) {

	if (change_folder_browser_type(dir,
#ifdef DIROPS
				       &local_browser
#else
				       &dummy_browser
#endif
				       )) {
	    
	    struct stat buf;
	    int l1 = strlen(item_d->expanded_dir);
	    int add_slash = 0;
	    struct string *
		buffer = new_string2(local_fs_charset,
				     s2us(item_d->expanded_dir));
			   	  	    
	    if (l1 > 0 && '/' != item_d->expanded_dir[l1-1]) {
		add_slash = 1;
	    }
	    
	    if (path_tail) {
		if (add_slash)
		    fill_ascii_to_string(buffer,1,'/'); 
		
		append_string(&buffer,path_tail,0);
	    }
	    
	    if (0 == stat(item_d->expanded_dir,&buf)) {
		
		if (
#ifdef S_ISDIR
		    S_ISDIR(buf.st_mode)
#else
		    S_IFDIR == (buf.st_mode & S_IFMT)
#endif
		    ) {

		    struct string * new_dirname    = dup_string(item_d->directory);
		    char          * new_sysname   = safe_strdup(item_d->expanded_dir);

		    if (add_slash) {
			char buf[2];
			buf[0] = '/';
			buf[1] = '\0';
			new_sysname = strmcat(new_sysname,buf);

			fill_ascii_to_string(new_dirname,1,'/');
		    }
			
		    if (dir->dirname) {
			DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: dir->dirname %S => %S\n",
					 dir->dirname,new_dirname));
			free_string(&(dir->dirname));
		    } else {
			DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: Setting dir->dirname = %S\n",
					 new_dirname));
		    }
		    dir->dirname = new_dirname; new_dirname = NULL;
		    
		    if (dir->sys_dir) {
			DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: dir->sys_dir %s => %s\n",
					 dir->sys_dir,new_sysname));
			
		    } else {
			DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: Setting dir->sys_dir = %s\n",
					 new_sysname));
			free(dir->sys_dir); dir->sys_dir = NULL;
		    }
		    dir->sys_dir = new_sysname; new_sysname = NULL;


		    if (dispname) {
			uint16          firstcode     = 0x0000;
			struct string * new_dispname  = NULL;
			
			if (string_len(item_d->directory) > 0)
			    firstcode = give_unicode_from_string(item_d->directory,0);
			
			switch(firstcode) {
			case 0x0024 /* '$' */:
			case 0x007E /* '~' */:
			case 0x003D /* '=' */:
			case 0x002B /* '+' */:
			case 0x007B /* '{' */:
			    new_dispname = dup_string(item_d->directory);
			    
			    if (path_tail) {
				if (add_slash)
				    fill_ascii_to_string(new_dispname,1,'/'); 
				
				append_string(&new_dispname,path_tail,0);
			    }
			    
			    break;
			default:
			    new_dispname = dup_string(buffer);
			    break;
			}
			
			if (*dispname)  {
			    DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: *dispname %S => %S\n",
					     *dispname,new_dispname));
			    free_string(dispname);		    
			} else {
			    DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: Setting *dispname = %S\n",
					     new_dispname));
			}
			*dispname = new_dispname; new_dispname = NULL;
		    }
		    
		    
		    
		    ret = 1;
		    
		} else {
		    DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: %s: Not a directory\n",
				     item_d->expanded_dir));
		}
		
	    } else {
		int err UNUSED_VAROK = errno;
		
		DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: %s: %s (errno=%d\n",
				 item_d->expanded_dir,strerror(err),err));
	    }

	    if (ret && new_buffer) {
		if (*new_buffer) {
		    DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: *new_buffer %S => %S\n",
				     *new_buffer,buffer));
		    
		    free_string(new_buffer);						
		} else {
		    DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: Setting *new_buffer = %S\n",
				     buffer));
		}
		*new_buffer = buffer; buffer = NULL;
		
	    } else
		free_string(&buffer);
	    
	} else {
	    DPRINT(Debug,16,(&Debug,
			     "hashtype_br_change_local: .. browser does not change type ?? ..\n"));
	}
    } else {	
	DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: .. hashmark dir not set\n"));
    }
    
    if (!ret && change_folder_browser_type(dir,old_type)) {
	DPRINT(Debug,16,(&Debug,"hashtype_br_change_local: Returned to old type\n"));
    }

    DPRINT(Debug,16,(&Debug,"hashtype_br_change_local=%d\n",
		     ret));

    return ret;
}


static int hashtype_brchdir_dir(item,dir,hashmark_data,path_tail,dispname)
     struct hashmark_item       * item;
     struct folder_browser      * dir;
     union hashmark_data        * hashmark_data;
     const struct string        * path_tail;  /* NULL if to root dir of hashmark */
     struct string             ** dispname;
{
    int ret = 0;
    
    struct string * new_buffer = NULL;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_brchdir_dir",
              "Bad magic number",0);

    if (hashtype_br_change_local(item,dir,path_tail,&new_buffer,dispname)) {


	ret = dir->type->browser_change_it(dir,new_buffer,dispname);
	
	if (!ret) {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_brchdir_dir: real browser_change_it [relative=%S] FAILED\n",
			     new_buffer));	
	}
	    	    
    } else {
	DPRINT(Debug,15,(&Debug,"hashtype_brchdir_dir: Not changed to local\n"));
    }
    
    if (new_buffer)
	free_string(&new_buffer);

    DPRINT(Debug,15,(&Debug,"hashtype_brchdir_dir=%d%s\n",
		     ret,
		     ret ? "" : " (failed)"));
    

    return ret;
}

static int hashtype_verify_racon_dir(item,passhm,C)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account *C;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_verify_racon_dir",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static int hashtype_passhm_open_ra_dir(item,passhm,ra,default_portlist,
				       give_service_entry_flag,force_port)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account      * ra;
     const PORTS                  default_portlist[];
     int                          give_service_entry_flag;
     PORTS                        force_port;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra_dir",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static int hashtype_passhm_open_ra2_dir(item,passhm,ra,give_service_entry_flag,est)
     const struct hashmark_item * item;
     struct browser_passhm      * passhm;
     struct remote_account      * ra;
     int                          give_service_entry_flag;
					     
     /* For enumerate_service_type */
     struct enum_service_type   * est;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_passhm_open_ra2_dir",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static int hashtype_select_on_item_dir P_((struct hashmark_item  * item,
					   struct folder_browser * dir,
					   union hashmark_data   * hashmark_data,
					   const struct string   * path_tail,
					   struct string        ** dispname,
					   int                   * newpos));
static int hashtype_select_on_item_dir(item,dir,hashmark_data,path_tail,dispname,newpos)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * path_tail;
     struct string        ** dispname;
     int                   * newpos;
{
    int ret = 0;
    struct string * new_buffer = NULL;

    if (hashtype_br_change_local(item,dir,path_tail,&new_buffer,dispname)) {
	
	ret = dir->type->browser_select_it(dir,new_buffer,
					   dispname,newpos);
    
	if (!ret) {
	    DPRINT(Debug,15,(&Debug,
			     " hashtype_select_on_item_dir: real browser_select_it [relative=%S] FAILED\n",
			     new_buffer));	
	}

    } else {
	DPRINT(Debug,15,(&Debug," hashtype_select_on_item_dir: Not changed to local\n"));
    }

    if (new_buffer)
	free_string(&new_buffer);


    return ret;
}

static int hashtype_select_this_item_dir P_((struct hashmark_item  * item,
					     struct folder_browser * dir,
					     union hashmark_data   * hashmark_data,
					     struct string        ** dispname,
					     int                   * newpos,
					     int                     fileflag));
static int hashtype_select_this_item_dir(item,dir,hashmark_data,dispname,newpos,
					 fileflag)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     struct string        ** dispname;
     int                   * newpos;
     int                     fileflag;
{
    int ret = 0;

    struct hashmark_dir * item_d;
    if (HASHMARK_DIR_magic != item->u.dir->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_select_this_item_dir",
	      "Bad magic number (hashmark_dir, item)",0);
    item_d = item->u.dir;

    if (item_d->expanded_dir && item_d->directory) {

	int F = local_selection_stat(item_d->expanded_dir,
				     /* Is this correct? */
				     browser_local_last_read(dir));
	struct string * X = NULL;
	
	if (0 == (F & BROWSER_NOFOLDER)) {
	    F |= fileflag;
	}

	if (item->hashmark_name)
	    X = format_string(FRM("#%S"),item->hashmark_name);
	else
	    X = dup_string(item_d->directory);


	if (*dispname)  {
	    DPRINT(Debug,16,(&Debug,"hashtype_select_this_item_dir: *dispname %S => %S\n",
			     *dispname,item_d->directory));
	    free_string(dispname);		    
	} else {
	    DPRINT(Debug,16,(&Debug,"hashtype_select_this_item_dir: Setting *dispname = %S\n",
			     item_d->directory));
	}
	*dispname = dup_string(item_d->directory);


	ret = 1;
	/* WARNING: set_dir_selection does not allocate strings -- 
	 *          it just assign pointers!
	 */
	set_hashmark(set_dir_selection(dir,
				       safe_strdup(item_d->expanded_dir),
				       X,F),
		     item);
    } else {	
	DPRINT(Debug,16,(&Debug,
			 "hashtype_select_this_item_dir: .. hashmark dir not set\n"));
    }

        
    return ret;
}

static int hashtype_selectbr_item_filedir(item,dir,hashmark_data,path_tail,dispname,newpos)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * path_tail;
     struct string        ** dispname;
     int                   * newpos;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selectbr_item_filedir",
              "Bad magic number",0);

    if (path_tail)
	ret = hashtype_select_on_item_dir(item,dir,hashmark_data,path_tail,dispname,newpos);
    else 
	ret =  hashtype_select_this_item_dir(item,dir,hashmark_data,dispname,newpos,0);
    
    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_filedir=%d%s\n",
		     ret,
		     ret ? "" : " (failed)"));

    return ret;
}

static int hashtype_selectbr_item_folderdir(item,dir,hashmark_data,path_tail,dispname,newpos)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * path_tail;
     struct string        ** dispname;
     int                   * newpos;
{
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selectbr_item_dir",
              "Bad magic number",0);

    if (path_tail)
	ret = hashtype_select_on_item_dir(item,dir,hashmark_data,path_tail,dispname,newpos);
    else 
	ret =  hashtype_select_this_item_dir(item,dir,hashmark_data,dispname,newpos,
					     BROWSER_MAILFILE);
    
    DPRINT(Debug,15,(&Debug,"hashtype_selectbr_item_folderdir=%d%s\n",
		     ret,
		     ret ? "" : " (failed)"));

    return ret;
}


/* folder-dir */
static struct folder_info * hashtype_folder_from_folderdir(item,dir,hashmark_data,
							 treat_as_spooled) 
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     int                     treat_as_spooled;
{
    struct hashmark_dir * item_d;
    struct folder_info *res = NULL;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_folder_from_folderdir",
              "Bad magic number",0);

     if (HASHMARK_DIR_magic != item->u.dir->magic) 
	 panic("CONNECTION PANIC",__FILE__,__LINE__,
	       "hashtype_folder_from_folderdir",
	       "Bad magic number (hashmark_dir)",0);
     item_d = item->u.dir;
    
    if (item_d->expanded_dir && item_d->directory) {
	enum folder_place place;
	
	/* Folder is hashmark directory ??? */

	res = mbx_new_folder();

	res -> cur_folder_sys  = safe_strdup(item_d->expanded_dir);
	res -> cur_folder_disp = dup_string(item_d->directory);
	
	res-> folder_type = get_folder_type(res -> cur_folder_sys,&place,
					    treat_as_spooled);
	res->folder_type->init_it(res);

    } else {	
	DPRINT(Debug,16,(&Debug,"hashtype_folder_from_folderdir: .. hashmark dir not set\n"));
    }

    DPRINT(Debug,15,(&Debug,"hashtype_folder_from_folderdir=%p%s  -- not supported\n",
		     res,
		     res ? "" : " (failed)"));

    return res;  /* Not really supported */
}

/* file-dir */
static struct folder_info * hashtype_folder_from_filed(item,dir,hashmark_data,
						       treat_as_spooled)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     int                     treat_as_spooled;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_folder_from_filed",
              "Bad magic number",0);


    return NULL;  /* UNSUPPORTED */
}

/* -1 == use fallback
    0 == lookup failed
    1 == ok

    caller must free *se, *username, *hostname, *addr
*/
static int hashtype_remote_lookup_dir(item,se,lookup_flags,username,
				      hostname,addr,useraddr_flags,
				      cancel_p)
     const struct hashmark_item  * item;
     struct service_entry      ** se;
     int                          lookup_flags; 
     char                      ** username;
     char                      ** hostname;
     struct address            ** addr;
     int                        * useraddr_flags;
     struct cancel_data        ** cancel_p
     /* May be NULL, Used if dns lookup was cancelable */;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_remote_lookup_dir",
              "Bad magic number",0);

    return 0; /* Not supported */
}

static int hashtype_selection_is_folderdir(item,dir,hashmark_data,folder)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     struct folder_info    * folder;
{
    struct hashmark_dir * item_d;
    int ret = 0;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selection_is_folderdir",
              "Bad magic number",0);

    if (HASHMARK_DIR_magic != item->u.dir->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      " hashtype_selection_is_folderdir",
              "Bad magic number (hashmark_dir, item)",0);
    item_d = item->u.dir;
  
    if (item_d->expanded_dir) {

	/* Folder is hashmark directory ??? */

	ret = 0 == strcmp(item_d->expanded_dir,
			  folder->cur_folder_sys);

    } else {	
	DPRINT(Debug,16,(&Debug,"hashtype_selection_is_folderdir: .. hashmark dir not set\n"));
    }

    DPRINT(Debug,15,(&Debug,"hashtype_selection_is_folderdir=%d%s  -- not supported\n",
		     ret,
		     ret ? "" : " (failed)"));

    return ret; 
}

static int hashtype_selection_is_filedir(item,dir,hashmark_data,folder)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     struct folder_info    * folder;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_selection_is_filedir",
              "Bad magic number",0);

    return 0;  /* Not supported */
}


static enum haskmark_valid_v hashtype_valid_on_user_dir(item)
    const struct hashmark_item  * item;
{
    struct hashmark_dir * item_d;
    enum haskmark_valid_v ret = hashmark_unusable;

    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_valid_on_user_dir",
              "Bad magic number",0);

    if (HASHMARK_DIR_magic != item->u.dir->magic) 
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_valid_on_user_dir",
              "Bad magic number (hashmark_dir, item)",0);
    item_d = item->u.dir;
    
    if (item_d->expanded_dir) {
	struct stat buf;

	if (0 == stat(item_d->expanded_dir,&buf)) {
	
	    if (
#ifdef S_ISDIR
		S_ISDIR(buf.st_mode)
#else
		S_IFDIR == (buf.st_mode & S_IFMT)
#endif
		) {

		if (0 == access(item_d->expanded_dir,EXECUTE_ACCESS)) {
		    
		    ret = hashmark_valid;
		
		} else 
		    goto fail;

	    } else {
		DPRINT(Debug,16,(&Debug,
				 "hashtype_valid_on_user_dir: %s: Not a directory\n",
				 item_d->expanded_dir));
	    }
	   
	} else {
	    int err UNUSED_VAROK;
	    
	fail:
	    err  = errno;
	    
	    DPRINT(Debug,16,(&Debug,"hashtype_valid_on_user_dir: %s: %s (errno=%d\n",
			     item_d->expanded_dir,strerror(err),err));
	}
    } else {
	DPRINT(Debug,16,(&Debug,"hashtype_valid_on_user_dir: .. hashmark dir not set\n"));
    }

    DPRINT(Debug,15,(&Debug,"hashtype_valid_on_user_dir=%d",
		     ret));

    switch(ret) {
    case hashmark_unusable: DPRINT(Debug,15,(&Debug," hashmark_unusable")); break;
    case hashmark_valid:    DPRINT(Debug,15,(&Debug," hashmark_valid")); break;
    case hashmark_default_menu: DPRINT(Debug,15,(&Debug," hashmark_default_menu")); break;
    case hashmark_hidden:   DPRINT(Debug,15,(&Debug," hashmark_hidden")); break;
    }

    DPRINT(Debug,15,(&Debug,"\n"));

    return ret;
}


static int hashtype_prepare_write_dir(item,dir,hashmark_data,ptr)
     struct hashmark_item  * item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     WRITE_STATE             ptr;
{
    if (HASHMARK_ITEM_magic != item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_prepare_write_dir",
              "Bad magic number",0);

    DPRINT(Debug,15,(&Debug,"hashtype_prepare_write_dir=0 (not supported)\n"));
    
    return 0; /* Not supported */
}

/* Return directory prefix and l_item,
   changes browser type
*/
static  struct string * hashtype_cat_hashmark_dir(h_item,dir,
						  hashmark_data,l_item)
     struct hashmark_item  * h_item;
     struct folder_browser * dir;
     union hashmark_data   * hashmark_data;
     const struct string   * l_item;
{
    struct string * ret = NULL;

    if (HASHMARK_ITEM_magic != h_item->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "hashtype_cat_hashmark_dir",
              "Bad magic number",0);

    
    if (hashtype_br_change_local(h_item,dir,
				 NULL /* path_tail */,
				 NULL /* new_buffer */,
				 NULL /* dispname   */)) {

	ret = dir->type->browser_cat_it(dir,l_item);

	if (!ret) {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_cat_hashmark_dir: real browser_cat_it FAILED\n"));
	    goto fallback;
	}	
    } else {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_dir: Not changed to local\n"));
    fallback:
	if (h_item->hashmark_name) {
	    DPRINT(Debug,15,(&Debug,
			     "hashtype_cat_hashmark_dir: Using hashmark name %S\n",
			     h_item->hashmark_name));

	    if (l_item)
		ret = format_string(FRM("#%S:%S"),
				    h_item->hashmark_name,
				    l_item);
	    else
		ret = format_string(FRM("#%S"),
				    h_item->hashmark_name);
	}
    }

    if (ret) {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_dir=%S\n",
			 ret));
    } else {
	DPRINT(Debug,15,(&Debug,
			 "hashtype_cat_hashmark_dir=NULL\n",
			 ret));

    }
        
    return ret;
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
