static char rcsid[] = "@(#)$Id: pop.c,v 2.48 2024/06/13 17:36:55 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.48 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "ss_imp.h"
#include "s_me.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"pop");

#ifdef REMOTE_MBX

/* Seems that h_errno is macro on AIX */
#ifndef h_errno
extern int h_errno;
#endif

#include <errno.h>

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
extern int errno;
#endif

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char *cs2us P_((const char *str));
static const unsigned char *cs2us(str) 
     const char *str;
{
    return (const unsigned char *)str;
}


#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

#define UIDL_ENTRY_magic	0xFA08

struct uidl_entry {
    unsigned short magic;    /* UIDL_ENTRY_magic */

    int refcount;

    
    char * uidl;
    int    status;
    
    unsigned int changed : 1;          /* == status not on disk */
    unsigned int     old : 1;          /* == old entry on disk */
    unsigned int  delete : 1;         /* Delete uidl when updating  */
};

#define UIDL_ENTRY_DEF_magic	0xFA09

struct uidl_entry_def {
    unsigned short magic;	      /*  UIDL_ENTRY_DEF_magic */

    int    status;
    unsigned int     old : 1;          /* == old entry on disk */  
};



static struct uidl_entry * malloc_uidl_entry P_((const char * uidl,
						 const struct uidl_entry_def * def));

S_(alloc_sort_item_f alloc_uidl_item)
static void alloc_uidl_item P_((union sort_item      * res,
				const union sort_key   key,
				const union sort_item_default def
				));
static void alloc_uidl_item(res,key,def)
     union sort_item      * res;
     const union sort_key   key;
     const union sort_item_default def;  
{
    res->uidl = malloc_uidl_entry(key.str,def.uidl);
}

static void free_uidl_entry P_((struct uidl_entry **uidl));

S_(free_sort_item_f free_uidl_item)
static void free_uidl_item P_((union sort_item      * ptr)); /* Decrements refcount */
static void free_uidl_item(ptr)
     union sort_item      * ptr; /* Decrements refcount */
{
    free_uidl_entry(& (ptr->uidl));
}

static void inc_uidl_entry_refcount P_((struct uidl_entry *ptr));

S_(inc_sort_item_refcount_f inc_uidl_item_refcount)
static void inc_uidl_item_refcount P_((union sort_item item));
static void inc_uidl_item_refcount(item)
     union sort_item item;
{
    inc_uidl_entry_refcount(item.uidl);
}

S_(sort_item_debug_name_f uidl_item_name)
static struct string * uidl_item_name   P_((const union sort_item item));
static struct string * uidl_item_name (item)
     const union sort_item item;
{
    struct uidl_entry *uidl = item.uidl;

    if (UIDL_ENTRY_magic != uidl->magic)
    	panic("MBX PANIC",__FILE__,__LINE__,
	      "uidl_item_name",
	      "Bad magic number (uidl_entry)",0);

    return new_string2(ASCII_SET,s2us(uidl->uidl));
}

S_(compare_sort_key_to_item_f uidl_key_cmp_item)
static int uidl_key_cmp_item P_((const union sort_key key,
				  const union sort_item item));
static int uidl_key_cmp_item(key,item)
     const union sort_key key;
     const union sort_item item;
{
    struct uidl_entry *uidl = item.uidl;
    int r;
    
    if (UIDL_ENTRY_magic != uidl->magic)
    	panic("MBX PANIC",__FILE__,__LINE__,
	      "uidl_key_cmp_item",
	      "Bad magic number (uidl_entry)",0);
    
    r = strcmp(key.str,uidl->uidl);

    if (r < 0) {
	DPRINT(Debug,16,(&Debug,
			 "uidl_key_cmp_item: name=\"%s\" < name=\"%s\"\n",
			 key.str,uidl->uidl));
    } else if (r > 0) {
	DPRINT(Debug,16,(&Debug,
			 "uidl_key_cmp_item: name=\"%s\" > name=\"%s\"\n",
			 key.str,uidl->uidl));
    } else {
	DPRINT(Debug,16,(&Debug,
			 "uidl_key_cmp_item: name=\"%s\" = name=\"%s\"\n",
			 key.str,uidl->uidl));
    }

    return r;
}

S_(sort_key_debug_name_f uidl_key_name)
static struct string * uidl_key_name P_((const union sort_key key));
static struct string * uidl_key_name(key)
     const union sort_key key;
{
    return new_string2(ASCII_SET,cs2us(key.str));
}

static struct sort_operation uidl_list_operation = {
    SORT_OPERATION_magic,
    alloc_uidl_item,
    free_uidl_item,
    inc_uidl_item_refcount,
    uidl_item_name,
    uidl_key_cmp_item,

    /* KEY operations */
    uidl_key_name
    
};

static void free_uidl_entry(uidl)
     struct uidl_entry **uidl;
{
    if (UIDL_ENTRY_magic != (*uidl)->magic)
    	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_uidl_entry",
	      "Bad magic number (uidl_entry)",0);

    if ((*uidl)->refcount < 1)
        panic("MBX PANIC",__FILE__,__LINE__,
              "free_uidl_entry",
              "Bad refcount",0);

    (*uidl)->refcount--;
    if ((*uidl)->refcount > 0) {
        /* Just reset this reference */
        
        *uidl = NULL;
        return;
    }

    if ((*uidl)->uidl) {
	free((*uidl)->uidl);
	(*uidl)->uidl = NULL;
    }
    
    (*uidl)->status  = 0;
    (*uidl)->changed = 0;

    (*uidl)->magic = 0;  /* Invalidate */
    free(*uidl);
    *uidl = NULL;
}

static struct uidl_entry * malloc_uidl_entry(uidl,def)
     const char *uidl;
     const struct uidl_entry_def * def;
{
    struct uidl_entry *ptr = safe_zero_alloc(sizeof (*ptr));

    ptr->uidl    = safe_strdup(uidl);
    ptr->status  = UNREAD | NEW;
    ptr->changed = 0;
    ptr->old     = 0;
    ptr->delete  = 0;
    
    if (def) {
	if (UIDL_ENTRY_DEF_magic != def->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "malloc_uidl_entry",
		  "Bad magic number (uidl_entry_def)",0);
	
	ptr->status = def->status;
	ptr->old    = def->old;
    }

    
    ptr->refcount = 1;
    ptr->magic = UIDL_ENTRY_magic;

    return ptr;
}

static void inc_uidl_entry_refcount(ptr)
     struct uidl_entry *ptr;
{
    if (UIDL_ENTRY_magic != ptr->magic)
    	panic("MBX PANIC",__FILE__,__LINE__,
	      "inc_uidl_entry_refcount",
	      "Bad magic number (uidl_entry)",0);
    
    ptr->refcount++;
}

/* Returns existing entry or creates new,
   increments refcount, caller must call free_uidl_entry()
 */

static struct uidl_entry * give_uidl P_(( struct sortlist  ** uidl_list,
					  const char * uidl,
					  const struct uidl_entry_def * def1,
					  size_t * idx,
					  int append_mode
					  ));
static struct uidl_entry * give_uidl(uidl_list,uidl,def1, idx,
				     append_mode)
     struct sortlist  ** uidl_list;
     const char * uidl;
     const struct uidl_entry_def * def1;
     size_t * idx;
     int append_mode;
{
    union sort_key     key;
    union sort_item    res;
    union sort_item_default def;
    size_t             res_idx = 0;

    struct uidl_entry * ret = NULL;
    enum sort_list_search_op      op = sort_list_search_create;

    
	
    DPRINT(Debug,14,(&Debug,
		     "give_uidl: uidl=\"%s\"\n",uidl));
    
    if (idx) {
	if (append_mode) {
	    res_idx = *idx;
	    op = sort_list_insert_hint;

	    DPRINT(Debug,14,(&Debug,
			     "give_uidl: insert hint %zu\n",
			     res_idx));
	}
	
	*idx = 0;
    }

    if (!*uidl_list) {
	DPRINT(Debug,14,(&Debug,
			 "give_uidl: Creating uidl_list\n"));

	*uidl_list = alloc_sort_list(&uidl_list_operation,
				     1 /* prealloc one item */);
    }

    key.str  = uidl;
    def.uidl = def1;
    res.dummy = NULL;

    /* Increments refcount, returns 1 if found */
    if (search_sort_list_item(*uidl_list,op,
			      key,def,&res,&res_idx,
			      NULL /* append_mode_rewrite hint */)) {

	DPRINT(Debug,14,(&Debug,
			 "give_uidl: search_sort_list_item found or created item #%zu\n",
			 res_idx));

	if (idx)
	    *idx  = res_idx;

	ret = res.uidl;
		
	DPRINT(Debug,14,(&Debug,
			 "give_uidl: uidl=\"%s\" = [%zu] uidl=\"%s\"\n",
			 uidl,res_idx,ret->uidl));

    } else {
	DPRINT(Debug,14,(&Debug,
			 "give_uidl: search_sort_list_item failed to create entry\n"));

	ret = malloc_uidl_entry(uidl,def1);

	DPRINT(Debug,14,(&Debug,
			 "give_uidl: uidl=\"%s\" = [no index] uidl=\"%s\"\n",
			 uidl,ret->uidl));	
	
    }
    
    DPRINT(Debug,14,(&Debug,
		     "give_uidl=%p (refcount=%d)\n",
		     ret,ret->refcount));
    return ret;
}

static void print_uidls P_((FILE *F, struct sortlist  * uidl_list));
static void print_uidls (F, uidl_list)
     FILE * F;
     struct sortlist  * uidl_list;
{
    size_t len;
    size_t i;
    size_t skipped = 0;

    /* 
     * Printed on order, so that initial load can append
     * to end of uidl_list and does not need move items when 
     * them are read.
     *
     *
     * NOTE: old tree based implementation used different order: 
     *
     * If wee print root first, and then other nodes, then
     * data will not go to tree on order when it is read to back.
     * That avoides degerated case of binary tree (ie. linear list)
     * if original tree was not degenerated.
     */

    if (!uidl_list)
	return;

    len = sort_list_len(uidl_list);

    DPRINT(Debug,14,(&Debug,
		     "print_uidl: [%p] len=%zu\n",
		     uidl_list,len));

    for (i = 0; i < len; i++) {
	union sort_item res;

	res.uidl = NULL;
	
	/* Increments refcount */
	get_sort_list_item(uidl_list,sort_list_get_normal,i,&res);
	
	if (res.uidl) {
	    if (UIDL_ENTRY_magic != res.uidl->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "print_uidl",
		      "Bad magic number (uidl_entry)",0);
	    
	    DPRINT(Debug,14,(&Debug,
			     "print_uidl: [%p] entry %zu = \"%s\"",
			     uidl_list,i,res.uidl->uidl));

	    if (res.uidl->delete) {
		DPRINT(Debug,14,(&Debug, " (delete)\n"));
		skipped++;
		continue;
	    }
	    DPRINT(Debug,14,(&Debug, "\n"));
	    
	    if (!ison(res.uidl->status, UNREAD)) {
		putc('R',F);
	    }
	    
	    if (ison(res.uidl->status, NEW)) {
		putc('N',F);
	    } else {
		putc('O',F);
	    }
	    if (ison(res.uidl->status, REPLIED_TO)) {
		putc('r',F);
	    }
	    fprintf(F," %s\n",res.uidl->uidl);

	    res.uidl->changed = 0;

	    free_uidl_entry(&(res.uidl));

	} else {
	    DPRINT(Debug,14,(&Debug,
			     "print_uidl: [%p] No item %zu\n",
			     uidl_list,i));
	}
    }

    if (skipped) {
	DPRINT(Debug,14,(&Debug,
			 "print_uidl: [%p] %zu items skipped (marked 'delete')\n",
			 uidl_list,skipped));
    }
}

typedef enum FLOCKING_status GrabLock_file_f P_((int flock_fd, struct dt_flags_info *flags));

static int GrabLock_file P_((const char *name, FILE * F,
			     GrabLock_file_f *Grab_file));
static int GrabLock_file(name,F,Grab_file)
     const char *name;
     FILE *F;
     GrabLock_file_f *Grab_file;
{
    enum FLOCKING_status ret = Grab_file(fileno(F),&mailbox_locking);
    
    if (FLOCKING_RETRY == ret) {
	int count = 0;
	
	lib_transient(CATGETS(elm_msg_cat, MeSet, MeLockingFile,
			      "Locking %s..."),
		      name);
	
	while (FLOCKING_RETRY == ret && count++ < 3) {
	    wait_for_timeout(1);
	    ret = Grab_file(fileno(F),&mailbox_locking);
	}
	
	if (FLOCKING_OK == ret) {
	    lib_transient(CATGETS(elm_msg_cat, MeSet, MeLockingFileOK,
				  "Locking %s... OK."),
			  name);

	}

	if (FLOCKING_RETRY == ret) {
	    return 0;
	}
    }
    
    if (FLOCKING_FAIL == ret) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeLockingFileFAIL,
			      "Locking %s... Failed."),
		      name);

	return 0;
    }
    
    return 1;
}

static const char UIDLS_HEADER[] = "ELMME+ UIDL\n";
static const int  UIDLS_HEADER_len = (sizeof UIDLS_HEADER) -1;

static int read_uidls P_((const char            * name,
			  FILE                  * F,
			  struct sortlist      ** uidl_list,
			  const struct uidl_entry_def * def));
static int read_uidls(name,F,uidl_list,def)
     const char            * name;
     FILE                  * F;
     struct sortlist      ** uidl_list;
     const struct uidl_entry_def * def;
{
    size_t read_count = 0;
    size_t updated = 0;
    char buffer[100];
    int ret;
    size_t idx = 0;   /* For append mode hint */

    if ((ret = mail_gets(buffer, sizeof buffer,F)) > 0) {
	if (ret != UIDLS_HEADER_len || 0 != memcmp(buffer,UIDLS_HEADER,
						   UIDLS_HEADER_len)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFileCorrupted,
			      "File %s is corrupted."),
		      name);
	    
	    return 0;
	}
    }
    
    if (*uidl_list) {
	size_t len = sort_list_len(*uidl_list);

	DPRINT(Debug,14,(&Debug,
			 "read_uidls: %zu items on uid_list already\n",
			 len));
	
    } else {
	size_t count = 0;
	size_t pos = ftell(F);
	enum syscall_status r;
	
	DPRINT(Debug,14,(&Debug,
			 "read_uidls: Creating uidl_list, counting lines from offset %ld at %s\n",
			 pos,name));

	
	while (0 < (ret = mail_gets(buffer, sizeof buffer,F))) {

	    /* So that truncate of file is not necessary */
	    if (0 == strcmp("--END\n",buffer))
		break;
	    count++;
	}
	
	DPRINT(Debug,14,(&Debug,
			 "read_uidls: Creating uidl_list, count=%zu of %s\n",
			 count,name));
	
	*uidl_list = alloc_sort_list(&uidl_list_operation,count);

	r = fseek(F,pos,SEEK_SET);
	switch (r) {
	    int err UNUSED_VAROK;
	case syscall_success:
	    DPRINT(Debug,14,(&Debug,
			     "read_uidls: Reading again from offset %ld at %s\n",
			     pos,name));
	    break;
	case syscall_error:
	    err = errno;
	    
	    DPRINT(Debug,14,(&Debug,
			     "read_uidls: Failed to seek to offset %ld at %s: %s\n",
			     pos,name,strerror(err)));

	    return 0;
	}
    }
	    
    while (0 < (ret = mail_gets(buffer, sizeof buffer,F))) {
	struct uidl_entry * A;
	int status = UNREAD;
	char * p;
	
	/* So that truncate of file is not necessary */
	if (0 == strcmp("--END\n",buffer))
	    break;

	if ('\n' == buffer[ret-1]) {
	    buffer[ret-1] = '\0';
	    ret--;
	}

	for (p = buffer; p - buffer < ret && ' ' != *p; p++) {
	    switch(*p) {
	    case 'R': 
		status &= ~UNREAD;
		break;
	    case 'O':
		status &= ~NEW;
		break;
	    case 'N':
		status |= NEW;
		break;
	    case 'r':
		status |= REPLIED_TO;
		break;
	    }
	}

	if (p-buffer >= ret) {
	    DPRINT(Debug,14,(&Debug,
			     "read_uidls: %s: Bad line: %s\n",
			     name,buffer)); 
	    continue;
	}
	p++;

	
	/* increments refcount, caller must call free_uidl_entry() 

	   Append mode uses idx as insertion position hint
	 */
	A = give_uidl(uidl_list,p,def,&idx,
		      1  /* Append mode */
		      );
	if (! A->changed) {
	    A -> status = status;
	    updated ++;
	}

	DPRINT(Debug,16,(&Debug,
			 "read_uidls: #%zu: \"%s\" -- status %d%s%s%s%s\n",
			 idx,
			 A->uidl,
			 A->status,
			 A->changed ? ", changed" : "",
			 A->old     ? ", old"     : "",
			 A->delete  ? ", delete"  : "",
			 def        ? ""          : ", no default"));
	
	free_uidl_entry(&A);

	read_count++;
	idx++;  /* For append mode position */
    }

    DPRINT(Debug,14,(&Debug,
		     "read_uidls: %zu status read (%zu updated) from %s\n",
		     read_count,updated,name));


    return 1;
}



static void read_old_uidls P_((const char *name,
			       struct sortlist  ** uidl_list));
static void read_old_uidls(name,uidl_list)
     const char *name;
     struct sortlist  ** uidl_list;
{
    FILE * F;
    
    int err = can_open(name, "r");

    static struct uidl_entry_def item_default = {
	UIDL_ENTRY_DEF_magic,

	UNREAD,
	1 /* old entry on disk */ 

    };
    

    DPRINT(Debug,14,(&Debug,
		     "read_old_uidls: name=%s\n",name));

    if (0 != err) {

	switch (err) {
	case ENOENT:
	    DPRINT(Debug,14,(&Debug,
			     "read_old_uidls: No file (OK)\n"));
	    
	    break;
	default:
	    lib_error(CATGETS(elm_msg_cat, ElmSet,ElmCantOpenFileErr,
			      "Can't open %s: %s"),
		      name,strerror(err));
	    break;
	}
	
	return;
    }
    
    F = fopen(name, "r");
    if (!F) {
	err = errno;
	lib_error(CATGETS(elm_msg_cat, ElmSet,ElmCantOpenFileErr,
			  "Can't open %s: %s"),
		  name,strerror(err));

	return;
    }

    if (! GrabLock_file(name,F,GrabRead_the_file)) {
	fclose(F);
	return;
    }

    if (! read_uidls(name,F,uidl_list,&item_default)) {
	DPRINT(Debug,14,(&Debug,"read_old_uidls: Reading failed\n"));

	Release_the_file(fileno(F),&mailbox_locking);
	fclose(F);
	return;
    }

    Release_the_file(fileno(F),&mailbox_locking);
    fclose(F);
    
    return;
}


static void remove_old_uidls P_((struct sortlist  ** uidl_list));
static void remove_old_uidls(uidl_list)
     struct sortlist  ** uidl_list;
{
    if (*uidl_list) {
	size_t marked = 0;
	size_t len = sort_list_len(*uidl_list);
	size_t i;
	
	DPRINT(Debug,14,(&Debug,
			 "remove_old_uidls: %zu entries\n",len));

	for (i = 0; i < len; i++) {

	    union sort_item res;
	    
	    res.uidl = NULL;
	    
	    
	    /* Increments refcount */
	    get_sort_list_item(*uidl_list,sort_list_get_normal,i,&res);
	    
	    if (res.uidl) {
		if (UIDL_ENTRY_magic != res.uidl->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "remove_old_uidls",
			  "Bad magic number (uidl_entry)",0);
		
		DPRINT(Debug,14,(&Debug,
				 "remove_old_uidls: entry %zu = \"%s\"",
				 i,res.uidl->uidl));

		if (res.uidl->old) {
		    DPRINT(Debug,14,(&Debug,", old -- marking for delete"));
		    res.uidl->delete = 1;
		    marked ++;
		} else if (! res.uidl->delete) {
		    DPRINT(Debug,14,(&Debug,", keeping"));
		}
		DPRINT(Debug,14,(&Debug,"\n"));

		free_uidl_entry(&(res.uidl));
	    }
	}

	if (marked) {
	    DPRINT(Debug,14,(&Debug,
			     "remove_old_uidls: %zu entries, %zu marked for delete\n",
			     len,marked));
	}
    }
}


static void update_uidls P_((const char *name,
			     struct sortlist  ** uidl_list));
static void update_uidls(name, uidl_list)
     const char *name;
     struct sortlist  ** uidl_list; 
{
    FILE * F;
    int err = can_open(name, "a");
        
    DPRINT(Debug,14,(&Debug,
		     "update_uidls: name=%s\n",name));

    
    if (0 != err) {
	lib_error(CATGETS(elm_msg_cat, ElmSet,ElmCantOpenFileErr,
			  "Can't open %s: %s"),
		  name,strerror(err));
	return;
    }

    F = open_or_create(name);
    if (!F) {
	err = errno;
	lib_error(CATGETS(elm_msg_cat, ElmSet,ElmCantOpenFileErr,
			  "Can't open %s: %s"),
		  name,strerror(err));

	return;
    }

    if (! GrabLock_file(name,F,Grab_the_file)) {
	fclose(F);
	return;
    }
    
    if (! read_uidls(name,F,uidl_list,NULL)) {
	DPRINT(Debug,14,(&Debug,"update_uidls: Reading failed\n"));

	Release_the_file(fileno(F),&mailbox_locking);
	fclose(F);

	return;
    }
        
    rewind(F);
#ifdef FTRUNCATE
    if (-1 == ftruncate(fileno(F),0)) {
	DPRINT(Debug,14,(&Debug,"update_uidls: ftruncate of %s failed\n",name));
    }
#endif
    fprintf(F,UIDLS_HEADER);
    print_uidls(F,*uidl_list);
    fprintf(F,"--END\n");   /* So that truncate of file is not necessary */
    fflush(F);
    Release_the_file(fileno(F),&mailbox_locking);
    fclose(F);

    if (*uidl_list) {
	size_t len = sort_list_len(*uidl_list);

	DPRINT(Debug,14,(&Debug,
			 "update_uidls: %zu entries\n",len));

	if (len > 0) {
	    size_t i = len -1;
	    size_t deleted = 0;
	    
	    while (1) {
		union sort_item res;

		res.uidl = NULL;
				
		
		/* Increments refcount */
		get_sort_list_item(*uidl_list,sort_list_get_normal,i,&res);
		
		if (res.uidl) {
		    if (UIDL_ENTRY_magic != res.uidl->magic)
			panic("MBX PANIC",__FILE__,__LINE__,
			      "update_uidls",
			      "Bad magic number (uidl_entry)",0);

		    DPRINT(Debug,14,(&Debug,
				     "update_uidls: entry %zu = \"%s\"",
				     i,res.uidl->uidl));
		    
		    if (res.uidl->delete) {
			DPRINT(Debug,14,(&Debug," -- deleting\n"));

			free_uidl_entry(&(res.uidl));

			/* Increments refcount */
			get_sort_list_item(*uidl_list,sort_list_get_remove,i,&res);

			if (res.uidl) {
			    if (UIDL_ENTRY_magic != res.uidl->magic)
				panic("MBX PANIC",__FILE__,__LINE__,
				      "update_uidls",
				      "Bad magic number (uidl_entry)",0);

			    DPRINT(Debug,14,(&Debug,
					     "update_uidls: entry %zu = \"%s\" deleted",
					     i,res.uidl->uidl));
			    deleted++;
			}
			
		    }
		    DPRINT(Debug,14,(&Debug,"\n"));

		    if (res.uidl) 
			free_uidl_entry(&(res.uidl));
		}
		    
		if (i > 0)
		    i--;
		else
		    break;		
	    }

	    if (deleted) {
		len = sort_list_len(*uidl_list);
		
		DPRINT(Debug,14,(&Debug,
				 "update_uidls: %zu entries deleted, %zu entries left\n",deleted,len));
		
	    }
	}
    }    
}

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


S_(ss_action_routine pop_read_stream)
static int pop_read_stream P_((struct streamsched              * ss,
			       union ss_action_routine_data      data,
			       const struct schedule_timelimit * now));
static int pop_read_stream (ss,data,now)
     struct streamsched              * ss;
     union ss_action_routine_data      data; 
     const struct schedule_timelimit * now;
{
    struct folder_info *folder = data.folder;
    int n, i;
    int close_it = 0;

    char * ERROR = NULL;
    
    if (!folder || 
	FOLDER_INFO_magic != folder->magic ||
	folder->folder_type != &pop_mbx ||
	PRIVATE_DATA_magic != folder->p->magic ||
	folder -> p->a.pop_mbx.C.stream != ss) {
	
	panic("MBX PANIC",__FILE__,__LINE__,"pop_read_stream",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug,
		     "pop_read_stream: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (POP_error == folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,13,(&Debug,
			 "pop_read_stream=0 (stream need closing)\n"));
	return 0;
    }
	    
    n = ReadFromStream(ss, &(folder -> p->a.pop_mbx.read_buffer),-1);

    if (n < 0) {      
	ERROR = RemoveStreamError(ss);
	DPRINT(Debug,13,(&Debug,
			 "pop_read_stream: read error %s\n",
			 ERROR ? ERROR : "(none)"));
    }
    
    DPRINT(Debug,50,(&Debug,
		     "Buffer content="));
    for (i = 0; i < folder -> p->a.pop_mbx.read_buffer.read_len; i++) {
	DPRINT(Debug,50,(&Debug,
			 " %d",
			 folder -> p->a.pop_mbx.read_buffer.read_buffer[i]));
    }
    DPRINT(Debug,50,(&Debug,
		     "\n (message?)="));
    for (i = 0; i < folder -> p->a.pop_mbx.read_buffer.read_len; i++) {
	DPRINT(Debug,50,(&Debug,
			 " %c",
			 isascii(folder -> p->a.pop_mbx.read_buffer.
				 read_buffer[i]) &&
			 isprint(folder -> p->a.pop_mbx.read_buffer.
				 read_buffer[i]) ? 
			 folder -> p->a.pop_mbx.read_buffer.read_buffer[i] :
			 '.'));
    }
    DPRINT(Debug,50,(&Debug,
		     "\n"));
    
    /* Parse response ... */
    
    if (POP_simple_response == folder->p->a.pop_mbx.pop_state ||
	POP_multiline_response == folder->p->a.pop_mbx.pop_state) {
	int linelen;
	
	DPRINT(Debug,13,(&Debug,
			 "pop_read_stream: Wanting response\n"));
	
	linelen = find_crlf(&(folder -> p->a.pop_mbx.read_buffer),1);
	if (!linelen)
	    goto check_read;
	
	folder -> p->a.pop_mbx.command_status =
	    safe_realloc(folder -> p->a.pop_mbx.command_status,
			 linelen);
	memcpy(folder -> p->a.pop_mbx.command_status,
	       folder -> p->a.pop_mbx.read_buffer.read_buffer,linelen);
	cut_line(&(folder -> p->a.pop_mbx.read_buffer),linelen);
	
	DPRINT(Debug,13,(&Debug,
			 "pop_read_stream: response=%s\n",
			 folder -> p->a.pop_mbx.command_status));
	
	if (folder -> p->a.pop_mbx.command_data) {
	    free(folder -> p->a.pop_mbx.command_data);
	    folder -> p->a.pop_mbx.command_data = NULL;
	    folder -> p->a.pop_mbx.data_len  = 0;
	}
	
	if (0 == memcmp("+OK",folder -> p->a.pop_mbx.command_status,3) &&
	    POP_multiline_response == folder->p->a.pop_mbx.pop_state)
	    folder->p->a.pop_mbx.pop_state = POP_multiline_data;
	else {
	    folder->p->a.pop_mbx.pop_state = POP_command_ready;
	    DPRINT(Debug,13,(&Debug,
			     "pop_read_stream: Command ready\n"));

	    /* Clear stalled message */
	    if (folder->p->a.pop_mbx.stalled &&
		folder->p->a.pop_mbx.C.host) {
		lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPresumed,
				      "POP connection to %s resumed."),
			      folder->p->a.pop_mbx.C.host);
		
	    }
	    folder->p->a.pop_mbx.stalled = 0;
	}
    }
    
    if (POP_multiline_data == folder->p->a.pop_mbx.pop_state) {
	int linelen;
	
	DPRINT(Debug,13,(&Debug,
			 "pop_read_stream: Wanting multiline data\n"));
	
	
	while (0 < (linelen = find_crlf(&(folder -> p->a.pop_mbx.read_buffer),
					0))) {
	    char * p = folder -> p->a.pop_mbx.read_buffer.read_buffer;
	    int plen = linelen;
	    if (p[0] == '.') {
		p++;
		plen--;
		if (plen <= 2) {   /* If CRLF only left then EOM */
		    cut_line(&(folder -> p->a.pop_mbx.read_buffer),
			     linelen);		   
		    folder->p->a.pop_mbx.pop_state = POP_command_ready;
		    DPRINT(Debug,13,(&Debug,
				     "pop_read_stream: Command ready (multiline data len = %d)\n",
				     folder -> p->a.pop_mbx.data_len));
		    break;
		}
	    }
	    folder -> p->a.pop_mbx.command_data = 
		safe_realloc(folder -> p->a.pop_mbx.command_data,
			     folder -> p->a.pop_mbx.data_len + plen);
	    memcpy(folder -> p->a.pop_mbx.command_data + 
		   folder -> p->a.pop_mbx.data_len,
		   p, plen);
	    folder -> p->a.pop_mbx.data_len += plen;
	    cut_line(&(folder -> p->a.pop_mbx.read_buffer),
		     linelen);		   
	}
    }
    
    if (folder -> p->a.pop_mbx.read_buffer.read_len) {
	switch(folder->p->a.pop_mbx.pop_state) {
	    int i;
	case POP_command_ready:	case POP_idle: case POP_simple_command: 
	case POP_multiline_command:
	    DPRINT(Debug,13,(&Debug,
			     "pop_read_stream: %d bytes data left! (bad state):\n",
			     folder -> p->a.pop_mbx.read_buffer.read_len));
	    DPRINT(Debug,13,(&Debug,
			     "pop_read_stream: extra data="));
	    for (i =0; i < folder -> p->a.pop_mbx.read_buffer.read_len; i++) {
		DPRINT(Debug,13,(&Debug,
				 " %d",
				 folder -> p->a.pop_mbx.read_buffer.
				 read_buffer[i]));
	    }
	    DPRINT(Debug,13,(&Debug,
			     "\npop_read_stream: (message?)="));
	    for (i =0; i < folder -> p->a.pop_mbx.read_buffer.read_len; i++) {
		DPRINT(Debug,13,(&Debug,
				 "%c",
				 isascii(folder -> p->a.pop_mbx.read_buffer.
					 read_buffer[i]) &&
				 isprint(folder -> p->a.pop_mbx.read_buffer.
					 read_buffer[i]) ?
				 folder -> p->a.pop_mbx.read_buffer.read_buffer[i] :
				 '.'));
	    }
	    DPRINT(Debug,13,(&Debug,
			     "\n"));

	    lib_error(CATGETS(elm_msg_cat, MeSet,MeUnexpextedResponse,
			      "Unexpected response from POP server"));
	    close_it = 1;
	    break;
	default:
	    break;
	}
    }

 check_read:
    if (n == 0) {
	lib_error(CATGETS(elm_msg_cat, MeSet,MeConnectionClosed,
			  "Connection closed to POP server"));
	close_it = 1;
    }
    if (n < 0 && ERROR) {
	lib_error(CATGETS(elm_msg_cat, MeSet,MeErrorReading,
			  "Error reading from POP server: %s"),
		  ERROR);
	close_it = 1;
    }

    if (ERROR) {
	free(ERROR);
	ERROR = NULL;
    }

    if (close_it) {
	folder->p->a.pop_mbx.pop_state = POP_error;
	DPRINT(Debug,13,(&Debug,
			 "pop_read_stream=0 (stream need closing)\n"));
	return 0;
    }
    DPRINT(Debug,13,(&Debug,
		     "pop_read_stream=1\n"));
    return 1;
}

static int pop_push_command P_((struct folder_info *folder,
				char *command, int simple));
static int pop_command_ok P_((struct folder_info *folder));
static void pop_clear_command P_((struct folder_info *folder));

S_(ss_action_routine pop_idle_timer)
static int pop_idle_timer P_((struct streamsched              * ss,
			      union ss_action_routine_data      data,
			      const struct schedule_timelimit * now));
static int pop_idle_timer (ss,data,now)
     struct streamsched              * ss;
     union ss_action_routine_data      data;
     const struct schedule_timelimit * now;
{
    struct folder_info *folder = data.folder;
    int status;
    
    if (!folder || 
	FOLDER_INFO_magic != folder->magic ||
	folder->folder_type != &pop_mbx ||
	PRIVATE_DATA_magic != folder->p->magic ||
	folder -> p->a.pop_mbx.C.stream != ss) {

	panic("MBX PANIC",__FILE__,__LINE__,"pop_idle_timer",
	      "Bad data argument !!!",0);
    }

    DPRINT(Debug,13,(&Debug,
		     "pop_idle_timer: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (POP_error == folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,13,(&Debug,
			 "pop_idle_timer=0 (stream need closing)\n"));
	return 0;
    }
    
    if (POP_command_ready == folder->p->a.pop_mbx.pop_state &&
	0 == strncmp(folder->p->a.pop_mbx.command,"NOOP",
		     sizeof folder->p->a.pop_mbx.command)) {
	DPRINT(Debug,13,(&Debug,
			 "pop_idle_timer: Previous idle command was finished.\n"));

	
	
	pop_clear_command(folder);
	
    } else if (POP_idle != folder->p->a.pop_mbx.pop_state) {

	if (0 == strncmp(folder->p->a.pop_mbx.command,"NOOP",
			 sizeof folder->p->a.pop_mbx.command)) {
	    struct sec_ms interval;
	    int r;

	    if (0 != (r = schedule_time_passed(& (folder->p->a.pop_mbx.NOOP_start),
					       now,
					       &interval))) {
           
		/* 'now' is time when wait ended, probably not current time */
		
		DPRINT(Debug,13,(&Debug, 
				 "POP_idle_timer: previous NOOP command executed for %d sec %d msec.\n",
				 interval.timeout_sec, interval.timeout_ms));
		
		if (folder->p->a.pop_mbx.C.host) {
		    folder->p->a.pop_mbx.stalled = 1;

		    if (interval.timeout_sec >= 0 &&
			interval.timeout_sec < 10 &&
			interval.timeout_ms  <= 999 &&
			0 != (r & SCHEDULE_TP_HAVE_MSEC)) {
			
			lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPstalledMS,
					      "POP connection to %s stalled for %d.%03d seconds."),
				      folder->p->a.pop_mbx.C.host,
				      interval.timeout_sec,
				      interval.timeout_ms);
		    } else if (interval.timeout_sec >= 0) {
			lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPstalledSec,
					      "POP connection to %s stalled for %d seconds."),
				      folder->p->a.pop_mbx.C.host,
				      interval.timeout_sec);                    
		    } 		    
		}
	    }
	}
	
	DPRINT(Debug,13,(&Debug,
			 "pop_idle_timer=1 (stream not idle: command %.*s)\n",
			 sizeof folder->p->a.pop_mbx.command,
			 folder->p->a.pop_mbx.command));
	return 1;
    }

    status = pop_push_command(folder,"NOOP",1);
    DPRINT(Debug,13,(&Debug,
		     "pop_idle_timer=%d\n",
		     status));

    folder->p->a.pop_mbx.NOOP_start = *now;
    
    return status;
}

S_(ss_action_routine pop_write_stream)
static int pop_write_stream P_((struct streamsched              * ss,
				union ss_action_routine_data      data,
				const struct schedule_timelimit * now));
static int pop_write_stream (ss,data,now)
     struct streamsched              * ss;
     union ss_action_routine_data      data;
     const struct schedule_timelimit * now;
{
    struct folder_info *folder = data.folder;
    int n;
    char * ERROR = NULL;

    if (!folder || 
	FOLDER_INFO_magic != folder->magic ||
	folder->folder_type != &pop_mbx ||
	PRIVATE_DATA_magic != folder->p->magic ||
	folder -> p->a.pop_mbx.C.stream != ss) {

	panic("MBX PANIC",__FILE__,__LINE__,"pop_write_stream",
	      "Bad data argument !!!",0);
    }
    
    DPRINT(Debug,13,(&Debug,
		     "pop_write_stream: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (!folder -> p->a.pop_mbx.write_buffer.write_len) {
	DPRINT(Debug,13,(&Debug,
			 "pop_write_stream: NO DATA TO WRITE!\n"));
    }

    if (POP_error == folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,13,(&Debug,
			 "pop_write_stream=0 (stream need closing)\n"));
	return 0;
    }
    
    if (POP_simple_command != folder->p->a.pop_mbx.pop_state &&
	POP_multiline_command != folder->p->a.pop_mbx.pop_state) {

	panic("MBX PANIC",__FILE__,__LINE__,"pop_write_stream",
	      "Bad state !!!",0);
    }

    n = WriteToStream(ss, & (folder -> p->a.pop_mbx.write_buffer));

    if (n > 0) {
	int i;

	DPRINT(Debug,70,(&Debug,
		    "Write content="));
	for (i = 0; i < n; i++) {
	    DPRINT(Debug,70,(&Debug," %d",
			     folder->p->a.pop_mbx.write_buffer.write_buffer[i]));
	}
	DPRINT(Debug,70,(&Debug,"\n    (text?)="));
	for (i = 0; i < n; i++) {
	    DPRINT(Debug,70,(&Debug," %c",
			     isascii(folder->p->a.pop_mbx.write_buffer.write_buffer[i]) 
			     &&
			     isprint(folder->p->a.pop_mbx.write_buffer.write_buffer[i]) 
			     ? 
			     folder->p->a.pop_mbx.write_buffer.write_buffer[i] :
			     '.'));
	}
	DPRINT(Debug,70,(&Debug,"\n"));

	cut_Write_Buffer(& (folder -> p->a.pop_mbx.write_buffer),n);
	
    } else if (n < 0) {
	ERROR = RemoveStreamError(ss);

	DPRINT(Debug,13,(&Debug,
			 "pop_write_stream: write error %s\n",
			 ERROR ? ERROR : "(none)"));
    }
    
    if (!folder -> p->a.pop_mbx.write_buffer.write_len) {
	if (POP_multiline_command == folder->p->a.pop_mbx.pop_state)
	    folder->p->a.pop_mbx.pop_state = POP_multiline_response;
	else
	    folder->p->a.pop_mbx.pop_state = POP_simple_response;

	if (ERROR) {
	    free(ERROR);
	    ERROR = NULL;
	}
	DPRINT(Debug,13,(&Debug,
			 "pop_write_stream=0 -- data written\n"));

	return 0;
    }

    if (ERROR) {
	lib_error(CATGETS(elm_msg_cat, MeSet,MeErrorWriting,
			  "Error writing to POP server: %s"),
		  ERROR);
	folder->p->a.pop_mbx.pop_state = POP_error;
	DPRINT(Debug,13,(&Debug,
			 "pop_write_stream=0 (stream need closing)\n"));

	free(ERROR);
	ERROR = NULL;
        return 0;	
    }

    DPRINT(Debug,13,(&Debug,
		     "pop_write_stream=1\n"));
    return 1;
}

S_(ss_badpid_routine pop_badpid_stream)
static enum badpid_status pop_badpid_stream
    P_((struct streamsched *ss,
	union  ss_action_routine_data data,
	int badpid));
static enum badpid_status pop_badpid_stream(ss,data,badpid)
     struct streamsched *ss;
     union  ss_action_routine_data data;
     int badpid;
{
    struct folder_info *folder = data.folder;

     if (!folder || 
	FOLDER_INFO_magic != folder->magic ||
	folder->folder_type != &pop_mbx ||
	PRIVATE_DATA_magic != folder->p->magic ||
	folder -> p->a.pop_mbx.C.stream != ss) {

	panic("MBX PANIC",__FILE__,__LINE__,"pop_badpid_stream",
	      "Bad data argument !!!",0);
    }

     DPRINT(Debug,13,(&Debug,
		      "pop_badpid_stream: folder=%p (%s)\n",
		      folder,folder->cur_folder_sys));

     folder->p->a.pop_mbx.pop_state    = POP_error;
     folder->p->a.pop_mbx.seen_badpid  = badpid;
     
     return badpid_remove       /* remove action */;
}

static void pop_push_cmd_fail P_((struct folder_info *folder));
static void pop_push_cmd_fail(folder)
     struct folder_info *folder;
{
    if (folder -> p->a.pop_mbx.C.stream ) {
	DPRINT(Debug,12,(&Debug,
			 "pop_push_cmd_fail: Closing stream"));
	if (folder -> p->a.pop_mbx.seen_badpid) {
	    DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
			     folder -> p->a.pop_mbx.seen_badpid));
	}	    
	DPRINT(Debug,12,(&Debug,"\n"));
	FreeStreamStack0(&(folder -> p->a.pop_mbx.C.stream),
			 folder -> p->a.pop_mbx.seen_badpid,
			 1 /* Force clearing of stack even when this is not last reference */
			 );
    }	
}

static int  pop_push_cmd_head P_((struct folder_info *folder));
static int  pop_push_cmd_head(folder)
     struct folder_info *folder;
{
    int status = 0;

    if (POP_not_logged == folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,12,(&Debug,
			 "pop_push_cmd_head: pop_state == POP_not_logged\n"));
	goto failure;
    }

    if (POP_error == folder->p->a.pop_mbx.pop_state) {

	DPRINT(Debug,12,(&Debug,
			 "pop_push_cmd_head: pop_state == POP_error\n"));

    failure:
	pop_push_cmd_fail(folder);
	status = 0;
	goto clean;
    }
    status = 1;
    
 clean:
    DPRINT(Debug,13,(&Debug,
		     "pop_push_cmd_head=%d\n",
		     status));
    
    return status;
}
				 
static int pop_push_cmd_tail P_((struct folder_info *folder,
				 char *command,
				 int simple));
static int pop_push_cmd_tail(folder,command,simple)
     struct folder_info *folder;
     char *command;
     int simple;
{
    int len = strlen(command);
    int status = 0;
    union ss_action_routine_data data;
    
    if (0 != folder->p->a.pop_mbx.write_buffer.write_len) {
	DPRINT(Debug,12,(&Debug,
			 "pop_push_cmd_tail: write buffer not empty (%d bytes left)\n",
			 folder->p->a.pop_mbx.write_buffer.write_len));
	status = 0;
	goto clean;
    }

    if (!folder -> p->a.pop_mbx.C.stream) {
	DPRINT(Debug,12,(&Debug,
			 "pop_push_cmd_tail: stream not open\n"));
	status = 0;
	goto clean;
    }

    if (simple)
	folder->p->a.pop_mbx.pop_state = POP_simple_command;
    else
	folder->p->a.pop_mbx.pop_state = POP_multiline_command;

    strncpy(folder->p->a.pop_mbx.command,command,
	    sizeof folder->p->a.pop_mbx.command);

    DPRINT(Debug,15,(&Debug,
		     "pop_push_cmd_tail: Command %.4s...\n",command));
    
    DPRINT(Debug,50,(&Debug,
		     "pop_push_cmd_tail: command=%s\n",command));

    
    folder->p->a.pop_mbx.write_buffer.write_buffer = 
	safe_realloc(folder->p->a.pop_mbx.write_buffer.write_buffer,len+2);
    memcpy(folder->p->a.pop_mbx.write_buffer.write_buffer,command,len);
    memcpy(folder->p->a.pop_mbx.write_buffer.write_buffer+len,"\r\n",2);
    folder->p->a.pop_mbx.write_buffer.write_len = len + 2;

    data.folder = folder;
    
    ConfigStream2(folder -> p->a.pop_mbx.C.stream,
		  pop_read_stream,pop_write_stream,
		  pop_idle_timer,
		  ss_nofree_action_rt_data,
		  ss_noinc_action_rt_data_refcount,
		  pop_badpid_stream,
		  pop_idle_alive_interval > MIN_POP_TIMEOUT ?
		  pop_idle_alive_interval : MIN_POP_TIMEOUT ,
		  data);
    status = 1;
    
 clean:
    DPRINT(Debug,13,(&Debug,
		     "pop_push_cmd_tail=%d\n",
		     status));    
    return status;
}

static int pop_push_command_c P_((struct folder_info *folder,
				  char *command,
				  int simple,
				  struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							       check of remeote mailbox
							    */));
static int pop_push_command_c(folder,command,simple,cd)
     struct folder_info *folder;
     char *command;
     int simple;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  check of remeote mailbox
			       */;
{
    int status = 0;

    DPRINT(Debug,12,(&Debug,
		"pop_push_command_c: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "pop_push_command_c",
	      "Bad magic number (private_data)",0);

    if (!pop_push_cmd_head(folder)) {
	status = 0;
	goto clean;
    }

    if (POP_idle != folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,7,(&Debug,
			 "pop_push_command_c: wrong state -- perhaps idle timer (command %.*s) -- waiting...\n",
			 sizeof folder->p->a.pop_mbx.command,
			 folder->p->a.pop_mbx.command));
	
	while (POP_command_ready != folder->p->a.pop_mbx.pop_state &&
	       POP_error       != folder->p->a.pop_mbx.pop_state &&
	       POP_not_logged != folder->p->a.pop_mbx.pop_state &&
	       NULL         != folder -> p->a.pop_mbx.C.stream) {

	    WaitStreamFor_c(folder -> p->a.pop_mbx.C.stream,SS_read_act,
			    cd);

	    if (cd && is_canceled(cd)) {
		DPRINT(Debug,12,(&Debug,
				 "pop_push_command_c: Wait canceled\n"));
		
		status = 0;
		goto clean;
	    }
	}

	DPRINT(Debug,7,(&Debug,
			"pop_push_command_c: ... wait ended\n"));
	if (POP_error == folder->p->a.pop_mbx.pop_state) {
	    DPRINT(Debug,12,(&Debug,
			     "pop_push_command_c: pop_state == POP_error\n"));
	    
	    goto failure;
	}

	if (POP_not_logged == folder->p->a.pop_mbx.pop_state) {

	    DPRINT(Debug,12,(&Debug,
			     "pop_push_command_c: pop_state == POP_not_logged\n"));

	failure:
	    pop_push_cmd_fail(folder);
	    status = 0;
	    goto clean;
	}
     }
	
    if (! pop_push_cmd_tail(folder,command,simple)) {
	status = 0;
	goto clean;
    }
    
    status = 1;
    
 clean:
    DPRINT(Debug,12,(&Debug,
		     "pop_push_command_c=%d\n",
		     status));
    return status;
}

static int pop_push_command(folder,command,simple)
     struct folder_info *folder;
     char *command;
     int simple;
{

    int status = 0;

    
    DPRINT(Debug,12,(&Debug,
		"pop_push_command: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "pop_push_command",
	      "Bad magic number (private_data)",0);

    if (!pop_push_cmd_head(folder)) {
	status = 0;
	goto clean;
    }
        
    if (POP_idle != folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,7,(&Debug,
			 "pop_push_command: wrong state -- perhaps idle timer (command %.*s) -- waiting...\n",
			 sizeof folder->p->a.pop_mbx.command,
			 folder->p->a.pop_mbx.command));
	
	while (POP_command_ready != folder->p->a.pop_mbx.pop_state &&
	       POP_error       != folder->p->a.pop_mbx.pop_state &&
	       POP_not_logged != folder->p->a.pop_mbx.pop_state &&
	       NULL         != folder -> p->a.pop_mbx.C.stream) {
	    WaitStreamFor(folder -> p->a.pop_mbx.C.stream,SS_read_act);
	    /* TODO:    Should user to have some way to interrupt progress ? */
	}
	DPRINT(Debug,7,(&Debug,
			 "pop_push_command: ... wait ended\n"));
	if (POP_error == folder->p->a.pop_mbx.pop_state) {
	    DPRINT(Debug,12,(&Debug,
			     "pop_push_command: pop_state == POP_error\n"));

	    goto failure;
	}
	if (POP_not_logged == folder->p->a.pop_mbx.pop_state) {

	    DPRINT(Debug,12,(&Debug,
			     "pop_push_command: pop_state == POP_not_logged\n"));

	failure:
	    pop_push_cmd_fail(folder);
	    status = 0;
	    goto clean;
	}	
    }

    if (! pop_push_cmd_tail(folder,command,simple)) {
	status = 0;
	goto clean;
    }
    
    status = 1;

 clean:
    DPRINT(Debug,12,(&Debug,
		     "pop_push_command=%d\n",
		     status));
    return status;
}


static int pop_cmd_ok_tail P_((struct folder_info *folder));
static int pop_cmd_ok_tail(folder)
     struct folder_info *folder;
{
    int status = 0;

    if (POP_error == folder->p->a.pop_mbx.pop_state) {
	if (folder -> p->a.pop_mbx.C.stream ) {
	    DPRINT(Debug,12,(&Debug,
			     "pop_cmd_ok_tail: Closing stream"));
	    if (folder -> p->a.pop_mbx.seen_badpid) {
		 DPRINT(Debug,12,(&Debug,", not our stream, owner pid %d",
				 folder -> p->a.pop_mbx.seen_badpid));
	     }	    
	     DPRINT(Debug,12,(&Debug,"\n"));
	     FreeStreamStack0(& (folder -> p->a.pop_mbx.C.stream),
			      folder -> p->a.pop_mbx.seen_badpid,
			      1 /* Force clearing of stack even when this is not last reference */);
	}	
	status = 0;
	goto clean;
    }

    if (0 == memcmp("+OK",folder -> p->a.pop_mbx.command_status,3))
	status = 1;
    else if (0 == memcmp("-ERR",folder -> p->a.pop_mbx.command_status,4)) {
	lib_error(CATGETS(elm_msg_cat, MeSet,MePopError,
			  "POP server: %s"),
		  folder -> p->a.pop_mbx.command_status+4);
	status = 0;
    } else {
	DPRINT(Debug,12,(&Debug,
			 "pop_cmd_ok_tail: UNEXPECTED RESPONSE: %s\n",
			 folder -> p->a.pop_mbx.command_status));
	folder->p->a.pop_mbx.pop_state = POP_error;	
	status = 0;
    }

 clean:
     DPRINT(Debug,13,(&Debug,
		      "pop_cmd_ok_tail=%d\n",
		      status));
    return status;
}

static int pop_command_ok_c P_((struct folder_info *folder,
				struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							     check of remote mailbox
							  */
				));

static int pop_command_ok_c(folder,cd)
     struct folder_info *folder;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  check of remeote mailbox
			       */;
{
    int status = 0;

    DPRINT(Debug,12,(&Debug,
		     "pop_command_ok_c: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "pop_command_ok_c",
	      "Bad magic number (private_data)",0);

    if (POP_idle == folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,12,(&Debug,
			 "pop_command_ok_c: WRONG STATE\n"));
	status = 0;
	goto clean;
    }

    while (POP_command_ready != folder->p->a.pop_mbx.pop_state &&
	   POP_error  != folder->p->a.pop_mbx.pop_state &&
	   NULL         != folder -> p->a.pop_mbx.C.stream) {
	WaitStreamFor_c(folder -> p->a.pop_mbx.C.stream,SS_read_act,
			cd);
	if (cd && is_canceled(cd)) {
	    DPRINT(Debug,12,(&Debug,
			     "pop_command_ok_c: Wait canceled\n"));
	    
	    status = 0;
	    goto clean;
	}
	
    }

    status =  pop_cmd_ok_tail(folder);

 clean:
    DPRINT(Debug,12,(&Debug,
		     "pop_command_ok_c=%d\n",
		     status));    
    return status;

}

static int pop_command_ok(folder)
     struct folder_info *folder;
{
    int status = 0;

    DPRINT(Debug,12,(&Debug,
		     "pop_command_ok: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "pop_command_ok",
	      "Bad magic number (private_data)",0);

    if (POP_idle == folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,12,(&Debug,
			 "pop_command_ok: WRONG STATE\n"));
	status = 0;
	goto clean;
    }

    while (POP_command_ready != folder->p->a.pop_mbx.pop_state &&
	   POP_error  != folder->p->a.pop_mbx.pop_state &&
	   NULL         != folder -> p->a.pop_mbx.C.stream) {
	WaitStreamFor(folder -> p->a.pop_mbx.C.stream,SS_read_act);
	/* TODO:    Should user to have some way to interrupt progress ? */
    }

    status =  pop_cmd_ok_tail(folder);

 clean:
    DPRINT(Debug,12,(&Debug,
		     "pop_command_ok=%d\n",
		     status));
    return status;
}

static void pop_clear_command(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,12,(&Debug,
		     "pop_clear_command: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "pop_clear_command",
	      "Bad magic number (private_data)",0);

    if (POP_idle == folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,12,(&Debug,
			 "pop_clear_command: Already clear!\n"));
    } else if (POP_command_ready != folder->p->a.pop_mbx.pop_state) {
	DPRINT(Debug,12,(&Debug,
			 "pop_clear_command: WRONG STATE!\n"));
	folder->p->a.pop_mbx.pop_state = POP_error;
    } else
	folder->p->a.pop_mbx.pop_state = POP_idle;

    strncpy(folder->p->a.pop_mbx.command,"",
	    sizeof folder->p->a.pop_mbx.command);

    if (folder -> p->a.pop_mbx.command_status)
	free(folder -> p->a.pop_mbx.command_status);
    folder -> p->a.pop_mbx.command_status = NULL;

    if (folder -> p->a.pop_mbx.command_data)
	free(folder -> p->a.pop_mbx.command_data);
    folder -> p->a.pop_mbx.command_data = NULL;
    folder -> p->a.pop_mbx.data_len = 0;
}
			      
static int mbx_close_pop P_((struct folder_info *folder,
			      enum close_mode mode,
			     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							  on remote mailbox
						       */));
static int mbx_close_pop(folder,mode,cd) 
     struct folder_info *folder;
     enum close_mode mode;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  on remote mailbox
			       */;
{
    int ret = 0;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_close_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));


    
    if(pop_push_command_c(folder,"QUIT",1,cd)) {
	
	if (pop_command_ok_c(folder,cd)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet,MePopError,
			      "POP server: %s"),
		      folder -> p->a.pop_mbx.command_status+4);

	    ret = 1;
	}
    }
    pop_clear_command(folder);

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "mbx_close_pop",
	      "Bad magic number (private_data)",0);

    if (folder -> p->a.pop_mbx.C.stream ) {
	DPRINT(Debug,12,(&Debug,
			 "mbx_close_pop: Closing stream\n"));
	
	FreeStreamStack2(& (folder -> p->a.pop_mbx.C.stream),
			 1 /* Force clearing of stack even when this is not last reference */);
    }	
    
    switch (mode) {
    case CLOSE_NORMAL:
	
	/* falltru */
    case CLOSE_LEAVE_LOCKED:
	if (!folder -> p->fh_temp) {
	    DPRINT(Debug,1,(&Debug,
		       "mbx_close_pop: tempfile (%s) not open -- will not unlink\n",
		       folder -> cur_tempfolder));
	} else if (unlink(folder -> cur_tempfolder) != 0) {
	    if (errno != ENOENT) {
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantUnlinkTemp,
				  "Sorry, can't unlink the temp file %s [%s]!\n\r"),
			  folder -> cur_tempfolder, strerror(errno));
	    }
	} else {
	    DPRINT(Debug,1,(&Debug, 
			    "close_folder: Unlinking tempfile (%s).\n",
			    folder -> cur_tempfolder));
	}      
    }

    DPRINT(Debug,11,(&Debug,
		     "mbx_close_pop=%d\n",
		     ret));
    
    return ret;
}

static int  mbx_lock_pop  P_((enum lock_direction direction,
			      struct folder_info *folder));
static int  mbx_lock_pop(direction,folder)
     enum lock_direction direction;
     struct folder_info *folder;
{
    int status=1;

    DPRINT(Debug,11,(&Debug,
		     "mbx_lock_pop=%d (dummy): folder=%p (%s)\n",
		     status,folder,folder->cur_folder_sys));
  
    return status;
}

static int mbx_unlock_pop P_((int interrupt, struct folder_info *folder));
static int mbx_unlock_pop(interrupt,folder) 
     int interrupt;
     struct folder_info *folder;
{
    int status = 1;

    DPRINT(Debug,11,(&Debug,
		     "mbx_unlock_pop=%d (dummy): folder=%p (%s)\n",
		     status,folder,folder->cur_folder_sys));

    return status;
}

static void mbx_init_pop P_((struct folder_info *folder));
static void mbx_init_pop(folder) 
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_init_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    folder -> cur_tempfolder[0] = '\0';
    /* remote_mbox generates tempfolder name */
    
    folder->p = malloc_mbx_private_data(NULL);

    zero_remote_account(&(folder -> p->a.pop_mbx.C));
    folder->p->a.pop_mbx.this_server = NULL;
    folder->p->a.pop_mbx.passhm      = NULL;
	
    zero_Read_Buffer(&(folder->p->a.pop_mbx.read_buffer));

    folder->p->a.pop_mbx.pop_state    = POP_not_logged;
    folder->p->a.pop_mbx.pop_flags    = 0;
    time(& (folder->p->a.pop_mbx.default_received_time));
    
    zero_Write_Buffer (& (folder->p->a.pop_mbx.write_buffer));

    folder->p->a.pop_mbx.command_status = NULL;
    folder->p->a.pop_mbx.command_data   = NULL;
    folder->p->a.pop_mbx.data_len       = 0;
    folder->p->a.pop_mbx.stat_size      = 0;
    folder->p->a.pop_mbx.stat_count      = 0;
    strncpy(folder->p->a.pop_mbx.command,"",
	    sizeof folder->p->a.pop_mbx.command);
    folder->p->a.pop_mbx.NOOP_start      = NO_schedule_timelimit;
    folder->p->a.pop_mbx.uidl_list       = NULL;
    folder->p->a.pop_mbx.seen_badpid     = 0;
    folder->p->a.pop_mbx.stalled         = 0;
}

static void pop_clear_buffers P_((struct folder_info *folder));
static void pop_clear_buffers(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,12,(&Debug,
		     "pop_clear_buffers: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));    
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"pop_clear_buffers",
	      "Bad magic number (private_data)",0);

    free_Read_Buffer(&(folder -> p->a.pop_mbx.read_buffer));

    free_Write_Buffer(&(folder -> p->a.pop_mbx.write_buffer));

    if (folder->p->a.pop_mbx.command_status)
	free(folder->p->a.pop_mbx.command_status);
    folder->p->a.pop_mbx.command_status = NULL;
    if (folder->p->a.pop_mbx.command_data)
	free(folder->p->a.pop_mbx.command_data);
    folder->p->a.pop_mbx.command_data = NULL;
    folder->p->a.pop_mbx.data_len       = 0;
    strncpy(folder->p->a.pop_mbx.command,"",
	    sizeof folder->p->a.pop_mbx.command);

    folder->p->a.pop_mbx.NOOP_start      = NO_schedule_timelimit;
    folder->p->a.pop_mbx.stalled         = 0;
}

S_(mbx_free_folder  mbx_free_pop)
static int mbx_free_pop P_((struct folder_info *folder,
			    struct cancel_data  * cd));
static int mbx_free_pop(folder,cd)
     struct folder_info *folder;
     struct cancel_data  * cd;
{
    int ret = 1;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_free_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));    

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_free_pop",
	      "Bad magic number (private_data)",0);

    clear_remote_account(&(folder -> p->a.pop_mbx.C));
    pop_clear_buffers(folder);

    if (folder -> p->a.pop_mbx.uidl_list) {
	free_sort_list(& (folder -> p->a.pop_mbx.uidl_list));
    }

    folder -> p->a.pop_mbx.pop_flags    = 0;

    if (folder->p->a.pop_mbx.this_server) {
	remove_remote_server_from_folder(folder,
					 folder->p->a.pop_mbx.this_server);
	folder->p->a.pop_mbx.this_server = NULL;
    }

    if (folder->p->a.pop_mbx.passhm)
	free_browser_passhm(& (folder->p->a.pop_mbx.passhm));    

    free_mbx_private_data(& (folder -> p));

    DPRINT(Debug,11,(&Debug,
		     "mbx_free_pop=%d\n",
		     ret));
    
    return ret;
}

#ifdef USE_DLOPEN
static struct pop_callbacks POP_FUNCTIONS = {
    pop_push_command,
    pop_command_ok,
    pop_clear_command 
};
#endif

S_(ra_verify_peer_cb_f pop_verify_peer_cb)
static int pop_verify_peer_cb P_((struct remote_account *ra,
				  union ra_verify_peer_cb_data *data,
				  const struct string * Server,
				  const struct string * server));
static int pop_verify_peer_cb(ra,data,Server,server)
   struct remote_account *ra;
   union ra_verify_peer_cb_data *data;
   const struct string * Server;
   const struct string * server;
{
    /* This is after CAPA command, so callback is not needed */

    return 1;
}

static int pop_open_connection P_((struct folder_info *folder));
static int pop_open_connection(folder)
     struct folder_info *folder;
{
    int status = 0;
    int sta1;
    char *data, *data1;
    int verify_peer_done = 0;
    union ra_verify_peer_cb_data cbdata;
    struct string * POP_server = NULL;
    union ss_action_routine_data action_data;

    /* give_dt_estr_as_str adds / to end */
    const char * str = give_dt_estr_as_str(&folders_e,"maildir",
					   NULL,NULL);

    
    DPRINT(Debug,12,(&Debug,
		     "pop_open_connection: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
	
    cbdata.folder = folder;  /* Not actually used */
    POP_server = format_string(CATGETS(elm_msg_cat, MeSet,MePOPserver,
				       "POP server"));

    /* Clear buffer from old data ... */
    pop_clear_buffers(folder);

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"pop_open_connection",
	      "Bad magic number (private_data)",0);

    if (NULL == folder->p->a.pop_mbx.C.stream) {

	
	enum itls_status have_initial_tls = itls_none;

	struct enum_service_type POP_ports_2 = NULL_enum_service_type;

	init_enum_service_type(&POP_ports_2,init_tls_default,STFLAG_is_pop,STFLAG_mbox);
	
	if (folder->p->a.pop_mbx.passhm) {

	    int r;

	    DPRINT(Debug,12,(&Debug,
			     "pop_open_connection: using passhm for open\n"));

	    r = browser_passhm_open_ra2(folder->p->a.pop_mbx.passhm,
					   &(folder->p->a.pop_mbx.C),
					   STFLAG_is_pop,					   
					   &POP_ports_2);

	    if (-1 == r) {
		DPRINT(Debug,12,(&Debug,
				 "pop_open_connection: doing regular open as fallback\n"));
		
		goto fallback_connect;
	    }
	    if (!r) {
		status = 0;
		goto clean;
	    }

	    /* canonify folder name .... */
	    free(folder->cur_folder_sys);
	    folder->cur_folder_sys = elm_message(FRM("%s@%s"),
						 folder->p->a.pop_mbx.C.username,
						 folder->p->a.pop_mbx.C.host);
	    free_string (&(folder->cur_folder_disp));
	    folder->cur_folder_disp = format_string(FRM("%s@%s"),
						    folder->p->
						    a.pop_mbx.C.username,
						    folder->p->a.pop_mbx.C.host);

	} else {

	    struct service_entry      * se;
	    struct service_entry      * result_se;
	    int ok;
	    int got_port;
	    const struct service_type * got_type;

	    struct cancel_data        * main_cancel /* Used if dns lookup was cancelable */;

	    	    
	fallback_connect:
	    
	    ok          = 0;
	    se          = NULL;
	    result_se   = NULL;
	    got_port    = PORT_end;
	    got_type    = NULL;
	    main_cancel = NULL;
	    
	    DPRINT(Debug,12,(&Debug,
			     "pop_open_connection: doing regular open\n"));

	    
	    se = 
		give_service_entry_can(folder->p->a.pop_mbx.C.host,
				       STFLAG_is_pop,itls_ignore,
				       NULL,&main_cancel);

	    if (!se) {
		if (main_cancel && is_canceled(main_cancel)) {
		    DPRINT(Debug,12,(&Debug,
				     "pop_open_connection: DNS lookup canceled\n"));
		} else  {
		    DPRINT(Debug,12,(&Debug,
				     "pop_open_connection: %s not found\n",
				     folder->p->a.pop_mbx.C.host));
		}
		
		goto clean_connect;
	    }
	    
	    /* canonify folder name .... */
	    free(folder->cur_folder_sys);
	    folder->cur_folder_sys = elm_message(FRM("%s@%s"),
						 folder->p->a.pop_mbx.C.username,
						 se->official_name);
	    free_string (&(folder->cur_folder_disp));
	    folder->cur_folder_disp = format_string(FRM("%s@%s"),
						    folder->p->
						    a.pop_mbx.C.username,
						    se->official_name);
	    
	    free(folder->p->a.pop_mbx.C.host);
	    folder->p->a.pop_mbx.C.host = safe_strdup(se->official_name);
	    
	    if (!connect_remote_account_2(&(folder->p->a.pop_mbx.C),
					  se,main_cancel,
					  
					  &result_se,
					  &got_type,
					  &got_port,

					  &POP_ports_2
					  )) {
	    clean_connect:
		status = 0;
		ok = 0;
	    } else {
		DPRINT(Debug,15,(&Debug,
				 "pop_open_connection: %s: connected, got port %d",
				 folder->p->a.pop_mbx.C.host,got_port));
		if (got_type) {
		    DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
				     got_type,
				     service_type_name(got_type),
				     service_type_defport(got_type)));
		}
		DPRINT(Debug,15,(&Debug,"\n"));
		    
		ok = 1;
	    }

	    /* 'se' can be NULL */
	    free_service_entry(&se);
	    free_service_entry(&result_se);
	    
	    if (main_cancel)
		free_cancel(& (main_cancel));
	    
	    if (!ok) {
		DPRINT(Debug,12,(&Debug,
				 "pop_open_connection: regular open failed or canceled\n"));
		goto clean;
	    }

	}


	switch((have_initial_tls = 
		remote_account_init_tls(&(folder->p->a.pop_mbx.C)))) {

	case itls_unsupported:
	case itls_failed:
	    status = 0;
	    goto clean;

	case itls_none:
	    break;

	case itls_done:
	    DPRINT(Debug,8,(&Debug,
			    "pop_open_connection: Initial TLS done (no STARTTLS)\n"));
	}

	
    }


    if (str && ! (folder -> p->a.pop_mbx.uidl_list)) {	
	char * fname = elm_message(FRM("%s.%s"),str,folder->cur_folder_sys);

	DPRINT(Debug,8,(&Debug,"pop_open_connection: Reading old UIDLS from %s\n",
			fname));
	
	read_old_uidls(fname,&(folder -> p->a.pop_mbx.uidl_list));
	free(fname);
    }

    /* We want server's initial response to connection */
    folder -> p->a.pop_mbx.pop_state = POP_simple_response;
    action_data.folder = folder;
    
    ConfigStream2(folder -> p->a.pop_mbx.C.stream,
		  pop_read_stream,ss_noaction_routine,pop_idle_timer,
		  ss_nofree_action_rt_data,
		  ss_noinc_action_rt_data_refcount,
		  pop_badpid_stream,
		  pop_idle_alive_interval > MIN_POP_TIMEOUT ?
		  pop_idle_alive_interval : MIN_POP_TIMEOUT,
		  action_data);
    if (!pop_command_ok(folder)) {
	/* Server was given non-OK response to connection ... */	    
	status = 0;
	goto clean;	    
    }
    if (POP_show_greeting)
	lib_error(CATGETS(elm_msg_cat, MeSet,MePopError,
                          "POP server: %s"),
                  folder -> p->a.pop_mbx.command_status+3);
    pop_clear_command(folder);
    


    
#ifdef USE_DLOPEN    
 retry_prelogin_capa:
#endif

    if (!pop_push_command(folder,"CAPA",0)) {
	status = 0;
	goto clean;
    }

    /* If command unimplemented, that is OK ... */
    if (pop_command_ok(folder)) {
#ifdef USE_DLOPEN
	struct POP_capa_libs * pop_capa_libs     = NULL;
	int                    pop_capa_libcount = 0;

	enum CAPA_phase  phase = capa_prelogin;

#endif
	int p1,p2;

	int TOP_found = 0;

	for (p1 = 0; p1 < folder->p->a.pop_mbx.data_len; p1 = p2) {
	    for (p2 = p1; p2 < folder->p->a.pop_mbx.data_len-1; p2++) {
		if ('\r' == folder->p->a.pop_mbx.command_data[p2] &&
		    '\n' == folder->p->a.pop_mbx.command_data[p2+1]) {
		    char * capa = & (folder->p->a.pop_mbx.command_data[p1]);
		    char * arg = NULL;
		    
		    folder->p->a.pop_mbx.command_data[p2] = '\0';

		    if (NULL != (arg = strchr(capa,' '))) {
			*arg = '\0';
			arg++;
		    }

		    DPRINT(Debug,2,(&Debug, 
				    "POP server capacity: '%s'%s%s\n",
				    capa,
				    arg ? " args: " : "",
				    arg ? arg : ""));

		    if (0 == strcmp(capa,"TOP"))
			TOP_found ++;

#ifdef USE_DLOPEN
		    probe_pop_capa_lib(&pop_capa_libs,
				       &pop_capa_libcount,
				       capa,
				       arg);
#endif

		    

		    p2 += 2;
		    break;
		}	       
	    }	    
	}
	pop_clear_command(folder);

	if (!TOP_found) {
	    DPRINT(Debug,8,(&Debug, "Disabling (possible) skipping mode\n"));
	    folder -> p->a.pop_mbx.pop_flags    |= POP_disable_skip;
	} else {
	    folder -> p->a.pop_mbx.pop_flags    &= ~POP_disable_skip;
	    if (pop_max_dl_size >= 0) {
		DPRINT(Debug,8,(&Debug, 
				"Skipping mode available (limit %d)\n",
				pop_max_dl_size
				));
	    }
	}

#ifdef USE_DLOPEN
	DPRINT(Debug,13,(&Debug,
			 "%d libraries available\n",pop_capa_libcount));
	if (!handle_pop_capa_libs(folder,&pop_capa_libs,
				  &pop_capa_libcount,
				  &phase,
				  &POP_FUNCTIONS)) {
	    /* Something gone wrong ... */

	    status = 0;
	    goto clean;	
	}

	switch(phase) {
	case capa_prelogin_again:
	    goto retry_prelogin_capa;
	case capa_logged:
	    goto is_already_logged;
	default:
	    break;
	}

#endif
    } else {
	pop_clear_command(folder);
    }

    if (! verify_peer_done) {
	if (remote_account_verify_peer(& (folder -> p->a.pop_mbx.C),&cbdata,
				       pop_verify_peer_cb,
				       POP_server,POP_server) < 1) {

	    DPRINT(Debug,8,(&Debug,
			    "pop_open_connection: Mailbox have verify failure\n"));

	    status = 0;
	    goto clean;
	}

	if (folder->p->a.pop_mbx.passhm) {
	    
	    if (! browser_passhm_verify_ra_con(folder->p->a.pop_mbx.passhm,
					       & (folder -> p->a.pop_mbx.C))) {
		
		DPRINT(Debug,8,(&Debug,
				"pop_open_connection: Mailbox have (hashmark) verify failure\n"));
		status = 0;
		goto clean;
	    }
	}
	
	verify_peer_done = 1;
    }


    

    data = elm_message(FRM("USER %s"),
		       folder -> p->a.pop_mbx.C.username);
    sta1 = pop_push_command(folder,data,1);
    free(data);
    
    if (!sta1) {
	status = 0;
	goto clean;	    
    }
    
    mbx_remote_login_msg(& (folder -> p->a.pop_mbx.C),rlogin_POP);
			 
    if (!pop_command_ok(folder)) {
	/* Server was given non-OK response to USER command ... */	    
	status = 0;
	goto clean;	    
    }
    pop_clear_command(folder);
    
    data1 = lib_prompt(1,CATGETS(elm_msg_cat, MeSet,MeLoginPasswd,
				 "Password for %s@%s:"),
		       folder -> p->a.pop_mbx.C.username,
		       folder -> p->a.pop_mbx.C.host);
    if (!data1) {
	status = 0;
	goto clean;	    
    }
    data = elm_message(FRM("PASS %s"),data1);
    sta1 = pop_push_command(folder,data,1);
    free(data);
    free(data1);
    
    if (!sta1) {
	status = 0;
	goto clean;	    
    }

    if (!pop_command_ok(folder)) {
	/* Server was given non-OK response to PASS command ... */	    
	status = 0;
	goto clean;	    
    }

#ifdef USE_DLOPEN
 is_already_logged:
#endif

    if (! verify_peer_done) {
	if (remote_account_verify_peer(& (folder -> p->a.pop_mbx.C),&cbdata,
				       pop_verify_peer_cb,
				       POP_server,POP_server) < 1) {
	    
	    DPRINT(Debug,8,(&Debug,
			    "pop_open_connection: Mailbox have verify failure\n"));

	    status = 0;
	    goto clean;
	}

	if (folder->p->a.pop_mbx.passhm) {
	    
	    if (! browser_passhm_verify_ra_con(folder->p->a.pop_mbx.passhm,
					       & (folder -> p->a.pop_mbx.C))) {
		
		DPRINT(Debug,8,(&Debug,
				"pop_open_connection: Mailbox have (hashmark) verify failure\n"));
		status = 0;
		goto clean;
	    }
	
	}

	verify_peer_done = 1;
    }

    status = 1;

    lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPLoginOK,
			  "POP login to %s as %s ... OK"),
		  folder -> p->a.pop_mbx.C.host,
		  folder -> p->a.pop_mbx.C.username);
    
 clean:
    pop_clear_command(folder);
    
    if (POP_server)
	free_string(&POP_server);


    if (!status) {
	if (folder->p->a.pop_mbx.C.stream) {
	    DPRINT(Debug,12,(&Debug,
			     "pop_open_connection: Closing stream\n"));
	    FreeStreamStack2(&(folder->p->a.pop_mbx.C.stream),
			     1 /* Force clearing of stack even when this is not last reference */);
	}
    }
    DPRINT(Debug,12,(&Debug,
		     "pop_open_connection=%d\n",
		     status));
    return status;
}

S_(mbx_sessionlock_folder mbx_sessionlock_pop)
static enum sessionlock_status mbx_sessionlock_pop
            P_((struct folder_info *folder,
		enum sessionlock_mode mode,
		int *errcode /* errno */,
		RECONNECT_MODE * reconnect_ptr
		));

static enum sessionlock_status mbx_sessionlock_pop(folder,mode,errcode,reconnect_ptr)
     struct folder_info *folder;
     enum sessionlock_mode mode;
     int *errcode /* errno */;
     RECONNECT_MODE * reconnect_ptr;
{
    enum sessionlock_status status = sessionlock_fail;
    enum sessionlock_status succeed = sessionlock_open;

    DPRINT(Debug,11,(&Debug,
		     "mbx_sessionlock_pop: folder=%p (%s) mode=%d",
		     folder,folder->cur_folder_sys));
    switch(mode) {
    case SESSIONLOCK_NORMAL:    DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NORMAL")); break;
    case SESSIONLOCK_REOPEN:    DPRINT(Debug,11,(&Debug, " SESSIONLOCK_REOPEN")); break;
    case SESSIONLOCK_TRUNCATE:  DPRINT(Debug,11,(&Debug, " SESSIONLOCK_TRUNCATE")); break;
    case SESSIONLOCK_CHECK:     DPRINT(Debug,11,(&Debug, " SESSIONLOCK_CHECK")); break;
    case SESSIONLOCK_NONE:      DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE")); break;
    case SESSIONLOCK_RECONNECT: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_RECONNECT")); break;
    case SESSIONLOCK_NONE_CHECKNEW: DPRINT(Debug,11,(&Debug, " SESSIONLOCK_NONE_CHECKNEW")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_pop",
	      "Bad magic number (private_data)",0);

    if (SESSIONLOCK_REOPEN == mode) {
	if (NULL != folder -> p->a.pop_mbx.C.stream ) {
	    if(pop_push_command(folder,"QUIT",1)) {
		pop_command_ok(folder);
		pop_clear_command(folder);
	    }
	    
	    DPRINT(Debug,12,(&Debug,
			     "mbx_sessionlock_pop: Closing stream\n"));
	    
	    FreeStreamStack2(&(folder -> p->a.pop_mbx.C.stream),
			     1 /* Force clearing of stack even when this is not last reference */);
	}		
    }

    if (NULL == folder->p->a.pop_mbx.C.stream ||
	folder -> p->a.pop_mbx.pop_state  == POP_not_logged) {

	if (reconnect_ptr && !*reconnect_ptr &&
	    SESSIONLOCK_REOPEN != mode) {
	    *reconnect_ptr =
		malloc_reconnect_mode();
	}
	    
	if (!pop_open_connection(folder)) {
	    status =  sessionlock_fail;
	    goto clean;
	}

	if (SESSIONLOCK_REOPEN == mode) {
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_pop: SESSIONLOCK_REOPEN: No reconnect handling\n"));
	} else {
	    succeed = sessionlock_reconnect;
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_pop: succeed = sessionlock_reconnect\n"));
	    
	    if (reconnect_ptr && *reconnect_ptr) {

		if (RECONNECT_MODE_magic != (*reconnect_ptr)->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_pop",
			  "Bad magic number (reconnect_mode)",0);
		
		switch (mbx_prepare_read_reconnect(folder,*reconnect_ptr,
						   & (folder->p->a.pop_mbx.C))) {
		    
		case prepare_reconnect_fail:
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_sessionlock_pop: Failing on mbx_prepare_read_reconnect\n"));
		    
		    status =  sessionlock_fail;
		    goto clean;
		case prepare_reconnect_ok:
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_sessionlock_pop:  OK mbx_prepare_read_reconnect\n"));
		    break;
		case prepare_reconnect_none:
		    break;
		}       	
	    }
	}

    } else if (SESSIONLOCK_RECONNECT == mode && reconnect_ptr && folder->was_reconnected) {
	/* Not needed ? */
	
	DPRINT(Debug,11,(&Debug, 
			 "mbx_sessionlock_pop: SESSIONLOCK_RECONNECT set, have reconnect_mode_ptr and folder is reconnected. Will move temp folder to reconnect_mode\n"));

	goto set_reconnect_mode;
	
    } else if (SESSIONLOCK_TRUNCATE == mode && reconnect_ptr && folder->was_reconnected) {
	/* Is this needed on POP ?? */
	
	DPRINT(Debug,11,(&Debug, 
			 "mbx_sessionlock_pop: SESSIONLOCK_TRUNCATE set, have reconnect_mode_ptr and folder is reconnected. Will move temp folder to reconnect_mode\n"));
	
    set_reconnect_mode:
	if (!*reconnect_ptr) {
	    *reconnect_ptr = malloc_reconnect_mode();
	}
	
	if (RECONNECT_MODE_magic != (*reconnect_ptr)->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"mbx_sessionlock_pop",
		  "Bad magic number (reconnect_mode)",0);
	
	switch (mbx_prepare_read_reconnect(folder,*reconnect_ptr,
					   & (folder->p->a.pop_mbx.C))) {
	    
	case prepare_reconnect_fail:
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_pop: Failing on mbx_prepare_read_reconnect\n"));
	    
	    status =  sessionlock_fail;
	    goto clean;
	case prepare_reconnect_ok:
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_pop:  OK mbx_prepare_read_reconnect\n"));
	    break;
	case prepare_reconnect_none:
	    break;
	}      
    }
	
    switch(mode) {
	int need_reopen;
    case SESSIONLOCK_RECONNECT:
	if (!folder -> p->fh_temp) {
	    DPRINT(Debug,11,(&Debug, 
			     "mbx_sessionlock_pop: SESSIONLOCK_RECONNECT need new temp file\n"));
	    goto create_it;
	}
	/* FALLTHRU */
    case SESSIONLOCK_NONE:
    case SESSIONLOCK_NONE_CHECKNEW:
	status = succeed;
	goto clean;
    case SESSIONLOCK_TRUNCATE:
	need_reopen = 1;
#ifdef FTRUNCATE  
	if (folder->p->fh_temp) {
	    int r;
	    rewind(folder->p->fh_temp);

	    r = ftruncate(fileno(folder->p->fh_temp),0);
	    if (0 == r) {
		need_reopen = 0;
		DPRINT(Debug,10,(&Debug,
				 "truncate_tempfolder: tempfile %s truncated (not recreated)\n",
				 folder->cur_tempfolder));

	    } else if (-1 == r) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, 
				  ElmSorryCantTruncateTemp,
				  "Sorry, can't truncate the temp file %s [%s]!"),
			  folder -> cur_tempfolder, strerror(err));
	    }
	    /* IF FAIL REOPEN IT INSTEAD */
	}
#endif
	if (!need_reopen)
	    break;  /* DONE */
	if (folder->p->fh_temp) {
	    fclose (folder->p->fh_temp);
	    folder->p->fh_temp = NULL;
	}

	if (0 != unlink(folder -> cur_tempfolder)) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantTruncateTemp,
			      "Sorry, can't truncate the temp file %s [%s]!"),
		      folder -> cur_tempfolder, strerror(err));    
	    status =  sessionlock_fail;
	    if (errcode)
		*errcode = err;
	    goto clean;
	}
	goto create_it;
    case  SESSIONLOCK_REOPEN:
	if (folder -> p->fh_temp) {
	    int temp_handle;
	    fclose(folder -> p->fh_temp);
	    folder -> p->fh_temp = NULL;

	    if ( (temp_handle = open(folder-> cur_tempfolder, 
				     O_RDWR,
				     0600)) == -1 ||
		 NULL == (folder -> p->fh_temp = fdopen(temp_handle,"w+"))) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmSorryCantReopenTemp,
				  "Sorry, can't reopen the temp file %s [%s]!"),
			  folder -> cur_tempfolder, strerror(err));  
		
		if (-1 != temp_handle)
		    close(temp_handle);
  
		status =  sessionlock_fail;
		if (errcode)
		    *errcode = err;
		goto clean;
	    }

	    break;
	}
	/* FALLTHRU */
    case SESSIONLOCK_NORMAL:
    case SESSIONLOCK_CHECK:
    create_it:
	if (!folder -> p->fh_temp) {
	    int err = 0;

	    /* If we are changing files and we are changing to a spool file,
	     * make sure there isn't a temp file for it, because if
	     * there is, someone else is using ELM to read the new file,
	     * and we don't want to be reading it at the same time.
	     */

	    if (sessionlock_create_tempfolder(folder,&err)) {

		DPRINT(Debug,1,(&Debug,
				"Creating tempfile (%s).\n",
				folder -> cur_tempfolder));
		
		elm_chown(folder->cur_tempfolder, userid, groupid,
			  NULL);
		chmod(folder -> cur_tempfolder, 0600);	
		/* shut off file for other people! */	    
		
	    } else {

		status =  sessionlock_fail;
		if (errcode)
		    *errcode = err;
		
		if (err == EEXIST) {
		    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmAlreadyRunningX,
				      "You seem to have ELM already reading this mail! You may not have two copies of\n\
ELM running simultaneously. If this is in error, then you'll need to remove \n\
the following file: %s"),
			      folder -> cur_tempfolder);
		    wait_for_timeout(5 + 6 * sleepmsg);
		    goto clean;	    
		}

		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFailedCreateTmpFolder,
				  "Failed to create %.50s: %.60s"),
			  folder -> cur_tempfolder,
			  strerror(err));
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmIGiveUp,
				  "Ahhhh... I give up."));
		
		goto clean;	    
	    }
	    
	}
	break;
    default:
	break;
    }
    status = succeed;
    
 clean:
    DPRINT(Debug,11,(&Debug,
		"mbx_sessionlock_pop=%d",
		status));
    switch (status) {
    case sessionlock_fail: DPRINT(Debug,11,(&Debug, " sessionlock_fail")); break;
    case sessionlock_open: DPRINT(Debug,11,(&Debug, " sessionlock_open")); break;
    case sessionlock_reconnect: DPRINT(Debug,11,(&Debug, " sessionlock_reconnect")); break;
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return status;
}

S_(mbx_flush_folder mbx_flush_pop)
static int mbx_flush_pop P_((struct folder_info *folder,
			     int *err_p  /* errno value */,
			     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							      on remote mailbox
						       */
			     ));
static int mbx_flush_pop(folder,err_p,cd) 
     struct folder_info *folder;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
							      on remote mailbox
			       */;
{
    int ret;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_flush_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    ret = mbx_flush_temp(folder,err_p);

    DPRINT(Debug,11,(&Debug,
		     "mbx_flush_pop=%d\n",ret));
    return ret;
}

S_(mbx_ferror_folder mbx_ferror_pop)
static int mbx_ferror_pop P_((struct folder_info *folder, int clean));
static int mbx_ferror_pop(folder,clean) 
     struct folder_info *folder; 
     int clean; 
{
    int status = 0;

    DPRINT(Debug,11,(&Debug,
		     "mbx_ferror_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_ferror_pop",
	      "Bad magic number (private_data)",0);

    if (POP_error == folder->p->a.pop_mbx.pop_state)
	status = 1;

    if (folder->p->fh_temp) {
	if (ferror(folder->p->fh_temp))
	    status = 1;
	if (clean)
	    clearerr(folder->p->fh_temp);
    }

    DPRINT(Debug,11,(&Debug,
		     "mbx_ferror_pop=%d\n",status));
    return status;
}

static int pop_got_line P_((struct folder_info *folder,
			    READ_STATE read_state_ptr,
			    char **buffer, int *len));

S_(mbx_prepare_read_folder mbx_prepare_read_pop)
static enum prepare_result mbx_prepare_read_pop P_((struct folder_info *folder,
				    enum prepare_mode mode,
				    READ_STATE read_state_ptr,
				    RECONNECT_MODE reconnect_ptr
				    ));
static enum prepare_result mbx_prepare_read_pop(folder,mode,read_state_ptr,reconnect_ptr)
     struct folder_info *folder;
     enum prepare_mode mode;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;   
{
    enum prepare_result status  = prepare_fail;
    enum prepare_result success = prepare_done;
    int r, tmp, i;
    int add_new_only = 0;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_read_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                    : read_state_ptr=%p\n",
		     read_state_ptr));
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_pop",
	      "Bad magic number (private_data)",0);
    
    switch (mode) {       	
    case PREPARE_NEW_ONLY:
    case PREPARE_NEW_ONLY_NOLOCK:
    case PREPARE_ACCESS:
	add_new_only = 1;
	DPRINT(Debug,11,(&Debug, 
			 "mbx_prepare_read_pop: Add new only\n"));
	break;
    case PREPARE_RECONNECT:	
    case PREPARE_NORMAL:
    case PREPARE_NOLOCK:
	if (reconnect_ptr) {
	    if (RECONNECT_MODE_magic != reconnect_ptr->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_pop",
		      "Bad magic number (reconnect_mode)",0);
	    
	    
	    switch (mbx_prepare_read_reconnect(folder,reconnect_ptr,
					       & (folder->p->a.pop_mbx.C))) {
	    case prepare_reconnect_fail:
		DPRINT(Debug,11,(&Debug, 
				 "mbx_prepare_read_pop: Failing on mbx_prepare_read_reconnect\n"));
		status = prepare_fail;
		goto clean;	
	    case prepare_reconnect_none:
		DPRINT(Debug,11,(&Debug, 
				 "mbx_prepare_read_pop: No mbx_prepare_read_reconnect\n"));
		success  = prepare_done;
		break;
	    case prepare_reconnect_ok:
		DPRINT(Debug,11,(&Debug, 
				 "mbx_prepare_read_pop: OK mbx_prepare_read_reconnect\n"));
		success = prepare_reconnected;
		break;
	    }	    	    	      	    
	}	
	break;	 
    }
    
    if (add_new_only) {
	/* start from unread messages */
	read_state_ptr->a.pop_mbx.msg_num = folder->p->a.pop_mbx.stat_count;
	if (!folder->p->fh_temp) {
	    DPRINT(Debug,11,(&Debug,
			     "mbx_prepare_read_pop -- NO temp file %s\n",
			     folder->cur_tempfolder));
	} else {
	    int r1 = fseek(folder -> p->fh_temp, 
			   folder -> mailfile_size, SEEK_SET);
	    
	    if (-1 == r1) {
		int err = errno;
		
		lib_error(CATGETS(elm_msg_cat, ElmSet, ElmCouldntSeekTempEnd,
				  "\nCouldn't fseek to end of temp mbox.\n"));
		lib_error(FRM("** %s. **\n"), strerror(err));

		DPRINT(Debug,11,(&Debug,
				 "mbx_prepare_read_pop: Failed to seek temp %s to end of file, offset %ld: %s (errno=%d)\n",
				 folder->cur_tempfolder,
				 folder -> mailfile_size,
				 strerror(err),
				 err));		
		
		
		status = prepare_fail;
		goto clean;	    
	    }  else if (0 == r1) {
		DPRINT(Debug,11,(&Debug,
				 "mbx_prepare_read_pop: Seeked temp %s to end of file, offset %ld\n",
				 folder->cur_tempfolder,
				 folder -> mailfile_size));
	    }	    
	}
	    
	read_state_ptr->fbytes = folder-> mailfile_size;  /* start correctly */
    } else { 
	/* start from beginning */
	read_state_ptr->a.pop_mbx.msg_num = 1;
    }
    
    if (!pop_push_command(folder,"STAT",1)) {
	status = prepare_fail;
	goto clean;
    }
    if (!pop_command_ok(folder)) {
	status = prepare_fail;
	goto clean;
    }
    tmp = folder -> p->a.pop_mbx.stat_size;
    if (2 != (r = sscanf(folder -> p->a.pop_mbx.command_status,"%*s %i %i",
		     & folder -> p->a.pop_mbx.stat_count,
		     & folder -> p->a.pop_mbx.stat_size))) {

	DPRINT(Debug,11,(&Debug,
			 "mbx_prepare_read_pop: Failed to parse STAT output (r = %d  != 2)\n",
			 r));
	lib_error(CATGETS(elm_msg_cat, MeSet,MeCantParseSTAT,
			  "Can't parse STAT output: %s"),
		  folder -> p->a.pop_mbx.command_status);
	status = prepare_fail;
	goto clean;
    }
    if (add_new_only) {
	/* Estimate new size for percent display */
	folder -> mailfile_size += folder -> p->a.pop_mbx.stat_size - tmp;
    } else {
	/* Real 
            folder -> mailfile_size does include skipped messages,
	    but here 
	    folder -> mailfile_size
	    includes skipped messages until folder is read.
	*/
	folder -> mailfile_size = folder -> p->a.pop_mbx.stat_size;
    }
    status = success;

    /* Try UIDL command */
    pop_clear_command(folder);
    DPRINT(Debug,11,(&Debug,
		     "   -- previous uidl count=%d, vector=%p\n",
		     read_state_ptr->a.pop_mbx.uidl_count,
		     read_state_ptr->a.pop_mbx.uidl_vector));
    for (i = 0; i < read_state_ptr->a.pop_mbx.uidl_count; i++) {
	if (read_state_ptr->a.pop_mbx.uidl_vector[i]) {
	    DPRINT(Debug,11,(&Debug,
			     "   ... was vector[%d]=%p (%s)\n",
			     i,
			     read_state_ptr->a.pop_mbx.uidl_vector[i],
			     read_state_ptr->a.pop_mbx.uidl_vector[i]));
	    free(read_state_ptr->a.pop_mbx.uidl_vector[i]);
	    read_state_ptr->a.pop_mbx.uidl_vector[i] = NULL;
	}
    }
    
    if (!pop_push_command(folder,"UIDL",0)) {
	status = prepare_fail;
	goto clean;
    }
    if (pop_command_ok(folder)) {      
	char * buffer = NULL;
	int buffer_len = 0;

	/* give_dt_estr_as_str adds / to end */
	const char * str = give_dt_estr_as_str(&folders_e,"maildir",
					       NULL,NULL);

	/* If command is unimplemeted that is OK ... */

	read_state_ptr->a.pop_mbx.uidl_count = 
	    folder -> p->a.pop_mbx.stat_count;
	read_state_ptr->a.pop_mbx.uidl_vector = 
	    safe_array_realloc(read_state_ptr->a.pop_mbx.uidl_vector,
			       read_state_ptr->a.pop_mbx.uidl_count,
			       sizeof (char *));
	for (i = 0; i < read_state_ptr->a.pop_mbx.uidl_count; i++)
	    read_state_ptr->a.pop_mbx.uidl_vector[i] = NULL;

	read_state_ptr->a.pop_mbx.data_idx = 0;
	while (pop_got_line(folder,read_state_ptr,&buffer,&buffer_len)) {
	    int l;
	    char *ptr;
	    
	    read_state_ptr->a.pop_mbx.data_idx += buffer_len;
	    
	    if (buffer_len > 0 && '\n' == buffer[buffer_len-1]) {
		buffer[buffer_len-1] = '\0';
		buffer_len--;
		if (buffer_len > 0 && '\r' == buffer[buffer_len-1]) {
		    buffer[buffer_len-1] = '\0';
		    buffer_len--;
		}
	    }
	    /* pop_got_line malloces also space for \0 */
	    buffer[buffer_len] = '\0';

	    l = strtol(buffer,&ptr,10);
	    if (ptr == buffer || ' ' != *ptr ||
		l < 1 || l > read_state_ptr->a.pop_mbx.uidl_count) {
		DPRINT(Debug,1,(&Debug,
				"Bad UIDL line: %s\n",buffer));
	    } else {
		struct uidl_entry *A;
		
		read_state_ptr->a.pop_mbx.uidl_vector[l-1]
		    = safe_strdup(ptr+1);
		DPRINT(Debug,18,(&Debug,
				 "UIDL(%ld) = %s\n",
				 l,
				 read_state_ptr->a.pop_mbx.uidl_vector[l-1]));
		
		/* Prefill */
		/* increments refcount, caller must call free_uidl_entry() */
		A = give_uidl(& (folder -> p->a.pop_mbx.uidl_list),
			      read_state_ptr->a.pop_mbx.uidl_vector[l-1],
			      NULL,NULL,0);
		
		A->old = 0;  /* Also on mailbox */
		free_uidl_entry(&A);
	    }

	    if (buffer) {
		free(buffer);
		buffer = NULL;
	    }
	}
	if (buffer) {
	    free(buffer);
	    buffer = NULL;
	}

	/* Remove UIDLs which are not seen on UIDL listing */
	remove_old_uidls(&(folder -> p->a.pop_mbx.uidl_list));
	
	if (str) {
	    char * fname = elm_message(FRM("%s.%s"),str,folder->cur_folder_sys);
	    update_uidls(fname,&(folder -> p->a.pop_mbx.uidl_list));
	    free(fname);
	}
    }
    
    /* Try LIST command */
    pop_clear_command(folder);
    DPRINT(Debug,11,(&Debug,
		     "   -- previous rfc822_size count=%d, vector=%p\n",
		     read_state_ptr->a.pop_mbx.rfc822_size_count,
		     read_state_ptr->a.pop_mbx.rfc822_size_vector));
    for (i = 0; i < read_state_ptr->a.pop_mbx.rfc822_size_count; i++)
	if (read_state_ptr->a.pop_mbx.rfc822_size_vector[i]) {
	    DPRINT(Debug,11,(&Debug,
			     "   ... was vector[%d]=%d\n",
			     i,
			     read_state_ptr->a.pop_mbx.rfc822_size_vector[i]));
	    read_state_ptr->a.pop_mbx.rfc822_size_vector[i] = -1;
	}

    if (pop_max_dl_size >= 0 &&
	! (folder -> p->a.pop_mbx.pop_flags & POP_disable_skip)
	) {
	if (!pop_push_command(folder,"LIST",0)) {
	    status = 0;
	    goto clean;
	}
	if (pop_command_ok(folder)) {
	    char * buffer = NULL;
	    int buffer_len = 0;

	    read_state_ptr->a.pop_mbx.rfc822_size_count = 
		folder -> p->a.pop_mbx.stat_count;
	    read_state_ptr->a.pop_mbx.rfc822_size_vector = 
		safe_array_realloc(read_state_ptr->a.pop_mbx.rfc822_size_vector,
				   read_state_ptr->a.pop_mbx.rfc822_size_count,
				   sizeof (int));
	    for (i = 0; i < read_state_ptr->a.pop_mbx.rfc822_size_count; i++)
		read_state_ptr->a.pop_mbx.rfc822_size_vector[i] = -1;
	    
	    read_state_ptr->a.pop_mbx.data_idx = 0;
	    while (pop_got_line(folder,read_state_ptr,&buffer,&buffer_len)) {
		int l;
		char *ptr;
		
		read_state_ptr->a.pop_mbx.data_idx += buffer_len;
		
		if (buffer_len > 0 && '\n' == buffer[buffer_len-1]) {
		    buffer[buffer_len-1] = '\0';
		    buffer_len--;
		    if (buffer_len > 0 && '\r' == buffer[buffer_len-1]) {
			buffer[buffer_len-1] = '\0';
			buffer_len--;
		    }
		}
		/* pop_got_line malloces also space for \0 */
		buffer[buffer_len] = '\0';
		
		l = strtol(buffer,&ptr,10);
		if (ptr == buffer || ' ' != *ptr ||
		    l < 1 || l > read_state_ptr->a.pop_mbx.rfc822_size_count) {
		    DPRINT(Debug,1,(&Debug,
				    "Bad LIST line: %s\n",buffer));
		} else {
		    read_state_ptr->a.pop_mbx.rfc822_size_vector[l-1]
			= atoi(ptr+1);
		    DPRINT(Debug,18,(&Debug,
				     "LIST(%ld) = %d\n",
				     l,
				     read_state_ptr->a.pop_mbx.rfc822_size_vector[l-1]));
		}
		
		if (buffer) {
		    free(buffer);
		    buffer = NULL;
		}
	    }
	    if (buffer) {
		free(buffer);
		buffer = NULL;
	    }
	}
    }
 
 clean:
    pop_clear_command(folder);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_read_pop=%d",status));
    switch (status) {
    case prepare_fail:  DPRINT(Debug,11,(&Debug, " prepare_fail")); break;
    case prepare_done:  DPRINT(Debug,11,(&Debug, " prepare_done"));
	if (reconnect_ptr) {
	    DPRINT(Debug,11,(&Debug, " (reconnect state not available)"));
	}

	break;
    case prepare_reconnected: DPRINT(Debug,11,(&Debug, " prepare_reconnected")); break;
    }   
    DPRINT(Debug,11,(&Debug, "\n"));

    return status;
}

S_(mbx_end_read_folder mbx_end_read_pop)
static int mbx_end_read_pop P_((struct folder_info *folder,
				READ_STATE read_state_ptr,
				RECONNECT_MODE reconnect_ptr,
				int silent));
static int mbx_end_read_pop(folder,read_state_ptr,
			    reconnect_ptr,silent)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     RECONNECT_MODE reconnect_ptr;  
     int silent;
{
    int status = 0;
    long size;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_read_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                : read_state_ptr=%p\n",
		     read_state_ptr));
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_read_pop",
	      "Bad magic number (private_data)",0);

    if (PREPARE_ACCESS != read_state_ptr->mode) {
	if (folder -> p->a.pop_mbx.stat_count > 
	    read_state_ptr->a.pop_mbx.msg_num) {
	    if (!silent) {
		lib_error(CATGETS(elm_msg_cat, MeSet,MeMessagesNotRead,
				  "Messages not read; current=%d, count=%d"),
			  read_state_ptr->a.pop_mbx.msg_num,
			  folder -> p->a.pop_mbx.stat_count);
	    }
	    status = 0;
	    goto clean;
	}
    }

    if (!folder->p->fh_temp) {
	DPRINT(Debug,11,(&Debug,
			 "mbx_end_read_pop=1 -- NO temp file %s\n",
			 folder->cur_tempfolder));
	return 1;
    }

    if ((ferror(folder->p->fh_temp)) || 
	(fflush(folder->p->fh_temp) == EOF)) {
	if (!silent) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, ElmSet, ElmFlushOnTempFailed,
			      "Flush on temp file %s failed: %s"), 
		      folder -> cur_tempfolder,
		      strerror(err));
	}
	DPRINT(Debug,1,(&Debug,
			"Can't flush on temp file %s!!\n",
			folder -> cur_tempfolder));
	status = 0;
	goto clean;
    }

    size = file_bytes(folder-> cur_tempfolder,NULL);
    if (-1L == size) {
	status = 0;
	goto clean;
    }
    /* Use tempfile size as folder size */
    folder-> mailfile_size = size;

    rewind(folder->p->fh_temp);
    status = 1;

 clean:
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_read_pop=%d\n",status));
    return status;
}

S_(info_zero_routine info_zero_pop_mbx)
static void info_zero_pop_mbx P_((struct mbx_hdr_info *info));
static void info_zero_pop_mbx(info)
     struct mbx_hdr_info *info;
{
    info->a.pop_mbx.msg_num = 0;
    info->a.pop_mbx.uidl    = NULL;
}

S_(info_free_routine  info_free_pop_mbx)
static void info_free_pop_mbx P_((struct mbx_hdr_info *info));
static void info_free_pop_mbx(info)
     struct mbx_hdr_info *info;
{
    info->a.pop_mbx.msg_num = 0;
    if (info->a.pop_mbx.uidl) {
	free (info->a.pop_mbx.uidl);
	info->a.pop_mbx.uidl = NULL;
    }
}

S_(info_status_routine info_pop_mbx_status)
static void info_pop_mbx_status P_((struct mbx_hdr_info *info,
				    char status_buffer[3]));
static void info_pop_mbx_status(info,status_buffer)
     struct mbx_hdr_info *info;
     char status_buffer[3];
{

}

static struct info_type pop_mbx_info = {
					 INFO_TYPE_magic, info_zero_pop_mbx, 
					 info_free_pop_mbx,
					 info_pop_mbx_status
};

S_(mbx_copy_envelope_folder mbx_copy_envelope_pop)
static enum copy_env_status mbx_copy_envelope_pop 
    P_((struct folder_info *folder,
	READ_STATE read_state_ptr,
	struct header_rec *entry,
	int force));
static enum copy_env_status mbx_copy_envelope_pop(folder,read_state_ptr,
						  entry,force)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     struct header_rec *entry;
     int force;
{
    enum copy_env_status status = copy_env_eof,i;
    char * data = NULL;
    char * return_path_data = NULL;
    char * from_separator   = NULL;

    int l;

    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                     : read_state_ptr=%p\n",
		     read_state_ptr));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_copy_envelope_pop",
	      "Bad magic number (private_data)",0);

    if (PREPARE_ACCESS == read_state_ptr->mode) {
	if(! entry->mbx_info ||
	   &pop_mbx_info != entry->mbx_info->type_code) {
	    DPRINT(Debug,11,(&Debug, 
			     "                     : Unable to access message\n"));
	    status = copy_env_eof;
	    goto clean;	    
	}

	/* Hack It */
	read_state_ptr->a.pop_mbx.msg_num = entry->mbx_info->a.pop_mbx.msg_num;
	
    } else {
	if (folder -> p->a.pop_mbx.stat_count < 
	    read_state_ptr->a.pop_mbx.msg_num) {
	    DPRINT(Debug,11,(&Debug,
			     "                     : End of messages\n"));
	    status = copy_env_eof;
	    goto clean;
	}

	/* Do not reset status if PREPARE_ACCESS */
	entry->status = VISIBLE | UNREAD | NEW;
	/* If UIDLs are supported we save them and use that
	   to determine if message is read before or new
	*/

    }

    read_state_ptr -> linecounter   = 0;      /* Linecounter of current
					         message */
    read_state_ptr->skipping = sm_reading;      /* not skipping */
    entry->offset = read_state_ptr -> fbytes; /* Offset of current message */
   
    DPRINT(Debug,12,(&Debug,
		     "                     : temp file offset %ld\n",
		     entry->offset));

    /* Use same time on all nessages so that mailbox order
       is used when received sorting order is requested.

       This time is required also when message is skipped.
    */
    entry->received_time = folder->p->a.pop_mbx.default_received_time;
    
    i = read_state_ptr->a.pop_mbx.msg_num ;
    if (pop_max_dl_size >= 0 &&
	i > 0 && i <= read_state_ptr->a.pop_mbx.rfc822_size_count &&
	read_state_ptr->a.pop_mbx.rfc822_size_vector[i-1] > pop_max_dl_size &&
	!force &&
	! (folder -> p->a.pop_mbx.pop_flags & POP_disable_skip)
	) {
	/* Need skip this */


	DPRINT(Debug,8,(&Debug,
                        "mbx_copy_envelope_pop: Size %d -- skipping this message (only read headers)\n",
			read_state_ptr->a.pop_mbx.rfc822_size_vector[i-1]));


	data = elm_message(FRM("TOP %d 0"),read_state_ptr->a.pop_mbx.msg_num);
	if (!pop_push_command(folder,data,0)) {
	    status = copy_env_eof;
	    goto clean;
	}
	if (!pop_command_ok(folder)) {
	    free(data);
	    data = NULL;

	    DPRINT(Debug,8,(&Debug,
			    "mbx_copy_envelope_pop: TOP command failed -- can't skip message\n"));

	    goto TOP_failed;
	}
	
	free(data);
	data = NULL;


	read_state_ptr->skipping = sm_skipping;
	entry->offset             = -1;

	read_state_ptr->skip_count++;
	read_state_ptr->skip_bytes += 
	    read_state_ptr->a.pop_mbx.rfc822_size_vector[i-1];

	/* Hack -- no line numbers available */
	entry->lines = -1;

    } else {
	char * T1;
    TOP_failed:

	data = elm_message(FRM("RETR %d"),read_state_ptr->a.pop_mbx.msg_num);
	if (!pop_push_command(folder,data,0)) {
	    status = copy_env_eof;
	    goto clean;
	}
	if (!pop_command_ok(folder)) {
	    status = copy_env_eof;
	    goto clean;
	}

	free(data);
	data = NULL;
        
	/* Very lazy method to get something to put "From " -separator */
	if (0 == memcmp("Return-Path: ",
			folder->p->a.pop_mbx.command_data,13)) {
	    int i;
	    char * XXX;
	    for (i=13; i < folder->p->a.pop_mbx.data_len; i++) {
		if ('\r' == folder->p->a.pop_mbx.command_data[i] ||
		    '\n' == folder->p->a.pop_mbx.command_data[i])
		    break;
	    }
	    XXX = safe_malloc(i-12);
	    memcpy(XXX,folder->p->a.pop_mbx.command_data+13,i-13);
	    XXX[i-13]='\0';

	    return_path_data = return_path_to_env_from_1(XXX);  /* result MALLOCED */

	    free(XXX);
	    if (! return_path_data)
		return_path_data = safe_strdup("nobody@localhost");	
	} else
	    return_path_data = safe_strdup("nobody@localhost");
	
	/* Make some kind mailbox separator line ... */	
	entry->env_from[0] = '\0'; /* Let parse it later */

	T1 = ctime(&entry->received_time);
	if (!T1) {
	    DPRINT(Debug,14,(&Debug, 
			     "mbx_copy_envelope_pop: ctime() failed\n"));

	    status = copy_env_eof;
	    goto clean;
	}
	
	
	from_separator = elm_message(FRM("From %s %s"),return_path_data,T1);

	 DPRINT(Debug,14,(&Debug, 
		     "mbx_copy_envelope_pop: generated %s",
		     from_separator));
    	 
	l = strlen(from_separator);
	/* ctime() will include terminating \n
	 * -- notice that on mailbox separator line we don't use \r\n
	 * that even when on 'binary' mail message lines are terminated with 
	 * \r\n
	 */
	if (!mbx_copy_line_to_temp(folder,from_separator,l)) {
	    status = copy_env_eof;
	    goto clean;
	}
	read_state_ptr -> fbytes += l;
	read_state_ptr -> linecounter++;
    }

    read_state_ptr->a.pop_mbx.data_idx = 0;
    change_rec_mbx_info(entry,&pop_mbx_info);
    entry->mbx_info->a.pop_mbx.msg_num = read_state_ptr->a.pop_mbx.msg_num;

    i = read_state_ptr->a.pop_mbx.msg_num ;
    if (i > 0 && i <= read_state_ptr->a.pop_mbx.uidl_count &&
	read_state_ptr->a.pop_mbx.uidl_vector[i-1]) {
	struct uidl_entry * A;
	
	entry->mbx_info->a.pop_mbx.uidl =
	    strmcpy(entry->mbx_info->a.pop_mbx.uidl,
		    read_state_ptr->a.pop_mbx.uidl_vector[i-1]);
	
	DPRINT(Debug,11,(&Debug,
			 "                     : (uidl)=%s\n",
			 entry->mbx_info->a.pop_mbx.uidl));

	/* increments refcount, caller must call free_uidl_entry() */
	A = give_uidl(& (folder -> p->a.pop_mbx.uidl_list),
		      entry->mbx_info->a.pop_mbx.uidl,NULL,NULL,0);
	if (PREPARE_ACCESS == read_state_ptr->mode) {
	    A->status = entry->status;
	} else {
	    entry->status = VISIBLE | A->status;
	}
	A->old = 0;  /* Also on mailbox */
	
	free_uidl_entry(&A);

    }

    status = copy_env_ok;

 clean:
    if (return_path_data)
	free(return_path_data);
    if (from_separator)
	free(from_separator);
    
    if (data) 
	free(data);
    data = NULL;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_pop=%d",
		     status));
    switch (status) {
    case copy_env_no_data:   DPRINT(Debug,11,(&Debug," copy_env_no_data")); break;
    case copy_env_format:    DPRINT(Debug,11,(&Debug," copy_env_format"));  break;
    case copy_env_eof:       DPRINT(Debug,11,(&Debug," copy_env_eof"));     break;
    case copy_env_ok:        DPRINT(Debug,11,(&Debug," copy_env_ok"));      break;
    }
    DPRINT(Debug,11,(&Debug,"\n"));
    
    return status;
}

S_(mbx_is_forwarded_folder mbx_is_forwarded_pop)
static const char * mbx_is_forwarded_pop P_((struct folder_info *folder,
					      READ_STATE read_state_ptr));
static const char * mbx_is_forwarded_pop(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_is_forwarded_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                     : read_state_ptr=%p\n",
		     read_state_ptr));

    DPRINT(Debug,11,(&Debug,
		     "mbx_is_forwarded_pop=NULL\n"));
    return NULL;
}

static int pop_got_line(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    int i;

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"pop_got_line",
	      "Bad magic number (private_data)",0);

    if (read_state_ptr->a.pop_mbx.data_idx >= 
	folder->p->a.pop_mbx.data_len) {
	DPRINT(Debug,65,(&Debug,
			 "pop_got_line=0\n"));
	return 0;
    }

    for (i = read_state_ptr->a.pop_mbx.data_idx;
	 i < folder->p->a.pop_mbx.data_len-1; i++) {
	if ('\r' == folder->p->a.pop_mbx.command_data[i] &&
	    '\n' == folder->p->a.pop_mbx.command_data[i+1]) {
	    i += 2;
	    goto found;
	}
    }
    i = folder->p->a.pop_mbx.data_len;
    DPRINT(Debug,10,(&Debug,"pop_got_line: EOLN NOT FOUND\n"));

 found:
    *len = i - read_state_ptr->a.pop_mbx.data_idx;
    *buffer = safe_malloc(*len+1);
    memcpy(*buffer,
	   folder->p->a.pop_mbx.command_data +
	   read_state_ptr->a.pop_mbx.data_idx,
	   *len);

    DPRINT(Debug,65,(&Debug,
		     "pop_got_line=1: len=%d, buffer=",
		     *len));
    DEBUG_PRINT_BUFFER(Debug,65,*len,s2us(*buffer));
    if (*len < 1 || !(*buffer) || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,65,(&Debug,
			 "\npop_got_line: NO NEWLINE\n"));
    } 
    return 1;
    
}

static void pop_accept_line P_((struct folder_info *folder,
					 READ_STATE read_state_ptr,
					 char **buffer, int *len));
static void pop_accept_line(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    if (!*buffer)
	return;

    if (*len < 1 || (*buffer)[*len -1] != '\n') {
    } else {
	read_state_ptr -> linecounter++;
    }
    switch (read_state_ptr -> skipping) {
    case sm_EOM:
	DPRINT(Debug,20,(&Debug, 
			 "pop_accept_line: end of message, skipping body\n"));
	break;
    case sm_reading:
	read_state_ptr -> fbytes             += *len;
	break;
    case sm_skipping:
	DPRINT(Debug,20,(&Debug, 
			 "pop_accept_line: skipping body\n"));
	break;
    }

    read_state_ptr->a.pop_mbx.data_idx   += *len;

    free(*buffer);
    *buffer = NULL;
    *len    = 0;
}

S_(mbx_copy_header_folder mbx_copy_header_pop)
static int mbx_copy_header_pop P_((struct folder_info *folder,
				   READ_STATE read_state_ptr,
				   char **buffer, int *len));
static int mbx_copy_header_pop(folder,read_state_ptr,buffer,len)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
{
    int status = 0;
    char *buffer1 = NULL;
    int len1,i;

    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_header_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                   : read_state_ptr=%p\n",
		     read_state_ptr));
	
 hide:
    if (buffer1) {
	free(buffer1);
	buffer1 = NULL;
    }


    if (!pop_got_line(folder,read_state_ptr,&buffer1,&len1)) {
	status = 0;
	goto clean;
    }

    if ((len1 == 1 && 0 == memcmp(buffer1,"\n",len1)) ||
	(len1 == 2 && 0 == memcmp(buffer1,"\r\n",len1))) {

	switch (read_state_ptr -> skipping) {
	case sm_EOM:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_pop: end of message, skipping body\n"));
	    break;
	case sm_reading:{
	    /* End of headers */
	    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
		status = 0;
		goto clean;
	    }
	}
	    break;
	case sm_skipping:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_pop: skipping body\n"));
	    break;
	}
	pop_accept_line(folder,read_state_ptr,
			&buffer1,&len1);
	status = 0;
	goto clean;
    }
    if (NULL == memchr(buffer1,':',len1)) {
	/* End of headers -- bad header */
	status = 0;
	goto clean;
    }
   
    do {
	append_buffer(buffer,len,buffer1,len1);
	switch (read_state_ptr -> skipping) {
	case sm_EOM:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_pop: end of message, skipping body\n"));
	    break;
	case sm_reading: {
	    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
		status = 0;
		goto clean;
	    }
	}
	    break;
	case sm_skipping:
	    DPRINT(Debug,20,(&Debug, 
			     "mbx_copy_header_pop: skipping body\n"));
	    break;
	}
	pop_accept_line(folder,read_state_ptr,
			&buffer1,&len1);
	if (!pop_got_line(folder,read_state_ptr,
			  &buffer1,&len1)) {
	    break;
	}
    } while (len1 > 0 && whitespace(buffer1[0]));

    i = read_state_ptr->a.pop_mbx.msg_num ;
    if (*buffer && *len > 6 &&
	header_cmp(*buffer, "Status", NULL) &&
	i > 0 && i <= read_state_ptr->a.pop_mbx.uidl_count &&
	read_state_ptr->a.pop_mbx.uidl_vector[i-1]) {
	DPRINT(Debug,11,(&Debug,
		    "                   : Hiding header %s",*buffer));
	DPRINT(Debug,11,(&Debug,
			 "                   : (uidl)=%s\n",
			 read_state_ptr->a.pop_mbx.uidl_vector[i-1]));
	free(*buffer);
	*buffer = NULL;
	*len = 0;
	goto hide;
    }
	
    status = 1;
   
 clean:
    if (buffer1) {
	free(buffer1);
	buffer1 = NULL;
    }

    if (!status) {

	if (*buffer) {
	    DPRINT(Debug,20,(&Debug,"mbx_copy_header_pop: discarding len=%d buffer=",
			     len));
	    
	    DEBUG_PRINT_BUFFER(Debug,20,*len,s2us(*buffer));
	    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
		DPRINT(Debug,20,(&Debug,
				 "\nmbx_copy_header_pop  <- NO NEWLINE\n"));
	    }
	
	    free(*buffer);
	    *buffer = NULL;
	}
    }
    
    DPRINT(Debug,50,(&Debug,
		     "mbx_copy_header_pop=%d | len=%d, buffer=",
		     status,*len));
    DEBUG_PRINT_BUFFER(Debug,50,*len,s2us(*buffer));
    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,50,(&Debug,
			 "\nmbx_copy_header_pop <- NO NEWLINE\n"));
    }
    return status;
}

S_(mbx_copy_body_folder mbx_copy_body_pop)
static int mbx_copy_body_pop P_((struct folder_info *folder,
				     READ_STATE read_state_ptr,
				     char **buffer, int *len,
				     long *content_remaining));

static int mbx_copy_body_pop(folder,read_state_ptr,buffer,len,
				 content_remaining)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     char **buffer; 
     int *len;
     long *content_remaining;
{ 
    char *buffer1 = NULL;
    int len1;
    int status = 0;

    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_body_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                 : read_state_ptr=%p\n",
		     read_state_ptr));

    switch (read_state_ptr -> skipping) {
    case sm_EOM: 
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_body_pop = 0 (end of message, skipping body)\n"));
	    return 0;
    case sm_reading:
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_body_pop: Reading body\n"));
	break;
    case sm_skipping:
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_body_pop = 0 (skipping body)\n"));
	return 0;
    }

    if (!pop_got_line(folder,read_state_ptr,
		      &buffer1,&len1)) {
	status = 0;
	goto clean;
    }

    append_buffer(buffer,len,buffer1,len1);
    if (!mbx_copy_line_to_temp(folder,buffer1,len1)) {
	status = 0;
	goto clean;
    }
    pop_accept_line(folder,read_state_ptr,
		     &buffer1,&len1);
    status = 1;

 clean:
    if (buffer1) {
	free(buffer1);
	buffer1 = NULL;
    }

    if (!status) {
	if (*buffer) {
	    free(*buffer);
	    *buffer = NULL;
	}
    }

    DPRINT(Debug,50,(&Debug,
		     "mbx_copy_body_pop=%d | len=%d, buffer=",
		     status,*len));
    DEBUG_PRINT_BUFFER(Debug,50,*len,s2us(*buffer));
    if (*len < 1 || !*buffer || (*buffer)[*len -1] != '\n') {
	DPRINT(Debug,50,(&Debug,
			 "\nmbx_copy_body_pop <- NO NEWLINE\n"));
    }
    return status;
}

S_(mbx_copy_envelope_end_folder mbx_copy_envelope_end_pop)
static enum copy_env_end_status mbx_copy_envelope_end_pop P_((struct folder_info *folder,
					 READ_STATE read_state_ptr,
					 long *newbytes,
					 int  *newmails
					 ));
static enum copy_env_end_status mbx_copy_envelope_end_pop(folder,read_state_ptr,
							  newbytes,newmails)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
     long *newbytes;
     int  *newmails;
{
    enum copy_env_end_status status = copy_env_end_mismatch;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_end_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                         : read_state_ptr=%p\n",
		     read_state_ptr));

    pop_clear_command(folder);
    read_state_ptr->a.pop_mbx.msg_num++;

    switch (read_state_ptr -> skipping) {
    case sm_EOM:
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_envelope_end_pop: end of message, skipping body\n"));
	break;

    case sm_reading: {
	/* On mailbox format there is empty line (\n\n)
	 * on end of mail...
	 */

	if (!mbx_copy_line_to_temp(folder,"\n",1)) {
	    status = copy_env_end_failure;
	    goto clean;
	}
	read_state_ptr -> fbytes ++;    /* Increment position */
    }
	break;
    case sm_skipping:
	DPRINT(Debug,20,(&Debug, 
			 "mbx_copy_envelope_end_pop: skipping body\n"));
	break;
    }
    status = copy_env_end_match;

 clean:
    read_state_ptr->skipping = sm_reading;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_end_pop=%d",
		     status));
    switch (status) {
    case copy_env_end_failure:  DPRINT(Debug,10,(&Debug," copy_env_end_failure"));  break;
    case copy_env_end_mismatch: DPRINT(Debug,10,(&Debug," copy_env_end_mismatch")); break;
    case copy_env_end_match:    DPRINT(Debug,10,(&Debug," copy_env_end_match"));    break;
    case copy_env_end_newmail:  DPRINT(Debug,10,(&Debug," copy_env_end_newmail"));  break;
    }
    if (newbytes) {
	DPRINT(Debug,11,(&Debug," *newbytes=%ld",
			 *newbytes));
    }
    if (newmails) {
	DPRINT(Debug,11,(&Debug," *newmails=%d",
			 *newmails));
    }
    DPRINT(Debug,10,(&Debug,"\n"));
    
    return status; 
}

S_(mbx_copy_envelope_reset_body mbx_copy_envelope_reset_body_pop)
static int mbx_copy_envelope_reset_body_pop P_((struct folder_info *folder,
						READ_STATE read_state_ptr));
static int mbx_copy_envelope_reset_body_pop(folder,read_state_ptr)
     struct folder_info *folder;
     READ_STATE read_state_ptr;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_reset_body_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "                                : read_state_ptr=%p\n",
		     read_state_ptr));
    DPRINT(Debug,11,(&Debug,
		     "mbx_copy_envelope_reset_body_pop=0\n"));
    return 0;
}

S_(mbx_folder_to_fd mbx_pop_to_fd)
static FILE * mbx_pop_to_fd P_((struct folder_info *folder,long offset));
static FILE * mbx_pop_to_fd(folder,offset)
     struct folder_info *folder;
     long offset;
{
    FILE * status = NULL;
    DPRINT(Debug,11,(&Debug,
		     "mbx_pop_to_fd: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_pop_to_fd",
	      "Bad magic number (private_data)",0);

    if (offset != -1L &&
	fseek(folder->p->fh_temp, offset , 
	      SEEK_SET) == -1) {
	int err = errno;
	lib_error(CATGETS(elm_msg_cat, ElmSet, 
			  ElmCouldntSeekBytesIntoFolder,
			  "\nCouldn't seek %ld bytes into folder.\n"),
		  offset);		
	lib_error(FRM("** %s. **\n"), strerror(err));
	DPRINT(Debug,1,(&Debug,
			"Error: Couldn't seek folder %s: (offset %ld) Errno %s (%s)\n",
			folder->cur_tempfolder, offset, 
			strerror(err), "mbx_pop_to_fd"));
	status = NULL;
	goto clean;
    }
    status = folder->p->fh_temp;

 clean:
    DPRINT(Debug,11,(&Debug,
		     "mbx_pop_to_fd=%s (%p)\n",
		     status ? "NON-NULL" : "NULL",
		     status));
    return status;
}

S_(mbx_new_mail_on_folder mbx_new_mail_on_pop)
static enum new_mail_stat mbx_new_mail_on_pop P_((struct folder_info *folder,
						  long *bytes, int *new_mailcount,
						  int *err_p  /* errno value */,
						  struct cancel_data  * cd
						  /* Allow cancelation (Ctrl-C)
						     check of remeote mailbox
						  */
						  ));
static enum new_mail_stat mbx_new_mail_on_pop(folder,bytes,new_mailcount,err_p,
					      cd)
     struct folder_info *folder;
     long *bytes;
     int  *new_mailcount /* -1 for local mailbox */;
     int *err_p  /* errno value */;
     struct cancel_data  * cd  /* Allow cancelation (Ctrl-C)
				  check of remeote mailbox
			       */;
{
    enum new_mail_stat status = no_new_mail;
    int count,size,r;

    DPRINT(Debug,11,(&Debug,
		     "mbx_new_mail_on_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (err_p)
	*err_p =0;
    
    /* THIS does NOT actually work -- new
       mail does not arrive when POP have
       mailbox open.
    */

    if (!pop_push_command_c(folder,"STAT",1,cd)) {
	status = failed_mbx_ign_new_mail;      /* Ignore failure, new mail check 
						  does not work for POP anyway */
	goto clean;
    }
    if (!pop_command_ok_c(folder,cd)) {
	status = new_mail_check_failed;
	goto clean;
    }

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_new_mail_on_pop",
	      "Bad magic number (private_data)",0);

    if (2 != (r = sscanf(folder -> p->a.pop_mbx.command_status,"%*s %i %i",
			 & count, & size))) {
	DPRINT(Debug,11,(&Debug,
			 "mbx_new_mail_on_pop: Failed to parse STAT output (r = %d  != 2)\n",
			 r));
	lib_error(CATGETS(elm_msg_cat, MeSet,MeCantParseSTAT,
			  "Can't parse STAT output: %s"),
		  folder -> p->a.pop_mbx.command_status);
	status = new_mail_check_failed;
	goto clean;
    }
    if (count > folder -> p->a.pop_mbx.stat_count)
	status = have_new_mail;
    if (bytes) {
	*bytes = size - folder -> p->a.pop_mbx.stat_size;
    }
    if (new_mailcount) {
	*new_mailcount = count - folder -> p->a.pop_mbx.stat_count;
    }

 clean:
    pop_clear_command(folder);

    DPRINT(Debug,11,(&Debug,
		     "mbx_new_mail_on_pop=%d",
		     status));
    switch (status) {
    case failed_mbx_ign_new_mail: DPRINT(Debug,11,(&Debug," failed_mbx_ign_new_mail")); break;
    case new_mail_check_failed:   DPRINT(Debug,11,(&Debug," new_mail_check_failed"));   break;
    case no_new_mail:             DPRINT(Debug,11,(&Debug," no_new_mail"));             break;
    case have_new_mail:           DPRINT(Debug,11,(&Debug," have_new_mail"));           break;
    case new_mail_reconnect:      DPRINT(Debug,11,(&Debug," new_mail_reconnect"));      break;
    }
    if (bytes) {
	DPRINT(Debug,11,(&Debug, " *bytes=%ld",
			 *bytes));
    }
    if (new_mailcount) {
	DPRINT(Debug,11,(&Debug, " *new_mailcount=%d",
			 *new_mailcount));
    }
    if (err_p) {
	DPRINT(Debug,11,(&Debug, " *err_p=%d",
			 *err_p));
	if (*err_p) {
	    DPRINT(Debug,11,(&Debug, " (%s)",
			     strerror(*err_p)));
	}
    }
    DPRINT(Debug,11,(&Debug, "\n"));
    
    return status;
}

S_(mbx_consider_remove_folder mbx_consider_remove_pop)
static int mbx_consider_remove_pop P_((struct folder_info *folder));
static int mbx_consider_remove_pop(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_consider_remove_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_consider_remove_pop=0\n"));
    return 0;
}

S_(mbx_prepare_keep_folder mbx_prepare_keep_pop)
static int mbx_prepare_keep_pop P_((struct folder_info *folder,
				    KEEP_STATE keep_state_ptr));
static int mbx_prepare_keep_pop(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    int status = 1;
 
    DPRINT(Debug,11,(&Debug,
		     "mbx_prepare_keep_pop=%d (dummy): folder=%p (%s)\n",
		     status,folder,folder->cur_folder_sys));
    
    return status;
}

S_(mbx_end_keep_folder mbx_end_keep_pop)
static int mbx_end_keep_pop P_((struct folder_info *folder,
				KEEP_STATE keep_state_ptr));
static int mbx_end_keep_pop(folder,keep_state_ptr)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
{
    /* give_dt_estr_as_str adds / to end */
    const char *str = give_dt_estr_as_str(&folders_e,"maildir",
					  NULL,NULL);

    int status = 0;

    DPRINT(Debug,11,(&Debug,
		     "mbx_end_keep_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    /* POP mailbox is needed to be closed so that
     * it will enter UPDATE state. sessionlock_folder()
     * will reopen it when needed
     */
    if (!pop_push_command(folder,"QUIT",1)) {
	status = 0;
	pop_clear_command(folder);
	goto clean;
    }
    status = pop_command_ok(folder);
    pop_clear_command(folder);

    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_end_keep_pop",
	      "Bad magic number (private_data)",0);

    if (folder->p->a.pop_mbx.C.stream) {
	DPRINT(Debug,12,(&Debug,
			 "mbx_end_keep_pop: Closing stream\n"));
	
	FreeStreamStack2(&(folder -> p->a.pop_mbx.C.stream),
			 1 /* Force clearing of stack even when this is not last reference */);
    }

    if (str) {
	char * fname = elm_message(FRM("%s.%s"),str,folder->cur_folder_sys);
	update_uidls(fname,&(folder -> p->a.pop_mbx.uidl_list));
	free(fname);
    }

 clean:
    DPRINT(Debug,11,(&Debug,
		     "mbx_end_keep_pop=%d\n",
		     status));
    return status;
}

S_(mbx_mark_keep_folder mbx_mark_keep_pop)
static int mbx_mark_keep_pop P_((struct folder_info *folder,
				 KEEP_STATE keep_state_ptr,
				 struct header_rec *entry,
				 int keep));
static int mbx_mark_keep_pop(folder,keep_state_ptr,entry,keep)
     struct folder_info *folder;
     KEEP_STATE keep_state_ptr;
     struct header_rec *entry;
     int keep;
{
    int ret = 1;

    DPRINT(Debug,11,(&Debug,
		     "mbx_mark_keep_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_mark_keep_pop",
	      "Bad magic number (private_data)",0);

    /* Store read/unread together
     * with UIDL
     */

    if(entry->mbx_info &&
       &pop_mbx_info == entry->mbx_info->type_code) {
	if (entry->mbx_info->a.pop_mbx.uidl) {
	    struct uidl_entry * A;

	    DPRINT(Debug,11,(&Debug,
			     "                     : (uidl)=%s\n",
			     entry->mbx_info->a.pop_mbx.uidl));

	    /* increments refcount, caller must call free_uidl_entry() */
	    A = give_uidl(& (folder -> p->a.pop_mbx.uidl_list),
			  entry->mbx_info->a.pop_mbx.uidl,NULL,NULL,0);
	    A->status  = entry->status; 
	    A->changed = 1;
	    		A->old = 0;  /* Also on mailbox */
	    free_uidl_entry(&A);

	}
	
	if (!keep &&
	    entry->mbx_info->a.pop_mbx.msg_num > 0) { 
	    char * data = NULL;
	    data = elm_message(FRM("DELE %d"),
			       entry->mbx_info->a.pop_mbx.msg_num);
	    if (pop_push_command(folder,data,1))
		ret = pop_command_ok(folder);
	    else
		ret = 0;
	    pop_clear_command(folder);
	}
    }

    DPRINT(Debug,11,(&Debug,
		     "mbx_mark_keep_pop=%d\n",ret));

    return ret;
}

S_(mbx_folder_type mbx_pop_type)
const char * mbx_pop_type P_((struct folder_info *folder));
const char * mbx_pop_type(folder)
     struct folder_info *folder;
{
    static char *mailbox = NULL;
    if (!mailbox)
	mailbox = catgets(elm_msg_cat, MeSet, MePopMailbox, "POP mailbox");

    DPRINT(Debug,11,(&Debug,
		"mbx_pop_type: folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    DPRINT(Debug,11,(&Debug,
		     "mbx_pop_type=%s\n",
		     mailbox));
    return mailbox;
}

S_(mbx_start_edit_folder mbx_start_edit_pop)
static int mbx_start_edit_pop P_((struct folder_info *folder, 
				  const char **buffer));
static int mbx_start_edit_pop(folder,buffer)
     struct folder_info *folder;
     const char **buffer;
{
    DPRINT(Debug,11,(&Debug,
		"mbx_start_edit_pop=0 (dummy): folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    return 0;
}

S_(mbx_end_edit_folder mbx_end_edit_pop)
static enum end_edit_fol_status mbx_end_edit_pop P_((struct folder_info *folder));
static enum end_edit_fol_status mbx_end_edit_pop(folder)
     struct folder_info *folder;
{
    DPRINT(Debug,11,(&Debug,
		"mbx_end_edit_pop= end_edit_none (dummy): folder=%p (%s)\n",
		folder,folder->cur_folder_sys));
    return end_edit_none;
}

S_( mbx_zero_rs_fields_folder mbx_zero_rs_fields_pop)
static void mbx_zero_rs_fields_pop P_((struct read_folder_state *rs));
static void mbx_zero_rs_fields_pop(rs)
     struct read_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_zero_rs_fields_pop: rs=%p\n",
		     rs));

    rs->a.pop_mbx.msg_num = 0;
    rs->a.pop_mbx.uidl_vector = NULL;
    rs->a.pop_mbx.uidl_count  = 0;
    rs->a.pop_mbx.rfc822_size_vector = NULL;
    rs->a.pop_mbx.rfc822_size_count  = 0;
}

S_(mbx_free_rs_fields_folder mbx_free_rs_fields_pop)
static void mbx_free_rs_fields_pop P_((struct read_folder_state *rs));
static void mbx_free_rs_fields_pop(rs)
     struct read_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_free_rs_fields_pop: rs=%p\n",
		     rs));

    rs->a.pop_mbx.msg_num = 0;
    if (rs->a.pop_mbx.uidl_vector) {
	int i;
	for (i = 0; i < rs->a.pop_mbx.uidl_count; i++)
	    if (rs->a.pop_mbx.uidl_vector[i]) {
		free(rs->a.pop_mbx.uidl_vector[i]);
		rs->a.pop_mbx.uidl_vector[i] = NULL;
	    }
	free(rs->a.pop_mbx.uidl_vector);
	rs->a.pop_mbx.uidl_vector = NULL;
    }
    rs->a.pop_mbx.uidl_count  = 0;

    if (rs->a.pop_mbx.rfc822_size_vector) {
	int i;
	for (i = 0; i < rs->a.pop_mbx.rfc822_size_count; i++)	    	    
	    rs->a.pop_mbx.rfc822_size_vector[i] = -1;          
	free(rs->a.pop_mbx.rfc822_size_vector);
	rs->a.pop_mbx.rfc822_size_vector = NULL;
    }
    rs->a.pop_mbx.rfc822_size_count  = 0;
}

S_(mbx_zero_ks_fields_folder mbx_zero_ks_fields_pop)
static void mbx_zero_ks_fields_pop P_((struct keep_folder_state *rs));
static void mbx_zero_ks_fields_pop(rs)
     struct keep_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_zero_ks_fields_pop: rs=%p\n",
		     rs));
}

S_(mbx_free_ks_fields_folder mbx_free_ks_fields_pop)
static void mbx_free_ks_fields_pop P_((struct keep_folder_state *rs));
static void mbx_free_ks_fields_pop(rs)
     struct keep_folder_state *rs;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_free_ks_fields_pop: rs=%p\n",
		     rs));
}

S_(mbx_get_folder_mode mbx_get_pop_mode)
static int mbx_get_pop_mode P_((struct folder_info *folder));
static int mbx_get_pop_mode(folder)
     struct folder_info *folder;
{
    int ret = FOLDER_MBOX;

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__," mbx_get_pop_mode",
	      "Bad magic number (private_data)",0);
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_get_pop_mode: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));

    if (NULL == folder->p->a.pop_mbx.C.stream ||
	folder -> p->a.pop_mbx.pop_state  == POP_not_logged)
	ret |= FOLDER_DISCONNECTED;
	
    DPRINT(Debug,11,(&Debug,
		     "mbx_get_pop_mode=%d",
		     ret));
    if (ison(ret,FOLDER_MBOX))            { DPRINT(Debug,11,(&Debug," FOLDER_MBOX")); }
    if (ison(ret,FOLDER_RDONLY))          { DPRINT(Debug,11,(&Debug," FOLDER_RDONLY")); }
    if (ison(ret,FOLDER_FILE))            { DPRINT(Debug,11,(&Debug," FOLDER_FILE")); }
    if (ison(ret,FOLDER_DISCONNECTED))    { DPRINT(Debug,11,(&Debug," FOLDER_DISCONNECTED")); }    
    DPRINT(Debug,11,(&Debug,"\n"));
    
    return ret;
}

S_(mbx_give_folder_server mbx_give_pop_server)
const struct remote_server * mbx_give_pop_server 
    P_((struct folder_info *folder));
const struct remote_server * mbx_give_pop_server(folder)
     struct folder_info *folder;
{
    struct remote_server * ret;

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_give_pop_server",
	      "Bad magic number (private_data)",0);

    if (folder->p->a.pop_mbx.this_server)
	return folder->p->a.pop_mbx.this_server;

    ret = give_remote_server_from_folder(folder,POP_SERVER_TYPE,
					 folder->p->a.pop_mbx.passhm);
    
    folder->p->a.pop_mbx.this_server = ret;

    return ret;
}

S_(mbx_comp_prev_hdr_folder mbx_comp_prev_hdr_pop)
static enum comp_prev_hdr_result mbx_comp_prev_hdr_pop
    P_((struct folder_info * folder,
	struct header_rec  * entry,
	struct header_rec  * prev_entry,
	RECONNECT_MODE    reconnect_ptr));
enum comp_prev_hdr_result mbx_comp_prev_hdr_pop(folder,entry,prev_entry,
						    reconnect_ptr)
     struct folder_info * folder;
     struct header_rec  * entry;
     struct header_rec  * prev_entry;
     RECONNECT_MODE    reconnect_ptr;
{
    enum comp_prev_hdr_result ret = comp_prev_hdr_pass;

    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_comp_prev_hdr_pop",
	      "Bad magic number (private_data)",0);

    if (entry->mbx_info &&
	&pop_mbx_info == entry->mbx_info->type_code &&
	prev_entry->mbx_info &&
	&pop_mbx_info == prev_entry->mbx_info->type_code
	) {

	if (entry->mbx_info->a.pop_mbx.uidl &&
	    prev_entry->mbx_info->a.pop_mbx.uidl) {

	    if (0 == strcmp(entry->mbx_info->a.pop_mbx.uidl,
			    prev_entry->mbx_info->a.pop_mbx.uidl)) {
		DPRINT(Debug,12,(&Debug,
				 "mbx_comp_prev_hdr_pop: UIDL %s matches\n",
				 entry->mbx_info->a.pop_mbx.uidl));
		ret = comp_prev_hdr_found;
	    } else {
		DPRINT(Debug,12,(&Debug,
				 "mbx_comp_prev_hdr_pop: entry UIDL %s != prev_entry UIDL %s\n",
				 entry->mbx_info->a.pop_mbx.uidl,
				 prev_entry->mbx_info->a.pop_mbx.uidl));
		ret = comp_prev_hdr_miss;
	    }
	}
    }
    
    DPRINT(Debug,12,(&Debug,"mbx_comp_prev_hdr_pop=%d",
		     ret));
    switch (ret) {
    case comp_prev_hdr_miss:  DPRINT(Debug,12,(&Debug," comp_prev_hdr_miss")); break;
    case comp_prev_hdr_pass:  DPRINT(Debug,12,(&Debug," comp_prev_hdr_pass")); break;
    case comp_prev_hdr_found: DPRINT(Debug,12,(&Debug," comp_prev_hdr_found")); break;
    }
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
}

S_(mbx_update_prev_hdr_folder mbx_update_prev_hdr_pop)
static void mbx_update_prev_hdr_pop  P_((struct folder_info * folder,
					 struct header_rec  * entry,
					 struct header_rec  * prev_entry,
					 RECONNECT_MODE    reconnect_mode_ptr,
					 READ_STATE        read_state_ptr
					 ));
static void mbx_update_prev_hdr_pop(folder,entry,prev_entry,
				    reconnect_mode_ptr,
				    read_state_ptr)
     struct folder_info * folder;
     struct header_rec  * entry;
     struct header_rec  * prev_entry;
     RECONNECT_MODE    reconnect_mode_ptr;
     READ_STATE        read_state_ptr;
{
    int ok = 0;

    /* Not really needed */
    
    DPRINT(Debug,12,(&Debug,"mbx_update_prev_hdr_pop\n"));

     if (entry->mbx_info &&
	 &pop_mbx_info == entry->mbx_info->type_code) {

	 int i = entry->mbx_info->a.pop_mbx.msg_num;

	 if (i > 0 && i <= read_state_ptr->a.pop_mbx.uidl_count &&
	     read_state_ptr->a.pop_mbx.uidl_vector[i-1] &&
	     entry->mbx_info->a.pop_mbx.uidl &&
	     0 == strcmp(read_state_ptr->a.pop_mbx.uidl_vector[i-1],
			 entry->mbx_info->a.pop_mbx.uidl)) {
	     DPRINT(Debug,12,(&Debug,
			      "mbx_update_prev_hdr_pop: hdr index #%d: Message %d have UIDL %s\n",
			      entry->index_number_X,
			      entry->mbx_info->a.pop_mbx.msg_num,
			      entry->mbx_info->a.pop_mbx.uidl));
	     ok = 1;	     
	 } else if (entry->mbx_info->a.pop_mbx.msg_num > 0) {
	     DPRINT(Debug,1,(&Debug,
			      "mbx_update_prev_hdr_pop: hdr index #%d: Message %d not match",
			      entry->index_number_X,
			      entry->mbx_info->a.pop_mbx.msg_num));
	     if (entry->mbx_info->a.pop_mbx.uidl) {
		 DPRINT(Debug,1,(&Debug,", UIDL %s",
				  entry->mbx_info->a.pop_mbx.uidl));
	     }

	      if (i > 0 && i <= read_state_ptr->a.pop_mbx.uidl_count &&
		  read_state_ptr->a.pop_mbx.uidl_vector[i-1]) {
		  DPRINT(Debug,1,(&Debug," != mbx %d UIDL %s",
				   i,read_state_ptr->a.pop_mbx.uidl_vector[i-1]));
	      }
	      DPRINT(Debug,1,(&Debug,"\n"));

	      entry->mbx_info->a.pop_mbx.msg_num = -1;
	 }

     }
     
     if (prev_entry->mbx_info &&
	 &pop_mbx_info == prev_entry->mbx_info->type_code &&
	 prev_entry->mbx_info->a.pop_mbx.uidl
	 ) {
	 
	 int i = prev_entry->mbx_info->a.pop_mbx.msg_num;
	 
	 if (i > 0 && i <= read_state_ptr->a.pop_mbx.uidl_count &&
	     read_state_ptr->a.pop_mbx.uidl_vector[i-1] &&
	     0 == strcmp(read_state_ptr->a.pop_mbx.uidl_vector[i-1],
			 prev_entry->mbx_info->a.pop_mbx.uidl)) {
	     
	     change_rec_mbx_info(entry,&pop_mbx_info);
	     
	     entry->mbx_info->a.pop_mbx.uidl =
		 strmcpy(entry->mbx_info->a.pop_mbx.uidl,
			 prev_entry->mbx_info->a.pop_mbx.uidl);
	     entry->mbx_info->a.pop_mbx.msg_num =
		 prev_entry->mbx_info->a.pop_mbx.msg_num;
	     
	     DPRINT(Debug,12,(&Debug,
			      "mbx_update_prev_hdr_pop: hdr index #%d: Setting UIDL %s message number %d\n",
			      entry->index_number_X,
			      entry->mbx_info->a.pop_mbx.uidl,
			      entry->mbx_info->a.pop_mbx.msg_num));
	 } else {
	     int message_num = -1;
	     int j;
	     
	     DPRINT(Debug,12,(&Debug,
			      "mbx_update_prev_hdr_pop: hdr index #%d: Scanning UIDL %s\n",
			      entry->index_number_X,prev_entry->mbx_info->a.pop_mbx.uidl));
	     
	     for (j = 1; j <=  read_state_ptr->a.pop_mbx.uidl_count; j++) {
		 if (read_state_ptr->a.pop_mbx.uidl_vector[j-1] &&
		     0 == strcmp(prev_entry->mbx_info->a.pop_mbx.uidl,
				 read_state_ptr->a.pop_mbx.uidl_vector[j-1])) {
		     message_num = j;
		     break;
		 }
	     }
	     
	     if (message_num > 0) {
		 change_rec_mbx_info(entry,&pop_mbx_info);
		 
		 entry->mbx_info->a.pop_mbx.uidl =
		     strmcpy(entry->mbx_info->a.pop_mbx.uidl,
			     prev_entry->mbx_info->a.pop_mbx.uidl);
		 entry->mbx_info->a.pop_mbx.msg_num = message_num;
		 
		 DPRINT(Debug,12,(&Debug,
				  "mbx_update_prev_hdr_pop: hdr index #%d: Setting UIDL %s message number %d\n",
				  entry->index_number_X,
				  entry->mbx_info->a.pop_mbx.uidl,
				  entry->mbx_info->a.pop_mbx.msg_num));
	     } else {
		 DPRINT(Debug,12,(&Debug,
				  "mbx_update_prev_hdr_pop: hdr index #%d: prev hdr index #%d, UIDL %s not found -- OK\n",
				  entry->index_number_X,
				  prev_entry->index_number_X,
				  prev_entry->mbx_info->a.pop_mbx.uidl));
		 
		 if (!ok) {
		     change_rec_mbx_info(entry,&pop_mbx_info);
		     
		     entry->mbx_info->a.pop_mbx.uidl =
			 strmcpy(entry->mbx_info->a.pop_mbx.uidl,
				 prev_entry->mbx_info->a.pop_mbx.uidl);
		     entry->mbx_info->a.pop_mbx.msg_num = -1;
		     
		     DPRINT(Debug,12,(&Debug,
				      "mbx_update_prev_hdr_pop: hdr index #%d: Setting UIDL %s message number %d (none)\n",
				      entry->index_number_X,
				      entry->mbx_info->a.pop_mbx.uidl,
				      entry->mbx_info->a.pop_mbx.msg_num));
		 }
	     }
	 }
     }
}

S_(mbx_give_message_msize_folder  mbx_give_message_msize_pop)
static int mbx_give_message_msize_pop P_((struct folder_info * folder,
					  struct header_rec  * entry,
					  unsigned long      * ret_size));
static int mbx_give_message_msize_pop(folder,entry,ret_size)
     struct folder_info * folder;
     struct header_rec  * entry;
     unsigned long      * ret_size;
{
    DPRINT(Debug,11,(&Debug,
		     "mbx_give_message_msize_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
        
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_give_message_msize_pop",
	      "Bad magic number (private_data)",0);
    
    /* Unsupported */

    return 0;

}

/* Return -1 if not available (local mail) */
S_(mbx_give_message_count_folder mbx_give_message_count_pop)
static int mbx_give_message_count_pop P_((struct folder_info * folder));
static int mbx_give_message_count_pop(folder)
     struct folder_info * folder;
{
    int ret;
    
    DPRINT(Debug,11,(&Debug,
		     "mbx_give_message_count_pop: folder=%p (%s)\n",
		     folder,folder->cur_folder_sys));
        
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_give_message_count_pop",
	      "Bad magic number (private_data)",0);

    ret = folder -> p->a.pop_mbx.stat_count;

    DPRINT(Debug,11,(&Debug,
		     "mbx_give_message_count_pop=%d\n",ret));

    return ret;
}

struct folder_type pop_mbx = { FOLDER_TYPE_magic, 
			       "POP",
                               mbx_close_pop,
			       mbx_lock_pop,
			       mbx_init_pop,
			       mbx_sessionlock_pop,
			       mbx_unlock_pop,
			       mbx_flush_pop,
			       mbx_ferror_pop,
			       mbx_prepare_read_pop,
			       mbx_end_read_pop,
			       mbx_copy_envelope_pop,
			       mbx_is_forwarded_pop,
			       mbx_copy_header_pop,
			       mbx_copy_body_pop,
			       mbx_copy_envelope_end_pop,
			       mbx_copy_envelope_reset_body_pop,
			       mbx_pop_to_fd,
			       mbx_new_mail_on_pop,
			       mbx_consider_remove_pop,
			       mbx_prepare_keep_pop,
			       mbx_end_keep_pop,
			       mbx_mark_keep_pop,
			       mbx_pop_type,
			       mbx_start_edit_pop,
			       mbx_end_edit_pop,
                               mbx_free_pop,
			       mbx_zero_rs_fields_pop,
			       mbx_free_rs_fields_pop,
			       mbx_zero_ks_fields_pop,
			       mbx_free_ks_fields_pop,
			       mbx_get_pop_mode,
			       mbx_give_pop_server,
			       mbx_comp_prev_hdr_pop,
			       mbx_update_prev_hdr_pop,
			       mbx_have_default_quota,
			       mbx_give_default_quotar_list,
			       mbx_give_message_msize_pop,
			       mbx_give_message_count_pop
};

struct remote_account * set_pop_connection(fh,X,passhm)
     struct folder_info *fh;
     struct remote_account *X;
     struct browser_passhm *passhm;
{
    if (fh->folder_type != &pop_mbx)
	panic("MBX PANIC",__FILE__,__LINE__,"set_pop_connection",
	      "Bad folder type",0);

    if (PRIVATE_DATA_magic != fh->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"set_pop_connection",
	      "Bad magic number (private_data)",0);

    clear_remote_account(&(fh->p->a.pop_mbx.C));
    fh->p->a.pop_mbx.C = *X;	    

    zero_remote_account(X);     /* Avoid double free */

    if (fh->p->a.pop_mbx.passhm)
	free_browser_passhm(& (fh->p->a.pop_mbx.passhm));  

    if (passhm &&
	browser_passhm_check_type(passhm, hmcon_pop)) {
	fh->p->a.pop_mbx.passhm = passhm;
	inc_browser_passhm_refcount(fh->p->a.pop_mbx.passhm);
    } else if (passhm) {
	DPRINT(Debug,10,(&Debug,"set_pop_connection: browser_passhm discarded\n"));	
    }

	    
    return &(fh->p->a.pop_mbx.C);    
}

/* increments refcount of passhm */
void pop_mbx_set_passhm(fh,passhm)
     struct folder_info *fh;
     struct browser_passhm * passhm;
{
    if (fh->folder_type != &pop_mbx)
	panic("MBX PANIC",__FILE__,__LINE__,"pop_mbx_set_passhm",
	      "Bad folder type",0);

    if (PRIVATE_DATA_magic != fh->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"pop_mbx_set_passhm",
	      "Bad magic number (private_data)",0);

    if (fh->p->a.pop_mbx.passhm)
	free_browser_passhm(& (fh->p->a.pop_mbx.passhm));  

    if (passhm &&
	browser_passhm_check_type(passhm, hmcon_pop)) {
	fh->p->a.pop_mbx.passhm = passhm;
	inc_browser_passhm_refcount(fh->p->a.pop_mbx.passhm);
    } else {
	if (passhm) {
	    DPRINT(Debug,10,(&Debug,"pop_mbx_set_passhm: browser_passhm discarded\n"));
	}
    }
}

const struct remote_account * give_pop_remote_account(fh)
     struct folder_info *fh;
{
    
    if (fh->folder_type != &pop_mbx)
	panic("MBX PANIC",__FILE__,__LINE__,"give_pop_remote_account",
	      "Bad folder type",0);

    if (PRIVATE_DATA_magic != fh->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"give_pop_remote_account",
	      "Bad magic number (private_data)",0);


    return &(fh->p->a.pop_mbx.C);
}
#endif

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
