static char rcsid[] = "@(#)$Id: quota.c,v 2.3 2024/06/23 07:38:36 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.3 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *       or Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "quota_imp.h"

DEBUG_VAR(Debug,__FILE__,"mbox");


struct mail_quota * malloc_mail_quota(quota_type)
     struct mail_quota_type * quota_type;
{
    struct mail_quota *ret =
	safe_zero_alloc(sizeof (*ret));

    ret->magic =  MAIL_QUOTA_magic;
    ret->refcount = 1;

    ret->quotarootl_list  = NULL;
    ret->quotarootl_count = 0;

    ret->quotaroot_list   = NULL;
    ret->quotaroot_count  = 0;

    ret->quotaitem_list   = NULL;
    ret->quotaitem_count  = 0;

    ret->p.dummy          = NULL;
       
    if (MAIL_QUOTA_TYPE_magic != quota_type->magic)
	  panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quota",
              "Bad magic number (mail_quota_type)",0);

    ret->quota_type = quota_type;

    ret->quota_type->mq_init_it(ret);
        
    return ret;
}

enum mq_freemode {
    mq_normal_free      = 0,
    mq_reset_mail_quota = 1
};

static void free_mail_quotarootl_internal P_((struct mail_quotaroot_list **quota_list,
					      enum mq_freemode reset_mail_quota));
static void free_mail_quotaroot_internal P_((struct mail_quotaroot **quotaroot,
					      enum mq_freemode reset_mail_quota));
static void free_mail_quota_item_internal P_((struct mail_quota_item **quota_item,
					      enum mq_freemode reset_mail_quota));

void free_mail_quota(quota)
     struct mail_quota **quota;
{
    if (MAIL_QUOTA_magic != (*quota)->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "free_mail_quota",
              "Bad magic number (mail_quota)",0);

    if ((*quota)->refcount < 1)
        panic("MBX PANIC",__FILE__,__LINE__,
              "free_mail_quota",
              "Bad refcount",0);
    
    (*quota)->refcount--;
    if ((*quota)->refcount > 0) {
        /* Just reset this reference */
        
        *quota = NULL;
        return;
    }
    
    if (MAIL_QUOTA_TYPE_magic != (*quota)->quota_type->magic)
	  panic("MBX PANIC",__FILE__,__LINE__,
              "free_mail_quota",
              "Bad magic number (mail_quota_type)",0);

    (*quota)->quota_type->mq_dest_it(*quota);

    if ((*quota)->quotarootl_list) {
	size_t i;

	for (i = 0; i < (*quota)->quotarootl_count; i++) {

	    if ((*quota)->quotarootl_list[i]) {
		if (MAIL_QUOTAROOT_LIST_magic != (*quota)->quotarootl_list[i]->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "free_mail_quota",
			  "Bad magic number (mail_quotaroot_list)",0);
		
		if (*quota != (*quota)->quotarootl_list[i]->mail_quota)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "free_mail_quota",
			  "Bad mail_quota link (mail_quotaroot_list)",0);

		/* This is refcounting link
		 */
		free_mail_quotarootl_internal(&((*quota)->quotarootl_list[i]),
					      mq_reset_mail_quota);
	    }
	}

	free((*quota)->quotarootl_list);
	(*quota)->quotarootl_list = NULL;
    }
    (*quota)->quotarootl_count = 0;

    if ((*quota)->quotaroot_list) {
	size_t i;

	for (i = 0; i < (*quota)->quotaroot_count; i++) {

	    if ((*quota)->quotaroot_list[i]) {
		if (MAIL_QUOTAROOT_magic != (*quota)->quotaroot_list[i]->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "free_mail_quota",
			  "Bad magic number (mail_quotaroot)",0);

		/* This is refcounting link
		 */
		free_mail_quotaroot_internal(& ((*quota)->quotaroot_list[i]),
					     mq_reset_mail_quota);
	    }
	}

	free((*quota)->quotaroot_list);
	(*quota)->quotaroot_list = NULL;
    }
    (*quota)->quotaroot_count = 0;

    if ((*quota)->quotaitem_list) {
	size_t i;

	for (i = 0; i < (*quota)->quotaitem_count; i++) {
	    if ((*quota)->quotaitem_list[i]) {

		if (MAIL_QUOTA_ITEM_magic != (*quota)->quotaitem_list[i]->magic)
		     panic("MBX PANIC",__FILE__,__LINE__,
			  "free_mail_quota",
			  "Bad magic number (mail_quota_item)",0);
		
		/* Reset backlink -- this is not refcounting link */
		free_mail_quota_item_internal(&  ((*quota)->quotaitem_list[i]),
					      mq_reset_mail_quota);
	    }
	}

	free((*quota)->quotaitem_list);
	(*quota)->quotaitem_list = NULL;
    }
    (*quota)->quotaitem_count = 0;

    if ((*quota)->p.dummy) {
	/* Should not be set -- this may leak memory */
	
	DPRINT(Debug,1,(&Debug, "free_mail_quota: (*quota)->p.dummy != NULL -- freeing !\n"));

	free((*quota)->p.dummy);
	(*quota)->p.dummy = NULL;
    }
    
    (*quota)->magic = 0; /* Invalidate */
    
  free(*quota);
  *quota = NULL;
  
}



void inc_mail_quota_refcount(quota)
     struct mail_quota *quota;
{
    if (MAIL_QUOTA_magic != quota->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "inc_mail_quota_refcount",
              "Bad magic number (mail_quota)",0);

    quota->refcount++;
}



struct mail_quotaroot_list * malloc_mail_quotaroot_list(mail_quota)
     struct mail_quota           * mail_quota;
{
    struct mail_quotaroot_list *ret =
	safe_zero_alloc(sizeof (*ret));
    size_t quotarootl_list_pos;
    size_t new_quotarootl_count;
    
    if (MAIL_QUOTA_magic != mail_quota->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quotaroot_list",
              "Bad magic number (mail_quota)",0);

    ret->magic = MAIL_QUOTAROOT_LIST_magic;
    ret->refcount = 2; /* 1 for returned pointer +
			  1 for mail_quota->quotarootl_list[] pointer
		       */
    ret->mail_quota = mail_quota;
    
    if (MAIL_QUOTA_TYPE_magic != mail_quota->quota_type->magic)
	  panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quotaroot_list",
              "Bad magic number (mail_quota_type)",0);
    ret->quota_type = mail_quota->quota_type;

    ret->p.dummy          = NULL;

    if (mail_quota->quotarootl_list) {
	size_t i;

	for (i = 0; i < mail_quota->quotarootl_count; i++) {

	    if (! mail_quota->quotarootl_list[i]) {
		quotarootl_list_pos = i;

		goto found_free_pos;
	    }
	}
    }

    new_quotarootl_count = mail_quota->quotarootl_count+1;

    mail_quota->quotarootl_list =
	safe_array_realloc(mail_quota->quotarootl_list,
			   new_quotarootl_count,
			   sizeof(mail_quota->quotarootl_list[0]));

    quotarootl_list_pos = mail_quota->quotarootl_count;
    mail_quota->quotarootl_list[quotarootl_list_pos] = NULL;
    mail_quota->quotarootl_count = new_quotarootl_count;
    
 found_free_pos:

    if (quotarootl_list_pos >= mail_quota->quotarootl_count ||
	mail_quota->quotarootl_list[quotarootl_list_pos])
	  panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quotaroot_list",
              "Bad quotarootl_list_pos",0);

    mail_quota->quotarootl_list[quotarootl_list_pos] = ret;
        
    ret->quota_type->mq_init_mqrl_it(mail_quota,ret);
    
    return ret;
}


static void free_mail_quotarootl_internal(quota_list,reset_mail_quota)
     struct mail_quotaroot_list **quota_list;
     enum mq_freemode reset_mail_quota;
{
    if (MAIL_QUOTAROOT_LIST_magic != (*quota_list)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quotarootl_internal",
	      "Bad magic number (mail_quotaroot_list)",0);

    if ((*quota_list)->refcount < 1)
        panic("MBX PANIC",__FILE__,__LINE__,
              "free_mail_quotarootl_internal",
              "Bad refcount",0);
    
    switch (reset_mail_quota) {
    case mq_normal_free: break;
    case  mq_reset_mail_quota:
	if ((*quota_list)->mail_quota) {

	    if (MAIL_QUOTA_magic != (*quota_list)->mail_quota->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "free_mail_quotarootl_internal",
		      "Bad magic number (mail_quota)",0);
	    
	    if (MAIL_QUOTA_TYPE_magic != (*quota_list)->quota_type->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "free_mail_quotarootl_internal",
		      "Bad magic number (mail_quota_type)",0);

	    (*quota_list)->quota_type->mq_disc_mqrl_it((*quota_list)->mail_quota,
						       *quota_list);
	    
	    (*quota_list)->mail_quota = NULL;
	} else
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "free_mail_quotarootl_internal",
		  "mail_quota link not set (mail_quotaroot_list)",0);
	break;
    }

    (*quota_list)->refcount--;
    if ((*quota_list)->refcount > 0) {
        /* Just reset this reference */
        
        *quota_list = NULL;
        return;
    }

    /* if struct mail_quota have link to here,
       refcount can not be 0
    */
    if ((*quota_list)->mail_quota) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quotarootl_internal",
	      "mail_quota link set (mail_quotaroot_list)",0);

    if (MAIL_QUOTA_TYPE_magic != (*quota_list)->quota_type->magic)
	  panic("MBX PANIC",__FILE__,__LINE__,
		"free_mail_quotarootl_internal",
              "Bad magic number (mail_quota_type)",0);

    (*quota_list)->quota_type->mq_dest_mqrl_it(*quota_list);


    if ((*quota_list)->p.dummy) {
	/* Should not be set -- this may leak memory */
	
	DPRINT(Debug,1,(&Debug,
			"free_mail_quotarootl_internal: (*quota_list)->p.dummy != NULL -- freeing !\n"));

	free((*quota_list)->p.dummy);
        (*quota_list)->p.dummy = NULL;
    }
       
    (*quota_list)->magic = 0; /* Invalidate */
    
    
    free(*quota_list);
    *quota_list = NULL;
}

void free_mail_quotaroot_list(quota_list)
     struct mail_quotaroot_list **quota_list;
{
    free_mail_quotarootl_internal(quota_list,mq_normal_free);
}

/* Returns 0 if  associated struct mail_quota is deallocated */
size_t quotaroot_list_len(quota_list)
     struct mail_quotaroot_list *quota_list;
{

    if (MAIL_QUOTAROOT_LIST_magic != quota_list->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quotaroot_list_len",
	      "Bad magic number (mail_quotaroot_list)",0);

    if (quota_list->mail_quota) {

	if (MAIL_QUOTA_magic != quota_list->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_list_len",
		  "Bad magic number (mail_quota)",0);
	    
	if (MAIL_QUOTA_TYPE_magic != quota_list->quota_type->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_list_len",
		  "Bad magic number (mail_quota_type)",0);

	return quota_list->quota_type->mq_listlen_mqrl_it(quota_list->mail_quota,
							  quota_list);
    }
    
    return 0;
}

/* Increments refcount 
   Returns NULL if  associated struct mail_quota is deallocated */

struct mail_quota *quotaroot_list_mquota(quota_list)
     struct mail_quotaroot_list *quota_list;
{
    if (MAIL_QUOTAROOT_LIST_magic != quota_list->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quotaroot_list_mquota",
	      "Bad magic number (mail_quotaroot_list)",0);

    if (quota_list->mail_quota) {
	struct mail_quota *ret = NULL;
	
	if (MAIL_QUOTA_magic != quota_list->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_list_len",
		  "Bad magic number (mail_quota)",0);

	ret = quota_list->mail_quota;
	inc_mail_quota_refcount(ret);

	return ret;
    }
    
    return NULL;
}

void inc_mail_quotaroot_list_refcount(quota_list)
     struct mail_quotaroot_list *quota_list;
{
    if (MAIL_QUOTAROOT_LIST_magic != quota_list->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "inc_mail_quotaroot_list_refcount",
	      "Bad magic number (mail_quotaroot_list)",0);
    
    quota_list->refcount++;
}



/* Increments refcount 
   Returns NULL if  associated struct mail_quota is deallocated */

struct mail_quotaroot *quotaroot_list_item(quota_list,idx,cd)
     struct mail_quotaroot_list *quota_list;
     size_t idx;
     struct cancel_data  * cd;
{
    if (MAIL_QUOTAROOT_LIST_magic != quota_list->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quotaroot_list_item",
	      "Bad magic number (mail_quotaroot_list)",0);

    if (quota_list->mail_quota) {
	size_t len;

	if (MAIL_QUOTA_magic != quota_list->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_list_item",
		  "Bad magic number (mail_quota)",0);
	    
	if (MAIL_QUOTA_TYPE_magic != quota_list->quota_type->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_list_item",
		  "Bad magic number (mail_quota_type)",0);

	len = quota_list->quota_type->mq_listlen_mqrl_it(quota_list->mail_quota,
							 quota_list);

	if (idx >= len)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_list_item",
		  "Bad index",0);
	    
	return quota_list->quota_type->mq_listitem_mqrl_it(quota_list->mail_quota,
							   quota_list,idx,cd);
    }
    
    return NULL;
}

void quotaroot_list_set_refresh(quota_list,refresh)
     struct mail_quotaroot_list *quota_list;
     enum quota_mode refresh;
{
    if (MAIL_QUOTAROOT_LIST_magic != quota_list->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quotaroot_list_set_refresh",
	      "Bad magic number (mail_quotaroot_list)",0);

    if (quota_list->mail_quota) {
	
	if (MAIL_QUOTA_magic != quota_list->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_list_set_refresh",
		  "Bad magic number (mail_quota)",0);
	
	if (MAIL_QUOTA_TYPE_magic != quota_list->quota_type->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_list_set_refresh",
		  "Bad magic number (mail_quota_type)",0);

	quota_list->quota_type->mq_refresh_mqrl_it(quota_list->mail_quota,
						   quota_list,refresh);
    }    
}

/* Previous quotaroot with same is removed from list */

struct mail_quotaroot * malloc_mail_quotaroot(mail_quota,name,name_token)
     struct mail_quota             * mail_quota;
     struct string                 * name /* may be NULL */;
     enum quota_reserved_quotaroot   name_token;
{
    struct mail_quotaroot * ret =
	safe_zero_alloc(sizeof (*ret));
    size_t quotaroot_list_pos;
    size_t new_quotaroot_count;

    if (MAIL_QUOTA_magic != mail_quota->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quotaroot",
              "Bad magic number (mail_quota)",0);

    ret->magic = MAIL_QUOTAROOT_magic;
    ret->refcount = 2; /* 1 for returned pointer +
			  1 for mail_quota->quotaroot_list[] pointer
		       */
    ret->mail_quota = mail_quota;
    
    if (MAIL_QUOTA_TYPE_magic != mail_quota->quota_type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quotaroot",
              "Bad magic number (mail_quota_type)",0);
    ret->quota_type = mail_quota->quota_type;
    if (name)
	ret->name       = dup_string(name);
    else
	ret->name       = NULL;
    ret->name_token = name_token;

    ret->p.dummy = NULL;
    
    if (mail_quota->quotaroot_list) {
	size_t i;

	/* Remove previous with same name */
	
	for (i = 0; i < mail_quota->quotaroot_count; i++) {

	    if (mail_quota->quotaroot_list[i]) {

		if (MAIL_QUOTAROOT_magic != mail_quota->quotaroot_list[i]->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,
			  "malloc_mail_quotaroot",
			  "Bad magic number (mail_quotaroot)",0);
		
		if (mail_quota->quotaroot_list[i]->name_token == name_token &&
		    ! mail_quota->quotaroot_list[i]->name &&
		    ! name) {

		    free_mail_quotaroot_internal(&(mail_quota->quotaroot_list[i]),
						 mq_reset_mail_quota);		    
		    
		} else if (mail_quota->quotaroot_list[i]->name_token == name_token &&
		    mail_quota->quotaroot_list[i]->name &&
		    name &&
		    0 == string_cmp(mail_quota->quotaroot_list[i]->name,
				    name,
				    -999 /* not comparable */)) {		    
		    free_mail_quotaroot_internal(&(mail_quota->quotaroot_list[i]),
						 mq_reset_mail_quota);		    
		}
	    }
	}

	for (i = 0; i < mail_quota->quotaroot_count; i++) {

	    if (! mail_quota->quotaroot_list[i]) {
		quotaroot_list_pos = i;

		goto found_free_pos;
	    }
	}
    }

    new_quotaroot_count = mail_quota->quotaroot_count+1;
    mail_quota->quotaroot_list =
	safe_array_realloc(mail_quota->quotaroot_list,
			   new_quotaroot_count,
			   sizeof(mail_quota->quotaroot_list[0]));

    quotaroot_list_pos = mail_quota->quotaroot_count;
    mail_quota->quotaroot_list[quotaroot_list_pos] = NULL;
    mail_quota->quotaroot_count = new_quotaroot_count;

 found_free_pos:

    if (quotaroot_list_pos >= mail_quota->quotaroot_count ||
	mail_quota->quotaroot_list[quotaroot_list_pos])
	panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quotaroot",
              "Bad quotaroot_list_pos",0);

    mail_quota->quotaroot_list[quotaroot_list_pos] = ret;
    
    ret->quota_type->mq_init_mqr_it(mail_quota,ret);

    DPRINT(Debug,20,(&Debug,"malloc_mail_quotaroot=%p; refcount=%d, index #%zu\n",
		     ret,ret->refcount,quotaroot_list_pos));
    
    return ret;
}

static void free_mail_quotaroot_internal(quotaroot,reset_mail_quota)
     struct mail_quotaroot **quotaroot;
     enum mq_freemode reset_mail_quota;
{
    if (MAIL_QUOTAROOT_magic != (*quotaroot)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quotaroot_internal",
	      "Bad magic number (mail_quotaroot)",0);

    if ((*quotaroot)->refcount < 1)
        panic("MBX PANIC",__FILE__,__LINE__,
              "free_mail_quotaroot_internal",
              "Bad refcount",0);
    
    switch (reset_mail_quota) {
    case mq_normal_free: break;
    case  mq_reset_mail_quota:
	if ((*quotaroot)->mail_quota) {

	    if (MAIL_QUOTA_magic != (*quotaroot)->mail_quota->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "free_mail_quotaroot_internal",
		      "Bad magic number (mail_quota)",0);

	    if (MAIL_QUOTA_TYPE_magic != (*quotaroot)->quota_type->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "free_mail_quotaroot_internal",
		      "Bad magic number (mail_quota_type)",0);
	    
	    (*quotaroot)->quota_type-> mq_disc_mqr_it((*quotaroot)->mail_quota,
						      *quotaroot);
	    (*quotaroot)->mail_quota = NULL;
	} else
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "free_mail_quotaroot_internal",
		  "mail_quota link not set (mail_quotaroot)",0);
	break;
    }

    (*quotaroot)->refcount--;
    if ((*quotaroot)->refcount > 0) {
        /* Just reset this reference */
        
        *quotaroot = NULL;
        return;
    }

    /* if struct mail_quota have link to here,
       refcount can not be 0
    */
    if ((*quotaroot)->mail_quota) 
	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quotaroot_internal",
	      "mail_quota link set (mail_quotaroot)",0);
    
    if (MAIL_QUOTA_TYPE_magic != (*quotaroot)->quota_type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quotaroot_internal",
	      "Bad magic number (mail_quota_type)",0);

    (*quotaroot)->quota_type->mq_dest_mqr_it(*quotaroot);

    if ((*quotaroot)->name)
	free_string(& ((*quotaroot)->name));
    
    if ((*quotaroot)->p.dummy) {
	/* Should not be set -- this may leak memory */

	DPRINT(Debug,1,(&Debug,
			"free_mail_quotaroot_internal: (*quotaroot)->p.dummy != NULL  -- freeing !\n"));

	free ((*quotaroot)->p.dummy);
	(*quotaroot)->p.dummy  = NULL;
    }
    
    (*quotaroot)->magic = 0; /* Invalidate */
    free(*quotaroot);
    *quotaroot = NULL;

}



void free_mail_quotaroot(quotaroot)
     struct mail_quotaroot **quotaroot;
{

    DPRINT(Debug,20,(&Debug,
		     "free_mail_quotaroot: [%p]\n",
		     *quotaroot));

    free_mail_quotaroot_internal(quotaroot,mq_normal_free);
}

void inc_mail_quotaroot_refcount(quotaroot) 
     struct mail_quotaroot *quotaroot;
{
    if (MAIL_QUOTAROOT_magic != quotaroot->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "inc_mail_quotaroot_refcount",
	      "Bad magic number (mail_quotaroot)",0);

    quotaroot->refcount++;

    DPRINT(Debug,20,(&Debug,
		     "inc_mail_quotaroot_refcount: [%p] refcount => %d\n",
		     quotaroot,quotaroot->refcount));
}

/* Caller must call free_string */
struct string *quotaroot_name(quotaroot,tok_r)
     struct mail_quotaroot *quotaroot;
     enum quota_reserved_quotaroot *tok_r /* return value */;
{
    if (MAIL_QUOTAROOT_magic != quotaroot->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quotaroot_name",
	      "Bad magic number (mail_quotaroot)",0);
    
    if (tok_r)
	*tok_r = quotaroot->name_token;

    if (quotaroot->name) {
	return dup_string(quotaroot->name);
    }

    return NULL;
}

size_t quotaroot_item_count(quotaroot)
     struct mail_quotaroot *quotaroot;
{
    if (MAIL_QUOTAROOT_magic != quotaroot->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quotaroot_item_count",
	      "Bad magic number (mail_quotaroot)",0);

    if (quotaroot->mail_quota) {
	
	if (MAIL_QUOTA_magic != quotaroot->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_item_count",
		  "Bad magic number (mail_quota)",0);
	    
	if (MAIL_QUOTA_TYPE_magic != quotaroot->quota_type->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_item_count",
		  "Bad magic number (mail_quota_type)",0);

	return  quotaroot->quota_type->mq_itemcount_mqr_it(quotaroot->mail_quota,
							   quotaroot);
    }

    return 0;
}

struct mail_quota_item *quotaroot_item(quotaroot,idx)
     struct mail_quotaroot *quotaroot;
     size_t idx;
{
    if (MAIL_QUOTAROOT_magic != quotaroot->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quotaroot_item",
	      "Bad magic number (mail_quotaroot)",0);

    if (quotaroot->mail_quota) {
	size_t count;
	
	if (MAIL_QUOTA_magic != quotaroot->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_item",
		  "Bad magic number (mail_quota)",0);
	    
	if (MAIL_QUOTA_TYPE_magic != quotaroot->quota_type->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_item",
		  "Bad magic number (mail_quota_type)",0);

	count = quotaroot->quota_type->mq_itemcount_mqr_it(quotaroot->mail_quota,
							   quotaroot);

	if (idx >= count) 
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_item",
		  "Bad magic number index",0);

	return quotaroot->quota_type->mq_item_mqr_it(quotaroot->mail_quota,
						     quotaroot,idx);
	
    }

    return NULL;
}

enum quota_quotaroot_status quotaroot_status(quotaroot)
     struct mail_quotaroot *quotaroot;
{
    if (MAIL_QUOTAROOT_magic != quotaroot->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quotaroot_status",
	      "Bad magic number (mail_quotaroot)",0);

    if (quotaroot->mail_quota) {
	if (MAIL_QUOTA_magic != quotaroot->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_status",
		  "Bad magic number (mail_quota)",0);
	    
	if (MAIL_QUOTA_TYPE_magic != quotaroot->quota_type->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quotaroot_status",
		  "Bad magic number (mail_quota_type)",0);

	return quotaroot->quota_type->mq_status_mqr_it(quotaroot->mail_quota,
						       quotaroot);
    }

    return qr_stat_none;
}

struct mail_quota_item *  malloc_mail_quota_item(mail_quota,name,name_token)
     struct mail_quota * mail_quota;
     struct string     * name      /* May be NULL */;
     enum quota_reserved_resource name_token;
{
    struct mail_quota_item * ret =
	safe_zero_alloc(sizeof (*ret));
    size_t quotaitem_list_pos;
    size_t new_quotaitem_count;

      if (MAIL_QUOTA_magic != mail_quota->magic)
        panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quota_item",
              "Bad magic number (mail_quota)",0);
             
    ret->magic = MAIL_QUOTA_ITEM_magic;
    ret->refcount = 1;   /* mail_quota->quotaitem_list[] not counted */
    ret->mail_quota = mail_quota;
    if (name)
	ret->name       = dup_string(name);
    else
	ret->name       = NULL;
    ret->name_token = name_token;

    ret->p.dummy    = NULL;
    
    if (MAIL_QUOTA_TYPE_magic != mail_quota->quota_type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quota_item",
              "Bad magic number (mail_quota_type)",0);
    ret->quota_type = mail_quota->quota_type;

    if (mail_quota->quotaitem_list) {
	size_t i;

	for (i = 0; i < mail_quota->quotaitem_count; i++) {
	    if (! mail_quota->quotaitem_list[i]) {		
		quotaitem_list_pos = i;

		goto found_free_pos;
	    }
	}	
    }

    new_quotaitem_count = mail_quota->quotaitem_count+1;
    mail_quota->quotaitem_list =
	safe_array_realloc(mail_quota->quotaitem_list,
			   new_quotaitem_count,
			   sizeof (mail_quota->quotaitem_list[0]));
    
    quotaitem_list_pos = mail_quota->quotaitem_count;
    mail_quota->quotaitem_list[quotaitem_list_pos] = NULL;
    mail_quota->quotaitem_count = new_quotaitem_count;
    
 found_free_pos:
    if (quotaitem_list_pos >=  mail_quota->quotaitem_count ||
	mail_quota->quotaitem_list[quotaitem_list_pos])
	panic("MBX PANIC",__FILE__,__LINE__,
              "malloc_mail_quota_item",
              "Bad quotaitem_list_pos",0);

    mail_quota->quotaitem_list[quotaitem_list_pos] = ret;
    
    ret->quota_type->mq_init_mqi_it(mail_quota,ret);
    
    return ret;
}

void inc_mail_quota_item_refcount(quota_item)
     struct mail_quota_item *quota_item;
{
    if (MAIL_QUOTA_ITEM_magic != quota_item->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quota_item_internal",
	      "Bad magic number (mail_quota_item)",0);

    quota_item->refcount++;
}



static void free_mail_quota_item_internal(quota_item,reset_mail_quota)
     struct mail_quota_item **quota_item;
     enum mq_freemode reset_mail_quota;
{

    if (MAIL_QUOTA_ITEM_magic != (*quota_item)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quota_item_internal",
	      "Bad magic number (mail_quota_item)",0);

    if ((*quota_item)->refcount < 1)
        panic("MBX PANIC",__FILE__,__LINE__,
              "free_mail_quota_internal",
              "Bad refcount",0);
       
    switch (reset_mail_quota) {
    case mq_normal_free:    break;
    case mq_reset_mail_quota:

	/* mail_quota is going to be released.
	   Disconnect linking between mail_quota and quota_item 
	*/
	
	if ((*quota_item)->mail_quota) {

	    if (MAIL_QUOTA_magic != (*quota_item)->mail_quota->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "free_mail_quota_item_internal",
		      "Bad magic number (mail_quota)",0);

	    if (MAIL_QUOTA_TYPE_magic != (*quota_item)->quota_type->magic)
		panic("MBX PANIC",__FILE__,__LINE__,
		      "free_mail_quota_item_internal",
		      "Bad magic number (mail_quota_type)",0);
	    
	     (*quota_item)->quota_type->mq_disc_mqi_it((*quota_item)->mail_quota,
						       *quota_item);
	     
	     (*quota_item)->mail_quota = NULL;
	} else
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "free_mail_quota_item_internal",
		  "mail_quota link not set (mail_quota_item)",0);
	    
	    
	/* not refounting -- so no free */
	*quota_item = NULL;
	return;	
    }


    (*quota_item)->refcount--;
    if ((*quota_item)->refcount > 0) {
        /* Just reset this reference */
        
        *quota_item = NULL;
        return;
    }

    if (MAIL_QUOTA_TYPE_magic != (*quota_item)->quota_type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quota_item_internal",
	      "Bad magic number (mail_quota_type)",0);

    if ((*quota_item)->mail_quota) {
	size_t i;

	/* Disconnect linking between mail_quota and quota_item */
	
	if (MAIL_QUOTA_magic != (*quota_item)->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "free_mail_quota_item_internal",
		  "Bad magic number (mail_quota)",0);
		
	/* Remove from mail_quota->quotaitem_list */

	for (i = 0; i < (*quota_item)->mail_quota->quotaitem_count; i++) {
	    if ((*quota_item)->mail_quota->quotaitem_list[i]) {

		if (MAIL_QUOTA_ITEM_magic != (*quota_item)->mail_quota->quotaitem_list[i]->magic)
		     panic("MBX PANIC",__FILE__,__LINE__,
			  "free_mail_quota_item_internal",
			  "Bad magic number (quotaitem_list... mail_quota_item)",0);

		if ((*quota_item) == (*quota_item)->mail_quota->quotaitem_list[i]) {
		    
		    (*quota_item)->mail_quota->quotaitem_list[i] = NULL;
		    goto item_found;
		}
	    }
	}

	panic("MBX PANIC",__FILE__,__LINE__,
	      "free_mail_quota_item_internal",
	      "*quota_item not found from quotaitem_list",0);
	
    item_found:

	(*quota_item)->quota_type->mq_disc_mqi_it((*quota_item)->mail_quota,
						  *quota_item);
	
	(*quota_item)->mail_quota = NULL;

    }
    
    (*quota_item)->quota_type->mq_dest_mqi_it(*quota_item);

    if ((*quota_item)->name)
	free_string(& ((*quota_item)->name));
    
    if ((*quota_item)->p.dummy) {
	/* Should not be set -- this may leak memory */

	DPRINT(Debug,1,(&Debug,
			"free_mail_quota_item_internal: ((*quota_item)->p.dummy != NULL  -- freeing !\n"));

	free((*quota_item)->p.dummy);
	(*quota_item)->p.dummy = NULL;
    }
    
    (*quota_item)->magic = 0; /* Invalidate */
    
    free(*quota_item);
    *quota_item = NULL;    
}

void free_mail_quota_item(quota_item)
     struct mail_quota_item **quota_item;
{
    free_mail_quota_item_internal(quota_item,mq_normal_free);
}


/* Caller must free_string() result */

struct string *quota_item_name(quota_item,tok_r)
     struct mail_quota_item *quota_item;
     enum quota_reserved_resource *tok_r /* return value */;
{
    if (MAIL_QUOTA_ITEM_magic != quota_item->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quota_item_name",
	      "Bad magic number (mail_quota_item)",0);


    if (tok_r)
	*tok_r = quota_item->name_token;

    if (quota_item->name) {
	return dup_string(quota_item->name);;
    }
    
    return NULL;
}

int quota_item_values(quota_item, usage_r, limit_r)
     struct mail_quota_item *quota_item;
     uint32 * usage_r /* return value */;
     uint32 * limit_r /* return value */;
{
    if (usage_r)
	*usage_r = 0;
    if (limit_r)
	*limit_r = 0;

    if (MAIL_QUOTA_ITEM_magic != quota_item->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "quota_item_values",
	      "Bad magic number (mail_quota_item)",0);
    
    if (quota_item->mail_quota) {
	
	if (MAIL_QUOTA_magic != quota_item->mail_quota->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quota_item_values",
		  "Bad magic number (mail_quota)",0);

	if (MAIL_QUOTA_TYPE_magic != quota_item->quota_type->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,
		  "quota_item_values",
		  "Bad magic number (mail_quota_type)",0);

	return quota_item->quota_type->mq_values_mqi_it(quota_item->mail_quota,
							quota_item,
							usage_r,limit_r);
    }
    
    return 0;
}

void mail_quota_unlink_con(quota,con)
     struct mail_quota *quota;
     struct connection_cache *con;
{
    if (MAIL_QUOTA_magic != quota->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "mail_quota_unlink_con",
	      "Bad magic number (mail_quota)",0);

    if (MAIL_QUOTA_TYPE_magic != quota->quota_type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,
	      "mail_quota_unlink_con",
	      "Bad magic number (mail_quota_type)",0);

    quota->quota_type->mq_unlink_con_it(quota,con);
}


/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
