static char rcsid[] = "@(#)$Id: remote_mbx.c,v 2.15 2022/08/04 05:53:03 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.15 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> (was hurtta+elm@posti.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/

#include "def_mbox.h"
#include "ss_imp.h"
#ifdef USE_DLOPEN
#include "shared_imp.h"
#endif
#include "s_me.h"
#include "s_elm.h"

DEBUG_VAR(Debug,__FILE__,"net");


#ifdef REMOTE_MBX

/* Seems that h_errno is macro on AIX */
#ifndef h_errno
extern int h_errno;
#endif

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif

static unsigned char *s2us P_((char *str));
static unsigned char *s2us(str) 
     char *str;
{
    return (unsigned char *)str;
}


void set_remote_tempmbox P_((struct folder_info *fh,
			     struct remote_account *X));
void set_remote_tempmbox(fh,X)
     struct folder_info *fh;
     struct remote_account *X;
{
    char * prefix = "mbox.";
    const char * tmp = give_dt_estr_as_str(&temp_dir_e,"tmpdir",NULL,NULL);

    static int counter = 0;

    /* In here we can use temp_dir instead of default_temp because
     * that file does not act as session lock ...
     */


    if (X->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "set_remote_tempmbox",
	      "Bad magic number",0);
    
    if (!tmp)
	tmp = "/tmp/";

    elm_sfprintf(fh-> cur_tempfolder,
		 sizeof fh -> cur_tempfolder,
		 FRM("%s%s%s.%s.%d.%d"),
		 tmp,prefix,username,X->host,getpid(),counter++);

    DPRINT(Debug,12,(&Debug,
		     "set_remote_tempmbox: cur_tempfolder=%s\n",
		     fh-> cur_tempfolder));
}

/* May free X (if succeed) or copy
   If make copy of X, then X is zeroed 
 */
int make_remote_mbox(fh,X,se,rest,rewrite,passhm,main_cancel)
     struct folder_info *fh; 
     struct remote_account *X;
     struct service_entry *se;
     char * rest;
     int rewrite;
     struct browser_passhm * passhm;
     struct cancel_data *main_cancel /* Used if dns lookup was
					cancelable 
				     */;
{
    struct remote_account *status = NULL;

    char * imap_only = NULL;  /* Flag and name of imap folder ... */

    const struct service_type      * got_type = NULL;
    int                              got_port = 0;  /* port number */

    static struct connection_cache * CX = NULL;
    struct service_entry           * result_se  = NULL;
    
    if (X->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "make_remote_mbox",
	      "Bad magic number",0);

    DPRINT(Debug,10,(&Debug,
		     "make_remote_mbox: fh=%X, rest=%s, rewrite=%d\n",
		     fh,rest ? rest : "<NULL>", rewrite));

    if (rewrite) {
	/* canonify folder name .... */
	if (rest) {
	    /* Inbox is case insensitive */
	    if (0 == istrcmp(":inbox",rest))
		rest = ":INBOX";
	    
	    imap_only = safe_strdup(rest);
	    free(fh->cur_folder_sys);
	    fh->cur_folder_sys = elm_message(FRM("%s@%s%s"),
					     X->username,
					     se->official_name,
					     imap_only);
 
	    /* No support of international folder names on here! */
	    free_string(&(fh->cur_folder_disp));
	    fh->cur_folder_disp = format_string(FRM("%s@%s%s"),
						X->username,
						se->official_name,
						imap_only); 
	    
	    rest = NULL; /* is no longer valid pointer if 
			    rest was pointing to fh->cur_folder_sys
			 */
	} else {
	    free(fh->cur_folder_sys);
	    fh->cur_folder_sys = elm_message(FRM("%s@%s"),
					     X->username,se->official_name);
	    free_string(&(fh->cur_folder_disp));
	    fh->cur_folder_disp = format_string(FRM("%s@%s"),
						X->username,
						se->official_name);    
	}
    }

    /* Only IMAP connections are cached */
    CX = locate_from_cache(X->username,X->host,&IMAP_connection,
			   0   /* Default port */, se );
    
    if (CX) {
	if (!imap_only)
	    imap_only = safe_strdup(":INBOX");
	fh -> folder_type = IMAP_MBX;

	DPRINT(Debug,8,(&Debug,
		   "make_remote_mbox: IMAP mailbox (from cache)\n"));
	
	fh->folder_type->init_it(fh);
	folder_from_connection(CX,fh);
	
	status = &(fh->p->a.imap_mbx.Ch->C);
	
	if (':' == imap_only[0]) 
	    fh->p->a.imap_mbx.folder = safe_strdup(imap_only+1);
	else
	    fh->p->a.imap_mbx.folder = safe_strdup(imap_only);
	
	imap_mbx_set_passhm(fh,passhm);

    } else {

	static const int port_masks [] = { STFLAG_is_imap, STFLAG_is_pop, 0 };
	static const int port_imaponly_masks [] = { STFLAG_is_imap, 0 };

	int stflag;
	enum itls_status have_initial_tls = itls_none;

	struct enum_service_type ports_2 = NULL_enum_service_type;
	struct service_entry      * seX;

	init_enum_service_list(&ports_2,init_tls_default,
			       imap_only ? port_imaponly_masks :  port_masks,
			       STFLAG_mbox);
	
	if (!connect_remote_account_2(X,se,main_cancel,
				    
				      &result_se,
				      &got_type,
				      &got_port,
				      
				      &ports_2))
	    goto fail;
	
	DPRINT(Debug,15,(&Debug,
			 "make_remote_mbox: %s: connected, got port %d",
			 X->host,got_port));
	if (got_type) {
	    DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
			     got_type,
			     service_type_name(got_type),
			     service_type_defport(got_type)));
	}
	DPRINT(Debug,15,(&Debug,"\n"));
	
	switch((have_initial_tls = remote_account_init_tls(X))) {

	case itls_unsupported:
	case itls_failed:
	    
		goto fail;

	case itls_none:
	    break;

	case itls_done:
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: Initial TLS done (no STARTTLS)\n"));
	}

	if (result_se) {
	    if (SERVICE_ENTRY_magic != result_se->magic)
		panic("MAILER PANIC",__FILE__,__LINE__,
		      "make_remote_mbox",
		      "Bad magic (service_entry)",0);
	    
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: Got service entry %p %s, type %p %s\n",
			    result_se,result_se->official_name,
			    result_se->service,service_type_name(result_se->service)));
	
	    seX = result_se;
	} else
	    seX = se;

	
	switch((stflag = STFLAG_service_type(seX))) {
	case STFLAG_is_imap:
	    goto is_imap;
	case STFLAG_is_pop:
	    goto is_pop;
	    
	default:
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: unknown stflag=x%04x (service=%p %s, port=%d)\n",
			    stflag,seX->service,service_type_name(seX->service), got_port));
	}

	if (got_type) {
	    int flag;
	    
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: got type %p %s\n",
			    got_type,service_type_name(got_type)));
	    
	    switch((flag = service_type_flags(got_type,STFLAG_type_mask))) {
	    case STFLAG_is_imap:
		goto is_imap;
	    case STFLAG_is_pop:
		goto is_pop;
	    default:
		DPRINT(Debug,8,(&Debug,
				"make_remote_mbox: unknown flag x%04x (type=%p %s)\n",
				flag,got_type,service_type_name(got_type)));
	    }
	}

	switch (got_port) {
	case PORT_pop3s:
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: pops service ... handled as pop\n"));
	    	    
	case PORT_pop3:
	is_pop:	    
	    fh -> folder_type = POP_MBX;
	    
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: POP mailbox (succeed)\n"));
	    
	    fh->folder_type->init_it(fh);
	    
	    status = set_pop_connection(fh,X,passhm);	    
	    break;

	case PORT_imaps:
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: imaps service ... handled as imap\n"));
	    
	case PORT_imap4:
	is_imap:
	    if (!imap_only)
		imap_only = safe_strdup(":INBOX");
	    fh -> folder_type = IMAP_MBX;
	    
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: IMAP mailbox (succeed)\n"));
	    
	    fh->folder_type->init_it(fh);
	    
	    status = set_imap_connection(fh,X,passhm);
	    
	    if (':' == imap_only[0]) 
		fh->p->a.imap_mbx.folder = safe_strdup(imap_only+1);
	    else
		fh->p->a.imap_mbx.folder = safe_strdup(imap_only);
	    break;
	default:
	    DPRINT(Debug,8,(&Debug,
			    "make_remote_mbox: Unknown mailbox type (service=%p %s, port=%d)\n",
			    seX->service,service_type_name(seX->service), got_port));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet,MeUnknownRemoteMailboxType,
			      "Remote mailbox type of %S unknown"),
		      fh->cur_folder_disp);
	    status = NULL;
	}		
    }

    if (status) {
	/* In here we can use temp_dir instead of default_temp because
	 * that file does not act as session lock ...
	 */

	set_remote_tempmbox(fh,status);

    } 
    
 fail:

    clear_remote_account(X);

    /* 'result_se' can be NULL */
    free_service_entry(&result_se);
    
    if (imap_only)
	free(imap_only);
      
    if (status) {
	DPRINT(Debug,12,(&Debug,
			 "make_remote_mbox=1\n"));
	return 1;
    }

    DPRINT(Debug,12,(&Debug,
		     "make_remote_mbox=0\n"));
    return 0;
}

int same_remote_account_host_user(ra1,ra2)
     const struct remote_account *ra1;
     const struct remote_account *ra2;
{
    int ret = 1;

    if (ra1->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "same_remote_account_host_user",
	      "Bad magic number (ra1)",0);
    if (ra1->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "same_remote_account_host_user",
	      "Bad magic number (ra2)",0);

    if (! ra1->username || ! ra2->username ||
	0 != strcmp(ra1->username,ra2->username)) {
	
	DPRINT(Debug,15,(&Debug,
			 "same_remote_account_host_user: Different username\n"));
	ret = 0;
    }

    if (! ra1->host || ! ra2->host ||
	0 != strcmp(ra1->host,ra2->host)) {
	
	DPRINT(Debug,15,(&Debug,
			 "same_remote_account_host_user: Different host\n"));
	ret = 0;
    }


    
    DPRINT(Debug,15,(&Debug,
		     "same_remote_account_host_user=%d: { user=%s, host=%s } %s { user=%s, host=%s }\n",
		     ret,
		     ra1->username ? ra1->username : "<NULL>",
		     ra1->host ? ra1->host : "<NULL>",
		     ret ? "==" : "<>",
		     ra2->username ? ra2->username : "<NULL>",
		     ra2->host ? ra2->host : "<NULL>"));
		     
    return ret;
}

int remote_account_match(ra,username,host) 
     const struct remote_account *ra;
     const char *username;
     const char *host;
{
    int ret = 1;

     if (ra->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "remote_account_match",
	      "Bad magic number (ra1)",0);
   
    if (! ra->username || ! username ||
	0 != strcmp(ra->username,username)) {
	
	DPRINT(Debug,15,(&Debug,
			 "remote_account_match: Different username\n"));
	ret = 0;
    }

    if (! ra->host || ! host ||
	0 != strcmp(ra->host,host)) {
	
	DPRINT(Debug,15,(&Debug,
			 "remote_account_match: Different host\n"));
	ret = 0;
    }

    DPRINT(Debug,15,(&Debug,
		     "remote_account_match=%d: { user=%s, host=%s } %s { user=%s, host=%s }\n",
		     ret,
		     ra->username ? ra->username : "<NULL>",
		     ra->host ? ra->host : "<NULL>",
		     ret ? "==" : "<>",
		     username ? username : "<NULL>",
		     host ? host : "<NULL>"));
		     
    return ret;

}


void mbx_remote_login_msg(C,t)
     struct remote_account *C;
     enum remote_login_kw t;
{
    int bits = 0;

    if (REMOTE_ACCOUNT_magic != C->magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "mbx_remote_login_msg",
	      "Bad magic number (remote_account)",0);

    
     DPRINT(Debug,11,(&Debug, 
		      "mbx_remote_login_msg: host %s username %s - message type %d",
		      C->host,C->username,t));
     switch (t) {
     case rlogin_IMAP: DPRINT(Debug,11,(&Debug, " rlogin_IMAP")); break;
     case rlogin_POP:  DPRINT(Debug,11,(&Debug, " rlogin_POP"));  break;
     }
     DPRINT(Debug,11,(&Debug, "\n"));
     
    StreamInfo(C->stream,SS_ssf,&bits,NULL,NULL);

    DPRINT(Debug,11,(&Debug, 
		     "mbx_remote_login_msg: StreamInfo SS_ssf gives %d bits\n",
		     bits));
    
    if (bits < 2) {
	switch (t) {
	case rlogin_IMAP: lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPLoginClear,
						"IMAP login to %s as %s ... (clear text)"),
					C->host,
					C->username);    
	    break;	    
	case rlogin_POP:  lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPLoginClear,
					       "POP login to %s as %s ... (clear text)"),
				       C->host,
				       C->username);
	    break;
	}

    } else {

	int verify_status = 0;
	struct string * peer_name = NULL;

	StreamInfo(C->stream,SS_verify,&verify_status,NULL,
		   &peer_name);

	DPRINT(Debug,11,(&Debug, 
			 "mbx_remote_login_msg: StreamInfo SS_verify verify status %d",
			 verify_status));
	if (peer_name) {
	    DPRINT(Debug,11,(&Debug, ", peer name %S",
			     peer_name));
	}
	DPRINT(Debug,11,(&Debug,"\n"));
	

	if (verify_status < 0) {
	    switch (t) {
	    case rlogin_IMAP: lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPLoginVerFail,
						    "IMAP login to %s as %s ... (VERIFY FAILED)"),
					    C->host,
					    C->username);
		break;
	    case rlogin_POP:  lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPLoginVerFail,
						    "POP login to %s as %s ... (VERIFY FAILED)"),
					    C->host,
					    C->username);
		break;
	    }	
	} else if (verify_status > 0)  {
	    /* XXX Is charset correct? */
	    struct string * temp = new_string2(system_charset,
					       s2us(C->host));
	    struct string * returned_host_name = NULL;

	    if (dt_flag_is_set(&use_tls,use_tls_display_host)) {
		SOCKADDR ip_literal;

		if (get_ip(&ip_literal,C->host)) {
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_remote_login_msg: %s is ip address -- no display-check-host\n",
				     C->host));
		    goto display_peer_name;
		}

		if (StreamVerifyName(C->stream,SS_check_host,
				     temp,
				     & returned_host_name)) {

		    DPRINT(Debug,11,(&Debug, 
				     "mbx_remote_login_msg: StreamVerifyName SS_check_host %S succeed",
				     temp));
		    

		    if (returned_host_name) {
			DPRINT(Debug,11,(&Debug, ", returned name %S\n",
					 returned_host_name));
			/* Optimize display */
			
			if (0 == string_cmp(returned_host_name,temp,99)) {
			    switch (t) {
			    case rlogin_IMAP: lib_transient(CATGETS(elm_msg_cat, MeSet,
								    MeIMAPLoginVerHost,
								    "IMAP login to %S (verified, host matches) as %s ..."),
							    returned_host_name,
							    C->username);
				break;				
			    case rlogin_POP:  lib_transient(CATGETS(elm_msg_cat, MeSet,
								    MePOPLoginVerHost,
								    "POP login to %S (verified, host matches) as %s ..."),
							    returned_host_name,
							    C->username);
				break;
			    }
			} else {
			    switch (t) {
			    case rlogin_IMAP: lib_transient(CATGETS(elm_msg_cat, MeSet,
								    MeIMAPLoginVerHostAs,
								    "IMAP login to %s as %s (verified, host matches %S) ..."),
							    C->host,
							    C->username,
							    returned_host_name);
				break;
			    case rlogin_POP:  lib_transient(CATGETS(elm_msg_cat, MeSet,
								    MePOPLoginVerHostAs,
								    "POP login to %s as %s (verified, host matches %S) ..."),
							    C->host,
							    C->username,
							    returned_host_name);
				break;
			    }
			}
			
			goto host_displayed;
		    } else {
			DPRINT(Debug,11,(&Debug, "\n"));

			/* if no name returned - check host result is not very useful */
			
		    }
		    		   
		} else {
		    /* Either verify failed or X509_check_host() not supported */
		    
		    DPRINT(Debug,11,(&Debug, 
				     "mbx_remote_login_msg: StreamVerifyName SS_check_host %S failed\n",
				     temp));
		}		
	    }

	display_peer_name:
	    
	    if (peer_name) {
		
		/* Optimize display */
		
		if (0 == string_cmp(peer_name,temp,99))  {
		    switch (t) {
		    case rlogin_IMAP: lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPLoginVer,
							    "IMAP login to %S (verified) as %s ..."),
						    peer_name,
						    C->username);
			break;
		    case rlogin_POP:  lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPLoginVer,
							    "POP login to %S (verified) as %s ..."),
						    peer_name,
						    C->username);
			break;
		    }
		} else {
		    switch (t) {
		    case rlogin_IMAP: lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPLoginVerAs,
							    "IMAP login to %s as %s ... (verified: %S)"),
						    C->host,
						    C->username,
						    peer_name);
			break;
		    case rlogin_POP:  lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPLoginVerAs,
							    "POP login to %s as %s ... (verified: %S)"),
						    C->host,
						    C->username,
						   peer_name);
			break;
		    }
		    
		}
		
	    } else {
		/* If no name, verify result is not very useful */
		
		switch (t) {
		case rlogin_IMAP: lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPLogin,
							"IMAP login to %s as %s ..."),
						C->host,
						C->username);
		    break;
		case rlogin_POP:  lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPLogin,
						       "POP login to %s as %s ..."),
						C->host,
						C->username);
		    break;
		}

	    }
	    
	host_displayed:
	    
	    if (returned_host_name)
		free_string(&returned_host_name);
	    
	    free_string(&temp);

	    
	} else {
	    switch (t) {
	    case rlogin_IMAP: lib_transient(CATGETS(elm_msg_cat, MeSet,MeIMAPLoginVerNot,
						    "IMAP login to %s as %s ... (not verified)"),
					   C->host,
					   C->username);
		break;
	    case rlogin_POP: lib_transient(CATGETS(elm_msg_cat, MeSet,MePOPLoginVerNot,
						   "POP login to %s as %s ... (not verified)"),
					   C->host,
					   C->username);
		break;
	    }
	}
	
	if (peer_name)
	    free_string(&peer_name);
    }

}
     

#endif


int remote_folder_type(fh)
     struct folder_info *fh;
{    
    int ret = 0;
    struct cancel_data *main_cancel = NULL /* Used if dns lookup was
					      cancelable */;

#ifdef REMOTE_MBX
    struct remote_account X;
    struct service_entry *se = NULL;
    char * rest;
    int z;
#endif

    if ('/' == fh->cur_folder_sys[0] ||
	0 == access(fh->cur_folder_sys,ACCESS_EXISTS)) {

	DPRINT(Debug,8,(&Debug,
			"remote_folder_type=0 (file: %s)\n",
			fh->cur_folder_sys));
	return 0;
    }

#ifdef REMOTE_MBX
    
    /* -1 == name not found or bad syntax or DNS lookup canceled
        0 == not a remote address
	1 == name found 
    */
    z = split_remote_name(fh->cur_folder_sys,&X,&se,&rest,
			  0 /* Both IMAP and POP are OK */,
			  &main_cancel);

    if (z < 0) {     /* ERROR */
	if (main_cancel && is_canceled(main_cancel)) {
	    DPRINT(Debug,12,(&Debug,
			     "remote_folder_type: DNS lookup canceled\n"));
	} else {
	    DPRINT(Debug,12,(&Debug,
			     "remote_folder_type: name not found or bad syntax\n"));
	}
	goto clean;	  
	
    } else if (z > 0) {
	
	/* May free X (if succeed) or copy */
	if (!make_remote_mbox(fh,&X,se,rest,1,NULL,
			      main_cancel)) {
	clean:
	    clear_remote_account(&X);
	    
	    fh -> folder_type = NO_NAME;
	    fh->folder_type->init_it(fh);
	}	

	/* 'se' can be NULL */
	free_service_entry(&se);
	
	ret = 1;      /* Is 'remote mailbox' even if it is canceled or failed ... */	    
    }
#else
    DPRINT(Debug,9,(&Debug,
		    "remote_folder_type: (remote mailboxes are not supported.)\n"));
#endif /* REMOTE_MBX */

    if (main_cancel)
	free_cancel(& (main_cancel));
    
    DPRINT(Debug,8,(&Debug,
		    "remote_folder_type=%d\n",ret));
    return ret;
}

enum prepare_reconnect_res mbx_prepare_read_reconnect(folder,reconnect_ptr,X)
     struct folder_info *folder;
     RECONNECT_MODE reconnect_ptr;
     struct remote_account *X;
{
    enum prepare_reconnect_res ret = prepare_reconnect_none;
    
    if (PRIVATE_DATA_magic != folder->p->magic) 
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_reconnect",
	      "Bad magic number (private_data)",0);
    if (RECONNECT_MODE_magic != reconnect_ptr->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"mbx_prepare_read_reconnect",
	      "Bad magic number (reconnect_mode)",0);
    if (X->magic  != REMOTE_ACCOUNT_magic)
	panic("CONNECTION PANIC",__FILE__,__LINE__,
	      "mbx_prepare_read_reconnect",
	      "Bad magic number (remote_account)",0);
    
    if (!folder->p->fh_temp) {
	DPRINT(Debug,12,(&Debug, 
			 "mbx_prepare_read_reconnect: NO temp file %s\n",
			 folder->cur_tempfolder));
	
	ret = prepare_reconnect_none;
	
    } else {

	if (EOF == fflush(folder->p->fh_temp)) {
	    int err = errno;
	    
	    DPRINT(Debug,12,(&Debug, 
			     "mbx_prepare_read_reconnect: failed to flush temp file %s: %s\n",
			     folder->cur_tempfolder,
			     strerror(err)));

	    ret = prepare_reconnect_fail;
	    goto clean;
	}
		
	if (! reconnect_ptr -> tempfolder_name &&
	    ! reconnect_ptr -> tempfolder_fh) {

	    long temp_size = file_bytes(folder-> cur_tempfolder,NULL);
	    if (-1L == temp_size) {
		ret = prepare_reconnect_fail;
		goto clean;
	    }
		    
	    if (temp_size > 0L) {
		DPRINT(Debug,12,(&Debug, 
				 "mbx_prepare_read_reconnect: temp file %s size %ld, need move to reconnect mode.\n",
				 folder->cur_tempfolder,
				 temp_size));

		reconnect_ptr -> tempfolder_size = temp_size;
		reconnect_ptr -> tempfolder_name =
		    strmcpy(reconnect_ptr -> tempfolder_name,
			    folder->cur_tempfolder);
		
		set_remote_tempmbox(folder,
				    X);
		
		if (0 == strcmp(reconnect_ptr -> tempfolder_name,folder->cur_tempfolder)) {
		    DPRINT(Debug,12,(&Debug, 
				     "mbx_prepare_read_reconnect: temp file %s -- no new name generated\n",
				     folder->cur_tempfolder));			     
		} else {
		    int err = 0;
		    reconnect_ptr -> tempfolder_fh = folder->p->fh_temp;
		    folder->p->fh_temp = NULL;
		    
		    if (sessionlock_create_tempfolder(folder,&err)) {
			DPRINT(Debug,12,(&Debug, 
					 "mbx_prepare_read_reconnect: temp file %s -- reconnect mode temp file %s\n",
					 folder->cur_tempfolder,
					 reconnect_ptr -> tempfolder_name));			    			     
			ret = prepare_reconnect_ok;
		    } else {
			
			if (folder->p->fh_temp) {
			    DPRINT(Debug,12,(&Debug, 
					     "mbx_prepare_read_reconnect: sessionlock_create_tempfolder failed, but %s opened?\n",
					     folder->cur_tempfolder));
			    fclose(folder->p->fh_temp);
			    folder->p->fh_temp = NULL;
			}
				 
			DPRINT(Debug,12,(&Debug, 
					 "mbx_prepare_read_reconnect: temp file %s -- new name (%s) not available",
					 reconnect_ptr -> tempfolder_name,
					 folder->cur_tempfolder));
			if (err) {
			    DPRINT(Debug,12,(&Debug, ": %s",
					     strerror(err)));
			}
			DPRINT(Debug,12,(&Debug, "\n"));
				 
			strfcpy(folder->cur_tempfolder,
				reconnect_ptr -> tempfolder_name,
				sizeof folder->cur_tempfolder);
			folder->p->fh_temp = reconnect_ptr -> tempfolder_fh;
			reconnect_ptr -> tempfolder_fh = NULL;
			
		    }
		}
	    } else {
		DPRINT(Debug,12,(&Debug, 
				 "mbx_prepare_read_reconnect: temp file %s size %ld; reconnect mode not used.\n",
				 folder->cur_tempfolder,
				 temp_size));
	    }
	} else {
	    DPRINT(Debug,12,(&Debug, 
			     "mbx_prepare_read_reconnect: reconnect mode temp file already set: %s\n",
			     reconnect_ptr -> tempfolder_name));

	    if (0 == strcmp(reconnect_ptr -> tempfolder_name,folder->cur_tempfolder)) {
		DPRINT(Debug,12,(&Debug, 
				 "mbx_prepare_read_reconnect: temp file %s -- no new name generated\n",
				 folder->cur_tempfolder));			     
	    } else {
		ret = prepare_reconnect_ok;
	    }
	    
	}
    }

 clean:
    DPRINT(Debug,12,(&Debug,
		     "mbx_prepare_read_reconnect=%d",
		     ret));
    switch (ret) {
    case prepare_reconnect_fail: DPRINT(Debug,12,(&Debug," prepare_reconnect_fail")); break;
    case prepare_reconnect_none: DPRINT(Debug,12,(&Debug," prepare_reconnect_none")); break;
    case prepare_reconnect_ok:   DPRINT(Debug,12,(&Debug," prepare_reconnect_ok"));   break;
    }    
    DPRINT(Debug,12,(&Debug,"\n"));
    
    return ret;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
