static char rcsid[] = "@(#)$Id: savefolder.c,v 2.52 2023/12/13 16:55:32 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.52 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************
 * Some local mailbox code based on Elm 2.4 src/file.c.
 * That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *			Copyright (c) 1988-1992 USENET Community Trust
 *			Copyright (c) 1986,1987 Dave Taylor
 *
 *****************************************************************************/

#include "def_mbox.h"
#include "mbxlocal_imp.h"
#include "hashmark.h"

#include "s_me.h"

DEBUG_VAR(Debug,__FILE__,"mbox");

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif

static unsigned char * s2us P_((char *str));
static unsigned char * s2us(str)
     char *str;
{
    return (unsigned char *)str;
}

static const unsigned char * cs2us P_((const char *str));
static const unsigned char * cs2us(str)
     const char *str;
{
    return (const unsigned char *)str;
}

static char *us2s P_((unsigned char *str));
static char *us2s(str) 
     unsigned char *str;
{
    return (char *)str;
}


/* ---------------------------------------------------------------------- */

enum DUMMY_special {
    DUMMY_none           = -1,
    DUMMY_root_directory = 0,
    DUMMY_INBOX,
    DUMMY_current_directory,
    DUMMY_Mail,
    DUMMY_attachment,
    DUMMY_received,
    DUMMY_sent,
    DUMMY_home,
    DUMMY_hashmark,
    DUMMY_dsn,

    DUMMY_count
};

#define DUMMY_BROWSER_magic	0xF512

struct DUMMY_BROWSER {
    unsigned short            magic;     /* DUMMY_BROWSER_magic */

    char                  * remote;
    struct browser_passhm * passhm;

    struct string * comment[DUMMY_count];
};

static struct DUMMY_BROWSER * malloc_DUMMY_BROWSER P_((void));
static struct DUMMY_BROWSER * malloc_DUMMY_BROWSER()
{
    enum DUMMY_special i;

    struct DUMMY_BROWSER * ret = safe_zero_alloc(sizeof (*ret));

    ret->magic            = DUMMY_BROWSER_magic;

    ret->remote = NULL;
    ret->passhm = NULL;

    for (i = 0; i < DUMMY_count; i++)
	ret->comment[i] = NULL;


    return ret;
}

static void free_DUMMY_BROWSER P_((struct DUMMY_BROWSER **ptr));
static void free_DUMMY_BROWSER(ptr)
     struct DUMMY_BROWSER **ptr;
{
    enum DUMMY_special i;    

    if (DUMMY_BROWSER_magic != (*ptr)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_DUMMY_BROWSER",
	      "Bad magic type",0);

    if ((*ptr)->remote) {
	free((*ptr)->remote);
	(*ptr)->remote = NULL;
    }

    if ((*ptr)->passhm)
	free_browser_passhm(& ((*ptr)->passhm));

    for (i = 0; i < DUMMY_count; i++)
	if ((*ptr)->comment[i])
	    free_string(& ((*ptr)->comment[i]));

    (*ptr)->magic = 0;        /* Invalidate */
    free(*ptr);
    *ptr = NULL;
}

/* -------------------------------- */

void dummy_browser_set_remote(dir,remote,passhm)
     struct folder_browser *dir;
     const char *remote /* may be NULL */;
     struct browser_passhm  * passhm;     
{
    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dummy_browser_set_remote",
	      "Bad magic type",0);


    if (dir->a.dummy_browser->remote)
	free(dir->a.dummy_browser->remote);
    dir->a.dummy_browser->remote = NULL;

    if (remote)
	dir->a.dummy_browser->remote = safe_strdup(remote);

    if (dir->a.dummy_browser->passhm)
	free_browser_passhm(& (dir->a.dummy_browser->passhm));
    dir->a.dummy_browser->passhm = passhm;
    if (dir->a.dummy_browser->passhm)
	inc_browser_passhm_refcount (dir->a.dummy_browser->passhm);

}

#if 0
const char * dummy_browser_get_remote(dir)
     struct folder_browser *dir;
{
    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dummy_browser_get_remote",
	      "Bad magic type",0);

    return dir->a.dummy_browser->remote;
}
#endif


/* -------------------------------*/

#define DUMMY_BR_entry_magic	0xF513

struct DUMMY_BROWSER_entry {
    unsigned short            magic;     /* DUMMY_BR_entry_magic */

    enum DUMMY_special        comment;
    struct hashmark_item    * hashmark_item;
    
};

static struct DUMMY_BROWSER_entry * malloc_DUMMY_BROWSER_entry P_((void));
static struct DUMMY_BROWSER_entry * malloc_DUMMY_BROWSER_entry() 
{
    struct DUMMY_BROWSER_entry * ret = safe_zero_alloc(sizeof (*ret));

    ret->magic            = DUMMY_BR_entry_magic;

    ret->comment          = DUMMY_none;
    ret->hashmark_item    = NULL;
           
    return ret;
}

static void free_DUMMY_BROWSER_entry P_((struct DUMMY_BROWSER_entry **entry));
static void free_DUMMY_BROWSER_entry(entry)
     struct DUMMY_BROWSER_entry **entry;
{
    if (DUMMY_BR_entry_magic != (*entry)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_DUMMY_BROWSER_entry",
	      "Bad magic type",0);

    if ((*entry)->hashmark_item)
	free_hashmark_item(& ((*entry)->hashmark_item));

    (*entry)->magic      = 0;  /* Invalidate */
    free(*entry);
    *entry = NULL;
}


static void set_comment P_((struct folder_browser *dir,
			    struct name_vector    *entry,
			    enum DUMMY_special    comment));
static void set_comment(dir,entry,comment) 
     struct folder_browser *dir;
     struct name_vector    *entry;
     enum DUMMY_special    comment;
{
    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"set_comment",
	      "Bad dir magic type",0);

    if (!entry->a.dummy_browser)
	entry->a.dummy_browser = malloc_DUMMY_BROWSER_entry();
    else if (DUMMY_BR_entry_magic != entry->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"set_comment",
	      "Bad entry magic type",0);

    if (DUMMY_none == comment) {
	entry->a.dummy_browser->comment = DUMMY_none;
	return;
    }

    if (comment < 0 || comment >= DUMMY_count)
	panic("MBX PANIC",__FILE__,__LINE__,"set_comment",
	      "Bad comment index",0);

    entry->a.dummy_browser->comment = comment;

    if (! dir->a.dummy_browser->comment[comment]) {
	struct string * text = NULL;

	switch(comment) {
	case DUMMY_current_directory:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, MeDummyEntryCurrentDir,
					  "[current directory]"));
	    break;
	case DUMMY_Mail:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, MeDummyEntryMail,
					  "[Mail directory]"));
	    break;
	case DUMMY_INBOX:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, MeDummyEntryINBOX,
					  "[incoming mailbox]"));
	    break;
	case DUMMY_received:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, MeDummyEntryReceived,
					  "[received folder]"));
	    break;
	case DUMMY_sent:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, MeDummyEntrySent,
					  "[sent folder]"));
	    break;
	case DUMMY_root_directory:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, MeDummyEntryRoot,
					  "[root directory]"));
	    break;
	case DUMMY_attachment:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, 
					  MeDummyEntryAttachment,
					  "[attachment (documents) directory]"));
	    break;
	case DUMMY_home:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, MeDummyEntryHome,
					  "[home directory]"));
	    break;

	case DUMMY_hashmark:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, 
					  MeDummyEntryHashmarks,
					  "[hashmarks listing]"));
	    break;

	case DUMMY_dsn:
	    text =  format_string(CATGETS(elm_msg_cat, MeSet, 
					  MeDummyEntryDsn,
					  "[Delivery Status Notifications folder]"));
	    break;

	case DUMMY_none:
	case DUMMY_count:
	    /* Not used */
	    break;
	}

	dir->a.dummy_browser->comment[comment] = text;

    }

}





/* ------------------------------ */

/* Dummy browser */

#include <errno.h>
#ifndef ANSI_C
extern int errno;
#endif


#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
#endif


S_(browser_zero_dir browser_zero_dummy)
static  void browser_zero_dummy P_((struct folder_browser *dir));
static  void browser_zero_dummy(dir) 
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug,"browser_zero_dummy: dir=%p\n", dir));
   
    dir->a.dummy_browser = malloc_DUMMY_BROWSER();
}

S_(browser_free_dir browser_free_dummy)
static void browser_free_dummy P_((struct folder_browser *dir));
static void browser_free_dummy(dir)
     struct folder_browser *dir;
{
    DPRINT(Debug,11,(&Debug,"browser_free_dummy: dir=%p\n", dir));

    if (dir->a.dummy_browser)
	free_DUMMY_BROWSER(& (dir->a.dummy_browser));
}

#ifdef REMOTE_MBX

 /* 0 == name not found
    1 == name found 
 */

static int dummy_give_remote_account P_((struct folder_browser *dir,
					 struct remote_account *ra,
					 struct service_entry **se,
					 const enum hmcon contype,
					 struct browser_passhm **passhm,
					 struct cancel_data         ** cancel_p /*  Used if dns lookup was
										    cancelable */
					 ));

static int dummy_give_remote_account(dir,ra,se,contype,passhm,cancel_p)
     struct folder_browser *dir;
     struct remote_account *ra;
     struct service_entry **se;
     const enum hmcon contype;
     struct browser_passhm      ** passhm;
     struct cancel_data         ** cancel_p;   
     
{
    int ret = 0;
    struct browser_passhm *passhm_1 = NULL;
    struct service_entry  *se_1     = NULL;
    struct remote_account  ra_1;
    int passhm_lookup = -1 /* fallback */;
    int flag = 0;

 
    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dummy_give_remote_account",
	      "Bad magic type",0);
    
    zero_remote_account(&ra_1);

    if (ra)
	clear_remote_account(ra);
    if (se)
	free_service_entry(se);
    if (passhm && *passhm)
	free_browser_passhm(passhm);
	
    switch (contype) {
    case hmcon_none: flag = STFLAG_mbox;  break;
    case hmcon_pop:  flag = STFLAG_is_pop;   break;
    case hmcon_imap: flag = STFLAG_is_imap;  break;
    }
    
    if (dir->a.dummy_browser->passhm) {

	if (hmcon_none == contype ||
	    browser_passhm_check_type(dir->a.dummy_browser->passhm,
				      contype)) {
	    char * username = NULL;
	    char * hostname = NULL;

	    passhm_1 = dir->a.dummy_browser->passhm;
	    inc_browser_passhm_refcount(passhm_1);

	    passhm_lookup = 
		browser_passhm_remote_lookup(passhm_1,&se_1,flag,
					     &username, &hostname,
					     NULL,NULL,cancel_p);
      	    
	    if (passhm_lookup > 0) {
		ra_1.host     = hostname; hostname = NULL;
		ra_1.username = username; username = NULL;
		
		if (!se_1)
		    panic("BROWSER PANIC",__FILE__,__LINE__,"dummy_give_remote_account",
			  "service_entry not set",0);

		ret = 1;

		DPRINT(Debug,15,(&Debug,
				 "dummy_give_remote_account: found host with browser_passhm_lookup\n"));


	    } else {

		if (0 == passhm_lookup && cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		    DPRINT(Debug,12,(&Debug,
				     "dummy_give_remote_account: DNS lookup canceled\n"));
		    
		    passhm_lookup = -2 /* canceled */;
		}

		if (hostname)
		    free(hostname);
		hostname = NULL;
		if (username)
		    free(username);
		username = NULL;
	    }
	} else {
	    DPRINT(Debug,15,(&Debug,
			     "dummy_give_remote_account: browser_passhm discarded\n"));
	}
    }

    if (-1 == passhm_lookup) {
	char *rest = NULL;
	int code;

	/* -1 == name not found or bad syntax
	    0 == not a remote address
	    1 == name found 
	    2 == no lookup (se == NULL)
	*/

	code = split_remote_name(dir->a.dummy_browser->remote,
				 &ra_1,
				 &se_1,
				 &rest,
				 flag,
				 cancel_p
				 );
	
	if (code < 0) {     /* ERROR */
	    if (cancel_p && *cancel_p && is_canceled(*cancel_p)) {
		DPRINT(Debug,12,(&Debug,
				 "dummy_give_remote_account: DNS lookup canceled; (%s)\n",
				 dir->a.dummy_browser->remote));		
	    } else  {
		DPRINT(Debug,12,(&Debug,
				 "dummy_give_remote_account: name not found or bad syntax; (%s)\n",
				 dir->a.dummy_browser->remote));
	    }		
	} else if (0 <= code) {
	    
	    if (code == 0 || rest)
		panic("BROWSER PANIC",__FILE__,__LINE__,"dummy_give_remote_account",
		      "Bad return from split_remote_name",0);
	    
	    if (!se_1)
		panic("BROWSER PANIC",__FILE__,__LINE__,"dummy_give_remote_account",
		      "service_entry not set",0);
	
	    DPRINT(Debug,15,(&Debug,
			     "dummy_give_remote_account: found host with split_remote_name\n"));


	    ret = 1;
	}
    }

    if (ret) {
	if (ra) {
	    *ra = ra_1;
	    zero_remote_account(&ra_1);
	}

	if (se && se_1) {
	    *se = se_1;
	    inc_service_entry_refcount(*se);
	}

	if (passhm && passhm_1) {
	    *passhm = passhm_1;
	    inc_browser_passhm_refcount(*passhm);
	}
    }

    clear_remote_account(&ra_1);

    /* can be NULL */
    free_service_entry(&se_1);  
    
    if (passhm_1)
	free_browser_passhm(& passhm_1);

    DPRINT(Debug,15,(&Debug,
		     "dummy_give_remote_account=%d\n",ret));
    
    return ret;
}


static int dummy_to_imap P_((struct folder_browser *dir));
static int dummy_to_imap(dir)
     struct folder_browser *dir;
{
    int                     ret = 0;
    struct remote_account   X;
    struct service_entry  * se = NULL;
    struct browser_passhm * passhm = NULL;
    struct cancel_data    * main_cancel = NULL /* Used if dns lookup was
					      cancelable */;
    int ret_dummy                     = 0;
    struct service_entry  * result_se = NULL;
    
    zero_remote_account(&X);
    
    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dummy_to_imap",
	      "Bad magic type",0);
            
    ret_dummy = dummy_give_remote_account(dir,&X,&se,hmcon_imap,&passhm,
					  &main_cancel);

    if (ret_dummy) {
    
	struct connection_cache *CX = NULL;

	int                         got_port = 0;
	const struct service_type * got_type = NULL;

	struct enum_service_type ports_imaponly_2 = NULL_enum_service_type;
	
	DPRINT(Debug,15,(&Debug,
			 "dummy_to_imap: Using user=%s host=%s\n",
			 X.username,X.host));

	init_enum_service_type(&ports_imaponly_2,init_tls_default,STFLAG_is_imap,STFLAG_mbox);
	
		
	/* Only IMAP connections are cached */
	CX = locate_from_cache(X.username,X.host,&IMAP_connection,
			       0  /* Default port */, se);
	if (CX) {
	    DPRINT(Debug,10,(&Debug,
			     "dummy_to_imap: Changing browser to IMAP browser\n"));
	    
	    change_folder_browser_type(dir,&imap_browser);
	    
	    /*  browser_from_connectionX sets CX = NULL */
	    browser_from_connectionX(&CX,dir);

	    /* passhm is null if wrong type */
	    imap_browser_set_passhm(dir,passhm);
	    	    
	    ret = 1;	  
	} else if (connect_remote_account_2(&X,se,main_cancel,
					    &result_se,
					    &got_type,
					    &got_port,
					    &ports_imaponly_2)) {
	    
	    enum itls_status have_initial_tls = itls_none;

	    DPRINT(Debug,15,(&Debug,
			     "dummy_to_imap: %s: connected, got port %d",
			     X.host,got_port));
	    if (got_type) {
		DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
				 got_type,
				 service_type_name(got_type),
				 service_type_defport(got_type)));
	    }
	    DPRINT(Debug,15,(&Debug,"\n"));
	    	    
	    switch((have_initial_tls = remote_account_init_tls(&X))) {
		
	    case itls_unsupported:
	    case itls_failed:
		
		goto fail;
		
	    case itls_none:
		break;
		
	    case itls_done:
		DPRINT(Debug,8,(&Debug,
				"dummy_to_imap: Initial TLS done (no STARTTLS)\n"));
	    }
	    
	    
	    DPRINT(Debug,10,(&Debug,
			     "dummy_to_imap: Changing browser to IMAP browser\n"));
	    
	    change_folder_browser_type(dir,&imap_browser);
	    
	    /* passhm is null if wrong type */

	    ret = join_connection_hm(give_imap_connection(dir),&X,
				     CON_greeting, passhm);
	    
	    imap_browser_set_passhm(dir,passhm);	
	} else if (main_cancel && is_canceled(main_cancel)) {
	    DPRINT(Debug,12,(&Debug,
			     "dummy_to_imap: connect canceled\n"));
	} 
	
    } else if (main_cancel && is_canceled(main_cancel)) {
	DPRINT(Debug,12,(&Debug,
			 "dummy_to_imap: DNS lookup canceled\n"));
    } 

 fail:
    clear_remote_account(&X);

    /* 'se' can be NULL */
    free_service_entry(&se);
    free_service_entry(&result_se);
    
    if (main_cancel)
	free_cancel(& (main_cancel));
    
    if (passhm)
	free_browser_passhm(&passhm);
	
    DPRINT(Debug,15,(&Debug,"dummy_to_imap=%d\n",ret));

    return ret;
}
#endif /* REMOTE_MBX */


/* Returns name relative to directory of browser */
static struct string * browser_descend_dummy P_((struct folder_browser *dir,
						 const struct string *rel_name));
static struct string * browser_descend_dummy(dir,rel_name)
     struct folder_browser *dir;
     const struct string *rel_name;
{
    int L, idx;
    static struct string * ret = NULL;

    int last_idx_ok = -1;

    DPRINT(Debug,12,(&Debug,"browser_descend_dummy: dir=%p, ", dir));

    if (!rel_name) {
	DPRINT(Debug,12,(&Debug,"rel_name=NULL\n"));
	DPRINT(Debug,12,(&Debug,"browser_descend_dummy=NULL\n"));
	return NULL;
    } 
    DPRINT(Debug,12,(&Debug,"rel_name=%S\n",rel_name));    

    L = string_len(rel_name);

    for (idx = 0; idx < L ; idx++) {
	uint16 code = give_unicode_from_string(rel_name,idx);

	if (0x002F /* '/' */ == code) {
	    int X1 = 0;
		
	    /* 1) First check is current path have same prefix ... */
	    struct string * A1 = clip_from_string(rel_name,&X1,idx+1);	    
	    if (dir->dirname) {
		int X2 = 0;
		struct string * A2 = clip_from_string(dir->dirname,&X2,idx+1); 

		if (X1 == X2 &&
		    0 == string_cmp(A1,A2,-99 /* Unknown indicator */ )) {

		    DPRINT(Debug,12,(&Debug,
				     "browser_descend_dummy: sep idx=%d -- so far ok\n",
				     idx));

		    last_idx_ok = idx;
		    free_string(&A1);
		    free_string(&A2);
		    continue;
		}
		
		free_string(&A2);
	    }

	    /* 2) Then check if current path have prefix of wanted */
	    if (!dir->dirname || 
		0 != string_cmp(A1,dir->dirname,
				-99 /* unknown indicator */)) {
		struct string * Lstr =  
		    convert_string(local_fs_charset,A1,0);
		unsigned char * str  = 
		    stream_from_string(Lstr,0,NULL);
		struct stat buf;

		DPRINT(Debug,12,(&Debug,
				 "browser_descend_dummy: sep idx=%d -- need change of directory\n",
				 idx));
		
		if (0 != stat(us2s(str),&buf) 
#ifdef S_ISDIR
		    || !S_ISDIR(buf.st_mode)
#endif
		    ) {
		    DPRINT(Debug,12,(&Debug,
				     "browser_descend_dummy: Failed -- bailing out\n"));

		    
		    free_string(&A1);
		    free_string(&Lstr);
		    free(str);
		    break;
		}
		 
		/* Make empty selection folder */
		clear_dir_vector(dir);

		if (dir->dirname)
		    free_string(&dir->dirname);
		dir->dirname = Lstr;
		Lstr = NULL;
		
		if (dir->sys_dir) {
		    free(dir->sys_dir);
		    dir->sys_dir = NULL;
		}
		dir->sys_dir = us2s(str);
		str = NULL;
	    }

	    DPRINT(Debug,12,(&Debug,
			     "browser_descend_dummy: sep idx=%d -- ok\n",
			     idx));
	    last_idx_ok = idx;		
	    free_string(&A1);
	}
    }

    DPRINT(Debug,12,(&Debug,
		     "browser_descend_dummy: Up to %d OK\n",last_idx_ok));

    if (last_idx_ok < 0) {

	DPRINT(Debug,12,(&Debug,
			 "browser_descend_dummy: No directory\n"));
	
	if (dir->dirname || dir->sys_dir) {
	    /* Make empty selection folder */
	    clear_dir_vector(dir);
	}

	if (dir->dirname)
	    free_string(&dir->dirname);

	if (dir->sys_dir) {
	    free(dir->sys_dir);
	    dir->sys_dir = NULL;
	}
    }


    last_idx_ok++;

    ret = clip_from_string(rel_name,&last_idx_ok,string_len(rel_name));
    
    if (dir->dirname) {
	DPRINT(Debug,10,(&Debug,
			 "*** %S as relative to %S is %S\n",
			 rel_name,dir->dirname,ret));
    }

    DPRINT(Debug,12,(&Debug,
		     "browser_descend_dummy=%S   -- consumed to %d\n",
		     ret,last_idx_ok));
    return ret;
}



static int expander P_((struct folder_browser *dir,
			struct string **buffer,
			struct string **relative_to_type));


S_(browser_change_v_dir browser_change_v_dummy)
static int browser_change_v_dummy P_((struct folder_browser *dir,
				      struct name_vector *X,
				      struct string **dispname));
static int browser_change_v_dummy(dir,X,dispname)
     struct folder_browser *dir;
     struct name_vector *X;
     struct string **dispname;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_change_v_dummy: dir=%p\n", dir));

     if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_v_dummy",
	      "Bad magic type (DUMMY_BROWSER)",0);

    if (X->a.dummy_browser) {

	if (DUMMY_BR_entry_magic != X->a.dummy_browser->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_change_v_dummy",
		  "Bad magic type (DUMMY_BROWSER_entry)",0);

	if (X->a.dummy_browser->hashmark_item) {
	    struct hashmark_item *item = X->a.dummy_browser->hashmark_item;

	    inc_hashmark_item_refcount(item);

	    DPRINT(Debug,11,(&Debug, 
			     "browser_change_v_dummy:  Changing browser to hashmark browser\n"));

	    if (change_folder_browser_type(dir,&hashmark_browser)) {
		       	
		ret = hashmark_browser_passthru_change(dir,item,dispname);		
	    } else {
		DPRINT(Debug,11,(&Debug, 
				 "browser_change_v_dummy: No change?\n"));
	    }

	    free_hashmark_item(&item);
	}
	
    } else if (dir->dirname) { 
	
	DPRINT(Debug,11,(&Debug, 
			 "browser_change_v_dummy: Directory %S (should not happen?)\n",
			 dir->dirname));

    } else {
	/* SPECIAL HANDLING FOR DEFAULT DIRECTORY */
	struct string *relative_to_type = NULL;
	
	if (*dispname)
	    free_string(dispname);
	*dispname = dup_string(X->disp_name);

	DPRINT(Debug,11,(&Debug, 
			 "browser_change_v_dummy: Expanding %S\n",
			 *dispname));
	
	if (expander(dir,dispname,&relative_to_type)) {
	    if (BROWSER_TYPE_magic != dir->type->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"browser_change_v_dummy",
		      "Bad magic number (browser_type) after expander",0);

	    ret = dir->type->browser_change_it(dir,relative_to_type,dispname);
	}
	
	if (relative_to_type)
	    free_string(&relative_to_type);	       	
    }

    DPRINT(Debug,11,(&Debug,"browser_change_v_dummy=%d\n", ret));
  
    return ret;
}

#define EXPAND_allow_file   ( 1 << selection_file )
#define EXPAND_allow_folder ( 1 << selection_folder )

static struct expand_tag_item {
    const char          * tag;
    const char          * variable;
    struct dt_estr_info * ptr;
    int                   allow_mask;
    
    enum DUMMY_special    dummy_text;
    int                   add_dir_vector_flags;
    
} expand_tag_list [] = {
    { dsn_mail_tag,
      "dsnmail",
      &dsn_mail_e,
      EXPAND_allow_folder,
      DUMMY_dsn,
      BROWSER_NODIR
    }
};
static int expand_tag_count = sizeof (expand_tag_list) / sizeof (expand_tag_list[0]);


static void browser_gen_default_menu P_((struct folder_browser *dir));
static void browser_gen_default_menu(dir)
     struct folder_browser *dir;
{
    int len = 0;
    struct hashmark_item **vector;
    int x;

    DPRINT(Debug,11,(&Debug,"browser_gen_default_menu: dir=%p\n", dir));
 
    vector =  hashmark_dir(&len,dir->sel_type);

    DPRINT(Debug,11,(&Debug,"browser_gen_default_menu: %d hashmarks\n",
		     len));

    /* Make top level selection folder */
    clear_dir_vector(dir);
    
    if (dir->dirname)
	free_string(&dir->dirname);
    if (dir->sys_dir) {
	free(dir->sys_dir);
	dir->sys_dir = NULL;
    }
    
    /* Local default directory (also was the last folder used on src/file.c,
          -- now !! is last folder used) */
    if (0 == access(".",READ_ACCESS)) {
	/* WARNING: add_dir_vector does not allocate strings -- 
	 *          it just assign pointers!
	 */
	set_comment(dir,
		    add_dir_vector(dir,
				   safe_strdup("."),new_string2(local_fs_charset,
								s2us(".")),
				   BROWSER_NOFOLDER),
		    DUMMY_current_directory);
    }
    
    switch(dir->sel_type) {
    case selection_folder:

	{
	    /* give_dt_estr_as_str adds / to end */
	    const char * folders_val = give_dt_estr_as_str(&folders_e,"maildir",
							   NULL,NULL);

	    /* Local folders directory */
	    if (folders_val && 0 == access(folders_val,READ_ACCESS)) {
		char * str = safe_strdup(folders_val);

		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!
		 */
		set_comment(dir,
			    add_dir_vector(dir,str,new_string2(system_charset,s2us("=")),
					   BROWSER_NOFOLDER),
			    DUMMY_Mail);
	    }
	}
	
	{
	    const char * default_val = 
		give_dt_estr_as_str(&defaultfile_e,
				    "incoming-mailbox",
				    NULL,NULL); /* XXX */

	    /* Incoming mailbox (may also be IMAP mailbox) */
	    if (default_val) {
		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!
		 */
		set_comment(dir,
			    add_dir_vector(dir,
					   safe_strdup(default_val),
					   new_string2(system_charset,s2us("!")),
					   BROWSER_NODIR),
			    DUMMY_INBOX);
			    
	    }	
	}

	{
	    const char * recvd_val = 
		give_dt_estr_as_str(&recvd_mail_e,
				    "receivedmail",
				    NULL,NULL); /* XXXX */

	    /*  ">" -- usually =received folder */
	    if (recvd_val && 0 == access(recvd_val,READ_ACCESS)) {
		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!
		 */
		set_comment(dir,
			    add_dir_vector(dir,
					   safe_strdup(recvd_val),
					   new_string2(system_charset,s2us(">")),
					   BROWSER_NODIR),
			    DUMMY_received);
	    }
	}

	{
	    const char * sent_val = give_dt_estr_as_str(&sent_mail_e,
							"sentmail",NULL,NULL);

	    /*  "<" -- usually =sent folder */
	    if (sent_val && 0 == access(sent_val,READ_ACCESS)) {
		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!
		 */
		set_comment(dir,
			    add_dir_vector(dir,
					   safe_strdup(sent_val),
					   new_string2(system_charset,s2us("<")),
					   BROWSER_NODIR),
			    DUMMY_sent);
	    }
	}

	{  /* Show extra mailbox dir if it is under home directory */


	    const char * mbx_dir = give_dt_estr_as_str(&extra_mailbox_dir_e,
						       "extra-mailbox-dir",
						       NULL,NULL);
	    int i;
	    
	    if (mbx_dir && 0 == access(mbx_dir,READ_ACCESS) &&
		home[0] && ( i = strlen(home) ) &&
		i < strlen(mbx_dir) &&
		0 == strncmp(mbx_dir,home,i) &&
		'/' == mbx_dir[i]) {

		struct string * X = new_string2(local_fs_charset,
						cs2us(mbx_dir+i));

		add_dir_vector(dir,
			       safe_strdup(mbx_dir),
			       format_string(FRM("~%S"),
					     X),
			       BROWSER_NOFOLDER);


		free_string(&X);
						       
	    }
	}


	break;
	
    case selection_file:
	/* Local root */
	if (0 == access("/",READ_ACCESS)) {
	    /* WARNING: add_dir_vector does not allocate strings -- 
	     *          it just assign pointers!
	     */

	    set_comment(dir,
			add_dir_vector(dir,
				       safe_strdup("/"),
				       new_string2(system_charset,s2us("/")),
				       BROWSER_NOFOLDER),
			DUMMY_root_directory);
	}

	{
	    const char * attachment_dir_val = 
		give_dt_estr_as_str(&attachment_dir_e,
				    "attachment-dir",NULL,NULL);

	    /* Attachment directory for {doc}/ */
	    if (attachment_dir_val &&
		0 == access(attachment_dir_val,READ_ACCESS)) {
		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!
		 */
		
		set_comment(dir,
			    add_dir_vector(dir,
					   elm_message(FRM("%s/"),attachment_dir_val),
					   new_string2(system_charset,s2us("{doc}/")),
					   BROWSER_NOFOLDER),
			    DUMMY_attachment);
	    }
	}
	break;

    case selection_url:
    case NUM_selection_type:
	/* Not used */
	break;	
    }

    for (x = 0; x < expand_tag_count; x++) {
	if (0 != (expand_tag_list[x].allow_mask & (1 << dir->sel_type)))  {
	    const char * tag_val =
		give_dt_estr_as_str(expand_tag_list[x].ptr,expand_tag_list[x].variable,
				    NULL,NULL);

	    if (tag_val && 0 == access(tag_val,READ_ACCESS)) {
		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!
		 */
		struct name_vector * entry = 
		    add_dir_vector(dir,
				   safe_strdup(tag_val),
				   new_string2(system_charset,cs2us(expand_tag_list[x].
								    tag)),
				   expand_tag_list[x].add_dir_vector_flags);

		if (expand_tag_list[x].dummy_text > DUMMY_none) {
		    set_comment(dir,entry,expand_tag_list[x].dummy_text);
		}
	    }
	}
    }
    
    /* User's local home directory */
    if (home[0] && 0 == access(home,READ_ACCESS)) {
	/* WARNING: add_dir_vector does not allocate strings -- 
	 *          it just assign pointers!
	 */

	char * str = safe_strdup(home);
	int l = strlen(str);
	if (str[l-1] != '/')
	    str = strmcat(str,"/");

	set_comment(dir,
		    add_dir_vector(dir,str,new_string2(system_charset,s2us("~")),
				   BROWSER_NOFOLDER),
		    DUMMY_home);
    }

   if (vector) {
	int i;

	set_comment(dir,
		    add_dir_vector(dir,
				   NULL,
				   new_string2(system_charset,s2us("#")),
				   BROWSER_NOFOLDER),
		    DUMMY_hashmark);
	
       
	for (i = 0; i < len; i++) {

	    if (hashmark_default_menu == hashmark_valid_on_user(vector[i])) {

		const struct string * N = 
		    hashmark_item_name(vector[i]);
		struct string *X = format_string(FRM("#%S"),N);

		/* WARNING: add_dir_vector does not allocate strings -- 
		 *          it just assign pointers!
		 */

		struct name_vector    *entry =
		    add_dir_vector(dir, 
				   /* Probably 'wrong' charset */
				   us2s(stream_from_string(X,1,NULL)),
				   X,
				   hashmark_browser_flags(vector[i]));

		if (!entry->a.dummy_browser)
		    entry->a.dummy_browser = malloc_DUMMY_BROWSER_entry();
		else if (DUMMY_BR_entry_magic != entry->a.dummy_browser->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"browser_gen_default_menu",
			  "Bad magic type (DUMMY_BROWSER_entry)",0);

		if (entry->a.dummy_browser->hashmark_item)
		    free_hashmark_item(& (entry->a.dummy_browser->hashmark_item));

		entry->a.dummy_browser->hashmark_item = vector[i];
		inc_hashmark_item_refcount(entry->a.dummy_browser->hashmark_item);
	    }

	    free_hashmark_item(& (vector[i]));
	}

	free(vector);
	vector = NULL;
   }
}

int browser_change_up_helper(X,sep)
     const struct string * X; 
     int sep;
{
    int L         = string_len(X);

    int L1  = 0;
    int idx;
    
    /* Ignore seperator if it is on last character ... */

    for (idx = 0; idx < L-1 ; idx++) {
	uint16 code = give_unicode_from_string(X,idx);
	
	if (sep == code) 
	    L1 = idx+1;
    }

    DPRINT(Debug,12,(&Debug,
		     "browser_change_up_helper=%d: sep=%04x X=%S\n",
		     L1,sep,X));

    return L1;
}

struct string * browser_disp_tail_helper(S,X,sep)
     const struct string *S;
     int X;
     int sep;
{
    struct string * ret = NULL;

    int L =  string_len(S);

    if (L > 0 && 
	give_unicode_from_string(S,L-1) == sep)
	L--;

    ret = clip_from_string(S,&X,L-X);

    DPRINT(Debug,12,(&Debug,
		     "browser_disp_tail_helper=%S: sep=%04x S=%S\n",
		     ret,sep,S));

    return ret;
}

S_(browser_change_up_dir browser_change_up_dummy)
static int browser_change_up_dummy P_((struct folder_browser *dir,
				       struct string **dispname,
				       struct string ** disp_tail));
static int browser_change_up_dummy(dir,dispname,disp_tail)
     struct folder_browser *dir;
     struct string **dispname;
     struct string ** disp_tail;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_change_up_dummy: dir=%p\n", dir));


    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__," browser_change_up_dummy",
	      "Bad magic type",0);


    if (dir->a.dummy_browser->remote) {
	/* Deselect remote mode ... */

	DPRINT(Debug,11,(&Debug,
			 "browser_change_up_dummy: Deselecting remote mode...\n"));
	free(dir->a.dummy_browser->remote);
	dir->a.dummy_browser->remote = NULL;
	if (dir->a.dummy_browser->passhm)
	    free_browser_passhm(& (dir->a.dummy_browser->passhm));

	ret = -1;
    } else {
	int L1  = browser_change_up_helper(dir->dirname,0x002F /* '/' */);

	if (0 == L1) {
	    ret = -1;
	} else {
	    int X = 0;
	    struct string * A1 = clip_from_string(dir->dirname,&X,L1);

	    if (disp_tail) 
		*disp_tail = browser_disp_tail_helper(dir->dirname,X,0x002F /* '/' */);
	    
	    if (*dispname)
		free_string(dispname);	    
	    *dispname = A1;

	    ret = 1;
	}

    }

    DPRINT(Debug,11,(&Debug,"browser_change_up_dummy=%d\n", ret));
    return ret;   
}

/* rel_dirname is relative to type -- not include user@hostname 

   rel_dirname is NULL is indicating that selection is on
   dir->a.dummy_browser->remote
 */

S_(browser_change_dir browser_change_dummy)
static int browser_change_dummy P_((struct folder_browser *dir,
				    const struct string *rel_dirname,
				    struct string **dispname));
static int browser_change_dummy(dir,rel_dirname,dispname)
     struct folder_browser *dir;
     const struct string *rel_dirname;
     struct string **dispname;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_change_dummy: dir=%p\n", dir));
    
    if (rel_dirname) {
	DPRINT(Debug,11,(&Debug,"browser_change_dummy: rel_dirname=%S\n", 
			 rel_dirname));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_change_dummy: rel_dirname=NULL\n"));
    }

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_change_dummy",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {     /* Do not change menu */

	switch(dir->sel_type) {

	case selection_file:
	    DPRINT(Debug,11,(&Debug,
			     "browser_change_dummy: File browser does not support remote directories\n"));

	    break;
	case selection_folder:
#ifdef REMOTE_MBX
	    /* On that point dir->a.dummy_browser->remote is user@host
	       indicating of POP or IMAP mailbox.
	       
	       But if user requires directory content from that it
	       must be IMAP server directory then.
	    */
	
	    if (dummy_to_imap(dir)) {			

		if (BROWSER_TYPE_magic != dir->type->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"browser_change_dummy",
			  "Bad magic number (browser_type) after dummy_to_imap",0);

		ret = dir->type->browser_change_it(dir,rel_dirname,dispname);
		goto clean;
	    }
#endif /* REMOTE_MBX */
	    break;
	case selection_url:
	case NUM_selection_type:
	    /* Not used */
	    break;
	}
	lib_error(CATGETS(elm_msg_cat, MeSet, MeRemoteDirNotAvail,
			  "Remote directries are not available in %s"),
		  dir->a.dummy_browser->remote);

    } else if (rel_dirname &&    
	       string_len(rel_dirname) > 0) {   
	/* Menu on case where there is no DIROPS ... */

	/* Dummy operation ... but for consistency ... */
	struct string * relative = browser_descend_dummy(dir,rel_dirname);

	struct string * Lstr =  convert_string(local_fs_charset,rel_dirname,0);
	unsigned char * str  = stream_from_string(Lstr,0,NULL);
	struct stat buf;

	if (relative)
	    free_string(&relative);

	DPRINT(Debug,12,(&Debug,
			 "browser_change_dummy: Trying stat %s\n",
			 str));

	if (0 != stat(us2s(str),&buf)) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirError,
			      "Directory %S: %s"),
		      Lstr, strerror(err));
#ifdef S_ISDIR
	} else if (!S_ISDIR(buf.st_mode)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeNoDirError,
			      "Not a directory: %S"),
		      Lstr);	    
#endif
	} else {
	    ret = 1;

	    /* Make empty selection folder */
	    clear_dir_vector(dir);

	    if (dir->dirname)
		free_string(&dir->dirname);
	    /* Copy just from edit buffer */
	    dir->dirname = dup_string(*dispname);

	    if (dir->sys_dir) {
		free(dir->sys_dir);
		dir->sys_dir = NULL;
	    }
 	    dir->sys_dir = us2s(str);
	    str = NULL;
	}

	if (Lstr)
	    free_string(&Lstr);
	if (str)
	    free(str);
    } else {       /* Default menu */
	ret = 1;
	browser_gen_default_menu(dir);
    }

#ifdef REMOTE_MBX
 clean:
#endif


    DPRINT(Debug,11,(&Debug,"browser_change_dummy=%d\n", ret));

    return ret;
}

int browser_select_generic(dir,relative_path,
			   rel_dirname,relative,Lstr,str,sep,default_charset)
     struct folder_browser *dir;
     struct string * relative_path;
     const struct string * rel_dirname;
     struct string * relative;
     struct string ** Lstr;
     char ** str;
     int sep;
     charset_t default_charset;
{
    int ret = -1;

    int i;
    int l1 = dir->sys_dir ? strlen(dir->sys_dir) : 0;
    int add_sep = 0;
    
    DPRINT(Debug,12,(&Debug,"browser_select_generic: dir=%p\n", 
		dir));
    
    if (!relative) {
	DPRINT(Debug,12,(&Debug,
			 "browser_select_generic: 'relative' (to directory) name is not given, using 'rel_dirname' (to service)\n"));
	goto recover;
    }

    if (sep == '\0' && l1 > 0) {
	DPRINT(Debug,12,(&Debug,
			 "browser_select_generic: Separator not given!\n"));
	goto recover;
    }
    
    if (l1 > 0 && sep != dir->sys_dir[l1-1]) {	
	add_sep = 1;
    }
    
    if (0 == string_len(relative)) {
	DPRINT(Debug,12,(&Debug,
			 "browser_select_generic: 'relative' is empty -- directory prefix is given\n"));
	ret = -2;

	if (! dir->sys_dir || !relative_path)
	    goto recover;
	
	
	*Lstr = dup_string(relative_path);
	if (add_sep)
	    fill_ascii_to_string(*Lstr,1,sep);
	
	*str = safe_strdup(dir->sys_dir);
	if (add_sep) {
	    char buf[2];
	    buf[0] = sep;
	    buf[1] = '\0';
	    *str = strmcat(*str,buf);
	}

	DPRINT(Debug,12,(&Debug,
			 "browser_select_generic: Using name %s (%S) from prefix, separator %c\n",
			 *str,*Lstr,sep));
	
	goto done;
    }


    /* Update cache ... */
    browser_vector_len(dir);

    for (i = 0; i < dir->vector_len; i++)
	if (0 == (string_cmp(relative,
			     dir->vector[i].disp_name,
			     -99 /* Unknown indicator */)))
	    break;

    if (i < dir->vector_len) {

	if (dir->sys_dir) {
	    *Lstr = dup_string(relative_path);
	    if (add_sep)
		fill_ascii_to_string(*Lstr,1,sep);
	    	    
	    append_string(Lstr, dir->vector[i].disp_name, 0);
		
	    *str = safe_strdup(dir->sys_dir);
	    if (add_sep) {
		char buf[2];
		buf[0] = sep;
		buf[1] = '\0';
		*str = strmcat(*str,buf);
	    }
	    *str = strmcat(*str,dir->vector[i].sys_name);
	} else {
	    *Lstr = dup_string(dir->vector[i].disp_name);
	    *str  = safe_strdup(dir->vector[i].sys_name);
	}

	ret = i;
	
	DPRINT(Debug,12,(&Debug,
			 "browser_select_generic: Using name %s (%S) from listing, separator %c\n",
			 *str,*Lstr,sep));
    } else {
    recover:
	*Lstr =  convert_string(default_charset,rel_dirname,0);
	*str  = us2s(stream_from_string(*Lstr,0,NULL));
	
	DPRINT(Debug,12,(&Debug,
			 "browser_select_generic: Using system name %s (%S)\n",
			 *str,*Lstr));
	if (rel_dirname)
	    DPRINT(Debug,11,(&Debug,"         ... rel_dirname was %S\n",
			     rel_dirname));

    }
    
    if (relative_path && relative)
	DPRINT(Debug,12,(&Debug,
			 "         ... relative path was %S and item %S\n",
			 relative_path,relative));
    
 done:
    DPRINT(Debug,12,(&Debug,"browser_select_generic=%d %s\n",
		     ret,
		     ret < 0 ? "(no index)" : "(index of listing)"));
    return ret;
}

S_(browser_select_dir browser_select_dummy)
static int browser_select_dummy P_((struct folder_browser *dir,
				    const struct string *rel_itemname,
				    struct string **dispname,
				    int *newpos));
static int browser_select_dummy(dir,rel_itemname,dispname,newpos)
     struct folder_browser *dir;
     const struct string *rel_itemname;
     struct string **dispname;
     int *newpos;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_select_dummy: dir=%p\n", dir));

    if (rel_itemname) {
	DPRINT(Debug,11,(&Debug,"browser_select_dummy: rel_itemname=%S\n", 
			 rel_itemname));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_select_dummy: rel_itemname=NULL\n"));
    }

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__," browser_select_dummy",
	      "Bad magic type",0);


    if (dir->a.dummy_browser->remote) {     /* Do not change menu */
	
	switch(dir->sel_type) {
	case selection_file:
	    DPRINT(Debug,11,(&Debug,
			     "browser_change_dummy: File browser does not support remote directories\n"));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeRemoteFileNotAvail,
			      "Remote file is not available in %s"),
		      dir->a.dummy_browser->remote);
	    break;
	case selection_folder:
#ifdef REMOTE_MBX
	    /* On that point dir->a.dummy_browser->remote is user@host
	       indicating of POP or IMAP mailbox.
	       
	       But if rel_itemname is non-NULL, then it must be on
	       IMAP server directory...
	    */
	
	    if (rel_itemname) {
		/* IMAP directory then ... */
		
		if (dummy_to_imap(dir)) {
		    
		    if (BROWSER_TYPE_magic != dir->type->magic)
			panic("MBX PANIC",__FILE__,__LINE__,"browser_change_dummy",
			      "Bad magic number (browser_type) after dummy_to_imap",0);

		    ret = dir->type->browser_select_it(dir,rel_itemname,
						       dispname,newpos);
		    goto clean;
		}	   
	    } else {
		/* Assume OK */
		
		/* Null selection when remote mode ... */
		set_dir_selection(dir,NULL,NULL,
				  BROWSER_EXIST       /* Assume existance */
				  |BROWSER_MAILFILE   /* Assume folder    */
				  );
		ret = 1;
		goto clean;
	    }	
#endif /* REMOTE_MBX */	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeRemoteFolderNotAvail,
			      "Remote folder is not available in %s"),
		      dir->a.dummy_browser->remote);
	    break;

	case selection_url:
	case NUM_selection_type:
	    /* Not used */
	    break;

	}    

    } else if (rel_itemname) {
	
	/* Make selection relative ... */
	struct string * relative = browser_descend_dummy(dir,rel_itemname);
	
	if (relative) {
	    ret = real_select_local(dir,rel_itemname,relative,newpos);
	    free_string(&relative);
	}       
    }

#ifdef REMOTE_MBX
 clean:
#endif
    
    DPRINT(Debug,11,(&Debug,"browser_select_dummy=%d\n", ret));
    
    return ret;
}


S_(browser_give_title_dir browser_give_title_dummy)
static struct string * browser_give_title_dummy P_((struct folder_browser *dir));
static struct string * browser_give_title_dummy(dir)
     struct folder_browser *dir;
{
    static struct string *ret;

    DPRINT(Debug,11,(&Debug,"browser_give_title_dummy: dir=%p\n", dir));

    if (dir->dirname)
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeNoDirListing,
				    "Directory listing for %S not available"),
				    dir->dirname);
    else 
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeDefaultDir,
				    "Default directory"));

    DPRINT(Debug,11,(&Debug,"browser_give_title_dummy=%S\n", ret));
    
    return ret;
}

S_(browser_separator_dir browser_separator_dummy)
static char browser_separator_dummy P_((struct folder_browser *dir));
static char browser_separator_dummy(dir)
     struct folder_browser *dir;
{

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_separator_dummy",
	      "Bad magic type",0);
       
    if (dir->a.dummy_browser->remote)
	return '\0';
    else
	return '/';
}

S_(browser_name_dir browser_name_dummy)
static struct string * browser_name_dummy P_((struct folder_browser *dir));
static struct string * browser_name_dummy(dir)
     struct folder_browser *dir;
{
    static struct string *ret;

    DPRINT(Debug,11,(&Debug,"browser_name_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_name_dummy",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeRemoteServer,
				    "Remote server %s"),
			    dir->a.dummy_browser->remote);
    } else if (dir->dirname) {
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeLocalDir,
				    "Local directory %S"),
			    dir->dirname);
    } else {
	ret = format_string(CATGETS(elm_msg_cat, MeSet,MeDefaultDir,
				    "Default directory"));
    }

    DPRINT(Debug,11,(&Debug,"browser_name_dummy=%S\n", ret));
    return ret;
}

S_(browser_cat_dir browser_cat_dummy)
static struct string * browser_cat_dummy P_((struct folder_browser *dir,
					     const struct string * item));
static struct string * browser_cat_dummy(dir,item)
     struct folder_browser *dir;
     const struct string * item;
{
    struct string * ret = NULL;
    int L;

    DPRINT(Debug,11,(&Debug,"browser_cat_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_cat_dummy",
	      "Bad magic type",0);

    
    if (dir->a.dummy_browser->remote) {
	switch(dir->sel_type) {
	case selection_file:
	    DPRINT(Debug,11,(&Debug,
			     "browser_cat_dummy: File browser does not support remote directories\n"));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeRemoteFileNotAvail,
			      "Remote file is not available in %s"),
		      dir->a.dummy_browser->remote);
	    break;
	case selection_folder:
#ifdef REMOTE_MBX
	    /* Must have IMAP folder !! */
	    if (dummy_to_imap(dir)) {

		if (BROWSER_TYPE_magic != dir->type->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"browser_cat_dummy",
			  "Bad magic number (browser_type) after dummy_to_imap",0);

		ret = dir->type->browser_cat_it(dir,item);
		goto clean;
	    }
#endif
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeRemoteFolderNotAvail,
			      "Remote folder is not available in %s"),
		      dir->a.dummy_browser->remote);
	    break;
	case selection_url:
	case NUM_selection_type:
	    /* Not used */
	    break;
	}
	goto clean;
    }

    if (dir->dirname &&
	(L = string_len(dir->dirname)) > 0) {
	
	ret = dup_string(dir->dirname);
	if (L > 0 && 
	    0x002F /* '/' */  !=  give_unicode_from_string(dir->dirname,
							   L-1))
	    fill_ascii_to_string(ret,1,'/');

	/* Some local directory! */
	if (item) {
	   append_string(&ret, item, 0);
	}
	
    } else {
	/* Default MENU ! */

	if (item)
	    ret = dup_string(item);
	else
	    ret = new_string(system_charset);
    }
    
 clean:
    if (ret) {
	DPRINT(Debug,11,(&Debug,"browser_cat_dummy=%S\n",ret));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_cat_dummy=NULL\n"));
    }
    return ret;
}

S_(browser_folder_from_dir browser_folder_from_dummy)
static struct folder_info * 
browser_folder_from_dummy P_((struct folder_browser *dir,
			      int treat_as_spooled));
static struct folder_info * browser_folder_from_dummy(dir, treat_as_spooled)
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info * res = NULL;
    struct cancel_data *main_cancel = NULL /* Used if dns lookup was
					      cancelable */;
    
    DPRINT(Debug,11,(&Debug,"browser_folder_from_dummy: dir=%p\n", 
		     dir));

    if (dir->sel_type != selection_folder)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_folder_from_dummy",
	      "Bad selection type",0);

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_folder_from_dummy",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {
#ifdef REMOTE_MBX

	if (dir->selection->sys_name || (dir->sys_dir && dir->sys_dir[0])) {
	    /* Must have IMAP folder !! */
	    if (dummy_to_imap(dir)) {

		if (BROWSER_TYPE_magic != dir->type->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"browser_folder_from_dummy",
			  "Bad magic number (browser_type) after dummy_to_imap",0);

		res = dir->type->browser_folder_from_it(dir,
							treat_as_spooled);
		goto clean;
	    }
	} else {
	    struct remote_account X;
	    struct service_entry *se       = NULL;
	    struct browser_passhm *passhm = NULL;

	    int ret_dummy = 0;
	    
	    zero_remote_account(&X);

	    ret_dummy = dummy_give_remote_account(dir,&X,&se,
						  hmcon_none  /* IMAP or POP */,
						  &passhm,
						  &main_cancel);
	    
	    if (ret_dummy) {
		
		DPRINT(Debug,15,(&Debug,
				 "browser_folder_from_dummy: Using user=%s host=%s\n",
				 X.username,X.host));
	    
		res = mbx_new_folder();
		
		res -> cur_folder_sys = 
		    elm_message(FRM("%s@%s"),
				X.username,X.host);
		res -> cur_folder_disp =
		    format_string(FRM("%s@%s"),
				  X.username,X.host);

		/* May free X (if succeed) or copy
		   If make copy of X, then X is zeroed 
		*/
		if (!make_remote_mbox(res,&X,se,NULL,1,
				      passhm,
				      main_cancel)) {
		    
		    res -> folder_type = NO_NAME;
		    res->folder_type->init_it(res);
		}	
           
		free_service_entry(&se);
		clear_remote_account(&X);
		if (passhm)
		    free_browser_passhm(&passhm);
		
		goto clean;		
	    } 
	    
            /* 'se' can be NULL */
	    free_service_entry(&se);
	    
	    clear_remote_account(&X);
	    if (passhm)
		free_browser_passhm(&passhm);

	    if (main_cancel && is_canceled(main_cancel)) {
		DPRINT(Debug,12,(&Debug,
				 "browser_folder_from_dummy: DNS lookup canceled\n"));
		goto clean;
	    } 
	}
#endif
	lib_error(CATGETS(elm_msg_cat, MeSet, MeRemoteFolderNotAvail,
			  "Remote folder is not available in %s"),
		  dir->a.dummy_browser->remote);

    } else 
	res = real_folder_from_local(dir, treat_as_spooled,0);

#ifdef REMOTE_MBX
 clean:
#endif
    if (main_cancel)
	free_cancel(& (main_cancel));
           
    if (res) {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_dummy=%p\n",res));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_folder_from_dummy=NULL\n"));
    }
    return res;
}


S_(browser_selection_is_folder browser_selection_is_dummy)
static int browser_selection_is_dummy P_((struct folder_browser *dir,
					  struct folder_info *folder));

static int browser_selection_is_dummy(dir,folder)
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int ret = 0;
    DPRINT(Debug,11,(&Debug,"browser_selection_is_dummy: dir=%p\n", 
		     dir));

    if (dir->sel_type != selection_folder)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_selection_is_dummy",
	      "Bad selection type",0);
    
    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_selection_is_dummy",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {
#ifdef REMOTE_MBX
	if (folder->folder_type == POP_MBX) {
	    DPRINT(Debug,11,(&Debug,
			     "browser_selection_is_dummy: Maybe POP ...\n"));
	    ret = 0 == strcmp(dir->a.dummy_browser->remote,
			      folder->cur_folder_sys);

	} else if (folder->folder_type == IMAP_MBX) {
	    DPRINT(Debug,11,(&Debug,
			     "browser_selection_is_dummy: Maybe IMAP ...\n"));  
	    if (dummy_to_imap(dir)) {

		if (BROWSER_TYPE_magic != dir->type->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"browser_selection_is_dummy",
			  "Bad magic number (browser_type) after dummy_to_imap",0);

		ret = dir->type->browser_selection_is_it(dir,folder);
	    } else {
		DPRINT(Debug,11,(&Debug,
				 "browser_selection_is_dummy: IMAP failed\n"));
		ret = 0;
	    }
	} else {
	   DPRINT(Debug,11,(&Debug,
			    "browser_selection_is_dummy: Folder is not remote\n"));
	    ret = 0;
	} 	    
#else
	DPRINT(Debug,11,(&Debug,
			 "browser_selection_is_dummy: Remote selection is not available\n"));
	ret = 0;
#endif
    } else {	
	ret = real_selection_is_local(dir,folder,0);
    }
    
    DPRINT(Debug,11,(&Debug,
		     "browser_selection_is_dummy=%d\n",ret));
    return ret;
}

int create_as_user(name)
     const char *name;
{
    int the_stat = 0, pid, w, sig;
    volatile int ret = 0;

    S__ status;
	
    if ((pid = 
#ifdef VFORK
	 vfork()
#else
	 fork()
#endif
	 ) == 0) {
	int filedes;

	if (-1 == setgid(groupid)) {
	    int err = errno;
	    fprintf(stderr,"create_as_user: setgid(%d) FAILED: %s\n",
		    groupid,strerror(err));
	    fflush(stderr);
	    _exit(err != 0? err : 1);	/* never return zero! */
	}
	if (-1 == setuid(userid)) {		/** back to normal userid **/
	    int err = errno;
	    fprintf(stderr,"create_as_user: setuid(%d) FAILED: %s\n",
		    userid,strerror(err));
	    fflush(stderr);
	    _exit(err != 0? err : 1);	/* never return zero! */
	}
	errno = 0;
	filedes = open(name, O_WRONLY | O_CREAT | O_EXCL, 0600);
	
	if (filedes < 0)
	    _exit(errno);
	else {
	    close(filedes);
	    _exit(0);
	}
    }
    
    errno = 0;
    while ((w = my_wait(pid,&status)) != pid && 
	   (w != -1 || EINTR == errno))
	;
    
    sig = convert_status(status,&the_stat);
    if (sig)
	ret = 0;
    else if (0 == the_stat)
	ret = 1;
    else
	lib_error(CATGETS(elm_msg_cat, MeSet, MeCreatingFileFailed,
			  "Creating of file %s failed: %s"),
		  name,strerror(the_stat));
    
    return ret;
}

S_(browser_create_selection_dir browser_create_selection_dummy)
static int browser_create_selection_dummy P_((struct folder_browser *dir));

static int browser_create_selection_dummy(dir)
     struct folder_browser *dir;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,
		     "browser_create_selection_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_create_selection_dummy",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {
	lib_error(CATGETS(elm_msg_cat, MeSet, MeCreatingRemoteFolderNotSupp,
			  "Creating of remote folder for %s not supported"),
		  dir->a.dummy_browser->remote);
	
    } else {
	ret = real_create_selection_local(dir,0);
    }

    DPRINT(Debug,11,(&Debug,"browser_create_selection_dummy=%d\n", ret));
    
    return ret;
}



/* We use same routines zero_ws_fields_local and free_ws_fields_local
 * both dummy and local browser!
 */
E_(zero_ws_fields_browser zero_ws_fields_local)
void zero_ws_fields_local(ptr)
     WRITE_STATE ptr;
{
    ptr->a.local.save_fd   = -1;
    ptr->a.local.save_file = NULL;
}

E_(free_ws_fields_browser free_ws_fields_local)
void free_ws_fields_local(ptr)
     WRITE_STATE ptr;
{
    /* ptr->a.local.save_file and ptr->a.local.save_fd shares same
       file descriptor!
    */

    if (ptr->a.local.save_file) {
	fclose(ptr->a.local.save_file);
	ptr->a.local.save_file = NULL;
	ptr->a.local.save_fd   = -1;
    }
    if (-1 != ptr->a.local.save_fd) {
	close(ptr->a.local.save_fd);
	ptr->a.local.save_fd  = -1;
    }
}

S_(browser_prepare_write_dir browser_prepare_write_dummy)
static int browser_prepare_write_dummy P_((struct folder_browser *dir,
					   WRITE_STATE ptr));
static int browser_prepare_write_dummy(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_prepare_write_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_prepare_write_dummy",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {

	switch(dir->sel_type) {
	case selection_file:
	    DPRINT(Debug,11,(&Debug,
			"browser_prepare_write_dummy: File browser does not support remote directories\n"));
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeWritingRemoteFileNotSupp,
			      "Writing to remote file %s not supported"),
		      dir->a.dummy_browser->remote);   
	    break;
	case selection_folder:
#ifdef REMOTE_MBX
	    /* Must have IMAP folder !! */
	    
	    if (BROWSER_TYPE_magic != dir->type->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"browser_prepare_write_dummy",
		      "Bad magic number (browser_type)",0);

	    dir->type->free_ws_fields_it(ptr);       /* Need reinit of state */
	    if (dummy_to_imap(dir)) {

		if (BROWSER_TYPE_magic != dir->type->magic)
		    panic("MBX PANIC",__FILE__,__LINE__,"browser_prepare_write_dummy",
			  "Bad magic number (browser_type) after dummy_to_imap",0);

		dir->type->zero_ws_fields_it(ptr);   /* Hopefully ok ....... */
		ret = dir->type->browser_prepare_write_it(dir,ptr);
		goto clean;
	    }
#endif /* REMOTE_MBX */
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeWritingRemoteFolderNotSupp,
			      "Writing to remote folder %s not supported"),
		      dir->a.dummy_browser->remote);
	    break;
	case selection_url:
	case NUM_selection_type:
	    /* Not used */
	    break;
	}

    } else if (dir->selection->sys_name) {
	char * name = NULL;
	
	if (dir->sys_dir && dir->sys_dir[0])
	    name = elm_message(FRM("%s/%s"),
					    dir->sys_dir,
					    dir->selection->sys_name);
	else
	    name = safe_strdup(dir->selection->sys_name);

	ret = real_prepare_write_local(dir,ptr,name);

	free(name);

    } else {

	DPRINT(Debug,11,(&Debug,
			 "browser_prepare_write_dummy: No selection name\n"));
    }


#ifdef REMOTE_MBX
 clean:
#endif

    DPRINT(Debug,11,(&Debug,"browser_prepare_write_dummy=%d\n", ret));

    return ret;
}

S_(browser_sync_write_dir browser_sync_write_dummy)
static int browser_sync_write_dummy P_((struct folder_browser *dir,
					WRITE_STATE ptr));
static int browser_sync_write_dummy(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_sync_write_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_sync_write_dummy",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_sync_write_dummy",
	      "Bad remote argument",0);
    } else if (dir->selection->sys_name) {
	char * name = NULL;
	
	if (dir->sys_dir && dir->sys_dir[0])
	    name = elm_message(FRM("%s/%s"),
					    dir->sys_dir,
					    dir->selection->sys_name);
	else
	    name = safe_strdup(dir->selection->sys_name);

	ret = real_sync_write_local(dir,ptr,name);

	free(name);
    } else {

	DPRINT(Debug,11,(&Debug,
			 "browser_sync_write_dummy: No selection name\n"));

    }
 
    DPRINT(Debug,11,(&Debug,"browser_sync_write_dummy=%d\n", ret));
    
    return ret;
}

S_(browser_end_write_dir browser_end_write_dummy)
static int browser_end_write_dummy P_((struct folder_browser *dir,
			      WRITE_STATE ptr));
static int browser_end_write_dummy(dir,ptr)
     struct folder_browser *dir;
     WRITE_STATE ptr;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_end_write_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_end_write_dummy",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_end_write_dummy",
	      "Bad remote argument",0);
    } else if (dir->selection->sys_name) {
	char * name = NULL;
	
	if (dir->sys_dir && dir->sys_dir[0])
	    name = elm_message(FRM("%s/%s"),
					    dir->sys_dir,
					    dir->selection->sys_name);
	else
	    name = safe_strdup(dir->selection->sys_name);

	ret = real_end_write_local(dir,ptr,name);

	free(name);
    } else {
	DPRINT(Debug,11,(&Debug,
			 "browser_end_write_dummy: No selection name\n"));


    }
 
    DPRINT(Debug,11,(&Debug,"browser_end_write_dummy=%d\n", ret));
    
    return ret;
}

S_(browser_tell_dir_ws browser_tell_dummy_ws)
static long browser_tell_dummy_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr));
static long browser_tell_dummy_ws(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    long ret = -1L;

    DPRINT(Debug,11,(&Debug,"browser_tell_dummy_ws: dir=%p\n", 
		     dir));


    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_tell_dummy_ws",
	      "Bad magic type",0);

    if (dir->a.dummy_browser->remote) {
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_tell_dummy_ws",
	      "Bad remote argument",0);
    } 

    ret = real_browser_tell_ws(dir,write_state_ptr);
    
    DPRINT(Debug,11,(&Debug,"browser_tell_dummy_ws=%ld\n",ret));

    return ret;
}

S_(browser_seek_dir_ws browser_seek_dummy_ws)
/* Returns 0 on failure */
static int browser_seek_dummy_ws P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      long pos));
static int browser_seek_dummy_ws(dir,write_state_ptr,pos)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     long pos;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_seek_dummy_ws: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_seek_dummy_ws",
	      "Bad magic type",0);


    if (dir->a.dummy_browser->remote) {
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_seek_dummy_ws",
	      "Bad remote argument",0);
    } 

    ret = real_browser_seek_ws(dir,write_state_ptr,pos);
    
    DPRINT(Debug,11,(&Debug,"browser_seek_dummy_ws=%d\n",ret));
    return ret;
}

S_(browser_write_dir_ws browser_write_dummy_ws)
static int browser_write_dummy_ws P_((struct folder_browser *dir,
				      WRITE_STATE ptr,
				      int l, const char *buffer));
static int browser_write_dummy_ws(dir,ptr,l,buffer)
     struct folder_browser *dir;
     WRITE_STATE ptr;
     int l; 
     const char *buffer;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_write_dummy_ws: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_write_dummy_ws",
	      "Bad magic type",0);


    if (dir->a.dummy_browser->remote) {
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_write_dummy_ws",
	      "Bad remote argument",0);
    } 
    ret = real_browser_write_ws(dir,ptr,l,buffer);
    
    DPRINT(Debug,11,(&Debug,"browser_write_dummy_ws=%d\n",ret));
    return ret;
}



S_(browser_start_we_dir browser_start_we_dummy)
static int browser_start_we_dummy P_((struct folder_browser *dir,
				      WRITE_STATE write_state_ptr,
				      int write_envelope,
				      struct header_rec *current_header,
				      int *env_flags));
static int browser_start_we_dummy(dir,write_state_ptr,
				  write_envelope,current_header,
				  env_flags)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
     int *env_flags;
{
    int ret = 1;

    DPRINT(Debug,11,(&Debug,"browser_start_we_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_start_we_dummy",
	      "Bad magic type",0);


    if (dir->a.dummy_browser->remote) {
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_start_we_dummy",
	      "Bad remote argument",0);
    } 

    *env_flags = 0;

    if (write_envelope)
	ret = real_start_we_local(dir,write_state_ptr,
				  current_header, env_flags);
	    
    DPRINT(Debug,11,(&Debug,"browser_start_we_dummy=%d\n",ret));
    return ret;
}


S_(browser_end_we_dir browser_end_we_dummy)
static int browser_end_we_dummy P_((struct folder_browser *dir,
				  WRITE_STATE write_state_ptr,
				  int write_envelope,
				  struct header_rec *current_header));
static int browser_end_we_dummy(dir,write_state_ptr,
			      write_envelope,current_header)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
{
    int ret = 1;
    DPRINT(Debug,11,(&Debug,"browser_end_we_dummy: dir=%p\n", 
		dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_end_we_dummy_ws",
	      "Bad magic type",0);


    if (dir->a.dummy_browser->remote) {
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_end_we_dummy",
	      "Bad remote argument",0);
    } 

    if (write_envelope)
	ret = real_end_we_local(dir,write_state_ptr,current_header);

    DPRINT(Debug,11,(&Debug,"browser_end_we_dummy=%d\n",ret));
    return ret;
}


S_(browser_make_ref_folder browser_make_ref_dummy)
static int browser_make_ref_dummy P_((struct folder_browser *dir,
				      char **refname, int *iscopy,
				      int is_text));
static int browser_make_ref_dummy(dir,refname,iscopy,is_text)
     struct folder_browser *dir;
     char **refname; 
     int *iscopy;
     int is_text;
{
    int ret = 0;

    DPRINT(Debug,11,(&Debug,"browser_make_ref_dummy: dir=%p\n", 
		     dir));

    if (dir->sel_type != selection_file)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "browser_make_ref_dummy",
	      "Bad selection type",0);

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_make_ref_dummy",
	      "Bad magic type",0);


    if (dir->a.dummy_browser->remote) {
       	DPRINT(Debug,11,(&Debug,
			 "browser_make_ref_dummy: File browser does not support remote directories\n"));
	
	lib_error(CATGETS(elm_msg_cat, MeSet, MeRemoteFileNotAvail,
			  "Remote file is not available in %s"),
		      dir->a.dummy_browser->remote);
    } else {
	ret = real_make_ref_local(dir,refname,iscopy,is_text); 
    }

    DPRINT(Debug,11,(&Debug,"browser_make_ref_dummy=%d\n",ret));
    
    return ret;
}

static void browser_update_dummy P_((struct folder_browser *dir));
static void browser_update_dummy(dir)
     struct folder_browser *dir;
{
    panic("BROWSER PANIC",__FILE__,__LINE__,"browser_update_dummy",
	  "browser_update_dummy() called",0);	
}

/* Call only if BROWSER_NEEDSTAT is set */
static void browser_dummy_do_stat P_((struct folder_browser *dir,
				      int idx));
static void browser_dummy_do_stat(dir,idx)
     struct folder_browser *dir;
     int idx;
{
    panic("BROWSER PANIC",__FILE__,__LINE__,"browser_dummy_do_stat",
	  "browser_dummy_do_stat() called",0);	
}


S_(browser_folder_sort_dir browser_folder_sort_dummy)
static void browser_folder_sort_dummy P_((struct folder_browser *dir,
					  print_sort_message * print));
static void browser_folder_sort_dummy(dir,print)
     struct folder_browser *dir;
     print_sort_message * print;
{
    /* Nothing */
}

S_(browser_free_name_vector_dir browser_free_name_vector_dummy)
static void browser_free_name_vector_dummy P_((struct folder_browser *dir,
					       struct name_vector *entry));
static void browser_free_name_vector_dummy(dir,entry)
     struct folder_browser *dir;
     struct name_vector *entry;
{
    if (entry->a.dummy_browser)
	free_DUMMY_BROWSER_entry(& (entry->a.dummy_browser));
}

/* Increments refcount */
S_(browser_gethm_name_vector_dir browser_gethm_name_vector_dummy)
static struct hashmark_item *
   browser_gethm_name_vector_dummy P_((struct folder_browser * dir,
				       struct name_vector    * entry));
static struct hashmark_item * browser_gethm_name_vector_dummy(dir,entry)
     struct folder_browser * dir;
     struct name_vector    * entry;
{
    DPRINT(Debug,11,(&Debug,"browser_gethm_name_vector_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_gethm_name_vector_dummy",
	      "Bad magic type",0);
    
    if (entry->a.dummy_browser) {
	if (DUMMY_BR_entry_magic != entry->a.dummy_browser->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_gethm_name_vector_dummy",
		  "Bad magic type",0);

	if (entry->a.dummy_browser->hashmark_item) {

	    inc_hashmark_item_refcount(entry->a.dummy_browser->hashmark_item);

	    DPRINT(Debug,11,(&Debug,"browser_gethm_name_vector_dummy=%p\n",
			     entry->a.dummy_browser->hashmark_item));

	    return entry->a.dummy_browser->hashmark_item;
	}
    }

    DPRINT(Debug,11,(&Debug,"browser_gethm_name_vector_dummy=0\n"));
    return 0;
}

    
S_(browser_sethm_name_vector_dir browser_sethm_name_vector_dummy)
static  int browser_sethm_name_vector_dummy P_((struct folder_browser * dir,
						struct name_vector    * entry,
						struct hashmark_item  * hm
						));

static int browser_sethm_name_vector_dummy(dir,entry,hm)
     struct folder_browser * dir;
     struct name_vector    * entry;
     struct hashmark_item  * hm;
{
    DPRINT(Debug,11,(&Debug,"browser_sethm_name_vector_dummy: dir=%p\n", 
		     dir));

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_sethm_name_vector_dummy",
	      "Bad magic type",0);

    if (!entry->a.dummy_browser)
	entry->a.dummy_browser = malloc_DUMMY_BROWSER_entry();
    else if (DUMMY_BR_entry_magic != entry->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_sethm_name_vector_dummy",
	      "Bad entry magic type",0);

    if (entry->a.dummy_browser->hashmark_item)
	free_hashmark_item(& (entry->a.dummy_browser->hashmark_item));
    entry->a.dummy_browser->hashmark_item = hm;
    inc_hashmark_item_refcount( entry->a.dummy_browser->hashmark_item);
    
    DPRINT(Debug,11,(&Debug,"browser_sethm_name_vector_dummy=1\n"));
    return 1;
}
    
S_(browser_line_idx_dir browser_line_idx_dummy)
static const struct string * browser_line_idx_dummy 
                      P_((struct folder_browser *dir,
			  struct name_vector *dummy,
			  const struct string  **comment_p));
static const struct string * browser_line_idx_dummy(dir,entry,comment_p)
     struct folder_browser *dir;
     struct name_vector *entry;
     const struct string  **comment_p;
{
    const struct string * res = NULL;

    enum DUMMY_special comment;

    if (! entry->a.dummy_browser)
	return NULL;

    if (DUMMY_BR_entry_magic != entry->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_line_idx_dummy",
	      "Bad magic type (DUMMY_BROWSER_entry)",0);

    comment = entry->a.dummy_browser->comment;

    if (DUMMY_none != comment) {

	if (comment < 0 || comment >= DUMMY_count)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_line_idx_dummy",
		  "Bad comment index",0);
	
	if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_line_idx_dummy",
		  "Bad dir magic type",0);

	if (comment_p)
	    *comment_p = dir->a.dummy_browser->comment[comment];

    } else if (entry->a.dummy_browser->hashmark_item) {
	
	if (comment_p)
	    *comment_p = hashmark_item_description(entry->a.dummy_browser->hashmark_item);

    }

    if (comment_p && *comment_p) {
	DPRINT(Debug,11,(&Debug,"browser_line_idx_dummy: comment=%S\n",
			 *comment_p));
    }

    if (res) {
	DPRINT(Debug,11,(&Debug,"browser_line_idx_dummy=%S\n",
			 res));
    } else {
	DPRINT(Debug,11,(&Debug,"browser_line_idx_dummy=NULL\n"));
    }
    
    return res;
}


S_(browser_reload_dir browser_reload_dummy)
static int browser_reload_dummy P_((struct folder_browser *dir));
static int browser_reload_dummy(dir)
     struct folder_browser *dir;
{

    int ret = 0;

    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__," browser_reload_dummy",
	      "Bad magic type",0);
 
    if (dir->a.dummy_browser->remote) {     /* Do not change menu */

	lib_error(CATGETS(elm_msg_cat, MeSet, MeRemoteDirNotAvail,
			  "Remote directries are not available in %s"),
		  dir->a.dummy_browser->remote);

    } else if (dir->sys_dir && dir->sys_dir[0]) {

	struct stat buf;
	char * str = dir->sys_dir;

	/* Menu on case where there is no DIROPS ... */

	if (0 != stat(str,&buf)) {
	    int err = errno;
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeDirError,
			      "Directory %S: %s"),
		      dir->dirname, strerror(err));
#ifdef S_ISDIR
	} else if (!S_ISDIR(buf.st_mode)) {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeNoDirError,
			      "Not a directory: %S"),
		      dir->dirname);	    
#endif
	} else {
	    ret = 1;
	    
	    /* Make empty selection folder */
	    clear_dir_vector(dir);

	}

    } else {       /* Default menu */
	ret = 1;
	browser_gen_default_menu(dir);
    }


    return ret;

}


struct browser_type dummy_browser = {
					     BROWSER_TYPE_magic, 
					     browser_zero_dummy,
					     browser_free_dummy,
					     browser_change_dummy,
					     browser_give_title_dummy,
					     browser_separator_dummy,
					     browser_name_dummy,
					     browser_cat_dummy,
					     browser_select_dummy,
					     browser_folder_from_dummy,
					     browser_change_v_dummy,
					     browser_change_up_dummy,
					     browser_create_selection_dummy,
					     zero_ws_fields_local,
					     free_ws_fields_local,
					     browser_prepare_write_dummy,
					     browser_end_write_dummy,
					     browser_tell_dummy_ws,
					     browser_seek_dummy_ws,
					     browser_write_dummy_ws,
					     browser_start_we_dummy,
					     browser_end_we_dummy,
					     browser_selection_is_dummy,
					     browser_make_ref_dummy,
					     browser_update_dummy,
					     browser_dummy_do_stat,
					     browser_sync_write_dummy,
					     browser_folder_sort_dummy,
					     browser_free_name_vector_dummy,
					     browser_reset_filter_generic,
		/* Not called, select_dir_item_by_idx() is used
		   only on fbrowser (Elm 2.5 stype file browser) */
					     browser_select_idx_generic,
					     browser_line_idx_dummy,
					     browser_reload_dummy,
					     browser_gethm_name_vector_dummy,
					     browser_sethm_name_vector_dummy
};


/* ---------------------------------------------------------------------- */

/* Folder browser */

void clear_dir_selection(dir)
     struct folder_browser *dir;
{

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"clear_dir_selection",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,12,(&Debug,
		     "clear_dir_selection: dir=%p; type=%p\n",
		     dir,dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"clear_dir_selection",
	      "Bad magic number (browser_type)",0);

    if (dir->selection) {
	if (dir->selection->sys_name) {
	    free(dir->selection->sys_name);
	    dir->selection->sys_name = NULL;
	}
	if (dir->selection->disp_name) 
	    free_string(&(dir->selection->disp_name));


	dir->type->browser_free_name_vector_it(dir,dir->selection);

	free(dir->selection);
	dir->selection = NULL;
    }
}

/* WARNING: set_dir_selection does not allocate strings -- 
 *          it just assign pointers!
 */
struct name_vector * set_dir_selection(dir,sys_name,disp_name,flags)
     struct folder_browser *dir;
     char *sys_name;
     struct string * disp_name;
     int flags;
{

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"set_dir_selection",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,12,(&Debug,
		     "set_dir_selection: dir=%p; type=%p%s\n",
		     dir,dir->type,
		     dir->selection ? " (clearing previous selection)" : ""));

    if (dir->selection) 
	clear_dir_selection(dir);

    DPRINT(Debug,12,(&Debug,
		     "set_dir_selection: sys_name=%s, flags=%X",
		     sys_name ? sys_name : "<NULL>",flags));
    if (disp_name) {
	DPRINT(Debug,12,(&Debug,
			 ", disp_name=%S\n",disp_name));
    } else {
	DPRINT(Debug,12,(&Debug,
			 ", disp_name=NULL\n"));
    }

    dir->selection = safe_zero_alloc(sizeof (struct name_vector));

    dir->selection->sys_name  = sys_name;
    dir->selection->disp_name = disp_name;
    dir->selection->flags     = flags;    
    dir->selection->a.dummy   = NULL;

    return dir->selection;

}

void clear_dir_vector(dir)
     struct folder_browser *dir;
{
    int i;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"clear_dir_vector",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,12,(&Debug,
		     "clear_dir_vector: dir=%p; type=%p\n",
		dir,dir->type));
    
    if (dir->vector) {
	if (dir->vector_len < 0) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"clear_dir_vector",
		  "Bad vector len (non-null vector)",0);	
	}

	for (i = 0; i < dir->vector_len; i++) {
	    if (dir->vector[i].sys_name) {
		free(dir->vector[i].sys_name);
		dir->vector[i].sys_name = NULL;
	    }
	    if (dir->vector[i].disp_name) 
		free_string(&(dir->vector[i].disp_name));

	    if (BROWSER_TYPE_magic != dir->type->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"clear_dir_vector",
		      "Bad magic number (browser_type)",0);

	    dir->type->browser_free_name_vector_it(dir,
						   & (dir->vector[i]));

	}
	free(dir->vector);
	dir->vector     = NULL;
	dir->vector_len = 0;
    } else {
	if (dir->vector_len > 0) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"clear_dir_vector",
		  "Bad vector len (null vector)",0);	
	}
	dir->vector_len = 0;
    }
}

/* WARNING: add_dir_vector does not allocate strings -- 
 *          it just assign pointers!
 */
struct name_vector *  add_dir_vector(dir, sys_name, disp_name, flags)
     struct folder_browser *dir;
     char *sys_name;
     struct string * disp_name;
     int flags;
{
    struct name_vector * ret = NULL;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"add_dir_vector",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,12,(&Debug,
		     "add_dir_vector: dir=%p; type=%p\n",
		     dir,dir->type));

    if (dir->vector_len < 0) {
	DPRINT(Debug,12,(&Debug,
			 "add_dir_vector: RESETTING vector_len=%d to 0\n",
			 dir->vector_len));

	if (dir->vector) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"add_dir_vector",
		  "Bad vector len (non-null vector)",0);
	}
	dir->vector_len = 0;
    }


    DPRINT(Debug,12,(&Debug,    
		     "add_dir_vector: [%d].sys_name=%s, [%d].disp_name=%S, [%d].flags=%X%s\n",
		     dir->vector_len,sys_name,
		     dir->vector_len,disp_name,
		     dir->vector_len,flags,
		     flags & BROWSER_NEEDSTAT ? " (need stat)" : ""));

    dir->vector = safe_array_realloc (dir->vector,
				      (dir->vector_len+1),
				      sizeof (dir->vector[0]));
    dir->vector[dir->vector_len].sys_name  = sys_name;
    dir->vector[dir->vector_len].disp_name = disp_name;
    dir->vector[dir->vector_len].flags     = flags;
    dir->vector[dir->vector_len].mtime     = 0;
    dir->vector[dir->vector_len].a.dummy   = NULL;

    ret = & (dir->vector[dir->vector_len]);

    dir->vector_len++;

    return ret;
}


static void free_browser P_((struct folder_browser **dir));
static void free_browser(dir)
     struct folder_browser **dir;
{
    DPRINT(Debug,15,(&Debug,"free_browser: Clearing dir vector\n"));

    if (FOLDER_BROWSER_magic != (*dir)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_browser",
	      "Bad magic number (folder_browser)",0);

    clear_dir_vector(*dir);

    if ((*dir)->selection)  {
	DPRINT(Debug,15,(&Debug,"free_browser: Clearing selection\n"));
	clear_dir_selection(*dir);
    }

    if ((*dir)->default_server) {
	DPRINT(Debug,15,(&Debug,"free_browser: Clearing default server\n"));
	remote_server_dec_dir(*dir);
    }

    if (BROWSER_TYPE_magic != (*dir)->type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_browser",
	      "Bad magic number (browser_type)",0);

    DPRINT(Debug,15,(&Debug,"free_browser: Freeing type specific data\n"));

    (*dir)->type->browser_free_it(*dir);
    
    DPRINT(Debug,15,(&Debug,"free_browser: Freeing rest\n"));

    if ((*dir)->dirname)
	free_string(&((*dir)->dirname));

    if ((*dir)->filter)
	free_string(&((*dir)->filter));

    if ((*dir)->sys_dir) {
	free((*dir)->sys_dir);
	(*dir)->sys_dir = NULL;
    }


    free(*dir);
    *dir = NULL;
}


void free_dir(dir)
     struct folder_browser **dir;
{
    if (FOLDER_BROWSER_magic != (*dir)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"free_dir",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,10,(&Debug,
		"free_dir: *dir=%p (%s); type=%p\n", 
		*dir,(*dir)->sys_dir ? (*dir)->sys_dir : "<NULL>",
		(*dir)->type));
    free_browser(dir);
}

static struct folder_browser * browser_malloc2 P_((
	 enum selection_type     sel_type,
	 struct browser_type    *type
	 ));
static struct folder_browser * browser_malloc2(sel_type,type)
     enum selection_type     sel_type;
     struct browser_type    *type;
{
    struct folder_browser * dir = safe_zero_alloc(sizeof (*dir));    

    dir->sys_dir       = NULL;
    dir->dirname       = NULL;
    dir->filter        = NULL;

    dir->sel_type      = sel_type;
    dir->vector        = NULL;
    dir->vector_len    = 0;
    dir->selection     = NULL;
    dir->magic         = FOLDER_BROWSER_magic;

    if (BROWSER_TYPE_magic != type->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_malloc2",
	      "Bad magic number (browser_type)",0);

    dir->type          = type;
    dir->type->browser_zero_it(dir);

    return dir;
}


static struct folder_browser * browser_malloc 
P_((enum selection_type  sel_type));

static struct folder_browser * browser_malloc(sel_type)
     enum selection_type  sel_type;
{
    return browser_malloc2(sel_type,&dummy_browser);
}

#ifdef REMOTE_MBX

uint16 browser_URL_default_port(schema)
     const struct browser_url_method  *schema;
{
    uint16 port;

    if (BROWSER_URL_method_magic != schema->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_URL_default_port",
	      "Bad schema magic",0);

    port = schema->url_default_port_it(schema);

    return port;
}

/* May not be 
      uin16 port
   because this is promoted to int
*/

struct folder_browser * browser_connect_URL(schema,user,password,host,port)
     const struct browser_url_method  *schema;
     const struct string              *user   /* May be NULL */;
     const struct string              *password /* May be NULL */;
     const char                       *host   /* May be NULL */;
     int                      port   /* May be NULL */;
{
    struct folder_browser * dir;
    struct connection_cache *con;

    if (BROWSER_URL_method_magic != schema->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_connect_URL",
	      "Bad schema magic",0);
    
    con = schema->url_make_it(schema,user,password,host,port);

    if (!con) {
	DPRINT(Debug,10,(&Debug,
			 "browser_connect_URL=NULL: url_make_it failed \n"));

	return NULL;
    }

    dir = browser_malloc2(selection_url,schema->type);

    /*  browser_from_connectionX sets con = NULL */
    browser_from_connectionX(&con,dir);

    DPRINT(Debug,10,(&Debug,
		     "browser_connect_URL=%p\n",
		     dir));

    return dir;
}

/* Returns 1 on succeed */
/* This is always absolute path -- from root as defined for URL */
int select_item_from_URL(schema,browser,elems_count,elems)
     const struct browser_url_method  *schema;
     struct folder_browser *browser;
     int elems_count;
     const struct string **elems;
{
    int ret;

    if (BROWSER_URL_method_magic != schema->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,
	      "select_item_from_URL_directory",
	      "Bad schema magic",0);

    /* Force type to correct */
    change_folder_browser_type(browser,schema->type);

    ret = schema->url_select_item_from_it_URL(schema,browser,elems_count,elems);

    return ret;
}
#endif

struct folder_browser *new_browser(sel_type)
     enum selection_type  sel_type;
{
    struct folder_browser * dir;

    DPRINT(Debug,10,(&Debug,
		"new_browser: sel_type = %d\n",sel_type));


    dir = browser_malloc(sel_type);

    DPRINT(Debug,10,(&Debug,
		     "new_browser=%p type = %p\n",dir,dir->type));
    return dir;
}

enum selection_type get_browser_sel_type(dir) 
     struct folder_browser *dir;
{
    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"get_browser_sel",
	      "Bad magic number (folder_browser)",0);
    
    return dir->sel_type;
}

int change_folder_browser_type(dir,new_type)
     struct folder_browser *dir; 
     struct browser_type *new_type;
{
    struct hashmark_item * selection_hashmark = NULL;
    
    if (BROWSER_TYPE_magic != new_type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"change_folder_browser_type",
	      "Bad new browser type (magic, browser_type)",0);

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"change_folder_browser_type",
	      "Bad magic number (folder_browser)",0);
    
    if (new_type == dir->type) 
	return 0;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"change_folder_browser_type",
	      "Bad magic (browser_type)",0);

    DPRINT(Debug,11,(&Debug,
		     "change_folder_browser_type: dir=%p Changing type from %p to %p\n",
		     dir,dir->type,new_type));


    DPRINT(Debug,15,(&Debug,"change_folder_browser_type: Clearing dir vector\n"));
    /* Must clear before type change !! */
    clear_dir_vector(dir);
    
    if (dir->selection)  {
	if (dir->selection->disp_name) {
	    DPRINT(Debug,15,(&Debug,
			     "change_folder_browser_type: selection %S\n",
			     dir->selection->disp_name));
	}

	/* Increments refcount */
	selection_hashmark
	    = dir->type->browser_gethm_name_vector_it(dir,dir->selection);
	if (selection_hashmark) {
	    DPRINT(Debug,15,(&Debug,
			     "change_folder_browser_type: Selection includes hashmark\n"));
	}
	
	DPRINT(Debug,15,(&Debug,
			 "change_folder_browser_type: clearing type specific selection data\n"));
	dir->type->browser_free_name_vector_it(dir,dir->selection);
    }

    DPRINT(Debug,15,(&Debug,"change_folder_browser_type: Freeing type specific data\n"));

    dir->type->browser_free_it(dir);
   
    /* Assurance for clean start */
    bzero((void *)&(dir->a), sizeof (dir->a));

    DPRINT(Debug,15,(&Debug,"change_folder_browser_type: Initializing type specific data\n"));

    dir->type = new_type;
    dir->type->browser_zero_it(dir);

    if (selection_hashmark) {
	if (dir->selection)  {
	    /* Returns 1 if hashmark saved
	       Increments refcount, if hashmark saved */
	    if (dir->type->browser_sethm_name_vector_it(dir,dir->selection,
							selection_hashmark)) {
		DPRINT(Debug,15,(&Debug,
				 "change_folder_browser_type: Saved hashmark to selection\n"));
	    } else {
		DPRINT(Debug,15,(&Debug,
				 "change_folder_browser_type: Can't save hashmark to selection\n"));
	    }							 
	}
	
	free_hashmark_item(& (selection_hashmark));
    }
	
    return 1;
}

#ifdef REMOTE_MBX
static void expander_set_dummy_user_host P_((struct folder_browser *dir,
				    char *user, char *host));
static void expander_set_dummy_user_host(dir,user,host)
     struct folder_browser *dir;
     char *user; 
     char *host;
{
    
    change_folder_browser_type(dir,&dummy_browser);
    
    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"expander_set_dummy_user_host",
	      "Bad magic type",0);
    
	    
    if (dir->a.dummy_browser->remote)
	free(dir->a.dummy_browser->remote);

    dir->a.dummy_browser->remote = 
	elm_message(FRM("%s@%s"),user,host);

    switch(dir->sel_type) {
    case selection_file:
	DPRINT(Debug,15,(&Debug,
			 "expander_set_dummy_user_host: %s is remote unsupported directory\n",
			 dir->a.dummy_browser->remote));
	break;
    case selection_folder:
	DPRINT(Debug,15,(&Debug,
			 "expander_set_dummy_user_host: %s is either POP or IMAP\n",
			 dir->a.dummy_browser->remote));
	break;
    case selection_url:
    case NUM_selection_type:
	/* Not used */
	break;
    
    }

}
#endif

static void expander_set_dummy_remote P_((struct folder_browser *dir,
				 char *remote));
static void expander_set_dummy_remote(dir,remote)
     struct folder_browser *dir;
     char *remote;
{
    change_folder_browser_type(dir,&dummy_browser);
    
    if (DUMMY_BROWSER_magic != dir->a.dummy_browser->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"expander_set_dummy_remote",
	      "Bad magic type",0);
    
    dir->a.dummy_browser->remote =
	strmcpy(dir->a.dummy_browser->remote,remote);

    switch(dir->sel_type) {
    case selection_file:
	DPRINT(Debug,15,(&Debug,
			 "expander_set_dummy_remote: %s is remote unsupported directory\n",
			 dir->a.dummy_browser->remote));
	break;
    case selection_folder:
	DPRINT(Debug,15,(&Debug,
			 "expander_set_dummy_remote: %s is either POP or IMAP\n",
			 dir->a.dummy_browser->remote));
	break;
    case selection_url:
    case NUM_selection_type:
	/* Not used */
	break;
    }

}    
    
/* ---------------------------------------------------------------------- */


/* NOTE:
   'spec' given on tail_expander is assumed to be mailbox.
   
   Therefore 'user@hostname' as IMAP mailbox is interpreted
   as 'user@hostname:INBOX'

*/

static int tail_expander P_((struct folder_browser *dir,
			     struct string **buffer,
			     int pos, const char *spec,
			     struct string **relative_to_type,
			     unsigned int defaultfile_keyword,
			     unsigned int defaultfile_cmask));
static int tail_expander(dir,buffer,pos,spec,relative_to_type,
			 defaultfile_keyword,defaultfile_cmask)
     struct folder_browser *dir;
     struct string **buffer;
     int pos; 
     const char *spec;
     struct string **relative_to_type;
     unsigned int defaultfile_keyword;
     unsigned int defaultfile_cmask;
{    
    int ret = 0;
    
    int code = 0;
    struct string * tempbuf = NULL;
#ifdef REMOTE_MBX
    struct service_entry *se        = NULL;
    struct service_entry *result_se = NULL;
#endif
    struct cancel_data *main_cancel = NULL /* Used if dns lookup was
					      cancelable */;
    
    /* Explicite disallow $MAIL to be interpreted as remote mailbox */
    if (0 != (defaultfile_cmask & FOLDER_is_local_file)) {

	DPRINT(Debug,15,(&Debug,
			 "tail_expander: FOLDER_is_local_file is set for value\n"));

	code = 0;   /* 0 == not a remote address */

    } else {

#ifdef REMOTE_MBX
	struct remote_account X;
	char *rest = NULL;
	
	
	DPRINT(Debug,15,(&Debug,
			 "tail_expander: Trying %s as remote address...\n",
			 spec));
    
	/* 'spec' is some folder ... */
    
	zero_remote_account(&X);

	/* -1 == name not found or bad syntax
	   0 == not a remote address
	   1 == name found 
	   2 == no lookup (se == NULL)
	*/


	code = split_remote_name(spec,&X,&se,&rest,
				 0 /* Both POP and IMAP are OK */,
				 &main_cancel
				 );
	
	if (code < 0) {     /* ERROR */
	    if (main_cancel && is_canceled(main_cancel)) {
		DPRINT(Debug,12,(&Debug,
				 "tail_expander: DNS lookup canceled\n"));
	    } else {
		DPRINT(Debug,12,(&Debug,
				 "tail_expander: name not found or bad syntax\n"));
	    }
	    goto fail;	    	   	    
	} else if (code > 0) {
	    
	    switch(dir->sel_type) {
	    case selection_file:
		/* Use dummy browser to store information ... (not supported) */
		
		DPRINT(Debug,15,(&Debug,
				 "tail_expander: user=%s host=%s -- remote unsupported file\n",
				 X.username,X.host));
		
		
		/* change type to dummy */
		expander_set_dummy_user_host(dir,X.username,X.host);
		
		clear_remote_account(&X);
		
		if (rest) {
		    switch(rest[0]) {
		    case ':':
			tempbuf = new_string2(system_charset,
					      s2us(rest+1));
			break;
		    case '/':
			tempbuf = new_string2(system_charset,
					      s2us(rest));
			break;
		    default:
			panic("BROWSER PANIC",__FILE__,__LINE__,
			      "tail_expander",
			      "Bad return from split_remote_name",0);
		    }
		} else		
		    /* Empty directory assumed ... */
		    tempbuf = new_string2(system_charset,s2us(""));
		
		break;
	    case selection_folder:
		
		if (!rest && pos >= string_len(*buffer)) {
		    /* Can be either IMAP or POP mailbox, use
		       dummy browser to store information 
		    */
		    
		    DPRINT(Debug,15,(&Debug,
				     "tail_expander: user=%s host=%s is either POP or IMAP\n",
				     X.username,X.host));
		    		    
		    /* change type to dummy */
		    expander_set_dummy_user_host(dir,X.username,X.host);
		    
		    clear_remote_account(&X);
		    
		    *relative_to_type = NULL;
		    
		} else {
		    struct connection_cache *XX;
		    
		    /* Other case it must be IMAP connection ... */
		    
		    DPRINT(Debug,15,(&Debug,
				     "tail_expander: Trying user=%s host=%s as IMAP\n",
				     X.username,X.host));
		    
		    if (dir->type == &imap_browser && 
			(XX=give_imap_connection(dir)) &&
			XX->C.username &&
			XX->C.host &&
			0 == strcmp(XX->C.username,X.username) &&
			0 == istrcmp(XX->C.host,X.host)) {
			DPRINT(Debug,15,(&Debug,
					 "tail_expander: already correct connection\n"));
		    } else {
			static struct connection_cache *CX;
			
			int got_port = 0;
			const struct service_type * got_type = NULL;
			
			struct enum_service_type ports_imaponly_2 = NULL_enum_service_type;

			init_enum_service_type(&ports_imaponly_2,init_tls_default,
					       STFLAG_is_imap,STFLAG_mbox);
				
			/* Only IMAP connections are cached */
			CX = locate_from_cache(X.username,X.host,
					       &IMAP_connection,
					       0 /* Default port */, se);
			if (CX) {
			    change_folder_browser_type(dir,&imap_browser);
			    
			    /*  browser_from_connectionX sets CX = NULL */
			    browser_from_connectionX(&CX,dir);
			    
			} else if (connect_remote_account_2(&X,se,main_cancel,
					    &result_se,
					    &got_type,
					    &got_port,
					    &ports_imaponly_2)) {
			    
			    enum itls_status have_initial_tls = itls_none;

			    DPRINT(Debug,15,(&Debug,
					     "tail_expander: %s: connected, got port %d",
					     X.host,got_port));
			    if (got_type) {
			      DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
					       got_type,
					       service_type_name(got_type),
					       service_type_defport(got_type)));
			    }
			    DPRINT(Debug,15,(&Debug,"\n"));

			    
			    switch((have_initial_tls = 
				    remote_account_init_tls(&X))) {
				
			    case itls_unsupported:
			    case itls_failed:
				clear_remote_account(&X);
				goto fail;
				
			    case itls_none:
				break;
				
			    case itls_done:
				DPRINT(Debug,8,(&Debug,
						"tail_expander: Initial TLS done (no STARTTLS)\n"));
			    }
			    
			    change_folder_browser_type(dir,&imap_browser);
			    
			    if (!join_connection(give_imap_connection(dir),&X,
						 CON_greeting)) {
				clear_remote_account(&X);
				goto fail;
			    }		
			} else {
			    clear_remote_account(&X);
			    goto fail;
			}
		    }
		    
		    /* Now we should have IMAP connection */
		    
		    if (rest) {
			switch(rest[0]) {
			case ':':
			    tempbuf = new_string2(imap_charset,
						  s2us(rest+1));
			    break;
			case '/':
			    tempbuf = new_string2(imap_charset,
						  s2us(rest));
			    break;
			default:
			    panic("BROWSER PANIC",__FILE__,__LINE__,
				  "tail_expander",
				  "Bad return from split_remote_name",0);
			}
		    } else		
			/* Sub directory of INBOX assumed */
			tempbuf = new_string2(imap_charset,s2us("INBOX"));
		}
		break;
	    case selection_url:
	    case NUM_selection_type:
		/* Not used */
		break;
	    }	
	}  
	clear_remote_account(&X);

#endif /* REMOTE_MBX */
    }

    if (0 == code) {

	change_folder_browser_type(dir,
#if DIROPS
		    &local_browser
#else
		    &dummy_browser
#endif
		    );


	tempbuf = new_string2(local_fs_charset,cs2us(spec));
    }

    if (pos < string_len(*buffer)) {
	char sep;
	int X = pos;
	struct string *XX;

	DPRINT(Debug,15,(&Debug,
		    "tail_expander: Changing directory for %s\n",
		    spec));

	if (BROWSER_TYPE_magic != dir->type->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"tail_expander",
		  "Bad magic (browser_type)",0);

	/* Change first directory specified on 'spec' */
	if (!dir->type->browser_change_it(dir,tempbuf,&tempbuf))
	    goto fail;
	
	sep = dir->type->browser_separator_it(dir);

	if ('\0' == sep) {
	    struct string * msgname = dir->type->browser_name_it(dir);
	    
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeFlatError,
			      "%S have no subdirectories"),
		      msgname);
	
	    free_string(&msgname);
	    goto fail;
	}

	XX = clip_from_string(*buffer,&X,string_len(*buffer));
	*relative_to_type = 
	    format_string(FRM("%S%c%S"),tempbuf,sep,XX);
	free_string(&XX);
    } else {         /* Directory given on 'spec' is current selection */
	*relative_to_type = tempbuf;
	tempbuf = NULL;
    }
    ret = 1;

 fail:
#ifdef REMOTE_MBX
    /* 'se' can be NULL */
    free_service_entry(&se);
    free_service_entry(&result_se);
#endif
    if (main_cancel)
	free_cancel(& (main_cancel));
              
    if (tempbuf)
	free_string(&tempbuf);


    if (*relative_to_type) {
	DPRINT(Debug,12,(&Debug,
			 "tail_expander: *relative_to_type=%S\n",
			 *relative_to_type));
    } else {
       	DPRINT(Debug,12,(&Debug,
			 "tail_expander: *relative_to_type=NULL\n"));
    }
    DPRINT(Debug,12,(&Debug,
		     "tail_expander=%d\n",ret));

    return ret;
}




      
static const char * expand_tag P_((struct folder_browser *dir,
				   struct string *buffer,
				   int haveend,
				   int *tagnum,
				   unsigned int *expand_keyword,
				   unsigned int *expand_cmask,
				   int *allow_mask));
static const char * expand_tag(dir,buffer,haveend,tagnum,
			       expand_keyword,expand_cmask,
			       allow_mask)
     struct folder_browser *dir;
     struct string *buffer;
     int haveend;
     int *tagnum;
     unsigned int *expand_keyword;
     unsigned int *expand_cmask;
     int *allow_mask;
{
    const char * tag_val = NULL;

    int i;
    int candinate = -1;
    

    DPRINT(Debug,15,(&Debug,"expand_tag: buffer=%S, haveend=%d\n",buffer,haveend));
    
    for (i = 0; i < expand_tag_count; i++) {
	int len = string_matches_ascii(buffer,
				       cs2us(expand_tag_list[i].tag),
				       SMA_return_len,SMA_op_match_prefix);

	if (len > 0) {
	    
	    DPRINT(Debug,15,(&Debug,"expand_tag: Matches #%d %s len=%d",
			     i,expand_tag_list[i].tag,len));

	    if (len == haveend+1) {
		DPRINT(Debug,15,(&Debug," == havend+1 = %d  OK\n",haveend+1));

		candinate = i;

		if (0 != (expand_tag_list[i].allow_mask & (1 << dir->sel_type)))  { 	       		
		    tag_val =
			give_dt_estr_as_str(expand_tag_list[i].ptr,expand_tag_list[i].variable,
					    expand_keyword,expand_cmask);

		    if (tag_val) {
			DPRINT(Debug,15,(&Debug,"expand_tag: Matches #%d variable %s gives %s\n",
					 i,expand_tag_list[i].variable,tag_val));

			if (tagnum)
			    *tagnum = i;
			if (allow_mask)
			    *allow_mask = expand_tag_list[i].allow_mask;
			goto found;
		    }		    
		}		
	    } else {
		DPRINT(Debug,15,(&Debug," != havend+1 = %d  IGNORED\n",haveend+1));
	    }	    
	}	
    }


    if (candinate >= 0) {
	DPRINT(Debug,15,(&Debug,
			 "expand_tag: Matches candinate #%d (expand_tag_count %d)\n",
			 candinate, expand_tag_count));
	
	if (candinate >= expand_tag_count)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"expand_tag",
		  "Bad return from split_remote_name",0);

	tag_val =
	    give_dt_estr_as_str(expand_tag_list[candinate].ptr,expand_tag_list[candinate].variable,
				expand_keyword,expand_cmask);
	
	if (tagnum)
	    *tagnum = candinate;
	if (allow_mask)
	    *allow_mask = expand_tag_list[candinate].allow_mask;

    }


 found:
    DPRINT(Debug,15,(&Debug,
		     "expand_tag="));
    if (tag_val) {
	DPRINT(Debug,15,(&Debug,"%Q",tag_val));	
	if (tagnum) {
	    DPRINT(Debug,15,(&Debug,", *tagmum=%d",*tagnum));
	}
	if (expand_keyword) {
	    DPRINT(Debug,15,(&Debug,", *expand_keyword=%d",*expand_keyword));
	}
	if (expand_cmask) {
	    DPRINT(Debug,15,(&Debug,", *expand_cmask=%d",*expand_cmask));
	}
	if (allow_mask) {
	    DPRINT(Debug,15,(&Debug,", *allow_mask=%d",*allow_mask));
	}	
	DPRINT(Debug,15,(&Debug,"\n"));
	
    } else {
	DPRINT(Debug,15,(&Debug,"NULL\n"));
    }
    
    return tag_val;
}


static int expander(dir,buffer,relative_to_type)
     struct folder_browser *dir; 
     struct string **buffer;
     struct string **relative_to_type;
{
    int ret = 0;
    int len;

    len = string_len(*buffer);
    *relative_to_type = NULL;

    if (len > 0) { 
	/* "!!" was as last folder used (was "." as last folder used)
	   "@alias" as default folder for user,
	   "="   as save by name and
	   "=?"  as conditionally save by name
	   must be implemented on src/file.c or src/savecopy.c
	   others we can implement on here ...

	   These include:
	   "."     as default directory
	   "~"     as home directory of current user
	   "~user" as home directory of given user
	   "&"     as Current remote (IMAP) server 
	   "="     as folder directory
	   "+"     as folder directory
	   "%"     as folder directory
	   "!"     as incoming mailbox
	   ">"     as received folder
	   "<"     as sent folder
	   "[dsn]" as dsn folder
	   "$"     as shell variable expansion (as local file)
	   "{rc}"  as .elm directory           (as local file)
	   "{lib}" as LIBHOME directory        (as local file)
	   "{etc}" as ETCHOME directory        (as local file)
	   "{doc}" as attachment-dir directory (as local file)
	   "#hashmark" as given on configuration
	*/

	uint16 code = give_unicode_from_string(*buffer,0);

	DPRINT(Debug,15,(&Debug,"expander: First char %04X\n",code));

	switch (code) {
	    int x, atpos;
	    int tailpos UNUSED_VAROK;
	    int idx;

	case 0x0023: /* # */

	    DPRINT(Debug,15,(&Debug,
			     "expander: #hashmark ?\n"));

	    idx = string_have_ascii_sep1(*buffer,
					 hashmark_RESERVED_SEPARATOR,
					 /* scan index 1 .. (len-1) */
					 1,len);

	    if (idx >= 0) {
		uint16 code1 = give_unicode_from_string(*buffer,idx);

		if (0x003A /* ':' */ != code1 &&
		    0x002F /* '/' */ == code1) {
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeBadHashmark,
				      "Bad hashmark: %S"),
			      *buffer);
		    break;
		}
	    }


	    /* change_folder_browser_type() does clear_dir_vector() */
	    if (change_folder_browser_type(dir,&hashmark_browser)) {
		DPRINT(Debug,15,(&Debug,
				 "    ... type changed\n"));

	    }

	    /* Clear directory listing ..... */
	    clear_dir_vector(dir);

	    if (len > 1) {

		int X1 = 1;

		struct string *tmpbuf =  clip_from_string(*buffer,&X1,len);


		*relative_to_type = tmpbuf;

	    }

	    ret = 1;

	    break;
	    
	case 0x002E: /* '.' */
	    if (1 == len) {   /* Default directory */
		change_folder_browser_type(dir,
#if DIROPS
			    &local_browser
#else
			    &dummy_browser
#endif
			    );

		*relative_to_type = dup_string(*buffer);
		ret = 1;
		break;
	    }
	    goto other;	 

	case 0x007E: /*  '~' */
	    if (1 == len) {   /* User's home directory */
		change_folder_browser_type(dir,
#if DIROPS
			    &local_browser
#else
			    &dummy_browser
#endif
			    );

		*relative_to_type = new_string2(local_fs_charset,
						s2us(home));
		ret = 1;
		break;
	    }

	    goto LOCAL_HANDLING;

	    
	case 0x0026: /* '&' */
	    /* Current remote (IMAP) server  */

	    DPRINT(Debug,15,(&Debug,
			     "expander: remote server\n"));
	    
	    /* change_folder_browser_type() does clear_dir_vector() */
	    if (change_folder_browser_type(dir,&remote_browser)) {
		DPRINT(Debug,15,(&Debug,
				 "    ... type changed\n"));

	    }


	    if (! dir->default_server) {

		remote_browser_failure(dir);
		clear_dir_vector(dir);

		break;
	    }
    
	    /* change_remote_browser_server() does clear_dir_vector()
	       in remote_browser_server is changed
	    */
	    change_remote_browser_server(dir);
	   
	    if (len > 1) {

		int X1 = 1;

		struct string *tmpbuf =  clip_from_string(*buffer,&X1,len);


		*relative_to_type = tmpbuf;

	    }

	    ret = 1;

	    break;

	case 0x003D: /* '=' */
	case 0x002B: /* '+' */
	case 0x0025: /* '%' */
	    
	    /* Local folder directory */
	    if (1 == len) {   

		/* give_dt_estr_as_str adds / to end */
		const char * folders_val = give_dt_estr_as_str(&folders_e,
							       "maildir",NULL,NULL);

		if (!folders_val)
		    ret = 0;
		else {		    
		    change_folder_browser_type(dir,
#ifdef DIROPS
				&local_browser
#else
				&dummy_browser
#endif
				);
		    
		    *relative_to_type = new_string2(local_fs_charset,
						    cs2us(folders_val));
		    ret = 1;
		}
		break;
	    }

	    goto LOCAL_HANDLING;

	    /* NOTE:
	       
	       tail_expander assumes that 'spec' argument
	       (defaultfile, recvd_mail, sent_mail)
	       is mailbox. Therefore 'username@hostname'
	       is interpreted as 'username@hostname:INBOX'
	       for IMAP mailbox.
	       
	    */
	    
	case 0x0021: {
	    /* '!'      incoming mailbox */

	    unsigned int defaultfile_keyword = 0;
	    unsigned int defaultfile_cmask   = 0;

	    const char * default_val = 
		give_dt_estr_as_str(&defaultfile_e,
				    "incoming-mailbox",
				    &defaultfile_keyword,
				    &defaultfile_cmask);

	    if (!default_val)
		ret = 0;
	    else {
		ret = tail_expander(dir,buffer,1,default_val,
				    relative_to_type,
				    defaultfile_keyword,
				    defaultfile_cmask);
		if (ret && dir->sel_type == selection_file) {
		    if (*relative_to_type && 0 < string_len(*relative_to_type)) 
			lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeFile1,
					  "Name %S (expands to %s, %S) is unsafe use as file"),
				  *buffer, default_val, *relative_to_type);
		    else
			lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeFile2,
					  "Name %S (expands to %s) is unsafe use as file"),
				  *buffer, default_val);
		    ret = 0;
		}
	    }
	}
	    break;
	case 0x003E: { 
	    /* '>'      received folder  */ 

	    /* Must have same flags on  incoming mailbox */

	    unsigned int defaultfile_keyword = 0;
	    unsigned int defaultfile_cmask   = 0;

	    const char * recvd_val = 
		give_dt_estr_as_str(&recvd_mail_e,
				    "receivedmail",
				    &defaultfile_keyword,
				    &defaultfile_cmask);   

	    if (!recvd_val)
		ret = 0 ;
	    else {
		ret = tail_expander(dir,buffer,1,recvd_val,
				    relative_to_type,
				    defaultfile_keyword,
				    defaultfile_cmask);

		if (ret && dir->sel_type == selection_file) {
		    if (*relative_to_type && 0 < string_len(*relative_to_type)) 
			lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeFile1,
					  "Name %S (expands to %s, %S) is unsafe use as file"),
				  *buffer, recvd_val, *relative_to_type);
		    else
			lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeFile2,
					  "Name %S (expands to %s) is unsafe use as file"),
				  *buffer, recvd_val);
		    ret = 0;
		}
	    }
	}
	    break;
	case 0x003C: {
	    /*  '<'     sent folder     */

	    /* Must have same flags on  incoming mailbox */

	    unsigned int defaultfile_keyword = 0;
	    unsigned int defaultfile_cmask   = 0;


	    const char * sent_val = 
		give_dt_estr_as_str(&sent_mail_e,
				    "sentmail",
				    & defaultfile_keyword,
				    & defaultfile_cmask);
	    
	    if (!sent_val)
		ret = 0 ;
	    else {
		
		ret = tail_expander(dir,buffer,1,sent_val,
				    relative_to_type,
				    defaultfile_keyword,
				    defaultfile_cmask);

		if (ret && dir->sel_type == selection_file) {
		    if (*relative_to_type && 0 < string_len(*relative_to_type)) 
			lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeFile1,
					  "Name %S (expands to %s, %S) is unsafe use as file"),
				  *buffer, sent_val, *relative_to_type);
		    else
			lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeFile2,
					  "Name %S (expands to %s) is unsafe use as file"),
				  *buffer, sent_val);
		    ret = 0;
		}
	    }
	}
	    break;
	    
	case 0x005B: {
	    int haveend = -1;
	    int tagnum = -1;
	    unsigned int expand_keyword = 0;
	    unsigned int expand_cmask   = 0;
	    int allow_mask = 0;
	    const char * tag_val;

	    for (x = 0; x < len; x++) {
		uint16 code1 = give_unicode_from_string(*buffer,x);

		if (0x005D /* ']' */  == code1) {
		    haveend = x;
		    break;
		}
	    }

	    if (haveend < 2) {
		lib_error(CATGETS(elm_msg_cat, MeSet, MeReservedForTags,
				  "Name %S have '[' reserved for tags"),
			  *buffer);
		ret = 0;
		break;
	    }

	    tag_val = expand_tag(dir,*buffer,
				 haveend,&tagnum,
				 &expand_keyword,
				 &expand_cmask,
				 &allow_mask);

	    if (!tag_val)
		ret = 0;
	    else {
		ret = tail_expander(dir,buffer,
				    haveend+1 /* pos for tail */,
				    tag_val,
				    relative_to_type,
				    expand_keyword,
				    expand_cmask);

		if (ret) {
		    int type_mask = (1 << dir->sel_type);
		    int is_safe = (allow_mask &  (1 << dir->sel_type));
		    
		    DPRINT(Debug,15,(&Debug,
				     "expander: %S (expands to %s) sel_type=%d allow_mask=%0x type_mask=%0x  -- %s\n",
				     *buffer, tag_val, dir->sel_type,allow_mask,type_mask,
				     is_safe ? "is safe" : "unsafe use"));
		    
		    switch (dir->sel_type) {
		    case selection_file:

			if (! is_safe) {
			    if (*relative_to_type && 0 < string_len(*relative_to_type)) 
				lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeFile1,
						  "Name %S (expands to %s, %S) is unsafe use as file"),
					  *buffer, tag_val, *relative_to_type);
			    else
				lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeFile2,
						  "Name %S (expands to %s) is unsafe use as file"),
					  *buffer, tag_val);
			    ret = 0;
			}
			break;
		    case selection_folder:
		    case selection_url:
		    case NUM_selection_type:
						
			if (! is_safe) {
			    if (*relative_to_type && 0 < string_len(*relative_to_type)) 
				lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeUse1,
						  "Name %S (expands to %s, %S) is unsafe use"),
					  *buffer, tag_val, *relative_to_type);
			    else
				lib_error(CATGETS(elm_msg_cat, MeSet, MeUnsafeUse2,
						  "Name %S (expands to %s) is unsafe use"),
					  *buffer, tag_val);
			    ret = 0;
			}
			break;			
		    }
		}		
	    }	    
	}
	    break;
	    
	default:
	other:

	    /* Look if it starts with user@hostname */
	    atpos = -1;
	    tailpos = len;

	    for (x = 0; x < len; x++) {
		uint16 code1 = give_unicode_from_string(*buffer,x);

		if (0x0040 /* '@' */   == code1)
		    atpos = x;
		else if (0x003A /* ':' */ == code1) {
		    tailpos = x+1;
		    break;
		} else if (0x002F /* '/' */ == code1) {
		    tailpos = x;
		    break;
		}
	    }

	    if (-1 != atpos) {
		if (0 == atpos || atpos+1 == x) {
		    lib_error(CATGETS(elm_msg_cat, MeSet,MeBadRemoteMailbox1,
				      "Bad remote mailbox: %S"),
			      *buffer);
		    break;
		}
		
		if (x >= len) {
		    struct string * Lstr = convert_string(system_charset,*buffer,0);
		    char * str = us2s(stream_from_string(Lstr,0,NULL));

		    /* Can be either IMAP or POP mailbox, use
		       dummy browser to store information 
		    */

		    expander_set_dummy_remote(dir,str);

		    free(str);
		    free_string(&Lstr);

		    ret = 1;
		    
		} else {
#ifdef REMOTE_MBX
		    struct remote_account   X;
		    struct service_entry  * se = NULL;
		    struct service_entry  * result_se = NULL;
		    char *rest = NULL;   /* Ignored -- need NOT to be free'ed
					  * 'rest' is pointer to area of 'str'
					  */

		    int ret_code;

		    struct string * Lstr = 
			convert_string(system_charset,*buffer,0);
		    unsigned char * str = 
			stream_from_string(Lstr,0,NULL);
		    struct cancel_data *main_cancel = NULL /* Used if dns lookup was
							      cancelable 
							   */;

		    /* Other case it must be IMAP connection ... */
		    
		    zero_remote_account(&X);

		    DPRINT(Debug,15,(&Debug,
				     "expander: Trying %s as remote address...\n",
				     str));
		    
		    /* -1 == name not found or bad syntax
		       0 == not a remote address
		       1 == name found 
		       2 == no lookup (se == NULL)
		    */

		    ret_code = split_remote_name(us2s(str),
						 &X,&se,&rest,
						 STFLAG_is_imap
						 /* Wanna IMAP */,
						 &main_cancel);

		    if (ret_code < 0) {     /* ERROR */
			if (main_cancel && is_canceled(main_cancel)) {
			    DPRINT(Debug,12,(&Debug,
					     "expander: DNS lookup canceled\n"));
			    
			} else {
			    DPRINT(Debug,12,(&Debug,
					     "expander: name not found or bad syntax\n"));
			}
			
		    } else if (0 <= ret_code) {

			if (ret_code == 0)
			    panic("BROWSER PANIC",__FILE__,__LINE__,"expander",
				  "Bad return from split_remote_name",0);

			switch(dir->sel_type) {
			    struct connection_cache *XX;

			case selection_file:
			    DPRINT(Debug,15,(&Debug,
					     "expander: user=%s host=%s -- remote unsupported file\n",
					     X.username,X.host));
			    
			    break;
			case selection_folder:
			    DPRINT(Debug,15,(&Debug,
					     "expander: Trying user=%s host=%s as IMAP\n",
					     X.username,X.host));

			    if (dir->type == &imap_browser && 
				(XX=give_imap_connection(dir)) &&
				XX->C.username &&
				XX->C.host &&
				0 == strcmp(XX->C.username,X.username) &&
				0 == istrcmp(XX->C.host,X.host)) {
				int X1 = tailpos;
				
				DPRINT(Debug,15,(&Debug,
						 "expander: already correct connection\n"));
				*relative_to_type = 
				    clip_from_string(*buffer,&X1,len);
				ret = 1;
			    } else {
				static struct connection_cache *CX = NULL;
			    
				int                         got_port = 0;
				const struct service_type * got_type = NULL;
				
				struct enum_service_type ports_imaponly_2 = NULL_enum_service_type;

				init_enum_service_type(&ports_imaponly_2,init_tls_default,
						       STFLAG_is_imap,STFLAG_mbox);
				
				/* Only IMAP connections are cached */
				CX = locate_from_cache(X.username,X.host,
						       &IMAP_connection,
						       0 /* Default port */,
						       se);
				if (CX) {
				    int X1 = tailpos;
				    
				    change_folder_browser_type(dir,&imap_browser);

				    /*  browser_from_connectionX sets CX = NULL */ 
				    browser_from_connectionX(&CX,dir);
				    ret = 1;
				    
				    *relative_to_type = 
					clip_from_string(*buffer,&X1,len);
				} else if (connect_remote_account_2(&X,se,main_cancel,
								    &result_se,
								    &got_type,
								    &got_port,
								    &ports_imaponly_2)) {

				    int X1 = tailpos;
				    enum itls_status have_initial_tls = itls_none;

				    DPRINT(Debug,15,(&Debug,
						     "expander: %s: connected, got port %d",
						     X.host,got_port));
				    if (got_type) {
					DPRINT(Debug,15,(&Debug,", type=%p %s defport %d",
							 got_type,
							 service_type_name(got_type),
							 service_type_defport(got_type)));
				    }
				    DPRINT(Debug,15,(&Debug,"\n"));
				    
				    switch((have_initial_tls = 
					    remote_account_init_tls(&X))) {
					
				    case itls_unsupported:
				    case itls_failed:
					
					goto fail;
					
				    case itls_none:
					break;
					
				    case itls_done:
					DPRINT(Debug,8,(&Debug,
							"expander: Initial TLS done (no STARTTLS)\n"));
				    }
				    
				    change_folder_browser_type(dir,&imap_browser);
				    
				    ret = join_connection(give_imap_connection(dir),
							  &X,CON_greeting);
				    if (ret)
					*relative_to_type = 
					    clip_from_string(*buffer,&X1,len);
				}
			    }
			    break;
			case selection_url:
			case NUM_selection_type:
			    /* Not used */
			    break;
			}
		    }

		fail:
		    free_string(&Lstr);
		    free(str);				    
		    clear_remote_account(&X);
		    
		    /* 'se' can be NULL */
		    free_service_entry(&se);
		    free_service_entry(&result_se);

		    if (main_cancel)
			free_cancel(& (main_cancel));
		    
#endif /* REMOTE_MBX */

		    if (!ret)
			lib_error(CATGETS(elm_msg_cat, MeSet, 
					  MeRemoteDirNotAvail1,
					  "Remote directries are not available in %.*S"),
				  x,*buffer);
		    
		}
	    } else {
		char expand_space[LONG_STRING];
		unsigned char * str;
		struct string * Lstr;
		int failcount = 0;

		/* Local mailbox handling */
		
		DPRINT(Debug,15,(&Debug,
				 "expander: does not start with 'user@hostname' atpos=%d, scan ended = %d (len=%d)\n",atpos,x,len));

		/* FALLTHRU */
	    case 0x0024: /* '$'  shell variable expansion (as local file) */
	    case 0x007B: /* '{'  {rc}  as .elm directory  (as local file)
			  *      {lib} as LIBHOME directory (as local file) 
			  *      {etc} as ETCHOME directory (as local file) 
			  *      {doc} as attachment-dir    (as local file) 
			  */

	    LOCAL_HANDLING:  /* goto label ... */
	    
		failcount = 0;
	        Lstr = convert_string2(local_fs_charset,*buffer,&failcount);

		if (failcount) {
		    const char *x = get_charset_MIME_name(local_fs_charset);
		    lib_error(CATGETS(elm_msg_cat, MeSet, 
				      MeNameNotValidAsLocal,
				      "Name %S not valid as local, %d errors with charset %s"),
			      *buffer,failcount,
			      x ? x : raw_local_fs_charset);
		}

	        str = stream_from_string(Lstr,0,NULL);

	       	DPRINT(Debug,15,(&Debug,
				 "expander: Expanding %s (%S) as local\n",
				 str,Lstr));
		
		if (expand_meta(expand_space,us2s(str),
				sizeof expand_space) >= 0) {
		    change_folder_browser_type(dir,
#ifdef DIROPS
				&local_browser
#else
				&dummy_browser
#endif
				);

		    *relative_to_type = new_string2(local_fs_charset,
						    s2us(expand_space));
		    if (0 == failcount)
			ret = 1;
		}
		free_string(&Lstr);
		free(str);		
	    }
	    break;
	}
    }

    if (*relative_to_type) {
	DPRINT(Debug,12,(&Debug,
			 "expander: *relative_to_type=%S\n",
			 *relative_to_type));
    } else {
       	DPRINT(Debug,12,(&Debug,
			 "expander: *relative_to_type=NULL\n"));
    }
    DPRINT(Debug,12,(&Debug,"expander=%d\n",ret));

    return ret;
}
    

int change_dir_helper(dir,buffer)
     struct folder_browser *dir; 
     struct string **buffer; 
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"change_dir_helper",
	      "Bad magic number (folder_browser)",0);

    if (*buffer &&
	string_len(*buffer) > 0) {
	struct string *relative_to_type = NULL;

	if (expander(dir,buffer,&relative_to_type)) {

	    if (BROWSER_TYPE_magic != dir->type->magic)
		panic("MBX PANIC",__FILE__,__LINE__,"change_dir_helper",
		      "Bad magic number (browser_type) after expander",0);
	    
	    ret = dir->type->browser_change_it(dir,relative_to_type,buffer);
	}

	if (relative_to_type)
	    free_string(&relative_to_type);

    } else {
	change_folder_browser_type(dir,&dummy_browser);

	if (BROWSER_TYPE_magic != dir->type->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"browser_change_dummy",
		  "Bad magic number (browser_type) after change_folder_browser_type",0);

	ret = dir->type->browser_change_it(dir,*buffer,buffer);
    }

    return ret;
}

/* Returns 1 if succesfully, and may canonify argument
   may change type of dir
 */
int change_dir(dir,buffer)
     struct folder_browser *dir; 
     struct string **buffer;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"change_dir",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "change_dir: dir=%p (%s); type=%p", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (*buffer) {
	if (!verify_string(*buffer)) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"change_dir",
		  "Bad buffer (string)",0);
	}
	DPRINT(Debug,10,(&Debug,", *buffer=%S\n",*buffer));

    } else {
	DPRINT(Debug,10,(&Debug,", *buffer=NULL\n"));
    }
	
    if (dir->filter)
	free_string(&(dir->filter));

    ret = change_dir_helper(dir,buffer);
	
    DPRINT(Debug,10,(&Debug,
		     "change_dir=%d; dir=%p (%s); type=%p", 
		     ret,
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));
    if (*buffer) {
	DPRINT(Debug,10,(&Debug,", *buffer=%S\n",*buffer));
    } else {
       	DPRINT(Debug,10,(&Debug,", *buffer=NULL\n"));
    }

    return ret;
}

int change_dir_to_entry(dir,entry,buffer)
     struct folder_browser *dir;
     int entry;
     struct string **buffer;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"change_dir_to_entry",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "change_dir_to_entry: dir=%p (%s); type=%p, entry=%d\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,
		     entry));

    if (*buffer) {
	if (!verify_string(*buffer)) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"change_dir_to_entry",
		  "Bad buffer (string)",0);
	}

	DPRINT(Debug,10,(&Debug,
			 "change_dir_to_entry- *buffer=%p\n",*buffer));

    } else {
       	DPRINT(Debug,10,(&Debug,
			 "change_dir_to_entry- *buffer=NULL\n"));
    }

    if (dir->filter)
	free_string(&(dir->filter));
    
    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"change_dir_to_entry",
	      "Bad magic (browser_type)",0);

    /* browser_vector_len() will update cache */

    if (entry < 0 || entry >= browser_vector_len(dir)) {
	DPRINT(Debug,10,(&Debug,
			 "change_dir_to_entry: entry (%d) not in range (0-%d)\n",
			 entry,dir->vector_len-1));

	ret = 0;
    } else { 
	
	ret = dir->type->browser_change_v_it(dir,&(dir->vector[entry]),buffer);

    }

    if (*buffer) {
	DPRINT(Debug,10,(&Debug,
			 "change_dir_to_entry=%d; *buffer=%S\n",
			 ret,*buffer));
    } else {
	DPRINT(Debug,10,(&Debug,
			 "change_dir_to_entry=%d; *buffer=NULL\n",ret));
    }
    return ret;
}

int change_dir_to_selection(dir,buffer)
     struct folder_browser *dir;
     struct string **buffer;
{
    int res = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"change_dir_to_selection",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "change_dir_to_selection: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"change_dir_to_selection",
	      "No selection",0);

    if (dir->filter)
	free_string(&(dir->filter));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"change_dir_to_selection",
	      "Bad magic (browser_type)",0);

    res = dir->type->browser_change_v_it(dir,dir->selection,buffer);

    DPRINT(Debug,10,(&Debug,"change_dir_to_selection=%d\n",res));

    return res;
}

int change_dir_up(dir,buffer,pos)
     struct folder_browser *dir;
     struct string **buffer;
     int *pos;
{
   int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"change_dir_up",
	      "Bad magic number (folder_browser)",0);
     
    DPRINT(Debug,10,(&Debug,
		     "change_dir_up: dir=%p (%s) pos=%p; type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     pos,
		     dir->type
		     ));
    
    if (*buffer) {
	if (!verify_string(*buffer)) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"change_dir_up",
		  "Bad buffer (string)",0);
	}

	DPRINT(Debug,10,(&Debug,
			 "change_dir_up- *buffer=%S\n",*buffer));
    } else {
       	DPRINT(Debug,10,(&Debug,
			 "change_dir_up- *buffer=NULL\n"));
    }

    if (dir->filter)
	free_string(&(dir->filter));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"change_dir_up",
	      "Bad magic (browser_type)",0);
    
    if (dir->dirname) {  
	struct string * disp_tail = NULL;

	/* Not possible to go up from default directory / menu */


	ret = dir->type->browser_change_up_it(dir,buffer,&disp_tail);
	if (ret < 0) {
	    DPRINT(Debug,10,(&Debug,
			     "change_dir_up: Returning to default menu...\n"));

	    change_folder_browser_type(dir,&dummy_browser);

	    browser_gen_default_menu(dir);
	    if (*buffer)
		free_string(buffer);
	    ret = 1;
	} else if (ret > 0) {
	    /* Do actual change ... */
	    ret = change_dir_helper(dir,buffer);

	    if (pos) {
		*pos = -1;

		if (disp_tail) {
		    int i;
		
		    DPRINT(Debug,10,(&Debug,
				     "change_dir_up: disp_tail=%S\n",disp_tail));

		    /* browser_vector_len() will update cache */
		    browser_vector_len(dir);
		    
		    for (i = 0; i < dir->vector_len; i++)
			if (0 == (string_cmp(disp_tail,
					     dir->vector[i].disp_name,
					     -99 /* Unknown indicator */)))
			    break;
	    
		    if (i < dir->vector_len) {
			DPRINT(Debug,10,(&Debug,
					 "change_dir_up: *pos => %d\n",
					 i));
			
			*pos = i;
		    }
		    
		}


		if (-1 == *pos) {
		    DPRINT(Debug,10,(&Debug,
				     "change_dir_up: *pos => -1   (NOT FOUND)\n"));		    
		}

	    }

	}

	if (disp_tail)
	    free_string(&disp_tail);
    } 

    if (*buffer) {
	DPRINT(Debug,10,(&Debug,
			 "change_dir_up=%d; *buffer=%S\n",ret,*buffer));
    } else {
	DPRINT(Debug,10,(&Debug,
			 "change_dir_up=%d; *buffer=NULL\n",ret));
    }
    return ret;
}

/* Returns 1 if succesfully -- updates *pos */
int reload_dir(dir, pos)
     struct folder_browser *dir;
     int *pos;
{
   int ret = 0;

   struct string * position = NULL;
   
   if (FOLDER_BROWSER_magic != dir->magic)
       panic("MBX PANIC",__FILE__,__LINE__,"reload_dir",
	     "Bad magic number (folder_browser)",0);
   
   DPRINT(Debug,10,(&Debug,
		    "reload_dir: dir=%p (%s) pos=%p; type=%p\n", 
		    dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		    pos,
		    dir->type
		    ));

    if (pos && *pos >= 0 && *pos < dir->vector_len) {
	position = dup_string(dir->vector[*pos].disp_name);
	
	DPRINT(Debug,10,(&Debug,
			 "reload_dir: *pos=%d position=%S\n",
			 *pos,position));
    }

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"reload_dir",
	      "Bad magic (browser_type)",0);
     
    ret =  dir->type->browser_reload_it(dir);

    if (pos) {
	*pos = -1;

	if (position) {
	    int i;

	    /* browser_vector_len() will update cache */
	    browser_vector_len(dir);
	    
	    for (i = 0; i < dir->vector_len; i++)
		if (0 == (string_cmp(position,
				     dir->vector[i].disp_name,
				     -99 /* Unknown indicator */)))
		    break;

	    if (i < dir->vector_len) {
		DPRINT(Debug,10,(&Debug,
				 "reload_dir: *pos => %d\n",
				 i));
		
		*pos = i;
	    }	    
	}

	if (-1 == *pos) {
	    DPRINT(Debug,10,(&Debug,
			     "reload_dir: *pos => -1   (NOT FOUND)\n"));		    
	}
    }

    if (position)
	free_string(&position);


    DPRINT(Debug,10,(&Debug,
		     "reload_dir=%d\n",ret));
    return ret;    
}

struct string * give_title_dir(dir,entry_count)
     struct folder_browser *dir;
     int *entry_count;
{
    struct string * ret;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"give_title_dir",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "give_title_dir: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    /* browser_vector_len() will update cache */

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"give_title_dir",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_give_title_it(dir);
    *entry_count = browser_vector_len(dir);;

    DPRINT(Debug,10,(&Debug,
		     "give_title_dir=%S  *entry_count=%d\n",
		     ret,*entry_count));
    return ret;
}

void  folder_sort_dir(dir,print,pos)
     struct folder_browser *dir;
     print_sort_message * print;
     int *pos;
{
    const struct string * entry = NULL;


    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"folder_sort_dir",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "folder_sort_dir: dir=%p (%s) pos=%p; type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     pos,
		     dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"folder_sort_dir",
	      "Bad magic (browser_type)",0);

    /* browser_vector_len() will update cache */
    browser_vector_len(dir);

    if (pos && *pos < dir->vector_len && *pos >= 0) {
	entry = dir->vector[*pos].disp_name;

	DPRINT(Debug,10,(&Debug,
			 "folder_sort_dir: *pos=%d entry=%S\n",
			 *pos,entry));

    }
    
    dir->type->browser_folder_sort_it(dir,print);

    if (pos) {
	*pos = -1;

	if (entry) {
	    int i;

	    for (i = 0; i < dir->vector_len; i++)
		if (0 == (string_cmp(entry,
				     dir->vector[i].disp_name,
				     -99 /* Unknown indicator */)))
		    break;
	    
	    if (i < dir->vector_len) {
		DPRINT(Debug,10,(&Debug,
				 "folder_sort_dir: *pos => %d\n",
				 i));

		*pos = i;
	    }
	}

	if (-1 == *pos) {

		DPRINT(Debug,10,(&Debug,
				 "folder_sort_dir: *pos => -1   (NOT FOUND)\n"));

	}
    }
}


/* return reference to array -- do not free_string() result !!! */
const struct string * give_line_dir(dir,idx,flags,comment)
     struct folder_browser *dir;
     int idx;
     int *flags;
     const struct string  **comment;
{
    const struct string * ret = NULL;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"folder_sort_dir",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "give_line_dir: dir=%p (%s); type=%p, idx=%d\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,
		     idx));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"give_line_dir",
	      "Bad magic (browser_type)",0);

    /* browser_vector_len() will update cache */

    if (idx < 0 || idx >= browser_vector_len(dir))
	panic("BROWSER PANIC",__FILE__,__LINE__,"give_line_dir",
	      "Bad directory list index",0);
    
    ret = dir->type->browser_line_idx_it(dir,& dir->vector[idx],comment);

    if (!ret)
	ret    = dir->vector[idx].disp_name;

    if (flags) {
	if (dir->vector[idx].flags & BROWSER_NEEDSTAT) {
	    DPRINT(Debug,10,(&Debug,
			     "give_line_dir: Updating flags -- doing stat\n"));
	    dir->type->browser_stat_routine(dir,idx);
	}
	
	*flags = dir->vector[idx].flags;

	DPRINT(Debug,10,(&Debug,
			 "give_line_dir=%S; *flags=%X",ret,*flags));

    } else {
	DPRINT(Debug,10,(&Debug,
			 "give_line_dir=%S",ret));

    }

    if (comment && *comment) {
	DPRINT(Debug,10,(&Debug,"; comment=%S",
			 *comment));
    }
    DPRINT(Debug,10,(&Debug,"\n"));

    return ret;
}



/* idx is used for selection, fills buffer */
int select_dir_item_by_idx(dir,idx,buffer)
     struct folder_browser *dir;
     int idx; 
     struct string **buffer;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"folder_sort_dir",
	      "Bad magic number (folder_browser)",0);
        
    DPRINT(Debug,10,(&Debug,
		     "select_dir_item_by_idx: dir=%p (%s); type=%p, idx=%d\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,
		     idx));
    
    /* browser_vector_len() will update cache */

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"select_dir_item_by_idx",
	      "Bad magic (browser_type)",0);
    
    if (idx < 0 || idx >= browser_vector_len(dir))
	panic("BROWSER PANIC",__FILE__,__LINE__,"select_dir_item_by_idx",
	      "Bad directory list index",0);
       
    if (dir->vector[idx].flags & BROWSER_NEEDSTAT) {
	DPRINT(Debug,10,(&Debug,
			 "select_dir_item_by_idx: Updating flags -- doing stat\n"));
	dir->type->browser_stat_routine(dir,idx);
    }

    ret = dir->type->browser_select_idx_it(dir,& dir->vector[idx],buffer);

    DPRINT(Debug,10,(&Debug,
		     "select_dir_item_by_idx=%d; dir=%p (%s); type=%p, *buffer=%p\n", 
		     ret,
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,
		     *buffer));
    return ret;
}

int select_dir_item_helper(dir,buffer,newpos)
     struct folder_browser *dir;
     struct string **buffer;
     int *newpos;
{ 
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"select_dir_item_helper",
	      "Bad magic number (folder_browser)",0);

    if (*buffer &&
	string_len(*buffer) > 0) {
	struct string *relative_to_type = NULL;
	
	if (expander(dir,buffer,&relative_to_type)) {

	    if (BROWSER_TYPE_magic != dir->type->magic)
		panic("BROWSER PANIC",__FILE__,__LINE__,
		      "select_dir_item_helper",
		      "Bad magic (browser_type)",0);

	    ret = dir->type->browser_select_it(dir,relative_to_type,buffer,
					       newpos);
	}

	if (relative_to_type)
	    free_string(&relative_to_type);

    } else {
	change_folder_browser_type(dir,&dummy_browser);

	if (BROWSER_TYPE_magic != dir->type->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"select_dir_item_helper",
		  "Bad magic (browser_type)",0);

	ret = dir->type->browser_select_it(dir,*buffer,buffer,
					   newpos);
    }

    return ret;
}


/* buffer is used selection */
int select_dir_item(dir,buffer,pos)
     struct folder_browser *dir;
     struct string **buffer;
     int *pos;
{
    int ret = 0;
    struct string       * position     = NULL;
    struct browser_type * old_type     = NULL;
    char                * old_sys_dir  = NULL;
    int                 newpos = -1;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"select_dir_item",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "select_dir_item: dir=%p (%s) pos=%p; type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     pos,
		     dir->type));
    
    if (pos && *pos >= 0 && *pos < dir->vector_len) {

	position    = dup_string(dir->vector[*pos].disp_name);
	old_type    = dir->type;
	old_sys_dir = dir->sys_dir ? safe_strdup(dir->sys_dir) : NULL; 
	
	DPRINT(Debug,10,(&Debug,
			 "select_dir_item: *pos=%d position=%S\n",
			 *pos,position));	
    }

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"select_dir_item",
	      "Bad magic (browser_type)",0);

    ret = select_dir_item_helper(dir,buffer, &newpos);


    if (pos) {
	*pos = -1;

	if (ret && newpos >= 0 && newpos < browser_vector_len(dir)) {

	    DPRINT(Debug,10,(&Debug,
			     "select_dir_item: *pos => %d (new position)\n",
			     newpos));
	    
	    *pos = newpos;

	} else if (position && old_type == dir->type &&
	    ( old_sys_dir == dir->sys_dir ||
	      (old_sys_dir && dir->sys_dir && 
	       0 == strcmp(old_sys_dir,dir->sys_dir)))) {
	    int i;
	    
	    /* browser_vector_len() will update cache */
	    browser_vector_len(dir);
	    
	    for (i = 0; i < dir->vector_len; i++)
		if (0 == (string_cmp(position,
				     dir->vector[i].disp_name,
				     -99 /* Unknown indicator */)))
		    break;
	    
	    if (i < dir->vector_len) {
		DPRINT(Debug,10,(&Debug,
				 "select_dir_item: *pos => %d\n",
				 i));
		
		*pos = i;
	    }	    	   
	}

	if (-1 == *pos) {
	    DPRINT(Debug,10,(&Debug,
			     "select_dir_item: *pos => -1   (NOT FOUND)\n"));		    
	}
    }

    if (position)
	free_string(&position);
    if (old_sys_dir)
	free(old_sys_dir);


    DPRINT(Debug,10,(&Debug,
		     "select_dir_item=%d; dir=%p (%s); type=%p, *buffer=%p\n", 
		     ret,
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,
		     *buffer));
    return ret;
}

int give_edit_buffer(dir,entry,buffer,fill_with_entry)
     struct folder_browser *dir;
     int entry; 
     struct string **buffer;
     int fill_with_entry;
{
    int ret = 1;
    struct string *res = NULL;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"select_dir_item_helper",
	      "Bad magic number (folder_browser)",0);

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"give_edit_buffer",
	      "Bad magic (browser_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "give_edit_buffer: dir=%p (%s); type=%p, entry=%d, (fill)=%d\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type,
		     entry,fill_with_entry));

    if (*buffer) {	
	if (!verify_string(*buffer)) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"give_edit_buffer",
		  "Bad buffer (string)",0);
	}

	DPRINT(Debug,10,(&Debug,
			 "give_edit_buffer- *buffer=%p\n",*buffer));
    } else {
	DPRINT(Debug,10,(&Debug,
			 "give_edit_buffer- *buffer=NULL\n"));
    }

    /* NOTE: dir->sys_dir does not include possible user@host
     *  but  dir->dirname includes possible user@host !
     *
     *  dir->dirname is NULL when default directory (menu) is selected
     *
     *  browser_vector_len() will update cache 
     */
    
    if (entry < 0 || 
	entry >= browser_vector_len(dir)) {

	DPRINT(Debug,10,(&Debug,
			 "give_edit_buffer: entry (%d) not in range (0-%d)\n",
			 entry,dir->vector_len-1));

	/* Do not fail on empty directory */
	if (fill_with_entry && entry != 0)	
	    ret = 0;
	
	fill_with_entry = 0;
    }

	    
    if (fill_with_entry) 
	res = dir->type->browser_cat_it(dir,
					dir->vector[entry].disp_name);
    else 
	res = dir->type->browser_cat_it(dir,NULL);
    
    if (res) {
	if (*buffer)
	    free_string(buffer);
	*buffer = res;
    } else {
	DPRINT(Debug,10,(&Debug,
			 "give_edit_buffer: failed -- NO CHANGE OF BUFFER!\n"));
	ret = 0;
    }
    
    if (*buffer) {
	DPRINT(Debug,10,(&Debug,
			 "give_edit_buffer=%d; *buffer=%S\n",ret,*buffer));
    } else {
	DPRINT(Debug,10,(&Debug,
			 "give_edit_buffer=%d; *buffer=NULL\n",ret));
    }
    return ret;
}

/* Returns 1 if changed, -1 error */
int dir_reset_filter(dir)
     struct folder_browser *dir;
{    
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dir_reset_filter",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "dir_reset_filter: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_reset_filter",
	      "Bad magic (browser_type)",0);

    if (dir->filter) {
	
	ret = dir->type->browser_reset_filter_it(dir);
    }


    DPRINT(Debug,10,(&Debug,"dir_reset_filter=%d\n",ret));

    return ret;
}

static void filter_director P_((struct folder_browser *dir));

/* Returns 1 if directory changed, -1 error */
int dir_change_filter(dir,filter)
     struct folder_browser *dir;
     const struct string *filter;
{    
    int ret = 0;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_reset_filter",
	      "Bad magic (browser_type)",0);

    DPRINT(Debug,10,(&Debug,
		     "dir_change_filter: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (dir->filter) {

	ret = dir->type->browser_reset_filter_it(dir);
	if (ret < 0)
	    goto fail;
    }

    if (filter) {
	dir->filter = dup_string(filter);
	
	/* browser_vector_len() applies filter_director()
	 * if dir->vector_len was -1 
	 */
	if (dir->vector_len >= 0)
	    filter_director(dir);

	ret = 1;
    }

 fail:
    DPRINT(Debug,10,(&Debug,
		     "dir_change_filter=%d\n",ret));

    return ret;
}


static void filter_director(dir)
     struct folder_browser *dir;
{
    int i;

    for (i = 0; i < dir->vector_len; i++) {
	if (!string_match(dir->vector[i].disp_name,
			  dir->filter,0)) {
	    int j;

	    if (dir->vector[i].sys_name) {
		free(dir->vector[i].sys_name);
		dir->vector[i].sys_name = NULL;
	    }
	    if (dir->vector[i].disp_name) 
		free_string(&(dir->vector[i].disp_name));
	    
	    for (j = i+1; j < dir->vector_len; j++) {
		dir->vector[j-1] = dir->vector[j];
		dir->vector[j].sys_name = NULL;
		dir->vector[j].disp_name = NULL;
	    }
	    dir->vector_len--;
	    i--;
	}
    }
}

const struct string * dir_give_filter(dir)
     struct folder_browser *dir;
{
    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dir_give_filter",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "dir_give_filter: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));
    
   
    if (dir->filter) {
	DPRINT(Debug,10,(&Debug,
			 "dir_give_filter=%S\n",
			 dir->filter));

	return dir->filter;
    }

    DPRINT(Debug,10,(&Debug,
		     "dir_give_filter=NULL\n"));

    return NULL;
}

int dir_give_count(dir)
     struct folder_browser *dir;
{
    int r;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dir_give_count",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "dir_give_count: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_give_count",
	      "Bad magic (browser_type)",0);

    /*
     *  browser_vector_len() will update cache 
     */
    r  = browser_vector_len(dir);

    DPRINT(Debug,10,(&Debug,
		     "dir_give_count=%d\n",r));
    return r;
}

int dir_autocomplete(dir,buffer,first_match)
     struct folder_browser *dir;
     struct string **buffer;
     int *first_match;
{
    int ret = 0;
    int len;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dir_autocomplete",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "dir_autocomplete: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (first_match)
	*first_match = -1;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_autocomplete",
	      "Bad magic (browser_type)",0);
    if (*buffer &&
	(len = string_len(*buffer)) > 0) {

	/*
	 *
	 *  browser_vector_len() will update cache 
	 */
	int count  = browser_vector_len(dir);

	int sep    = dir->type->browser_separator_it(dir);
	struct string * prefix = dir->type->browser_cat_it(dir,NULL);
	int pos = 0;
	int atpos = -1;
	int tailpos = len; 
	uint16 firstcode = 0;

	if (sep) {

	    int x;
	    for (x = 0; x < len; x++) {
		uint16 code1 = give_unicode_from_string(*buffer,x);
		
		if (x < tailpos) {
		    if (0x0040 /* '@' */   == code1)
			atpos = x;
		    else if (0x003A /* ':' */ == code1) {
			tailpos = x+1;
		    } else if (0x002F /* '/' */ == code1) {
			tailpos = x;
			
		    }
		}

		if (code1 == sep)
		    pos = x + 1;				
	    }
	}
	
	DPRINT(Debug,15,(&Debug,
			 "dir_autocomplete: sep=%d  pos=%d atpos=%d tailpos=%d len=%d\n",
			 sep,pos,atpos,tailpos,len));

	if (0 == pos) {
	    firstcode = give_unicode_from_string(*buffer,0);
	    
	    DPRINT(Debug,15,(&Debug,"dir_autocomplete: First char %04X\n",firstcode));
	    
	    switch (firstcode) {
	    case 0x0024 /* '$' */:
	    case 0x007E /* '~' */:
	    case 0x0026 /* '&' */:
	    case 0x003D /* '=' */:
	    case 0x002B /* '+' */:
	    case 0x0025 /* '%' */:
	    case 0x003E /* '>' */:
	    case 0x003C /* '<' */:
	    case 0x007B /* '{' */:
	    
		DPRINT(Debug,10,(&Debug,
				 "dir_autocomplete: No autocomplete for special values: %S\n",
				 *buffer));

		goto fail2;

	    case 0x0023: /* # */
		DPRINT(Debug,15,(&Debug,
				 "dir_autocomplete: #hashmark\n"));
		
		if (&hashmark_browser == dir->type) {
		    DPRINT(Debug,15,(&Debug,"dir_autocomplete: first char is prefix\n"));
		    pos = 1;
		} 
	    }
	}

	if (!prefix) {
	    DPRINT(Debug,10,(&Debug,
			     "dir_autocomplete: failed to get currect directory prefix\n"));
	} else if (0 == count) {

	    DPRINT(Debug,10,(&Debug,
			     "dir_autocomplete: no autocomplete on empty directory\n"));
	} else if (atpos != -1 && pos < tailpos) {
	    DPRINT(Debug,10,(&Debug,
			     "dir_autocomplete: no autocomplete on user@host\n"));

	} else if (pos < len) {

	    int X = 0;
	    struct string * first = clip_from_string(*buffer,&X,pos);
	    struct string * rest  = clip_from_string(*buffer,&X,len);
	    int           restlen = string_len(rest);
	    int a;

	    int foundcount = 0;
	    int idx        = -1;
	    int exact      = -1;

	    DPRINT(Debug,15,(&Debug,
			     "   buffer=%S\n",*buffer));
	    DPRINT(Debug,15,(&Debug,
			     "   first =%S\n",first));
	    DPRINT(Debug,15,(&Debug,
			     "   rest  =%S\n",rest));
	    
	    DPRINT(Debug,15,(&Debug,
			     "   dir   =%S\n",prefix));
	    
	    if (0 != string_cmp(prefix,first,99)) {
		DPRINT(Debug,10,(&Debug,
				 "dir_autocomplete: Directory prefix not match %S <> %S\n",
				 prefix,first));
		goto fail;
	    }

	    for (a = 0; a < count; a++) {
		
		int Y = 0;

		struct string * A = 
		    clip_from_string(dir->vector[a].disp_name,
				     &Y,restlen);

		if (0 == string_cmp(rest,A,99)) {
		    foundcount++;
		    idx = a;

		    DPRINT(Debug,15,(&Debug,"  match %d found: %S\n",
				     a,dir->vector[a].disp_name));

		    if (first_match &&
			-1 == *first_match)
			*first_match = a;

		}

		free_string(&A);

		if (0 == string_cmp(rest,dir->vector[a].disp_name,99)) {

		    DPRINT(Debug,15,(&Debug,"  complete match %d found: %S\n",
				     a,dir->vector[a].disp_name));

		    exact = a;

		}		   
	    }
	 

	    if (-1 == exact) {

		if (0 == foundcount) {

		    switch (firstcode) {
		    case 0x0023: /* # */
			DPRINT(Debug,15,(&Debug,
					 "dir_autocomplete: #hashmark\n"));
			
			if (&hashmark_browser != dir->type) {
			    DPRINT(Debug,15,(&Debug,
					     "dir_autocomplete: not hashmark browser; no autocomplete on special value\n"));
			    goto fail;
			}
			
			break;
		    }

		    lib_error(CATGETS(elm_msg_cat, MeSet, MeCompleteNotFound,
				      "Can't complete %S: not found"),
			      rest);
		} else if (1 < foundcount) 
		    lib_error(CATGETS(elm_msg_cat, MeSet, MeCompleteAmbiguous,
				      "Can't complete %S: ambiguous"),
			      rest);
		else {		
		    free_string(buffer);
		    *buffer = cat_strings(first,dir->vector[idx].disp_name,1);
		}

		ret = 1;   
	    }

	fail:

	    free_string(&rest);
	    free_string(&first);	    	   

	} else {
	    DPRINT(Debug,10,(&Debug,
			     "dir_autocomplete: no autocomplete data!\n"));
	}

    fail2:
	if (prefix)
	    free_string(&prefix);
    } 


    if (*buffer) {
	DPRINT(Debug,10,(&Debug,
			 "dir_autocomplete=%d; *buffer=%S\n",ret,*buffer));
    } else {
	DPRINT(Debug,10,(&Debug,
			 "dir_autocomplete=%d; *buffer=NULL\n",ret));
    }
	
    return ret;
}

static int buffer_found_wildcard P_((struct string *buffer));
static int buffer_found_wildcard(buffer)
     struct string *buffer;
{
    int len = string_len(buffer);
    int ret = 0;
    int x;

    DPRINT(Debug,12,(&Debug,
		     "buffer_found_wildcard: buffer=%S, len=%d\n",
		     buffer,len));

    for (x = 0; x < len; x++) {
	uint16 code1 = give_unicode_from_string(buffer,x);
	
	if (0x002A /* '*' */ == code1) {
	    ret = 1;
	    break;
	} else if (0x003F /* '?' */ == code1) {
	    ret = 1;
	    break;
	}
    }

    DPRINT(Debug,12,(&Debug,
		     "buffer_found_wildcard=%d\n",ret));

    return ret;
}


/* Changes directory
   Do not apply filter
   *buffer != NULL
 */

static struct string * internal_give_wildcard P_((struct folder_browser *dir,
						 struct string **buffer));
static struct string * internal_give_wildcard (dir,buffer)
     struct folder_browser *dir;
     struct string **buffer;
{
    struct string * ret = NULL;

    int len,x;
    struct string *relative_to_type = NULL;
    len = string_len(*buffer);

    if (expander(dir,buffer,&relative_to_type)) {
	int base_len = 0;
	int sep;
	int X = 0;
	struct string * new_relative = NULL;
	struct string * dispname = NULL;
	int disp_len = 0;
	int X2 = 0;
	int len1 = 0;
	
	if (BROWSER_TYPE_magic != dir->type->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"internal_give_wildcard",
		  "Bad magic (browser_type) after expander",0);
	
	sep = dir->type->browser_separator_it(dir);

	if (relative_to_type) {		
	    int x1;
	    
	    len1 = string_len(relative_to_type);
	    
	    for (x1 = 0; x1 < len1; x1++) {
		uint16 code1 = 
		    give_unicode_from_string(relative_to_type,x1);
		
		if (0x002A /* '*' */ == code1) {
		    break;
		} else if (0x003F /* '?' */ == code1) {
		    break;
		    
		} else if (sep && sep == code1) {
		    /* (separator) character set is assumed to be 
		       ASCII compatible on here! */
		    base_len = x1+1;
		}
	    }
	    
	    if (0 == base_len && 
		(dir->type == &dummy_browser
#if DIROPS
		 || dir->type == &local_browser
#endif
		 )) { 
		/* Special case: 
		   Treat adas*hjghj as filtering of current
		   directory  (and not filtering of default menu)
		*/
		
		new_relative = new_string2(system_charset,s2us("."));
		dispname = new_string2(system_charset,s2us("."));
		
		change_folder_browser_type(dir,
#if DIROPS
			    &local_browser
#else
			    &dummy_browser
#endif
			    );
				
	    } else {
		for (x = 0; x < len; x++) {
		    uint16 code1 = give_unicode_from_string(*buffer,x);
		    
		    if (0x002A /* '*' */ == code1) {
			break;
		    } else if (0x003F /* '?' */ == code1) {
			break;
		    } else if (sep && sep == code1) {
			/* (separator) character set is assumed to be 
			   ASCII compatible on here! */
			disp_len = x+1;
		    }
		}
		
		new_relative = clip_from_string(relative_to_type,&X,
						base_len);
		
		if (0 == disp_len) { 
		    /*  HACK for =* */
		    dispname = dup_string(new_relative);
		} else
		    dispname = clip_from_string(*buffer,&X2,disp_len);
	    }

	    
	} else {
	    lib_error(CATGETS(elm_msg_cat, MeSet, MeBadWildcard,
			      "Bad wildcard specification %S"),
		      *buffer);
	    
	    /* new_relative == NULL, but that is OK */
	    
	    dispname = dup_string(*buffer);
	}

	if (BROWSER_TYPE_magic != dir->type->magic)
	    panic("BROWSER PANIC",__FILE__,__LINE__,"internal_give_wildcard",
		  "Bad magic (browser_type)",0);


	if (dir->type->browser_change_it(dir,new_relative,&dispname)) {
	
	    if (relative_to_type)
		ret =  clip_from_string(relative_to_type,&X,len1);
	    else {

		DPRINT(Debug,10,(&Debug,"internal_give_wildcard: Direcory changed. Can not give filter.\n"));

	    }
	    
	}
       	
	if (new_relative)
	    free_string(&new_relative);
	free_string(&dispname);
    }
    
    if (relative_to_type)
	free_string(&relative_to_type);
              	
    return ret;
}

/* Changes directory if have wildcard,
   do not apply filter
   *changed == 1 if directory is changed
 */
struct string * dir_give_wildcard(dir,buffer, changed)
     struct folder_browser *dir;
     struct string **buffer;
     int *changed;
{
    struct string * ret = NULL;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dir_autocomplete",
	      "Bad magic number (folder_browser)",0);
        
    DPRINT(Debug,10,(&Debug,
		     "dir_give_wildcard: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));
    
    if (changed)
	*changed = 0;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_give_wildcard",
	      "Bad magic (browser_type)",0);

    if (*buffer) {
	int r = 0;
	
	if (!verify_string(*buffer)) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"dir_give_wildcard",
		  "Bad buffer (string)",0);
	}

	DPRINT(Debug,10,(&Debug,
			 "dir_give_wildcard- *buffer=%p\n",*buffer));

	r = buffer_found_wildcard(*buffer);

	if (!r) {
	    DPRINT(Debug,10,(&Debug,
			     "dir_give_wildcard: No wildcards\n"));
	    goto fail;
	} 

	/* Assumes that directory is changed */

	if (changed)
	    *changed = 1;

	ret = internal_give_wildcard(dir,buffer);

	if (ret) {
	    DPRINT(Debug,10,(&Debug,
			     "dir_give_wildcard: wildcard %S\n",ret));
	}

    }

 fail:

    return ret;
}

/* Resets filter
   Changes directory if have wildcard
   Applies filter
 */
int dir_is_wildcard(dir,buffer)
     struct folder_browser *dir;
     struct string **buffer;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dir_is_wirdcard",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "dir_is_wildcard: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (dir->filter)
	free_string(&(dir->filter));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_is_wildcard",
	      "Bad magic (browser_type)",0);

    if (*buffer) {
	
	struct string *W = NULL;

	if (!verify_string(*buffer)) {
	    panic("BROWSER PANIC",__FILE__,__LINE__,"dir_is_wildcard",
		  "Bad buffer (string)",0);
	}

	DPRINT(Debug,10,(&Debug,
		    "dir_is_wildcard- *buffer=%p\n",*buffer));

	ret = buffer_found_wildcard(*buffer);

	if (!ret) {
	    DPRINT(Debug,10,(&Debug,
			     "dir_is_wildcard: No wildcards\n"));
	    goto fail;
	}

	W = internal_give_wildcard(dir,buffer);

	if (W) {

	    DPRINT(Debug,10,(&Debug,
			     "dir_is_wildcard: wildcard %S\n",W));

	    dir->filter = W;


	    /* browser_vector_len() applies filter_director()
	     * if dir->vector_len was -1 
	     */
	    if (dir->vector_len >= 0)
		filter_director(dir);
	}
      
    } else {
	DPRINT(Debug,10,(&Debug,
			 "dir_is_wildcard- *buffer=NULL\n"));
    }

 fail:
    if (*buffer) {
	DPRINT(Debug,10,(&Debug,
			 "dir_is_wildcard=%d; *buffer=%S\n",ret,*buffer));
    } else {
	DPRINT(Debug,10,(&Debug,
			 "dir_is_wildcard=%d; *buffer=NULL\n",ret));
    }

    return ret;
}


struct folder_info * folder_from_dir_item(dir, treat_as_spooled)
     struct folder_browser *dir;
     int treat_as_spooled;
{
    struct folder_info * res = NULL;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"folder_from_dir_item",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "folder_from_dir_item: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"folder_from_dir_item",
	      "No selection",0);

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"folder_from_dir_item",
	      "Bad magic (browser_type)",0);

    res = dir->type->browser_folder_from_it(dir, treat_as_spooled);

    if (res) {
	DPRINT(Debug,10,(&Debug,
			 "folder_from_dir_item=%p (%s), type=%p\n",
			 res,res->cur_folder_sys,res->folder_type));
    } else {
	DPRINT(Debug,10,(&Debug,"folder_from_dir_item=NULL\n"));
    }

    return res;
}

int give_dir_flags(dir) 
    struct folder_browser *dir;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"give_dir_flags",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "folder_dir_flags: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (dir->selection) 
	ret |= BROWSER_SELECTED | dir->selection->flags;

    DPRINT(Debug,10,(&Debug, "folder_dir_flags=%X\n",ret));
    return ret;
}

int create_selection_dir(dir)
     struct folder_browser *dir;
{
    int res = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"create_selection_dir",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "create_selection_dir: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"create_selection_dir",
	      "No selection",0);

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"create_selection_dir",
	      "Bad magic (browser_type)",0);

    res = dir->type->browser_create_selection_it(dir);
    
    DPRINT(Debug,10,(&Debug,"create_selection_dir=%d\n",res));

    return res;
}

void malloc_keep_browser_write_state(ptr)
     WRITE_STATE * ptr;
{
    (*ptr) = safe_zero_alloc(sizeof (struct browser_write_state));

    (*ptr)->magic             = WS_magic;
}

void free_keep_browser_write_state(ptr)
     WRITE_STATE * ptr;
{
    if (*ptr) {

	if (WS_magic != (*ptr)->magic)
	    panic("MBX PANIC",__FILE__,__LINE__,"free_keep_browser_write_state",
		  "Bad magic type",0);


	/* bzero is defined hdrs/elm_defs.h */
	bzero((void *)*ptr,sizeof (struct browser_write_state));

	free(*ptr);
	*ptr = NULL;
    }	    
}

int prepare_write_folder(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE * write_state_ptr;
{
    int ret = 0;
    WRITE_STATE ptr = NULL;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"prepare_write_folder",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "prepare_write_folder: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"prepare_write_folder",
	      "No selection",0);

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"prepare_write_folder",
	      "Bad magic (browser_type)",0);

    malloc_keep_browser_write_state(&ptr);
    dir->type->zero_ws_fields_it(ptr);

    ret = dir->type->browser_prepare_write_it(dir,ptr);

    if (!ret) {
	dir->type->free_ws_fields_it(ptr);
	free_keep_browser_write_state(&ptr);
    }

    *write_state_ptr = ptr;

    DPRINT(Debug,10,(&Debug,"prepare_write_folder=%d\n",ret));
    return ret;
}

int sync_write_folder(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"sync_write_folder",
	      "Bad magic number (folder_browser)",0);

    if (write_state_ptr->magic  != WS_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"sync_write_folder",
	      "Bad magic",0);
    
    DPRINT(Debug,10,(&Debug,
		     "sync_write_folder: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"sync_write_folder",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_sync_write_it(dir,write_state_ptr);

    DPRINT(Debug,10,(&Debug,"sync_write_folder=%d\n",ret));

    return ret;   
}

int end_write_folder(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE * write_state_ptr;
{
    int ret = 0;
    WRITE_STATE ptr = *write_state_ptr;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"end_write_folder",
	      "Bad magic number (folder_browser)",0);
    
    if (ptr->magic  != WS_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"end_write_folder",
	      "Bad magic",0);

    DPRINT(Debug,10,(&Debug,
		     "end_write_folder: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"end_write_folder",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_end_write_it(dir,ptr);
    dir->type->free_ws_fields_it(ptr);
    free_keep_browser_write_state(&ptr);

    *write_state_ptr = ptr;

    DPRINT(Debug,10,(&Debug,"end_write_folder=%d\n",ret));
    return ret;   
}

struct string * dir_cat_filename(dir,filename)
     struct folder_browser *dir;
     const struct string *filename;
{
    struct string * ret = NULL;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dir_cat_filename",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "dir_cat_filename: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_cat_filename",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_cat_it(dir,filename);

    if (ret) {
	DPRINT(Debug,10,(&Debug,
			 "dir_cat_filename=%S\n",ret));
    } else {
	DPRINT(Debug,10,(&Debug,
			 "dir_cat_filename=NULL\n"));
    }
    return ret;
    

}

struct string * selection_name_dir(dir)
     struct folder_browser *dir;
{
    struct string * ret = NULL;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"selection_name_dir",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "selection_name_dir: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));
    
    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"selection_name_dir",
	      "No selection",0);

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"selection_name_dir",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_cat_it(dir,dir->selection->disp_name);

    if (ret) {
	DPRINT(Debug,10,(&Debug,
			 "selection_name_dir=%S\n",ret));
    } else {
	DPRINT(Debug,10,(&Debug,
			 "selection_name_dir=NULL\n"));
    }
    return ret;
}

void clear_selection_dir(dir)
     struct folder_browser *dir;
{
    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"clear_selection_dir",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "clear_selection_dir: dir=%p (%s); type=%p\n", 
		     dir,dir->sys_dir ? dir->sys_dir : "<NULL>",
		     dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"clear_selection_dir",
	      "Bad magic (browser_type)",0);

    clear_dir_selection(dir);
}

/* helper routines for fileio.c */
void start_fd_write_state(fd,dir,write_state_ptr)
     FILE *fd;
     struct folder_browser **dir;
     WRITE_STATE *write_state_ptr;
{
    WRITE_STATE ptr = NULL;
    
    DPRINT(Debug,10,(&Debug,"start_fd_write_state: fd=%p\n",fd));

    *dir = browser_malloc(selection_folder);
    
    malloc_keep_browser_write_state(&ptr);
    (*dir)->type->zero_ws_fields_it(ptr);

    ptr->a.local.save_file = fd;   /* WARNING: shared pointer! */

    *write_state_ptr = ptr;

    DPRINT(Debug,10,(&Debug,
		     "start_fd_write_state: *dir=%p; type=%p\n",
		     *dir,(*dir)->type));		
}

void end_fd_write_state(dir,write_state_ptr)
     struct folder_browser **dir;
     WRITE_STATE *write_state_ptr;
{
    WRITE_STATE ptr = *write_state_ptr;

    if (FOLDER_BROWSER_magic != (*dir)->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"end_fd_write_state",
	      "Bad magic number (folder_browser)",0);

    if (ptr->magic  != WS_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"end_fd_write_state",
	      "Bad magic",0);

    DPRINT(Debug,10,(&Debug,
		     "end_fd_write_state: *dir=%p; type=%p\n", 
		     *dir,(*dir)->type));

    ptr->a.local.save_file = NULL;   /* Reset shared pointer! */

    if (BROWSER_TYPE_magic != (*dir)->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"end_fd_write_state",
	      "Bad magic (browser_type)",0);

    (*dir)->type->free_ws_fields_it(ptr);
    free_keep_browser_write_state(&ptr);

    *write_state_ptr = ptr;

    free_browser(dir);
}

/* Returns -1 if not seekable, else position */
long tell_dir_write_state(dir,write_state_ptr)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
{
    long ret = -1;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"tell_dir_write_state",
	      "Bad magic number (folder_browser)",0);

    if (write_state_ptr->magic  != WS_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"tell_dir_write_state",
	      "Bad magic",0);

    DPRINT(Debug,10,(&Debug,
		     "tell_dir_write_state: *dir=%p; type=%p\n", 
		     dir,dir->type));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"tell_dir_write_state",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_tell_it_ws(dir,write_state_ptr);

    DPRINT(Debug,10,(&Debug,"tell_dir_write_state=%ld\n",ret));
    return ret;
}

/* Returns 0 on failure! */
int seek_dir_write_state(dir,write_state_ptr,pos)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     long pos;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"seek_dir_write_state",
	      "Bad magic number (folder_browser)",0);
    
    if (write_state_ptr->magic  != WS_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"seek_dir_write_state",
	      "Bad magic",0);

    DPRINT(Debug,10,(&Debug,
		     "seek_dir_write_state: *dir=%p; type=%p, pos=%ld\n", 
		     dir,dir->type,pos));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"seek_dir_write_state",
	      "Bad magic (browser_type)",0);


    ret = dir->type->browser_seek_it_ws(dir,write_state_ptr,pos);

    DPRINT(Debug,10,(&Debug,"seek_dir_write_state=%d\n",ret));
    return ret;
}

/* Return 0 on failure */
int write_dir_write_state(dir,write_state_ptr,l,buffer)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int l; 
     const char *buffer;
{    
    int ret = 0;
	
    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"write_dir_write_state",
	      "Bad magic number (folder_browser)",0);

    if (write_state_ptr->magic  != WS_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"write_dir_write_state",
	      "Bad magic",0);

    DPRINT(Debug,10,(&Debug,
		     "write_dir_write_state: *dir=%p; type=%p, len=%d\n", 
		     dir,dir->type,l));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"write_dir_write_state",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_write_it_ws(dir,write_state_ptr,l,buffer);

    DPRINT(Debug,10,(&Debug,"write_dir_write_state=%d\n",ret));
    return ret;
}

int write_envelope_start(dir,write_state_ptr,write_envelope,current_header,
			 env_flags)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
     int *env_flags;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"write_envelope_start",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "write_write_envelope_start: *dir=%p; type=%p, write_envelope=%d\n", 
		     dir,dir->type,write_envelope));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"write_envelope_start",
	      "Bad magic (browser_type)",0);
    if (write_state_ptr->magic  != WS_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"write_envelope_start",
	      "Bad magic",0);

    ret = dir->type->browser_start_we_it(dir,write_state_ptr,
					 write_envelope,current_header,
					 env_flags);

    DPRINT(Debug,10,(&Debug,
		     "write_write_envelope_start=%d    *env_flags=%d%s\n",
		     ret,*env_flags,
		     (*env_flags & WE_ADD_RETURN_PATH) ? 
		     " WE_ADD_RETURN_PATH" : ""));

    return ret;
}

int write_envelope_end(dir,write_state_ptr,write_envelope,current_header)
     struct folder_browser *dir;
     WRITE_STATE write_state_ptr;
     int write_envelope;
     struct header_rec *current_header;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"write_envelope_end",
	      "Bad magic number (folder_browser)",0);
    
    DPRINT(Debug,10,(&Debug,
		     "write_write_envelope_end: *dir=%p; type=%p, write_envelope=%d\n", 
		     dir,dir->type,write_envelope));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"write_envelope_end",
	      "Bad magic (browser_type)",0);
    if (write_state_ptr->magic  != WS_magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"write_envelope_end",
	      "Bad magic",0);


    ret = dir->type->browser_end_we_it(dir,write_state_ptr,
				       write_envelope,current_header);

    DPRINT(Debug,10,(&Debug,"write_write_envelope_end=%d\n",ret));

    return ret;
}

int selection_is_folder(dir,folder)
     struct folder_browser *dir;
     struct folder_info *folder;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"selection_is_folder",
	      "Bad magic number (folder_browser)",0);

    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"selection_is_folder",
	      "No selection",0);

    DPRINT(Debug,10,(&Debug,
		     "selection_is_folder: *dir=%p; type=%p, folder=%p\n", 
		     dir,dir->type,folder));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"selection_is_folder",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_selection_is_it(dir,folder);

    DPRINT(Debug,10,(&Debug,"selection_is_folder=%d\n",ret));

    return ret;
}

int dir_make_ref(dir,refname,iscopy,is_text) 
     struct folder_browser *dir;
     char **refname; 
     int *iscopy;
     int is_text;
{
    int ret = 0;

    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"dir_make_ref",
	      "Bad magic number (folder_browser)",0);

    if (!dir->selection) 
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_make_ref",
	      "No selection",0);

    DPRINT(Debug,10,(&Debug,
		     "dir_make_ref: *dir=%p; type=%p, is_text=%d\n", 
		     dir,dir->type,is_text));

    *refname = NULL;
    *iscopy  = 0;

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"dir_make_ref",
	      "Bad magic (browser_type)",0);

    ret = dir->type->browser_make_ref_it(dir,refname,iscopy,is_text);

    DPRINT(Debug,10,(&Debug,"dir_make_ref=%d;   *refname=%s, *iscopy=%d\n",
		     ret,
		     *refname ? *refname : "<NULL>",
		     *iscopy));
    return ret;
}


int browser_vector_len(dir)
     struct folder_browser *dir;
{
    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_vector_len",
	      "Bad magic number (folder_browser)",0);

    if (dir->vector_len >= 0) {
	
	DPRINT(Debug,12,(&Debug,
			 "browser_vector_len=%d, dir=%p\n",
			 dir->vector_len,dir));
	return dir->vector_len;
    }
    DPRINT(Debug,12,(&Debug,
		     "browser_vector_len: dir=%p   (updating cache)\n",
		     dir));

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_vector_len",
	      "Bad magic (browser_type)",0);

    dir->type->browser_update_it(dir);

    if (dir->filter)
	filter_director(dir);

    DPRINT(Debug,12,(&Debug,
		     "browser_vector_len=%d\n",
		     dir->vector_len));

    return dir->vector_len;
}


/* Sets username@server handle for & -prefix    */
void browser_set_remote_server(dir,server)
     struct folder_browser *dir;
     const struct remote_server *server;
{
    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_set_remote_server",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "browser_set_remote_server: *dir=%p; type=%p, ", 
		     dir,dir->type));
    
    if (server) {
	DPRINT(Debug,10,(&Debug,"server=%p\n"));
    } else {
	DPRINT(Debug,10,(&Debug,"server=NULL\n"));
    }

    if (BROWSER_TYPE_magic != dir->type->magic)
	panic("BROWSER PANIC",__FILE__,__LINE__,"browser_set_remote_server",
	      "Bad magic (browser_type)",0);
    
    if (dir->default_server) {
	DPRINT(Debug,12,(&Debug,
			 "browser_set_remote_server: previous server was %p\n",
			 dir->default_server));

	remote_server_dec_dir(dir);
    }

    if (server)
	remote_server_inc_dir(dir,server);
}

/* Gives browser's default remote server handle */
const struct remote_server * browser_get_remote_server(dir)
     struct folder_browser *dir;
{
    if (FOLDER_BROWSER_magic != dir->magic)
	panic("MBX PANIC",__FILE__,__LINE__,"browser_get_remote_server",
	      "Bad magic number (folder_browser)",0);

    DPRINT(Debug,10,(&Debug,
		     "browser_get_remote_server: *dir=%p; type=%p, ", 
		     dir,dir->type));
    
    if (dir->default_server) {
	DPRINT(Debug,10,(&Debug,
			 "browser_get_remote_server=%p\n",
			 dir->default_server));
	return dir->default_server;
    }

    DPRINT(Debug,10,(&Debug,
		     "browser_get_remote_server=NULL\n"));
    return NULL;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
