static char rcsid[] = "@(#)$Id: filelock.c,v 2.1 2021/07/13 07:58:37 hurtta Exp $";

/*****************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.1 $   $State: Exp $
 *
 *  Modified by: Kari Hurtta <hurtta+elm@siilo.FMI.FI>
 *           or  Kari Hurtta <elm@elmme-mailer.org>
 ******************************************************************************
 * Partially based on Elm 2.4 src/lock.c but code is rewritten.
 * That code was following copyright:
 *
 *  The Elm Mail System 
 *
 *                      Copyright (c) 1988-1992 USENET Community Trust
 *                      Copyright (c) 1986,1987 Dave Taylor
 *****************************************************************************
 * Moved locking from lib/mbox/localmbx.c. Also added support for blocking 
 * (waiting) lock.
 *
 ****************************************************************************/

#include "def_misc.h"
#include "s_me.h"

/* extra prototype */
#include "filelock.h"

#if ANSI_C
#define S_(x) static x;
#else
#define S_(x)
extern int errno;
#endif

#ifdef SYSCALL_LOCKING
#  if (defined(BSD_TYPE) || !defined(apollo))
#    include <sys/file.h>
#  endif
#endif

DEBUG_VAR(Debug,__FILE__,"file");


typedef enum FLOCKING_status lock_op_f P_((int                    fd,
					   enum FLOCKING_mode     op,
					   const char           * filename /* may be NULL */,
					   enum FLOCKING_blocking blockit,
					   int                  * errno_res));
S_(lock_op_f lock_op_dummy)
static  enum FLOCKING_status lock_op_dummy P_((int                    fd,
					       enum FLOCKING_mode     op,
					       const char           * filename /* may be NULL */,
					       enum FLOCKING_blocking blockit,
					       int                  * errno_res));
static  enum FLOCKING_status lock_op_dummy(fd,op,filename,blockit,errno_res)
     int                    fd;
     enum FLOCKING_mode     op;
     const char           * filename /* may be NULL */;
     enum FLOCKING_blocking blockit;
     int                  * errno_res;
{
    return FLOCKING_UNSUPPORTED;
}

#ifdef COMPILE_FCNTL_LOCKING
S_(lock_op_f lock_op_fcntl)
static  enum FLOCKING_status lock_op_fcntl P_((int                    fd,
					       enum FLOCKING_mode     op,
					       const char           * filename /* may be NULL */,
					       enum FLOCKING_blocking blockit,
					       int                  * errno_res));
static  enum FLOCKING_status lock_op_fcntl(fd,op,filename,blockit,errno_res)
     int                    fd;
     enum FLOCKING_mode     op;
     const char           * filename /* may be NULL */;
     enum FLOCKING_blocking blockit;
     int                  * errno_res;
{
    enum FLOCKING_status ret = FLOCKING_UNSUPPORTED;
    
    struct flock lock_info;

    enum {
	pass_trylock,
	pass_block,
	pass_done } a;

    int have_message = 0;
    int have_error = 0;
    
    /* bzero is defined hdrs/elm_defs.h */    
    bzero ((void *)&lock_info,sizeof lock_info);

    for (a = pass_trylock; a < pass_done; a++) {
	enum syscall_status r;
	const char *sop  UNUSED_VAROK = "BAD OP";
	const char *scmd UNUSED_VAROK = "BAD CMD";

	int cmd = 0;
	
	if (errno_res)
	    *errno_res = 0;

	switch (a) {
	case pass_trylock:  cmd = F_SETLK;  scmd = "F_SETLK";  break;
	case pass_block:    cmd = F_SETLKW; scmd = "F_SETLKW"; break;
	case pass_done:                                        break;
	}
	
	
	switch (op) {
	case FLOCKING_release:   lock_info.l_type = F_UNLCK; sop = "F_UNLCK"; break;
	case FLOCKING_exclusive: lock_info.l_type = F_WRLCK; sop = "F_WRLCK"; break;
	case FLOCKING_shared:    lock_info.l_type = F_RDLCK; sop = "F_RDLCK"; break;
	}

	lock_info.l_whence = 0;
	lock_info.l_start = 0;
	lock_info.l_len = 0;

	SIGDPRINT(Debug,13,(&Debug,"lock_op_fcntl: fcntl  fd=%d",
			 fd));
	if (filename) {
	    SIGDPRINT(Debug,13,(&Debug," %Q",filename));
	}
	SIGDPRINT(Debug,13,(&Debug,
			 " cmd=%d %s  lock_info.l_type=%d %s\n",
			    cmd,scmd,lock_info.l_type,sop));
	
	r = fcntl(fd,cmd,&lock_info);

	SIGDPRINT(Debug,13,(&Debug,"lock_op_fcntl: fcntl returns %d",r));
	
	switch(r) {
	    int err;
	case syscall_success:
	    SIGDPRINT(Debug,13,(&Debug," success"));
	    ret = FLOCKING_OK;
	    break;
	case syscall_error:
	    err = errno;
	    if (errno_res)
		*errno_res = err;
	    
	    SIGDPRINT(Debug,13,(&Debug," error"));
	    if (err) {
		SIGDPRINT(Debug,13,(&Debug,", errno %d",
				    err));
		switch (blockit) {
		case FLOCKING_interrupt:
		    break;
		case FLOCKING_non_block:
		case FLOCKING_block:
		    SIGDPRINT(Debug,13,(&Debug," %s",
					strerror(err)));
		    break;
		}		
	    }

	    if (err == EDEADLK && pass_block == a) {
		ret = FLOCKING_RETRY;
		
		SIGDPRINT(Debug,13,(&Debug,", deadlock\n"));
		break;
		
	    } else if ((err == EACCES) || (err == EAGAIN))
		ret = FLOCKING_RETRY;
	    else {
		ret = FLOCKING_FAIL;
		have_error = err;
	    }
	    break;
	}
	SIGDPRINT(Debug,13,(&Debug,"\n"));


	if (ret != FLOCKING_RETRY || pass_done == a)
	    goto done;

	switch (blockit) {
	case FLOCKING_interrupt:
	case FLOCKING_non_block:
	    goto done;
	case FLOCKING_block:
	    if (filename && !have_message && FLOCKING_release != op) {
		lib_transient(CATGETS(elm_msg_cat, MeSet, 
				      MeFileLocking,
				      "Locking %s..."),
			      filename);
		have_message = 1;
	    }
	}
    }

 done:
	
    switch (blockit) {
    case FLOCKING_interrupt:
	break;
    case FLOCKING_non_block:
    case FLOCKING_block:
	if (filename && have_message) {
	    if ( FLOCKING_OK == ret) {
		lib_transient(CATGETS(elm_msg_cat, MeSet, 
				      MeFileLockingOK,
				      "Locking %s... OK"),
			      filename);
	    } 
	    if (FLOCKING_FAIL == ret) {
		if (have_error)
		    lib_transient(CATGETS(elm_msg_cat, MeSet, 
					  MeFileLockingFail1,
					  "Locking %s... Fail: %s"),
				  filename,strerror(have_error));
		else
		    lib_transient(CATGETS(elm_msg_cat, MeSet, 
					  MeFileLockingFail,
					  "Locking %s... Fail"),
				  filename,strerror(have_error));	    		
	    }
	}
	break;
    }

    SIGDPRINT(Debug,13,(&Debug,"lock_op_fcntl=%d",ret));
    switch (ret) {
    case FLOCKING_FAIL:        SIGDPRINT(Debug,13,(&Debug," FLOCKING_FAIL"));        break;
    case FLOCKING_OK:          SIGDPRINT(Debug,13,(&Debug," FLOCKING_OK"));          break;
    case FLOCKING_RETRY:       SIGDPRINT(Debug,13,(&Debug," FLOCKING_RETRY"));       break;
    case FLOCKING_UNSUPPORTED: SIGDPRINT(Debug,13,(&Debug," FLOCKING_UNSUPPORTED")); break;
    }
    SIGDPRINT(Debug,13,(&Debug,"\n"));

    return ret;
}
#endif


#ifdef COMPILE_FLOCK_LOCKING
S_(lock_op_f lock_op_flock)
static  enum FLOCKING_status lock_op_flock P_((int                    fd,
					       enum FLOCKING_mode     op,
					       const char           * filename /* may be NULL */,
					       enum FLOCKING_blocking blockit,
					       int                  * errno_res));
static  enum FLOCKING_status lock_op_flock(fd,op,filename,blockit,errno_res)
     int                    fd;
     enum FLOCKING_mode     op;
     const char           * filename /* may be NULL */;
     enum FLOCKING_blocking blockit;
     int                  * errno_res;
{
    enum FLOCKING_status ret = FLOCKING_UNSUPPORTED;

    enum {
	pass_trylock,
	pass_block,
	pass_done } a;

    int have_message = 0;
    int have_error = 0;

    for (a = pass_trylock; a < pass_done; a++) {

	int cmd = 0;
	
	enum syscall_status r;
	const char *sop  UNUSED_VAROK = "BAD OP";
	const char *smod UNUSED_VAROK = "BAD MOD";

	switch (op) {
	case FLOCKING_release:   cmd = LOCK_UN;  sop = "LOCK_UN"; break;
	case FLOCKING_exclusive: cmd = LOCK_EX;  sop = "LOCK_EX"; break;
	case FLOCKING_shared:    cmd = LOCK_SH;  sop = "LOCK_SH"; break;
	}

	switch (a) {
	case pass_trylock: cmd |= LOCK_NB;    smod = "| LOCK_NB"; break;

	case pass_block: /* No modifier */    smod = "";          break;
	case pass_done:                                           break;
	}

	SIGDPRINT(Debug,13,(&Debug,"lock_op_flock: lockf fd=%d",
			 fd));
	if (filename) {
	    SIGDPRINT(Debug,13,(&Debug," %Q",filename));
	}

	SIGDPRINT(Debug,13,(&Debug,
			    " operation=%d %s %s\n",
			    cmd, sop,smod));

	r = flock(fd,cmd);

	SIGDPRINT(Debug,13,(&Debug,"lock_op_flock: flock returns %d",r));
	
	switch(r) {
	    int err;
	case syscall_success:
	    SIGDPRINT(Debug,13,(&Debug," success"));
	    ret = FLOCKING_OK;
	    break;
	case syscall_error:
	    err = errno;
	    if (errno_res)
		*errno_res = err;
	    
	    SIGDPRINT(Debug,13,(&Debug," error"));
	    if (err) {
		SIGDPRINT(Debug,13,(&Debug,", errno %d",
				    err));
		switch (blockit) {
		case FLOCKING_interrupt:
		    break;
		case FLOCKING_non_block:
		case FLOCKING_block:
		    SIGDPRINT(Debug,13,(&Debug," %s",
					strerror(err)));
		    break;
		}		
	    }
	    
	    if (err == EDEADLK && pass_block == a) {
		ret = FLOCKING_RETRY;
		
		SIGDPRINT(Debug,13,(&Debug,", deadlock\n"));
		break;
		
	    } else if ((err == EACCES) || (err == EAGAIN))
		ret = FLOCKING_RETRY;
	    else {
		ret = FLOCKING_FAIL;
		have_error = err;
	    }
	    break;
	}
	SIGDPRINT(Debug,13,(&Debug,"\n"));

	if (ret != FLOCKING_RETRY || pass_done == a)
	    goto done;
	
	switch (blockit) {
	case FLOCKING_interrupt:
	case FLOCKING_non_block:
	    goto done;
	case FLOCKING_block:
	    if (filename && !have_message && FLOCKING_release != op) {
		lib_transient(CATGETS(elm_msg_cat, MeSet, 
				      MeFileLocking,
				      "Locking %s..."),
			      filename);
		have_message = 1;
	    }
	}
	
    }

 done:

    switch (blockit) {
    case FLOCKING_interrupt:
	break;
    case FLOCKING_non_block:
    case FLOCKING_block:

	if (filename && have_message) {
	    if ( FLOCKING_OK == ret) {
		lib_transient(CATGETS(elm_msg_cat, MeSet, 
				      MeFileLockingOK,
				      "Locking %s... OK"),
			      filename);
	    } 
	    if (FLOCKING_FAIL == ret) {
		if (have_error)
		    lib_transient(CATGETS(elm_msg_cat, MeSet, 
					  MeFileLockingFail1,
					  "Locking %s... Fail: %s"),
				  filename,strerror(have_error));
		else
		    lib_transient(CATGETS(elm_msg_cat, MeSet, 
					  MeFileLockingFail,
					  "Locking %s... Fail"),
				  filename,strerror(have_error));	    		
	    }
	}
	break;
    }

    SIGDPRINT(Debug,13,(&Debug,"lock_op_flock=%d",ret));
    switch (ret) {
    case FLOCKING_FAIL:        SIGDPRINT(Debug,13,(&Debug," FLOCKING_FAIL"));        break;
    case FLOCKING_OK:          SIGDPRINT(Debug,13,(&Debug," FLOCKING_OK"));          break;
    case FLOCKING_RETRY:       SIGDPRINT(Debug,13,(&Debug," FLOCKING_RETRY"));       break;
    case FLOCKING_UNSUPPORTED: SIGDPRINT(Debug,13,(&Debug," FLOCKING_UNSUPPORTED")); break;
    }
    SIGDPRINT(Debug,13,(&Debug,"\n"));

    return ret;
}

#endif


static const struct lock_operations {
    enum mailbox_locking_v  locktype;
    lock_op_f             * lockop;
    const char            * debug_name;
} lock_operations[] = {
#ifdef COMPILE_FCNTL_LOCKING
    { lock_fcntl_flag, lock_op_fcntl, "fcntl", },
#endif
#ifdef COMPILE_FLOCK_LOCKING
    { lock_flock_flag, lock_op_flock, "flock" },
#endif
    { NUM_mailbox_locking_v, lock_op_dummy, NULL }
};

static  const int  lock_operations_count = (sizeof lock_operations) / sizeof (lock_operations[0]);


enum FLOCKING_status filelock_fd(fd,op,flags,filename,blockit,errno_res)
     int                    fd;
     enum FLOCKING_mode     op;
     struct dt_flags_info * flags;
     const char           * filename /* may be NULL */;
     enum FLOCKING_blocking blockit;
     int                  * errno_res;
{
    int i =  lock_operations_count /* Start position for FLOCKING_release !!! */;
    enum FLOCKING_status ret = FLOCKING_UNSUPPORTED;
    int lock_disabled = 0;


    switch (blockit) {
    case FLOCKING_interrupt:
	break;
    case FLOCKING_non_block:
    case FLOCKING_block:
	DEBUG_CHECK_INIT(Debug);
	break;
    }
    
    SIGDPRINT(Debug,13,(&Debug,
		     "filelock_fd: fd=%d op=%d",
		     fd,op));
    switch (op) {
    case FLOCKING_release:   SIGDPRINT(Debug,13,(&Debug, " FLOCKING_release"));     break;
    case FLOCKING_exclusive: SIGDPRINT(Debug,13,(&Debug, " FLOCKING_exclusive"));   break;
    case FLOCKING_shared:    SIGDPRINT(Debug,13,(&Debug, " FLOCKING_shared"));      break;
    }

    switch (blockit) {
	const char *s UNUSED_VAROK;

    case FLOCKING_interrupt:
	break;
    case FLOCKING_non_block:
    case FLOCKING_block:
	s = give_dt_flag_as_str(flags);
	SIGDPRINT(Debug,13,(&Debug," flags=%s",s));
	break;
    }
    
    if (filename) {
	SIGDPRINT(Debug,13,(&Debug," filename=%Q",filename));
    }

    SIGDPRINT(Debug,13,(&Debug," blockit=%d",blockit));    
    switch (blockit) {
    case FLOCKING_interrupt: SIGDPRINT(Debug,13,(&Debug," FLOCKING_interrupt")); break;
    case FLOCKING_non_block: SIGDPRINT(Debug,13,(&Debug," FLOCKING_non_block"));  break;
    case FLOCKING_block:     SIGDPRINT(Debug,13,(&Debug," FLOCKING_block"));      break;
    }
    
    SIGDPRINT(Debug,13,(&Debug,"\n"));

    if (errno_res)
	*errno_res = 0;

    
    
    switch (op) {
    case  FLOCKING_exclusive:
    case  FLOCKING_shared:
	
	for (i = 0;
	     i < lock_operations_count &&
		 lock_operations[i].locktype < NUM_mailbox_locking_v; i++) {

	    if (dt_flag_is_set(flags,lock_operations[i].locktype)) {
		int err = 0;
		enum FLOCKING_status x;
		
		SIGDPRINT(Debug,13,(&Debug,
				    "filelock_fd: %s %s ... %s\n",
				    op < 0 ? "unlocking" : "locking",
				    lock_operations[i].debug_name,
				    blockit ? " (blocking)" : ""));
		
		x = lock_operations[i].lockop(fd,op,filename,blockit,&err);
		
		SIGDPRINT(Debug,13,(&Debug,
				    "filelock_fd: %s gives %d",
				    lock_operations[i].debug_name,x));
		switch (x) {
		case FLOCKING_FAIL:        SIGDPRINT(Debug,13,(&Debug," FLOCKING_FAIL"));        break;
		case FLOCKING_OK:          SIGDPRINT(Debug,13,(&Debug," FLOCKING_OK"));          break;
		case FLOCKING_RETRY:       SIGDPRINT(Debug,13,(&Debug," FLOCKING_RETRY"));       break;
		case FLOCKING_UNSUPPORTED: SIGDPRINT(Debug,13,(&Debug," FLOCKING_UNSUPPORTED")); break;
		}
		
		if (err) {
		    SIGDPRINT(Debug,13,(&Debug," errno %d %s",
					err,strerror(err)));
		}
		
		if (FLOCKING_OK != x) {
		    ret = x;
		    if (errno_res)
			*errno_res = err;
		    
		    SIGDPRINT(Debug,13,(&Debug," -- canceling\n"));
		    break;
		}
		
		SIGDPRINT(Debug,13,(&Debug,"\n"));

		if (FLOCKING_UNSUPPORTED == ret) {
		    ret = x;
		}
		
	    } else 
		lock_disabled = 1;	    
	}

	if (lock_operations[i].locktype < NUM_mailbox_locking_v) {
	    SIGDPRINT(Debug,13,(&Debug,
				"filelock_fd: Interrupted at %s",
				lock_operations[i].debug_name));
	    
	    if (i > 0 && ret != FLOCKING_OK) {	   	    
		SIGDPRINT(Debug,13,(&Debug," -- unlocking\n"));
		
	    case FLOCKING_release:				
		for (i--; i >= 0; i--) {
		    
		    if (i < lock_operations_count &&
			lock_operations[i].locktype < NUM_mailbox_locking_v) {
			
			if (dt_flag_is_set(flags,lock_operations[i].locktype)) {
			
			    int err = 0;
			    enum FLOCKING_status x;
			    
			    SIGDPRINT(Debug,13,(&Debug,
						"filelock_fd: unlocking %s ...\n",
						lock_operations[i].debug_name));
			    
			    x = lock_operations[i].lockop(fd,FLOCKING_release,filename,blockit,&err);
			    
			    SIGDPRINT(Debug,13,(&Debug,
						"filelock_fd: %s gives %d",
						lock_operations[i].debug_name,x));
			    switch (x) {
			    case FLOCKING_FAIL:        SIGDPRINT(Debug,13,(&Debug," FLOCKING_FAIL"));
				break;
			    case FLOCKING_OK:          SIGDPRINT(Debug,13,(&Debug," FLOCKING_OK"));
				break;
			    case FLOCKING_RETRY:       SIGDPRINT(Debug,13,(&Debug," FLOCKING_RETRY"));
				break;
			    case FLOCKING_UNSUPPORTED: SIGDPRINT(Debug,13,(&Debug," FLOCKING_UNSUPPORTED"));
				break;			    
			    }
			    
			    if (err) {
				SIGDPRINT(Debug,13,(&Debug," errno %d %s",
						    err,strerror(err)));
			    }			
			    SIGDPRINT(Debug,13,(&Debug,"\n"));
			    
			    if (op ==  FLOCKING_release) {
				if (FLOCKING_UNSUPPORTED == ret ||
				    FLOCKING_FAIL == x ||
				    (x >  FLOCKING_OK && FLOCKING_OK == ret)) {
				    ret = x;
				    
				    if (FLOCKING_FAIL  == x ||
					FLOCKING_RETRY == x) {
					if (errno_res)
					    *errno_res = err;
				    }
				}
			    }
			    
			} else 
			    lock_disabled = 1;			
		    }
		    
		}
	    } else {
		SIGDPRINT(Debug,13,(&Debug,"\n"));
	    }
	    
	}

	break;
    }
	
    if (lock_disabled && FLOCKING_UNSUPPORTED == ret) {
	SIGDPRINT(Debug,13,(&Debug,"filelock_fd: Locking disabled.\n"));
	ret = FLOCKING_OK;
    }
    
    SIGDPRINT(Debug,13,(&Debug,
		     "filelock_fd=%d",
		     ret));
    switch (ret) {
    case FLOCKING_FAIL:        SIGDPRINT(Debug,13,(&Debug," FLOCKING_FAIL"));        break;
    case FLOCKING_OK:          SIGDPRINT(Debug,13,(&Debug," FLOCKING_OK"));          break;
    case FLOCKING_RETRY:       SIGDPRINT(Debug,13,(&Debug," FLOCKING_RETRY"));       break;
    case FLOCKING_UNSUPPORTED: SIGDPRINT(Debug,13,(&Debug," FLOCKING_UNSUPPORTED")); break;
    }
    if (errno_res) {
	SIGDPRINT(Debug,13,(&Debug,", *errno_res=%d",
			 *errno_res));
	switch (blockit) {
	case FLOCKING_interrupt:
	    break;
	case FLOCKING_non_block:
	case FLOCKING_block:
	    
	    if (*errno_res) {
	    SIGDPRINT(Debug,13,(&Debug," %s",strerror(*errno_res)));
	    }
	    break;
	}
    }    
    SIGDPRINT(Debug,13,(&Debug,"\n"));
    
    return ret;
}


    
/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */

  
