static char rcsid[] = "@(#)$Id: mediatype.c,v 2.11 2021/05/24 16:26:44 hurtta Exp $";

/******************************************************************************
 *  The Elm (ME+) Mail System  -  $Revision: 2.11 $   $State: Exp $
 *
 *  Author: Kari Hurtta <hurtta+elm@siilo.FMI.FI> 
 *                  (was hurtta+elm@posti.FMI.FI, hurtta+elm@ozone.FMI.FI)
 *      or  Kari Hurtta <elm@elmme-mailer.org>
 *****************************************************************************/


#include "def_misc.h"

DEBUG_VAR(Debug,__FILE__,"config");


#define   MIME_mask             0x00FFFF
#define   MIME_text             0x010000
#define   MIME_leaf             0x020000

#define MEDIA_TYPE_magic        0xFC03

struct media_type {
    unsigned short           magic;               /* MEDIA_TYPE_magic */
    enum mime_major_type     major_type_index;
    const char *             subtype;
    int                      flags;               /* MIME_xxx         */
    
    struct media_type_handle **handles;
    int                        handle_count;

    struct media_type * next;
};

/* MULTIPART types ------------------------------------------------------ */

static struct media_type multipart_8 =
{ MEDIA_TYPE_magic, MIME_TYPE_MULTIPART, "related", 0, 
  NULL, 0, NULL };                        /* RFC 2387 */
static struct media_type multipart_7 =
{ MEDIA_TYPE_magic, MIME_TYPE_MULTIPART, "parallel", 0, 
  NULL, 0, &multipart_8 };                /* RFC 2045 */
static struct media_type multipart_6 =
{ MEDIA_TYPE_magic, MIME_TYPE_MULTIPART, "encrypted", MIME_ENCRYPTED, 
  NULL, 0, &multipart_7, };               /* RFC 1847 */
static struct media_type multipart_5 =
{ MEDIA_TYPE_magic, MIME_TYPE_MULTIPART, "signed", MIME_SIGNED, 
  NULL, 0, &multipart_6 };                /* RFC 1847 */
static struct media_type multipart_4 =
{ MEDIA_TYPE_magic, MIME_TYPE_MULTIPART, "alternative", MIME_ALTERNATIVE, 
  NULL, 0, &multipart_5 };                /* RFC 2045 */ 
static struct media_type multipart_3 =
{ MEDIA_TYPE_magic, MIME_TYPE_MULTIPART, "digest", MIME_DIGEST, 
  NULL, 0, &multipart_4 };                /* RFC 2045 */
static struct media_type multipart_2 =
{ MEDIA_TYPE_magic, MIME_TYPE_MULTIPART, "report",  MIME_MIXED | MIME_REPORT, 
  NULL, 0, &multipart_3 };                /* RFC 3462 */
static struct media_type multipart_1 =
{ MEDIA_TYPE_magic, MIME_TYPE_MULTIPART, "mixed",   MIME_MIXED, 
  NULL, 0, &multipart_2 };                /* RFC 2045 */

/* MESSAGE types ----------------------------------------------------------- */

static struct media_type message_4 =
{ MEDIA_TYPE_magic, MIME_TYPE_MESSAGE, "disposition-notification", MIME_text, 
  NULL, 0, NULL };
static struct media_type message_3 =
{ MEDIA_TYPE_magic, MIME_TYPE_MESSAGE, "delivery-status", MIME_text, 
  NULL, 0, &message_4 };
static struct media_type message_2 =
{ MEDIA_TYPE_magic, MIME_TYPE_MESSAGE, "partial",  MIME_PARTIAL, 
  NULL, 0, &message_3 };
static struct media_type message_1 =
{ MEDIA_TYPE_magic, MIME_TYPE_MESSAGE, "rfc822",   MIME_RFC822, 
  NULL, 0, &message_2 };


/* APPLICATION types ------------------------------------------------------- */
struct media_type application_2 =
{ MEDIA_TYPE_magic, MIME_TYPE_APPLICATION, "octet-stream", 0, 
  NULL, 0, NULL };
struct media_type application_postscript =
{ MEDIA_TYPE_magic, MIME_TYPE_APPLICATION, "postscript", MIME_text, 
  NULL, 0, &application_2 };
static struct media_type application_1 =
{ MEDIA_TYPE_magic, MIME_TYPE_APPLICATION, "X-ELM-encode", MIME_text, 
  NULL, 0, &application_postscript };

/* TEXT types -------------------------------------------------------------- */

struct media_type text_plain =
{ MEDIA_TYPE_magic, MIME_TYPE_TEXT, "plain", MIME_text, 
  NULL, 0, NULL };

/* IMAGE  types ------------------------------------------------------------ */
struct media_type image_tiff = 
{ MEDIA_TYPE_magic, MIME_TYPE_IMAGE, "tiff",   0,
  NULL, 0, NULL };
struct media_type image_jpeg = 
{ MEDIA_TYPE_magic, MIME_TYPE_IMAGE, "jpeg",   0,
  NULL, 0, &image_tiff };
struct media_type image_gif = 
{ MEDIA_TYPE_magic, MIME_TYPE_IMAGE, "gif",   0,
  NULL, 0, &image_jpeg };

static struct major_type {
    const char        *   name;
    struct media_type *   list;

    int                   flags;           /* MIME_xxx */

    struct media_type_handle **handles;
    int                        handle_count;

} major_type [] = {
    /* As ordered on misclib.h */
    { "?unknown?",    NULL,             0,         NULL, 0 },
    { "application",  &application_1,   MIME_leaf, NULL, 0 },
    { "audio",        NULL,             MIME_leaf, NULL, 0 },
    { "image",        &image_gif,       MIME_leaf, NULL, 0 },
    { "message",      &message_1,       0,         NULL, 0 },
    { "multipart",    &multipart_1,     0,         NULL, 0 },
    { "text",         &text_plain,      MIME_leaf|MIME_text, NULL, 0 },
    { "video",        NULL,             MIME_leaf, NULL, 0 },
    { "font",         NULL,             MIME_leaf, NULL, 0 },
    { "model",        NULL,             MIME_leaf, NULL, 0 },
    { "example",      NULL,             0,         NULL, 0 },

    /* Non standard main types */
    { "x-world",      NULL,             MIME_leaf, NULL, 0 },   /* x-word/x-vmrl */
    { "chemical",     NULL,             MIME_leaf, NULL, 0 }, 

};

static struct major_type  * major_type_list = &(major_type[0]);
static int major_type_count = (sizeof major_type) / sizeof (major_type[0]);
#define  MAJOR_TYPE_MALLOCED   (major_type_list != &(major_type[0]))


/* WARNING: May return codes > MIME_TYPE_EXAMPLE if unknown or bad type */
enum mime_major_type get_major_type_code(T) 
     media_type_t T;
{
    if (!T)
	return MIME_TYPE_UNKNOWN;

    if (MEDIA_TYPE_magic != T->magic)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"get_major_type_code",
	      "Bad magic number",0);

    if (T->major_type_index < 0 ||
	T->major_type_index >= major_type_count)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"get_major_type_code",
	      "Bad major_type_index",0);

    return T->major_type_index;
}

char * TYPE_UNKNOWN_STRING = "?unknown?";

const char *         get_major_type_name(T)
     media_type_t T;
{
    if (!T)
	return TYPE_UNKNOWN_STRING;

    if (MEDIA_TYPE_magic != T->magic)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"get_major_type_name",
	      "Bad magic number",0);

    if (T->major_type_index < 0 ||
	T->major_type_index >= major_type_count)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"get_major_type_name",
	      "Bad major_type_index",0);

    return major_type_list[T->major_type_index].name;
}

const char *         get_major_type_name2(m)
     enum mime_major_type m;
{
    if (m < 0 || m >= major_type_count)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"get_major_type_name2",
	      "Bad major_type_index",0);

    return major_type_list[m].name;
}



int                  get_type_flags(T)
     media_type_t T;
{
    int ret;

    if (!T)
	return 0;

    if (MEDIA_TYPE_magic != T->magic)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"get_type_flags",
	      "Bad magic number",0);

    ret = T->flags & MIME_mask;

    if (T->major_type_index < 0 ||
	T->major_type_index >= major_type_count)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"get_type_flags",
	      "Bad major_type_index",0);

    DPRINT(Debug,12,(&Debug, 
		     "get_type_flags(%p)=%d     (%s/%s)\n",
		     T,ret,
		     major_type_list[T->major_type_index].name,
		     T->subtype));

    return ret;
}

const char *         get_subtype_name(T)
     media_type_t T;
{

    if (!T)
	return TYPE_UNKNOWN_STRING;

    if (MEDIA_TYPE_magic != T->magic)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"get_subtype_name",
	      "Bad magic number",0);

    if (! T->subtype)
	return TYPE_UNKNOWN_STRING;
  
    return T->subtype;
}

enum mime_major_type give_major_type(major,create)
     const char * major;
     int create;
{
    int i;

    if (NULL != strchr(major,'/'))
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"give_major_type",
	      "Bad major type",0);

    if (TYPE_UNKNOWN_STRING == major)
	return MIME_TYPE_UNKNOWN;
    	
    for (i = 0; i < major_type_count; i++)
	if (0 == istrcmp(major,major_type_list[i].name))
	    break;

    if (i >= major_type_count) {
	if (!create)
	    return MIME_TYPE_UNKNOWN;

	if (!MAJOR_TYPE_MALLOCED) {
	    major_type_list = safe_calloc((major_type_count+1),
					  sizeof (major_type_list[0]));
	    for (i = 0; i < major_type_count; i++)
		major_type_list[i] = major_type[i];
	} else
	    major_type_list = safe_array_realloc(major_type_list,
						 (major_type_count+1),
						 sizeof (major_type_list[0]));
	
	major_type_list[major_type_count].name = safe_strdup(major);
	major_type_list[major_type_count].list = NULL;

	/* Perhaps should be MIME_leaf, but this is not known
           because major type is unknown !!
	*/
	major_type_list[major_type_count].flags = 0;


	major_type_list[major_type_count].handles      = NULL;
	major_type_list[major_type_count].handle_count = 0;

	i = major_type_count++;

	DPRINT(Debug,10,(&Debug, 
			 "give_major_type: Created major type %d: %s\n",
			 i,major_type_list[i].name));
    }
    
    return (enum mime_major_type)i;
}

media_type_t give_media_type(major,minor,create)
     const char * major;
     const char * minor;
     int create;
{

    enum mime_major_type I1 = give_major_type(major,create);

    if (MIME_TYPE_UNKNOWN == I1 && !create)
	return NULL;

    return give_media_type2(I1,minor,create);
}

media_type_t give_media_type2(major_type_code,minor,create)
     enum mime_major_type major_type_code;
     const char * minor;
     int create;
{
    struct media_type * p;

    if (major_type_code < 0 ||
	major_type_code >= major_type_count)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"give_media_type2",
	      "Bad major_type_code",0);


    for (p = major_type_list[major_type_code].list; p; p = p->next) {
	if (major_type_code != p->major_type_index) 
	    panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"give_media_type",
		  "Bad major_type_index",0);
	if (0 == istrcmp(p->subtype,minor))
	    break;
    }

    if (!p) {
	if (!create)
	    return NULL;

	p =  safe_zero_alloc(sizeof (*p));

	p->magic             = MEDIA_TYPE_magic;
	p->major_type_index  = major_type_code;
	p->subtype           = safe_strdup(minor);
	p->flags             = 0;
	
	p->handles           = NULL;
	p->handle_count      = 0;

	if (major_type_list[major_type_code].flags & MIME_text)
	    p->flags  |=  MIME_text;

	p->next              = major_type_list[major_type_code].list;
	major_type_list[major_type_code].list = p;

	DPRINT(Debug,10,(&Debug, 
			 "give_media_type2: Created minor type %p: %s\n",
			 p,p->subtype));
    }

    DPRINT(Debug,12,(&Debug, 
		     "give_media_type2([%s]/%s)=%p\n",
		     major_type_list[p->major_type_index].name,
		     p->subtype,p));
    return p;
}

int give_text_type_code(T)
     media_type_t T;
{
    if (!T)
	return -1;

    if (MEDIA_TYPE_magic != T->magic)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"give_text_type_code",
	      "Bad magic number",0);

    if (T->major_type_index < 0 ||
	T->major_type_index >= major_type_count)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"give_text_type_code",
	      "Bad major_type_index",0);
    
    if (!(major_type_list[T->major_type_index].flags & MIME_leaf)) {
	DPRINT(Debug,11,(&Debug,"give_text_type_code=-1 for %p (%s, major %d)\n",
			 T,T->subtype,T->major_type_index));
	return -1;
    }    

    if (T->flags & MIME_text) {
	DPRINT(Debug,11,(&Debug,"give_text_type_code=1 for %p (%s, major %d)\n",
			 T,T->subtype,T->major_type_index));

	return 1;
    }

    DPRINT(Debug,11,(&Debug,"give_text_type_code=0 for %p (%s, major %d)\n",
		     T,T->subtype,T->major_type_index));
    return 0;
}

void register_mt_handler(T,H)
     media_type_t T;
     struct media_type_handle *H;
{
    if (MEDIA_TYPE_magic != T->magic)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"register_mt_handler",
	      "Bad magic number",0);

    if (handle_ANY == H->type)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"register_mt_handler",
	      "Bad handle type",0);

    T->handles =  safe_array_realloc(T->handles,
				     (T->handle_count+1),
				     sizeof (T->handles[0]));
    T->handles[T->handle_count] = H;

    DPRINT(Debug,11,(&Debug, 
		     "register_mt_handler: [%d] Adding handler %p for %p (%s), kind %d\n",
		     T->handle_count,H,T,T->subtype,H->type));
    T->handle_count++;
}

void register_mt_defhandler(major_type_code,H)
     int major_type_code;
     struct media_type_handle *H;
{
    if (major_type_code == MIME_TYPE_LEAF) {
	int i;

	for (i = 0; i <  major_type_count; i++) {
	    if (major_type_list[i].flags & MIME_leaf)
		register_mt_defhandler(i,H);
	}

	return;
    }

    if (major_type_code < 0 ||
	major_type_code >= major_type_count)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"register_mt_defhandler",
	      "Bad major_type_code",0);
    
    if (handle_ANY == H->type)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"register_mt_defhandler",
	      "Bad handle type",0);
    
    major_type_list[major_type_code].handles =  
	safe_array_realloc(major_type_list[major_type_code].handles,
			   (major_type_list[major_type_code].handle_count+1),
			   sizeof (major_type_list[major_type_code].handles[0]));
    major_type_list[major_type_code].
	handles[major_type_list[major_type_code].handle_count] = H;

    DPRINT(Debug,11,(&Debug, 
		     "register_mt_defhandler: [%d] Adding default handler %p for %d (%s), kind %d\n",
		     major_type_list[major_type_code].handle_count,
		     H,major_type_code,
		     major_type_list[major_type_code].name,
		     H->type));

    major_type_list[major_type_code].handle_count++;
}

int walk_mt_handler(T,H,is_default,kind)
     media_type_t T;
     struct media_type_handle **H;
     int *is_default;
     enum  mt_handle_type kind;
{
    if (!T) {
	*H = NULL;
	return 0;
    }


    if (MEDIA_TYPE_magic != T->magic)
	panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"walk_mt_handler",
	      "Bad magic number",0);

    if (0 == *is_default) {
	if (NULL == *H) {
	    int x;

	    DPRINT(Debug,11,(&Debug, 
			     "walk_mt_handler: Searching first handler for %p (%s), kind %d (%d handlers)\n",
			     T,T->subtype,kind,T->handle_count));

	    for (x = 0; x < T->handle_count; x++) {
		if (handle_ANY == kind || 
		    T->handles[x]->type == kind) {
		    *H = T->handles[x];
		    return 1;
		}
	    }	
	} else {
	    int i,x;
	    for (i = 0; i < T->handle_count; i++) {
		if (*H == T->handles[i] &&
		    (handle_ANY == kind || T->handles[i]->type == kind))
		    break;
	    }
	    if (i >= T->handle_count) {
		panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"walk_mt_handler",
		      "previous non-default handler not found",0);
		return 0;   /* NOT REACHED */
	    }

	    for (x = i+1; x < T->handle_count; x++) {
		if (handle_ANY == kind || 
		    T->handles[x]->type == kind) {
		    *H = T->handles[x];
		    return 1;
		}
	    }	
	}
	*is_default  = 1;
	*H           = NULL;   /* Not in default list */
    }

    if (1 == *is_default) {
	int idx = T->major_type_index;

	if (idx < 0 || idx >= major_type_count)
	    panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"walk_mt_handler",
		  "Bad major_type_index",0);

	if (NULL == *H) {
	    int x;

	    DPRINT(Debug,11,(&Debug, 
			     "walk_mt_handler: Searching default handler for %d (%s), kind %d (%d handlers)\n",
			     idx,major_type_list[idx].name,kind,major_type_list[idx].handle_count));
	    for (x = 0; x < major_type_list[idx].handle_count; x++) {
		if (handle_ANY == kind || 
		    major_type_list[idx].handles[x]->type == kind) {
		    *H = major_type_list[idx].handles[x];
		    return 1;
		}
	    }
	} else {
	    int i,x;
	    for (i = 0; i < major_type_list[idx].handle_count; i++) {
		if (*H == major_type_list[idx].handles[i] &&
		    (handle_ANY == kind || 
		     major_type_list[idx].handles[i]->type == kind))
		    break;
	    }
	    if (i >= major_type_list[idx].handle_count) {
		panic("MEDIA TYPE PANIC",__FILE__,__LINE__,"walk_mt_handler",
		      "previous defualt handler not found",0);
		return 0;   /* NOT REACHED */
	    }
	    for (x = i+1; x < major_type_list[idx].handle_count; x++) {
		if (handle_ANY == kind || 
		    major_type_list[idx].handles[x]->type == kind) {
		    *H = major_type_list[idx].handles[x];
		    return 1;
		}
	    }
	}
    }
    
    DPRINT(Debug,11,(&Debug, 
		     "walk_mt_handler: All handlers searched for %p (%s), kind %d\n",			     
		     T,T->subtype,kind));

    *H = NULL;
    return 0;
}

/*
 * Local Variables:
 *  mode:c
 *  c-basic-offset:4
 *  buffer-file-coding-system: iso-8859-1
 * End:
 */
